/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_audiorouting_lib.cpp
* @brief       CCA client-handler implementation for the service audio routing library.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/


#include "fc_sxm_tcl_base_app.h"
#include "fc_sxm_audiorouting_lib.h"
#include "fc_sxm_tcl_playback.h"
#include "fc_sxm_tcl_audio_app.h"
#include "fc_sxm_tcl_wsalerts_app.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_AUDIOROUTING_LIB
#include "trcGenProj/Header/fc_sxm_audiorouting_lib.cpp.trc.h"
#endif

/******************************************************************************/
/*                                                                            */
/* DEFINES                                                                    */
/*                                                                            */
/******************************************************************************/

/*
 constructor: call contructor of base-class accordingly
 */
fc_sxm_arl_tclISource::fc_sxm_arl_tclISource()
         : arl_tclISource(fc_sxm_tclApp::instance())

{
   _bMuteState = FALSE;
#if((FC_SXM_ENABLE_WATCHDOG_TIMER) || (FC_SXM_ENABLE_LCM_WATCHDOG_TIMER))
   _enSourceState = ARL_EN_ISRC_ACT_OFF;
#endif
   ETG_TRACE_USR4(("fc_sxm_arl_tclISource() entered."));
}

/*
 Destructor
 */
fc_sxm_arl_tclISource::~fc_sxm_arl_tclISource(tVoid)
{
   ETG_TRACE_USR4(("~fc_sxm_arl_tclISource() entered."));
}

/*******************************************************************************
 *
 * FUNCTION: tBool fc_sxm_arl_tclISource::bOnAllocate()
 *
 * DESCRIPTION: This function is called by the AUDIO-ROUTING-LIB after Allocate
 *              is processed.
 *
 * PARAMETER:[enSrcNum]:  (I) Source Number.
 *           [rfcoAllocRoute]: (I) Reference to Allocate route result
 *
 * RETURNVALUE: [tBool]: TRUE, if Application performed operations successfully,
 *              FALSE otherwise
 *
 ********************************************************************************
 * Overrides method arl_tclISource::bOnAllocate(..)
 *******************************************************************************/
tBool fc_sxm_arl_tclISource::bOnAllocate(arl_tenSource enSrcNum, const arl_tAllocRouteResult& /*rfcoAllocRoute*/)
{
   ETG_TRACE_USR4(("fc_sxm_arl_tclISource::bOnAllocate(),  enSrcNum = %d", enSrcNum));
   return TRUE;
}

/*******************************************************************************
 *
 * FUNCTION: tBool fc_sxm_arl_tclISource::bOnDeAllocate()
 *
 * DESCRIPTION: This function is called by the CALLED BY AUDIO-ROUTING-LIB
 *               after DeAllocate is processed.
 *              
 * PARAMETER: [enSrcNum]:  (I) Source Number..
 *
 * RETURNVALUE: [tBool]: TRUE, if Application performed operations successfully,
 *               FALSE otherwise.
 *
 ********************************************************************************
 * Overrides method arl_tclISource::bOnDeAllocate(..)
 *******************************************************************************/
tBool fc_sxm_arl_tclISource::bOnDeAllocate(arl_tenSource enSrcNum)
{
   ETG_TRACE_USR4(("fc_sxm_arl_tclISource::bOnDeAllocate(),  enSrcNum = %d", enSrcNum));
   return TRUE;
}

tBool fc_sxm_arl_tclISource::bPostSourceActivityToAudio(arl_tenSource enSrcNum, const arl_tSrcActivity& rfcoSrcActivity)
{
   fc_sxm_trMsgAudioMuteState bMuteState;
   bMuteState.eSourceActivity = e_SXM_EN_OFF;

   if(rfcoSrcActivity.enType == arl_tSrcActivity::FI_EN_OFF)
   {
      ETG_TRACE_USR4(("vSourceActivityResult() OFF Sent"));
      vSourceActivityResult(enSrcNum, ARL_EN_ISRC_ACT_OFF);
      bMuteState.eSourceActivity = e_SXM_EN_OFF;
#if((FC_SXM_ENABLE_WATCHDOG_TIMER) || (FC_SXM_ENABLE_LCM_WATCHDOG_TIMER))
      _enSourceState = ARL_EN_ISRC_ACT_OFF;
#endif
   }
   else if(rfcoSrcActivity.enType == arl_tSrcActivity::FI_EN_ON)
   {
      ETG_TRACE_USR4(("vSourceActivityResult() ON Sent"));
      vSourceActivityResult(enSrcNum, ARL_EN_ISRC_ACT_ON);
      bMuteState.eSourceActivity = e_SXM_EN_ON;
#if((FC_SXM_ENABLE_WATCHDOG_TIMER) || (FC_SXM_ENABLE_LCM_WATCHDOG_TIMER))
      _enSourceState = ARL_EN_ISRC_ACT_ON;
#endif
   }
   else
   {
      ETG_TRACE_USR4(("vSourceActivityResult() PAUSE Sent"));
      vSourceActivityResult(enSrcNum, ARL_EN_ISRC_ACT_PAUSE);
      bMuteState.eSourceActivity = e_SXM_EN_PAUSE;
#if((FC_SXM_ENABLE_WATCHDOG_TIMER) || (FC_SXM_ENABLE_LCM_WATCHDOG_TIMER))
      _enSourceState = ARL_EN_ISRC_ACT_PAUSE;
#endif
   }
   /* Post mute status to sxm audio thread */
   fc_sxm_tclAudioApp::instance()->vPostMsgNew(bMuteState);

   return TRUE;
}

tBool fc_sxm_arl_tclISource::bPostSourceActivityToTTSClient(arl_tenSource /*enSrcNum*/, const arl_tSrcActivity& rfcoSrcActivity)
{
   tBool bResult = FALSE;

   fc_sxm_trMsgWsAlertsAudioSource oSourceActivityUpdate;
   if(midw_fi_tcl_e8_SrcActivity::FI_EN_ON == rfcoSrcActivity.enType)
   {
      oSourceActivityUpdate.enSourceActivity = FC_SXM_EN_AUDIO_SOURCE_ON;
   }
   else if(midw_fi_tcl_e8_SrcActivity::FI_EN_PAUSE == rfcoSrcActivity.enType)
   {
      oSourceActivityUpdate.enSourceActivity = FC_SXM_EN_AUDIO_SOURCE_PAUSE;
   }
   else if(midw_fi_tcl_e8_SrcActivity::FI_EN_OFF == rfcoSrcActivity.enType)
   {
      oSourceActivityUpdate.enSourceActivity = FC_SXM_EN_AUDIO_SOURCE_OFF;
   }

   fc_sxm_tclWsAlertsApp::instance()->vPostMsgNew(oSourceActivityUpdate);
   return bResult;
}

/*******************************************************************************
 *
 * FUNCTION: tBool fc_sxm_arl_tclISource::bOnSrcActivity()
 *
 * DESCRIPTION: This function is called by the CALLED BY AUDIO-ROUTING-LIB
 *               on Source Activity start.
 *              
 * PARAMETER: [enSrcNum]:  (I) Source Number..
 *            [rfcoSrcActivity]: (I) Source Activity
 *
 * RETURNVALUE: [tBool]: TRUE, if source activity was successful,
 *               FALSE otherwise.
 *
 ********************************************************************************
 * Overrides method arl_tclISource::bOnSrcActivity(..)
 *******************************************************************************/
tBool fc_sxm_arl_tclISource::bOnSrcActivity(arl_tenSource enSrcNum, const arl_tSrcActivity& rfcoSrcActivity)
{
   tBool bResult = FALSE;
   ETG_TRACE_USR4(("fc_sxm_arl_tclISource::bOnSrcActivity() entered"));
   if(enSrcNum == ARL_SRC_XM)
   {
      bResult = bPostSourceActivityToAudio(enSrcNum, rfcoSrcActivity);
   }
   else if(enSrcNum == ARL_SRC_SXM_NEWS)
   {
      ETG_TRACE_USR4(("bOnSrcActivity(), enSrcNum=%d", enSrcNum));
      vSourceActivityResult(enSrcNum, rfcoSrcActivity);
   }
   else if(enSrcNum == ARL_SRC_SXM_TTS)
   {
      vSourceActivityResult(enSrcNum, rfcoSrcActivity);
      bResult = bPostSourceActivityToTTSClient(enSrcNum, rfcoSrcActivity);
   }
   else
   {
      ETG_TRACE_USR4(("fc_sxm_arl_tclISource::bOnSrcActivity() not for XM or SXM NEWS or XM TTS source!"));
   }
   return bResult;
}

/*******************************************************************************
 *
 * FUNCTION: tBool fc_sxm_arl_tclISource::bOnMuteState()
 *
 * DESCRIPTION: This function is called by the CALLED BY AUDIO-ROUTING-LIB
 *              when AudioRouting changed Mute state for given Source.
 *              
 *
 * PARAMETER: [enSrcNum]:  (I) Source Number..
 *            [enMuteState]: systems actual mute state
 *
 * RETURNVALUE: [tBool]: TRUE
 *                       FALSE 
 *
 ********************************************************************************
 * Overrides method arl_tclISource::bOnMuteState(..)
 *******************************************************************************/
tBool fc_sxm_arl_tclISource::bOnMuteState(arl_tenSource /*enSrcNum*/, arl_tenMuteState /*enMuteState*/)
{
   ETG_TRACE_USR4(("fc_sxm_arl_tclISource::bOnMuteState() entered"));
   return FALSE;
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_arl_tclISource::bSetMute()
 *
 * DESCRIPTION: This function is called by the CALLED BY SXM Audio
 *              when Muting and Demuting needs to be done
 *
 * PARAMETER: [bMuteFlag]:  TRUE - Mute, FALSE - Demute
 *
 * RETURNVALUE:
 *
 *******************************************************************************/
tVoid fc_sxm_arl_tclISource::vSetMute(tBool bMuteFlag)
{
   ETG_TRACE_USR4(("fc_sxm_arl_tclISource::fc_sxm_SetMute() entered, bMuteFlag = %u \t prevMuteState = %u", bMuteFlag, _bMuteState));

   if(_bMuteState != bMuteFlag)
   {
      /* Save the mute state */
      _bMuteState = bMuteFlag;

      /* Set the mute action */
      arl_tenMuteAction enMuteAction = bMuteFlag ? ARL_EN_MUTEACTION_MUTE : ARL_EN_MUTEACTION_DEMUTE;

      /* Perform Muting and Demuting */
      bSetMuteState(ARL_SRC_XM, enMuteAction);
   }
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_arl_tclISource::vSendSourceAvailability()
 *
 * DESCRIPTION: Interface can be used to set SXM source availability
 *
 * PARAMETER: srcAvailability, availabilityReason
 *
 * RETURNVALUE: tVoid
 *
 *******************************************************************************/
tVoid fc_sxm_arl_tclISource::vSendSourceAvailability(arl_tenSrcAvailability srcAvailability, arl_tenAvailabilityReason availabilityReason)
{
	tBool blResult = bSetSourceAvailability(srcAvailability, availabilityReason, ARL_SRC_XM);
	ETG_TRACE_USR1(("fc_sxm_arl_tclISource::vSendSourceAvailability() srcAvailability = %d availabilityReason = %d blResult = %d",
			srcAvailability, availabilityReason, blResult));
}

tVoid fc_sxm_arl_tclISource::vProcess(fc_sxm_trMsgCmdArlTestLoopBack const *prMsg) const
{
   SXM_ASSERT_RETURN(OSAL_NULL !=prMsg);
   ETG_TRACE_USR4(("fc_sxm_arl_tclISource::vProcess(fc_sxm_trMsgCmdArlTestLoopBack) u8TestVal=%d", prMsg->u8TestVal));
}

// test of internal loopback
tVoid fc_sxm_arl_tclISource::vDispatchMsg(fc_sxm_tclMessage const *poThreadMsg)
{
   ETG_TRACE_USR4(("fc_sxm_arl_tclISource::vDispatchMsg: SID=%x action=%d", ETG_CENUM(fc_sxm_tenServiceID,
      poThreadMsg->enGetServiceId()), poThreadMsg->u16GetActionOnly()));
   tU32 u32Action = poThreadMsg->u32GetAction();
   switch(u32Action)
   {
   SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgCmdArlTestLoopBack, fc_sxm_arl_tclISource::instance());
   default:
      break;
   }
}

tBool fc_sxm_arl_tclISource::bRequestAudioRoute(arl_tenSource enAudioSource, arl_tenActivity enActivity)
{
   return (bSetAudioRouteRequest(enAudioSource, enActivity));
}
