/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_twnow.cpp
* @brief       Implementation for Traffic and Weather Now service.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_audio_app_if.h"
#include "fc_sxm_tcl_twnow.h"
#include "fc_sxm_tcl_audio_properties.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_AUDIO_TWNOW
#include "trcGenProj/Header/fc_sxm_tcl_twnow.cpp.trc.h"
#endif

// Constructor for the class
fc_sxm_tclTWNow::fc_sxm_tclTWNow() :
		_poAudioApp(OSAL_NULL), _poTWNowProxy(OSAL_NULL), _bIsTWNowInitialized(FALSE), _bEnableTWNow(TRUE),_bIsTWNowPlayback(FALSE)
{
	// Default CTOR
	_vectMarkets.clear();
}

// Destructor for the class
fc_sxm_tclTWNow::~fc_sxm_tclTWNow()
{
	// DTOR
	_poAudioApp = OSAL_NULL;
	_poTWNowProxy = OSAL_NULL;
}

// Initialize all member variables
tVoid fc_sxm_tclTWNow::vInit()
{
	_vectMarkets.clear();
	_bIsTWNowInitialized = FALSE;
	_bEnableTWNow = TRUE;
	_bIsTWNowPlayback = FALSE;
}

// De-initialize all member variables
tVoid fc_sxm_tclTWNow::vDeInit()
{
	_vectMarkets.clear();
	_bIsTWNowInitialized = FALSE;
	_bEnableTWNow = TRUE;
}

// Clear All Member Data
tVoid fc_sxm_tclTWNow::vClearMemberData(tVoid)
{
	_vectMarkets.clear();
	_bIsTWNowInitialized = FALSE;
	_bEnableTWNow = TRUE;
	_bIsTWNowPlayback = FALSE;
	// Ensure TW_Now service is stopped as
	// the Decoder Object associated with the service would be stopped
	vStopTWNow();
}

// Set Audio App and TWNow Pointer References
tVoid fc_sxm_tclTWNow::vSetApp(Ifc_sxm_tclAudioApp* poAudioApp, Ifc_sxm_tclTWNowProxy* poTWNow)
{
	ETG_TRACE_USR4(("fc_sxm_tclTWNow::vSetApp()"));
	_poAudioApp = poAudioApp;
	_poTWNowProxy = poTWNow;
}

// Start TWNow service once decoder is up and is in ready state
tVoid fc_sxm_tclTWNow::vUpdateDecoderState( DECODER_OBJECT hDecoderObject,
		DECODER_STATE_ENUM enDecoderState)
{
	if (( FALSE == _bIsTWNowInitialized) &&
			(DECODER_STATE_READY == enDecoderState) &&
			(DECODER_INVALID_OBJECT != hDecoderObject) )
	{
		//Start TWNow Service
		_bIsTWNowInitialized = TRUE;
		const SMSAPI_RETURN_CODE_ENUM enReturnCode (enStartTWNow(hDecoderObject));
		if (SMSAPI_RETURN_CODE_SUCCESS == enReturnCode)
		{
			vGetAvailableMarkets();
		}
		else
		{
			vNotifyTWNowServiceState(enReturnCode, TRUE);
		}
	}
}

// Wrapper method to invoke TW_NOW.eStart()
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclTWNow::enStartTWNow(DECODER_OBJECT hDecoder)
{
	SMSAPI_RETURN_CODE_ENUM enReturnCode = (( hDecoder != DECODER_INVALID_OBJECT ) && (_poTWNowProxy != OSAL_NULL)) ?
			_poTWNowProxy->enStartTWNow(hDecoder, this) :
			SMSAPI_RETURN_CODE_ERROR ;
	ETG_TRACE_USR1(("Start TWNow Service \t enReturnCode = %u", ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enReturnCode)));
	return enReturnCode;
}

// Stop T&W Now service and release the object handle.
tVoid fc_sxm_tclTWNow::vStopTWNow() const
{
	DECODER_OBJECT hDecoder(hGetDecoder());
	if (( hDecoder != DECODER_INVALID_OBJECT ) && (_poTWNowProxy != OSAL_NULL))
	{
		_poTWNowProxy->vStopTWNow(hDecoder);
		ETG_TRACE_USR2(("Stop TWNow Service"));
		vNotifyTWNowServiceState(SMSAPI_RETURN_CODE_SUCCESS, FALSE);
	}
}

// Set TWNow Market
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclTWNow::enSetTWNowMarket(CID_OBJECT marketID) const
{
	DECODER_OBJECT hDecoder(hGetDecoder());
	SMSAPI_RETURN_CODE_ENUM enReturnCode = (( hDecoder != DECODER_INVALID_OBJECT ) && (_poTWNowProxy != OSAL_NULL)) ?
			_poTWNowProxy->enSetTWNowMarket(hDecoder, marketID) :
			SMSAPI_RETURN_CODE_ERROR ;
	ETG_TRACE_USR2(("Set TWNow Market \t enReturnCode = %u", ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enReturnCode) ));
	return enReturnCode;
}

// Get currently set TWNow market's CID object
CID_OBJECT fc_sxm_tclTWNow::hGetCurrentTWNowMarket() const
{
	CID_OBJECT marketID = CID_INVALID_OBJECT;
	DECODER_OBJECT hDecoder(hGetDecoder());
	if (( hDecoder != DECODER_INVALID_OBJECT ) && (_poTWNowProxy != OSAL_NULL))
	{
		// No point enquiring for Current Market conditions if decoder object is invalid
		marketID = _poTWNowProxy->hGetCurrentTWNowMarket(hDecoder);
		if ( CID_INVALID_OBJECT == marketID)
			ETG_TRACE_ERR(("TWNow Market Not Set YET"));
	}
	return marketID;
}

// Play the user provided TWNow BulletinID
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclTWNow::enPlayTWNowBulletin(tU32 u32BulletinID) const
{
	DECODER_OBJECT hDecoder(hGetDecoder());
	SMSAPI_RETURN_CODE_ENUM enReturnCode = (( hDecoder != DECODER_INVALID_OBJECT ) && (_poTWNowProxy != OSAL_NULL)) ?
			_poTWNowProxy->enPlayTWNowBulletin(hDecoder, u32BulletinID) :
			SMSAPI_RETURN_CODE_ERROR;
	ETG_TRACE_USR2(("Play TWNow BulletInID(%u) \t enReturnCode = %u",
			u32BulletinID, ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enReturnCode) ));
	return enReturnCode;
}

// Abort Playback of TWNow BulletinID
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclTWNow::enAbortTWNowBulletin(tU32 u32BulletinID) const
{
	//Abort playback of the Traffic/Weather report before it has been fully played and
	// return to the previous channel.
	DECODER_OBJECT hDecoder(hGetDecoder());
	SMSAPI_RETURN_CODE_ENUM enReturnCode = (( hDecoder != DECODER_INVALID_OBJECT ) && (_poTWNowProxy != OSAL_NULL)) ?
			_poTWNowProxy->enAbortTWNowBulletin(hDecoder) :
			SMSAPI_RETURN_CODE_ERROR;
	ETG_TRACE_USR2(("Abort TWNow BulletInID(%u) \t enReturnCode = %u",
			u32BulletinID, ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enReturnCode) ));
	return enReturnCode;
}

// Retrieve current available TWNowBulletinID
tU32 fc_sxm_tclTWNow::u32GetCurrentTWNowBulletinID() const
{
	// SMS call for getTWNowBulletinID would return TW_NOW_BULLETIN_ID value if
	// T&W Bulletin playback is in effect or TW_NOW_INVALID_BULLETIN_ID in case of normal tune.
	tU32 u32BulletinID = TW_NOW_INVALID_BULLETIN_ID;
	DECODER_OBJECT hDecoder(hGetDecoder());
	if (( hDecoder != DECODER_INVALID_OBJECT ) && (_poTWNowProxy != OSAL_NULL))
		u32BulletinID = _poTWNowProxy->u32GetCurrentTWNowBulletinID(hDecoder);
	ETG_TRACE_USR2(("TWNow Current Playing BulletinID = %u", u32BulletinID ));
	return u32BulletinID;
}

// Handle callbacks reported for TWNow Events
tVoid fc_sxm_tclTWNow::vHandleTWNowEvent(
		DECODER_OBJECT /*hDecoder*/,
		TW_NOW_BULLETIN_ID u32BulletinID,
		TW_NOW_BULLETIN_STATUS_ENUM enBulletinStatus,
		CID_OBJECT /* marketID */ ) const
{
	ETG_TRACE_USR2(("vHandleTWNowEvent \t _bEnableTWNow(%u) \t  BulletinID(%u) \t enBulletinStatus(%u)",
			_bEnableTWNow, u32BulletinID, ETG_CENUM(TW_NOW_BULLETIN_STATUS_ENUM, enBulletinStatus) ));
	if (_poAudioApp)
	{
		fc_sxm_trMsgTWNowEventAlert trFlashEvent(u32BulletinID, enBulletinStatus);
		_poAudioApp->vTWNowEventAlertMsg(trFlashEvent);
	}
}

// Callback from REPORT.bIterateContent for every market
tBool fc_sxm_tclTWNow::bAddTWNowMarket(const char* pacId, const char* pacName,CID_OBJECT hId)
{
	tBool bContinueIteration = FALSE;
	fc_sxm_trTWNowMarkets trMarket(pacId, pacName, hId);
	// Don't entertain invalid markets
	if (trMarket.bValidMarket())
	{
		bContinueIteration = TRUE;
		_vectMarkets.push_back(trMarket);
	}
	return bContinueIteration;
}

// Callback method to be invoked on expiry of available markets timer
tVoid fc_sxm_tclTWNow::vGetAvailableMarkets()
{
	ETG_TRACE_USR2(("fc_sxm_tclTWNow::vGetAvailableMarkets" ));
	// Timer for available markets enquiry to SMS is expired.
	// Check if we have received available markets
	if (TRUE == bGetAvailableMarkets())
	{
		vNotifyTWNowServiceState(SMSAPI_RETURN_CODE_SUCCESS, TRUE);
	}
	else if(_poTWNowProxy != OSAL_NULL)
	{
		const tU32 u32DelayTime = 5000U; // 5 seconds
		// TWNow market details are yet to be received OTA ... so restart the timer
		_poTWNowProxy->vStartPollingAvailableMarkets(u32DelayTime, this);
	}
}

// Destroy CID object instance
tVoid fc_sxm_tclTWNow::vDestroyCIDObject(CID_OBJECT hCID) const
{
	if (_poTWNowProxy != OSAL_NULL)
		_poTWNowProxy->vDestroyCIDObject(hCID);
}

// Compares 2 CID objects for equality
tS16 fc_sxm_tclTWNow::s16CompareCIDObjects(CID_OBJECT hCID1, CID_OBJECT hCID2) const
{
	return (( (_poTWNowProxy != OSAL_NULL) ?
			_poTWNowProxy->s16CompareCIDObjects(hCID1, hCID2)  :  -1 ));
}

// Notify clients about the change in TWNowServiceState
// Method would be invoked when 1) Starting of TW_NOW Service; 2) Change in service state; 3) Stop TWNOW
tVoid fc_sxm_tclTWNow::vNotifyTWNowServiceState (const SMSAPI_RETURN_CODE_ENUM enRetCode, tBool bStartService) const
{
	if (_poAudioApp)
	{
		fc_sxm_tcl_trAudioPropertyTWNowServiceStatus oProperty;
		switch (enRetCode)
		{
			case SMSAPI_RETURN_CODE_SUCCESS:
			{
				// SMS returns code Success for both succesful starting and stopping of TW_Now service
				oProperty.oFiMsg.ServiceStatus.enType = (bStartService) ?
						midw_ext_fi_tcl_e8_FlashServiceStatus::FI_EN_FLASH_SERVICE_STARTED :
						midw_ext_fi_tcl_e8_FlashServiceStatus::FI_EN_FLASH_SERVICE_STOPPED;
				break;
			}
			case SMSAPI_RETURN_CODE_SERVICE_ALREADY_STARTED:
			{
				// TW_Now service is already started for the Decoder object associated with service
				oProperty.oFiMsg.ServiceStatus.enType =
						midw_ext_fi_tcl_e8_FlashServiceStatus::FI_EN_FLASH_SERVICE_ALREADY_STARTED;
				break;
			}
			case SMSAPI_RETURN_CODE_ERROR:
			default:
			{
				oProperty.oFiMsg.ServiceStatus.enType =
						midw_ext_fi_tcl_e8_FlashServiceStatus::FI_EN_FLASH_SERVICE_ERROR;
				break;
			}
		}
		ETG_TRACE_USR1(("vNotifyTWNowServiceState \t bStartService(%u) \t enType(%u) \t enReturnCode(%u)",
				bStartService, oProperty.oFiMsg.ServiceStatus.enType,
				ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enRetCode) ));
		_poAudioApp->vTWNowServiceStatus(oProperty);
	}
}

// Get Decoder Object
DECODER_OBJECT fc_sxm_tclTWNow::hGetDecoder() const
{
	return (( (_poAudioApp)?
			_poAudioApp->oGetDecoderObject() :
			DECODER_INVALID_OBJECT));
}

// Get All Available Markets
tBool fc_sxm_tclTWNow::bGetAvailableMarkets()
{
	tBool bReportComplete = FALSE;
	// If we have already collected the complete list of available markets,
	// don't proceed with SMS call for getting market list
	if (((_vectMarkets.size() == 0) ) &&
			(_poTWNowProxy != OSAL_NULL))
	{
		_vectMarkets.clear();
		bReportComplete = _poTWNowProxy->bGetTWNowMarkets(this);
	}
	ETG_TRACE_USR2(("_bReportComplete(%u) \t _vectMarkets.size() = %u",
			bReportComplete, _vectMarkets.size()));
	return bReportComplete;
}

//Utility method that prints out the details of each market
tVoid fc_sxm_tclTWNow::vPrintMarkets() const
{
	SXM_FOREACH_CONST(vector <fc_sxm_trTWNowMarkets>, cIt, _vectMarkets)
	{
		cIt->vPrintMarketInfo();
	}
}

// Fill the complete market list to FI Entries
tVoid fc_sxm_tclTWNow::vFillMarkets(midw_ext_sxm_audiofi_tclMsgTWNowGetAvailableMarketsMethodResult& oMRes) const
{
	oMRes.Markets.Market.clear();
	tBool bValidMarket = FALSE;
	// Ensure list of markets is available beforehand
	if (_vectMarkets.size())
	{
		CID_OBJECT currentMarket(hGetCurrentTWNowMarket());
		// Iterate thru the market list to fill up market list based on FI element
		SXM_FOREACH_CONST(vector <fc_sxm_trTWNowMarkets>, cIt, _vectMarkets)
		{
			// Delegate work to method vFillMarketListEntry for filling up of individual FI element
			midw_ext_fi_tcl_MarketListEntry trMarketEntry;
			vFillMarketListEntry(trMarketEntry, *cIt, currentMarket, bValidMarket);
			oMRes.Markets.Market.push_back(trMarketEntry);
		}
		if ((CID_INVALID_OBJECT != currentMarket) && (FALSE == bValidMarket))
		{
			bDisableMonitoring();
		}
		vDestroyCIDObject(currentMarket);
	}
}

// Fill the details of one market entry to a FI Entries
tVoid fc_sxm_tclTWNow::vFillMarketListEntry(
		midw_ext_fi_tcl_MarketListEntry& marketEntry,
		const fc_sxm_trTWNowMarkets& market,
		CID_OBJECT currentMarket,
		tBool& bValidMarket) const
{
	// Fill up the FI entries from the given market values
	fc_sxm_tclTWNow::fc_sxm_vString2Fi(market.sAbbrName.c_str(), marketEntry.AbbrMarketName);
	fc_sxm_tclTWNow::fc_sxm_vString2Fi(market.sFullName.c_str(), marketEntry.FullMarketName);
	marketEntry.bIsActiveMarket = static_cast <tBool> (0 == s16CompareCIDObjects(currentMarket, market.hCID) );
	if (TRUE == marketEntry.bIsActiveMarket)
		bValidMarket = TRUE;
}

// Iterate through the list of TWNowMarkets, identify if the full name matches, SetMarket
tBool fc_sxm_tclTWNow::bSetMarket(const string& sMarketFullName)
{
	tBool bSuccess = FALSE;
	// Step 1: Ensure that MarketList is queried and available
	(tVoid) bGetAvailableMarkets();
	// Step 2: Iterate until the market's full name matches with the given input
	SXM_FOREACH_CONST(vector <fc_sxm_trTWNowMarkets>, cIt, _vectMarkets)
	{
		if (cIt->sFullName.compare(sMarketFullName) == 0)
		{
			// Step 3: Use Market's CID object to set market
			bSuccess =  static_cast <tBool>
			 ( SMSAPI_RETURN_CODE_SUCCESS == enSetTWNowMarket(cIt->hCID) );
			break;
		}
	}
	ETG_TRACE_USR2(("fc_sxm_tclTWNow::bSetMarket Exits with value (%u)", bSuccess));
	return bSuccess;
}

// Method De-selects any market from being monitored for TWNow Event
tBool fc_sxm_tclTWNow::bDisableMonitoring() const
{
	// To disable monitoring, set TWNow market to point to an invalid CID Object
	tBool bSuccess = static_cast <tBool>
	 ( SMSAPI_RETURN_CODE_SUCCESS == enSetTWNowMarket(CID_INVALID_OBJECT) );
	ETG_TRACE_USR2(("fc_sxm_tclTWNow::bDisableMonitoring Exits with value (%u)", bSuccess));
	return bSuccess;
}

// Copy the current market info to input argument "trMarket"
tBool fc_sxm_tclTWNow::bGetCurrentMarketInfo(fc_sxm_trTWNowMarkets& trMarket) const
{
	tBool bIsMarketFound = FALSE;
	if (_vectMarkets.size())
	{
		// Step 1: Retrieve current market's CID object
		CID_OBJECT hActiveMarket(hGetCurrentTWNowMarket());
		if (CID_INVALID_OBJECT != hActiveMarket)
		{
			// Step 2: Iterate until the market's CID matches with active market's CID
			SXM_FOREACH_CONST(vector <fc_sxm_trTWNowMarkets>, cIt, _vectMarkets)
			{
				if ( 0 == s16CompareCIDObjects(hActiveMarket, cIt->hCID) )
				{
					// Step 3: Copy market details
					trMarket = *cIt;
					bIsMarketFound = TRUE;
					break;
				}
			}
		}
		vDestroyCIDObject(hActiveMarket);
	}
	ETG_TRACE_USR2(("TWNow Current Market Information Availability (%u)", bIsMarketFound));
	return bIsMarketFound;
}

// Utility method that redirects FI Messages to be sent to Audio App
tVoid fc_sxm_tclTWNow::vSendTWNowFiMsg(fc_sxm_trAdressing const &rAddr,
		const midw_ext_sxm_audiofi_tclMsgBaseMessage& oMessage)
{
	// This method is executed for all posts of MethodResult
	if (OSAL_NULL != _poAudioApp)
	{
		_poAudioApp->vSendAudioFiMsg(rAddr, oMessage);
	}
}

// Utility method that redirects FI Error Messages to be sent to Audio App
tVoid fc_sxm_tclTWNow::vSendTWNowFiError(fc_sxm_trAdressing const &rAddr, tInt s32ErrorMsg)
{
	if (OSAL_NULL != _poAudioApp)
	{
		_poAudioApp->vSendAudioErrorMsg(rAddr, s32ErrorMsg);
	}
}

// Handle Method start for FI Method "TWNowSetMarket"
tVoid fc_sxm_tclTWNow::vProcess(fc_sxm_trMsgAudioMStartTWNowSetMarket const *prMsg)
{
	midw_ext_sxm_audiofi_tclMsgTWNowSetMarketMethodStart const &oMStart = prMsg->oFiMsg;

	tString chMarket = oMStart.Market.szGet(midw_ext_fi_tclString::FI_EN_UTF8);
	const string strMarketName(chMarket);
	ETG_TRACE_USR2(("fc_sxm_trMsgAudioMStartTWNowSetMarket (%s)", chMarket));

	midw_ext_sxm_audiofi_tclMsgTWNowSetMarketMethodResult oMRes;
	oMRes.Status = (oMStart.Status) ?
			bSetMarket(strMarketName) : bDisableMonitoring();
	vSendTWNowFiMsg(prMsg->rAdressing, oMRes);
	// Delete memory alloted for chMarket
	OSAL_DELETE [] chMarket;
}

// Handle Method start for FI Method "TWNowGetActiveMarket"
tVoid fc_sxm_tclTWNow::vProcess(fc_sxm_trMsgAudioMStartTWNowGetActiveMarket const *prMsg)
{
	fc_sxm_trTWNowMarkets trMarket;
	tBool bValidMarket = FALSE;
	// If no market is currently set, then return Error msg
	if (TRUE == bGetCurrentMarketInfo(trMarket))
	{
		// For the found market, fill up the market details
		midw_ext_sxm_audiofi_tclMsgTWNowGetActiveMarketMethodResult oMRes;
		vFillMarketListEntry(oMRes.Market, trMarket, trMarket.hCID, bValidMarket);
		ETG_TRACE_USR2(("fc_sxm_trMsgAudioMStartTWNowGetActiveMarket (%s)", trMarket.sFullName.c_str()));
		vSendTWNowFiMsg(prMsg->rAdressing, oMRes);
	}
	else
	{
		// Send error message if no valid market could be retrieved
		ETG_TRACE_ERR(("fc_sxm_trMsgAudioMStartTWNowGetActiveMarket ERROR"));
		vSendTWNowFiError(prMsg->rAdressing,
				(tInt) midw_ext_fi_tcl_e8_ErrorTypes::FI_EN_XMTUN_ERROR_INTERNAL_ERROR);
	}
}
// Handle Method start for FI Method "TWNowBulletinPlayback"
tVoid fc_sxm_tclTWNow::vProcess(fc_sxm_trMsgAudioMStartTWNowBulletinPlayback const *prMsg)
{
	midw_ext_sxm_audiofi_tclMsgTWNowBulletinPlaybackMethodStart const &oMStart = prMsg->oFiMsg;
	// Based on the action request, invoke either TWNOW's play or abort
	SMSAPI_RETURN_CODE_ENUM enReturnCode =
			( midw_ext_fi_tcl_e8_TWNow_Bulletin_PlayBackAction::FI_EN_TWNOW_BULLETIN_ACTION_PLAY ==
			oMStart.Action.enType ) ?
					enPlayTWNowBulletin(oMStart.BulletinID) :
					enAbortTWNowBulletin(oMStart.BulletinID);
	// SMS return codes for both Start and Abort of TWNow playback is the same
	if (SMSAPI_RETURN_CODE_SUCCESS == enReturnCode)
	{
		// Upon success, send method result
		midw_ext_sxm_audiofi_tclMsgTWNowBulletinPlaybackMethodResult oMRes;
		oMRes.Status = TRUE;
		oMRes.Action.enType = oMStart.Action.enType;
		ETG_TRACE_USR2(("fc_sxm_trMsgAudioMStartTWNowBulletinPlayback Status (%u) \t Type(%u)",
				oMRes.Status, oMRes.Action.enType));
		vSendTWNowFiMsg(prMsg->rAdressing, oMRes);
	}
	else
	{
		// error
		tInt s32Error = (tInt) midw_ext_fi_tcl_e8_ErrorTypes::FI_EN_XMTUN_ERROR_INTERNAL_ERROR;
		if ((SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED == enReturnCode) ||
				(SMSAPI_RETURN_CODE_API_NOT_ALLOWED == enReturnCode))
		{
			// In case if SMS reports Op-Not_Supported or API_Not_Allowed, then another error msg
			// is sent to as FI error msg so that a different advisory can be displayed to the user
			s32Error = (tInt) midw_ext_fi_tcl_e8_ErrorTypes::FI_EN_XMTUN_ERROR_INVALID_PARAMETER;
		}
		ETG_TRACE_ERR(("fc_sxm_trMsgAudioMStartTWNowBulletinPlayback ERROR (%d) \t enReturnCode (%u)",
				s32Error, enReturnCode));
		vSendTWNowFiError(prMsg->rAdressing, s32Error);
	}
}
// Handle Method start for FI Method "TWNowGetBulletinID"
tVoid fc_sxm_tclTWNow::vProcess(fc_sxm_trMsgAudioMStartTWNowGetBulletinID const *prMsg)
{
	// As this method is not utilized in A-IVI project scope 2 solution,
	// Utility function to print the list of available markets is added here.
	// TODO: Remove this print call once this method is used at-least by one client
	vPrintMarkets();
	midw_ext_sxm_audiofi_tclMsgTWNowGetBulletinIDMethodResult oMRes;
	oMRes.BulletinID = u32GetCurrentTWNowBulletinID();
	vSendTWNowFiMsg(prMsg->rAdressing, oMRes);
}

// Handle Method start for FI Method "TWNowGetAvailableMarkets"
tVoid fc_sxm_tclTWNow::vProcess(fc_sxm_trMsgAudioMStartTWNowGetAvailableMarkets const *prMsg)
{
	midw_ext_sxm_audiofi_tclMsgTWNowGetAvailableMarketsMethodResult oMRes;
	_vectMarkets.clear();
	// Get list of all available markets
	if ( bGetAvailableMarkets())
	{
		// Iterate thru the list of markets and fill details of all markets
		ETG_TRACE_USR2(("fc_sxm_trMsgAudioMStartTWNowGetAvailableMarkets VALID"));
		vFillMarkets(oMRes);
		vSendTWNowFiMsg(prMsg->rAdressing, oMRes);
	}
	else
	{
		// If the report content complete is not set, send Error message
		ETG_TRACE_ERR(("fc_sxm_trMsgAudioMStartTWNowGetAvailableMarkets ERROR"));
		vSendTWNowFiError(prMsg->rAdressing,
				(tInt) midw_ext_fi_tcl_e8_ErrorTypes::FI_EN_XMTUN_ERROR_INTERNAL_ERROR);
	}
}

// Handle method start for FI Method "TWNowEnable"
tVoid fc_sxm_tclTWNow::vProcess(fc_sxm_trMsgAudioMStartTWNowEnable const *prMsg)
{
	midw_ext_sxm_audiofi_tclMsgTWNowEnableMethodStart const &oMStart = prMsg->oFiMsg;
	_bEnableTWNow = oMStart.Status;
	// If oMStart.Status = TRUE; then user wants to enable TWNow notifications, otherwise
	// don't send any notification updates
	midw_ext_sxm_audiofi_tclMsgTWNowEnableMethodResult oMRes;
	oMRes.Status = TRUE;
	ETG_TRACE_USR2(("fc_sxm_trMsgAudioMStartTWNowEnable _bEnableTWNow(%u)", _bEnableTWNow));
	vSendTWNowFiMsg(prMsg->rAdressing, oMRes);
}

// Method to handle TWNOW event alerts reported from SMS
tVoid fc_sxm_tclTWNow::vProcess(fc_sxm_trMsgTWNowEventAlert const *prMsg)
{
	// A TWNow Event has been reported. Notify the event to HMI if _bEnableTWNow is set
	if ((prMsg) && ( TRUE == _bEnableTWNow ) && (_poAudioApp))
	{
		ETG_TRACE_USR2(("fc_sxm_trMsgTWNowEventAlert \t _bEnableTWNow(%u) \t  BulletinID(%u) \t enBulletinStatus(%u)",
						_bEnableTWNow, prMsg->u32BulletinID, ETG_CENUM(TW_NOW_BULLETIN_STATUS_ENUM, prMsg->enStatus) ));
		fc_sxm_tcl_trAudioPropertyTWNowBulletinStatus oProperty;
		oProperty.oFiMsg.BulletinID = prMsg->u32BulletinID;
		oProperty.oFiMsg.BulletinStatus.enType =
				static_cast <midw_ext_fi_tcl_e8_TWNow_Bulletin_Status::tenType> (prMsg->enStatus);
		_poAudioApp->vTWNowBulletinStatus(oProperty);
	}
}

// Method converts const char * to FI string
tVoid fc_sxm_tclTWNow::fc_sxm_vString2Fi(tCString sSrcString,  midw_ext_fi_tclString& rffiObj)
{
	if(OSAL_NULL != sSrcString)
	{
		// Convert the string to ISO-8859-1 format Call the function to convert the string to ISO-8859-1 format
		(tVoid) rffiObj.bSet(sSrcString, midw_ext_fi_tclString::FI_EN_ISO8859_1M);
	}
}

// Method handles DECODER_OBJECT_EVENT_TUNE notifications from SMS.
// Once this method is invoked by framework, a check is made if any TWNow event is currently played or not
// and appropriate property updates are sent to clients
tVoid fc_sxm_tclTWNow::vProcess(fc_sxm_trMsgAudioSmsEvtTuneState const * /* prMsg */ )
{
	if (_poAudioApp)
	{
		fc_sxm_tcl_trAudioPropertyTWNowPlaybackStatus oProperty;
		oProperty.oFiMsg.FlashEventPlayState.enType = midw_ext_fi_tcl_e8_FlashEventPlaybackStatus::FI_EN_FLASHEVENT_STOPPED;
		_bIsTWNowPlayback = FALSE;
		if (TW_NOW_INVALID_BULLETIN_ID != u32GetCurrentTWNowBulletinID())
		{
			// A valid TWNow bulletin id is returned by SMS if TWNow event is currently being played
			oProperty.oFiMsg.FlashEventPlayState.enType = midw_ext_fi_tcl_e8_FlashEventPlaybackStatus::FI_EN_FLASHEVENT_PLAYING;
			_bIsTWNowPlayback = TRUE;
		}
		ETG_TRACE_USR4(("Notify TWNow  fc_sxm_tcl_trAudioPropertyTWNowPlaybackStatus %u \t _bIsTWNowPlayback = %u",
				oProperty.oFiMsg.FlashEventPlayState.enType, _bIsTWNowPlayback));
		_poAudioApp->vTWNowPlaybackNotify(oProperty);
	}
}

// Utility method that prints the Abbreviated and Full Name of the reported Market or region
tVoid fc_sxm_trTWNowMarkets::vPrintMarketInfo() const
{
	string strMarket = "AbbrName = " + sAbbrName + "\t FullName = " + sFullName;;
	ETG_TRACE_USR4(("%s", strMarket.c_str()));
}
