/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_states.h
* @brief       Declaration for SMS, SRM, Module, Decoder and Audio states.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#ifndef _FC_SXM_TCL_SYSTEM_STATES_H_
#define _FC_SXM_TCL_SYSTEM_STATES_H_

#include "fc_sxm_tcl_ahl_object_list.h"
#include "fc_sxm_tcl_base_app.h"
#include "fc_sxm_client_spm_fi.h"
#include "fc_sxm_tcl_audio_member.h"

#include "fc_sxm_tcl_sxmapp_manager.h"

class fc_sxm_tclAudioApp;

#define FC_SXM_STATES_MIN_SMS_RESTART_WAIT_MS 1000

#define FC_SXM_STATES_SMS_RETRY_WAIT_MS 100
#define FC_SXM_STATES_MAX_SMS_START_RETRY 10
#define FC_SXM_STATES_MAX_SMS_STOP_RETRY 10

#define FC_SXM_STATES_SM_RETRY(TEXT, NUM_RETRY, FNCALL) \
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;   \
    for (int iRetry=0; iRetry<NUM_RETRY; iRetry++) {        \
        eReturnCode =FNCALL;                                            \
        fc_sxm_vTraceRetry(enGetType(), TEXT, eReturnCode, iRetry);     \
        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) {                \
            break;                                                      \
        }                                                               \
    }

// messages send from cca-context:
struct fc_sxm_trMsgAudioCmdNewAppState: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_STATES + 1);
    //constructor
    fc_sxm_trMsgAudioCmdNewAppState(tBool bCcaOn_=FALSE):
        bCcaOn(bCcaOn_)
    {
       vSetMsgPrioHigh();
    }
    tBool bCcaOn;
};

struct fc_sxm_trMsgAudioSetCriticalVoltage: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_STATES + 2);
    //constructor
    fc_sxm_trMsgAudioSetCriticalVoltage(tBool bCvOn_):
        bCvOn(bCvOn_)
    {
       vSetMsgPrioHigh();
    }
    tBool bCvOn;
};

// messges representing callbacks from sms
struct fc_sxm_trMsgAudioSmsEvtSrmState: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_STATES + 3);
    fc_sxm_trMsgAudioSmsEvtSrmState():eState(SRM_STATE_INVALID) {
       vSetMsgPrioHigh();
    }
    /*
      data-content as you like.
    */
    SRM_STATE_ENUM eState;
};

struct fc_sxm_trMsgAudioSmsEvtModuleState: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_STATES + 4);
    fc_sxm_trMsgAudioSmsEvtModuleState():eState(MODULE_STATE_INVALID),hModule(MODULE_INVALID_OBJECT){
       vSetMsgPrioHigh();
    }
    MODULE_STATE_ENUM eState;
    MODULE_OBJECT hModule;
};

struct fc_sxm_trMsgAudioSmsEvtDecoderState: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_STATES + 5);
    fc_sxm_trMsgAudioSmsEvtDecoderState():eState(DECODER_STATE_INVALID),hDecoder(DECODER_INVALID_OBJECT) {

       vSetMsgPrioHigh();
    }
    DECODER_STATE_ENUM eState;
    DECODER_OBJECT hDecoder;
};

// message from app-manager that all data-services are stopped
struct fc_sxm_trMsgAudioDataStopped: public fc_sxm_tclMessage {
    //constructor
    fc_sxm_trMsgAudioDataStopped(){
       vSetMsgPrioHigh();
    }
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_STATES + 6);
};

// messges representing callbacks from sms
struct fc_sxm_trMsgSmsSrmState: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_STATES + 7);
    /*
      data-content as you like.
    */
    SRM_STATE_ENUM eState;
};

// timeout-messges to delay start of data-services
struct fc_sxm_trMsgAudioDataStartDelayTimeout: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_STATES + 8);
    //constructor
    fc_sxm_trMsgAudioDataStartDelayTimeout(){
       vSetMsgPrioHigh();
    }
};

// timeout-messges to delay start of sms
struct fc_sxm_trMsgAudioSmsStartDelayTimeout: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_STATES + 9);
    //constructor
    fc_sxm_trMsgAudioSmsStartDelayTimeout(){
       vSetMsgPrioHigh();
    }
};

// timeout-messges to delay start of audio-service
struct fc_sxm_trMsgAudioAudioStartDelayTimeout: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_STATES + 10);
    //constructor
    fc_sxm_trMsgAudioAudioStartDelayTimeout(){
       vSetMsgPrioHigh();
    }
};

struct fc_sxm_trMsgAudioModuleStopRetryTimeout: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_STATES + 12);
    fc_sxm_trMsgAudioModuleStopRetryTimeout() { }
};

struct fc_sxm_trMsgAudioDecoderStopRetryTimeout: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_STATES + 13);
    fc_sxm_trMsgAudioDecoderStopRetryTimeout() { }
};

// Message for Reset SXM module
struct fc_sxm_trMsgAudioMStartResetSxmModule : public fc_sxm_tclMessage
{
    /* Service and service ID */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_STATES + 14);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgResetSXMModuleMethodStart oFiMsg;
};
struct fc_sxm_trMsgAudioSMSCfgError: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_STATES + 16);
    //constructor
    fc_sxm_trMsgAudioSMSCfgError()
    {
       vSetMsgPrioHigh();
    }
};

#if((FC_SXM_ENABLE_WATCHDOG_TIMER) || (FC_SXM_ENABLE_LCM_WATCHDOG_TIMER))
typedef enum {
   fc_sxm_enAudioInvalid,
   fc_sxm_enAudioReady,
   fc_sxm_enAudioStopped
} fc_sxm_tenAudioState;
#endif

typedef enum {
    fc_sxm_enSmsObjectState_STOPPED,
    fc_sxm_enSmsObjectState_STARTING,
    fc_sxm_enSmsObjectState_READY,
    fc_sxm_enSmsObjectState_UPDATING,
    fc_sxm_enSmsObjectState_STOPPING,
    fc_sxm_enSmsObjectState_ERROR
} fc_sxm_tenSmsObjectState;

typedef enum {
    fc_sxm_enSystemState_Error,
    fc_sxm_enSystemState_DefsetFactory,
    fc_sxm_enSystemState_OFF,     // cca-state off
    fc_sxm_enSystemState_INIT,
    fc_sxm_enSystemState_DownLoad,
    fc_sxm_enSystemState_DiagFreeze,
    fc_sxm_enSystemState_Normal,
    fc_sxm_enSystemState_INVALID
} fc_sxm_tenSystemState;

typedef enum {
	fc_sxm_enCcaState_On,
	fc_sxm_enCcaState_Off,
	fc_sxm_enCcaState_Invalid
} fc_sxm_tenCcaState;

typedef enum {
    en_fc_sxm_tclSmsSms,
    en_fc_sxm_tclSmsSrm,
    en_fc_sxm_tclSmsModule,
    en_fc_sxm_tclSmsDecoder,
    en_fc_sxm_tclDataProxy,
    en_fc_sxm_tclAudioProxy,
    en_fc_sxm_tclSmTypeMax
} fc_sxm_tenSmType;

typedef enum {
    fc_sxm_enMaskMode_Set,
    fc_sxm_enMaskMode_Add,
    fc_sxm_enMaskMode_Remove
} fc_sxm_tenMaskMode;

typedef enum {
    fc_sxm_enSmEvent_SmsInitial,
    fc_sxm_enSmEvent_SmsError,
    fc_sxm_enSmEvent_SmsStopped,
    fc_sxm_enSmEvent_SmsReady,
    fc_sxm_enSmEvent_SmsUpdating,
    fc_sxm_enSmEvent_ChildrenUp,
    fc_sxm_enSmEvent_ChildrenDown,
    fc_sxm_enSmEvent_ParentUp,
    fc_sxm_enSmEvent_ParentDown,
    fc_sxm_enSmEvent_Activate,
    fc_sxm_enSmEvent_DeActivate,
    fc_sxm_enSmEvent_EndSequence,
    fc_sxm_enSmEvent_MAX
} fc_sxm_tenSmEvent;

typedef enum {
    enErrorRecoveryMode_Idle,
    enErrorRecoveryMode_Pending,
    enErrorRecoveryMode_Active,
    enErrorRecoveryMode_Failed
} fc_sxm_tenErrorRecoveryMode;

class fc_sxm_tclSmsSm;
struct fc_sxm_trSmEvent {
    fc_sxm_tclSmsSm *poToSm;    
    fc_sxm_tclSmsSm *poArgSm;    
    fc_sxm_tenSmEvent enEvent;
    //constructor
    fc_sxm_trSmEvent():
        poToSm(OSAL_NULL),
        poArgSm(OSAL_NULL),
        enEvent(fc_sxm_enSmEvent_MAX)
    { }
    //copy constructor
    fc_sxm_trSmEvent(fc_sxm_trSmEvent const &rRight):
        poToSm(rRight.poToSm),
        poArgSm(rRight.poArgSm),
        enEvent(rRight.enEvent)
    { }
};

/* Message for SmEvent data */
struct fc_sxm_trMsgAudioSmEvent: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_STATES + 15);
    /* Constructor */
    fc_sxm_trMsgAudioSmEvent(fc_sxm_trSmEvent tSmEvent): 
      _tSmEvent(tSmEvent) {
      vSetMsgPrioHigh();
    };
    /* SmEvent data */
    fc_sxm_trSmEvent _tSmEvent;
};

tBool fc_sxm_bChangeMask(fc_sxm_tenMaskMode enMode, tU32 u32Mask, tU32 &ru32InOutMask);
fc_sxm_tenSystemState fc_sxm_enGetSystemState(void);
tBool fc_sxm_bIsCvOn(void);
extern "C" unsigned char fc_sxm_bIsSMSLIB_CvOn(void);

/*
  Base-class for all states.
  A state has one or zero parents and 0 to N children.
*/
class fc_sxm_tclSmsSm {
 public:
    fc_sxm_tclSmsSm();
    virtual ~fc_sxm_tclSmsSm(){_poParent = OSAL_NULL;}
        
    virtual tBool bStable();
    virtual fc_sxm_tenSmType enGetType() const =0;
    tVoid vActivate(tBool bActivate) {
        if (_bWantsActivate != bActivate) {
            _bWantsActivate = bActivate;
            vPostEvent(_bWantsActivate ? fc_sxm_enSmEvent_Activate : fc_sxm_enSmEvent_DeActivate);
        }
    }
    tVoid vTraceStart(fc_sxm_tenSmEvent enEvent) const ;
    tVoid vTraceEnd(fc_sxm_tenSmEvent enEvent) const ;
    tVoid vPostEvent(fc_sxm_tenSmEvent enEvent);

    tBool bWantActivate() const {
        return _bWantsActivate;
    }

    tVoid vHandleEvent(fc_sxm_tenSmEvent enEvent);

    virtual tBool bZombie() {return FALSE; };

 protected:
    virtual tVoid vHandleSmsInitial();
    virtual tVoid vHandleSmsError();
    virtual tVoid vHandleSmsStopped();
    virtual tVoid vHandleSmsReady();
    virtual tVoid vHandleSmsUpdating();
    virtual tVoid vHandleChildrenUp();
    virtual tVoid vHandleChildrenDown();
    virtual tVoid vHandleParentUp();
    virtual tVoid vHandleParentDown();
    virtual tVoid vHandleActivate();
    virtual tVoid vHandleDeActivate();

    virtual tBool bStartSmsObject() { return TRUE; };

    tVoid vEnterError();

    tVoid vRegisterChild(fc_sxm_tclSmsSm *poChild);
    tVoid vSetParent(fc_sxm_tclSmsSm *poParent);

    // tell us the changed state of a child
    // writes _bChildrenActive==TRUE, if at least one child is active.
    virtual tVoid vSetChildActive(fc_sxm_tclSmsSm *poChild, tBool bActive);

    // tell us the state of the parent.
    virtual tVoid vSetParentUp(tBool bUp);

    virtual tVoid vLinkSmsParent() {};
    fc_sxm_tenSmsObjectState _enState;
    fc_sxm_tclSmsSm *_poParent;
    tBool _bParentUp;
    deque<fc_sxm_tclSmsSm *>_lChildren;
    tBool _bChildrenActive;
    tBool _bWantsActivate;

    virtual tVoid vNotifyChildren(tBool bUp);
    tVoid vNotifyParent(tBool bActive);

    // call this to update the state
    tVoid vUpdateState(fc_sxm_tenSmsObjectState enState);

    // implement this: set new state and do all actions according to given state
    virtual tVoid vUpdateState_(fc_sxm_tenSmsObjectState enState)=0;

    virtual tVoid vStopSmsObject(){};
};


typedef enum {
    fc_sxm_enSmsState_STOPPED,
    fc_sxm_enSmsState_READY,
    fc_sxm_enSmsState_ERROR
}fc_sxm_tenSmsState;

class fc_sxm_tclSmsSms:
public fc_sxm_tclSmsSm,
    public fc_sxm_tclSingleton<fc_sxm_tclSmsSms>
{
    friend class fc_sxm_tclSingleton<fc_sxm_tclSmsSms>;
 public:
    // get sms type
    virtual fc_sxm_tenSmType enGetType() const {
        return en_fc_sxm_tclSmsSms; // return sms type
    }

 protected:
    virtual tVoid vHandleActivate();
    virtual tBool bStartSmsObject();
    virtual tVoid vStopSmsObject();

 private:
    fc_sxm_tclSmsSms();

    virtual tVoid vUpdateState_(fc_sxm_tenSmsObjectState enState);

    fc_sxm_tenSmsState _enSmsState;
};

class fc_sxm_tclSmsSrm:
public fc_sxm_tclSmsSm,
    public fc_sxm_tclSingleton<fc_sxm_tclSmsSrm>
{
    friend class fc_sxm_tclSingleton<fc_sxm_tclSmsSrm>;
 public:
    /*
      trigger: system-state
      trigger: Decoder-state
    */

    virtual fc_sxm_tenSmType enGetType() const {
        return en_fc_sxm_tclSmsSrm;
    }

    // srm-state from sms-callback
    tVoid vProcess(fc_sxm_trMsgAudioSmsEvtSrmState const *prMsg);


    SRM_OBJECT hGetSmsSrm() {
        return _hSRM;
    }
    
    virtual tBool bZombie() {return (_hSRM == 0) && (_enState!=fc_sxm_enSmsObjectState_STOPPED); };

 protected:
    virtual tBool bStartSmsObject();
    virtual tVoid vStopSmsObject();
    virtual tVoid vHandleSmsInitial();

 private:
    fc_sxm_tclSmsSrm();
    virtual tVoid vUpdateState_(fc_sxm_tenSmsObjectState enState);

    SRM_OBJECT      _hSRM;
    SRM_STATE_ENUM _enSmsState;
};

class fc_sxm_tclSmsModule:
public fc_sxm_tclSmsSm,
    public fc_sxm_tclSingleton<fc_sxm_tclSmsModule>
{
    friend class fc_sxm_tclSingleton<fc_sxm_tclSmsModule>;
    friend tVoid cb_vModuleEventCallback(MODULE_OBJECT hModule, 
                                         MODULE_EVENT_MASK tEventMask, 
                                         tVoid *pvEventCallbackArg);
 public:
    virtual fc_sxm_tenSmType enGetType() const {
        return en_fc_sxm_tclSmsModule;
    }

    // process module-state received in sms-callback
    tVoid vProcess(fc_sxm_trMsgAudioSmsEvtModuleState const *prMsg);

    tVoid vProcessTimer(fc_sxm_trMsgAudioModuleStopRetryTimeout const * /* prMsg */) {
        vStopSmsObject();
    }

    // set mask of wanted module-events
    tVoid vChangeMask(fc_sxm_tenMaskMode enMode, tU32 u32Mask);

    MODULE_OBJECT hGetSmsModule() {
        if (fc_sxm_bIsCvOn() || (_enState==fc_sxm_enSmsObjectState_STOPPING)) {
            return MODULE_INVALID_OBJECT;
        }
        return _hModule;
    }

    virtual tBool bZombie() {return (_hModule == 0) && (_enState!=fc_sxm_enSmsObjectState_STOPPED && _enState!=fc_sxm_enSmsObjectState_STOPPING); };

 protected:
    virtual tBool bStartSmsObject();
    virtual tVoid vStopSmsObject();
    virtual tVoid vLinkSmsParent();
    virtual tVoid vHandleSmsUpdating();
    virtual tVoid vHandleSmsInitial();

 private:
    fc_sxm_tclSmsModule();

    // send hardware-error according to MODULE-state to fc_sxm_tclAdvisories
    tVoid vSendHardwareError();

    virtual tVoid vUpdateState_(fc_sxm_tenSmsObjectState enState);
    fc_sxm_tclTimer<fc_sxm_tclAudioApp, fc_sxm_trMsgAudioModuleStopRetryTimeout, fc_sxm_tclSmsModule> _oModuleStopRetryTimer;

    MODULE_OBJECT _hModule;
    MODULE_OBJECT _hLastCallBackModule;
    MODULE_STATE_ENUM _enSmsState;
    tU32 _u32Mask;
    SRM_OBJECT _hSRM;
};


class fc_sxm_tclSmsDecoder:
public fc_sxm_tclSmsSm,
    public fc_sxm_tclSingleton<fc_sxm_tclSmsDecoder>
{
    friend class fc_sxm_tclSingleton<fc_sxm_tclSmsDecoder>;
 public:
    virtual fc_sxm_tenSmType enGetType() const {
        return en_fc_sxm_tclSmsDecoder;
    }

    // handle decoder-state received in sms-callback
    tVoid vProcess(fc_sxm_trMsgAudioSmsEvtDecoderState const *prMsg);
    tVoid vProcessTimer(fc_sxm_trMsgAudioDecoderStopRetryTimeout const * /* prMsg */) {
        vStopSmsObject();
    }

    // set mask of wanted module-events
    tVoid vChangeMask(fc_sxm_tenMaskMode enMode, tU32 u32Mask);

    // access to our sms-object
    DECODER_OBJECT hGetSmsDecoder() {
        return _hDecoder;
    }

    virtual tBool bZombie() {return (_hDecoder == 0) && (_enState!=fc_sxm_enSmsObjectState_STOPPED && _enState!=fc_sxm_enSmsObjectState_STOPPING); };

 protected:
    virtual tBool bStartSmsObject();
    virtual tVoid vStopSmsObject();
    virtual tVoid vLinkSmsParent();

 private:
    fc_sxm_tclSmsDecoder();

    // utility to stop the sms DECODER
    tVoid vStopSmsDecoder();

    // set new state and notify clients
    virtual tVoid vUpdateState_(fc_sxm_tenSmsObjectState enState);    
    fc_sxm_tclTimer<fc_sxm_tclAudioApp, fc_sxm_trMsgAudioDecoderStopRetryTimeout, fc_sxm_tclSmsDecoder> _oDecoderStopRetryTimer;

    DECODER_OBJECT _hDecoder;
    DECODER_STATE_ENUM _enSmsState;
    const tU32 _u32MaskStart;
    const tU32 _u32MaskReady;
    tU32 _u32Mask;
    SRM_OBJECT _hSRM;
    tU32 _u32NumStopRequests;

};

class fc_sxm_tclDataProxy:
public fc_sxm_tclSmsSm,
    public fc_sxm_tclSingleton<fc_sxm_tclDataProxy>
{
    friend class fc_sxm_tclSingleton<fc_sxm_tclDataProxy>;
 public:
    virtual fc_sxm_tenSmType enGetType() const {
        return en_fc_sxm_tclDataProxy;
    }

    // app-manager tells us, that data-services are stopped
    tVoid vProcess(fc_sxm_trMsgAudioDataStopped const *prMsg);

    tVoid vProcessTimer(fc_sxm_trMsgAudioDataStartDelayTimeout *prMsg);

 protected:
    virtual tVoid vHandleDeActivate();
    virtual tVoid vHandleChildrenDown();

 private:
    fc_sxm_tclDataProxy();

    // overwrite base-class becaus child is not a fc_sxm_tclSmsSm
    virtual tVoid vNotifyChildren(tBool bUp);

    // set new state and notify clients
    virtual tVoid vUpdateState_(fc_sxm_tenSmsObjectState enState);

    fc_sxm_tclTimer<fc_sxm_tclAudioApp, fc_sxm_trMsgAudioDataStartDelayTimeout, fc_sxm_tclDataProxy> _oDataDelayTimer;

};

class fc_sxm_tclAudioProxy:
public fc_sxm_tclSmsSm,
    public fc_sxm_tclSingleton<fc_sxm_tclAudioProxy>
{
    friend class fc_sxm_tclSingleton<fc_sxm_tclAudioProxy>;
 public:
    virtual fc_sxm_tenSmType enGetType() const {
        return en_fc_sxm_tclAudioProxy;
    }

    // child (fc_sxm_tclAudioApp) tells us that it is no longer active
    tVoid vSetAudioStopped();

    tVoid vProcessTimer(fc_sxm_trMsgAudioAudioStartDelayTimeout *prMsg);

    tVoid vRegisterDecoderCallbacks(tBool bRegister);
    tBool bAreHandleChnCatCallbacksAllowed() const {
        return _bChnCatCbRegistered;
    }

#if((FC_SXM_ENABLE_WATCHDOG_TIMER) || (FC_SXM_ENABLE_LCM_WATCHDOG_TIMER))
    fc_sxm_tenAudioState eGetAudioProxyState(tVoid) const { return(_enAudioState); }
 private:
    fc_sxm_tenAudioState _enAudioState;
#endif

 protected:
    virtual tVoid vLinkSmsParent();
    virtual tVoid vHandleDeActivate();
    virtual tVoid vHandleChildrenDown();

 private:
    fc_sxm_tclAudioProxy();

    // register/unregister additional sms-callbacks (CATEGORY and CHANNEL) 
    //Moving to public for the time being.Have to revert later
    //tVoid vRegisterDecoderCallbacks(tBool bRegister) const;

    virtual tVoid vNotifyChildren(tBool bUp);

    // set new state and notify clients
    virtual tVoid vUpdateState_(fc_sxm_tenSmsObjectState enState);

    DECODER_OBJECT _hDecoder;
    tBool _bChnCatCbRegistered;

    fc_sxm_tclTimer<fc_sxm_tclAudioApp, fc_sxm_trMsgAudioAudioStartDelayTimeout, fc_sxm_tclAudioProxy> _oAudioDelayTimer;

};

// priority of an activity, not used yet
typedef enum {
    fc_sxm_enActivityPrio_Highest,
    fc_sxm_enActivityPrio_High,
    fc_sxm_enActivityPrio_Normal,
    fc_sxm_enActivityPrio_Low,
    fc_sxm_enActivityPrio_Lowest
} fc_sxm_tenActivityPrio;

// state of an activity
typedef enum {
    fc_sxm_enActivityState_Idle,
    fc_sxm_enActivityState_Pending,
    fc_sxm_enActivityState_Running,
    fc_sxm_enActivityState_Terminating
} fc_sxm_tenActivityState;

/*
  base-class for an activity
  With an activity a specific system-state can be requested via bRequest().
  As soon as the system-state is reached, the activity is call with vRun().
  When the activity is done, it has to call vOnDone().
*/
class fc_sxm_tclActivityBase {
    friend class fc_sxm_tclSystemStates;
 public:
    fc_sxm_tclActivityBase(fc_sxm_tenSystemState enType=fc_sxm_enSystemState_INVALID, fc_sxm_tenActivityPrio enPrio=fc_sxm_enActivityPrio_Normal, tBool bInterruptible=FALSE);
       
    tBool bSetType(fc_sxm_tenSystemState enType);

    /* called by  user to request start of the activity*/
    tBool bRequest();

    tVoid vNotifyResult();

    /* called by user when activity is done */
    tVoid vOnDone() ;

    // called by framework to find out if activity can be stopped (not used yet)
    virtual tBool bIsInterruptible() {return TRUE; };
    
    fc_sxm_tenSystemState enGetType() const {
        return _enType; //return the type
    }
    fc_sxm_tenActivityPrio enGetPrio() const {
        //return the priority
        return _enPrio;
    }
    // Get the state
    fc_sxm_tenActivityState enGetState() const {
        return _enState;  
    }

 private:
    // called by framework when activity may start
    virtual tVoid vRun()=0;

    // called by framework to stopp an interruptible activity
    virtual tVoid vStop() { vOnDone();};

    fc_sxm_tenSystemState _enType;
    fc_sxm_tenActivityPrio _enPrio;
    fc_sxm_tenActivityState _enState;
    tBool _bInterruptible;
};


/* the main-class for all system-states.
   From here all other state-machines are controlled
*/
class fc_sxm_tclSystemStates:
public fc_sxm_tclAudioMember,
public fc_sxm_tclSingleton<fc_sxm_tclSystemStates>
{
    friend class fc_sxm_tclSingleton<fc_sxm_tclSystemStates>;
 public:
    // inner-class tclSmList storing all our state-machines
    class tclSmList:
    public fc_sxm_tclObjectList<fc_sxm_tclSmsSm> {
    public:
        // update all
        tVoid vUpdateSystemState(fc_sxm_tenSystemState enSystemState);
         
        // true, if all our SMs are stable
        tBool bStable();
    };

 public:
    ~fc_sxm_tclSystemStates();
    tVoid vInit();
    // tell us, if cca-app shall be ON, corresponds to APP_STATE_NORMAL
    tVoid vProcess(fc_sxm_trMsgAudioCmdNewAppState const *prMsg);
    tVoid vProcessTimer(fc_sxm_trMsgAudioSmsStartDelayTimeout const *prMsg);
    tVoid vProcess(fc_sxm_trMsgAudioSmEvent const *prMsg);

    // tell us, if we are in critical voltage
    tVoid vProcess(fc_sxm_trMsgAudioSetCriticalVoltage const *prMsg);
    tVoid vProcess(fc_sxm_trMsgAudioSMSCfgError const * /*prMsg*/);
    tVoid vProcess(fc_sxm_trMsgCvmErrorRecoveryDone const *prMsg);
    tVoid vProcess(fc_sxm_trMsgAudioCvmEvent const *prMsg);
    tVoid vProcess(fc_sxm_trMsgAudioMStartResetSxmModule const *prMsg);


    // getter for current system-state
    fc_sxm_tenSystemState enGetSystemState() const {
        return _enSystemState;
    }
     
    /* interface to be called by an activity 
       (returns TRUE, if activity has been accepted).
       In case the is allready an anctivity, the request will be rejected
    */
    tBool bRequestActivity(fc_sxm_tclActivityBase *poActivity);

    // will be call by an activity when it is done
    tVoid vActivityDone(fc_sxm_tclActivityBase *poActivity);

    static tBool bWantsActivate(fc_sxm_tenSmType enSmType, fc_sxm_tenSystemState enSytemState);
    tVoid vPostEvent(fc_sxm_trSmEvent const &rEvent) const;

    tVoid vHandleError(fc_sxm_tclSmsSm *poSM);
    tVoid vUpdateSystemState();
    // status of download recovery
    tBool bIsDownloadRecovery() const;
    // end of download recovery, update the flag
    tVoid vEndDownloadRecovery();
    // start the download recovery
    tVoid vStartDownloadRecovery();

    tBool bIsInErrorRecovery() const {
        return _enErrorRecoveryMode==enErrorRecoveryMode_Active || _enErrorRecoveryMode==enErrorRecoveryMode_Failed;
    }
    unsigned char bIsSMSLIBCvOn();

    tBool bIsCvOn() const {
        return _bCvOn;
    }
    tBool bIsSMSDecoderError() const {
        return _bSMSDecoderError;
    }
    tVoid vSetSMSDecoderError(tBool bError){
    	_bSMSDecoderError = bError;
    }
    tVoid vDeleteSmsCfg();

    // get type of current activity or fc_sxm_enSystemState_INVALID if there is no activity
    fc_sxm_tenSystemState enGetActivityType() const;

    tBool bIsDiagDefsetTEF() const;

 private:
    fc_sxm_tclSystemStates();

    // check if an activity shall be started or stopped
    tVoid vCheckAndRunActivities();

    // evaluate new state an take actions
    tVoid vUpdate(tBool bForced=FALSE);

    // notify all  our SMs about new system-state
    tVoid vPropagateSystemState(fc_sxm_tenSystemState enSystemState, tBool bForced);

    /* start a sequence. all messages posted until calling vEndSequence() will be processed
       before new system-state is evaluated viea vUpdate() or vCheckAndRunActivities().
    */
    tVoid vStartSequence() {
        _u32SequenceDepth++;
    }

    tBool bIsSequenceComplete() const {
        return (_u32SequenceDepth==0);
    }
    
    // see vStartSequence()
    tVoid vEndSequence() const {
        fc_sxm_trSmEvent rEvent=fc_sxm_trSmEvent();
        rEvent.enEvent=fc_sxm_enSmEvent_EndSequence;
        vPostEvent(rEvent);
    }

    // Mark the start and end of critical voltage handling
    tVoid SetCriticalVoltageStart();
    tVoid SetCriticalVoltageEnd();
    tVoid vRemove(string &cmd) const;


    fc_sxm_tenSystemState _enSystemState;
    tclSmList _listSMs;
    fc_sxm_tenCcaState _enCcaState;
    tBool _bCvOn;
    tBool _bFirstInit;
    tBool _bSMSDecoderError;

    fc_sxm_tclActivityBase *_poCurActivity;
    tBool _bEnteringActity;
    tBool _bTerminateActivity;
    tBool _bProcessingQ; // if Q is currently processed, another vProcessQ() will be ignored
    tU32 _u32SequenceDepth; // only when _u32SequenceDepth==0 system-state is settled and can be checked
    tBool _bDownloadRecovery;
    fc_sxm_tenErrorRecoveryMode _enErrorRecoveryMode;

    static const tBool _ActivateMatrix[en_fc_sxm_tclSmTypeMax][fc_sxm_enSystemState_INVALID];

    fc_sxm_tclTimer<fc_sxm_tclAudioApp, fc_sxm_trMsgAudioSmsStartDelayTimeout, fc_sxm_tclSystemStates> _oSmsDelayTimer;
};


#endif
