/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_sports_flash_proxy.cpp
* @brief       Implementation of interface between fc_sxm app and SMS library for 
*              SportsFlash service specific calls.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/
#include "fc_sxm_tcl_audio_app_if.h"
#include "fc_sxm_tcl_sports_flash_proxy.h"
#include "fc_sxm_sms_util.h"

fc_sxm_tclSportsFlashProxy::fc_sxm_tclSportsFlashProxy() : _poDecoder(OSAL_NULL)
{
	// Default CTOR
}

fc_sxm_tclSportsFlashProxy::~fc_sxm_tclSportsFlashProxy()
{
	// Default DTOR
	_poDecoder = OSAL_NULL;
}

// Set audio app's instance
tVoid fc_sxm_tclSportsFlashProxy::vSetDecoder(Ifc_sxm_tclDecoder* poDecoder)
{
	_poDecoder = poDecoder;
}

// Use all SMS calls specific for sports flash

// Start sports flash service
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclSportsFlashProxy::enStartSportsFlash( DECODER_OBJECT hDecoder,
		tUInt* pou32MaxSportsFlash,
		SPORTS_FLASH_GAME_EVENT_CALLBACK vGameEventCallback,
		tVoid* poGameEventCallbackArg) const
{
	// Acquire and initialize the Sports Flash service. It simultaneously creates
	// SPORTS_FLASH_OBJECT which is unique to the DECODER and is not provided to the caller
	return (SPORTS_FLASH.eStart(hDecoder, pou32MaxSportsFlash, vGameEventCallback, poGameEventCallbackArg)) ;
}

// Stop sports flash service
tVoid fc_sxm_tclSportsFlashProxy::vStopSportsFlash() const
{
	// Release an existing Sports Flash service and release the object handle
	DECODER_OBJECT hDecoder(hGetDecoder());
	return (SPORTS_FLASH.vStop(hDecoder));
}

// Add a favorite team to monitor for sports flash events
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclSportsFlashProxy::enAddFavoriteTeam(CID_OBJECT hLeagueId, CID_OBJECT hTeamId) const
{
	// Set up a favorite team to detect live games and also setup the seek monitor.
	DECODER_OBJECT hDecoder(hGetDecoder());
	SMSAPI_RETURN_CODE_ENUM enReturnCode = SMSAPI_RETURN_CODE_ERROR;
	if (DECODER_INVALID_OBJECT != hDecoder)
		enReturnCode = SPORTS_FLASH.eAddFavoriteTeam(hDecoder, hLeagueId, hTeamId);
	return enReturnCode;
}

// Remove a favorite team from the favorite teams list
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclSportsFlashProxy::enRemoveFavoriteTeam(CID_OBJECT hLeagueId, CID_OBJECT hTeamId) const
{
	// Stop the seek monitor for the given team
	DECODER_OBJECT hDecoder(hGetDecoder());
	SMSAPI_RETURN_CODE_ENUM enReturnCode = SMSAPI_RETURN_CODE_ERROR;
	if (DECODER_INVALID_OBJECT != hDecoder)
		enReturnCode = SPORTS_FLASH.eRemoveFavoriteTeam(hDecoder, hLeagueId, hTeamId);
	return enReturnCode;
}

// Remove all favorite teams
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclSportsFlashProxy::enRemoveAllFavoriteTeams() const
{
	// Seek monitor for sports flash events is set back to factory reset conditions
	DECODER_OBJECT hDecoder(hGetDecoder());
	SMSAPI_RETURN_CODE_ENUM enReturnCode = SMSAPI_RETURN_CODE_ERROR;
	if (DECODER_INVALID_OBJECT != hDecoder)
		enReturnCode = SPORTS_FLASH.eRemoveAllFavoriteTeams(hDecoder);
	return enReturnCode;
}

// Retrieve the current list of user's Favorite teams
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclSportsFlashProxy::enIterateFavoriteTeams(
		SPORTS_FLASH_TEAMS_ITERATOR_CALLBACK bTeamsIteratorCallBack,
		tVoid* poTeamsIteratorCallbackArg) const
{
	//Iterate through the list of all favorite teams
	DECODER_OBJECT hDecoder(hGetDecoder());
	SMSAPI_RETURN_CODE_ENUM enReturnCode = SMSAPI_RETURN_CODE_ERROR;
	if (DECODER_INVALID_OBJECT != hDecoder)
		enReturnCode = SPORTS_FLASH.eIterateFavoriteTeams(hDecoder, bTeamsIteratorCallBack, poTeamsIteratorCallbackArg);
	return enReturnCode;
}

// Retrieve Ongoing game events involving user specified Favorite teams filtered from duplicates
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclSportsFlashProxy::enIterateCurrentGames(
		SPORTS_FLASH_GAMES_ITERATOR_CALLBACK bGamesIteratorCallBack,
		tVoid* poGamesIteratorCallbackArg) const
{
	// Note: Events that have ended are automatically removed by SMS
	DECODER_OBJECT hDecoder(hGetDecoder());
	SMSAPI_RETURN_CODE_ENUM enReturnCode = SMSAPI_RETURN_CODE_ERROR;
	if (DECODER_INVALID_OBJECT != hDecoder)
		enReturnCode = SPORTS_FLASH.eIterateCurrentGames(hDecoder, bGamesIteratorCallBack, poGamesIteratorCallbackArg);
	return enReturnCode;
}

// Setup a list of game channels to monitor for Sports Flash Events.
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclSportsFlashProxy::enSetGamesMonitor(CHANNEL_ID *poChannelIDList,
		tU16 u16Count,
		SPORTS_FLASH_FLASH_EVENT_CALLBACK vFlashEventCallback,
		tVoid* poFlashEventCallbackArg) const
{
	// poChannelIDList contains the complete list of CHANNEL_IDs to monitor
	DECODER_OBJECT hDecoder(hGetDecoder());
	SMSAPI_RETURN_CODE_ENUM enReturnCode = SMSAPI_RETURN_CODE_ERROR;
	if (DECODER_INVALID_OBJECT != hDecoder)
		enReturnCode = SPORTS_FLASH.eSetGamesMonitor(hDecoder, poChannelIDList, u16Count, vFlashEventCallback, poFlashEventCallbackArg);
	return enReturnCode;
}

// Iterate & Retrieve the list of active flash events
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclSportsFlashProxy::enIterateCurrentFlashEvents(
		SPORTS_FLASH_EVENTS_ITERATOR_CALLBACK bEventsIteratorCallback,
		tVoid* poEventsIteratorCallbackArg) const
{
	// Each occured and not yet expired Flash Event can be retrieved
	DECODER_OBJECT hDecoder(hGetDecoder());
	SMSAPI_RETURN_CODE_ENUM enReturnCode = SMSAPI_RETURN_CODE_ERROR;
	if (DECODER_INVALID_OBJECT != hDecoder)
		enReturnCode = SPORTS_FLASH.eIterateCurrentFlashEvents(hDecoder, bEventsIteratorCallback, poEventsIteratorCallbackArg);
	return enReturnCode;
}

// Switch playback to the channel playing the Flash Event reported by the Monitor.
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclSportsFlashProxy::enPlayFlashEvent(SPORTS_FLASH_EVENT_ID tFlashEventID) const
{
	// A 5 minute timer is started on issue of this cmd to remain on the selected channel if the playback is not aborted
	DECODER_OBJECT hDecoder(hGetDecoder());
	SMSAPI_RETURN_CODE_ENUM enReturnCode = SMSAPI_RETURN_CODE_ERROR;
	if (DECODER_INVALID_OBJECT != hDecoder)
		enReturnCode = SPORTS_FLASH.ePlayFlashEvent(hDecoder, tFlashEventID);
	return enReturnCode;
}

// Abort playback of channel playing the flash event
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclSportsFlashProxy::enAbortFlashEvent() const
{
	// The 5 minute timer is stopped as a subsequent call to the API
	DECODER_OBJECT hDecoder(hGetDecoder());
	SMSAPI_RETURN_CODE_ENUM enReturnCode = SMSAPI_RETURN_CODE_ERROR;
	if (DECODER_INVALID_OBJECT != hDecoder)
		enReturnCode = SPORTS_FLASH.eAbortFlashEvent(hDecoder);
	return enReturnCode;
}

// Retrieve the currently active flash event ID
SPORTS_FLASH_EVENT_ID fc_sxm_tclSportsFlashProxy::u32CurrentFlashEventID() const
{
	// This API can be invoked in DECODER event callback once DECODER_EVENT_TUNE mask is reported.
	// If its invoked from other scenarios, SPORTS_FLASH_INVALID_EVENT_ID is returned
	DECODER_OBJECT hDecoder(hGetDecoder());
	SPORTS_FLASH_EVENT_ID u32EventID = (SPORTS_FLASH_EVENT_ID)((DECODER_INVALID_OBJECT != hDecoder) ?
			SPORTS_FLASH.tCurrentFlashEventID(hDecoder) :
			SPORTS_FLASH_INVALID_EVENT_ID);
	return u32EventID;
}

// Enable/Disable sending event notifications
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclSportsFlashProxy::enEnableNotification(tBool bEnable) const
{
	// The setting of sports flash alerts is stored in SMS configuration file
	DECODER_OBJECT hDecoder(hGetDecoder());
	SMSAPI_RETURN_CODE_ENUM enReturnCode = SMSAPI_RETURN_CODE_ERROR;
	if (DECODER_INVALID_OBJECT != hDecoder)
		enReturnCode = SPORTS_FLASH.eEnableNotification(hDecoder, bEnable);
	return enReturnCode;
}

// Retrieve current status (enabled/disabled) of event notifications
tBool fc_sxm_tclSportsFlashProxy::bIsNotificationEnabled() const
{
	tBool bEnabled = FALSE;
	DECODER_OBJECT hDecoder(hGetDecoder());
	if (SMSAPI_RETURN_CODE_SUCCESS != SPORTS_FLASH.eIsNotificationEnabled(hDecoder, &bEnabled))
		bEnabled = FALSE;
	return bEnabled;
}

// Fill Channel Attributes
tVoid fc_sxm_tclSportsFlashProxy::vExtractChannelAttributes(CHANNEL_ID channelID, fc_sxm_trSFChannel& trSFChannel) const
{
	// Extract channel attributes from DECODER....
	trSFChannel.vReset();
	DECODER_OBJECT hDecoder(hGetDecoder());
	if (DECODER_INVALID_OBJECT != hDecoder)
	{
		trSFChannel.tChannelID = channelID;
		// The return values for these calls, "SMSAPI_RETURN_CODE_ENUM" are ignored
		// as the utilization from return values are not clear
		(tVoid) DECODER.eIsChannelLocked(hDecoder, channelID, &trSFChannel.bIsLocked);
		(tVoid) DECODER.eIsChannelSkipped(hDecoder, channelID, &trSFChannel.bIsSkipped);
		(tVoid) DECODER.eIsChannelSubscribed(hDecoder, channelID, &trSFChannel.bIsSubscribed);
		(tVoid) DECODER.eIsChannelMature(hDecoder, channelID, &trSFChannel.bIsMature);
		(tVoid) DECODER.eIsChannelFreeToAir(hDecoder, channelID, &trSFChannel.bIsFreeToAir);
	}
}

// Extract Team Attributes
// Method returns TRUE if the Team is a SF Eligible Team
tBool fc_sxm_tclSportsFlashProxy::bExtractTeamInfo(LEAGUE_OBJECT leagueObj, TEAM_OBJECT teamObj, fc_sxm_trSFTeam& trSFTeam) const
{
	// Reset the team object
	trSFTeam.vReset();
	if (teamObj != TEAM_INVALID_OBJECT)
	{
		// Fill Team's -- Full Name, Nick name, Abbreviated Name
		fc_sxm_vCopySmsString2Stl((TEAM.hName(teamObj)), trSFTeam.sTeamName);
		fc_sxm_vCopySmsString2Stl((TEAM.hNickname(teamObj)), trSFTeam.sTeamNickName);
		fc_sxm_vCopySmsString2Stl((TEAM.hAbbreviation(teamObj)), trSFTeam.sTeamAbbrName);
		trSFTeam.hTeamObj = teamObj;
		// Extract Team's CID object, TableVersion number and TeamID
		trSFTeam.hTeamCID = TEAM.hId(teamObj);
		trSFTeam.s32TableVersion = TEAM.n32Version(teamObj);
		trSFTeam.bIsSFEligible = TEAM.bIsSportsFlashEligible(leagueObj, teamObj);
		trSFTeam.u16TeamID = TEAM.un16Id(teamObj);
		// Setting bIsMonitored status to TRUE forcefully
		trSFTeam.bIsMonitored = TRUE;
	}
	return trSFTeam.bIsSFEligible;
}

//Extract League Attributes
// Method returns TRUE if the League supports Sports Flash
tBool fc_sxm_tclSportsFlashProxy::bExtractLeagueInfo(LEAGUE_OBJECT leagueObj, fc_sxm_trSFLeague& trSFLeague) const
{
	// Reset league object
	trSFLeague.vReset();
	if ((leagueObj != LEAGUE_INVALID_OBJECT) &&
			((trSFLeague.bIsSFEnabled = LEAGUE.bIsSportsFlashEnabled(leagueObj)) == TRUE))
	{
		// Fill league type, Abbreaviated Name and Full Name
		fc_sxm_vCopySmsString2Stl((LEAGUE.hAbbreviation(leagueObj)), trSFLeague.sAbbrLeagueName);
		fc_sxm_vCopySmsString2Stl((LEAGUE.hName(leagueObj)), trSFLeague.sLeagueName);
		trSFLeague.enLeague = LEAGUE.eLeague(leagueObj);
		// Extract and Fill League's CID Object, Sport type, Table Version and League ID
		trSFLeague.hLeagueCID = LEAGUE.hId(leagueObj);
		trSFLeague.hLeagueObj = leagueObj;
		trSFLeague.enSports = LEAGUE.eSport(leagueObj);
		trSFLeague.s32TableVersion = LEAGUE.n32Version(leagueObj);
		trSFLeague.u8LeagueID = LEAGUE.un8Id(leagueObj);
	}
	return trSFLeague.bIsSFEnabled;
}

// Sports, League and Team Iterate content Callbacks

// Sports Iterator
tBool fc_sxm_tclSportsFlashProxy::bSportsIterateContent(SPORTS_CONTENT_ITERATOR_CALLBACK bSportsIteratorCB,
		tVoid* poSportsIteratorCallbackArg) const
{
	// Iterate through the currently available list of sports content using bSportsIteratorCB method.
	return (SPORTS.bIterateContent(bSportsIteratorCB, poSportsIteratorCallbackArg));
}

// League Iterator
tBool fc_sxm_tclSportsFlashProxy::bLeagueIterateContent(LEAGUE_CONTENT_ITERATOR_CALLBACK bLeagueIteratorCB,
		tVoid* poLeagueIteratorCallbackArg) const
{
	// Iterate through the currently available list of LEAGUE content using bLeagueIteratorCB method.
	return (LEAGUE.bIterateContent(bLeagueIteratorCB, poLeagueIteratorCallbackArg));
}

// Team Iterator
tBool fc_sxm_tclSportsFlashProxy::bTeamIterateContent( LEAGUE_OBJECT leagueObj,
		TEAM_CONTENT_ITERATOR_CALLBACK bTeamIteratorCB,
		tVoid* poTeamIteratorCallbackArg) const
{
	// Iterate through the currently available list of TEAM content of a LEAGUE using bTeamIteratorCB method
	return (TEAM.bIterateContent(leagueObj, bTeamIteratorCB, poTeamIteratorCallbackArg));
}

// Get Decoder Object
DECODER_OBJECT fc_sxm_tclSportsFlashProxy::hGetDecoder() const
{
	return (( (_poDecoder)?
			_poDecoder->oGetDecoderObject() :
			DECODER_INVALID_OBJECT));
}

// Compare 2 CID Objects for equality
tS16 fc_sxm_tclSportsFlashProxy::s16CompareCIDObjects(CID_OBJECT hCID1, CID_OBJECT hCID2) const
{
	return (CID.n16Equal(hCID1, hCID2));
}
