/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_rapid_channel_browse.cpp
* @brief       Implementation for Rapid Channel Browsing Functionality.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

/**
 * @headerfile fc_sxm_tcl_rapid_channel_browse.h "fc_sxm_tcl_rapid_channel_browse.h"
 */
#include "fc_sxm_tcl_rapid_channel_browse.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_AUDIO_RAPIDCHANNEL_BROWSE
#include "trcGenProj/Header/fc_sxm_tcl_rapid_channel_browse.cpp.trc.h"
#endif


/**
 * @constructor.
 */
fc_sxm_tclRapidChannelBrowse::fc_sxm_tclRapidChannelBrowse() : _u16Pivot((1u << 15))
{ // Default CTOR
}

/**
 * @destructor.
 */
fc_sxm_tclRapidChannelBrowse::~fc_sxm_tclRapidChannelBrowse()
{
	vReset();
}

/**
 * @brief Reset all member variables to default values
 *        UTest Completed.
 * @return void
 */
tVoid fc_sxm_tclRapidChannelBrowse::vReset()
{
	ETG_TRACE_USR2(("fc_sxm_tclRapidChannelBrowse::vReset"));
	_u16Pivot = (1u << 15);
	_enDirection.enType = midw_ext_fi_tcl_e8_BrowseDirection::FI_EN_BROWSE_INVALID;
	_enLoop.enType = midw_ext_fi_tcl_e8_BrowseChannelLoop::FI_EN_LOOP_AROUND;
	_vectConfig.clear();
	_vectChannels.clear();
}
/**
 * @brief Initialize all member variables to values passed as arguments. One of the four entry points for the class.
 *        UTest Completed.
 * @param[in] enDirection Browse Direction.
 * @param[in] configuration Browse Configuration.
 * @param[in] tunedChnID Current Tuned Channel ID.
 * @param[in] vectChannels List of Channel ID's that needs to be considered for Rapid Channel Browse.
 * @return void.
 *
 * @post All the member variables are initialized based on the arguments passed.
 *       Any minor errors to the input arguments are corrected before initialization.
 */
tVoid fc_sxm_tclRapidChannelBrowse::vInit(midw_ext_fi_tcl_e8_BrowseDirection enDirection,
		const midw_ext_fi_tcl_BrowseChannelConfig& configuration,
		CHANNEL_ID tunedChnID,
		const vector<CHANNEL_ID>& vectChannels)
{
	_enDirection = enDirection;
	_enLoop = configuration.Loop;
	_vectChannels = vectChannels;
	if (!bFindPivot(tunedChnID))
	{
		// This condition would get executed when the current tuned channel is either locked or skipped
		vCalibratePivot(tunedChnID);
	}
	ETG_TRACE_USR2(("fc_sxm_tclRapidChannelBrowse::vInit Direction = %u LoopType = %u pivot = %u", _enDirection.enType, _enLoop.enType, _u16Pivot));
	vFillConfig(configuration.Configuration);
}

/**
 * @brief Get the next timer expiry duration in milliseconds. One of the four entry points for the class.
 *        UTest Completed.
 * @return Timer expiry duration in milliseconds.
 * @retval Zero Stop Rapid Channel Browse Timer and put Rapid Channel Browse to OFF.
 *
 * @post HeadConfig's completed steps is incremented by one.
 */
tU16 fc_sxm_tclRapidChannelBrowse::u16GetNextDuration()
{
	if (_vectConfig[0].u16MaxSteps == _vectConfig[0].u16CompletedSteps )
	{
		// If one complete configuration cycle is completed, then remove it from the list
		vRemoveConfig();
	}

	if (0u != _vectConfig[0].u16Duration )
	{
		// If duration = 0, then continue with the same channel rate forever until Rapid Channel Browse Mode is stopped
		// Increment completed Steps
		_vectConfig[0].u16CompletedSteps++;
	}
	ETG_TRACE_USR2(("fc_sxm_tclRapidChannelBrowse::u16GetNextDuration  Rate = %u", _vectConfig[0].u16Rate));
	return (_vectConfig[0].u16Rate);
}

/**
 * @brief Get the next Channel ID for which the notification has to be sent to Clients. One of the four entry points for the class.
 *        UTest Completed
 * @return Channel ID of the next channel for which notification has to be sent.
 * @retval CHANNEL_INVALID_ID  stop sending updates and keep Rapid Channel Browse Mode OFF.
 * @retval Valid_Channel_ID keep sending updates and keep Rapid Channel Browse Mode ON.
 *
 * @post The pivot channel Index is modified.
 */
CHANNEL_ID fc_sxm_tclRapidChannelBrowse::u16GetNextChannelID()
{
	CHANNEL_ID chnID = CHANNEL_INVALID_ID;
	if (_enDirection.enType == midw_ext_fi_tcl_e8_BrowseDirection::FI_EN_BROWSE_DOWN)
	{
		chnID = u16ChannelDown();
	}
	else if (_enDirection.enType == midw_ext_fi_tcl_e8_BrowseDirection::FI_EN_BROWSE_UP)
	{
		chnID = u16ChannelUp();
	}
	ETG_TRACE_USR2(("fc_sxm_tclRapidChannelBrowse::u16GetNextChannelID = %u", chnID));
	return chnID;
}

/**
 * @brief Remove the Head Configuration stored because the maximum no of steps required to complete the Configuration has reached.
 *        UTest Completed.
 * @return void.
 *
 * @post the size of BrowseRate Configuration list is reduced by one.
 */
tVoid fc_sxm_tclRapidChannelBrowse::vRemoveConfig()
{
	if (_vectConfig.size())
	{
		// Erase the first item in the list
		_vectConfig.erase(_vectConfig.begin());
	}
}

/**
 * @brief Extract, Manipulate and Fill internal list of configurable browse parameters from the arguments passed.
 *        UTest indirectly completed from vInit method.
 * @param[in] cfg The Browse Rate Configuration parameters.
 * @return void.
 *
 * @post BrowseRate Configuration list is cleared and reinitialized based on the new values passed.
 */
tVoid fc_sxm_tclRapidChannelBrowse::vFillConfig(const vector<midw_ext_fi_tcl_BrowseRateConfiguration, allocator<midw_ext_fi_tcl_BrowseRateConfiguration> >& cfg)
{
	_vectConfig.clear();
	vector<midw_ext_fi_tcl_BrowseRateConfiguration, allocator<midw_ext_fi_tcl_BrowseRateConfiguration> >::const_iterator cIt = cfg.begin();
	// Read configurable parameters sent by Client and store it in internal list
	while(cIt != cfg.end())
	{
		fc_sxm_trBrowseRateConfiguration nextConfig(cIt->duration, cIt->rate, 0u);
		if (0u != cIt->duration)
		{
			vDiscernSteps(nextConfig);
		}
		_vectConfig.push_back(nextConfig);
		++cIt;
	}
}

/**
 * @brief From the list of channel id's stored and the current tuned Channel ID passed, find the pivot index.
 *        UTest Completed.
 * @param[in] tunedChannel Current Tuned Channel ID.
 * @return true if Pivot index could be identified.
 *
 * @post If true is returned, pivot channel index is saved. If false is returned, pivot channel index is set to default.
 */
bool fc_sxm_tclRapidChannelBrowse::bFindPivot(CHANNEL_ID tunedChannel)
{
	_u16Pivot = (1u << 15);
	vector<CHANNEL_ID>::const_iterator cIt = _vectChannels.begin();
	// Iterate thru the channel list to see if currentTunedChannel could be identified
	for (tU16 u16Index = 0u; cIt != _vectChannels.end(); ++cIt, ++u16Index)
	{
		if (tunedChannel == *cIt)
		{
			_u16Pivot = u16Index;
			break;
		}
	}
	ETG_TRACE_USR2(("fc_sxm_tclRapidChannelBrowse::pivot = %u", _u16Pivot));
	// Pivot index value could point to 1u << 15 if the current tuned channel is locked or a skipped channel
	return (_u16Pivot != (1u << 15));
}

/**
 * @brief If the pivot index could not be identified from method "bFindPivot", this method can set the pivot Index based on Channel ID.
 *        UTest Completed.
 * @param[in] tunedChannel Current Tuned Channel ID.
 * @return void.
 *
 * @pre Channel ID's stored in internal list are assumed to be in sorted order.
 * @post pivot channel index is saved.
 */
tVoid fc_sxm_tclRapidChannelBrowse::vCalibratePivot(CHANNEL_ID tunedChannel)
{
	// Method gets executed when current tuned channel is not browseable
	if ( midw_ext_fi_tcl_e8_BrowseDirection::FI_EN_BROWSE_DOWN == _enDirection.enType )
	{
		vCalibrateDown(tunedChannel);
	}
	else if ( midw_ext_fi_tcl_e8_BrowseDirection::FI_EN_BROWSE_UP == _enDirection.enType )
	{
		vCalibrateUp(tunedChannel);
	}
	ETG_TRACE_USR2(("fc_sxm_tclRapidChannelBrowse::vCalibratePivot pivot Index = %u", _u16Pivot));
}

/**
 * @brief If the pivot index could not be identified from method "bFindPivot", this method can set the pivot Index based on Channel ID for Browse Direction DOWN.
 *        UTest indirectly complete from vCalibratePivot.
 * @param[in] tunedChannel Current Tuned Channel ID.
 * @return void.
 *
 * @pre 1) Channel ID's stored in internal list are assumed to be in sorted order 2) Size of Channel ID list cannot be zero.
 * @post pivot channel index is saved.
 */
tVoid fc_sxm_tclRapidChannelBrowse::vCalibrateDown(CHANNEL_ID tunedChannel)
{
	// Size of _vectChannels cannot be zero
	if (1u != _vectChannels.size())
	{
		tU16 u16End = (tU16)(_vectChannels.size() - 1u);

		if ( (tunedChannel < _vectChannels[0]) || (tunedChannel > _vectChannels[u16End]))
		{
			_u16Pivot = 0u;
		}
		else
		{
			tU16 u16First = 0u;
			tU16 u16Second = 1u;
			do {
				// Iterate thru the list until a Range is found
				if ( (tunedChannel > _vectChannels[u16First]) && (tunedChannel < _vectChannels[u16Second]))
				{
					_u16Pivot = u16Second;
					break;
				}
				++u16First;
				++u16Second;
			} while (u16End != u16First);
		}
	}
	else
	{
		// Just one item in the list _vectChannels.
		_u16Pivot = 0u;
	}
}

/**
 * @brief If the pivot index could not be identified from method "bFindPivot", this method can set the pivot Index based on Channel ID for Browse Direction UP.
 *        UTest indirectly complete from vCalibratePivot.
 * @param[in] tunedChannel Current Tuned Channel ID.
 * @return void.
 *
 * @pre 1) Channel ID's stored in internal list are assumed to be in sorted order 2) Size of Channel ID list cannot be zero.
 * @post pivot channel index is saved.
 */
tVoid fc_sxm_tclRapidChannelBrowse::vCalibrateUp(CHANNEL_ID tunedChannel)
{
	// Size of _vectChannels cannot be zero
	if (1u != _vectChannels.size())
	{
		tU16 u16End = (tU16)(_vectChannels.size() - 1u);

		if ( (tunedChannel < _vectChannels[0]) || (tunedChannel > _vectChannels[u16End]))
		{
			_u16Pivot = u16End;
		}
		else
		{
			tU16 u16First = 0u;
			tU16 u16Second = 1u;
			do {
				// Iterate thru the list until a Range is found
				if ( (tunedChannel > _vectChannels[u16First]) && (tunedChannel < _vectChannels[u16Second]))
				{
					_u16Pivot = u16First;
					break;
				}
				++u16First;
				++u16Second;
			} while (u16End != u16First);
		}
	}
	else
	{
		// Just one item in the list _vectChannels.
		_u16Pivot = 0u;
	}
}

/**
 * @brief For the given BrowseRateConfiguration, calculate the MAX steps needed to complete the configuration.
 *        UTest Completed.
 * @param[in,out] config Configuration Parameter whose MAX steps needs to be calculated.
 * @return void.
 */
tVoid fc_sxm_tclRapidChannelBrowse::vDiscernSteps(fc_sxm_trBrowseRateConfiguration& config) const
{
	// Error- Handling... If Rate of channel change is made less than 100 for any configuration,
	// its forcefully reset to 500 milliseconds
	if (100u > config.u16Rate)
		config.u16Rate = 500;
	config.u16MaxSteps = (tU16)(config.u16Duration * 1000u) / (config.u16Rate);
	if (0.0F != (tF32)((config.u16Duration * 1000u) % (config.u16Rate)))
		++config.u16MaxSteps;
}


/**
 * @brief Find the next channel whose details have to be passed to Client if the Channel Browse Direction is UP.
 *        UTest Indirectly Completed from u16GetNextChannel.
 * @return Channel ID of the next channel for which notification has to be sent.
 * @retval CHANNEL_INVALID_ID  stop sending updates and keep Rapid Channel Browse Mode OFF.
 * @retval Valid_Channel_ID keep sending updates and keep Rapid Channel Browse Mode ON.
 *
 * @post pivot channel index is updated
 */
CHANNEL_ID fc_sxm_tclRapidChannelBrowse::u16ChannelUp()
{
	CHANNEL_ID nextChn = CHANNEL_INVALID_ID;
	// Using forward iterator to browse channel up
	vector<CHANNEL_ID>::iterator fIt = _vectChannels.begin() + _u16Pivot;
	if (fIt != _vectChannels.end())
	{
		++fIt;
		if (fIt == _vectChannels.end())
		{
			if ( _enLoop.enType == midw_ext_fi_tcl_e8_BrowseChannelLoop::FI_EN_LOOP_AROUND)
			{
				nextChn = _vectChannels.front();
				_u16Pivot = 0u;
			}
			else
			{
				_u16Pivot = (1u << 15);
			}
		}
		else
		{
			nextChn = *fIt;
			++_u16Pivot;
		}
	}
	return nextChn;
}

/**
 * @brief Find the next channel whose details have to be passed to Client if the Channel Browse Direction is UP.
 *        UTest Indirectly Completed from u16GetNextChannel
 * @return Channel ID of the next channel for which notification has to be sent.
 * @retval CHANNEL_INVALID_ID  stop sending updates and keep Rapid Channel Browse Mode OFF.
 * @retval Valid Channel ID keep sending updates and keep Rapid Channel Browse Mode ON.
 *
 * @post pivot channel index is updated
 */
CHANNEL_ID fc_sxm_tclRapidChannelBrowse::u16ChannelDown()
{
	CHANNEL_ID prevChn = CHANNEL_INVALID_ID;
	// Using reverse iterator to browse channel down
	vector<CHANNEL_ID>::reverse_iterator rIt = _vectChannels.rbegin() + (_vectChannels.size() -1 - _u16Pivot);
	if (rIt != _vectChannels.rend())
	{
		++rIt;
		if (rIt == _vectChannels.rend())
		{
			if ( _enLoop.enType == midw_ext_fi_tcl_e8_BrowseChannelLoop::FI_EN_LOOP_AROUND)
			{
				prevChn = _vectChannels.back();
				_u16Pivot = (tU16)(_vectChannels.size() -1u);
			}
			else
			{
				_u16Pivot = (1u << 15);
			}
		}
		else
		{
			prevChn = *rIt;
			--_u16Pivot;
		}
	}
	return prevChn;
}

