/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_playback.h
* @brief       Declaration for handling instant replay control functionalities, 
*              updating buffered data info such as progress bar, status bar, and play point indicators.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#ifndef FC_SXM_TCL_PLAYBACK_H
#define FC_SXM_TCL_PLAYBACK_H

#include "fc_sxm_common.h"
#include "fc_sxm_audio_types.h"
#include "fc_sxm_tcl_audio_app.h"
#include "fc_sxm_tcl_audio_properties.h"
#include "fc_sxm_tcl_playback_if.h"

/* Set time out duration or delay as 500 ms */
#define FC_SXM_IR_SEEK_TIME_DELAY 500
/* Set skip time out duration in ms to watch */
#define FC_SXM_IR_SKIP_TIME_DELAY 500

/* Speed in seconds at which audio must be jumped or seeked */
#define FC_SXM_IR_SEEK_SPEED_1    5   /* 0 sec to 3 sec */
#define FC_SXM_IR_SEEK_SPEED_2   10   /* 3 sec to 8 sec */
#define FC_SXM_IR_SEEK_SPEED_3   30   /* above 8 sec    */

/* Time slices to seek audio */
#define FC_SXM_IR_SEEK_TIMEOUT_COUNT_1      0
#define FC_SXM_IR_SEEK_TIMEOUT_COUNT_2   (3*2) 
#define FC_SXM_IR_SEEK_TIMEOUT_COUNT_3   (8*2)

/* Min seek duration */
#define FC_SXM_IR_MIN_SEEK_TIME 5
/* Min skip threshold */
#define FC_SXM_IR_MIN_SKIP_THRESHOLD 3

typedef enum {
    e_SXM_EN_OFF
   ,e_SXM_EN_PAUSE
   ,e_SXM_EN_ON
}eSXMSourceActivity;

/* MethodStart for audio playback control from CCA client */
struct fc_sxm_trMsgAudioMStartIRPlaybackControl : public fc_sxm_tclMessage
{
    /* Service and service ID */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_PLAYBACK + 1);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgInstantReplayControlMethodStart oFiMsg;
};

/* Message sent from SMS context containing current playback info */
struct fc_sxm_trMsgAudioSmsEvtPlaybackEvt : public fc_sxm_tclMessage
{
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_PLAYBACK + 2);
    fc_sxm_trMsgAudioSmsEvtPlaybackEvt()
       : IsPlayActive(FALSE)
       , u16TracksBefore(0)
       , u16TracksRemaining(0)
       , u32TimeFromTrackStart(0)
       , u32RecordDuration(0)
       , u32RecordElapsedTime(0)
       , u32RecordRemainingTime(0)
       , u8PlayPercentage(0)
       , u8FillPercentage(0)
       , ePlaybackEventMask(PLAYBACK_OBJECT_EVENT_NONE)
    , s32TimeOffset(0)
    , u32WarningOffset(0)
    { }
    tBool IsPlayActive;
    tU16  u16TracksBefore;
    tU16  u16TracksRemaining;
    UN32  u32TimeFromTrackStart;
    UN32  u32RecordDuration;
    UN32  u32RecordElapsedTime;
    UN32  u32RecordRemainingTime;
    UN8   u8PlayPercentage;
    UN8   u8FillPercentage;
    PLAYBACK_OBJECT_EVENT_MASK ePlaybackEventMask;

    N32 s32TimeOffset;
    UN32 u32WarningOffset;

    tVoid vPrint() const;
};

/*
 * messages send from cca-context from timer class to audio app 
 * for seek operation timeout updates.
 */
struct fc_sxm_trMsgIRSeekTimer : public fc_sxm_tclMessage
{
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_PLAYBACK + 3);
};

struct fc_sxm_trMsgAudioMuteState: public fc_sxm_tclMessage 
{
   SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_PLAYBACK + 4);
   eSXMSourceActivity eSourceActivity;
};

/* MethodStart for set current volume from CCA client */
struct fc_sxm_trMsgAudioMStartCurrentVolume : public fc_sxm_tclMessage
{
    /* Service and service ID */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_PLAYBACK + 5);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgSetCurrentVolumeMethodStart oFiMsg;
};

/*
 * messages send from cca-context from timer class to audio app 
 * for skip operation timeout updates.
 */
struct fc_sxm_trMsgIRSkipTimer : public fc_sxm_tclMessage
{
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_PLAYBACK + 6);
};

/* Record content struct containing buffer content info */
struct fc_sxm_trRecordContentInfo
{
   fc_sxm_trRecordContentInfo()
   {
      u32TimeFromTrackStart  = 0;
      u32RecordDuration      = 0;
      u32RecordElapsedTime   = 0;
      u32RecordRemainingTime = 0;
      u16TracksBefore        = 0;
      u16TracksRemaining     = 0;
      u8PlayPercentage       = 0;
      u8FillPercentage       = 0;
      vClear();
   }
   ~fc_sxm_trRecordContentInfo()
   {
      u32TimeFromTrackStart  = 0;
      u32RecordDuration      = 0;
      u32RecordElapsedTime   = 0;
      u32RecordRemainingTime = 0;
      u16TracksBefore        = 0;
      u16TracksRemaining     = 0;
      u8PlayPercentage       = 0;
      u8FillPercentage       = 0;
      vClear();
   }
   tVoid vClear(tVoid) 
   {
      bIsSeekActive    = FALSE;
      u32TimeOutCount  = 0;
      s32SeekDirection = 0;
      u32SeekTime      = 0;
   }
   tVoid vPrintRecordInfo(tVoid);

   tBool bIsSeekActive;          /* Flag to indicate seek state is active */
   UN32  u32TimeFromTrackStart;  /* Duration from the beginning of current track */
   UN32  u32RecordDuration;      /* Buffer record duration */
   UN32  u32RecordElapsedTime;   /* Buffer recorded elapsed or played time */
   UN32  u32RecordRemainingTime; /* Buffer recorded remaining time */
   N32   s32SeekDirection;       /* Current seek direction */
   N32   u32SeekTime;            /* Current seek speed in time or sec */
   UN32  u32TimeOutCount;        /* Seek timer timeout count */
   UN16  u16TracksBefore;        /* Total tracks before current track */
   UN16  u16TracksRemaining;     /* Total tracks remaining after current track */
   UN8   u8PlayPercentage;		 /* Percentage of buffer at which the play point is currently present */
   UN8   u8FillPercentage;		 /* Percentage of buffer containing stored data */
};

struct IRStatusControl
{
   tBool bIgnoreIRUpdate;
   tBool bValidate;
   tBool bSetMode;
   tBool bHandleIRMode;

   IRStatusControl()
      : bIgnoreIRUpdate(FALSE)
      , bValidate(FALSE)
      , bSetMode(FALSE)
      , bHandleIRMode(FALSE)
   { }
};

/* Playback class to handle playback control */
class fc_sxm_tclPlayback
   : public fc_sxm_tclAudioMember
   , public Ifc_sxm_tclPlayback
   , public fc_sxm_tclSingleton<fc_sxm_tclPlayback>
{
   public:
      ~fc_sxm_tclPlayback();
      tVoid vUpdateDecoderState(DECODER_OBJECT hDecoderObject, DECODER_STATE_ENUM enDecoderState);
      tVoid vClearMemberData(tVoid);
      tVoid vProcess(fc_sxm_trMsgAudioMStartIRPlaybackControl const *prMsg);
      tVoid vProcess(fc_sxm_trMsgAudioMStartCurrentVolume const *prMsg);
      tVoid vProcess(fc_sxm_trMsgAudioSmsEvtPlaybackEvt const *prMsg);
      tVoid vProcessTimer(fc_sxm_trMsgIRSeekTimer const *prMsg);
      tVoid vProcessTimer(fc_sxm_trMsgIRSkipTimer const *prMsg);
      tBool bIsReplayModeActive(tVoid);
      tVoid vPauseAudioPlayback(tVoid);
      tVoid vProcess(fc_sxm_trMsgAudioMuteState const *prMsg);
      tVoid vIgnoreIRUpdate(tBool bValidate, tBool bSetMode);
      /* Set Audio App Pointer Reference*/
      tVoid vSetApp(Ifc_sxm_tclAudioApp* audioApp);
      tBool bPlaybackStatus(const UN32 &, const tU32 &, const UN32 &, const midw_ext_fi_tcl_e8_IRPlaybackStatus&, const N32&) const;
      tVoid vHandleInstantReplayStatus();

   private:
      friend class fc_sxm_tclSingleton<fc_sxm_tclPlayback>;
      fc_sxm_tclPlayback();
      tVoid vChangeSeekSpeed();

      fc_sxm_tcl_trAudioPropertyInstantReplayStatus _tIRStatus;
      fc_sxm_tcl_trAudioPropertyInstantReplayStatus _tIROldStatus;

      fc_sxm_trRecordContentInfo _tRecordInfo;

      /* Timer to handle seek operation */
      fc_sxm_tclTimer<fc_sxm_tclAudioApp, 
                     fc_sxm_trMsgIRSeekTimer, 
                     fc_sxm_tclPlayback> _mIRSeekTimer;

      /* Timer to handle skip operation */
      fc_sxm_tclTimer<fc_sxm_tclAudioApp, 
                      fc_sxm_trMsgIRSkipTimer, 
                      fc_sxm_tclPlayback> _mIRSkipTimer;

      tU32 _u32BufferTime;
      tBool _bUpdateIRStatus;
      IRStatusControl _eIRStatusControl;
      tS16 _s16IRSkipCount;
	  tU32 _u32ThresholdValue;
      Ifc_sxm_tclAudioApp* _poAudioApp;

   protected:
};

#endif
