/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_content_alerts.h
* @brief       Class declaration for Artist and song content alerts functionality.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#ifndef FC_SXM_TCL_CONTENT_ALERT_H
#define FC_SXM_TCL_CONTENT_ALERT_H

#include "fc_sxm_tcl_content_alert_proxy_if.h"
#include "fc_sxm_tcl_audio_member.h"
#include "fc_sxm_audio_fi.h"

// Forward Declaration
class Ifc_sxm_tclAudioApp;

/* MethodStart for Save Favorite as either artist or song from CCA client */
struct fc_sxm_trMsgAudioMStartSaveFavorite :
   public fc_sxm_tclMessage
{
   SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_CONTENT_ALERT + 1);
   fc_sxm_trAdressing rAdressing;
   midw_ext_sxm_audiofi_tclMsgSaveFavoriteMethodStart oFiMsg;
};

/* MethodStart for Get Favorite List from CCA client */
struct fc_sxm_trMsgAudioMStartGetFavoriteList :
   public fc_sxm_tclMessage
{
   SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_CONTENT_ALERT + 2);
   fc_sxm_trAdressing rAdressing;
   midw_ext_sxm_audiofi_tclMsgGetFavoriteListMethodStart oFiMsg;
};

/* MethodStart for modify favorite from CCA client */
struct fc_sxm_trMsgAudioMStartModifyFavorite :
   public fc_sxm_tclMessage
{
   SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_CONTENT_ALERT + 3);
   fc_sxm_trAdressing rAdressing;
   midw_ext_sxm_audiofi_tclMsgModifyFavoriteMethodStart oFiMsg;
};

/* MethodStart for Get alert notification list from CCA client */
struct fc_sxm_trMsgAudioMStartGetAlertNoticeList :
   public fc_sxm_tclMessage
{
   SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_CONTENT_ALERT + 4);
   fc_sxm_trAdressing rAdressing;
   midw_ext_sxm_audiofi_tclMsgGetAlertNoticeListMethodStart oFiMsg;
};

struct fc_sxm_trMsgAudioSeekAlertEvent :
   public fc_sxm_tclMessage
{
   SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_CONTENT_ALERT + 7);
};

/* Message for Get Favorite */
struct fc_sxm_trMsgAudioGetFavorite
{
   AT_SEEK_ENUM enFavoriteType;
   midw_ext_sxm_audiofi_tclMsgGetFavoriteListMethodResult ofiFavoriteList;
};

/* Message for Get Favorite */
struct fc_sxm_trMsgAudioModifyFavorite
{
   AT_SEEK_ENUM enFavoriteType;
   midw_ext_fi_tcl_e8_Action::tenType enAction;
   string sContentName;
};

/* Message for ActiveAlertChannelList */
struct fc_sxm_trMsgAudioActiveAlertChannelList
{
   tU32 u32Size;
   vector<CHANNEL_ID> vecChannelList;
};

/******************************************************************************
 * CLASS        : fc_sxm_tclContentAlert                                      *
 * ---------------------------------------------------------------------------*
 * DESCRIPTION  : Class fc_sxm_tclContentAlert to handle all content alerts   *
 *                related following operations                                *
 *                - Save favorite as either artist or song                    *
 *                - Enable/Disable selected favorite                          *
 *                - Delete selected favorite                                  *
 *                - Block/Unblock favorite either artist or song              *
 *                - Enable/Disable all favorites                              *
 *****************************************************************************/
class fc_sxm_tclContentAlert :
   public fc_sxm_tclAudioMember,
   public fc_sxm_tclSingleton<fc_sxm_tclContentAlert>
{
public:
   virtual ~fc_sxm_tclContentAlert();

   tVoid vSetApp(Ifc_sxm_tclAudioApp* poAudioApp, Ifc_sxm_tclContentAlertProxy* poContentAlertProxy);
   tVoid vUpdateDecoderState(DECODER_OBJECT hDecoder, DECODER_STATE_ENUM enDecoderState);
   tVoid vClearMemberData(tVoid);

   tVoid vProcess(fc_sxm_trMsgAudioMStartSaveFavorite const *prMsg);
   tVoid vProcess(fc_sxm_trMsgAudioMStartGetFavoriteList const *prMsg);
   tVoid vProcess(fc_sxm_trMsgAudioMStartModifyFavorite const *prMsg);
   tVoid vProcess(fc_sxm_trMsgAudioMStartGetAlertNoticeList const *prMsg);

   tVoid vProcess(fc_sxm_trMsgAudioATAlerts const *prMsg);
   tVoid vProcess(fc_sxm_trMsgAudioSeekAlertEvent const *prMsg);
   tVoid vProcess(fc_sxm_trMsgAudioSeekCategoryUpdate const *prMsg);

private:
   friend class fc_sxm_tclSingleton<fc_sxm_tclContentAlert> ;
   fc_sxm_tclContentAlert();

#ifdef UNITTEST_FC_SXM
// For unit test product, to perform effective unit-testing of all private members,
// the private methods/variables are made public.
// In production-code, the methods and variables still remain private
public:
#endif
// Data Member methods
   DECODER_OBJECT hGetDecoder() const;
   // Send the content alert messages
   tVoid vSendContentAlertFiMsg(fc_sxm_trAdressing const &rAddr,
      const midw_ext_sxm_audiofi_tclMsgBaseMessage& oMessage);
   tVoid vSendContentAlertFiError(fc_sxm_trAdressing const &rAddr, tInt s32ErrorMsg);

   // Increment and decrement favorite count
   tVoid vIncrementFavoriteCount(AT_SEEK_ENUM enFavoriteType);
   tVoid vDecrementFavoriteCount(AT_SEEK_ENUM enFavoriteType);

   //
   tBool bIsFavoriteListFull(AT_SEEK_ENUM enFavoriteType, tU8 u8MaxFavoriteCount) const;
   tBool bIsFavoriteListEmpty(AT_SEEK_ENUM enFavoriteType) const;
   tBool bIsFavoriteTypeValid(AT_SEEK_ENUM enFavoriteType) const;
   tBool bIsModifyFavoriteRequestValid(const fc_sxm_trMsgAudioModifyFavorite &oModifyFavoriteRequest) const;
   tBool bIsFavoriteAlertBlocked(midw_ext_fi_tcl_e8_FavoriteType::tenType enFavoriteType) const;
   tBool bIsAlertAlreadyNotified(const midw_ext_fi_tcl_AlertInfo& oAlert) const;
   bool bIsAlertOnTuned(tU16 u16TunedChannelID, tU16 u16AlertChannelID) const;
   bool bIsRadioIDChannelTuned(tU16 u16TunedChannelID) const;
   bool bIsChannelInfoMatched(const midw_ext_fi_tcl_AlertInfo& oNotifiedAlert,
      const midw_ext_fi_tcl_AlertInfo& oNewAlert) const;
   bool bIsSongInfoMatched(const midw_ext_fi_tcl_AlertInfo& oNotifiedAlert,
      const midw_ext_fi_tcl_AlertInfo& oNewAlert) const;
   bool bIsActiveAlertExpired(CHANNEL_ID hAlertChannelID,
      const vector<CHANNEL_ID>& vChannelList) const;


   tVoid vSetBlockAlertsFlag(AT_SEEK_ENUM enFavoriteType, midw_ext_fi_tcl_e8_Action::tenType enAction);

   tU8 u8GetFavoriteCount(AT_SEEK_ENUM enFavoriteType) const;
   STRING_OBJECT hGetRegisteredContent(SEEK_CONTENT_OBJECT hSeekContent, AT_SEEK_ENUM enRegisteredContentType) const;
   string hGetRegisteredContentTextAsString(SEEK_CONTENT_OBJECT hSeekContent,
      AT_SEEK_ENUM enRegisteredContentType) const;
   midw_ext_fi_tclString hGetRegisteredContentTextAsFiString(SEEK_CONTENT_OBJECT hSeekContent,
      AT_SEEK_ENUM enRegisteredContentType) const;
   midw_ext_fi_tcl_e8_FavoriteStatus::tenType enGetSongAvailabilityStatus(SEEK_AVAILABILTY_ENUM enTitleAvailablity,
      SEEK_AVAILABILTY_ENUM enArtistAvailablity) const;
   midw_ext_fi_tcl_e8_FavoriteStatus::tenType enGetArtistAvailabilityStatus(SEEK_AVAILABILTY_ENUM enTitleAvailablity,
      SEEK_AVAILABILTY_ENUM enArtistAvailablity) const;
   midw_ext_fi_tcl_AlertInfo oGetAlertInfo(SEEK_EVENT_OBJECT hSeekEvent) const;
   midw_ext_fi_tcl_e8_FavoriteStatus::tenType enGetSaveFavoriteStatus(DECODER_OBJECT hDecoder,
      tU16 u16ChannelID,
      AT_SEEK_ENUM enFavoriteType,
      tU8 u8MaxFavoriteCount) const;
   midw_ext_fi_tcl_e8_Graphics::tenType enGetChannelGraphicsUpdate(tU16 u16ChannelID) const;
   midw_ext_fi_tcl_b8_ChannelStatus oGetChannelStatus(const tBool, const tBool, const tBool) const;

   midw_ext_fi_tcl_e8_FavoriteStatus::tenType enCheckArtistSongInfoAvailability(DECODER_OBJECT hDecoder,
      tU16 u16ChannelID,
      AT_SEEK_ENUM enFavoriteType) const;
   tBool bCheckAndGetFavorite(SEEK_CONTENT_OBJECT hSeekContent, fc_sxm_trMsgAudioGetFavorite *poGetFavorite) const;
   tBool bCheckAndModifyFavorite(SEEK_CONTENT_OBJECT hSeekContent, fc_sxm_trMsgAudioModifyFavorite const *prMsg);
   tVoid vCheckAndRemoveAlerts(CHANNEL_ID oChannelId,
      STRING_OBJECT oArtist,
      STRING_OBJECT oSong);
   N16 u16RemoveExpiredAlerts(vector<midw_ext_fi_tcl_AlertInfo>& vecAlertList,
      const vector<CHANNEL_ID>& vChannelList) const;

   tBool bRemoveExpiredAlert(vector<midw_ext_fi_tcl_AlertInfo>& vecActiveAlertList,
      CHANNEL_ID hExpiredChannelID,
      string sExpiredArtistName,
      string sExpiredSongName) const;
   void vRemoveAlertIfFavoriteDeleted(vector<midw_ext_fi_tcl_AlertInfo>& vecAlertList,
      AT_SEEK_ENUM enFavoriteType,
      string sDeleteContentName) const;

   //
   tVoid vUpdateFavoritesCount(SEEK_CONTENT_OBJECT hSeekContent);
   SMSAPI_RETURN_CODE_ENUM enUpdateFavorite(SEEK_CONTENT_OBJECT hSeekContent,
      midw_ext_fi_tcl_e8_Action::tenType enAction,
      AT_SEEK_ENUM enFavoriteType);
   tVoid vUpdateAlertList(tBool bIsAlertBlocked, const fc_sxm_trMsgAudioATAlertList& oAlerts);

   tVoid vHandleExpiredAlert(DECODER_OBJECT hDecoder);
   tVoid vHandleNewAlerts(DECODER_OBJECT hDecoder, fc_sxm_trMsgAudioATAlerts& oActiveContentAlertList) const;
   SMSAPI_RETURN_CODE_ENUM enHandleFavoriteDeletion(SEEK_CONTENT_OBJECT hSeekContent,
      AT_SEEK_ENUM enFavoriteType);
   tVoid vAddToActiveContentAlertList(const midw_ext_fi_tcl_AlertInfo& oAlert,
      fc_sxm_trMsgAudioATAlerts &oActiveContentAlertList) const;
   tVoid vNotifyIndividualAlert(const midw_ext_fi_tcl_AlertInfo& oAlert) const;
   tVoid vNotifyAlertExpirationEvent(CHANNEL_ID oChannelId,
      AT_SEEK_ENUM enAlertType,
      STRING_OBJECT oArtist,
      STRING_OBJECT oSong) const;
   tVoid vNotifyActiveContentAlertList(AT_SEEK_ENUM enAlertType);
   // Print methods
   tVoid vPrintFavoriteCount(AT_SEEK_ENUM enFavoriteType, tU8 u8RegisteredCount) const;
   tVoid vPrintContentStatusMessage(midw_ext_fi_tcl_e8_FavoriteStatus::tenType enFavoriteSaveStatus) const;
   tVoid vPrintModifyFavoriteStatus(midw_ext_fi_tcl_e8_Action::tenType enAction,
      AT_SEEK_ENUM enRegisteredFavoriteType,
      const string& sRegisteredContentName,
      SMSAPI_RETURN_CODE_ENUM enStatusCode) const;
   tVoid vPrintAlertInfo(const midw_ext_fi_tcl_AlertInfo& tAlerts) const;
   tVoid vPrintAlertList(const vector<midw_ext_fi_tcl_AlertInfo>& vecAlertList) const;

   // Method for operation on registered favorite
   tBool bSaveFavorite(DECODER_OBJECT hDecoder,
      fc_sxm_trMsgAudioMStartSaveFavorite const *prMsg,
      midw_ext_sxm_audiofi_tclMsgSaveFavoriteMethodResult *poMResult);
   tBool bGetFavoriteList(DECODER_OBJECT hDecoder,
      fc_sxm_trMsgAudioMStartGetFavoriteList const *prMsg,
      fc_sxm_trMsgAudioGetFavorite* poGetFavorite) const;
   N16 u16GetActiveAlertChannel(CHANNEL_OBJECT hChannel,
      fc_sxm_trMsgAudioActiveAlertChannelList *oActiveAlerts) const;
   tBool bRegisterFavorite(DECODER_OBJECT hDecoder,
      AT_SEEK_ENUM enFavoriteType,
      tU16 u16ChannelID);

   tBool bModifyFavorite(DECODER_OBJECT hDecoder,
      fc_sxm_trMsgAudioMStartModifyFavorite const *prMsg,
      midw_ext_sxm_audiofi_tclMsgModifyFavoriteMethodResult* poMResult);

   // Call back function to iterate through registered favorite list
   static tBool cb_bGetFavoriteList(DECODER_OBJECT hDecoder,
      SEEK_SERVICE_ENUM enSeekService,
      SEEK_CONTENT_OBJECT hSeekContent,
      void *pvSeekContentIteratorArg);
   static tBool cb_bGetFavoriteCount(DECODER_OBJECT hDecoder,
      SEEK_SERVICE_ENUM enSeekService,
      SEEK_CONTENT_OBJECT hSeekContent,
      void *pvSeekContentIteratorArg);
   static tBool cb_bModifyFavorite(DECODER_OBJECT hDecoder,
      SEEK_SERVICE_ENUM enSeekService,
      SEEK_CONTENT_OBJECT hSeekContent,
      void *pvSeekContentIteratorArg);
   static N16 cb_bGetActiveAlertChannelList(CATEGORY_OBJECT hCategory,
      CATEGORY_CHANNEL_INDEX tCurrentIndex,
      CHANNEL_OBJECT hChannel,
      void * pvIterateArg);

// Data Member Variables
   Ifc_sxm_tclAudioApp* _poAudioApp;
   Ifc_sxm_tclContentAlertProxy* _poContentAlertProxy;
   tBool _bIsSeekServiceInitialized;
   tBool _bIsArtistAlertBlocked;
   tBool _bIsSongAlertBlocked;
   tU8 _u8RegisteredArtistCount;
   tU8 _u8RegisteredSongCount;
   CATEGORY_ID _hSeekCategoryID;
   vector<midw_ext_fi_tcl_AlertInfo> _vecArtistList;
   vector<midw_ext_fi_tcl_AlertInfo> _vecSongList;
};

#endif   // FC_SXM_TCL_CONTENT_ALERT_H
