/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_channel_list.h
* @brief       Implementation for channel list related functionalities. 
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

/*=============================================================================
=======                            DEFINES                              =======
=============================================================================*/

#ifndef FC_SXM_TCL_CHANNEL_LIST_H
#define FC_SXM_TCL_CHANNEL_LIST_H

#define FC_SXM_DEFAULT_CHANNEL_BROWSING_STEP 1
/*=============================================================================
=======                            INCLUDES                             =======
=============================================================================*/

#include "fc_sxm_tcl_channel_list_if.h"
#include "fc_sxm_tcl_rapid_channel_browse.h"

/*=============================================================================
=======                             TYPES                               =======
=============================================================================*/

/* 
 * MethodStart for get channel list from CCA client 
 */
struct fc_sxm_trMsgAudioMStartGetChnList
    : public fc_sxm_tclMessage 
{
    /* Service and service ID */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio,
                    FC_SXM_AUDIO_ACTION_BASE_CHN_LIST + 1);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgGetChannelListMethodStart oFiMsg;
};

/* 
 * MethodStart for select channel from CCA client 
 */
struct fc_sxm_trMsgAudioMStartSelChn 
    : public fc_sxm_tclMessage 
{
    /* Service and service id */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, 
                    FC_SXM_AUDIO_ACTION_BASE_CHN_LIST + 2);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgSelectChannelMethodStart oFiMsg;
};

/*
 * MethodStart for select channel with dwell time from CCA client 
 */
struct fc_sxm_trMsgAudioMStartSelChnWitDwellTime 
    : public fc_sxm_tclMessage 
{
    /* Service and service id */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, 
                    FC_SXM_AUDIO_ACTION_BASE_CHN_LIST + 3);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgSelectChannelWithDwellTimeMethodStart oFiMsg;
};

/*
 * MethodStart for get channel information list from CCA client
 */
struct fc_sxm_trMsgAudioMStartGetChnInfo
: public fc_sxm_tclMessage
{
    /* Service and service id */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio,
                    FC_SXM_AUDIO_ACTION_BASE_CHN_LIST + 4);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgGetChannelInfoMethodStart oFiMsg;
};

/*
 * messages send from cca-context from timer class to audio app 
 * for channel list timeout updates.
 */
struct fc_sxm_trMsgChannelListTimer
    : public fc_sxm_tclMessage 
{
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, 
                    FC_SXM_AUDIO_ACTION_BASE_CHN_LIST + 10);
};

/*
 * messages send from cca-context from timer class to audio app 
 * for channel select timeout updates.
 */
struct fc_sxm_trMsgChannelSelectDwellTimer
    : public fc_sxm_tclMessage 
{
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, 
                    FC_SXM_AUDIO_ACTION_BASE_CHN_LIST + 11);
};

/* Message for channel objects events */
struct fc_sxm_trMsgChannelEventCallback: public fc_sxm_tclMessage
{
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_CHN_LIST + 15);
    fc_sxm_trMsgChannelEventCallback(CHANNEL_OBJECT hChannel_, CHANNEL_EVENT_MASK tEventMask_):
        tEventMask(tEventMask_),
        rChannel(hChannel_, tEventMask_)
    {};
        
    CHANNEL_EVENT_MASK tEventMask;
    fc_sxm_trChannel rChannel;
};

// Messages for test tone
struct fc_sxm_trMsgAudioTestToneEvent: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_CHN_LIST + 16);
    fc_sxm_trMsgAudioTestToneEvent():u8EnableTone(FALSE) {};
    tU8 u8EnableTone;
};

// MethodStart for DisableMatureChannels from CCA client
struct fc_sxm_trMsgAudioMStartDisableMatureChannels : public fc_sxm_tclMessage
{
    /* Service and service ID */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_CHN_LIST + 17);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgDisableMatureChannelsMethodStart oFiMsg;
};


// MethodStart for SkipChannels from CCA client
struct fc_sxm_trMsgAudioMStartSkipChannels : public fc_sxm_tclMessage
{
    /* Service and service ID */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_CHN_LIST + 18);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgSkipChannelMethodStart oFiMsg;
};

// MethodStart for SkipChannels from CCA client
struct fc_sxm_trMsgAudioMStartSetCategoryMonitor : public fc_sxm_tclMessage
{
    /* Service and service ID */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_CHN_LIST + 19);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgSetCategoryMonitorMethodStart oFiMsg;
};

// MethodStart for RapidChannelBrowseStart from CCA client
struct fc_sxm_trMsgAudioMStartRapidChannelBrowseStart : public fc_sxm_tclMessage
{
    /* Service and service ID */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_CHN_LIST + 20);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgRapidChannelBrowseStartMethodStart oFiMsg;
};

// MethodStart for RapidChannelBrowseStop from CCA client
struct fc_sxm_trMsgAudioMStartRapidChannelBrowseStop : public fc_sxm_tclMessage
{
    /* Service and service ID */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_CHN_LIST + 21);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgRapidChannelBrowseStopMethodStart oFiMsg;
};

struct fc_sxm_trMsgRapidChannelBrowseTimer
    : public fc_sxm_tclMessage
{
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio,
                    FC_SXM_AUDIO_ACTION_BASE_CHN_LIST + 22);
};

/*=============================================================================
=======                   PROTOTYPES & FUNCTIONS                        =======
=============================================================================*/

/*=============================================================================
=======                              METHODS                            =======
=============================================================================*/

/*=============================================================================
=======                              CLASSES                            =======
=============================================================================*/

/*****************************************************************************
 * CLASS        : fc_sxm_tclChannelList                                      *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION  : Class fc_sxm_tclChannelList to handle all channels related *
 *                information for sxm audio                                  *
 *****************************************************************************/
class Ifc_sxm_tclAudioApp;
struct fc_sxm_tcl_trAudioPropertyChannelInformation;

struct fc_sxm_trCategoryMonitorConfig
{
    CATEGORY_ID categoryID;
    CHANNEL_EVENT_MASK tEventMask;
    tBool bReportBrowsableChannels;
    tBool bEnableMonitoring;

    fc_sxm_trCategoryMonitorConfig(CATEGORY_ID catID = CATEGORY_INVALID_ID,
            CHANNEL_EVENT_MASK mask = CHANNEL_OBJECT_EVENT_NONE,
            tBool bReport = FALSE, tBool bEnable = FALSE)
    : categoryID(catID), tEventMask(mask), bReportBrowsableChannels(bReport), bEnableMonitoring(bEnable)
    { /* Default C-TOR */ }

    tVoid vReset(CATEGORY_ID catID = CATEGORY_INVALID_ID,
            CHANNEL_EVENT_MASK mask = CHANNEL_OBJECT_EVENT_NONE,
            tBool bReport = FALSE, tBool bEnable = FALSE) {
        // Method resets all values
        categoryID = catID;
        tEventMask = mask;
        bReportBrowsableChannels = bReport;
        bEnableMonitoring = bEnable;
    }

    tVoid vPrint() const;
};

class fc_sxm_tclChannelList
    : public fc_sxm_tclAudioMember
    , public Ifc_sxm_tclChannelList
    , public fc_sxm_tclSingleton<fc_sxm_tclChannelList>
{
    /*=========================================================================
    =======                     public section                          =======
    =========================================================================*/
    public:

        /*********************************************************************
         * DESCRIPTION : Class destructor                                    *
         * ----------------------------------------------------------------- *
         * PARAMETER   : None                                                *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        virtual ~fc_sxm_tclChannelList();

        /*********************************************************************
         * DESCRIPTION : Method to start channel list creation and initiate  *
         *               channel updates from SMS once decoder is in ready   *
         *               state                                               *
         * ----------------------------------------------------------------- *
         * PARAMETER   : hDecoderObject - Valid decoder object               *
         * PARAMETER   : enDecoderState - Current decoder state              *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        virtual tVoid vUpdateDecoderState(
            DECODER_OBJECT hDecoderObject,
            DECODER_STATE_ENUM enDecoderState);

        /*********************************************************************
         * DESCRIPTION : Update channel information in internal channel list *
         * ----------------------------------------------------------------- *
         * PARAMETER   : hChannelObject - Valid channel object               *
         * PARAMETER   : enEventMask    - Channel update event data          *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vUpdateChannelInfo(fc_sxm_trChannel const& rChannel,
                                 CHANNEL_EVENT_MASK enEventMask);
 
        /*********************************************************************
         * DESCRIPTION : Get current tuned channel ID                        *
         * ----------------------------------------------------------------- *
         * PARAMETER   : None                                                *
         * RETURNVALUE : CHANNEL_ID - Current tuned channel ID               *
         *********************************************************************/
        CHANNEL_ID u16GetCurrentTunedChannel(tVoid) const;

        /*********************************************************************
         * DESCRIPTION : Get SID of current tuned channel                    *
         * ----------------------------------------------------------------- *
         * PARAMETER   : None                                                *
         * RETURNVALUE : SERVICE_ID - Service ID of current tuned channel    *
         *********************************************************************/
        SERVICE_ID u16GetSIDOfTunedChannel(tVoid) const;
        SERVICE_ID u16GetSIDOfPreviewChannel(tVoid) const;

        /*********************************************************************
         * DESCRIPTION : Get details of preset channel and publish if tuned  *
         * ----------------------------------------------------------------- *
         * PARAMETER   : u16ChannelID  - Valid service ID for preset info    *
         * PARAMETER   : bChannelTuned - Flag to handle notifications        *
         * RETURNVALUE : midw_ext_fi_tcl_PresetListEntry - Preset info       *
         *********************************************************************/
        midw_ext_fi_tcl_PresetListEntry enGetPresetChannelInfo(
            CHANNEL_ID u16ChannelID, 
            tBool bChannelTuned, tTunable& tDisplayAdvisory);

        /* ================== CCA Request Handlers ========================= */

        /*********************************************************************
         * DESCRIPTION : CCA request to get channel list                     *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - Get channel list method start request       *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcess(fc_sxm_trMsgAudioMStartGetChnList const *prMsg);

        /*********************************************************************
         * DESCRIPTION : CCA request to select channel                       *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - Select channel method start request         *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcess(fc_sxm_trMsgAudioMStartSelChn const *prMsg);

        /*********************************************************************
         * DESCRIPTION : CCA request to select channel with delay            *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - Select channel with delay method start      *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcess(fc_sxm_trMsgAudioMStartSelChnWitDwellTime const *prMsg);

        /*********************************************************************
         * DESCRIPTION : CCA request to Disable/Enable Mature Channels       *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - DisableMature Channel method start request  *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcess(fc_sxm_trMsgAudioMStartDisableMatureChannels const *prMsg);

        /*********************************************************************
         * DESCRIPTION : CCA request to Skip Channels       *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - Skip channels method start request         *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcess(fc_sxm_trMsgAudioMStartSkipChannels const *prMsg);

        /*********************************************************************
         * DESCRIPTION : CCA request to get channel information list         *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - Skip channels method start request          *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcess(fc_sxm_trMsgAudioMStartGetChnInfo const *prMsg);

        /* ================ Timer utility functions ======================== */

        /*********************************************************************
         * DESCRIPTION : Process timeout of channel list update timer        *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - Channel list timer struct                   *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcessTimer(fc_sxm_trMsgChannelListTimer const *prMsg);

        /*********************************************************************
         * DESCRIPTION : Process timeout of channel select timer             *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - Channel select timer struct                 *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcessTimer(fc_sxm_trMsgChannelSelectDwellTimer const *prMsg);

        tVoid vProcessTimer(fc_sxm_trMsgRapidChannelBrowseTimer const *prMsg);

        tVoid vProcess(fc_sxm_trMsgAudioTestToneEvent const *prMsg);
        /*********************************************************************
         * DESCRIPTION : Process channel select timer time out               *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - Select channel with delay method request    *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vChannelSelectTimerNotifyTimeOut(
                  fc_sxm_trMsgAudioMStartSelChnWitDwellTime const *prMsg);

        /*********************************************************************
         * DESCRIPTION : Get channel ID of the last tuned channel from SMS   *
         *               and tune to that channel                            *
         * ----------------------------------------------------------------- *
         * PARAMETER   : None                                                *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vRestorePreviewChannel(tVoid);

        tVoid vNotifyChannelInformationUpdate(tFCSXM_SelectMode tSelectMode);

        fc_sxm_trChannel tGetChannelInfo(tVoid);

        /* Process channel object event callback */
        tVoid vProcess(fc_sxm_trMsgChannelEventCallback const *prMsg);

        tVoid vUpdateAllChannels(midw_ext_fi_tcl_ChannelList &rChannelList);

        /* ================== Public variables ========================= */
        
        /* Clear class member variables */
        tVoid vClearMemberData(tVoid);

        /* Update current tuned channel info */
        tVoid vUpdateCurrentTunedChannel(tVoid);

        /* Check if dwell timer is rumming */
        tBool bIsDwellTimerRunning(tVoid);

        /* Stop channel select dwell timer */
        tVoid vStopDwellTimer(tVoid);

        /* Set Audio App Pointer Reference*/
        tVoid vSetApp(Ifc_sxm_tclAudioApp* audioApp);

        /* Notify presets about the change */
        tVoid vUpdatePresetBands(fc_sxm_trChannel const &rChannel, CHANNEL_EVENT_MASK enEventMask) const;

        /* Notify presets about the change */
        tVoid vUpdateCategoryList() const;

        /* Get Channel Information for the given channel ID*/
        const fc_sxm_trChannel* trGetChannel(CHANNEL_ID channelID) const;

        /* Get channel's browse status*/
        tBool bGetChannelBrowseStatus(CHANNEL_ID) const;

        tVoid vProcess(fc_sxm_trMsgAudioMStartSetCategoryMonitor const *prMsg);
        tVoid vProcess(fc_sxm_trMsgAudioMStartRapidChannelBrowseStart const *prMsg);
        tVoid vProcess(fc_sxm_trMsgAudioMStartRapidChannelBrowseStop const *prMsg);

    /*=========================================================================
    =======                     private section                         =======
    =========================================================================*/
    private:

        /* ===================== Friend Classes ============================ */

        /* Defining singleton class as a friend class in channel list class */
        friend class fc_sxm_tclSingleton<fc_sxm_tclChannelList>;

        /* ==================== Member Functions =========================== */

        /*********************************************************************
         * DESCRIPTION : Class constructor                                   *
         * ----------------------------------------------------------------- *
         * PARAMETER   : None                                                *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        fc_sxm_tclChannelList();

        /*********************************************************************
         * DESCRIPTION : Add a new channel into internal channel list        *
         * ----------------------------------------------------------------- *
         * PARAMETER   : hChannelObject - Valid channel object               *
         * PARAMETER   : bGetAllInfo    - Flag to get all attributes or not  *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vAddNewChannel(fc_sxm_trChannel const& rChannel);

        /*********************************************************************
         * DESCRIPTION : Remove an existing channel from internal channel    *
         *               list                                                *
         * ----------------------------------------------------------------- *
         * PARAMETER   : hChannelObject - Valid channel object               *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vRemoveExistingChannel(fc_sxm_trChannel const& rChannel);

        /*********************************************************************
         * DESCRIPTION : Modify channel parameters from channel list         *
         * ----------------------------------------------------------------- *
         * PARAMETER   : hChannelObject   - Valid channel object             *
         * PARAMETER   : enChannelEvtMask - Channel update event data        *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vModifyChannelParams(fc_sxm_trChannel const& rChannel, 
                                   CHANNEL_EVENT_MASK enChannelEvtMask);

        /*********************************************************************
         * DESCRIPTION : Get channel ID from channel object                  *
         * ----------------------------------------------------------------- *
         * PARAMETER   : hChannelObject - Valid channel object               *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vUpdateChannelID(fc_sxm_trChannel const &rChannel);

        /*********************************************************************
         * DESCRIPTION : Notify channel list changes to all clients          *
         * ----------------------------------------------------------------- *
         * PARAMETER   : None                                                *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vNotifyChannelListUpdate(tVoid);

        /*********************************************************************
         * DESCRIPTION : Get channel ID from service ID from channel list    *
         * ----------------------------------------------------------------- *
         * PARAMETER   : u16ServiceID - Valid service ID to query            *
         * RETURNVALUE : CHANNEL_ID   - Valid queried channel ID             *
         *********************************************************************/
        CHANNEL_ID u16GetCidFromSidInChannelList(SERVICE_ID u16ServiceID);

        /*********************************************************************
         * DESCRIPTION : Get channel ID from channel index in channel list   *
         * ----------------------------------------------------------------- *
         * PARAMETER   : u16ListIndex - Valid list index to query            *
         * RETURNVALUE : CHANNEL_ID   - Valid queried channel ID             *
         *********************************************************************/
        CHANNEL_ID u16GetCidFromChannelListIndex(tU16 u16ListIndex);
        
        /*********************************************************************
         * DESCRIPTION : Get channel ID from relative index to current tuned *
         *               channel in channel list                             *
         * ----------------------------------------------------------------- *
         * PARAMETER   : s8ListRelativeIndex - Valid relative index to query *
         * RETURNVALUE : CHANNEL_ID          - Valid queried channel ID      *
         *********************************************************************/
        CHANNEL_ID u16GetCidFromChannelListRelativeIndex(
            tS8 s8ListRelativeIndex);
        
        /*********************************************************************
         * DESCRIPTION : Get channel ID from selection mode based on         *
         *               category type either from channel list or category  *
         *               list                                                *
         * ----------------------------------------------------------------- *
         * PARAMETER   : enSelectionMode - Channel select selection mode     *
         * PARAMETER   : u16CidOrSid     - Service ID or index               *
         * PARAMETER   : u16CategoryType  - Category type to query            *
         * PARAMETER   : u8Steps         - Relative index                    *
         * RETURNVALUE : CHANNEL_ID      - Valid queried channel ID          *
         *********************************************************************/
        CHANNEL_ID u16GetCidFromSelectionMode(
            midw_ext_fi_tcl_e8_SelectChannelMode enSelectionMode,
            tU16 u16CidOrSid,
            tU16 u16CategoryType,
            tU8 u8Steps);

        /* Check for current audio service subscription status */
        tVoid vCheckAudioSubscriptionStatus(tVoid);

        /* Check if channel info should be sent, even if identical to previous one */
        tBool bForceChannelInfoUpdate(tFCSXM_SelectMode enMode) const;

        CHANNEL_ID getCidForIndex(tU16 u16CategoryID, tU16 u16Id) const;
        CHANNEL_ID getCidForRelativeIndex(CATEGORY_ID u16CategoryID, CHANNEL_ID u16TunedChannelId, tS16 s16RelativeIndex) const;
        CHANNEL_ID getNextProgramCid(tU16 u16CategoryID, tS16 s16TunedChannelId) const;
        tVoid vNotifyTunedChannelPreset(SERVICE_ID serviceID, tBool bNeedsUpdate = FALSE) const;
        tU8 u8GetPresetsIndex(SERVICE_ID serviceID) const;
        tVoid vGetPresetIndices(map<SERVICE_ID, tU8>& ) const;
        tBool bIsReplayModeOn() const;
        tVoid vDisableAllMatureChannels(tBool bDisableAllMatureChn);
        tBool bSkipAllChannels();
        tBool bUnSkipAllChannels();
        tBool bSkipSelectedChannel(SERVICE_ID serviceID, CHANNEL_ID channelID);
        tBool bUnSkipSelectedChannel(SERVICE_ID serviceID, CHANNEL_ID channelID);

        tVoid vPrintChnInfoUpdate(const midw_ext_fi_tcl_ChannelInformation& oFiChn) const;
        tVoid vNotifySystemMsg(midw_ext_fi_tcl_e8_SystemMessage::tenType enCurrentAdvisory) const;
        tVoid vChannelInformationUpdate(fc_sxm_tcl_trAudioPropertyChannelInformation const& tChannelInfoProp, tFCSXM_SelectMode tSelectMode) const;

        tVoid vFillLowPrioAdvisoryMsg(CHANNEL_ID u16TuneChannelID, enAdvisoryMsgType enCheckAdvisory, midw_ext_fi_tcl_ChannelInformation &oFiChn, tFCSXM_SelectMode tSelectMode) const;
        tVoid vFillRadioIDAdvisoryMsg(fc_sxm_trChannel const &rChn, midw_ext_fi_tcl_ChannelInformation &oFiChn) const;
        tVoid vFillChannelAdvisoryMsg(fc_sxm_trChannel const &rChn, midw_ext_fi_tcl_ChannelInformation &oFiChn, tFCSXM_SelectMode tSelectMode) const;        
        tVoid vFillReplayModeElements(CHANNEL_ID channelID, midw_ext_fi_tcl_ChannelInformation &oFiChn) const;
        tVoid vUseChannel(CHANNEL_ID channelID) const;
        tVoid vApplyAdvisory(tFCSXM_SelectMode tSelectMode, CHANNEL_ID& u16TuneChannelID,
                enAdvisoryMsgType& enCheckAdvisory, midw_ext_fi_tcl_e8_SystemMessage::tenType& enCurrentAdvisory) const;
        tVoid vUpdateTunedChannelAdvisory(CHANNEL_ID channelID, const fc_sxm_trChannel& trChannel);
        tVoid vSetChannelBrowseStatus();

        tVoid vCheckSongInfoAvailibilityStatus(CHANNEL_ID hChannelID, tBool& bSaveArtistStatus, tBool& bSaveSongStatus) const;
        tBool bGetSaveSongInfoStatus(SEEK_AVAILABILTY_ENUM enSeekAvailablity) const;
        tVoid vNotifyCategoryChange(fc_sxm_trChannel const &rChannel, CHANNEL_EVENT_MASK enEventMask,tBool bOldSubscriptionState) const;
        tVoid vTriggerChannelListStatus(tBool bStatusChanged) const;

        tVoid vPrepareRapidChannelBrowse(CHANNEL_ID chnID, midw_ext_fi_tcl_e8_BrowseDirection enDirection,
                const midw_ext_fi_tcl_BrowseChannelConfig& configuration,
                const vector<CHANNEL_ID>& vectChannels);
        tVoid vStartRapidChannelBrowse();
        tVoid vNotifyRapidChannelBrowseInfo(CHANNEL_ID chnID);
        tVoid vFetchAllChannels(vector<CHANNEL_ID>& vectChnBrowse) const;
        tVoid vFetchCategoryChannels(vector<CHANNEL_ID>& vectChnBrowse, CATEGORY_ID tCategoryID) const;

        tVoid vStopRapidChannelBrowseTimer();
        tVoid vStartRapidChannelBrowseTimer(tU16 u16Duration);
        tTunable enTuneStatus(const fc_sxm_trChannel& trChannel, tBool bIsDirectTune) const;

        tVoid vHandleTunedChannelSkip();
        tVoid vHandleTunedChannelLock();
        tVoid vHandleSelectChannel(CHANNEL_ID &u16RequestedChannel, tU8 steps = FC_SXM_DEFAULT_CHANNEL_BROWSING_STEP);
        tVoid vGetSelectedChanId(CHANNEL_ID &u16RequestedChannel, tU8 steps, CHANNEL_ID chnId);
        tBool bAbortAdvancedAudioPlayback();
        tBool bLockChannel(fc_sxm_trChannel& trChannel);
        tBool bUnLockChannel(fc_sxm_trChannel& trChannel);
        tVoid vRetuneCurrentChannel();
        tVoid vSetMuteState (tBool bState);
        tBool bGetMuteStatus() const;

        /*Method to get channel lock state*/
        tBool bGetMatureChannelLockStatus() const { return _bEnableMatureChannels; }

        /* ==================== Member Variables =========================== */

        /* Utility variables */
        tBool _bChannelListReady;

        /* To handle channel list updates */
        tBool _bUpdateChannelList;
        tBool _bChannelListModified;
        tBool _bIsAudioServiceSubscribed;
        tBool _bChannelArtUpdate;
        tBool _bEnableMatureChannels;

        /* To handle current tuned channel */
        CHANNEL_ID   _u16TunedChannelChannelID;
        SERVICE_ID   _u16TunedChannelServiceID;
        CATEGORY_ID  _u16TunedChannelCategoryID;

        /* Store pending CCA requests */
        fc_sxm_trMsgAudioMStartSelChnWitDwellTime 
            _oPendingChnSelWitDelayRequest;

        /* Timer to handle channel select requests */
        fc_sxm_tclTimer<fc_sxm_tclAudioApp, 
                        fc_sxm_trMsgChannelSelectDwellTimer, 
                        fc_sxm_tclChannelList> _mChannelSelectDwellTimer;
        /* Timer to handle channel list updates */
        fc_sxm_tclTimer<fc_sxm_tclAudioApp, 
                        fc_sxm_trMsgChannelListTimer, 
                        fc_sxm_tclChannelList> _mChannelListUpdateTimer;

        /* Timer to notify if TWNow markets list has been received OTA */
        fc_sxm_tclTimer<fc_sxm_tclAudioApp,
                        fc_sxm_trMsgRapidChannelBrowseTimer,
                        fc_sxm_tclChannelList> _mRapidChannelBrowseTimer;

        map<CHANNEL_ID, fc_sxm_trChannel> _mChannelList;

        Ifc_sxm_tclAudioApp* _poAudioApp;
        fc_sxm_trCategoryMonitorConfig _trCategoryMonitorConfig;
        fc_sxm_tclRapidChannelBrowse _trRapidChnBrowse;

    /*=========================================================================
    =======                    protected section                        =======
    =========================================================================*/
    protected:

};

#endif

/*****************************************************************************
 * ----------------------------- END OF FILE ------------------------------- *
 *****************************************************************************/
