/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_channel.cpp
* @brief       Channel details extraction. 
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_audio_types.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_CHANNEL
#include "trcGenProj/Header/fc_sxm_tcl_channel.cpp.trc.h"
#endif

inline tBool bIsStringAvailable(string oStr) {
   return(oStr.size()!=0);
}  /* check if string is empty */

static void cb_vChannelArtAccessFunction(CHANNEL_ART_OBJECT /*hChannelArtObj*/,
                                         CHANNEL_ART_AVAILABLE_IMAGE_MASK tAvailableImages, 
                                         void* pvArg) 
{
   ETG_TRACE_USR4(("cb_vChannelArtAccessFunction()"));
   if(tAvailableImages & CHANNEL_ART_AVAILABLE_IMAGE_LOGO) {
      tBool *pbIsChannelArtAvailable = (tBool*)pvArg;
      *pbIsChannelArtAvailable = TRUE; 
   }
   if(tAvailableImages & CHANNEL_ART_AVAILABLE_IMAGE_ALBUM) {
   	  tBool *pbIsAlbumArtAvailable = (tBool*)pvArg;
   	  *pbIsAlbumArtAvailable = TRUE;
   }
}

fc_sxm_trChannel::fc_sxm_trChannel()
   : bIsChannelArtAvailable(FALSE)
   , bIsAlbumArtAvailable(FALSE)
   , u16ServiceId(SERVICE_INVALID_ID)
   , u16ChannelId(CHANNEL_INVALID_ID)
   , u16CategoryId(CATEGORY_INVALID_ID)
   , enSkip(enChnSkip_Default)
   , bIsChannelAvailable(FALSE)
   , bIsChannelFreeToAir(FALSE)
   , bIsChannelSubscribed(FALSE)
   , bIsPresetChannel(FALSE)
   , bIsChannelMature(FALSE)
   , bIsChannelLocked(FALSE)
   , bIsChannelSkipped(FALSE)
   , bIsChannelTuneMix(FALSE)
{
   oAudioChannelName.clear();
   oChannelCategoryName.clear();
   oChannelSongName.clear();
   oChannelArtistName.clear();
   vectCategories.clear();
}  /* Default constructor */

fc_sxm_trChannel::fc_sxm_trChannel(CHANNEL_OBJECT hChannelObject)
   : bIsChannelArtAvailable(FALSE)
   , u16ServiceId(SERVICE_INVALID_ID)
   , u16ChannelId(CHANNEL_INVALID_ID)
   , u16CategoryId(CATEGORY_INVALID_ID)
   , enSkip(enChnSkip_Default)
   , bIsChannelAvailable(FALSE)
   , bIsChannelFreeToAir(FALSE)
   , bIsChannelSubscribed(FALSE)
   , bIsPresetChannel(FALSE)
   , bIsChannelMature(FALSE)
   , bIsChannelLocked(FALSE)
   , bIsChannelSkipped(FALSE)
   , bIsChannelTuneMix(FALSE)
{
   vClearData();
   /* When a new channel is added, only channel Id, service Id and channel name 
      needs to be fetched all other information are updated as seperate events from SMS */
   vReadChannelId(hChannelObject);
   vReadServiceId(hChannelObject);
   vReadChannelName(hChannelObject);
   vReadArtistName(hChannelObject);
   vReadSongName(hChannelObject);
   vReadAttributeInfo(hChannelObject);
   vReadChannelCategories(hChannelObject);
   vReadPresetInfo(hChannelObject);

   bIsChannelAvailable=TRUE;
}  /* Copy constructor */

fc_sxm_trChannel::fc_sxm_trChannel(CHANNEL_OBJECT hChannel, CHANNEL_EVENT_MASK tEventMask)
   : bIsChannelArtAvailable(FALSE)
   , bIsAlbumArtAvailable(FALSE)
   , u16ServiceId(SERVICE_INVALID_ID)
   , u16ChannelId(CHANNEL_INVALID_ID)
   , u16CategoryId(CATEGORY_INVALID_ID)
   , enSkip(enChnSkip_Default)
   , bIsChannelAvailable(FALSE)
   , bIsChannelFreeToAir(FALSE)
   , bIsChannelSubscribed(FALSE)
   , bIsPresetChannel(FALSE)
   , bIsChannelMature(FALSE)
   , bIsChannelLocked(FALSE)
   , bIsChannelSkipped(FALSE)
   , bIsChannelTuneMix(FALSE)
{
   vClearData();
   vReadChannelId(hChannel);
   vReadServiceId(hChannel);
   tBool bDoTrace = etg_bIsTraceActive(TR_CLASS_FC_SXM_CHANNEL, 5);
   if (bDoTrace) {
       ETG_TRACE_USR1(("fc_sxm_trChannel::fc_sxm_trChannel cid=0x%08x sid=0x%08x mask=0x%08x", 
                       u16ChannelId, u16ServiceId, tEventMask));
   }
   
   if (CHANNEL_OBJECT_EVENT_REMOVED & tEventMask) {
      ETG_TRACE_USR1(("CHANNEL_OBJECT_EVENT_REMOVED"));
      return;
   }
   vReadPresetInfo(hChannel);
   
   bIsChannelAvailable=TRUE;

   if(CHANNEL_OBJECT_EVENT_CHANNEL_ID & tEventMask) {
      vReadAttributeInfo(hChannel);
      ETG_TRACE_USR1(("CHANNEL_OBJECT_EVENT_ATTRIBUTES: bIsChannelSubscribed = %d bIsChannelFreeToAir = %d", bIsChannelSubscribed, bIsChannelFreeToAir));
      vReadChannelName(hChannel);
      ETG_TRACE_USR1(("CHANNEL_OBJECT_EVENT_NAME: %s", oAudioChannelName.c_str()));
      vReadCategoryInfo(hChannel);
      ETG_TRACE_USR1(("CHANNEL_OBJECT_EVENT_CATEGORY: u16CategoryId = %u oChannelCategoryName = %s", u16CategoryId, oChannelCategoryName.c_str()));
      vReadSongName(hChannel);
      ETG_TRACE_USR1(("CHANNEL_OBJECT_EVENT_TITLE: %s", oChannelSongName.c_str()));
      vReadArtistName(hChannel);
      ETG_TRACE_USR1(("CHANNEL_OBJECT_EVENT_ARTIST: %s", oChannelArtistName.c_str()));
      vReadChannelArtAvail(hChannel);
      ETG_TRACE_USR1(("CHANNEL_OBJECT_EVENT_ART: bIsChannelArtAvailable = %d", bIsChannelArtAvailable));
      vReadChannelCategories(hChannel);
   }

   if(CHANNEL_OBJECT_EVENT_ATTRIBUTES & tEventMask) {
      vReadAttributeInfo(hChannel);
      ETG_TRACE_USR1(("CHANNEL_OBJECT_EVENT_ATTRIBUTES: bIsChannelSubscribed = %d bIsChannelFreeToAir = %d", bIsChannelSubscribed, bIsChannelFreeToAir));
   }
   if(CHANNEL_OBJECT_EVENT_NAME  & tEventMask) {
      vReadChannelName(hChannel);
      ETG_TRACE_USR1(("CHANNEL_OBJECT_EVENT_NAME: %s", oAudioChannelName.c_str()));
   }
   if(CHANNEL_OBJECT_EVENT_CATEGORY & tEventMask) {
      vReadCategoryInfo(hChannel);
      ETG_TRACE_USR1(("CHANNEL_OBJECT_EVENT_CATEGORY: u16CategoryId = %u oChannelCategoryName = %s", u16CategoryId, oChannelCategoryName.c_str()));
      vReadChannelCategories(hChannel);
   }
   if(CHANNEL_OBJECT_EVENT_TITLE & tEventMask) {
      vReadSongName(hChannel);
      ETG_TRACE_USR1(("CHANNEL_OBJECT_EVENT_TITLE: %s", oChannelSongName.c_str()));
   }
   if(CHANNEL_OBJECT_EVENT_ARTIST & tEventMask) {
      vReadArtistName(hChannel);
      ETG_TRACE_USR1(("CHANNEL_OBJECT_EVENT_ARTIST: %s", oChannelArtistName.c_str()));
   }
   if(CHANNEL_OBJECT_EVENT_ART & tEventMask) {
      vReadChannelArtAvail(hChannel);
      ETG_TRACE_USR1(("CHANNEL_OBJECT_EVENT_ART: bIsChannelArtAvailable = %d", bIsChannelArtAvailable));
   }
}

fc_sxm_trChannel::~fc_sxm_trChannel()
{
   vClearData();
}  /* Destructor */

tBool fc_sxm_trChannel::bIsValid() const
{
   return((CHANNEL_INVALID_ID != u16ChannelId) && (SERVICE_INVALID_ID != u16ServiceId));
}  /* Validate channel */

/* Check if channel is valid and tunable */
tTunable fc_sxm_trChannel::eIsTunable() const
{
	// A channel is tunable even if its locked or skipped
   tTunable eChannelTunable = e_UNAVAILABLE;
   if(FALSE == bIsChannelAvailable)
   {
      eChannelTunable = e_UNAVAILABLE;
   }
   else if((FALSE == bIsChannelSubscribed) && (FALSE == bIsChannelFreeToAir))
   {
      eChannelTunable = e_UNAUTHORIZED;
   }
   else
   {
      eChannelTunable = e_TUNABLE;
   }

   ETG_TRACE_USR2(("ChannelId         - %u", u16ChannelId));
   ETG_TRACE_USR2(("ChannelAvailable  - %u", bIsChannelAvailable));
   ETG_TRACE_USR2(("ChannelFreeToAir  - %u", bIsChannelFreeToAir));
   ETG_TRACE_USR2(("ChannelSubscribed - %u", bIsChannelSubscribed));
   ETG_TRACE_USR2(("ChannelLocked (%u) : ChannelSkipped(%u)", bIsChannelLocked, bIsChannelSkipped));
   ETG_TRACE_USR2(("eChannelTunable - %u", eChannelTunable));

   return eChannelTunable;
}

// Method returns channel's browse status
tTunable fc_sxm_trChannel::enBrowseStatus() const
{
	tTunable enAdvisory = e_UNAVAILABLE;
	if(FALSE == bIsChannelAvailable)
	{
		enAdvisory = e_UNAVAILABLE;
	}
	else if((FALSE == bIsChannelSubscribed) && (FALSE == bIsChannelFreeToAir))
	{
		enAdvisory = e_UNAUTHORIZED;
	}
	else if ((TRUE == bIsChannelLocked) || ( enChnSkip_Set == enSkip))
	{
		enAdvisory = e_LOCKED;
	}
	else if (TRUE == bIsChannelSkipped)
	{
		enAdvisory = e_SKIPPED;
	}
	else
	{
		enAdvisory = e_TUNABLE;
	}
	ETG_TRACE_USR2(("ChannelId(%u) \t ChannelAvailable(%u) \t Appropriate Advisory(%u)",
			u16ChannelId, bIsChannelAvailable, ETG_CENUM(tTunable, enAdvisory)));
	ETG_TRACE_USR2(("ChannelFreeToAir(%u) : ChannelSubscribed(%u) : ChannelLocked (%u) : ChannelSkipped(%u)",
			bIsChannelFreeToAir, bIsChannelSubscribed, bIsChannelLocked, bIsChannelSkipped));
	return enAdvisory;
}

tBool fc_sxm_trChannel::bIsSubscribedChannel() const
{
   return ((bIsChannelSubscribed) && (!bIsChannelFreeToAir));
}  /* Check if channel is subscribed channel and not a free to air channel */

tBool fc_sxm_trChannel::bIsChannelNameAvailable(tVoid) const
{
   return (bIsStringAvailable(oAudioChannelName));
}  /* Get channel name availability */

tBool fc_sxm_trChannel::bIsCategoryNameAvailable(tVoid) const
{
   return (bIsStringAvailable(oChannelCategoryName));
}  /* Get category name availaility */

tBool fc_sxm_trChannel::bIsSongNameAvailable(tVoid) const
{
   return (bIsStringAvailable(oChannelSongName));
}  /* Get song name availability */

tBool fc_sxm_trChannel::bIsArtistNameAvailable(tVoid) const
{
   return (bIsStringAvailable(oChannelArtistName));
}  /* Get artist name availability */

tVoid fc_sxm_trChannel::vChannelRemoved(tVoid)
{
   bIsChannelAvailable=FALSE;
}  /* Set channel unavailable */

tVoid fc_sxm_trChannel::vReadChannelId(CHANNEL_OBJECT hChannelObject)
{
   u16ChannelId = CHANNEL.tChannelId(hChannelObject);
}

tVoid fc_sxm_trChannel::vReadServiceId(CHANNEL_OBJECT hChannelObject)
{
   u16ServiceId = CHANNEL.tServiceId(hChannelObject);
}  /* Get service ID */


tVoid fc_sxm_trChannel::vReadAttributeInfo(CHANNEL_OBJECT hChannelObject)
{
   bIsChannelSubscribed = FALSE;
   bIsChannelFreeToAir  = FALSE;
   bIsChannelMature = FALSE;
   bIsChannelLocked = FALSE;
   bIsChannelSkipped = FALSE;

   (tVoid) CHANNEL.eIsSubscribed(hChannelObject, &bIsChannelSubscribed);
   (tVoid) CHANNEL.eIsFreeToAir(hChannelObject, &bIsChannelFreeToAir);
   (tVoid) CHANNEL.eIsMature(hChannelObject, &bIsChannelMature);
   (tVoid) CHANNEL.eIsLocked(hChannelObject, &bIsChannelLocked);
   (tVoid) CHANNEL.eIsSkipped(hChannelObject, &bIsChannelSkipped) ;

   bIsChannelTuneMix = CHANNEL.bQualifiedForTuneMix(hChannelObject);

   ETG_TRACE_USR4(("DATA Received from SMS FreeToAir - %d", bIsChannelSubscribed));
}

tVoid fc_sxm_trChannel::vReadChannelName(CHANNEL_OBJECT hChannelObject)
{
   oAudioChannelName.clear();
   fc_sxm_vCopySmsString2Stl(CHANNEL.hLongName(hChannelObject), oAudioChannelName);
   if( (FALSE == fc_sxm_pValidateString(oAudioChannelName.c_str())) || (oAudioChannelName.empty()) ) {
      if(CHANNEL_INVALID_ID != u16ChannelId) {
         tChar cDefaultChannelName[SXM_AUDIO_DISP_TEXT_LENGTH] = { '\0' };
         sprintf(cDefaultChannelName, "Channel %d", u16ChannelId);
         oAudioChannelName = cDefaultChannelName;
      } else {
	     oAudioChannelName.clear();
	  }
   }
}

tVoid fc_sxm_trChannel::vReadCategoryInfo(CHANNEL_OBJECT hChannelObject)
{
   u16CategoryId = CATEGORY_INVALID_ID;
   oChannelCategoryName.clear();
   CATEGORY_OBJECT hCategory = CHANNEL.hCategory(hChannelObject, 0);
   u16CategoryId = CATEGORY.tGetCategoryId(hCategory);
   fc_sxm_vCopySmsString2Stl(CATEGORY.hLongName(hCategory), oChannelCategoryName);
   if( (FALSE == fc_sxm_pValidateString(oChannelCategoryName.c_str())) || (oChannelCategoryName.empty()) ) {
      if( (FC_SXM_AUDIO_CATEGORY_TYPE_MIN <= u16CategoryId) && 
          (FC_SXM_AUDIO_CATEGORY_TYPE_MAX >= u16CategoryId) )     {
         tChar cChannelCategoryName[SXM_AUDIO_DISP_TEXT_LENGTH] = { '\0' };
         sprintf(cChannelCategoryName, "Category %d", u16CategoryId);
         oChannelCategoryName = cChannelCategoryName;
      } else {
         oChannelCategoryName.clear();
      }
   }
} 

tVoid fc_sxm_trChannel::vReadSongName(CHANNEL_OBJECT hChannelObject)
{
   oChannelSongName.clear();
   CD_OBJECT hCdo = CHANNEL.hCDO(hChannelObject);
   if(CD_INVALID_OBJECT != hCdo) {
      fc_sxm_vCopySmsString2Stl(CDO.hTitle(hCdo), oChannelSongName);
   }
}

tVoid fc_sxm_trChannel::vReadArtistName(CHANNEL_OBJECT hChannelObject)
{
   oChannelArtistName.clear();
   fc_sxm_vCopySmsString2Stl(CDO.hArtist(CHANNEL.hCDO(hChannelObject)), oChannelArtistName);
}

tVoid fc_sxm_trChannel::vReadChannelArtAvail(CHANNEL_OBJECT hChannelObject)
{
   CHANNEL_ART_OBJECT hChannelArtObj = CHANNEL.hArt(hChannelObject);
   if (CHANNEL_ART_INVALID_OBJECT != hChannelArtObj) {
	   if(SMSAPI_RETURN_CODE_SUCCESS != CHANNEL_ART.eUseArt(hChannelArtObj, cb_vChannelArtAccessFunction, &bIsChannelArtAvailable)) {
		   bIsChannelArtAvailable = FALSE;
		   ETG_TRACE_ERR(("CHANNEL_ART.eUseArt(cb_vChannelArtAccessFunction) failed"));
	   }
   }
   else {
	   ETG_TRACE_ERR(("Failed to get channel art object from channel object"));
   }
   CD_OBJECT hCDO = CHANNEL.hCDO(hChannelObject);
   if(CD_INVALID_OBJECT == hCDO){
   		ETG_TRACE_ERR(("Failed to get channel art object from channel object"));
   		bIsAlbumArtAvailable = FALSE;
   		return;
   	}
   CHANNEL_ART_OBJECT hAlbumArtObj = CDO.hArt(hCDO);
   if (CHANNEL_ART_INVALID_OBJECT != hAlbumArtObj) {
	   ETG_TRACE_USR4(("vReadChannelArtAvail - %d", bIsAlbumArtAvailable));
   	   if(SMSAPI_RETURN_CODE_SUCCESS != CHANNEL_ART.eUseArt(hAlbumArtObj, cb_vChannelArtAccessFunction, &bIsAlbumArtAvailable)) {
   		   ETG_TRACE_ERR(("CHANNEL_ART.eUseArt(cb_vChannelArtAccessFunction) failed"));
   	   }
   }
   else {
   	   ETG_TRACE_ERR(("Failed to get channel art object from channel object"));
   }
}

/* Read all the categories that the channel belongs to*/
tVoid fc_sxm_trChannel::vReadChannelCategories(CHANNEL_OBJECT hChannelObject)
{
	vectCategories.clear();
	N16 n16NumCats = 0;
	// Get the number of categories that this channel object belongs to
	if (SMSAPI_RETURN_CODE_SUCCESS == CHANNEL.eNumCategories(hChannelObject, &n16NumCats))  {
		// Push all the category information to vectCategories
		for (N16 n16Index = 0; n16Index < n16NumCats; n16Index++) {
			CATEGORY_OBJECT hCategory = CHANNEL.hCategory(hChannelObject, n16Index);
			if ( CATEGORY_INVALID_OBJECT == hCategory)  continue;
			fc_sxm_trChannelCategory category;
			category.u16CatId = CATEGORY.tGetCategoryId(hCategory);
			fc_sxm_vCopySmsString2Stl(CATEGORY.hLongName(hCategory), category.oCatName);
			category.enCatType = CATEGORY.eType(hCategory);
			vectCategories.push_back(category);
		}
	}
}

/* Method reads if the reported channel is a preset channel or not*/
tVoid fc_sxm_trChannel::vReadPresetInfo(CHANNEL_OBJECT hChannelObject)
{
	PRESET_BAND_OBJECT hBand = PRESET_BAND_INVALID_OBJECT;
	STRING_OBJECT hPresetName = STRING_INVALID_OBJECT;
	size_t tPresetIndex = 0;
	SMSAPI_RETURN_CODE_ENUM eReturnCode =  CHANNEL.ePreset(
			 hChannelObject,
			 &hBand,
			 &hPresetName,
			 &tPresetIndex );
	// If the channel belongs to a Valid preset band associated with the decoder object,
	// then its a preset channel
	bIsPresetChannel = static_cast <tBool>
	((SMSAPI_RETURN_CODE_SUCCESS == eReturnCode) && (PRESET_BAND_INVALID_OBJECT != hBand)) ;
}

tBool fc_sxm_trChannel::bBelongsToCategory(const CATEGORY_ID catID) const
{
	vector < fc_sxm_trChannelCategory >::const_iterator categoryCIT = vectCategories.begin();
	tBool bCategoryMatch = FALSE;
	for (; categoryCIT != vectCategories.end(); ++categoryCIT)
	{
		if (categoryCIT->u16CatId == catID)
		{
			bCategoryMatch = TRUE;
			break;
		}
	}
	return bCategoryMatch;
}

tVoid fc_sxm_trChannel::vClearData(tVoid)
{
   bIsChannelAvailable = FALSE;
   bIsChannelFreeToAir = FALSE;
   bIsChannelSubscribed = FALSE;
   bIsPresetChannel = FALSE;
   bIsChannelArtAvailable = FALSE;
   bIsAlbumArtAvailable = FALSE;
   bIsChannelMature = FALSE;
   bIsChannelLocked = FALSE;
   bIsChannelSkipped = FALSE;
   bIsChannelTuneMix = FALSE;

   u16ServiceId = SERVICE_INVALID_ID;
   u16ChannelId = CHANNEL_INVALID_ID;
   u16CategoryId = CATEGORY_INVALID_ID;
   oAudioChannelName.clear();
   oChannelCategoryName.clear();
   oChannelSongName.clear();
   oChannelArtistName.clear();
   vectCategories.clear();
}  /* Clear all member variables */



fc_sxm_trCategory::fc_sxm_trCategory()
   : _u16CategoryId(CATEGORY_INVALID_ID),
     _enCatType(CATEGORY_TYPE_INVALID)
{
   _oCategoryName.clear();          /* Clear category name */
   mCategoryChannelList.clear();    /* Clear category list */
}  /* Default constructor */

fc_sxm_trCategory::fc_sxm_trCategory(CATEGORY_ID u16CategoryID)
   : _u16CategoryId(u16CategoryID),
     _enCatType(CATEGORY_TYPE_INVALID)
{
   _oCategoryName.clear();          /* Clear category name */
   mCategoryChannelList.clear();    /* Clear channel list */
}  /* Copy constructor */

fc_sxm_trCategory::fc_sxm_trCategory(CATEGORY_OBJECT hCategory, CATEGORY_EVENT_MASK tEventMask)
   : _u16CategoryId(CATEGORY_INVALID_ID),
     _enCatType(CATEGORY_TYPE_INVALID)
{
   vClearData();
   vReadCategoryId(hCategory);
   tBool bDoTrace=etg_bIsTraceActive(TR_CLASS_FC_SXM_CHANNEL, 5);
   if (bDoTrace) {
       ETG_TRACE_USR1(("fc_sxm_trCategory::fc_sxm_trChannel id=0x%08x  mask=0x%08x", 
                       _u16CategoryId, tEventMask));

   }

   if(CATEGORY_INVALID_ID == _u16CategoryId) {
       ETG_TRACE_ERR(("Failed to get category id from category object"));
       return;
   }

    if (CATEGORY_OBJECT_EVENT_REMOVED & tEventMask) {
        ETG_TRACE_USR1(("CATEGORY_OBJECT_EVENT_REMOVED"));
        return;
    }

    if (CATEGORY_OBJECT_EVENT_NAME  & tEventMask) {
        vReadCategoryName(hCategory);
        if (bDoTrace) {
            ETG_TRACE_USR1(("CATEGORY_OBJECT_EVENT_NAME: %s", _oCategoryName.c_str()));
        }
    }

    vReadCategoryType(hCategory);
}

fc_sxm_trCategory::~fc_sxm_trCategory()
{
   vClearData();
}  /* Destructor */

tBool fc_sxm_trCategory::bIsCategoryNameAvailable(tVoid) const
{
   return (bIsStringAvailable(_oCategoryName));
}  /* Get category name availaility */

/* Check if category contains tunable channels */
tBool fc_sxm_trCategory::bIsCategoryAvailable(tVoid) const
{
   tBool bValidCategory = FALSE;
   map<CHANNEL_ID, fc_sxm_trChannel>::const_iterator channelIterator;
   for (channelIterator = mCategoryChannelList.begin();
       channelIterator != mCategoryChannelList.end();
       ++channelIterator)
   {
      if (TRUE == channelIterator->second.bIsBrowseable())
      {
         bValidCategory = TRUE;
         break;
      }
   }
   return bValidCategory;
}

tVoid fc_sxm_trCategory::vClearData(tVoid)
{
   _u16CategoryId = CATEGORY_INVALID_ID;
   _oCategoryName.clear();
   _enCatType = CATEGORY_TYPE_INVALID;
}  /* Clear all member variables */

tVoid fc_sxm_trCategory::vReadCategoryId(CATEGORY_OBJECT hCategory) {
    _u16CategoryId = CATEGORY.tGetCategoryId(hCategory);
}

tVoid fc_sxm_trCategory::vReadCategoryType(CATEGORY_OBJECT hCategory) {
	_enCatType = CATEGORY.eType(hCategory);
}

tVoid fc_sxm_trCategory::vReadCategoryName(CATEGORY_OBJECT hCategoryObject)
{
    fc_sxm_vCopySmsString2Stl(CATEGORY.hLongName(hCategoryObject), _oCategoryName);
    if( (FALSE == fc_sxm_pValidateString(_oCategoryName.c_str())) || (_oCategoryName.empty()) )
    {
      if( (FC_SXM_AUDIO_CATEGORY_TYPE_MIN <= /* OLD_CODE u16CategoryId */ _u16CategoryId) && 
          (FC_SXM_AUDIO_CATEGORY_TYPE_MAX >= /* OLD_CODE u16CategoryId */ _u16CategoryId) )     
      {
         tChar cName[SXM_AUDIO_DISP_TEXT_LENGTH] = { '\0' };
         sprintf(cName, "Category %d", _u16CategoryId);
         _oCategoryName = cName; 
      } else {
        _oCategoryName.clear();
      }
    } 
}

tVoid fc_sxm_trCategory::vGetCategoryName(CATEGORY_OBJECT hCategoryObject, string &roChannelCategoryName)
{
   fc_sxm_vCopySmsString2Stl(CATEGORY.hLongName(hCategoryObject), roChannelCategoryName);
   ETG_TRACE_USR4(("DATA Received from SMS Category Name - %s", roChannelCategoryName.c_str()));
}  /* Get category name */

