/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_category_list.h
* @brief       Category List Handler declaration.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

/*=============================================================================
=======                            DEFINES                              =======
=============================================================================*/

#ifndef FC_SXM_TCL_CATEGORY_LIST_H
#define FC_SXM_TCL_CATEGORY_LIST_H

/*=============================================================================
=======                            INCLUDES                             =======
=============================================================================*/

#include "fc_sxm_tcl_audio_member.h"
#include "fc_sxm_tcl_category_list_if.h"


/*=============================================================================
=======               CONSTANTS & MACROS FOR GENERAL PURPOSE            =======
=============================================================================*/

#define FC_SXM_AUDIO_CATEGORY_TYPE_ALL                     0x00
#define FC_SXM_AUDIO_CATEGORY_TYPE_BROADCAST_START         0x01  /* Min cat value from SMS */
#define FC_SXM_AUDIO_CATEGORY_TYPE_BROADCAST_END           0x3F  /* Max 63 cat from SMS */
#define FC_SXM_AUDIO_CATEGORY_TYPE_USER_VIRTUAL_START      0x100 /* Min 256 cat value from SMS */
#define FC_SXM_AUDIO_CATEGORY_TYPE_USER_VIRTUAL_END        0x13F /* Max 256 + 63 cat from SMS */
#define FC_SXM_AUDIO_CATEGORY_TYPE_CURRENT                 0xFF
/* Defining Max Number of Featured Favorite Band */
#define FC_SXM_AUDIO_FEATURED_FAV_NUM_BANDS                32

#define USER_CATEGORY_ID_MIN     FC_SXM_AUDIO_CATEGORY_TYPE_BROADCAST_START       /* Min UserID 1 */
#define USER_CATEGORY_ID_MAX     (FC_SXM_AUDIO_FEATURED_FAV_NUM_BANDS + FC_SXM_AUDIO_CATEGORY_TYPE_BROADCAST_END)  /* Max UserID: Max featured fav bands + 63 */

#define FC_SXM_AUDIO_CATEGORY_LIST_UPDATE_PERIOD  1000


/*=============================================================================
=======                             TYPES                               =======
=============================================================================*/

/* MethodStart for get category list from CCA client */
struct fc_sxm_trMsgAudioMStartGetCatList
: public fc_sxm_tclMessage
  {
    /* Service and service ID */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, 
            FC_SXM_AUDIO_ACTION_BASE_CAT_LIST + 1);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgGetCategoryListMethodStart oFiMsg;
  };
struct fc_sxm_trMsgCategoryListTimer: public fc_sxm_tclMessage
{
    /* Service and service ID */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio,
            FC_SXM_AUDIO_ACTION_BASE_CAT_LIST + 2);
};
/* Message for category objects events */
struct fc_sxm_trMsgCategoryEventCallback: public fc_sxm_tclMessage
{
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_CAT_LIST + 15);
    fc_sxm_trMsgCategoryEventCallback(CATEGORY_OBJECT hCategory_ = CATEGORY_INVALID_OBJECT, CATEGORY_EVENT_MASK tEventMask_ = CATEGORY_TYPE_BROADCAST):
        rCategory(hCategory_, tEventMask_),
        tEventMask(tEventMask_)
    { }

    fc_sxm_trCategory rCategory;
    CATEGORY_EVENT_MASK tEventMask;
};

/*=============================================================================
=======                   PROTOTYPES & FUNCTIONS                        =======
=============================================================================*/

/* Including audio app class */
class Ifc_sxm_tclAudioApp;

/*=============================================================================
=======                              METHODS                            =======
=============================================================================*/

/*=============================================================================
=======                              CLASSES                            =======
=============================================================================*/

/*****************************************************************************
 * CLASS        : fc_sxm_tclCategoryList                                     *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION  : Class fc_sxm_tclCategoryList to handle all category        *
 *                related information in sxm audio channels                  *
 *****************************************************************************/
class fc_sxm_tclCategoryList
: public fc_sxm_tclAudioMember
  , public Ifc_sxm_tclCategoryList
  , public fc_sxm_tclSingleton<fc_sxm_tclCategoryList>
{
    /*=========================================================================
    =======                     public section                          =======
    =========================================================================*/
public:

    /*********************************************************************
     * DESCRIPTION : Class destructor                                    *
     * ----------------------------------------------------------------- *
     * PARAMETER   : None                                                *
     * RETURNVALUE : None                                                *
     *********************************************************************/
    virtual ~fc_sxm_tclCategoryList();

    /*********************************************************************
     * DESCRIPTION : Method to start category list creation and initiate *
     *               category updates from SMS once decoder is in ready  *
     *               state                                               *
     * ----------------------------------------------------------------- *
     * PARAMETER   : hDecoderObject - Valid decoder object               *
     * PARAMETER   : enDecoderState - Current decoder state              *
     * RETURNVALUE : None                                                *
     *********************************************************************/
    virtual tVoid vUpdateDecoderState(
            DECODER_OBJECT hDecoderObject,
            DECODER_STATE_ENUM enDecoderState);

    /*********************************************************************
     * DESCRIPTION : Method Update category information in internal list *
     * ----------------------------------------------------------------- *
     * PARAMETER   : hDecoderObject - Valid decoder object from SMS      *
     * PARAMETER   : enEventMask    - Update info of category object     *
     * RETURNVALUE : None                                                *
     *********************************************************************/
    tVoid vUpdateCategoryInfo(fc_sxm_trCategory const &rCategory,
            CATEGORY_EVENT_MASK enEventMask);

    /*********************************************************************
     * DESCRIPTION : Update channel info in category list                *
     * ----------------------------------------------------------------- *
     * PARAMETER   : hCategory       - Valid category object             *
     * PARAMETER   : s16CurrentIndex - Channel index being iterated      *
     * PARAMETER   : hChannel        - Valid channel object              *
     * RETURNVALUE : N16             - Steps to iterate further          *
     *********************************************************************/
    N16 s16updateChannelInfo(CATEGORY_OBJECT hCategory,
            CATEGORY_CHANNEL_INDEX s16CurrentIndex,
            CHANNEL_OBJECT hChannel);

    /*********************************************************************
     * DESCRIPTION : Get category ID from category list index based on   *
     *               category type                                       *
     * ----------------------------------------------------------------- *
     * PARAMETER   : u16CategoryType - Category type to query             *
     * PARAMETER   : u16ListIndex   - Index in category list             *
     * RETURNVALUE : CHANNEL_ID    - Channel ID of list element queried  *
     *********************************************************************/
    CHANNEL_ID u16GetCidFromCategoryListIndex(tU16 u16CategoryType,
            tU16 u16ListIndex);

    /*********************************************************************
     * DESCRIPTION : Get channel ID from category list index based on    *
     *               relative index. This should also wrap to next list  *
     *               element if the index received is the last element   *
     *               in the list                                         *
     * ----------------------------------------------------------------- *
     * PARAMETER   : hCategoryType      - Category type to query       *
     * PARAMETER   : hChannelID         - Channel ID to query          *
     * PARAMETER   : lRelativeListIndex - Relative index               *
     * RETURNVALUE : CHANNEL_ID           - Channel ID of list element   *
     *********************************************************************/
    CHANNEL_ID u16GetCidFromCategoryListRelativeIndex(
            CATEGORY_ID hCategoryType,
            CHANNEL_ID hChannelID,
            tS16 lRelativeListIndex);

    /*********************************************************************
     * DESCRIPTION : Get channel ID from category list based on current  *
     *               program type                                        *
     * ----------------------------------------------------------------- *
     * PARAMETER   : u16CategoryType  - Category type to query            *
     * PARAMETER   : s16RelativeIndex - Relative index in category list  *
     * RETURNVALUE : CHANNEL_ID     - Channel ID of list element queried *
     *********************************************************************/
    CHANNEL_ID u16GetCidOfNextProgramFromCategoryList(
            tU16 u16CategoryType,
            tS16 s16RelativeIndex);

    /* ================== CCA Request Handlers ========================= */

    /*********************************************************************
     * DESCRIPTION : CCA request to get category list                    *
     * ----------------------------------------------------------------- *
     * PARAMETER   : prMsg - Get category list method start request      *
     * RETURNVALUE : None                                                *
     *********************************************************************/
    tVoid vProcess(fc_sxm_trMsgAudioMStartGetCatList const *prMsg);

    /* Process category object event callback */
    tVoid vProcess(fc_sxm_trMsgCategoryEventCallback const *prMsg);

    /* Clear class member variables */
    tVoid vClearMemberData(tVoid);

    /* Set Audio App Pointer Reference*/
    tVoid vSetApp(Ifc_sxm_tclAudioApp* audioApp);

    tVoid vCategoryListUpdate(tVoid);
    tVoid vProcessTimer(fc_sxm_trMsgCategoryListTimer const *prMsg);
    tVoid vStartCategoryListTimer(tVoid);
    tVoid vNotifyCategoryListContentUpdate(tVoid);
    // Get channels belonging to a category
    tVoid vFetchCategoryChannels(vector<CHANNEL_ID>& vectChnBrowse, CATEGORY_ID tCategoryID) const;

    /*=========================================================================
    =======                     private section                         =======
    =========================================================================*/
private:

    /* ===================== Friend Classes ============================ */

    /* Defining singleton class as a friend class in category list class */
    friend class fc_sxm_tclSingleton<fc_sxm_tclCategoryList>;

    /* ==================== Member Functions =========================== */

    /*********************************************************************
     * DESCRIPTION : Class constructor                                   *
     * ----------------------------------------------------------------- *
     * PARAMETER   : None                                                *
     * RETURNVALUE : None                                                *
     *********************************************************************/
    fc_sxm_tclCategoryList();

    /*********************************************************************
     * DESCRIPTION : Add new category to internal category list          *
     * ----------------------------------------------------------------- *
     * PARAMETER   : hCategoryObject - Valid category object from SMS    *
     * RETURNVALUE : None                                                *
     *********************************************************************/
    tVoid vAddNewCategory(fc_sxm_trCategory const &rCategory);

    /*********************************************************************
     * DESCRIPTION : Remove existing category from internal category     *
     *               list                                                *
     * ----------------------------------------------------------------- *
     * PARAMETER   : hCategoryObject - Valid category object from SMS    *
     * RETURNVALUE : None                                                *
     *********************************************************************/
    tVoid vRemoveExistingCategory(fc_sxm_trCategory const &rCategory);

    /*********************************************************************
     * DESCRIPTION : Modify category params from category list           *
     * ------------------------------------------------------------------*
     * PARAMETER   : hCategoryObject  - Valid category object from SMS   *
     * PARAMETER   : u32CategoryEvtMask - Category update event mask     *
     * RETURNVALUE : None                                                *
     *********************************************************************/
    tVoid vModifyCategoryParams(fc_sxm_trCategory const &roCatData,
            CATEGORY_EVENT_MASK hCategoryEvtMask);

    /* Notify category list update */
    tVoid vNotifyCategoryListUpdate(tBool bIsDecoderStateRequired = TRUE);

    /* Update channel list for each category */
    tVoid vUpdateCategoryChannels(tVoid);

    tU8 u8GetPresetsIndex(SERVICE_ID serviceID) const;

    /* Copy channel object contents into midw_ext_fi_tcl_ChannelListEntry FI structure*/
    tVoid vCopyChannelEntry(midw_ext_fi_tcl_ChannelListEntry& trChannelEntry,
            const fc_sxm_trChannel& channel) const ;

    /* method to check for valid category type range */
    tBool bIsValidRangeCategoryId(CATEGORY_ID nCatId) const;
    /* method to convert broadcast/User categoryId to UserId */
    USER_ID u16ConvertBroadcastCategoryIdtoUserId(CATEGORY_ID rCatId) const;
    USER_ID u16ConvertUserCategoryIdtoUserId(CATEGORY_ID rCatId);

    /* Method returns the channel browse status of the given channel ID*/
    tBool bChannelBrowseStatus(CHANNEL_ID tChannelID) const;

    tVoid vRemoveExistingUserId(CATEGORY_ID nCatId);
    /* ==================== Member Variables =========================== */

    /* Utility variables */
    tBool _bCategoryNotificationsReg;

    /* To handle category list updates */
    tBool _bUpdateCategoryList;

    tBool _bListUpdate;

    /* To hold current selected category type */
    CATEGORY_ID _u16CurrentSelectedCategoryID;
    CATEGORY_ID _u16MySeekCatId;
    /* Current decoder state */
    DECODER_STATE_ENUM _enCurrentDecoderState;

    map<CATEGORY_ID, fc_sxm_trCategory> _mCategoryList;

    map<USER_ID, CATEGORY_ID> _mCategoryUserIdList;

#ifdef FC_SXM_ENABLE_FEATURED_FAVORITE_SORTING
    tVoid vAddFeaturedFavoriteToSort(string sCategoryName, CATEGORY_ID u16CategoryId);
    tVoid vRemoveFeaturedFavorite(CATEGORY_ID nCatId);
    tVoid vSortFeaturedFavorite();
    map<string, CATEGORY_ID> _mFeaturedFavCategoryList;
#endif

    Ifc_sxm_tclAudioApp* _poAudioApp;

    tU32 _u32FeaturedFavCount;


    /* Timer to handle category list updates */
    fc_sxm_tclTimer<fc_sxm_tclAudioApp,
    fc_sxm_trMsgCategoryListTimer,
    fc_sxm_tclCategoryList> _mCategoryListUpdateTimer;

    /*=========================================================================
    =======                    protected section                        =======
    =========================================================================*/
protected:

};

#endif

/*****************************************************************************
 * ----------------------------- END OF FILE ------------------------------- *
 *****************************************************************************/
