/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_advisories.h
* @brief       Class declaration for handling advisories and declarations of internal 
*              messages intended for advisories implementation.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/


/*=============================================================================
=======                            DEFINES                              =======
=============================================================================*/

#ifndef FC_SXM_TCL_ADVISORIES_H
#define FC_SXM_TCL_ADVISORIES_H

/*=============================================================================
=======                            INCLUDES                             =======
=============================================================================*/

#include "fc_sxm_audio_types.h"
#include "fc_sxm_tcl_audio_member.h"
#include "fc_sxm_tcl_timer.h"
#include "fc_sxm_tcl_channel_list_if.h"

/*=============================================================================
=======               CONSTANTS & MACROS FOR GENERAL PURPOSE            =======
=============================================================================*/

/* Defining default audio channel */
#define FC_SXM_AUDIO_RADIO_ID_CHANNEL  00 
#define FC_SXM_AUDIO_DEFAULT_CHANNEL   01 

/*=============================================================================
=======                             TYPES                               =======
=============================================================================*/

/*
 * messages send from cca-context from timer class to audio app 
 * for advisory message timeout updates.
 */
struct fc_sxm_trMsgAdvisoryTimer
    : public fc_sxm_tclMessage
{
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, 
                    FC_SXM_AUDIO_ACTION_BASE_ADVISORY + 2);
};

/*
 * messages send from cca-context from timer class to audio app 
 * for loading advisory message timeout updates.
 */
struct fc_sxm_trMsgLoadingAdvisoryTimer
    : public fc_sxm_tclMessage
{
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, 
                    FC_SXM_AUDIO_ACTION_BASE_ADVISORY + 3);
};

/*
 * Message sent from SMS context containing current tuned channel info
 */
struct fc_sxm_trMsgAudioSmsEvtChannelEvt
    : public fc_sxm_tclMessage 
{
   fc_sxm_trMsgAudioSmsEvtChannelEvt() { }
   SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, 
                    FC_SXM_AUDIO_ACTION_BASE_ADVISORY + 4);
};

/*
 * Message sent from SMS context containing current tuned channel info
 */
struct fc_sxm_trMsgAudioSmsSubscriptionUpdatedEvt
	: public fc_sxm_tclMessage
{
	fc_sxm_trMsgAudioSmsSubscriptionUpdatedEvt() { }
	SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio,
					FC_SXM_AUDIO_ACTION_BASE_ADVISORY + 5);
};

/*=============================================================================
=======                   PROTOTYPES & FUNCTIONS                        =======
=============================================================================*/

/*=============================================================================
=======                              METHODS                            =======
=============================================================================*/

/*=============================================================================
=======                     FORWARD DECLARATIONS                        =======
=============================================================================*/
struct fc_sxm_trMsgAudioSmsEvtAntennaQuality;
struct fc_sxm_trMsgAudioSmsEvtSignalState;
struct fc_sxm_trMsgAudioSmsEvtTuneState;
class fc_sxm_tclAudioApp;
class Ifc_sxm_tclAudioApp;

/*=============================================================================
=======                              CLASSES                            =======
=============================================================================*/

/*****************************************************************************
 * CLASS        : fc_sxm_tclAdvisories                                       *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION  : Class to handle all advisory message related services in   *
 *                fc sxm audio component                                     *
 *****************************************************************************/
class fc_sxm_tclAdvisories
    : public fc_sxm_tclAudioMember
    , public fc_sxm_tclSingleton<fc_sxm_tclAdvisories>
{
    /*=========================================================================
    =======                     public section                          =======
    =========================================================================*/
    public:

        /*********************************************************************
         * DESCRIPTION : Class destructor                                    *
         * ----------------------------------------------------------------- *
         * PARAMETER   : None                                                *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        virtual ~fc_sxm_tclAdvisories();

        /*********************************************************************
         * DESCRIPTION : Method that will start advisory service once        *
         *               decoder is up and is in ready state                 *
         * ----------------------------------------------------------------- *
         * PARAMETER   : hDecoderObject : Valid decoder object from SXM      *
         * PARAMETER   : enDecoderState : Current decoder state              *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        virtual tVoid vUpdateDecoderState(
            DECODER_OBJECT hDecoderObject, 
            DECODER_STATE_ENUM enDecoderState);

        /*********************************************************************
         * DESCRIPTION : Method that will chack for antenna status and       *
         *               update the same to all registered clients           *
         * ----------------------------------------------------------------- *
         * PARAMETER   : hAntennaState - Antenna state enum value            *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vUpdateAntennaState(ANTENNA_STATE_ENUM hAntennaState);
        
        /*********************************************************************
         * DESCRIPTION : Antenna quality event handler                       *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - Antenna quality structure                   *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcess(fc_sxm_trMsgAudioSmsEvtAntennaQuality const *prMsg);

        /*********************************************************************
         * DESCRIPTION : Method that will chack for signal quality and       *
         *               update the same to all registered clients           *
         * ----------------------------------------------------------------- *
         * PARAMETER   : lSignalQuality - Signal quality data                *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vUpdateSignalQuality(const SIGNAL_QUALITY_STRUCT &lSignalQuality);
        
        /*********************************************************************
         * DESCRIPTION : Signal state event handler                          *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - Signal quality data                         *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcess(fc_sxm_trMsgAudioSmsEvtSignalState const *prMsg);

        /*********************************************************************
         * DESCRIPTION : Tune state event handler                            *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - Tune state data                             *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcess(fc_sxm_trMsgAudioSmsEvtTuneState const *prMsg);

        tVoid vProcess(fc_sxm_trMsgAudioSmsEvtChannelEvt const *prMsg);

        tVoid vProcess(fc_sxm_trMsgAudioSmsSubscriptionUpdatedEvt const *prMsg);

        tVoid vSetLoadingAdvisory(tVoid);

        tVoid vSetSXMInitializingAdvisory();
        tVoid vClearSXMInitializingAdvisory();
        /*********************************************************************
         * DESCRIPTION : Method that will update hardware status to all the  *
         *               registered clients                                  *
         * ----------------------------------------------------------------- *
         * PARAMETER   : hState - Module state enum                          *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vUpdateHardwareStatus(MODULE_STATE_ENUM hState);

        /*********************************************************************
         * DESCRIPTION : Method that will update channel status to all the   *
         *               registered clients                                  *
         * ----------------------------------------------------------------- *
         * PARAMETER   : hChannelState - Channel status info                 *
         * PARAMETER   : bLoadDefaultChannel - Flag to tune presious channel *
         *               or default channel                                  *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vUpdateChannelStatus(
            midw_ext_fi_tcl_e8_SystemMessage::tenType hChannelState,
            tBool bLoadDefaultChannel);

        /*********************************************************************
         * DESCRIPTION : Advisory timer time out handler                     *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - Time out event data                         *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcessTimer(fc_sxm_trMsgAdvisoryTimer const *prMsg);

        tVoid vProcessTimer(fc_sxm_trMsgLoadingAdvisoryTimer const *prMsg);
        
        /*********************************************************************
         * DESCRIPTION : Start Advisory timer                                *
         * ----------------------------------------------------------------- *
         * PARAMETER   : u16DelayTime - Delay time in ms                     *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vStartAdvisoryTimer(tU16 u16DelayTime);
        
        /*********************************************************************
         * DESCRIPTION : Stop Advisory timer                                 *
         * ----------------------------------------------------------------- *
         * PARAMETER   : None                                                *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vStopAdvisoryTimer(tVoid);
        
        /*********************************************************************
         * DESCRIPTION : Check weither advisory timer  is running or not     *
         * ----------------------------------------------------------------- *
         * PARAMETER   : None                                                *
         * RETURNVALUE : TRUE if timmer is running else FALSE                *
         *********************************************************************/
        tBool bIsAdvisoryTimerRunning(tVoid) const;

        /*********************************************************************
         * DESCRIPTION : Get current advisory type                           *
         * ----------------------------------------------------------------- *
         * PARAMETER   : None                                                *
         * RETURNVALUE : Current advisory set in advisory class              *
         *********************************************************************/
        midw_ext_fi_tcl_e8_SystemMessage::tenType 
            eGetCurrentAdvisoryType(tBool bOnlyChannelAdvisory, enAdvisoryMsgType &enCheckAdvisory) const;

        /* Clear class member variables */
        tVoid vClearMemberData(tVoid);

        /* Stop advisory timer and clear channel advisories */
        tVoid vStopAndClearChannelAdvisory(tVoid);

        /* Set Audio App Pointer Reference*/
		tVoid vSetApp(Ifc_sxm_tclAudioApp* audioApp);

		tBool bIsHighPriorityAdvisory() const;

    /*=========================================================================
    =======                     private section                         =======
    =========================================================================*/
    private:

        /* ===================== Friend Classes ============================ */

        /* Defining singleton class as a friend class in advisories class */
        friend class fc_sxm_tclSingleton<fc_sxm_tclAdvisories>;

        /* ==================== Member Functions =========================== */

        /*********************************************************************
         * DESCRIPTION : Class constructor                                   *
         * ----------------------------------------------------------------- *
         * PARAMETER   : None                                                *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        fc_sxm_tclAdvisories();

        // Method requests ChannelList object to restore playing preview channel
        tVoid vRestorePreviewChannel();

        // Method notifies ChannelList to update channel information
        tVoid vUpdateChannelInfo(tFCSXM_SelectMode enMode);

        enAdvisoryMsgType enGetAdvisoryType() const;
        tBool bIsChannelAdvisory() const;

        /* ==================== Member Variables =========================== */

        /* Current decoder state */
        DECODER_STATE_ENUM _enCurrentDecoderState;

        /* To handle system message updates */
        tU32 _mSystemMessageUpdates;

        /* To load default channel or previously tuned channel */
        tBool _mLoadDefaultChannel;

        /* To handle previous signal state */
        SIGNAL_QUALITY_ENUM _ePreviousSignalQuality;
        SIGNAL_STATE_ENUM   _ePreviousSignalState;

        /* Timer to handle advisory updates */
        fc_sxm_tclTimer<fc_sxm_tclAudioApp, 
                        fc_sxm_trMsgAdvisoryTimer, 
                        fc_sxm_tclAdvisories> _mAdvisoryTimer;

        /* Timer to handle loading advisory updates */
        fc_sxm_tclTimer<fc_sxm_tclAudioApp, 
                        fc_sxm_trMsgLoadingAdvisoryTimer, 
                        fc_sxm_tclAdvisories> _mLoadingAdvisoryTimer;

        Ifc_sxm_tclAudioApp* _poAudioApp;

    /*=========================================================================
    =======                    protected section                        =======
    =========================================================================*/
    protected:

};

#endif

/*****************************************************************************
 * ----------------------------- END OF FILE ------------------------------- *
 *****************************************************************************/
