/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_advisories.cpp
* @brief       Implementation for advisories class. 
*              File controls all advisory notifications being triggered from audio service.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

/*=============================================================================
=======                            DEFINES                              =======
=============================================================================*/

/*=============================================================================
=======                            INCLUDES                             =======
=============================================================================*/

#include "fc_sxm_types.h"
#include "fc_sxm_tcl_advisories.h"
#include "fc_sxm_tcl_audio_app.h"
#include "fc_sxm_tcl_audio_properties.h"
#include "fc_sxm_tcl_audio_app_if.h"
#include "fc_sxm_generic_utils.h"
#include "fc_sxm_tcl_audio_utils.h"

/*=============================================================================
=======               CONSTANTS & MACROS FOR GENERAL PURPOSE            =======
=============================================================================*/

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_ADVISORIES
#include "trcGenProj/Header/fc_sxm_tcl_advisories.cpp.trc.h"
#endif

/*=============================================================================
=======                             TYPES                               =======
=============================================================================*/

/*=============================================================================
=======                   PROTOTYPES & FUNCTIONS                        =======
=============================================================================*/

/*=============================================================================
=======                              METHODS                            =======
=============================================================================*/

/*****************************************************************************
 * FUNCTION    : fc_sxm_tclAdvisories                                        *
 * CLASS       : fc_sxm_tclAdvisories                                        *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Class constructor                                           *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/

fc_sxm_tclAdvisories::fc_sxm_tclAdvisories()
    : _enCurrentDecoderState(DECODER_STATE_INVALID)
    , _mSystemMessageUpdates(0U)
    , _mLoadDefaultChannel(FALSE)
    , _ePreviousSignalQuality(SIGNAL_QUALITY_INVALID)
    , _ePreviousSignalState(SIGNAL_STATE_UNKNOWN)
    , _poAudioApp(OSAL_NULL)
{
    ETG_TRACE_USR4(("fc_sxm_tclAdvisories constructor"));
}

/*****************************************************************************
 * FUNCTION    : ~fc_sxm_tclAdvisories                                       *
 * CLASS       : fc_sxm_tclAdvisories                                        *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Class destructor                                            *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
fc_sxm_tclAdvisories::~fc_sxm_tclAdvisories()
{
    ETG_TRACE_USR4(("fc_sxm_tclAdvisories destructor"));

    /* Set default values for member variables */
    _mSystemMessageUpdates = 0x00;
    _mLoadDefaultChannel = FALSE;
    _poAudioApp = OSAL_NULL;
}

/*****************************************************************************
 * FUNCTION    : vUpdateDecoderState                                         *
 * CLASS       : fc_sxm_tclAdvisories                                        *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method that will start advisory service once decoder is up  *
 *               and is in ready state                                       *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hDecoderObject - Valid decoder object from SXM              *
 * PARAMETER   : enDecoderState - Current decoder state                      *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclAdvisories::vUpdateDecoderState(DECODER_OBJECT /*hDecoderObject*/, DECODER_STATE_ENUM enDecoderState)
{
   ETG_TRACE_USR4(("fc_sxm_tclAdvisories::vUpdateDecoderState()  \t _mSystemMessageUpdates = %u", _mSystemMessageUpdates));

   /* Set current decoder state to ready */
   _enCurrentDecoderState = enDecoderState;
   DECODER_OBJECT hDecoder = (_poAudioApp) ?
		   _poAudioApp->oGetDecoderObject() : DECODER_INVALID_OBJECT;

   /* Register for signal quality and antenna status updates */
   if((DECODER_STATE_READY == enDecoderState) && (DECODER_INVALID_OBJECT != hDecoder)) {
      /* Update antenna state */
      vUpdateAntennaState(DECODER.eAntennaState(hDecoder, 1));

      /* Get signal quality */
      SIGNAL_QUALITY_STRUCT lSignalQuality;
      if(SMSAPI_RETURN_CODE_SUCCESS == DECODER.eSignalQuality(hDecoder, &lSignalQuality)) {
         /* Update signal quality */
         vUpdateSignalQuality(lSignalQuality);
      } else {
         ETG_TRACE_ERR(("Failed to get signal quality"));
      }
   }
}

/* Clear class member variables */
tVoid fc_sxm_tclAdvisories::vClearMemberData(tVoid)
{
   ETG_TRACE_USR4(("fc_sxm_tclAdvisories::vClearMemberData()  \t _mSystemMessageUpdates = %u", _mSystemMessageUpdates));

   /* Clear member variables */
   _mSystemMessageUpdates = 0x00;
   _mLoadDefaultChannel = FALSE;
}

/* Stop advisory timer and clear channel advisories */
tVoid fc_sxm_tclAdvisories::vStopAndClearChannelAdvisory(tVoid)
{
   ETG_TRACE_USR4(("fc_sxm_tclAdvisories::vStopAndClearChannelAdvisory()  \t _mSystemMessageUpdates = %u", _mSystemMessageUpdates));
   if(bIsAdvisoryTimerRunning())
   {
      vStopAdvisoryTimer();
      // Reset unauthorised system message
      fc_sxm_tclGenericUtils::vClearBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAUTHORIZED));
      // Reset channel unavailable system message
      fc_sxm_tclGenericUtils::vClearBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE));
      // Reset category unavailable system message
      fc_sxm_tclGenericUtils::vClearBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CATEGORY_NOT_FOUND));
	  // Reset channel mature system message
	  fc_sxm_tclGenericUtils::vClearBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_MATURE));
	  // Reset channel skip system message
	  fc_sxm_tclGenericUtils::vClearBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_LOCKED));
   }
}

/*****************************************************************************
 * FUNCTION    : vProcess                                                    *
 * CLASS       : fc_sxm_tclAdvisories                                        *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method that will chack for antenna status and update the    *
 *               same to all registered clients                              *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : prMsg - Decoder antenna state                               *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclAdvisories::vProcess(fc_sxm_trMsgAudioSmsEvtAntennaQuality const *prMsg) 
{
    ETG_TRACE_USR4(("fc_sxm_tclAdvisories::vProcess(fc_sxm_trMsgAudioSmsEvtAntennaQuality)"));
    vUpdateAntennaState(prMsg->eAntennaState);
}

/*****************************************************************************
 * FUNCTION    : vUpdateAntennaState                                         *
 * CLASS       : fc_sxm_tclAdvisories                                        *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method that will chack for antenna status and update the    *
 *               same to all registered clients                              *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hAntennaState - Antenna state enum value                    *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclAdvisories::vUpdateAntennaState(ANTENNA_STATE_ENUM hAntennaState)
{
   ETG_TRACE_USR4(("fc_sxm_tclAdvisories::vUpdateAntennaState()"));

   /* Data services antenna status property */
   fc_sxm_tcl_trDSAntennaStatus DSAntennaStatus;
   DSAntennaStatus.oFiRes.IsAntennaConnected = FALSE;  /* Set default value */

   switch(hAntennaState)
   {
      case ANTENNA_STATE_NOT_DETECTED:
      case ANTENNA_STATE_SHORTED: {
         ETG_TRACE_USR3(("Antenna state not connected"));
         /* Set antenna disconnected system message */
         fc_sxm_tclGenericUtils::vSetBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_ANTENNA_DISCONNECTED));
      } break;
      case ANTENNA_STATE_DETECTED:
      case ANTENNA_STATE_UNKNOWN:
      case ANTENNA_STATE_UNSUPPORTED: {
         ETG_TRACE_USR3(("Antenna state connected"));
         /* Reset antenna disconnected system message */
         fc_sxm_tclGenericUtils::vClearBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_ANTENNA_DISCONNECTED));
         DSAntennaStatus.oFiRes.IsAntennaConnected = TRUE;  /* Set data services antenna state as connected */
      } break;

      default:
        break;
   }

   Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
                		  _poAudioApp->poGetChannelListInstance(): NULL;
   SXM_ASSERT_RETURN (NULL != poChannelList)

   if(!bIsAdvisoryTimerRunning() && !poChannelList->bIsDwellTimerRunning() && (DECODER_STATE_READY == _enCurrentDecoderState)) {
	   poChannelList->vNotifyChannelInformationUpdate(e_ADVISORY_UPDATE);
   }

   ETG_TRACE_USR4(("DS Antenna State - %u  \t _mSystemMessageUpdates = %u", DSAntennaStatus.oFiRes.IsAntennaConnected, _mSystemMessageUpdates));
   /* Notify HMI regarding new antenna state */
   fc_sxm_tclAudioProperties::instance()->oDSAntennaStatus.vSet(DSAntennaStatus);
   fc_sxm_tclAudioProperties::instance()->oDSAntennaStatus.vNotify();
}

/*****************************************************************************
 * FUNCTION    : vProcess                                                    *
 * CLASS       : fc_sxm_tclAdvisories                                        *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Signal state event handler                                  *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : prMsg - Signal quality data                                 *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclAdvisories::vProcess(fc_sxm_trMsgAudioSmsEvtSignalState const *prMsg) 
{
    ETG_TRACE_USR4(("fc_sxm_tclAdvisories::vProcess(fc_sxm_trMsgAudioSmsEvtSignalState)"));
    vUpdateSignalQuality(prMsg->eSignalQuality);
}

/*****************************************************************************
 * FUNCTION    : vUpdateSignalQuality                                        *
 * CLASS       : fc_sxm_tclAdvisories                                        *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method that will check for signal quality and               *
 *               update the same to all registered clients                   *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : lSignalQuality - Signal quality data                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclAdvisories::vUpdateSignalQuality(const SIGNAL_QUALITY_STRUCT &lSignalQuality)
{
    //ETG_TRACE_USR4(("fc_sxm_tclAdvisories::vUpdateSignalQuality() eComposite(%u) \t eState(%u)",
         //ETG_CENUM(SIGNAL_QUALITY_ENUM, lSignalQuality.eComposite),
         //ETG_CENUM(SIGNAL_STATE_ENUM, lSignalQuality.eState)));

    /* New Code */
    if((SIGNAL_QUALITY_NONE != lSignalQuality.eComposite) && (SIGNAL_STATE_ACQUIRED == lSignalQuality.eState))
    {
        ETG_TRACE_USR3(("Signal acquired by decoder"));
        fc_sxm_tclGenericUtils::vClearBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_NO_SIGNAL));
        if(/*!poChannelList->bIsDwellTimerRunning()*/!bIsAdvisoryTimerRunning() &&
           ((lSignalQuality.eComposite != _ePreviousSignalQuality) || 
	         (lSignalQuality.eState != _ePreviousSignalState)))
        {
        	vUpdateChannelInfo(e_ADVISORY_UPDATE);
        }
    }
    else if((SIGNAL_QUALITY_NONE == lSignalQuality.eComposite)&&
             ((SIGNAL_STATE_UNKNOWN == lSignalQuality.eState) || (SIGNAL_STATE_ACQUIRING == lSignalQuality.eState)))
    {
        ETG_TRACE_USR3(("Signal not acquired by decoder"));
        fc_sxm_tclGenericUtils::vSetBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_NO_SIGNAL));

	    if( /*(*/!bIsAdvisoryTimerRunning() /*&& !poChannelList->bIsDwellTimerRunning())*/ && (DECODER_STATE_READY == _enCurrentDecoderState) &&
	        ((lSignalQuality.eComposite != _ePreviousSignalQuality) || 
	         (lSignalQuality.eState != _ePreviousSignalState)) )
	    {
	    	vUpdateChannelInfo(e_ADVISORY_UPDATE);
	    }
    }

    _ePreviousSignalQuality = lSignalQuality.eComposite;
    _ePreviousSignalState   = lSignalQuality.eState;

    ETG_TRACE_USR2(("_ePreviousSignalQuality - %d \t _ePreviousSignalState - %d \t _mSystemMessageUpdates - %u"
    		, _ePreviousSignalQuality, _ePreviousSignalState, _mSystemMessageUpdates));
}

/*****************************************************************************
 * FUNCTION    : vProcess                                                    *
 * CLASS       : fc_sxm_tclAdvisories                                        *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Tune state event handler                                    *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : prMsg - Tune state data                                     *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclAdvisories::vProcess(fc_sxm_trMsgAudioSmsEvtTuneState const *prMsg) 
{
   ETG_TRACE_USR4(("fc_sxm_tclAdvisories::vProcess(fc_sxm_trMsgAudioSmsEvtTuneState)"));

   if (OSAL_NULL != prMsg) 
   {
      /* Clear loading advisory in any case */
	   fc_sxm_tclGenericUtils::vClearBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_LOADING_XM));

      /* Get current advisory */
      enAdvisoryMsgType eCurrentAdvisory(enGetAdvisoryType());

      Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
                   		  _poAudioApp->poGetChannelListInstance(): NULL;
      SXM_ASSERT_RETURN(NULL != poChannelList)

      switch(prMsg->eTuneState) {
         case TUNE_STATE_TUNING_IN_PROGRESS: {
            ETG_TRACE_USR2(("TUNE_STATE_TUNING_IN_PROGRESS"));
            if((enHIGH_PRIO_ADVISORY != eCurrentAdvisory) && (!bIsAdvisoryTimerRunning() && !poChannelList->bIsDwellTimerRunning())) {
            	poChannelList->vNotifyChannelInformationUpdate(e_TUNING_CHANNEL);
            }
            // Display loading advisory after 1000 ms if selected channel is not tuned within 1000 ms
            _mLoadingAdvisoryTimer.vStart(FC_SXM_AUDIO_DEFAULT_LOADING_ADVISORY_WAIT_PERIOD);
         } break;
         case TUNE_STATE_TUNED: {
            ETG_TRACE_USR2(("TUNE_STATE_TUNED"));
            poChannelList->vUpdateCurrentTunedChannel();
            if(!bIsAdvisoryTimerRunning() && !poChannelList->bIsDwellTimerRunning()) {
            	poChannelList->vNotifyChannelInformationUpdate(e_TUNED_CHANNEL);
            }
            _mLoadingAdvisoryTimer.vStop();
         } break;
         case TUNE_STATE_UNKNOWN:
         case TUNE_STATE_INVALID:
         default: {
            ETG_TRACE_USR2(("default"));
         } break;
      }
   }
}

tVoid fc_sxm_tclAdvisories::vProcess(fc_sxm_trMsgAudioSmsEvtChannelEvt const * /* prMsg */ )
{
   ETG_TRACE_USR4(("fc_sxm_tclAdvisories::vProcess(fc_sxm_trMsgAudioSmsEvtChannelEvt)"));
   Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
                		  _poAudioApp->poGetChannelListInstance(): NULL;
   SXM_ASSERT_RETURN(NULL != poChannelList)
   poChannelList->vUpdateCurrentTunedChannel();
   if(!bIsAdvisoryTimerRunning() && !poChannelList->bIsDwellTimerRunning()) {
	   poChannelList->vNotifyChannelInformationUpdate(e_TUNED_CHANNEL);
   }
}

// Method handles decoder subscription updated event
tVoid fc_sxm_tclAdvisories::vProcess(fc_sxm_trMsgAudioSmsSubscriptionUpdatedEvt const * /* prMsg */ )
{
	ETG_TRACE_USR4(("fc_sxm_tclAdvisories::vProcess(fc_sxm_trMsgAudioSmsSubscriptionUpdatedEvt)"));
	// Set Subscription Updated Bit
	fc_sxm_tclGenericUtils::vSetBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_SUBSCRIPTION_UPDATED));
	vUpdateChannelInfo(e_SUBSCRIPTION_UPDATED);
	// Subscription Updated advisory notification is sent to HMI by now, so clearing Subscription Updated bit immediately
	fc_sxm_tclGenericUtils::vClearBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_SUBSCRIPTION_UPDATED));
}

tVoid fc_sxm_tclAdvisories::vSetLoadingAdvisory(tVoid)
{
	// This is a redundant method and has been replaced by method vSetSXMInitializingAdvisory()
   fc_sxm_tclGenericUtils::vSetBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_LOADING_XM));
   ETG_TRACE_USR4(("fc_sxm_tclAdvisories::vSetLoadingAdvisory() \t _mSystemMessageUpdates = %u", _mSystemMessageUpdates));
   vUpdateChannelInfo(e_FACTORY_RESET);
}

tVoid fc_sxm_tclAdvisories::vSetSXMInitializingAdvisory()
{
   fc_sxm_tclGenericUtils::vSetBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_SXM_INITIALIZING));
   ETG_TRACE_USR4(("fc_sxm_tclAdvisories::vSetSXMInitializingAdvisory() \t _mSystemMessageUpdates = %u", _mSystemMessageUpdates));
   vUpdateChannelInfo(e_FACTORY_RESET);
}

tVoid fc_sxm_tclAdvisories::vClearSXMInitializingAdvisory()
{
   fc_sxm_tclGenericUtils::vClearBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_SXM_INITIALIZING));
   ETG_TRACE_USR4(("fc_sxm_tclAdvisories::vClearSXMInitializingAdvisory() \t _mSystemMessageUpdates = %u", _mSystemMessageUpdates));
}

/*****************************************************************************
 * FUNCTION    : vUpdateHardwareStatus                                       *
 * CLASS       : fc_sxm_tclAdvisories                                        *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method that will update hardware status to all the          *
 *               registered clients                                          *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hState - Module state enum                                  *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclAdvisories::vUpdateHardwareStatus(MODULE_STATE_ENUM hState)
{
    switch(hState) {
        case MODULE_STATE_ERROR: {
            /* Set hardware error system message */
            fc_sxm_tclGenericUtils::vSetBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_HARDWARE_ERROR));
        } break;
        case MODULE_STATE_READY:
        default: {
            /* Reset hardware error system message */
        	fc_sxm_tclGenericUtils::vClearBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_HARDWARE_ERROR));
            break;
        }
    }
    ETG_TRACE_USR4(("fc_sxm_tclAdvisories::vUpdateHardwareStatus() state - %d \t _mSystemMessageUpdates = %u", hState, _mSystemMessageUpdates));
    vUpdateChannelInfo(e_ADVISORY_UPDATE);
}

/*****************************************************************************
 * FUNCTION    : vUpdateChannelStatus                                        *
 * CLASS       : fc_sxm_tclAdvisories                                        *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method that will update channel status to all the           *
 *               registered clients                                          *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hChannelState - Channel status info                         *
 * PARAMETER   : bLoadDefaultChannel - Flag to tune presious channel         *
 *               or default channel                                          *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclAdvisories::vUpdateChannelStatus(midw_ext_fi_tcl_e8_SystemMessage::tenType hChannelState, tBool bLoadDefaultChannel)
{
    ETG_TRACE_USR4(("fc_sxm_tclAdvisories::vUpdateChannelStatus() hChannelState = %u \t bLoadDefaultChannel = %u", hChannelState, bLoadDefaultChannel));
    switch(hChannelState) {
        case midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAUTHORIZED: {
            // Save default channel flag
            _mLoadDefaultChannel = bLoadDefaultChannel;

            //Set channel unauthorized message
            fc_sxm_tclGenericUtils::vSetBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAUTHORIZED));
        } break;
        case midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE: {
            // Set to true if previous channel is to be loaded on timer expiry
            _mLoadDefaultChannel = bLoadDefaultChannel;

            // Set channel unavailable message
            fc_sxm_tclGenericUtils::vSetBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE));
        } break;
        case midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CATEGORY_NOT_FOUND: {
            // Set category not found message
        	fc_sxm_tclGenericUtils::vSetBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CATEGORY_NOT_FOUND));
        } break;
        case midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_MATURE: {
			// Set channel mature message
			fc_sxm_tclGenericUtils::vSetBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_MATURE));
		} break;
        case midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_LOCKED: {
			// Set channel locked message
			fc_sxm_tclGenericUtils::vSetBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_LOCKED));
		} break;
        default:
            break;
    }
    vStartAdvisoryTimer(FC_SXM_AUDIO_DEFAULT_ADVISORY_UPDATE_PERIOD);
}

/*****************************************************************************
 * FUNCTION    : vStartAdvisoryTimer                                         *
 * CLASS       : fc_sxm_tclAdvisories                                        *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Start Advisory timer                                        *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclAdvisories::vStartAdvisoryTimer(tU16 u16DelayTime) 
{
    ETG_TRACE_USR4(("vStartAdvisoryTimer() u16DelayTime = %u", u16DelayTime));
    _mAdvisoryTimer.vStart(u16DelayTime);
}

/*****************************************************************************
 * FUNCTION    : vStopAdvisoryTimer                                          *
 * CLASS       : fc_sxm_tclAdvisories                                        *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Stop Advisory timer                                         *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclAdvisories::vStopAdvisoryTimer(tVoid)
{
    ETG_TRACE_USR4(("vStopAdvisoryTimer()"));
    _mAdvisoryTimer.vStop();
}

/*****************************************************************************
 * FUNCTION    : bIsAdvisoryTimerRunning                                     *
 * CLASS       : fc_sxm_tclAdvisories                                        *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Check weither advisory timer  is running or not             *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : TRUE if timmer is running else FALSE                        *
 *****************************************************************************/
tBool fc_sxm_tclAdvisories::bIsAdvisoryTimerRunning(tVoid) const
{
	tBool bIsTimerActive = _mAdvisoryTimer.bIsRunning();
    ETG_TRACE_USR4(("bIsAdvisoryTimerRunning() bIsTimerActive = %u", bIsTimerActive));
    return bIsTimerActive;
}

/*****************************************************************************
 * FUNCTION    : eGetCurrentAdvisoryType                                     *
 * CLASS       : fc_sxm_tclAdvisories                                        *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Get current advisory type                                   *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : Current advisory set in advisory class                      *
 *****************************************************************************/
midw_ext_fi_tcl_e8_SystemMessage::tenType fc_sxm_tclAdvisories::eGetCurrentAdvisoryType(tBool bOnlyChannelAdvisory, enAdvisoryMsgType &enCheckAdvisory) const
{
    ETG_TRACE_USR4(("fc_sxm_tclAdvisories::eGetCurrentAdvisoryType() _mSystemMessageUpdates = (%u) bOnlyChannelAdvisory = (%u)",
    		_mSystemMessageUpdates, bOnlyChannelAdvisory));

    /* Set current system message to no error */
    midw_ext_fi_tcl_e8_SystemMessage::tenType enCurrentSystemMsg;
    fc_sxm_tclAudioUtils::vSetChannelAdvisoryMsgType(_mSystemMessageUpdates, enCurrentSystemMsg, enCheckAdvisory);

    ETG_TRACE_USR3(("Channel advisory type Set: _mSystemMessageUpdates = (%u) enCurrentSystemMsg = (%u) enCheckAdvisory = (%u)",
        		_mSystemMessageUpdates, enCurrentSystemMsg, ETG_CENUM(enAdvisoryMsgType, enCheckAdvisory)));

    if(!bOnlyChannelAdvisory)
    {
    	fc_sxm_tclAudioUtils::vSetHighPrioAdvisoryMsgType(_mSystemMessageUpdates, enCurrentSystemMsg, enCheckAdvisory);
    	ETG_TRACE_USR3(("HighPrio advisory type Set: _mSystemMessageUpdates = (%u) enCurrentSystemMsg = (%u) enCheckAdvisory = (%u)",
    	        		_mSystemMessageUpdates, enCurrentSystemMsg, ETG_CENUM(enAdvisoryMsgType, enCheckAdvisory)));
    }

    return enCurrentSystemMsg;
}

// Method returns the currently set display advisory type
enAdvisoryMsgType fc_sxm_tclAdvisories::enGetAdvisoryType() const
{
	enAdvisoryMsgType enAdvisory;
	if (TRUE == bIsHighPriorityAdvisory())
		enAdvisory = enHIGH_PRIO_ADVISORY;
	else if (TRUE == bIsChannelAdvisory())
		enAdvisory = enCHANNEL_ADVISORY;
	else
		enAdvisory = enNO_ADVISORY;
	ETG_TRACE_USR3(("fc_sxm_tclAdvisories::enGetAdvisoryType returns %u", ETG_CENUM(enAdvisoryMsgType, enAdvisory)));
	return enAdvisory;
}

// Method returns TRUE if Channel specific advisory is applicable
tBool fc_sxm_tclAdvisories::bIsChannelAdvisory() const
{
	// Check if status bit is set for channel attributes
	tBool bIsChannelAdvisoryMsg = fc_sxm_tclAudioUtils::bIsChannelAdvisory(_mSystemMessageUpdates);
	ETG_TRACE_USR3(("fc_sxm_tclAdvisories::bIsChannelAdvisory(%u)  \t _mSystemMessageUpdates = %u", bIsChannelAdvisoryMsg, _mSystemMessageUpdates));
	return bIsChannelAdvisoryMsg;
}

// Method returns TRUE if any High Priority advisory is applicable
tBool fc_sxm_tclAdvisories::bIsHighPriorityAdvisory() const
{
	// Check if status bit is set for high priority attributes
	tBool bIsHighPrioAdvisoryMsg = fc_sxm_tclAudioUtils::bIsHighPriorityAdvisory(_mSystemMessageUpdates);
	ETG_TRACE_USR3(("fc_sxm_tclAdvisories::bIsHighPrioAdvisory  %u  \t  _mSystemMessageUpdates = %u", bIsHighPrioAdvisoryMsg, _mSystemMessageUpdates));
	return bIsHighPrioAdvisoryMsg;
}

/*****************************************************************************
 * FUNCTION    : vProcess(fc_sxm_trMsgAdvisoryTimer)                         *
 * CLASS       : fc_sxm_tclAdvisories                                        *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Advisory timer time out function                            *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : prMsg - Pointer to advisory timer struct                    *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclAdvisories::vProcessTimer(fc_sxm_trMsgAdvisoryTimer const * /* prMsg */ )
{
    ETG_TRACE_USR4(("fc_sxm_tclAdvisories::vProcess"
                    "(fc_sxm_trMsgAdvisoryTimer)"));

    /* Stop the timer and notify time out to advisory class */
    _mAdvisoryTimer.vStop();
    tBool bShouldRestorPrvChn =  fc_sxm_tclAudioUtils::bRestoreToPreviewChannel(_mSystemMessageUpdates);
    // If channel unauthorized bit is set then reset and load back previous or default channel
    fc_sxm_tclAudioUtils::vClearAllChannelAdvisories(_mSystemMessageUpdates);
    if (bShouldRestorPrvChn)
    {
    	vRestorePreviewChannel();
    }
    vUpdateChannelInfo(e_ADVISORY_TIMEOUT);
}

/*****************************************************************************
 * FUNCTION    : vProcess(fc_sxm_trMsgLoadingAdvisoryTimer)                  *
 * CLASS       : fc_sxm_tclAdvisories                                        *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Loading advisory timer time out function                    *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : prMsg - Pointer to load advisory timer struct               *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
 
tVoid fc_sxm_tclAdvisories::vProcessTimer(fc_sxm_trMsgLoadingAdvisoryTimer const * /* prMsg */ )
{
   ETG_TRACE_USR4(("fc_sxm_tclAdvisories::vProcess(fc_sxm_trMsgLoadingAdvisoryTimer)"));
   fc_sxm_tclGenericUtils::vSetBit(_mSystemMessageUpdates, static_cast <tU32> (midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_LOADING_XM));

   /* Get current advisory */
   enAdvisoryMsgType eCurrentAdvisory = enGetAdvisoryType();
   if(enHIGH_PRIO_ADVISORY != eCurrentAdvisory) {
	   vUpdateChannelInfo(e_TUNING_CHANNEL);
   }
}

// Set Audio App Pointer Reference
tVoid fc_sxm_tclAdvisories::vSetApp(Ifc_sxm_tclAudioApp* audioApp)
{
	ETG_TRACE_USR4(("fc_sxm_tclChannelList::vSetApp()"));
	_poAudioApp = audioApp;
}

// Method requests ChannelList object to restore playing preview channel
tVoid fc_sxm_tclAdvisories::vRestorePreviewChannel()
{
	Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
	                 		  _poAudioApp->poGetChannelListInstance(): NULL;
	SXM_ASSERT_RETURN(NULL != poChannelList)
	if(TRUE == _mLoadDefaultChannel)
	{
		// Reset flag
		_mLoadDefaultChannel = FALSE;
		// Tune to last tuned channel
		poChannelList->vRestorePreviewChannel();
	}
}

// Method notifies ChannelList to update channel information
tVoid fc_sxm_tclAdvisories::vUpdateChannelInfo(tFCSXM_SelectMode enMode)
{
	Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
		                 		  _poAudioApp->poGetChannelListInstance(): NULL;
	SXM_ASSERT_RETURN(NULL != poChannelList)
	poChannelList->vNotifyChannelInformationUpdate(enMode);
}

/*****************************************************************************
 * ----------------------------- END OF FILE ------------------------------- *
 *****************************************************************************/
