/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_audio_types.h
* @brief       File containing channel and category structure declarations. These 
*              structures holds all properties related to every individual 
*              channels and categories respectively.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

/*=============================================================================
=======                            DEFINES                              =======
=============================================================================*/

#ifndef FC_SXM_AUDIO_TYPES_H
#define FC_SXM_AUDIO_TYPES_H

/*=============================================================================
=======                            INCLUDES                             =======
=============================================================================*/

#include "fc_sxm_sms.h"
#include "fc_sxm_sms_util.h"

/*=============================================================================
=======               CONSTANTS & MACROS FOR GENERAL PURPOSE            =======
=============================================================================*/

// message-ids to be used by class of audio
#define FC_SXM_AUDIO_ACTION_BASE_APP            0
#define FC_SXM_AUDIO_ACTION_BASE_STATES         100
#define FC_SXM_AUDIO_ACTION_BASE_DIAG           200
#define FC_SXM_AUDIO_ACTION_BASE_CHN_LIST       300
#define FC_SXM_AUDIO_ACTION_BASE_CAT_LIST       400
#define FC_SXM_AUDIO_ACTION_BASE_PRESETS        500
#define FC_SXM_AUDIO_ACTION_BASE_ADVISORY       600
#define FC_SXM_AUDIO_ACTION_BASE_PLAYBACK       700
#define FC_SXM_AUDIO_ACTION_BASE_CONTENT_ALERT  800
#define FC_SXM_AUDIO_ACTION_BASE_FLASHEVENTS    900 // Applicable for SportsFlash and TWNow
#define SXM_AUDIO_DISP_TEXT_LENGTH              50

#define FC_SXM_AUDIO_CATEGORY_TYPE_MIN          0x01 /* Min cat value from SMS */
#define FC_SXM_AUDIO_CATEGORY_TYPE_MAX          0x3F /* Max 63 cat from SMS */

#define FC_SXM_AUDIO_CATEGORY_TYPE_USER_VIRTUAL_START      0x100 /* Min 256 cat value from SMS */
#define FC_SXM_AUDIO_CATEGORY_TYPE_USER_VIRTUAL_END        0x13F /* Max 256 + 63 cat from SMS */


/*=============================================================================
=======                             TYPES                               =======
=============================================================================*/

typedef enum Tunable
{
   e_UNAVAILABLE = 0,
   e_UNAUTHORIZED   ,
   e_LOCKED ,
   e_SKIPPED ,
   e_TUNABLE        
}tTunable;

/* Check for specific advisory type based on different user selections */
typedef enum AdvisoryMsg
{
    enRADIO_ID_DISPLAY   = 0, /* Check for anll advisories          */ 
    enHIGH_PRIO_ADVISORY    , /* Check for high priority advisories */
    enCHANNEL_ADVISORY      , /* Check for low priority advisories  */
    enNO_ADVISORY             /* Check for no advisories            */
} enAdvisoryMsgType;

/*
 * Contains the category information of channels
 */
struct fc_sxm_trChannelCategory
{
public:
	CATEGORY_ID   u16CatId;
	string oCatName;
	CATEGORY_TYPE_ENUM enCatType;

	fc_sxm_trChannelCategory(CATEGORY_ID catID = 0, string name = "", CATEGORY_TYPE_ENUM type = CATEGORY_TYPE_INVALID) :
		u16CatId(catID), oCatName(name), enCatType(type) { /* Default constructor */ }
};


typedef enum enChannelSkipStatus
{
	enChnSkip_Default = 0, /* Channel's skip status until modified by user after system Re-start*/
	enChnSkip_Set, /* Channel's skip status value when Channel's skip attribute is set to TRUE*/
	enChnSkip_Unset  /* Channel's skip status value when Channel's skip attribute is set to TRUE*/
} enChnSkipStatus;

/*
 * Contains information of an audio channel 
 */
struct fc_sxm_trChannel
{
   /*==========================================================================
   =======                      Public section                          =======
   ==========================================================================*/

   public:

      /* ====================== Member functions =========================== */

      /* Default constructor */
      fc_sxm_trChannel();
      /* constructor */
      fc_sxm_trChannel(CHANNEL_OBJECT hChannelObject);

      /* constructor */
      fc_sxm_trChannel(CHANNEL_OBJECT hChannelObject, CHANNEL_EVENT_MASK tEventMask);

      /* Destructor */
      ~fc_sxm_trChannel();
      /* Check if channel object is valid */
      tBool bIsValid() const;
      /* Check if channel is valid and tunable */
      tTunable eIsTunable() const;
      /* Get channel's Browse Status*/
      tTunable enBrowseStatus() const;
      /* Check if channel is subscribed channel and not a free to air channel */
      tBool bIsSubscribedChannel() const;

      /* Is channel name available */
      tBool bIsChannelNameAvailable(tVoid) const;
      /* Is category name available */
      tBool bIsCategoryNameAvailable(tVoid) const;
      /* Is song name available */
      tBool bIsSongNameAvailable(tVoid) const;
      /* Is artist name available */
      tBool bIsArtistNameAvailable(tVoid) const;

      /* Update channel availability status */
      tVoid vChannelRemoved(tVoid);
	  
      tVoid vCopyAttributes(fc_sxm_trChannel const &rChannel) {
         bIsChannelSubscribed = rChannel.bIsChannelSubscribed;
         bIsChannelFreeToAir = rChannel.bIsChannelFreeToAir;
         bIsPresetChannel = rChannel.bIsPresetChannel;
         bIsChannelMature = rChannel.bIsChannelMature ;
         bIsChannelLocked = rChannel.bIsChannelLocked ;
         bIsChannelSkipped = rChannel.bIsChannelSkipped ;
         bIsChannelTuneMix = rChannel.bIsChannelTuneMix ;
      }
	  
      tVoid vCopyCategoryInfo(fc_sxm_trChannel const &rChannel) {
         u16CategoryId = rChannel.u16CategoryId;
         oChannelCategoryName = rChannel.oChannelCategoryName;
         vectCategories.clear();
         vectCategories = rChannel.vectCategories ;
      }

      inline tBool bIsPreset() const {
    	  return bIsPresetChannel;
      }

      inline tBool bIsMature() const {
    	  return bIsChannelMature;
      }

      inline tBool bIsLocked() const {
    	  return bIsChannelLocked;
      }

      // Method returns the skip status of the channel
      tBool bIsSkipped() const {
    	  tBool bSkipStatus;
    	  if (enChnSkip_Default == enSkip)
    		  bSkipStatus =  bIsChannelSkipped;
    	  else if ( enChnSkip_Set == enSkip)
    		  bSkipStatus = TRUE;
    	  else
    		  bSkipStatus = FALSE;
    	  return bSkipStatus;
      }

      inline tBool bIsAuthorized() const {
    	  return ((bIsChannelFreeToAir | bIsChannelSubscribed));
      }

      inline tBool bIsTuneMixQualified() const {
    	  return bIsChannelTuneMix;
      }

      inline tVoid vLockChannel(tBool bLock) {
    	  bIsChannelLocked = bLock;
      }

      inline tVoid vSkipChannel(tBool bSkip) {
    	  bIsChannelSkipped = bSkip;
    	  enSkip = (bIsChannelSkipped) ? enChnSkip_Set : enChnSkip_Unset ;
      }

      /*inline tBool bIsSubscribed() const {
    	  return ( bIsChannelFreeToAir | bIsChannelSubscribed);
      }*/

      inline tBool bIsAvailable() const {
    	  return bIsChannelAvailable;
      }

      inline tBool bIsBrowseable() const {
    	  return ((e_TUNABLE ==  enBrowseStatus()));
      }

      inline tBool bIsFreeToAir() const {
    	  return bIsChannelFreeToAir;
      }

      inline tBool bIsSubscribed() const {
    	  return bIsChannelSubscribed;
      }

      tBool bBelongsToCategory(const CATEGORY_ID catID) const;

      /* ======================= Member variables ========================== */

      tBool         bIsChannelArtAvailable; // Channel Art availability flag
      tBool         bIsAlbumArtAvailable;  // Album Art availability flag
      SERVICE_ID    u16ServiceId;            // Service id
      CHANNEL_ID    u16ChannelId;            // Channel id
      CATEGORY_ID   u16CategoryId;           // Category id
      string        oAudioChannelName;       // Channel name
      string        oChannelCategoryName;    // Category name
      string        oChannelSongName;        // Song name
      string        oChannelArtistName;      // Artist name
      vector < fc_sxm_trChannelCategory > vectCategories; // A channel may belong to any no. of categories
      enChnSkipStatus enSkip;

   /*==========================================================================
   =======                     Private section                          =======
   ==========================================================================*/

   private:

      /* ====================== Member functions =========================== */

      /* Clear all member variable */
      tVoid vClearData(tVoid);

      /* Get channel name */
      static tVoid vGetChannelName(CHANNEL_OBJECT hChannelObject, string &roChannelName);
      /* Get channel category info */
      static tVoid vGetCatgeoryInfo(CHANNEL_OBJECT hChannelObject, CATEGORY_ID &ru16CategoryId, string &roCategoryName);
      /* Get channel artist name */
      static tVoid vGetArtistInfo(CHANNEL_OBJECT hChannelObject, string &roArtistName);
      /* Get channel song name */
      static tVoid vGetSongInfo(CHANNEL_OBJECT hChannelObject, string &roSongName);
      /* Get channel attributes */
      static tVoid vGetAttributeInfo(CHANNEL_OBJECT hChannelObject, tBool &rbSubscribed, tBool &rbFreeToAir);
      /* Get channel art info */
      static tVoid vGetArtInfo(CHANNEL_OBJECT hChannelObject, tBool &rbArtAvailable);
      /* Set channel availability status */
      static tVoid vSetChannelAvailable(tBool &rbAvailable);
      /* Reset channel availability status */
      static tVoid vResetChannelAvailable(tBool &rbAvailable);

      tVoid vReadChannelId(CHANNEL_OBJECT hChannelObject);
      tVoid vReadServiceId(CHANNEL_OBJECT hChannelObject);
      tVoid vReadAttributeInfo(CHANNEL_OBJECT hChannelObject);
      tVoid vReadChannelName(CHANNEL_OBJECT hChannelObject);
      tVoid vReadCategoryInfo(CHANNEL_OBJECT hChannelObject);
      tVoid vReadSongName(CHANNEL_OBJECT hChannelObject);
      tVoid vReadArtistName(CHANNEL_OBJECT hChannelObject);
      tVoid vReadChannelArtAvail(CHANNEL_OBJECT hChannelObject);
      /* Read all categories to which the channel is part of*/
      tVoid vReadChannelCategories(CHANNEL_OBJECT hChannelObject);
      /* Method reads if the reported channel is a preset channel or not*/
      tVoid vReadPresetInfo(CHANNEL_OBJECT hChannelObject);

      /* ======================= Member variables ========================== */

      tBool         bIsChannelAvailable;     // Availability flag
      tBool         bIsChannelFreeToAir;     // Free to Air flag
      tBool         bIsChannelSubscribed;    // Subscription flag
      tBool         bIsPresetChannel;       // Preset channel status

      tBool         bIsChannelMature;        // Mature flag
      tBool         bIsChannelLocked;        // Channel Lock status
      tBool         bIsChannelSkipped;      // Channel skip status
      tBool         bIsChannelTuneMix;      // Tune Mix qualified status

   /*==========================================================================
   =======                    Protected section                         =======
   ==========================================================================*/

   protected:
};

/*
 * Structure containing all information of an audio category
 */
struct fc_sxm_trCategory
{
   /*==========================================================================
   =======                      Public section                          =======
   ==========================================================================*/

   public:

      /* ====================== Member functions =========================== */

      /* constructor */
      fc_sxm_trCategory(CATEGORY_ID u16CategoryID);

      fc_sxm_trCategory(CATEGORY_OBJECT hCategory, CATEGORY_EVENT_MASK tEventMask);

      /* Destructor */
      ~fc_sxm_trCategory();

      /* Is category name available */
      tBool bIsCategoryNameAvailable(tVoid) const;


      /* Check if category contains tunable channels */
      tBool bIsCategoryAvailable(tVoid) const;

    tVoid vReadCategoryId(CATEGORY_OBJECT hCategory);
    tVoid vReadCategoryName(CATEGORY_OBJECT hCategoryObject);
    tVoid vReadCategoryType(CATEGORY_OBJECT hCategory);

      /* ====================== Member variables =========================== */

      CATEGORY_ID  _u16CategoryId;         /* Category Id */
      string       _oCategoryName;         /* Category name */
      CATEGORY_TYPE_ENUM _enCatType;   /* Category Type */

      /* Category channel list */
      map<CHANNEL_ID, fc_sxm_trChannel> mCategoryChannelList;

   /*==========================================================================
   =======                     Private section                          =======
   ==========================================================================*/

   private:

      /* Default constructor */
      fc_sxm_trCategory();

      /* Clear all member variables */
      tVoid vClearData(tVoid);

      /* Get category name */
      static tVoid vGetCategoryName(CATEGORY_OBJECT hCategoryObject, string &roChannelCategoryName);

   /*==========================================================================
   =======                    Protected section                         =======
   ==========================================================================*/

   protected:
};

/*=============================================================================
=======                   PROTOTYPES & FUNCTIONS                        =======
=============================================================================*/

/*=============================================================================
=======                              METHODS                            =======
=============================================================================*/

/*=============================================================================
=======                              CLASSES                            =======
=============================================================================*/

#endif

/*****************************************************************************
 * ----------------------------- END OF FILE ------------------------------- *
 *****************************************************************************/


