/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_diaghandler.cpp
* @brief       SXM Diagnosis handler.
*              Maintains ownership of all helper classes that implements different 
*                   functionalities of SXM Diagnosis.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_main.h"

#include "fc_sxm_tcl_sms_init.h"
/*Taken from prototyping.cpp.Required only for prototyping:start*/
#include "fc_sxm_tcl_timer.h"
#include "fc_sxm_sms_util.h"
#include "fc_sxm_tcl_sxmapp_manager.h"
#include "fc_sxm_tcl_base_app.h"
/*Taken from prototyping.cpp.Required only for prototyping:end*/

#include "fc_sxm_service_sxm_audio.h"
#include "fc_sxm_audio_fi.h"
#include "fc_sxm_tcl_audio_app.h"
#include "fc_sxm_tcl_audio_properties.h"
#include "fc_sxm_audiorouting_lib.h"

#include "fc_sxm_diag_if.h"
#include "fc_sxm_tcl_diag_handler_test.h"

//#define TEST_REG

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_DIAGHANDLER
#include "trcGenProj/Header/fc_sxm_tcl_diaghandler.cpp.trc.h"
#endif

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_fc_sxm_if.h"

//extern "C" void SMSLIB_vPrintCache(void);

typedef enum {
    fc_sxm_enTtfisCmdsDiag_SystemStateActivityStart=0xF0,
    fc_sxm_enTtfisCmdsDiag_SystemStateActivityStop=0xF1,
    fc_sxm_enTtfisCmdsDiag_PackageId_Start=0xF2,
    fc_sxm_enTtfisCmdsDiag_Defset=0xF3,
    fc_sxm_enTtfisCmdsDiag_Download=0xF4,
    fc_sxm_enTtfisCmdsDiag_RestoreSmsHb=0xF5,
    fc_sxm_enTtfisCmdsDiag_SmsLog=0xF6,
    fc_sxm_enTtfisCmdsDiag_TestMode_Start=0xF7,
    /* command to test loopback-msg to cca-client or server (example:arl) */
    fc_sxm_enTtfisCmdsDiag_TestArlLoopBack=0xF8,
    fc_sxm_enTtfisCmdsDiag_TestExtMode=0xF9,
    fc_sxm_enTtfisCmdsDiag_PrintFuelCache=0xFA,
    fc_sxm_enTtfisCmdsDiag_GetVersionFromRegistry=0xFB
} fc_sxm_tenTtfisCmdsDiag;


tVoid fc_sxm_tclDiagTestActivity::vRun() {
    //Logging the status of test activity
    ETG_TRACE_USR4(("fc_sxm_tclDiagTestActivity(%d)::vRun",
                    ETG_CENUM(fc_sxm_tenSystemState, enGetType())));
}



/*******************************************************************************
 *
 * FUNCTION: fc_sxm_tclDiagHandler::
 *             fc_sxm_tclDiagHandler()
 *
 * DESCRIPTION: Constructor .
 *              
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
fc_sxm_tclDiagHandler::fc_sxm_tclDiagHandler()
{
    fc_sxm_tclDiagDefset::instance();
    ETG_TRACE_USR4(("fc_sxm_tclDiagHandler constructor"));
}

/*******************************************************************************
 *
 * FUNCTION: fc_sxm_tclDiagHandler::
 *             ~fc_sxm_tclDiagHandler()
 *
 * DESCRIPTION: Destructor .
 *              
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
fc_sxm_tclDiagHandler::~fc_sxm_tclDiagHandler()
{
    OSAL_DELETE fc_sxm_tclDiagDefset::instance(); //deallocate
    ETG_TRACE_USR4(("fc_sxm_tclDiagHandler destructor"));
}


tVoid fc_sxm_tclDiagHandler::vInit() {
    ETG_TRACE_USR4(("fc_sxm_tclDiagHandler vInit"));
     //initialise the member lists
    _oMemberList.poAdd(fc_sxm_tclDiagTmHandler::instance());
    _oMemberList.poAdd(fc_sxm_tclDiagUsrDataClear::instance());
    _oMemberList.poAdd(fc_sxm_tclDiagReporter::instance());
    //Add the diag defset handler
    _oMemberList.poAdd(fc_sxm_tclDiagDefset::instance());
    _oMemberList.poAdd(fc_sxm_tclDiagLog::instance());
    _oMemberList.poAdd(fc_sxm_tclDiagSmsLog::instance());
    //Add the external diagnosis mode handler
    _oMemberList.poAdd(fc_sxm_tclDiagExtDiagnosisMode::instance());
}

tVoid fc_sxm_tclDiagHandler::vDeInit() {
    ETG_TRACE_USR4(("fc_sxm_tclDiagHandler vDeInit"));
    _oMemberList.vClear(); //clear the memberlist
}
/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagHandler::bDispatchMsg
 *(
 *  fc_sxm_tclMessage const *poThreadMsg
 *) const
 *
 * DESCRIPTION: This is the dispatch method for Diagnosis.All diagnosis message 
 *              exchange happens through this dispatch message.All the diagnosis 
 *              related messages will be delegated to this dispatch function by 
 *              audio diapatch function
 *
 * PARAMETER: poThreadMsg : The pointer to the populated message
 *
 * RETURNVALUE: tBool - True : If action found
 *                      False:If action not found
 *
 *******************************************************************************/
tBool fc_sxm_tclDiagHandler::bDispatchMsg
(
 fc_sxm_tclMessage const *poThreadMsg
 ) const
{
    ETG_TRACE_USR4(("fc_sxm_tclDiagHandler::bDispatchMsg: SID=%x action=%d",
                    ETG_CENUM(fc_sxm_tenServiceID, poThreadMsg->enGetServiceId()), 
                    poThreadMsg->u16GetActionOnly()));

    tBool bRetVal = TRUE;

    tU32 u32Action=poThreadMsg->u32GetAction();
    switch (u32Action) {
        //SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgEvtAudioSmsModuleSubStatusChanged,fc_sxm_tclDiagTmHandler::instance());

        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgCmdClearUserData, fc_sxm_tclDiagUsrDataClear::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgCmdAppTtfisCmd,fc_sxm_tclDiagHandler::instance());
        
        SXM_MSGQ_DISPATCH_TO_TIMEROBJ(fc_sxm_trMsgCmdClearUserDataFinalize, fc_sxm_tclDiagUsrDataClear::instance());

        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgCmdDiagPrintReport, fc_sxm_tclDiagReporter::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgCmdDiagPrintReportDone, fc_sxm_tclDiagReporter::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgDiagMStartTestModeEnable, fc_sxm_tclDiagTmHandler::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgDiagMFetchUpcValue, fc_sxm_tclDiagTmHandler::instance());

        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgDiagSmsDbError, fc_sxm_tclDiagDefset::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgDiagRestoreSmsDb, fc_sxm_tclDiagDefset::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgDiagDefset, fc_sxm_tclDiagDefset::instance());

        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgDiagDownloadMStart, fc_sxm_tclDiagDownload::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgDiagModuleUpdateEvent, fc_sxm_tclDiagDownload::instance());

        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgDiagSxmDbReplaced,fc_sxm_tclDiagDefset::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgFiDiaglogNextTestStatus, fc_sxm_tclDiagLog::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgFiDiaglogMethodResultSaveTest, fc_sxm_tclDiagLog::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgDiagExtMode, fc_sxm_tclDiagExtDiagnosisMode::instance());

        default:
            bRetVal = FALSE;
    }

    return bRetVal;
}


/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagHandler::vProcess
 *(
 * fc_sxm_trMsgCmdAppTtfisCmd const *prMsg
 * )
 *
 * DESCRIPTION: This is TTFIs commands for system state activity
 *
 * PARAMETER: fc_sxm_trMsgCmdAppTtfisCmd const *prMsg : The message to be processed
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagHandler::vProcess
(
 fc_sxm_trMsgCmdAppTtfisCmd const *prMsg
 )
{
    fc_sxm_tenTtfisCmdsDiag enMsgCode=(fc_sxm_tenTtfisCmdsDiag)prMsg->u8MsgCode;
    ETG_TRACE_USR4(("fc_sxm_tclDiag::vProcess(fc_sxm_trMsgCmdAppTtfisCmd) code=%d",
                    ETG_CENUM(fc_sxm_tenTtfisCmdsDiag, enMsgCode)));
    tU8 u8Param1 = prMsg->au8Data[0];
    //based on msgcode execute the command
    switch (enMsgCode) {
        case fc_sxm_enTtfisCmdsDiag_SystemStateActivityStart: //command to test SystemStateActivityStart
        {
            fc_sxm_tenSystemState enSystemState=(fc_sxm_tenSystemState)u8Param1;
            ETG_TRACE_USR4(("fc_sxm_tclDiag::SystemStateActivityStart: %d",
                            ETG_CENUM(fc_sxm_tenSystemState, enSystemState)));
            _oTestActivity.bSetType(enSystemState);
            tBool bAccepted=_oTestActivity.bRequest();
            ETG_TRACE_USR4(("fc_sxm_tclDiag::SystemStateActivityStart: bAccepted=%d",
                            bAccepted));
        }
        break;
        case fc_sxm_enTtfisCmdsDiag_SystemStateActivityStop: //command to test SystemStateActivityStop
            ETG_TRACE_USR4(("fc_sxm_tclDiag::SystemStateActivityStop"));
            // end activity
            _oTestActivity.vOnDone();
            fc_sxm_tclDiagHandler::instance()->vHandleResult(fc_sxm_enDiagResult_Ok);
            break;
        case fc_sxm_enTtfisCmdsDiag_PackageId_Start: //command to test Diag_PackageId_Start
        {
            ETG_TRACE_USR4(("fc_sxm_tclDiag::fc_sxm_enTtfisCmdsDiag_PackageId_Start"));
            midw_ext_sxm_audiofi_tclMsgDiagUpcValuesMethodStart oPGet;
            oPGet.GetVal = u8Param1;
            fc_sxm_vSendSelfMessage(oPGet);
        }
        break;


        case fc_sxm_enTtfisCmdsDiag_Defset:   //command to test Diag_Defset
        {
            fc_sxm_tenDiagDefsetType enType=(fc_sxm_tenDiagDefsetType)u8Param1;
            fc_sxm_tenDefsetState enState = (fc_sxm_tenDefsetState)prMsg->au8Data[1];
            ETG_TRACE_USR4(("fc_sxm_enTtfisCmdsDiag_Defset type=%d,state = %d",
                            ETG_CENUM(fc_sxm_tenDiagDefsetType, enType),ETG_CENUM(fc_sxm_tenDefsetState, enState)));
            fc_sxm_trMsgDiagDefset rMsg(enType,enState);
            fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);
        }
        break;

        case fc_sxm_enTtfisCmdsDiag_Download:  //command to test Diag_Download
        {
            	midw_ext_sxm_audiofi_tclMsgFirmwareDownloadMethodStart oPStart;
            	oPStart.bForced = (u8Param1!=0);
            	char acTmp[100];
            	oPStart.Version = OSALUTIL_szSaveStringNCopy(acTmp,
                        (char const *)&prMsg->au8Data[1],
                        30);
                 ETG_TRACE_USR4(("fc_sxm_enTtfisCmdsDiag_Download oSwVersion=%s",
                		 oPStart.Version.szValue));
                 oPStart.File = OSALUTIL_szSaveStringNCopy(acTmp,
                         (char const *)&prMsg->au8Data[31],
                         100);
                 ETG_TRACE_USR4(("fc_sxm_enTtfisCmdsDiag_Download oFileName=%s",
                		 oPStart.File.szValue));
                 fc_sxm_vSendSelfMessage(oPStart);
        }
        break;

        case fc_sxm_enTtfisCmdsDiag_RestoreSmsHb:  //command to test Diag_RestoreSmsHb
        {
            ETG_TRACE_USR4(("fc_sxm_tclDiag::fc_sxm_enTtfisCmdsDiag_RestoreSmsHb"));
            midw_ext_sxm_audiofi_tclMsgDiagSxmResetMethodStart oPGet;
            oPGet.tU8Request = u8Param1;
            fc_sxm_vSendSelfMessage(oPGet);
        }
        break;

        case fc_sxm_enTtfisCmdsDiag_SmsLog: //command to test SmsLog
        {
            fc_sxm_tenSmsLogCmd enCmd = (fc_sxm_tenSmsLogCmd)u8Param1;
            char acTmp[100];
            string oSmsObjName=
                OSALUTIL_szSaveStringNCopy(acTmp,
                                           (char const *)&prMsg->au8Data[1],
                                           30);
            ETG_TRACE_USR4(("fc_sxm_tclDiag::fc_sxm_enTtfisCmdsDiag_SmsLog"));
            fc_sxm_tclDiagSmsLog::instance()->vHandleLogCmd(enCmd, oSmsObjName);
        }
        break;

    case fc_sxm_enTtfisCmdsDiag_TestMode_Start:  //command to test TestMode_Start
        {
            midw_ext_sxm_audiofi_tclMsgDiagTestModeEnableMethodStart oPGet;
            oPGet.bEnable = u8Param1;
            fc_sxm_vSendSelfMessage(oPGet);
        }
        break;

        case fc_sxm_enTtfisCmdsDiag_TestArlLoopBack:  //command to test Diag_TestArlLoopBack
        {
            fc_sxm_trMsgCmdArlTestLoopBack rMsg;
            rMsg.u8TestVal=5;
            fc_sxm_vPostMsgToCcaObj(fc_sxm_arl_tclISource::instance(), rMsg);
        }
        break;

         case fc_sxm_enTtfisCmdsDiag_TestExtMode: //Test the external diagnosis mode
        {
            midw_ext_sxm_audiofi_tclMsgExternalDiagnosisModeMethodStart oPGet;
            oPGet.Mode.enType = (midw_ext_fi_tcl_e8_SxmExtDiagMode::tenType)u8Param1;            
            fc_sxm_vSendSelfMessage(oPGet);
        }
        break;

        case fc_sxm_enTtfisCmdsDiag_PrintFuelCache:
        {
            //SMSLIB_vPrintCache();
        }
        break;

        case fc_sxm_enTtfisCmdsDiag_GetVersionFromRegistry:
        {
            //Datapool objs
            dp_tclFcSxmDPRADIOID sRadioId;
            dp_tclFcSxmDPHWREV  sHwRev;
            dp_tclFcSxmDPSWREV  sSwRev;

            char aParam[32];
            sRadioId.u8GetData(aParam, 32);
            ETG_TRACE_USR4(("fc_sxm_tclDiagHandler::sRadioId.u8GetData: %s", aParam));

            aParam[0] = '\0';
            sHwRev.u8GetData(aParam, 32);
            ETG_TRACE_USR4(("fc_sxm_tclDiagHandler::sHwRev.u8GetData: %s", aParam));

            aParam[0] = '\0';
            sSwRev.u8GetData(aParam, 32);
            ETG_TRACE_USR4(("fc_sxm_tclDiagHandler::sSwRev.u8GetData: %s", aParam));
        }
        break;

        default:
            ETG_TRACE_USR4(("fc_sxm_tclDiag::vProcess(fc_sxm_trMsgCmdTtfisCmd) unknown code=0x%02x",
                            enMsgCode));
            break;
    }

}

tVoid fc_sxm_tclDiagHandler::vHandleResult(fc_sxm_tenDiagResult enResult) {
   //print diagresult
    ETG_TRACE_USR4(("fc_sxm_tclDiag::vHandleResult(enResult) enResult=%d",
                    ETG_CENUM(fc_sxm_tenDiagResult, enResult)));
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagHandler::vUpdateDecoderState()
 *
 * DESCRIPTION: This function is used by the Audio app to update the decoder 
 *              state to FC_SXM Diagnosis
 *
 * PARAMETER: [IN] hDecoderObject - Handle to Decoder
 *            [IN] enDecoderState - State of the decoder
 *
 * RETURNVALUE: None
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagHandler::vUpdateDecoderState(
                                                 DECODER_OBJECT hDecoderObject, 
                                                 DECODER_STATE_ENUM enDecoderState)
{
    ETG_TRACE_USR4(("fc_sxm_tclDiagHandler::vUpdateDecoderState()"));
    (tVoid)hDecoderObject; //To remove Lint warning

    if(DECODER_STATE_READY == enDecoderState)
    {
        /* Update antenna state */
        vUpdateAntennaState(
                            DECODER.eAntennaState(
                                                  fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), 1));

        /* Get signal quality */
        SIGNAL_QUALITY_STRUCT lSignalQuality;
        if(SMSAPI_RETURN_CODE_SUCCESS == 
           DECODER.eSignalQuality(
                                  fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), 
                                  &lSignalQuality))
        {
            /* Update signal quality */
            vUpdateSignalQuality(lSignalQuality);
        }
        else
        {
            ETG_TRACE_ERR(("fc_sxm_tclDiagHandler :Failed to get signal quality"));

            lSignalQuality.eState = SIGNAL_STATE_UNKNOWN;

            /* Update signal quality */
            vUpdateSignalQuality(lSignalQuality);
        }   
    }
}


/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagHandler::vUpdateAntennaState
 *(
 * ANTENNA_STATE_ENUM hAntennaState
 *)
 *
 * DESCRIPTION: This function is used to update the antenna state
 *
 * PARAMETER: [IN] ANTENNA_STATE_ENUM hAntennaState
 *
 * RETURNVALUE: None
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagHandler::vUpdateAntennaState
(
 ANTENNA_STATE_ENUM hAntennaState
 ) const
{
    ETG_TRACE_USR4(("fc_sxm_tclDiagHandler::vUpdateAntennaState()"));

    midw_ext_sxm_audiofi_tclMsgSxmDiagAntennaStateStatus &rFiData =fc_sxm_tclDiagProperties::instance()->oAntennaState.rAccess().oFiMsg;
    rFiData.AntennaStatus.enType = (midw_ext_fi_tcl_e8_Sxm_AntennaState::tenType) hAntennaState ;
    fc_sxm_tclDiagProperties::instance()->oAntennaState.vRelease();

    fc_sxm_tclDiagProperties::instance()->oAntennaState.vNotify(); //notify anteena state  
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagHandler::vUpdateSignalQuality
 *(
 * SIGNAL_QUALITY_STRUCT lSignalQuality
 *)
 *
 * DESCRIPTION: This function is used to update the signal quality state
 *
 * PARAMETER: [IN]  SIGNAL_QUALITY_STRUCT lSignalQuality
 *
 * RETURNVALUE: None
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagHandler::vUpdateSignalQuality
(
 SIGNAL_QUALITY_STRUCT lSignalQuality
 ) const
{
    ETG_TRACE_USR4(("fc_sxm_tclDiagHandler::vUpdateSignalQuality()"));

   fc_sxm_tcl_trDiagPropertySignalQuality lSignalQualityUpdate;
    if(SIGNAL_STATE_ACQUIRED == lSignalQuality.eState)
    {
        lSignalQualityUpdate.oFiMsg.SignalQualityStatus.enType = (midw_ext_fi_tcl_e8_Sxm_SignalQuality::tenType)(lSignalQuality).eComposite;
    }
    else
    {
        lSignalQualityUpdate.oFiMsg.SignalQualityStatus.enType = midw_ext_fi_tcl_e8_Sxm_SignalQuality::FI_EN_FC_SXM_DIAG_SQ_NO_SIGNAL;
    }
    fc_sxm_tclDiagProperties::instance()->oSignalQuality.vSetAndNotify(lSignalQualityUpdate);
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagHandler::vProcess
 *           (
 *            fc_sxm_trMsgAudioSmsEvtAntennaQuality const *prMsg
 *            )
 *
 * DESCRIPTION: This function is used by the Audio app to update the Antenna 
 *              state to FC_SXM Diagnosis
 *
 * PARAMETER: [IN] u8AntennaState  - The antenna state got from SMS
 *                                   FC_SXM_DIAG_ANT_UNSUPPORTED
 *                                   FC_SXM_DIAG_ANT_DETECTED 
 *                                   FC_SXM_DIAG_ANT_NOT_DETECTED 
 *                                   FC_SXM_DIAG_ANT_SHORTED 
 *                                   FC_SXM_DIAG_ANT_UNKNOWN   
 *            
 *
 * RETURNVALUE: None
 *
 *******************************************************************************/ 
tVoid fc_sxm_tclDiagHandler::vProcess
(
 fc_sxm_trMsgAudioSmsEvtAntennaQuality const *prMsg
 ) const
{
    ETG_TRACE_USR1(("fc_sxm_tclDiagHandler::vUpdateAntennaState() on event"));
    //Update antenna state
    vUpdateAntennaState(prMsg -> eAntennaState);
   //Send antenna status diaglog
    fc_sxm_tclDiagLog::instance()->vSendMethodStartSaveTestResult(FALSE);
}

tVoid fc_sxm_tclDiagHandler::vProcess(fc_sxm_trMsgAudioDiagSetSXMAntennaDTC const * /* prMsg*/ ) const
{
    ETG_TRACE_USR1(("fc_sxm_tclDiagHandler::fc_sxm_trMsgAudioDiagSetSXMAntennaDTC()"));

    //Send the current antenna status to diaglog
    fc_sxm_tclDiagLog::instance()->vSendMethodStartSaveTestResult(TRUE);
}


/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagHandler::vProcess
 *           (
 *            fc_sxm_trMsgAudioSmsEvtSignalState const *prMsg
 *            )
 *
 * DESCRIPTION: This function is used by the Audio app to update the Signal 
 *              quality to FC_SXM Diagnosis
 *
 * PARAMETER: [IN] u8SignalQuality  - The signal quality got from SMS
 *                                   FC_SXM_DIAG_SQ_NO_SIGNAL
 *                                   FC_SXM_DIAG_SQ_VERY_WEAK_SIGNAL
 *                                   FC_SXM_DIAG_SQ_WEAK_SIGNAL
 *                                   FC_SXM_DIAG_SQ_MARGINAL_SIGNAL
 *                                   FC_SXM_DIAG_SQ_GOOD_SIGNAL 
 *            
 *
 * RETURNVALUE: None
 *
 *******************************************************************************/ 
tVoid fc_sxm_tclDiagHandler::vProcess
(
 fc_sxm_trMsgAudioSmsEvtSignalState const *prMsg
 ) const
{
    ETG_TRACE_USR1(("fc_sxm_tclDiagHandler::vUpdateSignalQuality() on event"));
    //update signal quality
    vUpdateSignalQuality(prMsg -> eSignalQuality);
}


/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagHandler::vProcess
 *           (
 *            fc_sxm_trMsgAudioSmsEvtModuleState const *prMsg
 *            )
 *
 * DESCRIPTION: This function is used to check the module state and carry out 
 *              functionlities on module state ready
 *
 * PARAMETER: fc_sxm_trMsgAudioSmsEvtModuleState const *prMsg 
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagHandler::vProcess
(
 fc_sxm_trMsgAudioSmsEvtModuleState const *prMsg
 ) 
{
    ETG_TRACE_USR4(("fc_sxm_tclDiagHandler::fc_sxm_trMsgAudioSmsEvtModuleState"));
    if (prMsg->eState != MODULE_STATE_READY) {
        return;
    }
    
    /* fetch version-info */
    vFetchModuleVerInfo(&_rModuleVerInfo);
    /*Write to registry on module state ready*/
    vWriteSxmVerRegistry(_rModuleVerInfo);
    
}


/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagHandler::vFetchModuleVerInfo
 *(
 *  fc_sxm_trModuleVerInfo *prVerInfo
 *)const
 *
 * DESCRIPTION: This function is used to fetch the S65 Module version numbers from the 
 *              module
 *
 * PARAMETER: fc_sxm_trModuleVerInfo *prVerInfo : Structure to store version info
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagHandler::vFetchModuleVerInfo
(
 fc_sxm_trModuleVerInfo *prVerInfo
 ) const
{
    MODULE_OBJECT hModule= fc_sxm_tclSmsModule::instance()->hGetSmsModule();
    fc_sxm_vCopySmsString2Stl( MODULE.hESN(hModule),prVerInfo->oRadioId); //Radio Id
    fc_sxm_vCopySmsString2Stl(MODULE_VERSION.hHardwareRevision(hModule),prVerInfo->oHwRev); //HW VERsion info
    fc_sxm_vCopySmsString2Stl(MODULE_VERSION.hSoftwareRevision(hModule),prVerInfo->oSwRev); //SW version info
}


/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagHandler::vWriteSxmVerRegistryKey
 *(
 * fc_sxm_tclRegUtil &oReg, 
 * const char *pcKey,
 * string const &oValString
 * ) const
 *
 * DESCRIPTION: This function is used to write single key to version registry 
 *              used by vWriteSxmVerRegistry()
 *
 * PARAMETER: fc_sxm_tclRegUtil &oReg,  //Handle to the Registry utility class
 *            const char *pcKey,        //The key to identify the registry
 *            string const &oValString  //the data to write to the registry
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagHandler::vWriteSxmVerRegistryKey
(
 fc_sxm_tclRegUtil &oReg, 
 const char *pcKey,
 string const &oValString
 ) const
{
    ETG_TRACE_USR4(("vWriteSxmVerRegistryKey: %25s =%s",pcKey, oValString.c_str()));
    if(TRUE != oReg.bWrite(pcKey,const_cast<char*>(oValString.c_str()),(tU32)((oValString.length() + 1))))
    {
        ETG_TRACE_USR2(("vWriteSxmVerRegistry: Registry write failed key=%30s val=%s", pcKey, oValString.c_str()));
    }
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagHandler::vWriteSxmVerRegistry
 *(
 * fc_sxm_trModuleVerInfo const &rVerInfo
 * ) const
 *
 * DESCRIPTION: This function is used to write the S65 Module version numbers to registry
 *              so that it can be read by the FC_Diagnosis
 *
 * PARAMETER: fc_sxm_trModuleVerInfo const &rVerInfo //Version info 
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagHandler::vWriteSxmVerRegistry
(
 fc_sxm_trModuleVerInfo const &rVerInfo
 ) const
{
    ETG_TRACE_USR4(("fc_sxm_tclDiagHandler::vWriteSxmVerRegistry()"));
    
    fc_sxm_tclRegUtil oReg(SXM_DIAG_REG_VERSIONS_PATH);

    /*Write Module ID to registry*/
    vWriteSxmVerRegistryKey(oReg, SXM_DIAG_REG_SERIAL_NUM_KEY, rVerInfo.oRadioId);

    /*Write Hardware version to registry*/
    vWriteSxmVerRegistryKey(oReg, SXM_DIAG_REG_HW_VER_KEY, rVerInfo.oHwRev);

    /*Write Software version to registry*/
    vWriteSxmVerRegistryKey(oReg, SXM_DIAG_REG_SW_VER_KEY, rVerInfo.oSwRev);

    string strHwId;
    MODULE_OBJECT hModule = fc_sxm_tclSmsModule::instance()->hGetSmsModule();
    fc_sxm_vCopySmsString2Stl(MODULE_VERSION.hTypeID(hModule), strHwId);
    vWriteSxmVerRegistryKey(oReg, "SXM_TYPE", strHwId);

    //Datapool objs
    dp_tclFcSxmDPRADIOID sRadioId;
    dp_tclFcSxmDPHWREV  sHwRev;
    dp_tclFcSxmDPSWREV  sSwRev;

	ETG_TRACE_USR4(("fc_sxm_tclDiagHandler::vWriteSxmVerRegistry()rVerInfo.oRadioId: %s", const_cast<char *>((rVerInfo.oRadioId).c_str())));
	ETG_TRACE_USR4(("fc_sxm_tclDiagHandler::vWriteSxmVerRegistry()rVerInfo.oHwRev: %s", const_cast<char *>((rVerInfo.oHwRev).c_str())));
	ETG_TRACE_USR4(("fc_sxm_tclDiagHandler::vWriteSxmVerRegistry()rVerInfo.oSwRev: %s", const_cast<char *>((rVerInfo.oSwRev).c_str())));

	//Writing RadioId, H/W Ver., S/W Ver. into Datapool registry
    sRadioId.vSetData(const_cast<char *>((rVerInfo.oRadioId).c_str()));
    sHwRev.vSetData(const_cast<char *>((rVerInfo.oHwRev).c_str()));
    sSwRev.vSetData(const_cast<char *>((rVerInfo.oSwRev).c_str()));

    /*Write Package ID to registry : TBD..Clarification from SXM pending*/   


#ifdef TEST_REG
    fc_sxm_tclDiagHandler_test::instance() -> vTraceRegValues();
#endif

}






