/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_diag_usr_data_clear.cpp
* @brief       Implementation for for clearing user data.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_tcl_timer.h"

#include "fc_sxm_main.h"
#include "fc_sxm_tcl_sxmapp_manager.h"

#include "fc_sxm_tcl_diaghandler.h"
#include "fc_sxm_tcl_diag_defset.h"
#include "fc_sxm_tcl_diag_usr_data_clear.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_DIAGHANDLER
#include "trcGenProj/Header/fc_sxm_tcl_diag_usr_data_clear.cpp.trc.h"
#endif


tVoid fc_sxm_tclDiagUsrDataClear::vProcess(fc_sxm_trMsgCmdClearUserData const *prMsg) {
    (tVoid)prMsg;//not used resolving prio 2 lint
    ETG_TRACE_USR3(("fc_sxm_tclDiagUsrDataClear::vProcess(fc_sxm_trMsgCmdClearUserData)"));
    bStart();
}


fc_sxm_tclDiagUsrDataClear::fc_sxm_tclDiagUsrDataClear()
{
	/*Open the semaphore*/
    _oSem.vOpen();
}


fc_sxm_tclDiagUsrDataClear::~fc_sxm_tclDiagUsrDataClear() {
	/*Close the semaphore*/
    _oSem.vClose();
}


tBool fc_sxm_tclDiagUsrDataClear::bStart() {
    ETG_TRACE_USR3(("fc_sxm_tclDiagUsrDataClear::bStart()"));
    /*Get the semaphore*/
    _oSem.vGet();
    if (bIsActive()) {
        _oSem.vPost();
        ETG_TRACE_ERR(("fc_sxm_tclDiagUsrDataClear::bStart() failed, already active"));
        return FALSE;
    }
    /*clear the map results*/
    _mapResults.clear();
    fc_sxm_trMsgCmdAppClearUserData rMsg;
    map<fc_sxm_tenServiceID, fc_sxm_tclBaseApp*> const &mapOfApps = fc_sxm_tclAppManager::instance()->oGetMapOfApps();
    /*Iterate through the map containing service information*/
    SXM_FOREACH_MAP_CONST(fc_sxm_tenServiceID, fc_sxm_tclBaseApp*, iter, mapOfApps) {
        _mapResults[iter->first]=fc_sxm_enDiagResult_Pending;
        ETG_TRACE_USR4(("fc_sxm_tclDiagUsrDataClear::bStart(): request for %d",
                        ETG_CENUM(fc_sxm_tenServiceID, iter->first)));
        iter->second->vPostMsgNew(rMsg);
    }
#define FC_SXM_DIAG_RESET_USER_DATA_SV_TIMEOUT_MS 3000
    _oSupervisonTimer.vStart(FC_SXM_DIAG_RESET_USER_DATA_SV_TIMEOUT_MS);
    _oSem.vPost();
    ETG_TRACE_USR3(("fc_sxm_tclDiagUsrDataClear::bStart() END"));

    return TRUE;
}

// thread-save, each app has to report its result here
tVoid fc_sxm_tclDiagUsrDataClear::vSetResult(fc_sxm_tenServiceID enServiceId, fc_sxm_tenDiagResult enRes) {
    ETG_TRACE_USR3(("fc_sxm_tclDiagUsrDataClear::vSetResult(%d, %d)",
                    ETG_CENUM(fc_sxm_tenServiceID, enServiceId),
                    ETG_CENUM(fc_sxm_tenDiagResult, enRes)));
    /*Get the semaphore*/
    _oSem.vGet();
    if (!bIsActive()) {
        ETG_TRACE_ERR(("fc_sxm_tclDiagUsrDataClear::vSetResult(): not active"));
        _oSem.vPost();
        return;
    }
    /*Get the result of all services*/
    SXM_IF_FIND_MAP(fc_sxm_tenServiceID, fc_sxm_tenDiagResult, iter, _mapResults, enServiceId) {
        iter->second=enRes;
    }
    tBool bReady= bIsReady();
    _oSem.vPost();
    if (bReady) {
        ETG_TRACE_USR3(("fc_sxm_tclDiagUsrDataClear::vSetResult READY"));

        fc_sxm_trMsgCmdClearUserDataFinalize rMsg; /*Post the message*/
        fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);
    }
}


tVoid fc_sxm_tclDiagUsrDataClear::vProcessTimer(fc_sxm_trMsgCmdClearUserDataFinalize const *prMsg) {
    (tVoid)prMsg;
    /*Get the semaphore*/
    _oSem.vGet();
    _oSupervisonTimer.vStop(); /*Stop the timer*/
    tBool bSuccess=bIsSuccess();
    ETG_TRACE_USR3(("fc_sxm_tclDiagUsrDataClear::fc_sxm_trMsgCmdClearUserDataFinalize: bSuccess=%d",
                    bSuccess));
    _oSem.vPost();
    /*Handle the result*/
    fc_sxm_tclDiagDefset::instance()->vHandleResult(bSuccess ? fc_sxm_enDiagResult_Ok : fc_sxm_enDiagResult_Failed);
}

// check if action is running
tBool fc_sxm_tclDiagUsrDataClear::bIsActive() const {
    return _oSupervisonTimer.bIsRunning();
}

// check if we have received all results
tBool fc_sxm_tclDiagUsrDataClear::bIsReady() const {
    SXM_FOREACH_MAP_CONST(fc_sxm_tenServiceID, fc_sxm_tenDiagResult, iter, _mapResults) {
        if (iter->second == fc_sxm_enDiagResult_Pending) {
            return FALSE;
        }
    }
    return TRUE;
}

// check if all app have answered OK or no action
tBool fc_sxm_tclDiagUsrDataClear::bIsSuccess() const {
	/*Iterate through the map */
    SXM_FOREACH_MAP_CONST(fc_sxm_tenServiceID, fc_sxm_tenDiagResult, iter, _mapResults) {
        if (iter->second != fc_sxm_enDiagResult_Ok && iter->second != fc_sxm_enDiagResult_NoAction) {
            return FALSE;
        }
    }
    return TRUE;
}
