/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_diag_smslog.cpp
* @brief       SMS library logs handler. 
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_main.h"
#include "fc_sxm_tcl_diag_smslog.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_DIAGHANDLER
#include "trcGenProj/Header/fc_sxm_tcl_diag_smslog.cpp.trc.h"
#endif

#ifdef printf
#undef printf
#endif

tVoid fc_sxm_tclDiagSmsLog::vHandleLogCmd(fc_sxm_tenSmsLogCmd enCmd, string oSmsObjName) const {
    ETG_TRACE_USR4(("vHandleLogCmd:  %d", enCmd));

#if OSAL_LOG==1    
    if (enCmd==fc_sxm_enSmsLogCmd_List) {
        vHandleList();
        return;
    }
    OSAL_OBJECT_HDL hLog= hGetSmsLogHandle(oSmsObjName);
    if (OSAL_INVALID_OBJECT_HDL == hLog) {
        return;
    }
    switch (enCmd) {
        case fc_sxm_enSmsLogCmd_Clear:
            vHandleClear(hLog, oSmsObjName, TRUE);
            break;
        case fc_sxm_enSmsLogCmd_PrintToConsole:
            vHandlePrint(hLog, oSmsObjName, FALSE);
            break;
        case fc_sxm_enSmsLogCmd_PrintToFile:
            vHandlePrint(hLog, oSmsObjName, TRUE);
            break;
        case fc_sxm_enSmsLogCmd_Enable:
            vHandleClear(hLog, oSmsObjName, TRUE);
            break;
        case fc_sxm_enSmsLogCmd_Disable:
            vHandleClear(hLog, oSmsObjName, FALSE);
            break;
        case fc_sxm_enSmsLogCmd_List:
            break;
        case fc_sxm_enSmsLogCmd_LAST:
        default:
            ETG_TRACE_ERR(("vHandleLogCmd: invalid cmd  %d", enCmd));
                
            break;
    }
#else 
    ETG_TRACE_ERR(("vHandleLogCmd: Please compile with option OSAL_LOG==1 to use logging"));

#endif
}

OSAL_OBJECT_HDL fc_sxm_tclDiagSmsLog::hGetSmsLogHandle(string oSmsObjName) const {
    OSAL_OBJECT_HDL hLog=OSAL_INVALID_OBJECT_HDL;
#if OSAL_LOG==1    
    hLog = OSAL.hLogGetHandleByName(oSmsObjName.c_str());
    if (hLog == OSAL_INVALID_OBJECT_HDL) {
        ETG_TRACE_ERR(("could not get LogHandle for %s", oSmsObjName.c_str()));
    }
#endif
    return hLog;
}


tVoid fc_sxm_tclDiagSmsLog::vHandleList() const {
    printf("### AVAILABLE LOGS: ###\n");
    ETG_TRACE_ERR(("### AVAILABLE LOGS: ###"));
#if OSAL_LOG==1    
    OSAL.eLogList(FALSE);
#endif
    printf("### AVAILABLE LOGS END ###\n");
    ETG_TRACE_ERR(("### AVAILABLE LOGS END ###"));
    
}

tVoid fc_sxm_tclDiagSmsLog::vHandleClear(OSAL_OBJECT_HDL hLog, string oSmsObjName, tBool bEnable) const {
    // Disable the log...
    ETG_TRACE_USR4(("### CLEAR LOG %s ###", oSmsObjName.c_str()));
    printf("### CLEAR LOG %s ###\n", oSmsObjName.c_str());
#if OSAL_LOG==1    
    OSAL.vLogDisable(hLog);  
    // Clear the log...
    if (OSAL_SUCCESS != OSAL.eLogClear(hLog)) {
        ETG_TRACE_ERR(("could not clear log %s", oSmsObjName.c_str()));
    }
    else {
        if (bEnable) {
            // Enable the log
            OSAL.vLogEnable(hLog);
        }
    }
#endif
}

tVoid fc_sxm_tclDiagSmsLog::vHandlePrint(OSAL_OBJECT_HDL hLog, string oSmsObjName, tBool bToFile) const {
#if OSAL_LOG==1    
    char acData[128]; // arbitrary frame length
    tS32 s32Num=0;
    tU32 u32Pos = 0;
#ifdef FILE
#undef FILE
#endif
#ifdef fopen
#undef fopen
#endif
#ifdef stdout
#undef stdout
#endif
    FILE *outfile=stdout;
    if (bToFile) {
        string fileName=FC_SXM_SMS_TEMP_DIR_PATH;
        fileName+="/";
        fileName+=oSmsObjName;
        fileName+=".log";
        outfile=fopen(fileName.c_str(), "w");
        if (!outfile) {
            ETG_TRACE_ERR(("could not open file %s", fileName.c_str()));
            return;
        }
    }
        
    // We now have a log handle
    ETG_TRACE_ERR(("START LOG (bToFile=%d) %s", bToFile, oSmsObjName.c_str()));
    fprintf(outfile, "### START LOG %s ###\n", oSmsObjName.c_str());

    do
    {
        // Grab a frame at a time...
        s32Num = sizeof(acData) - 1;
        s32Num = OSAL.n32LogRead(hLog, &acData[0], u32Pos, s32Num);
             
        // Print if we have something at all
        if(s32Num > 0)
        {
            // NULL terminate frame
            acData[s32Num] = '\0';
                 
            // Increment the next position
            u32Pos += s32Num;
                 
            // print it
            fprintf(outfile, "%s", acData);
                 
            // Go get some more
            continue;
        }
        else if(s32Num < 0)
        {
            // We already know the handle is valid
            // so the log must be locked instead
            // wait and try again...
            OSAL.eTaskDelay(1);
            continue;
        }
             
    } while(s32Num != 0);
    ETG_TRACE_ERR(("### END LOG %30s (%d bytes)###", oSmsObjName.c_str(), u32Pos));
    fprintf(outfile, "\n### END LOG %s ###\n", oSmsObjName.c_str());
    if (bToFile) {
        fclose(outfile);
    }

    // clear the log to make space for new entries
    vHandleClear(hLog, oSmsObjName);
#endif
}



