/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_diag_log.cpp
* @brief       SXM Diagnosis handler for Diag log. Writes the Diagnosis DTC.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_main.h"

#include "fc_sxm_audio_fi.h"
#include "fc_sxm_tcl_audio_app.h"
#include "fc_sxm_tcl_audio_properties.h"

#include "fc_sxm_diag_if.h"
#include "fc_sxm_client_diaglog.h"
#include "fc_sxm_tcl_diag_log.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_DIAGHANDLER
#include "trcGenProj/Header/fc_sxm_tcl_diag_log.cpp.trc.h"
#endif


/*******************************************************************************
 *
 * FUNCTION: fc_sxm_tclDiagLog::
 *             fc_sxm_tclDiagLog()
 *
 * DESCRIPTION: Constructor .
 *
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
fc_sxm_tclDiagLog::fc_sxm_tclDiagLog():
m_eAntennaState(ANTENNA_STATE_UNKNOWN),m_bFirstUpdate(FALSE)
{
	ETG_TRACE_USR1(("fc_sxm_tclDiagLog constructor"));
}

/*******************************************************************************
 *
 * FUNCTION: fc_sxm_tclDiagLog::
 *             ~fc_sxm_tclDiagLog()
 *
 * DESCRIPTION: Destructor .
 *
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
fc_sxm_tclDiagLog::~fc_sxm_tclDiagLog()
{
	ETG_TRACE_USR1(("fc_sxm_tclDiagLog destructor"));

	m_eAntennaState = ANTENNA_STATE_UNKNOWN;
	m_bFirstUpdate = FALSE;
}


/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagLog::vProcess
 *(
 *  fc_sxm_trMsgFiDiaglogNextTestStatus const *prMsg
 *)
 *
 * DESCRIPTION: This is the function to handle the dialog test request
 *
 * PARAMETER: fc_sxm_trMsgFiDiaglogNextTestStatus : The pointer to the populated message
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagLog::vProcess
(
		fc_sxm_trMsgFiDiaglogNextTestStatus const *prMsg
)
{
	ETG_TRACE_USR1(("fc_sxm_tclDiagLog :fc_sxm_trMsgFiDiaglogNextTestStatus"));

	if(TRUE == (prMsg->oFiMsg.Status))
	{
		vSendMethodStartSaveTestResult(TRUE);
	}
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagLog::vProcess
 *(
 *  fc_sxm_trMsgFiDiaglogMethodResultSaveTest const *prMsg
 *) const
 *
 * DESCRIPTION: This is the function to handle the dialog test acknowledgment
 *
 * PARAMETER: fc_sxm_trMsgFiDiaglogMethodResultSaveTest : The pointer to the populated message
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagLog::vProcess
(
		fc_sxm_trMsgFiDiaglogMethodResultSaveTest const *prMsg
)
{
	//0=accepted,1=Rejected
	ETG_TRACE_USR4(("fc_sxm_tclDiagLog :MethodResultSaveTest result = %d",(prMsg->oFiMsg.Status.enType)));
}



/*******************************************************************************
 *
 * FUNCTION: tVoid fc_sxm_tclDiagLog::vSendMethodStartSaveTestResult
 *(
 *  tBool bUpdateAlways
 *)
 *
 * DESCRIPTION: This is the function to send the update to Diaglog
 *
 * PARAMETER: tBool bUpdateAlways : Indicate whether we need to update always the
 *                                  diaglog irrespective of other condition check
 *
 * RETURNVALUE: NONE
 *
 *******************************************************************************/
tVoid fc_sxm_tclDiagLog::vSendMethodStartSaveTestResult
(
		tBool bUpdateAlways
)
{
	ANTENNA_STATE_ENUM eAntennaState = DECODER.eAntennaState(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), 1);

	ETG_TRACE_USR4(("fc_sxm_tclDiagLog:vSendMethodStartSaveTestResult:bUpdateAlways=%d,eAntennaState=%d,m_eAntennaState=%d,m_bFirstUpdate=%d",
			bUpdateAlways,eAntennaState,m_eAntennaState,m_bFirstUpdate));

	if(
			(FALSE == m_bFirstUpdate) /*Diaglog wants the first update after bootup in any case*/

			||

			(eAntennaState != m_eAntennaState) /*Diaglog wants the update only if there is a change*/

			||

			(TRUE == bUpdateAlways) /*Diaglog wants the update in any case if its requested from Diaglog*/
	)
	{
		if((eAntennaState == ANTENNA_STATE_UNSUPPORTED) || (eAntennaState == ANTENNA_STATE_UNKNOWN))
		{
			ETG_TRACE_USR4(("Antenna state ignored"));
			return;
		}

		m_bFirstUpdate = TRUE;

		/*Update only if antenna state is changed from previous state.Exception only if bUpdateAlways=TRUE*/
		m_eAntennaState = eAntennaState;

		midw_diaglogfi_tclMsgSaveTestResultMethodStart oSxmData;

		midw_fi_tcl_TestResult oTestResultAntennaOpen;
		midw_fi_tcl_TestResult oTestResultAntennaShortGround;
		midw_fi_tcl_TestResult oTestResultAntennaShortBattery;

      // set default values
		oTestResultAntennaOpen.TroubleCode  =  ITC_XM_TUNER_ANT1_OPEN;
        oTestResultAntennaOpen.Result.enType =  midw_fi_tcl_e8_TestResult::FI_EN_NORESULT;

        oTestResultAntennaShortGround.TroubleCode  = ITC_XM_TUNER_ANT1_SHORT_GND;
        oTestResultAntennaShortGround.Result.enType =  midw_fi_tcl_e8_TestResult::FI_EN_NORESULT;

        oTestResultAntennaShortBattery.TroubleCode  = ITC_XM_TUNER_ANT1_SHORT_BAT;
        oTestResultAntennaShortBattery.Result.enType =  midw_fi_tcl_e8_TestResult::FI_EN_PASSED;

		switch(m_eAntennaState)
		{
		case ANTENNA_STATE_DETECTED:
			oTestResultAntennaOpen.Result.enType =  midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
			oTestResultAntennaShortGround.Result.enType =  midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
			break;
		case ANTENNA_STATE_SHORTED:
			oTestResultAntennaOpen.Result.enType =  midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
			oTestResultAntennaShortGround.Result.enType =  midw_fi_tcl_e8_TestResult::FI_EN_FAILED;
			break;
		case ANTENNA_STATE_NOT_DETECTED:
			oTestResultAntennaOpen.Result.enType =  midw_fi_tcl_e8_TestResult::FI_EN_FAILED;
			oTestResultAntennaShortGround.Result.enType =  midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
			break;
		default:
			break;
		}

		/*Update the list.Ready to be send to Diaglog*/
		oSxmData.TestResultList.TestResultList.push_back(oTestResultAntennaOpen);
		oSxmData.TestResultList.TestResultList.push_back(oTestResultAntennaShortGround);
		oSxmData.TestResultList.TestResultList.push_back(oTestResultAntennaShortBattery);

      ETG_TRACE_USR2(("fc_sxm_tclDiagLog :vSendMethodStartSaveTestResult Sending Fi message"));
		// call the send-function of the client-handler:
		fc_sxm_tclDiagLogClient::instance()->enSendFiMessage(oSxmData);
	}
	else
	{
		ETG_TRACE_USR4(("fc_sxm_tclDiagLog :vSendMethodStartSaveTestResult update not required"));
	}


}





