/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_diag_ext_mode.cpp
* @brief       SXM Diagnosis external diagnosis mode(also called CTI mode) handler.
*              SXM uses CTI mode to test X65 chip on the DUT. Mainly used in type approval.
*              Support from FC_SXM is to prevent any reset strategy of the X65 module by FC_SXM or SMS
*              when this test is conducted. The reset happens because when SXM connects external
*              application through CTI the X65 module of DUT will start communicating with them and 
*              FC_SXM -SMS finds the Module as unreponsive and initiate a reset of the X65 module.
*              Hence The implementation involves
*              1.)Completely shutting down the SMS.
*              2.)Start just the X65 module.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_main.h"

#include "fc_sxm_service_sxm_audio.h"  
#include "fc_sxm_tcl_sms_init.h"
#include "fc_sxm_tcl_diag_ext_mode.h"
#include "fc_sxm_tcl_advisories.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_DIAGHANDLER
#include "trcGenProj/Header/fc_sxm_tcl_diag_ext_mode.cpp.trc.h"
#endif

/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagExtModeActivity::vRun()
*
* DESCRIPTION: This method is called after the requested activity is done.
*              Here after the activity of shutting down SMS,we start the X65 module
*
* PARAMETER: const *prMsg : fc_sxm_trMsgDiagRestoreSmsDb 
*
* RETURNVALUE: NONE
*
*******************************************************************************/
tVoid fc_sxm_tclDiagExtModeActivity::vRun() 
{
   ETG_TRACE_USR1(("fc_sxm_tclDiagExtModeActivity::vRun START"));

   fc_sxm_tclSMSInit::instance()->vStartX65Module();

   /* send the answer to the client using adressing-infomation: */
   fc_sxm_tclDiagExtDiagnosisMode::instance() -> vHandleExtDiagModeResult();
}



/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagExtDiagnosisMode::fc_sxm_tclDiagExtDiagnosisMode()
*
* DESCRIPTION: Construcote
*
* PARAMETER: NONE
*
* RETURNVALUE: NONE
*
*******************************************************************************/
fc_sxm_tclDiagExtDiagnosisMode::fc_sxm_tclDiagExtDiagnosisMode()
:m_bIsExtDiagModeOn(FALSE)
{
   ETG_TRACE_USR1(("fc_sxm_tclDiagExtDiagnosisMode::constructor"));
}



/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagExtDiagnosisMode::vProcess
*(
*  fc_sxm_trMsgDiagExtMode const *prMsg
*)
*
* DESCRIPTION: This method is called on receiving trigger for external diag mode from HMI.
*
* PARAMETER: const *prMsg : fc_sxm_trMsgDiagExtMode 
*
* RETURNVALUE: NONE
*
*******************************************************************************/
tVoid fc_sxm_tclDiagExtDiagnosisMode::vProcess(fc_sxm_trMsgDiagExtMode const *prMsg)
{
   ETG_TRACE_USR1(("fc_sxm_tclDiagExtDiagnosisMode::entered"));

   m_rAdressing = prMsg->rAdressing;
   //Check : request is to enter the CTI mode
   if( (midw_ext_fi_tcl_e8_SxmExtDiagMode::FI_EN_FC_SXM_EXTERNAL_DIAG_MODE_START) == (prMsg->oFiMsg.Mode.enType))
   {
      ETG_TRACE_USR2(("fc_sxm_tclDiagExtDiagnosisMode::ENTER"));

      /*Set to true so that the state handler can keep the 
        Audio service available for this special case*/
      m_bIsExtDiagModeOn = TRUE;

      if (!m_oExtDiagModeActivity.bRequest())
      {
         ETG_TRACE_ERR(("fc_sxm_tclDiagExtDiagnosisMode: bRequest failed"));
      }
	  else
	  {
		 fc_sxm_tclAdvisories::instance()->vSetSXMInitializingAdvisory(); 
	  }
   } //Check : request is to exit the CTI mode
   else if((midw_ext_fi_tcl_e8_SxmExtDiagMode::FI_EN_FC_SXM_EXTERNAL_DIAG_MODE_EXIT) == (prMsg->oFiMsg.Mode.enType))
   {
      ETG_TRACE_USR2(("fc_sxm_tclDiagExtDiagnosisMode::EXIT"));
      //Stop the X65 module as we have started it for Diag mode
      fc_sxm_tclSMSInit::instance()->vStopX65Module();
      /*Inform state handler that the External diag mode has ended*/
      m_bIsExtDiagModeOn = FALSE;

      m_oExtDiagModeActivity.vOnDone();
      /* send the answer to the client using adressing-infomation: */
      vHandleExtDiagModeResult();
	  fc_sxm_tclAdvisories::instance()->vClearSXMInitializingAdvisory();
   }
   else //The request is unknown
   {
      ETG_TRACE_ERR(("fc_sxm_tclDiagExtDiagnosisMode::UNKNOWN REQUEST"));
   }

}


/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagExtDiagnosisMode::vHandleDbRestoreResult
*
*
* DESCRIPTION: This function is used to send the method result for external diagnosis mode
*
* PARAMETER: NONE
*
* RETURNVALUE: NONE
*
*******************************************************************************/
tVoid fc_sxm_tclDiagExtDiagnosisMode::vHandleExtDiagModeResult()
{
	ETG_TRACE_USR4(("fc_sxm_tclDiagDefset::vHandleExtDiagModeResult"));

	midw_ext_sxm_audiofi_tclMsgExternalDiagnosisModeMethodResult oFiTxObj;
    fc_sxm_tclAudioService::instance()->enSendFiMessage(m_rAdressing, oFiTxObj);
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclDiagExtDiagnosisMode::bGetExtDiagModeStatus
*
*
* DESCRIPTION: This function is used to send the status of external diag mode
*
* PARAMETER: NONE
*
* RETURNVALUE: tBool
*              TRUE : External Diag mode ON
*              FALSE : External Diag mode OFF
*
*******************************************************************************/
tBool fc_sxm_tclDiagExtDiagnosisMode::bGetExtDiagModeStatus()
{
   return m_bIsExtDiagModeOn;
}

