/************************************************************************
* FILE:         fc_dataservices_DABClientHandler.cpp
* PROJECT:      G3g
* SW-COMPONENT: fc_dataservices
* ----------------------------------------------------------------------
*
* DESCRIPTION:  CCA, DBus Message Handler Class for 
*				fc_dataservices component
*
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
* HISTORY:
* Date      | Author            | Modification
* 24.09.2014 | Raghavendra Nannuru Vutkuru (RBEI/ECV2)    | initial version
* 13.07.2015 | Raghavendra Nannuru Vutkuru (RBEI/ECV2)    | changed name
* to fc_dataservicesDABClientHandler and moved out DBus Handling to
* fc_dataservice_DBusStub
*************************************************************************/

#include "fc_dataservices_DABClientHandler.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC 
#include "etrace_if.h"

#define ETG_S_IMPORT_INTERFACE_GENERIC 
#include "etg_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DATASRVCS_MSG_HANDLER
#include "trcGenProj/Header/fc_dataservices_DABClientHandler.cpp.trc.h" 
#endif

namespace fc_dataservices { namespace Server {		

	/******************************************************************************
	*
	* FUNCTION: fc_dataservices_DABClientHandler::fc_dataservices_DABClientHandler 
	*
	* DESCRIPTION: Constructor.
	*			   Creates DSM handler.
	*			   Sends property Upreg for required messages to DAB
	*
	* PARAMETER: const ::boost::shared_ptr< asf::core::Proxy >& proxy
	*
	* RETURNVALUE: None.
	*
	*******************************************************************************/
	fc_dataservices_DABClientHandler::fc_dataservices_DABClientHandler() :    
m_pDABFI( ::com::dab::MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy::createProxy ("ccaFIPort", *this)){

	ETG_TRACE_USR4((" fc_dataservices_DABClientHandler:"));

	m_pDispatcher = tclDispatcher::instance();
	m_pDSMHandler = fc_dataservices_DSMHandler::instance();

	m_pDispatcher->vStartDispatcherThread();
	bMsgInProcess = FALSE;
	bDABFMlinkingStatus = FALSE;	

}

/******************************************************************************
*
* FUNCTION: fc_dataservices_DABClientHandler::~fc_dataservices_DABClientHandler()
*
* DESCRIPTION: Destructor.Deletes DSM handler 
*
* PARAMETER: None
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_dataservices_DABClientHandler::~fc_dataservices_DABClientHandler() {	 
	ETG_TRACE_USR4((" ~fc_dataservices_DABClientHandler"));
	vFreeObjects();
	
}

/******************************************************************************
*
* FUNCTION: fc_dataservices_DABClientHandler::onAvailable
*
* DESCRIPTION: Callback when DAB service is aviailable
*
* PARAMETER: const ::boost::shared_ptr< asf::core::Proxy >& proxy,
*			 const ServiceStateChange &stateChange
*			 
*
* RETURNVALUE: None.
*
*******************************************************************************/
void fc_dataservices_DABClientHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const ServiceStateChange &stateChange){

	(tVoid)proxy;
	(tVoid)stateChange;

	ETG_TRACE_USR4((" DAB Service is available"));
	
	if (m_pDABFI)
	{
	/*subscribe to DAB service property upreg*/
	m_pDABFI->sendFID_DAB_G_ADR_MSGUpReg(*this);
	m_pDABFI->sendFID_DAB_G_CHANNEL_INFOUpReg(*this);
	m_pDABFI->sendFID_DAB_G_LOCAL_OFFSETUpReg(*this);
	}
	if(m_pDSMHandler!= NULL){
	m_pDSMHandler->vStartDSMTriggerTimer();
	m_pDSMHandler->vSetDABClientAvailability(TRUE);
	}
}

/******************************************************************************
*
* FUNCTION: fc_dataservices_DABClientHandler::onFID_DAB_G_ADR_MSGError
*
* DESCRIPTION: Callback when DAB service is not aviailable
*
* PARAMETER: const ::boost::shared_ptr< asf::core::Proxy >& proxy,
*			 const ServiceStateChange &stateChange			 
*
* RETURNVALUE: None.
*
*******************************************************************************/
void fc_dataservices_DABClientHandler::onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const ServiceStateChange &stateChange){
	(tVoid)proxy;
	(tVoid)stateChange;

	ETG_TRACE_USR4((" DAB Service is NOT available"));
	
	/*release upreg if service is not available*/
	if (m_pDABFI)
	{
		m_pDABFI-> sendFID_DAB_G_ADR_MSGRelUpRegAll ();
		m_pDABFI-> sendFID_DAB_G_CHANNEL_INFORelUpRegAll ();
		m_pDABFI-> sendFID_DAB_G_LOCAL_OFFSETRelUpRegAll ();
	}
	if(m_pDSMHandler!= NULL){
	m_pDSMHandler->vStopDSMTriggerTimer();
	m_pDSMHandler->vSetDABClientAvailability(FALSE);
	}

}

/******************************************************************************
*
* FUNCTION: fc_dataservices_DABClientHandler::onFID_DAB_G_ADR_MSGError
*
* DESCRIPTION: handles ADR msg property error from DAB Tuner
*
* PARAMETER: &proxy ,&error
*			 
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid fc_dataservices_DABClientHandler::onFID_DAB_G_ADR_MSGError(
	const ::boost::shared_ptr<  ::com::dab::MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& proxy,
	const boost::shared_ptr<  ::com::dab::MIDW_EXT_DABTUNER_FI::FID_DAB_G_ADR_MSGError >& error){

		(tVoid)error;
		(tVoid)proxy;

}

/******************************************************************************
*
* FUNCTION: fc_dataservices_DABClientHandler::onFID_DAB_S_DSM_MSGError
*
* DESCRIPTION: handles DSM msg property error from DAB Tuner
*
* PARAMETER: &proxy ,&error
*			 
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid fc_dataservices_DABClientHandler::onFID_DAB_S_DSM_MSGError(
	const ::boost::shared_ptr< ::com::dab::MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& proxy,
	const boost::shared_ptr< ::com::dab::MIDW_EXT_DABTUNER_FI::FID_DAB_S_DSM_MSGError >& error) {
		(tVoid)error;
		(tVoid)proxy;	

}

/******************************************************************************
*
* FUNCTION: fc_dataservices_DABClientHandler::onFID_DAB_S_DSM_MSGResult
*
* DESCRIPTION: handles DSM msg result from DAB Tuner
*
* PARAMETER: &proxy ,&result
*			 
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid fc_dataservices_DABClientHandler::onFID_DAB_S_DSM_MSGResult(
	const ::boost::shared_ptr< ::com::dab::MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& proxy, 
	const boost::shared_ptr< ::com::dab::MIDW_EXT_DABTUNER_FI::FID_DAB_S_DSM_MSGResult >& result){
		(tVoid)proxy;
		(tVoid)result;

}

/******************************************************************************
*
* FUNCTION: fc_dataservices_DABClientHandler::onFID_DAB_G_ADR_MSGStatus
*
* DESCRIPTION: Sends message received from DAB Tuner to DSM handler
*
* PARAMETER: &proxy ,&status
*			 
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid fc_dataservices_DABClientHandler::onFID_DAB_G_ADR_MSGStatus(
	const ::boost::shared_ptr< ::com::dab::MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& proxy,
	const ::boost::shared_ptr<  ::com::dab::MIDW_EXT_DABTUNER_FI::FID_DAB_G_ADR_MSGStatus >& status){
		(tVoid)proxy;


		ETG_TRACE_USR4((" Received onFID_DAB_G_ADR_MSGStatus"));

		tU16 u16FktId= status->getTunerStatus().getU16FktID();

		tU32 u32MsgLength=(tU32)(status->getTunerStatus().getMsgData().size());

		vector<tU8> vectorData= status->getTunerStatus().getMsgData();



		ETG_TRACE_USR4((" Received Msg FktID %x Length %d ",u16FktId, u32MsgLength));

		if((u32MsgLength>0)&&
			(m_pDispatcher!= OSAL_NULL)){

				ADRMessage rMessage;
				rMessage.u16FktId = u16FktId;
				rMessage.u32MsgLength = u32MsgLength;
				rMessage.vSetData(vectorData);	

				bMsgInProcess = TRUE;
				m_pDispatcher->m_rQueue.push(rMessage);
				//ETG_TRACE_USR4((" pushed Msg FktID sizeof queue %d",((m_pDispatcher->m_rQueue).size()) ));
				int rc = pthread_cond_signal (&m_pDispatcher->m_rCondition); 									
				ETG_TRACE_USR4((" Cond signal Result %d",rc ));		
				bMsgInProcess = FALSE;

		}
}

/******************************************************************************
*
* FUNCTION: fc_dataservices_DABClientHandler::vonDSMMessage
*
* DESCRIPTION: Sends requests from DSM to DAB tuner
*
* PARAMETER: ::com::dab::midw_ext_dabtuner_fi_types::T_DAB_ADRMsg*			 
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid fc_dataservices_DABClientHandler::vonDSMMessage( ::com::dab::midw_ext_dabtuner_fi_types::T_DAB_ADRMsg rMsg)
{

	ETG_TRACE_USR4((" Received vonDSMMessage FktID %x ",rMsg.getU16FktID ()));		

	m_pDABFI->sendFID_DAB_S_DSM_MSGStart(*this,rMsg);

}

/******************************************************************************
*
* FUNCTION: fc_dataservices_DABClientHandler::vUpdateStationLogoList
*
* DESCRIPTION: Sends logo list to CCA clients and DBUS clients
*
* PARAMETER: ::vector< ::com::dab::midw_ext_dabtuner_fi_types::T_DAB_LogoElement> 
*			 rLogoData	 
*
* RETURNVALUE: None.
*
*******************************************************************************/

tVoid fc_dataservices_DABClientHandler::vUpdateStationLogoList(const vector< ::com::dab::midw_ext_dabtuner_fi_types::T_DAB_LogoElement> &rLogoData,const vector<tclDABServiceManager::tEPGContentIdDataType> &rDSList){	

	vector< ::com::dab::midw_ext_dabtuner_fi_types::T_DAB_ContentID> rdatalist ;
	rdatalist.resize(rDSList.size());
	for(unsigned int i=0; i<rDSList.size();i++){
		::com::dab::midw_ext_dabtuner_fi_types::T_DAB_ContentID content((rDSList[i].r_service & 0xFFFF),rDSList[i].b_scids);		
		rdatalist.push_back(content);
	}
	m_pDABFI->sendFID_DAB_S_DATASERVICES_DATAStart(*this,rLogoData, rdatalist );

}

/******************************************************************************
*
* FUNCTION: fc_dataservices_DABClientHandler::onFID_DAB_G_CHANNEL_INFOStatus
*
* DESCRIPTION: gives current playing service info
*
* PARAMETER: &proxy ,&status 
*
* RETURNVALUE: None.
*
*******************************************************************************/

tVoid fc_dataservices_DABClientHandler::onFID_DAB_G_CHANNEL_INFOStatus(const ::boost::shared_ptr< ::com::dab::MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& proxy,
	const boost::shared_ptr< ::com::dab::MIDW_EXT_DABTUNER_FI::FID_DAB_G_CHANNEL_INFOStatus >& status){
		(tVoid)proxy;

		::com::dab::midw_ext_dabtuner_fi_types::T_DAB_ChannelInfo rChnInfo;
		rChnInfo = status->getChannelInfo ();
		tU32 u32ServiceId = rChnInfo.getU32ServId();
		
		tBool bfmfollowing= rChnInfo.getBFmServiceFollowActive();
		/*if DAB->FM linking is active, clear SLS after 120sec */
		if(bDABFMlinkingStatus != bfmfollowing)
		{
			bDABFMlinkingStatus = bfmfollowing;
			/*start timer if DAB->FM linking is active*/
			if(bDABFMlinkingStatus)
			{
				SLSTriggerTimer.start(*this,120000,Timer::Once);
			}/*stop timer if DAB->FM linking is not active and if timer is already started*/
			else if(SLSTriggerTimer.isActive())
			{
				SLSTriggerTimer.stop();
			}
		}
		
		string radiotext = rChnInfo.getRRadioText().getSRadioTextData();
		/*if DAb signal is lost clear SLS based on radio text which will be cleared after 120 sec*/
		if((m_pDSMHandler != NULL)&&
			(u32ServiceId !=0)&&
			((radiotext.empty())&&
			 (rChnInfo.getEnState ()== ::com::dab::midw_ext_dabtuner_fi_types::T_e8_DAB_ChannelInfoState__DAB_UNAVAIL)
			)
		  ){

				m_pDSMHandler->vTriggerSlideUpdate(SLIDE_STATE_CLEAR);

				ETG_TRACE_USR4((" Clear slide triggered	"));
				
		} 

}
/******************************************************************************
*
* FUNCTION: fc_dataservices_DABClientHandler::onFID_DAB_G_LOCAL_OFFSETStatus
*
* DESCRIPTION: gives local offset to GMT 
*
* PARAMETER: &proxy ,&status 
*
* RETURNVALUE: None.
*
*******************************************************************************/
void fc_dataservices_DABClientHandler::onFID_DAB_G_LOCAL_OFFSETStatus(const ::boost::shared_ptr< ::com::dab::MIDW_EXT_DABTUNER_FI::MIDW_EXT_DABTUNER_FIProxy >& proxy, 
		const ::boost::shared_ptr< ::com::dab::MIDW_EXT_DABTUNER_FI::FID_DAB_G_LOCAL_OFFSETStatus >& status){
	(tVoid)proxy;
		tS64 localOffset = status->getLocalOffset ();
		ETG_TRACE_USR4((" fc_dataservices_DABClientHandler offset %u :",localOffset));
		if(m_pDSMHandler != NULL)
		{
			m_pDSMHandler->setLocalTimeOffset(localOffset);
		}
	}
/******************************************************************************
*
* FUNCTION: fc_dataservices_DABClientHandler::onExpired
*
* DESCRIPTION: gives current playing service info
*
* PARAMETER:  Timer& ,  ::boost::shared_ptr<TimerPayload>
*
* RETURNVALUE: None.
*
*******************************************************************************/
void fc_dataservices_DABClientHandler::onExpired(Timer& timer, ::boost::shared_ptr<TimerPayload> data){
	(tVoid)timer;
	(tVoid)data;
	ETG_TRACE_USR4(("SLS clear trigger timer expired"));
	m_pDSMHandler->vTriggerSlideUpdate(SLIDE_STATE_CLEAR);
}
/******************************************************************************
*
* FUNCTION: fc_dataservices_DABClientHandler::vCloseDataBase
*
* DESCRIPTION: handles closing of database and setting of status variables
*
* PARAMETER: 
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid fc_dataservices_DABClientHandler::vCloseDataBase(){
	
	if((m_pDSMHandler != NULL)&&
		(!m_pDSMHandler->bGetDispatcherStatus())){
		ETG_TRACE_USR4((" Direct closure of Database" ));				
		m_pDSMHandler->vSetDBstatus(FALSE);
		m_pDSMHandler->m_pDSM->oDB()->vClose();
		
	}
	else{
		if(m_pDispatcher!= NULL){
			ADRMessage rMessage;
			rMessage.enMsgType = MESSAGE_CLOSE_DB;
			rMessage.u32MsgLength = 0;	
			m_pDispatcher->m_rQueue.push(rMessage);
	//		ETG_TRACE_USR4((" pushed Msg Close DB Event %d",((m_pDispatcher->m_rQueue).size()) ));
			int rc = pthread_cond_signal (&m_pDispatcher->m_rCondition); 									
			ETG_TRACE_USR4((" Cond signal Result %d",rc ));		
		 
		}
	}
}
tVoid fc_dataservices_DABClientHandler::vFreeObjects(){
	if(OSAL_NULL !=m_pDSMHandler)
	{
		OSAL_DELETE m_pDSMHandler;
		m_pDSMHandler=OSAL_NULL;
		ETG_TRACE_USR4((" END ~m_pDSMHandler"));
	}
	if(OSAL_NULL !=m_pDispatcher)
	{
		ETG_TRACE_USR4((" STARt ~m_pDispatcher"));
		OSAL_DELETE m_pDispatcher;
		m_pDispatcher=OSAL_NULL;
		ETG_TRACE_USR4((" END ~m_pDispatcher"));
	}
	
}
}}
