#ifndef AARSLIB_H
#define AARSLIB_H
/*!
*******************************************************************************
  @file         AARSLIB.h

  @brief        <b>AaRS message library</b>

*******************************************************************************
*/
#include <stdio.h>
#ifndef AARSLIB__OMIT_AARSLIBCONFIG
#include "aarsLibConfig.h"
#endif /* AARSLIB__OMIT_AARSLIBCONFIG */

#define MACRO_AARSLIB_DEFINE_METHOD(_opcode_,_name_,_namestr_,_baseclass_)               \
    static const tU16 OP_CODE = _opcode_;                                                   \
    static const tChar* methodname() { return _namestr_; }                                        \
    t##_name_( tU32 msglen, tU8* msgbody, tBool freemsgbodyflag = false ) : _baseclass_(OP_CODE, msglen, msgbody, freemsgbodyflag) { }      \
    t##_name_( tMessage& inmsg ) : _baseclass_(inmsg) { }                                   \
    t##_name_( ) : _baseclass_(OP_CODE, 0, AARSLIB_NULL, false) { }

#define MACRO_AARSLIB_SIMPLE_COMMAND_SETGET(_type_,_name_,_basetype_,_start_,_size_)     \
    private:                                                                                \
    tU32  start__##_name_() { return _start_; }                                             \
    tU32  size__##_name_() { return _size_; }                                               \
    tU32  end__##_name_() { return start__##_name_() + size__##_name_(); }                  \
    public:                                                                                 \
    tVoid  set__##_name_( _type_ value ) { set_##_basetype_(_start_,(_basetype_)value,_size_); }        \
    _type_ _name_( void ) { return (_type_)get_##_basetype_(_start_); }

#define MACRO_AARSLIB_SIMPLE_RESPONSE_SETGET(_type_,_name_,_basetype_,_start_,_size_)    \
    private:                                                                                \
    tU32  start__##_name_() { return _start_; }                                             \
    tU32  size__##_name_() { return _size_; }                                               \
    tU32  end__##_name_() { return start__##_name_() + size__##_name_(); }                  \
    private:                                                                                \
    public:                                                                                 \
    _type_ _name_( void ) { return (_type_)get_##_basetype_(_start_); }

namespace AaRSLib
{

namespace MSG
{

/*!
*******************************************************************************
  @brief        <b>class tMessage</b>
*******************************************************************************
*/
class tMessage
{
protected:
    tU32 _msglen;
    tU8  _msgbody_allocated;
    tU8* _msgbody;
    tU16 _msgopcode;
    tU8  _optype;
    tU8  _instanceid;

public:
    /*!
    ***************************************************************************
      @brief        <b>constructor</b>
    ***************************************************************************
    */
    tMessage( tU16 msgopcode, tU32 msglen, tU8* msgbody, tBool freemsgbodyflag = false, tU8 instanceid = 0 )
    {
        _optype = 0;
        _msgopcode = msgopcode;
        _msglen = msglen;
        _msgbody = msgbody;
        _instanceid = instanceid;
        if( _msglen == 0 )
        {
            _msgbody = AARSLIB_NULL;
        }
        if((_msgbody == AARSLIB_NULL )||( freemsgbodyflag == true ))
        {
            _msgbody_allocated = 1; // store if body shall be freed while destructing
        }
        else
        {
            _msgbody_allocated = 0;
        }
    }

    /*!
    ***************************************************************************
      @brief        <b>constructor</b>
    ***************************************************************************
    */
    tMessage( const tMessage& inmsg )
    {
        _msglen = 0;
        _msgbody = AARSLIB_NULL;
        _msgbody_allocated = 1;
        _msgopcode = inmsg.opcode();
        _optype = inmsg.optype();
        _instanceid = inmsg.instanceid();
        set_tPU8(0,inmsg.get_tPU8(0),inmsg.size());
    }

#if 0
	/*
	 * removed for fixed warning with double copy constructors
	 */
    /*!
    ***************************************************************************
      @brief        <b>constructor</b>
    ***************************************************************************
    */
    tMessage( tMessage& inmsg )
    {
        _msglen = 0;
        _msgbody = AARSLIB_NULL;
        _msgbody_allocated = 1;
        _msgopcode = inmsg.opcode();
        _optype = inmsg.optype();
        _instanceid = inmsg.instanceid();
        set_tPU8(0,inmsg.get_tPU8(0),inmsg.size());
    }
#endif

    /*!
    ***************************************************************************
      @brief        <b>assignment operator</b>
    ***************************************************************************
    */
    tMessage & operator = (const tMessage &_otMessage)
    {
       if(this != &_otMessage)
       {
           if(( _msgbody != AARSLIB_NULL ) && ( _msgbody_allocated == 1 ))
           {
               AARSLIB_FREE((tVoid*)_msgbody);
               _msgbody = AARSLIB_NULL;
           }
           _msglen = 0;
		   _msgopcode = _otMessage.opcode();
		   _optype = _otMessage.optype();
		   _instanceid = _otMessage.instanceid();
		   set_tPU8(0,_otMessage.get_tPU8(0),_otMessage.size());
       }

       return *this;
    }

#if 0
	/*
	 * This non-const copy constructor might be needed for avoding
	 * an implicit one, defined by the compiler.
	 * But, I saw, without this one, the const variant is needed.
	 */
    /*!
    ***************************************************************************
      @brief        <b>assignment operator</b>
    ***************************************************************************
    */
    tMessage & operator = (tMessage &_otMessage)
    {
       if(this != &_otMessage)
       {
           if(( _msgbody != AARSLIB_NULL ) && ( _msgbody_allocated == 1 ))
           {
               AARSLIB_FREE((tVoid*)_msgbody);
               _msgbody = AARSLIB_NULL;
           }
           _msglen = 0;
           _msgopcode = _otMessage.opcode();
           _optype = _otMessage.optype();
           _instanceid = _otMessage.instanceid();
           set_tPU8(0,_otMessage.get_tPU8(0),_otMessage.size());
       }

       return *this;
    }
#endif

    /*!
    ***************************************************************************
      @brief        <b>constructor</b>
    ***************************************************************************
    */
    tMessage( )
    {
        _msglen = 0;
        _msgbody = AARSLIB_NULL;
        _msgbody_allocated = 0;
    }

    /*!`
    ***************************************************************************
      @brief        <b>destructor</b>
    ***************************************************************************
    */
    ~tMessage()
    {
        if(( _msgbody != AARSLIB_NULL ) && ( _msgbody_allocated == 1 ))
        {
            AARSLIB_FREE((tVoid*)_msgbody);
            _msgbody = AARSLIB_NULL;
        }
        _msglen = 0;
        _msgopcode = 0;
    }

#ifndef CONFIG_AARSLIB_DONT_USE_STD_STRING
    /*!
    ***************************************************************************
      @brief        <b>function toString</b>
    ***************************************************************************
    */
    std::string toString()
    {
        std::string str = string_sprintf("optype=%d opcode=0x%04x msgbody=0x",
                                         optype(),
                                         opcode());
        for(int i=0;i<(int)_msglen;i++)
        {
            str += string_sprintf("%02x",_msgbody[i]);
        }
        return str;
    }
#endif

    /*!
    ***************************************************************************
      @brief        <b>function size</b>
    ***************************************************************************
    */
    tU8 optype() const
    {
        return _optype;
    }

    void set__optype(tU8 optype)
    {
        _optype = optype;
    }

    /*!
    ***************************************************************************
      @brief        <b>function size</b>
    ***************************************************************************
    */
    tU32 size() const
    {
        return _msglen;
    }

    /*!
    ***************************************************************************
      @brief        <b>function size</b>
    ***************************************************************************
    */
    tPU8 body() const
    {
        return _msgbody;
    }

    /*!
    ***************************************************************************
      @brief        <b>function fromByteArray</b>
    ***************************************************************************
    */
    tU16 opcode() const
    {
        return _msgopcode;
    }

    /*!
    ***************************************************************************
      @brief        <b>function msgbody</b>
    ***************************************************************************
    */
    tPU8 msgbody() const
    {
        return _msgbody;
    }

    /*!
    ***************************************************************************
      @brief        <b>function instanceid</b>
    ***************************************************************************
    */
    tU8 instanceid() const
    {
        return _instanceid;
    }

    /*!
    ***************************************************************************
      @brief        <b>function instanceid</b>
    ***************************************************************************
    */
    tVoid set__instanceid(tU8 instanceid)
    {
        _instanceid = instanceid;
    }

    /*!
    ***************************************************************************
      @brief        <b>function methodname</b>
    ***************************************************************************
    */
    tChar* methodname();

    /*!
    ***************************************************************************
      @brief        <b> == operator</b>
    ***************************************************************************
    */
    tBool operator==(const tMessage& rhs) const
    {
        if( opcode() != rhs.opcode() )
        {
            return false;
        }
        if( size() != rhs.size() )
        {
            return false;
        }
        if( instanceid() != rhs.instanceid() )
        {
            return false;
        }
        if( size()==0 && rhs.size()==0 )
        {
            return true;
        }
        {
            tU8* a = body();
            tU8* b = rhs.body();
            int n = size();
            if( memcmp(a,b,n) == 0 )
            {
                return true;
            }
        }
        return false;
    }
	
#ifndef CONFIG_AARSLIB_DONT_USE_STD_STRING
    /*!
    ***************************************************************************
      @brief        <b>function toString</b>
    ***************************************************************************
    */
    std::string string_sprintf(const std::string fmt, ...) {
        int size = 100;
        std::string str;
        va_list ap;
        while (1) {
            str.resize(size);
            va_start(ap, fmt);
            int n = vsnprintf((char *)str.c_str(), size, fmt.c_str(), ap);
            va_end(ap);
            if (n > -1 && n < size) {
                str.resize(n);
                return str;
            }
            if (n > -1)
                size = n + 1;
            else
                size *= 2;
        }
        return str;
    }
#endif


    //Implemented for variable length messages
    void setMsgBody(tU8 *_msgbuf)
    {
          _msgbody = _msgbuf;
    }


    /*!
    ***************************************************************************
      @brief        <b>function set_MsgLength</b>
    ***************************************************************************
    */
    void set_MsgLength( tU32 msglen )
    {
        if(( _msgbody_allocated == 0 )&&(_msgbody == AARSLIB_NULL ))
        {
            /* when no body buffer was allocated before, then realloc it.
             */
            _msgbody_allocated = 1;
        }
        if( _msgbody_allocated == 1 )
        {
            if( _msglen != msglen )
            {
                if( _msgbody != AARSLIB_NULL )
                {
                    _msgbody = (tU8*)AARSLIB_REALLOC(_msgbody,msglen);
                }
                else
                {
                    _msgbody = (tU8*)AARSLIB_MALLOC(msglen);
                }
                if( _msgbody == AARSLIB_NULL )
                {
                    _msglen = 0;
                }
                else
                {
                    _msglen = msglen;
                }
            }
        }
    }

    /*!
    ***************************************************************************
      @brief        <b>function set_tPChar</b>
    ***************************************************************************
    */
    tU32 set_tPChar( tU32 offset, tPChar v, tU32 vs )
    {
        if( _msglen < (offset+vs) )
        {
            set_MsgLength(offset+vs);
        }
        if( _msgbody != AARSLIB_NULL )
        {
            if( _msglen >= (offset+vs) )
            {
                AARSLIB_MEMCOPY(&_msgbody[offset],v,vs);
            }
        }
        return offset + vs;
    }

    /*!
    ***************************************************************************
      @brief        <b>function set_tPU8</b>
    ***************************************************************************
    */
    tU32 set_tPU8( tU32 offset, tPU8 v, tU32 vs )
    {
        if( vs != 0 )
        {
            if( _msglen < (offset+vs) )
            {
                set_MsgLength(offset+vs);
            }
            if( _msgbody != AARSLIB_NULL )
            {
                if( _msglen >= (offset+vs) )
                {
                    AARSLIB_MEMCOPY(&_msgbody[offset],v,vs);
                }
            }
        }
        return offset + vs;
    }

    /*!
    ***************************************************************************
      @brief        <b>function set_tU8</b>
    ***************************************************************************
    */
    tU32 set_tU8( tU32 offset, tU8 v, tU32 vs )
    {
        if( _msglen < offset+1 )
        {
            set_MsgLength(offset+1);
        }
        if( _msgbody != AARSLIB_NULL )
        {
            if( _msglen >= (offset+vs) )
            {
                _msgbody[offset] = v;
            }
        }
        return offset + 1;
    }
    tU32 set_tU8( tU32 offset, tU8 v )
    {
        return set_tU8(offset,v,1);
    }

    /*!
    ***************************************************************************
      @brief        <b>function set_tS8</b>
    ***************************************************************************
    */
    tU32 set_tS8( tU32 offset, tS8 v, tU32 vs )
    {
        if( _msglen < offset+1 )
        {
            set_MsgLength(offset+1);
        }
        if( _msgbody != AARSLIB_NULL )
        {
            if( _msglen >= (offset+vs) )
            {
                _msgbody[offset] = v;
            }
        }
        return offset + 1;
    }
    tU32 set_tS8( tU32 offset, tS8 v )
    {
        return set_tS8(offset,v,1);
    }

    /*!
    ***************************************************************************
      @brief        <b>function set_tU16</b>
    ***************************************************************************
    */
    tU32 set_tU16( tU32 offset, tU16 v, tU32 vs )
    {
        (tVoid)vs;
        set_tU8(offset+0,(tU8)(v>>8)&0xff,1);
        set_tU8(offset+1, (tU8)(v>>0)&0xff,1);
        return offset + 2;
    }
    tU32 set_tU16( tU32 offset, tU16 v )
    {
        return set_tU16(offset,v,2);
    }

    /*!
    ***************************************************************************
      @brief        <b>function set_tS16</b>
    ***************************************************************************
    */
    tU32 set_tS16( tU32 offset, tS16 v, tU32 vs )
    {
        (tVoid)vs;
        set_tU8(offset+0, (tU8)(v>>8)&0xff,1);
        set_tU8(offset+1, (tU8)(v>>0)&0xff,1);
        return offset + 2;
    }
    tU32 set_tS16( tU32 offset, tU16 v )
    {
        return set_tS16(offset,v,2);
    }

    /*!
    ***************************************************************************
      @brief        <b>function set_tU32</b>
    ***************************************************************************
    */
    tU32 set_tU32( tU32 offset, tU32 v, tU32 vs )
    {
        (tVoid)vs;
        set_tU8(offset+0, (tU8)(v>>24)&0xff,1);
        set_tU8(offset+1,(v>>16)&0xff,1);
        set_tU8(offset+2,(v>>8)&0xff,1);
        set_tU8(offset+3,(v>>0)&0xff,1);
        return offset + 4;
    }
    tU32 set_tU32( tU32 offset, tU32 v )
    {
        return set_tU32(offset,v,4);
    }
    /*!
    ***************************************************************************
      @brief        <b>function set_tS32</b>
    ***************************************************************************
    */
    tS32 set_tS32( tU32 offset, tS32 v, tS32 vs )
    {
        (tVoid)vs;
        set_tU8(offset+0, (tU8)(v>>24)&0xff,1);
        set_tU8(offset+1, (tU8)(v>>16)&0xff,1);
        set_tU8(offset+2, (tU8)(v>>8)&0xff,1);
        set_tU8(offset+3, (tU8)(v>>0)&0xff,1);
        return offset + 4;
    }
    tU32 set_tS32( tU32 offset, tS32 v )
    {
        return set_tS32(offset,v,4);
    }
    /*!
    ***************************************************************************
      @brief        <b>function get_tPChar</b>
    ***************************************************************************
    */
    tPChar get_tPChar( tU32 offset ) const
    {
        if( _msgbody == AARSLIB_NULL )
        {
            return AARSLIB_NULL;
        }
        if( offset < _msglen )
        {
            return (tPChar)(&_msgbody[offset]);
        }
        return AARSLIB_NULL;
    }

    /*!
    ***************************************************************************
      @brief        <b>function get_tPU8</b>
    ***************************************************************************
    */
    tPU8 get_tPU8( tU32 offset ) const
    {
        if( _msgbody == AARSLIB_NULL )
        {
            return AARSLIB_NULL;
        }
        if( offset < _msglen )
        {
            return (tPU8)&_msgbody[offset];
        }
        return AARSLIB_NULL;
    }

public:
    /*!
    ***************************************************************************
      @brief        <b>function get_tU8</b>
    ***************************************************************************
    */
    tU8 get_tU8( tU32 offset ) const
    {
        if( _msgbody == AARSLIB_NULL )
        {
            return 0;
        }
        if( offset < _msglen )
        {
            return _msgbody[offset]&0xff;
        }
        return 0;
    }

    /*!
    ***************************************************************************
      @brief        <b>function get_tS8</b>
    ***************************************************************************
    */
    tS8 get_tS8( tU32 offset ) const
    {
        if( _msgbody == AARSLIB_NULL )
        {
            return 0;
        }
        if( offset < _msglen )
        {
            return _msgbody[offset]&0xff;
        }
        return 0;
    }

//protected:
    /*!
    ***************************************************************************
      @brief        <b>function get_tU16</b>
    ***************************************************************************
    */
    tU16 get_tU16( tU32 offset ) const
    {
        return (tU16)((get_tU8(offset)<<8)|get_tU8(offset+1));
    }

    /*!
    ***************************************************************************
      @brief        <b>function get_tS16</b>
    ***************************************************************************
    */
    tS16 get_tS16( tU32 offset ) const
    {
        return (tS16)((get_tU8(offset)<<8)|get_tU8(offset+1));
    }

    /*!
    ***************************************************************************
      @brief        <b>function get_tU32</b>
    ***************************************************************************
    */
    tU32 get_tU32( tU32 offset ) const
    {
        return (get_tU16(offset)<<16)|get_tU16(offset+2);
    }
    /*!
    ***************************************************************************
      @brief        <b>function get_tS32</b>
    ***************************************************************************
    */
    tS32 get_tS32( tU32 offset ) const
    {
        return (get_tS16(offset)<<16)|get_tS16(offset+2);
    }
};

/*!
*******************************************************************************
  @brief        <b>class tCommandMessage</b>
*******************************************************************************
*/
class tCommandMessage : public AaRSLib::MSG::tMessage
{
public:
    /*!
    ***************************************************************************
      @brief        <b>constructor</b>
    ***************************************************************************
    */
    tCommandMessage( tMessage& inmsg )
        : AaRSLib::MSG::tMessage(inmsg)
    {

    }

    tCommandMessage( tU16 msgopcode, tU32 msglen, tU8* msgbody,tBool freemsgbodyflag = false )
        : AaRSLib::MSG::tMessage(msgopcode, msglen, msgbody, freemsgbodyflag)
    {
    }
}; /* ... class tCommandMessage */

/*!
*******************************************************************************
  @brief        <b>class tResponseMessage</b>
*******************************************************************************
*/
class tResponseMessage : public AaRSLib::MSG::tMessage
{
public:
    /*!
    ***************************************************************************
      @brief        <b>constructor</b>
    ***************************************************************************
    */
    tResponseMessage( tMessage& inmsg )
        : AaRSLib::MSG::tMessage(inmsg)
    {

    }

    tResponseMessage( tU16 msgopcode, tU32 msglen, tU8* msgbody,tBool freemsgbodyflag = false )
        : AaRSLib::MSG::tMessage(msgopcode, msglen, msgbody, freemsgbodyflag)
    {
    }
}; /* ... class tResponseMessage */

/*!
*******************************************************************************
  @brief        <b>class tDatachannelMessage</b>
*******************************************************************************
*/
class tDatachannelMessage : public AaRSLib::MSG::tMessage
{
public:
    /*!
    ***************************************************************************
      @brief        <b>constructor</b>
    ***************************************************************************
    */
    tDatachannelMessage( tMessage& inmsg )
        : AaRSLib::MSG::tMessage(inmsg)
    {

    }

    tDatachannelMessage( tU16 msgopcode, tU32 msglen, tU8* msgbody, tBool freemsgbodyflag = false )
        : AaRSLib::MSG::tMessage(msgopcode, msglen, msgbody,freemsgbodyflag)
    {
    }
}; /* ... class tDatachannelMessage */

} /* ... namespace AaRSLib::MSG */

} /* ... namespace AaRSLib */

/*!
*******************************************************************************
 EOF
*******************************************************************************
*/
#endif // AARSLIB_H
