/******************************************************************************
 *
 * FILE: tclDSMApp.cpp
 * PROJECT: AaRS
 * SW-COMPONENT: Data Service Manager
 *
 * DESCRIPTION: This file contains the DSM class interface specification
 *
 * AUTHOR: CM-AI/ECB2-Scholz
 *
 * COPYRIGHT: (c) 2010 Robert Bosch Multimedia GmbH
 * HISTORY:
 * Date | Rev. | Author | Modification
 * ----------------------------------------------------------------------------
 * 23.04.2014 | 1.0 | CM-AI/ECB2-Scholz | Initial revision
 *
 *****************************************************************************/
#include <stdarg.h>  // for va_start, etc
#include <memory>    // for std::unique_ptr
#include "tclDSM.h"
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DATASRVCS_DSM_DAB_MOT_APP
#include "trcGenProj/Header/tclDSMAppDABMOT.cpp.trc.h"
#endif

using namespace dsm;

#if defined(ENABLE_GTEST)
tU16 tclDSMDABMOTApp::_u16GTestTransportIdOffset = 0;
#endif

#if !defined(DSM_PERF_ENTRY) || !defined(DSM_PERF_EXIT)
/* create dummy PERF macros, when not defined
 */
#define DSM_PERF_ENTRY()
#define DSM_PERF_EXIT(_name_)
#endif

#ifndef CONFIG_DSM__IGNORE_COMPRESSED_MOT_DIRECTORIES
#error handling of compressed MOT directories not implemented
#endif

#define MOT_DIRECTORY_PARAMID__BODYSIZE             0x40
#define MOT_DIRECTORY_PARAMID__HEADERSIZE           0x41
#define MOT_DIRECTORY_PARAMID__CONTENTTYPE          0x42
#define MOT_DIRECTORY_PARAMID__CONTENTSUBTYPE       0x43
#define MOT_DIRECTORY_PARAMID__DATACAROUSELPERIOD   0x44

#ifdef CONFIG_DSM__ENABLE_DAB
/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tclDSMDABMOTApp::tclDSMDABMOTApp( tclDSM* dsm,tU8 u8TunerIndex, std::string uri )
    : tclDSMDABApp(dsm,u8TunerIndex,uri)
{
    _num_inserted_datagroups = 0;
    _num_ignored_datagroups = 0;
    _bMOTDirectoryMode = FALSE;
    _u64MOTHeaderModeAppCompleteCTime = 0;
    _bMOTHeaderModeAppCompleteSend = FALSE;
    _tMissingMOTDirObjects.clear();
    _u32CntTryDecodeNxtMOTDir = 0;
    _u8_no_transp_id_cnt = 0;
    _u16NextMOTDirUpdateValue = FALSE;
    _u16NextMOTDirRunGC = FALSE;
    _u32NextMOTDirTransportId = 0;
    _u16NextMOTDirObjectsLeft = 0;

#ifdef CONFIG_DSM_ENABLE_USE_MOT_OBJECT_COMPLETE_CACHE
    _tMOTObjCompleteStateCache.clear();
#endif /* CONFIG_DSM_ENABLE_USE_MOT_OBJECT_COMPLETE_CACHE */

    ETG_TRACE_USR1(("%p.tclDSMDABMOTApp(%p,%d,%s)",(void*)this,dsm,u8TunerIndex,uri.c_str()));

#ifdef TR_ENABLE
#if 0
    {
        int rc;
        sqlite3_stmt* pStmt = NULL;

        rc = _dsm->oDB()->dbPrepare(
                    "SELECT a,dgtype,transpid,age_received,complete FROM dab_mot_object WHERE a=? ORDER BY transpid,dgtype",
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            int n=0;
            rc = _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                 1,
                                                 s64GetA());
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 5 )
                    {
                        ETG_TRACE_USR1(("%p.tclDSMDABMOTApp() n=%d a=%d available dgtype=%d transpid=0x%04x age=%d complete=%d",
                                        (void*)this,
                                        n++,
                                        tclDB::dbColumnInt( pStmt, 0 ),
                                        tclDB::dbColumnInt( pStmt, 1 ),
                                        tclDB::dbColumnInt( pStmt, 2 ),
                                        tclDB::dbColumnInt( pStmt, 3 ),
                                        tclDB::dbColumnInt( pStmt, 4 )));
                    }
                }
                else if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    break;
                }
            }
            rc = _dsm->oDB()->dbReset(pStmt);
        }

    }
#endif
#endif


    /* In MOT Header Mode,
     * when 10 seconds now new DG received AND N DG were ignored,
     * assume App is complete */
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tclDSMDABMOTApp::~tclDSMDABMOTApp()
{
#ifdef TR_ENABLE
#if 0
    {
        int rc;
        sqlite3_stmt* pStmt = NULL;

        rc = _dsm->oDB()->dbPrepare(
                    "SELECT a,dgtype,transpid,age_received,complete FROM dab_mot_object WHERE a=? ORDER BY transpid,dgtype",
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            int n=0;
            rc = _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                 1,
                                                 s64GetA());
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 5 )
                    {
                        ETG_TRACE_USR1(("%p.~tclDSMDABMOTApp() n=%d a=%d available dgtype=%d transpid=0x%04x age=%d complete=%d",
                                        (void*)this,
                                        n++,
                                        tclDB::dbColumnInt( pStmt, 0 ),
                                        tclDB::dbColumnInt( pStmt, 1 ),
                                        tclDB::dbColumnInt( pStmt, 2 ),
                                        tclDB::dbColumnInt( pStmt, 3 ),
                                        tclDB::dbColumnInt( pStmt, 4 )));
                    }
                }
                else if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    break;
                }
            }
            rc = _dsm->oDB()->dbReset(pStmt);
        }
    }
#endif
#endif

    ETG_TRACE_USR1(("%p.~tclDSMDABMOTApp()",this));
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABMOTApp::vInitMOTDirectory()
{
    if( s64GetA() != DSM_INVALID_APPID )
    {
        /* When application is started, go through the MOT directory and check what files are missing.
         * After this bHandleMOTObjComplete might check when redoing the MOT dir check is nessesary.
         */
        if( iBeginSavePoint() == SQLITE_OK )
        {
            if( bTryDecodeNxtMOTDir(s64GetA()) == TRUE )
            {
                iCommitSavePoint();
            }
            else if(_dsm->oDB()->dbGetAutoCommit() == 0)
            {
                iRoolbackSavePoint();
            }
        }
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
typedef struct _blobConcatStepContextType_
{
    unsigned char*                    p;
    unsigned int                      n;
    int                               id;
    struct _blobConcatStepContextType_*     next;
}blobConcatStepContextType;
/*
** blob_concat(EXPR, ?SEPARATOR?)
*/

#if defined(ENABLE_GTEST)
tU64 u64DSMDABMOTApp_blobConcatStepCnt = 0;
tU64 u64DSMDABMOTApp_blobConcatFinalizeCnt = 0;
#endif

static void tclDSMDABMOTApp_blobConcatStep(
        sqlite3_context *context,
        int argc,
        sqlite3_value **argv
        ){
    blobConcatStepContextType *pAccum;
    //assert( argc==1||argc==2 );
    if( sqlite3_value_type(argv[0])==SQLITE_NULL ) return;
    pAccum = (blobConcatStepContextType*)sqlite3_aggregate_context(context, sizeof(blobConcatStepContextType));
    //   SELECT * from v_dab_mot_file;
    if( pAccum ){
        const unsigned char *zVal;
        int nVal;
        int id = 0;

        if( argc >= 2 )
        {
            id = sqlite3_value_int(argv[1]);
        }

#if defined(ENABLE_GTEST)
        u64DSMDABMOTApp_blobConcatStepCnt++;
#endif

        nVal = sqlite3_value_bytes(argv[0]);
        if( nVal != 0 )
        {
            blobConcatStepContextType *x;
            blobConcatStepContextType *n;
            blobConcatStepContextType *t = NULL;
            zVal = (const unsigned char*) tclDB::dbValueBlob(argv[0]);
            x = pAccum;
            while(x->next != NULL)
            {
                if( x->next->id > id )
                {
                    t = x->next;
                    break;
                }
                x = x->next;
            }

            n = (blobConcatStepContextType*)sqlite3_malloc(sizeof(blobConcatStepContextType));
            if(n != NULL)
            {
                n->id = id;
                n->p = (unsigned char*)tclDB::dbMalloc(nVal);
                n->n = nVal;
                n->next = t;
                memcpy(n->p,zVal,nVal);
            }
            x->next = n;
        }
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
static void tclDSMDABMOTApp_blobConcatFinalize(sqlite3_context *context){
    blobConcatStepContextType *pAccum;
    pAccum = (blobConcatStepContextType*)sqlite3_aggregate_context(context, sizeof(blobConcatStepContextType));
    if( pAccum != NULL ){
        blobConcatStepContextType *x = pAccum;
        unsigned char* p = NULL;
        int n=0;

        while(x != NULL)
        {
            n += x->n;
            x = x->next;
        }

        if( n != 0 )
        {
#if defined(ENABLE_GTEST)
            u64DSMDABMOTApp_blobConcatFinalizeCnt++;
#endif

            p = (unsigned char*)tclDB::dbMalloc(n);
            if( p != NULL )
            {
                x = pAccum;
                n = 0;
                while(x != NULL)
                {
                    memcpy(&p[n],x->p,x->n);
                    sqlite3_free(x->p);
                    x->p = NULL;
                    n += x->n;
                    x->n = 0;
                    x = x->next;
                }
                x = pAccum->next;
                while(x != NULL)
                {
                    blobConcatStepContextType* t = x->next;
                    sqlite3_free(x);
                    x = t;
                }
                sqlite3_result_blob(context,
                                    p,
                                    n,
                                    sqlite3_free);
                return;
            }
        }
    }
    sqlite3_result_null(context);
    return;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
static void tclDSMDABMOTApp_blob2Int(
        sqlite3_context *context,
        int argc,
        sqlite3_value **argv
        ){
    sqlite_int64 v = 0;
    int i, n;
    const unsigned char *pBlob;
    //assert( argc==1 );
    (void)argc;
    pBlob = (const unsigned char *)tclDB::dbValueBlob(argv[0]);
    n = tclDB::dbValueBytes(argv[0]);
    //assert( pBlob==tclDB::dbValueBlob(argv[0]) );  /* No encoding change */
    for(i=0;i<n;i++)
    {
        v = (v << 8)|pBlob[i];
    }
    tclDB::dbResultInt64(context, v);
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABMOTApp::vInitializeTables(tclDSM* dsm)
{
    SQLITE_API int rc;

    ETG_TRACE_USR1(("vInitializeTables(%p) ...",dsm));

    /* define SDL functions needed for this module
     */
    rc = dsm->oDB()->dbCreateFunction("blob_concat", 1, SQLITE_DETERMINISTIC, NULL, NULL, tclDSMDABMOTApp_blobConcatStep, tclDSMDABMOTApp_blobConcatFinalize, NULL);
    rc = dsm->oDB()->dbCreateFunction("blob_concat", 2, SQLITE_DETERMINISTIC, NULL, NULL, tclDSMDABMOTApp_blobConcatStep, tclDSMDABMOTApp_blobConcatFinalize, NULL);
    rc = dsm->oDB()->dbCreateFunction("blob2Int", 1, SQLITE_DETERMINISTIC, NULL, tclDSMDABMOTApp_blob2Int, NULL, NULL, NULL);

#ifdef CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE
#ifdef CONFIG_DSM_ENABLE_USE_dab_mot_object_complete_TABLE
#error disable CONFIG_DSM_ENABLE_USE_dab_mot_object_complete_TABLE when CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE is enabled
#endif
#ifdef CONFIG_DSM_ENABLE_USE_dab_datagroup_mot_age_received_TABLE
#error disable CONFIG_DSM_ENABLE_USE_dab_datagroup_mot_age_received_TABLE when CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE is enabled
#endif

    /*
     * create tables dab_datagroup_mot
     */
    rc = dsm->oDB()->dbExec(
                "CREATE TABLE IF NOT EXISTS "
                /**/ "dab_mot_object ("
                /***/ "o INTEGER PRIMARY KEY,"
                /***/ "a INTEGER NOT NULL REFERENCES dab_data_application(o) ON DELETE CASCADE,"
                /***/ "dgtype,"
                /***/ "transpid NOT NULL,"
                /***/ "age_received,"
                /***/ "complete,"
            #ifndef CONFIG_DSM_ENABLE_SKIP_DATAGROUP_extfield
                /***/ "extfield,"
            #endif
            #ifndef CONFIG_DSM_ENABLE_SKIP_DATAGROUP_eua
                /***/ "eua,"
            #endif
                /**/ "UNIQUE(a,dgtype,transpid))",
                NULL,NULL,NULL);

    /*
     * create tables dab_mot_object_segm
     */
    rc = dsm->oDB()->dbExec(
                "CREATE TABLE IF NOT EXISTS "
                /**/ "dab_mot_object_segm ("
                /***/ "o INTEGER NOT NULL REFERENCES dab_mot_object(o) ON DELETE CASCADE,"
                /***/ "a INTEGER NOT NULL,"
                /***/ "segmnum NOT NULL,"
                /***/ "last,"
                /***/ "segmsize,"
                /***/ "segm,"
                /**/ "UNIQUE(o,segmnum))",
                NULL,NULL,NULL);

#ifdef CONFIG_DSM__ENABLE_dab_data_application__size_COLUMN
    rc = dsm->oDB()->dbExec(
                "CREATE TRIGGER IF NOT EXISTS "
                /**/ "tr_dab_mot_object_segm_01 "
                /***/ "AFTER INSERT ON dab_mot_object_segm "
                /***/ "BEGIN "
                /****/ "UPDATE dab_data_application SET size=size+(100+length(new.segm)) WHERE o=new.a; "
                /***/ "END",
                NULL,NULL,NULL);
    rc = dsm->oDB()->dbExec(
                "CREATE TRIGGER IF NOT EXISTS "
                /**/ "tr_dab_mot_object_segm_02 "
                /***/ "BEFORE DELETE ON dab_datagroup_mot "
                /***/ "BEGIN "
                /****/ "UPDATE dab_data_application SET size=size-(100+length(new.segm)) WHERE o=old.a; "
                /***/ "END",
                NULL,NULL,NULL);
#endif /* CONFIG_DSM__ENABLE_dab_data_application__size_COLUMN */

    rc = dsm->oDB()->dbExec(
                "CREATE TRIGGER IF NOT EXISTS "
                /**/ "tr_dab_mot_object_complete_01 "
                /***/ "AFTER INSERT ON dab_mot_object_segm "
                /***/ "WHEN ((SELECT ((min(segmnum)=0)AND(sum(last)=1)AND(max(last)=1)AND((count(segmnum)*(count(segmnum)-1)/2))=(sum(segmnum))) AS complete FROM dab_mot_object_segm WHERE o=new.o GROUP BY o))!=0 "
                /***/ "BEGIN "
                /****/ "UPDATE dab_mot_object SET complete=1 WHERE o=new.o; "
                /***/ "END",
                NULL,NULL,NULL);

    rc = dsm->oDB()->dbExec(
                "CREATE TRIGGER IF NOT EXISTS "
                /**/ "tr_dab_mot_object_complete_02 "
                /***/ "AFTER INSERT ON dab_mot_object_segm "
                /***/ "WHEN ((SELECT ((min(segmnum)=0)AND(sum(last)=1)AND(max(last)=1)AND((count(segmnum)*(count(segmnum)-1)/2))=(sum(segmnum))) AS complete FROM dab_mot_object_segm WHERE o=new.o GROUP BY o))=0 "
                /***/ "BEGIN "
                /****/ "UPDATE dab_mot_object SET complete=0 WHERE o=new.o; "
                /***/ "END",
                NULL,NULL,NULL);

#else /* CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE */

    /*
     * create tables dab_datagroup_mot*
     */
    rc = dsm->oDB()->dbExec(
                "CREATE TABLE IF NOT EXISTS "
                /**/ "dab_datagroup_mot ("
                /***/ "o INTEGER PRIMARY KEY,"
                /***/ "a NOT NULL REFERENCES dab_data_application(o) ON DELETE CASCADE,"
                /***/ "dgtype,"
                /***/ "transpid NOT NULL,"
                /***/ "segmnum NOT NULL,"
                /***/ "last,"
                /***/ "segmsize,"
                /***/ "segm,"
            #ifndef CONFIG_DSM_ENABLE_USE_dab_datagroup_mot_age_received_TABLE
                /***/ "age_received,"
            #endif
            #ifndef CONFIG_DSM_ENABLE_SKIP_DATAGROUP_extfield
                /***/ "extfield,"
            #endif
            #ifndef CONFIG_DSM_ENABLE_SKIP_DATAGROUP_eua
                /***/ "eua,"
            #endif
                /**/ "UNIQUE(a,dgtype,transpid,segmnum))",
                NULL,NULL,NULL);

#ifdef CONFIG_DSM__ENABLE_dab_data_application__size_COLUMN
#error when this feature is enabled, some operation are very slow
#endif

#ifdef CONFIG_DSM__ENABLE_dab_data_application__size_COLUMN
    rc = dsm->oDB()->dbExec(
                "CREATE TRIGGER IF NOT EXISTS "
                /**/ "tr_dab_datagroup_mot_01 "
                /***/ "AFTER INSERT ON dab_datagroup_mot "
                /***/ "BEGIN "
                /****/ "UPDATE dab_data_application SET size=size+(100+(SELECT length(segm) FROM dab_datagroup_mot WHERE o=new.o)) WHERE o=new.a; "
                /***/ "END",
                NULL,NULL,NULL);
    rc = dsm->oDB()->dbExec(
                "CREATE TRIGGER IF NOT EXISTS "
                /**/ "tr_dab_datagroup_mot_02 "
                /***/ "BEFORE DELETE ON dab_datagroup_mot "
                /***/ "BEGIN "
                /****/ "UPDATE dab_data_application SET size=size-(100+(SELECT length(segm) FROM dab_datagroup_mot WHERE o=old.o)) WHERE o=old.a; "
                /***/ "END",
                NULL,NULL,NULL);
#endif /* CONFIG_DSM__ENABLE_dab_data_application__size_COLUMN */

#ifdef CONFIG_DSM_ENABLE_USE_dab_datagroup_mot_age_received_TABLE
    rc = dsm->oDB()->dbExec(
                "CREATE TABLE IF NOT EXISTS "
                /**/ "dab_datagroup_mot_age_received ("
                /***/ "o PRIMARY KEY NOT NULL REFERENCES dab_datagroup_mot(o) ON DELETE CASCADE,"
                /***/ "age_received)",
                NULL,NULL,NULL);

    /*
     * When inserting a new row in dab_datagroup_mot, create an age row too
     */
    rc = dsm->oDB()->dbExec(
                "CREATE TRIGGER IF NOT EXISTS "
                /**/ "tr_dab_mot_object_complete_04 "
                /***/ "AFTER INSERT ON dab_datagroup_mot "
                /***/ "BEGIN "
                /****/ "INSERT OR IGNORE INTO dab_datagroup_mot_age_received (o,age_received) VALUES (new.o,0); "
                /***/ "END",
                NULL,NULL,NULL);
#endif /* CONFIG_DSM_ENABLE_USE_dab_datagroup_mot_age_received_TABLE */

#ifdef CONFIG_DSM_ENABLE_USE_dab_mot_object_complete_TABLE
    rc = dsm->oDB()->dbExec(
                "CREATE TABLE IF NOT EXISTS "
                /**/ "dab_mot_object_complete ("
                /***/ "a NOT NULL REFERENCES dab_data_application(o) ON DELETE CASCADE,"
                /***/ "dgtype,"
                /***/ "transpid NOT NULL,"
                /**/ "UNIQUE(a,dgtype,transpid))",
                NULL,NULL,NULL);

    rc = dsm->oDB()->dbExec(
                "CREATE TRIGGER IF NOT EXISTS "
                /**/ "tr_dab_mot_object_complete_01 "
                /***/ "AFTER DELETE ON dab_datagroup_mot "
                /***/ "BEGIN "
                /****/ "DELETE FROM dab_mot_object_complete WHERE a=old.a AND dgtype=old.dgtype AND transpid=old.transpid; "
                /***/ "END",
                NULL,NULL,NULL);

    rc = dsm->oDB()->dbExec(
                "CREATE TRIGGER IF NOT EXISTS "
                /**/ "tr_dab_mot_object_complete_02 "
                /***/ "AFTER INSERT ON dab_datagroup_mot "
                /***/ "WHEN ((SELECT ((min(segmnum)=0)AND(sum(last)=1)AND(max(last)=1)AND((count(segmnum)*(count(segmnum)-1)/2))=(sum(segmnum))) AS complete FROM dab_datagroup_mot WHERE a=new.a AND dgtype=new.dgtype AND transpid=new.transpid GROUP BY a,transpid,dgtype))!=0 "
                /***/ "BEGIN "
                /****/ "REPLACE INTO dab_mot_object_complete (a,dgtype,transpid) VALUES (new.a,new.dgtype,new.transpid); "
                /***/ "END",
                NULL,NULL,NULL);

    rc = dsm->oDB()->dbExec(
                "CREATE TRIGGER IF NOT EXISTS "
                /**/ "tr_dab_mot_object_complete_03 "
                /***/ "AFTER INSERT ON dab_datagroup_mot "
                /***/ "WHEN ((SELECT ((min(segmnum)=0)AND(sum(last)=1)AND(max(last)=1)AND((count(segmnum)*(count(segmnum)-1)/2))=(sum(segmnum))) AS complete FROM dab_datagroup_mot WHERE a=new.a AND dgtype=new.dgtype AND transpid=new.transpid GROUP BY a,transpid,dgtype))=0 "
                /***/ "BEGIN "
                /****/ "DELETE FROM dab_mot_object_complete WHERE a=new.a AND dgtype=new.dgtype AND transpid=new.transpid; "
                /***/ "END",
                NULL,NULL,NULL);
#endif /* CONFIG_DSM_ENABLE_USE_dab_mot_object_complete_TABLE */
#endif /* CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE */

    rc = dsm->oDB()->dbExec(
                "CREATE TABLE IF NOT EXISTS "
                /**/ "dab_mot_directory ("
                /***/ "o INTEGER PRIMARY KEY,"
                /***/ "a NOT NULL REFERENCES dab_data_application(o) ON DELETE CASCADE,"
                /***/ "transpid NOT NULL,"
                /***/ "dab_timestamp,"
                /***/ "period"
                /**/ ")",
                NULL,NULL,NULL);

    /*the mot directory parameter table contains the information from the mot directory parameter*/
    /*and the information from every parameter in the entries (e.g. for each mot object*/
    /*param id in the mot protocol is just 6 bit*/
    /*so we use values >6 bit for other purposes */
    /*body size       will have PLI=10 (32 bit value) param id 0100 0000*/
    /*header size     will have PLI=10 (32 bit value) param id 0100 0001*/
    /*content type    will have PLI=10 (32 bit value) param id 0100 0002*/
    /*content subtype will have PLI=10 (32 bit value) param id 0100 0003*/

    rc = dsm->oDB()->dbExec(
                "CREATE TABLE IF NOT EXISTS "
                /**/ "dab_mot_directory_param ("
                /***/ "a NOT NULL REFERENCES dab_data_application(o) ON DELETE CASCADE,"
                /***/ "d REFERENCES dab_mot_directory(o) ON DELETE CASCADE DEFERRABLE INITIALLY DEFERRED,"
                /***/ "transpid NOT NULL,"
                /***/ "paramid,"
                /***/ "datafield)",
                NULL,NULL,NULL);

    rc = dsm->oDB()->dbExec(
                "CREATE TABLE IF NOT EXISTS "
                /**/ "dab_mot_directory_cname ("
                /***/ "a NOT NULL REFERENCES dab_data_application(o) ON DELETE CASCADE,"
                /***/ "d REFERENCES dab_mot_directory(o) ON DELETE CASCADE DEFERRABLE INITIALLY DEFERRED,"
                /***/ "transpid NOT NULL,"
                /***/ "cset,"
                /***/ "contentname TEXT)",
                NULL,NULL,NULL);

#ifdef CONFIG_DSM__ENABLE_dab_data_application__size_COLUMN
    rc = dsm->oDB()->dbExec(
                "CREATE TRIGGER IF NOT EXISTS "
                /**/ "tr_dab_mot_directory_param_01 "
                /***/ "AFTER INSERT ON dab_mot_directory_param "
                /***/ "BEGIN "
                /****/ "UPDATE dab_data_application SET size=size+20 WHERE o=new.a; "
                /***/ "END",
                NULL,NULL,NULL);
    rc = dsm->oDB()->dbExec(
                "CREATE TRIGGER IF NOT EXISTS "
                /**/ "tr_dab_mot_directory_param_02 "
                /***/ "BEFORE DELETE ON dab_mot_directory_param "
                /***/ "BEGIN "
                /****/ "UPDATE dab_data_application SET size=size-20 WHERE o=old.a; "
                /***/ "END",
                NULL,NULL,NULL);
    rc = dsm->oDB()->dbExec(
                "CREATE TRIGGER IF NOT EXISTS "
                /**/ "tr_dab_mot_directory_cname_01 "
                /***/ "AFTER INSERT ON dab_mot_directory_cname "
                /***/ "BEGIN "
                /****/ "UPDATE dab_data_application SET size=size+20 WHERE o=new.a; "
                /***/ "END",
                NULL,NULL,NULL);
    rc = dsm->oDB()->dbExec(
                "CREATE TRIGGER IF NOT EXISTS "
                /**/ "tr_dab_mot_directory_cname_02 "
                /***/ "BEFORE DELETE ON dab_mot_directory_cname "
                /***/ "BEGIN "
                /****/ "UPDATE dab_data_application SET size=size-20 WHERE o=old.a; "
                /***/ "END",
                NULL,NULL,NULL);
#endif /* CONFIG_DSM__ENABLE_dab_data_application__size_COLUMN */

#if defined(CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE)
    rc = dsm->oDB()->dbExec(
                "CREATE VIEW IF NOT EXISTS "
                /**/ "v_dab_mot_object "
                /**/ "AS "
                /**/ "SELECT "
                     "O.a AS a,"
                     "O.dgtype AS dgtype,"
                     "O.transpid AS transpid,"
                     "O.complete AS complete,"
                     "(SELECT blob_concat(S.segm,S.segmnum) FROM dab_mot_object_segm AS S WHERE O.o=S.o) AS object "
                     "FROM "
                     "dab_mot_object AS O "
                     "WHERE "
                     "O.complete=1",
                NULL,NULL,NULL);
#elif defined(CONFIG_DSM_ENABLE_USE_dab_mot_object_complete_TABLE)
    rc = dsm->oDB()->dbExec(
                "CREATE VIEW IF NOT EXISTS "
                /**/ "v_dab_mot_object "
                /**/ "AS "
                /**/ "SELECT "
                     "X.a,"
                     "X.dgtype,"
                     "X.transpid,"
                     "1 AS complete,"
                     "(SELECT blob_concat(DG.segm,DG.segmnum) FROM dab_datagroup_mot AS DG WHERE X.a=DG.a AND X.dgtype=DG.dgtype AND X.transpid=DG.transpid) AS object "
                     "FROM "
                     "dab_mot_object_complete AS X",
                NULL,NULL,NULL);
    /* Warning: this view does NOT cover incomplete objects !!!
     */
#else /* CONFIG_DSM_ENABLE_USE_dab_mot_object_complete_TABLE */
    rc = dsm->oDB()->dbExec(
                "CREATE VIEW IF NOT EXISTS "
                /**/ "v_dab_mot_object "
                /**/ "AS "
                /**/ "SELECT "
                     "a,"
                     "dgtype,"
                     "transpid,"
                     "((min(segmnum)=0)AND(sum(last)=1)AND(max(last)=1)AND((count(segmnum)*(count(segmnum)-1)/2))=(sum(segmnum))) AS complete,"
                     "blob_concat(segm,segmnum) AS object "
                     "FROM "
                     "dab_datagroup_mot "
                     "GROUP BY a,transpid,dgtype",
                NULL,NULL,NULL);
#endif /* CONFIG_DSM_ENABLE_USE_dab_mot_object_complete_TABLEB */

    rc = dsm->oDB()->dbExec(
                "CREATE VIEW IF NOT EXISTS "
                /**/ "v_dab_mot_application "
                /**/ "AS "
                /**/ "SELECT "
                     "o,"
                     "min_received,"
                     "age_received,"
                     "age_used,"
                     "datetime(dab_timestamp,'unixepoch') AS dab_timestamp,"
                     "datetime(dab_utc_timestamp,'unixepoch') AS dab_utc_timestamp,"
                     "datetime(host_timestamp,'unixepoch') AS host_timestamp,"
                     "label,"
            #ifdef CONFIG_DSM_ENABLE_LACY_v_dab_mot_application_COMPLETE_FLAG
                     "complete,"
            #else
                     "(SELECT (CASE WHEN ((SELECT count() FROM dab_mot_directory WHERE A=A.o)!=0 AND count()=0) THEN 1 WHEN complete=1 THEN 1 ELSE 0 END) FROM dab_mot_directory_param AS P WHERE P.a=A.o AND P.paramid=64 AND P.transpid NOT IN (SELECT O.transpid FROM v_dab_mot_object AS O WHERE O.a=P.a AND O.complete=1) ) AS complete,"
            #endif
            #ifdef CONFIG_DSM__ENABLE_dab_data_application__size_COLUMN
                     "size,"
            #else
                     "("
            #if defined(CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE)
                     "  (SELECT sum(segmsize)+count()*50 FROM dab_mot_object_segm WHERE a=A.o) "
            #else
                     "  (SELECT sum(segmsize)+count()*50 FROM dab_datagroup_mot WHERE a=A.o) "
            #endif
                     "+ (SELECT count()*50 FROM dab_mot_directory_param WHERE a=A.o) "
                     "+ (SELECT count()*200 FROM dab_mot_directory_cname WHERE a=A.o) "
                     ") AS size,"
            #endif
                     "nxtmotdir,"
                     "nxtmotdirobjectsleft,"
                     "uri,"
                     "w_user_app_type,"
                     "r_user_app_data,"
                     "w_scidi,"
                     "w_xpad_app_type,"
                     "e_label,"
                     "s_label,"
                     "c_label,"
                     "u_label FROM dab_data_application AS A",
                NULL,NULL,NULL);

    rc = dsm->oDB()->dbExec(
                "CREATE VIEW IF NOT EXISTS "
                /**/ "v_dab_mot_file "
                /**/ "AS "
                /**/ "SELECT "
                     "O.a AS a,"
                     "O.transpid AS transpid,"
                     "O.complete AS complete,"
                     "(SELECT datafield FROM dab_mot_directory_param AS PV "
                     "WHERE "
                     "O.a=PV.a AND "
                     "O.transpid=PV.transpid AND "
                     "PV.paramid=16) AS mime,"
                     "(SELECT datafield FROM dab_mot_directory_param AS PV WHERE O.a=PV.a AND O.transpid=PV.transpid AND PV.paramid=17) AS compressiontype,"
                     "(SELECT datetime(PV.datafield,'unixepoch') FROM dab_mot_directory_param AS PV WHERE O.a=PV.a AND O.transpid=PV.transpid AND PV.paramid=13) AS uniquebodyversion,"
                     "(SELECT datetime(PT.datafield,'unixepoch') FROM dab_mot_directory_param AS PT WHERE O.a=PT.a AND O.transpid=PT.transpid AND PT.paramid=9) AS exptime,"
                     "P.contentname AS name,"
                     "(CASE WHEN O.complete THEN length(O.object) ELSE 0 END) AS length,"
                     "(CASE WHEN O.complete THEN O.object END) AS body "
                     "FROM v_dab_mot_object AS O,"
                     "dab_mot_directory_cname AS P "
                     "WHERE O.a=P.a AND O.transpid=P.transpid AND (O.dgtype=4 OR O.dgtype=5) AND O.complete!=0",
                NULL,NULL,NULL);

#ifndef CONFIG_DSM_AVOID_ANY_INDEX
#if defined(CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE)
    rc = dsm->oDB()->dbExec(
                "CREATE INDEX IF NOT EXISTS i_dab_mot_object_1 ON dab_mot_object (a,transpid)",
                NULL,NULL,NULL);
#else
    rc = dsm->oDB()->dbExec(
                "CREATE INDEX IF NOT EXISTS i_dab_datagroup_mot_1 ON dab_datagroup_mot (a,transpid)",
                NULL,NULL,NULL);
#endif
    rc = dsm->oDB()->dbExec(
                "CREATE INDEX IF NOT EXISTS i_dab_mot_directory_param_1 ON dab_mot_directory_param (a,transpid,paramid)",
                NULL,NULL,NULL);

    rc = dsm->oDB()->dbExec(
                "CREATE INDEX IF NOT EXISTS i_dab_mot_directory_cname_1 ON dab_mot_directory_cname (a,contentname,transpid)",
                NULL,NULL,NULL);
#endif /* CONFIG_DSM_AVOID_ANY_INDEX */

    (void)rc;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
void tclDSMDABMOTApp::vForceSetObjectComplete( tS64 a, tU8 u8_dgtype, tU16 u16_transport_id )
{
#ifdef CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE
    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;

    ETG_TRACE_USR4(("%p.vForceSetObjectComplete(%d,%d,%04x) ...",(void*)this,(int)a,u8_dgtype,u16_transport_id));

    rc = _dsm->oDB()->dbPrepare(
                "UPDATE dab_mot_object SET complete=1 WHERE a=? AND transpid=? AND dgtype=?",
                -1,
                &pStmt,
                NULL);
    if( pStmt != NULL )
    {
        rc = _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                             1,
                                             a);
        rc = _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                           2,
                                           u16_transport_id);
        rc = _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                           3,
                                           u8_dgtype);
        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                break;
            }
        }
        rc = _dsm->oDB()->dbReset(pStmt);
    }
#else
#error this is not implemenbted yet
#endif
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
void tclDSMDABMOTApp::vClearDGSegemt( tS64 a, tU8 u8_dgtype, tU16 u16_transport_id )
{
    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;

    ETG_TRACE_USR4(("%p.vClearDGSegemt(%d,%d,%04x) ...",(void*)this,(int)a,u8_dgtype,u16_transport_id));

    rc = _dsm->oDB()->dbPrepare(
            #if defined(CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE)
                "DELETE FROM dab_mot_object_segm WHERE o IN (SELECT o FROM dab_mot_object WHERE a=? AND transpid=? AND dgtype=?)",
            #else
                "UPDATE dab_datagroup_mot SET segm=NULL WHERE a=? AND transpid=? AND dgtype=?",
            #endif
                -1,
                &pStmt,
                NULL);
    if( pStmt != NULL )
    {
        rc = _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                             1,
                                             a);
        rc = _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                           2,
                                           u16_transport_id);
        rc = _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                           3,
                                           u8_dgtype);
        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                break;
            }
        }
        rc = _dsm->oDB()->dbReset(pStmt);
    }

#ifdef ENABLE_DELETE_MOTDIR_PARAM_WHEN_OBJECT_SEGM_IS_CLEARED
    /* delete all related MOT dir parameter except the body size
     */
    rc = _dsm->oDB()->dbPrepare(
                "DELETE FROM dab_mot_directory_param WHERE a=? AND transpid=? AND paramid!=?",
                -1,
                &pStmt,
                NULL);
    if( pStmt != NULL )
    {
        rc = _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                             1,
                                             a);

        rc = _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                           2,
                                           u16_transport_id);

        rc = _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                           3,
                                           MOT_DIRECTORY_PARAMID__BODYSIZE);

        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                break;
            }
        }
        rc = _dsm->oDB()->dbReset(pStmt);
    }
#endif
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABMOTApp::vHandle60sCTimeTick()
{
    ETG_TRACE_USR1(("%p.vHandle60sCTimeTick() ...",(void*)this));

    /* increment age_received counter for all DGs in current application
     */
    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;
        rc = _dsm->oDB()->dbPrepare(
            #if defined(CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE)
                    "UPDATE dab_mot_object SET age_received=age_received+1 WHERE a=? AND age_received<99",
            #elif defined(CONFIG_DSM_ENABLE_USE_dab_datagroup_mot_age_received_TABLE)
                    "UPDATE dab_datagroup_mot_age_received SET age_received=age_received+1 WHERE o IN (SELECT o FROM dab_datagroup_mot WHERE a=?) AND age_received<99",
            #else
                    "UPDATE dab_datagroup_mot SET age_received=age_received+1 WHERE a=? AND age_received<255",
            #endif
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            1,
                                            s64GetA());
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    break;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }

    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;
        rc = _dsm->oDB()->dbPrepare(
            #if defined(CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE)
                    "DELETE FROM dab_mot_object WHERE a=? AND age_received>=?",
            #elif defined(CONFIG_DSM_ENABLE_USE_dab_datagroup_mot_age_received_TABLE)
                    "DELETE FROM dab_datagroup_mot WHERE a=? AND o IN ( SELECT o FROM dab_datagroup_mot_age_received WHERE age_received>=? )",
            #else
                    "DELETE FROM dab_datagroup_mot WHERE a=? AND age_received>=?",
            #endif
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            1,
                                            s64GetA());

            _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                          2,
                                          CONFIG_DAB_MOT_APP__MAX_DG_AGE_MINUTES);
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    if( _dsm->oDB()->dbChanges() != 0 )
                    {
                        ETG_TRACE_USR1(("%p.vHandle60sCTimeTick() %d rows changed",
                                        (void*)this,
                                        _dsm->oDB()->dbChanges()));
                    }
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    break;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }

    tclDSMDABApp::vHandle60sCTimeTick();
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABMOTApp::vHandleGarbageCollection()
{
    tBool ret;

    ETG_TRACE_USR1(("%p.vHandleGarbageCollection() ...",(void*)this));

    ret = tclDSMDABApp::vHandleGarbageCollection();

    ETG_TRACE_USR1(("%p.vHandleGarbageCollection() ... %d done",
                    (void*)this,(int)ret));
    return ret;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
void tclDSMDABMOTApp::vCheckAndSendAppComplete(tBool motdirdecoded)
{
    if( _bMOTDirectoryMode == FALSE )
    {
        if( u64CTime() >= _u64MOTHeaderModeAppCompleteCTime )
        {
            if( _bMOTHeaderModeAppCompleteSend == FALSE )
            {
                int rc;
                sqlite3_stmt* pStmt = NULL;
                rc = _dsm->oDB()->dbPrepare(
                            "UPDATE dab_data_application SET complete=1 WHERE o=?",
                            -1,
                            &pStmt,
                            NULL);
                if( pStmt != NULL )
                {
                    _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                    1,
                                                    s64GetA());
                    while(1)
                    {
                        rc = _dsm->oDB()->dbStep(pStmt);
                        if( rc == SQLITE_DONE )
                        {
                            break;
                        }
                        else if( tclDB::dbOnStepError(rc) )
                        {
                            break;
                        }
                    }
                    _dsm->oDB()->dbReset(pStmt);
                    pStmt = NULL;
                }

                _bMOTHeaderModeAppCompleteSend = TRUE;
                _dsm->vBroadCastEvent( DSM_EV_APPLICATION_COMPLETE, string_sprintf("%s",tGetURI().data()) );
                _dsm->log("bTryDecodeMOTDirectory","DSM_EV_APPLICATION_COMPLETE %s",tGetURI().data());
            }
        }
    }
    else if(motdirdecoded == TRUE)
    {
        int rc;
        sqlite3_stmt* pStmt = NULL;

        ETG_TRACE_USR1(("%p.vCheckAndSendAppComplete(a=%d) nxtmotdirobjectsleft=%d ...",
                        (void*)this,(int)s64GetA(),_tMissingMOTDirObjects.size()));

        rc = _dsm->oDB()->dbPrepare(
                    "UPDATE dab_data_application SET complete=1,nxtmotdir=?,nxtmotdirobjectsleft=? WHERE o=?",
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindNull(pStmt,
                                           1);
            _dsm->oDB()->tclDB::dbBindNull(pStmt,
                                           2);
            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            3,
                                            s64GetA());
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    break;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
            pStmt = NULL;
        }

        _dsm->vBroadCastEvent( DSM_EV_APPLICATION_COMPLETE, string_sprintf("%s",tGetURI().data()) );
        _dsm->log("bTryDecodeMOTDirectory","DSM_EV_APPLICATION_COMPLETE %s",tGetURI().data());
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tU32 tclDSMDABMOTApp::u32GetAppSize()
{
    DSM_PERF_ENTRY();

    tU32 u32AppSize = 0;
    /*
     * query the actual app size ...
     */
    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;
    rc = _dsm->oDB()->dbPrepare(
            #ifdef CONFIG_DSM__ENABLE_dab_data_application__size_COLUMN
                "SELECT size FROM dab_data_application WHERE o=?",
            #else /* CONFIG_DSM__ENABLE_dab_data_application__size_COLUMN */
                "SELECT size FROM v_dab_mot_application WHERE o=?",
            #endif /* CONFIG_DSM__ENABLE_dab_data_application__size_COLUMN */
                -1,
                &pStmt,
                NULL);
    if( pStmt != NULL )
    {
        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                        1,
                                        s64GetA());
        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( rc == SQLITE_ROW )
            {
                if( tclDB::dbColumnCount(pStmt) == 1 )
                {
                    u32AppSize = tclDB::dbColumnInt( pStmt, 0 );
                }
                break;
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                break;
            }
        }
        _dsm->oDB()->dbReset(pStmt);
    }
    DSM_PERF_EXIT("tclDSMDABMOTApp::u32GetAppSize");
    return u32AppSize;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABMOTApp::vHandleMsgR_DATACHANNEL_FRAME(AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME msg)
{
    if( bGetErrorFlag() == TRUE )
    {
        ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) other error",(void*)this));
        return;
    }

    if(u64CTime() == 0 )
    {
        /* having a current DDMAPI_TIMESTAMP is mandatory for any decoding ...
         */
        ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) invalid time",(void*)this));
        return;
    }

    if( s64GetA() == 0 )
    {
        /* having a valid application handle is mandatrory ...
         */
        ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) invalid A",(void*)this));
        return;
    }

    if( _dsm->oDB()->bIsToBig() )
    {
        /* database + WAL is to big, ignore ...
         */
        ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) DB is to big",(void*)this));
        return;
    }

    ETG_TRACE_USR4(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) ...",(void*)this));

    tPU8 p_msc_dg = msg.r_body();

    _doRollbackTransaction = FALSE;

    if (msg.l_body_size() > 0)
    {
        /*check for journaline DG*/
        if (((*p_msc_dg)&0xff)==0x40)
        {
            /*todo here the journaline DGs*/
            _bErrorFlag = TRUE; /* ignore any further datagroups for this app instance */

            ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) error journaline DG",(void*)this));
            return;
        }

        /*so this must be a mot DG*/
        if( ( msg.b_api_source() == AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME::b_api_source__SOURCE_API_DDM )&&
                ( msg.b_data_source() == AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME::b_data_source__SOURCE_TYPE_SERV_COMP)&&
                (( msg.b_data_type() == AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME::b_data_type__DATA_TYPE_DATA_GROUPS )||
                 ( msg.b_data_type() == AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME::b_data_type__DATA_TYPE_PAD_DATA_GROUPS ))&&
                (( msg.l_header_size() == 2 )||(msg.l_header_size() == 4 ))&&
                ( msg.l_body_size() >= 007 )&&
                (((*p_msc_dg)&0x30)==0x30)) /*check the segment flag and user access flag presence*/
        {
            /* handle datagroups ...*/
            SQLITE_API int rc;
            tS64 a = 0;

            //tU32 u32_header_size = msg.l_header_size();

            //tPU8 p_header   = msg.r_header();
            //tU16 u16_scidi         = (p_header[0]<<8)|(p_header[1]);
            //tU16 u16_xpad_app_type = 0;

            if (msg.b_data_type() == AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME::b_data_type__DATA_TYPE_PAD_DATA_GROUPS)
            {
                ETG_TRACE_USR4(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) PAD",(void*)this));
                //u16_xpad_app_type  = (p_header[2]<<8)|(p_header[3]);
            }

            a = s64GetA();

            if( a == DSM_INVALID_APPID )
            {
                ETG_TRACE_USR4(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) a=-1",(void*)this));
            }
            else
            {
                tBool bCallHandleObjComplete = FALSE;
                tBool bIsMOTObjCompleteBefore = FALSE;
                tBool bResetAgeReceivedCounter = FALSE;

                tU32 u32_body_size = msg.l_body_size();

                tU8  u8_DGtype   = (*p_msc_dg) & 0x0f;
                tU8  u8_ext_flag = (*p_msc_dg) & 0x80;
                //tU8  crc_flag = (*p_msc_dg) & 0x40;
                //tU8  u8_segmflag = (*p_msc_dg) & 0x20;
                //tU8  u8_ua__flag = (*p_msc_dg) & 0x10;
                tU8  u8_crc_size = ((*p_msc_dg) & 0x40) >> 5;

#ifndef CONFIG_DSM_ENABLE_SKIP_DATAGROUP_extfield
                tU16 u16_ext_field;
#endif

                tU8  u8_last;
                tU16 u16_segm_number;

                tU8  u8_transport_id_flag;
                tU16 u16_transport_id = 0;

                tU8  u8_eua_len;
#ifndef CONFIG_DSM_ENABLE_SKIP_DATAGROUP_eua
                tPU8 p_eua=NULL;
#endif

                /*segmentation header from data group data field*/
                //tU8  u8_repetition_count;
                tU16 u16_segment_size;

                tPU8 p_segment;

                /*check for the extension field*/
                if (u8_ext_flag == 0x80)
                {
#ifndef CONFIG_DSM_ENABLE_SKIP_DATAGROUP_extfield
                    u16_ext_field = (p_msc_dg[2]<<8)|(p_msc_dg[3]);
#endif
                    p_msc_dg+=4;
                    u32_body_size-=4;
                }
                else
                {
                    p_msc_dg+=2;
                    u32_body_size-=2;
                }

                u8_last = ((*p_msc_dg) & 0x80)>>7;
                u16_segm_number = (tU16)((((p_msc_dg[0]) & 0x7f)<<8)|(p_msc_dg[1]));

                p_msc_dg+=2;
                u32_body_size-=2;

                u8_transport_id_flag = (*p_msc_dg) & 0x10;

                /*the transport id is mandatory for mot*/
                if (u8_transport_id_flag!=0x10)
                {
                    _u8_no_transp_id_cnt++;
                    if(_u8_no_transp_id_cnt >= 10)
                    {
                        _bErrorFlag = TRUE; /* ignore any further datagroups for this app instance */
                        _u8_no_transp_id_cnt=0;
                    }
                    ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) other error",(void*)this));
                    return;
                }
                else
                {
                    _u8_no_transp_id_cnt=0;
                }

                u8_eua_len = (tU8)(((*p_msc_dg) & 0x0f)-2);

                p_msc_dg++;
                u32_body_size--;

                u16_transport_id = (tU16)(((p_msc_dg[0])<<8)|(p_msc_dg[1]));
#if defined(ENABLE_GTEST)
                u16_transport_id = (u16_transport_id + tclDSMDABMOTApp::_u16GTestTransportIdOffset) & 0xffff;
#endif

                p_msc_dg+=2;
                u32_body_size-=2;

                if ((u8_eua_len>0)&&(u8_eua_len<14))
                {
#ifndef CONFIG_DSM_ENABLE_SKIP_DATAGROUP_eua
                    p_eua=p_msc_dg;
#endif
                    p_msc_dg+=(u8_eua_len);
                    u32_body_size-=(u8_eua_len);
                }

                //u8_repetition_count = ((*p_msc_dg)&0xe0)>>5;
                u16_segment_size = (tU16)((((p_msc_dg[0]) & 0x1f)<<8)|(p_msc_dg[1]));

                p_segment = &(p_msc_dg[2]);

                if (u32_body_size < (tU32)(u16_segment_size+u8_crc_size))
                {
                    // it may happen that due to circumstances a datagroup is truncated therefore ignore
                    // datagroups which indicate a segment size (incl. CRC) that is larger than the real body size
                    // but dont set the error flag because this would make several services unusable

                    // _bErrorFlag = TRUE; /* ignore any further datagroups for this app instance */
                    ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) body size error",(void*)this));
                    return;
                }

#ifdef TR_ENABLE_ALLOW_MORE_THEN_10_PARAMETERS
                ETG_TRACE_USR4(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) handle DG (%d.%d.0x%04x.%d.%d) a=%d u8_DGtype=%d u8_ext_flag=%d u8_last=%d u16_segm_number=%d u16_transport_id=0x%04x(%d) u8_eua_len=%d u16_segment_size=%d u8_last=%d u32_body_size=%d u8_eua_len=%d u8_transport_id_flag=%d",
                                (void*)this,
                                (int)a,u8_DGtype,u16_transport_id,u16_segm_number,u8_last,
                                (int)a,u8_DGtype,u8_ext_flag,u8_last,u16_segm_number,u16_transport_id,u16_transport_id,u8_eua_len,u16_segment_size,u8_last,u32_body_size,u8_eua_len,u8_transport_id_flag));
#else                                
                ETG_TRACE_USR4(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) handle DG (%d.%d.0x%04x.%d.%d)",
                                (void*)this,
                                (int)a,u8_DGtype,u16_transport_id,u16_segm_number,u8_last));
#endif                                

                if(( u16_segm_number > 256 )&&( u8_last == 0 ))
                {
                    ETG_TRACE_USR4(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) %d segment number %d is out of range, disable data service",
                                     (void*)this,
                                    u16_segment_size,
                                    u8_last));

                    vDeleteAppData(s64GetA());
                    _bErrorFlag = TRUE; /* ignore any further datagroups for this app instance */
                    return;
                }

                if(( u16_segment_size < 128 )&&( u8_last == 0 ))
                {
                    ETG_TRACE_USR4(("%p.vHandleMsgR_DATACHANNEL_FRAME(...)%d first or intermediate segment %d is to small, disable data service",
                                     (void*)this,
                                    u16_segment_size,
                                    u8_last));

                    vDeleteAppData(s64GetA());
                    _bErrorFlag = TRUE; /* ignore any further datagroups for this app instance */
                    return;
                }

#if 1
                {
                    tU32 object_size_limit = u32ObjectSizeLimit(u8_DGtype,u16_transport_id);
                    if( object_size_limit != 0 )
                    {
                        /* if there any limit ...
                         */
                        if( u8_last == 0 )
                        {
                            /*
                             * check is possible for any non last objects
                             */
                            tU32 estimated_object_size = (1+u16_segm_number)*u16_segment_size;
                            if( estimated_object_size > object_size_limit )
                            {
#ifdef CONFIG_DSM__ENABLE_DAB_EPG
                                if( (tclDSMDABEPGApp*)dynamic_cast<tclDSMDABEPGApp*>(this) != NULL )
                                {
                                    /*
                                     * For any objects with unsupported size.
                                     * Clear the data segemnts and continue application processing.
                                     * While decoding the MOT directory, this object might be
                                     * identified by ignoreable (e.g. wrong profile subset).
                                     * In this case, the complete flag is set.
                                     * If this object is not obsolete, the application will not become
                                     * complete.
                                     */
                                    ETG_TRACE_USR4(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) %d>%d => vClearDGSegemt",
                                                    (void*)this,
                                                    estimated_object_size,
                                                    object_size_limit));
                                    vClearDGSegemt(a,u8_DGtype,u16_transport_id);
                                    /*
                                     * ignore this datagroup
                                     */
                                    return;
                                }
                                else
#endif
                                {
                                    /*now set the error flag for the application to stop decoding and retrigger app start after time out*/
				    // TODO Gerold, DSMSLSHandlerTest.tclDABSLSHandler_DR_D_Ensemble_ObjectSizeLimitReceiveSomeSLS test fails with this error handling code
                                    //_bmaxObjectSizeViolated = TRUE;
                                    //_bErrorFlag             = TRUE;
                                    //return;
                                }
                            }
                        }
                    }
                }
#endif

#ifdef CONFIG_DSM__IGNORE_COMPRESSED_MOT_DIRECTORIES
                if( u8_DGtype == 0x07 ) /* MOT Directory Compressed */
                {
                    ETG_TRACE_USR4(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) compressed MOT dicrectory DG ignored",(void*)this));
                    return;
                }
#endif

                {
                    /*now we have all parameter for the MOT MSC DG*/
                    sqlite3_stmt* pStmt = NULL;
                    /*
                     * MOT object GC
                     * check if an old objects with different segmsize but same transpid blocks the place and delete it
                     */
                    rc = _dsm->oDB()->dbPrepare(
                                /*
                                 * ?1=a
                                 * ?2=dgtype
                                 * ?3=segmnumber
                                 * ?4=transportid
                                 * ?5=segmsize
                                 */
            #if defined(CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE)
                                /* delete all MOT objects ... */
                                "DELETE FROM dab_mot_object WHERE "
                                "o IN ("
                                /**/ "SELECT O.o "
                                /**/ "FROM dab_mot_object AS O "
                                /**/ "WHERE "
                                /**/ "O.a=?1 AND "
                                /**/ "O.transpid=?4 AND "
                                /**/ "("
                                /* ... delete object with changed dgtype ... */
                                /****/ "((3==?2 OR 4==?2) AND (O.dgtype==6 OR O.dgtype==7)) OR "
                                /****/ "((6==?2 OR 7==?2) AND (O.dgtype==3 OR O.dgtype==4))"
                                /**/ ")"
                                /**/ ") OR "
                                "o IN ("
                                /**/ "SELECT O.o "
                                /**/ "FROM dab_mot_object AS O,dab_mot_object_segm AS S "
                                /**/ "WHERE "
                                /**/ "O.o=S.o AND "
                                /**/ "O.a=?1 AND "
                                /**/ "O.transpid=?4 AND "
                                /**/ "O.dgtype=?2 AND "
                                /**/ "S.segmnum=?3 AND "
                                /* ... delete object with changed segment size ... */
                                /**/ "S.segmsize!=?5"
                                /**/ ")",
            #else
                                "DELETE FROM dab_datagroup_mot WHERE a=?1 AND dgtype=?2 AND transpid IN "
                                "(SELECT transpid FROM dab_datagroup_mot WHERE "
                                "a=?1 AND dgtype=?2 AND segmnum=?3 AND transpid=?4 AND segmsize!=?5)",
            #endif
                                -1,
                                &pStmt,
                                NULL);
                    if( pStmt != NULL )
                    {
                        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                        1,
                                                        a);

                        _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                      2,
                                                      u8_DGtype);

                        _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                      3,
                                                      u16_segm_number);

                        _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                      4,
                                                      u16_transport_id);

                        _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                      5,
                                                      u16_segment_size);

                        while(1)
                        {
                            rc = _dsm->oDB()->dbStep(pStmt);
                            if( rc == SQLITE_DONE )
                            {
                                break;
                            }
                            else if( tclDB::dbOnStepError(rc) )
                            {
                                _dsm->oDB()->dbReset(pStmt);

                                _bErrorFlag = TRUE; /* ignore any further datagroups for this app instance */

                                ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) error can't delete obsolete datagroups",(void*)this));
                                return;
                            }
                        }
                        _dsm->oDB()->dbReset(pStmt);
                        pStmt = NULL;
                    }

                    bIsMOTObjCompleteBefore = bIsMOTObjComplete(a,u8_DGtype,u16_transport_id);

#ifdef CONFIG_DSM_ENABLE_CHECK_dab_datagroup_mot_BEFORE_INSERT
                    {
                        tBool bSkipDatagroup = FALSE;

                        if( bIsMOTObjCompleteBefore == TRUE )
                        {
                            bSkipDatagroup = TRUE;
                            ETG_TRACE_USR4(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) dg allready complete, skip it",(void*)this));
                        }

                        if( bSkipDatagroup == TRUE )
                        {
                            /* now reset the DG age_received timestamp
                             */
                            pStmt = NULL;

                            /* reset the age_received counter ...
                             */
                            rc = _dsm->oDB()->dbPrepare(
            #if defined(CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE)
                                        "UPDATE dab_mot_object SET age_received=0 WHERE age_received!=0 AND a=? AND dgtype=? AND transpid=?",
            #elif defined(CONFIG_DSM_ENABLE_USE_dab_datagroup_mot_age_received_TABLE)
                                        "UPDATE dab_datagroup_mot_age_received SET age_received=0 WHERE age_received!=0 AND o IN ( SELECT o FROM dab_datagroup_mot WHERE a=? AND dgtype=? AND segmnum=? AND transpid=? )",
            #else
                                        "UPDATE dab_datagroup_mot SET age_received=0 WHERE a=? AND dgtype=? AND segmnum=? AND transpid=? AND age_received!=0",
            #endif
                                        -1,
                                        &pStmt,
                                        NULL);
                            if( pStmt != NULL )
                            {
                                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                                1,
                                                                a);

                                _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                              2,
                                                              u8_DGtype);

#if defined(CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE)
                                _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                              3,
                                                              u16_transport_id);
#else
                                _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                              3,
                                                              u16_segm_number);

                                _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                              4,
                                                              u16_transport_id);
#endif


                                while(1)
                                {
                                    rc = _dsm->oDB()->dbStep(pStmt);
                                    if( rc == SQLITE_DONE )
                                    {
                                        break;
                                    }
                                    else if( tclDB::dbOnStepError(rc) )
                                    {
                                        if( rc != SQLITE_OK )
                                        {
                                            ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) can't change age_received",(void*)this));
                                        }
                                        break;
                                    }
                                }
                                _dsm->oDB()->dbReset(pStmt);
                                pStmt = NULL;
                            }
                            ETG_TRACE_USR4(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) skipped",(void*)this));
                            return;
                        }
                    }
#endif /* CONFIG_DSM_ENABLE_CHECK_dab_datagroup_mot_BEFORE_INSERT */

                    /* create savepoint for possible rollback ...
                     */
                    if( iBeginSavePoint() != SQLITE_OK )
                    {
                        //_dsm->oDB()->dbReset(pStmt);
                        ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) can't create savepoint",(void*)this));
                        return;
                    }

#ifdef CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE
                    tS64 dab_mot_object_o = DSM_INVALID_ROWID;
                    pStmt = NULL;
                    rc = _dsm->oDB()->dbPrepare(
                                "INSERT OR IGNORE INTO dab_mot_object ("
                                "a,"
                                "dgtype,"
                                "transpid,"
                                "age_received"
            #ifndef CONFIG_DSM_ENABLE_SKIP_DATAGROUP_eua
                                ",eua,"
            #endif
            #ifndef CONFIG_DSM_ENABLE_SKIP_DATAGROUP_extfield
                                ",extfield"
            #endif
                                ") "
                                "VALUES("
                                "?,"
                                "?,"
                                "?,"
                                "0"
            #ifndef CONFIG_DSM_ENABLE_SKIP_DATAGROUP_eua
                                ",?"
            #endif
            #ifndef CONFIG_DSM_ENABLE_SKIP_DATAGROUP_extfield
                                ",?"
            #endif
                                ")",
                                -1,
                                &pStmt,
                                NULL);
                    if( pStmt != NULL )
                    {
                        int uArg = 1;
                        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                        uArg++,
                                                        a);

                        _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                      uArg++,
                                                      u8_DGtype);

                        _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                      uArg++,
                                                      u16_transport_id);
                        while(1)
                        {
                            rc = _dsm->oDB()->dbStep(pStmt);
                            if( rc == SQLITE_CONSTRAINT )
                            {
                                ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) INSERT INTO dab_datagroup_mot ... SQLITE_CONSTRAINT",(void*)this));

                                /* Abort due to constraint violation */
                                _bErrorFlag = TRUE; /* ignore any further datagroups for this app instance */
                                _dsm->oDB()->dbReset(pStmt);
                                if(_dsm->oDB()->dbGetAutoCommit() == 0)
                                {
                                    rc = iRoolbackSavePoint();
                                    if( rc != SQLITE_OK )
                                    {
                                        ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) can't rollback savepoint",(void*)this));
                                    }
                                }
                                ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) other error",(void*)this));
                                return;
                            }
                            else if( rc == SQLITE_DONE )
                            {
                                if(  _dsm->oDB()->tclDB::dbChanges() > 0 )
                                {
                                    dab_mot_object_o = _dsm->oDB()->tclDB::dbLastInsertRowId();
                                }
                                break;
                            }
                            else if( tclDB::dbOnStepError(rc) )
                            {
                                ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) INSERT INTO dab_datagroup_mot ... other step error",(void*)this));
                                _dsm->oDB()->dbReset(pStmt);
                                if(_dsm->oDB()->dbGetAutoCommit() == 0)
                                {
                                    rc = iRoolbackSavePoint();
                                    if( rc != SQLITE_OK )
                                    {
                                        ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) can't rollback savepoint",(void*)this));
                                    }
                                }
                                ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) other error",(void*)this));
                                return;
                            }
                        }
                        _dsm->oDB()->dbReset(pStmt);
                    }

                    if (dab_mot_object_o == DSM_INVALID_ROWID )
                    {
                        pStmt = NULL;
                        rc = _dsm->oDB()->dbPrepare(
                                    "SELECT o FROM dab_mot_object WHERE a=? AND dgtype=? AND transpid=?",
                                    -1,
                                    &pStmt,
                                    NULL);
                        if( pStmt != NULL )
                        {
                            int uArg = 1;
                            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                            uArg++,
                                                            a);

                            _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                          uArg++,
                                                          u8_DGtype);

                            _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                          uArg++,
                                                          u16_transport_id);
                            while(1)
                            {
                                rc = _dsm->oDB()->dbStep(pStmt);
                                if( rc == SQLITE_DONE )
                                {
                                    break;
                                }
                                else if( rc == SQLITE_ROW )
                                {
                                    if( tclDB::dbColumnCount(pStmt) == 1 )
                                    {
                                        dab_mot_object_o = tclDB::dbColumnInt64( pStmt, 0 );
                                    }
                                    break;
                                }
                                else if( tclDB::dbOnStepError(rc) )
                                {
                                    ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) INSERT INTO dab_datagroup_mot ... other step error",(void*)this));
                                    _dsm->oDB()->dbReset(pStmt);
                                    if(_dsm->oDB()->dbGetAutoCommit() == 0)
                                    {
                                        rc = iRoolbackSavePoint();
                                        if( rc != SQLITE_OK )
                                        {
                                            ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) can't rollback savepoint",(void*)this));
                                        }
                                    }
                                    ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) other error",(void*)this));
                                    return;
                                }
                            }
                            _dsm->oDB()->dbReset(pStmt);
                        }
                    }

                    if (dab_mot_object_o == DSM_INVALID_ROWID )
                    {
                        _bErrorFlag = TRUE; /* ignore any further datagroups for this app instance */
                        _dsm->oDB()->dbReset(pStmt);
                        if(_dsm->oDB()->dbGetAutoCommit() == 0)
                        {
                            rc = iRoolbackSavePoint();
                            if( rc != SQLITE_OK )
                            {
                                ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) can't rollback savepoint",(void*)this));
                            }
                        }
                        ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) other error",(void*)this));
                    }
#endif
                    pStmt = NULL;
                    rc = _dsm->oDB()->dbPrepare(
                                /* age_received set to 0 by trigger                                                                     */
            #ifdef CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE
                                "INSERT OR IGNORE INTO dab_mot_object_segm ("
                                "o,"
                                "a,"
            #else /* CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE */
                                "INSERT OR IGNORE INTO dab_datagroup_mot ("
                                "a,"
                                "dgtype,"
                                "transpid,"
            #ifndef CONFIG_DSM_ENABLE_USE_dab_datagroup_mot_age_received_TABLE
                                "age_received,"
            #endif
            #ifndef CONFIG_DSM_ENABLE_SKIP_DATAGROUP_eua
                                "eua,"
            #endif
            #ifndef CONFIG_DSM_ENABLE_SKIP_DATAGROUP_extfield
                                "extfield,"
            #endif
            #endif /* CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE */
                                "last,"
                                "segmnum,"
                                "segmsize,"
                                "segm"
                                ") "
                                "VALUES("
            #ifdef CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE
                                "?,"
                                "?,"
            #else /* CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE */
                                "?,"
                                "?,"
                                "?,"
            #ifndef CONFIG_DSM_ENABLE_USE_dab_datagroup_mot_age_received_TABLE
                                "?,"
            #endif
            #ifndef CONFIG_DSM_ENABLE_SKIP_DATAGROUP_eua
                                "?,"
            #endif
            #ifndef CONFIG_DSM_ENABLE_SKIP_DATAGROUP_extfield
                                "?,"
            #endif
            #endif /* CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE */
                                "?,"
                                "?,"
                                "?,"
                                "?"
                                ")",
                                -1,
                                &pStmt,
                                NULL);
                    if( pStmt != NULL )
                    {
                        int uArg = 1;

#ifdef CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE
                        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                        uArg++,
                                                        dab_mot_object_o);
                        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                        uArg++,
                                                        a);
#else /* CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE */
                        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                        uArg++,
                                                        a);

                        _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                      uArg++,
                                                      u8_DGtype);

                        _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                      uArg++,
                                                      u16_transport_id);

#ifndef CONFIG_DSM_ENABLE_USE_dab_datagroup_mot_age_received_TABLE
                        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                        uArg++,
                                                        0);
#endif

#ifndef CONFIG_DSM_ENABLE_SKIP_DATAGROUP_eua
                        if ((u8_eua_len>0)&&(p_eua!=NULL))
                        {
                            _dsm->oDB()->tclDB::dbBindBlob(pStmt,
                                                           uArg++,
                                                           p_eua,
                                                           u8_eua_len,
                                                           NULL);
                        }
                        else
                        {
                            _dsm->oDB()->tclDB::dbBindNull(pStmt,
                                                           uArg++);
                        }
#endif
#ifndef CONFIG_DSM_ENABLE_SKIP_DATAGROUP_extfield
                        if (u8_ext_flag == 0x80)
                        {
                            _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                          uArg++,
                                                          u16_ext_field);
                        }
                        else
                        {
                            _dsm->oDB()->tclDB::dbBindNull(pStmt,
                                                           uArg++);
                        }
#endif
#endif /* CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE */

                        _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                      uArg++,
                                                      u8_last);

                        _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                      uArg++,
                                                      u16_segm_number);

                        _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                      uArg++,
                                                      u16_segment_size);

                        _dsm->oDB()->tclDB::dbBindBlob(pStmt,
                                                       uArg++,
                                                       p_segment,
                                                       u16_segment_size,
                                                       NULL);

                        while(1)
                        {
                            rc = _dsm->oDB()->dbStep(pStmt);
                            if( rc == SQLITE_CONSTRAINT )
                            {
                                ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) INSERT INTO dab_datagroup_mot ... SQLITE_CONSTRAINT",(void*)this));

                                /* Abort due to constraint violation */
                                _bErrorFlag = TRUE; /* ignore any further datagroups for this app instance */
                                _dsm->oDB()->dbReset(pStmt);
                                if(_dsm->oDB()->dbGetAutoCommit() == 0)
                                {
                                    rc = iRoolbackSavePoint();
                                    if( rc != SQLITE_OK )
                                    {
                                        ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) can't rollback savepoint",(void*)this));
                                    }
                                }
                                ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) other error",(void*)this));
                                return;
                            }
                            else if( rc == SQLITE_DONE )
                            {
                                /* check if new DG was inserted or ignored ...
                                 */
#ifndef CONFIG_DSM_ENABLE_SKIP_DATAGROUP_extfield
								ETG_TRACE_USR4(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) INSERT INTO dab_datagroup_mot ... u16_ext_field=0x%04x u16_transport_id=0x%04x(%d) ... changes=%d,%d,%d",
									(void*)this,
									u16_ext_field,  u16_transport_id, u16_transport_id, 
									_dsm->oDB()->dbChanges(),
									_num_inserted_datagroups,
									u64CTime() > _u64MOTHeaderModeAppCompleteCTime));

								ETG_TRACE_USR4(("a=%d u8_DGtype=%d u8_ext_flag=%d u8_last=%d u16_segm_number=%d u8_eua_len=%d u16_segment_size=%d", (int)a, 
									u8_DGtype, u8_ext_flag, u8_last, u16_segm_number, u8_eua_len, u16_segment_size));

#else
								ETG_TRACE_USR4(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) INSERT INTO dab_datagroup_mot ... u16_transport_id=0x%04x(%d) ... changes=%d,%d,%d",
									(void*)this,
									u16_transport_id, u16_transport_id,
									_dsm->oDB()->dbChanges(),
									_num_inserted_datagroups,
									u64CTime() > _u64MOTHeaderModeAppCompleteCTime));

								ETG_TRACE_USR4(("a=%d u8_DGtype=%d u8_ext_flag=%d u8_last=%d u16_segm_number=%d u8_eua_len=%d u16_segment_size=%d", (int)a,
									u8_DGtype, u8_ext_flag, u8_last, u16_segm_number, u8_eua_len, u16_segment_size));
#endif
                                if( _dsm->oDB()->dbChanges() != 0 )
                                {
                                    _num_inserted_datagroups++;
                                    if( u64CTime() > _u64MOTHeaderModeAppCompleteCTime )
                                    {
                                        vClearCompleteFlag();
                                    }
                                    _u64MOTHeaderModeAppCompleteCTime = u64CTime() + CONFIG_DAB_MOT_APP__HEADER_MODE_APP_COMPLETE_TIMEOUT;
                                    _bMOTHeaderModeAppCompleteSend = FALSE;
                                    bCallHandleObjComplete = TRUE;
                                }
                                else
                                {
                                    _num_ignored_datagroups++;
                                    bResetAgeReceivedCounter = TRUE;
                                }

                                break;
                            }
                            else if( tclDB::dbOnStepError(rc) )
                            {
                                ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) INSERT INTO dab_datagroup_mot ... other step error",(void*)this));
                                _dsm->oDB()->dbReset(pStmt);
                                if(_dsm->oDB()->dbGetAutoCommit() == 0)
                                {
                                    rc = iRoolbackSavePoint();
                                    if( rc != SQLITE_OK )
                                    {
                                        ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) can't rollback savepoint",(void*)this));
                                    }
                                }
                                ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) other error",(void*)this));
                                return;
                            }
                        }
                        _dsm->oDB()->dbReset(pStmt);
                    }
                }

                if( bResetAgeReceivedCounter == TRUE )
                {
                    sqlite3_stmt* pStmt = NULL;
                    rc = _dsm->oDB()->dbPrepare(
            #ifdef CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE
                                "UPDATE dab_mot_object SET age_received=0 WHERE age_received>0 AND a=? AND dgtype=? AND transpid=?",
            #elif defined(CONFIG_DSM_ENABLE_USE_dab_datagroup_mot_age_received_TABLE)
                                "UPDATE dab_datagroup_mot_age_received SET age_received=0 WHERE age_received>0 AND o IN (SELECT o FROM dab_datagroup_mot WHERE a=? AND dgtype=? AND transpid=?)",
            #else
                                "UPDATE dab_datagroup_mot SET age_received=0 WHERE age_received>0 AND a=? AND dgtype=? AND transpid=?",
            #endif
                                -1,
                                &pStmt,
                                NULL);
                    if( pStmt != NULL )
                    {
                        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                        1,
                                                        s64GetA());

                        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                        2,
                                                        u8_DGtype);

                        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                        3,
                                                        u16_transport_id);
                        while(1)
                        {
                            rc = _dsm->oDB()->dbStep(pStmt);
                            if( rc == SQLITE_DONE )
                            {
                                break;
                            }
                            else if( tclDB::dbOnStepError(rc) )
                            {
                                _dsm->oDB()->dbReset(pStmt);
                                if(_dsm->oDB()->dbGetAutoCommit() == 0)
                                {
                                    rc = iRoolbackSavePoint();
                                    if( rc != SQLITE_OK )
                                    {
                                        ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) can't rollback savepoint",(void*)this));
                                    }
                                }
                                ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) other error",(void*)this));
                                return;
                            }
                        }
                        _dsm->oDB()->dbReset(pStmt);
                    }
                }

                if( TRUE == _doRollbackTransaction )
                {
#ifdef CONFIG_DSM_ENABLE_USE_MOT_OBJECT_COMPLETE_CACHE
                    _tMOTObjCompleteStateCache.erase((u8_DGtype<<16) | u16_transport_id );
#endif /* CONFIG_DSM_ENABLE_USE_MOT_OBJECT_COMPLETE_CACHE */
                    if(_dsm->oDB()->dbGetAutoCommit() == 0)
                    {
                        rc = iRoolbackSavePoint();
                        if( rc != SQLITE_OK )
                        {
                            ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) can't rollback savepoint",(void*)this));
                        }
                    }
                }
                else if( bCallHandleObjComplete == TRUE )
                {
#ifdef CONFIG_DSM_ENABLE_USE_MOT_OBJECT_COMPLETE_CACHE
                    _tMOTObjCompleteStateCache.erase((u8_DGtype<<16) | u16_transport_id );
#endif /* CONFIG_DSM_ENABLE_USE_MOT_OBJECT_COMPLETE_CACHE */
                    /*prevent rollback in case of max object size violated*/
                    if(( FALSE == bHandleMOTObjComplete(bIsMOTObjCompleteBefore,a,u8_DGtype,u16_transport_id,u8_last)&&bCallHandleObjComplete)&&
                            (_bmaxObjectSizeViolated == FALSE))
                    {
                        /* rollback savepoint because complete not handled
                         */

#ifdef CONFIG_DSM_ENABLE_USE_MOT_OBJECT_COMPLETE_CACHE
                        _tMOTObjCompleteStateCache.erase((u8_DGtype<<16) | u16_transport_id );
#endif /* CONFIG_DSM_ENABLE_USE_MOT_OBJECT_COMPLETE_CACHE */

                        if(_dsm->oDB()->dbGetAutoCommit() == 0)
                        {
                            rc = iRoolbackSavePoint();
                            if( rc != SQLITE_OK )
                            {
                                ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) can't rollback savepoint",(void*)this));
                            }
                        }
                    }
                    else
                    {
                        /* commit savepoint
                         */
                        rc = iCommitSavePoint();
                        if( rc != SQLITE_OK )
                        {
                            ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) can't commit savepoint",(void*)this));
                        }
                        vCheckAndSendAppComplete(FALSE);
                    }
                }
                else
                {
                    /* commit savepoint
                     */
                    rc = iCommitSavePoint();
                    if( rc != SQLITE_OK )
                    {
                        ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) can't commit savepoint",(void*)this));
                    }
                    vCheckAndSendAppComplete(FALSE);
                }
            }
        }

        if( FALSE == _doRollbackTransaction )
        {
            vUpdateNextMOTDir();
        }

        if( FALSE == _doRollbackTransaction )
        {
            /*
             * Check App Size
             */
            if( u32AppSizeLimit() != 0 ) /* check if app size is not unlimited */
            {
                tU32 u32AppSize = u32GetAppSize();

                /*
                 * check it ...
                 */
                if( u32AppSize > 0 )
                {
                    //printf("iAppSize=%d u32AppSizeLimit=%d\n",iAppSize,u32AppSizeLimit());
                    if( u32AppSizeLimit() < u32AppSize )
                    {
                        ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) APP is to big",(void*)this));

                        /*
                         * because app is to big, delete all app related data ...
                         */

                        vDeleteAppData(s64GetA());
                        _bErrorFlag = TRUE; /* ignore any further datagroups for this app instance */
                        ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) app size error",(void*)this));
                        return;
                    }
                }
            }
            if (_bmaxObjectSizeViolated == TRUE)
            {
                ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) Object is to big",(void*)this));

                /*
                 * because object is to big, delete all app related data ...
                 */

                vDeleteAppData(s64GetA());
                _bErrorFlag = TRUE; /* ignore any further datagroups for this app instance */
                ETG_TRACE_FATAL(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) max obj size error",(void*)this));
                return;
            }
        }
    }
    ETG_TRACE_USR4(("%p.vHandleMsgR_DATACHANNEL_FRAME(...) done",
                    (void*)this));
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABMOTApp::vHandleMsgR_TIMESTAMP(AaRSDABLib::API::DDM::tR_TIMESTAMP msg)
{
    tclDSMDABApp::vHandleMsgR_TIMESTAMP(msg);
    if( _u64MOTHeaderModeAppCompleteCTime == 0 )
    {
        _u64MOTHeaderModeAppCompleteCTime =
                u64CTime() + CONFIG_DAB_MOT_APP__HEADER_MODE_APP_COMPLETE_TIMEOUT;
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABMOTApp::vHandleMsgR_SELECT_URI(AaRSDABLib::API::DDM::tR_SELECT_URI msg)
{
    tclDSMDABApp::vHandleMsgR_SELECT_URI(msg);
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABMOTApp::bTryDecodeNxtMOTDir( tS64 a )
{
    if( a == DSM_INVALID_APPID )
    {
        return FALSE;
    }

    _u32CntTryDecodeNxtMOTDir++;

    ETG_TRACE_USR1(("%p.bTryDecodeNxtMOTDir(a=%d) _u32CntTryDecodeNxtMOTDir=%d ...",(void*)this,(int)a,_u32CntTryDecodeNxtMOTDir));

    tU32 nxtmotdir = 0;
    if( (nxtmotdir = u32GetSavedNxtMOTDirId(a)) != 0 )
    {
        if( bProcessCompleteMOTObj(a,(nxtmotdir>>16)&0xff,(nxtmotdir>>0)&0xffff) == FALSE )
        {
            ETG_TRACE_USR1(("%p.bTryDecodeNxtMOTDir(a=%d) _u32CntTryDecodeNxtMOTDir=%d ... abort => FALSE",(void*)this,(int)a,_u32CntTryDecodeNxtMOTDir));
            return FALSE;
        }
    }

    ETG_TRACE_USR1(("%p.bTryDecodeNxtMOTDir(a=%d) _u32CntTryDecodeNxtMOTDir=%d ... done => TRUE",(void*)this,(int)a,_u32CntTryDecodeNxtMOTDir));
    return TRUE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
std::string tclDSMDABMOTApp::tMissingObjects( )
{
    std::string r = "( ";
    for(std::list<tU16>::iterator i = _tMissingMOTDirObjects.begin();
        i != _tMissingMOTDirObjects.end();
        i++)
    {
        char s[32];
        sprintf(s,"0x%04x(%d) ",*i,*i);
        r += s;
    }
    r += ")" ;
    return r;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABMOTApp::bHandleMOTObjComplete( tBool bIsCompleteBefore, tS64 a, tU8 u8DGType, tU16 u16_transport_id, tU8 u8Last )
{
    tBool bRes = TRUE;
    tBool bIsComplete = bIsMOTObjComplete(a,u8DGType,u16_transport_id);

    ETG_TRACE_USR3(("%p.bHandleMOTObjComplete(bIsComplete=%d,bIsCompleteBefore=%d,a=%d,u8DGType=0x%02x,u16_transport_id=0x%04x(%d),u8Last=%d) bIsComplete=%d nMissingObjs=%s ... ",
                    (void*)this,
                    (int)bIsComplete,(int)bIsCompleteBefore,(int)a,u8DGType,u16_transport_id,u16_transport_id,u8Last,
                    (int)bIsComplete,tMissingObjects().c_str()));

    (void)u8Last; /* not used */

    if( bIsCompleteBefore == FALSE )
    {
        if(( u8DGType == 0x06 )||( u8DGType == 0x07 ))
        {
            /* in case of a MOT directory DG becomes complete, commit the transaction.
             */
            iCommitSavePoint();
            /* start new transaction because, further rollback might be needed
             */
            iBeginSavePoint();
        }
    }

    if( bIsComplete == TRUE )
    {
        bRes = bProcessCompleteMOTObj(a,u8DGType,u16_transport_id);
    }

    if( bRes == TRUE )
    {
        if((_tMissingMOTDirObjects.size() != 0 )&&(bIsComplete == true))
        {
            if(( u8DGType == 0x04 )||( u8DGType == 0x05 ))
            {
                _tMissingMOTDirObjects.remove(u16_transport_id);
                //printf("_tMissingMOTDirObjects.remove(%08x) %d #2\n",u16_transport_id,_tMissingMOTDirObjects.size());
                if( _tMissingMOTDirObjects.size() == 0 )
                {
                    /* all missing Ids received, re-decode previous MOTDir
                     */
                    iCommitSavePoint();

                    if( iBeginSavePoint() == SQLITE_OK )
                    {
                        bRes = bTryDecodeNxtMOTDir(a);
                    }
                }
                else if( _u32NextMOTDirTransportId != 0 )
                {
                    _u16NextMOTDirUpdateValue = TRUE;
                    _u16NextMOTDirObjectsLeft = (tU32)_tMissingMOTDirObjects.size();
                }
            }
        }
    }

    /*
     * check if application is still/allready complete
     */
    if( bIsComplete == TRUE )
    {
        if(( u8DGType == 0x06 )||( u8DGType == 0x07 ))
        {
            if( _u32NextMOTDirTransportId == 0 )
            {
                if( bIsAppComplete(a) )
                {
                    /* no next MOT directory, then application is still complete
                     */
                    _dsm->vBroadCastEvent( DSM_EV_APPLICATION_ALLREADY_COMPLETE, string_sprintf("%s",tGetURI().data()) );
                    _dsm->log("bHandleMOTObjComplete","DSM_EV_APPLICATION_ALLREADY_COMPLETE %s",tGetURI().data());
                }
            }
        }
    }

    ETG_TRACE_USR3(("%p.bHandleMOTObjComplete(bIsCompleteBefore=%d,a=%d,u8DGType=0x%02x,u16_transport_id=0x%04x(%d),u8Last=%d) bIsComplete=%d nMissingObjs=%s ... done",
                    (void*)this,
                    (int)bIsCompleteBefore,(int)a,u8DGType,u16_transport_id,u16_transport_id,u8Last,
                    (int)bIsComplete,tMissingObjects().c_str()));

    return bRes;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABMOTApp::bIsMOTObjComplete( tS64 a, tU8 u8_dgtype, tU16 u16_transport_id )
{
    SQLITE_API int rc;
    sqlite3_stmt* pStmt;
    tBool complete = FALSE;

#ifdef CONFIG_DSM_ENABLE_USE_MOT_OBJECT_COMPLETE_CACHE
    if( _tMOTObjCompleteStateCache.count((u8_dgtype<<16) | u16_transport_id) != 0 )
    {
        return _tMOTObjCompleteStateCache[ (u8_dgtype<<16) | u16_transport_id ];
    }
#endif /* CONFIG_DSM_ENABLE_USE_MOT_OBJECT_COMPLETE_CACHE */

    rc = _dsm->oDB()->dbPrepare(
            #if defined(CONFIG_DSM_ENABLE_USE_dab_mot_object_complete_TABLE)
                "SELECT 1 FROM dab_mot_object_complete WHERE a=? AND dgtype=? AND transpid=? LIMIT 1",
            #else /* CONFIG_DSM_ENABLE_USE_dab_mot_object_complete_TABLEB */
                "SELECT complete FROM v_dab_mot_object WHERE a=? AND dgtype=? AND transpid=? LIMIT 1",
            #endif /* CONFIG_DSM_ENABLE_USE_dab_mot_object_complete_TABLEB */
                -1,
                &pStmt,
                NULL);

    if( pStmt != NULL )
    {
        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                        1,
                                        a);

        _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                      2,
                                      u8_dgtype);

        _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                      3,
                                      u16_transport_id);
        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( rc == SQLITE_ROW )
            {
                if( tclDB::dbColumnCount(pStmt) == 1 )
                {
                    complete = (tclDB::dbColumnInt( pStmt, 0 ))==0?FALSE:TRUE;
                }
                break;
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                break;
            }
        }
        _dsm->oDB()->dbReset(pStmt);
    }

    ETG_TRACE_USR3(("%p.bIsMOTObjComplete(a=%d,u8DGType=0x%02x,u16_transport_id=0x%04x(%d)) bIsComplete=%d",
                    (void*)this,
                    (int)a,u8_dgtype,u16_transport_id,u16_transport_id,(int)complete));

#ifdef CONFIG_DSM_ENABLE_USE_MOT_OBJECT_COMPLETE_CACHE
    _tMOTObjCompleteStateCache[ (u8_dgtype<<16) | u16_transport_id ] = complete;
#endif /* CONFIG_DSM_ENABLE_USE_MOT_OBJECT_COMPLETE_CACHE */

    return complete;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABMOTApp::bGetObjParam( tS64 a, tU16 u16_transport_id, tU32 u32ParamId, tU32& u32Value )
{
    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;

    rc = _dsm->oDB()->dbPrepare(
                "SELECT P.datafield FROM dab_mot_directory_param AS P WHERE a=?1 AND transpid=?2 AND paramid=?3 LIMIT 1",
                -1,
                &pStmt,
                NULL);

    if( pStmt != NULL )
    {
        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                        1,
                                        a);

        _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                      2,
                                      u16_transport_id);

        _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                      3,
                                      u32ParamId);

        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( rc == SQLITE_ROW )
            {
                if( tclDB::dbColumnCount(pStmt) == 1 )
                {
                    u32Value = tclDB::dbColumnInt( pStmt, 0 );
                    _dsm->oDB()->dbReset(pStmt);
                    return TRUE;
                }
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                _dsm->oDB()->dbReset(pStmt);
                return FALSE;
            }
        }
        _dsm->oDB()->dbReset(pStmt);
    }
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABMOTApp::bGetObjParams( tS64 a, tU16 u16_transport_id, tU32& u32BodySize, tU8& u8ContentType, tU16& u16ContentSubType )
{
    tU32 p=0;
    if( bGetObjParam(a,u16_transport_id,MOT_DIRECTORY_PARAMID__BODYSIZE,p) == TRUE )
    {
        u32BodySize = p;
        if( bGetObjParam(a,u16_transport_id,MOT_DIRECTORY_PARAMID__CONTENTTYPE,p) == TRUE )
        {
            u8ContentType = (tU8)p;
            if( bGetObjParam(a,u16_transport_id,MOT_DIRECTORY_PARAMID__CONTENTSUBTYPE,p) == TRUE )
            {
                u16ContentSubType = (tU16)p;
                return TRUE;
            }
        }
    }
    return FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABMOTApp::bProcessCompleteMOTObj( tS64 a, tU8 u8DGType, tU16 u16_transport_id )
{
    tBool bRet = TRUE;
    tBool bProcessObject = FALSE;

    switch(u8DGType)
    {
    case 0x03: /* MOT Header */
        bProcessObject = TRUE;
        break;
    case 0x04: /* Unscrambled DG */
        bProcessObject = TRUE;
        break;
    case 0x05: /* Scrambled DG */
        break;
    case 0x06: /* MOT Directory Uncompressed */
        bProcessObject = TRUE;
        break;
    case 0x07: /* MOT Directory Compressed */
        bProcessObject = TRUE;
        break;
    }

    ETG_TRACE_USR3(("%p.bProcessCompleteMOTObj(a=%d,u8DGType=0x%02x,u16_transport_id=0x%04x,bProcessObject=%d) ...",
                    (void*)this,(int)a,u8DGType,u16_transport_id,(int)bProcessObject));

    if( bProcessObject == TRUE )
    {
        /* object completness detected ...
        */
        tBool bSendEvent = FALSE;
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;
        const unsigned char* uri = NULL;
        const tU8* object_body = NULL;
        tU32 object_body_bytes = 0;
        tU32 object_size_limit = 0;

        tU32 u32UserAppType = 0;

        rc = _dsm->oDB()->dbPrepare(
                    "SELECT A.uri,A.w_user_app_type,O.object FROM dab_data_application AS A, v_dab_mot_object AS O WHERE A.o=O.a AND O.transpid=?1 AND A.o=?2 AND O.dgtype=?3 AND O.complete=1 LIMIT 1",
                    -1,
                    &pStmt,
                    NULL);

        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                          1,
                                          u16_transport_id);

            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            2,
                                            a);

            _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                          3,
                                          u8DGType);

            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 3 )
                    {
                        uri = tclDB::dbColumnText( pStmt, 0 );
                        u32UserAppType = tclDB::dbColumnInt( pStmt, 1 );
                        object_body_bytes = tclDB::dbColumnBytes( pStmt, 2 );
                        if( object_body_bytes != 0 )
                        {
                            object_body = (const tU8*)tclDB::dbColumnBlob( pStmt, 2 );
                        }
                    }
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    _dsm->oDB()->dbReset(pStmt);
                    return FALSE;
                }
            }

            /*get object limit*/
            object_size_limit = u32ObjectSizeLimit(u8DGType,u16_transport_id);

            /*if to large set error flag*/
            if((object_size_limit != 0)&&
                    (object_body_bytes >= object_size_limit))
            {
                ETG_TRACE_USR1(("%p.bProcessCompleteMOTObj(a=%d,u8DGType=0x%02x,u16_transport_id=0x%04x,object_body_bytes=%d) no Params; abort => vClearDGSegemt",
                                (void*)this,(int)a,u8DGType,u16_transport_id,(int)object_body_bytes));
                                
                vClearDGSegemt(a,u8DGType,u16_transport_id);
                _dsm->oDB()->dbReset(pStmt);

#ifdef CONFIG_DSM__ENABLE_DAB_EPG
                if( (tclDSMDABEPGApp*)dynamic_cast<tclDSMDABEPGApp*>(this) != NULL )
                {
                    /*
                     * For any EPG objects with unsupported size.
                     * Clear the data segemnts and continue application processing.
                     * While decoding the MOT directory, this object might be
                     * identified by ignoreable (e.g. wrong profile subset).
                     * In this case, the complete flag is set.
                     * If this object is not obsolete, the application will not become
                     * complete.
                     */
                    return TRUE;
                }
                else
#endif
                {
                    /*now set the error flag for the application to stop decoding and retrigger app start after time out*/
                    _bmaxObjectSizeViolated = TRUE;
                    _bErrorFlag             = TRUE;
                    return FALSE;
                }
            }

            if( uri != NULL )
            {
                if( object_body != NULL )
                {
                    switch(u8DGType)
                    {
                    case 0x03: /* MOT Header */
                        bSendEvent = bTryDecodeMOTHeader(a,u16_transport_id,u8DGType,u32UserAppType,object_body_bytes,(const tU8*)object_body);
                        break;
                    case 0x04: /* Unscrambled DG */
                    {
                        tU32 u32BodySize=0;
                        tU8  u8ContentType=0;
                        tU16 u16ContentSubType=0;
                        if( bGetObjParams(a,u16_transport_id,u32BodySize,u8ContentType,u16ContentSubType) == FALSE )
                        {
                            ETG_TRACE_USR1(("%p.bProcessCompleteMOTObj(a=%d,u8DGType=0x%02x,u16_transport_id=0x%04x,bProcessObject=%d) no Params; abort",
                                            (void*)this,(int)a,u8DGType,u16_transport_id,(int)bProcessObject));
                            if( _doRollbackTransaction == TRUE )
                            {
                                if(_dsm->oDB()->dbGetAutoCommit() == 0)
                                {
                                    rc = iRoolbackSavePoint();
                                }
                                _dsm->oDB()->dbReset(pStmt);
                                return FALSE;
                            }
                        }
                        else
                        {
                            if( u32BodySize != object_body_bytes )
                            {
                                /* ignore the MOT object, when body size given in MOT header is different to real size.
                                 */
                                ETG_TRACE_USR1(("%p.bProcessCompleteMOTObj(a=%d,u8DGType=0x%02x,u16_transport_id=0x%04x,bProcessObject=%d) wrong BodySize (%d!=%d); abort",
                                                (void*)this,(int)a,u8DGType,u16_transport_id,(int)bProcessObject,
                                                (int)u32BodySize,(int)object_body_bytes));
                                if( _doRollbackTransaction == TRUE )
                                {
                                    if(_dsm->oDB()->dbGetAutoCommit() == 0)
                                    {
                                        rc = iRoolbackSavePoint();
                                    }
                                    _dsm->oDB()->dbReset(pStmt);
                                    return FALSE;
                                }
                            }
                            else
                            {
                                switch( u32UserAppType )
                                {
#ifdef CONFIG_DSM__ENABLE_DAB_EPG
                                case DAB_SERVICE_MANAGER_UAPPTYPE_EPG:
                                    if(( u8ContentType == 7 )&&
                                            (
                                                (u16ContentSubType == 0)||
                                                (u16ContentSubType == 1)||
                                                (u16ContentSubType == 2))
                                            )
                                    {
                                        /* check for EPG content type
                                         */
                                        tclDSMDABEPGApp* _this = (tclDSMDABEPGApp*)dynamic_cast<tclDSMDABEPGApp*>(this);
                                        if( _this != NULL )
                                        {
#ifdef CONFIG_DSM_ENABLE_bTryDecodeEPGObjectIfNessesary_OPTIMIZIATION
                                            if( _this->bTryDecodeEPGObjectIfNessesary(NULL,a,u16_transport_id,object_body,object_body_bytes) == TRUE )
#else
                                            if( _this->bTryDecodeEPGObjectIfNessesary(NULL,a,u16_transport_id) == TRUE )
#endif
                                            {
                                                if( _doRollbackTransaction == TRUE )
                                                {
                                                    if(_dsm->oDB()->dbGetAutoCommit() == 0)
                                                    {
                                                        rc = iRoolbackSavePoint();
                                                    }
                                                    _dsm->oDB()->dbReset(pStmt);
                                                    return FALSE;
                                                }
                                            }
                                            else
                                            {
                                                if(_dsm->oDB()->dbGetAutoCommit() == 0)
                                                {
                                                    rc = iRoolbackSavePoint();
                                                }
                                                _dsm->oDB()->dbReset(pStmt);
                                                return FALSE;
                                            }
                                        }
                                    }
                                    break;
#endif /* CONFIG_DSM__ENABLE_DAB_EPG */
#ifdef CONFIG_DSM__ENABLE_DAB_SLS
                                case DAB_SERVICE_MANAGER_UAPPTYPE_SLS:
                                {
                                    tclDSMDABSLSApp* _this = (tclDSMDABSLSApp*)dynamic_cast<tclDSMDABSLSApp*>(this);
                                    if( _this != NULL )
                                    {
                                        _this->bTryHandleSLSObject(u16_transport_id);
                                    }
                                }
                                    break;
#endif /* CONFIG_DSM__ENABLE_DAB_SLS */
                                default:
                                    break;
                                }
                            }
                        }
                        bSendEvent = TRUE;
                    }
                        break;
#if 0
                    case 0x05: /* Scrambled DG */
                        break;
#endif
                    case 0x06: /* MOT Directory Uncompressed */
                        _bMOTDirectoryMode = TRUE;
                        bSendEvent = bTryDecodeMOTDirectory(a,u16_transport_id,u8DGType,u32UserAppType,object_body_bytes,(tU8*)object_body);
                        break;
                    case 0x07: /* MOT Directory Compressed */
                        _bMOTDirectoryMode = TRUE;
                        bSendEvent = bTryDecodeMOTDirectory(a,u16_transport_id,u8DGType,u32UserAppType,object_body_bytes,(tU8*)object_body);
                        break;
                    }
                }
                if( bSendEvent == TRUE )
                {
                    _dsm->vBroadCastEvent( DSM_EV_APPLICATION_OBJECT_COMPLETE, string_sprintf("%s?transpid=0x%04x",uri,u16_transport_id) );
                }
                else
                {
                    bRet = FALSE;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }
    return bRet;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
int tclDSMDABMOTApp::s32TryDecodeMOTHeaderElement( tS64 a, tS64 d, tU16 u32EntryTransportId, tU32 object_body_bytes, const tU8* object_body, tU32 bitidx, tU16* pU32ContentType, tU16* pU32ContentSubType )
{
    /*the mot directory parameter table contains the information from the mot directory parameter*/
    /*and the information from every parameter in the entries (e.g. for each mot object*/
    /*param id in the mot protocol is just 6 bit*/
    /*so we use values >6 bit for other purposes */
    /*body size       will have PLI=10 (32 bit value) param id 0100 0000*/
    /*header size     will have PLI=10 (32 bit value) param id 0100 0001*/
    /*content type    will have PLI=10 (32 bit value) param id 0100 0002*/
    /*content subtype will have PLI=10 (32 bit value) param id 0100 0003*/

    /* BodySize: This 28-bit field, coded as an unsigned binary number, indicates the total size of the body in bytes.
         */
    tU32 u32BodySize = u32BitStreamValue(object_body,object_body_bytes,bitidx,28);
    bitidx+=28;

    /* HeaderSize: This 13-bit field, coded as an unsigned binary number,
         * indicates the total size of the header information in bytes including the header core size of 7 bytes.
         */
    tU32 u32HeaderSize = u32BitStreamValue(object_body,object_body_bytes,bitidx,13);
    bitidx+=13;

    if( u32HeaderSize < 7 )
    {
        /* return if u32HeaderSize for the minimum limit
         */
        DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
        return -1;
    }

    /* ContentType: This 6-bit field indicates the main category of the body's content.
         * The interpretation of this field shall be defined in TS 101 756 [7], table 17.
         */
    tU16 u16ContentType = (tU16)u32BitStreamValue(object_body,object_body_bytes,bitidx,6);
    bitidx+=6;

    if( pU32ContentType != NULL )
    {
        *pU32ContentType = u16ContentType;
    }

    /* ContentSubType: This 9-bit field indicates the exact type of the body's content depending
         * on the value of the field ContentType. The interpretation of this field shall be defined
         * in TS 101 756 [7], table 17.
         */
    tU16 u16ContentSubType = (tU16)u32BitStreamValue(object_body,object_body_bytes,bitidx,9);
    bitidx+=9;

    if( pU32ContentSubType != NULL )
    {
        *pU32ContentSubType = u16ContentSubType;
    }

    if( u32HeaderSize <= 7 )
    {
        DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
        return -1;
    }

    if( ((bitidx>>3)+(u32HeaderSize-7)) > object_body_bytes )
    {
        DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
        return -1;
    }

    if( bDirectoryParamFilter((tU8)u16ContentType,u16ContentSubType,(tU8)MOT_DIRECTORY_PARAMID__BODYSIZE) == TRUE )
    {
        if( bTryDecodeMOTDirectoryParameter(a,d,u32EntryTransportId,0x02,MOT_DIRECTORY_PARAMID__BODYSIZE,0,NULL,u32BodySize) == FALSE )
        {
            DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
            return -1;
        }
    }

#if 0
    if( bDirectoryParamFilter(u16ContentType,u16ContentSubType,MOT_DIRECTORY_PARAMID__HEADERSIZE) == TRUE )
    {
        if( bTryDecodeMOTDirectoryParameter(a,d,u32EntryTransportId,0x02,MOT_DIRECTORY_PARAMID__HEADERSIZE,0,NULL,u32HeaderSize) == FALSE )
        {
            DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
            return -1;
        }
    }
#endif

    if( bDirectoryParamFilter((tU8)u16ContentType,u16ContentSubType,(tU8)MOT_DIRECTORY_PARAMID__CONTENTTYPE) == TRUE )
    {
        if( bTryDecodeMOTDirectoryParameter(a,d,u32EntryTransportId,0x02,MOT_DIRECTORY_PARAMID__CONTENTTYPE,0,NULL,u16ContentType) == FALSE )
        {
            DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
            return -1;
        }
    }

    if( bDirectoryParamFilter((tU8)u16ContentType,u16ContentSubType,(tU8)MOT_DIRECTORY_PARAMID__CONTENTSUBTYPE) == TRUE )
    {
        if( bTryDecodeMOTDirectoryParameter(a,d,u32EntryTransportId,0x02,MOT_DIRECTORY_PARAMID__CONTENTSUBTYPE,0,NULL,u16ContentSubType) == FALSE )
        {
            DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
            return -1;
        }
    }

    u32HeaderSize = u32HeaderSize - 7;

    if( (bitidx+(u32HeaderSize*8)) > (object_body_bytes*8) )
    {
        /* return if u32HeaderSize exceeds the maximum limit
         */
        DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
        return -1;
    }

    for( tU32 j=0;j<(tU32)u32HeaderSize;)
    {
        tU32 u32PLI = u32BitStreamValue(object_body,object_body_bytes,bitidx,2);
        tU32 u32DatafieldLength = 0;
        bitidx+=2;

        tU32 u32ParamId= u32BitStreamValue(object_body,object_body_bytes,bitidx,6);
        bitidx+=6;
        j += 1;

        if( u32PLI == 0 )
        {
            u32DatafieldLength = 0;
        }
        else if( u32PLI == 1 )
        {
            if( bDirectoryParamFilter((tU8)u16ContentType,u16ContentSubType,(tU8)u32ParamId) == TRUE )
            {
                if( bTryDecodeMOTDirectoryParameter(a,d,u32EntryTransportId,u32PLI,u32ParamId,0,NULL,u32BitStreamValue(object_body,object_body_bytes,bitidx,8)) == FALSE )
                {
                    DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
                    return -1;
                }
            }
            bitidx+=8;
            u32DatafieldLength = 1;
        }
        else if( u32PLI == 2 )
        {
            if( bDirectoryParamFilter((tU8)u16ContentType,u16ContentSubType,(tU8)u32ParamId) == TRUE )
            {
                if( bTryDecodeMOTDirectoryParameter(a,d,u32EntryTransportId,u32PLI,u32ParamId,0,NULL,u32BitStreamValue(object_body,object_body_bytes,bitidx,32)) == FALSE )
                {
                    DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
                    return -1;
                }
            }
            bitidx+=32;
            u32DatafieldLength = 4;
        }
        else if( u32PLI == 3 )
        {
            if( u32BitStreamValue(object_body,object_body_bytes,bitidx++,1) == 0 )
            {
                u32DatafieldLength = u32BitStreamValue(object_body,object_body_bytes,bitidx,7);
                bitidx+=7;
                j += 1;
            }
            else
            {
                u32DatafieldLength = u32BitStreamValue(object_body,object_body_bytes,bitidx,15);
                bitidx+=15;
                j += 2;
            }

            if( ((bitidx>>3)+(u32DatafieldLength)) > object_body_bytes )
            {
                DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
                return -1;
            }

            if( bDirectoryParamFilter((tU8)u16ContentType,u16ContentSubType,(tU8)u32ParamId) == TRUE )
            {
                if( bTryDecodeMOTDirectoryParameter(a,d,u32EntryTransportId,u32PLI,u32ParamId,u32DatafieldLength,&object_body[bitidx>>3],0) == FALSE )
                {
                    //DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
                    return -1;
                }
            }

            bitidx += (u32DatafieldLength<<3);

        }
        j += u32DatafieldLength;
    }

    return bitidx;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABMOTApp::bTryDecodeMOTHeader( tS64 a, tU16 u32EntryTransportId, tU8 u8DGType, tU32 u32UserAppType, tU32 object_body_bytes, const tU8* object_body )
{
    tU32 bitidx = 0;
    SQLITE_API int rc;

    (void)u8DGType;
    (void)u32UserAppType;
    tU16 u16ContentType = 0;
    tU16 u16ContentSubType = 0;

    ETG_TRACE_USR1(("%p.bTryDecodeMOTHeader(a=%d,0x%08x,%d,%d) ...",
                    (void*)this,(int)a,
                    u32EntryTransportId,
                    u8DGType,
                    u32UserAppType));

    if( s32TryDecodeMOTHeaderElement(a,-1,u32EntryTransportId,object_body_bytes,object_body,bitidx,&u16ContentType,&u16ContentSubType) < 0 )
    {
        if(_dsm->oDB()->dbGetAutoCommit() == 0)
        {
            rc = iRoolbackSavePoint();
        }
        return FALSE;
    }
    else if( _doRollbackTransaction == FALSE )
    {
        /* success ...
         */
#ifdef CONFIG_DSM__ENABLE_DAB_EPG
#ifdef CONFIG_DSM__ENABLE_DAB_EPG_IN_HEADER_MODE
        {
            tclDSMDABEPGApp* _this = (tclDSMDABEPGApp*)dynamic_cast<tclDSMDABEPGApp*>(this);
            if( _this != NULL )
            {
                if( _this->bTryDecodeEPGObjectIfNessesary(NULL,a,u32EntryTransportId) == FALSE )
                {

                }
            }
        }
#else /* CONFIG_DSM__ENABLE_DAB_EPG_IN_HEADER_MODE */
        /*
         * no EPG without MOT directory
         */
#endif /* CONFIG_DSM__ENABLE_DAB_EPG_IN_HEADER_MODE */
#endif /* CONFIG_DSM__ENABLE_DAB_EPG */

#ifdef CONFIG_DSM__ENABLE_DAB_SLS
        {
            tclDSMDABSLSApp* _this = (tclDSMDABSLSApp*)dynamic_cast<tclDSMDABSLSApp*>(this);
            if( _this != NULL )
            {
                _this->bTryHandleSLSObject(u32EntryTransportId);
            }
        }
#endif /* CONFIG_DSM__ENABLE_DAB_SLS */
    }

    if( _doRollbackTransaction == TRUE )
    {
        if(_dsm->oDB()->dbGetAutoCommit() == 0)
        {
            rc = iRoolbackSavePoint();
        }
        return FALSE;
    }
    (void)rc;
    return TRUE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABMOTApp::vDeleteMOTDirectory( )
{
    SQLITE_API int rc;

    sqlite3_stmt* pStmt = NULL;
    rc = _dsm->oDB()->dbPrepare(
                "DELETE FROM dab_mot_directory WHERE a=?",
                -1,
                &pStmt,
                NULL);
    if( pStmt != NULL )
    {
        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                        1,
                                        s64GetA());
        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                break;
            }
        }
        _dsm->oDB()->dbReset(pStmt);
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABMOTApp::vClearCompleteFlag( )
{
    SQLITE_API int rc;

    sqlite3_stmt* pStmt = NULL;
    rc = _dsm->oDB()->dbPrepare(
                "UPDATE dab_data_application SET complete=0 WHERE o=?",
                -1,
                &pStmt,
                NULL);
    if( pStmt != NULL )
    {
        _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                        1,
                                        s64GetA());
        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( tclDB::dbOnStepError(rc) )
            {
                break;
            }
        }
        _dsm->oDB()->dbReset(pStmt);
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tU32 tclDSMDABMOTApp::u32GetSavedNxtMOTDirId( tS64 a )
{
    if( a != DSM_INVALID_APPID )
    {
        SQLITE_API int rc;
        sqlite3_stmt* pStmt = NULL;

        rc = _dsm->oDB()->dbPrepare(
                    "SELECT nxtmotdir FROM dab_data_application WHERE o=?",
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            1,
                                            a);
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 1 )
                    {
                        if( tclDB::dbColumnType(pStmt,0) == SQLITE_INTEGER )
                        {
                            tU32 nxtmotdir = tclDB::dbColumnInt(pStmt,0);
                            _dsm->oDB()->dbReset(pStmt);
                            return nxtmotdir;
                        }
                    }
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    _dsm->oDB()->dbReset(pStmt);
                    return 0;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }
    return 0;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABMOTApp::vUpdateNextMOTDir( )
{
    tS64 a = s64GetA();

    if( a != DSM_INVALID_APPID )
    {
#if 0
#ifdef TR_ENABLE
    {
        int rc;
        sqlite3_stmt* pStmt = NULL;

        rc = _dsm->oDB()->dbPrepare(
                    "SELECT a,dgtype,transpid,age_received,complete FROM dab_mot_object WHERE a=? ORDER BY transpid,dgtype",
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            int n=0;
            rc = _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                 1,
                                                 s64GetA());
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 5 )
                    {
                        ETG_TRACE_USR1(("%p.vUpdateNextMOTDir() n=%d a=%d available dgtype=%d transpid=0x%04x age=%d complete=%d",
                                        (void*)this,
                                        n++,
                                        tclDB::dbColumnInt( pStmt, 0 ),
                                        tclDB::dbColumnInt( pStmt, 1 ),
                                        tclDB::dbColumnInt( pStmt, 2 ),
                                        tclDB::dbColumnInt( pStmt, 3 ),
                                        tclDB::dbColumnInt( pStmt, 4 )));
                    }
                }
                else if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    break;
                }
            }
            rc = _dsm->oDB()->dbReset(pStmt);
        }

    }
#endif
#endif

        if( _u16NextMOTDirUpdateValue == TRUE )
        {
            SQLITE_API int rc;
            sqlite3_stmt* pStmt = NULL;
            _u16NextMOTDirUpdateValue = FALSE;

            ETG_TRACE_USR1(("%p.vUpdateNextMOTDir(a=%d) _u32NextMOTDirTransportId=%06x nxtmotdirobjectsleft=%d nMissingObjs=%s ...",
                            (void*)this,(int)a,
                            _u32NextMOTDirTransportId,
                            _tMissingMOTDirObjects.size(),
                            tMissingObjects().c_str()));

            rc = _dsm->oDB()->dbPrepare(
                        "UPDATE dab_data_application SET nxtmotdir=?,nxtmotdirobjectsleft=? WHERE o=?",
                        -1,
                        &pStmt,
                        NULL);
            if( pStmt != NULL )
            {
                int j=1;
                _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                              j++,
                                              _u32NextMOTDirTransportId);
                _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                              j++,
                                              _u16NextMOTDirObjectsLeft);
                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                j++,
                                                a);
                while(1)
                {
                    rc = _dsm->oDB()->dbStep(pStmt);
                    if( rc == SQLITE_DONE )
                    {
                        break;
                    }
                    else if( tclDB::dbOnStepError(rc) )
                    {
                        break;
                    }
                }
                _dsm->oDB()->dbReset(pStmt);
                pStmt = NULL;
            }

        }

        if( _u16NextMOTDirRunGC == TRUE )
        {
            _u16NextMOTDirRunGC = FALSE;
            tU32 u32SavedNextMOTDirid = u32GetSavedNxtMOTDirId(a);
            if(( u32SavedNextMOTDirid != 0 )&&( u32SavedNextMOTDirid==_u32NextMOTDirTransportId ))
            {
                vDeleteAllNotCurrentMOTDirectoryObjects(false);
            }
        }
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDSMDABMOTApp::vDeleteAllNotCurrentMOTDirectoryObjects(tBool bIgnoreOwnCompleteFlag)
{
    DSM_PERF_ENTRY();
    tS64 a = s64GetA();

    if( a != DSM_INVALID_APPID )
    {
        tBool bDoIt = bIgnoreOwnCompleteFlag;
        if( bDoIt == false )
        {
            bDoIt = bIsAppComplete(a);
        }
        if( bDoIt == true )
        {
            SQLITE_API int rc;
            sqlite3_stmt* pStmt = NULL;

            /*delete all old MOT MSC DGs where the transport ids are not in the new directory AND
              don't delete the datagroups of the directory itself*/
            rc = _dsm->oDB()->dbPrepare(
            #ifdef CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE
                        "DELETE FROM dab_mot_object WHERE "
                        "a=?1 AND "
                        "o NOT IN ("
                        "SELECT "
                        "O.o AS o "
                        "FROM "
                        "dab_mot_object AS O,"
                        "dab_mot_directory AS D,"
                        "dab_mot_directory_param AS P "
                        "WHERE "
                        "O.a=?1 AND "
                        "D.a=O.a AND O.transpid=P.transpid AND P.a=D.a AND (P.paramid=?2 OR P.paramid=?3)"
                        ")",
            #else
                        "DELETE FROM dab_datagroup_mot WHERE "
                        "a=?1 AND "
                        "transpid NOT IN ("
                        "SELECT "
                        "P.transpid as transpid "
                        "FROM "
                        "dab_mot_directory AS D,"
                        "dab_mot_directory_param AS P "
                        "WHERE "
                        "D.a=?1 AND P.a=D.a AND (P.paramid=?2 OR P.paramid=?3)"
                        ")",
            #endif
                        -1,
                        &pStmt,
                        NULL);
            if( pStmt != NULL )
            {
                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                1,
                                                a);

                _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                              2,
                                              MOT_DIRECTORY_PARAMID__BODYSIZE);

                _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                              3,
                                              MOT_DIRECTORY_PARAMID__DATACAROUSELPERIOD);

                while(1)
                {
                    rc = _dsm->oDB()->dbStep(pStmt);
                    if( rc == SQLITE_DONE )
                    {
                        break;
                    }
                    else if( tclDB::dbOnStepError(rc) )
                    {
                        break;
                    }
                }
                _dsm->oDB()->dbReset(pStmt);
            }
        }
    }
    DSM_PERF_EXIT("tclDSMDABMOTApp::vDeleteAllNotCurrentMOTDirectoryObjects");
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABMOTApp::bTryDecodeMOTDirectory( tS64 a, tU16 u16_transport_id, tU8 u8DGType, tU32 u32UserAppType, tU32 object_body_bytes, tU8* object_body )
{
    SQLITE_API int rc;
    tBool success;

    DSM_PERF_ENTRY();

    ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectory(a=%d,u8DGType=%d,u16_transport_id=0x%04x) _u32NextMOTDirTransportId=0x%06x ...",
                    (void*)this,(int)a,u8DGType,u16_transport_id,(int)_u32NextMOTDirTransportId));

    if( (tU32)(((u8DGType&0xff)<<16)|(u16_transport_id&0xffff)) == _u32NextMOTDirTransportId )
    {
        if(_tMissingMOTDirObjects.size() != 0 )
        {
            /*
             * processing is not needed, because we allready know what is missing
             */
            DSM_PERF_EXIT("tclDSMDABMOTApp::bTryDecodeMOTDirectory");
            ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectory(a=%d) u16_transport_id=%08x #1 skip processing ...",(void*)this,(int)a,_u32NextMOTDirTransportId));
            return FALSE;
        }
    }

    ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectory(a=%d) u16_transport_id=%08x #1 ...",(void*)this,(int)a,u16_transport_id));
    vDeleteMOTDirectory();

    ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectory(a=%d) u16_transport_id=%08x #2 ...",(void*)this,(int)a,u16_transport_id));
    vClearCompleteFlag();

    ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectory(a=%d) u16_transport_id=%08x #3 ...",(void*)this,(int)a,u16_transport_id));
    success = bTryDecodeMOTDirectoryCore(a,u16_transport_id,u8DGType,u32UserAppType,object_body_bytes,object_body);

#if 1
    if( success == TRUE)
    {
        success = bTryDecodeMOTDirectoryAppSpecificPreHandling( a, u16_transport_id, u8DGType );
    }
#endif

    if( success == TRUE)
    {
        std::list<tU32> list_epg_transpid;
        sqlite3_stmt* pStmt = NULL;
        _tMissingMOTDirObjects.clear();

        ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectory(a=%d) u16_transport_id=%08x #4 ...",(void*)this,(int)a,u16_transport_id));

#ifdef TR_ENABLE
#if 0
        {
            tU32 u32MOTDirCSUM = 0;
            int u32NMissing=0;
            int u32NAvailable=0;
            int u32NComplete=0;
            int rc;
            sqlite3_stmt* pStmt = NULL;

            rc = _dsm->oDB()->dbPrepare(
                        "SELECT P.transpid,(SELECT O.complete FROM v_dab_mot_object AS O WHERE O.a=D.a AND O.transpid=P.transpid AND O.dgtype=4) AS complete,(SELECT O.o FROM dab_mot_object AS O WHERE O.a=D.a AND O.transpid=P.transpid AND O.dgtype=4),(SELECT O.age_received FROM dab_mot_object AS O WHERE O.a=D.a AND O.transpid=P.transpid AND O.dgtype=4) FROM dab_mot_directory AS D,dab_mot_directory_param AS P WHERE D.transpid=? AND D.o=P.d AND D.a=? AND P.paramid=? ORDER BY P.transpid ASC",
                        -1,
                        &pStmt,
                        NULL);
            if( pStmt != NULL )
            {
                int n=0;
                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                1,
                                                u16_transport_id);

                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                2,
                                                a);

                _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                              3,
                                              MOT_DIRECTORY_PARAMID__BODYSIZE);
                while(1)
                {
                    rc = _dsm->oDB()->dbStep(pStmt);
                    if( rc == SQLITE_ROW )
                    {
                        if( tclDB::dbColumnCount(pStmt) == 4 )
                        {
                            if( tclDB::dbColumnType(pStmt, 1) == SQLITE_NULL )
                            {
                                ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectory() a=%d n=%d directory transpid=0x%04x o=%d age=%d (%d)",
                                                (void*)this,
                                                (int)s64GetA(),
                                                ++n,
                                                tclDB::dbColumnInt( pStmt, 0 ),
                                                tclDB::dbColumnInt( pStmt, 2 ),
                                                tclDB::dbColumnInt( pStmt, 3 ),
                                                ++u32NMissing));
                            }
                            else
                            {
                                if( tclDB::dbColumnInt( pStmt, 1 ) != 0 )
                                {
                                    u32NComplete++;
                                }
                                ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectory() a=%d n=%d directory transpid=0x%04x o=%d complete=%d age=%d (%d,%d)",
                                                (void*)this,
                                                (int)s64GetA(),
                                                ++n,
                                                tclDB::dbColumnInt( pStmt, 0 ),
                                                tclDB::dbColumnInt( pStmt, 2 ),
                                                tclDB::dbColumnInt( pStmt, 1 ),
                                                tclDB::dbColumnInt( pStmt, 3 ),
                                                ++u32NAvailable,u32NComplete));
                            }
                            u32MOTDirCSUM += tclDB::dbColumnInt( pStmt, 0 );
                        }
                    }
                    else if( rc == SQLITE_DONE )
                    {
                        break;
                    }
                    else if( tclDB::dbOnStepError(rc) )
                    {
                        break;
                    }
                }
                rc = _dsm->oDB()->dbReset(pStmt);
            }
            ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectory() a=%d u32MOTDirCSUM=0x%08x u32NAvailable=%d u32NComplete=%d u32NMissing=%d",
                            (void*)this,
                            (int)s64GetA(),
                            u32MOTDirCSUM,u32NAvailable,u32NComplete,u32NMissing));
        }
#endif
#endif

        /* check if all files complete for new directory, if not store new
         * u16_transport_id into nxtmotdir and rollback.
         */
        rc = _dsm->oDB()->dbPrepare(
            #if defined(CONFIG_DSM_ENABLE_USE_dab_mot_object_complete_TABLE) && defined(CONFIG_DSM_ENABLE_USE_dab_mot_object_complete_IN_bTryDecodeMOTDirectory)
                    "SELECT P.transpid FROM dab_mot_directory AS D,dab_mot_directory_param AS P WHERE D.transpid=? AND D.o=P.d AND D.a=? AND P.paramid=? AND P.transpid NOT IN (SELECT O.transpid FROM dab_mot_object_complete AS O WHERE O.a=P.a)",
            #else /* CONFIG_DSM_ENABLE_USE_dab_mot_object_complete_TABLEB */
                    "SELECT P.transpid FROM dab_mot_directory AS D,dab_mot_directory_param AS P WHERE D.transpid=? AND D.o=P.d AND D.a=? AND P.paramid=? AND P.transpid NOT IN (SELECT O.transpid FROM v_dab_mot_object AS O WHERE O.a=P.a AND O.complete=1)",
            #endif /* CONFIG_DSM_ENABLE_USE_dab_mot_object_complete_TABLEB */
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            1,
                                            u16_transport_id);

            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            2,
                                            a);

            _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                          3,
                                          MOT_DIRECTORY_PARAMID__BODYSIZE);

            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_ROW )
                {
                    if( tclDB::dbColumnCount(pStmt) == 1 )
                    {
                        tU16 u16MissingTranspId = (tU16)tclDB::dbColumnInt( pStmt, 0 );
                        _tMissingMOTDirObjects.insert(_tMissingMOTDirObjects.end(),u16MissingTranspId);
                        //printf("_tMissingMOTDirObjects.insert(%08x) dir=%08x\n",u16MissingTranspId,u16_transport_id);
                    }
                }
                else if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    break;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
        //app_is_complete = 1;

        ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectory(a=%d) u16_transport_id=%08x #5 ...",(void*)this,(int)a,u16_transport_id));
        /* genrate a list for EPG objects to be decode
         */
        {
            list_epg_transpid.clear();
            if( u32UserAppType == DAB_SERVICE_MANAGER_UAPPTYPE_EPG )
            {
                /* search for not decoded EPG objects ...
                 */
                rc = _dsm->oDB()->dbPrepare(
            #if defined (CONFIG_DSM_ENABLE_USE_dab_mot_object_complete_TABLE) && defined(CONFIG_DSM_ENABLE_USE_dab_mot_object_complete_IN_bTryDecodeMOTDirectory)
                            "SELECT O.dgtype,O.transpid FROM dab_mot_object_complete AS O,dab_mot_directory_param AS P WHERE O.a=? AND O.a=P.a AND O.dgtype=4 AND O.transpid=P.transpid AND P.paramid=? AND P.datafield=7 AND O.transpid NOT IN (SELECT transpid FROM dab_epg_root WHERE a=O.a)",
            #else
                            "SELECT O.dgtype,O.transpid FROM v_dab_mot_object AS O,dab_mot_directory_param AS P WHERE O.a=? AND O.a=P.a AND O.dgtype=4 AND O.transpid=P.transpid AND P.paramid=? AND P.datafield=7 AND O.complete=1 AND O.transpid NOT IN (SELECT transpid FROM dab_epg_root WHERE a=O.a)",
            #endif
                            -1,
                            &pStmt,
                            NULL);
                if( pStmt != NULL )
                {
                    _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                    1,
                                                    a);
                    _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                  2,
                                                  MOT_DIRECTORY_PARAMID__CONTENTTYPE);
                    while(1)
                    {
                        rc = _dsm->oDB()->dbStep(pStmt);
                        if( rc == SQLITE_DONE )
                        {
                            break;
                        }
                        else if( rc == SQLITE_ROW )
                        {
                            if( tclDB::dbColumnCount(pStmt) == 2 )
                            {
                                tU32 id = 0;
                                id |= (tclDB::dbColumnInt( pStmt, 0 )<<16);
                                id |= (tclDB::dbColumnInt( pStmt, 1 )<<0);
                                list_epg_transpid.insert( list_epg_transpid.end(),
                                                          id);
                            }
                        }
                        else if( tclDB::dbOnStepError(rc) )
                        {
                            break;
                        }
                    }
                    _dsm->oDB()->dbReset(pStmt);
                }
            }
        }

        ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectory(a=%d) u16_transport_id=%08x #6 nxtmotdirobjectsleft=%d nMissingObjs=%s ...",
                        (void*)this,(int)a,u16_transport_id,_tMissingMOTDirObjects.size(),tMissingObjects().c_str()));

        if(_tMissingMOTDirObjects.size() != 0 )
        {
            ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectory(a=%d) u16_transport_id=%08x #7 objects missing ...",(void*)this,(int)a,u16_transport_id));
            /*
             * rollback anything because i was not able to complete
             */

            _u16NextMOTDirRunGC = TRUE;
            _u16NextMOTDirUpdateValue = TRUE;
            _u32NextMOTDirTransportId = (((u8DGType&0xff)<<16)|(u16_transport_id&0xffff));
            _u16NextMOTDirObjectsLeft = (tU32)_tMissingMOTDirObjects.size();

            ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectory(a=%d,u8DGType=%d,u16_transport_id=0x%04x) _u32NextMOTDirTransportId=0x%06x (set) ...",
                            (void*)this,(int)a,u8DGType,u16_transport_id,(int)_u32NextMOTDirTransportId));

            DSM_PERF_EXIT("tclDSMDABMOTApp::bTryDecodeMOTDirectory");
            return FALSE;
        }

        _u32NextMOTDirTransportId = 0;
        ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectory(a=%d,u8DGType=%d,u16_transport_id=0x%04x) _u32NextMOTDirTransportId=0x%06x (cleared) ...",
                        (void*)this,(int)a,u8DGType,u16_transport_id,(int)_u32NextMOTDirTransportId));

        rc = _dsm->oDB()->dbPrepare(
                    "UPDATE dab_data_application SET complete=1,nxtmotdir=?,nxtmotdirobjectsleft=? WHERE o=?",
                    -1,
                    &pStmt,
                    NULL);
        if( pStmt != NULL )
        {
            int j=1;
            _dsm->oDB()->tclDB::dbBindNull(pStmt,
                                           j++);
            _dsm->oDB()->tclDB::dbBindNull(pStmt,
                                           j++);
            _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                            j++,
                                            a);
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    break;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
            pStmt = NULL;
        }

        ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectory(a=%d) u16_transport_id=%08x #8 nxtmotdirobjectsleft=%d ...",
                        (void*)this,(int)a,u16_transport_id,_tMissingMOTDirObjects.size()));

        if( _doRollbackTransaction == FALSE )
        {
            vDeleteAllNotCurrentMOTDirectoryObjects(TRUE/* complete flag not set yet, so ignore it*/);
        }

        ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectory(a=%d) u16_transport_id=%08x #9 ...",(void*)this,(int)a,u16_transport_id));

        if(( _tMissingMOTDirObjects.size() == 0 )&&( _doRollbackTransaction == FALSE ))
        {
            if((u32UserAppType == DAB_SERVICE_MANAGER_UAPPTYPE_EPG )&&(list_epg_transpid.size()>0))
            {
                tU32 u32EPGByteLimit = u32AppSizeLimit();
                u32EPGByteLimit = (u32EPGByteLimit==0)?0xffffffff:u32EPGByteLimit;
                /* decode new EPG objects
                 */
                std::list<tU32>::iterator it;
                for (it=list_epg_transpid.begin(); it!=list_epg_transpid.end(); ++it)
                {
                    tU32 id = *it;
                    if( ((id>>16)&0xff) == 4 )
                    {
#ifdef CONFIG_DSM__ENABLE_DAB_EPG
                        tclDSMDABEPGApp* _this = (tclDSMDABEPGApp*)dynamic_cast<tclDSMDABEPGApp*>(this);
                        if( _this != NULL )
                        {
                            if( _this->bTryDecodeEPGObjectIfNessesary(&u32EPGByteLimit,a,id&0xffff) == FALSE )
                            {
                                break;
                            }
                        }
#endif /* CONFIG_DSM__ENABLE_DAB_EPG */
                    }
                }
            }
        }

        ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectory(a=%d) u16_transport_id=%08x #10 ...",(void*)this,(int)a,u16_transport_id));

#ifdef ENABLE_CLEAR_MOTDIR_DG_SEGM_WHEN_DECODED
        if( _doRollbackTransaction == FALSE )
        {
            SQLITE_API int rc;
            sqlite3_stmt* pStmt = NULL;
            rc = _dsm->oDB()->dbPrepare(
            #ifdef CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE
                        "DELETE FROM dab_mot_object_segm WHERE o IN (SELECT o FROM dab_mot_object WHERE a=? AND transpid=? AND dgtype=?)",
            #else
                        "UPDATE dab_datagroup_mot SET segm=NULL WHERE a=? AND transpid=? AND dgtype=?",
            #endif
                        -1,
                        &pStmt,
                        NULL);
            if( pStmt != NULL )
            {
                rc = _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                     1,
                                                     a);
                rc = _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                   2,
                                                   u16_transport_id);
                rc = _dsm->oDB()->tclDB::dbBindInt(pStmt,
                                                   3,
                                                   u8DGType);

                while(1)
                {
                    rc = _dsm->oDB()->dbStep(pStmt);
                    if( rc == SQLITE_DONE )
                    {
                        break;
                    }
                    else if( tclDB::dbOnStepError(rc) )
                    {
                        break;
                    }
                }
                rc = _dsm->oDB()->dbReset(pStmt);
            }
        }
#endif

        ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectory(a=%d) u16_transport_id=%08x #11 ...",(void*)this,(int)a,u16_transport_id));

        if( _doRollbackTransaction == TRUE )
        {
            if(_dsm->oDB()->dbGetAutoCommit() == 0)
            {
                rc = iRoolbackSavePoint();
            }
        }
        else
        {
        }

        vCheckAndSendAppComplete(TRUE);

        DSM_PERF_EXIT("tclDSMDABMOTApp::bTryDecodeMOTDirectory");
        return TRUE;
    }
    else
    {
        if(_dsm->oDB()->dbGetAutoCommit() == 0)
        {
            rc = iRoolbackSavePoint();
        }
        DSM_PERF_EXIT("tclDSMDABMOTApp::bTryDecodeMOTDirectory");
        return FALSE;
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
#define MOTDIR_PARAMID_1            1
#define MOTDIR_PARAMID_TRIGGERTIME  5
#define MOTDIR_PARAMID_6            6
#define MOTDIR_PARAMID_9            9
#define MOTDIR_PARAMID_CNAME        12
#define MOTDIR_PARAMID_VERSION      13
#define MOTDIR_PARAMID_16           16
#define MOTDIR_PARAMID_17           17
#define MOTDIR_PARAMID_32           32
#define MOTDIR_PARAMID_34           34
#define MOTDIR_PARAMID_11           11
tBool tclDSMDABMOTApp::bTryDecodeMOTDirectoryParameter( tS64 a, tS64 d, tU16 u16_transport_id, tU32 u32PLI, tU32 u32ParamId, tU32 u32DatafieldLength, const tU8* pDatafield, tS64 u64DataValue )
{
    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;

    if( pDatafield != NULL )
    {
    #ifdef TR_ENABLE_ALLOW_MORE_THEN_10_PARAMETERS
        if( u32ParamId == MOTDIR_PARAMID_CNAME )
        {
            ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectoryParameter(%lld,%lld,%d/0x%04x,u32ParamId=0x%04x) pDatafield[%d]=%02x.%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c parameter ... ",
                            (void*)this,
                            a,
                            d,
                            u16_transport_id,
                            u16_transport_id,
                            u32ParamId,
                            u32DatafieldLength,
                            ((u32DatafieldLength>0)?((char*)pDatafield[0]):""),
                            ((u32DatafieldLength>1)?((char*)pDatafield[1]):""),
                            ((u32DatafieldLength>2)?((char*)pDatafield[2]):""),
                            ((u32DatafieldLength>3)?((char*)pDatafield[3]):""),
                            ((u32DatafieldLength>4)?((char*)pDatafield[4]):""),
                            ((u32DatafieldLength>5)?((char*)pDatafield[5]):""),
                            ((u32DatafieldLength>6)?((char*)pDatafield[6]):""),
                            ((u32DatafieldLength>7)?((char*)pDatafield[7]):""),
                            ((u32DatafieldLength>8)?((char*)pDatafield[8]):""),
                            ((u32DatafieldLength>9)?((char*)pDatafield[9]):""),
                            ((u32DatafieldLength>10)?((char*)pDatafield[10]):""),
                            ((u32DatafieldLength>11)?((char*)pDatafield[11]):""),
                            ((u32DatafieldLength>12)?((char*)pDatafield[12]):""),
                            ((u32DatafieldLength>13)?((char*)pDatafield[13]):""),
                            ((u32DatafieldLength>14)?((char*)pDatafield[14]):""),
                            ((u32DatafieldLength>15)?((char*)pDatafield[15]):""),
                            ((u32DatafieldLength>16)?((char*)pDatafield[16]):"")));
        }
        else
        {
            ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectoryParameter(%lld,%lld,%d/0x%04x,u32ParamId=0x%04x) pDatafield[%d]=%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x parameter ... ",
                            (void*)this,
                            a,
                            d,
                            u16_transport_id,
                            u16_transport_id,
                            u32ParamId,
                            u32DatafieldLength,
                            ((u32DatafieldLength>0)?(pDatafield[0]):0),
                            ((u32DatafieldLength>1)?(pDatafield[1]):0),
                            ((u32DatafieldLength>2)?(pDatafield[2]):0),
                            ((u32DatafieldLength>3)?(pDatafield[3]):0),
                            ((u32DatafieldLength>4)?(pDatafield[4]):0),
                            ((u32DatafieldLength>5)?(pDatafield[5]):0),
                            ((u32DatafieldLength>6)?(pDatafield[6]):0),
                            ((u32DatafieldLength>7)?(pDatafield[7]):0),
                            ((u32DatafieldLength>8)?(pDatafield[8]):0),
                            ((u32DatafieldLength>9)?(pDatafield[9]):0),
                            ((u32DatafieldLength>10)?(pDatafield[10]):0),
                            ((u32DatafieldLength>11)?(pDatafield[11]):0),
                            ((u32DatafieldLength>12)?(pDatafield[12]):0),
                            ((u32DatafieldLength>13)?(pDatafield[13]):0),
                            ((u32DatafieldLength>14)?(pDatafield[14]):0),
                            ((u32DatafieldLength>16)?(pDatafield[15]):0)));
        }
        #else
            ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectoryParameter(%lld,%lld,%d/0x%04x,u32ParamId=0x%04x) pDatafield[%d]=... parameter ... ",
                            (void*)this,
                            a,
                            d,
                            u16_transport_id,
                            u16_transport_id,
                            u32ParamId,
                            u32DatafieldLength));
        #endif
    }
    else
    {
        ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectoryParameter(%lld,%lld,%d/0x%04x,u32ParamId=0x%04x) u64DataValue=%ul parameter ... ",
                        (void*)this,
                        a,
                        d,
                        u16_transport_id,
                        u16_transport_id,
                        u32ParamId,
                        (unsigned long)u64DataValue));
    }

    if( u32ParamId == MOTDIR_PARAMID_CNAME )
    {
        /* Content Name:
         */
        rc = _dsm->oDB()->dbPrepare(
                    "REPLACE INTO dab_mot_directory_cname (a,d,transpid,cset,contentname) VALUES(?,?,?,?,?)",
                    -1,
                    &pStmt,
                    NULL);

        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt64(pStmt, 1, a);

            if( d == -1 )
            {
                _dsm->oDB()->tclDB::dbBindNull(pStmt, 2 );
            }
            else
            {
                _dsm->oDB()->tclDB::dbBindInt64(pStmt, 2, d);
            }

            _dsm->oDB()->tclDB::dbBindInt(pStmt, 3, u16_transport_id);

            if( u32DatafieldLength == 0 )
            {
                _dsm->oDB()->dbReset(pStmt);
                return TRUE;
            }
            else
            {
                if( u32PLI == 0 )
                {
                    _dsm->oDB()->dbReset(pStmt);
                    return TRUE;
                }
                else if(( u32DatafieldLength > 1)&&(pDatafield != NULL))
                {
                    _dsm->oDB()->tclDB::dbBindInt(pStmt, 4, (pDatafield[0]>>0)%0xf);

                    _dsm->oDB()->tclDB::dbBindText(pStmt,
                                                   5,
                                                   (const char*)&pDatafield[1],
                            u32DatafieldLength-1,
                            NULL);
                }
                else
                {
                    _dsm->oDB()->dbReset(pStmt);
                    return TRUE;
                }
            }

            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    _dsm->oDB()->dbReset(pStmt);
                    rc = iRoolbackSavePoint();
                    return FALSE;
                }
            }

            _dsm->oDB()->dbReset(pStmt);
        }
        return TRUE;
    }

    if( u32ParamId == MOTDIR_PARAMID_TRIGGERTIME )
    {
        tS64 s64UTCTriggerTime=0;
        tS32 mjd=0;
        tS32 utc=0;

        /* first convert the trigger time to utc_triggertime*/
        if(( u32DatafieldLength != 0 )&&
                (pDatafield != NULL))
        {
            /*check validity flag*/
            if ((pDatafield[0]&0x80)==0)
            {
                s64UTCTriggerTime = 0 ;
            }
            else
            {
                mjd = ((pDatafield[0]&0x7f)<<10)|
                        ((pDatafield[1]&0xff)<<2)|
                        (pDatafield[2]&0xc0)>>6;

                if (u32DatafieldLength==6)
                {
                    utc=((pDatafield[2]&0x07)<<24)|
                            ((pDatafield[3]&0xff)<<16)|
                            ((pDatafield[4]&0xff)<<8)|
                            (pDatafield[5]&0xff);
                }
                else if (u32DatafieldLength==4)
                {
                    utc=((pDatafield[2]&0x07)<<24)|
                            ((pDatafield[3]&0xff)<<16);
                }
                else
                {
                    /*invalid*/
                    utc=0;
                }

                if(( mjd >= MJD_1_1_1970 )&&(utc != 0 ))
                {
                    s64UTCTriggerTime = (tU64)(mjd - MJD_1_1_1970) * 24 * 60 * 60;

                    s64UTCTriggerTime += ((0x1f&(utc >> (6+6+10))) * (60*60));	/* add hours */
                    s64UTCTriggerTime += ((0x3f&(utc >> (6+10))) * (60));  	    /* add minutes */
                    s64UTCTriggerTime += ((0x3f&(utc >> (10))) * (1));  		/* add seconds */
                }
            }

            /* TriggerTime:
         */
            /* process Directory Extensions ...
         */
            rc = _dsm->oDB()->dbPrepare(
                        "REPLACE INTO dab_mot_directory_param (a,d,transpid,paramid,datafield) VALUES(?,?,?,?,?)",
                        -1,
                        &pStmt,
                        NULL);

            if( pStmt != NULL )
            {
                _dsm->oDB()->tclDB::dbBindInt64(pStmt, 1, a);

                if( d == -1 )
                {
                    _dsm->oDB()->tclDB::dbBindNull(pStmt, 2 );
                }
                else
                {
                    _dsm->oDB()->tclDB::dbBindInt64(pStmt, 2, d);
                }

                _dsm->oDB()->tclDB::dbBindInt(pStmt, 3, u16_transport_id);

                _dsm->oDB()->tclDB::dbBindInt(pStmt, 4, u32ParamId);

                _dsm->oDB()->tclDB::dbBindInt64(pStmt, 5, s64UTCTriggerTime);

                while(1)
                {
                    rc = _dsm->oDB()->dbStep(pStmt);
                    if( rc == SQLITE_DONE )
                    {
                        break;
                    }
                    else if( tclDB::dbOnStepError(rc) )
                    {
                        _dsm->oDB()->dbReset(pStmt);
                        return FALSE;
                    }
                }
                _dsm->oDB()->dbReset(pStmt);
            }
            return TRUE;
        }
    }

    {
        /* process Directory Extensions ...
         */
        rc = _dsm->oDB()->dbPrepare(
                    "REPLACE INTO dab_mot_directory_param (a,d,transpid,paramid,datafield) VALUES(?,?,?,?,?)",
                    -1,
                    &pStmt,
                    NULL);

        if( pStmt != NULL )
        {
            _dsm->oDB()->tclDB::dbBindInt64(pStmt, 1, a);

            if( d == -1 )
            {
                _dsm->oDB()->tclDB::dbBindNull(pStmt, 2 );
            }
            else
            {
                _dsm->oDB()->tclDB::dbBindInt64(pStmt, 2, d);
            }

            _dsm->oDB()->tclDB::dbBindInt(pStmt, 3, u16_transport_id);

            _dsm->oDB()->tclDB::dbBindInt(pStmt, 4, u32ParamId);

            if( u32DatafieldLength == 0 )
            {
                _dsm->oDB()->tclDB::dbBindInt64(pStmt,5,
                                                u64DataValue);
            }
            else if( pDatafield == NULL )
            {
                _dsm->oDB()->tclDB::dbBindNull(pStmt,5);
            }
            else
            {
                if( u32PLI == 0 )
                {
                    _dsm->oDB()->dbReset(pStmt);
                    return FALSE;
                }
                else if( u32PLI == 1 )
                {
                    _dsm->oDB()->tclDB::dbBindInt(pStmt,5,
                                                  u32BitStreamValue(pDatafield,u32DatafieldLength,0,8));
                }
                else if( u32PLI == 2 )
                {
                    _dsm->oDB()->tclDB::dbBindInt64(pStmt,5,
                                                    u32BitStreamValue(pDatafield,u32DatafieldLength,0,32));
                }
                else if(( u32PLI == 3 )&&(u32DatafieldLength != 0))
                {
                    _dsm->oDB()->tclDB::dbBindBlob(pStmt,
                                                   5,
                                                   pDatafield,
                                                   u32DatafieldLength,
                                                   NULL);
                }
            }
            while(1)
            {
                rc = _dsm->oDB()->dbStep(pStmt);
                if( rc == SQLITE_DONE )
                {
                    break;
                }
                else if( tclDB::dbOnStepError(rc) )
                {
                    _dsm->oDB()->dbReset(pStmt);
                    return FALSE;
                }
            }
            _dsm->oDB()->dbReset(pStmt);
        }
    }

    return TRUE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDSMDABMOTApp::bTryDecodeMOTDirectoryCore( tS64 a, tU16 u16_transport_id, tU8 u8DGType, tU32 u32UserAppType, tU32 object_body_bytes, tU8* object_body )
{
    (void)u32UserAppType;
    if( u8DGType == 0x07 /* MOT Directory Compressed */ )
    {
        /* todo */
        DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
    }
    else if( u8DGType == 0x06 /* MOT Directory Uncompressed */ )
    {
        tU32 bitidx = 0;
        if( u32BitStreamValue(object_body,object_body_bytes,bitidx,1) != 0 )
        {
            /* CF (CompressionFlag): This bit shall be set to 0. */
            DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
            return FALSE;
        }
        bitidx+=1;

        if( u32BitStreamValue(object_body,object_body_bytes,bitidx,1) != 0 )
        {
            /* Rfu: This 1-bit field shall be reserved for future use of the remainder of the structure.
             * The bit shall be set to zero for the currently specified definition of the MOT directory.
             * An MOT decoder has to verify the value of this bit.
             */
            DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
            return FALSE;
        }
        bitidx+=1;

        /* Indicates the total size of the MOT directory in bytes.
         */
        tU32 u32DirectorySize = u32BitStreamValue(object_body,object_body_bytes,bitidx,30);
        bitidx+=30;

        if( u32DirectorySize > object_body_bytes )
        {
            DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
            return FALSE;
        }

        /* Indicates the total number N of objects described by the directory.
         */
        tU32 u32NumberOfObjects = u32BitStreamValue(object_body,object_body_bytes,bitidx,16);
        bitidx+=16;

        /* Indicates the maximum time in tenths of a second for the data carousel to complete a cycle.
         * It is the longest time taken for any object in the data carousel to be retransmitted.
         * A value of 0 shall indicate that the DataCarouselPeriod is undefined.
         * NOTE: This is the case if the data carousel only makes one turn or the bit-rate changes dynamically.
         */
        tU32 u32DataCarouselPeriod = u32BitStreamValue(object_body,object_body_bytes,bitidx,24);
        bitidx+=24;

        if( u32BitStreamValue(object_body,object_body_bytes,bitidx,1) != 0 )
        {
            /* Rfu: This 1-bit field shall be reserved for future use of the remainder of the structure.
             * The bit shall be set to zero for the currently specified definition of the MOT directory.
             * An MOT decoder has to verify the value of this bit.
             */
            DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
            return FALSE;
        }
        bitidx+=1;

        /* Rfa: This 2-bit field shall be reserved for future additions.
         * The bits shall be set to zero until they are defined.
         */
        bitidx+=2;

        /* SegmentSize: Indicates the SegmentSize in bytes that will be used for the segmentation
         * of objects within the MOT data carousel.
         * A value of 0 indicates that objects within the data carousel may have different segmentation sizes.
         * The last segment of an object may be smaller than this SegmentSize.
         */
        //tU32 u32SegmentSize = u32BitStreamValue(object_body,object_body_bytes,bitidx,13);
        bitidx+=13;

        /* DirectoryExtensionLength: Indicates the total number of following directory extension bytes.
         */
        tU32 u32DirectoryExtensionLength = u32BitStreamValue(object_body,object_body_bytes,bitidx,16);
        bitidx+=16;

        /* Insert Line in dab_mot_directory table
         */
        {
            tS64 d = 0;

            SQLITE_API int rc;
            sqlite3_stmt* pStmt = NULL;
            rc = _dsm->oDB()->dbPrepare(
                        "REPLACE INTO dab_mot_directory (a,transpid,dab_timestamp,period) VALUES(?,?,?,?)",
                        -1,
                        &pStmt,
                        NULL);
            if( pStmt != NULL )
            {
                _dsm->oDB()->tclDB::dbBindInt64(pStmt, 1, a);

                _dsm->oDB()->tclDB::dbBindInt(pStmt, 2, u16_transport_id);

                _dsm->oDB()->tclDB::dbBindInt64(pStmt,
                                                3,
                                                u64CTimeUTC());

                _dsm->oDB()->tclDB::dbBindInt(pStmt, 4, u32DataCarouselPeriod);

                ETG_TRACE_USR1(("%p.bTryDecodeMOTDirectoryCore(%lld,%d/0x%04x) dab_timestamp=%lld decode MOT directory ... ",
                                (void*)this,
                                a,
                                u16_transport_id,
                                u16_transport_id,
                                u64CTimeUTC()));

                while(1)
                {
                    rc = _dsm->oDB()->dbStep(pStmt);
                    if( rc == SQLITE_DONE )
                    {
                        d = sqlite3_last_insert_rowid(_dsm->oDB()->pHandle());
                        break;
                    }
                    else if( tclDB::dbOnStepError(rc) )
                    {
                        _dsm->oDB()->dbReset(pStmt);
                        DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
                        return FALSE;
                    }
                }
                _dsm->oDB()->dbReset(pStmt);
                pStmt = NULL;
            }

            if( bTryDecodeMOTDirectoryParameter(a,d,u16_transport_id,0,MOT_DIRECTORY_PARAMID__DATACAROUSELPERIOD,0,NULL,u32DataCarouselPeriod) == FALSE )
            {
                _dsm->oDB()->dbReset(pStmt);
                DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
                return FALSE;
            }

            if( d == 0 )
            {
                DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
            }
            else
            {
                for( int i=0;i<(int)u32DirectoryExtensionLength;)
                {
                    tU32 u32PLI = u32BitStreamValue(object_body,object_body_bytes,bitidx,2);
                    bitidx+=2;

                    tU32 u32ParamId= u32BitStreamValue(object_body,object_body_bytes,bitidx,6);
                    bitidx+=6;

                    tU32 u32DatafieldLength = 0;
                    if( u32PLI == 0 )
                    {
                        u32DatafieldLength = 0;
                    }
                    else if( u32PLI == 1 )
                    {
                        u32DatafieldLength = 1;
                        if( bTryDecodeMOTDirectoryParameter(a,d,u16_transport_id,u32PLI,u32ParamId,0,NULL,u32BitStreamValue(object_body,object_body_bytes,bitidx,8)) == FALSE )
                        {
                            _dsm->oDB()->dbReset(pStmt);
                            DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
                            return FALSE;
                        }
                    }
                    else if( u32PLI == 2 )
                    {
                        u32DatafieldLength = 4;
                        if( bTryDecodeMOTDirectoryParameter(a,d,u16_transport_id,u32PLI,u32ParamId,0,NULL,u32BitStreamValue(object_body,object_body_bytes,bitidx,32)) == FALSE )
                        {
                            _dsm->oDB()->dbReset(pStmt);
                            DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
                            return FALSE;
                        }
                    }
                    else if( u32PLI == 3 )
                    {
                        if( u32BitStreamValue(object_body,object_body_bytes,bitidx++,1) == 0 )
                        {
                            u32DatafieldLength = u32BitStreamValue(object_body,object_body_bytes,bitidx,7);
                            bitidx+=7;
                            i += 1;
                        }
                        else
                        {
                            u32DatafieldLength = u32BitStreamValue(object_body,object_body_bytes,bitidx,15);
                            bitidx+=15;
                            i += 2;
                        }
                        if( ((bitidx>>3) + u32DatafieldLength) > object_body_bytes )
                        {
                            _dsm->oDB()->dbReset(pStmt);
                            DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
                            return FALSE;
                        }

                        if( bTryDecodeMOTDirectoryParameter(a,d,u16_transport_id,u32PLI,u32ParamId,u32DatafieldLength,&object_body[bitidx>>3],0) == FALSE )
                        {
                            _dsm->oDB()->dbReset(pStmt);
                            DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
                            return FALSE;
                        }
                    }

                    bitidx += (u32DatafieldLength<<3);

                    i += 1;
                    i += u32DatafieldLength;
                }

                /* process Directory Entrys ...
                 */
                for( int o=0;o<(int)u32NumberOfObjects;o++)
                {
                    tU16 u16EntryTransportId = (tU16)u32BitStreamValue(object_body,object_body_bytes,bitidx,16);
#if defined(ENABLE_GTEST)
                    u16EntryTransportId = (u16EntryTransportId + tclDSMDABMOTApp::_u16GTestTransportIdOffset) & 0xffff;
#endif
                    bitidx+=16;

                    {
                        int consumed_bits = s32TryDecodeMOTHeaderElement(a,d,u16EntryTransportId,object_body_bytes,object_body,bitidx,NULL,NULL);
                        if( consumed_bits >= 0 )
                        {
                            bitidx = consumed_bits;
                        }
                        else
                        {
                            _dsm->oDB()->dbReset(pStmt);
                            return FALSE;
                        }
                    }
                }

                if( (bitidx>>3) == object_body_bytes )
                {
                    return TRUE;
                }
                if( (bitidx>>3) <= u32DirectorySize )
                {
                    return TRUE;
                }
                DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
            }
            DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
        }
        DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
    }
    DSM_MOT_DIRECTORY_FATAL_DECODING_ERROR();
    return FALSE;
}

#endif /* CONFIG_DSM__ENABLE_DAB */

/******************************************************************************
 * EOF
 *****************************************************************************/

