/******************************************************************************
*
* FILE: tclDSM.h
* PROJECT: AaRS
* SW-COMPONENT: Data Service Manager
*
* DESCRIPTION: This file contains the DSM class interface specification
*
* AUTHOR: CM-AI/ECB2-Scholz
*
* COPYRIGHT: (c) 2010 Robert Bosch Multimedia GmbH
* HISTORY:
* Date | Rev. | Author | Modification
* ----------------------------------------------------------------------------
* 23.04.2014 | 1.0 | CM-AI/ECB2-Scholz | Initial revision
*
*****************************************************************************/

/*
 * @startuml
 * namespace AaRSLib {
 *         namespace MSG {
 *                 class tMessage {
 *                 }
 *                 tCommandMessage <|-- tMessage
 *                 tResponseMessage <|-- tMessage
 *         }
 * }
 * @enduml
 * @startuml
 * namespace AaRSDABLib {
 *         namespace API {
 *                 namespace DC {
 *                         class tR_DATACHANNEL_FRAME {
 *                         }
 *                 }
 *
 *                 namespace RDM {
 *                 }
 *                 namespace DB {
 *                         tC_QUERY <|-- AaRSLib.MSG.tCommandMessage
 *                         tR_QUERY <|-- AaRSLib.MSG.tResponseMessage
 *                 }
 *                 namespace DDM {
 *                 }
 *         }
 * }
 * @enduml
 * @startuml
 * namespace dsm {
 *
 *         tclDSM o-- dsm.tclDB
 *         tclDSM o-- tclDABServiceManager
 *         tclDSM o-- tclDRMServiceManager
 *         tclDSM o-- tclDSMListener
 *         class tclDSM {
 *           dsm.tclDB* oDB()
 *           dsm.tclDABServiceManager* oDAB();
 *           dsm.tclDRMServiceManager* oDRM();
 *           tVoid vHandle() : called cyclic by owner
 *           ~ tVoid vBroadCastEvent( tDSMEvent ev, String uri )
 *         }
 *         class tclDB {
 *           - sqlite3* hnd
 *           tBool bOpen( tChar* fname )
 *           tVoid vClose( )
 *           sqlite3* pHandle();
 *         }
 *         tclServiceManager - tclDSM : using
 *         class tclServiceManager {
 *             - tclDSM* _dsm
 *                 tclServiceManager( tclDSM* dsm )
 *                 tVoid vReset() : reset internal state and restart
 *                 tVoid vShutdown() : sync database file and shutdown
 *                 tVoid vDeleteAll() : delete all data
 *             tBool bGetObject( String uri, tPU8 obj_buffer, tU32 obj_buffer_length )
 *             tBool bDataServiceComplete( String uri )
 *             - tVoid vHandle() : called DSM.vHandle
 *         }
 *         tclServiceManager <|-- tclDABServiceManager : using
 *         tclDABServiceManager o-- tclDABServiceManagerCommandReceiver
 *         class tclDABServiceManager {
 *             * provides SQLite table dab_data_service
 *             * provides SQLite view v_dab_mot_object
 *             * provides SQLite view v_dab_jrnl_object
 *                 tclDABServiceManager( tclDSM* dsm )
 *                 tVoid vSetCommandReceiver( tclDABServiceManagerCommandReceiver* meca_cmd_receiver )
 *                 tVoid vHandleMeCaResponse( AaRSLib.MSG.tMessage* msg )
 *                 tVoid vSetServiceEnabled( tBool onoff, String uri )
 *                 tVoid vSetEPGDecodingEnabled( tBool onoff )
 *                 tVoid vSetSLSDecodingEnabled( tBool onoff )
 *                 tVoid vSetBWSDecodingEnabled( tBool onoff )
 *                 tVoid vSetJRNLDecodingEnabled( tBool onoff )
 *             - tVoid sqlite_func_MOT_object() : called by SQL query
 *             - tVoid sqlite_func_JRNL_object() : called by SQL query
 *         }
 *         interface tclDABServiceManagerCommandReceiver {
 *                 tVoid vSendMeCaCommand( AaRSLib.MSG.tMessage* msg )
 *         }
 *         tclServiceManager <|-- tclDRMServiceManager : using
 *         class tclDRMServiceManager {
 *                 tclDRMServiceManager(tclDSM* dsm)
 *                 note: r.f.u.
 *         }
 *         interface tclDSMListener {
 *                 tVoid vDSMEvent( tDSMEvent ev, String uri )
 *         }
 *         enum tDSMEvent {
 *           DSM_EV_APPLICATION_STARTED
 *           DSM_EV_APPLICATION_STOPPED
 *           DSM_EV_APPLICATION_CANCELED
 *           DSM_EV_APPLICATION_COMPLETE
 *           DSM_EV_APPLICATION_OBJECT_COMPLETE
 *         }
 * }
 * dsm.tclDSMListener <|-- tclFCDataService : implements
 * dsm.tclDABServiceManagerCommandReceiver <|-- tclFCDataService : implements
 * tclFCDataService o-- dsm.tclDSM : owns
 * class tclFCDataService {
 *         - tclDSM* _dsm
 * }
 * @enduml
 *
 */

#ifndef _TCLDSM_HEADER_
#define _TCLDSM_HEADER_

#define CONFIG_DSM__ENABLE_DAB

#include <string>
#include <list>
#include <map>
#include <vector>
#include "DSMConfig.h"
#include "sqlite3.h"
#include "aarsLib.h"
#include "aarsDABLib.h"

#define DSM_PARAM_KEY__CURRENT_UTC_TIMESTAMP  1
#define DSM_PARAM_KEY__CURRENT_UTC_TIMEOFFSET 2

#ifndef DSM_TRACE_PERF
#define DSM_TRACE_PERF(_source_,_cnt_,_time_,_timemax_,_timesum_)
#endif

#define DSM_INVALID_APPID -1
#define DSM_INVALID_ROWID -1

namespace dsm
{
  /*
   * some forward definition
   */
  class tclDB;
  class tclDSM;
  class tclDSMTimeProvider;
  class tclDSMEventListener;
  class tclServiceManager;
  class tclDSMApp;
#ifdef CONFIG_DSM__ENABLE_DAB
  class tclDABServiceManager;
  class tclDABServiceManagerCommandReceiver;
  class tclDABSLSHandler;
  class tclDSMDABWizard;
  class tclDSMDABApp;
  class tclDSMDABMOTApp;
  class tclDSMDABEPGApp;
#ifdef CONFIG_DSM__ENABLE_DAB_JOURNALINE
  class tclDSMDABJRLNApp;
#endif
  class tclDSMDABSLSApp;
  class tclDSMDABBWSApp;
#endif /* CONFIG_DSM__ENABLE_DAB */

  /******************************************************************************
   ******************************************************************************
   * class tclDB ...
   ******************************************************************************
   *****************************************************************************/
  class tclDB
  {
    /* public methods & data ...
     */
  public:
    tclDB(class tclDSM* dsm);
    tBool bOpen( const tChar* fname );
    tBool bOpen( const tChar* fname, tU32 u32FileLimit, tBool bTruncate = false, tBool bInMemory = false );
    tU32 u32FileLimit( ) { return _u32FileLimit; };
    tVoid vSetFileLimit( tU32 u32FileLimit );
    tVoid vSetUserVersion( int user_version );
    tBool bIsNew();
    tBool bIsFull();
    tBool bIsToBig();
    tVoid vDeleteAll( );
    tVoid vClose();
    tBool bIsOpen() { return (_db!=NULL?TRUE:FALSE); };
    tVoid vWALCheckPoint();
    tBool bIsLocked();
    sqlite3* pHandle();
    tU32 u32FileErrorCounter( ) { tU32 r = _u32FileErrorCounter; _u32FileErrorCounter = 0; return r; };
    tVoid vDebugPrintSQLiteErrorMsg(int rc);
    static tVoid vDebugPrintSQLiteErrorStr(int rc);
    tBool bIsInMemory() {
        return _bInMemory;
    };

#if defined(ENABLE_GTEST)
  public:
      static tBool _bReportSQLITE3_ERROR;
#endif

    tS32 s32dbExecSimple(const char *zSql);

  protected:
    tVoid vFinalizeAllStatements();

  private:
    sqlite3_stmt* pStmtPrepareCache;

  private:
    char    _dbfilename[1024];
    static int _u8SQLiteInitialized;
    tBool   _bInMemory;

  public:
    SQLITE_API tU32 u32dbExecSimple(const char *zSql);
    SQLITE_API int dbExec(const char *zSql,sqlite3_callback xCallback,void *pArg,char **pzErrMsg);
#ifdef CONFIG_DSM_ENABLE_EXPLAIN_SQL
    SQLITE_API int dbExplainSQL(const char *zSql);
#endif
    SQLITE_API int dbPrepare(const char *zSql,int nBytes,sqlite3_stmt **ppStmt,const char **pzTail);
    SQLITE_API int dbStep(sqlite3_stmt *pStmt);
    SQLITE_API int dbReset(sqlite3_stmt *pStmt);
    SQLITE_API int dbFinalize(sqlite3_stmt *pStmt);

    SQLITE_API int dbCreateFunction(
      const char *zFunc,
      int nArg,
      int enc,
      void *p,
      void (*xFunc)(sqlite3_context*,int,sqlite3_value **),
      void (*xStep)(sqlite3_context*,int,sqlite3_value **),
      void (*xFinal)(sqlite3_context*),
      void (*xDestroy)(void *)
    );

    static SQLITE_API tBool dbOnStepError(int rc);

    static SQLITE_API int dbColumnCount(sqlite3_stmt* pStmt);
    static SQLITE_API const char *dbColumnName(sqlite3_stmt*, int iCol);

    SQLITE_API int dbGetAutoCommit();

    SQLITE_API int dbChanges();
    SQLITE_API tS64 dbLastInsertRowId();

    static void* dbMalloc(int bytes);
    static void dbFree(void* p);

    static SQLITE_API int dbBindBlob(sqlite3_stmt*, int, const void*, int n, void(*)(void*));
    static SQLITE_API int dbBindDouble(sqlite3_stmt*, int, double);
    static SQLITE_API int dbBindInt(sqlite3_stmt*, int, int);
    static SQLITE_API int dbBindInt64(sqlite3_stmt*, int, tS64);
    static SQLITE_API int dbBindNull(sqlite3_stmt*, int);
    static SQLITE_API int dbBindText(sqlite3_stmt*, int, const char*, int n, void(*)(void*));
    static SQLITE_API int dbBindText16(sqlite3_stmt*, int, const void*, int, void(*)(void*));
    static SQLITE_API int dbBindValue(sqlite3_stmt*, int, const sqlite3_value*);
    static SQLITE_API int dbBindZeroblob(sqlite3_stmt*, int, int n);

    static SQLITE_API const void *dbColumnBlob(sqlite3_stmt*, int iCol);
    static SQLITE_API int dbColumnBytes(sqlite3_stmt*, int iCol);
    static SQLITE_API int dbColumnBytes16(sqlite3_stmt*, int iCol);
    static SQLITE_API double dbColumnDouble(sqlite3_stmt*, int iCol);
    static SQLITE_API int dbColumnInt(sqlite3_stmt*, int iCol);
    static SQLITE_API tS64 dbColumnInt64(sqlite3_stmt*, int iCol);
    static SQLITE_API const unsigned char *dbColumnText(sqlite3_stmt*, int iCol);
    static SQLITE_API const void *dbColumnText16(sqlite3_stmt*, int iCol);
    static SQLITE_API int dbColumnType(sqlite3_stmt*, int iCol);
    static SQLITE_API sqlite3_value *dbColumnValue(sqlite3_stmt*, int iCol);

    static SQLITE_API const void *dbValueBlob(sqlite3_value* v);
    static SQLITE_API int dbValueBytes(sqlite3_value* v);
    static SQLITE_API int dbValueBytes16(sqlite3_value* v);
    static SQLITE_API double dbValueDouble(sqlite3_value* v);
    static SQLITE_API int dbValueInt(sqlite3_value* v);
    static SQLITE_API tS64 dbValueInt64(sqlite3_value* v);
    static SQLITE_API const unsigned char *dbValueText(sqlite3_value* v);
    static SQLITE_API const void *dbValueText16(sqlite3_value* v);
    static SQLITE_API const void *dbValueText16le(sqlite3_value* v);
    static SQLITE_API const void *dbValueText16be(sqlite3_value* v);
    static SQLITE_API int dbValueType(sqlite3_value* v);
    static SQLITE_API int dbValueNumericType(sqlite3_value* v);

    static SQLITE_API void dbResultInt64(sqlite3_context* ctx, sqlite3_int64 v);
    static SQLITE_API void dbResultInt(sqlite3_context* ctx, int v);

    /* protected methods & data ...
     */
  protected:
    /* private methods & data ...
     */
  private:
    tVoid vInitializeTables();
    sqlite3*            _db;
    tBool               _bIsNew;
    class tclDSM*       _dsm;
    tU32                _u32PageSize;
    tU32                _u32FileLimit;
    tU32                _u32PageCount;
    tU32                _u32MaxPageCount;
    tU32                _u32UsedPageCount;
    tU32                _u32IsFullPageCount;
    tU32                _u32FreeListCount;
    tU32                _u32FileErrorCounter;
    tU32                _u32StmtReuseCounter;
    tU32                _u32StmtReuseStrCmpCounter;
    tU32                _u32StmtPrepareCounter;
    tU32                _u32IsFullCounter;
    tU32                _u32IsToBigCounter;
  };

  /******************************************************************************
   ******************************************************************************
   * class tclServiceManager ...
   ******************************************************************************
   *****************************************************************************/
  class tclServiceManager
  {
    /* public methods & data ...
     */
  public:
    tclServiceManager(class tclDSM* dsm);
    /* protected methods & data ...
     */
  protected:
    /* private methods & data ...
     */
#ifdef CONFIG_DSM_DATACHANNEL_FRAME_DAB_PERF_LIMIT
      tU64          _u64HandleDatachannelFramePrevTimestamp;
      tU64          _u32HandleDatachannelSkiped;
      tDouble       _dHandleDatachannelPeriod;
      tDouble       _dHandleDatachannelProcessingTime;
#endif

  private:
    class tclDSM* _dsm;
  };

#ifdef CONFIG_DSM__ENABLE_DAB
  /******************************************************************************
   ******************************************************************************
   * class tclDABServiceManager ...
   ******************************************************************************
   *****************************************************************************/
  class tclDABServiceManager : public tclServiceManager
  {
    /* public methods & data ...
     */
  public:
    tclDABServiceManager();
    tclDABServiceManager(class tclDSM* dsm);
    ~tclDABServiceManager();

    tclDABServiceManager(const tclDABServiceManager& obj);
    tclDABServiceManager& operator =(const tclDABServiceManager& obj);
    tVoid vInitializeTables();
    tVoid vSetCommandReceiver( tclDABServiceManagerCommandReceiver* cmdrx );
    tVoid vSendDSMMeCaCommand( tU8 u8TunerIdx, AaRSLib::MSG::tCommandMessage msg);
    //class tclDABServiceManagerCommandReceiver* oCmdReceiver() { return _cmd_receiver; };
    tVoid vSwitchOff();
    tVoid vSetNumTuner( tU8 u32NumTuner = 1 );
    tVoid vDeleteAppsWithFutureHostTimeStamp();
    tU64 u64GetMaxAppHostTimestamp();
#ifdef CONFIG_DSM__ENABLE_DAB_WIZARD
    class tclDSMDABWizard* oWizard() {
        return _wizard;
    }
#endif /* CONFIG_DSM__ENABLE_DAB_WIZARD */

#define DAB_SERVICE_MANAGER_UAPPTYPE_UNDEF  0xffff
#define DAB_SERVICE_MANAGER_UAPPTYPE_SLS    2
#define DAB_SERVICE_MANAGER_UAPPTYPE_BWS    3
#define DAB_SERVICE_MANAGER_UAPPTYPE_EPG    7
#define DAB_SERVICE_MANAGER_UAPPTYPE_JRNL   1098

    tVoid vSetDecodingCapability( tU8 u8TunerIdx, tU32 u32NumCurrentENSJobs, tU32 u32NumOtherENSJobs );

#define DAB_SERVICE_MANAGER_STRATEGY__CURRENT_SERVICE_COMPONENT (1<<0)
#define DAB_SERVICE_MANAGER_STRATEGY__CURRENT_SERVICE           (1<<1)
#define DAB_SERVICE_MANAGER_STRATEGY__CURRENT_ENSEMBLE          (1<<2)
#define DAB_SERVICE_MANAGER_STRATEGY__OTHER_ENSEMBLE            (1<<3)

    tVoid vSetDecodingStrategy( tU8 u8TunerIdx, std::list<tU16>  u16UserAppType_list, tU32 u32Strategy );
    tVoid vHandleMeCaResponse( tU8 u8TunerIdx, AaRSLib::MSG::tResponseMessage msg);
    tPVoid bGetMultimediaObject( tU32* u32ResultBytes, tU32 r_ensemble = 0, tU8 b_service_type = 0, tU32 r_service = 0, tU8 b_scids = 0xff, tU8 b_xpad_app_type = 0, tU8* multimedia_type = 0, tU32* w = 0, tU32* h = 0, tPChar* mimeValue =  0, tU64* pResA = NULL, std::string* pResURL = NULL );

    struct tEPGContentIdDataType
    {
		
		tEPGContentIdDataType(){
	    r_ensemble = 0;
        b_service_type = 0;
        r_service = 0;
        b_scids = 0;
        b_xpad_app_type = 0;
	    }
        tU32 r_ensemble;
        tU8 b_service_type;
        tU32 r_service;
        tU8 b_scids;
        tU8 b_xpad_app_type;
    };

    tBool bGetMultimediaObjects( std::vector<tU8>& resContentId, std::string& resMimeType, std::vector<tU8>& resFileBody, const char* uri, sqlite3_stmt** pContentIdStmt, tU8* multimedia_type = 0, tU32* multimedia_w = 0, tU32* multimedia_h = 0);
    tBool bGetMultimediaObject( std::string& resMimeType, std::vector<tU8>& resFileBody, const char* uri, std::vector<tU8> ContentId, tU8* pMultimediaType, tU32* pW, tU32* pH, tU64* pResA = NULL, std::string* pResURL = NULL );

#ifndef CONFIG_DSM__ENABLE_SPECIAL_HANDLING_FOR_HR_IMAGE_UNRESTRICTED_HEIGHT_BUT_NO_WIDTH
#define CONFIG_DSM__ENABLE_SPECIAL_HANDLING_FOR_HR_IMAGE_UNRESTRICTED_HEIGHT_BUT_NO_WIDTH 0 /* by default, don't handle HR use case */
#endif

#ifndef CONFIG_DSM__ENABLE_SPECIAL_HANDLING_FOR_IMAGE_TYPE_1
#define CONFIG_DSM__ENABLE_SPECIAL_HANDLING_FOR_IMAGE_TYPE_1 0 /* by default, ignore images with type=1 */
#endif

#ifndef CONFIG_DSM__GET_MULTIMEDIA_OBJECT_ASCPECT_RATIO_ERROR_LIMIT
#define CONFIG_DSM__GET_MULTIMEDIA_OBJECT_ASCPECT_RATIO_ERROR_LIMIT 0 /* by default, no aspect ration acceptance window */
#endif

    typedef enum
    {
        logo_undef_0            = 0,
#if (CONFIG_DSM__ENABLE_SPECIAL_HANDLING_FOR_IMAGE_TYPE_1 == 1)
        logo_undef_1            = 1, /* used in some streams */
#endif
        logo_unrestricted       = 0x02,
        logo_mono_square        = 0x03,
        logo_colour_square      = 0x04,
        logo_mono_rectangle     = 0x05,
        logo_colour_rectangle   = 0x06
    } tDABEPGLogoImageType;

    typedef struct
    {
        int             quality_error; /* image quality */
        tS64            a;        /* application id  */
        char            Uri[128]; /* application uri */
        char            Url[128]; /* object filename */
        tDABEPGLogoImageType    t;/* image type      */
        int             w;        /* image width     */
        int             h;        /* image height    */
    } tGetMultimediaObjectFoundMultimediaObjectType; /* exported for testing */

    static int bGetMultimediaObject_MatchBestImage( tS64 a,
                                                    tGetMultimediaObjectFoundMultimediaObjectType* found_mobj,
                                                    tDABEPGLogoImageType image_t, /* image type */
                                                    int image_w,  /* image width */
                                                    int image_h, /* image height */
                                                    tDABEPGLogoImageType wanted_t, /* wanted type */
                                                    int wanted_w, /* wanted width */
                                                    int wanted_h, /* wanted height */
                                                    char* pUrl, int nUrl, /* filename */
                                                    char* pUri, int nUri  /* uri */
                                                    ); /* exported for testing */

#if defined(CONFIG_DSM__ENABLE_DAB_EPG) && (defined(CONFIG_DSM__ENABLE_DAB_EPG_BASIC_SCHEDULE) || defined(CONFIG_DSM__ENABLE_DAB_EPG_ALL))
    typedef struct _tEPGScheduleInfoType_
    {
        tU64            u64CurrTime;
        tU64            u64StartTime;
        tU64            u16Duration;
        std::string     sMediumName;
        std::string     sLongName;
#if defined(CONFIG_DSM__ENABLE_DAB_EPG_SCHEDULE_shortDescription) || defined(CONFIG_DSM__ENABLE_DAB_EPG_ALL)
        std::string     sShortDescription;
#endif
    }tEPGScheduleInfoType;
    tBool bGetPrevEPGScheduleInfos( sqlite3_stmt** pStmt, tEPGContentIdDataType& pContentIdData, tEPGScheduleInfoType& info, tU32 u32SpanSeconds = (12*60*60) /* 1/2 day in the past */ );
    tBool bGetNextEPGScheduleInfos( sqlite3_stmt** pStmt, tEPGContentIdDataType& pContentIdData, tEPGScheduleInfoType& info, tU32 u32SpanSeconds = (24*60*60) /* 1 day in the future */ );
    tBool bHasEPGScheduleInfo( tEPGContentIdDataType& ContentIdData );
    tU32 u32GetEPGScheduleInfoContentIds( std::vector<tEPGContentIdDataType>& result );
#define DSM_PROVIDE_FUNCTION_tclDABServiceManager_u32GetEPGScheduleInfoContentIds
#endif

    tVoid vInformUsed( tS64 s64AppId );
    tVoid vInformUsed( std::string uri );
    tVoid vHandle();
    tVoid vCloseDB();
    tVoid vDestructApp( tS64 a );
    tVoid vDestructAllApps( );
    tVoid vDeleteAll();

    tVoid vTriggerAppCompleteEvent( );
    tBool bAppComplete( const char* uri );
    tBool bGetFile( const char* uri, const char* filename, std::vector<tU8>& resFileBody );
    tBool bGetFile( const char* uri, const char* filename, std::string& resMimeType, std::vector<tU8>& resFileBody );
    tBool bGetFile( tS64 s64AppId, const char* filename, std::string& resMimeType, std::vector<tU8>& resFileBody );

    tS64 s64GetAppId( const char* uri );
    tU16  u16GetUserAppType( const char* uri );
    tU16  u16GetUserAppType( tS64 s64AppId );
    tBool bIsAppComplete( tS64 s64AppId );
    tU32  u32GetNumMOTObjects( tS64 s64AppId );
    tU32  u32getNumApps();
    tBool bGetRecentTranspId( tS64 s64AppId, tU16& u16TransportId, tS64 s64AppTimeOffset )  ;
    tBool bGetFileTransPortId( tS64 s64AppId, const char* pFileName, tU16& u16TransportId );
    tBool bGetFileParam( tS64 s64AppId, tU16 u16TransportId, tS32 id, tU32& value );
    tBool bGetFileParam( tS64 s64AppId, tU16 u16TransportId, tS32 id, std::string& value );
    tBool bGetFileParam( tS64 s64AppId, tU16 u16TransportId, tS32 id, std::vector<tU8>& value );
    tBool bGetFileName( tS64 s64AppId, tU16 u16TransportId, std::string& name );
    tBool bGetFileBody( tS64 s64AppId, tU16 u16TransportId, std::vector<tU8>& body );

    tBool bGetBWSRootPageName( tS64 s64AppId, std::string& rName, tU8 u8BWSProfileId = 0 );
    tBool bGetBWSRootPageName( const char* uri, std::string& rName, tU8 u8BWSProfileId = 0 );

    /* protected methods & data ...
     */
  protected:
    /* private methods & data ...
     */
  private:
    tVoid vIncreateAgeCounter();
    tVoid vHandleMsgR_TIMESTAMP(tU8 u8TunerIdx,AaRSDABLib::API::DDM::tR_TIMESTAMP msg);
    tVoid vHandleMsgR_DATACHANNEL_FRAME(tU8 u8TunerIdx,AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME msg);
    tVoid vHandleMsgR_SELECT_URI(tU8 u8TunerIdx,AaRSDABLib::API::DDM::tR_SELECT_URI msg);

    class tclDSM* 								_dsm;
    class tclDABServiceManagerCommandReceiver*  _cmd_receiver;
    tU8		 	  								_number_tuner;
    std::list<tclDSMDABApp*>                    _active_apps;

    tU64                                        _u64Timeout1000mS;
    tU64                                        _u64Timeout60000mS;

    tU64                                        _u64TimeStamp;
    //tU64                                        _u64TimeOffset;

#ifdef CONFIG_DSM__ENABLE_DAB_WIZARD
    class tclDSMDABWizard*                      _wizard;
#endif /* CONFIG_DSM__ENABLE_DAB_WIZARD */

    /******************************************************************************
     ******************************************************************************
     * class tclSLSHandler ...
     ******************************************************************************
     *****************************************************************************/
    class tclDABSLSHandler
    {
      /* public methods & data ...
       */
    public:
        tclDABSLSHandler(tclDABServiceManager* parent);

        tVoid vHandleMeCaResponse( tU8 u8TunerIdx, AaRSLib::MSG::tResponseMessage msg);

        tVoid vHandle();
        tVoid vCheckforExpireandDelete(tU16 u16current_transpid,tS64 s64AppTimeOffset);
        tVoid vCheckforExpireandCompleteandDelete(tU16 u16current_transpid,tS64 s64AppTimeOffset);

      /* protected methods & data ...
       */
    protected:
      /* private methods & data ...
       */
        tS64 _s64app_id;
        tU16 _u16sid;
        tU16 _u16scids;
        tU32 _u32transpid;

    private:
        tclDABServiceManager* _parent;
    };

  protected:
    class tclDABSLSHandler                      _sls_handler;
  };
#endif /* CONFIG_DSM__ENABLE_DAB */

#ifdef CONFIG_DSM__ENABLE_DAB
  /******************************************************************************
   ******************************************************************************
   * class tclDABServiceManagerCommandReceiver ...
   ******************************************************************************
   *****************************************************************************/
  class tclDABServiceManagerCommandReceiver
  {
    /* public methods & data ...
     */
  public:
      virtual ~tclDABServiceManagerCommandReceiver( ) { };
      virtual tVoid vSendDSMMeCaCommand( tU8 u8TunerIdx, AaRSLib::MSG::tCommandMessage msg) { (void)u8TunerIdx; (void)msg; };
      /* protected methods & data ...
     */
  protected:
    /* private methods & data ...
     */
  private:
  };
#endif /* CONFIG_DSM__ENABLE_DAB */

  /******************************************************************************
   ******************************************************************************
   * type tDSMEvent ...
   ******************************************************************************
   *****************************************************************************/
  typedef enum {
      DSM_EV_APPLICATION_STARTED,
      DSM_EV_APPLICATION_STOPPED,
      DSM_EV_APPLICATION_CANCELED,
      DSM_EV_APPLICATION_OBJECT_COMPLETE,
      DSM_EV_APPLICATION_COMPLETE,
      DSM_EV_APPLICATION_ALLREADY_COMPLETE,
      DSM_EV_APPLICATION_EPG_OBJECT_DECODED,
      DSM_EV_APPLICATION_SLS_CLEAR,
      DSM_EV_APPLICATION_SLS_UPDATE,
      DSM_EV_APPLICATION_ERROR,
  } tDSMEvent;

  /******************************************************************************
   ******************************************************************************
   * class tclDSMTimeProvider ...
   ******************************************************************************
   *****************************************************************************/
  class tclDSMTimeProvider
  {
    /* public methods & data ...
     */
  public:
    virtual tU64 u64DSMGetUTCmS() { return 0; };
    virtual ~tclDSMTimeProvider() { };

    /* protected methods & data ...
     */
  protected:
    /* private methods & data ...
     */
  private:
  };

  /******************************************************************************
   ******************************************************************************
   * class tclDSMEventListener ...
   ******************************************************************************
   *****************************************************************************/
  class tclDSMEventListener
  {
  public:
      virtual ~tclDSMEventListener() { /* NOP*/ };
      /* public methods & data ...
     */
  public:
      virtual tVoid vDSMEvent( tDSMEvent ev, std::string uri ) { (void)ev; (void)uri; };
      /* protected methods & data ...
     */
  protected:
      /* private methods & data ...
     */
  private:
  };

  /******************************************************************************
   ******************************************************************************
   * class tclDSM ...
   ******************************************************************************
   *****************************************************************************/
  class tclDSM
  {
    /* public methods & data ...
     */
  public:
    tclDSM(tclDSMTimeProvider* dsmTimeProvider,tclDSMEventListener* tclEventListener=0);
    tclDSM(tclDSM& obj){ (void)obj; };
	virtual ~tclDSM();
    tVoid operator =(const tclDSM& obj){ (void)obj; };
    virtual tVoid vAddDSMEventListener(tclDSMEventListener* tclEventListener);
    tVoid vRemoveDSMEventListener(tclDSMEventListener* tclEventListener);
    tVoid vHandle();
    tU64 u64ElapsedTimeMS();
    tU64 u64GetMaxAppHostTimestamp();
    tVoid vDeleteAppsWithFutureHostTimeStamp();
    tclDB* oDB();
    tVoid vBroadCastEvent( tDSMEvent ev, std::string uri );
    tVoid vSendBroadCastEvents();
#ifdef CONFIG_DSM__ENABLE_DAB
    tclDABServiceManager* oDAB();
#endif /* CONFIG_DSM__ENABLE_DAB */
    tVoid log(const  char* src_str, const  char* fmt_str, ...);

    /* protected methods & data ...
     */
  protected:
    /* private methods & data ...
     */

  private:
    tclDB* _db;
    std::list<tclDSMEventListener*> _dsmeventlistener;
    tclDSMTimeProvider* _dsmtimeprovider;
#ifdef CONFIG_DSM__ENABLE_DAB
    tclDABServiceManager* _dab;
#endif /* CONFIG_DSM__ENABLE_DAB */

    typedef struct
    {
        tDSMEvent ev;
        std::string uri;
    } tDSMEventObj;
   std::list<tDSMEventObj> _event_queue;
  };

#ifdef CONFIG_DSM__ENABLE_DAB_WIZARD
  /******************************************************************************
   ******************************************************************************
   * class tclDSMDABWizard ...
   ******************************************************************************
   *****************************************************************************/
  class tclDSMDABWizard
  {
      /* public methods & data ...
       */
public:
      /* constructor
       */
      tclDSMDABWizard( tclDSM* dsm );
	
      tclDSMDABWizard( tclDSMDABWizard& obj){ (void)obj; }
      tclDSMDABWizard& operator = (const tclDSMDABWizard& obj){ return (tclDSMDABWizard&)obj; } //Coverity 74710
      /* destructor
       */
      virtual ~tclDSMDABWizard();

      /* create db tables if nessesary
       */
      static tVoid vInitializeTables(tclDSM* dsm);

      /* do some work
       */
      tVoid vHandle();

      /* handle a incomming MeCa response
       */

      virtual tU8 u8GetCurrentReceiveableDataServiceList(tBool force);
      tVoid vHandleMeCaResponse( tU8 u8TunerIdx, AaRSLib::MSG::tResponseMessage msg);

       /* set number of tuner systems
        * (e.g. FG1-ADR,FG2-ADR,BG-ADR)=3
        */
      tVoid vSwitchOff( );
      tVoid vSetNumTuner( tU8 u8NumTuner );
      tU8   u8GetNumTuner();

      /* set the wanted decoding strategy
       */
      tVoid vSetDecodingCapability( tU8 u8TunerIdx, tU32 u32NumCurrentENSJobs, tU32 u32NumOtherENSJobs );
      tVoid vSetDecodingStrategy( tU8 u8TunerIdx, std::list<tU16> u16UserAppType, tU32 u32Strategy );
      tBool bNotSelected(std::string uri);

      /* handle event from DSM e.g. DSM_EV_APPLICATION_COMPLETE
       */
      tVoid vDSMEvent( tDSMEvent ev, std::string uri );
  public:
      class tclReceiveableApp
      {
      public:
          tU16    u16eid;
          tU16    u16tmid;
          tU16    u16subchid;
          tU16    u16paddr;
          tU8     u8scids;
          tU32    u32sid;
          tU16    u16user_app_type;
          tU8     u8current_comp;
          tU8     u8current_serv;
          tU8     u8current_ens;
          std::string uri;
          tU8     label[20];
          tU8     label_bytes;
          tU8     e_label[20];
          tU8     e_label_bytes;
          tU8     s_label[20];
          tU8     s_label_bytes;
          tU8     c_label[20];
          tU8     c_label_bytes;
          tU8     u_label[20];
          tU8     u_label_bytes;
          tU8     u_app_data[20];
          tU8     u_app_data_bytes;
      };
      typedef std::list<tclReceiveableApp> tclReceiveableAppList;

      class tclStoredApp
      {
      public:
          std::string uri;
          int         age_received;
          tU64        host_timestamp;
      };
      typedef std::list<tclStoredApp> tclStoredAppList;
#if defined(ENABLE_GTEST)
      tclReceiveableAppList TESTgetReceivebleApps()
      {return _receiveable_apps;}
#endif

  protected:
      virtual tclStoredAppList getStoredApps();

      /* protected methods & data ...
       */
  protected:
      /* private methods & data ...
       */

      /* current last asked data service list
       */
      AaRSDABLib::API::DB::tR_QUERY* _tDataServiceList;

      tclReceiveableAppList  _receiveable_apps;    /* ADR list    */
      virtual tclReceiveableAppList getReceivebleApps();

  private:
      tclDSM*  _dsm;
      tU8      _u8NumTuner;
      tU8      _u8WizardDBQueryTag;
      tU64     _u64DataServiceListTimeout;

      /* sub-class implements ONE decoding statemachine
       */
      class tclDSMDABWizardJob : public tclDSMEventListener
      {
        public:
          /* constructor
           */
          tclDSMDABWizardJob(tU8 u8TunerIdx, std::list<tU16>  u16UserAppType, tU32 u32Strategy, tU8 u8DBQueryTag,tclDSMDABWizard* parent);

          /* destructor
           */
          ~tclDSMDABWizardJob();

          /* handle event from DSM e.g. DSM_EV_APPLICATION_COMPLETE
           */
          tVoid vDSMEvent( tDSMEvent ev, std::string uri );

          /* do the work
           */
          tVoid vHandle();

          /* handle incomming DAB MeCa response
           */
          tVoid vHandleMeCaResponse( tU8 u8TunerIdx, AaRSLib::MSG::tResponseMessage message);
          tVoid vHandleNewDataServiceList(tU8 u8TunerIdx);
          tVoid vSwitchOff();
          std::string sGetURI();

      private:
          tU8    u8TunerIdx_md;
          tU8    u8JobIdx_md;
          std::list<tU16>    u16UserAppType_list_md;
          tU32   u32Strategy_md;
          tU8    u8WizardJobDBQueryTag_md;
          tU64   u64StateTimeout_md;

          tclDSMDABWizard*                          _parent;

          /* decide the URI to be select and select it
           */
          tBool bSelectNextApp( tU8 u8TunerIdx );
          tBool bFilterStrategy( tclReceiveableApp& app );

          typedef enum {
              DAB_WIZARD_STATE__START = 1,
              DAB_WIZARD_STATE__WAITING_FOR_SERVICE_LIST = 2,
              DAB_WIZARD_STATE__WAITING_FOR_APP_SELECTION = 3,
              DAB_WIZARD_STATE__WAITING_FOR_APP_ACTIVATION = 4,
              DAB_WIZARD_STATE__WAITING_FOR_APP_COMPLETE = 5,
              DAB_WIZARD_STATE__DONE = 6,
              DAB_WIZARD_STATE__WAITING_FOR_APP_DESELECTION = 7,
              DAB_WIZARD_STATE__INIT = 0
          } tCurrentStateType;
          tCurrentStateType                                  tCurrentState;
          AaRSDABLib::API::DDM::tR_SELECT_URI::t__w_cmd      w_cmd;
          AaRSDABLib::API::DDM::tR_SELECT_URI::t__w_response w_response;

          /* my current URI
           */
          std::string   r_uri;
          tU16          u16user_app_type_current;
      }; // sub class tclDSMDABWizardJob

      std::list<tclDSMDABWizardJob*>       _wizard_job_list;

      /* black list management
       */
      class tclBlackListApp
      {
      public:
          bool operator == (const tclBlackListApp &rhs)
          {
              return (rhs.uri == uri);
          }
          std::string uri;
          tU64        host_timestamp;
      };
      std::list<tclBlackListApp> _blacklist_apps;
      tBool bIsBlackListApp( std::string uri );
      tVoid vAddToBlacklist( std::string uri );
      tVoid vDeleteOlderAppsFromBlackList( tU64 host_timestamp );
  };
#endif /* CONFIG_DSM__ENABLE_DAB_WIZARD */

  /******************************************************************************
   ******************************************************************************
   * class tclDSMApp ...
   ******************************************************************************
   *****************************************************************************/
  class tclDSMApp
  {
      /* public methods & data ...
       */
  public:
      tclDSMApp( tclDSM* dsm );
      virtual ~tclDSMApp( ) { };

      virtual void vHandle() { };

      static tVoid vInitializeTables(tclDSM* dsm);

      tBool bGetErrorFlag() { return _bErrorFlag; };

      static tU32 u32BitStreamValue( const tU8* bitstream, tU32 bitstreambytes, tU32 pos, tU32 bits );
      static tU32 u32AppProfileSupported(tU16 u16user_app_type, tU8* u_app_data, tU8 u_app_data_bytes)
      {
          switch(u16user_app_type)
          {
             case DAB_SERVICE_MANAGER_UAPPTYPE_SLS:
              /* by default enable all*/
              return 1;
             case DAB_SERVICE_MANAGER_UAPPTYPE_BWS:
              /* by default enable all*/
              return 1;
             case DAB_SERVICE_MANAGER_UAPPTYPE_EPG:
#ifdef  CONFIG_DSM__ENABLE_DAB_EPG_ALL
              return 1;
#elif (defined (CONFIG_DSM__ENABLE_DAB_EPG_BASIC_SCHEDULE) || defined (CONFIG_DSM__ENABLE_DAB_EPG_BASIC_STATIONLOGOS))
              /*check if the user app data contain the basic profile*/
              {
                int idx=0;

                /*in case of no user app data assume basic profile*/
                if (u_app_data_bytes == 0)
                {
                    return 1;
                }

                while (idx<u_app_data_bytes)
                {
                    /* The user application data field for the SPI user application is a sequence
                     * of 1-byte values, each being a ProfileID,
                     * indicating which profile(s) of the SPI service are carried there.
                     * If there is more than one ProfileID, then the list shall
                     * be sorted in ascending order with the lowest ProfileID first.
                     * The ProfileIDs are defined in table 13. Any remaining
                     * values for the ProfileID are reserved for future use and shall
                     * not be processed by devices.*/

                    /* 0x01 Basic profile
                     * 0x02 Advanced profile*/

                    if(u_app_data[idx]==0x01)
                        return 1;

                    /* If user application specific parameters other than the list of profiles
                     * are carried within the user application data field,
                     * then the data field shall commence with the list of ProfileIDs,
                     * followed by a single 0x00 after the end of the list of
                     * profiles and before the other user application specific parameters. */
                    if(u_app_data[idx]==0x00)
                        return 0;
                    idx++;
                }
                return 0;
              }
#endif
             default:
              return 0;
          }
      };

      virtual tU32 u32AppSizeLimit() { return 0; }; /* by default unlimited */
      virtual tU32 u32ObjectSizeLimit(tU8 u8DGType,tU16 u16TranspID) {
          (void)u8DGType;
          (void)u16TranspID;
          return 0; }; /* by default unlimited */

      int iBeginSavePoint();
      int iCommitSavePoint();
      int iRoolbackSavePoint();

      /* protected methods & data ...
       */
  protected:

      std::string string_sprintf(const std::string fmt, ...);

      tclDSM*       _dsm;
      tBool         _doRollbackTransaction;
      tBool         _bErrorFlag;
      tBool         _bmaxObjectSizeViolated;

      /* private methods & data ...
       */
  private:
  };

  /******************************************************************************
   ******************************************************************************
   * class tclDSMDABApp ...
   ******************************************************************************
   *****************************************************************************/
  class tclDSMDABApp : public tclDSMApp
  {
      /* public methods & data ...
       */
  public:
      tclDSMDABApp( tclDSM* dsm, tU8 u8TunerIndex, std::string uri );
      virtual ~tclDSMDABApp( );

      virtual void vHandle() { tclDSMApp::vHandle(); };

      static tVoid vInitializeTables(tclDSM* dsm);

      tU8 u8GetTunerIdx() { return _u8TunerIndex; };
      tU16 u16tGetSCIDI() { return _u16SCIDI; };
      tU16 u16tGetXPADAppType() { return _u16XPADAppType; };
      std::string tGetURI() { return _uri; };
      tU64 u64CTime() { return _ctime; };
      tU64 u64CTimeUTC() { return _ctime_utc; };
      tS64 s64GetA();
      tS64 s64GetApplicationTime();
      tVoid vDeleteAppData( tU64 a );
      static tVoid vDeleteAppData( tclDSM* dsm, tU64 a );
      tBool bIsAppComplete( tS64 s64AppId );

      virtual tVoid vHandleMsgR_DATACHANNEL_FRAME(AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME msg);
      virtual tVoid vHandleMsgR_TIMESTAMP(AaRSDABLib::API::DDM::tR_TIMESTAMP msg);
      virtual tVoid vHandleMsgR_SELECT_URI(AaRSDABLib::API::DDM::tR_SELECT_URI msg);

      virtual tBool vHandleGarbageCollection();
      virtual tVoid vHandle60sCTimeTick();

      /* protected methods & data ...
       */
  protected:

  private:
      tS64          _a;
      tU64          _ctime;
      tU64          _ctime_60s;
      tU64          _ctime_utc;
      tU64          _ctime_old_host_time;
      tU8           _u8TunerIndex;
      tU16          _u16SCIDI;
      tU16          _u16XPADAppType;
      std::string   _uri;
  };

#ifdef CONFIG_DSM__ENABLE_DAB
  /******************************************************************************
   ******************************************************************************
   * class tclDSMDABMOTApp ...
   ******************************************************************************
   *****************************************************************************/
  class tclDSMDABMOTApp : public tclDSMDABApp
  {
      /* public methods & data ...
       */
  public:
      tclDSMDABMOTApp( tclDSM* dsm, tU8 u8TunerIndex, std::string uri );
      virtual ~tclDSMDABMOTApp( );

      virtual void vHandle() { tclDSMDABApp::vHandle(); };

      static tVoid vInitializeTables(tclDSM* dsm);

      virtual tBool vHandleGarbageCollection();
      virtual tVoid vHandle60sCTimeTick();

      tU32 u32GetAppSize();

      /* protected methods & data ...
       */
  protected:
      tVoid vInitMOTDirectory();
      virtual tBool bDirectoryParamFilter( tU8 u8ContentType, tU16 u16ContentSubType, tU8 u8ParamId) {
          (void)u8ContentType;
          (void)u16ContentSubType;
          (void)u8ParamId;
#ifdef CONFIG_DSM_ENABLE_MOT_DIRECTORY_PARAM_FILTER
          return FALSE;
#else
          return TRUE;
#endif /* CONFIG_DSM_ENABLE_MOT_DIRECTORY_PARAM_FILTER */
      }

      tVoid vHandleMsgR_DATACHANNEL_FRAME(AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME msg);
      tVoid vHandleMsgR_TIMESTAMP(AaRSDABLib::API::DDM::tR_TIMESTAMP msg);
      tVoid vHandleMsgR_SELECT_URI(AaRSDABLib::API::DDM::tR_SELECT_URI msg);

      void vClearDGSegemt( tS64 a, tU8 u8_dgtype, tU16 u16_transport_id );
      void vForceSetObjectComplete( tS64 a, tU8 u8_dgtype, tU16 u16_transport_id );

      void vCheckAndSendAppComplete(tBool motdirdecoded);

      tBool bGetObjParam( tS64 a, tU16 u16_transport_id, tU32 u32ParamId, tU32& u32Value );
      tBool bGetObjParams( tS64 a, tU16 u16_transport_id, tU32& u32BodySize, tU8& u8ContentType, tU16& u16ContentSubType );

      tBool bIsMOTObjComplete( tS64 a, tU8 u8_dgtype, tU16 u16_transport_id );
      tBool bProcessCompleteMOTObj( tS64 a, tU8 u8DGType, tU16 u16_transport_id );
      tBool bHandleMOTObjComplete( tBool bIsCompleteBefore, tS64 a, tU8 u32DGType, tU16 u16_transport_id, tU8 u8Last );
      tBool bTryDecodeNxtMOTDir( tS64 a );
      tU32  u32GetSavedNxtMOTDirId( tS64 a );
      tVoid vUpdateNextMOTDir();
      tVoid vDeleteAllNotCurrentMOTDirectoryObjects(tBool bIgnoreOwnCompleteFlag);

      int   s32TryDecodeMOTHeaderElement( tS64 a, tS64 d, tU16 u32EntryTransportId, tU32 object_body_bytes, const tU8* object_body, tU32 bitidx, tU16* pU16ContentType, tU16* pU16ContentSubType );
      tBool bTryDecodeMOTHeader( tS64 a, tU16 u16_transport_id, tU8 u8DGType, tU32 u32UserAppType, tU32 object_body_bytes, const tU8* object_body);
      tBool bTryDecodeMOTDirectory( tS64 a, tU16 u16_transport_id, tU8 u8DGType, tU32 u32UserAppType, tU32 object_body_bytes, tU8* object_body);
      virtual tBool bTryDecodeMOTDirectoryAppSpecificPreHandling( tS64 a, tU16 u16_transport_id, tU8 u8DGType ) {
          (void)a;
          (void)u16_transport_id;
          (void)u8DGType;
          return true;
      };

      tBool bTryDecodeMOTDirectoryCore( tS64 a, tU16 u16_transport_id, tU8 u8DGType, tU32 u32UserAppType, tU32 object_body_bytes, tU8* object_body);
      tBool bTryDecodeMOTDirectoryParameter( tS64 a, tS64 d, tU16 u16_transport_id, tU32 u32PLI, tU32 u32ParamId, tU32 u32DatafieldLength, const tU8* pDatafield, tS64 u64DataValue );

#if defined(ENABLE_GTEST)
  public:
      static tU16 _u16GTestTransportIdOffset;
#endif

  private:
      std::string tMissingObjects( );
      tVoid vDeleteMOTDirectory( );
      tVoid vClearCompleteFlag( );

      /* private methods & data ...
       */
  private:
      tU8           _u8_no_transp_id_cnt;
      tU32          _num_inserted_datagroups;
      tU32          _num_ignored_datagroups;
      tBool         _bMOTDirectoryMode;
      tU64          _u64MOTHeaderModeAppCompleteCTime;
      tBool         _bMOTHeaderModeAppCompleteSend;
      std::list<tU16> _tMissingMOTDirObjects;
      tBool         _u16NextMOTDirUpdateValue;
      tBool         _u16NextMOTDirRunGC;
      tU32          _u32NextMOTDirTransportId;
      tU32          _u16NextMOTDirObjectsLeft;
#ifdef CONFIG_DSM_ENABLE_USE_MOT_OBJECT_COMPLETE_CACHE
      std::map<tU32,tBool> _tMOTObjCompleteStateCache;
#endif /* CONFIG_DSM_ENABLE_USE_MOT_OBJECT_COMPLETE_CACHE */
      tU32          _u32CntTryDecodeNxtMOTDir;
  };

#ifdef CONFIG_DSM__ENABLE_DAB_EPG
  /******************************************************************************
   ******************************************************************************
   * class tclDSMDABEPGApp ...
   ******************************************************************************
   *****************************************************************************/  
  class tclDSMDABEPGApp : public tclDSMDABMOTApp
  {
      /* public methods & data ...
       */
public:
      tclDSMDABEPGApp( tclDSM* dsm, tU8 u8TunerIndex, AaRSDABLib::API::DDM::tR_SELECT_URI msg);
      virtual ~tclDSMDABEPGApp();

      virtual void vHandle() { tclDSMDABMOTApp::vHandle(); };

      static tVoid vInitializeTables(tclDSM* dsm);
#ifdef CONFIG_DSM__EPG_OBJECT_SIZE_LIMIT
      static tVoid vSetObjectSizeLimit(tU32 limit) {_u32_object_size_limit = limit;};
      virtual tU32 u32ObjectSizeLimit(tU8 u8DGType,tU16 u16TranspID);
#endif

#ifdef CONFIG_DSM__EPG_APP_SIZE_LIMIT
      static tVoid vSetAppSizeLimit(tU32 limit) {_u32_app_size_limit = limit;};
      virtual tU32 u32AppSizeLimit() { return _u32_app_size_limit; };
#endif

      virtual tBool bTryDecodeMOTDirectoryAppSpecificPreHandling( tS64 a, tU16 u16_transport_id, tU8 u8DGType );

      typedef struct
      {
        tS64  a;
        tU16  u16_transport_id;
        tU32  object_body_bytes;
        tU8*  object_body;
        char* token_ptr[0x20];
        tU32  token_length[0x20];
        tU32  u32NumDecodedElements;
        sqlite3_stmt* pStmt;
#ifdef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
        tU8*  pu8DefaultContentID;
        tU32  u32DefaultContentIDBytes;
        tU8*  pu8ContentID;
        tU32  u32ContentIDBytes;
#endif /* CONFIG_DSM__ENABLE_dab_epg_schedule_scope */
      } tDecodeEPGObjectStatus;

#ifdef CONFIG_DSM__ENABLE_dab_epg_schedule_scope
      tBool bTryDecodeEPGObjectInsertScope( tDecodeEPGObjectStatus* status, tS64 schedule_o, unsigned char* contentID, tU32 contentIDBytes );
#endif

      tBool bTryDecodeEPGObjectInsertRoot( tDecodeEPGObjectStatus* status, tS64 dab_epg_o, tU8 u8Tag );
      tBool bTryDecodeEPGObjectIfNessesary( tU32* p32EPGByteLimit, tS64 a, tU16 u16_transport_id );
#ifdef CONFIG_DSM_ENABLE_bTryDecodeEPGObjectIfNessesary_OPTIMIZIATION
      tBool bTryDecodeEPGObjectIfNessesary( tU32* p32EPGByteLimit, tS64 a, tU16 u16_transport_id, const void* object_body, tU32 object_body_bytes );
#endif

      int u32EPGObjectCompressionType( tS64 a, tU16 u16_transport_id );
      tBool bTryDecodeEPGObject( tDecodeEPGObjectStatus* status );

      tBool vHandleGarbageCollection();
      virtual tVoid vHandle60sCTimeTick();

      tPVoid bGetMultimediaObject( tU32* u32ResultBytes, tU32 r_ensemble = 0, tU8 b_service_type = 0, tU32 r_service = 0, tU8 b_scids = 0xff, tU8 b_xpad_app_type = 0, tU8* multimedia_type = 0, tU32* w = 0, tU32* h = 0, tPChar* mimeValue =  0 );

      static std::string pInsertCDATAToken( tDecodeEPGObjectStatus* status, const char* cdata, tU32 cdata_bytes, tU8 ptag=0 );
      static tBool bUTF8Validate( const std::string& string );

      /* protected methods & data ...
       */
protected:
#ifdef CONFIG_DSM_ENABLE_MOT_DIRECTORY_PARAM_FILTER
      virtual tBool bDirectoryParamFilter( tU8 u8ContentType, tU16 u16ContentSubType, tU8 u8ParamId);
#endif /* CONFIG_DSM_ENABLE_MOT_DIRECTORY_PARAM_FILTER */
      tVoid vHandleMsgR_DATACHANNEL_FRAME(AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME msg);
      tVoid vHandleMsgR_TIMESTAMP(AaRSDABLib::API::DDM::tR_TIMESTAMP msg);
      tVoid vHandleMsgR_SELECT_URI(AaRSDABLib::API::DDM::tR_SELECT_URI msg);

      tBool bTryDecodeEPGObject( tDecodeEPGObjectStatus* status, tU32 level, tU8 ptag, tS64 p_o, tU32 p_addr, tU32 addr, tU32 taglen );

      /* private methods & data ...
       */
  private:
#ifdef CONFIG_DSM__EPG_OBJECT_SIZE_LIMIT
      static tU32 _u32_object_size_limit;  // = CONFIG_DSM__EPG_OBJECT_SIZE_LIMIT;
#endif
#ifdef CONFIG_DSM__EPG_APP_SIZE_LIMIT
      static tU32 _u32_app_size_limit;  // = CONFIG_DSM__EPG_APP_SIZE_LIMIT;
#endif
  };
#endif /* CONFIG_DSM__ENABLE_DAB_EPG */

#ifdef CONFIG_DSM__ENABLE_DAB_BWS
  /******************************************************************************
   ******************************************************************************
   * class tclDSMDABBWSApp ...
   ******************************************************************************
   *****************************************************************************/
  class tclDSMDABBWSApp : public tclDSMDABMOTApp
  {
      /* public methods & data ...
       */
public:
      tclDSMDABBWSApp( tclDSM* dsm, tU8 u8TunerIndex, AaRSDABLib::API::DDM::tR_SELECT_URI msg);
      virtual ~tclDSMDABBWSApp();

      virtual void vHandle() { tclDSMDABMOTApp::vHandle(); };

      static tVoid vInitializeTables(tclDSM* dsm);

      /* protected methods & data ...
       */
  protected:
#ifdef CONFIG_DSM_ENABLE_MOT_DIRECTORY_PARAM_FILTER
      virtual tBool bDirectoryParamFilter( tU8 u8ContentType, tU16 u16ContentSubType, tU8 u8ParamId);
#endif /* CONFIG_DSM_ENABLE_MOT_DIRECTORY_PARAM_FILTER */
      virtual tVoid vHandleMsgR_DATACHANNEL_FRAME(AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME msg);
      virtual tVoid vHandleMsgR_TIMESTAMP(AaRSDABLib::API::DDM::tR_TIMESTAMP msg);
      virtual tVoid vHandleMsgR_SELECT_URI(AaRSDABLib::API::DDM::tR_SELECT_URI msg);

      /* private methods & data ...
       */
  private:
  };
#endif /* CONFIG_DSM__ENABLE_DAB_BWS */

#ifdef CONFIG_DSM__ENABLE_DAB_SLS
  /******************************************************************************
   ******************************************************************************
   * class tclDSMDABSLSApp ...
   ******************************************************************************
   *****************************************************************************/
  class tclDSMDABSLSApp : public tclDSMDABMOTApp
  {
      /* public methods & data ...
       */
public:
      tclDSMDABSLSApp( tclDSM* dsm, tU8 u8TunerIndex, AaRSDABLib::API::DDM::tR_SELECT_URI msg);
      virtual ~tclDSMDABSLSApp();

      virtual void vHandle();

      bool bTryHandleSLSObject(tU16 u16_transport_id);

      static tVoid vInitializeTables(tclDSM* dsm);
#ifdef CONFIG_DSM__SLS_OBJECT_SIZE_LIMIT
      static tVoid vSetObjectSizeLimit(tU32 limit) {_u32_object_size_limit = limit;};
      virtual tU32 u32ObjectSizeLimit(tU8 u8DGType,tU16 u16TranspID) {
          (void)u8DGType;
          (void)u16TranspID;
          return _u32_object_size_limit;
      };
#endif

#ifdef CONFIG_DSM__SLS_APP_SIZE_LIMIT
      static tVoid vSetAppSizeLimit(tU32 limit) {_u32_app_size_limit = limit;};
      virtual tU32 u32AppSizeLimit() { return _u32_app_size_limit; };
#endif

      /* protected methods & data ...
       */
  protected:
#ifdef CONFIG_DSM__KEEP_ONLY_SLS_FROM_CURRENT_SERVICE
        tVoid DeleteAllSLS_except_current();
#endif        

#ifdef CONFIG_DSM_ENABLE_MOT_DIRECTORY_PARAM_FILTER
      virtual tBool bDirectoryParamFilter( tU8 u8ContentType, tU16 u16ContentSubType, tU8 u8ParamId);
#endif /* CONFIG_DSM_ENABLE_MOT_DIRECTORY_PARAM_FILTER */
      virtual tVoid vHandleMsgR_DATACHANNEL_FRAME(AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME msg);
      virtual tVoid vHandleMsgR_TIMESTAMP(AaRSDABLib::API::DDM::tR_TIMESTAMP msg);
      virtual tVoid vHandleMsgR_SELECT_URI(AaRSDABLib::API::DDM::tR_SELECT_URI msg);

      /* private methods & data ...
       */
  private:
#ifdef CONFIG_DSM__SLS_OBJECT_SIZE_LIMIT
      static tU32 _u32_object_size_limit;  // = CONFIG_DSM__SLS_OBJECT_SIZE_LIMIT;
#endif
#ifdef CONFIG_DSM__SLS_APP_SIZE_LIMIT
      static tU32 _u32_app_size_limit;  // = CONFIG_DSM__SLS_APP_SIZE_LIMIT;
#endif
  };
#endif /* CONFIG_DSM__ENABLE_DAB_SLS */

#ifdef CONFIG_DSM__ENABLE_DAB_JOURNALINE
  /******************************************************************************
   ******************************************************************************
   * class tclDSMDABJRLNApp ...
   ******************************************************************************
   *****************************************************************************/
  class tclDSMDABJRLNApp : public tclDSMDABApp
  {
      /* public methods & data ...
       */
public:
      tclDSMDABJRLNApp( tclDSM* dsm, tU8 u8TunerIndex, AaRSDABLib::API::DDM::tR_SELECT_URI msg);
      virtual ~tclDSMDABJRLNApp();

      virtual void vHandle() { tclDSMDABApp::vHandle(); };

      static tVoid vInitializeTables(tclDSM* dsm);

      virtual tVoid vHandleMsgR_DATACHANNEL_FRAME(AaRSDABLib::API::DC::tR_DATACHANNEL_FRAME msg);
      virtual tVoid vHandleMsgR_TIMESTAMP(AaRSDABLib::API::DDM::tR_TIMESTAMP msg);
      virtual tVoid vHandleMsgR_SELECT_URI(AaRSDABLib::API::DDM::tR_SELECT_URI msg);

      /* protected methods & data ...
       */
  protected:
      /* private methods & data ...
       */
  private:
  };
#endif /* CONFIG_DSM__ENABLE_DAB_JOURNALINE */

#endif /* CONFIG_DSM__ENABLE_DAB */

};

#define MJD_1_1_1970 40587

#endif /* _TCLDSM_HEADER_ */
 /******************************************************************************
  * EOF
  *****************************************************************************/

