/******************************************************************************
 *
 * FILE: tclDSMApp.cpp
 * PROJECT: AaRS
 * SW-COMPONENT: Data Service Manager
 *
 * DESCRIPTION: This file contains the DSM class interface specification
 *
 * AUTHOR: CM-AI/ECB2-Scholz
 *
 * COPYRIGHT: (c) 2010 Robert Bosch Multimedia GmbH
 * HISTORY:
 * Date | Rev. | Author | Modification
 * ----------------------------------------------------------------------------
 * 23.04.2014 | 1.0 | CM-AI/ECB2-Scholz | Initial revision
 *
 *****************************************************************************/
#include <stdarg.h>  // for va_start, etc
#include <stdexcept>
#include <memory>    // for std::unique_ptr
#include "tclDSM.h"

// ETG defines
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

// defines and include of generated code
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DATASRVCS_DSM_DB
#include "trcGenProj/Header/tclDB.cpp.trc.h"
#endif

using namespace dsm;

#if defined(ENABLE_GTEST)
tBool tclDB::_bReportSQLITE3_ERROR = true;
#endif


/* SQLite version check
 */
#if (SQLITE_VERSION_NUMBER<3008006)
#error SQLite 3.8.6 or newer required because needed support for HEX literale
#endif

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
#ifdef ENABLE_SQLITE3_LOG
static void dsm_sqlite3_log(void* argptr,int rc,const char* str)
{
    int rc_byte = (rc & 0xff);
    if( rc_byte == SQLITE_FULL )
    {
        ETG_TRACE_USR1(("dsm_sqlite3_log(rc=%d,%s) SQLITE_FULL",rc,str));
    }
    else if( rc_byte == SQLITE_BUSY )
    {
        ETG_TRACE_USR1(("dsm_sqlite3_log(rc=%d,%s) SQLITE_BUSY",rc,str));
    }
    else if( rc_byte == SQLITE_CONSTRAINT )
    {
        ETG_TRACE_USR1(("dsm_sqlite3_log(rc=%d,%s) SQLITE_CONSTRAINT",rc,str));
    }
    else if( rc_byte == SQLITE_SCHEMA )
    {
        ETG_TRACE_USR1(("dsm_sqlite3_log(rc=%d,%s) SQLITE_SCHEMA",rc,str));
    }
    else if( rc_byte == SQLITE_CANTOPEN )
    {
        ETG_TRACE_USR1(("dsm_sqlite3_log(rc=%d,%s) SQLITE_CANTOPEN",rc,str));
    }
    else if( rc_byte == SQLITE_NOTADB )
    {
        ETG_TRACE_USR1(("dsm_sqlite3_log(rc=%d,%s) SQLITE_NOTADB",rc,str));
    }
    else if( rc_byte == SQLITE_NOTICE )
    {
        ETG_TRACE_USR1(("dsm_sqlite3_log(rc=%d,%s) SQLITE_NOTICE",rc,str));
    }
    else if( rc_byte == SQLITE_WARNING )
    {
        ETG_TRACE_USR1(("dsm_sqlite3_log(rc=%d,%s) SQLITE_WARNING",rc,str));
    }
    else if( rc_byte == SQLITE_CORRUPT )
    {
        ETG_TRACE_USR1(("dsm_sqlite3_log(rc=%d,%s) SQLITE_CORRUPT",rc,str));
    }
    else if( rc_byte == SQLITE_NOTICE )
    {
        ETG_TRACE_USR1(("dsm_sqlite3_log(rc=%d,%s) SQLITE_NOTICE",rc,str));
    }
    else
    {
        ETG_TRACE_FATAL(("dsm_sqlite3_log(rc=%d,%s)",rc,str));
        DSM_DEBUG_SQLITE3_ERROR();
#if defined(ENABLE_GTEST)
        if( tclDB::_bReportSQLITE3_ERROR == true )
        {
            printf("dsm_sqlite3_log(rc=%d/0x%x,%s)",rc_byte,rc,str);
            FAIL();
        }
#endif
    }
    (void)argptr;
    (void)str;
}
#endif

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
#ifdef ENABLE_SQLITE3_TRACE
static void dsm_sqlite3_trace(void* argptr,const char* str)
{
    if( str != NULL )
    {
#ifdef TR_ENABLE
#if TR_ENABLE != 0
        ETG_TRACE_USR1(("dsm_sqlite3_trace() %s",str));
#else
        printf("dsm_sqlite3_trace() %s\n",str);
#endif
#else
        printf("dsm_sqlite3_trace() %s\n",str);
#endif
    }
    (void)argptr;
}
#endif

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
#ifdef ENABLE_SQLITE3_PROFILE
static void dsm_sqlite3_profile(void* argptr,const char* str,sqlite3_uint64 t)
{
    if((str != NULL)&&(t>50000000UL))
    {
        ETG_TRACE_USR1(("dsm_sqlite3_profile() %ld ms %s",(long)t/1000000UL,str));
    }
    (void)t;
    (void)argptr;
}
#endif
int tclDB::_u8SQLiteInitialized = 0;

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tclDB::tclDB(class tclDSM* dsm)
{
	int rc;
    if( (sizeof(sqlite_int64) != 8)||(sizeof(sqlite_uint64) != 8) )
    {
        throw std::length_error( "sizeof(sqlite_int64/sqlite_uint64) must be 8 bytes, check type definition" );
    }

    _db = NULL;
    _bIsNew = TRUE;
    _dsm = dsm;
    _u32PageCount = 0;
    _u32FreeListCount = 0;
    _u32MaxPageCount = 0;
    _u32IsFullPageCount = 0;
    _u32UsedPageCount = 0;
    _u32PageSize = CONFIG_USE_SQLITE_PAGE_SIZE;
    _u32FileLimit = DEFAULT_DSM_FILE_LIMIT;
    _u32StmtReuseCounter = 0;
    _u32StmtReuseStrCmpCounter = 0;
    _u32FileErrorCounter = 0;
    _u32StmtPrepareCounter = 0;
    _u32IsFullCounter = 0;
    _u32IsToBigCounter = 0;
    pStmtPrepareCache = NULL;
	_bInMemory = FALSE;

    ETG_TRACE_USR1(("%p.tclDB(%p)",(void*)this,dsm));
    if( _u8SQLiteInitialized == 0 )
    {
        _u8SQLiteInitialized = 1;
#ifdef ENABLE_SQLITE3_LOG
        sqlite3_config(SQLITE_CONFIG_LOG,dsm_sqlite3_log,NULL);
#endif

        sqlite3_config(SQLITE_CONFIG_LOOKASIDE, CONFIG_USE_LOOKASIDE_SIZE, CONFIG_USE_LOOKASIDE_NUMBER);

#if defined(CONFIG_USE_SCRATCH_SIZE) && defined(CONFIG_USE_SCRATCH_NUMBER)
        {
            static tU8 scratch[4096*16];
            sqlite3_config(SQLITE_CONFIG_SCRATCH, scratch, CONFIG_USE_SCRATCH_SIZE, CONFIG_USE_SCRATCH_NUMBER);
        }
#endif

#ifdef CONFIG_USE_NUM_PAGES_FOR_PAGECACHE
        {
            static tU8 pagemem[CONFIG_USE_SQLITE_PAGE_SIZE*CONFIG_USE_NUM_PAGES_FOR_PAGECACHE];
            sqlite3_config(SQLITE_CONFIG_PAGECACHE, pagemem, CONFIG_USE_SQLITE_PAGE_SIZE,CONFIG_USE_NUM_PAGES_FOR_PAGECACHE);
        }
#endif
        rc = sqlite3_initialize();
		(void)rc;
        sqlite3_soft_heap_limit64(4*1024*1024);
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDB::vDeleteAll( )
{
/*
 * not implemented yet
 */
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDB::vFinalizeAllStatements( )
{
    /* check for all still not finalized statements.
     */
    pStmtPrepareCache = NULL;
    {
        sqlite3_stmt* pStmt = NULL;
        do
        {
            pStmt = sqlite3_next_stmt(_db,NULL);
            if( pStmt != NULL )
            {
                dbFinalize(pStmt);
            }
        } while(pStmt != NULL );
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDB::vClose( )
{
    ETG_TRACE_USR1(("%p.vClose()",(void*)this));
    if( _db != NULL )
    {
        SQLITE_API int rc;

#ifdef CONFIG_DSM__ENABLE_DAB
        if( _dsm->oDAB() != NULL )
        {
            _dsm->oDAB()->vCloseDB();
        }
#endif /* CONFIG_DSM__ENABLE_DAB */

        vFinalizeAllStatements();

        vWALCheckPoint();

        vFinalizeAllStatements();

        if( _bInMemory == true )
        {
            sqlite3 *pFile;             /* Database connection opened on zFilename */
            /* Open the database file identified by zFilename. */
            rc = sqlite3_open(_dbfilename, &pFile);

            if( rc==SQLITE_OK )
            {
                sqlite3_backup *pBackup;    /* Backup handle used to copy data */
                /* Open the sqlite3_backup object used to accomplish the transfer */
                pBackup = sqlite3_backup_init(pFile, "main", _db , "main");
                if( pBackup )
                {
                    /* Each iteration of this loop copies 5 database pages from database
              ** pDb to the backup database. If the return value of backup_step()
              ** indicates that there are still further pages to copy, sleep for
              ** 250 ms before repeating. */
                    do {
                        rc = sqlite3_backup_step(pBackup, 10);
                        if( rc==SQLITE_OK )
                        {
                            /* one chunk copied, continue ...
                             */
                        }
                        else if( rc==SQLITE_BUSY || rc==SQLITE_LOCKED )
                        {
                            /* target file is locked by some other application,
                             * in this case abort the backup.
                             */
                            break;
                        }
                    } while( rc==SQLITE_OK || rc==SQLITE_BUSY || rc==SQLITE_LOCKED );
                    /*
                     * Any other write error (SQLITE_READONLY, SQLITE_NOMEM, SQLITE_BUSY, SQLITE_LOCKED, or an SQLITE_IOERR_XXX) shall stop the backup.
                     */

                    /* Release resources allocated by backup_init(). */
                    (void)sqlite3_backup_finish(pBackup);
                }
                rc = sqlite3_errcode(pFile);
				(void)rc;
            }
            /* Close the database connection opened on database file zFilename
             * and return the result of this function.
             */
            (void)sqlite3_close(pFile);
        }

        rc = sqlite3_close_v2(_db);
        vDebugPrintSQLiteErrorMsg(rc);
        _db = NULL;
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDB::vDebugPrintSQLiteErrorMsg(int rc)
{
    if( IS_SQLITE_ERROR(rc) )
    {
        ETG_TRACE_USR1(("vDebugPrintSQLiteErrorMsg(rc=%d) SQLITE_ERROR %s",rc,sqlite3_errmsg(pHandle())));
        DSM_DEBUG_SQLITE3_ERROR();
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDB::vDebugPrintSQLiteErrorStr(int rc)
{
    if( IS_SQLITE_ERROR(rc) )
    {
        ETG_TRACE_USR1(("vDebugPrintSQLiteErrorStr(rc=%d) SQLITE_ERROR %s",rc,sqlite3_errstr(rc)));
        DSM_DEBUG_SQLITE3_ERROR();
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDB::vWALCheckPoint()
{
    if( _db != NULL )
    {
        int rc;
        rc = _dsm->oDB()->dbExec("PRAGMA incremental_vacuum(128)", NULL, NULL,
                                 NULL);
        vDebugPrintSQLiteErrorMsg(rc);
    }

#ifdef ENABLE_SQLITE_WAL
    if( _db != NULL )
    {
        int rc;
        int nLog=0;       /* OUT: Size of WAL log in frames */
        int nCkpt=0;      /* OUT: Total number of frames checkpointed */
        rc = sqlite3_wal_checkpoint_v2(_dsm->oDB()->pHandle(),
                                       NULL,
                                       SQLITE_CHECKPOINT_PASSIVE,
                                       &nLog,
                                       &nCkpt);
        vDebugPrintSQLiteErrorMsg(rc);
    }
#endif
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDB::bIsLocked( )
{
    int rc;
    int nLog=0;       /* OUT: Size of WAL log in frames */
    int nCkpt=0;      /* OUT: Total number of frames checkpointed */
    rc = sqlite3_wal_checkpoint_v2(_dsm->oDB()->pHandle(),
                                   NULL,
                                   SQLITE_CHECKPOINT_PASSIVE,
                                   &nLog,
                                   &nCkpt);
    if( rc == SQLITE_LOCKED )
    {
        return true;
    }
    return false;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDB::vSetUserVersion( int user_version )
{
    int rc;
    char* sql = sqlite3_mprintf("PRAGMA user_version=%d",user_version);
    if( sql != NULL )
    {
        rc = dbExec( sql,
                     NULL,NULL,NULL);
        vDebugPrintSQLiteErrorMsg(rc);
        sqlite3_free(sql);
    }
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDB::bOpen( const tChar* fname )
{
    ETG_TRACE_USR1(("%p.bOpen(%s)",(void*)this,fname));
    return bOpen( fname, DEFAULT_DSM_FILE_LIMIT );
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDB::bOpen( const char* fname,tU32 u32FileLimit, tBool bTruncate, tBool bInMemory )
{
    tBool bDBIsOK = FALSE;
    SQLITE_API int rc;

#ifdef ENABLE_DSM_DB_FORCE_IN_MEMORY
    bInMemory = TRUE;
#endif
    _bInMemory = bInMemory;

    if(strlen(fname)<=sizeof(_dbfilename))
      {
        strcpy(_dbfilename, fname);
      }
    else
      {
        return FALSE;
      }

    _u32FileLimit = u32FileLimit;

    ETG_TRACE_USR1(("%p.bOpen(%d,%d,%d,%s)",(void*)this,u32FileLimit,(int)bTruncate,(int)bInMemory,fname));
    vClose();

    if((bTruncate == false )&&(_bInMemory == true))
    {
        /* load db from file
         */
        _bInMemory = false;
        if( (rc = sqlite3_open_v2(fname,&_db,SQLITE_OPEN_READWRITE,NULL)) != SQLITE_OK )
        {
            rc = sqlite3_close_v2(_db);
            vDebugPrintSQLiteErrorMsg(rc);
            _db = NULL;
        }
        else if( rc == SQLITE_OK )
        {
#ifdef ENABLE_SQLITE3_TRACE
            sqlite3_trace(_db, dsm_sqlite3_trace, NULL);
#endif
#ifdef ENABLE_SQLITE3_PROFILE
            sqlite3_profile(_db, dsm_sqlite3_profile, NULL);
#endif
        }

        /* quick check ...
         */
        {
            /* temporary open inport db for checking ...
             */
            sqlite3_stmt* pStmt = NULL;
            rc = _dsm->oDB()->dbPrepare(
                        "PRAGMA quick_check",
                        -1,
                        &pStmt,
                        NULL);
            if( rc != SQLITE_NOTADB )
            {
                vDebugPrintSQLiteErrorMsg(rc);
            }
            else
            {
                sqlite3_vfs* vfs = sqlite3_vfs_find(NULL);
                dbReset(pStmt);
                vClose();

                ETG_TRACE_USR1(("bOpen() quick_check=NOTADB"));

                if( vfs != NULL )
                {
                    vfs->xDelete(vfs,fname,1);
                }
            }

            if( pStmt != NULL )
            {
                while(1)
                {
                    rc = dbStep(pStmt);
                    vDebugPrintSQLiteErrorMsg(rc);
                    if( rc == SQLITE_DONE )
                    {
                        break;
                    }
                    else if( rc == SQLITE_ROW )
                    {
                        if( tclDB::dbColumnCount(pStmt) == 1 )
                        {
                            if( strcmp("ok",(char*)tclDB::dbColumnText( pStmt, 0 )) == 0 )
                            {
                                ETG_TRACE_USR1(("bOpen() quick_check=OK"));
                                bDBIsOK = TRUE;
                            }
                            else
                            {
                                ETG_TRACE_USR1(("bOpen() quick_check=NOK"));
                            }
                        }
                        break;
                    }
                }
                dbReset(pStmt);
            }

            /*
             * close it
             */
            vClose();
            _bInMemory = true;

            /*
             * import db from file into a in-memory db
             */
            if( (rc = sqlite3_open_v2(":memory:",&_db,SQLITE_OPEN_READWRITE,NULL)) != SQLITE_OK )
            {
                rc = sqlite3_close_v2(_db);
                vDebugPrintSQLiteErrorMsg(rc);
                _db = NULL;
            }
            else if( rc == SQLITE_OK )
            {
#ifdef ENABLE_SQLITE3_TRACE
                sqlite3_trace(_db, dsm_sqlite3_trace, NULL);
#endif
#ifdef ENABLE_SQLITE3_PROFILE
                sqlite3_profile(_db, dsm_sqlite3_profile, NULL);
#endif

                if( bDBIsOK == true )
                {
                    sqlite3 *pFile;             /* Database connection opened on zFilename */
                    /* Open the database file identified by zFilename. */
                    rc = sqlite3_open(fname, &pFile);

                    if( rc==SQLITE_OK )
                    {
                        sqlite3_backup *pBackup;    /* Backup handle used to copy data */
                        /* Open the sqlite3_backup object used to accomplish the transfer */
                        pBackup = sqlite3_backup_init(_db, "main", pFile, "main");
                        if( pBackup )
                        {
                            /* Each iteration of this loop copies 5 database pages from database
                      ** pDb to the backup database. If the return value of backup_step()
                      ** indicates that there are still further pages to copy, sleep for
                      ** 250 ms before repeating. */
                            do {
                                rc = sqlite3_backup_step(pBackup, 10);
                                if( rc==SQLITE_OK )
                                {
                                    /* one chunk copied, continue ...
                                     */
                                }
                                else if( rc==SQLITE_BUSY || rc==SQLITE_LOCKED )
                                {
                                    /* target file is locked by some other application,
                                     * in this case abort the backup.
                                     */
                                    break;
                                }
                            } while( rc==SQLITE_OK || rc==SQLITE_BUSY || rc==SQLITE_LOCKED );
                            /*
                             * Any other write error (SQLITE_READONLY, SQLITE_NOMEM, SQLITE_BUSY, SQLITE_LOCKED, or an SQLITE_IOERR_XXX) shall stop the backup.
                             */

                            /* Release resources allocated by backup_init(). */
                            (void)sqlite3_backup_finish(pBackup);
                        }
                        rc = sqlite3_errcode(pFile);

                        if( rc==SQLITE_OK )
                        {
                            bDBIsOK = TRUE;
                        }
                    }
                    /* Close the database connection opened on database file zFilename
                  ** and return the result of this function. */
                    (void)sqlite3_close(pFile);
                }
            }
        }
    }
    else if((bTruncate == false )&&(_bInMemory == false))
    {
        /* load db from file
         */
        if( (rc = sqlite3_open_v2(fname,&_db,SQLITE_OPEN_READWRITE,NULL)) != SQLITE_OK )
        {
            rc = sqlite3_close_v2(_db);
            vDebugPrintSQLiteErrorMsg(rc);
            _db = NULL;
        }
        else if( rc == SQLITE_OK )
        {
#ifdef ENABLE_SQLITE3_TRACE
            sqlite3_trace(_db, dsm_sqlite3_trace, NULL);
#endif
#ifdef ENABLE_SQLITE3_PROFILE
            sqlite3_profile(_db, dsm_sqlite3_profile, NULL);
#endif
        }

        /* quick check ...
         */
        if(( _db != NULL )&&(bInMemory==false))
        {
            sqlite3_stmt* pStmt = NULL;
            rc = _dsm->oDB()->dbPrepare(
                        "PRAGMA quick_check",
                        -1,
                        &pStmt,
                        NULL);
            if( rc != SQLITE_NOTADB )
            {
                vDebugPrintSQLiteErrorMsg(rc);
            }
            else
            {
                sqlite3_vfs* vfs = sqlite3_vfs_find(NULL);
                dbReset(pStmt);
                vClose();

                ETG_TRACE_USR1(("bOpen() quick_check=NOTADB"));

                if( vfs != NULL )
                {
                    vfs->xDelete(vfs,fname,1);
                }
            }
            if( pStmt != NULL )
            {
                while(1)
                {
                    rc = dbStep(pStmt);
                    vDebugPrintSQLiteErrorMsg(rc);
                    if( rc == SQLITE_DONE )
                    {
                        break;
                    }
                    else if( rc == SQLITE_ROW )
                    {
                        if( tclDB::dbColumnCount(pStmt) == 1 )
                        {
                            if( strcmp("ok",(char*)tclDB::dbColumnText( pStmt, 0 )) == 0 )
                            {
                                ETG_TRACE_USR1(("bOpen() quick_check=OK"));
                                bDBIsOK = TRUE;
                            }
                            else
                            {
                                ETG_TRACE_USR1(("bOpen() quick_check=NOK"));
                            }
                        }
                        break;
                    }
                }
                dbReset(pStmt);
            }
        }

        if((_db != NULL)&&(bDBIsOK == FALSE))
        {
            sqlite3_vfs* vfs = sqlite3_vfs_find(NULL);
            vClose();
            if( vfs != NULL )
            {
                if( vfs->xDelete(vfs,fname,1) == SQLITE_OK )
                {
                    return  bOpen(fname,u32FileLimit);
                }
            }
            return FALSE;
        }

        /* database user version check ...
         */
        if((_db != NULL)&&(bInMemory==false))
        {
            sqlite3_stmt* pStmt = NULL;
            int user_version = 0;
            rc = _dsm->oDB()->dbPrepare(
                        "PRAGMA user_version",
                        -1,
                        &pStmt,
                        NULL);
            vDebugPrintSQLiteErrorMsg(rc);
            if( pStmt != NULL )
            {
                while(1)
                {
                    rc = dbStep(pStmt);
                    vDebugPrintSQLiteErrorMsg(rc);
                    if( rc == SQLITE_DONE )
                    {
                        break;
                    }
                    else if( rc == SQLITE_ROW )
                    {
                        if( tclDB::dbColumnCount(pStmt) == 1 )
                        {
                            user_version = tclDB::dbColumnInt( pStmt, 0 );
                            if( (user_version>>16) != SQLITE_USER_VERSION_MAJOR )
                            {
                                sqlite3_vfs* vfs = sqlite3_vfs_find(NULL);
                                dbReset(pStmt);
                                vClose();
                                if( vfs != NULL )
                                {
                                    rc = vfs->xDelete(vfs,fname,1);
                                    if( rc == SQLITE_OK )
                                    {
                                        return  bOpen(fname,u32FileLimit);
                                    }
                                }
                                return FALSE;
                            }
                        }
                        break;
                    }
                }
                dbReset(pStmt);
            }
        }
    }

    if( _db == NULL )
    {
        if( ( bTruncate == true ) )
        {
            sqlite3_vfs* vfs = sqlite3_vfs_find(NULL);
            if( vfs != NULL )
            {
                rc = vfs->xDelete(vfs,fname,1);
            }
        }
        if( (rc = sqlite3_open_v2(fname,&_db,SQLITE_OPEN_CREATE|SQLITE_OPEN_READWRITE,NULL)) != SQLITE_OK )
        {
            vClose();
        }
        else if( rc == SQLITE_OK )
        {
#ifdef ENABLE_SQLITE3_TRACE
            sqlite3_trace(_db, dsm_sqlite3_trace, NULL);
#endif
#ifdef ENABLE_SQLITE3_PROFILE
            sqlite3_profile(_db, dsm_sqlite3_profile, NULL);
#endif
        }
        if( _db != NULL )
        {
            char sql[32];
            sprintf(sql,"PRAGMA page_size=%d;",CONFIG_USE_SQLITE_PAGE_SIZE);
            rc = dbExec( sql,
                         NULL,NULL,NULL);
            vDebugPrintSQLiteErrorMsg(rc);
        }

        if( _db != NULL )
        {
            vSetUserVersion(SQLITE_USER_VERSION);
        }

        _bIsNew = true;
    }
    else
    {
        _bIsNew = false;
    }

    /* set busy timeout ...
     */
    if( _db != NULL )
    {
        sqlite3_busy_timeout(_db, 10);
    }

    /* initialize tables ...
     */
    if( _db != NULL )
    {
        vInitializeTables();
    }

    return _db!=NULL?TRUE:FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDB::vSetFileLimit( tU32 u32FileLimit )
{
    _u32FileLimit = u32FileLimit;
#if 1
    if( _db != NULL )
    {
        SQLITE_API int rc;
        char sql[64];
        sprintf(sql,"PRAGMA max_page_count=%d",_u32FileLimit/_u32PageSize);

        _u32MaxPageCount = _u32FileLimit/_u32PageSize;
        if( (_u32MaxPageCount*_u32PageSize) >= (4*1024*1024) )
        {
            /* GC shall reserve 1MB for new data ...
             */
            _u32IsFullPageCount = _u32MaxPageCount - ((1*1024*1024)/_u32PageSize);
        }
        else
        {
            /* GC shall reserve 1/4 for new data ...
             */
            _u32IsFullPageCount = _u32MaxPageCount * 3/4;
        }
        rc = dbExec( sql,NULL,NULL,NULL);
#ifdef CONFIG_DSM_ENABLE_USE_INCREMENTAL_VACUUM_ON_INIT
        rc = dbExec("PRAGMA incremental_vacuum(128)", NULL, NULL,NULL);
#else /* CONFIG_DSM_ENABLE_USE_INCREMENTAL_VACUUM_ON_INIT */
        rc = dbExec( "VACUUM",NULL,NULL,NULL);
#endif /* CONFIG_DSM_ENABLE_USE_INCREMENTAL_VACUUM_ON_INIT */
        (void)rc;
    }
#endif
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDB::bIsNew()
{
    return _bIsNew;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDB::bIsFull()
{
    int rc;
    sqlite3_stmt* pStmt = NULL;
    /* Ask for the number of pages ...
     */
    rc = _dsm->oDB()->dbPrepare(
                "PRAGMA page_count",
                -1,
                &pStmt,
                NULL);
    vDebugPrintSQLiteErrorMsg(rc);
    _u32PageCount = 0;
    if( pStmt != NULL )
    {
        while(1)
        {
            rc = dbStep(pStmt);
            vDebugPrintSQLiteErrorMsg(rc);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( rc == SQLITE_ROW )
            {
                if( tclDB::dbColumnCount(pStmt) == 1 )
                {
                    _u32PageCount = tclDB::dbColumnInt( pStmt, 0 );
                }
                break;
            }
        }
        dbReset(pStmt);
        pStmt = NULL;
    }
    /* Ask for the number of pages in free list ...
     */
    rc = _dsm->oDB()->dbPrepare(
                "PRAGMA freelist_count",
                -1,
                &pStmt,
                NULL);
    vDebugPrintSQLiteErrorMsg(rc);
    _u32FreeListCount = 0;
    if( pStmt != NULL )
    {
        while(1)
        {
            rc = dbStep(pStmt);
            vDebugPrintSQLiteErrorMsg(rc);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( rc == SQLITE_ROW )
            {
                if( tclDB::dbColumnCount(pStmt) == 1 )
                {
                    _u32FreeListCount = tclDB::dbColumnInt( pStmt, 0 );
                }
                break;
            }
        }
        dbReset(pStmt);
        pStmt = NULL;
    }
    if( _u32FreeListCount > _u32PageCount )
    {
        _u32PageCount = _u32FreeListCount;
    }
    _u32UsedPageCount = _u32PageCount-_u32FreeListCount;

    if(_u32UsedPageCount >= _u32IsFullPageCount)
    {
        _u32IsFullCounter++;
    }

    ETG_TRACE_USR3(("bIsFull() status u32MaxPageCount=%d(%d) u32PageCount=%d u32FreeListCount=%d => u32UsedPageCount=%d < u32IsFullPageCount=%d u32IsFullCounter=%d u32IsToBigCounter=%d",
                    _u32MaxPageCount,_u32FileLimit,
                    _u32PageCount,
                    _u32FreeListCount,
                    _u32UsedPageCount,
                    _u32IsFullPageCount,
                    _u32IsFullCounter,
                    _u32IsToBigCounter));

    return (_u32UsedPageCount >= _u32IsFullPageCount)?TRUE:FALSE;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tBool tclDB::bIsToBig()
{
    if( bIsFull() )
    {
        if(_u32UsedPageCount >= _u32MaxPageCount)
        {
            _u32IsToBigCounter++;
        }
        ETG_TRACE_USR3(("bIsToBig() status u32IsToBigCounter=%d",
                        _u32IsToBigCounter));
        return (_u32UsedPageCount >= _u32MaxPageCount)?TRUE:FALSE;
    }
    return FALSE;
}

/******************************************************************************
 * sqlite3_* wrapper functions ...
 *****************************************************************************/
tS32 tclDB::s32dbExecSimple(const char *zSql)
{
    tS32 retVal = 0;
    sqlite3_stmt* pStmt01 = NULL;
    int rc = dbPrepare(zSql,-1, &pStmt01, NULL);

    if( pStmt01 != NULL )
    {
        while(1)
        {
            rc = dbStep(pStmt01);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( rc == SQLITE_ROW )
            {
                retVal = dbColumnInt( pStmt01,0 );
                break;
            }
            else if( dbOnStepError(rc) )
            {
                break;
            }
        }
        dbReset(pStmt01);
    }
    return retVal;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
tVoid tclDB::vInitializeTables()
{
    SQLITE_API int rc;
    if( _db == NULL )
    {
        return;
    }

#ifdef SQLITE_DEBUG
#ifdef ENABLE_SQLITE3_TRACE
    rc = dbExec(
                "PRAGMA vdbe_trace=true",
                NULL,NULL,NULL);
    vDebugPrintSQLiteErrorMsg(rc);
#endif /* ENABLE_SQLITE3_TRACE */
#endif /* SQLITE_DEBUG */

    rc = dbExec(
                "PRAGMA temp_store=MEMORY",
                NULL,NULL,NULL);
    vDebugPrintSQLiteErrorMsg(rc);

    vSetFileLimit(u32FileLimit());

#if defined(CONFIG_USE_SQLITE3_SYNCHRONOUS_OFF)
    rc = dbExec(
                "PRAGMA synchronous=OFF",
                NULL,NULL,NULL);
    vDebugPrintSQLiteErrorMsg(rc);
#elif defined(CONFIG_USE_SQLITE3_SYNCHRONOUS_NORMAL)
    rc = dbExec(
                "PRAGMA synchronous=NORMAL",
                NULL,NULL,NULL);
    vDebugPrintSQLiteErrorMsg(rc);
#endif

    rc = dbExec(
                "PRAGMA auto_vacuum=INCREMENTAL",
                NULL,NULL,NULL);
    vDebugPrintSQLiteErrorMsg(rc);

    rc = dbExec(
                "PRAGMA automatic_index=false",
                NULL,NULL,NULL);
    vDebugPrintSQLiteErrorMsg(rc);

#ifdef CONFIG_USE_NUM_PAGES_FOR_PAGECACHE
     {
        char sql[64];
        sprintf(sql,"PRAGMA cache_size=%d",CONFIG_USE_NUM_PAGES_FOR_PAGECACHE);
        rc = dbExec(
                    sql,
                    NULL,NULL,NULL);
        vDebugPrintSQLiteErrorMsg(rc);
    }
#endif

#ifdef CONFIG_DSM_ENABLE_USE_LOCKING_MODE_EXCLUSIVE
    rc = dbExec(
                "PRAGMA locking_mode=EXCLUSIVE",
                NULL,NULL,NULL);
    vDebugPrintSQLiteErrorMsg(rc);
#endif

#ifdef ENABLE_SQLITE_WAL
    rc = dbExec(
                "PRAGMA journal_mode=WAL",
                NULL,NULL,NULL);
    vDebugPrintSQLiteErrorMsg(rc);

    rc = sqlite3_wal_autocheckpoint(_db,64*1024/_u32PageSize);
    vDebugPrintSQLiteErrorMsg(rc);
#endif

    rc = dbExec(
                "PRAGMA journal_size_limit=1048578",
                NULL,NULL,NULL);
    vDebugPrintSQLiteErrorMsg(rc);

    rc = dbExec(
                "PRAGMA foreign_keys=true",
                NULL,NULL,NULL);
    vDebugPrintSQLiteErrorMsg(rc);

    rc = dbExec(
                "PRAGMA recursive_triggers=true",
                NULL,NULL,NULL);
    vDebugPrintSQLiteErrorMsg(rc);

#ifdef CONFIG_DSM__ENABLE_DAB
    _dsm->oDAB()->vInitializeTables();
#endif /* CONFIG_DSM__ENABLE_DAB */

    /* create sqlite_stat1 table
     */
    rc = dbExec("ANALYZE sqlite_master",NULL,NULL,NULL);
    vDebugPrintSQLiteErrorMsg(rc);

    {
        const char* sqlite_stat1[] = {
            "INSERT INTO sqlite_stat1 (tbl,idx,stat) VALUES ('dab_epg_root','i_dab_epg_root_1','1580 1');",
            "INSERT INTO sqlite_stat1 (tbl,idx,stat) VALUES ('dab_epg','i_dab_epg_5','1540 1540');",
            "INSERT INTO sqlite_stat1 (tbl,idx,stat) VALUES ('dab_epg','i_dab_epg_4','220 220');",
            "INSERT INTO sqlite_stat1 (tbl,idx,stat) VALUES ('dab_epg','i_dab_epg_2','316660 2');",
    #if 0
            /* this index makes no sense, because it is not very selective and
             * creates many flat updates while DG reception
             */
            "INSERT INTO sqlite_stat1 (tbl,idx,stat) VALUES ('dab_epg','i_dab_epg_1','58668 2347');",
    #endif
            "INSERT INTO sqlite_stat1 (tbl,idx,stat) VALUES ('dsm_param','','2');",
            "INSERT INTO sqlite_stat1 (tbl,idx,stat) VALUES ('dsm_log','','100');",
            "INSERT INTO sqlite_stat1 (tbl,idx,stat) VALUES ('dab_data_application','sqlite_autoindex_dab_data_application_1','17 1');",
    #ifdef CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE
            "INSERT INTO sqlite_stat1 (tbl,idx,stat) VALUES ('dab_mot_object','i_dab_mot_object_1','5505 7 6 5');",
    #else
            "INSERT INTO sqlite_stat1 (tbl,idx,stat) VALUES ('dab_datagroup_mot','i_dab_datagroup_mot_1','5505 7 6 5');",
            "INSERT INTO sqlite_stat1 (tbl,idx,stat) VALUES ('dab_datagroup_mot','sqlite_autoindex_dab_datagroup_mot_1','5505 324 145 5 1');",
    #ifdef CONFIG_DSM_ENABLE_USE_dab_datagroup_mot_age_received_TABLE
            "INSERT INTO sqlite_stat1 (tbl,idx,stat) VALUES ('dab_datagroup_mot_age_received','sqlite_autoindex_dab_datagroup_mot_age_received_1','5505 1');",
    #endif
    #ifdef CONFIG_DSM_ENABLE_USE_dab_mot_object_complete_TABLE
            "INSERT INTO sqlite_stat1 (tbl,idx,stat) VALUES ('dab_mot_object_complete','sqlite_autoindex_dab_mot_object_complete_1','1173 69 33 1');",
    #endif
    #endif /* CONFIG_DSM_ENABLE_USE_dab_mot_object_TABLE */
            "INSERT INTO sqlite_stat1 (tbl,idx,stat) VALUES ('dab_mot_directory','','13');",
#ifndef CONFIG_DSM_ENABLE_MOT_DIRECTORY_PARAM_BY_VTAB
            "INSERT INTO sqlite_stat1 (tbl,idx,stat) VALUES ('dab_mot_directory_param','i_dab_mot_directory_param_1','3600 180 1 1');",
            "INSERT INTO sqlite_stat1 (tbl,idx,stat) VALUES ('dab_mot_directory_cname','i_dab_mot_directory_cname_1','440 22 1 1');",
#endif
            NULL
        };

        rc = dbExec("DELETE FROM sqlite_stat1",NULL,NULL,NULL);
        int i=0;
        while( sqlite_stat1[i] != NULL )
        {
            rc = dbExec(sqlite_stat1[i],NULL,NULL,NULL);
            i++;
        }
    }

    _dsm->log("oDB","started");
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
sqlite3* tclDB::pHandle()
{
    return _db;
}

/******************************************************************************
 * FUNCTION:
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * HISTORY:
 *****************************************************************************/
SQLITE_API tBool tclDB::dbOnStepError(int rc)
{
    if( rc == SQLITE_ERROR )
    {
        return TRUE;
    }
    else if( rc == SQLITE_CONSTRAINT )
    {
        return TRUE;
    }
    else if( IS_SQLITE_ERROR(rc) )
    {
        return TRUE;
    }
    else if((rc == SQLITE_FULL)||(rc == SQLITE_IOERR)||(rc == SQLITE_NOMEM)||(rc == SQLITE_BUSY)||(rc == SQLITE_CONSTRAINT))
    {
        return TRUE;
    }
    return FALSE;
}

/******************************************************************************
 * sqlite3_* wrapper functions ...
 *****************************************************************************/
SQLITE_API tU32 tclDB::u32dbExecSimple(const char *zSql)
{
    tU32 u32Value = 0;
    SQLITE_API int rc;
    sqlite3_stmt* pStmt = NULL;

    rc = dbPrepare(zSql,
                                -1,
                                &pStmt,
                                NULL);
    if( pStmt != NULL )
    {
        while(1)
        {
            rc = _dsm->oDB()->dbStep(pStmt);
            if( rc == SQLITE_DONE )
            {
                break;
            }
            else if( rc == SQLITE_ROW )
            {
                if( dbColumnCount(pStmt) == 1 )
                {
                    u32Value = (tU32)dbColumnInt64( pStmt, 0 );
                }
            }
            else if( dbOnStepError(rc) )
            {
                break;
            }
        }
        dbReset(pStmt);
    }
    return u32Value;
}

/******************************************************************************
 * sqlite3_* wrapper functions ...
 *****************************************************************************/
SQLITE_API int tclDB::dbExec(const char *zSql,sqlite3_callback xCallback,void *pArg,char **pzErrMsg)
{
    int rc;

    if( _db == NULL )
    {
        return SQLITE_ERROR;
    }

    rc = sqlite3_exec(_db,zSql,xCallback,pArg,pzErrMsg);

    if(rc == SQLITE_BUSY)
    {
        return rc;
    }
    else if((rc == SQLITE_FULL)||(rc == SQLITE_IOERR)||(rc == SQLITE_NOMEM))
    {
        _u32FileErrorCounter++;
        return rc;
    }
    else if( rc == SQLITE_CONSTRAINT)
    {
        return rc;
    }
    else
    {
        vDebugPrintSQLiteErrorMsg(rc);
    }

    return rc;
}

#ifdef CONFIG_DSM_ENABLE_EXPLAIN_SQL
SQLITE_API int tclDB::dbExplainSQL(const char *zSql)
{
    int rc;
    char *zExplain;                 /* SQL with EXPLAIN QUERY PLAN prepended */
    sqlite3_stmt *pExplain;         /* Compiled EXPLAIN QUERY PLAN command */

    zExplain = sqlite3_mprintf("EXPLAIN QUERY PLAN %s", zSql);
    if( zExplain==0 ) return SQLITE_NOMEM;

#ifdef SQLITE_PREPARE_PERSISTENT
    rc = sqlite3_prepare_v3(_db, zExplain, -1, SQLITE_PREPARE_PERSISTENT, &pExplain, 0);
#else
    rc = sqlite3_prepare_v2(_db, zExplain, -1, &pExplain, 0);
#endif
    sqlite3_free(zExplain);
    if( rc!=SQLITE_OK ) return rc;

    while( SQLITE_ROW==sqlite3_step(pExplain) )
    {
      int iSelectid = sqlite3_column_int(pExplain, 0);
      int iOrder = sqlite3_column_int(pExplain, 1);
      int iFrom = sqlite3_column_int(pExplain, 2);
      const char *zDetail = (const char *)sqlite3_column_text(pExplain, 3);

      ETG_TRACE_USR1(("%d %d %d %s", iSelectid, iOrder, iFrom, zDetail));

      //printf("%d %d %d %s\n", iSelectid, iOrder, iFrom, zDetail);
    }

    return sqlite3_finalize(pExplain);
}
#endif

SQLITE_API int tclDB::dbPrepare(const char *zSql,int nBytes,sqlite3_stmt **ppStmt,const char **pzTail)
{    
    int rc;

    if( _db == NULL )
    {
        return SQLITE_ERROR;
    }

    if( ppStmt == NULL )
    {
		return SQLITE_ERROR;
	}
	if( pStmtPrepareCache != NULL )
	{
		const char* sql = sqlite3_sql(pStmtPrepareCache);
		if(sql!=NULL)
		{
			_u32StmtReuseStrCmpCounter++;
			if( strcmp(sql,zSql) == 0 )
			{
				if( sqlite3_stmt_busy(pStmtPrepareCache) == 0 )
				{
					*ppStmt = pStmtPrepareCache;
					_u32StmtReuseCounter++;
					return SQLITE_OK;
				}
			}
		}
	}
	pStmtPrepareCache = NULL;
	do
	{
		pStmtPrepareCache = sqlite3_next_stmt(_db,pStmtPrepareCache);
		if( pStmtPrepareCache != NULL )
		{
			const char* sql = sqlite3_sql(pStmtPrepareCache);
			_u32StmtReuseStrCmpCounter++;
			if( strcmp(sql,zSql) == 0 )
			{
				if( sqlite3_stmt_busy(pStmtPrepareCache) == 0 )
				{
					*ppStmt = pStmtPrepareCache;
					_u32StmtReuseCounter++;
					return SQLITE_OK;
				}
			}
		}
	} while(pStmtPrepareCache != NULL );


#ifdef CONFIG_DSM_ENABLE_EXPLAIN_SQL
    dbExplainSQL(zSql);
#endif

#ifdef SQLITE_PREPARE_PERSISTENT
    rc = sqlite3_prepare_v3(_db,zSql,nBytes,SQLITE_PREPARE_PERSISTENT,ppStmt,pzTail);
#else
    rc = sqlite3_prepare_v2(_db,zSql,nBytes,ppStmt,pzTail);
#endif
    _u32StmtPrepareCounter++;

    pStmtPrepareCache = *ppStmt;

    if(rc == SQLITE_BUSY)
    {
        return rc;
    }
    else if((rc == SQLITE_FULL)||(rc == SQLITE_IOERR)||(rc == SQLITE_NOMEM))
    {
        _u32FileErrorCounter++;
        return rc;
    }
    else if( rc == SQLITE_CONSTRAINT)
    {
        return rc;
    }
    else
    {
        vDebugPrintSQLiteErrorMsg(rc);
    }

#if defined(ENABLE_GTEST) && (defined(TR_ENABLE))
#if TR_ENABLE==1
    if( rc == SQLITE_OK )
    {
        if( ppStmt != NULL )
        {
            if( *ppStmt != NULL )
            {
                int FULLSCAN_STEP = sqlite3_stmt_status(*ppStmt,SQLITE_STMTSTATUS_FULLSCAN_STEP,true);
                int SORT = sqlite3_stmt_status(*ppStmt,SQLITE_STMTSTATUS_SORT,true);
                int AUTOINDEX = sqlite3_stmt_status(*ppStmt,SQLITE_STMTSTATUS_AUTOINDEX,true);
                int VM_STEP = sqlite3_stmt_status(*ppStmt,SQLITE_STMTSTATUS_VM_STEP,true);
                ETG_TRACE_USR1(("dbPrepare.stmt_status FULLSCAN_STEP=%d SORT=%d AUTOINDEX=%d VM_STEP=%d (%s)",
                                FULLSCAN_STEP,
                                SORT,
                                AUTOINDEX,
                                VM_STEP,
                                sqlite3_sql(*ppStmt)));
            }
        }
    }
#endif
#endif

    return rc;
}

SQLITE_API int tclDB::dbStep(sqlite3_stmt *pStmt)
{
    int rc;
    rc = sqlite3_step(pStmt);

    if(rc == SQLITE_BUSY)
    {
        return rc;
    }
    else if((rc == SQLITE_FULL)||(rc == SQLITE_IOERR)||(rc == SQLITE_NOMEM))
    {
        _u32FileErrorCounter++;
        return rc;
    }
    else if( rc == SQLITE_CONSTRAINT)
    {
        return rc;
    }
    else
    {
        vDebugPrintSQLiteErrorMsg(rc);
    }

    return rc;
}

SQLITE_API int tclDB::dbReset(sqlite3_stmt *pStmt)
{
    int rc;

    if( bIsOpen() == FALSE )
    {
        return SQLITE_ERROR;
    }

#if defined(ENABLE_GTEST) && defined(TR_ENABLE)
#if TR_ENABLE==1
    if( pStmt != NULL )
    {
        int FULLSCAN_STEP = sqlite3_stmt_status(pStmt,SQLITE_STMTSTATUS_FULLSCAN_STEP,true);
        int SORT = sqlite3_stmt_status(pStmt,SQLITE_STMTSTATUS_SORT,true);
        int AUTOINDEX = sqlite3_stmt_status(pStmt,SQLITE_STMTSTATUS_AUTOINDEX,true);
        int VM_STEP = sqlite3_stmt_status(pStmt,SQLITE_STMTSTATUS_VM_STEP,true);
        ETG_TRACE_USR1(("dbReset.stmt_status FULLSCAN_STEP=%d SORT=%d AUTOINDEX=%d VM_STEP=%d (%s)",
                        FULLSCAN_STEP,
                        SORT,
                        AUTOINDEX,
                        VM_STEP,
                        sqlite3_sql(pStmt)));
    }
#endif
#endif

    rc = sqlite3_reset(pStmt);
    if(rc == SQLITE_BUSY)
    {
        return rc;
    }

    else if((rc == SQLITE_FULL)||(rc == SQLITE_IOERR)||(rc == SQLITE_NOMEM))
    {
        _u32FileErrorCounter++;
        return rc;
    }
    else if( rc == SQLITE_CONSTRAINT)
    {
        return rc;
    }
    else
    {
        vDebugPrintSQLiteErrorMsg(rc);
    }
    return rc;
}

SQLITE_API int tclDB::dbFinalize(sqlite3_stmt *pStmt)
{
    int rc;

    if( bIsOpen() == FALSE )
    {
        return SQLITE_ERROR;
    }

    rc = sqlite3_finalize(pStmt);
    if((rc == SQLITE_FULL)||(rc == SQLITE_IOERR)||(rc == SQLITE_NOMEM)||(rc == SQLITE_BUSY))
    {
        _u32FileErrorCounter++;
        return rc;
    }
    else if( rc == SQLITE_CONSTRAINT)
    {
        return rc;
    }
    else
    {
        vDebugPrintSQLiteErrorMsg(rc);
    }
    return rc;
}

SQLITE_API int tclDB::dbCreateFunction(
        const char *zFunc,
        int nArg,
        int enc,
        void *p,
        void (*xFunc)(sqlite3_context*,int,sqlite3_value **),
        void (*xStep)(sqlite3_context*,int,sqlite3_value **),
        void (*xFinal)(sqlite3_context*),
        void (*xDestroy)(void *)
        )
{
    if( _db == NULL )
    {
        return SQLITE_ERROR;
    }
    return sqlite3_create_function_v2(_db,zFunc,nArg,enc,p,xFunc,xStep,xFinal,xDestroy);
}

SQLITE_API int tclDB::dbBindBlob(sqlite3_stmt* pStmt, int col, const void* v, int n, void(*cb)(void*))
{
    int rc;
    rc = sqlite3_bind_blob(pStmt,col,v,n,cb);
    vDebugPrintSQLiteErrorStr(rc);
    return rc;
}

SQLITE_API int tclDB::dbBindDouble(sqlite3_stmt* pStmt, int col, double v)
{
    int rc;
    rc = sqlite3_bind_double(pStmt,col,v);
    vDebugPrintSQLiteErrorStr(rc);
    return rc;
}

SQLITE_API int tclDB::dbBindInt(sqlite3_stmt* pStmt, int col, int v)
{
    int rc;
    rc = sqlite3_bind_int(pStmt,col,v);
    vDebugPrintSQLiteErrorStr(rc);
    return rc;
}

SQLITE_API int tclDB::dbBindInt64(sqlite3_stmt* pStmt, int col, tS64 v)
{
    int rc;
    rc = sqlite3_bind_int64(pStmt,col,v);
    vDebugPrintSQLiteErrorStr(rc);
    return rc;
}

SQLITE_API int tclDB::dbBindNull(sqlite3_stmt* pStmt, int col)
{
    int rc;
    rc = sqlite3_bind_null(pStmt,col);
    vDebugPrintSQLiteErrorStr(rc);
    return rc;
}

SQLITE_API int tclDB::dbBindText(sqlite3_stmt* pStmt, int col, const char* v, int n, void(*cb)(void*))
{
    int rc;
    rc = sqlite3_bind_text(pStmt,col,v,n,cb);
    vDebugPrintSQLiteErrorStr(rc);
    return rc;
}

SQLITE_API int tclDB::dbBindText16(sqlite3_stmt* pStmt, int col, const void* v, int n, void(*cb)(void*))
{
    int rc;
    rc = sqlite3_bind_text16(pStmt,col,v,n,cb);
    vDebugPrintSQLiteErrorStr(rc);
    return rc;
}

SQLITE_API int tclDB::dbBindValue(sqlite3_stmt* pStmt, int col, const sqlite3_value* v)
{
    int rc;
    rc = sqlite3_bind_value(pStmt,col,v);
    vDebugPrintSQLiteErrorStr(rc);
    return rc;
}

SQLITE_API int tclDB::dbBindZeroblob(sqlite3_stmt* pStmt, int col, int n)
{
    int rc;
    rc = sqlite3_bind_zeroblob(pStmt,col,n);
    vDebugPrintSQLiteErrorStr(rc);
    return rc;
}

SQLITE_API int tclDB::dbColumnCount(sqlite3_stmt* pStmt)
{
    return sqlite3_column_count(pStmt);
}

SQLITE_API const char *tclDB::dbColumnName(sqlite3_stmt* pStmt, int iCol)
{
    return sqlite3_column_name(pStmt,iCol);
}

SQLITE_API int tclDB::dbGetAutoCommit()
{
    if( _db == NULL )
    {
        return SQLITE_ERROR;
    }
    return sqlite3_get_autocommit(_db);
}

SQLITE_API int tclDB::dbChanges()
{
    if( _db == NULL )
    {
        return SQLITE_ERROR;
    }
    return sqlite3_changes(_db);
}

SQLITE_API tS64 tclDB::dbLastInsertRowId()
{
    if( _db == NULL )
    {
        return 0;
    }
    return sqlite3_last_insert_rowid(_db);
}

void* tclDB::dbMalloc(int bytes)
{
    return sqlite3_malloc(bytes);
}
void tclDB::dbFree(void* p)
{
    return sqlite3_free(p);
}
SQLITE_API void tclDB::dbResultInt64(sqlite3_context* ctx, sqlite3_int64 v)
{
    sqlite3_result_int64(ctx,v);
}
SQLITE_API void tclDB::dbResultInt(sqlite3_context* ctx, int v)
{
    sqlite3_result_int(ctx,v);
}

SQLITE_API const void *tclDB::dbColumnBlob(sqlite3_stmt* pStmt, int iCol)
{
    return sqlite3_column_blob(pStmt,iCol);
}

SQLITE_API int tclDB::dbColumnBytes(sqlite3_stmt* pStmt, int iCol)
{
    return sqlite3_column_bytes(pStmt,iCol);
}

SQLITE_API int tclDB::dbColumnBytes16(sqlite3_stmt* pStmt, int iCol)
{
    return sqlite3_column_bytes16(pStmt,iCol);
}

SQLITE_API double tclDB::dbColumnDouble(sqlite3_stmt* pStmt, int iCol)
{
    return sqlite3_column_double(pStmt,iCol);
}

SQLITE_API int tclDB::dbColumnInt(sqlite3_stmt* pStmt, int iCol)
{
    return sqlite3_column_int(pStmt,iCol);
}

SQLITE_API tS64 tclDB::dbColumnInt64(sqlite3_stmt* pStmt, int iCol)
{
    return sqlite3_column_int64(pStmt,iCol);
}

SQLITE_API const unsigned char *tclDB::dbColumnText(sqlite3_stmt* pStmt, int iCol)
{
    return sqlite3_column_text(pStmt,iCol);
}

SQLITE_API const void *tclDB::dbColumnText16(sqlite3_stmt* pStmt, int iCol)
{
    return sqlite3_column_text16(pStmt,iCol);
}

SQLITE_API int tclDB::dbColumnType(sqlite3_stmt* pStmt, int iCol)
{
    return sqlite3_column_type(pStmt,iCol);
}

SQLITE_API sqlite3_value *tclDB::dbColumnValue(sqlite3_stmt* pStmt, int iCol)
{
    return sqlite3_column_value(pStmt,iCol);
}

SQLITE_API const void *tclDB::dbValueBlob(sqlite3_value* v)
{
    return sqlite3_value_blob(v);
}

SQLITE_API int tclDB::dbValueBytes(sqlite3_value* v)
{
    return sqlite3_value_bytes(v);
}

SQLITE_API int tclDB::dbValueBytes16(sqlite3_value* v)
{
    return sqlite3_value_bytes16(v);
}

SQLITE_API double tclDB::dbValueDouble(sqlite3_value* v)
{
    return sqlite3_value_double(v);
}

SQLITE_API int tclDB::dbValueInt(sqlite3_value* v)
{
    return sqlite3_value_int(v);
}

SQLITE_API tS64 tclDB::dbValueInt64(sqlite3_value* v)
{
    return sqlite3_value_int64(v);
}

SQLITE_API const unsigned char *tclDB::dbValueText(sqlite3_value* v)
{
    return sqlite3_value_text(v);
}

SQLITE_API const void *tclDB::dbValueText16(sqlite3_value* v)
{
    return sqlite3_value_text16(v);
}

SQLITE_API const void *tclDB::dbValueText16le(sqlite3_value* v)
{
    return sqlite3_value_text16le(v);
}

SQLITE_API const void *tclDB::dbValueText16be(sqlite3_value* v)
{
    return sqlite3_value_text16be(v);
}

SQLITE_API int tclDB::dbValueType(sqlite3_value* v)
{
    return sqlite3_value_type(v);
}

SQLITE_API int tclDB::dbValueNumericType(sqlite3_value* v)
{
    return sqlite3_value_numeric_type(v);
}

/******************************************************************************
 * EOF
 *****************************************************************************/

