/************************************************************************
 * FILE:        fc_dabtuner_stringUtil.cpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of fc_dabtuner_stringUtil
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      		 | Author                       | Modification
   
				
 *************************************************************************/

#include "fc_dabtuner_stringUtil.hpp"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_UTIL
#include "trcGenProj/Header/fc_dabtuner_stringUtil.cpp.trc.h"
#endif   

#include "dbus/dbus.h"

namespace DAB {
static const tU16 sc_RDS2Unicode[256] = {
    0x0020, 0x0118, 0x012e, 0x0172, 0x0102, 0x0116, 0x010e, 0x0218, 0x021a, 0x010a, 0x0020, 0x0020, 0x0120, 0x0139, 0x017b, 0x0143,
    0x0105, 0x0119, 0x012f, 0x0173, 0x0103, 0x0117, 0x010f, 0x0219, 0x021b, 0x010b, 0x0147, 0x011a, 0x0121, 0x013a, 0x017c, 0x0020,
    0x0020, 0x0021, 0x0022, 0x0023, 0x0142, 0x0025, 0x0026, 0x0027, 0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
    0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f,
    0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f,
    0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x005b, 0x016e, 0x005d, 0x0141, 0x005f,
    0x0104, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f,
    0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x00ab, 0x016f, 0x00bb, 0x013d, 0x0126,
    0x00e1, 0x00e0, 0x00e9, 0x00e8, 0x00ed, 0x00ec, 0x00f3, 0x00f2, 0x00fa, 0x00f9, 0x00d1, 0x00c7, 0x015e, 0x00df, 0x00a1, 0x0178,
    0x00e2, 0x00e4, 0x00ea, 0x00eb, 0x00ee, 0x00ef, 0x00f4, 0x00f6, 0x00fb, 0x00fc, 0x00f1, 0x00e7, 0x015f, 0x011f, 0x0131, 0x00ff,
    0x0136, 0x0145, 0x00a9, 0x0122, 0x011e, 0x011b, 0x0148, 0x0151, 0x0150, 0x20ac, 0x00a3, 0x0024, 0x0100, 0x0112, 0x012A, 0x016a,
    0x0137, 0x0146, 0x013b, 0x0123, 0x013c, 0x0130, 0x0144, 0x0171, 0x0170, 0x00bf, 0x013e, 0x00b0, 0x0101, 0x0113, 0x012b, 0x016b,
    0x00c1, 0x00c0, 0x00c9, 0x00c8, 0x00cd, 0x00cc, 0x00d3, 0x00d2, 0x00da, 0x00d9, 0x0158, 0x010c, 0x0160, 0x017d, 0x00d0, 0x013f,
    0x00c2, 0x00c4, 0x00ca, 0x00cb, 0x00ce, 0x00cf, 0x00d4, 0x00d6, 0x00db, 0x00dc, 0x0159, 0x010d, 0x0161, 0x017e, 0x0111, 0x0140,
    0x00c3, 0x00c5, 0x00c6, 0x0152, 0x0177, 0x00dd, 0x00d5, 0x00d8, 0x00de, 0x014a, 0x0154, 0x0106, 0x015a, 0x0179, 0x0164, 0x00f0,
    0x00e3, 0x00e5, 0x00e6, 0x0153, 0x0175, 0x00fd, 0x00f5, 0x00f8, 0x00fe, 0x014b, 0x0155, 0x0107, 0x015b, 0x017a, 0x0165, 0x0127
};

DAB_stringUtil::DAB_stringUtil()
{
	//Condition to prevent multiple initialization of the map.
	if(m_mapFMRdsEBU2Unicode.size() > 0)
		return;
	/** Filling the map */
	m_mapFMRdsEBU2Unicode[0x24]= 0xa4;
	m_mapFMRdsEBU2Unicode[0x5e]= 0x2014;
	m_mapFMRdsEBU2Unicode[0x60]= 0x2016;
	m_mapFMRdsEBU2Unicode[0x7f]= TUN_CHAR_SPACE;
	m_mapFMRdsEBU2Unicode[0x7e]= 0xaf;
	m_mapFMRdsEBU2Unicode[0x80]= 0xe1;
	m_mapFMRdsEBU2Unicode[0x81]= 0xe0;
	m_mapFMRdsEBU2Unicode[0x82]= 0xe9;
	m_mapFMRdsEBU2Unicode[0x83]= 0xe8;
	m_mapFMRdsEBU2Unicode[0x84]= 0xed;
	m_mapFMRdsEBU2Unicode[0x85]= 0xec;
	m_mapFMRdsEBU2Unicode[0x86]= 0xf3;
	m_mapFMRdsEBU2Unicode[0x87]= 0xf2;
	m_mapFMRdsEBU2Unicode[0x88]= 0xfa;
	m_mapFMRdsEBU2Unicode[0x89]= 0xf9;
	m_mapFMRdsEBU2Unicode[0x8a]= 0xd1;
	m_mapFMRdsEBU2Unicode[0x8b]= 0xc7;
	m_mapFMRdsEBU2Unicode[0x8c]= 0x15e;
	m_mapFMRdsEBU2Unicode[0x8d]= 0xdf;
	m_mapFMRdsEBU2Unicode[0x8e]= 0xa1;
	m_mapFMRdsEBU2Unicode[0x8f]= 0x132;

	m_mapFMRdsEBU2Unicode[0x90]= 0xe2;
	m_mapFMRdsEBU2Unicode[0x91]= 0xe4;
	m_mapFMRdsEBU2Unicode[0x92]= 0xea;
	m_mapFMRdsEBU2Unicode[0x93]= 0xeb;
	m_mapFMRdsEBU2Unicode[0x94]= 0xee;
	m_mapFMRdsEBU2Unicode[0x95]= 0xef;
	m_mapFMRdsEBU2Unicode[0x96]= 0xf4;
	m_mapFMRdsEBU2Unicode[0x97]= 0xf6;
	m_mapFMRdsEBU2Unicode[0x98]= 0xfb;
	m_mapFMRdsEBU2Unicode[0x99]= 0xfc;
	m_mapFMRdsEBU2Unicode[0x9a]= 0xf1;
	m_mapFMRdsEBU2Unicode[0x9b]= 0xe7;
	m_mapFMRdsEBU2Unicode[0x9c]= 0x15f;
	m_mapFMRdsEBU2Unicode[0x9d]= 0x11f;
	m_mapFMRdsEBU2Unicode[0x9e]= 0x131;
	m_mapFMRdsEBU2Unicode[0x9f]= 0x133;


	m_mapFMRdsEBU2Unicode[0xa0]= 0xaa;
	m_mapFMRdsEBU2Unicode[0xa1]= 0x03b1;
	m_mapFMRdsEBU2Unicode[0xa2]= 0xa9;
	m_mapFMRdsEBU2Unicode[0xa3]= 0x2030;
	m_mapFMRdsEBU2Unicode[0xa4]= 0x11e;
	m_mapFMRdsEBU2Unicode[0xa5]= 0x11b;
	m_mapFMRdsEBU2Unicode[0xa6]= 0x148;
	m_mapFMRdsEBU2Unicode[0xa7]= 0x151;
	m_mapFMRdsEBU2Unicode[0xa8]= 0x03c0;
	m_mapFMRdsEBU2Unicode[0xa9]= 0x20ac;
	m_mapFMRdsEBU2Unicode[0xaa]= 0xa3;
	m_mapFMRdsEBU2Unicode[0xab]= 0x24;
	m_mapFMRdsEBU2Unicode[0xac]= 0x2190;
	m_mapFMRdsEBU2Unicode[0xad]= 0x2191;
	m_mapFMRdsEBU2Unicode[0xae]= 0x2192;
	m_mapFMRdsEBU2Unicode[0xaf]= 0x2193;

	m_mapFMRdsEBU2Unicode[0xb0]= 0xba;
	m_mapFMRdsEBU2Unicode[0xb1]= 0xb9;
	m_mapFMRdsEBU2Unicode[0xb2]= 0xb2;
	m_mapFMRdsEBU2Unicode[0xb3]= 0xb3;
	m_mapFMRdsEBU2Unicode[0xb4]= 0xb1;
	m_mapFMRdsEBU2Unicode[0xb5]= 0x130;
	m_mapFMRdsEBU2Unicode[0xb6]= 0x144;
	m_mapFMRdsEBU2Unicode[0xb7]= 0x171;
	m_mapFMRdsEBU2Unicode[0xb8]= 0xb5;
	m_mapFMRdsEBU2Unicode[0xb9]= 0xbf;
	m_mapFMRdsEBU2Unicode[0xba]= 0xf7;
	m_mapFMRdsEBU2Unicode[0xbb]= 0xb0;
	m_mapFMRdsEBU2Unicode[0xbc]= 0xbc;
	m_mapFMRdsEBU2Unicode[0xbd]= 0xbd;
	m_mapFMRdsEBU2Unicode[0xbe]= 0xbe;
	m_mapFMRdsEBU2Unicode[0xbf]= 0xa7;

	m_mapFMRdsEBU2Unicode[0xc0]= 0xc1;
	m_mapFMRdsEBU2Unicode[0xc1]= 0xc0;
	m_mapFMRdsEBU2Unicode[0xc2]= 0xc9;
	m_mapFMRdsEBU2Unicode[0xc3]= 0xc8;
	m_mapFMRdsEBU2Unicode[0xc4]= 0xcd;
	m_mapFMRdsEBU2Unicode[0xc5]= 0xcc;
	m_mapFMRdsEBU2Unicode[0xc6]= 0xd3;
	m_mapFMRdsEBU2Unicode[0xc7]= 0xd2;
	m_mapFMRdsEBU2Unicode[0xc8]= 0xda;
	m_mapFMRdsEBU2Unicode[0xc9]= 0xd9;
	m_mapFMRdsEBU2Unicode[0xca]= 0x158;
	m_mapFMRdsEBU2Unicode[0xcb]= 0x10c;
	m_mapFMRdsEBU2Unicode[0xcc]= 0x160;
	m_mapFMRdsEBU2Unicode[0xcd]= 0x17d;
	m_mapFMRdsEBU2Unicode[0xce]= 0x110;
	m_mapFMRdsEBU2Unicode[0xcf]= 0x013f;

	m_mapFMRdsEBU2Unicode[0xd0]= 0xc2;
	m_mapFMRdsEBU2Unicode[0xd1]= 0xc4;
	m_mapFMRdsEBU2Unicode[0xd2]= 0xca;
	m_mapFMRdsEBU2Unicode[0xd3]= 0xcb;
	m_mapFMRdsEBU2Unicode[0xd4]= 0xce;
	m_mapFMRdsEBU2Unicode[0xd5]= 0xcf;
	m_mapFMRdsEBU2Unicode[0xd6]= 0xd4;
	m_mapFMRdsEBU2Unicode[0xd7]= 0xd6;
	m_mapFMRdsEBU2Unicode[0xd8]= 0xdb;
	m_mapFMRdsEBU2Unicode[0xd9]= 0xdc;
	m_mapFMRdsEBU2Unicode[0xda]= 0x159;
	m_mapFMRdsEBU2Unicode[0xdb]= 0x10d;
	m_mapFMRdsEBU2Unicode[0xdc]= 0x161;
	m_mapFMRdsEBU2Unicode[0xdd]= 0x17e;
	m_mapFMRdsEBU2Unicode[0xde]= 0x111;
	m_mapFMRdsEBU2Unicode[0xdf]= 0x0140;

	m_mapFMRdsEBU2Unicode[0xe0]= 0xc3;
	m_mapFMRdsEBU2Unicode[0xe1]= 0xc5;
	m_mapFMRdsEBU2Unicode[0xe2]= 0xc6;
	m_mapFMRdsEBU2Unicode[0xe3]= 0x152;
	m_mapFMRdsEBU2Unicode[0xe4]= 0x0177;
	m_mapFMRdsEBU2Unicode[0xe5]= 0xdd;
	m_mapFMRdsEBU2Unicode[0xe6]= 0xd5;
	m_mapFMRdsEBU2Unicode[0xe7]= 0xd8;
	m_mapFMRdsEBU2Unicode[0xe8]= 0xde;
	m_mapFMRdsEBU2Unicode[0xe9]= 0x014a;
	m_mapFMRdsEBU2Unicode[0xea]= 0x154;
	m_mapFMRdsEBU2Unicode[0xeb]= 0x106;
	m_mapFMRdsEBU2Unicode[0xec]= 0x15a;
	m_mapFMRdsEBU2Unicode[0xed]= 0x179;
	m_mapFMRdsEBU2Unicode[0xee]= 0x0166;
	m_mapFMRdsEBU2Unicode[0xef]= 0xf0;


	m_mapFMRdsEBU2Unicode[0xf0]= 0xe3;
	m_mapFMRdsEBU2Unicode[0xf1]= 0xe5;
	m_mapFMRdsEBU2Unicode[0xf2]= 0xe6;
	m_mapFMRdsEBU2Unicode[0xf3]= 0x153;
	m_mapFMRdsEBU2Unicode[0xf4]= 0x0175;
	m_mapFMRdsEBU2Unicode[0xf5]= 0xfd;
	m_mapFMRdsEBU2Unicode[0xf6]= 0xf5;
	m_mapFMRdsEBU2Unicode[0xf7]= 0xf8;
	m_mapFMRdsEBU2Unicode[0xf8]= 0xfe;
	m_mapFMRdsEBU2Unicode[0xf9]= 0x014b;
	m_mapFMRdsEBU2Unicode[0xfa]= 0x155;
	m_mapFMRdsEBU2Unicode[0xfb]= 0x107;
	m_mapFMRdsEBU2Unicode[0xfc]= 0x15b;
	m_mapFMRdsEBU2Unicode[0xfd]= 0x17a;
	m_mapFMRdsEBU2Unicode[0xfe]= 0x167;

}

/*************************************************************************
 *
 * FUNCTION:     tU16 u16ConvertFromRdsEBU2Unicode( )
 * 
 * DESCRIPTION:  
 *
 * PARAMETER:    
 *
 * RETURNVALUE:  
 *
 *************************************************************************/
tU16 DAB_stringUtil::u16ConvertFromRdsEBU2Unicode( tU8 u8RdsChar )
{ 
    // Initialize with invalid unicode character
    tU16 u16UnicodeValue = DABTUN_INVALID_UNICODE_CHARACTER;

    //-----------------------------------------
    // Characters with value less than 0xff 
    //-----------------------------------------
	//comparison is always true due to limited range of data type
    //if( u8RdsChar <= (tU8)0xff )
        u16UnicodeValue = sc_RDS2Unicode[u8RdsChar];

    return u16UnicodeValue;

}

/*************************************************************************
 *
 * FUNCTION:     tU32 u32ConvertFromUnicode2UTF8( )
 * 
 * DESCRIPTION:  
 *
 * PARAMETER:    
 *
 * RETURNVALUE:  
 *
 *************************************************************************/
tU32 DAB_stringUtil::u32ConvertFromUnicode2UTF8( tU16 u16UnicodeChar )
{ 
    tU32 u32UTF8Char = (tU32)u16UnicodeChar;

    // If the Unicode character is less than 128(0x7f), equivalent UTF8 is one byte long
    // No conversion is required.
    if( u16UnicodeChar < (tU16) 128 )
    {
        u32UTF8Char = (tU32)u16UnicodeChar;

    }
    // If the Unicode data value is between 128( 0x7f) and 2047 (0x7ff), 
    // equivalent UTF8 is two byte long.
    else if( (u16UnicodeChar >= (tU16) 128) && (u16UnicodeChar <= (tU16) 2047) )
    {
        // Get the UTF8 MSB:
        u32UTF8Char = 192 + (u16UnicodeChar / 64 ); 

        u32UTF8Char = (tU32)(u32UTF8Char << 8);

        // OR with UTF8 LSB:
        u32UTF8Char |= ( 128 + (u16UnicodeChar % 64 ) );

    }
     /* 3 byte UTF 8 character.*/
	//comparison is always true due to limited range of data type
	//else if( (u16UnicodeChar >= (tU16) 2048) && (u16UnicodeChar <= (tU16) 65535) )
    else 
    {
      tU16 u16TempUTF8Char = 0;
      // Get the UTF8 MSB. Byte1 = 224 + (ud div 4096)
      u32UTF8Char = (tU32) 224 + (u16UnicodeChar / 4096 ); 

      u32UTF8Char = (u32UTF8Char << 16);

      // Get the 2nd UTF8 MSB. Byte2 = 128 + ((ud div 64) mod 64)
      u16TempUTF8Char = (tU16) (128 + ((u16UnicodeChar / 64 ) % 64)); 

      u16TempUTF8Char = ( tU16 )(u16TempUTF8Char << 8);

      u32UTF8Char |= (tU32) u16TempUTF8Char;

      // Get the LSB. Byte3 = 128 + (ud mod 64)
      u32UTF8Char |= ( 128 + (u16UnicodeChar % 64 ) );
    }
    
    return u32UTF8Char;
}


/*************************************************************************
 *
 * FUNCTION:     tU32 u32ConvertFromRdsEBU2UTF8( )
 * 
 * DESCRIPTION:  
 *
 * PARAMETER:    
 *
 * RETURNVALUE:  
 *
 *************************************************************************/
tU32 DAB_stringUtil::u32ConvertFromRdsEBU2UTF8( tU8 u8RdsChar )
{ 
    tU32 u32UTF8Char = DABTUN_INVALID_UTF8_CHAR;
     
    // Convert from RDS character to equivalent Unicode format.
    tU16 u16UniCodeChar = u16ConvertFromRdsEBU2Unicode( u8RdsChar );

    // Convert Unicode to UTF8 format.
    u32UTF8Char = u32ConvertFromUnicode2UTF8( u16UniCodeChar );     
    
    return u32UTF8Char;
}


tU32 DAB_stringUtil::u32ConvertMecaStringToUtf8( tenDab_CharSet enCharSet, const tChar *acInText,  tU8 *pu8OutString, tU32 u32MaxLen) 
{     
    tU32 u32InLen=(tU32)OSAL_u32StringLength(acInText);

    if (!u32MaxLen) {
        u32MaxLen=0xFF;
    }

    ETG_TRACE_USR1((" u32ConvertMecaStringToUtf8 enCharSet %d u32MaxLen %d u32InLen=%d",enCharSet,u32MaxLen,u32InLen));

    if ( (enCharSet==enDab_CharSetUTF8) || 
         (enCharSet==enDab_CharSetASCII) ) {
        if (u32InLen>u32MaxLen) {
            u32InLen=u32MaxLen;
        }
        
        if(dbus_validate_utf8((char*)acInText,0)){
			OSAL_pvMemoryCopy(pu8OutString, acInText,u32InLen);

        }else{
        	u32InLen=0;
        }
        pu8OutString[u32InLen]='\0';
        ETG_TRACE_USR1((" u32ConvertMecaStringToUtf8 pu8OutString %s",pu8OutString));
        return u32InLen;
    }

    tU32 u32LeadingSpaceCount = 0;
    tU32 u32TrailingSpaceCount = u32InLen;
    while( (u32LeadingSpaceCount <u32InLen) && acInText[u32LeadingSpaceCount] == ' ') {
        u32LeadingSpaceCount++;
    }

    while( u32TrailingSpaceCount && acInText[u32TrailingSpaceCount-1] == ' ') {
        u32TrailingSpaceCount--;
    }

	return u32GetTrimmedString(u32LeadingSpaceCount, u32TrailingSpaceCount,acInText,u32MaxLen,pu8OutString);
}
	
tU32  DAB_stringUtil::u32GetTrimmedString(tU32 u32LeadingSpaceCount, tU32 u32TrailingSpaceCount, const tChar *acInText, tU32 u32MaxLen,tU8 *pu8OutString){
	
	tU32 u32UTFChar = '\0';
    tU32  u32Count     = 0;
    tU32 u32Temp     = 0;
    for(tU32 u32Index = u32LeadingSpaceCount; u32Index < u32TrailingSpaceCount; u32Index++) {
        u32UTFChar = u32ConvertFromRdsEBU2UTF8(acInText[u32Index]);

        // if single byte character MSB starts with 0     
        if ( u32UTFChar < 0x80 ) // Single character 
        {     
            pu8OutString[u32Count] = (tU8) u32UTFChar;
            u32Count++;
            if (u32Count>u32MaxLen) {
                break;
            }
        }
        else if( (u32UTFChar >= 128) && (u32UTFChar <= 65635) )
        {         
            if (u32Count+2>u32MaxLen) {
                break;
            }
            //Extract 2nd Byte MSB from the word
            u32Temp = ( (u32UTFChar) & (0xff00) );      
            u32Temp = (u32Temp >> 8);      
            pu8OutString[u32Count] = (tU8) u32Temp;
            u32Count++;      
            pu8OutString[u32Count] = (tU8) ( (u32UTFChar) & (0x00ff) );      // Extract the LSB byte.
            u32Count++;
        }
        /* 3 byte UTF 8 character.*/
        else if( u32UTFChar > 65635 )
        {
            if (u32Count+3>u32MaxLen) {
                break;
            }
            //Extract 3nd Byte MSB from the word
            u32Temp = ( (u32UTFChar) & (0xff0000) );      
            u32Temp = (u32Temp >> 16);      
            pu8OutString[u32Count] = (tU8) u32Temp;
            u32Count++;
            u32Temp = ( (u32UTFChar) & (0xff00) );      
            u32Temp = (u32Temp >> 8);      
            pu8OutString[u32Count] = (tU8) u32Temp;
            u32Count++;
            pu8OutString[u32Count] = (tU8) ( (u32UTFChar) & (0x00ff) );      // Extract the LSB byte.
            u32Count++;
        }
    }  
    // Null character at the End of the PSName extraction
    pu8OutString[u32Count] = '\0';
    ETG_TRACE_USR1((" u32GetTrimmedString b4 validate u32Count %d pu8OutString %s",u32Count,pu8OutString));
    if(!dbus_validate_utf8((char*)pu8OutString,0)){
    	u32Count=0;
    	pu8OutString[u32Count] = '\0';
    }
    ETG_TRACE_USR1((" u32GetTrimmedString after validate u32Count %d pu8OutString %s",u32Count,pu8OutString));
    return u32Count;

}
// Converting EBU characters to UTF-8. PSARCCB30-1911
tU32 DAB_stringUtil::u32ConvertEBUStringToUtf8( const tChar *acInText, tU32 u32TextLen,  tU8 *pu8OutString, tU32 u32MaxLen){
	tU32 u32UTFChar = '\0';
	tU32 u32Count   = 0;
	tU32 u32Temp    = 0;
	tU32 u32Index   = 0;
	if (u32TextLen == 0) {
		return 0;
	}
	if (!u32MaxLen) {
		u32MaxLen=0xFF;
	}

	 tU32 u32LeadingSpaceCount = 0;
	 tU32 u32InLen=(tU32)OSAL_u32StringLength(acInText);
	 tU32 u32TrailingSpaceCount = u32InLen; // Bug 664285
	   while( (acInText[u32LeadingSpaceCount]) == ' ' && (u32LeadingSpaceCount < u32TextLen))
	   {
		   ETG_TRACE_USR1(("  u32ConvertEBUStringToUtf8::u32LeadingSpaceCount %d",u32LeadingSpaceCount));

	      u32LeadingSpaceCount++;
	   }
	   while( (u32TrailingSpaceCount > 0) && (acInText[u32TrailingSpaceCount - 1]) == ' ')
	   {
		   ETG_TRACE_USR1(("  u32ConvertEBUStringToUtf8::u32TrailingSpaceCount %d",u32TrailingSpaceCount));

	      u32TrailingSpaceCount--;
	   }

	   u32Index=u32LeadingSpaceCount;

	 while (u32Index < u32TrailingSpaceCount && acInText[u32Index] != '\0')
	    {
	       u32UTFChar = u32ConvertFromRdsEBU2UTF8(acInText[u32Index]);
		   if ( u32UTFChar < 0x80 ) // Single character
		   {
			   pu8OutString[u32Count] = (tU8)u32UTFChar;
			   u32Count++;
		   }
		   else if( (u32UTFChar >= 128) && (u32UTFChar <= 65535) ) // Value is 2Byte long
		   {
			   u32Temp = ( (u32UTFChar) & (0xff00) );
			   u32Temp = (u32Temp >> 8);
			   pu8OutString[u32Count] = (tU8) u32Temp;
			   u32Count++;
			   pu8OutString[u32Count] = (tU8) ( (u32UTFChar) & (0x00ff) );      // Extract the LSB byte.
			   u32Count++;
		   }
		   u32Index++;
	    }
	 // Null character at the End of the PSName extraction
	 pu8OutString[u32Count] = '\0';

	if(!dbus_validate_utf8((char*)pu8OutString,0)){
		u32Count=0;
		pu8OutString[u32Count] = '\0';
	}

	 return u32Count;
}

tU32 DAB_stringUtil::u32ConvertUnicodeStringToUtf8( const tChar *acInText, tU32 u32TextLen,  tU8 *pu8OutString, tU32 u32MaxLen) 
{ 
    tU32 u32UTFChar = '\0';
    tU32 u32Count   = 0;
    tU32 u32Temp    = 0;
    tU32 u32Index   = 0;

    if (u32TextLen == 0) {
        return 0;
    }

    if (!u32MaxLen) {
        u32MaxLen=0xFF;
    }

	 tU32 u32LeadingSpaceCount = 0;
	 tU32 u32TrailingSpaceCount = u32TextLen;


	   while(((u32LeadingSpaceCount+1) < u32TextLen) && (DABDRV_GET_U16(&acInText[u32LeadingSpaceCount])) == TUN_UNICODE_CHAR_SPACE)
	   {
	      u32LeadingSpaceCount=u32LeadingSpaceCount+2;
	      ETG_TRACE_USR1(("  u32ConvertEBUStringToUtf8::u32LeadingSpaceCount %d",u32LeadingSpaceCount));
	   }
	   while((u32TrailingSpaceCount-1 > 0)&& (DABDRV_GET_U16(&acInText[u32TrailingSpaceCount-2])) == TUN_UNICODE_CHAR_SPACE )
	   {
		  u32TrailingSpaceCount = u32TrailingSpaceCount-2;
		  ETG_TRACE_USR1(("  u32ConvertEBUStringToUtf8::u32TrailingSpaceCount %d",u32TrailingSpaceCount));
	   }

	   u32Index=u32LeadingSpaceCount;


    while (u32Index < u32TrailingSpaceCount && DABDRV_GET_U16(&acInText[u32Index]) != '\0')
    {
        tU16 u16UniCodeChar = DABDRV_GET_U16(&acInText[u32Index]);
        u32UTFChar = u32ConvertFromUnicode2UTF8(u16UniCodeChar);

        // if single byte character MSB starts with 0     
        if ( u32UTFChar < 0x80 ) // Single character 
        {     
            pu8OutString[u32Count] = (tU8)u32UTFChar;
            u32Count++;
        }
        else if( (u32UTFChar >= 128) && (u32UTFChar <= 65535) ) // Value is 2Byte long
        {         
            u32Temp = ( (u32UTFChar) & (0xff00) );      
            u32Temp = (u32Temp >> 8);      
            pu8OutString[u32Count] = (tU8) u32Temp;      
            u32Count++;      
            pu8OutString[u32Count] = (tU8) ( (u32UTFChar) & (0x00ff) );      // Extract the LSB byte.
            u32Count++;
        }
        else if( u32UTFChar > 65535 )// Value is 3Byte long
        {         
            
            u32Temp = ( (u32UTFChar) & (0xff0000) );      
            u32Temp = (u32Temp >> 16);      
            pu8OutString[u32Count] = (tU8) u32Temp;      
            u32Count++;      
            u32Temp = ( (u32UTFChar) & (0xff00) );      
            u32Temp = (u32Temp >> 8);      
            pu8OutString[u32Count] = (tU8) u32Temp;      
            u32Count++;      
            pu8OutString[u32Count] = (tU8) ( (u32UTFChar) & (0x00ff) );      // Extract the LSB byte.
            u32Count++;
        }
        u32Index=u32Index+2;
    }  
    // Null character at the End of the PSName extraction
    pu8OutString[u32Count] = '\0';

	if(!dbus_validate_utf8((char*)pu8OutString,0)){
		u32Count=0;
		pu8OutString[u32Count] = '\0';
	}
    return u32Count;

}


tU8 DAB_stringUtil::getUt8CharLengthinBytes(char c){
	if (c < 128)
	  {
		return 1;
	  }
	  else if ((c & 0xe0) == 0xc0)
	  {
		  return  2;
	  }
	  else if ((c & 0xf0) == 0xe0)
	  {
		  return  3;
	  }
	  else if ((c & 0xf8) == 0xf0)
	  {
		  return  4;
	  }
	  else if ((c & 0xfc) == 0xf8)
	  {
		  return  5;
	  }
	  else if ((c & 0xfe) == 0xfc)
	  {
		  return  6;
	  }
	  else
	  {
		  return  0;
	  }
}
//FM RDS Conversion

/*************************************************************************
*
* FUNCTION:     DAB_stringUtil::u32ConvertFromFMRdsEBU2UTF8( )
*
* DESCRIPTION:  Converts the RDS data in Air to UTF8 equivalent
*
* PARAMETER:    tU8 - RDS character received in air
*
* RETURNVALUE:  tU32 - UTF8 character.
*
*************************************************************************/
tU32 DAB_stringUtil::u32ConvertFromFMRdsEBU2UTF8( tU8 u8RdsChar )
{
    tU32 u32UTF8Char = (tU32) TUN_INVALID_UTF8_CHAR;

    // Convert from RDS character to equivalent Unicode format.
    tU16 u16UniCodeChar = u16ConvertFromFMRdsEBU2Unicode( u8RdsChar );

    // Convert Unicode to UTF8 format.
    u32UTF8Char = u32ConvertFromUnicode2UTF8( u16UniCodeChar );

    return u32UTF8Char;

}

/*************************************************************************
*
* FUNCTION:     DAB_stringUtil::u16ConvertFromFMRdsEBU2Unicode( )
*
* DESCRIPTION:  Convert the Rds data character received in air (EBU ISO 646 )
*               to Unicode data.
*
* PARAMETER:    RDS character
*
* RETURNVALUE:  Unicode equivalent value.
*
*************************************************************************/
tU16 DAB_stringUtil::u16ConvertFromFMRdsEBU2Unicode( tU8 u8RdsChar )
{
	//call the constructor to initialize the map.
	DAB_stringUtil stringUtil;

	mapFMRdsEBU2Unicode::const_iterator pos;
	if(!m_mapFMRdsEBU2Unicode.empty())
	{
		/** Find the RdsChar in map*/
		pos = m_mapFMRdsEBU2Unicode.find(u8RdsChar);
	}

	/** Check if iterator reached end */
		if(m_mapFMRdsEBU2Unicode.end() != pos )
		{
			/** Return corresponding unicode value */
			return pos->second ;
		}
		else if((u8RdsChar >= 0x20) && (u8RdsChar <= 0x7d))
		{
			/** Return corresponding unicode value */
			return (tU16)u8RdsChar;
		}
		else
		{
			/** Invalid RdsChar */
			return TUN_INVALID_UNICODE_CHARACTER;
		}

	return 0;
}

/*************************************************************************
*
* FUNCTION:     DAB_stringUtil::vConvertRDSPSNametoUtf8(  )
*
* DESCRIPTION:  This function converts the RDS PSname to the UTF8string
*
* PARAMETER:    tU8* aPSName, tString &sUTF8String
*
* RETURNVALUE:   tVoid
*
*06.1.07 pus1kor
*         Initial version
*
*************************************************************************/
tVoid DAB_stringUtil::vConvertRDSPSNametoUtf8( const char *aPSName, char* s8abuffer)
{


   tU32 u32UTFChar = '\0';

   tU16 u16Temp =0;
   tU8 u8Count = 0;

   if(aPSName ==NULL)
   {
      return;
   }

   ETG_TRACE_USR1(("  vConvertRDSPSNametoUtf8::vConvertRDSPSNametoUtf8 () -> Start"));

   tU8 u8LeadingSpaceCount = 0;
   tU8 u8TrailingSpaceCount = 8;


   while( (aPSName[u8LeadingSpaceCount] == ' ' ||
      aPSName[u8LeadingSpaceCount] == '\0') && (u8LeadingSpaceCount < 8))
   {
	   ETG_TRACE_USR1(("  vConvertRDSPSNametoUtf8::Entering into  u8LeadingSpaceCount()"));

      u8LeadingSpaceCount++;
   }
   while( (aPSName[u8TrailingSpaceCount - 1] == ' ' ||
      aPSName[u8TrailingSpaceCount - 1] == '\0') && (u8TrailingSpaceCount > 0) )
   {
	   ETG_TRACE_USR1(("  vConvertRDSPSNametoUtf8::Entering into  u8TrailingSpaceCount()"));

      u8TrailingSpaceCount--;
   }


   for(tU8 u8Index = u8LeadingSpaceCount; u8Index < u8TrailingSpaceCount; u8Index++)
   {
      u32UTFChar = u32ConvertFromFMRdsEBU2UTF8(aPSName[u8Index]);

      ETG_TRACE_USR1(( " tun_HMIManager::vConvertRDSPSNametoUtf8 () -> u32UTFChar=%04x"
                      ,u32UTFChar ));

      // if single byte character MSB starts with 0


      // Single character
      if ( u32UTFChar < 0x80 )
      {
         s8abuffer[u8Count] = (tS8) u32UTFChar;
         u8Count++;

         ETG_TRACE_USR1(( " tun_HMIManager::vConvertRDSPSNametoUtf8 () -> Single character" ));
      }
      else if(u32UTFChar < TUN_INVALID_UTF8_CHAR)
         // Value is 2Byte long
      {

         //Extract 2nd Byte MSB from the word
         u16Temp = ( (u32UTFChar) & (0xff00) );

         u16Temp = (tU16)(u16Temp >> 8);

         s8abuffer[u8Count] = (tS8) u16Temp;

         u8Count++;

         // Extract the LSB byte.
         s8abuffer[u8Count] = (tS8) ( (u32UTFChar) & (0x00ff) );

         u8Count++;

         ETG_TRACE_USR1(( " vConvertRDSPSNametoUtf8::vConvertRDSPSNametoUtf8 () -> Value is 2Byte long" ));
      }
      else if(u32UTFChar == TUN_INVALID_UTF8_CHAR)
      {
    	  s8abuffer[u8Count] = (tS8) 0x20;
         u8Count++;

         ETG_TRACE_USR1(( " vConvertRDSPSNametoUtf8::vConvertRDSPSNametoUtf8 () -> Single character" ));

      }
      else
      { //UTF8 character is 3 byte long. extract the third byte into the array.
         tU32 u32Temp = ((u32UTFChar) & (0xff0000));

         //Extract the 3rd byte of UTF8 character into the array.
         u32Temp = u32Temp >> 16;
         s8abuffer[u8Count] = (tS8) u32Temp;
         //Character count increased to indicate addition of a byte
         u8Count++;

         //Extract the 2nd byte of UTF8 character into the array
         u32Temp = ((u32UTFChar) & (0x00ff00));
         u32Temp = u32Temp >> 8;
         s8abuffer[u8Count] = (tS8) u32Temp;
         //Character count increased to
         u8Count++;

         //Extrace the 1st byte of UTF8 character into the array.
         u32Temp = ((u32UTFChar) & (0x0000ff));
         s8abuffer[u8Count] = (tS8) u32Temp;
         //Character count increased to
         u8Count++;
      }

   }



   if(!dbus_validate_utf8((const char*)s8abuffer,0))
	{
	   u8Count=0;
	}

   // Null character at the End of the PSName extraction
   s8abuffer[u8Count] = '\0';

	}
}
