/************************************************************************
 * FILE:        fc_dabtuner_dispatcher.hpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of fc_dabtuner_dispatcher
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      		 | Author                       | Modification
   
				
 *************************************************************************/

#ifndef DAB_DISPATCHER_HPP
#define DAB_DISPATCHER_HPP

#include "fc_dabtuner_config.h"
#include "fc_dabtuner_config.hpp"
#include "fc_dabtuner_stl.h"
#include "fc_dabtuner_trace.h"
#include "fc_dabtuner_assert.h"
#include "fc_dabtuner_sem.h"
#include "fc_dabtuner_messageHandler.h"
#include "fc_dabtuner_prioQueue.hpp"
#include "fc_dabtuner_timer.hpp"
#include "fc_dabtuner_event.hpp"
#include "fc_dabtuner_typeinfo.h"
#include "fc_dabtuner_trace.h"
#include "fc_dabtuner_profiler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_UTIL_DISP
#include "trcGenProj/Header/fc_dabtuner_dispatcher.hpp.trc.h"
#endif

namespace DAB {
#define DAB_DISP_MAX_STRING_LEN 30
    static tCString DAB_C_DispatchThreadName  = DAB_DISPATCHER_THREAD_NAME;
    static tCString DAB_C_DispatchEvent  = DAB_DISPATCHER_EVENT_NAME;
    static tCString DAB_C_TerminateSemName  = DAB_DISPATCHER_TERMINATE_SEM_NAME;
    static tCString DAB_C_MsgQSemName  = DAB_DISPATCHER_MSG_Q_SEM_NAME;
    static tCString DAB_C_SubscrQSemName  = DAB_DISPATCHER_SUBSCR_Q_SEM_NAME;

	extern tU32 DAB_u32CurTraceCheck;

    // needed data that have to be put to Q to enable later processing of message
    struct DAB_tCallQEntry {
        DAB_MsgCallerBase *poCaller; // pointer to object that shall handle the msg
        DAB_Message *pMsg;
		DAB_tCallQEntry()
		{
			poCaller = NULL;
			pMsg = NULL;
		}
    };

    inline tVoid DAB_PrintThreadInfo(char const *szName) {

#ifdef DAB_USE_OSAL_THREAD_INFO
        tS32 s32OsalRet;
        OSAL_trThreadControlBlock rTcb;
        s32OsalRet = OSAL_s32ThreadControlBlock(OSAL_ThreadWhoAmI(), &rTcb);

        DAB_ASSERT_RETURN(OSAL_OK == s32OsalRet);

        ETG_TRACE_USR4(("ThreadFn(%14s)(pid=%d):Prio=%d name=%s",
                        szName,
                        rTcb.id,
                        rTcb.u32Priority,
                        rTcb.szName));
#else
        (tVoid)szName;
#endif
    }

template<int TRC_CLASS>
class DAB_TraceChecker:public DAB_MessageHandlerBase {
    public:
    DAB_TRACE_INFO_IMPL((tU16)TRC_CLASS);
};

    class DAB_tclDispatcher  {

    public:
        static DAB_TraceChecker<FC_DABTUNER_TR_UTIL_DISP> _oTraceChecker;

        tVoid vClose() {
            if (_hThreadId!=OSAL_ERROR)
            {
                tS32 s32OsalRet;
            
                s32OsalRet = OSAL_s32EventPost(_hEvent, 
                                               (OSAL_tEventMask) enKillEvtMask,
                                               OSAL_EN_EVENTMASK_OR);
                DAB_ASSERT_RETURN(OSAL_OK == s32OsalRet);
                _oTerminateSem.vGet();
                _hThreadId=OSAL_ERROR;
            }
        }


        //lint -esym(1704, DAB_tclDispatcher::DAB_tclDispatcher) prio3 constructor has private access specification
        DAB_tclDispatcher():
            _u32FreeEvts( ~((tU32)enTimerEvtMask | (tU32)enMsgEvtMask | (tU32)enKillEvtMask) )
        {
            _bInitialized=FALSE;
            _oPrioQueue.clear();
            _poTmrMgr = OSAL_NULL;
            _poEvtMgr = OSAL_NULL;
            _hEvent = OSAL_C_INVALID_HANDLE;
            _hThreadId = OSAL_ERROR;
            _u8NumUsrEvents=0;
            _u32InstanceId=_u32NumInstances;
            _u32CallDepth=0;
            _poCentralSem=OSAL_NULL;
            _oTerminateSem.vOpen(cMakeName(DAB_C_TerminateSemName), 0);
            _oPrioQueue.vSetProtection(cMakeName(DAB_C_MsgQSemName) );
            if (!_u32NumInstances) {
                _oSubscrQSem.vOpen(DAB_C_SubscrQSemName);
            }
            _u32NumInstances++;
        };
        
        //copy constructor
      DAB_tclDispatcher(DAB_tclDispatcher const &poDispatcher):
        	_bInitialized(poDispatcher._bInitialized),
        	_poTmrMgr(poDispatcher._poTmrMgr),
        	_poEvtMgr(poDispatcher._poEvtMgr),
        	_hEvent(poDispatcher._hEvent),
        	_hThreadId(poDispatcher._hThreadId),
        	_u8NumUsrEvents(poDispatcher._u8NumUsrEvents),
        	_u32InstanceId(poDispatcher._u32InstanceId),
        	_u32CallDepth(poDispatcher._u32CallDepth),
        	_poCentralSem(poDispatcher._poCentralSem),
        	_u32FreeEvts(poDispatcher._u32FreeEvts)      	
        {
        	
        };


		//assignment operator
		DAB_tclDispatcher &operator=(DAB_tclDispatcher const &poDispatcher){
			if (this == &poDispatcher) {
				return *this;
			}
			_bInitialized=poDispatcher._bInitialized;
        	_poTmrMgr=poDispatcher._poTmrMgr;
        	_poEvtMgr=poDispatcher._poEvtMgr;
        	_hEvent=poDispatcher._hEvent;
        	_hThreadId=poDispatcher._hThreadId;
        	_u8NumUsrEvents=poDispatcher._u8NumUsrEvents;
        	_u32InstanceId=poDispatcher._u32InstanceId;
        	_u32CallDepth=poDispatcher._u32CallDepth;
        	_poCentralSem=poDispatcher._poCentralSem; 			
			return *this;			
		}


        static tVoid vSetDefaultDispathcher(DAB_tclDispatcher *poDispatcher) {
                DAB_poDefaultDispatcher=poDispatcher;
        }

        static DAB_tclDispatcher *poGetDefaultDispatcher() {
            return DAB_poDefaultDispatcher;
        }

        //lint -esym(1551, DAB_tclDispatcher::~DAB_tclDispatcher) prio3: Function may throw exception in destructor
        virtual ~DAB_tclDispatcher() {

            vClose();
            OSAL_s32EventDelete(cMakeName(DAB_C_DispatchEvent));
            _oPrioQueue.clear();

            _u8NumUsrEvents=0;
        

            _bInitialized=FALSE;

            if (OSAL_NULL != _poTmrMgr) {
                OSAL_DELETE _poTmrMgr;
                _poTmrMgr = OSAL_NULL;
            }
            if (OSAL_NULL != _poEvtMgr) {
                OSAL_DELETE _poEvtMgr;
                _poEvtMgr = OSAL_NULL;
            }

            _poCentralSem=OSAL_NULL;

            if (this==DAB_poDefaultDispatcher) {
                DAB_poDefaultDispatcher=OSAL_NULL;
            }
            if (_u32InstanceId==0) {
                _oSubscrQSem.vClose();
            }

        };

        tVoid vInit(tU32 u32Prio, dab_sem *poCentralSem=OSAL_NULL) {
        
            DAB_ASSERT_RETURN(FALSE== _bInitialized);
            DAB_PrintThreadInfo("ccaThread");
#if 0
            OSAL_s32ThreadPriority(OSAL_ThreadWhoAmI(), DAB_GET_CONFIG().u32CcaPrio);
            DAB_PrintThreadInfo("ccaThread2");
#endif
            tS32 s32OsalRet;
            s32OsalRet = OSAL_s32EventCreate(cMakeName(DAB_C_DispatchEvent), 
                                             &_hEvent);
            DAB_ASSERT_RETURN(OSAL_OK == s32OsalRet);

            _poTmrMgr = new DAB_tclTimerMgr(_hEvent, (tU32)enTimerEvtMask);
            DAB_ASSERT_RETURN(OSAL_NULL != _poTmrMgr);
            _poEvtMgr = new DAB_tclEventMgr(_hEvent, _u32FreeEvts);
            DAB_ASSERT_RETURN(OSAL_NULL != _poEvtMgr);

            OSAL_trThreadAttribute    rThreadAttribute 
                = {
                cMakeName(DAB_C_DispatchThreadName),
                u32Prio,
                10000,
                vThreadStarter,
                this      
            };
            _hThreadId=OSAL_ThreadSpawn ( &rThreadAttribute );
            DAB_ASSERT_RETURN(OSAL_ERROR != _hThreadId);
            _poCentralSem=poCentralSem;
            _bInitialized=TRUE;

        };




        // post msg to queue, concrete type is needed to do a deep copy
        template<class M>
        void vPostMsg(DAB_MsgCallerBase *poCaller, M *pMsg, DAB_tenMsgPrio enPrio=DAB_eMsgPrioNormal) {
            ETG_TRACE_USR4(("vPostMsg[%u] Start enPrio %d", _u32InstanceId, (tU8)enPrio));
            DAB_tCallQEntry entry;
            entry.poCaller=poCaller;
            entry.pMsg=new M(*pMsg);
            vPostEntry(&entry, OSAL_NULL == entry.poCaller ? DAB_enCallMode_PostAll:DAB_enCallMode_Post);
            ETG_TRACE_USR4(("vPostMsg[%u]:End", _u32InstanceId));
        }

        void vPostMsgNoCopy(DAB_MsgCallerBase *poCaller, DAB_Message *pMsg, DAB_tenMsgPrio=DAB_eMsgPrioNormal) {
            ETG_TRACE_USR4(("vPostMsgNoCopy[%u] Start", _u32InstanceId));
            DAB_tCallQEntry entry;
            entry.poCaller=poCaller;
            entry.pMsg=pMsg;
            vPostEntry(&entry, OSAL_NULL == entry.poCaller ? DAB_enCallMode_PostAllNoCopy:DAB_enCallMode_PostNoCopy);
            ETG_TRACE_USR4(("vPostMsgNoCopy[%u]:End", _u32InstanceId));
        }


    


        tVoid vNotify(DAB_tclDispatcher const *poSrcDisp, DAB_Message *pMsg, DAB_tenCallMode enCallMode=DAB_enCallMode_CallAll) const {
            DAB_ASSERT_RETURN(OSAL_NULL != pMsg);
            // todo: start critical section
            static tU32 u32NumNotify_=0;
            u32NumNotify_++;
            tU32 u32NumNotify=u32NumNotify_;
            // todo: end critical section
            tBool bDoTrace=_oTraceChecker.bDoTraceUsr1();
            tBool bDoTraceMsg=FALSE;
            list<DAB_MsgCallerBase *> *poSubscriberList=pMsg->poGetSubscriberList();
            _oSubscrQSem.vGet();
            list<DAB_MsgCallerBase *>::iterator iterBegin=poSubscriberList->begin();
            list<DAB_MsgCallerBase *>::iterator iterEnd=poSubscriberList->end();
            list<DAB_MsgCallerBase *>::iterator iter=iterBegin;
            if (bDoTrace) {
                for(iter=iterBegin;iter!=iterEnd;++iter) {
                    if (!(*iter)->_bActive) {
                        continue;
                    }
                    DAB_MessageHandlerBase *pMh = OSAL_NULL; // Coverity
                    pMh=(*iter)->poGetTarget();
                    if (pMh==OSAL_NULL || poSrcDisp != pMh->poGetDispatcher()) {
                        continue;
                    }
                    if (pMh->bDoTraceUsr4()) {
                        ETG_TRACE_USR1(("DAB_tclDispatcher[%d]:start notify(%d) u32NumNotify=%d with msg follows:",
                                        _u32InstanceId,
                                        ETG_CENUM(DAB_tenCallMode, enCallMode),
                                        u32NumNotify));
                        pMsg->vTrace();
                        bDoTraceMsg=TRUE;
                        break;
                    }
                }
            }

            iter=iterBegin;
            _oSubscrQSem.vPost();
            while(iter != iterEnd) {
                DAB_MessageHandlerBase *pMh=OSAL_NULL;
                if ((*iter)->_bActive) {
                    pMh=(*iter)->poGetTarget();
                    if (poSrcDisp != pMh->poGetDispatcher()) {
                        pMh=OSAL_NULL;
                    }
                }

                if (OSAL_NULL != pMh) {
                    tBool bDoTraceMh=FALSE;
                    if (bDoTrace && pMh->bDoTraceUsr4()) {
                        bDoTraceMh=TRUE;
                        ETG_TRACE_USR1(("DAB_tclDispatcher[%u]:notify(u32NumNotify=%d) %s",
                                        _u32InstanceId,
                                        u32NumNotify, pMh->pcGetName()));
                        pMh->vTraceState();
                    }
                    (*iter)->vCallShort(pMsg);
                    if (bDoTraceMh) {
                        pMh->vTraceState();
                    }
                }
                _oSubscrQSem.vGet();
                ++iter;
                _oSubscrQSem.vPost();


            }
            if (bDoTraceMsg) {
                ETG_TRACE_USR1(("DAB_tclDispatcher[%u]:end notify(%d)",
                                _u32InstanceId,
                                u32NumNotify));
            }
        }

        tVoid vCallMsg(DAB_MsgCallerBase *poCaller, DAB_Message *pMsg, DAB_tenCallMode enCallMode=DAB_enCallMode_Call) const {
            DAB_ASSERT_RETURN(OSAL_NULL != poCaller);
            DAB_ASSERT_RETURN(OSAL_NULL != pMsg);
            poCaller->vCall(pMsg, enCallMode);
        }

        tVoid vLock() {
            if (OSAL_NULL != _poCentralSem) {
                _poCentralSem->vGet();
            }
        }
    
        tVoid vUnlock() {
            if (OSAL_NULL != _poCentralSem) {
                _poCentralSem->vPost();
            }
        }

        //lint -esym(1763,DAB::DAB_tclDispatcher::poGetTmrMgr) "Member function '... const' marked as const indirectly modifies class"
        DAB_tclTimerMgr *poGetTmrMgr() const {
            return _poTmrMgr;
        }

        //lint -esym(1763,DAB::DAB_tclDispatcher::poGetEvtMgr) "Member function '... const' marked as const indirectly modifies class"
        DAB_tclEventMgr *poGetEvtMgr() const {
            return _poEvtMgr;
        }

        static tVoid vLockSubscrQ() {
            _oSubscrQSem.vGet();
        }
        static tVoid vUnLockSubscrQ() {
            _oSubscrQSem.vPost();
        }
        tU32 u32GetInstanceId() const {
            return _u32InstanceId;
        }

    private:

        typedef enum {
            enTimerEvtMask=0x01,
            enMsgEvtMask=0x02,
            enKillEvtMask=0x04
        } tenEventMask;



        tVoid vPostEntry(DAB_tCallQEntry const *pEntry, DAB_tenCallMode enCallMode) {
           
            tS32 s32OsalRet;
            tBool bDoTrace=_oTraceChecker.bDoTraceUsr1();
            if (bDoTrace) {
                if (OSAL_NULL != pEntry->poCaller) {
                    DAB_vTracePreCall(pEntry->poCaller->poGetTarget(), pEntry->pMsg, enCallMode);
                }else {
                    DAB_vTracePreCall(OSAL_NULL, pEntry->pMsg, enCallMode);
                }
                
            }
            _oPrioQueue.push(*pEntry);
            s32OsalRet = OSAL_s32EventPost(_hEvent, 
                                           (OSAL_tEventMask) enMsgEvtMask,
                                           OSAL_EN_EVENTMASK_OR);
            DAB_ASSERT_RETURN(OSAL_OK == s32OsalRet);
        
        };

        static tVoid vThreadStarter(tVoid* pvArg) {
            DAB_tclDispatcher *poThis= (DAB_tclDispatcher *)pvArg;
            DAB_ASSERT_RETURN(OSAL_NULL != poThis);
            poThis->ThreadFn();
        };

        tVoid ThreadFn() {
            tS32 s32OsalRet;

            DAB_PrintThreadInfo("dispThread2");

            for (;;) {
                OSAL_tEventMask ResultMask;

                s32OsalRet =  OSAL_s32EventWait(_hEvent, 
                                                0xFFFFFFFF, 
                                                OSAL_EN_EVENTMASK_OR,
                                                OSAL_C_TIMEOUT_FOREVER ,
                                                &ResultMask);
                DAB_ASSERT_RETURN(OSAL_OK == s32OsalRet);

                ETG_TRACE_USR4_CLS((FC_DABTUNER_TR_UTIL_SYNC, "ThreadFn[%u]:calling semGet",
                                    _u32InstanceId));
                vLock();
                ETG_TRACE_USR2(("ThreadFn[%u]:got evt=0x%08x",
                                _u32InstanceId,
                                ResultMask));
                DAB_u32CurTraceCheck++;
                ETG_TRACE_USR4_CLS((FC_DABTUNER_TR_UTIL_SYNC, "ThreadFn[%u]:semGet OK",
                                    _u32InstanceId));
                s32OsalRet = OSAL_s32EventPost(_hEvent, 
                                               ~ResultMask,
                                               OSAL_EN_EVENTMASK_AND);
                // todo:make trace, no return
                DAB_ASSERT_RETURN(OSAL_OK == s32OsalRet);

                if (ResultMask & (tU32)enKillEvtMask) {
                    ETG_TRACE_USR1(("ThreadFn[%u]:got terminate evt",
                                    _u32InstanceId));
                    vUnlock();
                    _oTerminateSem.vPost();
                    break;

                }

            
                if (ResultMask & (tU32)enTimerEvtMask) {
                    _poTmrMgr->vHandleTimerEvent();
                }
                if (ResultMask & (tU32)enMsgEvtMask) {
                    vProcessMsgs();
                }
                if (ResultMask & _u32FreeEvts) {
                    _poEvtMgr->vProcess(ResultMask);
                }

                ETG_TRACE_USR2(("ThreadFn[%u]:All Events done",
                                _u32InstanceId));

                ETG_TRACE_USR4_CLS((FC_DABTUNER_TR_UTIL_SYNC, "ThreadFn[%u]:calling semGet semPost",
                                    _u32InstanceId));
                vUnlock();
            }
        };

        tVoid vProcessMsgs() {
            tBool bDoTrace=_oTraceChecker.bDoTraceUsr4();
            if (bDoTrace) {
                ETG_TRACE_USR4(("vProcessMsgs[%u] START",
                                _u32InstanceId));
            }
            DAB_tCallQEntry oEntry;
            static DAB_Profiler rProfiler("vProcessMsgs");

            while (_oPrioQueue.bPop(&oEntry)) {
                vUnlock();// yield
                vLock();

                rProfiler.vStart((tU32)(uintptr_t)oEntry.pMsg);
                DAB_ASSERT_RETURN(OSAL_NULL != oEntry.pMsg);
                DAB_MsgCallerBase  *poCaller =oEntry.poCaller;
                if (OSAL_NULL == poCaller) {
                    if (bDoTrace) {
                        ETG_TRACE_USR4(("vProcessMsgs[%u] notify START",
                                        _u32InstanceId));
                    }
                    vNotify(this, oEntry.pMsg, DAB_enCallMode_ExecPostAll);
                    if (bDoTrace) {
                        ETG_TRACE_USR4(("vProcessMsgs[%u] notify END",
                                        _u32InstanceId));
                    }
                } else {
                    if (bDoTrace) {
                        ETG_TRACE_USR4(("vProcessMsgs[%u] call START",
                                        _u32InstanceId));
                    }
                    vCallMsg(poCaller, oEntry.pMsg, DAB_enCallMode_ExecPost);
                    if (bDoTrace) {
                        ETG_TRACE_USR4(("vProcessMsgs[%u] call END",
                                        _u32InstanceId));
                    }
                }
                delete oEntry.pMsg;            
                rProfiler.u32GetMs();
            }
            if (bDoTrace) {
                ETG_TRACE_USR4(("vProcessMsgs[%u] END",
                                _u32InstanceId));
            }
        };

        tChar *cMakeName(tChar const *cPrefix) {
            OSALUTIL_s32SaveNPrintFormat(_cNameBuf, DAB_DISP_MAX_STRING_LEN, "%s_%d", 
                                         cPrefix,
                                         _u32InstanceId);
            return _cNameBuf;
        }

        tBool _bInitialized;
        DAB_tclPrioQueue<DAB_tCallQEntry> _oPrioQueue;
		DAB_tclTimerMgr *_poTmrMgr;
        DAB_tclEventMgr *_poEvtMgr;
        OSAL_tEventHandle _hEvent;
        OSAL_tThreadID _hThreadId;
		tU8 _u8NumUsrEvents;
        tU32 _u32InstanceId;
        tU32 _u32CallDepth;
		dab_sem *_poCentralSem;
		const tU32 _u32FreeEvts;
        tChar _cNameBuf[DAB_DISP_MAX_STRING_LEN];
		static tU32 _u32NumInstances;
        dab_sem _oTerminateSem;
        static dab_sem _oSubscrQSem;
        vector<DAB_MessageHandlerBase *> _aUsrEventMhs;
    };






    tVoid  DAB_vRegisterEvent(DAB_MessageHandlerBase *pMh, tU8 *pu8EvtId);


    void DAB_vPostEvent(tU8 u8EvtId);


    // #### interface for classes not derived from messageHandlerCommon
    template<class M>
    inline tVoid DAB_vCallMsg(M *pMsg) {
        DAB_tclDispatcher::poGetDefaultDispatcher()->vNotify(DAB_tclDispatcher::poGetDefaultDispatcher(), pMsg);
    }





    template<class D, class M>
    inline tVoid DAB_vSubscribe(tBool bFirst=FALSE) {
        DAB_MsgCallerBase * poCaller = DAB_poGetMsgCaller<D,M>();
        list<DAB_MsgCallerBase *> *poSubscriberList = DAB::DAB_poGetSubscriberList<M>();
        DAB_tclDispatcher::vLockSubscrQ();
        list<DAB_MsgCallerBase *>::iterator iter= find(poSubscriberList->begin(), poSubscriberList->end(), poCaller);
#ifdef DAB_USE_TYPE_INFO
        ETG_TRACE_USR4(("DAB_vSubscribe list=%x D=%20s M=%s",
                        poSubscriberList,
                        typeid(D).name(),
                        typeid(M).name()));
#else
        //    ETG_TRACE_USR4(("DAB_vSubscribe list=%x",
        //                    poSubscriberList));
#endif
        if (iter == poSubscriberList->end()) {
            if (!bFirst) {
                poSubscriberList->push_back(poCaller);
            }
            else {
                poSubscriberList->push_front(poCaller);
            }
        } else {
            (*iter)->_bActive=TRUE;
            //        ETG_TRACE_USR4(("DAB_vSubscribe reactivated"));
        }
        DAB_tclDispatcher::vUnLockSubscrQ();

    }


    template<class D, class M>
    inline void DAB_vUnSubscribe() {
        DAB_MsgCallerBase * poCaller = DAB_poGetMsgCaller<D,M>();
        list<DAB_MsgCallerBase *> *poSubscriberList = DAB::DAB_poGetSubscriberList<M>();
        DAB_tclDispatcher::vLockSubscrQ();

        list<DAB_MsgCallerBase *>::iterator iter= find(poSubscriberList->begin(), poSubscriberList->end(), poCaller);
        if (iter != poSubscriberList->end()) {
            (*iter)->_bActive=FALSE;
        }
        DAB_tclDispatcher::vUnLockSubscrQ();
            
    };

    tBool DAB_bDoDispTraceUsr1();
    tBool DAB_bDoDispTraceUsr4();
    tVoid DAB_vTracePreCall(DAB_MessageHandlerBase const *pMh, DAB_Message const *pMsg, DAB_tenCallMode enCallMode);
    tVoid DAB_vTracePostCall(DAB_MessageHandlerBase const *pMh);

    extern  DAB_tclDispatcher *DAB_poDefaultDispatcher;


}

#endif
