/*******************************************************************************
* FILE:          fc_dabtuner_fiConvert.cpp
* PROJECT:        g3g
* SW-COMPONENT:   DABTUNER
*-------------------------------------------------------------------------------
*
* DESCRIPTION:    Implementation of fc_dabtuner_fiConvert
* 
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
* HISTORY:
* Date      | Author              | Modification

*
*******************************************************************************/

#include "fc_dabtuner_util.h"

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#define FC_DABTUNER_S_IMPORT_INTERFACE_MSG
#include "fc_dabtuner_if.h"

#include "fc_dabtuner_fiConvert.h"

#include "fc_dabtuner_stringUtil.hpp"
#include "dabmeca_db.hpp"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_SERVICE 
#include "trcGenProj/Header/fc_dabtuner_fiConvert.cpp.trc.h"
#endif
#include "dbus/dbus.h"
using namespace DAB;

// conversions for HMI Testmode
static tCString pcGetFrequencyTable(tU8 u8Index)
{
    static char scFrequencyTable[9];
    switch(u8Index)
    {
    case 0x11: (tVoid)OSAL_szStringCopy(scFrequencyTable,"EUROPE"); break;
    case 0x12: (tVoid)OSAL_szStringCopy(scFrequencyTable, "CANADA"); break;
    case 0x13: (tVoid)OSAL_szStringCopy(scFrequencyTable, "EU_NEW"); break;
    case 0x14: (tVoid)OSAL_szStringCopy(scFrequencyTable, "EU_B3"); break;
    case 0x15: (tVoid)OSAL_szStringCopy(scFrequencyTable, "EU_B3NEW"); break;
    case 0x16: (tVoid)OSAL_szStringCopy(scFrequencyTable, "EU_LB"); break;
    case 0x17: (tVoid)OSAL_szStringCopy(scFrequencyTable, "EU_KOREA"); break;
    case 0x18: (tVoid)OSAL_szStringCopy(scFrequencyTable, "EU_CHINA"); break;
    case 0x19: (tVoid)OSAL_szStringCopy(scFrequencyTable, "EU_C_B3"); break;
    case 0x1a: (tVoid)OSAL_szStringCopy(scFrequencyTable, "EU_C_LB"); break;
	case 0x1b: (tVoid)OSAL_szStringCopy(scFrequencyTable, "NZEALND"); break;
    default:
        (tVoid)OSAL_szStringCopy(scFrequencyTable, "NOT USED");
    }
    return scFrequencyTable;
};

static tCString pcGetAudioMode(tU8 u8Index)
{
    static char scAudioMode[4];
    switch(u8Index)
    {
    case 0x00: (tVoid)OSAL_szStringCopy(scAudioMode,"ST"); break;
    case 0x01: (tVoid)OSAL_szStringCopy(scAudioMode, "JS"); break;
    case 0x02: (tVoid)OSAL_szStringCopy(scAudioMode, "DC"); break;
    case 0x03: (tVoid)OSAL_szStringCopy(scAudioMode, "SC"); break;
    default:
        sprintf(scAudioMode, "%d", u8Index); 
    }
    return scAudioMode;
};

static tCString pcGetAudioCodec(tU8 u8Index)
{
    static char scAudioCodec[5];
    switch(u8Index)
    {
    case 0x01: (tVoid)OSAL_szStringCopy(scAudioCodec,"DAB"); break;
    case 0x11: (tVoid)OSAL_szStringCopy(scAudioCodec, "DAB+"); break;
    case 0x21: (tVoid)OSAL_szStringCopy(scAudioCodec, "DMBA"); break;
    default:
        sprintf(scAudioCodec, "%d", u8Index); 
    }
    return scAudioCodec;
};

static tU8 u8GetAudioQualityIndicatorMapping(tU8 u8Index)
{
    tU8 u8RetVal = 0;
    switch(u8Index)
    {
    case 0x01: u8RetVal =  5; break;
    case 0x02: u8RetVal = 15; break;
    case 0x03: u8RetVal = 20; break;
    case 0x04: u8RetVal = 25; break;
    case 0x05: u8RetVal = 30; break;
    case 0x06: u8RetVal = 35; break;
    case 0x07: u8RetVal = 40; break;
    case 0x08: u8RetVal = 50; break;
    case 0x09: u8RetVal = 60; break;
    case 0x0a: u8RetVal = 70; break;
    case 0x0b: u8RetVal = 80; break;
    case 0x0c: u8RetVal = 90; break;
    case 0x0d: u8RetVal = 99; break;
    default:
        ;
    }
    return u8RetVal;
};

static tU8 u8GetSignalQualityMapping(tU8 u8Index)
{
    tU8 u8RetVal = 0;
    switch(u8Index)
    {
    case 0x01: u8RetVal =  1; break;
    case 0x02: u8RetVal = 20; break;
    case 0x03: u8RetVal = 40; break;
    case 0x04: u8RetVal = 60; break;
    case 0x05: u8RetVal = 80; break;
    case 0x06: u8RetVal = 99; break;
    default:
        ;
    }
    return u8RetVal;
};

midw_ext_fi_tcl_e8_DAB_Response 
DAB_tclFiConvert::vConvert2Fi(DAB_tenResult enDabRes) {
    midw_ext_fi_tcl_e8_DAB_Response enFiRes;
    enFiRes.enType=(midw_ext_fi_tcl_e8_DAB_Response::tenType)enDabRes;
    return enFiRes;
}


midw_ext_dabtunerfi_tclMsgFID_DAB_G_TUNER_STATUSStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trTunerStatusProperty const &rTunerStatus) {
    midw_ext_fi_tcl_DAB_TunerStatus oFiMember;
    oFiMember.FreqTuned.u32Frequency = rTunerStatus.u32Frequency;
    // todo check string-conversion
	// Validating UTF-8 characters. PSARCCB30-1911
    if(dbus_validate_utf8(rTunerStatus.sFreqLabel,0)){
    	oFiMember.FreqTuned.sFreqLabel.bSet(rTunerStatus.sFreqLabel,midw_ext_fi_tclString::FI_EN_UTF8);
    }else{
    	oFiMember.FreqTuned.sFreqLabel.bSet("",midw_ext_fi_tclString::FI_EN_UTF8);
    }
    oFiMember.b16Flags.u16Value = rTunerStatus.u16TunerStatusFlags;
    oFiMember.sFrequencyTable.bSet(pcGetFrequencyTable((tU8)rTunerStatus.enFrequencyTable),midw_ext_fi_tclString::FI_EN_UTF8);
    oFiMember.u8SignalQuality = u8GetSignalQualityMapping((tU8)rTunerStatus.u8SignalQuality);
    oFiMember.u8AudioQuality = u8GetAudioQualityIndicatorMapping((tU8)rTunerStatus.u8AudioQuality);
    oFiMember.u32CurrentEnsembleId = rTunerStatus.u32CurrentEnsembleId;
    oFiMember.u32CurrentServiceId = rTunerStatus.u32CurrentServiceId;
    oFiMember.u16CurrentSCIDI = rTunerStatus.u16CurrentSCIDI;
	oFiMember.e8TunerActivityType.enType = (midw_ext_fi_tcl_e8_DAB_TunerActivityType::tenType)rTunerStatus.enTunerActivityType;
    midw_ext_dabtunerfi_tclMsgFID_DAB_G_TUNER_STATUSStatus oFiRes;
    oFiRes.TunerStatus=oFiMember;
    ETG_TRACE_USR4(("DAB_tclFiConvert::vTunerStatus2Fi u32Frequency=%d FreqTableId=%d sFreqTable=%9s "
                    "sFreqLabel=%4s u16TunerStatusFlags=0x%x ",
                    oFiMember.FreqTuned.u32Frequency,
                    rTunerStatus.enFrequencyTable,
                    oFiMember.sFrequencyTable.szGet(midw_ext_fi_tclString::FI_EN_UTF8),
                    oFiMember.FreqTuned.sFreqLabel.szGet(midw_ext_fi_tclString::FI_EN_UTF8),
                    oFiMember.b16Flags.u16Value));
    ETG_TRACE_USR4(("DAB_tclFiConvert::vTunerStatus2Fi cont."
                    "u8SignalQuality=%d u8AudioQuality=%d u32CurrentEnsembleId=0x%x u32CurrentServiceId=0x%x u16CurrentSCIDI=0x%x e8TunerActivityType=%d",
                    oFiMember.u8SignalQuality,
                    oFiMember.u8AudioQuality,
                    oFiMember.u32CurrentEnsembleId,
                    oFiMember.u32CurrentServiceId,
                    oFiMember.u16CurrentSCIDI,
					oFiMember.e8TunerActivityType.enType));
    ETG_TRACE_USR4(("DAB_tclFiConvert::vTunerStatus2Fi cont."
                    "Flags: sync=%d ensValid=%d srvValid=%d compValid=%d mute=%d tuning=%d learnRunning=%d",
                    oFiMember.b16Flags.bDAB_TUNERSTATUSFLAGS_SYNC(),
                    oFiMember.b16Flags.bDAB_TUNERSTATUSFLAGS_ENS_VALID(),
                    oFiMember.b16Flags.bDAB_TUNERSTATUSFLAGS_AUDIO_SERVICE_VALID(),
                    oFiMember.b16Flags.bDAB_TUNERSTATUSFLAGS_SERVICE_COMP_VALID(),
                    oFiMember.b16Flags.bDAB_TUNERSTATUSFLAGS_MUTE(),
                    oFiMember.b16Flags.bDAB_TUNERSTATUSFLAGS_TUNING(),
                    oFiMember.b16Flags.bDAB_TUNERSTATUSFLAGS_LEARN()));

    return oFiRes;
}

midw_ext_dabtunerfi_tclMsgFID_DAB_G_TESTMODE_DATAStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trTestModeDataProperty const &rTestModeData) {
    midw_ext_fi_tcl_DAB_TestModeData oFiMember;
    oFiMember.u32CurrentServID=rTestModeData.u32CurrentServId;
    oFiMember.u32MSCBER=rTestModeData.u32MSCBER;
    oFiMember.u32FICBER=rTestModeData.u32FICBER;
	oFiMember.u32NETBER=rTestModeData.u32NETBER;
    oFiMember.u8RSFEC=rTestModeData.u8RSFEC;
    oFiMember.u8DABMode=rTestModeData.u8DabMode;
    oFiMember.u8ProtectionLevel=rTestModeData.u8ProtectionLevel;
    oFiMember.u8ConcealmentLevel=rTestModeData.u8ConcealmentLevel;
    oFiMember.e8ServiceLinkingMode.enType=(midw_ext_fi_tcl_e8_DAB_ServiceLinkingMode::tenType)rTestModeData.e8ServiceLinkingMode;
    oFiMember.u16AltFMPI=rTestModeData.u16AltFMPI;
    oFiMember.u32AltFMFrequency=rTestModeData.u32AltFMFrequency;
    oFiMember.u8AltFMQuality=rTestModeData.u8AltFMQuality;
    oFiMember.b8Flags.u8Value=rTestModeData.u8Flags;
    oFiMember.s8FieldStrength=(tS8)rTestModeData.u32FieldStrength; // todo, check format u8->u32
    oFiMember.u8ECC = rTestModeData.u8ECC;    //Added by jab4kor
	oFiMember.u16TotalNumberLinks = rTestModeData.u16TotalNumberLinks;// Added for Link info by KVA4KOR
 	oFiMember.bSync = rTestModeData.bSync;
	ETG_TRACE_USR4(("DAB_tclFiConvert::vTestModeData2Fi u16TotalNumberLinks=%d u8RSFEC=%d",
                    oFiMember.u16TotalNumberLinks, oFiMember.u8RSFEC));
	for (tU16 u16Index=0;u16Index<oFiMember.u16TotalNumberLinks;u16Index++) {
        trLinkListElem const &rLinkListElem =rTestModeData.vecLinkList[u16Index];
        midw_ext_fi_tcl_DAB_LinkListElem oFiLinkListElem;
		oFiLinkListElem.e8LinkType.enType=(midw_ext_fi_tcl_e8_DAB_LinkType::tenType)rLinkListElem.enLinkType;
		oFiLinkListElem.u32ID=rLinkListElem.u32Id;
		oFiLinkListElem.u32EnsembleId=rLinkListElem.u32EnsembleId;
		oFiLinkListElem.u8Quality=rLinkListElem.u8Quality;
        ETG_TRACE_USR4(("   DAB_tclFiConvert::trMeca_RDbServiceGetLinkInfo Element %d "
                        "e8LinkType=%d u32ID=0x%x u32EnsembleId=%d u8Quality=%d",
                    u16Index,
                    rLinkListElem.enLinkType,
                    oFiLinkListElem.u32ID,
                    oFiLinkListElem.u32EnsembleId,
                    oFiLinkListElem.u8Quality ));
		oFiMember.LinkList.push_back(oFiLinkListElem);
    }

	for (tU16 u16Index=0;u16Index<rTestModeData.vecAltFreqInfo.size();u16Index++) {
		trAltFreqInfoList const &rAltFreqElem =rTestModeData.vecAltFreqInfo[u16Index];
	        midw_ext_fi_tcl_DAB_AltFreqInfo oFiAltFreqElem;
	        oFiAltFreqElem.u32Frequency=rAltFreqElem.u32Freq;
	        oFiAltFreqElem.s8FieldStrength=(tU8)rAltFreqElem.s32FieldStrength;
	        oFiAltFreqElem.u8Quality=rAltFreqElem.u8Quality;
	        ETG_TRACE_USR4(("   DAB_tclFiConvert::trAltFreqInfoList Element %d "
	                        "u32Freq=%d s32FieldStrength=%d u8Quality=%d",
	                    u16Index,
	                    oFiAltFreqElem.u32Frequency,
	                    oFiAltFreqElem.s8FieldStrength,
	                    oFiAltFreqElem.u8Quality ));
			oFiMember.DAB_DMB_AltFreqInfo.push_back(oFiAltFreqElem);
	    }

    midw_ext_dabtunerfi_tclMsgFID_DAB_G_TESTMODE_DATAStatus oFiRes;
    oFiRes.TestModeData=oFiMember;
    ETG_TRACE_USR4(("DAB_tclFiConvert::vTestModeData2Fi u32CurrentServId=0x%x u32MSCBER=0x%x u32FICBER=0x%x "
                    "u8DABMode=0x%x u8ProtectionLevel=%d u8ConcealmentLevel=%d e8ServiceLinkingMode=0x%x"
                    "u16AltFMPI=0x%x u32AltFMFrequency=%d u8ECC = %x" ,
                    oFiMember.u32CurrentServID,
                    oFiMember.u32MSCBER,
                    oFiMember.u32FICBER,
                    oFiMember.u8DABMode,
                    oFiMember.u8ProtectionLevel,
                    oFiMember.u8ConcealmentLevel,
                    oFiMember.e8ServiceLinkingMode.enType,
                    oFiMember.u16AltFMPI,
                    oFiMember.u32AltFMFrequency,
                    oFiMember.u8ECC));
    ETG_TRACE_USR4(("DAB_tclFiConvert::vTestModeData2Fi u8AltFMQuality=%d b8Flags=0x%x s8FieldStrength=%d bSync=%d",                    
                    oFiMember.u8AltFMQuality,
                    oFiMember.b8Flags.u8Value,
                    oFiMember.s8FieldStrength,
					oFiMember.bSync));
    return oFiRes;
}


// TODO !!!!!!!!!!!!!!!
midw_ext_dabtunerfi_tclMsgFID_DAB_G_BG_TESTMODE_DATAStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trBgTestModeDataProperty const &rBgTestModeData) {
    midw_ext_fi_tcl_DAB_BG_TestModeData oFiMember;
    oFiMember.u32BgFrequency=rBgTestModeData.u32BgFrequency;
    oFiMember.u32BgCurrentEnsID=rBgTestModeData.u32BgCurrentEnsID;
    oFiMember.u32BgFICBER=rBgTestModeData.u32BgFICBER;
    oFiMember.b16BgFlags.u16Value=rBgTestModeData.u16BgFlags;
    oFiMember.sBgDabMode.bSet(rBgTestModeData.sBgDabMode,midw_ext_fi_tclString::FI_EN_UTF8);
    
    midw_ext_dabtunerfi_tclMsgFID_DAB_G_BG_TESTMODE_DATAStatus oFiRes;
    oFiRes.BG_TestModeData=oFiMember;
    ETG_TRACE_USR4(("DAB_tclFiConvert::vBgTestModeData2Fi u32BgFrequency=%d u32BgCurrentEnsID=0x%x u32BgFICBER=%d b16BgFlags=0x%x sBgDabMode=%s",
                    oFiMember.u32BgFrequency,
                    oFiMember.u32BgCurrentEnsID,
                    oFiMember.u32BgFICBER,
                    oFiMember.b16BgFlags.u16Value,
                    oFiMember.sBgDabMode.szValue));
    return oFiRes;
}

midw_ext_dabtunerfi_tclMsgFID_DAB_G_ACTIVE_COMPONENT_INFOStatus
DAB_tclFiConvert::vConvert2Fi(DAB_trComponentInfoProperty const &rComponentInfo) {
    midw_ext_dabtunerfi_tclMsgFID_DAB_G_ACTIVE_COMPONENT_INFOStatus oFiRes;
    vLabel2Fi(rComponentInfo.rComponentLabel,oFiRes.ComponentInfo.ComponentLabel);
    oFiRes.ComponentInfo.e8ComponentType.enType = (midw_ext_fi_tcl_e8_DAB_ComponentType::tenType)rComponentInfo.enComponentType;
    oFiRes.ComponentInfo.e8ServiceType.enType = (midw_ext_fi_tcl_e8_DAB_ServiceType::tenType)rComponentInfo.enServiceType;
    oFiRes.ComponentInfo.e8TMIdType.enType = (midw_ext_fi_tcl_e8_DAB_TMIdType::tenType)rComponentInfo.enTMIIdType;
    oFiRes.ComponentInfo.u16SCIDI = rComponentInfo.u16Scidi;
	oFiRes.ComponentInfo.u8SCIDS = rComponentInfo.u8Scids;
    oFiRes.ComponentInfo.u8AudioDataServiceComponentType = rComponentInfo.u8AudioDataServiceComponentType;
    oFiRes.ComponentInfo.u8ComponentId = rComponentInfo.u8ComponentId;
    oFiRes.ComponentInfo.u8Subchannel = rComponentInfo.u8Subchannel;
	// Passing Component Index -- Chait
    oFiRes.ComponentInfo.u16ComponentIndex = rComponentInfo.u16ComponentIndex;

    ETG_TRACE_USR4(("DAB_tclFiConvert::DAB_trComponentInfoProperty "
                    "u8CompId=0x%x label=%17s "
                    "enComponentType=%d enServiceType=%d enTMIIdType=%d u16SCIDI=%x "
                    "u8AudioDataServiceComponentType=%d u8Subchannel=%d",
                    rComponentInfo.u8ComponentId,
                    rComponentInfo.rComponentLabel.pcGetCString(),
                    ETG_CENUM(DAB::DAB_tenComponentType ,rComponentInfo.enComponentType),
                    ETG_CENUM(DAB::DAB_tenServiceType ,rComponentInfo.enServiceType),
                    ETG_CENUM(DAB::DAB_tenTMIdType ,rComponentInfo.enTMIIdType),
                    rComponentInfo.u16Scidi,
                    rComponentInfo.u8AudioDataServiceComponentType,
                    rComponentInfo.u8Subchannel));
	ETG_TRACE_USR4(("DAB_tclFiConvert::DAB_trComponentInfoProperty "
                        "u16ComponentIndex=%d",
                        rComponentInfo.u16ComponentIndex));

    return oFiRes;
}


midw_ext_dabtunerfi_tclMsgFID_DAB_G_ACTIVE_SERVICE_INFOStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trServiceInfoProperty const &rChnInfo) {

    midw_ext_dabtunerfi_tclMsgFID_DAB_G_ACTIVE_SERVICE_INFOStatus oFiRes;
    midw_ext_fi_tcl_DAB_ServiceInfo &oFiSrvInfo=oFiRes.ServiceInfo;
    oFiSrvInfo.u32ServId=rChnInfo.rProgService.u32GetSID();
    vLabel2Fi(rChnInfo.rLabel,oFiSrvInfo.ServiceLabel);
    oFiSrvInfo.e8ServiceType.enType=midw_ext_fi_tcl_e8_DAB_ServiceType::FI_EN_DAB_SERVICETYPE_AUDIO; // do we show channel-info for data-services
    oFiSrvInfo.u8NumberOfAudioComponents=rChnInfo.u8NumberOfAudioComponents;

    oFiSrvInfo.u8NumberOfDataComponents=0;
    if (rChnInfo.u8NumberOfComponents>rChnInfo.u8NumberOfAudioComponents) {
        oFiSrvInfo.u8NumberOfDataComponents=(tU8)(rChnInfo.u8NumberOfComponents-rChnInfo.u8NumberOfAudioComponents);
    }

    oFiSrvInfo.b16AvailableAnnoTypesMask.u16Value=rChnInfo.u16AvailableAnnoTypesMask;

    oFiSrvInfo.u8SampleRate = (tU8)rChnInfo.enSampleRate;
    oFiSrvInfo.u16BitRate=rChnInfo.u16BitRateKbps;
	oFiSrvInfo.e8EncodingType.enType=(midw_ext_fi_tcl_e8_DAB_Encoding::tenType)rChnInfo.enAudioCoding;
    oFiSrvInfo.sAudioMode.bSet(pcGetAudioMode((tU8)rChnInfo.enAudioMode),midw_ext_fi_tclString::FI_EN_UTF8);
    oFiSrvInfo.sAudioCodec.bSet(pcGetAudioCodec((tU8)rChnInfo.enAudioCodec),midw_ext_fi_tclString::FI_EN_UTF8);
    //oFiSrvInfo.b8Flags.vSetDAB_SERVICEINFO_DRC(rChnInfo.u8ServiceInfoFlags);
    oFiSrvInfo.b8Flags.u8Value=rChnInfo.u8ServiceInfoFlags;
    oFiSrvInfo.u8PresetListElementID=rChnInfo.u8PresetNumber;
	// Passing Service Index and PTY Code -- Chait
    oFiSrvInfo.u16ServiceIndex=rChnInfo.u16ServiceIndex;
	oFiSrvInfo.u8PTYCode=rChnInfo.u8PtyCode;

	if(rChnInfo.bPSFlag){
		oFiSrvInfo.sAdvancedAudioCodec = "PS";
	}
	else if(rChnInfo.bSBRFlag){
    		oFiSrvInfo.sAdvancedAudioCodec = "SBR";
    }
	else{
		oFiSrvInfo.sAdvancedAudioCodec = "";
	}
 // todo: is this ok (receptionQual==AudioQual)
    // todo:u8PTY?, e8SampleRate?, b8Flags

   ETG_TRACE_USR4(("DAB_tclFiConvert::DAB_trServiceInfoProperty->ACTIVE_SERVICE_INFO "
                   "u32ServId=0x%x label=%17s "
                   "u8NumberOfAudioComponents=%d u8NumberOfDataComponents=%d u16AvailableAnnoTypesMask=0x%04x " 
                   "u16BitRateKbps=%d enAudioMode=%d sAudioMode=%5s",
                   oFiSrvInfo.u32ServId,
                   oFiSrvInfo.ServiceLabel.sLabel.szValue,
                   oFiSrvInfo.u8NumberOfAudioComponents,
                   oFiSrvInfo.u8NumberOfDataComponents,
                   oFiSrvInfo.b16AvailableAnnoTypesMask.u16Value,
                   oFiSrvInfo.u16BitRate,
                   ETG_CENUM(tenMeca_RdmAudioMode, rChnInfo.enAudioMode),
                   oFiSrvInfo.sAudioMode.szValue));
   ETG_TRACE_USR4(("u8PresetListElementID=%d  sAudioCodec=%s ",
                   oFiSrvInfo.u8PresetListElementID,
                   oFiSrvInfo.sAudioCodec.szValue));
	ETG_TRACE_USR4(("DAB_tclFiConvert::DAB_trServiceInfoProperty "
                           "u16ServiceIndex=%d u8PTYCode=%d  b8Flags=%d sAdvancedAudioCodec=%s",
                           oFiSrvInfo.u16ServiceIndex, oFiSrvInfo.u8PTYCode,
						   oFiSrvInfo.b8Flags.u8Value,
                           oFiSrvInfo.sAdvancedAudioCodec.szValue));
   return oFiRes;
}


midw_ext_dabtunerfi_tclMsgFID_DAB_G_ACTIVE_ENSEMBLE_INFOStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trEnsembleInfoProperty const &rEnsembleInfo) {
    midw_ext_dabtunerfi_tclMsgFID_DAB_G_ACTIVE_ENSEMBLE_INFOStatus oFiRes;

    vLabel2Fi(rEnsembleInfo.rEnsembleLabel,oFiRes.EnsembleInfo.EnsembleLabel);
    oFiRes.EnsembleInfo.u32EnsembleId = rEnsembleInfo.u32EnsembleId;
    oFiRes.EnsembleInfo.u32Frequency = rEnsembleInfo.u32Frequency;
    oFiRes.EnsembleInfo.u8NumberOfAudioServiceComponents = rEnsembleInfo.u8NumberOfAudioServiceComponents;
    oFiRes.EnsembleInfo.u8NumberOfAudioServices = rEnsembleInfo.u8NumberOfAudioServices;
    oFiRes.EnsembleInfo.u8NumberOfDataServiceComponents = rEnsembleInfo.u8NumberOfDataServiceComponents;
    oFiRes.EnsembleInfo.u8NumberOfDataServices = rEnsembleInfo.u8NumberOfDataServices;
    oFiRes.EnsembleInfo.u8ReceptionQuality = rEnsembleInfo.u8ReceptionQuality;
    oFiRes.EnsembleInfo.u8NumberOfEnsembles = rEnsembleInfo.u8NumberOfEnsemble;
	oFiRes.EnsembleInfo.bReception = rEnsembleInfo.bReception;

    ETG_TRACE_USR4(("DAB_tclFiConvert::DAB_trEnsembleInfoProperty "
                    "u32EnsembleId=0x%x label=%17s "
                    "u32Frequency=%d u8NumberOfAudioServices=%d u8NumberOfDataServices=%d "
                    "u8NumberOfDataServiceComponents=%d u8NumberOfDataServiceComponents=%d "
                    "u8ReceptionQuality=%d u8NumberOfEnsemble=%d",
                    rEnsembleInfo.u32EnsembleId,
                    rEnsembleInfo.rEnsembleLabel.pcGetCString(),
                    rEnsembleInfo.u32Frequency,
                    rEnsembleInfo.u8NumberOfAudioServices,
                    rEnsembleInfo.u8NumberOfDataServices,
                    rEnsembleInfo.u8NumberOfAudioServiceComponents,
                    rEnsembleInfo.u8NumberOfDataServiceComponents,
                    rEnsembleInfo.u8ReceptionQuality,
                    rEnsembleInfo.u8NumberOfEnsemble));
   return oFiRes;

}

midw_ext_dabtunerfi_tclMsgFID_DAB_G_CHANNEL_INFOStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trChnInfoProperty const &rChnInfo) {
    midw_ext_dabtunerfi_tclMsgFID_DAB_G_CHANNEL_INFOStatus oFiRes;
    midw_ext_fi_tcl_DAB_ChannelInfo &oFiChnInfo=oFiRes.ChannelInfo;

    oFiChnInfo.u32ServId=rChnInfo.rMecaId.u32GetSID();
    oFiChnInfo.enState.enType = (midw_ext_fi_tcl_e8_DAB_ChannelInfoState::tenType)rChnInfo.enChnState;
    oFiChnInfo.bMuted=rChnInfo.bMuted;
    vLabel2Fi(rChnInfo.rLabel,oFiChnInfo.ServiceLabel);
    oFiChnInfo.u8NumberOfAudioComponents=rChnInfo.u8NumberOfAudioComponents;
    oFiChnInfo.b16AvailableAnnoTypesMask.u16Value=rChnInfo.u16AvailableAnnoTypesMask;
    oFiChnInfo.u8PresetListElementID=rChnInfo.u8PresetNumber;
    oFiChnInfo.u16FrozenListElementID=rChnInfo.u16FrozenListId;
    oFiChnInfo.bComponentMode=rChnInfo.bComponentMode;
    oFiChnInfo.bFmServiceFollowActive=rChnInfo.bFmServiceFollowActive;
    oFiChnInfo.u8ScidsValue=rChnInfo.u8Scids;

// SUZUKI - 28364 Disable chaneel TP status when DAB-FM linking is active
    if(rChnInfo.bFmServiceFollowActive == 1)
    {
        oFiChnInfo.b16AvailableAnnoTypesMask.u16Value=0;
    }

	if(	TRUE == rChnInfo.bFmServiceFollowActive	)
	{
		oFiChnInfo.rRadioText.bValid = oFiChnInfo.rRadioText.sRadioTextData.bSet((tCString)rChnInfo.rRadioText.sRadioText, midw_ext_fi_tclString::FI_EN_UTF8);
		oFiChnInfo.rRadioText.u8CharSet=(tU8)rChnInfo.rRadioText.enMeca_CharSet;
		ETG_TRACE_USR4(("DAB_tclFiConvert::vRadioText2Fi %s",oFiChnInfo.rRadioText.sRadioTextData.szValue));
	}
	else
	{
		vRadioText2Fi(rChnInfo.rRadioText, oFiChnInfo.rRadioText);
		oFiChnInfo.u8NumberOfDLPElements = (tU8)rChnInfo.vectorRadioTextPlus.size();


		if(rChnInfo.vectorRadioTextPlus.size() != 0)
		{
			for (tU8 u8Index=0;u8Index<rChnInfo.vectorRadioTextPlus.size();u8Index++) {
				midw_ext_fi_tcl_DAB_RadioTextPlus oFiElement;
				oFiElement.u8ContentType = rChnInfo.vectorRadioTextPlus[u8Index].u8ContentType;
				vRadioText2Fi(rChnInfo.vectorRadioTextPlus[u8Index].sContent,oFiElement.Content);

				ETG_TRACE_USR4(("   DAB_tclFiConvert::DAB_trChnInfoProperty rChnInfo.vectorRadioTextPlus[u8Index].sContent = %s ",
					rChnInfo.vectorRadioTextPlus[u8Index].sContent.sRadioText));
				ETG_TRACE_USR4(("   DAB_tclFiConvert::DAB_trChnInfoProperty Element = %d "
					"u8ContentType=%d",
					u8Index,
					oFiElement.u8ContentType));

				oFiChnInfo.DLPlusList.push_back(oFiElement);
			}
		}
	}
	if(!dbus_validate_utf8(oFiChnInfo.rRadioText.sRadioTextData.szValue,0))
	{
		ETG_TRACE_USR4(("DAB_tclFiConvert::vRadioText2Fi Invalid Radiotext"));
		oFiChnInfo.rRadioText.bValid = FALSE;
		oFiChnInfo.rRadioText.sRadioTextData.bSet("", midw_ext_fi_tclString::FI_EN_UTF8);
	   oFiChnInfo.rRadioText.u8CharSet=0;	
	}
	oFiChnInfo.bAFAvailable =rChnInfo.bAFPresent;
	ETG_TRACE_USR4(("   DAB_tclFiConvert::DAB_trChnInfoProperty RadioTextPlus u8NumberOfDLPElements = %d AFStatus =%d",
		oFiChnInfo.u8NumberOfDLPElements,
		oFiChnInfo.bAFAvailable));


 // todo: is this ok (receptionQual==AudioQual)
    // todo:u8PTY?, e8SampleRate?, b8Flags

   ETG_TRACE_USR4(("DAB_tclFiConvert::DAB_trChnInfoProperty->SERVICE_INFO "
                   "enChnState=%d u32ServId=0x%x label=%20s "
                   "u8NumberOfAudioComponents=%d u16AvailableAnnoTypesMask=0x%04x "
                   "u8PresetListElementID=%d u16FrozenListElementID=%d bComponentMode=%d " 
                   "bFmServiceFollowActive=%d ...",
                   ETG_CENUM(tenChnState, rChnInfo.enChnState),
                   oFiChnInfo.u32ServId,
                   oFiChnInfo.ServiceLabel.sLabel.szValue,
                   oFiChnInfo.u8NumberOfAudioComponents,
                   oFiChnInfo.b16AvailableAnnoTypesMask.u16Value,
                   oFiChnInfo.u8PresetListElementID,
                   oFiChnInfo.u16FrozenListElementID,
                   oFiChnInfo.bComponentMode,
                   oFiChnInfo.bFmServiceFollowActive));
   ETG_TRACE_USR4(("DAB_tclFiConvert::DAB_trChnInfoProperty (cont): bMuted=%d u8ScidsValue=%d",
                   oFiChnInfo.bMuted,oFiChnInfo.u8ScidsValue));
   	oFiChnInfo.sLogoLink.bSet(rChnInfo.sLogoLink.c_str(),midw_ext_fi_tclString::FI_EN_UTF8);
	 ETG_TRACE_USR4(("DAB_tclFiConvert::DAB_trChnInfoProperty (cont): logo link =%s",                 
				   oFiChnInfo.sLogoLink.szValue));

   return oFiRes;
}


//midw_ext_dabtunerfi_tclMsgFID_DAB_G_EXT_SERVICE_FOLLOWING_STATUSStatus 
//DAB_tclFiConvert::vConvert2Fi(DAB_trTsuSwitchProperty const &rTsuSwitch) {
//    midw_ext_dabtunerfi_tclMsgFID_DAB_G_EXT_SERVICE_FOLLOWING_STATUSStatus oFiRes;
//    oFiRes.e8SwitchTuner.enType= (rTsuSwitch.enRecomTunerId==enMeca_TsuTunerId_MAIN_FM_TUNER) ?
//        midw_ext_fi_tcl_e8_DAB_SwitchTuner::FI_EN_DAB_SWITCH_TO_FM : 
//        midw_ext_fi_tcl_e8_DAB_SwitchTuner::FI_EN_DAB_SWITCH_TO_DAB;
//        
//    oFiRes.b8ExtServFollowingFlags.u8Value=0;
//    oFiRes.bSwitchRecommended=(rTsuSwitch.enRecomTunerId != rTsuSwitch.enSelectedTunerId);
//    oFiRes.bUserRequest=rTsuSwitch.bUserRequest;
//    oFiRes.u32PI=(tU16)rTsuSwitch.rRecomService.u32GetSID();
//
//    ETG_TRACE_USR4(("DAB_tclFiConvert::DAB_trTsuSwitchProperty "
//                   "e8SwitchTuner=%d  b8ExtServFollowingFlags=0x%x bSwitchRecommended=%d "
//                   "bUserRequest=0x%x oFiRes.u32PI=%d U32Id=%x",
//                   oFiRes.e8SwitchTuner.enType,
//                   oFiRes.b8ExtServFollowingFlags.u8Value,
//                   oFiRes.bSwitchRecommended,
//                   oFiRes.bUserRequest,
//                   oFiRes.u32PI,
//                   rTsuSwitch.rRecomService._u32Id));
//
//    return oFiRes;
//}


midw_ext_dabtunerfi_tclMsgFID_DAB_G_ANNO_STATUSStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trAnnoStatusProperty const &rAnnoStatus) {
    midw_ext_dabtunerfi_tclMsgFID_DAB_G_ANNO_STATUSStatus oFiRes;
    midw_ext_fi_tcl_DAB_AnnoStatus &oFiAnnoStatus= oFiRes.AnnoStatus;
    oFiAnnoStatus.AnnoConfigTypes.u16Value=rAnnoStatus.u16CfgAnnoTypesMask;
    oFiAnnoStatus.AnnoConfigSource.enType=(midw_ext_fi_tcl_e8_DAB_AnnouncementSource::tenType)rAnnoStatus.enCfgAnnoSource;

    oFiAnnoStatus.AnnoSupportedTypes.u16Value=rAnnoStatus.u16AvailableAnnoTypesMask;
    oFiAnnoStatus.AnnoActiveTypes.u16Value=rAnnoStatus.u16ActiveAnnoTypesMask;
    oFiAnnoStatus.AnnoActiveSource.enType=(midw_ext_fi_tcl_e8_DAB_AnnouncementSource::tenType)rAnnoStatus.enActiveAnnoSource;;
    oFiAnnoStatus.AnnoActiveState.enType=(midw_ext_fi_tcl_e8_DAB_AnnouncementStatus::tenType)rAnnoStatus.enActiveAnnoStatus;
    oFiAnnoStatus.AnnoSelectState.enType=(midw_ext_fi_tcl_e8_DAB_AnnouncementSelectState::tenType)rAnnoStatus.enAnnoSelectState;
    oFiAnnoStatus.AnnoSelectedTypes.u16Value=rAnnoStatus.u16SelectedAnnoTypesMask;
    vLabel2Fi(rAnnoStatus.rAnnoServiceLabel, oFiAnnoStatus.AnnoServiceLabel);
	for(tU8 i = 0; i < rAnnoStatus.vectorAnnoList.size(); i++)
	{	midw_ext_fi_tcl_DAB_AnnoElement oAnnoElem;
		oAnnoElem.u16AnnoType = rAnnoStatus.vectorAnnoList[i].u16AnnoTypesMask;
		oAnnoElem.AnnoState.enType=(midw_ext_fi_tcl_e8_DAB_AnnouncementStatus::tenType)rAnnoStatus.vectorAnnoList[i].enAnnoStatus;
    oFiAnnoStatus.AnnouncementList.push_back(oAnnoElem);
		  ETG_TRACE_USR4(("DAB_tclFiConvert::DAB_trAnnoStatusProperty vector Elem %d"
                   "AnnoTypes=0x%x AnnoState=0x%x",                 
                   i,
                  oAnnoElem.u16AnnoType,
                   oAnnoElem.AnnoState.enType));
    
	}
    ETG_TRACE_USR4(("DAB_tclFiConvert::DAB_trAnnoStatusProperty "
                   "AnnoConfigTypes=0x%x AnnoSupportedTypes=0x%x "
                   "AnnoActiveTypes=0x%x AnnoActiveSource=0x%x "
                   "AnnoActiveState=0x%x AnnoSelectState=0x%x AnnoSelectedTypes=0x%x ListSize=%d",
                   oFiAnnoStatus.AnnoConfigTypes.u16Value,
                   oFiAnnoStatus.AnnoSupportedTypes.u16Value,
                   oFiAnnoStatus.AnnoActiveTypes.u16Value,
                   oFiAnnoStatus.AnnoActiveSource.enType,
                   oFiAnnoStatus.AnnoActiveState.enType,
                   oFiAnnoStatus.AnnoSelectState.enType,
                   oFiAnnoStatus.AnnoSelectedTypes.u16Value,
				   (tU8)oFiAnnoStatus.AnnouncementList.size()));
    return oFiRes;
}

midw_ext_dabtunerfi_tclMsgFID_DAB_G_SETUP_STATUSStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trSetupStatusProperty const &rSetupStatus) {
    midw_ext_dabtunerfi_tclMsgFID_DAB_G_SETUP_STATUSStatus oFiRes;
    oFiRes.b32SetupFlags.u32Value=rSetupStatus.u32SetupFlags;
    oFiRes.e8ServiceLinkingMode.enType=(midw_ext_fi_tcl_e8_DAB_ServiceLinkingMode::tenType)rSetupStatus.e8ServiceLinkingMode;
    oFiRes.u8ConcealmentLevel=rSetupStatus.u8ConcealmentLevel;
    oFiRes.u8DABFMLinkingThreshold=rSetupStatus.u8DABFMLinkingThreshold;
    oFiRes.u8DABFMLinkBackDelay=rSetupStatus.u8DABFMLinkBackDelay;
    oFiRes.u8DABFMLinkingBackThreshold=rSetupStatus.u8DABFMLinkingBackThreshold;
    oFiRes.u8DABFMLinkingBackValidationTime=rSetupStatus.u8DABFMLinkingBackValidationTime;
    oFiRes.u8EnsInsertThreshold=rSetupStatus.u8EnsInsertThreshold;
    oFiRes.u8EnsDeleteThreshold=rSetupStatus.u8EnsDeleteThreshold;


    ETG_TRACE_USR4(("DAB_tclFiConvert::DAB_trSetupStatusProperty "
                   "b32SetupFlags=0x%x u8ConcealmentLevel=%d"
                   "u8DABFMLinkingThreshold=%d u8DABFMLinkBackDelay=%d,"
                   "u8DABFMLinkingBackThreshold=%d u8DABFMLinkingBackValidationTime=%d u8EnsInsertThreshold=%d u8EnsDeleteThreshold=%d",
                   oFiRes.b32SetupFlags.u32Value,
                   oFiRes.u8ConcealmentLevel,
                   oFiRes.u8DABFMLinkingThreshold,
                   oFiRes.u8DABFMLinkBackDelay,
                   oFiRes.u8DABFMLinkingBackThreshold,
                   oFiRes.u8DABFMLinkingBackValidationTime,
                   oFiRes.u8EnsInsertThreshold,
                   oFiRes.u8EnsDeleteThreshold));

    ETG_TRACE_USR4(("DAB_tclFiConvert::DAB_trSetupStatusProperty e8ServiceLinkingMode=%d",
    		ETG_CENUM(DAB_tenServiceLinkingMode ,oFiRes.e8ServiceLinkingMode.enType)));

    return oFiRes;
}


// not more needed in lcn2kai
/*
midw_ext_dabtunerfi_tclMsgFID_DAB_G_MUTE_STATUSStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trMuteStatusProperty const &rMuteStatus) {
    midw_ext_dabtunerfi_tclMsgFID_DAB_G_MUTE_STATUSStatus oFiRes;
    oFiRes.e8MuteStatus.enType = (rMuteStatus.bMuteDab && rMuteStatus.bMuteDabTA) ? 
        midw_ext_fi_tcl_e8_DAB_MuteStatus::FI_EN_DAB_MUTE : 
        midw_ext_fi_tcl_e8_DAB_MuteStatus::FI_EN_DAB_DEMUTE;
    
    ETG_TRACE_USR4(("DAB_tclFiConvert::DAB_trMuteStatusProperty "
                    "e8MuteStatus=0x%x",
                    oFiRes.e8MuteStatus.enType ));
    return oFiRes;
}
*/

midw_ext_dabtunerfi_tclMsgFID_DAB_G_AVAILABILITYStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trAvailabilityProperty const &rAvailability) {
    midw_ext_dabtunerfi_tclMsgFID_DAB_G_AVAILABILITYStatus oFiRes;
    oFiRes.bIsDABAvailable = rAvailability.bIsDABAvailable; 
    oFiRes.e8SourceState.enType=(midw_ext_fi_tcl_e8_DAB_SourceState::tenType)rAvailability.enSourceState;

    ETG_TRACE_USR4(("DAB_tclFiConvert::DAB_trAvailabilityProperty "
                   "bIsDABAvailable=%d e8SourceState=0x%x",
                   oFiRes.bIsDABAvailable,
                   oFiRes.e8SourceState.enType ));

    return oFiRes;
}


tVoid DAB_tclFiConvert::vConvert2Fi(trMsgSrvRspSetChnList const &oMsg,
                                    midw_ext_dabtunerfi_tclMsgFID_DAB_S_SET_SRV_LISTMethodResult &oFiRes) {
    oFiRes.e8MethodResult=DAB_tclFiConvert::vConvert2Fi(oMsg.enRes);
    oFiRes.bOpen=oMsg.bOpen;
    oFiRes.u16InitialActivatedElementId=oMsg.u16ActivatedElemId;
    oFiRes.u16TotalNumberOfElements=oMsg.u16TotalNumberOfElements;
    oFiRes.u16SentNumberOfElements=oMsg.u16NumElemToSend;

    ETG_TRACE_USR4(("DAB_tclFiConvert::trMsgSrvRspSetChnList "
                    "u16InitialActivatedElementId=%d u16TotalNumberOfElements=%d u16SentNumberOfElements=%d",
                    oFiRes.u16InitialActivatedElementId,
                    oFiRes.u16TotalNumberOfElements,
                    oFiRes.u16SentNumberOfElements));

    for (tU16 u16Index=0;u16Index<oMsg.u16NumElemToSend;u16Index++) {
        trChnListElem const &rListElem =oMsg.vcChnList[oMsg.u16FirstIndexToSend+u16Index];
        midw_ext_fi_tcl_DAB_ChnListElement oFiListElem;
        oFiListElem.u16ListElementID=rListElem.u16Id;
        oFiListElem.u32DABId=rListElem.rMecaId.u32GetSID();
        oFiListElem.u8CompScids=(tU8)rListElem.u16Scids;
        vLabel2Fi(rListElem.rLabel, oFiListElem.Label);
		vLabel2Fi(rListElem.rEnsembleLabel, oFiListElem.EnsembleLabel);
        oFiListElem.bReception=rListElem.bReception;
        oFiListElem.u8PresetListElemId=rListElem.u8PresetNumber;
		oFiListElem.u8PTY=rListElem.u8PTY;
		oFiListElem.u8NumberOfComponents=rListElem.u8NumberOfComponents;
		
		if(dbus_validate_utf8(rListElem.sLogoLink.c_str(),0)){
		oFiListElem.sLogoLink.bSet(rListElem.sLogoLink.c_str(),midw_ext_fi_tclString::FI_EN_UTF8);
		}
		else{
			oFiListElem.sLogoLink.bSet("",midw_ext_fi_tclString::FI_EN_UTF8);
		}
		if(dbus_validate_utf8(rListElem.sFreqLabel.c_str(),0)){
		oFiListElem.sFreqLabel.bSet(rListElem.sFreqLabel.c_str(),midw_ext_fi_tclString::FI_EN_UTF8);
		}else{
			oFiListElem.sFreqLabel.bSet("",midw_ext_fi_tclString::FI_EN_UTF8);
		}
		
		oFiListElem.bEPGSupport = rListElem.u8DSSupport;
        oFiRes.ChannelList.push_back(oFiListElem);
		ETG_TRACE_USR4(("   DAB_tclFiConvert::trMsgSrvRspSetChnList Element %d u8PTY=%d"
                        "u16ListElementID=%d u32DABId=0x%x Label=%17s bReception=%d presetNr=%d u8CompScids %d EnsembleLabel=%s ",
                    u16Index,
					oFiListElem.u8PTY,
                    oFiListElem.u16ListElementID,
                    oFiListElem.u32DABId,
                    rListElem.rLabel.pcGetCString(),
                        oFiListElem.bReception,
                        oFiListElem.u8PresetListElemId,
                        oFiListElem.u8CompScids,
						rListElem.rEnsembleLabel.pcGetCString()));
		ETG_TRACE_USR4(("   DAB_tclFiConvert::supported dataservices %d",
						oFiListElem.bEPGSupport));
	     ETG_TRACE_USR4(("   DAB_tclFiConvert::trMsgSrvRspSetChnList LogoLink=%s",
						rListElem.sLogoLink.c_str()));
     
    }

}

tVoid DAB_tclFiConvert::vConvert2Fi(trMsgSrvRspValidateEPGData const &oMsg,
                                    midw_ext_dabtunerfi_tclMsgFID_DAB_S_VALIDATE_EPG_LISTMethodResult &oFiRes) {

	oFiRes.bEPGAvailability = oMsg.bEPGPresent;
	ETG_TRACE_USR4(("DAB_tclFiConvert::bEPGPresent %d",
						oFiRes.bEPGAvailability));
}

tVoid DAB_tclFiConvert::vConvert2Fi(trMsgSrvRspSetEnsmbleList const &oMsg,
                                    midw_ext_dabtunerfi_tclMsgFID_DAB_S_SET_ENS_LISTMethodResult &oFiRes) {
    oFiRes.e8MethodResult=DAB_tclFiConvert::vConvert2Fi(oMsg.enRes);
    oFiRes.bOpen=oMsg.bOpen;
    oFiRes.u16InitialActivatedElementId=oMsg.u16ActivatedElemId;
    oFiRes.u16TotalNumberOfElements=oMsg.u16TotalNumberOfElements;
    oFiRes.u16SentNumberOfElements=oMsg.u16NumElemToSend;

    ETG_TRACE_USR4(("DAB_tclFiConvert::trMsgSrvRspSetEnsmbleList "
                    "u16InitialActivatedElementId=%d u16TotalNumberOfElements=%d u16SentNumberOfElements=%d",
                    oFiRes.u16InitialActivatedElementId,
                    oFiRes.u16TotalNumberOfElements,
                    oFiRes.u16SentNumberOfElements));

    for (tU16 u16Index=0;u16Index<oMsg.u16NumElemToSend;u16Index++) {
        trEnsembleListElem const &rListElem =oMsg.vcEnsmbleList[oMsg.u16FirstIndexToSend+u16Index];
        midw_ext_fi_tcl_DAB_EnsembleInfo oFiListElem;
		oFiListElem.u8ListElementId = (tU8)rListElem.u16Id;
		oFiListElem.u32EnsembleId = rListElem.rEnsemble.u32GetID();
		oFiListElem.u32Frequency = rListElem.u32Frequency;
		vLabel2Fi(rListElem.rEnsembleLabel, oFiListElem.EnsembleLabel);
		oFiListElem.u8NumberOfAudioServices = rListElem.u8NumberOfAudioServices;
		oFiListElem.u8NumberOfDataServices = rListElem.u8NumberOfDataServices;
		oFiListElem.u8ReceptionQuality = rListElem.u8ReceptionQuality;
	    
        oFiRes.EnsembleList.push_back(oFiListElem);
        ETG_TRACE_USR4(("   DAB_tclFiConvert::trMsgSrvRspSetEnsmbleList Element %d "
                        "u16ListElementID=%d u32EnsembleId=0x%x Label=%17s"
						"NoOfAudioServices=%d NoOfDataServices=%d",
						u16Index,
						oFiListElem.u8ListElementId,
						oFiListElem.u32EnsembleId,
						rListElem.rEnsembleLabel.pcGetCString(),
						oFiListElem.u8NumberOfAudioServices,
						oFiListElem.u8NumberOfDataServices ));
    }

}
tVoid DAB_tclFiConvert::vConvert2Fi(trMsgSrvRspSrcState const &oMsg,
		midw_ext_dabtunerfi_tclMsgFID_DAB_S_SOURCE_STATEMethodResult &oFiResult)
{
	oFiResult.e8MethodResult.enType=(midw_ext_fi_tcl_e8_DAB_Response::tenType)oMsg.enRes;
	oFiResult.e8SourceState.enType=(midw_ext_fi_tcl_e8_DAB_SourceState::tenType)oMsg.enSourceState;
	ETG_TRACE_USR4(("DAB_tclFiConvert::trMsgSrvRspSrcState "
	                    "e8SourceState=%d",
	                    oFiResult.e8SourceState.enType));
}

tVoid DAB_tclFiConvert::vConvert2Fi(DAB_trCompListProperty const &oMsg,
                                    midw_ext_dabtunerfi_tclMsgFID_DAB_G_COMP_LISTStatus &oFiRes) {

    oFiRes.bOpen=oMsg.bOpen;
    oFiRes.u16ActivatedElementId=oMsg.bActivatedElemId;
    oFiRes.u16NumberOfComponents=(tU16)oMsg.vectorCompEntries.size();

    ETG_TRACE_USR4(("DAB_tclFiConvert::DAB_trCompListProperty "
                    "u16ActivatedElementId=%d u16NumberOfComponents=%d",
                    oFiRes.u16ActivatedElementId,
                    oFiRes.u16NumberOfComponents));

    for (tU16 u16Index=0;u16Index<oFiRes.u16NumberOfComponents;u16Index++) {
        trChnListElem const &rListElem=oMsg.vectorCompEntries[u16Index];
        midw_ext_fi_tcl_DAB_ChnListElement oFiListElem;
        oFiListElem.u16ListElementID=rListElem.u16Id;
        oFiListElem.u32DABId=rListElem.rMecaId.u32GetSID();
        vLabel2Fi(rListElem.rLabel, oFiListElem.Label);
        oFiListElem.bReception=TRUE;
        oFiRes.ComponentList.push_back(oFiListElem);
        ETG_TRACE_USR4(("   DAB_tclFiConvert::DAB_trCompListProperty Element %d "
                    "u16ListElementID=%d u32DABId=0x%08x Label=%17s bReception=%d",
                    u16Index,
                    oFiListElem.u16ListElementID,
                    oFiListElem.u32DABId,
                    rListElem.rLabel.pcGetCString(),
                    oFiListElem.bReception ));
    }

}


tVoid DAB_tclFiConvert::vConvert2Fi(trMsgSrvRspUpdateChnList const &oMsg, 
                                    midw_ext_dabtunerfi_tclMsgFID_DAB_G_SRV_LIST_UPDATEStatus &oFiRes) {
    oFiRes.bActive=oMsg.bChnListActive;
    oFiRes.bUpdateRunning=oMsg.bUpdateRunning;
    oFiRes.u16ActivatedElementId=oMsg.u16ActivatedElemId;

    ETG_TRACE_USR4(("DAB_tclFiConvert::trMsgSrvRspUpdateChnList "
                    "bChnListActive=%d bUpdateRunning=%d u16ActivatedElemId=%d",
                    oFiRes.bActive,
                    oFiRes.bUpdateRunning,
                    oFiRes.u16ActivatedElementId));

    for (tU16 u16Index=0;u16Index<oMsg.u16NumChanges;u16Index++) {
        midw_ext_fi_tcl_DAB_ListElementAvailInfo oFiAvailInfo;
        trSrvIdAndQuality const &rAvailInfo = oMsg.vcChangedAvailablities[u16Index];
        oFiAvailInfo.u16ListElementID=(tU16)rAvailInfo.u16ListElemId;
        oFiAvailInfo.bAvailable=rAvailInfo.bReception;
        oFiRes.ChangedSrvAvailList.push_back(oFiAvailInfo);

        ETG_TRACE_USR4(("   DAB_tclFiConvert::trMsgSrvRspUpdateChnList Element %d "
                    "u16ListElementID=%d bAvailable=%d",
                    u16Index,
                    oFiAvailInfo.u16ListElementID,
                    oFiAvailInfo.bAvailable));
    }
}

tVoid DAB_tclFiConvert::vConvert2Fi(DAB::trMsgSrvRspGetTIIList const &oMsg,
									midw_ext_dabtunerfi_tclMsgFID_DAB_S_GET_TIILISTMethodResult &oFiRes) {
    oFiRes.e8MethodResult=DAB_tclFiConvert::vConvert2Fi(oMsg.enRes);
	tU16 u16Size = (tU16)oMsg.vecTiiList.size();
    ETG_TRACE_USR4(("DAB_tclFiConvert::trMsgSrvRspGetTIIList u16TotalNumberOfElements=%d",
                   u16Size));
    for (tU16 u16Index=0;u16Index<u16Size;u16Index++) {
        trMecaTiiInfo const &rListElem =oMsg.vecTiiList[u16Index];
        midw_ext_fi_tcl_DAB_TIIListElement oFiListElem;
        oFiListElem.u8MainId=rListElem.u8MainId;
        oFiListElem.u8SubId=rListElem.u8SubId;
		oFiListElem.u32Latitude=rListElem.u32Latitude;
		oFiListElem.u32Longitude=rListElem.u32Longitude;
        oFiRes.TIIList.push_back(oFiListElem);
        ETG_TRACE_USR4(("   DAB_tclFiConvert::trMsgSrvRspGetTIIList u8SubId %d u8MainId=%d"
                        "u32Latitude=%d u32Longitude=0x%x ",
                   oFiListElem.u8MainId,
					oFiListElem.u8SubId,
                    oFiListElem.u32Latitude,
                    oFiListElem.u32Longitude));
    }
}


tVoid DAB_tclFiConvert::vConvert2Fi(trMsgSrvRspExpIdValueU32 const &oMsg, midw_ext_dabtunerfi_tclMsgFID_DAB_S_GET_FIELDSTRENGTHMethodResult &oFiRes){
	oFiRes.e8MethodResult = DAB_tclFiConvert::vConvert2Fi(oMsg.enRes);
	oFiRes.s8FieldStrength = (tU8)oMsg.u32Value;
}

tVoid DAB_tclFiConvert::vLabel2Fi(trMecaLabel const &rLabel, midw_ext_fi_tcl_DAB_Label &oFiLabel) {
    rLabel.vTrace();
    if (rLabel.bIsLabelOk() && dbus_validate_utf8(rLabel.pcGetCString(),0)) {
        oFiLabel.sLabel.bSet(rLabel.pcGetCString(), midw_ext_fi_tclString::FI_EN_UTF8);// todo: check for ISO
        oFiLabel.sLabelShort.bSet(rLabel.pcGetCStringShort(), midw_ext_fi_tclString::FI_EN_UTF8);// todo: check for ISO
     oFiLabel.u16LabelMask=rLabel._u16LabelMask;
    } else {
        oFiLabel.sLabel.bSet("", midw_ext_fi_tclString::FI_EN_UTF8);
        oFiLabel.sLabelShort.bSet("", midw_ext_fi_tclString::FI_EN_UTF8);
  	oFiLabel.u16LabelMask=0;
    }
}

tVoid DAB_tclFiConvert::vRadioText2Fi(trRadioText const &rRadioText, midw_ext_fi_tcl_DAB_RadioText &oFiRadioText) {
    oFiRadioText.bValid=FALSE;
    if (rRadioText.bValid) {
        oFiRadioText.bValid=bConvertRTextTotString( rRadioText.enMeca_CharSet, rRadioText.sRadioText, &oFiRadioText.sRadioTextData, rRadioText.u8RadioTextLen);
		oFiRadioText.u8CharSet=(tU8)rRadioText.enMeca_CharSet;
        ETG_TRACE_USR4(("DAB_tclFiConvert::vRadioText2Fi %s",oFiRadioText.sRadioTextData.szValue));
    }
    if (!rRadioText.bValid) {
        oFiRadioText.sRadioTextData.bSet("", midw_ext_fi_tclString::FI_EN_UTF8);
	   oFiRadioText.u8CharSet=0;
    }

}

tVoid DAB_tclFiConvert::vRdbId2Fi(DAB_trTmcRdbId const &rTmcRdbId, midw_ext_fi_tcl_DAB_RdbId &oFiTmcRdbId) {
    oFiTmcRdbId.bEncrypted = rTmcRdbId.bEncrypted;
    oFiTmcRdbId.u8Cc  = rTmcRdbId.u8Cc;
    oFiTmcRdbId.u8Ltn = rTmcRdbId.u8Ltn;
    oFiTmcRdbId.u8Sid = rTmcRdbId.u8Sid;
    oFiTmcRdbId.u16Bsa = rTmcRdbId.u16Bsa;
    ETG_TRACE_USR4(("DAB_tclFiConvert::vRdbId2Fi "
                        "RDBID =0x00%2x%2x%2x",
                        oFiTmcRdbId.u8Cc,
                        oFiTmcRdbId.u8Ltn,
                        oFiTmcRdbId.u8Sid));
}

tVoid DAB_tclFiConvert::vFi2RdbId(midw_ext_fi_tcl_DAB_RdbId const &oFiTmcRdbId, DAB_trTmcRdbId &rTmcRdbId) {
    rTmcRdbId.bEncrypted = oFiTmcRdbId.bEncrypted;
    rTmcRdbId.u8Cc = oFiTmcRdbId.u8Cc;
    rTmcRdbId.u8Ltn = oFiTmcRdbId.u8Ltn;
    rTmcRdbId.u8Sid = oFiTmcRdbId.u8Sid;
    rTmcRdbId.u16Bsa = oFiTmcRdbId.u16Bsa;
    ETG_TRACE_USR4(("DAB_tclFiConvert::vFi2RdbId "
                        "RDBID =0x00%2x%2x%2x",
                        rTmcRdbId.u8Cc,
                        rTmcRdbId.u8Ltn,
                        rTmcRdbId.u8Sid));
}
tVoid DAB_tclFiConvert::vTmcData2Fi(DAB_trTmcDataListElement const &rTmcDataListElement, midw_ext_fi_tcl_DAB_TmcInfo &oFiTmcInfo) {
    vRdbId2Fi(rTmcDataListElement.RdbId,oFiTmcInfo.RdbId);
    //oFiTmcInfo.RdbId  = rTmcDataListElement.RdbId;
    oFiTmcInfo.s16Ecc = rTmcDataListElement.s16Ecc;
    oFiTmcInfo.u16Aid = rTmcDataListElement.u16Aid;
    for(tU8 i=0; i<5; i++) {
        oFiTmcInfo.TmcData.push_back(rTmcDataListElement.au8TmcData[i]);
    }
    oFiTmcInfo.DabEncryption.bEncrypted = rTmcDataListElement.TmcEncryption.bEncrypted;
    oFiTmcInfo.DabEncryption.u8EncId = rTmcDataListElement.TmcEncryption.u8EncId;
    oFiTmcInfo.DabEncryption.u8TestBits = rTmcDataListElement.TmcEncryption.u8TestBits;
    oFiTmcInfo.DabMsgSource.DabMediumTypeId.enType = (midw_ext_fi_tcl_e8_DAB_TmcMediumType::tenType)rTmcDataListElement.DabMsgSource.DabMediumTypeId;
    oFiTmcInfo.DabMsgSource.u8Number = rTmcDataListElement.DabMsgSource.u8Number;
    oFiTmcInfo.sProgramServiceName = rTmcDataListElement.sProgramServiceName;
    oFiTmcInfo.sServiceProviderName = rTmcDataListElement.sServiceProviderName;
    ETG_TRACE_USR4(("DAB_tclFiConvert::DAB_trTmcRdbId "
                        "RDBID =0x00%2x%2x%2x",
                        oFiTmcInfo.s16Ecc,
                        oFiTmcInfo.RdbId.u8Ltn,
                        oFiTmcInfo.RdbId.u8Sid));
}

tVoid DAB_tclFiConvert::vConvert2Fi(DAB::trMsgSrvRspGetPresetList const &oPresetList, 
                                    midw_ext_dabtunerfi_tclMsgFID_DAB_S_GET_PRESET_LISTMethodResult &oFiRes) {
    oFiRes.u8FirstElementID=oPresetList.u8FirstElementID;
    oFiRes.u8NumberOfElements=oPresetList.u8NumberOfElements;
    oFiRes.e8MethodResult=DAB_tclFiConvert::vConvert2Fi(oPresetList.enRes);

    ETG_TRACE_USR4(("DAB_tclFiConvert::trMsgSrvRspGetPresetList "
                    "u8FirstElementID=%d u8NumberOfElements=%d",
                    oFiRes.u8FirstElementID,
                    oFiRes.u8NumberOfElements));

    for (tU16 u16Index=(tU16)(oPresetList.u8FirstElementID-1);u16Index< (tU16)(oPresetList.u8NumberOfElements);u16Index++) {
        midw_ext_fi_tcl_DAB_PresetListElement oFiElement;
        oFiElement.u8PresetElementID=(tU8)(/*oPresetList.u8FirstElementID+*/u16Index+1);
        oFiElement.u32ServiceID=oPresetList.vecPresetInfos[u16Index].rPSID.u32GetSID();
		oFiElement.u32Frequency=oPresetList.vecPresetInfos[u16Index].u32Frequency;
        vLabel2Fi(oPresetList.vecPresetInfos[u16Index].rLabel, oFiElement.ServiceLabel);
		
        if(dbus_validate_utf8(oPresetList.vecPresetInfos[u16Index].sLogoLink.c_str(),0)){
		oFiElement.sLogoLink.bSet(oPresetList.vecPresetInfos[u16Index].sLogoLink.c_str(),midw_ext_fi_tclString::FI_EN_UTF8);
        }else{
        	oFiElement.sLogoLink.bSet("",midw_ext_fi_tclString::FI_EN_UTF8);
        }
		ETG_TRACE_USR4(("   DAB_tclFiConvert::trMsgSrvRspGetPresetList Logo link %s",
                    oFiElement.sLogoLink.szValue));
	
        oFiRes.PresetList.push_back(oFiElement);
        ETG_TRACE_USR4(("   DAB_tclFiConvert::trMsgSrvRspGetPresetList Element %d "
                    "u8PresetElementID=%d Label=%17s",
                    u16Index,
                    oFiElement.u8PresetElementID,
                    oPresetList.vecPresetInfos[u16Index].rLabel.pcGetCString()));
    }
}
midw_ext_dabtunerfi_tclMsgFID_DAB_G_PRESET_LISTStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trPresetListProperty const &rPresetList){

	midw_ext_dabtunerfi_tclMsgFID_DAB_G_PRESET_LISTStatus oFiRes;

	oFiRes.u8FirstElementID=1;
    oFiRes.u8NumberOfElements=rPresetList.u8NumberOfElements;

	ETG_TRACE_USR4(("DAB_tclFiConvert::DAB_trPresetListProperty "
                    "u8NumberOfElements=%d",                  
                    (tU16)rPresetList.vecPresetListInfo.size()));

    for (tU16 u16Index=0;u16Index<rPresetList.u8NumberOfElements;u16Index++) {
        midw_ext_fi_tcl_DAB_PresetListElement oFiElement;
        oFiElement.u8PresetElementID=(tU8)(u16Index+1);
        oFiElement.u32ServiceID=rPresetList.vecPresetListInfo[u16Index].rPSID.u32GetSID();
		oFiElement.u32Frequency=rPresetList.vecPresetListInfo[u16Index].u32Frequency;
        vLabel2Fi(rPresetList.vecPresetListInfo[u16Index].rLabel, oFiElement.ServiceLabel);
		
        if(dbus_validate_utf8(rPresetList.vecPresetListInfo[u16Index].sLogoLink.c_str(),0))
		oFiElement.sLogoLink.bSet(rPresetList.vecPresetListInfo[u16Index].sLogoLink.c_str(),midw_ext_fi_tclString::FI_EN_UTF8);
        else{
        	oFiElement.sLogoLink.bSet("",midw_ext_fi_tclString::FI_EN_UTF8);
        }
		ETG_TRACE_USR4(("   DAB_tclFiConvert::DAB_trPresetListProperty Logo link %s",
                    oFiElement.sLogoLink.szValue));
		
        oFiRes.PresetList.push_back(oFiElement);
        ETG_TRACE_USR4(("   DAB_tclFiConvert::DAB_trPresetListProperty Element %d "
                    "u8PresetElementID=%d Label=%17s",
                    u16Index,
                    oFiElement.u8PresetElementID,
                    rPresetList.vecPresetListInfo[u16Index].rLabel.pcGetCString()));
    }

	return oFiRes;
}

tVoid DAB_tclFiConvert::vConvert2Fi(DAB::trMsgSrvRspRdbidWishlist const &oRdbidWishlist, 
                                    midw_ext_dabtunerfi_tclMsgFID_DAB_S_RDBID_WISH_LISTMethodResult &oFiRes) {

    ETG_TRACE_USR4(("   DAB_tclFiConvert::trMsgSrvRspRdbidWishlist size %d", (tU16)oRdbidWishlist.vecRdbIdList.size()));

    if (oRdbidWishlist.vecRdbIdList.size() != 0) {
        for(tU32 i = 0; i < oRdbidWishlist.vecRdbIdList.size(); i++)
        {
            DAB_trTmcRdbId const &rTmcRdbId=oRdbidWishlist.vecRdbIdList[i];
            midw_ext_fi_tcl_DAB_RdbId oTmcRdbId;
            vRdbId2Fi(rTmcRdbId,oTmcRdbId);
            oFiRes.RdbIDList.push_back(oTmcRdbId);
        }
    }
}

tVoid DAB_tclFiConvert::vConvert2Fi(DAB::trMsgDrvRspGetExpApi const &oGetExpApi, 
                                    midw_ext_dabtunerfi_tclMsgFID_DAB_S_GET_EXP_API_INFOMethodResult &oFiMsg) {
    ETG_TRACE_USR4(("   DAB_tclFiConvert::trMsgDrvRspGetExpApi sExpLabel.Description=%s", oGetExpApi.pcDescription));
    ETG_TRACE_USR4(("   DAB_tclFiConvert::trMsgDrvRspGetExpApi sExpLabel.Value=%s", oGetExpApi.pcValueLabel));

    oFiMsg.u16ExpApiId = oGetExpApi.u16ExpApiId;
    oFiMsg.sLabel.bSet(oGetExpApi.pcDescription,midw_ext_fi_tclString::FI_EN_UTF8);
    oFiMsg.sValue.bSet(oGetExpApi.pcValueLabel,midw_ext_fi_tclString::FI_EN_UTF8);
}

tenDefsetType DAB_tclFiConvert::getFromFi(midw_ext_fi_tcl_e8_DefSetMode const &oDefsetMode) {
        //lint -e{788} enum constant '...' not used within defaulted switch
    switch (oDefsetMode.enType) {
        case midw_ext_fi_tcl_e8_DefSetMode::FI_EN_DEFSET_SRC_SET_TEF: 
            return enDefsetType_TEF;
        case midw_ext_fi_tcl_e8_DefSetMode::FI_EN_DEFSET_SRC_SET_USER: 
            return enDefsetType_USER;
        default: 
            return enDefsetType_INVALID;
    }
}

tVoid DAB_tclFiConvert::vConvert2Fi(trMsgDrvRspDbQueryTestmode const &rDbQueryTestmode, 
                                    midw_ext_dabtunerfi_tclMsgFID_DAB_S_DB_QUERYMethodResult &oFiRes) {

    oFiRes.u8Adr3DbQuery = rDbQueryTestmode.u8Query;
    oFiRes.sDbQuery.bSet(rDbQueryTestmode.sQuery.c_str(),midw_ext_fi_tclString::FI_EN_UTF8);

    ETG_TRACE_USR4(("DAB_tclFiConvert::vConvert2Fi(trMsgDrvRspDbQueryTestmode) u8AdrDbQuery=%d ",
                    oFiRes.u8Adr3DbQuery));
    //ETG_TRACE_USR4(("DAB_tclFiConvert::vConvert2Fi(trMsgDrvRspDbQueryTestmode) sDbQuery=%s ",
    //                oFiRes.sDbQuery.szGet(midw_ext_fi_tclString::FI_EN_UTF8)));
}

tVoid DAB_tclFiConvert::vConvert2Fi(trMsgSrvRspGetTpegUriList const &rTpegUriList, midw_ext_dabtunerfi_tclMsgFID_DAB_G_TPEG_URI_LISTStatus &oFiRes){

#ifdef VARIANT_S_FTR_ENABLE_TPEG
    for (tU8 u8Index=0;u8Index<rTpegUriList.vcTpegUriList.size();u8Index++) {
        DAB_trTpegUriListElement const &rTpegUriListElement=rTpegUriList.vcTpegUriList[u8Index];

        midw_ext_fi_tcl_DAB_TpegUriChannel oTpegChannel;
        oTpegChannel.sTpegUriChn.bSet(rTpegUriListElement.sUri.c_str(),midw_ext_fi_tclString::FI_EN_UTF8);
        oTpegChannel.DabUriSelectState.enType=(midw_ext_fi_tcl_e8_DAB_UriSelectState::tenType)rTpegUriListElement.enUriSelectState;
        oTpegChannel.bActive = rTpegUriListElement.bActive;

        ETG_TRACE_USR4(("   DAB_tclFiConvert::trMsgSrvRspGetTpegUriList bActive %d SelectState 0x%02x UriChannel %s",
                        oTpegChannel.bActive,
                        ETG_CENUM(DAB::DAB_tenUriSelectState ,rTpegUriListElement.enUriSelectState),
                        oTpegChannel.sTpegUriChn.szGet(midw_ext_fi_tclString::FI_EN_UTF8)));

        oFiRes.TpegUriChannelList.push_back(oTpegChannel);
    }
#else
	(tVoid)rTpegUriList; //to remove lint warning
	(tVoid)oFiRes; //to remove lint warning
#endif// VARIANT_S_FTR_ENABLE_TPEG
}

midw_ext_dabtunerfi_tclMsgFID_DAB_G_TMC_DATAStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trTmcDataListProperty const &rTmcData) {

    midw_ext_dabtunerfi_tclMsgFID_DAB_G_TMC_DATAStatus oFiRes;

    for (tU8 u8Index=0;u8Index<rTmcData.vectorTmcDataListElements.size();u8Index++) {
            DAB_trTmcDataListElement const &rTmcDataListElement=rTmcData.vectorTmcDataListElements[u8Index];
            midw_ext_fi_tcl_DAB_TmcInfo oTmcInfo;
            vTmcData2Fi(rTmcDataListElement,oTmcInfo);
            oFiRes.TmcData.push_back(oTmcInfo);
        }
    return oFiRes;
}

midw_ext_dabtunerfi_tclMsgFID_DAB_G_TMC_AVAIL_LISTStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trTmcAvailListProperty const &rTmcAvailList) {

    midw_ext_dabtunerfi_tclMsgFID_DAB_G_TMC_AVAIL_LISTStatus oFiRes;

    ETG_TRACE_USR4(("DAB_tclFiConvert::DAB_trTmcAvailListProperty"));

    for (tU8 u8EnsIndex=0;u8EnsIndex<rTmcAvailList.vectorTmcAvailEntries.size();u8EnsIndex++) {
        trTmcAvailListElem const &rTmcAvailListElem = rTmcAvailList.vectorTmcAvailEntries[u8EnsIndex];


        midw_ext_fi_tcl_DAB_EnsembleTmcInfo oFiMember;
        oFiMember.u32EnsembleId = rTmcAvailListElem.rEnsembleId.u32GetID();
        oFiMember.u8Quality = rTmcAvailListElem.u8Quality;
        oFiMember.u8NumberOfRdbIds = (tU8)rTmcAvailListElem.vectorRdbIdEntries.size();

        midw_ext_fi_tcl_DAB_RdbId oRdbId;

        ETG_TRACE_USR4(("   DAB_tclFiConvert::DAB_trTmcAvailListProperty u8EnsIndex %d "
                        "u32EnsembleId=0x%x u8Quality=%d u8NumberOfRdbIds=%d",
                        u8EnsIndex,
                        oFiMember.u32EnsembleId,
                        oFiMember.u8Quality,
                        oFiMember.u8NumberOfRdbIds));

        for (tU8 u8Index=0;u8Index<oFiMember.u8NumberOfRdbIds;u8Index++) {
            DAB_trTmcRdbId const &rTmcRdbId=rTmcAvailListElem.vectorRdbIdEntries[u8Index];
            midw_ext_fi_tcl_DAB_RdbId oTmcRdbId;
            vRdbId2Fi(rTmcRdbId,oTmcRdbId);
            oFiMember.RdbIdList.push_back(oTmcRdbId);
        }

        oFiRes.EnsembleTmcInfoList.push_back(oFiMember);
    }

    return oFiRes;
}

midw_ext_dabtunerfi_tclMsgFID_DAB_G_TMC_TPEG_DATAStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trTmcTpegInfoProperty const &rTmcTpegInfo) {

    midw_ext_dabtunerfi_tclMsgFID_DAB_G_TMC_TPEG_DATAStatus oFiRes;
#ifdef VARIANT_S_FTR_ENABLE_TPEG
    oFiRes.TestModeData.u32NumTmcServices = (tU8)rTmcTpegInfo.u8NumOfTmcServices;
    oFiRes.TestModeData.u32NumTpegServices = (tU8)rTmcTpegInfo.u8NumOfTpegServices;
    oFiRes.TestModeData.u32RcvTmcMsgs = rTmcTpegInfo.u32ReceivedTmcMsgs;
    oFiRes.TestModeData.u32RcvTpegMsgs = rTmcTpegInfo.u32ReceivedTpegMsgs;
    oFiRes.TestModeData.DabTime.u32DabMday =rTmcTpegInfo.rLastReceivedUriMsgTime.u32DabMday;
    oFiRes.TestModeData.DabTime.u32DabMon  =rTmcTpegInfo.rLastReceivedUriMsgTime.u32DabMon;
    oFiRes.TestModeData.DabTime.u32DabYear  =rTmcTpegInfo.rLastReceivedUriMsgTime.u32DabYear;
    oFiRes.TestModeData.DabTime.u32DabHour  =rTmcTpegInfo.rLastReceivedUriMsgTime.u32DabHour;
    oFiRes.TestModeData.DabTime.u32DabMin = rTmcTpegInfo.rLastReceivedUriMsgTime.u32DabMin;
    oFiRes.TestModeData.DabTime.u32DabSec = rTmcTpegInfo.rLastReceivedUriMsgTime.u32DabSec;
    oFiRes.TestModeData.DabTime.u32DabWday = rTmcTpegInfo.rLastReceivedUriMsgTime.u32DabWday;
    oFiRes.TestModeData.DabTime.u32DabYday = rTmcTpegInfo.rLastReceivedUriMsgTime.u32DabYday;
    oFiRes.TestModeData.DabTime.u32DabUtcOff = rTmcTpegInfo.rLastReceivedUriMsgTime.u32DabUtcOff;

    ETG_TRACE_USR4(("   DAB_tclFiConvert::DAB_trTmcTpegInfoProperty u2NumTmcServices %d "
                        "u32NumTpegServices=%d u32RcvTmcMsgs=%d u32RcvTpegMsgs=%d "
                        "date %02d.%02d.%02d  time %02d:%02d:%02d",
                        oFiRes.TestModeData.u32NumTmcServices,
                        oFiRes.TestModeData.u32NumTpegServices,
                        oFiRes.TestModeData.u32RcvTmcMsgs,
                        oFiRes.TestModeData.u32RcvTpegMsgs,
                        oFiRes.TestModeData.DabTime.u32DabMday,
                        oFiRes.TestModeData.DabTime.u32DabMon,
                        oFiRes.TestModeData.DabTime.u32DabYear,
                        oFiRes.TestModeData.DabTime.u32DabHour,
                        oFiRes.TestModeData.DabTime.u32DabMin,
                        oFiRes.TestModeData.DabTime.u32DabSec));

#else
    (tVoid)rTmcTpegInfo;
#endif// VARIANT_S_FTR_ENABLE_TPEG
    return oFiRes;
}

midw_ext_dabtunerfi_tclMsgFID_DAB_G_MUTE_REQUESTStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trMuteRequestProperty const &rMuteRequest) {

    midw_ext_dabtunerfi_tclMsgFID_DAB_G_MUTE_REQUESTStatus oFiRes;
    oFiRes.bRequestMute = rMuteRequest.bMuteRequest;
    oFiRes.bTa = rMuteRequest.bTA;
    oFiRes.u8Attenuation = rMuteRequest.u8Attenuation;
    oFiRes.bMute_Interrupt = rMuteRequest.bMute_Interrupt;
    if (oFiRes.bTa == TRUE) {
        ETG_TRACE_USR4(("   DAB_tclFiConvert:: MUTE_REQUEST DAB TA %d",
                        rMuteRequest.bMuteRequest));
    }
    else {
        ETG_TRACE_USR4(("   DAB_tclFiConvert:: MUTE_REQUEST DAB AUDIO %d",
                        rMuteRequest.bMuteRequest));
    }

    ETG_TRACE_USR4(("   DAB_tclFiConvert::DAB_trMuteRequestProperty bMuteRequest %d bTA %d",
                        rMuteRequest.bMuteRequest,
                        rMuteRequest.bTA));

    return oFiRes;
}

midw_ext_dabtunerfi_tclMsgFID_DAB_G_LEARNUPDATE_PROGRESSStatus
DAB_tclFiConvert::vConvert2Fi(DAB_trLearnUpdateStatusProperty const &rLearnUpdate) {

	midw_ext_dabtunerfi_tclMsgFID_DAB_G_LEARNUPDATE_PROGRESSStatus oFiRes;
    oFiRes.u8LearnPercentage = rLearnUpdate.u8LearnPercentage;

    ETG_TRACE_USR4(("DAB_tclFiConvert::DAB_trLearnUpdateProperty "
                    "u8FreqPerc=%d ",
                    oFiRes.u8LearnPercentage));
    return oFiRes;
}

midw_ext_dabtunerfi_tclMsgFID_DAB_G_ADR_MSGStatus
DAB_tclFiConvert::vConvert2Fi(DAB_trADRMsgProperty const &rADRMsg){
	midw_ext_dabtunerfi_tclMsgFID_DAB_G_ADR_MSGStatus oFiStatusRes;
	midw_ext_fi_tcl_DAB_ADRMsg &oFiRes = oFiStatusRes.TunerStatus;
	oFiRes.u16FktID = rADRMsg.u16FktId;
	for(tU16 i = 0; i < rADRMsg.vectorADRMsg.size(); i++)
	{	
		tU8 oElem=rADRMsg.vectorADRMsg[i];	
		oFiRes.MsgData.push_back(oElem);	
	}
	return oFiStatusRes;
}

midw_ext_dabtunerfi_tclMsgFID_DAB_G_LOCAL_OFFSETStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trLocalOffsetProperty const &rLocalOffset){
	midw_ext_dabtunerfi_tclMsgFID_DAB_G_LOCAL_OFFSETStatus oFiStatus;
	oFiStatus.LocalOffset = rLocalOffset.s64Offset;
	
	return oFiStatus;
}
midw_ext_dabtunerfi_tclMsgFID_DAB_G_SERVICE_FOLLOWING_STATUSStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trMtcServiceFollowingProperty const &rServiceFollowing) {

    midw_ext_dabtunerfi_tclMsgFID_DAB_G_SERVICE_FOLLOWING_STATUSStatus oFiRes;
    oFiRes.e8ServiceFollowingTuner.enType= (rServiceFollowing.enMtcTuner==enMeca_MtcTuner_MAIN_DAB_TUNER) ?
        midw_ext_fi_tcl_e8_DAB_ServiceFollowingTuner::FI_EN_MAIN_DAB_TUNER : 
        midw_ext_fi_tcl_e8_DAB_ServiceFollowingTuner::FI_EN_MAIN_FM_TUNER;

    ETG_TRACE_USR4(("   DAB_tclFiConvert::DAB_trMtcServiceFollowingProperty enMtcTuner %d ",
                        rServiceFollowing.enMtcTuner));

    return oFiRes;
}

midw_ext_dabtunerfi_tclMsgFID_DAB_G_MTC_SETMODE_STATUSStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trMTCSetModeProperty const &rMTCMode) {

    midw_ext_dabtunerfi_tclMsgFID_DAB_G_MTC_SETMODE_STATUSStatus oFiRes;
	oFiRes.Mode.u8Value = rMTCMode.enMtcMode;

    ETG_TRACE_USR4((" DAB_tclFiConvert::vConvert2Fi(DAB_trMTCSetModeProperty): MTCMode = %d )",
                     ETG_CENUM(DAB_tenMtcMode, rMTCMode.enMtcMode)));

    return oFiRes;
}


//midw_ext_dabtunerfi_tclMsgFID_DAB_G_MTC_REQUEST_SWITCH_STATUSStatus 
//DAB_tclFiConvert::vConvert2Fi(DAB_trMTCSwitchStatusProperty const &rSwitchMode) {
//
//    midw_ext_dabtunerfi_tclMsgFID_DAB_G_MTC_REQUEST_SWITCH_STATUSStatus oFiRes;
//	oFiRes.Request.enType= (midw_ext_fi_tcl_e8_MTC_Request::tenType)rSwitchMode.enMtcRequestSwitch;
//
//   ETG_TRACE_USR4(("  DAB_tclFiConvert::vConvert2Fi(DAB_trMTCSwitchStatusProperty): SwitchMode = %d )",
//                     ETG_CENUM(DAB_tenMtcRequestSwitch, rSwitchMode.enMtcRequestSwitch)));
//
//    return oFiRes;
//}

// New FI Changes3

midw_ext_dabtunerfi_tclMsgFID_DAB_G_MTC_TUNER_STATUSStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trMTCTunerStatusProperty const &rTunerStatus) {

    midw_ext_dabtunerfi_tclMsgFID_DAB_G_MTC_TUNER_STATUSStatus oFiRes;
	oFiRes.State.enType = (midw_ext_fi_tcl_e8_MTC_Tuner_Status::tenType)rTunerStatus.e8MtcTunerSwitchStatus;

   ETG_TRACE_USR4(("  DAB_tclFiConvert::vConvert2Fi(DAB_trMTCTunerStatusProperty): SwitchMode = %d )",
                     ETG_CENUM(DAB_tenMtcTunerSwitchStatus, rTunerStatus.e8MtcTunerSwitchStatus)));
   
    return oFiRes;
}

// New FI Changes4

//midw_ext_dabtunerfi_tclMsgFID_DAB_G_MTC_ANNOUNCEMENTStatus 
//DAB_tclFiConvert::vConvert2Fi(DAB_trMTCAnnouncementProperty const &rAnnoStatus) {
//
//    midw_ext_dabtunerfi_tclMsgFID_DAB_G_MTC_ANNOUNCEMENTStatus oFiRes;
//	oFiRes.ActivityState.enType = (midw_ext_fi_tcl_e8_MTC_Tuner_ActivityState::tenType)rAnnoStatus.enMtcActivityState;
//
//  ETG_TRACE_USR4(("DAB_tclFiConvert::vConvert2Fi(DAB_trMTCAnnouncementProperty): poAnnoState = %d )",
//                     ETG_CENUM(DAB_tenMtcActivityState, rAnnoStatus.enMtcActivityState)));
//
//    return oFiRes;
//}

// New FI Changes5

midw_ext_dabtunerfi_tclMsgFID_DAB_G_MTC_SURVEILLANCE_MONITORStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trMTCSurveillanceMonitorProperty const &rSurveillanceMonitor) {

    midw_ext_dabtunerfi_tclMsgFID_DAB_G_MTC_SURVEILLANCE_MONITORStatus oFiRes;
	oFiRes.StationType.enType = (midw_ext_fi_tcl_e8_StationType::tenType)rSurveillanceMonitor.enMtcStationType;
	oFiRes.StationFrequency = rSurveillanceMonitor.u32MtcStationFreq;
	oFiRes.StationId = rSurveillanceMonitor.u32MtcStationId;
	oFiRes.EnsID = rSurveillanceMonitor.u32MtcEnsID;
	oFiRes.TPState.enType = (midw_ext_fi_tcl_e8_TPState::tenType)rSurveillanceMonitor.enMtcTPState;

	tU8 u8abuffer[32];
	::memset(u8abuffer, '\0', 32);
	
	//PSARCC30-2905
	DAB_stringUtil::vConvertRDSPSNametoUtf8((const tChar*)rSurveillanceMonitor.sMtcStationNameFM,(char*)u8abuffer);
//	u32ConvertEBUStringToUtf8::u32ConvertEBUStringToUtf8((const tChar*)rSurveillanceMonitor.sMtcStationNameFM,8,u8abuffer);
	oFiRes.StationNameFM.bSet((const tChar*)u8abuffer,midw_ext_fi_tclString::FI_EN_UTF8);
	ETG_TRACE_USR4(("FM Name %s",u8abuffer));
	tU8 u8buffer[64];
	::memset(u8buffer, '\0', 64);
	//In PSA DAB station name is taken from Chninfo, So to reduce extra validations the code has been commented out.
//	DAB_stringUtil::u32ConvertMecaStringToUtf8(enDab_CharSetUTF8, (const tChar*)rSurveillanceMonitor.sStationNameDAB, u8buffer, DAB_LABEL_MAX_NUM_CHAR);
	oFiRes.StationNameDAB.bSet((const tChar*)u8buffer,midw_ext_fi_tclString::FI_EN_UTF8);

	 ETG_TRACE_USR4(("  DAB_trMTCSurveillanceMonitorProperty: e8MtcMtcStationType =%d "
			"  e8MtcStationFreq=%d  e8MtcStationId=%d  e8MtcEnsID=%d"
            "  e8MtcTPState=%d  sMtcStationNameFM=%s " ,
             ETG_CENUM(DAB_tenMtcStationType, rSurveillanceMonitor.enMtcStationType), rSurveillanceMonitor.u32MtcStationFreq,
			 rSurveillanceMonitor.u32MtcStationId, rSurveillanceMonitor.u32MtcEnsID, 
			 ETG_CENUM(DAB_tenMtcTPState, rSurveillanceMonitor.enMtcTPState), rSurveillanceMonitor.sMtcStationNameFM));  
	 ETG_TRACE_USR4(("sStationNameDAB=%s", oFiRes.StationNameDAB.szGet(midw_ext_fi_tclString::FI_EN_UTF8)));

    return oFiRes;
}



tBool DAB_tclFiConvert::bConvertRTextTotString( tenMeca_CharSet enMeca_CharSet, const tChar* apRText, midw_ext_fi_tclString* sUTF8String, tU8 u8Len)
{ 
    (tVoid)u8Len;

    // Buffer to get the PS string. ( 4 * 128 = 512, 4bytes per character )
    tU8 u8abuffer[DABTUN_RADIOTEXT_BUFFER];  
    // Initialize the buffer with '\0'
    ::memset(u8abuffer, '\0', DABTUN_RADIOTEXT_BUFFER);

    DAB_stringUtil::u32ConvertMecaStringToUtf8(DAB_enGetDabCharSet(enMeca_CharSet), apRText, u8abuffer, DABTUN_RADIOTEXT_BUFFER-1);

    // Copy the buffer contents.
    return sUTF8String->bSet((tCString)u8abuffer, midw_ext_fi_tclString::FI_EN_UTF8);
}


tVoid DAB_tclFiConvert::vConvert2Fi(tU8 const *pu8InUtf8Buf, midw_ext_fi_tclString &sUTF8FiOutString) {
    sUTF8FiOutString.bSet((tCString)pu8InUtf8Buf, midw_ext_fi_tclString::FI_EN_UTF8);
}

tVoid DAB_tclFiConvert::vConvertFromFi(midw_ext_dabtunerfi_tclMsgFID_DAB_S_SELECT_TMCMethodStart const *prFiMsg, trMsgSrvCmdSelectTmc *prMsg) {
    prMsg->rEns=trMecaEnsemble(prFiMsg->u32EnsembleId);
    if (prFiMsg->RdbIDList.size()) {
        vConvertFromFi(prFiMsg->RdbIDList[0], &prMsg->rRdbId);
    }
}

tVoid DAB_tclFiConvert::vConvertFromFi(midw_ext_fi_tcl_DAB_RdbId const &rFiRdbId, DAB_trTmcRdbId *prRdbId) {
    prRdbId->u8Cc = rFiRdbId.u8Cc;
    prRdbId->u8Ltn = rFiRdbId.u8Ltn;
    prRdbId->u16Bsa = rFiRdbId.u16Bsa;
    prRdbId->u8Sid = rFiRdbId.u8Sid;
    prRdbId->bEncrypted = rFiRdbId.bEncrypted;
}

tVoid DAB_tclFiConvert::vConvertFromFi(midw_ext_dabtunerfi_tclMsgFID_DAB_S_RDBID_WISH_LISTMethodStart const *prFiMsg, trMsgSrvCmdRdbidWishlist *rRdbidWishlist) {

    ETG_TRACE_USR4(("   DAB_tclFiConvert::trMsgSrvCmdRdbidWishlist size %d", (tU16)prFiMsg->RdbIDList.size()));

    if (prFiMsg->RdbIDList.size() != 0) {
        for(tU32 i = 0; i < prFiMsg->RdbIDList.size(); i++)
        {
            midw_ext_fi_tcl_DAB_RdbId const &oTmcRdbId=prFiMsg->RdbIDList[i];
            DAB_trTmcRdbId rTmcRdbId;
            vFi2RdbId(oTmcRdbId, rTmcRdbId);
            rRdbidWishlist->vecRdbIdList.push_back(rTmcRdbId);
        }
    }
}

tVoid DAB_tclFiConvert::vConvertFromFi(midw_ext_dabtunerfi_tclMsgFID_DAB_S_SET_TPEG_CHNMethodStart const *prFiMsg, trMsgSrvCmdSelectTpegUri *rSelectTpegUri) {

    ETG_TRACE_USR4(("   DAB_tclFiConvert::trMsgSrvCmdSelectTpegUri bActive=%d", prFiMsg->bActive));

    rSelectTpegUri->bSelected = prFiMsg->bActive;
    rSelectTpegUri->sSelectUri = prFiMsg->sTpegUriChn.sTpegUriChn.szGet(midw_ext_fi_tclString::FI_EN_UTF8);
}


midw_ext_dabtunerfi_tclMsgFID_DAB_G_TPEG_DATAStatus 
DAB_tclFiConvert::vConvert2Fi(DAB_trTpegDataListProperty const &rTpegData) {

    midw_ext_dabtunerfi_tclMsgFID_DAB_G_TPEG_DATAStatus oFiRes;

    for (tU8 u8Index=0;u8Index<rTpegData.vecTpegDataListElements.size();u8Index++) {
            DAB_trTpegDataListElement const &rTpegDataListElement=rTpegData.vecTpegDataListElements[u8Index];
            midw_ext_fi_tcl_DAB_TpegInfo oTpegInfo;
            vTpegData2Fi(rTpegDataListElement,oTpegInfo);
            oFiRes.TpegData.push_back(oTpegInfo);
        }
    ETG_TRACE_USR4(("DAB_tclFiConvert::vConvert2Fi(DAB_trTpegDataListProperty) Number of TPEG Msgs = %d ",
                        (tU16)oFiRes.TpegData.size()));

    return oFiRes;
}

tVoid DAB_tclFiConvert::vTpegData2Fi(DAB_trTpegDataListElement const &rTpegDataListElement, midw_ext_fi_tcl_DAB_TpegInfo &oFiTpegInfo) {
    
    oFiTpegInfo.TpegData = rTpegDataListElement.vecTpegDataListElement;

    ETG_TRACE_USR4(("DAB_tclFiConvert::vTpegData2Fi Number of Elements = %d ",
                       (tU16) rTpegDataListElement.vecTpegDataListElement.size()));
}

midw_ext_dabtunerfi_tclMsgFID_DAB_G_LIST_UPDATEStatus
DAB_tclFiConvert::vConvert2Fi(DAB_trMsgSrvListUpdateProperty const &oMsg) {

midw_ext_dabtunerfi_tclMsgFID_DAB_G_LIST_UPDATEStatus oFiRes;

for (tU16 u16Index=0;u16Index<oMsg.vcAvailSrvList.size();u16Index++) {
        trChnListElem const &rListElem = oMsg.vcAvailSrvList[u16Index];
        midw_ext_fi_tcl_DAB_ChnListElement oFiListElem;
        oFiListElem.u16ListElementID=rListElem.u16Id;
        oFiListElem.u32DABId=rListElem.rMecaId.u32GetSID();
        oFiListElem.u8CompScids= (tU8)rListElem.u16Scids;
        vLabel2Fi(rListElem.rLabel, oFiListElem.Label);
		vLabel2Fi(rListElem.rEnsembleLabel, oFiListElem.EnsembleLabel);
        oFiListElem.bReception=rListElem.bReception;
        oFiListElem.u8PresetListElemId=rListElem.u8PresetNumber;
		oFiListElem.u8PTY=rListElem.u8PTY;
		oFiListElem.u8NumberOfComponents=rListElem.u8NumberOfComponents;
		
		if(dbus_validate_utf8(rListElem.sLogoLink.c_str(),0)){
		oFiListElem.sLogoLink.bSet(rListElem.sLogoLink.c_str(),midw_ext_fi_tclString::FI_EN_UTF8);
		}else{
			oFiListElem.sLogoLink.bSet("",midw_ext_fi_tclString::FI_EN_UTF8);
		}
		if(dbus_validate_utf8(rListElem.sFreqLabel.c_str(),0)){
		oFiListElem.sFreqLabel.bSet(rListElem.sFreqLabel.c_str(),midw_ext_fi_tclString::FI_EN_UTF8);
		}else{
			oFiListElem.sFreqLabel.bSet("",midw_ext_fi_tclString::FI_EN_UTF8);
		}
		
        oFiRes.AvailSrvList.push_back(oFiListElem);
		ETG_TRACE_USR4(("   DAB_tclFiConvert::DAB_trMsgSrvListUpdateProperty Element %d u8PTY=%d"
                        "u16ListElementID=%d u32DABId=0x%x Label=%17s bReception=%d presetNr=%d u8CompScids %d EnsembleLabel=%s ",
                    u16Index,
					oFiListElem.u8PTY,
                    oFiListElem.u16ListElementID,
                    oFiListElem.u32DABId,
                    rListElem.rLabel.pcGetCString(),
                        oFiListElem.bReception,
                        oFiListElem.u8PresetListElemId,
                        oFiListElem.u8CompScids,
						rListElem.rEnsembleLabel.pcGetCString()));
	     ETG_TRACE_USR4(("   DAB_tclFiConvert::DAB_trMsgSrvListUpdateProperty LogoLink=%s",
						rListElem.sLogoLink.c_str()));
     
    }
return oFiRes;
}

midw_ext_dabtunerfi_tclMsgFID_DAB_G_SRVLIST_NOTIFICATIONStatus
DAB_tclFiConvert::vConvert2Fi(DAB_trMsgSrvListUpdateNotification const &oMsg) {

    midw_ext_dabtunerfi_tclMsgFID_DAB_G_SRVLIST_NOTIFICATIONStatus oFiRes;
    oFiRes.bListUpdate = oMsg.bListUpdate;
    ETG_TRACE_USR4(("DAB_tclFiConvert::vConvert2Fi(DAB_trMsgSrvListUpdateNotification bListUpdate = %d ",
                        oMsg.bListUpdate));
    return oFiRes;
    
}
