/************************************************************************
* FILE:         fc_dabtuner_config.hpp
* PROJECT:      g3g
* SW-COMPONENT: DAB Tuner
*----------------------------------------------------------------------
*
* DESCRIPTION:  Implementation of fc_dabtuner_config
*              
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
* HISTORY:      
* Date      | Author             | Modification

*
*************************************************************************/

#ifndef FC_DABTUNER_CONFIG_HPP
#define FC_DABTUNER_CONFIG_HPP

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "fc_dabtuner_config.h"
#include "fc_dabtuner_singleton.h"
#include "dabdrv_typeEnums.h"

#define DAB_GET_CONFIG() DAB::fc_dabtuner_config::instance()->rGetConfig()

#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
#define FEATURE_PSA_RCC 1
//#else
//#define FEATURE_PSA_RCC 0
#endif

#ifdef VARIANT_S_FTR_ENABLE_FEATURE_VW_MIBE
#define FEATURE_VW_MIBE 1
#else
#define FEATURE_VW_MIBE 0
#endif

namespace DAB {

    //Enumeration for KDS configuration parameters
typedef enum
{
  DAB_KDS_INVALID_PARAMETER   = 255
 ,DAB_KDS_PARAMETER_SET = 0
 ,DAB_KDS_CM_CONFIG_STRING
 ,DAB_KDS_CODING_BAND
 ,DAB_KDS_CODING_VARIANT
 ,DAB_KDS_EOL_PARAM_RADIO
 ,DAB_KDS_EOL_PARAM_RADIONUM
 ,DAB_KDS_GENERIC_PARAM_SET
} tenKDSParameter;

#define DIA_CODING_VALUE 0XD10
#define DIA_CODING_VALUE_LEN 25
#define DAB_KDS_PSA_EOL_RADIO	 0xA127
#define DAB_KDS_PSA_EOL_RADIO_LEN 4
#define DAB_KDS_PSA_EOL_RADIONUM 0xA128
#define DAB_KDS_PSA_EOL_RADIONUM_LEN 2
#define MASK_LOWER_NIBBLE 0x0F
#define MASK_UPPER_NIBBLE 0xF0
#define NUMBER_TWO 2
#define NUMBER_FOUR 4
#define NUMBER_FIVE 5
#define NUMBER_SIX 6
#ifdef VARIANT_S_FTR_ENABLE_AARSPROXY
const tU8 cleoSBRVariant = 0x2A;
const tU8 aivi2SBRVariant = 0x2B;
#endif
/*============================================================================
 * Base class for KDS entries
 *--------------------------------------------------------------------------*/
struct tun_trKds
{
  virtual tVoid vSetToDefaultValues( tVoid) = 0;
  virtual tVoid vParse( tPU8 pu8KdsBuffer, tU32 u32Length) = 0;
  virtual tVoid vTrace( tVoid) = 0;
};
/*============================================================================
 * Indices for KDS byte stream 'HWConfigurationString'
 *--------------------------------------------------------------------------*/
typedef enum
{
  enKdsHWConfigurationString_idx_OEMType                 =   3, enKdsHWConfigurationString_len_OEMType               =   1,
  enKdsHWConfigurationString_idx_TunerMode               =   4, enKdsHWConfigurationString_len_TunerMode               =   1,
  enKdsHWConfigurationString_idx_FmDabHdTunerMode        =   5, enKdsHWConfigurationString_len_FmDabHdTunerMode        =   1
} tenKdsHWConfigurationString;                                    /* ------- */
#define DAB_CONFIG_KDS_HW_CONFIGURATION_STRING_LENGTH             64
/*============================================================================
 * KDS type FmDabHdTunerMode
 *--------------------------------------------------------------------------*/
typedef enum
{
  enFmDabHdTunerMode_reserved_0x00          = 0x00,
  enFmDabHdTunerMode_TunVar1_2FE_DDA        = 0x01,
  enFmDabHdTunerMode_TunVar2_1FE            = 0x02,
  enFmDabHdTunerMode_TunVar3_2FE            = 0x03,
  enFmDabHdTunerMode_reserved_0x04          = 0x04,
  enFmDabHdTunerMode_TunVar5_3FE_DDA        = 0x05,
  enFmDabHdTunerMode_TunVar6_3FE_DAB        = 0x06,
  enFmDabHdTunerMode_TunVar7_4FE_DDA_DAB    = 0x07,
  enFmDabHdTunerMode_TunVar8_2FE_HD         = 0x08,
  enFmDabHdTunerMode_TunVar9_2FE_DDA_HD     = 0x09,
  enFmDabHdTunerMode_TunVar10_3FE_DDA_HD    = 0x0A,
  enFmDabHdTunerMode_TunVar11_3FE_DAB_SBM   = 0x0B,
  enFmDabHdTunerMode_TunVar12_4FE_DDA_DAB_SBM   = 0x0C,
  enFmDabHdTunerMode_TunVar13_2FE_DDA_DDS		= 0x0D,
  enFmDabHdTunerMode_TunVar14_3FE_DDA_DDS_DAB   = 0x0E,
  enFmDabHdTunerMode_TunVar15_3FE_DDA_DDS_DAB   = 0x0F,
  enFmDabHdTunerMode_TunVar16_2FE			    = 0x10,
  enFmDabHdTunerMode_TunVar17_3FE_DAB			= 0x11,
  enFmDabHdTunerMode_TunVar18_2FE_DDA_DDS		= 0x12,
  enFmDabHdTunerMode_TunVar19_2FE_DDA_DDS		= 0x13,
  enFmDabHdTunerMode_TunVar20_3FE_DDA_DDS_DAB	= 0x14,
  enFmDabHdTunerMode_TunVar21_3FE_DDA_DDS_DAB   = 0x15,
  enFmDabHdTunerMode_TunVar22_3FE_DDA_DDS_DAB   = 0x16,
  enFmDabHdTunerMode_TunVar23					= 0x17,
  enFmDabHdTunerMode_TunVar24					= 0x18,
  enFmDabHdTunerMode_TunVar25					= 0x19,
  enFmDabHdTunerMode_TunVar26					= 0x1a,
  enFmDabHdTunerMode_TunVar27					= 0x1b,
  enFmDabHdTunerMode_TunVar28					= 0x1c,
  enFmDabHdTunerMode_TunVar29					= 0x1d,
  enFmDabHdTunerMode_TunVar30					= 0x1e,
  enFmDabHdTunerMode_TunVar31					= 0x1f,
  enFmDabHdTunerMode_TunVar32_2FE_DDA			= 0x20,
  enFmDabHdTunerMode_TunVar33_3FE_DDA_DAB		= 0x21,
  enFmDabHdTunerMode_TunVar34_2FE_DAB			= 0x22,
  enFmDabHdTunerMode_TunVar35_2FE_DDA_DDS		= 0x23,
  enFmDabHdTunerMode_TunVar36					= 0x24,
  enFmDabHdTunerMode_TunVar37_3FE_DDA_DDS_DAB	= 0x25,
  enFmDabHdTunerMode_TunVar38_2FE_AM_1FE		= 0x26,
  enFmDabHdTunerMode_TunVar39_3FE_AM_1FE_DAB	= 0x27,
  enFmDabHdTunerMode_TunVar40_4FE_AM_1FE_DDA_DAB = 0x28, 
  enFmDabHdTunerMode_TunVar41_4FE_AM_1FE_DDA_DAB = 0x29,
  enFmDabHdTunerMode_TunVar42_3FE_AM_1FE_DDA_HD	 = 0x2a,
  enFmDabHdTunerMode_TunVar43_3FE_AM_1FE_DDA	 = 0x2b,
  enFmDabHdTunerMode_TunVar44_2FE_AM_1FE_DDA	 = 0x2c,
  enFmDabHdTunerMode_TunVar45_2FE_AM_2FE		 = 0x2d
} tenFmDabHdTunerMode;

struct trKdsHWConfigurationString : public tun_trKds
{
  tBool bKdsDataRead;
  tU8   u8OEMType;
  tU8   u8TunerMode;
  tenFmDabHdTunerMode enFmDabHdTunerMode;
  
  trKdsHWConfigurationString( tVoid);
  virtual tVoid vSetToDefaultValues( tVoid);
  virtual tVoid vParse( tPU8 pu8KdsBuffer, tU32 u32Length);
  virtual tVoid vTrace( tVoid);
};
struct trKdsCodingData : public tun_trKds
{
  tBool bKdsDataRead;
  tU8   u8TunerBand1;   
  tU8   u8TunerBand2;   
  tBool bAlarmEnabled;
  trKdsCodingData( tVoid);

  virtual tVoid vSetToDefaultValues( tVoid);
  virtual tVoid vParse( tPU8 pu8KdsBuffer, tU32 u32Length);
  virtual tVoid vTrace( tVoid);
};
//struct trKdsCodingVariant : public tun_trKds
//{
//  tBool bKdsDataRead;
//  tU8   u8OEMType;
//  trKdsCodingVariant( tVoid);
//
//  virtual tVoid vSetToDefaultValues( tVoid);
//  virtual tVoid vParse( tPU8 pu8KdsBuffer, tU32 u32Length);
//  virtual tVoid vTrace( tVoid);
//};
#ifdef FEATURE_PSA_RCC
struct trKdsPSAEOLParamRadio : public tun_trKds
{
  tBool bKdsDataRead;
  tU8   u8Radio_CountryZone;
  tBool bRadio_RdsFollowUp;
  trKdsPSAEOLParamRadio( tVoid);

  virtual tVoid vSetToDefaultValues( tVoid);
  virtual tVoid vParse( tPU8 pu8KdsBuffer, tU32 u32Length);
  virtual tVoid vTrace( tVoid);
};
struct trKdsPSAEOLParamRadioNum : public tun_trKds
{
  tBool bKdsDataRead;
  tU8   u8RadioNum_BandActivation;
  tU8   u8RadioNum_BandActivityThreshold;
  trKdsPSAEOLParamRadioNum( tVoid);

  virtual tVoid vSetToDefaultValues( tVoid);
  virtual tVoid vParse( tPU8 pu8KdsBuffer, tU32 u32Length);
  virtual tVoid vTrace( tVoid);
};
#endif
typedef enum {
	enCodingTunerBand1_OFF				= 0x0,
	enCodingTunerBand1_EU_BAND3_WO_NEW  = 0x1,
	enCodingTunerBand1_EU_BAND3         = 0x2,
	enCodingTunerBand1_CANADA_L_BAND    = 0x3,
	enCodingTunerBand1_KOREA_BAND3      = 0x4,
	enCodingTunerBand1_CHINA_BAND3      = 0x5,
	enCodingTunerBand1_DOWNLOADTABLE1   = 0x6,
	enCodingTunerBand1_NEWZEALAND_BAND3 = 0x7
}tenCodingTunerBand1;
typedef enum {
	enCodingTunerBand2_OFF				= 0x0,
	enCodingTunerBand2_L_BAND			= 0x1,
	enCodingTunerBand1_DOWNLOADTABLE2   = 0x2
}tenCodingTunerBand2;

typedef enum {
    enConfigServiceLinking_OFF = 0x0UL,
    enConfigServiceLinking_DAB = 0x1UL,
    enConfigServiceLinking_DAB_FM = 0x2UL,
    enConfigServiceLinking_FM = 0x3UL
} tenConfigServiceLinking;
typedef enum
{
  enKdsCodingValue_TunerBand          =   10,
  enKdsCodingValue_Alarm              =   13, 
  enKdsOEMType						  =   3
} tenKdsVariantCoding;

#ifdef FEATURE_PSA_RCC
typedef enum
{
  enConfigCountryZone_EU            =   0x00,
  enConfigCountryZone_JP            =   0x01,
  enConfigCountryZone_SA            =   0x02,
  enConfigCountryZone_Asia_CH       =   0x03,
  enConfigCountryZone_UAE           =   0x04,
  enConfigCountryZone_USA           =   0x05,
  enConfigCountryZone_KOR           =   0x06,
  enConfigCountryZone_CAN           =   0x07
} tenConfigCountryZone;

typedef enum
{
  enConfigBandActivation_DeactiveAll            =   0x00,
  enConfigBandActivation_LBandActive            =   0x01,
  enConfigBandActivation_Band3Active            =   0x02,
  enConfigBandActivation_ActivateAll            =   0x03
} tenConfigBandActivation;

typedef enum
{
  enKdsPSAEOLRadio_Cluster            =   0,
  enKdsPSAEOLRadio_Activity           =   1,
  enKdsPSAEOLRadio_Antenna            =   2,
  enKdsPSAEOLRadio_Country            =   3
} tenKdsPSAEOLParamRadio;

typedef enum
{
  enKdsPSAEOL_BandActivation          =   0,
  enKdsPSAEOL_ActivityThreshold       =   1
} tenKdsPSAEOLParamRadioNum;
#endif

typedef enum
{
	enKdsOEMType_VW          =   0x00,
	enKdsOEMType_SKODA       =   0x02,
	enKdsOEMType_NISSAN      =   0x06,
	enKdsOEMType_SUZUKI      =   0x0C
} tenKdsOEMType;

typedef enum
{
  enKdsAIVIVariantType_NotConfigured =   0,
  enKdsAIVIVariantType_Navi          =   1,
  enKdsAIVIVariantType_DA            =   2,
  enKdsAIVIVariantType_No3D          =   3
} tenKdsAIVIVariantType;

/*============================================================================
 * Indices for KDS byte stream 'DAB Tuner Parameter Set 1'
 *--------------------------------------------------------------------------*/
typedef enum
{
	enKdsDabTunerParameter_idx_Tag							   =   0, enKdsDabTunerParameter_len_Tag                        =  8,
	enKdsDabTunerParameter_idx_MajorVersion					   =   8, enKdsDabTunerParameter_len_MajorVersion               =  1,
	enKdsDabTunerParameter_idx_MinorVersion					   =   9, enKdsDabTunerParameter_len_MinorVersion               =  2,
	enKdsDabTunerParameter_idx_FeatureControl1				   =  11, enKdsDabTunerParameter_len_FeatureControl1           =  1,
	enKdsDabTunerParameter_idx_FeatureControl2				   =  12, enKdsDabTunerParameter_len_FeatureControl2           =  1,
	enKdsDabTunerParameter_idx_ServiceFollowConfig    		   =  13, enKdsDabTunerParameter_len_ServiceFollowConfig           =  1,
	enKdsDabTunerParameter_idx_ConcealmentLevel	    		   =  14, enKdsDabTunerParameter_len_ConcealmentLevel           =  1,
	enKdsDabTunerParameter_idx_TsuMatrixDabFm				   =  15, enKdsDabTunerParameter_len_TsuMatrixDabFm             =  32,
	enKdsDabTunerParameter_idx_TsuMatrixFmDab				   =  47, enKdsDabTunerParameter_len_TsuMatrixFmDab             =  32,
	enKdsDabTunerParameter_idx_DabFmSwitchBackDelay			   =  79, enKdsDabTunerParameter_len_DabFmSwitchBackDelay             =  1,
	enKdsDabTunerParameter_idx_DabFmSmoothingT_Up			   =  80, enKdsDabTunerParameter_len_DabFmSmoothingT_Up             =  1,
	enKdsDabTunerParameter_idx_DabFmSmoothingT_Down			   =  81, enKdsDabTunerParameter_len_DabFmSmoothingT_Down             =  1,
	enKdsDabTunerParameter_idx_DabFmComparisionDelay		   =  82, enKdsDabTunerParameter_len_DabFmComparisionDelay             =  1,
	enKdsDabTunerParameter_idx_DabFmSmoothingT_Long			   =  83, enKdsDabTunerParameter_len_DabFmSmoothingT_Long             =  1,
	enKdsDabTunerParameter_idx_DabFmQualityThreshold		   =  84, enKdsDabTunerParameter_len_DabFmQualityThreshold             =  1,
	enKdsDabTunerParameter_idx_FmDABSwitchBackDelay			   =  85, enKdsDabTunerParameter_len_FmDABSwitchBackDelay             =  1,
	enKdsDabTunerParameter_idx_FmDABSmoothingT_Up			   =  86, enKdsDabTunerParameter_len_FmDABSmoothingT_Up             =  1,
	enKdsDabTunerParameter_idx_FmDABSmoothingT_Down			   =  87, enKdsDabTunerParameter_len_FmDABSmoothingT_Down             =  1,
	enKdsDabTunerParameter_idx_FmDABPSIDChangeDelay			   =  88, enKdsDabTunerParameter_len_FmDABPSIDChangeDelay             =  1,
	enKdsDabTunerParameter_idx_FmDABSmoothingT_Long			   =  89, enKdsDabTunerParameter_len_FmDABSmoothingT_Long             =  1,
	enKdsDabTunerParameter_idx_FmDABQualityThreshold		   =  90, enKdsDabTunerParameter_len_FmDABQualityThreshold             =  1,
	/*    byte 91 - 93 reserved    */
	enKdsDabTunerParameter_idx_RDMQConfig_QDecreaseFactor      =  94, enKdsDabTunerParameter_len_RDMQConfig_QDecreaseFactor               =  1,
	enKdsDabTunerParameter_idx_RDMQConfig_Mode				   =  95, enKdsDabTunerParameter_len_RDMQConfig_Mode               =  1,
	enKdsDabTunerParameter_idx_RDMQConfig_Count				   =  96, enKdsDabTunerParameter_len_RDMQConfig_Count               =  1,
	enKdsDabTunerParameter_idx_Ensemble_RecIndicator		   =  97, enKdsDabTunerParameter_len_Ensemble_RecIndicator               =  1,
	enKdsDabTunerParameter_idx_Ensemble_RecIndicatorSyncDelay  =  98, enKdsDabTunerParameter_len_Ensemble_RecIndicatorSyncDelay               =  1,
	enKdsDabTunerParameter_idx_EnsembleVerAlg_TimeTickDivider  =  99, enKdsDabTunerParameter_len_EnsembleVerAlg_TimeTickDivider               =  1,
	enKdsDabTunerParameter_idx_EnsembleVerAlg_TimePeriod	   =  100, enKdsDabTunerParameter_len_EnsembleVerAlg_TimePeriod              =  1,
	enKdsDabTunerParameter_idx_EnsembleVerAlg_MinFICBERTime	   =  101, enKdsDabTunerParameter_len_EnsembleVerAlg_MinFICBERTime              =  1,
	enKdsDabTunerParameter_idx_EnsembleVerAlg_MinFICBERLevel   =  102, enKdsDabTunerParameter_len_EnsembleVerAlg_MinFICBERLevel              =  1,
	enKdsDabTunerParameter_idx_EnsembleVerAlg_TargetCntrValue  =  103, enKdsDabTunerParameter_len_EnsembleVerAlg_TargetCntrValue              =  1,
	enKdsDabTunerParameter_idx_EnsembleQualAlg_MinFICBERTime   =  104, enKdsDabTunerParameter_len_EnsembleQualAlg_MinFICBERTime              =  1,
	enKdsDabTunerParameter_idx_EnsembleQualAlg_MinFICBERLevel  =  105, enKdsDabTunerParameter_len_EnsembleQualAlg_MinFICBERLevel              =  1,	
	enKdsDabTunerParameter_idx_BGSnooping1					   =  106, enKdsDabTunerParameter_len_BGSnooping1              =  1,	
	enKdsDabTunerParameter_idx_BGSnooping_LearnFieldstrength   =  107, enKdsDabTunerParameter_len_BGSnooping_LearnFieldstrength              =  1,
	enKdsDabTunerParameter_idx_BGSnooping_UnLearnFieldstrength =  108, enKdsDabTunerParameter_len_BGSnooping_UnLearnFieldstrength              =  1,
	enKdsDabTunerParameter_idx_BGSnooping_LearnCntrMaxValue	   =  109, enKdsDabTunerParameter_len_LearnCntrMaxValue              =  1,
	enKdsDabTunerParameter_idx_BGSnooping_LearnCntrValue	   =  110, enKdsDabTunerParameter_len_LearnCntrValue              =  1,
	enKdsDabTunerParameter_idx_BGSnooping_UnLearnCntrValue	   =  111, enKdsDabTunerParameter_len_UnLearnCntrValue              =  1,
	enKdsDabTunerParameter_idx_BGSnooping_UnLearnCntrMinValue  =  112, enKdsDabTunerParameter_len_UnLearnCntrMinValue              =  1,
	/*    byte 91 - 93 reserved    */
	enKdsDabTunerParameter_idx_AnnouncementConfig              =  118, enKdsDabTunerParameter_len_AnnouncementConfig              =   1,
	/*    byte 119 - 121 reserved    */
	enKdsDabTunerParameter_idx_DabScanConfig				   =  122, enKdsDabTunerParameter_len_DabScanConfig              =   1,
	enKdsDabTunerParameter_idx_ScanMinSignalQuality			   =  123, enKdsDabTunerParameter_len_ScanMinSignalQuality             =   1,
	enKdsDabTunerParameter_idx_ScanAudioThreshold			   =  124, enKdsDabTunerParameter_len_ScanAudioThreshold              =   1,
	enKdsDabTunerParameter_idx_ScanPlayDuration				   =  125, enKdsDabTunerParameter_len_ScanPlayDuration              =   1,
	/*    byte 126 - 143 reserved    */
	enKdsDabTunerParameter_idx_EnsembleListSorting			   =  144, enKdsDabTunerParameter_len_EnsembleListSorting              =   1,
	enKdsDabTunerParameter_idx_ServiceListSorting			   =  145, enKdsDabTunerParameter_len_ServiceListSorting              =   1,
	enKdsDabTunerParameter_idx_ComponentListSorting			   =  146, enKdsDabTunerParameter_len_ComponentListSorting              =   1,
	enKdsDabTunerParameter_idx_MaxStations					   =  147, enKdsDabTunerParameter_len_MaxStations              =   2,
	enKdsDabTunerParameter_idx_StationListConfig			   =  149, enKdsDabTunerParameter_len_StationListConfig              =   1,
	/*    byte 150 - 153 reserved    */
	enKdsDabTunerParameter_idx_t_FmDab_SwitchMute_Dab		   =  154, enKdsDabTunerParameter_len_t_FmDab_SwitchMute_Dab               =   1,
	enKdsDabTunerParameter_idx_t_FmDab_SwitchMute_DabPlus	   =  155, enKdsDabTunerParameter_len_t_FmDab_SwitchMute_DabPlus               =   1,
	enKdsDabTunerParameter_idx_t_FmDab_SwitchMute_Dmba		   =  156, enKdsDabTunerParameter_len_t_FmDab_SwitchMute_Dmba               =   1,
	enKdsDabTunerParameter_idx_t_BGLearnStart				   =  157, enKdsDabTunerParameter_len_t_BGLearnStart				 =   1,
	enKdsDabTunerParameter_idx_t_LearnStart_SF_Off			   =  158, enKdsDabTunerParameter_len_t_LearnStart_SF_Off				 =   1,
	enKdsDabTunerParameter_idx_t_ManMode_SF_Off				   =  159, enKdsDabTunerParameter_len_t_ManMode_SF_Off				 =   1,
	enKdsDabTunerParameter_idx_t_Station_Select_SF_Off		   =  160, enKdsDabTunerParameter_len_t_Station_Select_SF_Off				 =   1,
	enKdsDabTunerParameter_idx_t_SLS_Signal_Lost			   =  161, enKdsDabTunerParameter_len_t_SLS_Signal_Lost				 =   1,
	enKdsDabTunerParameter_idx_t_Clear_RT_Delay				   =  162, enKdsDabTunerParameter_len_t_Clear_RT_Delay				 =   1,
	enKdsDabTunerParameter_idx_t_Clear_RT_AfterSyncLost		   =  163, enKdsDabTunerParameter_len_t_Clear_RT_AfterSyncLost				 =   1,
	enKdsDabTunerParameter_idx_LSMConfig_EID				   =  164, enKdsDabTunerParameter_len_LSMConfig_EID				 =   2,
	enKdsDabTunerParameter_idx_LSMConfig_EnsembleLabel		   =  166, enKdsDabTunerParameter_len_LSMConfig_EnsembleLabel				 =   16,
	enKdsDabTunerParameter_idx_LSMConfig_EnsShortLabelMask	   =  182, enKdsDabTunerParameter_len_LSMConfig_EnsShortLabelMask				 =   2,
	enKdsDabTunerParameter_idx_LSMConfig_SID				   =  184, enKdsDabTunerParameter_len_LSMConfig_SID				 =   2,
	enKdsDabTunerParameter_idx_LSMConfig_ServiceLabel		   =  186, enKdsDabTunerParameter_len_LSMConfig_ServiceLabel				 =   16,
	enKdsDabTunerParameter_idx_LSMConfig_SrvcShortLabelMask	   =  202, enKdsDabTunerParameter_len_LSMConfig_SrvcShortLabelMask				 =   2,
	enKdsDabTunerParameter_idx_LSMConfig_CompID				   =  204, enKdsDabTunerParameter_len_LSMConfig_CompID				 =   2,
	enKdsDabTunerParameter_idx_LSMConfig_CompLabel			   =  206, enKdsDabTunerParameter_len_LSMConfig_CompLabel				 =   16,
	enKdsDabTunerParameter_idx_LSMConfig_CompShortLabelMask	   =  222, enKdsDabTunerParameter_len_LSMConfig_CompShortLabelMask				 =   2,
	enKdsDabTunerParameter_idx_LSMConfig_Channel			   =  224, enKdsDabTunerParameter_len_LSMConfig_Channel				 =   3,
	enKdsDabTunerParameter_idx_DabBandConfig                   =  227, enKdsDabTunerParameter_len_DabBandConfig                    =   1,
	enKdsDabTunerParameter_idx_CRC							   =  228, enKdsDabTunerParameter_len_CRC					 =   2
  
} tenKdsDabTunerParameter;                                                                                                       /* ------- */

const tU8 DAB_CONFIG_KDS_SET_DAB_TUNER_LEN = 230;
const tU8 DAB_CONFIG_VERSIONID_LEN     = 11; /* enKdsDabTunerParameter_len_Tag + enKdsDabTunerParameter_len_MajorVersion + enKdsDabTunerParameter_len_MinorVersion */
const tU8 DAB_CONFIG_NUMBER_OF_PRESETS =  6;
/*==========================================================================*/


struct trKdsDabTunerParameter : public tun_trKds
{
  tBool bKdsDataRead;
  tChar sTag[ 8];
  tChar sVersion[ 3];

  tU8   u8FeatureControl1;          /* Bit 0     SF (DAB/DAB)	            default = ON      */
                                    /* Bit 1     SF (DAB/FM)	            default = ON      */
                                    /* Bit 2     SF (FM/DAB		            default = OFF     */
                                    /* Bit 3-6   Reserved									  */                                    
                                    /* Bit 7     DMB-A						default = OFF     */

  tU8   u8FeatureControl2;          /* Bit 0     TPEG						default = OFF     */
                                    /* Bit 1     EPG						default = ON      */
                                    /* Bit 2     SLS						default = OFF     */
                                    /* Bit 3	 BWS						default = OFF     */
                                    /* Bit 4     Intellitext				default = OFF     */
                                    /* Bit 5     Journaline					default = OFF     */                                    
                                    /* Bit 6-7   Reserved									  */ 

  
  tU8   u8ServiceFollowingConfig;   /* Bit 0     SF AltFreq					default = ON	  */
                                    /* Bit 1     SF AltEnsemble				default = ON      */
                                    /* Bit 2     SF Hardlink				default = ON	  */
                                    /* Bit 3	 SF Search					default = ON      */
                                    /* Bit 4     Reserved									  */
                                    /* Bit 5     SF Sticky Component		default = OFF     */                                    
                                    /* Bit 6-7   Reserved									  */ 


  tU8   u8ConcealmentLevel;         /* concealment level									  */

  tU8   u8TsuMatrixDabFm[32];       /* TSU switch matrix DAB -> FM                            */
  tU8   u8TsuMatrixFmDab[32];       /* TSU switch matrix FM  -> DAB                           */

  tU8   u8DabFmSwitchBackDelay;		/* TSU time constants DabFmSwitchBackDelay                */
  tU8   u8DabFmSmoothingT_Up;		/* TSU time constants DabFmSmoothingT_Up                  */
  tU8   u8DabFmSmoothingT_Down;		/* TSU time constants DabFmSmoothingT_Down                */
  tU8   u8DabFmComparisionDelay;	/* TSU time constants DabFmComparisionDelay               */
  tU8   u8DabFmSmoothingT_Long	;	/* TSU time constants DabFmSmoothingT_Long	              */
  tU8   u8DabFmQualityThreshold	;	/* TSU time constants DabFmQualityThreshold	              */

  tU8   u8FmDABSwitchBackDelay;		/* TSU time constants FmDABSwitchBackDelay                */
  tU8   u8FmDABSmoothingT_Up;		/* TSU time constants FmDABSmoothingT_Up                  */
  tU8   u8FmDABSmoothingT_Down;		/* TSU time constants FmDABSmoothingT_Down                */
  tU8   u8FmDABPSIDChangeDelay;		/* TSU time constants FmDABPSIDChangeDelay                */
  tU8   u8FmDABSmoothingT_Long;		/* TSU time constants FmDABSmoothingT_Long	              */
  tU8   u8FmDABQualityThreshold;	/* TSU time constants FmDABQualityThreshold	              */

  tU8   u8TsuTimeConstantsDabFm[6]; /* TSU time constants                                   */
  tU8   u8TsuTimeConstantsFmDab[6]; /* TSU time constants                                   */

  tU8   u8RdmQualityConfigDecreaseSpeedFactorCount; 

  tU8    u8RdmQualityConfigDecreaseSpeedFactor[DAB_CONFIG_DEFAULT_QUALITY_DECREASE_SPEED_FACTOR_COUNT];    

  tU8   u8RdmQualityConfigMode;     /* "RDM Quality Config" quality strategy				  */
									/*[0x00] RMD strategy 0									  */
									/*[0x01] RMD strategy 1 (VW strategy)					  */
  tU8   u8RdmQualityConfigCount;    /* "RDM Quality Config" count of quality config bytes     */	

  tU8   u8Ensemble_RecIndicator;		   /* [0x00] Disable
											  [0x01] Enable Normal
											  [0x02] Enable Sticky Rx Indicator				*/

  tU8   u8Ensemble_RecIndicatorSyncDelay;	  /* [0x00] 0
										      [0x01] 1 
											  [0x02] 2 
											  [0x03] None */

  tU8   u8EnsembleVerAlg_TimeTickDivider;
  tU8   u8EnsembleVerAlg_TimePeriod;
  tU8   u8EnsembleVerAlg_MinFICBERTime;
  tU8   u8EnsembleVerAlg_MinFICBERLevel;
  tU8   u8EnsembleVerAlg_TargetCntrValue;
  tU8   u8EnsembleQualAlg_MinFICBERTim;
  tU8   u8EnsembleQualAlg_MinFICBERLevel;

  tU8   u8BGSnooping1;	                 /* 0 Enable
											1 Quick Hide Ensemble
											2 Real Unlearn Ensemble
											3 Use For Showing Not-Receiveable Indicator
											4 Use For Clearing Not-Receiveable Indicator
											5 - 7 (reserved)								*/

 tU8   u8BGSnooping_LearnFieldstrength;
 tU8   u8BGSnooping_UnLearnFieldstrength;
 tU8   u8BGSnooping_LearnCntrMaxValue;
 tU8   u8BGSnooping_LearnCntrValue;
 tU8   u8BGSnooping_UnLearnCntrValue;
 tU8   u8BGSnooping_UnLearnCntrMinValue;

 tU8   u8RdmQualityConfig[16];     /* "RDM Quality Config"                                 */

  tU8   u8AnnouncementConfig;			/* Bit 0  Resume interrupted announcemnet  default = OFF */
										/* Bit 1-7   reserved                                    */


  tU8   u8DabScanConfig;				/* Bit 0	SearchUp Scan											default = OFF  */
										/* Bit 1    Scan Station List										default = OFF  */ 
										/* Bit 2    Supress Double Services									default = OFF  */
										/* Bit 3    Use Min Signal Quality									default = OFF  */
										/* Bit 4    Use Min Audio Quality									default = OFF  */
										/* Bit 5    Use Min Play Duration									default = OFF  */
										/* Bit 6    Scan Forever											default = OFF  */
										/* Bit 7    Select last scanned service when LSM is not available   default = OFF  */   

 tU8 u8ScanMinSignalQualType;			/*[0x00] No DAB Signal
										[0x01] BER Very High
										[0x02] BER High
										[0x03] BER Medium
										[0x04] BER Good 
										[0x05] BER Very Good
										[0x06] Best Reception Quality*/

 tU8 u8ScanAudioThreshold; 
 tU8 u8ScanPlayDuration;

 tU8 u8EnsembleList_SortingType;		/*[0x00] By Label
									      [0x01] By ID
										  [0x02] By Frequency*/

 tU8 u8ServiceList_SortingType;			/*[0x00] By Label
										  [0x01] By ID*/

 tU8 u8ComponentList_SortingType;		/*[0x00] By Label
										  [0x01] By ID*/

 tU16 u16MaxStations;
 tU8 u8StationListConfig;				/* Bit0 -Components Included In Station List*/

 tU8 u8T_FMDABSwitchMuteDab;
 tU8 u8T_FMDABSwitchMuteDabPlus;
 tU8 u8T_FMDABSwitchMuteDmba;
 tU8 u8T_BG_LearnStart;
 tU8 u8T_Learn_Start_SF_Off;
 tU8 u8T_Man_Mode_SF_Off;
 tU8 u8T_Station_Select_SF_Off;
 tU8 u8T_SLS_Signal_Lost;
 tU8 u8T_Clear_RT_Delay;
 tU8 u8T_Clear_RT_After_Sync_Lost;
 
 tU16 u16LSMConfig_EID;
 tU8  u16LSMConfig_EnsembleLabel[16];
 tU16 u16LSMConfig_EnsembleShortLabelMask;
 tU16 u16LSMConfig_SID;
 tU8  u16LSMConfig_ServiceLabel[16];
 tU16 u16LSMConfig_ServiceShortLabelMask;
 tU16 u16LSMConfig_CompID;
 tU8  u16LSMConfig_ComponentLabel[16];
 tU16 u16LSMConfig_ComponentShortLabelMask; 
 tU8  u16LSMConfig_Channel[3];

 tU8   u8DabBandConfig;              /* Bit 0-3   DABFrequencyTable_1_Type     
											 [0x00] OFF 
											 [0x01] EU Band III (-N)
											 [0x02] EU Band III
											 [0x03] Canada L-Band
											 [0x04] Korea Band III
											 [0x05] China Band III
											 [0x06] reserved
											 [0x07] New Zealand Band III
											 [0x08] Download Table 1         */
                                     /* Bit 4-7   DABFrequencyTable_2_Type      
											 [0x00] OFF 
											 [0x01] L-Band
											 [0x02] reserved
											 [0x03] Download Table 2*/              
 tU16 u16CRC;

  //tU8   u8Pty;                      /* Bit 0-4   PTY                          default = 0   */
  //                                  /* Bit 5     DAB present                                */
  //                                  /* Bit 6-7   rfu                                        */

  //tU16  u16AnnoSelection;           /* Announcement Types                                   */
  //tU8   u8PtyPreset[6];             /* PTY preset 1-6                                       */
  //tU8   u8NoOfAsStation;            /* Number of stored autostore stations                  */
  //tU8   u8LastAsStation;            /* last selected autostore station                      */
  //tU8   u8AsPreset[6];              /* AS preset 1-6                                        */
  //tU16  u16AnnoFlags;               /* announcement new flag                                */
  //tU8   u8AnnoSource;               /* announvement source  default = current service       */
  //tU8   u8ConcealmentLevel;         /* concealment level                                    */
  //tU32  u32LSMFrequency;            /* last service frequency                               */
  //tU32  u32LSMAsid;                 /* last service ASID                                    */
  //tU8   u8LSMPreset;                /* last preset                                          */
  //tU16  u16AnnoSupport;             /* announvement support                                 */
  //tU32  u32ServiceLabelInfo;        /* service label info                                   */
  //tU8   u8ServiceLabel[16];         /* service label                                        */
  //tU32  u32EnsembleLabelInfo;       /* ensemble label info                                  */
  //tU8   u8EnsembleLabel[16];        /* ensemble label                                       */
  //tU16  u16EnsembleId;              /* ensemble EID                                         */


 

  //tU8   u8RdmQualityConfigMode;     /* "RDM Quality Config" quality strategy                */
  //tU8   u8RdmQualityConfigCount;    /* "RDM Quality Config" count of quality config bytes   */
  //tU8   u8RdmQualityConfigDecreaseSpeedFactorCount; 
  //tU8    u8RdmQualityConfigDecreaseSpeedFactor[DAB_CONFIG_DEFAULT_QUALITY_DECREASE_SPEED_FACTOR_COUNT];    
  //tU8   u8RdmQualityConfig[16];     /* "RDM Quality Config"                                 */
  //tU8   u8DabMode;                  /* enabled support of DAB, DAB+, DMB-A                  */
  //tU16  au16ExpertApi[3];           /* default expert api values for testmode               */
  //tU8   u8RdmSignalQualityLevel;    /* "RDM Signal Quality Level"                */

  //tU8 u8MuteRampAttForSeek;			/* Mute Ramp Attenuation  for Seek */
  //tU8 u8MuteRampAttForStationChg;	/* Mute Ramp Attenuation  for Absolute Station change */
  //tU8 u8MuteRampAttForDirectFreq;   /* Mute Ramp Attenuation  for Direct Frequency */
  //tU8 u8MuteRampAttForManualTune;   /* Mute Ramp Attenuation  for Manual Tuning */
  //tU8 u8TimeToReleaseAtt;   		/* Mute Ramp Attenuation  for Manual Tuning */

  trKdsDabTunerParameter( tVoid);

  virtual tVoid vSetToDefaultValues( tVoid);
  virtual tVoid vParse( tPU8 pu8KdsBuffer, tU32 u32Length);
  virtual tVoid vTrace( tVoid);
};

    struct trDabConfig {
        tU32 u32CcaPrio;
        tU32 u32DispatcherPrio;
        tU32 u32DispatcherStackSize;
        tU32 u32IpnPrio;
        tU32 u32IpnStackSize;
        tBool bDabModulePresent;
        tBool bDabEnable;
        trDabConfig();
    };

    class fc_dabtuner_config: 
        public DAB_Singleton <fc_dabtuner_config>
    {
        friend class DAB_Singleton<fc_dabtuner_config>;


    public:
        trDabConfig const &rGetConfig() {return rDabConfig;};
        tVoid vReloadConfigration();

        tBool bGetSfDabDabEnable() { return bGetKdsBit(m_rKdsDabTunerParameter.u8FeatureControl1,0);};
        tBool bGetSfDabFmEnable()  { return bGetKdsBit(m_rKdsDabTunerParameter.u8FeatureControl1,1);};
		tBool bGetSfFmDabEnable() { return bGetKdsBit(m_rKdsDabTunerParameter.u8FeatureControl1,2);};
        tBool bGetDmbAEnable() { return bGetKdsBit(m_rKdsDabTunerParameter.u8FeatureControl1,7);};

		tBool bGetDABEPGEnable(){return bGetKdsBit(m_rKdsDabTunerParameter.u8FeatureControl2,1);};
		tBool bGetDABSLSEnable(){return bGetKdsBit(m_rKdsDabTunerParameter.u8FeatureControl2,2);};
		/* LSM Config */
     /*   tU32 u32GetLastFreq() { return m_rKdsDabTunerParameter.u32LSMFrequency;};*/
        tU32 u32GetLastAsid() { return m_rKdsDabTunerParameter.u8FeatureControl1;};
        tU8 const *pcu8GetLastSrvLabel() {return m_rKdsDabTunerParameter.u16LSMConfig_ServiceLabel;};
        tU8 const *pcu8GetLastEnsLabel() {return m_rKdsDabTunerParameter.u16LSMConfig_EnsembleLabel;};
        tU32 u32GetLastEid() { return m_rKdsDabTunerParameter.u16LSMConfig_EID;};

		/*TSU Matrix*/
        tU8 const *pcu8GetTsuMatrixDabFm(){return m_rKdsDabTunerParameter.u8TsuMatrixDabFm;};
        tU8 const *pcu8GetTsuMatrixFmDab(){return m_rKdsDabTunerParameter.u8TsuMatrixFmDab;};
		tU8 const *pcu8GetTsuConstantsDabFm(){return m_rKdsDabTunerParameter.u8TsuTimeConstantsDabFm;};
        tU8 const *pcu8GetTsuConstantsFmDab(){return m_rKdsDabTunerParameter.u8TsuTimeConstantsFmDab;};
        tU8 *pcu8GetRdmQualityStrategy(){return &m_rKdsDabTunerParameter.u8RdmQualityConfigMode;};
        tU8 *pcu8GetRdmQualityConfigDataLength(){return &m_rKdsDabTunerParameter.u8RdmQualityConfigCount;};
        tU8 const *pcu8GetRdmQualityConfig(){return m_rKdsDabTunerParameter.u8RdmQualityConfig;};
		tU8 const *pcu8GetRdmQualitySpeedFactor(){return m_rKdsDabTunerParameter.u8RdmQualityConfigDecreaseSpeedFactor;};
        tU8 u8GetConcealmentLevel(){return m_rKdsDabTunerParameter.u8ConcealmentLevel;};
#ifdef FEATURE_PSA_RCC
		tU8 u8GetCountryZone(){return m_rKdsPSAEOLParamRadio.u8Radio_CountryZone ;};
		tU8 u8GetActiveBand(){return m_rKdsPSAEOLParamRadioNum.u8RadioNum_BandActivation ;};		
#endif


		tBool bGetRdsFollowStatus(){
			#ifdef FEATURE_PSA_RCC
				return m_rKdsPSAEOLParamRadio.bRadio_RdsFollowUp;
			#else
				return TRUE;
			#endif
		
		};



        tVoid vSetConcealmentLevel(tU8 u8ConcealmentLevel){m_rKdsDabTunerParameter.u8ConcealmentLevel= u8ConcealmentLevel;};
        tVoid vSetDabFmLinkingThreshold(tU8 u8DabFmLinkingThreshold){m_rKdsDabTunerParameter.u8DabFmQualityThreshold = u8DabFmLinkingThreshold;};
        tVoid vSetDabFmLinkBackDelay(tU8 u8DabFmLinkBackDelay){m_rKdsDabTunerParameter.u8DabFmSwitchBackDelay= u8DabFmLinkBackDelay;};
        tVoid vSetDabFmLinkBackThreshold(tU8 u8DabFmLinkBackThreshold){m_rKdsDabTunerParameter.u8FmDABQualityThreshold= u8DabFmLinkBackThreshold;};
        tVoid vSetDABFMLinkingBackValidationTime(tU8 u8DabFmLinkingBackValidationTime){m_rKdsDabTunerParameter.u8DabFmComparisionDelay= u8DabFmLinkingBackValidationTime;};
        tVoid vSetEnsembleQuality(tU8 u8EnsembleQuality){m_rKdsDabTunerParameter.u8RdmQualityConfig[8]= u8EnsembleQuality;
				m_rKdsDabTunerParameter.u8EnsembleQualAlg_MinFICBERLevel= u8EnsembleQuality;};
        tVoid vSetDABScanThreshold(tU8 u8ScanQs){m_rKdsDabTunerParameter.u8ScanMinSignalQualType= u8ScanQs;};
        tVoid vSetSfDabDabEnable(tBool bSet);
        tVoid vSetSfDabFmEnable(tBool bSet);
		tVoid vSetSfFmDabEnable(tBool bSet);
        tenConfigServiceLinking enReadSfMode();
        DAB_tenMtcMode enReadMtcMode();
		tBool bGetAlarmEnable() { return m_rKdsDabCodingData.bAlarmEnabled;};
		tU8 enGetFrequencyTable(){return m_u8FreqeuncyTable;};
		tU8 u8GetAnnoConfig(){return m_rKdsDabTunerParameter.u8AnnouncementConfig;};
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
        tBool bGetClearPreset(){return bClearPreset;};
#endif

#ifdef FEATURE_PSA_RCC
		tVoid vSetEOLBasedFrequencyTable(tU8 u8ActiveBand, tU8 u8CountryZone);
#endif
		tVoid vSetDabFrequencyTable_Part1(tU8 u8BandConfig);
		tVoid vSetDabFrequencyTable_Part2(tU8 u8Band1, tU8 u8Band2);
    tVoid vSetFrequencyTableBasedOnVariantSelection();
		tU8 u8GetDABScanThreshold(){return m_rKdsDabTunerParameter.u8ScanMinSignalQualType;};
	/*	tU8 u8GetTimeToReleaseAtt(){return m_rKdsDabTunerParameter.u8TimeToReleaseAtt;};*/
		tU8 u8GetOEMType() { return m_rKdsHWConfigurationString.u8OEMType;};
		tenFmDabHdTunerMode enGetFmDabHdTunerMode() { return m_rKdsHWConfigurationString.enFmDabHdTunerMode ;};
          //Configuration function for various KDS parameters
        tVoid vReadKDSConfig( tenKDSParameter enKDSParameter);
        tU8 u8NumberOfPresets()
        {
            return (tU8)(u8NumberofBanks * u8NumOfPresetsPerBank);
        }

        tU8 u8NumberOfPresetsPerBank()
        {
            return u8NumOfPresetsPerBank;
        }

        tVoid vTraceAll(){
            m_rKdsHWConfigurationString.vTrace();
            m_rKdsDabTunerParameter.vTrace();
        };
		
		inline tBool bIsSBRVariant() const {
			return m_bIsSBRVariant;
		}
		
    private:
        tVoid vGetRegistryValue (tU32 *pu32KeyData, char const *cszSection, char const *cszKey) const;
        fc_dabtuner_config();
        trDabConfig rDabConfig;
        tBool bGetKdsBit(tU8 u8StartByte, tU8 u8BitPos)const;
        tU8   u8GetKdsBits(tU8 u8KdsByte, tU8 u8BitPos, tU8 u8NumBits) const;
        tVoid vSetKdsBit(tU8 u8StartByte, tU8 u8BitPos) const;
		tVoid vReadKdsEolSysConfiguration1();
        tU8 u8NumberofBanks;
        tU8 u8NumOfPresetsPerBank;
        tU8 bMixedPresetList;
		
        trKdsHWConfigurationString  m_rKdsHWConfigurationString;
        trKdsDabTunerParameter      m_rKdsDabTunerParameter;
		trKdsCodingData m_rKdsDabCodingData;
		//trKdsCodingVariant m_rKdsDabCodingVariant;
#ifdef FEATURE_PSA_RCC		
		trKdsPSAEOLParamRadio	 m_rKdsPSAEOLParamRadio;
		trKdsPSAEOLParamRadioNum m_rKdsPSAEOLParamRadioNum;
#endif
		tU8							m_u8FreqeuncyTable;
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
        tBool                       bClearPreset;
#endif
		tBool m_bIsSBRVariant = false;
    };

} // namespace DAB
#endif
