/************************************************************************
* FILE:         fc_dabtuner_DiagLibHandler.cpp
* PROJECT:      g3g
* SW-COMPONENT: 
*----------------------------------------------------------------------
*
* DESCRIPTION: Demo Server diagnosis handler using the new diagnosis
* library.
* In order to use diaglib to provide diagnostic services, include a
* diaghandler object like this one in your component. Adjust it to your
* needs and create an instance of it in your main application. This class
* does not use traces by purpose in order not to pollute the important
* information for the diaglib-use. Look at the other parts of this
* demoserver to get an idea of the general concept of application, CCA-
* communication and trace functionality. This part concentrates on the
* diaglib.
* Parts that need modification by you are marked by special comments.
* depending on the file.
* - ccademo_diaghandler.x              => // +++ Comment +++
* - ccademo_IncreaseDecreaseHandler.x  => // +++ Comment +++
* - ccademo_main.x                     => // +++diaglib+++ comment
* - ccademo_msgtypes.h                 => // +++diaglib+++ comment
*              
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
* HISTORY:      
* Date      | Author                 | Modification
* 15.07.08  | CM-AI/PJ-FO45 Evers    | Inital version
* 21.08.08  | CM-AI/PJ-FO45 Evers    | Finalize and comment
* 05.06.09  | CM-DI/PJ-VW33 hassmann  | Adaption for dabtuner
*************************************************************************/
//-----------------------------------------------------------------------------
// defines
//-----------------------------------------------------------------------------

// +++ Look at the diaglib header file Version.h and
// change the major number here to the one given there +++
#define DAB_U8_EXPECTED_DIAGLIB_MAJOR 1

#include "fc_dabtuner_util.h"
#include "fc_dabtuner_main.h"
#include "fc_dabtuner_trace.h"

#include "fc_dabtuner_DiagLibHandler.h"
#include "fc_dabtuner_DiagLibSysSetListener.h"
#include "fc_dabtuner_DiagLibIoControlListener.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_SERVICE 
#include "trcGenProj/Header/fc_dabtuner_DiagLibHandler.cpp.trc.h"
#endif

using namespace DAB;

/*************************************************************************
*
* FUNCTION: Constructor
* 
* DESCRIPTION: Init Members and setup connection with diaglib
*
* PARAMETER: poCcademoMainApp Pointer to main application
*
* RETURNVALUE: n/a
*
*************************************************************************/
fc_dabtuner_DiagLibHandler::fc_dabtuner_DiagLibHandler():
        oDabServiceConfig(CCA_C_U16_SRV_DIAGLIB,
                          CCA_C_U16_APP_FC_DABTUNER,
                          "DiagLib_Srv",
                          1, 0, 0)
{
    m_poGWMain = OSAL_NULL;
	m_poFactory = OSAL_NULL;


    m_poDiagIF = OSAL_NULL;
	    m_poCcaIF = OSAL_NULL;
    //_poSysSetIF = OSAL_NULL;
    m_poSystemSetIF = OSAL_NULL;
    m_poIoControlIF = OSAL_NULL;


    // +++ Initializes listeners that you provide +++
    m_poSystemSetListener = OSAL_NULL;
    m_poIoControlListener = OSAL_NULL;
}

tVoid fc_dabtuner_DiagLibHandler::vInit() {
    m_poGWMain=fc_dabtuner_tclApp::instance();
#ifndef DABTUNER_UTEST
    m_poFactory = diaglib::tclDiagnosisInterfaceFactory::poGetInstance();


    if (m_poFactory != OSAL_NULL)
    {
        tU8 u8MajorVersion;
        tU8 u8MinorVersion;

        if (m_poFactory->bGetVersion(u8MajorVersion, u8MinorVersion) == TRUE)
        {
            // +++ Trace info diaglib version number if wanted +++
            if (u8MajorVersion != DAB_U8_EXPECTED_DIAGLIB_MAJOR)
            {
                // +++ Trace error/warning that major version differs from expected one +++
                ETG_TRACE_ERR(("fc_dabtuner_DiagLibHandler:constructor:Wrong Major-Vesion: Major=%d, Minor=%d",
                                u8MajorVersion, u8MinorVersion));
            }
        }
        else
        {
            // +++ Trace warning +++
            ETG_TRACE_ERR(("fc_dabtuner_DiagLibHandler:constructor:Could not get ServiceVersion"));
        }

		
        // +++ Provide pointer to your Main application that derives from ail_tclAppInterface as tVoid* +++
        m_poDiagIF = m_poFactory->poCreateDiagnosisInterface( diaglib::EN_PROTOCOL_CCA, static_cast<tVoid*>(m_poGWMain) );

    }
    else
    {
        // +++ Trace error +++
        ETG_TRACE_ERR(("fc_dabtuner_DiagLibHandler:constructor:Could not create m_poFactory"));
    }

	
    if (m_poDiagIF != OSAL_NULL)
    {
        m_poDiagIF->bGetInterface<diaglib::tclCCAServiceIF>(&m_poCcaIF);
        m_poDiagIF->bGetInterface<diaglib::tclSystemSetIF>(&m_poSystemSetIF);
        m_poDiagIF->bGetInterface<diaglib::tclIoControlIF>(&m_poIoControlIF);
    }
    else
    {
        ETG_TRACE_ERR(("fc_dabtuner_DiagLibHandler:constructor:Null-pointer m_poDiagIF"));
    }

    // +++ Check if an interface failed creation +++
    if ( m_poCcaIF == OSAL_NULL
         || m_poSystemSetIF == OSAL_NULL 
         || m_poIoControlIF == OSAL_NULL)
    {
        ETG_TRACE_ERR(("fc_dabtuner_DiagLibHandler:constructor:Null-pointer m_poDiagIF=%p, m_poSystemSetIF=%p m_poIoControlIF=%p",
                       m_poCcaIF, m_poSystemSetIF, m_poIoControlIF));

    }
    else
    {
        // +++ Create your listener objects here +++
        m_poSystemSetListener = new fc_dabtuner_DiagLibSysSetListener(m_poSystemSetIF,m_poGWMain);
        if (OSAL_NULL == m_poSystemSetListener)
        {
            ETG_TRACE_ERR(("fc_dabtuner_DiagLibHandler:constructor:Null-pointer m_poSystemSetListener"));
        }

        m_poIoControlListener = new fc_dabtuner_DiagLibIoControlListener(m_poIoControlIF,m_poGWMain);
        if (OSAL_NULL == m_poIoControlListener)
        {
            ETG_TRACE_ERR(("fc_dabtuner_DiagLibHandler:constructor:Null-pointer m_poIoControlListener"));
        }


        // +++ Register your listeners here +++

        /* Default setting requests by TEF (Blaupunkt production) are handled by the 
           default listener we created before */
        m_poSystemSetIF->vRegisterListener( SYSSETID_TUNE_DAB,
                                            diaglib::EN_TYPE_DEFAULT_TEF,
                                            m_poSystemSetListener );
        m_poSystemSetIF->vRegisterListener( SYSSETID_TUNE_DAB,
                                            diaglib::EN_TYPE_DEFAULT_HMI,
                                            m_poSystemSetListener );
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
        m_poSystemSetIF->vRegisterListener( SYSSETID_TUNE_DAB,
                                                    diaglib::EN_TYPE_UM_RESET_ALL ,
                                                    m_poSystemSetListener );
        m_poSystemSetIF->vRegisterListener( SYSSETID_TUNE_DAB,
                                                    diaglib::EN_TYPE_UM_RESET_USER ,
                                                    m_poSystemSetListener );
        m_poSystemSetIF->vRegisterListener( SYSSETID_TUNE_DAB,
                                                    diaglib::EN_TYPE_UM_SWITCH_USER ,
                                                    m_poSystemSetListener );
#endif
          m_poSystemSetIF->vRegisterListener( SYSSETID_TUNE_DAB,
          										diaglib::EN_TYPE_CODING ,
                                                     m_poSystemSetListener );

        m_poIoControlIF->vRegisterListener( IOCONTROLID_DAB_TUNER_FREQUENCY_SET,
                                            m_poIoControlListener );
        m_poIoControlIF->vRegisterListener( IOCONTROLID_DAB_TUNER_SERVICEID_SET,
                                            m_poIoControlListener );
        m_poIoControlIF->vRegisterListener( IOCONTROLID_DIAG_REMOTE_CONTROL,
                                            m_poIoControlListener );

        vSubscribe<DAB::trMsgSrvResultRsp>();
        vSubscribe<DAB::trMsgSrvRspTestOperation>();
    }
#endif

}

/*************************************************************************
*
* FUNCTION:tVoid Destructor
* 
* DESCRIPTION: make sure we are unregistered
*
* PARAMETER:  void
*
* RETURNVALUE: void
*
*************************************************************************/
fc_dabtuner_DiagLibHandler::~fc_dabtuner_DiagLibHandler()
{
    ETG_TRACE_USR4(("fc_dabtuner_DiagLibHandler::~fc_dabtuner_DiagLibHandler"));

    m_poFactory = OSAL_NULL;

    m_poGWMain = OSAL_NULL;
#ifndef DABTUNER_UTEST
    m_poFactory = diaglib::tclDiagnosisInterfaceFactory::poGetInstance();

	
    if ( m_poFactory != NULL)
    {
        // +++ Destroy your diaglib instance by calling the following function and providind ONE of +++
        // +++ your interfaces. All interfaces are invalid after this function returned true. +++
        if ( (!m_poFactory->bDeleteConnectionAndFreeInterfaces( m_poDiagIF )) == TRUE )
        {
            ETG_TRACE_ERR(("fc_dabtuner_DiagLibHandler::~fc_dabtuner_DiagLibHandler: bDeleteConnectionAndFreeInterfaces failed"));
        }
    }
    else
    {
        // +++ Trace error +++
        ETG_TRACE_ERR(("fc_dabtuner_DiagLibHandler::~fc_dabtuner_DiagLibHandler: null-pointer m_poDiagIF"));
    }
#endif
    m_poDiagIF = OSAL_NULL;
    m_poCcaIF = OSAL_NULL;
    m_poSystemSetIF = OSAL_NULL;
    m_poIoControlIF = OSAL_NULL;
   
    m_poFactory = OSAL_NULL;



    // +++ Delete heap instances that are members of your object, for example listeners or worker threads/objects  +++
    if (OSAL_NULL != m_poSystemSetListener)
    {
        delete m_poSystemSetListener;
        m_poSystemSetListener = OSAL_NULL;
    }
    if (OSAL_NULL != m_poIoControlListener)
    {
        delete m_poIoControlListener;
        m_poIoControlListener = OSAL_NULL;
    }
}

trServerConfig &fc_dabtuner_DiagLibHandler::roGetConfig() {
    return oDabServiceConfig;
}

/*************************************************************************
*
* FUNCTION: vOnNewAppState(tU32 u32OldAppState, tU32 u32AppState)
* 
* DESCRIPTION: handles state - change messages form the spm
*
* PARAMETER: old state, new state
*
* RETURNVALUE: void
*
*************************************************************************/
tVoid fc_dabtuner_DiagLibHandler::vOnNewAppState(tU32 u32AppState)
{
    tU32 u32OldAppState=0;
    if (OSAL_NULL!= m_poGWMain) {
        u32OldAppState = m_poGWMain->_u32OldAppState;
    }
    ETG_TRACE_USR4(("fc_dabtuner_DiagLibHandler::vOnNewAppState: old=%d, new=%d",
                    u32OldAppState, u32AppState));

	
    if (u32OldAppState != u32AppState)
    {
        if ( m_poCcaIF == OSAL_NULL )
        {
            // +++ Trace Error +++
            ETG_TRACE_ERR(("fc_dabtuner_DiagLibHandler::vOnNewAppState: null-pointer m_poCcaIF"));
            return;
        }
        else
        {
            m_poCcaIF->vOnNewAppState(u32OldAppState, u32AppState);
        }
    }
	

    ETG_TRACE_USR4(("fc_dabtuner_DiagLibHandler::vOnNewAppState: END"));
}

/*************************************************************************
*
* FUNCTION: u16GetServiceId()
* 
* DESCRIPTION: Returns the service ID of the diag service
*
* PARAMETER: none
*
* RETURNVALUE: tU16 service ID
*
*************************************************************************/
tU16 fc_dabtuner_DiagLibHandler::u16GetServiceId()
{
    //ETG_TRACE_USR4(("fc_dabtuner_DiagLibHandler::u16GetServiceId"));
    tU16 _u16RetValue = 0;
	
    if ( m_poCcaIF == OSAL_NULL ) {
        // +++ Trace Error +++
        ETG_TRACE_ERR(("fc_dabtuner_DiagLibHandler::u16GetServiceId: null-pointer m_poCcaIF"));
    }
    else {
        _u16RetValue = m_poCcaIF->u16GetServiceId();
    }
	
    //ETG_TRACE_USR4(("fc_dabtuner_DiagLibHandler::u16GetServiceId==0x%x ", _u16RetValue));
    return _u16RetValue;
}

/*************************************************************************
*
* FUNCTION: vDispatchMessage_(amt_tclServiceData* poMessage)
* 
* DESCRIPTION: Dispatches the incoming message to the diaglib
*
* PARAMETER: poMessage Incoming message
*
* RETURNVALUE: none
*
*************************************************************************/
tVoid fc_dabtuner_DiagLibHandler::vDispatchMessage_(amt_tclServiceData *poMessage)
{
	(tVoid)poMessage;
    //ETG_TRACE_USR4(("fc_dabtuner_DiagLibHandler::vDispatchMessage_"));
		
    if ( m_poCcaIF == OSAL_NULL ) {
        // +++ Trace Error +++
        ETG_TRACE_ERR(("fc_dabtuner_DiagLibHandler::vDispatchMessage_: null-pointer m_poCcaIF"));
    }
    else {
        amt_tclServiceData oSrvData(poMessage);
        m_poCcaIF->vDispatchMessage(&oSrvData);
    }
	
    //ETG_TRACE_USR4(("fc_dabtuner_DiagLibHandler::vDispatchMessage_ END"));
}

/*************************************************************************
*
* FUNCTION: bGetServiceVersion ( tU16& rfu16MajorVersion, 
                                 tU16& rfu16MinorVersion, 
                                 tU16& rfu16PatchVersion)
* 
* DESCRIPTION: Reads out the internal service version
*
* PARAMETER: out rfu16MajorVersion - Major service version
*            out rfu16MinorVersion - Minor service version
*            out rfu16PatchVersion - Patch service version
*
* RETURNVALUE: tBool - true if version is returned correctly, false if not
*
*************************************************************************/
tBool fc_dabtuner_DiagLibHandler::bGetServiceVersion ( tU16& rfu16MajorVersion, 
                                                           tU16& rfu16MinorVersion, 
                                                           tU16& rfu16PatchVersion)
{
    ETG_TRACE_USR4(("fc_dabtuner_DiagLibHandler::bGetServiceVersion"));


    tBool _bRetValue = FALSE;
		
    if ( m_poCcaIF == OSAL_NULL )
    {
        // +++ Trace Error +++
        ETG_TRACE_ERR(("fc_dabtuner_DiagLibHandler::bGetServiceVersion: null-pointer m_poCcaIF"));
    }
    else
    {
        _bRetValue = m_poCcaIF->bGetServiceVersion(rfu16MajorVersion, 
                                                   rfu16MinorVersion, 
                                                   rfu16PatchVersion );
    }
	

    ETG_TRACE_USR4(("fc_dabtuner_DiagLibHandler::bGetServiceVersion: major=%u, minor=%u, path=%u, retVal=%u",
                    rfu16MajorVersion, rfu16MinorVersion, rfu16PatchVersion, _bRetValue));

    return _bRetValue;
}

tVoid fc_dabtuner_DiagLibHandler::vProcess(DAB::trMsgSrvResultRsp *poSrvRspDefSet) {

    ETG_TRACE_USR4(("fc_dabtuner_DiagLibHandler::vProcess(trMsgSrvResultRsp)"));

    if (m_poSystemSetListener==OSAL_NULL) {
        return;
    }
    m_poSystemSetListener->vOnSysSetResult(poSrvRspDefSet);
}

tVoid fc_dabtuner_DiagLibHandler::vProcess(DAB::trMsgSrvRspTestOperation *poSrvRspTestOperation) {
    ETG_TRACE_USR4(("fc_dabtuner_DiagLibHandler::vProcess(trMsgSrvRspTestOperation)"));
    if (m_poIoControlListener==OSAL_NULL) {
        return;
    }
    m_poIoControlListener->vOnIoControlResult(poSrvRspTestOperation);
}

