/*******************************************************************************
 * Modulename:         rfc3072_helper.c
 * Revision:           $Id: rfc3072_helper.c,v 1.28 2015/07/09 12:41:34 shz2hi Exp $
 * Module function:    helper Functions for RFC3072
 *
 * Project:            AaRS
 * Created:            22-11-2011, Gerold Bertram
 *
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2011
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is forbidden.
 *******************************************************************************
 */
// enable lints from CF4 delivery to handle sql-statements correct which come from adr3
//lint -e793
//Info 793: prio2: ANSI/ISO limit of 31 'significant characters in an external identifier' exceeded -- processing is unaffected

#ifdef CONFIG_COMPILE_DABLIB
#include "dablibtypes.h"
#include "DBAPITypes.h"
#endif

#include "rfc3072_helper.h"

#undef ENABLE_RFC3072_MODULE_TEST

#ifndef NULL
#define NULL 0
#endif

/*!
 *******************************************************************************
 @brief        <b>sdxf_chunk_get_id</b>

 Returns chunk ID

 @param[in]    sdxf_data, pointer to chunk
 @return       chunk identification
 *******************************************************************************
 */
uint16
sdxf_chunk_get_id(const uint8* sdxf_data )
{
  uint16 id = 0;

  if (sdxf_data != NULL)
    {
      id = (uint16)((*sdxf_data << 8) | *(sdxf_data + 1));
    }

  return id;
}

/*!
 *******************************************************************************
 @brief        <b>sdxf_chunk_get_flags</b>

 Returns chunk flags

 @param[in]    sdxf_data, pointer to chunk
 @return       chunk flags
 *******************************************************************************
 */
uint8
sdxf_chunk_get_flags(const uint8* sdxf_data)
{
  uint8 flags = 0;

  if (sdxf_data != NULL)
    {
      flags = *(sdxf_data + 2);
    }

  return flags;
}

/*!
 *******************************************************************************
 @brief        <b>sdxf_chunk_get_chunk_body_length</b>

 Returns the length of chunk data

 @param[in]    sdxf_data, pointer to chunk
 @return       chunk body length
 *******************************************************************************
 */
uint32
sdxf_chunk_get_chunk_body_length(uint8* sdxf_data)
{
  uint32 length;

  if (sdxf_data == NULL)
    {
      return (0);
    }

  /*check for short chunk*/
  if (RFC3072_FLAGS_SHORT & (sdxf_chunk_get_flags(sdxf_data)))
    {
      return 3;
    }
  else
    {
      length = (uint32)(((*(sdxf_data + 3)) << 16) | ((*(sdxf_data + 4)) << 8)
          | (*(sdxf_data + 5)));
      return length;
    }
}

/*!
 *******************************************************************************
 @brief        <b>sdxf_get_chunk_body_pointer</b>

 Returns a pointer to a chunk content

 @param[in]    sdxf_data, pointer to chunk
 @return       chunk body pointer
 *******************************************************************************
 */
uint8*
sdxf_get_chunk_body_pointer(uint8* sdxf_data)
{
  if (sdxf_data == NULL)
    {
      return (NULL);
    }

  /*check for short chunk*/
  if (RFC3072_FLAGS_SHORT & (sdxf_chunk_get_flags(sdxf_data)))
    {
      return sdxf_data + 3;
    }
  else
    {
      return sdxf_data + 6;
    }
}

/*!
 *******************************************************************************
 @brief        <b>sdxf_chunk_get_info</b>

 Returns a chunk

 @param[in]    sdxf_data, pointer to chunk
 @return       chunk info structure
 *******************************************************************************
 */
SDXFChunkInfo
sdxf_chunk_get_info(uint8* sdxf_data)
{
  SDXFChunkInfo chunk;

  if (sdxf_data != NULL)
    {
      chunk.chunkID = sdxf_chunk_get_id(sdxf_data);
      chunk.flags = sdxf_chunk_get_flags(sdxf_data);
      chunk.length = sdxf_chunk_get_chunk_body_length(sdxf_data);
      chunk.content = sdxf_get_chunk_body_pointer(sdxf_data);
    }
  else
    {
      chunk.chunkID = 0;
      chunk.flags = 0;
      chunk.length = 0;
      chunk.content = NULL;
    }
  return chunk;
}

/*!
 *******************************************************************************
 @brief        <b>sdxf_chunk_get_datatype</b>

 Returns a chunk

 @param[in]    sdxf_data, pointer to chunk
 @return       data type of chunk
 *******************************************************************************
 */
int
sdxf_chunk_get_datatype(uint8* sdxf_data)
{
  switch (sdxf_chunk_get_flags(sdxf_data))
    {
  case (RFC3072_FLAGS_SHORT | RFC3072_FLAGS_DATATYPE_NUMERIC):
    return RFC3072_FLAGS_DATATYPE_NUMERIC;
  case (RFC3072_FLAGS_DATATYPE_NUMERIC):
    return RFC3072_FLAGS_DATATYPE_NUMERIC;
  case (RFC3072_FLAGS_DATATYPE_BITSTRING):
    return RFC3072_FLAGS_DATATYPE_BITSTRING;
  case (RFC3072_FLAGS_DATATYPE_CHARACTER):
    return RFC3072_FLAGS_DATATYPE_CHARACTER;
  case (RFC3072_FLAGS_DATATYPE_STRUCTURE):
    return RFC3072_FLAGS_DATATYPE_STRUCTURE;
  default:
      ;
    }
  return 0;
}

/*!
 *******************************************************************************
 @brief        <b>sdxf_chunk_get_blob</b>

 Returns the blob (content) of the chunk

 @param[in]    sdxf_data, pointer to chunk
 @return       chunk blob pointer
 *******************************************************************************
 */
SDXFBlob
sdxf_chunk_get_blob(uint8 *chunk)
{
  SDXFBlob the_blob;

  if (chunk == NULL)
    {
      the_blob.content = NULL;
      the_blob.length = 0;
    }
  else
    {
      the_blob.content = sdxf_get_chunk_body_pointer(chunk);
      the_blob.length = sdxf_chunk_get_chunk_body_length(chunk);
    }

  return the_blob;
}

/*!
 *******************************************************************************
 @brief        <b>sdxf_chunk_get_numeric_sint32</b>

 Returns the numeric value of the chunk

 @param[in]    sdxf_data, pointer to chunk
 @return       int32 numeric value
 *******************************************************************************
 */
sint32
sdxf_chunk_get_numeric_sint32(uint8 *chunk)
{
  uint32 len;
  uint8 *content;
  int32 value = 0;

  uint32 i;

  if (chunk == NULL)
    {
      return -1;
    }

  len = sdxf_chunk_get_chunk_body_length(chunk);
  content = sdxf_get_chunk_body_pointer(chunk);

  if (len <= 4)
    {
      for (i = 0; i < len; i++)
        {
          value = (value << 8) | content[i];
        }
      return value;
    }
  return -1;
}

/*!
 *******************************************************************************
 @brief        <b>sdxf_chunk_get_numeric_uint32</b>

 Returns the numeric value of the chunk

 @param[in]    sdxf_data, pointer to chunk
 @return       uint32 numeric value
 *******************************************************************************
 */
uint32
sdxf_chunk_get_numeric_uint32(uint8 *chunk)
{
  return (uint32) sdxf_chunk_get_numeric_sint32(chunk);
}

/*!
 *******************************************************************************
 @brief        <b>sdxf_row_get_chunk</b>

 Return the pointer to the chunk of the sdxf row pointed
 by row identified by chunk_idx

 @param[in]    sdxf_data, pointer to table
 @param[in]    sdxf_length, table length
 @param[in]    row, pointer to row chunk
 @param[in]    column_idx, row number (0,1,2,3,4,...)
 @return       pointer to chunk
 *******************************************************************************
 */
uint8*
sdxf_row_get_chunk(const uint8* sdxf_data, uint32 sdxf_length, uint8* row,
    uint32 column_idx)
{
  SDXFChunkInfo row_chunk, the_chunk;

  if (sdxf_data == NULL)
    {
      return (NULL);
    }
  if (sdxf_length < 6)
    {
      return (NULL);
    }

  /*check ptr validity*/
  if ((sdxf_data < row) && (row < (sdxf_data + sdxf_length)))
    {
      row_chunk = sdxf_chunk_get_info(row);

      /*now we have the requested row, check for the chunk*/
      if ((row_chunk.flags & RFC3072_FLAGS_DATATYPE_MASK)
          == RFC3072_FLAGS_DATATYPE_STRUCTURE)
        { /*do it only if we have a structured row*/
          uint32 the_chunk_pos = 6;

          /*go through the list of chunks*/
          while (the_chunk_pos < (6+row_chunk.length))
            {
              the_chunk = sdxf_chunk_get_info(row + the_chunk_pos);

              /*check if this chunk contains the correct chunk id which least 12 bit are equal to chunk_column*/
              if (column_idx == (uint32) (the_chunk.chunkID & 0xfff))
			  {
                return ((uint8*) row + the_chunk_pos);
			  }

              if (the_chunk.flags & RFC3072_FLAGS_SHORT)
			  {
                the_chunk_pos += 6;
			  }
              else
			  {
                the_chunk_pos += (6 + the_chunk.length);
			  }
            }
        }
    }
  return NULL;
}

/*!
 *******************************************************************************
 @brief        <b>sdxf_table_get_chunk</b>

 Returns the ptr to chunk identified by row_idx and column_idx

 @param[in]    sdxf_data, pointer to table
 @param[in]    sdxf_length, table length
 @param[in]    row_idx, column number (0,1,2,3,4,...)
 @param[in]    column_idx, row number (0,1,2,3,4,...)
 @return       pointer to chunk
 *******************************************************************************
 */
uint8*
sdxf_table_get_chunk(uint8* sdxf_data, uint32 sdxf_length, uint32 row_idx,
    uint32 column_idx)
{
  SDXFChunkInfo table_chunk, row_chunk, the_chunk;
  uint32 cur_row = 0;
  uint32 position;

  if (sdxf_length < 6)
    {
      return (NULL);
    }
  if (sdxf_data == NULL)
    {
      return (NULL);
    }

  table_chunk = sdxf_chunk_get_info(sdxf_data);

  /*first: go through the rows*/
  if ((table_chunk.flags & RFC3072_FLAGS_DATATYPE_MASK)
      == RFC3072_FLAGS_DATATYPE_STRUCTURE)
    {
      position = 6;

      /*go through the rows*/
      do
        {
          if (position < sdxf_length)
            {
              /*get the row chunk*/
              row_chunk = sdxf_chunk_get_info(sdxf_data + position);

              if (cur_row < row_idx)
                {/*we need to step over this row*/
                  if (row_chunk.flags & RFC3072_FLAGS_SHORT)
                    position += 6;
                  else
                    position += (6 + row_chunk.length);

                  cur_row++;
                }
              else
                {/*we found the row indexed by row_idx, so leave the while loop*/
                  break;
                }
            }
          else
            {/*maybe someone took the wrong sdxf ptr or to large row_idx*/
              return NULL;
            }
        }
      while (1);

      position += 6;

      /*now we have the requested row, check for the chunk*/
      if ((row_chunk.flags & RFC3072_FLAGS_DATATYPE_MASK)
          == RFC3072_FLAGS_DATATYPE_STRUCTURE)
        { /*do it only if we have a structured row*/
          uint32 the_chunk_pos = position;

          /*go through the list of chunks*/
          while (the_chunk_pos < (position + row_chunk.length))
            {
              the_chunk = sdxf_chunk_get_info(sdxf_data + the_chunk_pos);

              /*check if this chunk contains the correct chunk id which least 12 bit are equal to chunk_column*/
              if (column_idx == (uint32) (the_chunk.chunkID & 0xfff))
                return ((uint8*) sdxf_data + the_chunk_pos);

              if (the_chunk.flags & RFC3072_FLAGS_SHORT)
                the_chunk_pos += 6;
              else
                the_chunk_pos += (6 + the_chunk.length);
            }
        }
    }
  return NULL;
}

/*!
 *******************************************************************************
 @brief        <b>sdxf_table_get_num_columns</b>

 Returns the number of comlumns

 @param[in]    sdxf_data, pointer to table
 @param[in]    sdxf_length, table length
 @return       number of columns in table
 *******************************************************************************
 */
uint32
sdxf_table_get_num_columns(uint8* sdxf_data, uint32 sdxf_length)
{
  /*mask all chunk ids in all rows with 0x0fff*/
  /*and find the largest value. This is the number of columns*/
  SDXFChunkInfo table_chunk, row_chunk, the_chunk;
  uint32 num_colmn = 0;
  uint32 position = 0;

  if (sdxf_length < 6)
    {
      return (0);
    }
  if (sdxf_data == NULL)
    {
      return (0);
    }
  table_chunk = sdxf_chunk_get_info(sdxf_data);

  if ((table_chunk.flags & RFC3072_FLAGS_DATATYPE_MASK)
      == RFC3072_FLAGS_DATATYPE_STRUCTURE)
    {
      position = 6;

      /*go through the list of chunks*/
      while (position < sdxf_length)
        {
          row_chunk = sdxf_chunk_get_info(sdxf_data + position);

          position += 6;

          /*now we have the row, in this row go through all chunks an find largest column*/
          if ((row_chunk.flags & RFC3072_FLAGS_DATATYPE_MASK)
              == RFC3072_FLAGS_DATATYPE_STRUCTURE)
            { /*do it only if we have a structured row*/
              uint32 the_chunk_pos = position;

              /*go through the list of chunks*/
              while (the_chunk_pos < (position + row_chunk.length))
                {
                  the_chunk = sdxf_chunk_get_info(sdxf_data + the_chunk_pos);

                  if (num_colmn < (uint32) (the_chunk.chunkID & 0xfff))
                    {
                      num_colmn = the_chunk.chunkID & 0xfff;
                    }

                  if (the_chunk.flags & RFC3072_FLAGS_SHORT)
                    {
                      the_chunk_pos += 6;
                    }
                  else
                    {
                      the_chunk_pos += (6 + the_chunk.length);
                    }
                }
            }

          if (RFC3072_DATA_ID(row_chunk.chunkID) != RFC3072_DATA_ID_HEADLINE)
            {
              /*
               * After headline is processed, no further column counting is needed
               */
              return num_colmn;
            }

          if ((row_chunk.flags & RFC3072_FLAGS_SHORT) == 0)
            {
              position += row_chunk.length;
            }

        }
    }
  return num_colmn;
}

/*!
 *******************************************************************************
 @brief        <b>sdxf_table_get_num_rows</b>

 Returns the number of rows

 @param[in]    sdxf_data, pointer to table
 @param[in]    sdxf_length, table length
 @return       number of rows in table
 *******************************************************************************
 */
uint32
sdxf_table_get_num_rows(uint8* sdxf_data, uint32 sdxf_length)
{
  SDXFChunkInfo table_chunk, row_chunk;
  uint32 num_rows = 0;
  uint32 position = 0;

  if (sdxf_length < 6)
    {
      return (0);
    }
  if (sdxf_data == NULL)
    {
      return (0);
    }

  table_chunk = sdxf_chunk_get_info(sdxf_data);

  if ((table_chunk.flags & RFC3072_FLAGS_DATATYPE_MASK)
      == RFC3072_FLAGS_DATATYPE_STRUCTURE)
    {
      position = 6;

      /*go through the list of chunks*/
      while (position < sdxf_length)
        {
          row_chunk = sdxf_chunk_get_info(sdxf_data + position);

          if (row_chunk.flags & RFC3072_FLAGS_SHORT)
            {
              position += 6;
            }
          else
            {
              position += (6 + row_chunk.length);
            }

          if (RFC3072_DATA_ID(row_chunk.chunkID) != RFC3072_DATA_ID_HEADLINE)
            {
              num_rows++;
            }
        }
    }
  return num_rows;
}

/*!
 *******************************************************************************
 @brief        <b>sdxf_table_get_row</b>

 Return the pointer to the row of the sdxf table pointed
 by sdxf data

 @param[in]    sdxf_data, pointer to table
 @param[in]    sdxf_length, table length
 @param[in]    row_idx, row index
 @return       pointer to row chunk
 *******************************************************************************
 */
uint8*
sdxf_table_get_row(uint8* sdxf_data, uint32 sdxf_length, uint32 row_idx)
{
  SDXFChunkInfo table_chunk, row_chunk;
  uint32 num_rows = 0;
  uint32 position = 0;

  if (sdxf_data == NULL)
    {
      return (NULL);
    }
  if (sdxf_length < 6)
    {
      return (NULL);
    }

  table_chunk = sdxf_chunk_get_info(sdxf_data);

  if ((table_chunk.flags & RFC3072_FLAGS_DATATYPE_MASK)
      == RFC3072_FLAGS_DATATYPE_STRUCTURE)
    {
      position = 6;

      /*go through the list of chunks*/
      while (position < sdxf_length)
        {
          row_chunk = sdxf_chunk_get_info(sdxf_data + position);

          /*
           * ignore headline rows
           */
          if (RFC3072_DATA_ID(row_chunk.chunkID) != RFC3072_DATA_ID_HEADLINE)
            {
              /*check if we have the wanted row*/
              if (row_idx == num_rows)
                {
                  return (sdxf_data + position);
                }
            }

          if (row_chunk.flags & RFC3072_FLAGS_SHORT)
            {
              position += 6;
            }
          else
            {
              position += (6 + row_chunk.length);
            }

          /*
           * ignore headline rows
           */
          if (RFC3072_DATA_ID(row_chunk.chunkID) != RFC3072_DATA_ID_HEADLINE)
            {
              num_rows++;
            }
        }
    }
  return NULL;
}

/*!
 *******************************************************************************
 @brief        <b>sdxf_table_get_headline_row</b>

 Return the pointer to the headline row of the sdxf table pointed
 by sdxf data

 @param[in]    sdxf_data, pointer to table
 @param[in]    sdxf_length, table length
 @return       pointer to headline row chunk
 *******************************************************************************
 */
uint8*
sdxf_table_get_headline_row(uint8* sdxf_data, uint32 sdxf_length)
{
  SDXFChunkInfo table_chunk, row_chunk;
  uint32 position = 0;

  if (sdxf_data == NULL)
    {
      return (NULL);
    }
  if (sdxf_length < 6)
    {
      return (NULL);
    }

  table_chunk = sdxf_chunk_get_info(sdxf_data);

  if ((table_chunk.flags & RFC3072_FLAGS_DATATYPE_MASK)
      == RFC3072_FLAGS_DATATYPE_STRUCTURE)
    {
      position = 6;
      row_chunk = sdxf_chunk_get_info(sdxf_data + position);

      if (RFC3072_DATA_ID(row_chunk.chunkID) == RFC3072_DATA_ID_HEADLINE)
        {
          return (sdxf_data + position);
        }
    }
  return NULL;
}

/*!
 *******************************************************************************
 @brief        <b>sdxf_labelblob_2_apilabel</b>

 Converts the label blob into the label structure

 @param[in]    label, pointer to APILabel
 @param[in]    label_blob, pointer to SDXF label blob
 @return       label valid or not
 *******************************************************************************
 */
uint32
sdxf_labelblob_2_apilabel(APILabel *label, SDXFBlob label_blob)
{
  uint32 i;

  if ((label_blob.length > 4) && (label != NULL))
    {
        for (i = 0; (i<(label_blob.length - 4))&&(i<(sizeof(label->b_label)-1)); i++)
        {
          label->b_label[i] = label_blob.content[i];
        }
        label->b_label[i] = '\0';
      label->w_8_char_abbreviation = (uint16)((label_blob.content[label_blob.length - 2]
          << 8) | label_blob.content[label_blob.length - 1]);

      label->b_text_control_field = ((label_blob.content[label_blob.length - 3]>>4)&0xf);
      label->b_charset = (label_blob.content[label_blob.length - 3]&0xf);
      label->b_label_status = 1;

      return 1;
    }
  else
    {
      if (label != NULL)
        {
          label->b_label_status = 0;
        }
      return 0;
    }
}

/*!
 *******************************************************************************
 @brief        <b>sdxf_apiserviceblob_2_apiservice</b>

 Converts the api_service blob into the api_service structure

 @param[in]    serv, pointer to APIService
 @param[in]    api_service_blob, pointer to SDXF service blob
 @return       success
 *******************************************************************************
 */
uint32
sdxf_apiserviceblob_2_apiservice(APIService *serv, SDXFBlob api_service_blob)
{
  if ((api_service_blob.length == 5) && (serv != NULL))
    {
      if (api_service_blob.content[0] == API_SERVICE_TYPE__DAB_DATA_SERVICE)
        {
          serv->l_data_service_id = (uint32)((api_service_blob.content[1] << 24)
              | (api_service_blob.content[2] << 16)
              | (api_service_blob.content[3] << 8)
              | api_service_blob.content[4]);
          return 1;
        }
      else if (api_service_blob.content[0]
          == API_SERVICE_TYPE__DAB_PROG_SERVICE)
        {
          serv->r_programme_service.b_programme_service_type
              = api_service_blob.content[1];
          serv->r_programme_service.b_ecc = api_service_blob.content[2];
          serv->r_programme_service.w_service_id = (uint16)((api_service_blob.content[3]
              << 8) | api_service_blob.content[4]);
          return 1;
        }

      return 0;
    }
  else
    {
      return 0;
    }
}

/*!
 *******************************************************************************
 @brief        <b>sdxf_apiserviceblob_2_apiservicetype</b>

 Converts the api_service blob into the api_service_type

 @param[in]    api_serv_type, api service type
 @param[in]    api_service_blob, pointer to SDXF service blob
 @return       success
 *******************************************************************************
 */
uint32
sdxf_apiserviceblob_2_apiservicetype(uint8 *api_serv_type,
    SDXFBlob api_service_blob)
{
  if ((api_service_blob.length == 5) && (api_serv_type != NULL))
    {
      *api_serv_type = api_service_blob.content[0];
      return (1);
    }
  else
    {
      return (0);
    }
}

/*!
 *******************************************************************************
 @brief        <b>sdxf_check_structure</b>

 Check the chunk structure

 @param[in]    sdxf_data        pointer to SDXF structure
 @param[in]    sdxf_length      length of SDXF structure
 @param[in]    printmode
 @return       -1 => error
 *******************************************************************************
 */
int
sdxf_check_structure(uint8* sdxf_data, uint32 sdxf_length)
{
  SDXFChunkInfo this_chunk;

  if (sdxf_length < 6)
    {
      return (-1);
    }

  if (sdxf_data == NULL)
    {
      return (-1);
    }

  /*evaluate the chunk*/
  this_chunk = sdxf_chunk_get_info(sdxf_data);

  /*check if we have subsequent chunks inside*/
  if ((this_chunk.flags & RFC3072_FLAGS_DATATYPE_MASK)
      == RFC3072_FLAGS_DATATYPE_STRUCTURE)
    {
      /*go through the list of chunks*/
      while ((sdxf_data + sdxf_length) > this_chunk.content)
        {
          uint32 this_content_len = sdxf_chunk_get_chunk_body_length(
              this_chunk.content);

          if (sdxf_chunk_get_flags(this_chunk.content) & RFC3072_FLAGS_SHORT)
            {
              sdxf_check_structure(this_chunk.content, 6);
              this_chunk.content += 6;
            }
          else
            {
              sdxf_check_structure(this_chunk.content, this_content_len + 6);
              this_chunk.content += (this_content_len + 6);
            }
        }
    }
  else
    { /*we have found the elementary content*/

    }

  return 0;
}

/*!
 *******************************************************************************
@brief        <b>sdxf_print_chunk_strlen</b>

*******************************************************************************
*/
uint32 sdxf_print_chunk_strlen( char* str, uint32 maxlen )
{
	uint32 len = 0;
	while( (*str++!=0)&&(len<maxlen) )
	{
		len++;
	}
	return len;
}

/*!
*******************************************************************************
@brief        <b>sdxf_print_chunk__print_value</b>

*******************************************************************************
*/
uint32 sdxf_print_chunk__print_value( tSDXFPrintChunkConfig* config, tSDXFPrintColumnFormatConfig* format_hint, uint8* chunk )
{
	uint32 len = 0;
	int chunk_data_type = sdxf_chunk_get_datatype(chunk);
	switch (chunk_data_type)
	{
	case RFC3072_FLAGS_DATATYPE_CHARACTER:
		{		
			char* string = (char*) sdxf_get_chunk_body_pointer(chunk);
			len = sdxf_chunk_get_chunk_body_length(chunk);
			{
				uint32 i;
				for(i=0;(i<len)&&(i<(config->tmp_buffer_size-1));i++)
				{
					if((string[i]>=' ')&&(string[i]<=127))
					{
						config->tmp_buffer[i] = string[i];
					}
					else
					{
						config->tmp_buffer[i] = '.';
					}
				}
				config->tmp_buffer[i] = 0;
			}
			break;
		}
	case RFC3072_FLAGS_DATATYPE_NUMERIC:
		{		
			uint32 value = sdxf_chunk_get_numeric_uint32(chunk);
			boolean special_format_defined = FALSE;
			if( format_hint != NULL )
			{
				switch( format_hint->mode )
				{
				case COLUMN_FORMAT_MODE_SIGNED_DEZIMAL:
					sprintf(config->tmp_buffer,"%d",value);
					special_format_defined = TRUE;
					break;
				case COLUMN_FORMAT_MODE_UNSIGNED_DEZIMAL:
					sprintf(config->tmp_buffer,"%d",(uint32)value);
					special_format_defined = TRUE;
					break;
				case COLUMN_FORMAT_MODE_UINT8_HEX:
					sprintf(config->tmp_buffer,"0x%02x",value);
					special_format_defined = TRUE;
					break;
				case COLUMN_FORMAT_MODE_UINT16_HEX:
					sprintf(config->tmp_buffer,"0x%04x",value);
					special_format_defined = TRUE;
					break;
				case COLUMN_FORMAT_MODE_UINT32_HEX:
					sprintf(config->tmp_buffer,"0x%08x",value);
					special_format_defined = TRUE;
					break;
				case COLUMN_FORMAT_MODE_AUTO_HEX:
					if(value < 0x100)
					{
						sprintf(config->tmp_buffer,"0x%02x",value);
					}
					else if(value < 0x10000)
					{
						sprintf(config->tmp_buffer,"0x%04x",value);
					}
					else if(value < 0x1000000)
					{
						sprintf(config->tmp_buffer,"0x%06x",value);
					}
					else
					{
						sprintf(config->tmp_buffer,"0x%08x",value);
					}
					special_format_defined = TRUE;
					break;
				default:
					config->tmp_buffer[0] = 0;
					break;
				}
			}
			if( special_format_defined == FALSE )
			{
				if(value <= 9)
				{
					sprintf(config->tmp_buffer,"%d",value);
				}
				else if(value < 0x100)
				{
					sprintf(config->tmp_buffer,"%u(0x%02x)",value,value);
				}
				else if(value < 0x10000)
				{
					sprintf(config->tmp_buffer,"%u(0x%04x)",value,value);
				}
				else
				{
					sprintf(config->tmp_buffer,"%u(0x%08x)",value,value);
				}
			}
			len = sdxf_print_chunk_strlen(config->tmp_buffer,config->tmp_buffer_size-1);
			break;
		}
	case RFC3072_FLAGS_DATATYPE_BITSTRING:
		{		
			len = sdxf_chunk_get_chunk_body_length(chunk);
			if( len != 0 )
			{
				boolean special_format_defined = FALSE;
				uint8* pointer = sdxf_get_chunk_body_pointer(chunk);

				if( format_hint != NULL )
				{
					switch( format_hint->mode )
					{
					case COLUMN_FORMAT_MODE_DABLABEL:
						{
							uint32 i;
							for(i=0;(i<16)&&(i<config->tmp_buffer_size-1);i++)
							{
								if( (((uint8*)pointer)[i]<32) || (((uint8*)pointer)[i]>127) )
								{
									config->tmp_buffer[i] = '.';
								}
								else
								{
									config->tmp_buffer[i] = pointer[i];
								}
							}
							config->tmp_buffer[16] = 0;
						}
						special_format_defined = TRUE;
						break;
					default:
						break;
					} /* switch( format_hint->mode ) */
				} /* if( format_hint != NULL ) */

				if( special_format_defined == FALSE ) 
				{
					char* s = config->tmp_buffer;
					s += sprintf(config->tmp_buffer,"0x");
					{
						uint32 i;
						for(i=0;(i<len)&&((config->tmp_buffer_size-(s-config->tmp_buffer)-1)>3);i++)
						{
							s += sprintf(s,
								"%02x",pointer[i]);
						}
					}
				} /* if( special_format_defined == FALSE ) */
				len = sdxf_print_chunk_strlen(config->tmp_buffer,config->tmp_buffer_size);
			} /* if( len != 0 ) */
			break;
		}
	case RFC3072_FLAGS_DATATYPE_STRUCTURE:
		{		
			len = 0;
			break;
		}
	default:
		break;
	}
	return len;
}

/*!
*******************************************************************************
@brief        <b>sdxf_print_lineappend</b>

*******************************************************************************
*/
void sdxf_print_lineappend( tSDXFPrintChunkConfig* config, uint32 column_index,uint32 len, char* str )
{
	uint32 chars = 0;
	if( column_index == 1 )
	{
		if( config->column_splitter != 0 )
		{
			chars += 1;
		}
	}
	chars += len;
	if( config->column_splitter != 0 )
	{
		chars += 1;
	}
	else
	{
		chars = (uint32)(chars + strlen(" "));
	}
	if(chars>=(config->line_buffer_size-config->line_buffer_used))
	{
		config->line_buffer_is_full = 1;
		return;
	}

	if( column_index == 1 )
	{
		if( config->column_splitter != 0 )
		{
			config->line_buffer[config->line_buffer_used] = config->column_splitter;
			config->line_buffer_used += 1;
			config->line_buffer[config->line_buffer_used] = 0;
		}
	}
	if( config->column[column_index].width != 0 )
	{
		uint32 i = len;
		while( i++ < config->column[column_index].width )
		{
			config->line_buffer[config->line_buffer_used] = ' ';
			config->line_buffer_used += 1;
			config->line_buffer[config->line_buffer_used] = 0;
		}
	}
	if(len != 0)
	{
		memcpy(&config->line_buffer[config->line_buffer_used],str,len);
		config->line_buffer_used += len;
		config->line_buffer[config->line_buffer_used] = 0;
	}
	if( config->column_splitter != 0 )
	{
		config->line_buffer[config->line_buffer_used] = config->column_splitter;
		config->line_buffer_used += 1;
		config->line_buffer[config->line_buffer_used] = 0;
	}
	else
	{
		config->line_buffer[config->line_buffer_used] = ' ';
		config->line_buffer_used += 1;
		config->line_buffer[config->line_buffer_used] = 0;
	}
}

/*!
*******************************************************************************
@brief        <b>sdxf_print_chunk</b>

*******************************************************************************
*/
char* sdxf_print_chunk( tSDXFPrintChunkConfig* config, uint32 l_result_length, uint8* r_result )
{
	boolean inc_state = FALSE;
	if( config->max_num_columns > SDXF_CONFIG_MAX_NUM_COLUMNS )
	{
		config->max_num_columns = SDXF_CONFIG_MAX_NUM_COLUMNS;
	} /* if( config->max_num_columns > SDXF_CONFIG_MAX_NUM_COLUMNS ) */

	if( ( config->line_buffer == NULL )||
		( config->line_buffer_size < SDXF_CONFIG_MIN_BUFFER_SIZE )||
		( config->tmp_buffer == NULL )||
		( config->tmp_buffer_size < SDXF_CONFIG_MIN_BUFFER_SIZE ) )
	{
		return NULL;
	}

	if( config->state == 0 )
	{
		int c;
		for(c=0;c<SDXF_CONFIG_MAX_NUM_COLUMNS;c++)
		{
			config->column[c].format_hint.mode = COLUMN_FORMAT_MODE_UNDEF;
			config->column[c].format_hint.name = NULL;
			config->column[c].width = 0;
		}

		config->table_num_columns = sdxf_table_get_num_columns(r_result,l_result_length);
	} /* if( config->state == 0 ) */

	if( config->table_num_columns == 0 )
	{
		return NULL;
	}

	strcpy(config->line_buffer,"");
	config->line_buffer_used = 0;
	config->line_buffer_is_full = 0;

	{
		boolean row_is_headline = FALSE;
		uint32 row_index = 0;
		uint8* row_chunk = NULL;

		/*
		* check for headline ...
		*/
		if(( config->state == 0 )||( config->state == 1 ))
		{
			inc_state = TRUE;
			row_is_headline = TRUE;
			row_chunk = sdxf_table_get_headline_row(r_result, l_result_length);
		}

		if(( config->state == 1 )&&( row_chunk == NULL )&&(row_is_headline==TRUE))
		{
			/*
			 * no headline available
			 */
			config->state = 2;
			row_is_headline = FALSE;
			inc_state = FALSE;
		}

		if( config->state == 2 )
		{
			config->state++;
			if( config->headline_splitter != 0 )
			{
				uint32 column_index;
				for (column_index=1; (column_index<=config->max_num_columns)&&(column_index<=config->table_num_columns)&&(config->line_buffer_used<config->line_buffer_size)&&(column_index<SDXF_CONFIG_MAX_NUM_COLUMNS); column_index++)
				{
					uint32 i = 0;
					uint32 n = config->column[column_index].width + 1;
					if(column_index==1) n++;
					while( (i++ < n)&&(config->line_buffer_used<config->line_buffer_size) )
					{
						config->line_buffer[config->line_buffer_used] = config->headline_splitter;
						config->line_buffer_used += 1;
						config->line_buffer[config->line_buffer_used] = 0;
					}
				}
				return config->line_buffer;
			}
		}

		if( config->state > 2 )
		{
			inc_state = TRUE;
			row_index = config->state-3;
		}

		if( row_chunk == NULL )
		{
			/*
			* check for EOL
			*/
			row_is_headline = FALSE;
			row_chunk = sdxf_table_get_row(r_result, l_result_length, row_index);

			if(( config->state > 1 )&&( row_chunk == NULL ))
			{
				return NULL;
			}
		}

		while( row_chunk != NULL )
		{
			/*
			* parse row
			*/
			uint32 column_index;
			for (column_index=1; (column_index<=config->max_num_columns)&&(column_index<=config->table_num_columns)&&(column_index<SDXF_CONFIG_MAX_NUM_COLUMNS); column_index++)
			{
				uint32 len = 0;
				uint8* chunk = sdxf_row_get_chunk(r_result, l_result_length, row_chunk, column_index);
				if (chunk == NULL)
				{
					len = 0;
				}
				else
				{
					int chunk_data_type = sdxf_chunk_get_datatype(chunk);

					config->tmp_buffer[0] = 0;

					switch (chunk_data_type)
					{
					case RFC3072_FLAGS_DATATYPE_CHARACTER:
						{		
							/*
							* search format hint ...
							*/
							if( row_is_headline == TRUE )
							{
								char* string = (char*) sdxf_get_chunk_body_pointer(chunk);
								len = sdxf_chunk_get_chunk_body_length(chunk);
								if( string != NULL )
								{
								len = sdxf_print_chunk_strlen(string,len);
								if(len>config->column[column_index].width)
								{
									config->column[column_index].width = len;
								}

								if( config->column[column_index].format_hint.mode == COLUMN_FORMAT_MODE_UNDEF )
								{
									boolean format_found = FALSE;
									tSDXFPrintColumnFormatConfig* hint = config->column_format_hint;
									while( (hint != NULL)&&(hint->mode !=COLUMN_FORMAT_MODE_UNDEF)&&(hint->name!=NULL) )
									{
										if( strlen(hint->name) == len )
										{
											if(strncmp(string,hint->name,strlen(hint->name))==0)
											{
												config->column[column_index].format_hint.mode = hint->mode;
												config->column[column_index].format_hint.name = hint->name;
												format_found = TRUE;
												break;
											}
										}
										hint++;
									}
									if( format_found == FALSE )
									{
										if( string != NULL )
										{
											if( strlen(string) > 0 )
											{
												switch( string[0] )
												{
												case 'd':
													config->column[column_index].format_hint.mode = COLUMN_FORMAT_MODE_SIGNED_DEZIMAL;
													config->column[column_index].format_hint.name = "int";
													format_found = TRUE;
													break;
												case 'u':
													config->column[column_index].format_hint.mode = COLUMN_FORMAT_MODE_UNSIGNED_DEZIMAL;
													config->column[column_index].format_hint.name = "uint";
													format_found = TRUE;
													break;
												case 'x':
													config->column[column_index].format_hint.mode = COLUMN_FORMAT_MODE_AUTO_HEX;
													config->column[column_index].format_hint.name = "hex";
													format_found = TRUE;
													break;
												default:
													break;
												}
											}
										}
									}
								}
							}
							}

							/**Vnd4kor: To solve prio 1 lint warning . Instead of sending NULL , Null pointer is passed
							to function sdxf_print_chunk__print_value*/
							tSDXFPrintColumnFormatConfig* poSDXFNullPointer = NULL;
							len = sdxf_print_chunk__print_value(
								config,
								( config->state > 1 )?&config->column[column_index].format_hint:poSDXFNullPointer,
								chunk);

							if(len>config->column[column_index].width)
							{
								config->column[column_index].width = len;
							}
						}
						break;
					case RFC3072_FLAGS_DATATYPE_NUMERIC:
					case RFC3072_FLAGS_DATATYPE_BITSTRING:
					case RFC3072_FLAGS_DATATYPE_STRUCTURE:
						if( row_is_headline == FALSE )
						{
							len = sdxf_print_chunk__print_value(
								config,
								&config->column[column_index].format_hint,
								chunk);

							if(len>config->column[column_index].width)
							{
								config->column[column_index].width = len;
							}
						}
						break;
					default:
						break;
					}

				} /* if (chunk != NULL) */
				if( config->state > 0 )
				{
					sdxf_print_lineappend( config, column_index,len, config->tmp_buffer );
				}
			} /* for (column_index = 1; column_index <= num_columns; column_index++) */

			if( config->state == 0 )
			{
				/*
				* next row ...
				*/
				row_chunk = sdxf_table_get_row(r_result, l_result_length, row_index++);
				row_is_headline = FALSE;
			}
			else
			{
				row_chunk = NULL;
			}
		} /* while( row_chunk != NULL ) */

	} /* if( config->state == 0 ) */

	if( inc_state == TRUE )
	{
		config->state++;
	}

	if( config->state == 1 )
	{
		return sdxf_print_chunk( config, l_result_length, r_result);
	}
	else if( config->line_buffer_used == 0 )
	{
		return NULL;
	}

	return config->line_buffer;
}

/*!
 *******************************************************************************
 @brief        <b>sdxf_db_query_example_callback_function</b>

 Just an example

 *******************************************************************************
 */
#ifdef ENABLE_RFC3072_MODULE_TEST
static void
sdxf_db_query_example_callback_function(/*ObjectRef*/int oref,
    uint8 b_response, uint8 b_tag, uint8 b_stmt_id, uint8 b_rfu1,
    uint16 w_num_result_tables, uint16 w_num_result_rows,
    uint16 w_num_result_columns, uint8 b_result_protocol,
    uint8 b_result_option, uint32 l_result_length, uint8* r_result)
{

	{
		tSDXFPrintColumnFormatConfig column_format_hint[] =
				SDXF_WELL_KNOWN_COLUMN_FORMAT_HINTS;

		char* line = NULL;
		char line_buffer[140];
		tSDXFPrintChunkConfig print_config;
		print_config.state = 0;
		print_config.print_header = TRUE;
		print_config.line_buffer = line_buffer;
		print_config.line_buffer_size = sizeof(line_buffer);
		print_config.column_format_hint = column_format_hint;
		print_config.line_header = "=> [";
		print_config.line_footer = "]";
		print_config.column_splitter = '|';
		print_config.max_num_columns = SDXF_CONFIG_MAX_NUM_COLUMNS;
		while( (line=sdxf_print_chunk(&print_config,l_result_length,r_result)) != NULL )
		{
			printf("%s\n",line);
		}
	}



  if ((b_response == /*DBAPI_R_QUERY__B_RESPONSE__PARA_OK*/1)
      && (b_result_protocol
          == /*DBAPI_R_QUERY__B_RESULT_PROTOCOL__PARA_SDXF_TABLE*/0x30)
      && (l_result_length != 0) && (r_result != NULL))
    {
      int row_index;
      int num_rows = sdxf_table_get_num_rows(r_result, l_result_length);
      for (row_index = 0; row_index < num_rows; row_index++)
        {
          uint8* row_chunk = sdxf_table_get_row(r_result, l_result_length,
              row_index);
          int column_index;
          int num_column =
              sdxf_table_get_num_columns(r_result, l_result_length);
          for (column_index = 1; column_index <= num_column; column_index++)
            {
              uint8* chunk = sdxf_row_get_chunk(r_result, l_result_length,
                  row_chunk, column_index);
              if (chunk != NULL)
                {
                  switch (sdxf_chunk_get_datatype(chunk))
                    {
                  case RFC3072_FLAGS_DATATYPE_NUMERIC:
                    {
                      uint32 value = sdxf_chunk_get_numeric_uint32(chunk);
                      /*
                       * do something with chunk_value
                       */
                      (void) value; /* make lint happy */
                    }
                    break;
                  case RFC3072_FLAGS_DATATYPE_BITSTRING:
                    {
                      uint8* pointer = sdxf_get_chunk_body_pointer(chunk);
                      /*
                       * do something with chunk_value
                       */
                      (void) pointer; /* make lint happy */
                    }
                    break;
                  case RFC3072_FLAGS_DATATYPE_CHARACTER:
                    {
                      char* string = (char*) sdxf_get_chunk_body_pointer(chunk);
                      /*
                       * do something with chunk_value
                       */
                      (void) string; /* make lint happy */
                    }
                    break;
                  case RFC3072_FLAGS_DATATYPE_STRUCTURE:
                    {
                      uint8* subtable = (char*) sdxf_get_chunk_body_pointer(
                          chunk);
                      /*
                       * do something with chunk_value
                       */
                      (void) subtable; /* make lint happy */
                    }
                    break;
                  default:
                    break;
                    }
                }
            }
        }
    }
}
#endif /* ENABLE_RFC3072_MODULE_TEST */

