/************************************************************************
 * FILE:        dabmeca_rdm.hpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of dabmeca_rdm
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      	| Author                       | Modification
				  
				
 *************************************************************************/

#ifndef DABMECA_RDM_HPP
#define DABMECA_RDM_HPP

#include "fc_dabtuner_util.h"
#include "dabdrv_meca.h"
#include "dabmeca_db.hpp"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_UTIL_MSG
#include "trcGenProj/Header/dabmeca_rdm.hpp.trc.h"
#endif

namespace DAB {
    // generic

#define DAB_BIT_ZERO_VALUE 1
#define DAB_BIT_TWO_VALUE 4
    typedef enum {
        DAB_enDir_None = 0x00,
        DAB_enDir_UP = 0x01,
        DAB_enDir_NEXT = 0x01,
        DAB_enDir_DOWN = 0x02,
        DAB_enDir_PREV = 0x02
    } DAB_tenDir;


    typedef enum {
        enMeca_RdmMode_GET = 0x00,
        enMeca_RdmMode_DAB_FG_AUDIO = 0x01,
        enMeca_RdmMode_DAB_DMB = 0x02,
        enMeca_RdmMode_DAB_BG_AUDIO = 0x03,  
        enMeca_RdmMode_DAB_INACTIVE = 0x04,
		enMeca_RdmMode_DAB_BG_ANNO_SURVEILLANCE = 0x05,
		enMeca_RdmMode_DAB_FG_ANNO_SURVEILLANCE = 0x06
    } tenMeca_RdmMode;

    // _DAB_C_RDM_SET_MODE
    struct trMeca_CRdmSetMode:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_SET_MODE;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 1;
            pRawOutput->au8Data[0]=(tU8)enRdmMode;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmSetMode:enRdmMode=%u", 
                            ETG_CENUM(tenMeca_RdmMode, enRdmMode)));
        };
        tenMeca_RdmMode enRdmMode;
    };

    // DAB_*_R_RDM_SET_MODE
    struct trMeca_RRdmSetMode:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 1) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enRdmMode=(tenMeca_RdmMode)pu8Data[0];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmSetMode: enRdmMode=%u", 
                            ETG_CENUM(tenMeca_RdmMode, enRdmMode)));
        };
        tenMeca_RdmMode enRdmMode;
    };

    typedef enum {
        enMeca_RdmClass_ENSEMBLE = 0x01,
        enMeca_RdmClass_SERVICE = 0x02,
        enMeca_RdmClass_SERVICE_COMPONENT = 0x02
    } tenMeca_RdmClass;


    // _DAB_C_RDM_SET_FILTER
    struct trMeca_CRdmSetFilter:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_SET_FILTER;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 3;
            pRawOutput->au8Data[0]=0; // rfu
            pRawOutput->au8Data[1]=(tU8)enRdmClass;
            pRawOutput->au8Data[2]=u8FilterId;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmSetMode:enRdmClass=0x%x u8FilterId=0x%x", 
                            ETG_CENUM(tenMeca_RdmClass, enRdmClass),
                            u8FilterId));
        };
        tenMeca_RdmMode enRdmClass;
        tU8 u8FilterId;
    };

    typedef enum {
        enMeca_RdmFilterState_SET_FILTER_REJECTED = 0x00,
        enMeca_RdmFilterState_SET_FILTER_OK = 0x01,
        enMeca_RdmFilterState_GET_FILTER = 0xFF
    } tenMeca_RdmFilterState;

    //DAB_*_R_RDM_SET_FILTER
    struct trMeca_RRdmSetFilter:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 4) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enRdmClass=(tenMeca_RdmMode)pu8Data[0];
            u8FilterId=pu8Data[1];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmSetFilter: enRdmClass=%u u8FilterId=0x%x enFilterState=%x", 
                            ETG_CENUM(tenMeca_RdmClass, enRdmClass),
                            u8FilterId,
                            ETG_CENUM(tenMeca_RdmFilterState, enFilterState)));
        };

        tenMeca_RdmMode enRdmClass;
        tU8 u8FilterId;
        tenMeca_RdmFilterState enFilterState;

    };


    //_DAB_C_RDM_GET_SIGNAL_QUALITY
    struct trMeca_CRdmGetSignalQuality:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_GET_SIGNAL_QUALITY;
            pRawOutput->enOpType = enOpType_GET;
            pRawOutput->u16DataLen = 1;
            pRawOutput->au8Data[0]=u8Cmd; // 0x00==SIGNAL_QUALITY
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmGetSignalQuality:u8Cmd=0x%x", 
                            u8Cmd));
        };
        tU8 u8Cmd;

    };

    typedef enum {
        enMeca_RdmSignalQualityIndicator_NO_DAB_SIGNAL = 0x00,
        enMeca_RdmSignalQualityIndicator_BER_VERY_HIGH = 0x01,
        enMeca_RdmSignalQualityIndicator_BER_HIGH = 0x02,
        enMeca_RdmSignalQualityIndicator_BER_MEDIUM = 0x03,
        enMeca_RdmSignalQualityIndicator_BER_GOOD = 0x04,
        enMeca_RdmSignalQualityIndicator_BER_VERY_GOOD = 0x05,
        enMeca_RdmSignalQualityIndicator_BEST = 0x06
    } tenMeca_RdmSignalQualityIndicator;

    // DAB_*_R_RDM_GET_SIGNAL_QUALITY
    struct trMeca_RRdmGetSignalQuality:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            if (rMostHdr._u16MecaLen != 2) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            u8Cmd=pu8Data[0];
            enQualityIndicator=(tenMeca_RdmSignalQualityIndicator)pu8Data[1];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmGetSignalQuality: u8Cmd=0x%x enQualityIndicator=%u", 
                            u8Cmd,
                            ETG_CENUM(tenMeca_RdmSignalQualityIndicator, enQualityIndicator)));
        };

        tU8 u8Cmd;
        tenMeca_RdmSignalQualityIndicator enQualityIndicator;

    };

    typedef enum {
        tenMeca_RdmQualityConfigCmd_GET=0x00,
        tenMeca_RdmQualityConfigCmd_SET=0x01
    } tenMeca_RdmQualityConfigCmd;

    //_DAB_C_RDM_QUALITY_CONFIG
    struct trMeca_CRdmQualityConfig:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_QUALITY_CONFIG;
            pRawOutput->enOpType = enOpType_SETGET;
            pRawOutput->u16DataLen = (tU16)(u8DataFieldLen + 4);
            pRawOutput->au8Data[0]= (tU8)enMecaQualityConfigCmd;
            pRawOutput->au8Data[1]= u8QualityStrategy; // 0x00==default
            pRawOutput->au8Data[2]= 0;                // rfu
            pRawOutput->au8Data[3]= u8DataFieldLen;
            OSAL_pvMemoryCopy(&pRawOutput->au8Data[4],au8DataField, u8DataFieldLen<20 ? u8DataFieldLen:20);
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmQualityConfig: u8DataFieldLen=%d u8QualityStrategy=%d au8DataField=%*p", 
                            u8DataFieldLen,
                            u8QualityStrategy,
                            ETG_LIST_LEN(u8DataFieldLen),ETG_LIST_PTR_T8(au8DataField)));

        };
        tenMeca_RdmQualityConfigCmd enMecaQualityConfigCmd;
        tU8 u8QualityStrategy;
        tU8 u8DataFieldLen;
        tU8 au8DataField[20];
    };


    // DAB_*_R_RDM_QUALITY_CONFIG
    struct trMeca_RRdmQualityConfig:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const * pu8Data) {
            _rMostHdr=rMostHdr;
            u8QualityStrategyCmd=pu8Data[0];
            u8QualityStrategy=pu8Data[1];
            //pu8Data[2]; //rfu
            u8DataFieldLen=pu8Data[3];

            ETG_TRACE_USR1(("  trMeca_RRdmQualityConfig: u8DataFieldLen=%d rMostHdr._u16MecaLen=%d", 
                            u8DataFieldLen,
                            rMostHdr._u16MecaLen));

            /*if (rMostHdr._u16MecaLen != 4+u8DataFieldLen) {
                return FALSE;
            }*/

            OSAL_pvMemoryCopy(au8DataField, &pu8Data[4], u8DataFieldLen);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmQualityConfig: u8DataFieldLen=%d u8QualityStrategy=%d au8DataField=%*p", 
                            u8DataFieldLen,
                            u8QualityStrategy,
                            ETG_LIST_LEN(u8DataFieldLen),ETG_LIST_PTR_T8(au8DataField)));
        };

        tU8 u8QualityStrategyCmd;
        tU8 u8QualityStrategy;
        tU8 u8DataFieldLen;
        tU8 au8DataField[20];

    };

    //_DAB_C_RDM_GET_FREQUENCY
    struct trMeca_CRdmGetFrequency:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_GET_FREQUENCY;
            pRawOutput->enOpType = enOpType_GET;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmGetSignalQuality"));
        };
    };

    // DAB_*_R_RDM_GET_FREQUENCY
    struct trMeca_RRdmGetFrequency:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            if (rMostHdr._u16MecaLen != 4) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            u32Frequency=DABDRV_GET_U32(&pu8Data[0]);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmGetSignalQuality: u32Frequency=%u", 
                            u32Frequency));
        };

        tU32 u32Frequency;

    };

    typedef enum {
        enMeca_RdmTuneCmd_STOP=0x01,
        enMeca_RdmTuneCmd_TUNE_ONE_STEP=0x02,
        enMeca_RdmTuneCmd_SEARCH=0x03,
        enMeca_RdmTuneCmd_SEARCH_ENDLESS=0x04,
         enMeca_RdmTuneCmd_ENS=0x05,
		enMeca_RdmTuneCmd_SKIM=0x06
    } tenMeca_RdmTuneCmd;

    //_DAB_C_RDM_TUNE
    struct trMeca_CRdmTune:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_TUNE;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 6;
            pRawOutput->au8Data[0]=(tU8)enTuneCmd;
            pRawOutput->au8Data[1]=(tU8)enDir;
            DABDRV_SET_U32(&pRawOutput->au8Data[2],u32StartFrequency);
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmTune:enTuneCmd=%x, enDir=%u, u32StartFrequency=%u", 
                            ETG_CENUM(tenMeca_RdmTuneCmd, enTuneCmd),
                            ETG_CENUM(DAB_tenDir, enDir),
                            u32StartFrequency));
        };
        tenMeca_RdmTuneCmd enTuneCmd;
        DAB_tenDir enDir;
        tU32 u32StartFrequency;
    };

    typedef enum {
        enMeca_RdmTuneState_STOP=0x01,
        enMeca_RdmTuneState_TUNING=0x02,
        enMeca_RdmTuneState_SEARCHING=0x03,
        enMeca_RdmTuneState_SEARCHING_ENS=0x04,
		enMeca_RdmTuneState_SKIMMING=0x06,
        enMeca_RdmTuneState_NO_ENS_AVAILABLE=0x10,
        enMeca_RdmTuneState_SEARCHING_DONE=0x11,
        enMeca_RdmTuneState_TUNING_DONE=0x12
    } tenMeca_RdmTuneState;

    // DAB_*_R_RDM_TUNE
    struct trMeca_RRdmTune:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            if (rMostHdr._u16MecaLen != 2) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enTuneState = (tenMeca_RdmTuneState)pu8Data[0];
            enDir = (DAB_tenDir)enDir;
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmTune: enTuneState=%x enDir=%u", 
                            ETG_CENUM(tenMeca_RdmTuneState, enTuneState),
                            ETG_CENUM(DAB_tenDir, enDir)));
        };

        tenMeca_RdmTuneState enTuneState;
        DAB_tenDir enDir;

    };

    //_DAB_C_RDM_TUNE_FREQUENCY
    struct trMeca_CRdmTuneFrequency:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
        trMeca_CRdmTuneFrequency() {
            u32Frequency = 0;
        }
        trMeca_CRdmTuneFrequency(tU32 u32Frequency_) {
            u32Frequency = u32Frequency_;
        }
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_TUNE_FREQUENCY;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 4;
            DABDRV_SET_U32(&pRawOutput->au8Data[0],u32Frequency);
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmTuneFrequency:u32Frequency=%u", 
                            u32Frequency));
        };
        tU32 u32Frequency;
    };


    typedef enum {
        enMeca_RdmFrequencyState_FREQUENCY_INVALID= 0x00,
        enMeca_RdmFrequencyState_FREQUENCY_OK     = 0x01,
        enMeca_RdmFrequencyState_TUNING_DONE      = 0x12
    } tenMeca_RdmFrequencyState;

    // DAB_*_R_RDM_TUNE_FREQUENCY
    struct trMeca_RRdmTuneFrequency:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            if (rMostHdr._u16MecaLen != 1) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enFrequencyState = (tenMeca_RdmFrequencyState)pu8Data[0];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmTuneFrequency: enFrequencyState=%x", 
                            ETG_CENUM(tenMeca_RdmFrequencyState, enFrequencyState)));
        };
    
        tenMeca_RdmFrequencyState enFrequencyState;
    };

            struct trMeca_CRdmTuneFrequencyLabel:
                public  trMsgMecaOut {
                DAB_DISPATCH_IMPL
                trMeca_CRdmTuneFrequencyLabel() {
                	sfreqlabel = ("");
                }
                trMeca_CRdmTuneFrequencyLabel(string sfreqlabel_) {
                	sfreqlabel = sfreqlabel_;
                }
                virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
                    pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
                    pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_TUNE_FREQUENCY_LABEL;
                    pRawOutput->enOpType = enOpType_SET;
                    pRawOutput->u16DataLen = (tU16)(sfreqlabel.size()+1);//including 0 terminator
					ETG_TRACE_USR1(("  trMeca_CRdmTuneFrequencyLabel:sFrequencyLabel=%s",sfreqlabel.c_str()));
					for(int i=0; i<(int)pRawOutput->u16DataLen; i++)
                    {
                    	pRawOutput->au8Data[i]=sfreqlabel[i];
                    }
					pRawOutput->au8Data[pRawOutput->u16DataLen]='\0';
                }
                virtual tVoid vTrace() const {
                    ETG_TRACE_USR4(("  trMeca_CRdmTuneFrequencyLabel:sFrequencyLabel=%s",
                    		sfreqlabel.c_str()));
                };
                string sfreqlabel;
            };
            struct trMeca_RRdmTuneFrequencyLabel:
            public trMsgMecaInput {
            	DAB_DISPATCH_IMPL
                virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
                    if (rMostHdr._u16MecaLen != 1) {
                        return FALSE;
                    }
                    _rMostHdr=rMostHdr;
                    enFrequencyState = (tenMeca_RdmFrequencyState)pu8Data[0];
                    return TRUE;
                }
                virtual tVoid vTrace() const {
                    ETG_TRACE_USR1(("  trMeca_RRdmTuneFrequencyLabel: enFrequencyState=%x",
                                    ETG_CENUM(tenMeca_RdmFrequencyState, enFrequencyState)));
                };
                tenMeca_RdmFrequencyState enFrequencyState;
            };



    //_DAB_C_RDM_TUNE_EID
    struct trMeca_CRdmTuneEid:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_TUNE_EID;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 4;
            rEnsemble.vSerialize(&pRawOutput->au8Data[0]);
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmTuneEid: ens=0x%08x",
                            rEnsemble.u32GetID()));
        };
        trMecaEnsemble rEnsemble;
    };

    typedef enum {
        enMeca_RdmEnsembleState_ENSEMBLE_INVALID   = 0x00,
        enMeca_RdmEnsembleState_TUNING_ENSEMBLE    = 0x01,
        enMeca_RdmEnsembleState_ENSEMBLE_NOT_FOUND = 0x02,
        enMeca_RdmEnsembleState_TUNING_DONE        = 0x12
    } tenMeca_RdmEnsembleState;


    // DAB_*_R_RDM_TUNE_EID
    struct trMeca_RRdmTuneEid:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            if (rMostHdr._u16MecaLen != 5) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enEnsembleState = (tenMeca_RdmEnsembleState)pu8Data[0];
            rEnsemble.vParse(&pu8Data[1]);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmTuneEid: ens=0x%08x enEnsembleState=%x",
                            rEnsemble.u32GetID(),
                            ETG_CENUM(tenMeca_RdmEnsembleState, enEnsembleState)));
        };

        tenMeca_RdmEnsembleState enEnsembleState;
        trMecaEnsemble rEnsemble;

    };



    //_DAB_C_RDM_GET_EID
    struct trMeca_CRdmGetEid:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_GET_EID;
            pRawOutput->enOpType = enOpType_GET;
            pRawOutput->u16DataLen = 0;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmGetEid"));
        };
    };

    // DAB_*_R_RDM_GET_EID
    struct trMeca_RRdmGetEid:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            if (rMostHdr._u16MecaLen != 4) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            rEnsemble.vParse(&pu8Data[0]);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmGetEid:EID=0x%08x",
                            rEnsemble.u32GetID()));
        };

        trMecaEnsemble rEnsemble;

    };

    //_DAB_C_RDM_GET_FREQUENCY_LABEL
    struct trMeca_CRdmGetFrequencyLabel:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_GET_FREQUENCY_LABEL;
            pRawOutput->enOpType = enOpType_GET;
            pRawOutput->u16DataLen = 0;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmGetFrequencyLabel"));
        };
    };

    // DAB_*_R_RDM_GET_FREQUENCY_LABEL
    struct trMeca_RRdmGetFrequencyLabel:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            OSALUTIL_szSaveStringNCopy(sFrequencyLabel,(tCString)&pu8Data[0], DABTUN_FREQLABEL_BUFFER);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmGetFrequencyLabel:%s",
                            sFrequencyLabel)) 
                };

        tChar sFrequencyLabel[DABTUN_FREQLABEL_BUFFER];

    };


    typedef enum {
        enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE   = 0x11,
        enMeca_RdmFrequencyTable_FREQUENCY_TABLE_CANADA   = 0x12,
        enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE_NEW   = 0x13,
        enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE_BAND3   = 0x14,
        enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE_BAND3_NEW   = 0x15,
        enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE_LBAND   = 0x16,
        enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE_KOREA   = 0x17,
        enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE_CHINA   = 0x18,
		enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE_CHINA_BAND3   = 0x19,
        enMeca_RdmFrequencyTable_FREQUENCY_TABLE_EUROPE_CHINA_LBAND   = 0x1a,
		enMeca_RdmFrequencyTable_FREQUENCY_TABLE_NEWZEALAND   = 0x1b,
        enMeca_RdmFrequencyTable_FREQ_LIST_EUROPE_BAND3_5A12D = 0x1c,
        enMeca_RdmFrequencyTable_FREQ_LIST_EUROPE_BAND3_5A12D_NEW = 0x1d
    } tenMeca_RdmFrequencyTable;




    //_DAB_C_RDM_SET_DAB_FREQUENCY_TABLE
    struct trMeca_CRdmSetDabFrequencyTable:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_SET_DAB_FREQUENCY_TABLE;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 1;
            pRawOutput->au8Data[0]=(tU8)enFrequencyTable;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmSetDabFrequencyTable enFrequencyTable=%x",
                            ETG_CENUM(tenMeca_RdmFrequencyTable, enFrequencyTable)));
        };
        tenMeca_RdmFrequencyTable enFrequencyTable;
    };


    // DAB_*_R_RDM_SET_DAB_FREQUENCY_TABLE
    struct trMeca_RRdmSetDabFrequencyTable:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 1) {
                return FALSE;
            }
            enFrequencyTable = (tenMeca_RdmFrequencyTable)pu8Data[0];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmSetDabFrequencyTable enFrequencyTable=%x",
                            ETG_CENUM(tenMeca_RdmFrequencyTable, enFrequencyTable)));
        };

        tenMeca_RdmFrequencyTable enFrequencyTable;
    };


    //_DAB_C_RDM_GET_DAB_FREQUENCY_TABLE
    struct trMeca_CRdmGetDabFrequencyTable:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        trMeca_CRdmGetDabFrequencyTable() {};
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_GET_DAB_FREQUENCY_TABLE;
            pRawOutput->enOpType = enOpType_GET;
            pRawOutput->u16DataLen = 0;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmGetDabFrequencyTable"));
        };
    };


    // DAB_*_R_RDM_GET_DAB_FREQUENCY_TABLE
    struct trMeca_RRdmGetDabFrequencyTable:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 1) {
                return FALSE;
            }
            enFrequencyTable = (tenMeca_RdmFrequencyTable)pu8Data[0];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmGetDabFrequencyTable enFrequencyTable=%x",
                            ETG_CENUM(tenMeca_RdmFrequencyTable, enFrequencyTable)));
        };

        tenMeca_RdmFrequencyTable enFrequencyTable;
    };


    //_DAB_C_RDM_GET_DAB_AUDIO_QUALITY
    struct trMeca_CRdmGetDabAudioQuality:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_GET_DAB_AUDIO_QUALITY;
            pRawOutput->enOpType = enOpType_GET;
            pRawOutput->u16DataLen = 1;
            pRawOutput->au8Data[0]=u8Cmd;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmGetDabAudioQuality u8Cmd=0x%x",
                            u8Cmd));
        };
        tU8 u8Cmd;
    };
    
    typedef enum {
        enMeca_RdmAudioQualityIndicator_NO_INFO_AVAIL   = 0x00, //    0%
        enMeca_RdmAudioQualityIndicator_NO_DAB_SIGNAL   = 0x01, //    0%
        enMeca_RdmAudioQualityIndicator_TOO_MANY_ERRORS = 0x02, //    2%
        enMeca_RdmAudioQualityIndicator_VERY_POOR       = 0x03, //    5%
        enMeca_RdmAudioQualityIndicator_POOR            = 0x04, //   10%
        enMeca_RdmAudioQualityIndicator_BETTER_THAN_04  = 0x05, //   20%
        enMeca_RdmAudioQualityIndicator_BETTER_THAN_05  = 0x06, //   30%
        enMeca_RdmAudioQualityIndicator_BETTER_THAN_06  = 0x07, //   40%
        enMeca_RdmAudioQualityIndicator_BETTER_THAN_07  = 0x08, //   50%
        enMeca_RdmAudioQualityIndicator_BETTER_THAN_08  = 0x09, //   60%
        enMeca_RdmAudioQualityIndicator_BETTER_THAN_09  = 0x0a, //   70%
        enMeca_RdmAudioQualityIndicator_BETTER_THAN_0A  = 0x0b, //   80%
        enMeca_RdmAudioQualityIndicator_BETTER_THAN_0B  = 0x0c, //   90%
        enMeca_RdmAudioQualityIndicator_BEST            = 0x0d   // 100%
    } tenMeca_RdmAudioQualityIndicator;

    // DAB_*_R_RDM_GET_DAB_AUDIO_QUALITY
    struct trMeca_RRdmGetDabAudioQuality:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 2) {
                return FALSE;
            }
            u8Cmd= pu8Data[0];
            enAudioQuality = (tenMeca_RdmAudioQualityIndicator)pu8Data[1];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmGetDabAudioQuality u8Cmd=%x enAudioQuality=%x",
                            u8Cmd,
                            ETG_CENUM(tenMeca_RdmAudioQualityIndicator, enAudioQuality)));
        };

        tU8 u8Cmd;
        tenMeca_RdmAudioQualityIndicator enAudioQuality;
    };


    //_DAB_C_RDM_GET_AUDIO_STATUS
    struct trMeca_CRdmGetAudioStatus:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_GET_AUDIO_STATUS;
            pRawOutput->enOpType = enOpType_GET;
            pRawOutput->u16DataLen = 1;
            pRawOutput->au8Data[0]=(tU8)enRdmMode;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmGetAudioStatus enRdmMode=%x",
                            ETG_CENUM(tenMeca_RdmMode, enRdmMode)));
        };
        tenMeca_RdmMode enRdmMode;
    };

    typedef enum {
        enMeca_RdmDrcMode_OFF      = 0x00,
        enMeca_RdmDrcMode_ON       = 0x01,
        enMeca_RdmDrcMode_RFU2     = 0x02,
        enMeca_RdmDrcMode_RFU3     = 0x03
    } tenMeca_RdmDrcMode;

    typedef enum {
        enMeca_RdmMusicSpeech_NOT_SIGNALLED = 0x00,
        enMeca_RdmMusicSpeech_MUSIC         = 0x01,
        enMeca_RdmMusicSpeech_SPEEC         = 0x02,
        enMeca_RdmMusicSpeech_RFU3          = 0x03
    } tenMeca_RdmMusicSpeech;


    typedef enum {
        enMeca_RdmAudioMode_STEREO       = 0x00,
        enMeca_RdmAudioMode_JOINT_STEREO = 0x01,
        enMeca_RdmAudioMode_DUAL_CHANNEL = 0x02,
        enMeca_RdmAudioMode_MONO         = 0x03,
		enMeca_RdmAudioMode_MultiChannel = 0x04
    } tenMeca_RdmAudioMode;

    typedef enum {
        enMeca_RdmDrcInfo_INVALID     = 0x00,
        enMeca_RdmDrcInfo_NO_INFO     = 0x01,
        enMeca_RdmDrcInfo_ALWAYS_ZERO = 0x02,
        enMeca_RdmDrcInfo_OK          = 0x03
    } tenMeca_RdmDrcInfo;

    typedef enum {
        enMeca_RdmAlgorithm_MPEG1 = 0x00,
        enMeca_RdmAlgorithm_MPEG2 = 0x01
    } tenMeca_RdmAlgorithm;

    typedef enum {
    	 enMeca_RdmChannelType_DAB = 0x01,
    	 enMeca_RdmChannelType_DABPLUS = 0x02,
    	 enMeca_RdmChannelType_DMB = 0x03
    }tenMeca_RdmChannelType;
    typedef enum {
    	 enMeca_MuteFlag_Unmute = 0,
    	 enMeca_MuteFlag_Mute = 1
    }tenMeca_MuteFlag;
    typedef enum {
      	 enMeca_MSFlag_Music = 0,
      	 enMeca_MSFlag_Speech = 1
      }tenMeca_MSFlag;
      typedef enum {
			 enMeca_EqualizerLevel_Min = 0,
			 enMeca_EqualizerLevel_Low = 1,
			 enMeca_EqualizerLevel_Medium = 2,
			 enMeca_EqualizerLevel_High = 3,
			 enMeca_EqualizerLevel_Max = 4
	   }tenMeca_EqualizerLevel;
	   typedef enum {
		     enMeca_RdmDMBAudioMode_Stereo = 0,
			 enMeca_RdmDMBAudioMode_JointStereo = 1,
			 enMeca_RdmDMBAudioMode_DualChannel = 2,
			 enMeca_RdmDMBAudioMode_Mono = 3,
			 enMeca_RdmDMBAudioMode_MultiChannel = 4
	   }tenMeca_RdmDMBAudioMode;
	   typedef enum {
			     enMeca_RdmCoreAudioCoding_MPEG1 = 0,
				 enMeca_RdmCoreAudioCoding_MPEG2 = 1,
				 enMeca_RdmCoreAudioCoding_HEAAC = 2,
				 enMeca_RdmCoreAudioCoding_BSAC = 3
		}tenMeca_RdmCoreAudioCoding;
		 typedef enum {
				 enMeca_RdmDRCStatus_Invalid = 0,
				 enMeca_RdmDRCStatus_No_DRC = 1,
				 enMeca_RdmDRCStatus_DRC_Zero = 2,
				 enMeca_RdmDRCStatus_DRC_NonZero = 3
		 }tenMeca_RdmDRCStatus;
		 typedef enum {
				 enMeca_RdmOriginalFlag_Copy = 0,
				 enMeca_RdmOriginalFlag_Originale = 1
		 }tenMeca_RdmOriginalFlag;
		 typedef enum {
				 enMeca_RdmCopyrightFlag_NoCopyright = 0,
				 enMeca_RdmCopyrightFlag_Copyright = 1
		 }tenMeca_RdmCopyrightFlag;
		 typedef enum {
				 enMeca_RdmDACRate_SampRate8 = 0,
				 enMeca_RdmDACRate_SampRate24 = 2,
				 enMeca_RdmDACRate_SampRate32 = 3,
				 enMeca_RdmDACRate_SampRate48 = 4,
				 enMeca_RdmDACRate_SampRate44 = 6
		 }tenMeca_RdmDACRate;
		 typedef enum {
				 enMeca_RdmSBRFlag_SBRNotUsed = 0,
				 enMeca_RdmSBRFlag_SBRUsed = 1
		 }tenMeca_RdmSBRFlag;
		 typedef enum {
				 enMeca_RdmAAC_ChannelMode_AACMono = 0,
				 enMeca_RdmAAC_ChannelMode_AACStereo = 1
		 }tenMeca_RdmAAC_ChannelMode;
		 typedef enum {
				 enMeca_RdmPSFlag_PSUnUsed = 0,
				 enMeca_RdmPSFlag_PSUsed = 1
		 }tenMeca_RdmPSFlag;
		 typedef enum {
				 enMeca_RdmMPEGSurround_None = 0,
				 enMeca_RdmMPEGSurround_MPEGSurround = 1
		 }tenMeca_RdmMPEGSurround;
    inline tU16 u16EtsBitRate2kbs(tU8 u8MecaBitRate) {
        switch (u8MecaBitRate) {
            case 1: return 32;
            case 2: return 48;
            case 3: return 56;
            case 4: return 64;
            case 5: return 80;
            case 6: return 96;
            case 7: return 112;
            case 8: return 128;
            case 9: return 160;
            case 10: return 192;
            case 11: return 224;
            case 12: return 256;
            case 13: return 320;
            case 14: return 384;
            default:return 0;
        }
    }
                

    struct DAB_rDMBAudioStatus{
		 tVoid vParse(tU8 const *pu8Data) {
			enChannelType=(tenMeca_RdmChannelType)pu8Data[0];
			u8SubChId = pu8Data[1];
			u8AudioBERLevel=pu8Data[2];
			u8FECCorrLevel=pu8Data[3];
			u8DMBConcealmentLevel=pu8Data[4];
			u8DMBProtectionLevel = pu8Data[5];
			enMuteFlag= (tenMeca_MuteFlag)pu8Data[6];
			u8HeaderCrcErrCnt = pu8Data[7];
			u8ScfCrcErrCnt = pu8Data[8];
			enEqualizerLevel = (tenMeca_EqualizerLevel)pu8Data[9];
			enMSFlag = (tenMeca_MSFlag)pu8Data[10];
			enDMBAudioMode = (tenMeca_RdmDMBAudioMode)pu8Data[11];
			enCoreAudioConfig = (tenMeca_RdmCoreAudioCoding)pu8Data[12];
			enDRCStatus = (tenMeca_RdmDRCStatus)pu8Data[13];
			u8DRCScaling=pu8Data[14];
			u16AudioBitrate=pu8Data[15];
			enOriginalFlag=(tenMeca_RdmOriginalFlag)pu8Data[17];
			enCopyrightFlag=(tenMeca_RdmCopyrightFlag)pu8Data[18];
			enDACRate=(tenMeca_RdmDACRate)pu8Data[19];
			enSBRFlag=(tenMeca_RdmSBRFlag)pu8Data[20];
			enAAC_ChannelMode=(tenMeca_RdmAAC_ChannelMode)pu8Data[21];
			enPSFlag=(tenMeca_RdmPSFlag)pu8Data[22];
			enMPEGSurround=(tenMeca_RdmMPEGSurround)pu8Data[23];
		}
		 tVoid vTrace() const {
			ETG_TRACE_USR1(("  trMeca_RRdmGetAudioStatus enChannelType=%d,u8SubChId=%d,enDMBAudioMode=%d,u16AudioBitrate=%d ",
				enChannelType,
				u8SubChId,
				enDMBAudioMode,
				u16AudioBitrate));
		}
    	tenMeca_RdmChannelType enChannelType;
    	tU8 u8SubChId;
    	tU8 u8AudioBERLevel;
    	tU8 u8FECCorrLevel;
    	tU8 u8DMBConcealmentLevel;
    	tU8 u8DMBProtectionLevel;
    	tenMeca_MuteFlag enMuteFlag;
    	tU8 u8HeaderCrcErrCnt;
    	tU8 u8ScfCrcErrCnt;
    	tenMeca_EqualizerLevel enEqualizerLevel;
    	tenMeca_MSFlag enMSFlag;
    	tenMeca_RdmDMBAudioMode enDMBAudioMode;
    	tenMeca_RdmCoreAudioCoding enCoreAudioConfig;
    	tenMeca_RdmDRCStatus enDRCStatus;
    	tU8 u8DRCScaling;
    	tU16 u16AudioBitrate;
    	tenMeca_RdmOriginalFlag enOriginalFlag;
    	tenMeca_RdmCopyrightFlag enCopyrightFlag;
    	tenMeca_RdmDACRate enDACRate;
    	tenMeca_RdmSBRFlag enSBRFlag;
    	tenMeca_RdmAAC_ChannelMode enAAC_ChannelMode;
    	tenMeca_RdmPSFlag enPSFlag;
    	tenMeca_RdmMPEGSurround enMPEGSurround;
    };
    struct DAB_rAudioStatus {
        tVoid vParse(tU8 const *pu8Data) {
            //enRdmMode=(tenMeca_RdmMode)pu8Data[0];
            enDrcMode = (tenMeca_RdmDrcMode)((pu8Data[0] & (DAB_b0 | DAB_b1)));
            enMusicSpeech=(tenMeca_RdmMusicSpeech)((pu8Data[0] & (DAB_b2 | DAB_b3)) >> 2);
            enAudioMode=(tenMeca_RdmAudioMode)((pu8Data[0] & (DAB_b4 | DAB_b5)) >> 4);
            enDrcInfo=(tenMeca_RdmDrcInfo)((pu8Data[0] & (DAB_b6 | DAB_b7)) >> 6);
            enAlgorithm = (tenMeca_RdmAlgorithm)(pu8Data[1] & DAB_b0);
            u16BitRateKbps= u16EtsBitRate2kbs((pu8Data[1]>>1) & 0x0f);
            bOriginal = (pu8Data[1] & DAB_b6) ? TRUE:FALSE;
            bCopyRight = (pu8Data[1] & DAB_b7) ? TRUE:FALSE;
        }
        tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmGetAudioStatus enDrcMode=%x enMusicSpeech=%x "
                            "enAudioMode=%x enDrcInfo=%x enAlgorithm=%x u16BitRateKbps=%x "
                            "bOriginal=%x bCopyRight=%x",
                            ETG_CENUM(tenMeca_RdmDrcMode, enDrcMode),
                            ETG_CENUM(tenMeca_RdmMusicSpeech, enMusicSpeech),
                            ETG_CENUM(tenMeca_RdmAudioMode, enAudioMode),
                            ETG_CENUM(tenMeca_RdmDrcInfo, enDrcInfo),
                            ETG_CENUM(tenMeca_RdmAlgorithm, enAlgorithm),
                            u16BitRateKbps,
                            bOriginal,
                            bCopyRight));
        
        }
        //tenMeca_RdmMode enRdmMode;
        tenMeca_RdmDrcMode enDrcMode;
        tenMeca_RdmMusicSpeech enMusicSpeech;
        tenMeca_RdmAudioMode enAudioMode;
        tenMeca_RdmDrcInfo enDrcInfo;
        tenMeca_RdmAlgorithm enAlgorithm;
        tU16 u16BitRateKbps;
        tBool bOriginal;
        tBool bCopyRight;
    };

    // DAB_*_R_RDM_GET_AUDIO_STATUS
    struct trMeca_RRdmGetAudioStatus:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 3) {
                return FALSE;
            }
            enRdmMode = (tenMeca_RdmMode)pu8Data[0];
            
            if(enRdmMode == enMeca_RdmMode_DAB_DMB){
            	rDMBAudioStatus.vParse(&pu8Data[3]);
            }
			else{
				rAudioStatus.vParse(&pu8Data[1]);
            }
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmGetAudioStatus:"));
            rAudioStatus.vTrace();
        };
    
        //tU8 u8Rfu;
        tenMeca_RdmMode enRdmMode;
        DAB_rAudioStatus rAudioStatus;
        DAB_rDMBAudioStatus rDMBAudioStatus;
    };



    //DAB_*_C_RDM_AUDIO_GET_ASID
    struct trMeca_CRdmAudioGetAsid:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_AUDIO_GET_ASID;
            pRawOutput->enOpType = enOpType_GET;
            pRawOutput->u16DataLen = 0;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmAudioGetAsid"));
        };
        tenMeca_RdmMode enRdmMode;
    };

    // DAB_*_R_RDM_AUDIO_GET_ASID
    struct trMeca_RRdmAudioGetAsid:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 4) {
                return FALSE;
            }
            rMecaProgrammeService.vParse(pu8Data);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmAudioGetAsid: sid=0x%08x",
                            rMecaProgrammeService.u32GetSID()));
        };
        trMecaProgrammeService rMecaProgrammeService;
    };


    //DAB_*_C_RDM_AUDIO_PLAY_ASID
    struct trMeca_CRdmAudioPlayAsid:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_AUDIO_PLAY_ASID;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 4;
            rMecaProgrammeService.vSerialize(&pRawOutput->au8Data[0]);
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmAudioPlayAsid sid=0x%08x",
                            rMecaProgrammeService.u32GetSID()));
        };
        trMecaProgrammeService rMecaProgrammeService;

    };

    typedef enum {
        enMeca_RdmServiceState_SERVICE_INVALID = 0x00,
        enMeca_RdmServiceState_SERVICE_SELECTION_DONE = 0x01,
		enMeca_RdmServiceState_SERVICE_NO_NEXTSERVICE = 0x02,
		enMeca_RdmServiceState_SERVICE_NO_PREVIOUSSERVICE = 0x03,
        enMeca_RdmServiceState_WAITING_FOR_SERVICE = 0x04,
        enMeca_RdmServiceState_PSID_NOT_AVAILABLE = 0x06
    } tenMeca_RdmServiceState;

    // DAB_*_R_RDM_AUDIO_PLAY_ASID
    struct trMeca_RRdmAudioPlayAsid:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 5) {
                return FALSE;
            }
            rMecaProgrammeService.vParse(pu8Data);
            enServiceState = (tenMeca_RdmServiceState) pu8Data[4];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmAudioPlayAsid: enServiceState=%d asid=0x%08x",
                            ETG_CENUM(tenMeca_RdmServiceState, enServiceState),
                            rMecaProgrammeService.u32GetSID()));
        };
        trMecaProgrammeService rMecaProgrammeService;
        tenMeca_RdmServiceState enServiceState;
    };


    typedef enum {
        enMeca_RdmServiceCommand_PLAY_FIRST    = 0x01,
        enMeca_RdmServiceCommand_PLAY_LAST     = 0x02,
        enMeca_RdmServiceCommand_PLAY_PREVIOUS = 0x03,
        enMeca_RdmServiceCommand_PLAY_NEXT     = 0x04
    } tenMeca_RdmServiceCommand;

    //DAB_*_C_RDM_AUDIO_PLAY_SERVICE
    struct trMeca_CRdmAudioPlayService:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    

        trMeca_CRdmAudioPlayService(tenMeca_RdmServiceCommand enServiceCommand_=enMeca_RdmServiceCommand_PLAY_FIRST) :
            enServiceCommand(enServiceCommand_)
        {}

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_AUDIO_PLAY_SERVICE;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 1;
            pRawOutput->au8Data[0]= (tU8)enServiceCommand;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmAudioPlayService enServiceCommand=%x",
                            ETG_CENUM(tenMeca_RdmServiceCommand, enServiceCommand)));
        };
        tenMeca_RdmServiceCommand enServiceCommand;

    };

    // DAB_*_R_RDM_AUDIO_PLAY_SERVICE
    struct trMeca_RRdmAudioPlayService:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 5) {
                return FALSE;
            }
            rMecaProgrammeService.vParse(pu8Data);
            enServiceState = (tenMeca_RdmServiceState) pu8Data[4];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmAudioPlayService: enServiceState=%d sid=0x%08x",
                            ETG_CENUM(tenMeca_RdmServiceState, enServiceState),
                            rMecaProgrammeService.u32GetSID()));
        };
        trMecaProgrammeService rMecaProgrammeService;
        tenMeca_RdmServiceState enServiceState;
    };


    typedef enum {
        enMeca_RdmComponentCommand_PLAY_FIRST    = 0x01,
        enMeca_RdmComponentCommand_PLAY_LAST     = 0x02,
        enMeca_RdmComponentCommand_PLAY_PREVIOUS = 0x03,
        enMeca_RdmComponentCommand_PLAY_NEXT     = 0x04
    } tenMeca_RdmComponentCommand;

    //DAB_*_C_RDM_AUDIO_PLAY_COMPONENT
    struct trMeca_CRdmAudioPlayComponent:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_AUDIO_PLAY_COMPONENT;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 1;
            pRawOutput->au8Data[0]= (tU8)enComponentState;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmAudioPlayComponent enComponentState=%x",
                            ETG_CENUM(tenMeca_RdmComponentState, enComponentState)));
        };
        tenMeca_RdmComponentCommand enComponentState;

    };

    typedef enum {
        enMeca_RdmComponentState_NO_COMPONENT           = 0x00,
        enMeca_RdmComponentState_PRIMARY_COMPONENT_OK   = 0x01,
        enMeca_RdmComponentState_NO_PRIMARY             = 0x02,
        enMeca_RdmComponentState_SECONDARY_COMPONENT_OK = 0x03,
        enMeca_RdmComponentState_NO_SECONDARY           = 0x04
    } tenMeca_RdmComponentState;

    // DAB_*_R_RDM_AUDIO_PLAY_COMPONENT
    struct trMeca_RRdmAudioPlayComponent:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 1) {
                return FALSE;
            }
            enComponentState = (tenMeca_RdmComponentState) pu8Data[0];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmAudioPlayComponent:enComponentState=%x",
                            ETG_CENUM(tenMeca_RdmServiceState, enComponentState)));
        };
        tenMeca_RdmComponentState enComponentState;
    };

    //DAB_*_C_RDM_AUDIO_GET_SCIDI
    struct trMeca_CRdmAudioGetScidi:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        trMeca_CRdmAudioGetScidi() {}
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_AUDIO_GET_SCIDI;
            pRawOutput->enOpType = enOpType_GET;
            pRawOutput->u16DataLen = 0;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmAudioGetScidi"));
        };
    };


    // DAB_*_R_RDM_AUDIO_GET_SCIDI
    struct trMeca_RRdmAudioGetScidi:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 2) {
                return FALSE;
            }
            u16SCIDI = DABDRV_GET_U16(&pu8Data[0]);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmAudioGetScidi:u16SCIDI=0x%04x",
                            u16SCIDI));
        };
        tU16 u16SCIDI;
    };


    //DAB_*_C_RDM_AUDIO_COMPONENT_PLAY
    struct trMeca_CRdmAudioComponentPlay:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_AUDIO_COMPONENT_PLAY;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 8;
            DABDRV_SET_U16(&pRawOutput->au8Data[0],u16SCIDI);
            pRawOutput->au8Data[2]= u8SCIDS; 
            pRawOutput->au8Data[3]= (tU8)enMeca_ServiceType_AUDIO_SERVICE;
            rMecaProgrammeService.vSerialize(&pRawOutput->au8Data[4]);
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmAudioComponentPlay: u8SCIDS=%d u16SCIDI=0x%04x sid=0x%08x:",
							u8SCIDS,
                            u16SCIDI,
                            rMecaProgrammeService.u32GetSID()));
        
        };
        tU16 u16SCIDI;
		tU8 u8SCIDS;
        trMecaProgrammeService rMecaProgrammeService;
    };


    // DAB_*_C_RDM_AUDIO_COMPONENT_PLAY
    struct trMeca_RRdmAudioComponentPlay:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 1) {
                return FALSE;
            }
            bComponentOk = pu8Data[0] == 1 ? TRUE:FALSE;
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmAudioComponentPlay:bComponentOk=%u",
                            ETG_CENUM(DAB_tenBool, bComponentOk)));
        };
        tBool bComponentOk;
    };


    //DAB_*_C_RDM_SET_DYNAMIC_RANGE_CONTROL
    struct trMeca_CRdmSetDynamicRangeControl:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_SET_DYNAMIC_RANGE_CONTROL;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 1;
            pRawOutput->au8Data[0]= bDrcOn; // rfu
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmSetDynamicRangeControl: bDrcOn=%u:",
                            ETG_CENUM(DAB_tenOnOff, bDrcOn)));
        };
        tBool bDrcOn;
    };


    // DAB_*_C_RDM_SET_DYNAMIC_RANGE_CONTROL
    struct trMeca_RRdmSetDynamicRangeControl:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 1) {
                return FALSE;
            }
            bDrcOn = pu8Data[0] == 1 ? TRUE:FALSE;
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmSetDynamicRangeControl: bDrcOn=%u:",
                            ETG_CENUM(DAB_tenOnOff, bDrcOn)));
        };
        tBool bDrcOn;
    };


    //DAB_*_C_RDM_GET_CURR_REGIONS
    struct trMeca_CRdmGetCurrRegions:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_GET_CURR_REGIONS;
            pRawOutput->enOpType = enOpType_GET;
            pRawOutput->u16DataLen = 0;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmGetCurrRegions:"));
        };
    };


    struct trMecaExtdRegionId {
        tBool bGlobal;
        tU16 u16RegionId;
		trMecaExtdRegionId()
		{
			bGlobal = TRUE;
			u16RegionId = 0;
		}
        tVoid vParse(tU8 const *pu8Data) {
            //bGlobal =TRUE; //Coverity
            u16RegionId = DABDRV_GET_U16(&pu8Data[0]);
            u16RegionId=(tU16)(u16RegionId<<2);
        }
        tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMecaExtdRegionId: bGlobal=%u u16RegionId=0x%04x",
                            bGlobal,
                            u16RegionId));

        }
    };

    // DAB_*_R_RDM_GET_CURR_REGIONS
    struct trMeca_RRdmGetCurrRegions:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen < 2) {
                return FALSE;
            }
            tU8 u8NumRegions=pu8Data[1];
            if (rMostHdr._u16MecaLen !=  2+u8NumRegions*2) {
                return FALSE;
            }
            for (int i=0;i<u8NumRegions;i++) {
                trMecaExtdRegionId rRegionId;
                rRegionId.vParse(&pu8Data[2+i*2]);
                lRegions.push_back(rRegionId);
            }
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmGetCurrRegions: numRegions=%d",
                            (tU16)(lRegions.size())));
            for (vector<trMecaExtdRegionId>::const_iterator iter =lRegions.begin();iter!=lRegions.end();iter++) {
                (*iter).vTrace();
            }
        };
        vector<trMecaExtdRegionId>lRegions;
    };


    //DAB_*_C_RDM_GET_COARSE_POSITION
    struct trMeca_CRdmGetCoarsePosition:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_GET_COARSE_POSITION;
            pRawOutput->enOpType = enOpType_GET;
            pRawOutput->u16DataLen = 0;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmGetCoarsePosition:"));
        };
    };


    // DAB_*_C_RDM_GET_COARSE_POSITION
    struct trMeca_RRdmGetCoarsePosition:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 8) {
                return FALSE;
            }
            u32Latitude=DABDRV_GET_U32(&pu8Data[0]);
            u32Longitude=DABDRV_GET_U32(&pu8Data[4]);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmSetDynamicRangeControl: u32Latitude=%u u32Longitude=%u",
                            u32Latitude,
                            u32Longitude));
        };
        tU32 u32Latitude;
        tU32 u32Longitude;
    };

    //DAB_*_C_RDM_GET_CURR_TII_LIST
    struct trMeca_CRdmGetCurrTiiList:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_GET_CURR_TII_LIST;
            pRawOutput->enOpType = enOpType_GET;
            pRawOutput->u16DataLen = 0;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmGetCurrTiiList:"));
        };
    };


    struct trMecaTiiInfo {
        tU8 u8MainId;
        tU8 u8SubId;
        tU32 u32Latitude;
        tU32 u32Longitude;
        tVoid vParse(tU8 const *pu8Data) {
            u8MainId=pu8Data[0];
            u8SubId=pu8Data[1];
            u32Latitude = DABDRV_GET_U32(&pu8Data[2]);
            u32Longitude = DABDRV_GET_U32(&pu8Data[6]);
        }
        tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMecaTiiInfo: u8MainId=0x%x u8SubId=0x%x u32Latitude=%u u32Longitude=%u",
                            u8MainId,
                            u8SubId,
                            u32Latitude,
                            u32Longitude));

        }
    };
    // DAB_*_R_RDM_GET_CURR_TII_LIST
    struct trMeca_RRdmGetCurrTiiList:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen < 1) {
                return FALSE;
            }
            tU8 u8NTI=pu8Data[0];
            if (rMostHdr._u16MecaLen !=  1+u8NTI*10) {
                return FALSE;
            }
            for (int i=0;i<u8NTI;i++) {
                trMecaTiiInfo rTiiInfo;
                rTiiInfo.vParse(&pu8Data[1+i*10]);
                lTiiInfos.push_back(rTiiInfo);
            }
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmGetCurrTiiList: numTiiInfos=%d",
                            (tU16)(lTiiInfos.size())));
            for (vector<trMecaTiiInfo>::const_iterator iter =lTiiInfos.begin();iter!=lTiiInfos.end();iter++) {
                (*iter).vTrace();
            }
        };
        vector<trMecaTiiInfo>lTiiInfos;
    };


    //DAB_*_C_RDM_GET_RDM_STATUS
    struct trMeca_CRdmGetRdmStatus:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_GET_RDM_STATUS;
            pRawOutput->enOpType = enOpType_GET;
            pRawOutput->u16DataLen = 0;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmGetRdmStatus:"));
        };
    };
    
    typedef  enum {
        enRdmStatusBits_SYNC=0x01,
        enRdmStatusBits_ENS_VALID=0x02,
        enRdmStatusBits_AUDIO_SRV_VALID=0x04,
        enRdmStatusBits_ADUIO_COMP_VALID=0x08,
        enRdmStatusBits_MUTE=0x10,
        enRdmStatusBits_TUNING=0x20
    } tenRdmStatusBits;

    struct trMeca_RdmStatus {
        tBool _bWithChanges;
        tU8 u8Status;
        tU8 u8Changes;

        trMeca_RdmStatus():
            _bWithChanges(FALSE),
            u8Status(0),
            u8Changes(0)
        {}

        tVoid vParse(tU8 const *pu8RdmStatus, tBool bWithChanges=FALSE) {
            _bWithChanges = bWithChanges;
            u8Status = pu8RdmStatus[0];
            u8Changes = _bWithChanges ? pu8RdmStatus[1]:0;
        }


        tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RdmStatus: bSync=%u bEnsValid=%u bAudioServiceValid=%u "
                            "bAudioServiceComponentValid=%u bMuteOn=%u bTuning=%u",
                            DAB_BOOL_FROM_BIT(u8Status, (tU8)enRdmStatusBits_SYNC)?1:0,
                            DAB_BOOL_FROM_BIT(u8Status, (tU8)enRdmStatusBits_ENS_VALID)?1:0,
                            DAB_BOOL_FROM_BIT(u8Status, (tU8)enRdmStatusBits_AUDIO_SRV_VALID)?1:0,
                            DAB_BOOL_FROM_BIT(u8Status, (tU8)enRdmStatusBits_ADUIO_COMP_VALID)?1:0,
                            DAB_BOOL_FROM_BIT(u8Status, (tU8)enRdmStatusBits_MUTE)?1:0,
                            DAB_BOOL_FROM_BIT(u8Status, (tU8)enRdmStatusBits_TUNING)?1:0));
            if (_bWithChanges) {
                ETG_TRACE_USR1(("  trMeca_RdmStatus Changed: bSync=%u bEnsValid=%u bAudioServiceValid=%u "
                                "bAudioServiceComponentValid=%u bMuteOn=%u bTuning=%u",
                                DAB_BOOL_FROM_BIT(u8Changes, DAB_b0)?1:0,
                                DAB_BOOL_FROM_BIT(u8Changes, DAB_b1)?1:0,
                                DAB_BOOL_FROM_BIT(u8Changes, DAB_b2)?1:0,
                                DAB_BOOL_FROM_BIT(u8Changes, DAB_b3)?1:0,
                                DAB_BOOL_FROM_BIT(u8Changes, DAB_b4)?1:0,
                                DAB_BOOL_FROM_BIT(u8Changes, DAB_b5)?1:0));
            }
        }

        tBool bIsMuted() const {
            return ((u8Status & (tU8)enRdmStatusBits_MUTE)!=0);
        }
        tBool bIsSync() const {
            return (((u8Status & (tU8)enRdmStatusBits_SYNC)!=0)&&((u8Status & (tU8)enRdmStatusBits_AUDIO_SRV_VALID)!=0));
        }
        tBool bIsEnsValid() const{
        	return((u8Status & (tU8)enRdmStatusBits_ENS_VALID)!=0);
        }

        tBool bIsTuning() const {
        	return((u8Status & (tU8)enRdmStatusBits_TUNING)!=0);
        }
    };

    // DAB_*_R_RDM_GET_RDM_STATUS
    struct trMeca_RRdmGetRdmStatus:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 3) {
                return FALSE;
            }
            enDabMode = (tenMeca_RdmMode)pu8Data[0];
            rRdmStatus.vParse(&pu8Data[1], TRUE);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmGetRdmStatus: members follow:"));
            rRdmStatus.vTrace();
        };


        tenMeca_RdmMode enDabMode;
        trMeca_RdmStatus rRdmStatus;

    };



    //DAB_*_C_RDM_GET_RDM_INFO
    struct trMeca_CRdmGetRdmInfo:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        trMeca_CRdmGetRdmInfo() {}
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_GET_RDM_INFO;
            pRawOutput->enOpType = enOpType_GET;
            pRawOutput->u16DataLen = 1;
            pRawOutput->au8Data[0]= 1; // INFO_DAB
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmGetRdmInfo"));
        };
    };


    // DAB_*_R_RDM_GET_RDM_INFO
    /* todo: change according to received signal, not according to meca:
       1.) label-len is variable
       2.) enAudioQuality has only 1 byte (specified: 2)
       3.) signal_qualitity has only one byte (specified: 2)
    */
    
    struct trMeca_RRdmGetRdmInfo:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        trMeca_RRdmGetRdmInfo():
            rEnsemble(),
            rProgrammeService(),
            u16SCIDI(0),
            rRdmStatus(),
            u32Frequency(0),
            enAudioQuality(enMeca_RdmAudioQualityIndicator_NO_INFO_AVAIL),
            enQualityIndicator(enMeca_RdmSignalQualityIndicator_NO_DAB_SIGNAL) 
        {
            sFrequencyLabel[0]=0;
        }
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen < 22) {
                return FALSE;
            }
            if (pu8Data[0]!= 0x01) {
                return FALSE;
            }
            rEnsemble.vParse(&pu8Data[1]);
            rProgrammeService.vParse(&pu8Data[5]);
            u16SCIDI = DABDRV_GET_U16(&pu8Data[9]);
            rRdmStatus.vParse(&pu8Data[11]);
            u32Frequency = DABDRV_GET_U32(&pu8Data[12]);
            OSALUTIL_szSaveStringNCopy(sFrequencyLabel,(tCString)&pu8Data[16], rMostHdr._u16MecaLen-16);
            tU32 u32LabelLen=(tU32)OSAL_u32StringLength(sFrequencyLabel);
            if (u32LabelLen>DABTUN_FREQLABEL_BUFFER) {
                return FALSE;
            }
            enAudioQuality = (tenMeca_RdmAudioQualityIndicator)pu8Data[20];
            enQualityIndicator=(tenMeca_RdmSignalQualityIndicator)pu8Data[21];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmGetRdmInfo: sid=0x%08x eid=0x%08x u16SCIDI=0x%04x u32Frequency=%u sFrequencyLabel=%4s, "
                            "enAudioQuality=%x enQualityIndicator=%x",
                            rProgrammeService.u32GetSID(),
                            rEnsemble.u32GetID(),
                            u16SCIDI,
                            u32Frequency,
                            sFrequencyLabel,
                            ETG_CENUM(tenMeca_RdmAudioQualityIndicator, enAudioQuality),
                            ETG_CENUM(tenMeca_RdmSignalQualityIndicator, enQualityIndicator)));
            rRdmStatus.vTrace();

        };
        trMecaEnsemble rEnsemble;
        trMecaProgrammeService rProgrammeService;
        tU16 u16SCIDI;
        trMeca_RdmStatus rRdmStatus;
        tU32 u32Frequency;
        tChar sFrequencyLabel[6];
        tenMeca_RdmAudioQualityIndicator enAudioQuality;
        tenMeca_RdmSignalQualityIndicator enQualityIndicator;
    };

    typedef enum {
        enMeca_RdmNotifyCommand_SET          = 0x01,
        enMeca_RdmNotifyCommand_DELETE       = 0x02,
        enMeca_RdmNotifyCommand_SET_ALL      = 0x03,
        enMeca_RdmNotifyCommand_DELETE_ALL   = 0x04,
        enMeca_RdmNotifyCommand_GET_SELECTED = 0x05
    } tenMeca_RdmNotifyCommand;

    typedef enum {
        enMeca_RdmNotifyClass_RDM   = 0x01,
        enMeca_RdmNotifyClass_TUNER = 0x02,
        enMeca_RdmNotifyClass_DAB   = 0x03,
        enMeca_RdmNotifyClass_AUDIO = 0x05
    } tenMeca_RdmNotifyClass;

    typedef enum {
        enMeca_RdmNotifyMessageMask_GET_RDM_STATUS   = DAB_b0, // RDM
        enMeca_RdmNotifyMessageMask_GET_RDM_INFO     = DAB_b1,  //RDM
        enMeca_RdmNotifyMessageMask_GET_TUNER_SIGNAL_QUALITY_LEVEL  = DAB_b0 | DAB_b1 | DAB_b2,   // TUNER
        enMeca_RdmNotifyMessageMask_GET_TUNER_FREQUENCY             = DAB_b3,   // TUNER
        enMeca_RdmNotifyMessageMask_GET_DAB_EID                     = DAB_b0,   // DAB
        enMeca_RdmNotifyMessageMask_GET_DAB_FREQUENCY_LABEL         = DAB_b1 | DAB_b2,   // DAB
        enMeca_RdmNotifyMessageMask_GET_AUDIO_STATUS                = DAB_b3,   // AUDIO
        enMeca_RdmNotifyMessageMask_GET_AUDIO_SCIDI                 = DAB_b4,   // AUDIO
        enMeca_RdmNotifyMessageMask_GET_AUDIO_PSID                  = DAB_b5    // AUDIO
        // todo check masks with ADR
    } tenMeca_RdmNotifyMessageMask;

    //DAB_*_C_RDM_SET_AUTO_NOTIFICATION
    struct trMeca_CRdmSetAutoNotification:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_SET_AUTO_NOTIFICATION;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 3;
            pRawOutput->au8Data[0]=(tU8)enNotifyCommand;
            pRawOutput->au8Data[1]=(tU8)enNotifyClass;
            pRawOutput->au8Data[2]=u8NotifyMask;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmSetAutoNotification: enNotifyCommand=%x enNotifyClass=%x u8NotifyMask=0x%x",
                            ETG_CENUM(tenMeca_RdmNotifyCommand, enNotifyCommand),
                            ETG_CENUM(tenMeca_RdmNotifyClass, enNotifyClass),
                            u8NotifyMask));
        };
        tenMeca_RdmNotifyCommand enNotifyCommand;
        tenMeca_RdmNotifyClass enNotifyClass;
        tU8 u8NotifyMask;
    };

    // DAB_*_R_RDM_SET_AUTO_NOTIFICATION
    struct trMeca_RRdmSetAutoNotification:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 2) {
                return FALSE;
            }
            enNotifyClass = (tenMeca_RdmNotifyClass)pu8Data[0];
            u8NotifyMask = pu8Data[1];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmSetAutoNotification: enNotifyClass=%x u8NotifyMask=%x",
                            ETG_CENUM(tenMeca_RdmNotifyClass, enNotifyClass),
                            u8NotifyMask));
        };
        tenMeca_RdmNotifyClass enNotifyClass;
        tU8 u8NotifyMask;

    };

    typedef enum {
        enMeca_RdmLearnCommand_INVALID = 0x00, // for internal use, not meca
        enMeca_RdmLearnCommand_NORMAL_START = 0x01,
        enMeca_RdmLearnCommand_NORMAL_STOP  = 0x02,
        enMeca_RdmLearnCommand_UPDATE_START = 0x03,
        enMeca_RdmLearnCommand_UPDATE_STOP  = 0x04,
		enMeca_RdmLearnCommand_LEARN_FOREVER_START  = 0x05,
		enMeca_RdmLearnCommand_LEARN_FOREVER_STOP  = 0x06,
        enMeca_RdmLearnCommand_BG_NORMAL_LEARN_START  = 0x81,
        enMeca_RdmLearnCommand_BG_NORMAL_LEARN_STOP   = 0x82,
        enMeca_RdmLearnCommand_BG_UPDATE_LEARN_START  = 0x83,
        enMeca_RdmLearnCommand_BG_UPDATE_LEARN_STOP   = 0x84,
        enMeca_RdmLearnCommand_BG_FOREVER_LEARN_START = 0x85,
        enMeca_RdmLearnCommand_BG_FOREVER_LEARN_STOP  = 0x86,
        enMeca_RdmLearnCommand_REJECTED     = 0xFF
    } tenMeca_RdmLearnCommand;

    //DAB_*_C_RDM_LEARN
    struct trMeca_CRdmLearn:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL


        trMeca_CRdmLearn(tenMeca_RdmLearnCommand enLearnCommand_=enMeca_RdmLearnCommand_INVALID) {
            enLearnCommand=enLearnCommand_;
        }
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_LEARN;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 1;
            pRawOutput->au8Data[0]=(tU8)enLearnCommand;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmLearn: enLearnCommand=%x",
                            ETG_CENUM(tenMeca_RdmLearnCommand, enLearnCommand)));
        };
        tenMeca_RdmLearnCommand enLearnCommand;
    };

    // DAB_*_R_RDM_LEARN
    struct trMeca_RRdmLearn:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 1) {
                return FALSE;
            }
            enLearnCommand = (tenMeca_RdmLearnCommand)pu8Data[0];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmLearn: enLearnCommand=%x",
                            ETG_CENUM(tenMeca_RdmLearnCommand, enLearnCommand)));
        };
        tenMeca_RdmLearnCommand enLearnCommand;
    };

	//_DAB_C_RDM_GET_DAB_FREQUENCY_INFO
	struct trMeca_CRdmFrequencyInfo:
		public  trMsgMecaOut {
		DAB_DISPATCH_IMPL


		trMeca_CRdmFrequencyInfo() {}
		virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
			pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
			pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_GET_FREQUENCY_INFO;
			pRawOutput->enOpType = enOpType_GET;
			pRawOutput->u16DataLen = 1;
		}

		virtual tVoid vTrace() const {
			ETG_TRACE_USR1(("  trMeca_CRdmFrequencyInfo"));
		};
	};

	// DAB_*_R_RDM_GET_DAB_FREQUENCY_INFO
	struct trMeca_RRdmFrequencyInfo:
		public trMsgMecaInput {
		DAB_DISPATCH_IMPL

		virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
			_rMostHdr=rMostHdr;
			if (rMostHdr._u16MecaLen < 9) {
				return FALSE;
			}
			u32Frequency=DABDRV_GET_U32(&pu8Data[0]);

			tU32 u32LabelLen=0;
			OSALUTIL_szSaveStringNCopy(sFrequencyLabel,(tCString)&pu8Data[4],rMostHdr._u16MecaLen-8);
		u32LabelLen=/*OSAL_u32StringLength(sFrequencyLabel);*/rMostHdr._u16MecaLen-8;


			u8FirstFreqIndex=pu8Data[u32LabelLen+4];
			u8CurrFreqIndex=pu8Data[u32LabelLen+5];
			u8LastFreqIndex=pu8Data[u32LabelLen+6];
			u8FreqTable=pu8Data[u32LabelLen+7];

			return TRUE;
		}
		virtual tVoid vTrace() const {
			ETG_TRACE_USR1(("  trMeca_RRdmFrequencyInfo: u32frequency=%x u8FirstFreqIndex=%d u8CurrFreqIndex=%d u8LastFreqIndex=%d u8FreqTable=%d sFrequencyLabel=%s",
								u32Frequency,
								u8FirstFreqIndex,
								u8CurrFreqIndex,
								u8LastFreqIndex,
								u8FreqTable,
								sFrequencyLabel));
		};
		tU32 u32Frequency;
		tChar sFrequencyLabel[DABTUN_FREQLABEL_BUFFER];
		tU8 u8FirstFreqIndex;
		tU8 u8CurrFreqIndex;
		tU8 u8LastFreqIndex;
		tU8 u8FreqTable;
	};

    typedef enum {
        enMeca_RdmServFollowMode_None    = 0, // not meca
        enMeca_RdmServFollowMode_ALT_FREQUENCY    = DAB_b0,
        enMeca_RdmServFollowMode_ALT_ENSEMBLE     = DAB_b1,
        enMeca_RdmServFollowMode_ALT_HARD_LINKING = DAB_b2,
        enMeca_RdmServFollowMode_ALT_SEARCH       = DAB_b3,
        enMeca_RdmServFollowMode_ALT_STICKYCOMPONENT=DAB_b5
    } tenMeca_RdmServFollowMode;

	typedef enum {
		enMeca_OneShot_Audio_Play_Asid = 0,
		enMeca_OneShot_Audio_Comp_Play = 1
	}tenMeca_RdmWStartTrigger;
	typedef enum {
		enMeca_OneShot_End_State_End = 0,
		enMeca_OneShot_End_State_Change = 1,
		enMeca_OneShot_End_Tune = 2
	}tenMeca_RdmWEndTrigger;
    //DAB_*_C_RDM_SERV_FOLLOW_SET_MODE
    struct trMeca_CRdmServFollowSetMode:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_SERV_FOLLOW_SET_MODE;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 1;
            pRawOutput->au8Data[0]=u8ServFollowMode;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmServFollowSetMode: u8ServFollowMode=%x (AltFreq=%u, AltEns=%u, AltHardLinking=%u, AltSearch=%u,AltStkComp=%u)",
                            u8ServFollowMode,
                            (u8ServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_FREQUENCY) ? TRUE : FALSE,
                            (u8ServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_ENSEMBLE) ? TRUE : FALSE,
                            (u8ServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_HARD_LINKING) ? TRUE : FALSE,
                            (u8ServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_SEARCH) ? TRUE : FALSE,
                            (u8ServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_STICKYCOMPONENT) ? TRUE : FALSE));
        }
        tU8 u8ServFollowMode;
    };


    // DAB_*_R_RDM_FOLLOW_SET_MODE
    struct trMeca_RRdmServFollowSetMode:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 1) {
                return FALSE;
            }
            u8ServFollowMode = pu8Data[0];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmServFollowSetMode: u8ServFollowMode=%x (AltFreq=%u, AltEns=%u, AltHardLinking=%u, AltSearch=%u)",
                            u8ServFollowMode,
                            (u8ServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_FREQUENCY) ? TRUE : FALSE,
                            (u8ServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_ENSEMBLE) ? TRUE : FALSE,
                            (u8ServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_HARD_LINKING) ? TRUE : FALSE,
                            (u8ServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_SEARCH) ? TRUE : FALSE));
        }
        tU8 u8ServFollowMode;
    };


    struct trMeca_CRdmServFollowSetOneShotMode:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_SERV_FOLLOW_SET_ONESHOT_MODE;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 8;
            pRawOutput->au8Data[0]=u8OneShotServFollowMode;
			pRawOutput->au8Data[1]=u8b_rfu;
			DABDRV_SET_U16(&pRawOutput->au8Data[2],u16WStartTrigger);
			DABDRV_SET_U16(&pRawOutput->au8Data[4],u16WEndTrigger);
			DABDRV_SET_U16(&pRawOutput->au8Data[6],u16W_rfu);
        }
		 virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmServFollowSetOneShotMode: u8ServFollowMode=%x (AltFreq=%u, AltEns=%u, AltHardLinking=%u, AltSearch=%u)",
                            u8OneShotServFollowMode,
                            (u8OneShotServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_FREQUENCY) ? TRUE : FALSE,
                            (u8OneShotServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_ENSEMBLE) ? TRUE : FALSE,
                            (u8OneShotServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_HARD_LINKING) ? TRUE : FALSE,
                            (u8OneShotServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_SEARCH) ? TRUE : FALSE));
			ETG_TRACE_USR1(("  trMeca_CRdmServFollowSetOneShotMode:u16WStartTrigger: u16WStartTrigger=%x (OneShot_Audio_Play_Asid=%u,OneShot_Audio_Comp_Play=%u)",
                            u16WStartTrigger,
                            (u16WStartTrigger & (tU16)enMeca_OneShot_Audio_Play_Asid) ? TRUE : FALSE,
                            (u16WStartTrigger & (tU16)enMeca_OneShot_Audio_Comp_Play) ? TRUE : FALSE));
			ETG_TRACE_USR1(("  trMeca_CRdmServFollowSetOneShotMode:u16WEndTrigger: u16WEndTrigger=%x (enMeca_OneShot_End_State_End=%u,enMeca_OneShot_End_State_Change=%u, enMeca_OneShot_End_Tune=%u)",
                            u16WEndTrigger,
                            (u16WEndTrigger & (tU16)enMeca_OneShot_End_State_End) ? TRUE : FALSE,
                            (u16WEndTrigger & (tU16)enMeca_OneShot_End_State_Change) ? TRUE : FALSE,
							(u16WEndTrigger & (tU16)enMeca_OneShot_End_Tune) ? TRUE : FALSE));
		 }
        tU8 u8OneShotServFollowMode;
		tU8 u8b_rfu;
		tU16 u16WStartTrigger;
		tU16 u16WEndTrigger;
		tU16 u16W_rfu;
    };
    struct trMeca_RRdmServFollowSetOneShotMode:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 8) {
                return FALSE;
            }
            u8OneShotServFollowMode = pu8Data[0];
			u8b_rfu = pu8Data[1];
			u16WStartTrigger = DABDRV_GET_U16(&pu8Data[2]);
			u16WEndTrigger = DABDRV_GET_U16(&pu8Data[4]);
			u16W_rfu = DABDRV_GET_U16(&pu8Data[6]);
            return TRUE;
        }
		 virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmServFollowSetOneShotMode: u8ServFollowMode=%x (AltFreq=%u, AltEns=%u, AltHardLinking=%u, AltSearch=%u)",
                            u8OneShotServFollowMode,
                            (u8OneShotServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_FREQUENCY) ? TRUE : FALSE,
                            (u8OneShotServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_ENSEMBLE) ? TRUE : FALSE,
                            (u8OneShotServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_HARD_LINKING) ? TRUE : FALSE,
                            (u8OneShotServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_SEARCH) ? TRUE : FALSE));
			ETG_TRACE_USR1((" trMeca_RRdmServFollowSetOneShotMode: u16WStartTrigger: u16WStartTrigger=%x (OneShot_Audio_Play_Asid=%u,OneShot_Audio_Comp_Play=%u)",
                            u16WStartTrigger,
                            (u16WStartTrigger & (tU16)enMeca_OneShot_Audio_Play_Asid) ? TRUE : FALSE,
                            (u16WStartTrigger & (tU16)enMeca_OneShot_Audio_Comp_Play) ? TRUE : FALSE));
			ETG_TRACE_USR1((" trMeca_RRdmServFollowSetOneShotMode: u16WEndTrigger: u16WEndTrigger=%x (enMeca_OneShot_End_State_End=%u,enMeca_OneShot_End_State_Change=%u, enMeca_OneShot_End_Tune=%u)",
                            u16WEndTrigger,
                            (u16WEndTrigger & (tU16)enMeca_OneShot_End_State_End) ? TRUE : FALSE,
                            (u16WEndTrigger & (tU16)enMeca_OneShot_End_State_Change) ? TRUE : FALSE,
							(u16WEndTrigger & (tU16)enMeca_OneShot_End_Tune) ? TRUE : FALSE));
		 }
        tU8 u8OneShotServFollowMode;
		tU8 u8b_rfu;
		tU16 u16WStartTrigger;
		tU16 u16WEndTrigger;
		tU16 u16W_rfu;
    };
    typedef enum {
        enMeca_RdmServFollowState_STATE_START    = 0x01,
        enMeca_RdmServFollowState_STATE_CHANGE     = 0x02,
        enMeca_RdmServFollowState_STATE_END = 0x03
    } tenMeca_RdmServFollowState;


    // DAB_*_R_RDM_FOLLOW_ACTIVE
    struct trMeca_RRdmServFollowActive:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 2) {
                return FALSE;
            }
            u8ServFollowMode = pu8Data[0];

            enServFollowState = (tenMeca_RdmServFollowState)pu8Data[1];
            if (u8ServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_FREQUENCY) {
                enMaxMode=enMeca_RdmServFollowMode_ALT_FREQUENCY;
            }
            else if (u8ServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_ENSEMBLE) {
                enMaxMode=enMeca_RdmServFollowMode_ALT_ENSEMBLE;
            }
            else if (u8ServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_HARD_LINKING) {
                enMaxMode=enMeca_RdmServFollowMode_ALT_HARD_LINKING;
            }
            else if (u8ServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_SEARCH) {
                enMaxMode=enMeca_RdmServFollowMode_ALT_SEARCH;
            }
            else  {
                enMaxMode=enMeca_RdmServFollowMode_None;
            }
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmServFollowActive: u8ServFollowMode=%x (AltFreq=%u, AltEns=%u, AltHardLinking=%u, AltSearch=%u) enServFollowState=%x",
                            u8ServFollowMode,
                            (u8ServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_FREQUENCY) ? TRUE : FALSE,
                            (u8ServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_ENSEMBLE) ? TRUE : FALSE,
                            (u8ServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_HARD_LINKING) ? TRUE : FALSE,
                            (u8ServFollowMode & (tU8)enMeca_RdmServFollowMode_ALT_SEARCH) ? TRUE : FALSE,
                            ETG_CENUM(tenMeca_RdmServFollowState, enServFollowState)));
        };
        tU8 u8ServFollowMode;
        tenMeca_RdmServFollowState enServFollowState;
        tenMeca_RdmServFollowMode enMaxMode;

    };


    //DAB_*_C_RDM_STATION_PLAY
    struct trMeca_CRdmStationPlay:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
//PSARCC30-1099, PSARCC30-1875 : PlayLSM is sent to ADR with option ignore Frequency. 
//If ignore frequency option is enabled for PlayLSM then it ingnores the frequence and remians in the previously tune frequency.
//As per the mail from ADR team it should be ignored for LSM.
        trMeca_CRdmStationPlay(tU8 u8StationNumber_=0,tU8 u8Option_=0)
            {
				u8StationNumber= u8StationNumber_;
				if (u8StationNumber)
					u8Option = u8Option_ | DAB_BIT_ZERO_VALUE;
				else
					u8Option = u8Option_ | DAB_BIT_TWO_VALUE;

				//u8Option(u8Option_ | (u8StationNumber?DAB_BIT_ZERO_VALUE : DAB_BIT_TWO_VALUE)) //IGNORE_STATION_FREQUENCY flag is enabled by setting bit 0 to TRUE PSARCC21-3352
			}																	//Don't send IGNORE_FREQ for preset 0.
        																	//PSARCC30-2064 DAB_BIT_TWO_VALUE is added, If there is FREE LSM in ADR it tries to play available FM channel
        																	//As per discussion with ADR team if BIT 2 is set ADR clears FM Rdminfo if DAB LSM is Empty
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_STATION_PLAY;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 2;
            pRawOutput->au8Data[0]=u8StationNumber;
            pRawOutput->au8Data[1]=u8Option;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmStationPlay: u8StationNumber=%u u8Option=%u",
                            u8StationNumber,u8Option));
        };
        tU8 u8StationNumber;
        tU8 u8Option;
    };

    typedef enum {
        enMeca_RdmStationState_STATION_NUMBER_INVALID    = 0x00,
        enMeca_RdmStationState_STATION_INVALID    = 0x01,
        enMeca_RdmStationState_STATION_FREE     = 0x02,
        enMeca_RdmStationState_STATION_PLAY_OK = 0x03,
        enMeca_RdmStationState_STATION_PLAY_ERROR = 0x04,
        enMeca_RdmStationState_STATION_STORE_OK = 0x05,
        enMeca_RdmStationState_WAITING_FOR_STATION_PLAY = 0x06,
        enMeca_RdmStationState_STATION_SERVICE_INVALID = 0x07,
        enMeca_RdmStationState_STATION_ENS_FREQ_INVALID = 0x08,
        enMeca_RdmStationState_STATION_SET_OK = 0x09
    } tenMeca_RdmStationState;

    // DAB_*_R_RDM_STATION_PLAY
    struct trMeca_RRdmStationPlay:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 2) {
                return FALSE;
            }
            enStationState = (tenMeca_RdmStationState)pu8Data[0];
            u8StationNumber = pu8Data[1];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmStationPlay: enStationState=%x  u8StationNumber=%u",
                            ETG_CENUM(tenMeca_RdmStationState, enStationState),
                            u8StationNumber));
        };

        tenMeca_RdmStationState enStationState;
        tU8 u8StationNumber;
    };


    //DAB_*_C_RDM_STATION_STORE
    struct trMeca_CRdmStationStore:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_STATION_STORE;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 1;
            pRawOutput->au8Data[0]=u8StationNumber;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmStationStore: u8StationNumber=%u",
                            u8StationNumber));
        };
        tU8 u8StationNumber;
    };



    // DAB_*_R_RDM_STATION_STORE
    struct trMeca_RRdmStationStore:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 2) {
                return FALSE;
            }
            enStationState = (tenMeca_RdmStationState)pu8Data[0];
            u8StationNumber = pu8Data[1];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmStationStore: enStationState=%x  u8StationNumber=%u",
                            ETG_CENUM(tenMeca_RdmStationState, enStationState),
                            u8StationNumber));
        };

        tenMeca_RdmStationState enStationState;
        tU8 u8StationNumber;
    };

    //DAB_*_C_RDM_STATION_GET_PRESET_STATION
    struct trMeca_CRdmStationGetPresetStation:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_STATION_GET_PRESET_STATION;
            pRawOutput->enOpType = enOpType_GET;
            pRawOutput->u16DataLen = 1;
            pRawOutput->au8Data[0]=u8StationNumber;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmStationGetPresetStation: u8StationNumber=%u",
                            u8StationNumber));
        };
        tU8 u8StationNumber;
    };




    // DAB_*_R_RDM_STATION_GET_PRESET_STATION
    struct trMeca_RRdmStationGetPresetStation:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 30) {
                return FALSE;
            }
            u8StationNumber = pu8Data[0];
            enServiceType = (tenMeca_ServiceType)pu8Data[1];
            rProgrammeService.vParse(&pu8Data[2], enServiceType);
            u32Frequency = DABDRV_GET_U32(&pu8Data[6]);
            rLabel.vParse(&pu8Data[10]);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmStationGetPresetStation: u8StationNumber=%u enServiceType=%x sid=0x%08x u32Frequency=%u Label follows:",
                            u8StationNumber,
                            ETG_CENUM(tenMeca_ServiceType, enServiceType),
                            rProgrammeService.u32GetSID(),
                            u32Frequency));
            rLabel.vTrace();
        };

        tU8 u8StationNumber;
        tenMeca_ServiceType enServiceType;
        trMecaProgrammeService rProgrammeService;
        tU32 u32Frequency;
        trMecaLabel rLabel;

    };

    //DAB_*_C_RDM_STATION_SET_PRESET_STATION
    struct trMeca_CRdmStationSetPresetStation:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        trMeca_CRdmStationSetPresetStation(){
        	dataLength=30;
        	u8StationNumber=0;
        	u32Frequency=0;
        	scids=0;
        	rfu=0;
        	enServiceType=enMeca_ServiceType_INVALID;
        }

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_STATION_SET_PRESET_STATION;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = dataLength;
            pRawOutput->au8Data[0]=u8StationNumber;
            pRawOutput->au8Data[1]=(tU8) enServiceType;
            rProgrammeService.vSerialize(&pRawOutput->au8Data[2]);
            DABDRV_SET_U32(&pRawOutput->au8Data[6],u32Frequency);
            rLabel.vSerialize(&pRawOutput->au8Data[10]);
            if(dataLength == 76){//Secondary service
            	rEnsemble.vSerialize(&pRawOutput->au8Data[30]);
            	rEnsLabel.vSerialize(&pRawOutput->au8Data[34]);
            	pRawOutput->au8Data[54] = scids;
            	pRawOutput->au8Data[55] = 0x01;
            	rCompLabel.vSerialize(&pRawOutput->au8Data[56]);
            }
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmStationSetPresetStation: u8StationNumber=%u sid=0x%x enServiceType=%x u32Frequency=%u follows:",
                            u8StationNumber,
                            rProgrammeService.u32GetSID(),
                            ETG_CENUM(tenMeca_ServiceType, enServiceType),
                            u32Frequency));
            rLabel.vTrace();
        };
        tU8 u8StationNumber;
        tenMeca_ServiceType enServiceType;
        trMecaProgrammeService rProgrammeService;
        tU32 u32Frequency;
        trMecaLabel rLabel;
        trMecaEnsemble rEnsemble;
        trMecaLabel rEnsLabel;
        tU8 scids;
        tU8 rfu;
        trMecaLabel rCompLabel;
        tU16 dataLength;
    };


    // DAB_*_R_RDM_STATION_SET_PRESET_STATION
    struct trMeca_RRdmStationSetPresetStation:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 2) {
                return FALSE;
            }
            enStationState = (tenMeca_RdmStationState)pu8Data[0];
            u8StationNumber = pu8Data[1];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmStationSetPresetStation: enStationState=%x  u8StationNumber=%u",
                            ETG_CENUM(tenMeca_RdmStationState, enStationState),
                            u8StationNumber));
        };

        tenMeca_RdmStationState enStationState;
        tU8 u8StationNumber;
    };

	//DAB_*_C_RDM_STATION_SET_USER
	struct trMeca_CRdmStationSetUser:
		public  trMsgMecaOut {
		DAB_DISPATCH_IMPL

		virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
			pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
			pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_STATION_SET_USER;
			pRawOutput->enOpType = enOpType_SET;
			pRawOutput->u16DataLen = 4;
			pRawOutput->au8Data[0]=u8UserId;
			pRawOutput->au8Data[1]=u8Flag;
			pRawOutput->au8Data[2]=tU8PresetCount;
			pRawOutput->au8Data[3]=u8Rfu2;

		}

		virtual tVoid vTrace() const {
			ETG_TRACE_USR1(("  trMeca_CRdmStationSetUser: u8UserId=%x u8Flag=%d tU8PresetCount=%d u8Rfu2=%d follows:",
							u8UserId,
							u8Flag,
							tU8PresetCount,
							u8Rfu2));
		};
		tU8 u8UserId;
		tU8 u8Flag;
		tU8 tU8PresetCount;
		tU8 u8Rfu2; //Unused
	};


	// DAB_*_R_RDM_STATION_SET_USER
	struct trMeca_RRdmStationSetUser:
		public trMsgMecaInput {
		DAB_DISPATCH_IMPL

		virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
			_rMostHdr=rMostHdr;
			if (rMostHdr._u16MecaLen != 4) {
				return FALSE;
			}
			u8UserId = pu8Data[0];
			u8Flag = pu8Data[1];
			tU8PresetCount = pu8Data[2];
			u8Rfu2 = pu8Data[3];
			return TRUE;
		}
		virtual tVoid vTrace() const {
			ETG_TRACE_USR1(("  trMeca_RRdmStationSetUser: u8UserId=%x  u8Flag=%d tU8PresetCount=%d u8Rfu2=%d",
					u8UserId,
					u8Flag,
					tU8PresetCount,
					u8Rfu2));
		};

		tU8 u8UserId;
		tU8 u8Flag;
		tU8 tU8PresetCount;
		tU8 u8Rfu2; //Unused
	};


#if 0
    //DAB_*_C_RDM_STATION_GET_PRESET_STATION_NUMBER
    struct trMeca_CRdmStationGetPresetStationNumber:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_STATION_GET_PRESET_STATION_NUMBER;
            pRawOutput->enOpType = enOpType_GET;
            pRawOutput->u16DataLen = 30;
            pRawOutput->au8Data[0]=0;
            pRawOutput->au8Data[1]=(tU8) enServiceType;
            rProgrammeService.vSerialize(&pRawOutput->au8Data[2]);
            DABDRV_SET_U32(&pRawOutput->au8Data[6],u32Frequency);
            rLabel.vSerialize(&pRawOutput->au8Data[10]);
            pRawOutput->au8Data[30] = u8FirstStationNumber;
            pRawOutput->au8Data[31] = u8NumberStations;
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmStationGetPresetStationNumber: enServiceType=%x sid=0x%08x u32Frequency=%u "
                            "u8FirstStationNumber=%u u8NumberStations=%u follows: ",
                            ETG_CENUM(tenMeca_ServiceType, enServiceType),
                            rProgrammeService.u32GetSID(),
                            u32Frequency,
                            u8FirstStationNumber,
                            u8NumberStations));
            rLabel.vTrace();
        };
        tenMeca_ServiceType enServiceType;
        trMecaProgrammeService rProgrammeService;
        tU32 u32Frequency;
        trMecaLabel rLabel;
        tU8 u8FirstStationNumber;
        tU8 u8NumberStations;
    };

    typedef enum {
        enMeca_RdmPresetStationState_STATION_NOT_AVAILABLE = 0x00,
        enMeca_RdmPresetStationState_STATION_FOUND         = 0x01,
        enMeca_RdmPresetStationState_COMMAND_REJECTED      = 0xFF
    } tenMeca_RdmPresetStationState;

    // DAB_*_R_RDM_STATION_GET_PRESET_STATION_NUMBER
    struct trMeca_RRdmStationGetPresetStationNumber:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 3) {
                return FALSE;
            }
            enPresetStationState = (tenMeca_RdmPresetStationState)pu8Data[0];
            u8StationNumber = pu8Data[1];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmStationGetPresetStationNumber: enPresetStationState=%x  u8StationNumber=%u",
                            ETG_CENUM(tenMeca_RdmPresetStationState, enPresetStationState),
                            u8StationNumber));
        };

        tenMeca_RdmPresetStationState enPresetStationState;
        tU8 u8StationNumber;
    };
#endif

    //DAB_*_C_RDM_STATION_TRAVELSTORE
    struct trMeca_CRdmStationTravelStore:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
    
        trMeca_CRdmStationTravelStore(){
        bStart=TRUE;
        u8Mode=0;
        u8FilterId=0;
        u8FirstStationNumber=0;
        u8NumberStations=0;
        u8DestinationStationNumber=0;
        }
        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_STATION_TRAVELSTORE;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 6;
            pRawOutput->au8Data[0]=bStart;
            pRawOutput->au8Data[1]=u8Mode; // STORE_DB_SERV_LIST
            pRawOutput->au8Data[2]=u8FilterId;
            pRawOutput->au8Data[3]=u8FirstStationNumber;
            pRawOutput->au8Data[4]=u8NumberStations;
            pRawOutput->au8Data[5]=u8DestinationStationNumber; // destination station number
        }
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmStationTravelStore: bStart=%u u8FilterId=0x%x u8FirstStationNumber=%u u8NumberStations=%u u8DestinationStationNumber=%d",
                            bStart,
                            u8FilterId,
                            u8FirstStationNumber,
                            u8NumberStations,
							u8DestinationStationNumber));
        };
        tBool bStart;
        tU8 u8Mode;
        tU8 u8FilterId;
        tU8 u8FirstStationNumber;
        tU8 u8NumberStations;
        tU8 u8DestinationStationNumber;

    };

    typedef enum {
        enMeca_RdmTravelStoreState_STOPPED          = 0x00,
        enMeca_RdmTravelStoreState_STARTED          = 0x01,
        enMeca_RdmTravelStoreState_DONE             = 0x02,
        enMeca_RdmTravelStoreState_COMMAND_REJECTED = 0xFF
    } tenMeca_RdmTravelStoreState;

    // DAB_*_R_RDM_STATION_TRAVELSTORE
    struct trMeca_RRdmStationTravelStore:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 6) {
                return FALSE;
            }
            enTravelStoreState = (tenMeca_RdmTravelStoreState)pu8Data[0];
            u8Mode=pu8Data[1];
            u8FilterId = pu8Data[2];
            u8FirstStationNumber = pu8Data[3];
            u8NumberStations = pu8Data[4];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmStationTravelStore: enTravelStoreState=%x u8Mode=%d u8FilterId=0x%x u8FirstStationNumber=%u u8NumberStations=%u",
                            ETG_CENUM(tenMeca_RdmTravelStoreState, enTravelStoreState),
                            u8Mode,
                            u8FilterId,
                            u8FirstStationNumber,
                            u8NumberStations));
        };

        tenMeca_RdmTravelStoreState enTravelStoreState;
        tU8 u8Mode;
        tU8 u8FilterId;
        tU8 u8FirstStationNumber;
        tU8 u8NumberStations;
    };


    /* missing:
       _DAB_C_RDM_SET_AUX_DATA
       DAB_*_R_RDM_SET_AUX_DATA
       _DAB_C_RDM_GET_AUX_DATA
       DAB_*_R_RDM_GET_AUX_DATA
    */
	typedef enum {
		enMeca_RdmScan_START        = 0x01,
		enMeca_RdmScan_STOP         = 0x02,
		enMeca_RdmScan_ABORT        = 0x03,
		enMeca_RdmScan_PAUSE        = 0x04,
		enMeca_RdmScan_RESUME       = 0x05,
		enMeca_RdmScan_NEXT         = 0x06
	} tenMeca_RdmScanCommand;

	 struct trMeca_CRdmScan:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL
			virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
				pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
				pRawOutput->u16MsgType = MECA_DISPATCHER_C_RDM_SCAN;
				pRawOutput->enOpType = enOpType_SET;
				pRawOutput->u16DataLen = 13;
				pRawOutput->au8Data[0] = (tU8)enCmd;
				pRawOutput->au8Data[1]= 1;
				DABDRV_SET_U32(&pRawOutput->au8Data[2],u32StartFrequency);
				pRawOutput->au8Data[6]= 0;//rfu
				pRawOutput->au8Data[7]=5;
				DABDRV_SET_U16(&pRawOutput->au8Data[8],0x0100);
				DABDRV_SET_U16(&pRawOutput->au8Data[10],0x0401);
				pRawOutput->au8Data[12]=(tU8)u8ScanThreshold;
		}
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CRdmScan ScanType=%x",
                            ETG_CENUM(tenMeca_RdmScanCommand, enCmd)));
        };
        tenMeca_RdmScanCommand enCmd;		
		tU32 u32StartFrequency;		
		tU8 u8ScanThreshold;		
    };
   
	typedef enum {
		enMeca_RdmScanState_STARTED       			  = 0x01,
		enMeca_RdmScanState_DONE          			  = 0x02,
		enMeca_RdmScanState_ABORT         			  = 0x03,
		enMeca_RdmScanState_PAUSE         			  = 0x04,
		enMeca_RdmScanState_RESUME        			  = 0x05,
		enMeca_RdmScanState_SEARCHING_ENSEMBLE        = 0x10,
		enMeca_RdmScanState_SCANNING_ENSEMBLE         = 0x11,
		enMeca_RdmScanState_SKIPPING_ENSEMBLE         = 0x12,
		enMeca_RdmScanState_PLAYING_SERVICE           = 0x13
	} tenMeca_RdmScanState;

     struct trMeca_RRdmScan:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            _rMostHdr=rMostHdr;
            if (rMostHdr._u16MecaLen != 3) {
                return FALSE;
            }
            enCmd = (tenMeca_RdmScanState)pu8Data[0];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RRdmScan: enCmd=%x ",
                            ETG_CENUM(tenMeca_RdmScanState, enCmd)));
        };
        tenMeca_RdmScanState enCmd;
    };
} // namespace DAB
#define RDM_DONE
#endif
