/************************************************************************
 * FILE:        dabmeca_ddm.hpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of dabmeca_ddm
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      	| Author                       | Modification
				  
				
 *************************************************************************/

#ifndef DABMECA_DDM_HPP
#define DABMECA_DDM_HPP

#include "fc_dabtuner_util.h"
#include "dabdrv_meca.h"
#include "dabmeca_db.hpp"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_UTIL_MSG
#include "trcGenProj/Header/dabmeca_ddm.hpp.trc.h"
#endif

namespace DAB {

    typedef enum {
        enMeca_DdmCommand_STOP  = 0x0000,
        enMeca_DdmCommand_START = 0x0001,
        enMeca_DdmCommand_GET = 0x0002,
        enMeca_DdmCommand_START_NEXT = 0x0003,
		enMeca_DdmCommand_STOP_ALL = 0x00FF
    } tenMeca_DdmCommand;

    typedef enum {
        enMeca_DdmPacketSource_SCIDI   = 0x0000,
        enMeca_DdmPacketSource_SubChId = 0x0001
    } tenMeca_DdmPacketSource;




    // _DAB_C_DDM_PACKET
    struct trMeca_CDdmPacket:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_DDM_PACKET;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 6;
            DABDRV_SET_U16(&pRawOutput->au8Data[0],(tU16)enCommand);
            DABDRV_SET_U16(&pRawOutput->au8Data[2],enSource);
            DABDRV_SET_U16(&pRawOutput->au8Data[4],u16Id);

        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CDdmPacket: enCommand=%x enSource=0x%x u16Id=0x%x",
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            ETG_CENUM(tenMeca_DdmPacketSource, enSource),
                            u16Id));
        };
    
        tenMeca_DdmCommand enCommand;
        tenMeca_DdmPacketSource enSource;
        tU16 u16Id;
    

    };

    typedef enum {
        enMeca_DdmResponse_OK                           = 0x0000,
        enMeca_DdmResponse_COMMAND_NOT_VALID            = 0x0001,
        enMeca_DdmResponse_SCIDI_NOT_VALID              = 0x0002,
        enMeca_DdmResponse_SubChId_NOT_VALID            = 0x0003,
        enMeca_DdmResponse_NO_RESOURCES                 = 0x0004,
        enMeca_DdmResponse_URL_NOT_VALID                = 0x0005,
        enMeca_DdmResponse_TIME_OUT                     = 0x0006,
        enMeca_DdmResponse_NO_CAROUSEL                  = 0x0007,
        enMeca_DdmResponse_NO_DECODER                   = 0x0008,
        enMeca_DdmResponse_NOT_SUPPORTED                = 0x0009,
        enMeca_DdmResponse_END_CAROUSEL                 = 0x000a,
        enMeca_DdmResponse_NO_HOMEPAGE                  = 0x000b,
        enMeca_DdmResponse_NO_MATCH_OBJECT              = 0x000c,
        enMeca_DdmResponse_NO_OBJECT                    = 0x000d,
        enMeca_DdmResponse_PROFILE_NOT_VALID            = 0x000e,
        enMeca_DdmResponse_NO_OBJSTREAM                 = 0x000f,
        enMeca_DdmResponse_NO_MPEG2_SYNC                = 0x0010,
        enMeca_DdmResponse_DATA_ALREADY_SELECTED        = 0x0011,
        enMeca_DdmResponse_DATA_NOT_SELECTED            = 0x0012,
        enMeca_DdmResponse_SERVICE_ID_NOT_VALID         = 0x0013,
        enMeca_DdmResponse_URI_PROCESSING_IN_PROGRESS   = 0x0020,
        enMeca_DdmResponse_URI_ACTIVATED                = 0x0021,
        enMeca_DdmResponse_URI_ACTIVATE_FAIL_SCHED      = 0x0022,
        enMeca_DdmResponse_URI_ACTIVATE_FAIL_RECEPT     = 0x0023,
        enMeca_DdmResponse_URI_DEACTIVATED_BY_SCHED     = 0x0024,
        enMeca_DdmResponse_URI_DEACTIVATED_BY_SYNC_LOSS = 0x0025,
        enMeca_DdmResponse_URI_FAIL_NO_SCIDI            = 0x0026,
        enMeca_DdmResponse_URI_FAIL_TDC_ENABLE          = 0x0027,
        enMeca_DdmResponse_URI_FAIL_NO_DB_INFO          = 0x0028,
        enMeca_DdmResponse_OTHER_ERROR                  = 0xFFFF
    } tenMeca_DdmResponse;


    // DAB_*_R_DDM_PACKET
    struct trMeca_RDdmPacket:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            if (rMostHdr._u16MecaLen != 8) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enCommand=(tenMeca_DdmCommand)DABDRV_GET_U16(&pu8Data[0]);
            enResponse=(tenMeca_DdmResponse)DABDRV_GET_U16(&pu8Data[2]);
            enSource=(tenMeca_DdmPacketSource)DABDRV_GET_U16(&pu8Data[4]);
            u16Id = DABDRV_GET_U16(&pu8Data[6]);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RDdmPacket: enCommand=%x enResponse=%x enSource=0x%x u16Id=0x%x",
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            ETG_CENUM(tenMeca_DdmResponse, enResponse),
                            ETG_CENUM(tenMeca_DdmPacketSource, enSource),
                            u16Id));
        };
    
        tenMeca_DdmCommand enCommand;
        tenMeca_DdmResponse enResponse;
        tenMeca_DdmPacketSource enSource;
        tU16 u16Id;

    };

#define DABMECA_MAX_EUA_FIELD_LEN 100
#define DAB_MIN(a, b) (a)<(b) ? (a) : (b);
    // _DAB_C_DDM_DATAGROUP
    struct trMeca_CDdmDataGroup:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            tU16 u16FieldLen = (tU16) DAB_MIN(u16EuaFieldLength,(tU16)DABMECA_MAX_EUA_FIELD_LEN);
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_DDM_DATAGROUP;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = (tU16)(8 + u16FieldLen);
            DABDRV_SET_U16(&pRawOutput->au8Data[0],(tU16)enCommand);
            DABDRV_SET_U16(&pRawOutput->au8Data[2],u16SCIDI);
            DABDRV_SET_U16(&pRawOutput->au8Data[4],u16DataGroupTypeMask);
            DABDRV_SET_U16(&pRawOutput->au8Data[6],u16FieldLen);
            if (u16FieldLen) {
                OSAL_pvMemoryCopy(&pRawOutput->au8Data[8],au8EuaField, u16FieldLen);
            }


        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CDdmDataGroup: enCommand=%x u16SCIDI=0x%x u16DataGroupTypeMask=0x%x u16EuaFieldLength=%d au8EuaField=%*p",
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            u16SCIDI,
                            u16DataGroupTypeMask,
                            u16EuaFieldLength,
                            ETG_LIST_LEN(u16EuaFieldLength), ETG_LIST_PTR_T8(au8EuaField)));
        };
    
        tenMeca_DdmCommand enCommand;
        tU16 u16SCIDI;
        tU16 u16DataGroupTypeMask;
        tU16 u16EuaFieldLength;
        tU8 au8EuaField[DABMECA_MAX_EUA_FIELD_LEN];
    };



    // DAB_*_R_DDM_DATAGROUP
    struct trMeca_RDdmDataGroup:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            if (rMostHdr._u16MecaLen < 10) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            u16EuaFieldLength = DABDRV_GET_U16(&pu8Data[8]);
            if (u16EuaFieldLength > DABMECA_MAX_EUA_FIELD_LEN) {
                u16EuaFieldLength = 0;
                return FALSE;
            }
        
            enCommand=(tenMeca_DdmCommand)DABDRV_GET_U16(&pu8Data[0]);
            enResponse=(tenMeca_DdmResponse)DABDRV_GET_U16(&pu8Data[2]);
            u16SCIDI=DABDRV_GET_U16(&pu8Data[4]);
            u16EuaFieldLength=DABDRV_GET_U16(&pu8Data[6]);
            OSAL_pvMemoryCopy(au8EuaField,&pu8Data[10], u16EuaFieldLength);
            return TRUE;
        }
        virtual tVoid vTrace() const {
      /*      ETG_TRACE_USR1(("  trMeca_RDdmDataGroup: enCommand=%x enResponse=%x u16SCIDI=0x%x u16DataGroupTypeMask=0x%x u16EuaFieldLength=%d au8EuaField=%x",
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            ETG_CENUM(tenMeca_DdmResponse, enResponse),
                            u16SCIDI,
                            u16DataGroupTypeMask,
                            u16EuaFieldLength,
                            ETG_LIST_LEN(u16EuaFieldLength), ETG_LIST_PTR_T8(au8EuaField)));*/
        };
    

        tenMeca_DdmCommand enCommand;
        tenMeca_DdmResponse enResponse;
        tU16 u16SCIDI;
        tU16 u16DataGroupTypeMask;
        tU16 u16EuaFieldLength;
        tU8 au8EuaField[DABMECA_MAX_EUA_FIELD_LEN];
    };



    // _DAB_C_DDM_PAD_DATAGROUP
    struct trMeca_CDdmPadDataGroup:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            tU16 u16FieldLen = (tU16) DAB_MIN(u16EuaFieldLength,(tU16)DABMECA_MAX_EUA_FIELD_LEN);
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_DDM_PAD_DATAGROUP;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = (tU16)(12 + u16FieldLen);
            OSAL_pvMemorySet(pRawOutput->au8Data,0,pRawOutput->u16DataLen);

            DABDRV_SET_U16(&pRawOutput->au8Data[0],(tU16)enCommand);
            DABDRV_SET_U16(&pRawOutput->au8Data[2],u16SCIDI);
            DABDRV_SET_U16(&pRawOutput->au8Data[4],u16XpadApplicationType);
            DABDRV_SET_U16(&pRawOutput->au8Data[6],u16DataGroupTypeMask);
            DABDRV_SET_U16(&pRawOutput->au8Data[8],u16FieldLen);
            if (u16FieldLen) {
                OSAL_pvMemoryCopy(&pRawOutput->au8Data[10],au8EuaField, u16FieldLen);
            }
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CDdmPadDataGroup: enCommand=%x u16SCIDI=0x%x u16XpadApplicationType=0x%x u16DataGroupTypeMask=0x%x u16EuaFieldLength=%d au8EuaField=%*p",
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            u16SCIDI,
                            u16XpadApplicationType,
                            u16DataGroupTypeMask,
                            u16EuaFieldLength,
                            ETG_LIST_LEN(u16EuaFieldLength), ETG_LIST_PTR_T8(au8EuaField)));
        };
    
        tenMeca_DdmCommand enCommand;
        tU16 u16SCIDI;
        tU16 u16XpadApplicationType;
        tU16 u16DataGroupTypeMask;
        tU16 u16EuaFieldLength;
        tU8 au8EuaField[DABMECA_MAX_EUA_FIELD_LEN];
    };



    // DAB_*_R_DDM_PAD_DATAGROUP
    struct trMeca_RDdmPadDataGroup:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            if (rMostHdr._u16MecaLen < 14) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            u16EuaFieldLength = DABDRV_GET_U16(&pu8Data[8]);
            if (u16EuaFieldLength > DABMECA_MAX_EUA_FIELD_LEN) {
                u16EuaFieldLength = 0;
                return FALSE;
            }
        
            enCommand=(tenMeca_DdmCommand)DABDRV_GET_U16(&pu8Data[0]);
            enResponse=(tenMeca_DdmResponse)DABDRV_GET_U16(&pu8Data[2]);
            u16SCIDI=DABDRV_GET_U16(&pu8Data[4]);
            u16XpadApplicationType=DABDRV_GET_U16(&pu8Data[6]);
            u16DataGroupTypeMask=DABDRV_GET_U16(&pu8Data[8]);
            u16EuaFieldLength=DABDRV_GET_U16(&pu8Data[10]);
            OSAL_pvMemoryCopy(au8EuaField,&pu8Data[12], u16EuaFieldLength);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RDdmPadDataGroup: enCommand=%x enResponse=%x u16SCIDI=0x%x u16XpadApplicationType=0x%x u16DataGroupTypeMask=0x%x u16EuaFieldLength=%d au8EuaField=%*p",
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            ETG_CENUM(tenMeca_DdmResponse, enResponse),
                            u16SCIDI,
                            u16XpadApplicationType,
                            u16DataGroupTypeMask,
                            u16EuaFieldLength,
                            ETG_LIST_LEN(u16EuaFieldLength), ETG_LIST_PTR_T8(au8EuaField))); 
        };
    

        tenMeca_DdmCommand enCommand;
        tenMeca_DdmResponse enResponse;
        tU16 u16SCIDI;
        tU16 u16XpadApplicationType;

        tU16 u16DataGroupTypeMask;
        tU16 u16EuaFieldLength;
        tU8 au8EuaField[DABMECA_MAX_EUA_FIELD_LEN];
    };


    // _DAB_C_DDM_STREAM
    struct trMeca_CDdmStream:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_DDM_STREAM;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 4;
            DABDRV_SET_U16(&pRawOutput->au8Data[0],(tU16)enCommand);
            DABDRV_SET_U16(&pRawOutput->au8Data[2],u16SCIDI);

        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CDdmStream: enCommand=%x u16SCIDI=0x%x",
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            u16SCIDI));
        };
    
        tenMeca_DdmCommand enCommand;
        tU16 u16SCIDI;
    

    };


    // DAB_*_R_DDM_STREAM
    struct trMeca_RDdmStream:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            if (rMostHdr._u16MecaLen != 8) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enCommand=(tenMeca_DdmCommand)DABDRV_GET_U16(&pu8Data[0]);
            enResponse=(tenMeca_DdmResponse)DABDRV_GET_U16(&pu8Data[2]);
            u16SCIDI = DABDRV_GET_U16(&pu8Data[4]);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RDdmStream: enCommand=%x enResponse=%x u16SCIDI=0x%x",
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            ETG_CENUM(tenMeca_DdmResponse, enResponse),
                            u16SCIDI));
        };
    
        tenMeca_DdmCommand enCommand;
        tenMeca_DdmResponse enResponse;
        tU16 u16SCIDI;

    };


    // _DAB_C_DDM_PAD
    struct trMeca_CDdmPad:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_DDM_PAD;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 4;
            DABDRV_SET_U16(&pRawOutput->au8Data[0],(tU16)enCommand);
            DABDRV_SET_U16(&pRawOutput->au8Data[2],u16SCIDI);

        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CDdmPad: enCommand=%x u16SCIDI=0x%x",
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            u16SCIDI));
        };
    
        tenMeca_DdmCommand enCommand;
        tU16 u16SCIDI;
    

    };


    // DAB_*_R_DDM_PAD
    struct trMeca_RDdmPad:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            if (rMostHdr._u16MecaLen != 6) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enCommand=(tenMeca_DdmCommand)DABDRV_GET_U16(&pu8Data[0]);
            enResponse=(tenMeca_DdmResponse)DABDRV_GET_U16(&pu8Data[2]);
            u16SCIDI = DABDRV_GET_U16(&pu8Data[4]);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RDdmPad: enCommand=%x enResponse=%x u16SCIDI=0x%x",
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            ETG_CENUM(tenMeca_DdmResponse, enResponse),
                            u16SCIDI));
        };
    
        tenMeca_DdmCommand enCommand;
        tenMeca_DdmResponse enResponse;
        tU16 u16SCIDI;

    };


    // _DAB_C_DDM_APP_TYPE_PAD
    struct trMeca_CDdmAppTypePad:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_DDM_APP_TYPE_PAD;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 6;
            DABDRV_SET_U16(&pRawOutput->au8Data[0],(tU16)enCommand);
            DABDRV_SET_U16(&pRawOutput->au8Data[2],u16SCIDI);
            DABDRV_SET_U16(&pRawOutput->au8Data[4],u16AppType);

        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CDdmAppTypePad: enCommand=%x u16SCIDI=0x%x u16AppType=0x%x",
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            u16SCIDI,
                            u16AppType));
        };
    
        tenMeca_DdmCommand enCommand;
        tU16 u16SCIDI;
        tU16 u16AppType;
    

    };


    // DAB_*_R_DDM_APP_TYPE_PAD
    struct trMeca_RDdmAppTypePad:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            if (rMostHdr._u16MecaLen != 8) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enCommand=(tenMeca_DdmCommand)DABDRV_GET_U16(&pu8Data[0]);
            enResponse=(tenMeca_DdmResponse)DABDRV_GET_U16(&pu8Data[2]);
            u16SCIDI = DABDRV_GET_U16(&pu8Data[4]);
            u16AppType = DABDRV_GET_U16(&pu8Data[6]);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RDdmAppTypePad: enCommand=%x enResponse=%x u16SCIDI=0x%x, u16AppType=0x%x",
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            ETG_CENUM(tenMeca_DdmResponse, enResponse),
                            u16SCIDI,
                            u16AppType));
        };
    
        tenMeca_DdmCommand enCommand;
        tenMeca_DdmResponse enResponse;
        tU16 u16SCIDI;
        tU16 u16AppType;

    };



    // _DAB_C_DDM_RADIOTEXT_CURRENT
    struct trMeca_CDdmRadioTextCurrent:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_DDM_RADIOTEXT_CURRENT;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 5;
            DABDRV_SET_U16(&pRawOutput->au8Data[0],(tU16)enCommand);
            pRawOutput->au8Data[2] = (tU8)((bExtractControlChars ? 1 : 0) | (bAutoMode ? 2 : 0) | (bRepeatMode ? 4 : 0));
            pRawOutput->au8Data[3] = u8StartPos;
            pRawOutput->au8Data[4] = u8SegmentLength;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CDdmRadioTextCurrent: enCommand=%x bExtractControlChars=%d bAutoMode=%d "
                            "u8StartPos=%d u8SegmentLength=%d",
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            bExtractControlChars,
                            bAutoMode,
                            u8StartPos,
                            u8SegmentLength));
        };
    
        tenMeca_DdmCommand enCommand;
        tBool bExtractControlChars;
        tBool bAutoMode;
		tBool bRepeatMode;
        tU8 u8StartPos;
        tU8 u8SegmentLength;
    };



    // DAB_*_R_DDM_RADIOTEXT_CURRENT
    struct trMeca_RDdmRadioTextCurrent:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            if (rMostHdr._u16MecaLen < 10) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enCommand=(tenMeca_DdmCommand)DABDRV_GET_U16(&pu8Data[0]);
            enResponse=(tenMeca_DdmResponse)DABDRV_GET_U16(&pu8Data[2]);
            bExtractControlChars=DAB_BOOL_FROM_BIT(pu8Data[4], DAB_b0);
            bAutoMode=DAB_BOOL_FROM_BIT(pu8Data[4], DAB_b1);
			bRepeatMode=DAB_BOOL_FROM_BIT(pu8Data[4], DAB_b2);

            bSegment=DAB_BOOL_FROM_BIT(pu8Data[5], DAB_b0);
            bNew=DAB_BOOL_FROM_BIT(pu8Data[5], DAB_b2);
            bComplete=DAB_BOOL_FROM_BIT(pu8Data[5], DAB_b1);
            u8StartPos = pu8Data[6];
            u8Length = pu8Data[7];
            u8SegmentLength = pu8Data[8];
            enCharSet = (tenMeca_CharSet)pu8Data[9];
            if (rMostHdr._u16MecaLen != 10 + u8SegmentLength) {
                return FALSE;
            }
            OSAL_pvMemoryCopy(u8RadioTextSegment,&pu8Data[10], u8SegmentLength);
            u8RadioTextSegment[u8SegmentLength]=0;
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RDdmRadioTextCurrent: enCommand=%x enResponse=%x bExtractControlChars=%d "
                            " bAutoMode=%d bRepeatMode=%d bSegment=%d bNew=%d bComplete=%d u8StartPos=%d u8Length=%d",
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            ETG_CENUM(tenMeca_DdmResponse, enResponse),
                            bExtractControlChars,
                            bAutoMode,
							bRepeatMode,
                            bSegment,
                            bNew,
                            bComplete,
                            u8StartPos,
                            u8Length));
            ETG_TRACE_USR1(("  trMeca_RDdmRadioTextCurrent (cont): u8SegmentLength=%d enCharSet=%x u8RadioTextSegment=%*p",
                            u8SegmentLength,
                            ETG_CENUM(tenMeca_CharSet, enCharSet),
                            ETG_LIST_LEN(u8SegmentLength), ETG_LIST_PTR_T8(u8RadioTextSegment)));
        };
    
        tenMeca_DdmCommand enCommand;
        tenMeca_DdmResponse enResponse;
        tBool bExtractControlChars;
        tBool bAutoMode;
		tBool bRepeatMode;
        tBool bSegment;
        tBool bNew;
        tBool bComplete;
        tU8 u8StartPos;
        tU8 u8Length;
        tU8 u8SegmentLength;
        tenMeca_CharSet enCharSet;
        tU8 u8RadioTextSegment[0xFF+1];

    };

    typedef enum {
        enMeca_DdmResponse_TMC_GROUP_3A          = 0x03,
        enMeca_DdmResponse_TMC_GROUP_8A          = 0x08
    } tenMeca_DdmResponse_TMC_GROUP;


    typedef enum {
		enMeca_DdlCommand_RemoveLabel		= 0x0001,
		enMeca_DdlCommand_DLPlus          	= 0x0002,
		enMeca_DdlCommand_Non_Command		= 0xffff
    } tenMeca_DdlCommand;

    struct trDLPlusTag
	{
		tU8 u8ContentType;
		tU8 u8StartMarker;
		tU8 u8LengthMarker;

		trDLPlusTag():
			u8ContentType(0),
			u8StartMarker(0),
			u8LengthMarker(0)
		{
		}
	};

#define DABMECA_RADIOTEXT_CMD_CID_LEN    4
#define DABMECA_RADIOTEXT_CMD_ITBIT_LEN  1
#define DABMECA_RADIOTEXT_CMD_IRBIT_LEN  1
#define DABMECA_RADIOTEXT_CMD_NUMTAG_LEN 2

	struct trCommandHead
	{
    	trCommandHead():
    		CId(0),ITBit(0),IRBit(0),NumOfTags(0)
    	{ }

    	tU8 CId 		;//: DABMECA_RADIOTEXT_CMD_CID_LEN;
		tU8 ITBit  		;//: DABMECA_RADIOTEXT_CMD_ITBIT_LEN;
		tU8 IRBit  		;//: DABMECA_RADIOTEXT_CMD_IRBIT_LEN;
		tU8 NumOfTags 	;//: DABMECA_RADIOTEXT_CMD_NUMTAG_LEN;
	};



    // DAB_*_R_DDM_RADIOTEXT_CURRENT_CMD
	struct trMeca_RDdmRadioTextCurrentCmd:
		public trMsgMecaInput {
		DAB_DISPATCH_IMPL

		virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
			if (rMostHdr._u16MecaLen < 10) {
				return FALSE;
			}

			_rMostHdr=rMostHdr;
			enDlCommand=(tenMeca_DdlCommand)DABDRV_GET_U16(&pu8Data[0]);
			enServiceType=(tenMeca_ServiceType)pu8Data[3];
			rProgrammeService.vParse(&pu8Data[4]);
			u16SCIDI = DABDRV_GET_U16(&pu8Data[8]);
			u16Length=DABDRV_GET_U16(&pu8Data[10]);

			if(enDlCommand == enMeca_DdlCommand_DLPlus)
			{
				(tVoid)OSAL_pvMemoryCopy(u8CommandField, &pu8Data[12], u16Length);

				rDLPlusCH.NumOfTags = (0x3 & u8CommandField[0]);
				rDLPlusCH.IRBit = (0x1 & (u8CommandField[0]>>2));
				rDLPlusCH.ITBit = (0x1 & (u8CommandField[0]>>3));
				rDLPlusCH.CId =  (tU8)(0xF & (u8CommandField[0]>>4));

				for (tU8 i=0, tagPos=1;i<=rDLPlusCH.NumOfTags;i++,tagPos=(tU8)(tagPos+3)) {
							trDLPlusTag tempTag;
							tempTag.u8ContentType = u8CommandField[tagPos];
							tempTag.u8StartMarker = u8CommandField[tagPos+1];
							tempTag.u8LengthMarker = u8CommandField[tagPos+2];

							lDLPlusTag.push_back(tempTag);
				}
			}
			else
			{
				ETG_TRACE_USR1(("  trMeca_RDdmRadioTextCurrentCmd: enDlCommand=%x ",
									ETG_CENUM(tenMeca_DdlCommand, enDlCommand)));
			}


			return TRUE;
		}
		virtual tVoid vTrace() const {
			ETG_TRACE_USR1(("  trMeca_RDdmRadioTextCurrentCmd: enDlCommand=%x  enServiceType=%d sid=0x%08x u8Length=%d u16SCIDI=0x%x NumOfTags=%d",
					ETG_CENUM(tenMeca_DdlCommand, enDlCommand),
					ETG_CENUM(tenMeca_ServiceType, enServiceType),
					rProgrammeService.u32GetSID(),
					u16Length,
					u16SCIDI,
					rDLPlusCH.NumOfTags));

		};

		tenMeca_DdlCommand enDlCommand;
		tenMeca_ServiceType enServiceType;
		trMecaProgrammeService rProgrammeService;
		tU16 u16SCIDI;
		tU16 u16Length;
		tU8 u8CommandField[0xFF+1];
		trCommandHead rDLPlusCH;

		vector<trDLPlusTag> lDLPlusTag;
	};


    // _DAB_C_DDM_TMC_SC
    struct trMeca_CDdmTmcSc:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_DDM_TMC_SC;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 10;
            DABDRV_SET_U16(&pRawOutput->au8Data[0],(tU16)enCommand);
            pRawOutput->au8Data[2]=u8Flags;
            pRawOutput->au8Data[3]=(tU8)enServiceType;
            rMecaProgrammeService.vSerialize(&pRawOutput->au8Data[4]);
            DABDRV_SET_U16(&pRawOutput->au8Data[8],u16SCIDI);

        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CDdmTmcSc: enServiceType=%d sid=0x%08x enCommand=%x u8Flags=0x%x u16SCIDI=0x%x",
                            ETG_CENUM(tenMeca_ServiceType, enServiceType),
                            rMecaProgrammeService.u32GetSID(),
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            u8Flags,
                            u16SCIDI));

        };
    
        tenMeca_ServiceType enServiceType;
        tenMeca_DdmCommand enCommand;
        tU8 u8Flags;
        trMecaProgrammeService rMecaProgrammeService;
        tU16 u16SCIDI;

    

    };


    // DAB_*_R_DDM_TMC_SC
    struct trMeca_RDdmTmcSc:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            if (rMostHdr._u16MecaLen != 12) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enCommand=(tenMeca_DdmCommand)DABDRV_GET_U16(&pu8Data[0]);
            enResponse=(tenMeca_DdmResponse)DABDRV_GET_U16(&pu8Data[2]);
            u8Flags=pu8Data[4];
            rMecaProgrammeService.vParse(&pu8Data[6]);
            enServiceType= (tenMeca_ServiceType)pu8Data[5];
            u16SCIDI = DABDRV_GET_U16(&pu8Data[10]);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RDdmTmcSc: enServiceType=%d sid=0x%08x enCommand=%x enResponse=%x u8Flags=0x%x u16SCIDI=0x%x",
                            ETG_CENUM(tenMeca_ServiceType, enServiceType),
                            rMecaProgrammeService.u32GetSID(),
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            ETG_CENUM(tenMeca_DdmResponse, enResponse),
                            u8Flags,
                            u16SCIDI));
        };
    
        tenMeca_ServiceType enServiceType;
        tenMeca_DdmCommand enCommand;
        tenMeca_DdmResponse enResponse;
        tU8 u8Flags;
        trMecaProgrammeService rMecaProgrammeService;

        tU16 u16SCIDI;

    };



    // _DAB_C_DDM_TMC_CURRENT
    struct trMeca_CDdmTmcCurrent:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_DDM_TMC_CURRENT;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 4;
            DABDRV_SET_U16(&pRawOutput->au8Data[0],(tU16)enCommand);
            pRawOutput->au8Data[2]=u8FilterId;
            pRawOutput->au8Data[3]=u8Flags;

        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CDdmTmcCurrent: enCommand=%x u8FilterId=0x%x u8Flags=0x%x",
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            u8FilterId,
                            u8Flags));

        };
    
        tenMeca_DdmCommand enCommand;
        tU8 u8FilterId;
        tU8 u8Flags;
    };


    // DAB_*_R_DDM_TMC_CURRENT
    struct trMeca_RDdmTmcCurrent:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            if (rMostHdr._u16MecaLen != 17) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enCommand=(tenMeca_DdmCommand)DABDRV_GET_U16(&pu8Data[0]);
            enResponse=(tenMeca_DdmResponse)DABDRV_GET_U16(&pu8Data[2]);
            u8FilterId=pu8Data[4];
            rMecaProgrammeService.vParse(&pu8Data[6]);
            enServiceType= (tenMeca_ServiceType)pu8Data[5];
            u16SCIDI = DABDRV_GET_U16(&pu8Data[10]);
            u16Aid = DABDRV_GET_U16(&pu8Data[12]);
            u16RdbId = DABDRV_GET_U16(&pu8Data[14]);
            u8Flags = pu8Data[16];
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RDdmTmcCurrent: enServiceType=%d sid=0x%08x enCommand=%x enResponse=%x u8FilterId=0x%x u16SCIDI=0x%x "
                            "u16Aid=0x%x u16RdbId=0x%x u8Flags=0x%x",
                            ETG_CENUM(tenMeca_ServiceType, enServiceType),
                            rMecaProgrammeService.u32GetSID(),
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            ETG_CENUM(tenMeca_DdmResponse, enResponse),
                            u8FilterId,
                            u16SCIDI,
                            u16Aid,
                            u16RdbId,
                            u8Flags));
        };
    
        tenMeca_ServiceType enServiceType;
        tenMeca_DdmCommand enCommand;
        tenMeca_DdmResponse enResponse;
        tU8 u8FilterId;

        trMecaProgrammeService rMecaProgrammeService;

        tU16 u16SCIDI;
        tU16 u16Aid;
        tU16 u16RdbId;
        tU8 u8Flags;

    };

    // DAB_*_R_DDM_TMC_EVENT
    struct trMeca_RDdmTmcEvent:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            ETG_TRACE_USR1(("  trMeca_RDdmTmcEvent:bParse _u16MecaLen=%d",rMostHdr._u16MecaLen));
            if (rMostHdr._u16MecaLen != 14) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            rMecaProgrammeService.vParse(&pu8Data[2]);
            enServiceType = (tenMeca_ServiceType)pu8Data[1];
            s16Ecc = (tS16)DABDRV_GET_U16(&pu8Data[3]);
            u16SCIDI = DABDRV_GET_U16(&pu8Data[6]);
            enTmcGroup = (tenMeca_DdmResponse_TMC_GROUP)pu8Data[8];
            OSAL_pvMemoryCopy(u8TmcData,&pu8Data[9], 5);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RDdmTmcEvent: enServiceType=%d sid=0x%08x s16Ecc=0x%x u16SCIDI=0x%x enTmcGroup=0x%x u8TmcData=%*p",
                            ETG_CENUM(tenMeca_ServiceType, enServiceType),
                            rMecaProgrammeService.u32GetSID(),
                            s16Ecc,
                            u16SCIDI,
                            ETG_CENUM(tenMeca_DdmResponse_TMC_GROUP, enTmcGroup),
                            ETG_LIST_LEN(5), ETG_LIST_PTR_T8(u8TmcData)));

        };
    

        tenMeca_ServiceType enServiceType;
        trMecaProgrammeService rMecaProgrammeService;

        tS16 s16Ecc;
        tU16 u16SCIDI;
        tenMeca_DdmResponse_TMC_GROUP enTmcGroup;
        tU8 u8TmcData[5];
    };

    // _DAB_C_DDM_RDI_SUBCH
    struct trMeca_CDdmRdiSubCh:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_DDM_RDI_SUBCH;
            pRawOutput->enOpType = enOpType_SET;
            pRawOutput->u16DataLen = 5;
            DABDRV_SET_U16(&pRawOutput->au8Data[0],(tU16)enCommand);
            DABDRV_SET_U16(&pRawOutput->au8Data[2],u16SubCh);
            pRawOutput->au8Data[4]=0;

        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CDdmRdiSubCh: enCommand=%x u16SubCh=0x%x",
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            u16SubCh));
        };
    
        tenMeca_DdmCommand enCommand;
        tU16 u16SubCh;
    };



    // DAB_*_R_DDM_RDI_SUBCH
    struct trMeca_RDdmRdiSubCh:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            if (rMostHdr._u16MecaLen != 7) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            enCommand=(tenMeca_DdmCommand)DABDRV_GET_U16(&pu8Data[0]);
            enResponse=(tenMeca_DdmResponse)DABDRV_GET_U16(&pu8Data[2]);
            u16SubCh=DABDRV_GET_U16(&pu8Data[4]);
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RDdmRdiSubCh: enCommand=%x enResponse=%x u16SubCh=0x%x",
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            ETG_CENUM(tenMeca_DdmResponse, enResponse),
                            u16SubCh));
        };
    
        tenMeca_DdmCommand enCommand;
        tenMeca_DdmResponse enResponse;
        tU16 u16SubCh;


    };


    // _DAB_C_DDM_SELECT_URI  
    struct trMeca_CDdmSelectUri:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            tU16 u16UriLenght = (tU16)cBuffer.size();
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = MECA_DISPATCHER_C_DDM_SELECT_URI;
            pRawOutput->enOpType = enOpType_GET;
            pRawOutput->u16DataLen = (tU8)(12+u16UriLenght+2);
            DABDRV_SET_U16(&pRawOutput->au8Data[0],(tU16)enCommand);
            pRawOutput->au8Data[2]=u8TOn;                   // b-t_on
            pRawOutput->au8Data[3]=u8TPause;                   // b-t_pause
            pRawOutput->au8Data[4]=u8Priority;                   // b-priority
            pRawOutput->au8Data[5]=u8TDecodeLevel;                   // b-decode_level
            DABDRV_SET_U16(&pRawOutput->au8Data[6],u16Rfu1);  // w-rfu1
            DABDRV_SET_U16(&pRawOutput->au8Data[8],u16Rfu2);  // w-rfu2
            DABDRV_SET_U16(&pRawOutput->au8Data[10],u16UriLenght);// w-uri-len
            for (tU16 i=0;i<u16UriLenght;i++) {
                pRawOutput->au8Data[12+i] = cBuffer[i];
            }
            DABDRV_SET_U16(&pRawOutput->au8Data[12+u16UriLenght],u16Rfu);
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_CDdmSelectUri: selected enCommand=%x ",
                                ETG_CENUM(tenMeca_DdmCommand, enCommand)));
        };
    
		tVoid vUpdate(tU8* pu8Data){
			  enCommand=(tenMeca_DdmCommand)DABDRV_GET_U16(&pu8Data[0]);
			  u8TOn=pu8Data[2];
			  u8TPause =pu8Data[3];
			  u8Priority =pu8Data[4];
			  u8TDecodeLevel=pu8Data[5];
			  u16Rfu1=DABDRV_GET_U16(&pu8Data[6]);
			  u16Rfu2=DABDRV_GET_U16(&pu8Data[8]);
			  u16URILength=DABDRV_GET_U16(&pu8Data[10]);
			  ETG_TRACE_USR1(("  trMeca_CDdmSelectUri: u16URILength=%d ",
                                u16URILength));
			  for (tU16 i=0;i<u16URILength;i++) {
				  cBuffer.push_back(pu8Data[12+i]);
			  }
			   ETG_TRACE_USR1(("  trMeca_CDdmSelectUri: u16URILength After =%d ",
                                u16URILength));
				u16Rfu=DABDRV_GET_U16(&pu8Data[12+u16URILength]);
				 ETG_TRACE_USR1(("  trMeca_CDdmSelectUri: u16URILength=%d ",
                                u16URILength));
		}
        tenMeca_DdmCommand enCommand;
		tU8 u8TOn;
		tU8 u8TPause;
		tU8 u8Priority;
		tU8 u8TDecodeLevel;
		tU16 u16Rfu1;
		tU16 u16Rfu2;
		tU16 u16URILength;
		tU16 u16Rfu;
        std::vector<char> cBuffer;
    };



    // DAB_*_R_DDM_SELECT_URI
    struct trMeca_RDdmSelectUri:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {

            _rMostHdr=rMostHdr;
            enCommand=(tenMeca_DdmCommand)DABDRV_GET_U16(&pu8Data[0]);
            enResponse=(tenMeca_DdmResponse)DABDRV_GET_U16(&pu8Data[2]);
            u8TimeOn = pu8Data[4];
            u8TimePause = pu8Data[5];
            u8Priority = pu8Data[6];
            u8DecodeLevel = pu8Data[7];
            u16UriLen = DABDRV_GET_U16(&pu8Data[12]);

            for (tU8 i=0;i<u16UriLen;i++) {
                lu8UriData.push_back(pu8Data[10+i]);
            }
            
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RDdmSelectUri: enCommand=%x enResponse=%x",
                            ETG_CENUM(tenMeca_DdmCommand, enCommand),
                            ETG_CENUM(tenMeca_DdmResponse, enResponse)));
        };
    
        tenMeca_DdmCommand enCommand;
        tenMeca_DdmResponse enResponse;
        tU8 u8TimeOn;
        tU8 u8TimePause;
        tU8 u8Priority;
        tU8 u8DecodeLevel;
        tU16 u16UriLen;
        vector<tU8> lu8UriData; 

    };

const tU16 MOST_FB_DATACHANNEL_MSG = 0x0d0c;

    typedef enum {
        enMeca_DcSourceApi_Undefined = 0x00,
        enMeca_DcSourceApi_DDM       = 0x01,
        enMeca_DcSourceApi_Rfu       = 0x02,
        enMeca_DcSourceApi_Expert    = 0x03
    } tenMeca_DcSourceApi;

    typedef enum {
        enMeca_DcSourceType_Undefined         = 0x00,
        enMeca_DcSourceType_Service_Component = 0x01,
        enMeca_DcSourceType_Subchannel        = 0x02,
        enMeca_DcSourceType_FIDC              = 0x03,
        enMeca_DcSourceType_FIC               = 0x04
    } tenMeca_DcSourceType;

    typedef enum {
        enMeca_DcDataType_Undefined         = 0x00,
        enMeca_DcDataType_Packets           = 0x01,
        enMeca_DcDataType_Data_Groups       = 0x02,
        enMeca_DcDataType_Stream            = 0x03,
        enMeca_DcDataType_PAD               = 0x06,
        enMeca_DcDataType_PAD_Data_Groups   = 0x07,
        enMeca_DcDataType_APP_TYPE_PAD      = 0x08,
        enMeca_DcDataType_FIBs              = 0x09,
        enMeca_DcDataType_TDC               = 0x0c
    } tenMeca_DcDataType;

// DAB_*_R_DDM_DATACHANNEL
    struct trMeca_RDdmDataChannel:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {

            _rMostHdr=rMostHdr;

            if (MOST_FB_DATACHANNEL_MSG != DABDRV_GET_U16(&pu8Data[0])) {
                return FALSE;
            }

            enMeca_DcSourceApi = (tenMeca_DcSourceApi)pu8Data[2];
            enMeca_DcSourceType = (tenMeca_DcSourceType)pu8Data[3];
            enMeca_DcDataType = (tenMeca_DcDataType)pu8Data[4];
            u32ExtLength = DABDRV_GET_U32(&pu8Data[5]);
            if (u32ExtLength != 0) {
                for (tU32 i=0;i<u32ExtLength;i++) {
                    lu8ExtData.push_back(pu8Data[9+i]);
                }
            }
            u32DataLength = DABDRV_GET_U32(&pu8Data[9+u32ExtLength]);;
            if (u32DataLength != 0) {
                for (tU32 i=0;i<u32DataLength;i++) {
                    lu8Data.push_back(pu8Data[13+u32ExtLength+i]);
                }
            }
            return TRUE;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMeca_RDdmDataChannel: enMeca_DcSourceApi=%x enMeca_DcSourceType=%x enMeca_DcDataType=%x extLength=%d DataLength=0x%x",
                            ETG_CENUM(tenMeca_DcSourceApi, enMeca_DcSourceApi),
                            ETG_CENUM(tenMeca_DcSourceType, enMeca_DcSourceType),
                            ETG_CENUM(tenMeca_DcDataType, enMeca_DcDataType),
                            u32ExtLength,
                            u32DataLength));
        };
    
        tenMeca_DcSourceApi enMeca_DcSourceApi;
        tenMeca_DcSourceType enMeca_DcSourceType;
        tenMeca_DcDataType enMeca_DcDataType;
        tU32 u32ExtLength;
        vector<tU8> lu8ExtData; 
        tU32 u32DataLength;
        vector<tU8> lu8Data; 

    };



} // namespace DAB


#endif
