/************************************************************************
 * FILE:        dabdrv_types.hpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of dabdrv_types
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      	| Author                       | Modification
				  
				
 *************************************************************************/

#ifndef DABDRV_TYPES_H
#define DABDRV_TYPES_H
#include "dabmeca_rdm.hpp"
#include "dabmeca_tsu.hpp"
#include "dabmeca_exp.hpp"
#include "dabdrv_typeEnums.h"
#include "fc_dabtuner_util.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_UTIL_MSG 
#include "trcGenProj/Header/dabdrv_types.hpp.trc.h"
#endif

#ifndef RDM_DONE
#error
#endif


#ifndef __RFC3072_HELPER_H__
#include "rfc3072_helper.h"
#endif

namespace DAB {

    inline tU32 DAB_u32RemoveCC(tU32 u32WithCc) {
        return u32WithCc & 0xFF00FFFF;
    }

    inline tVoid DAB_vRemoveCC(tU32 *pu32WithCc) {
        *pu32WithCc &= 0xFF00FFFF;
    }

    inline tBool DAB_bIdEqNoCC(tU32 id1, tU32 id2) {
        return ((id1 & 0xFF00FFFF) == (id2 & 0xFF00FFFF));
    }
    struct trMecaId {
        tU32 _u32Id;
        tenMeca_ServiceType _enServiceType;

        explicit trMecaId(tU32 u32Sid=0, tenMeca_ServiceType enServType=enMeca_ServiceType_AUDIO_SERVICE):
            _u32Id(u32Sid),
            _enServiceType(enServType)
        {}

        trMecaId(trMecaProgrammeService const &rProgService) {
            _u32Id = rProgService._u32Id;
            _enServiceType=rProgService._enServiceType;
        }
        trMecaId(trMecaEnsemble const &rEnsemble) {
            _u32Id = rEnsemble._u32Id;
            _enServiceType=enMeca_ServiceType_INVALID;
        }

        tBool bIsValid() const {
            return (_u32Id != 0);
        }

        tVoid vInvalidate() {
            _u32Id=0;
        }
        tU16 u16GetScidi() const {
            return (tU16)_u32Id;
        };

        tU32 u32GetSID() const {
            return _u32Id;
        };

        tU16 u16GetSID() const {
            return (tU16)_u32Id;
        };

        trMecaProgrammeService rGetProgService() const {
            trMecaProgrammeService rProgService;
            rProgService._u32Id=_u32Id;
            return rProgService;
        }
        tU8 u8GetSubChannel() const  {
            return (tU8)(_u32Id & 0x3F);
        }

        // persistant data serialization
        tVoid vSerialize(DAB_Serializer &rSerializer) const {
            rSerializer.vSerialize(_u32Id);
        }
        
        tVoid vParse(DAB_Parser &rParser) {
            rParser.vParse(_u32Id);
        }



    };

	 struct trChnListKey {
        trMecaId _rMecaId;

#ifdef COMP_SERV_LIST
		tU16 _u16Scids;

		trChnListKey():
			_rMecaId(),
			_u16Scids(0)
		{}

		trChnListKey(trMecaId rMecaId, tU16 u16Scids=0):
			_rMecaId(rMecaId._u32Id,rMecaId._enServiceType)	,
			_u16Scids(u16Scids)
		{}

		trChnListKey(trMecaProgrammeService const &rProgService):
			_rMecaId(rProgService._u32Id,rProgService._enServiceType),
			_u16Scids(rProgService._u16Scids)
		{}

#else
		trChnListKey():
			_rMecaId()
		{}

		trChnListKey(trMecaId rMecaId):
			_rMecaId(rMecaId._u32Id,rMecaId._enServiceType)		
		{}

		trChnListKey(trMecaProgrammeService const &rProgService):
			_rMecaId(rProgService._u32Id,rProgService._enServiceType)	
		{}
#endif

		// persistant data serialization
        tVoid vSerialize(DAB_Serializer &rSerializer) const {
			rSerializer.vSerialize(_rMecaId._u32Id);
			rSerializer.vSerialize(_u16Scids);
        }
        
        tVoid vParse(DAB_Parser &rParser) {
			rParser.vParse(_rMecaId._u32Id);
			rParser.vParse(_u16Scids);
        }
	

	 };

    inline tBool operator==(const trMecaId& left, const trMecaId& right) {
        return DAB_ID_EQ(left._u32Id, right._u32Id);
    }
#ifdef COMP_SERV_LIST
    inline tBool operator==(const trChnListKey& left, const trChnListKey& right) {
		return (DAB_ID_EQ(left._rMecaId._u32Id, right._rMecaId._u32Id) && (left._u16Scids == right._u16Scids));
    }
#else
	inline tBool operator==(const trChnListKey& left, const trChnListKey& right) {
		return DAB_ID_EQ(left._rMecaId._u32Id, right._rMecaId._u32Id);
    }
#endif

    inline tBool operator!=(const trMecaId& left, const trMecaId& right) {
        return DAB_ID_NEQ(left._u32Id, right._u32Id);
    }
#ifdef COMP_SERV_LIST
	inline tBool operator!=(const trChnListKey& left, const trChnListKey& right) {
		return (DAB_ID_NEQ(left._rMecaId._u32Id, right._rMecaId._u32Id) && (left._u16Scids != right._u16Scids));
    }
#else
	inline tBool operator!=(const trChnListKey& left, const trChnListKey& right) {
		return DAB_ID_NEQ(left._rMecaId._u32Id, right._rMecaId._u32Id);
    }
#endif

    inline tBool operator<(const trMecaId& left, const trMecaId& right) {
        return DAB_ID_LESS(left._u32Id, right._u32Id);
    }

	inline tBool operator>(const trMecaId& left, const trMecaId& right) {
        return DAB_ID_GREATER(left._u32Id, right._u32Id);
    }
#ifdef COMP_SERV_LIST
	inline tBool operator<(const trChnListKey& left, const trChnListKey& right) {
		return (left._rMecaId<right._rMecaId || (left._rMecaId==right._rMecaId && left._u16Scids<right._u16Scids));
	}
#else
	inline tBool operator<(const trChnListKey& left, const trChnListKey& right) {
		return (left._rMecaId<right._rMecaId);
	}
#endif

    inline tBool operator==(const trMecaProgrammeService& left, const trMecaId& right) {
        return DAB_ID_EQ(left._u32Id, right._u32Id);
    }
	inline tBool operator==(const trMecaProgrammeService& left, const trChnListKey& right) {
		return DAB_ID_EQ(left._u32Id, right._rMecaId._u32Id);
    }

    inline tBool operator!=(const trMecaProgrammeService& left, const trMecaId& right) {
        return DAB_ID_NEQ(left._u32Id, right._u32Id);
    }

    inline tBool operator<(const trMecaProgrammeService& left, const trMecaId& right) {
        return DAB_ID_LESS(left._u32Id, right._u32Id);
    }

    inline tBool operator==(const trMecaId& left, const trMecaProgrammeService& right) {
        return DAB_ID_EQ(left._u32Id, right._u32Id);
    }
    inline tBool operator==(const trChnListKey& left, const trMecaProgrammeService& right) {
		return DAB_ID_EQ(left._rMecaId._u32Id, right._u32Id);
    }
    inline tBool operator!=(const trMecaId& left, const trMecaProgrammeService& right) {
        return DAB_ID_NEQ(left._u32Id, right._u32Id);
    }

    inline tBool operator<(const trMecaId& left, const trMecaProgrammeService& right) {
        return DAB_ID_LESS(left._u32Id, right._u32Id);
    }

    typedef enum {
            DAB_enProfile_Change = 1,
            DAB_enProfile_Delete = 2,
            DAB_enProfileCopy = 3
        } DAB_tenProfileUpdateType;

    struct trChnListElem {

        trChnListElem():
            u16Id(0),
            rMecaId(),
            //            rProgService(),
            u16Scidi(0),
			u16Scids(0),
            bReception(FALSE),
            rLabel(),
			rEnsembleLabel(),
            u8PresetNumber(0),
			u8PTY(0),
			u8NumberOfComponents(0),
			bPSFlag(TRUE),			
			sFreqLabel(""),
			sLogoLink("/0"),
			u8DSSupport(0)
        {}

            //            rProgService(),
        tU16 u16Id;
        trMecaId rMecaId;
        //        trMecaProgrammeService rProgService;
        tU16 u16Scidi;
		tU16 u16Scids;
        tBool bReception;
        trMecaLabel rLabel;
		trMecaLabel rEnsembleLabel;
        tU8 u8PresetNumber;
		tU8 u8PTY;
		tU8 u8NumberOfComponents;
		tBool bPSFlag;
		string sFreqLabel;

		string sLogoLink;
		
		tU8 u8DSSupport;

        tBool bIsValid() const {
            return rMecaId._u32Id != 0;
        }
        bool operator!=(trChnListElem const& b) const
        {
            DAB_RET_1_IF_NEQ(u16Id);
            DAB_RET_1_IF_NEQ(rMecaId);
            DAB_RET_1_IF_NEQ(u16Scidi);
			DAB_RET_1_IF_NEQ(u16Scids);
            DAB_RET_1_IF_NEQ(bReception);
            DAB_RET_1_IF_NEQ(rLabel);
			DAB_RET_1_IF_NEQ(rEnsembleLabel);
            DAB_RET_1_IF_NEQ(u8PresetNumber);
			DAB_RET_1_IF_NEQ(u8PTY);
			DAB_RET_1_IF_NEQ(u8NumberOfComponents);
			DAB_RET_1_IF_NEQ(bPSFlag);
            return FALSE;
        }
			 bool operator==(trChnListElem const& b) const
        {       
            DAB_RET_0_IF_NEQ(u16Id);
            DAB_RET_0_IF_NEQ(rMecaId);
            DAB_RET_0_IF_NEQ(u16Scidi);
			DAB_RET_0_IF_NEQ(u16Scids);
            DAB_RET_0_IF_NEQ(bReception);
            DAB_RET_0_IF_NEQ(rLabel);
			DAB_RET_0_IF_NEQ(rEnsembleLabel);
            DAB_RET_0_IF_NEQ(u8PresetNumber);
			DAB_RET_0_IF_NEQ(u8PTY);
			DAB_RET_0_IF_NEQ(u8NumberOfComponents);
			DAB_RET_0_IF_NEQ(bPSFlag);
            return FALSE;
        }
    };

	 struct trEnsembleListElem {

        trEnsembleListElem():
			u16Id(0),
            rEnsemble(),
            rEnsembleLabel(),
            u32Frequency(0),
			u8NumberOfAudioServices(0),
			u8NumberOfDataServices(0),
			u8ReceptionQuality(0),
			bReception(FALSE),
			sSihash(""),
			sFreqLabel("")

        {}
			tU16 u16Id;
			trMecaEnsemble rEnsemble;
			DAB::trMecaLabel rEnsembleLabel;
			tU32 u32Frequency;
			tU8 u8NumberOfAudioServices;
			tU8 u8NumberOfDataServices;
			tU8 u8ReceptionQuality;
			tBool bReception;
			string sSihash;
			string sFreqLabel;
			
			
        
        bool operator!=(trEnsembleListElem const& b) const
        {
			DAB_RET_1_IF_NEQ(u16Id);
            DAB_RET_1_IF_NEQ(rEnsemble);
            DAB_RET_1_IF_NEQ(rEnsembleLabel);
            DAB_RET_1_IF_NEQ(u32Frequency);
            DAB_RET_1_IF_NEQ(u8NumberOfAudioServices);
            DAB_RET_1_IF_NEQ(u8NumberOfDataServices);
            DAB_RET_1_IF_NEQ(u8ReceptionQuality);
			DAB_RET_1_IF_NEQ(bReception);
            return FALSE;
        }

		  tVoid vTrace() const {
            ETG_TRACE_USR1(("  trEnsembleListElem:rEnsembleId=0x%08x u8ReceptionQuality=0x%02x AudioServices=%d"
				"DataServices=%d rLabel=%s",
							rEnsemble.u16GetEID(),
							u8ReceptionQuality,
							u8NumberOfAudioServices,
							u8NumberOfDataServices,							
                            rEnsembleLabel.pcGetCString()));
            
        }
    };

	 struct trEnsListElemInfo {
		 trEnsembleListElem rEnsembleElem;
		 tU16 u16FrozenListId;
		 trEnsListElemInfo():
		 rEnsembleElem(),
			 u16FrozenListId()

		 {
		 }
		   tVoid vTrace() const {
            ETG_TRACE_USR1(("  trEnsListElemInfo:rEnsembleId=0x%08x , frozenlistId=%d",
				            rEnsembleElem.rEnsemble.u16GetEID(),
							u16FrozenListId));

			rEnsembleElem.vTrace();
        }

	 };


    struct trChnListChnInfo {
        tBool bValid;
        trMecaId rMecaId;
		tU16 u16Scids;
        //        trMecaProgrammeService rProgService;
        tBool bLabelPresent;
        trMecaLabel rLabel;
        tU8 u8ReceptionQuality;
        tU16 u16FrozenListId;
        tU8 u8PresetNumber;
        tU16 u16AvailableAnnoTypesMask;
		tU8 u8PTY;
		tBool bPSFlag;
        trChnListChnInfo():
            bValid(FALSE),
            rMecaId(0),
			u16Scids(0),
            //            rProgService(),
            bLabelPresent(FALSE),
            rLabel(),
            u8ReceptionQuality(0),
            u16FrozenListId(),
            u8PresetNumber(0),
            u16AvailableAnnoTypesMask(0),
			u8PTY(0),
			bPSFlag(TRUE)
        {}

        operator trChnListElem() const {
            trChnListElem rRes;
            rRes.u16Id= u16FrozenListId;
            rRes.rMecaId=rMecaId;
			rRes.u16Scids=u16Scids;
            //                rRes.rProgService=rProgService;
            rRes.bReception=bIsReceptionOk(u8ReceptionQuality);
            rRes.rLabel=rLabel;
			rRes.u8PTY= u8PTY;
			rRes.bPSFlag = bPSFlag;
            return rRes;
        }
        tVoid vTrace() const {
            ETG_TRACE_USR1(("  trChnListChnInfo:bValid=%d rMecaId=0x%08x bLabelPresent=%d u8ReceptionQuality=0x%02x "
                            "u16FrozenListId=%d u8PresetNumber=%d u16AvailableAnnoTypesMask=0x%04x u16Scids = %d bPSFlag=%d rLabel=%s",
                            bValid,
                            rMecaId.u32GetSID(),
                            bLabelPresent,
                            u8ReceptionQuality,
                            u16FrozenListId,
                            u8PresetNumber,
                            u16AvailableAnnoTypesMask,
							u16Scids,
							bPSFlag,
                            rLabel.pcGetCString()));
            
        }

    };

	struct trReceptionState {
		tU8 u8ReceptionQuality;
		tBool bReception;


		trReceptionState(tU8 u8ReceptionQuality_=0, tBool bReception_=FALSE){
			u8ReceptionQuality=u8ReceptionQuality_;
			bReception=bReception_;
		}

		 // persistant data serialization
        tVoid vSerialize(DAB_Serializer &rSerializer) const {
            rSerializer.vSerialize(u8ReceptionQuality);
            rSerializer.vSerialize(bReception);
		};

		 tVoid vParse(DAB_Parser &rParser) {
            rParser.vParse(u8ReceptionQuality);
            rParser.vParse(bReception);
		};

		tVoid vTrace() const {
			ETG_TRACE_USR4(("   trReceptionState:u8ReceptionQuality=%d bReception=%d",
							u8ReceptionQuality,
							bReception));                
        };
	};

	 struct trMsgSrvCmdAutoStore:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvCmdAutoStore() {
            bStart=FALSE;
			u8MaxElements=fc_dabtuner_config::instance()->u8NumberOfPresets() ;//tobe read from KDS
			enAutostoreType=enAutostore_Normal;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdAutoStore: bStart=%d u8MaxElements=%d enAutostoreType=%d",
                            bStart,
                            u8MaxElements,
                            enAutostoreType));
        }
        tBool bStart;
		tU8 u8MaxElements;
		tenAutostoreType enAutostoreType;
    };
    struct trMsgSrvCmdStorePreset:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvCmdStorePreset() {
            u8PresetNumber=0;
            rPSID.vInvalidate();
            u32EnsFrequency = 0;
            bDelete=FALSE;

        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdStorePreset: u8PresetNumber=%d bDelete=%d rPSID=0x%08x u32EnsFrequency=%d",
                            u8PresetNumber,
                            bDelete,
                            rPSID.u32GetSID(),
                            u32EnsFrequency
                            ));
        }

        tU8 u8PresetNumber;
        tBool bDelete;
        trMecaProgrammeService rPSID;
        tU32 u32EnsFrequency;

    };
	struct trMsgSrvCmdClearPresets:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
			trMsgSrvCmdClearPresets() :
		u8PresetNumber(1),
		enDeleteAction(DAB_enPresetListDelete_AllElems)
        {
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdClearPresets"));
        }
		tU8 u8PresetNumber;
		DAB_tenPresetListDeleteAction enDeleteAction;
    };

    
    struct trMsgSrvCmdRecallPreset:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvCmdRecallPreset() {
            u8PresetNumber=0;
            enRes = DAB_enResult_OK;
        }
        trMsgSrvCmdRecallPreset(tU8 u8PresetNumber_) {
            u8PresetNumber=u8PresetNumber_;
            enRes = DAB_enResult_OK;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdRecallPreset: u8PresetNumber=%d enRes=%d",
                            u8PresetNumber,
                            ETG_CENUM(DAB_tenResult, enRes)));
        }

        tU8 u8PresetNumber;
        DAB_tenResult enRes;

    };
#ifdef MUTE_ACTIVE
	struct trMsgSrvCmdLoadPreset:
		   public DAB_Message
	   {
		   DAB_DISPATCH_IMPL
		   trMsgSrvCmdLoadPreset() {
			   u8PresetNumber=0;
			   enRes = DAB_enResult_OK;
		   }
		   trMsgSrvCmdLoadPreset(tU8 u8PresetNumber_) {
			   u8PresetNumber=u8PresetNumber_;
			   enRes = DAB_enResult_OK;
		   }
		   virtual tVoid vTrace() const {
			   ETG_TRACE_USR1(("trMsgSrvCmdLoadPreset: u8PresetNumber=%d enRes=%d",
							   u8PresetNumber,
							   ETG_CENUM(DAB_tenResult, enRes)));
		   }

		   tU8 u8PresetNumber;
		   DAB_tenResult enRes;

	   };
#endif
    struct trMsgSrvCmdSetPresetBank:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvCmdSetPresetBank() {
            u8PresetBank=0;
            enRes = DAB_enResult_OK;
        }
        trMsgSrvCmdSetPresetBank(tU8 u8PresetBank_) {
            u8PresetBank=u8PresetBank_;
            enRes = DAB_enResult_OK;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdSetPresetBank: u8PresetBank=%d enRes=%d",
                            u8PresetBank,
                            ETG_CENUM(DAB_tenResult, enRes)));
        }

        tU8 u8PresetBank;
        DAB_tenResult enRes;
    };


    struct trMsgSrvRspAnnoSelect:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvRspAnnoSelect(DAB_tenResult enRes_) {
            enRes=enRes_;
        }
        trMsgSrvRspAnnoSelect() {
            enRes=DAB_enResult_OK;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvRspAnnoSelect:enRes=%d",
                            ETG_CENUM(DAB_tenResult, enRes)));
        }
        DAB_tenResult enRes;
    };
    
    struct trMsgSrvCmdGetPresetList:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvCmdGetPresetList() {
            u8FirstElementID=0;
            u8NumberOfElements = 0;
        }
        trMsgSrvCmdGetPresetList(tU8 u8FirstElementID_, tU8 u8NumberOfElements_) {
            u8FirstElementID=u8FirstElementID_;
            u8NumberOfElements = u8NumberOfElements_;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdGetPresetList: u8FirstElementID=%d u8NumberOfElements=%d",
                            u8FirstElementID,
                            u8NumberOfElements));
        }

        tU8 u8FirstElementID;
        tU8 u8NumberOfElements;
    };

    struct trMsgDrvCmdPresetChanged:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvCmdPresetChanged(tU8 u8PresetNumber_=0, trMecaProgrammeService const &rPSID_=trMecaProgrammeService()):
            u8PresetNumber(u8PresetNumber_),
            rPSID(rPSID_)
        {}

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdPresetChanged: u8PresetNumber=%d SID=0x%08x",
                            u8PresetNumber, rPSID.u32GetSID()));
        }

        tU8 u8PresetNumber;
        trMecaProgrammeService rPSID;
    };
   
    struct trPresetInfo {
        trPresetInfo():
            u8StationNumber(0),
            rPSID(),
            u32Frequency(0),
			u32EnsembleId(0),
			u8Scid(0),
            rLabel(),
			rCompLabel(),
			rEnsembleLabel(),
			sLogoLink("/0")
        {}
 
        tU8 u8StationNumber;
        trMecaProgrammeService rPSID;
        tU32 u32Frequency;
		tU32 u32EnsembleId;
		tU8 u8Scid;
        trMecaLabel rLabel;
		trMecaLabel rCompLabel;
		trMecaLabel rEnsembleLabel;
		string sLogoLink;

        bool operator!=(trPresetInfo const& b) const
        {
            DAB_RET_1_IF_NEQ(u8StationNumber);
            DAB_RET_1_IF_NEQ(rPSID);
            DAB_RET_1_IF_NEQ(u32Frequency);
            DAB_RET_1_IF_NEQ(rLabel);
            return FALSE;
        }
		 bool operator==(trPresetInfo const& b) const
        {
            DAB_RET_0_IF_NEQ(u8StationNumber);
            DAB_RET_0_IF_NEQ(rPSID);
            DAB_RET_0_IF_NEQ(u32Frequency);
            DAB_RET_0_IF_NEQ(rLabel);
            return FALSE;
        }
        trPresetInfo(trMeca_RRdmStationGetPresetStation const &rRdmGetPresetStation) {
            rPSID= rRdmGetPresetStation.rProgrammeService;
            rLabel=rRdmGetPresetStation.rLabel;
            u8StationNumber=rRdmGetPresetStation.u8StationNumber;
            u32Frequency=rRdmGetPresetStation.u32Frequency;
        }
        virtual tVoid vTrace() const {
			ETG_TRACE_USR1(("    trPresetInfo: u8StationNumber=%d rPSID=0x%08x u32Frequency=%d scids=%d rLabel is: ",
                            u8StationNumber,
                            rPSID.u32GetSID(),
                            u32Frequency,
                            rPSID.u16GetScids()));

			rLabel.vTrace();
        }
    };


    struct trMsgSrvRspGetPresetList:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvRspGetPresetList():
            u8FirstElementID(0),
            u8NumberOfElements(0),
            enRes(DAB_enResult_OK),
            vecPresetInfos() 
        {}
        
        trMsgSrvRspGetPresetList(tU8 u8FirstElementID_, tU8 u8NumberOfElements_):
            u8FirstElementID(u8FirstElementID_),
            u8NumberOfElements(u8NumberOfElements_),
            enRes(DAB_enResult_OK),
            vecPresetInfos()  
        {}

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvRspGetPresetList: enRes=%d, u8FirstElementID=%d u8NumberOfElements=%d",
                            ETG_CENUM(DAB_tenResult, enRes),
                            u8FirstElementID,
                            u8NumberOfElements));
        }

        tU8 u8FirstElementID;
        tU8 u8NumberOfElements;
        DAB_tenResult enRes;
        vector<trPresetInfo> vecPresetInfos;
    };
    
	struct trMsgSrvSavePresetList:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvSavePresetList():
             vecPresetInfos() 
        {}  
        vector<trPresetInfo> vecPresetInfos;
    };


    struct trMsgDrvSetTestMode:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvSetTestMode(){
            bEnable=FALSE;
            enRes = DAB_enResult_OK;
        }
        trMsgDrvSetTestMode(tBool bEnable_){
            bEnable=bEnable_;
            enRes = DAB_enResult_OK;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvSetTestMode: bEnable=%d",
                            bEnable));
        };
        tBool bEnable;
        DAB_tenResult enRes;
    };

    struct trMsgDrvSetBgTestMode:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvSetBgTestMode(){
            bEnable=FALSE;
            enRes = DAB_enResult_OK;
        }
        trMsgDrvSetBgTestMode(tBool bEnable_){
            bEnable=bEnable_;
            enRes = DAB_enResult_OK;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvSetBgTestMode: bEnable=%d",
                            bEnable));
        };
        tBool bEnable;
        DAB_tenResult enRes;
    };

    struct trMsgDrvTmcTpegData:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvTmcTpegData(){
            bEnable=FALSE;
            enRes = DAB_enResult_OK;
        }
        trMsgDrvTmcTpegData(tBool bEnable_){
            bEnable=bEnable_;
            enRes = DAB_enResult_OK;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvTmcTpegData: bEnable=%d",
                            bEnable));
        };
        tBool bEnable;
        DAB_tenResult enRes;
    };

    struct trMsgSrvGetExpApi:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
            trMsgSrvGetExpApi():
                u16ExpApi(0),
                enRes(DAB_enResult_OK)
        {}
        trMsgSrvGetExpApi(tU16 u16ExpApi_):
                u16ExpApi(u16ExpApi_),
                enRes(DAB_enResult_OK)
        {}
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvGetExpApi: u16ExpApi=%d",
                            u16ExpApi));
        };
        tU16 u16ExpApi;
        DAB_tenResult enRes;
    };

    struct trMsgDrvRspGetExpApi:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvRspGetExpApi():
            u16ExpApiId(0),
            bDescriptionValid(FALSE),
            bValueValid(FALSE){
                pcDescription[0] = 0;
                pcValueLabel[0] = 0;
            }
        
        trMsgDrvRspGetExpApi(tU16 u16ExpApiId_, tChar* pcDescription_, tChar* pcValueLabel_):
            u16ExpApiId(u16ExpApiId_),
            bDescriptionValid(FALSE),
            bValueValid(FALSE){

                if (pcDescription_==OSAL_NULL) {
                    pcDescription[0]= 0;
                    return;
                }
                
                tU32 u32Len=(tU32)(OSAL_u32StringLength(pcDescription_));
                if (u32Len>20) {
                    u32Len=20;
                }
                OSAL_pvMemoryCopy(pcDescription, pcDescription_, u32Len);
                pcDescription[u32Len-1]=0;


                if (pcValueLabel_==OSAL_NULL) {
                    pcValueLabel[0]= 0;
                }
                
                if (pcValueLabel_==OSAL_NULL) {
                    return;
                }
                u32Len=(tU32)(OSAL_u32StringLength(pcValueLabel_));
                if (u32Len>20) {
                    u32Len=20;
                }
                OSAL_pvMemoryCopy(pcValueLabel, pcValueLabel_, u32Len);
                pcDescription[u32Len-1]=0;
            }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvRspGetExpApi: u16ExpApiId=%x pcDescription=%s",
                            u16ExpApiId,
                            pcDescription));
        }

        tU16 u16ExpApiId;
        tChar pcDescription[20];
        tChar pcValueLabel[20];
        tBool bDescriptionValid;
        tBool bValueValid;
    };

    
    struct trMsgDrvRspDbQueryTestmode:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvRspDbQueryTestmode(){
                u8Query = 1;
                sQuery = "/0";
            }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvRspDbQueryTestmode: sQuery=%s",
                            sQuery.c_str()));
        }
        tU8 u8Query;
        std::string sQuery;
    };    

    struct trMsgDrvSetMuteStatus:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvSetMuteStatus():
            enMuteState(enARLMuteState_Muted),
            enDabSource(enDabSource_Dab),
            enRes(DAB_enResult_OK)
            {}
        
        trMsgDrvSetMuteStatus( tenARLMuteState enMuteState_, tenDabSource enDabSource_ ):
            enMuteState(enMuteState_),
            enDabSource(enDabSource_),
            enRes(DAB_enResult_OK){
            }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvSetMuteStatus: enDabSource=%d enMuteState=%d",
                            enDabSource,
                            enMuteState));
        }

        tenARLMuteState enMuteState;
        tenDabSource enDabSource;
        DAB_tenResult enRes;
    };
#ifdef MUTE_ACTIVE
        struct trMsgDrvMuteState:
            public DAB_Message
        {
            DAB_DISPATCH_IMPL
            trMsgDrvMuteState():
                enMuteState(enRequiredMuteState_None),
                enDabSource(enDabSource_Dab),
                enRes(DAB_enResult_OK)
                {}

            trMsgDrvMuteState( tenRequiredMuteState enMuteState_, tenDabSource enDabSource_ ):
                enMuteState(enMuteState_),
                enDabSource(enDabSource_),
                enRes(DAB_enResult_OK){
                }

            virtual tVoid vTrace() const {
                ETG_TRACE_USR1(("trMsgDrvMuteState: enDabSource=%d enMuteState=%d",
                                enDabSource,
                                enMuteState));
            }

            tenRequiredMuteState enMuteState;
            tenDabSource enDabSource;
            DAB_tenResult enRes;
        };
#endif

    struct trMsgEnableMuteReqProperty: 
        public DAB_Message 
    {
        DAB_DISPATCH_IMPL
        trMsgEnableMuteReqProperty(tBool bMuteReqEnabled_):
        bMuteReqEnabled(bMuteReqEnabled_) {
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  trMsgEnableMuteReqProperty:bMuteReqEnabled=%d",
                            bMuteReqEnabled));
        };
        tBool bMuteReqEnabled;
    };

    struct trMsgSrvCmdGetTestModeData:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdGetTestModeData"));
        };
    };

    struct trMsgSrvRspGetTestModeData:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvRspGetTestModeData"));
        };
    };
    

    struct trMsgDrvActivateComponentInfo:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvActivateComponentInfo(){
            bEnable=FALSE;
        }
        trMsgDrvActivateComponentInfo(tBool bEnable_){
            bEnable=bEnable_;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvActivateComponentInfo: bEnable=%d",
                            bEnable));
        };
        tBool bEnable;
    };
    
    struct trMsgDrvActivateServiceInfo:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvActivateServiceInfo(){
            bEnable=FALSE;
        }
        trMsgDrvActivateServiceInfo(tBool bEnable_){
            bEnable=bEnable_;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvActivateServiceInfo: bEnable=%d",
                            bEnable));
        };
        tBool bEnable;
    };

    struct trMsgDrvActivateEnsembleInfo:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvActivateEnsembleInfo(){
            bEnable=FALSE;
        }
        trMsgDrvActivateEnsembleInfo(tBool bEnable_){
            bEnable=bEnable_;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvActivateEnsembleInfo: bEnable=%d",
                            bEnable));
        };
        tBool bEnable;
    };

    struct trMsgDrvRspCurrentEnsemble:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvRspCurrentEnsemble(trMecaEnsemble const &rEnsemble_=trMecaEnsemble()):
            rEnsemble(rEnsemble_)
        {}
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvRspCurrentEnsemble: EID=0x%08x",
                            rEnsemble.u32GetID()));
        };
        trMecaEnsemble rEnsemble;
    };


    struct trMsgSrvCmdSetChnList:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        trMsgSrvCmdSetChnList(tenListOperation enOperation_=tenListOperation_INVALID,tenChnListType enListType_ =enChnList_INVALID, tU16 u16FirstElem_=0, 
		tU16 u16NumElem_=0,tU32 u32PTYFilter_=0x3fffffff,tenServiceListSortingType enSrvListSortingType_=enServiceListSortingType_LABEL, tU16 u16EnsIndex_=0){
            enOperation=enOperation_;
			enListType=enListType_;
            u16FirstElem = u16FirstElem_;
            u16NumElem = u16NumElem_;
			u32PTYFilter =u32PTYFilter_;
			enSrvListSortingType =enSrvListSortingType_;
			u16EnsIndex = u16EnsIndex_;

        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdSetChnList: op=%d listtype=%d u16FirstElem=%d u16NumElem=%d u32PTYFilter=%u, sorting-=%d u16EnsIndex=%d",
                            ETG_CENUM(tenListOperation, enOperation),
							ETG_CENUM(tenChnListType, enListType),
                            u16FirstElem,
                            u16NumElem,
							u32PTYFilter,
							ETG_CENUM(tenServiceListSortingType, enSrvListSortingType),
							u16EnsIndex));
        };
        tenListOperation enOperation;
		tenChnListType enListType;
        tU16 u16FirstElem;
        tU16 u16NumElem;
		tU32 u32PTYFilter;
		tenServiceListSortingType enSrvListSortingType;
		tU16 u16EnsIndex;
    };

	 struct trMsgSrvCmdSetEnsmbleList:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        trMsgSrvCmdSetEnsmbleList(tenListOperation enOperation_=tenListOperation_INVALID, tU16 u16FirsElem_=0, tU16 u16NumElem_=0,
		tenEnsembleListSortingType enEnsListSortingType_=enEnsembleListSortingType_LABEL){
            enOperation=enOperation_;
            u16FirstElem = u16FirsElem_;
            u16NumElem = u16NumElem_;
			enEnsListSortingType=enEnsListSortingType_;

        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdSetEnsmbleList: op=%d u16FirstElem=%d u16NumElem=%d sorting=%d",
                            ETG_CENUM(tenListOperation, enOperation),
                            u16FirstElem,
                            u16NumElem,
							ETG_CENUM(tenEnsembleListSortingType, enEnsListSortingType)));
        };
        tenListOperation enOperation;
        tU16 u16FirstElem;
        tU16 u16NumElem;
		tenEnsembleListSortingType enEnsListSortingType;
    };

	struct trMsgSrvRspSetChnList:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvRspSetChnList(vector<trChnListElem> const &vcChnList_):vcChnList(vcChnList_){
            enRes=DAB_enResult_OK;
            bOpen=FALSE;
            u16ActivatedElemId=0;
            u16TotalNumberOfElements=0;
            u16NumElemToSend=0;
            u16FirstIndexToSend=0;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvRspSetChnList: enRes=%d bOpen=%d u16ActivatedElemId=%d u16TotalNumberOfElements=%d u16NumElemToSend=%d u16FirstIndexToSend=%dServiceList:",
                            ETG_CENUM(DAB_tenResult, enRes),
                            bOpen, 
                            u16ActivatedElemId,
                            u16TotalNumberOfElements,
                            u16NumElemToSend,
                            u16FirstIndexToSend
                            ));
            DAB_FOREACH_CONST(vector<trChnListElem>, iter, vcChnList) {
                ETG_TRACE_USR1((    "  %08x %d %17s ", 
                                    iter->rMecaId.u32GetSID(), 
                                    iter->bReception,
                                    iter->rLabel.pcGetCString()));
            }
        };

        DAB_tenResult enRes;
        tBool bOpen;
        tU16 u16ActivatedElemId;
        tU16 u16TotalNumberOfElements;
        tU16 u16NumElemToSend;
        tU16 u16FirstIndexToSend;
        //lint -esym(1725, DAB::trMsgSrvRspSetChnList::vcChnList) "class member is a reference"
        vector<trChnListElem> const &vcChnList;
        
    };


	struct trMsgSrvValidateEPGData:
        public DAB_Message
    {
		 DAB_DISPATCH_IMPL
        trMsgSrvValidateEPGData(){}
	};


	struct trMsgSrvRspValidateEPGData:
        public DAB_Message
    {
		 DAB_DISPATCH_IMPL
        trMsgSrvRspValidateEPGData(){
            enRes=DAB_enResult_OK;
            bEPGPresent=TRUE;
        }

		DAB_tenResult enRes;
		tBool bEPGPresent;
	};


	struct trMsgSrvRspSetEnsmbleList:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvRspSetEnsmbleList(vector<trEnsembleListElem> const &vcEnsmbleList_):vcEnsmbleList(vcEnsmbleList_){
            enRes=DAB_enResult_OK;
            bOpen=FALSE;
            u16ActivatedElemId=0;
            u16TotalNumberOfElements=0;
            u16NumElemToSend=0;
            u16FirstIndexToSend=0;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvRspSetEnsmbleList: enRes=%d bOpen=%d u16ActivatedElemId=%d u16TotalNumberOfElements=%d u16NumElemToSend=%d u16FirstIndexToSend=%dEnsembleList:",
                            ETG_CENUM(DAB_tenResult, enRes),
                            bOpen, 
                            u16ActivatedElemId,
                            u16TotalNumberOfElements,
                            u16NumElemToSend,
                            u16FirstIndexToSend
                            ));
            DAB_FOREACH_CONST(vector<trEnsembleListElem>, iter, vcEnsmbleList) {
                ETG_TRACE_USR1((    "  %08x %d %17s ", 
                                    iter->rEnsemble.u16GetEID(),
                                    iter->u8ReceptionQuality,
                                    iter->rEnsembleLabel.pcGetCString()));
            }
        };

        DAB_tenResult enRes;
        tBool bOpen;
        tU16 u16ActivatedElemId;
        tU16 u16TotalNumberOfElements;
        tU16 u16NumElemToSend;
        tU16 u16FirstIndexToSend;
        //lint -esym(1725, DAB::trMsgSrvRspSetChnList::vcChnList) "class member is a reference"
        vector<trEnsembleListElem> const &vcEnsmbleList;
        
    };

    struct trMsgSrvCmdSetCompList:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvCmdSetCompList(){
            enOperation=tenListOperation_INVALID;
            enRes=DAB_enResult_OK;
        }
        trMsgSrvCmdSetCompList(tenListOperation enOperation_){
            enOperation=enOperation_;
            enRes=DAB_enResult_OK;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdSetCompList: op=%d enRes=%d",
                            ETG_CENUM(tenListOperation, enOperation),
                            ETG_CENUM(DAB_tenResult, enRes)));
        };
        tenListOperation enOperation;
        DAB_tenResult enRes;
    };

    struct trSrvIdAndQuality {
        tU16 u16ListElemId;
        tBool bReception;
    };


    struct trMsgSrvRspUpdateChnList:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvRspUpdateChnList():vcChangedAvailablities(){
            bChnListActive=FALSE;
            bUpdateRunning=FALSE;
            u16ActivatedElemId=0;
            u16NumChanges=0;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvRspUpdateChnList: bChnListActive=%d bUpdateRunning=%d u16ActivatedElemId=%d u16NumChanges=%d",
                            bChnListActive,
                            bUpdateRunning,
                            u16ActivatedElemId,
                            u16NumChanges));
            DAB_FOREACH_CONST(vector<trSrvIdAndQuality>, iter, vcChangedAvailablities) {
                ETG_TRACE_USR1((    "  %04x %d", 
                                    iter->u16ListElemId, 
                                    iter->bReception));
            }
        };

        tBool bChnListActive;
        tBool bUpdateRunning;
        tU16 u16ActivatedElemId;
        tU16 u16NumChanges;
        vector<trSrvIdAndQuality> vcChangedAvailablities;
        
    };

    
    inline tBool DAB_bDoSrvLinkingDab(DAB_tenServiceLinkingMode enServiceLinkingMode) {
        return enServiceLinkingMode==DAB_enServiceLinkingMode_DAB ||
            enServiceLinkingMode==DAB_enServiceLinkingMode_DAB_FM;
    }

    inline tBool DAB_bDoSrvLinkingFm(DAB_tenServiceLinkingMode enServiceLinkingMode) {
        return ( (enServiceLinkingMode==DAB_enServiceLinkingMode_DAB_FM) || 
                 (enServiceLinkingMode==DAB_enServiceLinkingMode_FM ) );
    }

    struct trMsgSrvCmdSetup:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvCmdSetup():
            u32SetupMask(0),
            u32SetupFlags(0),
            bUseConcealmentLevel(FALSE),
            u8ConcealmentLevel(0),
            bUseServiceLinkingMode(FALSE),
            e8ServiceLinkingMode(DAB_enServiceLinkingMode_OFF),
            bSwitchMeasurementMode(FALSE),
            u8LinkingSetupMask(enLinkingSetupMask_None),
            u8DABFMLinkingThreshold(0),
            u8DABFMLinkingBackThreshold(0),
            u8DABFMLinkingBackValidationTime(0),
            u8DABFMLinkBackDelay(0),
            bUseEnsInsertThreshold(FALSE),
            u8EnsInsertThreshold(0),
            bUseEnsDeleteThreshold(FALSE),
            u8EnsDeleteThreshold(0),
            bBgLearnUpdateTime(FALSE),
            u8BgLearnUpdateTime(0),
            bDABScanThreshold(FALSE),
            u8DABScanThreshold(0),
            enRes(DAB_enResult_OK),
            bIsTestMode(FALSE)
            
        {}



        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdSetup:u32SetupMask=0x%04x u32SetupFlags=0x%04x "
                            "bUseConcealmentLevel=%d u8ConcealmentLevel=%x"
                            "bUseServiceLinkingMode=%d e8ServiceLinkingMode=%x bSwitchMeasurementMode=%d "
                            "u8LinkingSetupMask=%d u8DABFMLinkingThreshold=%d u8DABFMLinkingBackThreshold=%d ",
                            u32SetupMask,
                            u32SetupFlags,
                            bUseConcealmentLevel,
                            u8ConcealmentLevel,
                            bUseServiceLinkingMode,
                            ETG_CENUM(DAB_tenServiceLinkingMode, e8ServiceLinkingMode),
                            bSwitchMeasurementMode,
                            u8LinkingSetupMask,
                            u8DABFMLinkingThreshold,
                            u8DABFMLinkingBackThreshold
                            ));
            ETG_TRACE_USR1(("trMsgSrvCmdSetup:u8DABFMLinkingBackValidationTime=%d u8DABFMLinkBackDelay=%d bUseEnsInsertThreshold=%d"
            				"u8EnsInsertThreshold=%d bUseEnsDeleteThreshold=%d u8EnsDeleteThreshold=%d bBgLearnUpdateTime=%d u8BgLearnUpdateTime=%d",
                            u8DABFMLinkingBackValidationTime,
                            u8DABFMLinkBackDelay,
                            bUseEnsInsertThreshold,
                            u8EnsInsertThreshold,
                            bUseEnsDeleteThreshold,
                            u8EnsDeleteThreshold,
                            bBgLearnUpdateTime,
                            u8BgLearnUpdateTime));

			ETG_TRACE_USR1(("trMsgSrvCmdSetup:bDABScanThreshold=%d u8DABScanThreshold=%d enRes=%d",
								bDABScanThreshold,
								u8DABScanThreshold,
								ETG_CENUM(DAB_tenResult, enRes)));
        }
        tU32 u32SetupMask;
        tU32 u32SetupFlags;
        tBool bUseConcealmentLevel;
        tU8 u8ConcealmentLevel;
        tBool bUseServiceLinkingMode;
        DAB_tenServiceLinkingMode e8ServiceLinkingMode;
        tBool bSwitchMeasurementMode;
        tenLinkingSetupMask u8LinkingSetupMask;
        tU8 u8DABFMLinkingThreshold;
        tU8 u8DABFMLinkingBackThreshold;
        tU8 u8DABFMLinkingBackValidationTime;
        tU8 u8DABFMLinkBackDelay;
        tBool bUseEnsInsertThreshold;
        tU8 u8EnsInsertThreshold;
        tBool bUseEnsDeleteThreshold;
        tU8 u8EnsDeleteThreshold;
        tBool bBgLearnUpdateTime;
        tU8 u8BgLearnUpdateTime;
        tBool bDABScanThreshold;
        tU8 u8DABScanThreshold;
        DAB_tenResult enRes;
        tBool bIsTestMode;
    };

    
    struct trMsgDrvCmdEnableDabSrvFollow:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvCmdEnableDabSrvFollow(tBool bEnable_=TRUE):
            bEnable(bEnable_)
        {}


        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdEnableDabSrvFollow:bEnable=%d",
                            bEnable));
        }
        tBool bEnable;
    };

    struct trMsgSrvCmdAnnoConfig:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvCmdAnnoConfig():
            u16AnnoTypesMask(0),
            enAnnoSource(enAnnoSource_NONE),
            bEnable(FALSE),
            enRes(DAB_enResult_OK)
        {}


        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdAnnoConfig:u16AnnoTypesMask=0x%04x enAnnoSource=%d bEnable=%d enRes=%d",
                            u16AnnoTypesMask,
                            ETG_CENUM(tenAnnoSource, enAnnoSource),
                            bEnable,
                            ETG_CENUM(DAB_tenResult, enRes)));
        }
        tU16 u16AnnoTypesMask;
        tenAnnoSource enAnnoSource;
        tBool bEnable;
        DAB_tenResult enRes;
    };

    struct trMsgDrvCmdSelectAnno:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvCmdSelectAnno(tBool bSelect_=TRUE ,tU16 u16AnnoMask_=0):
            bSelect(bSelect_),
			u16AnnoMask(u16AnnoMask_)
        {}


        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdSelectAnno:bSelect=%d u16AnnoMask=%d",
                            bSelect,
							u16AnnoMask));
        }
        tBool bSelect;
		tU16 u16AnnoMask;
    };


    // todo: check valid len of radio-text (128 byte + 0-termination?)
    struct trRadioText {
        tBool bValid;
        tenMeca_CharSet enMeca_CharSet;
        tU8 u8RadioTextLen;
        tChar sRadioText[257]; 
        trRadioText():
            bValid(FALSE),
            enMeca_CharSet(enMeca_CharSet_EBU_LATIN),
            u8RadioTextLen(0)
        {
            sRadioText[0]='\0';
        };

        trRadioText(trRadioText const &b):
            bValid(b.bValid),
            enMeca_CharSet(b.enMeca_CharSet),
            u8RadioTextLen(b.u8RadioTextLen)
        {
            OSAL_pvMemoryCopy(sRadioText, b.sRadioText, b.u8RadioTextLen);
            sRadioText[b.u8RadioTextLen]='\0';
        };

        trRadioText &operator=(trRadioText const &b) {
            if (this == &b)  {
                return *this; 
            }
            bValid=b.bValid;
            enMeca_CharSet=b.enMeca_CharSet;
            u8RadioTextLen=b.u8RadioTextLen;
            OSAL_pvMemoryCopy(sRadioText, b.sRadioText, b.u8RadioTextLen);
            sRadioText[b.u8RadioTextLen]='\0';
            return *this; 
        };

        trRadioText(tU8 const *pu8RadioText, tU8 u8RadioTextLen_) {
            enMeca_CharSet = enMeca_CharSet_EBU_LATIN;
            sRadioText[0]='\0';
            u8RadioTextLen=u8RadioTextLen_;
            bValid=TRUE;
            if (u8RadioTextLen>128) {
                u8RadioTextLen=128;
            }
            if (OSAL_NULL == pu8RadioText) {
                bValid=FALSE;
                u8RadioTextLen=0;
            }
			/**vnd4kor: solving prio 2 lint warning*/
			if(NULL != pu8RadioText )
			{
	            OSAL_pvMemoryCopy(sRadioText, pu8RadioText, u8RadioTextLen);
			}
            //todo:check if meca-radio-text is null-terminated (then use [u8RadioTextLen-1])
            sRadioText[u8RadioTextLen]='\0';
        }

        trRadioText(tChar const *pcRadioText) {
            bValid=TRUE;
            if (pcRadioText==OSAL_NULL) {
                pcRadioText= "";
                bValid=FALSE;
            }
            u8RadioTextLen=(tU8)OSAL_u32StringLength(pcRadioText);
            if (u8RadioTextLen>128) {
                u8RadioTextLen=128;
            }
            enMeca_CharSet = enMeca_CharSet_EBU_LATIN;
            OSAL_pvMemoryCopy(sRadioText, pcRadioText, u8RadioTextLen);
            sRadioText[u8RadioTextLen]='\0';
        }


        bool operator!=(trRadioText const& b) const {
            DAB_RET_1_IF_NEQ(bValid);
            DAB_RET_1_IF_NEQ(u8RadioTextLen);
            return OSAL_s32MemoryCompare(sRadioText, b.sRadioText, u8RadioTextLen) ? TRUE : FALSE;
        }

        tVoid vTrace() const {
            ETG_TRACE_USR1(("  trRadioText:bValid=%d u8RadioTextLen=%d radioText=%s",
                            bValid,
                            u8RadioTextLen,
                            sRadioText));
            
        }
    };

    struct trMsgDrvCmdRegisterFmPi:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvCmdRegisterFmPi(tBool bRegister_=FALSE):
            bRegister(bRegister_)
        {}

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdRegisterFmPi:bRegister=%d",
                            bRegister));
        }

        tBool bRegister;
    };

    struct trMsgDrvCmdRegisterFmRT:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvCmdRegisterFmRT(tBool bRegister_=FALSE):
            bRegister(bRegister_)
        {}

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdRegisterFmRT:bRegister=%d",
                            bRegister));
        }

        tBool bRegister;
    };

    struct trMsgSrvRspFmRT:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvRspFmRT(char const *sRT=OSAL_NULL):rRadioText(sRT)
        {}

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvRspFmRT:rRadioText follows:"));
            rRadioText.vTrace();
        }
        trRadioText rRadioText;
    };

    struct trMsgDrvFmServiceFollow:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvFmServiceFollow(tBool bFmSrvFollowActive_):bFmSrvFollowActive(bFmSrvFollowActive_)
        {}

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvFmServiceFollow:bFmSrvFollowActive: %d",bFmSrvFollowActive));
        }
        tBool bFmSrvFollowActive;
    };

	struct trMsgSrvCmdMtcTunerstatus:
		public DAB_Message
	{
		DAB_DISPATCH_IMPL
		trMsgSrvCmdMtcTunerstatus(tenMeca_MtcTuner _enTunerStatus=enMeca_MtcTuner_MAIN_DAB_TUNER):enTunerStatus(_enTunerStatus)
		{}

		virtual tVoid vTrace() const {
			ETG_TRACE_USR1(("trMsgSrvCmdMtcTunerstatus:enTunerStatus: %d",enTunerStatus));
		}
		tenMeca_MtcTuner  enTunerStatus;
	};


    struct trMsgSrvCmdFmQuality:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvCmdFmQuality() {
            u8Quality=0;
        }

        trMsgSrvCmdFmQuality(tU8 u8Quality_) {
            u8Quality=u8Quality_;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdFmQuality:u8Quality=%d",
                            u8Quality));
        }

        tU8 u8Quality;
    };

   struct trMsgSrvCmdChnSelect: 



        public DAB_Message 
    {
        DAB_DISPATCH_IMPL
        tenChnSelectMode enChnSelectMode;
        tU32 u32Value;
        tU16 u16Scids;
        tBool bNegative;


        trMsgSrvCmdChnSelect():
            enChnSelectMode(enChnSelectMode_Invalid),
            u32Value(0),
            u16Scids(0),
            bNegative(FALSE)
        {}

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMsgSrvCmdChnSelect: enChnSelectMode=0x%x u32Value=0x%x (%u) u16Scids=%d bNegative=%d" ,
                            ETG_CENUM(tenChnSelectMode, enChnSelectMode),
                            u32Value,
                            u32Value,
                            u16Scids,
                            bNegative
                            ));
        };
        




    };
#ifdef MUTE_ACTIVE
	struct trMsgSrvCmdSrvSelect:
		   public DAB_Message
	   {
		   DAB_DISPATCH_IMPL
		   tenChnSelectMode enChnSelectMode;
		   tU32 u32Value;
		   tBool bNegative;
		   tU16 u16Scids;
		   tenMuteOperation enMuteOp;

		   trMsgSrvCmdSrvSelect():
			   enChnSelectMode(enChnSelectMode_Invalid),
			   u32Value(0),
			   bNegative(FALSE),
			   u16Scids(0),
			   enMuteOp(enMuteOperation_None)
		   {}

		   virtual tVoid vTrace() const {
			   ETG_TRACE_USR1(("  trMsgSrvCmdSrvSelect: enChnSelectMode=0x%x u32Value=0x%x (%u) bNegative=%d u16Scids=%d enMuteOp=%d" ,
							   ETG_CENUM(tenChnSelectMode, enChnSelectMode),
							   u32Value,
							   u32Value,
							   bNegative,
							   u16Scids,
							   ETG_CENUM(tenMuteOperation, enMuteOp)
							   ));
		   };
	   };

#endif

   struct trMsgSrvCmdEnsSelect: 



        public DAB_Message 
    {
        DAB_DISPATCH_IMPL
        tenEnsSelectMode enEnsSelectMode;
        tU32 u32Value;
        tBool bNegative;


        trMsgSrvCmdEnsSelect():
            enEnsSelectMode(enEnsSelectMode_Invalid),
            u32Value(0),
            bNegative(FALSE)
        {}

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMsgSrvCmdEnsSelect: enEnsSelectMode=0x%x u32Value=0x%x (%u) bNegative=%d" , 
                            ETG_CENUM(tenEnsSelectMode, enEnsSelectMode),
                            u32Value,
                            u32Value,
                            bNegative
                            ));
        };
        




    };


    struct trMsgDrvIndPreChnSelect: 
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        tBool bCompMode;
        tBool bIdChanged;
        tenChnSelectSource enChnSelectSource;

        trMsgDrvIndPreChnSelect(tBool bCompMode_=FALSE, tBool bIdChanged_=TRUE, tenChnSelectSource enChnSelectSource_= enChnSelectSource_ChnSelect):
            bCompMode(bCompMode_),
            bIdChanged(bIdChanged_),
            enChnSelectSource(enChnSelectSource_)

        {}
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMsgDrvIndPreChnSelect: bCompMode=%d tBool bIdChanged=%d enChnSelectSource=%d",
                            bCompMode,
                            bIdChanged,
                            ETG_CENUM(tenChnSelectSource, enChnSelectSource)));
        }
    };

    struct trMsgDrvIndPostChnSelect: 
        public DAB_Message 
    {
        DAB_DISPATCH_IMPL
        tBool bCompMode;
        trMsgDrvIndPostChnSelect(tBool bCompMode_=FALSE):
            bCompMode(bCompMode_)
        {}
        
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMsgDrvIndPostChnSelect: bCompMode=%d",
                            bCompMode));
        }
    };

    struct trMsgDrvCmdChnSelect:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        trMsgDrvCmdChnSelect(trMsgSrvCmdChnSelect const &rSrvCmdChnSelect):
            enChnSelectSource(enChnSelectSource_ChnSelect),
            enChnSelectMode(rSrvCmdChnSelect.enChnSelectMode),
            u32Value(rSrvCmdChnSelect.u32Value),
            bNegative(rSrvCmdChnSelect.bNegative),
            u16scids(rSrvCmdChnSelect.u16Scids),
            u32Frequency(0),
            enRes(DAB_enResult_INPROCESS)
        {}

        trMsgDrvCmdChnSelect():
            enChnSelectSource(enChnSelectSource_ChnSelect),
            enChnSelectMode(enChnSelectMode_Invalid),
            u32Value(0),
            bNegative(FALSE),
            u16scids(0),
            u32Frequency(0),
            enRes(DAB_enResult_INPROCESS)
        {}


        /* modes:
           up/down/current
           refType:
           listId/PSID
           trMecaProgrammeService rPSID
           
        */
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMsgDrvCmdChnSelect: enChnSelectSource=%d enChnSelectMode=0x%x u32Value=0x%x (%u) bNegative=%d u32Frequency=%u enRes=%d" ,
                            ETG_CENUM(tenChnSelectSource, enChnSelectSource),
                            ETG_CENUM(tenChnSelectMode, enChnSelectMode),
                            u32Value,
                            u32Value,
                            bNegative,
                            u32Frequency,
                            ETG_CENUM(DAB_tenResult, enRes)
                            ));

            ETG_TRACE_USR1(("u16scids=%d",u16scids));
        }

        tenChnSelectSource enChnSelectSource;
        tenChnSelectMode enChnSelectMode;
        tU32 u32Value;
        tBool bNegative;
        tU16 u16scids;
        tU32 u32Frequency;
        DAB_tenResult enRes;
    };
 
	 struct trMsgDrvCmdEnsSelect:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        trMsgDrvCmdEnsSelect(trMsgSrvCmdEnsSelect const &rSrvCmdEnsSelect):
            enEnsSelectMode(rSrvCmdEnsSelect.enEnsSelectMode),
            u32Value(rSrvCmdEnsSelect.u32Value),
            bNegative(rSrvCmdEnsSelect.bNegative),
            u32Frequency(0),
            enRes(DAB_enResult_INPROCESS)
        {}

        trMsgDrvCmdEnsSelect():
            enEnsSelectMode(enEnsSelectMode_Invalid),
            u32Value(0),
            bNegative(FALSE),
            u32Frequency(0),
            enRes(DAB_enResult_INPROCESS)
        {}


        /* modes:
           up/down/current
           refType:
           listId/PSID
           trMecaProgrammeService rPSID
           
        */
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMsgDrvCmdEnsSelect: enEnsSelectMode=0x%x u32Value=0x%x (%u) bNegative=%d u32Frequency=%u enRes=%d" , 
                            ETG_CENUM(tenEnsSelectMode, enEnsSelectMode),
                            u32Value,
                            u32Value,
                            bNegative,
                            u32Frequency,
                            ETG_CENUM(DAB_tenResult, enRes)
                            ));
        }

        tenEnsSelectMode enEnsSelectMode;
        tU32 u32Value;
        tBool bNegative;
        tU32 u32Frequency;
        DAB_tenResult enRes;
    };
    struct trMsgSrvCmdCriticalVoltage:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvCmdCriticalVoltage(tBool bIsCritical_=FALSE):
            bIsCritical(bIsCritical_)
        {}
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdCriticalVoltage: bIsCritical=%d",
                            bIsCritical));
        }
        tBool bIsCritical;
    };

#ifdef MUTE_ACTIVE
	struct trMsgSrvCmdEnsSeek:
		   public DAB_Message
	{
	   DAB_DISPATCH_IMPL
	   DAB_tenTuneType enTuneType;
	   DAB_tenDir	enDirection;
       DAB_tenResult	enRes;


       trMsgSrvCmdEnsSeek():
		   enTuneType(DAB_enTuneType_INVALID),
		   enDirection(DAB_enDir_None),
		   enRes(DAB_enResult_OK)
	   {}

	   virtual tVoid vTrace() const {
	               ETG_TRACE_USR1(("  trMsgSrvCmdEnsSeek: enDir=0x%x" ,
	            		   enDirection));
	           };
	};
#endif

	struct trMsgDrvCmdRdmTune:
		   public DAB_Message
	{
	   DAB_DISPATCH_IMPL
	   DAB_tenTuneType enTuneType;
	   DAB_tenDir	enDirection;
	   DAB_tenResult	enRes;


	   trMsgDrvCmdRdmTune():
		   enTuneType(DAB_enTuneType_INVALID),
		   enDirection(DAB_enDir_None),
		   enRes(DAB_enResult_OK)
	   {}

	   virtual tVoid vTrace() const {
				   ETG_TRACE_USR1(("  trMsgDrvCmdRdmTune: enDir=0x%x" ,
						   enDirection));
			   };
	};

    struct trMsgSrvCmdTestOperation: 
        public DAB_Message 
    {
        DAB_DISPATCH_IMPL
        DAB_tenTestOperation enTestOperation;
        tU32 u32Value1;
        tU32 u32Value2;
        tU32 u32Value3;
        tBool bRejected;

        trMsgSrvCmdTestOperation():
            enTestOperation(DAB_enTestOperation_INVALID),
            u32Value1(0),
            u32Value2(0),
            u32Value3(0),
            bRejected(FALSE),
            enRes(DAB_enResult_OK)
        {}

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMsgSrvCmdTestOperation: enTestOperation=0x%x u32Value=0x%08x (%u) u32Value2=0x%08x u32Value3=0x%08x bRejected=%d" , 
                            ETG_CENUM(DAB_tenTestOperation, enTestOperation),
                            u32Value1,
                            u32Value1,
                            u32Value2,
                            u32Value3,
                            bRejected));
        };
        DAB_tenResult enRes;        
    };
    struct trMsgSrvCmdTuneFreqlabel:
        public DAB_Message
        {
                DAB_DISPATCH_IMPL
                string sfreqlabel;
                trMsgSrvCmdTuneFreqlabel():
                	sfreqlabel(""),
                    enRes(DAB_enResult_OK)
                {}
                virtual tVoid vTrace() const {
                    ETG_TRACE_USR1(("  trMsgSrvCmdTuneFreqlabel: FreqLabel=%s", sfreqlabel.c_str()));
                };
                DAB_tenResult enRes;
           };
#ifdef MUTE_ACTIVE
	struct trMsgDrvCmdTuneFreqlabel:
		public DAB_Message
		{
				DAB_DISPATCH_IMPL
				string sfreqlabel;
				trMsgDrvCmdTuneFreqlabel():
					sfreqlabel(""),
					enRes(DAB_enResult_OK)
				{}
				virtual tVoid vTrace() const {
					ETG_TRACE_USR1(("  trMsgDrvCmdTuneFreqlabel: FreqLabel=%s", sfreqlabel.c_str()));
				};
				DAB_tenResult enRes;
		   };
#endif

#ifdef MUTE_ACTIVE
	struct trMsgSrvCmdManualModeOp:
		public DAB_Message
	{
		DAB_DISPATCH_IMPL
		DAB_tenTestOperation enTestOperation;
		tU32 u32Value1;
		tU32 u32Value2;
		tU32 u32Value3;
		tBool bRejected;

		trMsgSrvCmdManualModeOp():
			enTestOperation(DAB_enTestOperation_INVALID),
			u32Value1(0),
			u32Value2(0),
			u32Value3(0),
			bRejected(FALSE),
			enRes(DAB_enResult_OK)
		{}

		virtual tVoid vTrace() const {
			ETG_TRACE_USR1(("  trMsgSrvCmdManualModeOp: enTestOperation=0x%x u32Value=0x%08x (%u) u32Value2=0x%08x u32Value3=0x%08x bRejected=%d" ,
							ETG_CENUM(DAB_tenTestOperation, enTestOperation),
							u32Value1,
							u32Value1,
							u32Value2,
							u32Value3,
							bRejected));
		};
		DAB_tenResult enRes;
	};
#endif
    struct trMsgSrvRspTestOperation: 
        public DAB_Message 
    {
        DAB_DISPATCH_IMPL
        DAB_tenTestOperation enTestOperation;
        DAB_tenResult enResult;

        trMsgSrvRspTestOperation(DAB_tenTestOperation enTestOperation_=DAB_enTestOperation_INVALID,
                                 DAB_tenResult enResult_=DAB_enResult_OK):
            enTestOperation(enTestOperation_),
            enResult(enResult_)
        {}

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMsgSrvRspTestOperation: enTestOperation=0x%x enResult=%d" , 
                            ETG_CENUM(DAB_tenTestOperation, enTestOperation),
                            ETG_CENUM(DAB_tenResult, enResult)));
        };
        
    };

    struct trMsgSrvDbQueryTestmode: 
        public DAB_Message 
    {
        DAB_DISPATCH_IMPL
        tU8 u8Adr3DbQuery;

        trMsgSrvDbQueryTestmode():
            u8Adr3DbQuery(1),
            enRes(DAB_enResult_OK)
        {}

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMsgSrvDbQueryTestmode: u8Adr3DbQuery=%d" , 
                            u8Adr3DbQuery));
        };
        DAB_tenResult enRes;        
    };

    struct trMsgDrvIndRemoteCtrl: 
        public DAB_Message 
    {
        DAB_DISPATCH_IMPL
        tBool bOn;
        trMsgDrvIndRemoteCtrl(tBool bOn_=FALSE):
            bOn(bOn_)
        {}

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMsgDrvIndRemoteCtrl: bOn%d" , 
                            bOn));
        };
        
    };
        
    struct trMsgSrvCmdStopAnno: 
        public DAB_Message 
    {
        DAB_DISPATCH_IMPL
        DAB_tenResult enResult;

        trMsgSrvCmdStopAnno():
            enResult(DAB_enResult_OK)
        {}

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMsgSrvCmdStopAnno: enResult=%d" , 
                            ETG_CENUM(DAB_tenResult, enResult)));
        };
        
    };

    struct DAB_trTmcRdbId
    {
        tU8 u8Cc;
        tU8 u8Ltn;
        tU16 u16Bsa;
        tU8 u8Sid;
        tBool bEncrypted;

        DAB_trTmcRdbId():
            u8Cc(0), u8Ltn(0),
            u16Bsa(0), u8Sid(0),
            bEncrypted(FALSE)
        {}

        tU32 u32GetVal() const {
            return (tU32)u8Cc<<24 | (tU32)u8Ltn<<16 | (tU32)u8Sid<<8 | ((tU8)bEncrypted ? 1 : 0);
        }
        bool operator!=(DAB_trTmcRdbId const &b) const {
            DAB_RET_1_IF_NEQ(u8Cc);
            DAB_RET_1_IF_NEQ(u8Ltn);
            DAB_RET_1_IF_NEQ(u16Bsa);
            DAB_RET_1_IF_NEQ(u8Sid);
            DAB_RET_1_IF_NEQ(bEncrypted);
            return FALSE;
        }

        bool operator<(DAB_trTmcRdbId const &b) const {
            DAB_RET_1_IF_LWR(u8Cc);
            DAB_RET_1_IF_LWR(u8Ltn);
            DAB_RET_1_IF_LWR(u16Bsa);
            DAB_RET_1_IF_LWR(u8Sid);
            DAB_RET_1_IF_LWR(bEncrypted);
            return FALSE;
        }

        bool operator==(DAB_trTmcRdbId const &b) const {
            return !(*this!=b);
        }

    };

    struct trMsgSrvRspChnSelect:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvRspChnSelect(DAB_tenResult enRes_) {
            enRes=enRes_;
        }
        trMsgSrvRspChnSelect() {
            enRes=DAB_enResult_OK;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvRspChnSelect:enRes=%d",
                            ETG_CENUM(DAB_tenResult, enRes)));
        }

        DAB_tenResult enRes;

    };

	struct trMsgSrvRspEnsSelect:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvRspEnsSelect(DAB_tenResult enRes_) {
            enRes=enRes_;
        }
        trMsgSrvRspEnsSelect() {
            enRes=DAB_enResult_OK;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvRspEnsSelect:enRes=%d",
                            ETG_CENUM(DAB_tenResult, enRes)));
        }

        DAB_tenResult enRes;

    };



    struct trMsgSrvCmdCompSelect: 
        public DAB_Message 
    {
        DAB_DISPATCH_IMPL
        tenChnSelectMode enChnSelectMode;
        tU32 u32Value;
        tBool bNegative;


        trMsgSrvCmdCompSelect():
            enChnSelectMode(enChnSelectMode_Invalid),
            u32Value(0),
            bNegative(FALSE)
        {}

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMsgSrvCmdCompSelect: enChnSelectMode=0x%x u32Value=0x%x (%u) bNegative=%d" , 
                            ETG_CENUM(tenChnSelectMode, enChnSelectMode),
                            u32Value,
                            u32Value,
                            bNegative
                            ));
        };
        
    };




    struct trMsgSrvRspCompSelect:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvRspCompSelect(DAB_tenResult enRes_) {
            enRes=enRes_;
        }
        trMsgSrvRspCompSelect() {
            enRes=DAB_enResult_OK;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvRspCompSelect:enRes=%d",
                            ETG_CENUM(DAB_tenResult, enRes)));
        }

        DAB_tenResult enRes;

    };

    struct trMsgSrvRspSetPresetStation:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvRspSetPresetStation(DAB_tenResult enRes_) {
            enRes=enRes_;
        }
        trMsgSrvRspSetPresetStation() {
            enRes=DAB_enResult_OK;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvRspSetPresetStation:enRes=%d",
                            ETG_CENUM(DAB_tenResult, enRes)));
        }

        DAB_tenResult enRes;

    };
  struct trMsgSrvRspAutostore:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvRspAutostore(DAB_tenResult enRes_,tU8 u8Stations_) {
            enRes=enRes_;
			u8NumberOfFoundStations=u8Stations_;
        }
        trMsgSrvRspAutostore() {
            enRes=DAB_enResult_OK;
			u8NumberOfFoundStations=0;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvRspAutostore:enRes=%d",
                            ETG_CENUM(DAB_tenResult, enRes)));
        }
        DAB_tenResult enRes;
		tU8 u8NumberOfFoundStations;
    };

	  struct trMsgSrvRspSetUser:
			public DAB_Message
		{
			DAB_DISPATCH_IMPL
			trMsgSrvRspSetUser(DAB_tenProfileUpdateType _enUpdateType,tU8 _u8UserId, DAB_tenResult _enRes) {
				enUpdateType=_enUpdateType;
				u8UserId=_u8UserId;
				enRes=_enRes;
			}
			trMsgSrvRspSetUser() {
				enUpdateType=DAB_enProfile_Change;
				u8UserId=0;
				enRes=DAB_enResult_OK;
			}
			virtual tVoid vTrace() const {
				ETG_TRACE_USR1(("trMsgSrvRspSetUser:enUpdateType=%d u8UserId=%d enRes=%d",
								ETG_CENUM(DAB_tenProfileUpdateType, enUpdateType),
								u8UserId,
								ETG_CENUM(DAB_tenResult, enRes)));
			}
			DAB_tenProfileUpdateType enUpdateType;
			tU8 u8UserId;
			DAB_tenResult enRes;
		};
    struct trMsgDrvCmdMtcConfig:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvCmdMtcConfig() {
            bTsuOn=FALSE;
        }
        trMsgDrvCmdMtcConfig(tBool bTsuOn_) {
            bTsuOn=bTsuOn_;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdMtcConfig:bTsuOn=%d",
                            bTsuOn));
        }

        tBool bTsuOn;

    };

    struct trMsgDrvCmdTsuSetCurrSrv:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvCmdTsuSetCurrSrv() {
            enTunerId=enMeca_TsuTunerId_MAIN_DAB_TUNER;
            rProgrammeService.vInvalidate();
        }
        trMsgDrvCmdTsuSetCurrSrv(DAB::tenMeca_TsuTunerId enTunerId_, 
                                 DAB::trMecaProgrammeService const &rProgrammeService_) {
            enTunerId=enTunerId_;
            rProgrammeService=rProgrammeService_;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdTsuSetCurrSrv:enTunerId=%d ...",
                            ETG_CENUM(DAB::tenMeca_TsuTunerId, enTunerId)));
            rProgrammeService.vTrace();
        }

        tenMeca_TsuTunerId enTunerId;
        trMecaProgrammeService rProgrammeService;
    };


    struct trMsgDrvRspTsuSwitchRecom:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvRspTsuSwitchRecom() {
            enTunerId=enMeca_TsuTunerId_MAIN_DAB_TUNER;
            rProgrammeService.vInvalidate();
        }
        trMsgDrvRspTsuSwitchRecom(DAB::tenMeca_TsuTunerId enTunerId_, 
                                  DAB::trMecaProgrammeService const &rProgrammeService_) {
            enTunerId=enTunerId_;
            rProgrammeService=rProgrammeService_;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvRspTsuSwitchRecom:enTunerId=%d ...",
                            ETG_CENUM(DAB::tenMeca_TsuTunerId, enTunerId)));
            rProgrammeService.vTrace();
        }

        tenMeca_TsuTunerId enTunerId;
        trMecaProgrammeService rProgrammeService;
    };


    struct trMsgDrvCmdTsuSelectFmPsid:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvCmdTsuSelectFmPsid():
            bMonitor(FALSE), 
            rProgrammeService()
        {}
        trMsgDrvCmdTsuSelectFmPsid(DAB::trMecaProgrammeService const &rProgrammeService_):
            bMonitor(TRUE),
            rProgrammeService(rProgrammeService_)
        {}

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdTsuSelectFmPsid: psid=0x%08x bMonitor=%d...",
                            rProgrammeService.u32GetSID(),
                            bMonitor));
        }

        tBool bMonitor;
        trMecaProgrammeService rProgrammeService;
    };



    struct trMsgClientStatusSourceActivity:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgClientStatusSourceActivity():
            enSourceActivity(enSourceActivity_Inactive),
            bTA(FALSE)
        {}
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgClientStatusSourceActivity:activity=%d, bTA=%d",
                            ETG_CENUM(tenSourceActivity, enSourceActivity),
                            bTA));
        }
        tenSourceActivity enSourceActivity;
        tBool bTA;
    };

    struct trMsgDrvCmdSourceActivity: public trMsgClientStatusSourceActivity {
        DAB_CLONE_IMPL(trMsgDrvCmdSourceActivity, trMsgClientStatusSourceActivity);
    };

 struct trMsgDrvCmdFrozenEnsListStatus:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvCmdFrozenEnsListStatus() {
            bOpen=FALSE;
        }
        trMsgDrvCmdFrozenEnsListStatus(tBool bOpen_) {
            bOpen=bOpen_;
        }


        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdFrozenEnsListStatus:bOpen=%d",
                            bOpen));
        }

        tBool bOpen;
    };


    struct trMsgDrvCmdFrozenChnListStatus:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvCmdFrozenChnListStatus() {
            bOpen=FALSE;
        }
        trMsgDrvCmdFrozenChnListStatus(tBool bOpen_) {
            bOpen=bOpen_;
        }


        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdFrozenChnListStatus:bOpen=%d",
                            bOpen));
        }

        tBool bOpen;
    };

    struct trMsgDrvCmdFrozenCompListStatus:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvCmdFrozenCompListStatus() {
            bOpen=FALSE;
        }
        trMsgDrvCmdFrozenCompListStatus(tBool bOpen_) {
            bOpen=bOpen_;
        }


        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdFrozenCompListStatus:bOpen=%d",
                            bOpen));
        }

        tBool bOpen;
    };



    struct trMsgSrvCmdLearn:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvCmdLearn(tenLearnType enLearnType_=enLearnType_OFF, tBool bOffLearn_=FALSE, tBool bRollback_=FALSE) {
            enLearnType = enLearnType_;
            bOffLearn=bOffLearn_;
            bRollback = bRollback_;
            enRes = DAB_enResult_OK;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMsgSrvCmdLearn:enLearnType=%d bOffLearn=%d bRollback=%d enRes=%d",
                            ETG_CENUM(tenLearnType, enLearnType),
                            bOffLearn,
                            bRollback,
                            ETG_CENUM(DAB_tenResult, enRes)));
        };
        tenLearnType enLearnType;
        tBool bRollback;
        tBool bOffLearn;
        DAB_tenResult enRes;

    };
#ifdef MUTE_ACTIVE
        struct trMsgSrvCmdManualLearnOp:
            public DAB_Message
        {
            DAB_DISPATCH_IMPL
            trMsgSrvCmdManualLearnOp(tenLearnType enLearnType_=enLearnType_OFF, tBool bOffLearn_=FALSE, tBool bRollback_=FALSE) {
                enLearnType = enLearnType_;
                bOffLearn=bOffLearn_;
                bRollback = bRollback_;
                enRes = DAB_enResult_OK;
            }
            virtual tVoid vTrace() const {
                ETG_TRACE_USR1(("  trMsgSrvCmdManualLearnOp:enLearnType=%d bOffLearn=%d bRollback=%d enRes=%d",
                                ETG_CENUM(tenLearnType, enLearnType),
                                bOffLearn,
                                bRollback,
                                ETG_CENUM(DAB_tenResult, enRes)));
            };
            tenLearnType enLearnType;
            tBool bRollback;
            tBool bOffLearn;
            DAB_tenResult enRes;

        };
#endif


    struct trMsgDrvEvtLearnDone:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        trMsgDrvEvtLearnDone():
            bCompleteScan(FALSE)
        {}
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMsgDrvEvtLearnDone bCompleteScan=%d",
                            bCompleteScan));
        };
        tBool bCompleteScan;
    };

    struct trMsgDrvEvtForcedLearnStart:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMsgDrvEvtForcedLearnStart"));
        };
    };

    struct trMsgSrvCmdDefSet:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvCmdDefSet(tenDefsetType enDefsetType_=enDefsetType_INVALID,
                          DAB_tenResult enRes_=DAB_enResult_OK):
            enDefsetType(enDefsetType_),
            enRes(enRes_)
        {}

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdDefSet:enDefsetType=%d enRes=%d",
                            ETG_CENUM(tenDefsetType, enDefsetType),
                            ETG_CENUM(DAB_tenResult, enRes)));
        }
        tenDefsetType enDefsetType;
        DAB_tenResult enRes;

    };

    struct trMsgSrvResultRsp:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        trMsgSrvResultRsp(DAB_tenResult enRes_=DAB_enResult_FAILED):
            enRes(enRes_)
        {}


        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvResultRsp:enRes=%d",
                            ETG_CENUM(DAB_tenResult, enRes)));
        }
        DAB_tenResult enRes;
    };


	struct trMsgSrvCmdSetUser:
		 public DAB_Message
	 {
		 DAB_DISPATCH_IMPL
		 trMsgSrvCmdSetUser(tU8 u8UserId_=0,
		 DAB_tenProfileUpdateType enUpdateType_ =DAB_enProfile_Change,
							tU8 u8DestUserId_ = 0,						   
						   DAB_tenResult enRes_=DAB_enResult_OK):
			 u8UserId(u8UserId_),
			 enUpdateType(enUpdateType_),
			 u8DestUserId(u8DestUserId_),
			 enRes(enRes_)
		 {}

		 virtual tVoid vTrace() const {
			 ETG_TRACE_USR1(("trMsgSrvCmdSetUser:enUpdateType=%d u8UserId=%d u8DestUserId=%d enRes=%d",
							 ETG_CENUM(DAB_tenProfileUpdateType, enUpdateType),
					 	 	 u8UserId,
							 u8DestUserId,
							 ETG_CENUM(DAB_tenResult, enRes)));
		 }
		 tU8 u8UserId;
		 DAB_tenProfileUpdateType enUpdateType;
		 tU8 u8DestUserId;
		 DAB_tenResult enRes;
	 };

	//Diagnosis Status
	struct trMsgSrvCmdDiagStatus:
		 public DAB_Message
	 {
		 DAB_DISPATCH_IMPL
		 trMsgSrvCmdDiagStatus(tenDiagAction _enDiagStaus=enDiagAction_FINISH,
						   DAB_tenResult enRes_=DAB_enResult_OK):
			 enDiagStaus(_enDiagStaus),
			 enRes(enRes_)
		 {}

		 virtual tVoid vTrace() const {
			 ETG_TRACE_USR1(("trMsgSrvCmdDiagStatus:enDiagStaus=%d enRes=%d",
					 	 	 ETG_CENUM(tenDiagAction, enDiagStaus),
							 ETG_CENUM(DAB_tenResult, enRes)));
		 }
		 tenDiagAction enDiagStaus;
		 DAB_tenResult enRes;
	 };

//Diagnosis Status
struct trMsgSrvCmdCoding:
	 public DAB_Message
 {
	 DAB_DISPATCH_IMPL
	 trMsgSrvCmdCoding(DAB_tenResult enRes_=DAB_enResult_OK):
	 bValue(TRUE),
	 enRes(enRes_)
	 {}

	 tBool bValue;
	 DAB_tenResult enRes;
 };


	 /*struct trMsgSrvRspSetUser:
		 public DAB_Message
	 {
		 DAB_DISPATCH_IMPL

		 trMsgSrvRspSetUser(DAB_tenResult enRes_=DAB_enResult_FAILED):
			 enRes(enRes_)
		 {}


		 virtual tVoid vTrace() const {
			 ETG_TRACE_USR1(("trMsgSrvRspSetUser:enRes=%d",
							 ETG_CENUM(DAB_tenResult, enRes)));
		 }
		 DAB_tenResult enRes;
	 };*/




    struct trMsgDrvCmdCompMode:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvCmdCompMode(tBool bCompMode_=FALSE):
            bCompMode(bCompMode_)
        {}


        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdCompMode:bCompMode%d",
                            bCompMode));
        }
        tBool bCompMode;
    };



    struct trLinkListElem {
        trLinkListElem():
            enLinkType(enLinkType_DAB_NOLINK),
            u32Id(0),
            u32EnsembleId(0),
            u8Quality(0)
        {
            sFreqLabel = "/0";
        }
        tenLinkType enLinkType;
        std::string sFreqLabel;
        tU32 u32Id;
        tU32 u32EnsembleId;
        tU8  u8Quality;
        
        
        bool operator!=(trLinkListElem const& b) const
        {
            DAB_RET_1_IF_NEQ(enLinkType);
            DAB_RET_1_IF_NEQ(sFreqLabel);
            DAB_RET_1_IF_NEQ(u32Id);
            DAB_RET_1_IF_NEQ(u32EnsembleId);
            DAB_RET_1_IF_NEQ(u8Quality);
            
            return FALSE;
        }
		 bool operator==(trLinkListElem const& b) const
        {
            DAB_RET_0_IF_NEQ(enLinkType);
            DAB_RET_0_IF_NEQ(sFreqLabel);
            DAB_RET_0_IF_NEQ(u32Id);
            DAB_RET_0_IF_NEQ(u32EnsembleId);
            DAB_RET_0_IF_NEQ(u8Quality);
            return FALSE;
        }
    };

    struct trAltFreqInfoList {
    	trAltFreqInfoList():
			u32Freq(0),
			s32FieldStrength(0),
			u8Quality(0)
		{
		}

		tU32 u32Freq;
		tS32 s32FieldStrength;
		tU8  u8Quality;

		bool operator!=(trAltFreqInfoList const& b) const
		{
			DAB_RET_1_IF_NEQ(u32Freq);
			DAB_RET_1_IF_NEQ(s32FieldStrength);
			DAB_RET_1_IF_NEQ(u8Quality);

			return FALSE;
		}
		 bool operator==(trAltFreqInfoList const& b) const
		{
			DAB_RET_0_IF_NEQ(u32Freq);
			DAB_RET_0_IF_NEQ(s32FieldStrength);
			DAB_RET_0_IF_NEQ(u8Quality);

			return FALSE;
		}
    };

#if 0
    // todo: this msg shall be sent by chnInfo
    struct trMsgDrvCmdChannelStable:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL


        trMsgDrvCmdChannelStable(tBool bStable_) {
            bStable=bStable_;
        }

        trMsgDrvCmdChannelStable() {
            bStable=FALSE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdChannelStable:bStable=%d",
                            bStable));
        }

        tBool bStable;
    };
#endif

    
    struct trMsgDrvCmdChnSelectState:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvCmdChnSelectState(tenChnState enChnState_=enChnState_Unavail, tBool bCompMode_=FALSE):
            enChnState(enChnState_),
            bCompMode(bCompMode_)
        {}


        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdChnSelectState:enChnState%d bCompMode=%d",
                            ETG_CENUM(tenChnState, enChnState),
                            bCompMode));
        }
        tBool bStable() const {
            return enChnState==enChnState_Stable;
        }
        tenChnState enChnState;
        tBool bCompMode;
    };

    struct trMsgDrvIndActivityChangesChannel:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvIndActivityChangesChannel"));
        }
    };

    struct trMsgDrvIndNumComponentsOfSrv:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvIndNumComponentsOfSrv(trMecaId const &rMecaId_=trMecaId(), tU8 u8NumComponents_=0):
            rMecaId(rMecaId_),
            u8NumComponents(u8NumComponents_)
        {}
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvIndNumComponentsOfSrv rMecaId=0x%08x u8NumComponents=%d",
                            rMecaId.u32GetSID(),
                            u8NumComponents));
        }
        trMecaId rMecaId;
        tU8 u8NumComponents;
    };


    struct trMsgDrvCmdSetDabSrvFollowClass:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        trMsgDrvCmdSetDabSrvFollowClass(tenSrvFollowClass enSrvFollowClass_=enSrvFollowClass_Off){
            enSrvFollowClass=enSrvFollowClass_;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdSetDabSrvFollowClass: enSrvFollowClass=%d",
                            ETG_CENUM(tenSrvFollowClass, enSrvFollowClass)));
        };

        tenSrvFollowClass enSrvFollowClass;
    };

    struct trMsgDrvRspSetDabSrvFollowClass:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        trMsgDrvRspSetDabSrvFollowClass(tenSrvFollowClass enSrvFollowClass_=enSrvFollowClass_Off){
            enSrvFollowClass=enSrvFollowClass_;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvRspSetDabSrvFollowClass: enSrvFollowClass=%d",
                            ETG_CENUM(tenSrvFollowClass, enSrvFollowClass)));
        };

        tenSrvFollowClass enSrvFollowClass;
    };


    struct trMsgDrvCmdConfigDabSrvFollow:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        trMsgDrvCmdConfigDabSrvFollow(tenSrvFollowClass enSrvFollowClass_=enSrvFollowClass_Normal, tU8 u8ServFollowMode_=0){
            enSrvFollowClass=enSrvFollowClass_;
            u8ServFollowMode=u8ServFollowMode_;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdConfigDabSrvFollow: enSrvFollowClass=%d u8ServFollowMode=0x%x",
                            ETG_CENUM(tenSrvFollowClass, enSrvFollowClass),
                            u8ServFollowMode));
        };
        tenSrvFollowClass enSrvFollowClass; 
        tU8 u8ServFollowMode;
    };


    struct trMsgDrvCmdChnListPsidMonitor:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvCmdChnListPsidMonitor(trMecaId const &rMecaId_=trMecaId()):
            rMecaId(rMecaId_) 
        {}
        
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdChnListPsidMonitor:rMecaId=0x%08x",
                            rMecaId.u32GetSID()));
        }
        trMecaId rMecaId;
    };

    struct trMsgDrvRspChnListPsidMonitor:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvRspChnListPsidMonitor(trMecaId const &rMecaId_=trMecaId()):
            rMecaId(rMecaId_)
        {}
        
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvRspChnListPsidMonitor:rMecaId=0x%08x",
                            rMecaId.u32GetSID()));
        }
        trMecaId rMecaId;
    };

    struct trMsgDrvCmdCompListScidiMonitor:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvCmdCompListScidiMonitor(trMecaId const &rMecaId_=trMecaId()):
            rMecaId(rMecaId_)
        {}
        
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdCompListScidiMonitor:rMecaId=0x%04x",
                            rMecaId.u16GetScidi()));
        }
        trMecaId rMecaId;
    };

    struct trMsgDrvRspCompListScidiMonitor:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvRspCompListScidiMonitor(trMecaId const &rMecaId_=trMecaId()):
            rMecaId(rMecaId_),
            bNew(FALSE),
            bLost(FALSE)
        {}
        
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvRspCompListScidiMonitor:rMecaId=0x%04x bNew=%d bLost=%d",
                            rMecaId.u16GetScidi(), bNew, bLost));
        }
        trMecaId rMecaId;
        tBool bNew;
        tBool bLost;
    };



    struct trMsgDrvIndCompModeFailed:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvIndCompModeFailed"));
        }
    };

    struct trMsgDrvCmdChnListSetCurrentSrv:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvCmdChnListSetCurrentSrv(trMecaId const &rMecaId_=trMecaId()):
            rMecaId(rMecaId_) 
        {}
        
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdChnListSetCurrentSrv:rMecaId=0x%08x",
                            rMecaId.u32GetSID()));
        }
        trMecaId rMecaId;
    };

    struct trMsgDrvRspChnListCurrentSrvChanged:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvRspChnListCurrentSrvChanged(trMecaId const &rMecaId_=trMecaId()):
            rMecaId(rMecaId_),
            u8Scids(0)
        {}
        
        trMsgDrvRspChnListCurrentSrvChanged(trMecaProgrammeService const &rMecaPrgService):
                   rMecaId(rMecaPrgService._u32Id),
                   u8Scids((tU8)(rMecaPrgService._u16Scids))
               {}

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvRspChnListCurrentSrvChanged:rMecaId=0x%08x",
                            rMecaId.u32GetSID()));
        }
        trMecaId rMecaId;
        tU8 u8Scids;
    };



    struct trMsgDrvCmdSetDbNotifyMode:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvCmdSetDbNotifyMode(tBool bOn_):
            bOn(bOn_)
        {}
        
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdSetDbNotifyMode:bOn=%d",
                            bOn));
        }
        tBool bOn;
    };

    struct trMsgDrvRspSetDbNotifyMode:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvRspSetDbNotifyMode(tBool bOn_):
            bOn(bOn_)
        {}
        
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvRspSetDbNotifyMode:bOn=%d",
                            bOn));
        }
        tBool bOn;
    };


    struct trMsgDrvCmdSetRdmNotifyMode:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvCmdSetRdmNotifyMode(tenRdmNotifyMode enNotifyMode_=enRdmNotifyMode_Normal):
            enNotifyMode(enNotifyMode_)
        {}
        
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdSetRdmNotifyMode:enNotifyMode=%d",
                            ETG_CENUM(tenRdmNotifyMode, enNotifyMode)));
        }
        tenRdmNotifyMode enNotifyMode;
    };

    struct trMsgDrvRspSetRdmNotifyMode:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvRspSetRdmNotifyMode(tenRdmNotifyMode enNotifyMode_=enRdmNotifyMode_Normal):
            enNotifyMode(enNotifyMode_)
        {}
        
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvRspSetRdmNotifyMode:enNotifyMode=%d",
                            ETG_CENUM(tenRdmNotifyMode, enNotifyMode)));
        }
        tenRdmNotifyMode enNotifyMode;
    };

    struct trMsgDrvSetTmcMode:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvSetTmcMode(){
            bEnable=FALSE;
        }
        trMsgDrvSetTmcMode(tBool bEnable_){
            bEnable=bEnable_;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("dabdrv_trMsgTmcMode: bEnable=%d",
                            bEnable));
        };
        tBool bEnable;
    };

    struct trMsgDrvSetTpegMode:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvSetTpegMode(){
            bEnable=FALSE;
        }
        trMsgDrvSetTpegMode(tBool bEnable_){
            bEnable=bEnable_;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("dabdrv_trMsgTpegMode: bEnable=%d",
                            bEnable));
        };
        tBool bEnable;
    };
    struct trMsgDrvLoadTpegSimFile:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvLoadTpegSimFile()
        {}
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvLoadTpegSimFile"));
        };
    };
    struct trMsgDrvIndEnsSrvListChanged:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvIndEnsSrvListChanged"));
        };

    };

    struct trMsgSrvCmdSelectTmc:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        trMsgSrvCmdSelectTmc():
            rEns(),
            rRdbId(),
            enRes(DAB_enResult_OK)
        {};


        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdSelectTmc: bActivate=%d rEns=0x%08x u32RdbId=0x%08x enRes=%d",
                            bActivate(),
                            rEns.u32GetID(),
                            rRdbId.u32GetVal(),
                            ETG_CENUM(DAB_tenResult, enRes)));
        };

        trMecaEnsemble rEns;
        //vector<DAB_trTmcRdbId> vecRdbIdList;
        DAB_trTmcRdbId rRdbId;
        DAB_tenResult enRes;

        tBool bActivate() const {
            return rEns.bIsValid() && rRdbId.u32GetVal();
        }
    };

    struct trMsgSrvCmdRdbidWishlist:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        trMsgSrvCmdRdbidWishlist()
        {};

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdRdbidWishlist"));
        };

        vector<DAB_trTmcRdbId> vecRdbIdList;
    };

    struct trMsgSrvRspRdbidWishlist:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        trMsgSrvRspRdbidWishlist()
        {};

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvRspRdbidWishlist"));
        };

        vector<DAB_trTmcRdbId> vecRdbIdList;
    };

    struct trMsgSrvCmdStorePersData:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        trMsgSrvCmdStorePersData(){};


        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdStorePersData"));
        };

    };

    struct trMsgDrvCmdSysStore:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        trMsgDrvCmdSysStore( tU8 u8UsrId_=0, tBool bQuick_=FALSE ):
            u8UsrId(u8UsrId_), // not used by now
            bQuick(bQuick_)
        {};


        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdSysStore  u8UsrId=%d bQuick=%d",
                            u8UsrId, bQuick));
        };
        tU8 u8UsrId;
        tBool bQuick;

    };

    struct trMsgDrvCmdSysReStore:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        trMsgDrvCmdSysReStore(tU8 u8UsrId_=0):
            u8UsrId(u8UsrId_)
        {};


        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdSysReStore u8UsrId=%d",
                            u8UsrId));
        };
        tU8 u8UsrId;

    };


    struct trMsgDrvCmdApplyUsrData:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        trMsgDrvCmdApplyUsrData(){};


        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdApplyUsrData"));
        };

    };


    struct trTmcData {
        tBool bValid;
        tU16 u16Scidi;
        tU8  u8TmcData[5]; 
        tU8  u8TmcGroup;
        trTmcData():
            bValid(FALSE),
            u16Scidi(0),
            u8TmcGroup(0)
        {
            u8TmcData[0]=0;
        };

        trTmcData(tU8 const *pu8TmcData):
            bValid(FALSE),
            u16Scidi(0),
            u8TmcGroup(0)
        {
            u8TmcData[0]=0;
            
            if (OSAL_NULL != pu8TmcData) {
                bValid=TRUE;
            }
			/**vnd4kor: Solving prio 2 lint warnings*/
            if(NULL != pu8TmcData )
			{
				OSAL_pvMemoryCopy(u8TmcData, pu8TmcData, 5);
			}
        }
        bool operator!=(trTmcData const& b) const {
            DAB_RET_1_IF_NEQ(bValid);
            DAB_RET_1_IF_NEQ(u16Scidi);
            DAB_RET_1_IF_NEQ(u8TmcGroup);
            return OSAL_s32MemoryCompare(u8TmcData, b.u8TmcData, 5) ? TRUE : FALSE;
        }

        tVoid vTrace() const {
            ETG_TRACE_USR1(("  trTmcData:bValid=%d u16Scidi=%d u8TmcGroup=%d u8TmcData=%x %x %x %x %x",
                            bValid,
                            u16Scidi,
                            u8TmcGroup,
                            u8TmcData[0],u8TmcData[1],u8TmcData[2],u8TmcData[3],u8TmcData[4]));
            
        }
    };

    struct trTmcAvailListElem {
        
        trTmcAvailListElem():
            rEnsembleId(),u8Quality(0),u8NumberOfRdbIds(0),
            u32Scidi(0),u8Ecc(0),vectorRdbIdEntries()
        {}

        trMecaEnsemble rEnsembleId;
        tU8  u8Quality;
        tU8  u8NumberOfRdbIds;
        tU32 u32Scidi;
        trMecaProgrammeService rProgrammeService;
        tU8  u8Ecc;
        vector<DAB::DAB_trTmcRdbId>vectorRdbIdEntries;

        bool operator!=(trTmcAvailListElem const& b) const
        {
            DAB_RET_1_IF_NEQ(rEnsembleId);
            DAB_RET_1_IF_NEQ(u8Quality);
            DAB_RET_1_IF_NEQ(u32Scidi);
            DAB_RET_1_IF_NEQ(u8Ecc);
            DAB_RET_1_IF_NEQ(u8NumberOfRdbIds);
            DAB_RET_1_IF_NEQ(vectorRdbIdEntries);
            return FALSE;
        }

		 bool operator==(trTmcAvailListElem const& b) const
        {
            DAB_RET_0_IF_NEQ(rEnsembleId);
            DAB_RET_0_IF_NEQ(u8Quality);
            DAB_RET_0_IF_NEQ(u32Scidi);
            DAB_RET_0_IF_NEQ(u8Ecc);
            DAB_RET_0_IF_NEQ(u8NumberOfRdbIds);
            DAB_RET_0_IF_NEQ(vectorRdbIdEntries);
            return FALSE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  trTmcAvailListElem: rEnsembleId=0x%08x u8Quality=%d numElem=%d",
                            rEnsembleId.u32GetID(),
                            u8Quality,
                            (tU16)(vectorRdbIdEntries.size())));
        };
    };

    struct trMsgSrvCmdGetTmcLearnData: public DAB_Message
    {
        DAB_DISPATCH_IMPL
        
        vector<DAB::trTmcAvailListElem>vectorTmcAvailEntries;
        
        trMsgSrvCmdGetTmcLearnData():
            vectorTmcAvailEntries()
        {}
        
        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  trMsgSrvCmdGetTmcLearnData: numElem=%d",
                            (tU16)(vectorTmcAvailEntries.size())));
        };
    };

    struct trMsgDrvIndRadioText: public DAB_Message
    {
        DAB_DISPATCH_IMPL
        
        trRadioText rRadioText;
        
        trMsgDrvIndRadioText(trRadioText const &rRadioText_=trRadioText()):
            rRadioText(rRadioText_)
        {}
        
        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  trMsgDrvIndRadioText: rRadioText follows"));
                           rRadioText.vTrace();
        };
    };

    struct trRTPlusInfo {
    	trRTPlusInfo():
    			u8ContentType(0),
    			sContent()
            {
            }
    			tU8 u8ContentType;
    			trRadioText sContent;

    			bool operator!=(trRTPlusInfo const& b) const
				{
					DAB_RET_1_IF_NEQ(u8ContentType);
					DAB_RET_1_IF_NEQ(sContent);
					return FALSE;
				}
				bool operator==(trRTPlusInfo const& b) const
				{
					DAB_RET_0_IF_NEQ(u8ContentType);
					DAB_RET_0_IF_NEQ(sContent);
					return FALSE;
				}
            virtual tVoid vTrace() const {
                ETG_TRACE_USR1(("    trRTPlusInfo: u8ContentType=%d",
								u8ContentType));
            }
        };

    struct trMsgDrvIndRadioTextPlus: public DAB_Message
	{
		DAB_DISPATCH_IMPL

		vector<trRTPlusInfo> vRDTextPlus;
        DAB_tenResult enRes;

		trMsgDrvIndRadioTextPlus():
			vRDTextPlus(),
			enRes(DAB_enResult_OK)
		{	}
	};

    struct trActiveEnsembleInfo {
        tBool bValid;
        tU32 u32EnsembleId;
        tBool bLabelPresent;
        trMecaLabel rEnsembleLabel;
        tU32 u32Frequency;
        tU8  u8NumberOfAudioServices;
        tU8  u8NumberOfDataServices;
        tU8  u8ReceptionQuality;

        trActiveEnsembleInfo():
            bValid(FALSE),
            u32EnsembleId(0),
            bLabelPresent(FALSE),
            rEnsembleLabel(),
            u32Frequency(0),
            u8NumberOfAudioServices(0),
            u8NumberOfDataServices(0),
            u8ReceptionQuality(0)
        {}

#if 0
        operator trActiveEnsembleInfo() {
            trActiveEnsembleInfo rRes;
            rRes.u32EnsembleId = u32EnsembleId;
            rRes.rEnsembleLabel = rEnsembleLabel;
            rRes.u32Frequency = u32Frequency;
            rRes.u8NumberOfAudioServices = u8NumberOfAudioServices;
            rRes.u8NumberOfDataServices = u8NumberOfDataServices;
            rRes.u8ReceptionQuality = u8ReceptionQuality;
            return rRes;
        }
#endif
        tVoid vTrace() const {
            ETG_TRACE_USR1(("  trActiveEnsembleInfo:bValid=%d u32EnsembleId=0x%x bLabelPresent=%d u32Frequency=%d "
                            "u8NumberOfAudioServices=%d u8NumberOfDataServices=%d u8ReceptionQuality=0x%04x",
                            bValid,
                            u32EnsembleId,
                            bLabelPresent,
                            u32Frequency,
                            u8NumberOfAudioServices,
                            u8NumberOfDataServices,
                            u8ReceptionQuality));
            
        }

    };


    struct trActiveComponentInfo {
        tBool bValid;
        tU32 u32ComponentId;
        tBool bLabelPresent;
        trMecaLabel rComponentLabel;
        tU16 u16SCIDI;
        tU8 u8Subchannel;
        DAB_tenComponentType e8ComponentType;
        DAB_tenServiceType e8ServiceType;
        DAB_tenTMIdType e8TMIdType;
        tU8 u8AudioDataServiceComponentType;

        trActiveComponentInfo():
            bValid(FALSE),
            u32ComponentId(0),
            bLabelPresent(FALSE),
            rComponentLabel(),
            u16SCIDI(0),
            u8Subchannel(0),
            e8ComponentType(DAB_enComponentType_PRIMARY),
            e8ServiceType(DAB_enServiceType_UNKNOWN),
            e8TMIdType(DAB_enTMIdType_MSC_AUDIO_STREAM),
            u8AudioDataServiceComponentType(0)
        {}
#if 0
        operator trActiveComponentInfo() {
            trActiveComponentInfo rRes;
            rRes.u32ComponentId = u32ComponentId;
            rRes.rComponentLabel = rComponentLabel;
            rRes.u16SCIDI = u16SCIDI;
            rRes.u8Subchannel = u8Subchannel;
            rRes.e8ComponentType = e8ComponentType;
            rRes.e8ServiceType = e8ServiceType;
            rRes.e8TMIdType = e8TMIdType;
            rRes.u8AudioDataServiceComponentType = u8AudioDataServiceComponentType;
            return rRes;
        }
#endif
        tVoid vTrace() const {
            ETG_TRACE_USR1(("  trActiveComponentInfo:bValid=%d u32ComponentId=0x%x bLabelPresent=%d u16SCIDI=%d "
                            "u8Subchannel=%d ComponentType=%d ServiceType=%d TMIdType=%d A/DSCTY=%d",
                            bValid,
                            u32ComponentId,
                            bLabelPresent,
                            u16SCIDI,
                            u8Subchannel,
                            ETG_CENUM(DAB_tenComponentType, e8ComponentType),
                            ETG_CENUM(DAB_tenServiceType, e8ServiceType),
                            ETG_CENUM(DAB_tenTMIdType, e8TMIdType),
                            u8AudioDataServiceComponentType));
            
        }       

    };

    inline tBool bIsChnStateSelecting(tenChnState enChnState) {
        tBool bActive=FALSE;
        //lint -e{788} enum constant '...' not used within defaulted switch
        switch (enChnState) {
            case enChnState_Selecting:
            case enChnState_SelectingWait:
            case enChnState_SelectingSearch:
                bActive=TRUE;
                break;
            default:
                break;
        }
        return bActive;
    }

    template<class D>
    class DAB_tclClassCmp {
    public:
        virtual tBool operator()(D &rRight, D &rLeft) const =0;
    };

    class trTmcAvailListElemSigQualBigger: public DAB_tclClassCmp<trTmcAvailListElem>
    {
    public:
        virtual tBool operator()(trTmcAvailListElem &rRight, trTmcAvailListElem &rLeft) const {
            return rRight.u8Quality>rLeft.u8Quality;
        };
    };

    struct trMsgSrvCmdSelectTpegUri:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL

        trMsgSrvCmdSelectTpegUri(string sSelectUri_="", tBool bSelected_=FALSE):
        sSelectUri(sSelectUri_),
        bSelected(bSelected_)
        {};
    
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdSelectTpegUri"));
        };

        std::string sSelectUri;
        tBool bSelected;
    };


    struct trMsgSrvCmdGetTpegUriList:
    	public DAB_Message
    {
    	DAB_DISPATCH_IMPL

        trMsgSrvCmdGetTpegUriList()
        {};

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdGetTpegUriList"));
        };
    };

    struct DAB_trTpegUriListElement
    {
        std::string sUri;
        DAB_tenUriSelectState enUriSelectState;
        tBool bActive;

        DAB_trTpegUriListElement():
            enUriSelectState(DAB_enUriSelectState_NOT_SELECTED)
        {
            sUri = "/0";
            bActive = FALSE;
        };

        bool operator!=(DAB_trTpegUriListElement const& b) const
        {
            DAB_RET_1_IF_NEQ(enUriSelectState);
            DAB_RET_1_IF_NEQ(sUri);
            DAB_RET_1_IF_NEQ(bActive);
            return FALSE;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("DAB_trTpegUriListElement"));
        };
    };

    struct trMsgSrvRspGetTpegUriList:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvRspGetTpegUriList(vector<DAB_trTpegUriListElement> const &vcTpegUriList_):vcTpegUriList(vcTpegUriList_){
            enRes=DAB_enResult_OK;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvRspGetTpegUriList: enRes=%d ",
                            ETG_CENUM(DAB_tenResult, enRes)
                            ));
            DAB_FOREACH_CONST(vector<DAB_trTpegUriListElement>, iter, vcTpegUriList) {
                ETG_TRACE_USR1((    "  %50s ", 
                                    iter->sUri.c_str()));
            }
        };

        DAB_tenResult enRes;
        vector<DAB_trTpegUriListElement> const &vcTpegUriList;
        
    };

    struct DAB_trTime
    {
        tU32 u32DabSec;      /* seconds */
        tU32 u32DabMin;      /* minutes */
        tU32 u32DabHour;     /* hours */
        tU32 u32DabMday;     /* day of the month */
        tU32 u32DabMon;      /* month */
        tU32 u32DabYear;     /* year */
        tU32 u32DabWday;     /* day of the week */
        tU32 u32DabYday;     /* day in the year */
        tU32 u32DabUtcOff;   /* difference between UTC and local time */

        DAB_trTime():
            u32DabSec(0),
            u32DabMin(0),
            u32DabHour(12),
            u32DabMday(1),
            u32DabMon(1),
            u32DabYear(2014),
            u32DabWday(1),
            u32DabYday(0),
            u32DabUtcOff(0)
        {};
       
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("DAB_trTime %02d.%02d.%04d  %02d:%02d:%02d ",
                                    u32DabMday,
                                    u32DabMon,
                                    u32DabYear,
                                    u32DabHour,
                                    u32DabMin,
                                    u32DabSec));
        };

    };
	struct trMsgDrvCmdNotifyLearnComplete:
		   public DAB_Message
	   {
		   DAB_DISPATCH_IMPL
		   trMsgDrvCmdNotifyLearnComplete()
		   {}

		   virtual tVoid vTrace() const {
			   ETG_TRACE_USR1(("trMsgDrvCmdNotifyLearnComplete"));
		   }
	   };
    /*
      struct trLinkList{

      trLinkList():
      vecLinkList() 
      {}
        
      virtual tVoid vTrace() const {
      ETG_TRACE_USR1(("trLinkList: "
      ));
      }

      vector<trLinkListElem> vecLinkList;
      };
    */

	struct trMsgDrvCmdDabFrequencyTable:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvCmdDabFrequencyTable(tU8 u8FrequencyTable_=0):
            u8FrequencyTable(u8FrequencyTable_)
        {};
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvCmdDabFrequencyTable u8FrequencyTable=%d",
                            u8FrequencyTable));
        };
        tU8 u8FrequencyTable;
    };
	struct trMsgSrvCmdScan:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvCmdScan(tBool bStartScan_=FALSE):
            bStartScan(bStartScan_)
        {};
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvCmdScan bStartScan=%d",
                            bStartScan));
        };
        tBool bStartScan;
    };
	  struct trMsgDrvDbQuery: 
        public DAB_Message 
    {
        DAB_DISPATCH_IMPL
        tenDbQueryType enDbQuery;
        trMsgDrvDbQuery(tenDbQueryType enDbQuery_=Query_InitialAutostore):
            enDbQuery(enDbQuery_),
            u8Param1(0)
        {}
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMsgDrvDbQuery: enDbQuery=%x" , 
                            ETG_CENUM(tenDbQueryType, enDbQuery)));
        };
        tU8 u8Param1;
    };
  	  struct trMsgDrvDbQueryTrigger:
          public DAB_Message
      {
          DAB_DISPATCH_IMPL
          tenDbQueryTriggerType enDbQueryTrigger;
          trMsgDrvDbQueryTrigger(tenDbQueryTriggerType enDbQueryTrigger_=QueryTrigger_GlobalList):
        	  enDbQueryTrigger(enDbQueryTrigger_)
          {}
          virtual tVoid vTrace() const {
              ETG_TRACE_USR1(("  trMsgDrvDbQuery: enDbQuery=%x" ,
                              ETG_CENUM(tenDbQueryTriggerType, enDbQueryTrigger)));
          };
      };
   struct trMsgDrvUpdateLink:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgDrvUpdateLink(){          
        }
        trMsgDrvUpdateLink(trLinkListElem rElem_){
			rLinkListElem = rElem_;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgDrvUpdateLink: enLinkType=%x u32Id=%x",
				 ETG_CENUM(tenLinkType, rLinkListElem.enLinkType),
				rLinkListElem.u32Id));
        };
        trLinkListElem rLinkListElem;
    };

	struct trMsgDrvAltFreqInfo:
		 public DAB_Message
	 {
		 DAB_DISPATCH_IMPL
		 trMsgDrvAltFreqInfo(){
		 }
		 trMsgDrvAltFreqInfo(trAltFreqInfoList rElem_){
			 rAltFreqInfo = rElem_;
		 }
		 virtual tVoid vTrace() const {
			 ETG_TRACE_USR1(("trMsgDrvUpdateLink: u32Frequency=%06u s32FieldStrength=%d u8Quality=%x",
					 rAltFreqInfo.u32Freq,
					 rAltFreqInfo.s32FieldStrength,
					 rAltFreqInfo.u8Quality));
		 };
		 trAltFreqInfoList rAltFreqInfo;
	 };

	struct trEnsembleInfo:
		 public DAB_Message
	 {
		 DAB_DISPATCH_IMPL
		 trEnsembleInfo():
		 lEnsembleList()
		{}

		 trEnsembleInfo(vector<trMecaEnsembleListElement> const &lEnsembleList_):lEnsembleList(lEnsembleList_){}

		 virtual tVoid vTrace() const {
			 ETG_TRACE_USR1(("trEnsembleInfo"));
		 };
		 vector<trMecaEnsembleListElement>lEnsembleList;
	 };

	struct trEnsembleServiceList {
			 trEnsembleServiceList():
				 lServiceList(),
				 rEnsemble()
			{
			}
			 set<trMecaProgrammeService>lServiceList;
			 trMecaEnsemble rEnsemble;

			virtual tVoid vTrace() const {
				ETG_TRACE_USR1(("    trEnsembleServiceList"));
			}
		};


	 struct trEnsembleServiceInfo:
		 public DAB_Message
	 {
		 DAB_DISPATCH_IMPL
		 trEnsembleServiceInfo():
		 vEnsembleServiceMap()
		{};

		 virtual tVoid vTrace() const {
			 ETG_TRACE_USR1(("trEnsembleServiceInfo"));
		 };
		 map<trMecaEnsemble,trEnsembleServiceList> vEnsembleServiceMap;
	 };



	 struct trServiceInfo:
		 public DAB_Message
	 {
		 DAB_DISPATCH_IMPL
		 trServiceInfo():
		 u32Eid(0),
		 rProgrammeService(),
		 u8PresetNum(0),
		 ServiceLabel(),
		 ComponentLabel(),
		 EnsembleLabel(),
		 u16Scids(0),
		 u8PSFlag(0),
		 bNotRXFlag(0),
		 u8ASUFlags(0),
		 u8PTY(0),
		 sFreqLabel("")
		{}


		 virtual tVoid vTrace() const {
			 ETG_TRACE_USR1(("trServiceInfo"));
		 };

			tU32 u32Eid;
		    trMecaProgrammeService rProgrammeService;
			tU8 u8PresetNum;
			APILabel ServiceLabel;
			APILabel ComponentLabel;
			trMecaLabel EnsembleLabel;
			tU16 u16Scids;
			tU8 u8PSFlag;
			tBool bNotRXFlag;
			tU8 u8ASUFlags;
			tU8 u8PTY;
			string sFreqLabel;
	 };

	 struct trServiceInformation:
		 public DAB_Message
	 {
		 DAB_DISPATCH_IMPL
		 trServiceInformation():
		 vecServiceInfo()
		{};

		 virtual tVoid vTrace() const {
			 ETG_TRACE_USR1(("trServiceInformation"));
		 };
		 vector<trServiceInfo> vecServiceInfo;
	 };

   struct trMsgDrvDeleteMemory:
	   public DAB_Message
   {
	   DAB_DISPATCH_IMPL
	   trMsgDrvDeleteMemory(tenMeca_DbConfig_ValueId enDeleteAction_=enMeca_DbConfig_ValueId_delete_All){
		   enDeleteAction = enDeleteAction_;
	   }
	   virtual tVoid vTrace() const {
		   ETG_TRACE_USR1(("trMsgDrvDeleteMemory: enDeleteAction=%x ",
			   ETG_CENUM(tenMeca_DbConfig_ValueId, enDeleteAction)));
	   };
	   tenMeca_DbConfig_ValueId enDeleteAction;
   };
    struct trMsgSrvRspScan:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvRspScan(DAB_tenResult enRes_) {
            enRes=enRes_;
        }
        trMsgSrvRspScan() {
            enRes=DAB_enResult_OK;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvRspScan:enRes=%d",
                            ETG_CENUM(DAB_tenResult, enRes)));
        }
        DAB_tenResult enRes;
    };
	struct trMsgSrvGetTIIList:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvGetTIIList() {
        }
        virtual tVoid vTrace() const {
			ETG_TRACE_USR1(("trMsgSrvGetTIIList:"));
        }
    };
	struct trMsgSrvRspGetTIIList:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvRspGetTIIList(vector<trMecaTiiInfo> const &vcTIiList_):vecTiiList(vcTIiList_){
            enRes=DAB_enResult_OK;            
        }
		trMsgSrvRspGetTIIList():
		enRes(DAB_enResult_OK),
			vecTiiList(){
		}
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvRspGetTIIList: enRes=%d TotalNumberOfElements=%d ",
                            ETG_CENUM(DAB_tenResult, enRes),
                            (tU16)(vecTiiList.size())
                            ));          
        };
        DAB_tenResult enRes;      
        vector<trMecaTiiInfo>vecTiiList;
    };

struct trMsgSrvRspExpIdValueU32:
		public DAB_Message
	{
		DAB_DISPATCH_IMPL
		trMsgSrvRspExpIdValueU32(tenMeca_ExpValueId _expValueId,  tU32 _u32Val):enValueId(_expValueId),u32Value(_u32Val){
		enRes = DAB_enResult_OK;
		}
		trMsgSrvRspExpIdValueU32():
			enValueId(enMeca_ExpValueId_FieldStrength),
			u32Value(0),
			enRes(DAB_enResult_OK)
		{

		}
		virtual tVoid vTrace() const {
			ETG_TRACE_USR1(("trMsgSrvRspIdValueU32: enValueId=%d u32Value=%d ",
							ETG_CENUM(tenMeca_ExpValueId, enValueId),
							u32Value
							));
		};

	   tenMeca_ExpValueId enValueId;
	   tU32 u32Value;
	   DAB_tenResult enRes;

	};
struct trMsgSrvRspSrcState:
		public DAB_Message
		{
			DAB_DISPATCH_IMPL
			trMsgSrvRspSrcState(DAB_tenSourceState _enSourceState):enSourceState(_enSourceState){
			enRes = DAB_enResult_OK;
			}
			trMsgSrvRspSrcState():
			enSourceState(DAB_enSourceState_BG),
			enRes(DAB_enResult_OK)
			{
			}
			virtual tVoid vTrace() const {
				ETG_TRACE_USR1(("trMsgSrvRspSrcState: enSourceState=%d ",
					ETG_CENUM(DAB_tenSourceState, enSourceState)
					));
			};

			DAB_tenSourceState enSourceState;
			DAB_tenResult enRes;

		};
		struct trMsgSrvCmdRspTuneFreqlabel:
				public DAB_Message
		{
			DAB_DISPATCH_IMPL
			trMsgSrvCmdRspTuneFreqlabel(){
				enRes = DAB_enResult_OK;
			}
			virtual tVoid vTrace() const {
				ETG_TRACE_USR1(("trMsgSrvCmdRspTuneFreqlabel: enRes= %d",
						ETG_CENUM(DAB_tenResult, enRes)));
			};
			DAB_tenResult enRes;
		};

		struct trMsgSrvCmdRspSavePresetList:
				public DAB_Message
		{
			DAB_DISPATCH_IMPL
			trMsgSrvCmdRspSavePresetList(){
				enRes = DAB_enResult_OK;
			}
			virtual tVoid vTrace() const {
				ETG_TRACE_USR1(("trMsgSrvCmdRspSavePresetList: enRes= %d",
						ETG_CENUM(DAB_tenResult, enRes)));
			};
			DAB_tenResult enRes;
		};
	struct trMsgSrvCmdExpSelect:
		public DAB_Message
	{
		DAB_DISPATCH_IMPL
		trMsgSrvCmdExpSelect(tenMeca_ExpValueId _enValId, tenMeca_ExpAction _enAction):enValueId(_enValId), enAction(_enAction){

		}
		trMsgSrvCmdExpSelect(): enValueId(enMeca_ExpValueId_FieldStrength),
							enAction(enMeca_ExpAction_UPDATE_VALUE){

		}
		virtual tVoid vTrace() const {
			ETG_TRACE_USR1(("trMsgSrvCmdExpSelect: valueId=%d action=%d",
					enValueId,
					enValueId));
		}
	    tenMeca_ExpValueId enValueId;
	    tenMeca_ExpAction enAction;
	};	

	struct trMsgSrvDataList:
        public DAB_Message
	{
		DAB_DISPATCH_IMPL
        trMsgSrvDataList(){
                    
        }	
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("trMsgSrvLogoList: TotalNumberOfElements=%d ",
                                                 (tU16)(mapLogoList.size())
                            ));          
        };
      
      
		map<tU32,string> mapLogoList;
		multimap<tU32,tU8> mmapContentIdList;
	};

	struct trMsgSrvRspCmdLearn:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        trMsgSrvRspCmdLearn() {
            enRes = DAB_enResult_OK;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1(("  trMsgSrvRspCmdLearn: enRes=%d",
                            ETG_CENUM(DAB_tenResult, enRes)));
        };

        DAB_tenResult enRes;

    };

	struct trMsgSrvCmdListUpdateStatus:
		public DAB_Message
	{
		DAB_DISPATCH_IMPL
		trMsgSrvCmdListUpdateStatus(tBool bStatus) {
			bEnableListUpadteStatus = bStatus;
		}
		virtual tVoid vTrace() const {
			ETG_TRACE_USR1(("  trMsgSrvCmdListUpdateStatus: =%d",
					bEnableListUpadteStatus));
		};

		tBool bEnableListUpadteStatus;

	};

} // namespace DAB
#endif
