/************************************************************************
 * FILE:         dabdrv_tpeg.cpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of dabdrv_tpeg
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      	| Author                       | Modification
				  
				
 *************************************************************************/

#include "dabdrv_tpeg.hpp"
#include "dabdrv_main.hpp"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_DRV_TPEG
#include "trcGenProj/Header/dabdrv_tpeg.cpp.trc.h"
#endif

#ifndef __RFC3072_HELPER_H__
#include "rfc3072_helper.h"
#endif


#ifdef VARIANT_S_FTR_ENABLE_TPEG
#define THREE (tU8)3
#endif


namespace DAB {
    // a timer-event
#define TPEG_FILE_LOCATION "/dev/ffs2/dab/INRIX.tpg"
#define TPEG_MSG_UPDATE_TIMER 1000
#define TPEG_TRANSPORT_FRAME_HEADER_SIZE 7

    struct dabdrv_msgTpegMsgUpdate:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_MSG, 
                                "  dabdrv_msgTpegMsgUpdate"));
        };

    };

    struct dabdrv_msgTpegUriActivatedTimer:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL


        virtual tVoid vTrace() const {
            ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_MSG, 
                                "  dabdrv_msgTpegUriActivatedTimer"));
        };

    };

    struct dabdrv_msgTpegSupervisionTimer:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL


        virtual tVoid vTrace() const {
            ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_MSG, 
                                "  dabdrv_msgTpegSupervisionTimer"));
        };

    };

}

using namespace DAB;
dabdrv_tpeg::dabdrv_tpeg() {

    _bTpegActive = FALSE;
    bUseSimulationData = FALSE;
    bSendTpegData = FALSE;
    _bTpegDataReceived = FALSE;
    _sSelectUri="";

    vSubscribe<trMsgDrvStartComponent>();
    vSubscribe<trMsgDrvStopComponent>();

    vSubscribe<trMsgSrvCmdSelectTpegUri>();
    vSubscribe<trMsgSrvCmdGetTpegUriList>();
    vSubscribe<trMeca_RDbQuery>();
    vSubscribe<trMeca_RDdmSelectUri>();
    vSubscribe<trMeca_RDdmDataChannel>();
    vSubscribe<trMsgDrvSetTpegMode>();
    vSubscribe<dabdrv_msgTpegMsgUpdate>();
    vSubscribe<trMsgDrvLoadTpegSimFile>();
}

tVoid dabdrv_tpeg::vInit() {
    ETG_TRACE_USR1(("dabdrv_tpeg::vInit"));
     _oTpegMsgUpdateTmr.vInit(instance(),dabdrv_msgTpegMsgUpdate());
     _oTpegUriActivatedTimer.vInit(instance(),dabdrv_msgTpegUriActivatedTimer());
     _oTpegSupervisionTimer.vInit(instance(),dabdrv_msgTpegSupervisionTimer());
}

tVoid dabdrv_tpeg::vDeInit() {
    ETG_TRACE_USR1(("dabdrv_tpeg::vDeInit"));
    _oTpegMsgUpdateTmr.vDeInit();
    _oTpegUriActivatedTimer.vDeInit();
    _oTpegSupervisionTimer.vDeInit();
}

tVoid dabdrv_tpeg::vTraceState() const {

}

tVoid dabdrv_tpeg::vProcess(trMsgDrvStartComponent *poStartComponent) {
    (tVoid)poStartComponent;
#ifdef VARIANT_S_FTR_ENABLE_TPEG
    // prepare DB_QUERY
    trMeca_CDbQuery rCDbQuery;
    rCDbQuery.enDbQueryCmd = enMeca_DbQueryCmd_PREPARE;
    rCDbQuery.u8QueryTag = (tU8)enMeca_QUERY_TAG_GET_TPEG_CHN_LIST;
    rCDbQuery.u8QueryStateId = (tU8)enMeca_DB_QUERY_STMT_ID_RECEIVEABLE_TPEG_CHN_LIST;
    //const tChar line[] = "SELECT uri FROM v1f0_dab_user_app_information where user_app_type=4";
    //const tChar line[] = "SELECT dab_uri(C.o,U.user_app_x_pad_data) FROM dab_service_component AS C, dab_user_app_information as U, v1f0_dab_ensemble as E ON U.e=E.o AND C.o=U.c WHERE user_app_type=4 AND (E.quality_word&0xC0)=0";
    const tChar line[] = "SELECT dab_uri(C.o,U.user_app_x_pad_data) FROM dab_service_component AS C, dab_user_app_information as U, v1f0_dab_ensemble as E,  dab_subchannel AS X  ON U.e=E.o AND C.o=U.c WHERE U.e=X.e AND C.subchid=X.id AND X.datarate<=64 AND U.user_app_type=4 AND (E.quality_word&0xC0)=0 AND C.tmid=3 GROUP BY C.e,C.id";

    tU32 u32Len=(tU32)(OSAL_u32StringLength(line));
    rCDbQuery.lu8SqlStatement.reserve(u32Len+2);
    for (tU32 i=0; i < u32Len; i++) {
        rCDbQuery.lu8SqlStatement.push_back(line[i]);
    }

    dabdrv_mecaIf::instance()->vSendMecaCommand(rCDbQuery);

    trMsgSrvCmdGetTpegUriList rMsg;
    DAB_vCallMsg(rMsg);
#endif
}

tVoid dabdrv_tpeg::vProcess(trMsgDrvStopComponent *poStopComponent) {
    (tVoid)poStopComponent;
    _oTpegUriActivatedTimer.vStop();
    _oTpegSupervisionTimer.vStop();
}

tVoid dabdrv_tpeg::vProcess(dabdrv_msgTpegMsgUpdate *poTpegMsgUpdate) {
    (tVoid)poTpegMsgUpdate;

    ETG_TRACE_USR1(("dabdrv_tpeg::vProcess(dabdrv_msgTpegMsgUpdate)"));

    if (bUseSimulationData) {
        // get next messages from INRIX file
        static tU16 u16StaticCounter = 0;
        std::vector<DAB_trTpegDataListElement>::iterator iter = m_oTpegDataList.begin();
        for (tU8 i = 0; i< u16StaticCounter; i++) {
            iter++;
        }
        u16StaticCounter++;
        if ( u16StaticCounter >= m_oTpegDataList.size() ) { 
            u16StaticCounter = 0;
        }

        DAB_trTpegDataListElement vecTpegData;
        vecTpegData.vecTpegDataListElement = iter->vecTpegDataListElement;
        
        tU16 u16TpegFrameSize = (tU16)(7 + (DABDRV_GET_U16(iter->vecTpegDataListElement)));
        ETG_TRACE_USR1(("dabdrv_tpeg::vProcess: Frame Size: %d", u16TpegFrameSize ));

        oTpegDataList.vecTpegDataListElements.push_back(vecTpegData);
        dabdrv_properties::instance()->oTpegDataListProperty.vSet(oTpegDataList);
        oTpegDataList.vecTpegDataListElements.clear();
        _oTpegMsgUpdateTmr.vStart(TPEG_MSG_UPDATE_TIMER);
    }
}

tVoid dabdrv_tpeg::vProcess(dabdrv_msgTpegUriActivatedTimer *poTimeOutMsg) {
    (tVoid)poTimeOutMsg;

    if (( TRUE ==_bTpegDataReceived )||(_sSelectUri.empty())){
        return;
    }

    // no tpeg frames received within 30 sec -> set uri-service to state DAB_enUriSelectState_NO_RESOURCES
    // update SelectState in internal Uri-list
    std::vector<DAB_trTpegUriListElement>::iterator itComp;
    for( itComp = m_oTpegUriList.begin() ; itComp < m_oTpegUriList.end() ; itComp ++ ) {

        ETG_TRACE_USR1(("dabdrv_tpeg::vProcess(dabdrv_msgTpegUriActivatedTimer) Comp uri %s", _sSelectUri.c_str()));

        if(/*equal*/0 == OSAL_s32MemoryCompare( itComp->sUri.c_str(), 
                                                _sSelectUri.c_str(),
                                                OSAL_u32StringLength(_sSelectUri.c_str()))) {

            if ( itComp->enUriSelectState != DAB_enUriSelectState_NO_RESOURCES ) {
                itComp->enUriSelectState = DAB_enUriSelectState_NO_RESOURCES;
                // update uri-list
                trMsgSrvRspGetTpegUriList rRsp(m_oTpegUriList);
                DAB_vCallMsg(rRsp);
            }
        }
    }
}

tVoid dabdrv_tpeg::vProcess(dabdrv_msgTpegSupervisionTimer *poTimeOutMsg) {
    (tVoid)poTimeOutMsg;

    // no response received after uri service selection within 5 min -> set uri-service to state DAB_enUriSelectState_NO_RESOURCES
    // update SelectState in internal Uri-list
    std::vector<DAB_trTpegUriListElement>::iterator itComp;
    for( itComp = m_oTpegUriList.begin() ; itComp < m_oTpegUriList.end() ; itComp ++ ) {


        if(/*equal*/0 == OSAL_s32MemoryCompare( itComp->sUri.c_str(), 
                                                _sSelectUri.c_str(),
                                                OSAL_u32StringLength(_sSelectUri.c_str()))) {
        ETG_TRACE_USR1(("dabdrv_tpeg::vProcess(dabdrv_msgTpegSupervisionTimer) Comp uri %s",  _sSelectUri.c_str()));

            if ( itComp->enUriSelectState != DAB_enUriSelectState_NO_RESOURCES ) {
                itComp->enUriSelectState = DAB_enUriSelectState_NO_RESOURCES;
                // update uri-list
                trMsgSrvRspGetTpegUriList rRsp(m_oTpegUriList);
                DAB_vCallMsg(rRsp);
                break;
            }
        }
    }
}

tVoid dabdrv_tpeg::vSetTpegActivity(tBool bTpegActive ) {
    _bTpegActive = bTpegActive;
     if (_bTpegActive) {
        _oTpegMsgUpdateTmr.vStart(10*TPEG_MSG_UPDATE_TIMER);
    }
    else {
        _oTpegMsgUpdateTmr.vStop();
    }
}

// enable or disable tpeg-mode
tVoid dabdrv_tpeg::vProcess(trMsgDrvSetTpegMode* poMsgDrvSetTpegMode) {
    
    ETG_TRACE_USR1(("dabdrv_tpeg::vProcess(trMsgDrvSetTpegMode) TPEG activ = %d", poMsgDrvSetTpegMode->bEnable));
    vSetTpegActivity(poMsgDrvSetTpegMode->bEnable);
}

tVoid dabdrv_tpeg::vProcess(trMsgDrvLoadTpegSimFile* /*poMsgDrvLoadTpegSimFile*/) {
    
    std::string sTpegFilename = TPEG_FILE_LOCATION;
    ETG_TRACE_USR4(("dabdrv_tpeg::vProcess(trMsgDrvLoadTpegSimFile) loading tpegfilename: %s", sTpegFilename.c_str()));

    bUseSimulationData = TRUE;
    m_oTpegDataList  = ofilereader.oLoadFile(sTpegFilename);

    _oTpegMsgUpdateTmr.vStart(TPEG_MSG_UPDATE_TIMER);
}

tVoid dabdrv_tpeg::vProcess(trMsgSrvCmdSelectTpegUri* poSelectTpegUri) {

    ETG_TRACE_USR4(("dabdrv_tpeg::vProcess(trMsgSrvCmdSelectTpegUri) select uri %s", poSelectTpegUri->sSelectUri.c_str()));

    // select TPEG URI
    trMeca_CDdmSelectUri rCDdmSelectUri;
	rCDdmSelectUri.u8TOn=4;
    rCDdmSelectUri.u8TPause=0;
    rCDdmSelectUri.u8Priority=0;
    rCDdmSelectUri.u8TDecodeLevel=2;
    rCDdmSelectUri.u16Rfu=0;
    rCDdmSelectUri.u16Rfu2=0;
    rCDdmSelectUri.u16Rfu=0;
	
    tU32 u32Len = 0;
    
    _sSelectUri.clear();
    _sSelectUri = poSelectTpegUri->sSelectUri;

    // check for empty string and send STOP_ALL
    if (poSelectTpegUri->sSelectUri.empty()) {
        rCDdmSelectUri.enCommand = enMeca_DdmCommand_STOP_ALL;
        ETG_TRACE_USR1(("dabdrv_tpeg::vProcess(trMsgSrvCmdGetTpegUriList) empty string !!!!!"));
    }
    else {
        rCDdmSelectUri.enCommand = poSelectTpegUri->bSelected ? enMeca_DdmCommand_START:enMeca_DdmCommand_STOP;
        u32Len = (tU32)(poSelectTpegUri->sSelectUri.size());
        _oTpegSupervisionTimer.vStart(DAB_TPEG_SUPERVISION_TIMER_MS);
        ETG_TRACE_USR1(("dabdrv_tpeg::vProcess(trMsgSrvCmdGetTpegUriList) Supervisiontimer started !!!!!"));
    }


    // stop timer to control if messages will be received by selected uri service
    if ( ( rCDdmSelectUri.enCommand == enMeca_DdmCommand_STOP) ||
         ( rCDdmSelectUri.enCommand == enMeca_DdmCommand_STOP_ALL) ) {
        _oTpegUriActivatedTimer.vStop();
        _oTpegSupervisionTimer.vStop();
        _bTpegDataReceived = FALSE;
    }

    for (tU32 i=0; i <= u32Len; i++) {
        rCDdmSelectUri.cBuffer.push_back(poSelectTpegUri->sSelectUri[i]);
    }
    rCDdmSelectUri.u16URILength = (tU16)u32Len;
    dabdrv_mecaIf::instance()->vSendMecaCommand(rCDdmSelectUri);

    // update bActive status of uri service
    std::vector<DAB_trTpegUriListElement>::iterator itComp;
    for( itComp = m_oTpegUriList.begin() ; itComp < m_oTpegUriList.end() ; itComp ++ ) {

        if (rCDdmSelectUri.enCommand == enMeca_DdmCommand_STOP_ALL) {
            itComp->bActive = FALSE;
        }
        else {
            if(/*equal*/0 == OSAL_s32MemoryCompare( itComp->sUri.c_str(), 
                                                    poSelectTpegUri->sSelectUri.c_str(),
                                                    OSAL_u32StringLength(poSelectTpegUri->sSelectUri.c_str()))) {
                itComp->bActive = poSelectTpegUri->bSelected;
            }
        }
    }
}


tVoid dabdrv_tpeg::vProcess(trMsgSrvCmdGetTpegUriList* /*poGetTpegUriList*/) {

    ETG_TRACE_USR4(("dabdrv_tpeg::vProcess(trMsgSrvCmdGetTpegUriList)"));
#ifdef VARIANT_S_FTR_ENABLE_TPEG
	// send URI-list
	trMeca_CDbQueryTrigger rCDbQueryTrigger;
	//rCDbQueryTrigger.enMeca_DbQueryTriggerCmd = enMeca_DbQueryTriggerCmdRsp_ADD;
	rCDbQueryTrigger.u8QueryTag = (tU8)enMeca_QUERY_TAG_GET_TPEG_CHN_LIST;
	rCDbQueryTrigger.u8Id = (tU8)enMeca_DB_QUERY_STMT_ID_RECEIVEABLE_TPEG_CHN_LIST;
    rCDbQueryTrigger.u8QueryStateId= (tU8)enMeca_DB_QUERY_STMT_ID_RECEIVEABLE_TPEG_CHN_LIST;
	rCDbQueryTrigger.enMeca_DbQueryAction = enMeca_DbQueryTriggerAction_EXEC_DB_QUERY_AND_SEND_RESULT;

	tU16 u16QueryTriggerList[THREE] = {(tU16)enMeca_DbQueryTriggerList_TR__ADDED,
								  (tU16)enMeca_DbQueryTriggerList_TR__SERVICE_DATA_COMPLETE,
								  (tU16)enMeca_DbQueryTriggerList_TR__DAB_ENSEMBLE__QUALITY_WORD};

	OSAL_pvMemoryCopy(rCDbQueryTrigger.au8TriggerList, u16QueryTriggerList, sizeof(u16QueryTriggerList));
	rCDbQueryTrigger.u16TriggerListSize = 3;

    /*rCDbQueryTrigger.lu16TriggerList.push_back((tU16)enMeca_DbQueryTriggerEvent_ADDED);
    rCDbQueryTrigger.lu16TriggerList.push_back((tU16)enMeca_DbQueryTriggerEvent_DATA_COMPLETE);
    rCDbQueryTrigger.lu16TriggerList.push_back((tU16)enMeca_DbQueryTriggerEvent_ENS_QUALITY_WORD);*/
    dabdrv_mecaIf::instance()->vSendMecaCommand(rCDbQueryTrigger);
#endif
}

tVoid dabdrv_tpeg::vProcess(trMeca_RDbQuery* poQuery) {
    
    ETG_TRACE_USR1(("dabdrv_tpeg::vProcess(trMeca_RDbQuery)"));

    tU32 u32Columns = 0;
    tU32 u32Rows    = 0;
    tU32 row_index  = 0;

    if ( poQuery->u8QueryTag == (tU8)enMeca_QUERY_TAG_GET_TPEG_CHN_LIST ) 
    {
        ETG_TRACE_USR1(("dabdrv_tpeg::vProcess QueryTag = enMeca_QUERY_TAG_GET_TPEG_CHN_LIST"));

        tU32 l_result_length = (tU32)(poQuery->lu8UserData.size());
        if ( l_result_length == 0 )
            return;

        tU8* r_result = OSAL_NEW tU8[l_result_length]; 
        if (r_result == NULL)
            return;

        for(tU32 k=0; k<l_result_length; k++)
            r_result[k] = poQuery->lu8UserData[k];

        // get count of columns
        u32Columns = sdxf_table_get_num_columns(r_result, l_result_length);
        // get count of rows
        u32Rows = sdxf_table_get_num_rows(r_result, l_result_length);
        
        ETG_TRACE_USR1(("dabdrv_tpeg::vProcess(trMeca_RDbQuery) Columns: %d  Rows: %d",u32Columns, u32Rows));

        DAB_trTmcAvailListProperty rProperty= dabdrv_properties::instance()->oTmcAvailListProperty.oGet();

        uint8* row_chunk = NULL;
        row_chunk = sdxf_table_get_row(r_result, l_result_length, row_index++);

        ETG_TRACE_USR1(("dabdrv_tpeg::vProcess(trMeca_RDbQuery) No    |  URI"));

        vector<string> vcTpegUriList;
        
        while( row_chunk != NULL ){
        	tU8* chunk = sdxf_row_get_chunk(r_result, l_result_length+1, row_chunk, 1);

        	if (chunk == NULL)
        		break;

        	tChar* tpeg_string = (tChar*) sdxf_get_chunk_body_pointer(chunk);
            tU32 u32length = sdxf_chunk_get_chunk_body_length(chunk);
            tpeg_string[u32length]=0;

            std::string sTpegUriString(tpeg_string);
            vcTpegUriList.push_back(sTpegUriString);

            ETG_TRACE_USR1(("dabdrv_tpeg::vProcess(trMeca_RDbQuery)  %d     |  %s", row_index, sTpegUriString.c_str()));

        	row_chunk = sdxf_table_get_row(r_result, l_result_length, row_index++);
        }

        // update TPEG Uri list
        bCompareTpegReceivedUriList(vcTpegUriList);
        bSendTpegData=bIsSelectStateActivated();
        
        trMsgSrvRspGetTpegUriList rRsp(m_oTpegUriList);
        DAB_vCallMsg(rRsp);
    }
}

tVoid dabdrv_tpeg::vProcess(trMeca_RDdmSelectUri* poDdmSelectUri) {
    
    ETG_TRACE_USR1(("dabdrv_tpeg::vProcess(trMeca_RDdmSelectUri) enResponse=%x",
                    ETG_CENUM(tenMeca_DdmResponse, poDdmSelectUri->enResponse)));

    switch(poDdmSelectUri->enResponse){
        case enMeca_DdmResponse_OK:
        case enMeca_DdmResponse_URI_PROCESSING_IN_PROGRESS:
            return;
        case enMeca_DdmResponse_URI_ACTIVATED:
            // uri service is activated -> start timer for supervising if tpeg frames will be received
            _oTpegSupervisionTimer.vStop();
            _oTpegUriActivatedTimer.vStart(DAB_TPEG_URI_ACTIVATED_TIMER_MS);
            _bTpegDataReceived = FALSE;
            break;
        case enMeca_DdmResponse_URI_ACTIVATE_FAIL_SCHED:
        case enMeca_DdmResponse_URI_ACTIVATE_FAIL_RECEPT:
        case enMeca_DdmResponse_URI_DEACTIVATED_BY_SCHED:
        case enMeca_DdmResponse_URI_DEACTIVATED_BY_SYNC_LOSS:
            _oTpegUriActivatedTimer.vStop();
            _oTpegSupervisionTimer.vStop();
            _bTpegDataReceived = FALSE;
            break;
        default:
            {}
    }

    // update SelectState in internal Uri-list
    std::vector<DAB_trTpegUriListElement>::iterator itComp;
    for( itComp = m_oTpegUriList.begin() ; itComp < m_oTpegUriList.end() ; itComp ++ ) {

        ETG_TRACE_USR1(("dabdrv_tpeg::vProcess(trMeca_RDdmSelectUri) Comp   uri %s", itComp->sUri.c_str()));

        if(/*equal*/0 == OSAL_s32MemoryCompare( itComp->sUri.c_str(), 
                                                _sSelectUri.c_str(),
                                                OSAL_u32StringLength(_sSelectUri.c_str()))) {

            if ( itComp->enUriSelectState != (DAB_tenUriSelectState)poDdmSelectUri->enResponse ){
                itComp->enUriSelectState = (DAB_tenUriSelectState)poDdmSelectUri->enResponse;
                // update uri-list
                trMsgSrvRspGetTpegUriList rRsp(m_oTpegUriList);
                DAB_vCallMsg(rRsp);
            }
        }
    }

    bSendTpegData=bIsSelectStateActivated();
    /*if ( poDdmSelectUri->enResponse != enMeca_DdmResponse_OK ) {

        ETG_TRACE_USR1(("dabdrv_tpeg::vProcess(trMeca_RDdmSelectUri) UriLen=%d",
                            poDdmSelectUri->u16UriLen));

        if ( poDdmSelectUri->u16UriLen != 0 ) {

            trMsgSrvCmdSelectTpegUri rMsg;
            rMsg.bSelected=FALSE;
            for (tU16 i=0; i < poDdmSelectUri->u16UriLen; i++) {
                rMsg.sSelectUri.push_back(poDdmSelectUri->lu8UriData[i]);
            }
            ETG_TRACE_USR4(("dabdrv_tpeg::vProcess(trMeca_RDdmSelectUri) sSelectUri=%s",
                            rMsg.sSelectUri.c_str()));
            DAB_vCallMsg(rMsg);
        }
    }*/
}


tVoid dabdrv_tpeg::vProcess(trMeca_RDdmDataChannel* poDdmDatachannel) {
    
    if (FALSE == bIsSelectStateActivated()) {
        ETG_TRACE_USR1(("bIsSelectStateActivated"));
        return;
    }

    ETG_TRACE_USR1(("  dabdrv_tpeg::vProcess(trMeca_RDdmDataChannel) enMeca_DcSourceApi=%x enMeca_DcSourceType=%x enMeca_DcDataType=%x extLength=%d DataLength=0x%x",
                            ETG_CENUM(tenMeca_DcSourceApi, poDdmDatachannel->enMeca_DcSourceApi),
                            ETG_CENUM(tenMeca_DcSourceType, poDdmDatachannel->enMeca_DcSourceType),
                            ETG_CENUM(tenMeca_DcDataType, poDdmDatachannel->enMeca_DcDataType),
                            poDdmDatachannel->u32ExtLength,
                            poDdmDatachannel->u32DataLength));

    if ( poDdmDatachannel->u32DataLength != 0 ) {

        switch (poDdmDatachannel->enMeca_DcDataType) {
        case enMeca_DcDataType_TDC:
        {
            if ( poDdmDatachannel->lu8ExtData.size() >=2 ) {
                //tU16 u16Scidi = DABDRV_GET_U16(&poDdmDatachannel->lu8ExtData[0]);
                ETG_TRACE_USR4(("dabdrv_tpeg::vProcess(trMeca_RDdmDataChannel) u32DataLength=%x",
                                        poDdmDatachannel->u32DataLength));

                if ( TRUE == _bTpegActive ) {
                    DAB_trTpegDataListElement vecTpegData;
                    vecTpegData.vecTpegDataListElement = poDdmDatachannel->lu8Data;

                    //tU16 u16FrameStart = DABDRV_GET_U16(&poDdmDatachannel->lu8Data[0]);
                    //ETG_TRACE_USR1(("dabdrv_tpeg::vProcess: Tpeg-Frame Start: %x", u16FrameStart ));
                    tU16 u16TpegFrameSize = DABDRV_GET_U16(vecTpegData.vecTpegDataListElement);
                    ETG_TRACE_USR1(("dabdrv_tpeg::vProcess: Tpeg-Frame size: %d", u16TpegFrameSize ));
                    tU8 u8TpegFrameType = poDdmDatachannel->lu8Data[6];
                    ETG_TRACE_USR1(("dabdrv_tpeg::vProcess: Tpeg-Frame Type: %d", u8TpegFrameType ));

                    oTpegDataList.vecTpegDataListElements.push_back(vecTpegData);
                }
                _bTpegDataReceived = TRUE;
                vUpdateTpegReceiveTime();
                // check if vector is full
                //if ( (oTmcDataList.vectorTmcDataListElements.size() == 50) || (_enTmcState == enTmcState_SendTmcData) ) 
                {
                    dabdrv_properties::instance()->oTpegDataListProperty.vSet(oTpegDataList);

                    oTpegDataList.vecTpegDataListElements.clear();
                    //_enTpegState = enTpegState_CollectingTpegData;
                }
            }
            break;
        }
        case enMeca_DcDataType_Undefined:
        case enMeca_DcDataType_Packets:
        case enMeca_DcDataType_Data_Groups:
        case enMeca_DcDataType_Stream:
        case enMeca_DcDataType_PAD:
        case enMeca_DcDataType_PAD_Data_Groups:
        case enMeca_DcDataType_APP_TYPE_PAD:
        case enMeca_DcDataType_FIBs:
        default:
            ETG_TRACE_USR1(("  dabdrv_tpeg::vProcess(trMeca_RDdmDataChannel) !!!!  NOT HANDLED: %x ", 
                             ETG_CENUM(tenMeca_DcDataType, poDdmDatachannel->enMeca_DcDataType)));
        }
    }
}

tVoid dabdrv_tpeg::vUpdateTpegReceiveTime() {
    // update time for last received tpeg-message
    OSAL_trTimeDate rCurrentTime;
    tS32 sRet = OSAL_s32ClockGetTime( &rCurrentTime );

    ETG_TRACE_USR4(("dabdrv_tpeg::vUpdateTpegReceiveTime(): TIME: %2d:%2d:%2d  %2d.%2d.%2d %i",
                                                rCurrentTime.s32Hour,
                                                rCurrentTime.s32Minute,
                                                rCurrentTime.s32Second,
                                                rCurrentTime.s32Day,
                                                rCurrentTime.s32Month,
                                                rCurrentTime.s32Year,sRet));

    DAB_trTmcTpegInfoProperty rProperty= dabdrv_properties::instance()->oTmcTpegInfoProperty.oGet();
    rProperty.u32ReceivedTpegMsgs++;
    rProperty.rLastReceivedUriMsgTime.u32DabSec = rCurrentTime.s32Second;
    rProperty.rLastReceivedUriMsgTime.u32DabMin = rCurrentTime.s32Minute;
    rProperty.rLastReceivedUriMsgTime.u32DabHour = rCurrentTime.s32Hour;
    rProperty.rLastReceivedUriMsgTime.u32DabMday = rCurrentTime.s32Day;
    rProperty.rLastReceivedUriMsgTime.u32DabMon = rCurrentTime.s32Month;
    rProperty.rLastReceivedUriMsgTime.u32DabYear = rCurrentTime.s32Year+1900;
    rProperty.rLastReceivedUriMsgTime.u32DabWday = rCurrentTime.s32Weekday;
    rProperty.rLastReceivedUriMsgTime.u32DabYday = rCurrentTime.s32Yearday;
    rProperty.rLastReceivedUriMsgTime.u32DabUtcOff = rCurrentTime.s32Daylightsaving;
    dabdrv_properties::instance()->oTmcTpegInfoProperty.vSet(rProperty);
}

tBool dabdrv_tpeg::bCompareTpegReceivedUriList(vector<string> &vcTpegUriList) {

    tBool bUpdated = FALSE;
    tU32 u32Len = (tU32)(vcTpegUriList.size());
    if (u32Len==0)
        return FALSE;

    if (m_oTpegUriList.size() == 0) {
        // copy received uri-list to internal
        for (tU32 i=0; i < u32Len; i++) {
            DAB_trTpegUriListElement vUriElement;
            vUriElement.sUri =  vcTpegUriList[i];
            vUriElement.enUriSelectState = DAB_enUriSelectState_NOT_SELECTED;
            m_oTpegUriList.push_back(vUriElement);
        }
        return TRUE;
    }

    // check internal uri-list if old uri-services can be removed
    std::vector<DAB_trTpegUriListElement>::iterator it;
    for( it = m_oTpegUriList.begin() ; it < m_oTpegUriList.end() ; it ++ ) {

        tBool bEntry = FALSE;

        std::vector<std::string>::iterator itComp;
        for( itComp = vcTpegUriList.begin() ; itComp < vcTpegUriList.end() ; itComp ++ ) {

            if(/*equal*/0 == OSAL_s32MemoryCompare( it->sUri.c_str(), 
                                                    itComp->c_str(),
                                                    OSAL_u32StringLength(itComp->c_str()))) {
                bEntry = TRUE;
            }
        }
        if (!bEntry) {
            // remove from internal list
            ETG_TRACE_USR1(("dabdrv_tpeg::bCompareTpegReceivedUriList  remove item %s", it->sUri.c_str()));
            it = m_oTpegUriList.erase(it);
            bUpdated = TRUE;
        }
    }

    // now add new entries from received uri-list
    std::vector<std::string>::iterator itAdd;
    for( itAdd = vcTpegUriList.begin() ; itAdd < vcTpegUriList.end() ; itAdd ++ ) {

        tBool bAddEntry = TRUE;

        std::vector<DAB_trTpegUriListElement>::iterator itComp;
        for( itComp = m_oTpegUriList.begin() ; itComp < m_oTpegUriList.end() ; itComp ++ ) {
            
            //ETG_TRACE_USR1(("dabdrv_tpeg::bCompareTpegReceivedUriList  item-list %s", itComp->sUri.c_str()));
            //ETG_TRACE_USR1(("dabdrv_tpeg::bCompareTpegReceivedUriList  add -list %s", itAdd->c_str()));

            if(/*equal*/0 == OSAL_s32MemoryCompare( itComp->sUri.c_str(), 
                                                    itAdd->c_str(),
                                                    OSAL_u32StringLength(itAdd->c_str()))) {
                //ETG_TRACE_USR1(("dabdrv_tpeg::bCompareTpegReceivedUriList  false"));
                bAddEntry = FALSE;
            }
        }
        if (bAddEntry) {
            ETG_TRACE_USR1(("dabdrv_tpeg::bCompareTpegReceivedUriList  add item %s", itAdd->c_str()));
            // add to internal list
            DAB_trTpegUriListElement vUriElement;
            vUriElement.sUri =  itAdd->c_str();
            vUriElement.enUriSelectState = DAB_enUriSelectState_NOT_SELECTED;
            m_oTpegUriList.push_back(vUriElement);
            bUpdated = TRUE;
        }
    }
    return bUpdated;
}


tBool dabdrv_tpeg::bIsSelectStateActivated() {

    tBool bRetVal = FALSE;

    std::vector<DAB_trTpegUriListElement>::iterator it;
    for( it = m_oTpegUriList.begin() ; it < m_oTpegUriList.end() ; it ++ ) {
        if(it->enUriSelectState == DAB_enUriSelectState_URI_ACTIVATED) { 
            ETG_TRACE_USR1(("bIsSelectStateActivated"));
            bRetVal = TRUE;
        }
    }
    return bRetVal;
}

/**************************************************************************//**
* Constructor
******************************************************************************/
dabdrv_filereader::dabdrv_filereader() : u32Position(0)
{  
}

/**************************************************************************//**
* Loads the input file to a vector containing all pattern information
******************************************************************************/
std::vector<DAB_trTpegDataListElement>  dabdrv_filereader::oLoadFile(std::string sFileLocation) 
{  
   std::vector<DAB_trTpegDataListElement> oPatterns;   
   
   OSAL_tIODescriptor oInputFile = OSAL_IOOpen(sFileLocation.c_str(), (OSAL_tenAccess) (OSAL_EN_READONLY|OSAL_EN_BINARY));
   if (oInputFile != OSAL_ERROR) 
   {
      oPatterns = vProcessFile(oInputFile);
      
      OSAL_s32IOClose(oInputFile); 
      oInputFile = 0;
   } 
   else 
   {
       ETG_TRACE_USR1(("oLoadFile: Problem opening file: %s",  sFileLocation.c_str()));
   }

   OSAL_s32IOClose(oInputFile);
   oInputFile = 0;

   return oPatterns;
}


/**************************************************************************//**
* Processes the entire buffer
******************************************************************************/
std::vector<DAB_trTpegDataListElement> dabdrv_filereader::vProcessFile(OSAL_tIODescriptor oInputFile)
{
   std::vector<DAB_trTpegDataListElement> oTpegFrame;

   tU32 lSize = u32GetFileSize(oInputFile);

   ETG_TRACE_USR1(("dabdrv_filereader::vProcessFile size: %d",  lSize));
   
   tString cBuffer = vReadFileIntoBuffer(lSize, oInputFile);
   
   u32Position = 0;
   while (u32Position < lSize)
   {
      //ETG_TRACE_USR1(("dabdrv_filereader::vProcessFile Position: %d", u32Position ));
      oTpegFrame.push_back(oGetTpegFrameFromBuffer(cBuffer)); //lint !e429 oPattern is returned
   } //lint !e429 oPattern is returned

   OSAL_s32IOClose(oInputFile);
   oInputFile = 0;
   OSAL_vMemoryFree(cBuffer);

   return oTpegFrame;
}

/**************************************************************************//**
* Reads the entire file to a buffer
******************************************************************************/
tString dabdrv_filereader::vReadFileIntoBuffer(tU32 lSize, OSAL_tIODescriptor oInputFile)
{
   tString cBuffer = (tString) OSAL_pvMemoryAllocate(lSize);
   (tVoid)OSAL_s32IORead(oInputFile, (tPS8)cBuffer, lSize);

   return cBuffer;
} //lint !e1762

/**************************************************************************//**
* Gets the entire file size
******************************************************************************/
tU32 dabdrv_filereader::u32GetFileSize(OSAL_tIODescriptor oInputFile) {
   tU32 lSize = 0;
   tS32 s32return = OSALUTIL_s32FGetSize(oInputFile);
   if(s32return > 0)
   {
	   lSize = s32return;
   }

   return lSize;
} //lint !e1762

DAB_trTpegDataListElement dabdrv_filereader::oGetTpegFrameFromBuffer(tString cBuffer) {
   DAB_trTpegDataListElement oTpegFrameBuffer;
   if (cBuffer != NULL ) {
      //tU16 u16FrameStart = DABDRV_GET_U16(&cBuffer[u32Position]);
      //ETG_TRACE_USR1(("oGetTpegFrameFromBuffer: Tpeg-Frame Start: %x", u16FrameStart ));
      tU16 u16TpegFrameSize = (tU16)(TPEG_TRANSPORT_FRAME_HEADER_SIZE + DABDRV_GET_U16(&cBuffer[u32Position+2]));
      //ETG_TRACE_USR1(("oGetTpegFrameFromBuffer: Tpeg-Frame size: %d", u16TpegFrameSize ));

      for (tU32 i = 0; i < u16TpegFrameSize; i++) {
          oTpegFrameBuffer.vecTpegDataListElement.push_back(cBuffer[u32Position]);
          u32Position++;
      }
   }

   return oTpegFrameBuffer;
}
