/************************************************************************
 * FILE:   dabdrv_properties.hpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  header of dabdrv_properties
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      		 | Author                       | Modification
   
				
 *************************************************************************/

#ifndef DABDRV_PROPERTIES_HPP
#define DABDRV_PROPERTIES_HPP

#include "fc_dabtuner_util.h"
#include "dabdrv_meca.h"
#include "dabmeca_ans.hpp"
#include "dabmeca_ddm.hpp"
#include "dabmeca_tsu.hpp"
#include "dabdrv_main.hpp"
#include "fc_dabtuner_property.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_UTIL_MSG
#include "trcGenProj/Header/dabdrv_properties.hpp.trc.h"
#endif

namespace DAB {

// todo: add traces
struct DAB_trADRMsgProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL
	tU16 u16FktId;
   vector<tU8>vectorADRMsg;
    DAB_trADRMsgProperty():
   u16FktId(0),
       vectorADRMsg()
    {}
    bool operator!=(DAB_trADRMsgProperty const& b) const
    {
		DAB_RET_1_IF_NEQ(u16FktId);
        DAB_RET_1_IF_NEQ(vectorADRMsg);
        return FALSE;
    }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  DAB_trADRMsgProperty: ADRMsg Fkt id %x ssize %d",
                u16FktId,            
                            (tU16)(vectorADRMsg.size())));
        };
};

// todo: add traces
struct DAB_trLocalOffsetProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL
	tS64 s64Offset;
  
  DAB_trLocalOffsetProperty():
   s64Offset(0)       
    {}
    bool operator!=(DAB_trLocalOffsetProperty const& b) const
    {
		DAB_RET_1_IF_NEQ(s64Offset);       
        return FALSE;
    }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  DAB_trLocalOffsetProperty: Offset %d",
                s64Offset));
        };
};
struct DAB_trServiceInfoProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL

    DAB::trMecaProgrammeService rProgService;
    tBool bLabelPresent;
    DAB::trMecaLabel rLabel;
    tU8 u8NumberOfComponents;
    tU8 u8NumberOfAudioComponents;
    tU16 u16AvailableAnnoTypesMask;
    DAB::DAB_tenSampleRate enSampleRate;
    tU16 u16BitRateKbps;
    DAB::tenMeca_RdmAudioMode enAudioMode;
    DAB::tenMeca_RdmMode enAudioCodec;
    tU8 u8ServiceInfoFlags;
    tU8 u8PresetNumber;
	// Added Service index and PTY Code -- Chait
    tU16 u16ServiceIndex; 
    tU8 u8PtyCode;
    tBool bSBRFlag;
    tBool bPSFlag;
	tenMeca_RdmCoreAudioCoding enAudioCoding;

    DAB_trServiceInfoProperty():
        rProgService(),
        bLabelPresent(FALSE),
        rLabel(),
        u8NumberOfComponents(0),
        u8NumberOfAudioComponents(0),
        u16AvailableAnnoTypesMask(0),
        enSampleRate(DAB_enSampleRate_48KHZ),
        u16BitRateKbps(0),
        enAudioMode(DAB::enMeca_RdmAudioMode_STEREO), // todo: add mode "unknown"
        enAudioCodec(enMeca_RdmMode_DAB_FG_AUDIO),
        u8ServiceInfoFlags(0),
        u8PresetNumber(0),
		u16ServiceIndex(0),
		u8PtyCode(0),
		bSBRFlag(FALSE),
		bPSFlag(FALSE),
		enAudioCoding(enMeca_RdmCoreAudioCoding_MPEG1)
    {}
   
    bool operator!=(DAB_trServiceInfoProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(rProgService);
        DAB_RET_1_IF_NEQ(bLabelPresent);
        DAB_RET_1_IF_NEQ(rLabel);
        DAB_RET_1_IF_NEQ(u8NumberOfComponents);
        DAB_RET_1_IF_NEQ(u8NumberOfAudioComponents);
        DAB_RET_1_IF_NEQ(u16AvailableAnnoTypesMask);
        DAB_RET_1_IF_NEQ(enSampleRate);
        DAB_RET_1_IF_NEQ(u16BitRateKbps);
        DAB_RET_1_IF_NEQ(enAudioMode);
        DAB_RET_1_IF_NEQ(enAudioCodec);
        DAB_RET_1_IF_NEQ(u8ServiceInfoFlags);
        DAB_RET_1_IF_NEQ(u8PresetNumber);
		DAB_RET_1_IF_NEQ(u16ServiceIndex);
        DAB_RET_1_IF_NEQ(u8PtyCode);		
        DAB_RET_1_IF_NEQ(bSBRFlag);
        DAB_RET_1_IF_NEQ(bPSFlag);
		DAB_RET_1_IF_NEQ(enAudioCoding);

        return FALSE;
    }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  DAB_trServiceInfoProperty: rProgService.sid=0x%08x  bLabelPresent=%d  "
                            "  u8NumberOfComponents=%d  u8NumberOfAudioComponents=%d u16AvailableAnnoTypesMask=%x  enSampleRate=%d  u16BitRateKbps=%d"
                            "  enAudioMode=%d  enAudioCodec=%d  u8ServiceInfoFlags=%x ...",
                            rProgService.u32GetSID(),
                            bLabelPresent,
                            u8NumberOfComponents,
                            u8NumberOfAudioComponents,
                            u16AvailableAnnoTypesMask,
                            enSampleRate,
                            u16BitRateKbps,
                            enAudioMode,
                            enAudioCodec,
                            u8ServiceInfoFlags));

            ETG_TRACE_USR4(("  DAB_trServiceInfoProperty(cont): u8PresetNumber=%d enAudioCoding=%d Label=%17s",
                            u8PresetNumber, 
							enAudioCoding,
                            rLabel.pcGetCString()));
			
			ETG_TRACE_USR4(("  DAB_trServiceInfoProperty(cont): u16ServiceIndex=%d u8PtyCode=%d bSBRFlag=%d bPSFlag=%d", u16ServiceIndex, u8PtyCode, bSBRFlag, bPSFlag));

        };

};

struct  DAB_trChnStateProperty: public DAB_Message
{
    DAB_DISPATCH_IMPL
    
    DAB_trChnStateProperty(tenChnState enChnState_=enChnState_Unavail, tBool bMuted_=TRUE):
        enChnState(enChnState_),
        bMuted(bMuted_)
    {}

    bool operator!=(DAB_trChnStateProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(enChnState);
        DAB_RET_1_IF_NEQ(bMuted);
        
        return FALSE;
    };

    virtual tVoid vTrace() const {
        ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_MSG, 
                            "   DAB_trChnStateProperty: enChnState=%d bMuted=%d",
                            ETG_CENUM(tenChnState, enChnState),
                            bMuted));
    };
    DAB::tenChnState enChnState;
    tBool bMuted;
};

struct DAB_trChnInfoProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL

    DAB::trMecaId rMecaId;
    DAB::tenChnState enChnState;
    tBool bMuted;
    tBool bLabelPresent;
    DAB::trMecaLabel rLabel;
    tU8 u8NumberOfAudioComponents;
    tU16 u16AvailableAnnoTypesMask;
    tU8 u8PresetNumber;
    tU16 u16FrozenListId;
    tBool bComponentMode;
    tBool bFmServiceFollowActive;
    DAB::trRadioText rRadioText;
    vector<DAB::trRTPlusInfo>vectorRadioTextPlus;
	tU8 u8PTY;
	tBool bAFPresent;
	string sLogoLink;
	tU8 u8Scids;

    DAB_trChnInfoProperty():
        rMecaId(),
        enChnState(DAB::enChnState_Unavail),
        bMuted(FALSE),
        bLabelPresent(FALSE),
        rLabel(),
        u8NumberOfAudioComponents(0),
        u16AvailableAnnoTypesMask(0),
        u8PresetNumber(0),
        u16FrozenListId(0),
        bComponentMode(FALSE),
        bFmServiceFollowActive(FALSE),
        rRadioText(),
        vectorRadioTextPlus(),
		u8PTY(0),
		bAFPresent(FALSE),
		sLogoLink("/0"),
		u8Scids(0)
    {}
   
    bool operator!=(DAB_trChnInfoProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(rMecaId);
        DAB_RET_1_IF_NEQ(enChnState);
        DAB_RET_1_IF_NEQ(bMuted);
        DAB_RET_1_IF_NEQ(bLabelPresent);
        DAB_RET_1_IF_NEQ(rLabel);
        DAB_RET_1_IF_NEQ(u8NumberOfAudioComponents);
        DAB_RET_1_IF_NEQ(u16AvailableAnnoTypesMask);
        DAB_RET_1_IF_NEQ(u8PresetNumber);
        DAB_RET_1_IF_NEQ(u16FrozenListId);
        DAB_RET_1_IF_NEQ(bComponentMode);
        DAB_RET_1_IF_NEQ(bFmServiceFollowActive);
        DAB_RET_1_IF_NEQ(u8Scids);
		if(OSAL_s32StringCompare(b.rRadioText.sRadioText,rRadioText.sRadioText) != 0) 
			return TRUE;
		DAB_RET_1_IF_NEQ(sLogoLink);
		//DAB_RET_1_IF_NEQ(rRadioText.sRadioText);
        DAB_RET_1_IF_NEQ(vectorRadioTextPlus);
		DAB_RET_1_IF_NEQ(u8PTY);
		DAB_RET_1_IF_NEQ(bAFPresent);

        return FALSE;
    }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  DAB_trChnInfoProperty:rMecaId=0x%x enChnState=%d u8NumberOfAudioComponents=%d "
                            "u16AvailableAnnoTypesMask=0x%x u8PresetNumber=%d u16FrozenListId=%d bComponentMode=%d bFmServiceFollowActive=%d ...",
                            rMecaId.u32GetSID(),
                            ETG_CENUM(tenChnState, enChnState),
                            u8NumberOfAudioComponents,
                            u16AvailableAnnoTypesMask,
                            u8PresetNumber,
                            u16FrozenListId,
                            bComponentMode,
                            bFmServiceFollowActive));
            ETG_TRACE_USR4(("  DAB_trChnInfoProperty (cont): bLabelPresent=%d u8PTY=%d bAFPresent=%d u8Scids=%d sLabel=%s radioText follows",
                            bLabelPresent,
							u8PTY,
							bAFPresent,
							u8Scids,
							rLabel.pcGetCString()));
                            
            rRadioText.vTrace();
        };

};


struct DAB_trTestModeDataProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL

    tU32 u32CurrentServId;
    tU32 u32BBE;
    tU32 u32MSCBER;
    tU32 u32FICBER;
	tU32 u32NETBER;
    tU8  u8RSFEC;
    tU8 u8DabMode;
    tU8 u8ProtectionLevel;
    tU8 u8ConcealmentLevel;
    DAB::DAB_tenServiceLinkingMode e8ServiceLinkingMode;
    tU16 u16AltFMPI;
    tU32 u32AltFMFrequency;
    tU8 u8AltFMQuality;
    tU8 u8Flags;
    tU32 u32FieldStrength;
    tU16 u16TotalNumberLinks;
    tU16 u16ActivatedLink;
    tBool bDataValid;
    tU8 u8ECC;       //Added by jab4kor
    vector<DAB::trLinkListElem>vecLinkList;
	tBool bSync;
	vector<DAB::trAltFreqInfoList>vecAltFreqInfo;
    //DAB::trLinkList rLinkList;

    DAB_trTestModeDataProperty() {
        bDataValid=FALSE;
        u32CurrentServId= 0;
        u32BBE = 0;
        u32MSCBER = 0;
        u32FICBER = 0;
		u32NETBER = 0;
        u8RSFEC = 0;
        u8DabMode = 0;
        u8ProtectionLevel =0;
        u8ECC = 0;  		//Added by jab4kor

        fc_dabtuner_config *poConfig=fc_dabtuner_config::instance();
        if (OSAL_NULL ==poConfig) {
            ETG_TRACE_FATAL(("DAB_trTestModeDataProperty config not available !!!"));
            u8ConcealmentLevel = DAB_CONFIG_DEFAULT_CONCEALMENT_LEVEL;
        }
        else {
            u8ConcealmentLevel = poConfig->u8GetConcealmentLevel();
        }
        e8ServiceLinkingMode = DAB::DAB_enServiceLinkingMode_DAB_FM;
        u16AltFMPI = 0;
        u32AltFMFrequency = 0;
        u8AltFMQuality = 0;
        u8Flags = 0;
        u32FieldStrength = 0;
        u16TotalNumberLinks = 0;
        u16ActivatedLink = 0;
		bSync =FALSE;
    }
    bool operator!=(DAB_trTestModeDataProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(u32CurrentServId);
        DAB_RET_1_IF_NEQ(u32BBE); // exp 0x0200
        DAB_RET_1_IF_NEQ(u32MSCBER); // exp 0x0204
        DAB_RET_1_IF_NEQ(u32FICBER); // exp 0x0207
		DAB_RET_1_IF_NEQ(u32NETBER); // exp 0x020A
        DAB_RET_1_IF_NEQ(u8RSFEC); // todo exp ???
        DAB_RET_1_IF_NEQ(u8DabMode); // exp 0x020F
        DAB_RET_1_IF_NEQ(u8ProtectionLevel); // db trMeca_RDbSubchannelGetInfo ???
        DAB_RET_1_IF_NEQ(u8ConcealmentLevel); // aud trMeca_RAudSetConcealmentLevel ???
        DAB_RET_1_IF_NEQ(e8ServiceLinkingMode); // internal usage
        DAB_RET_1_IF_NEQ(u16AltFMPI); // todo
        DAB_RET_1_IF_NEQ(u32AltFMFrequency); // todo
        DAB_RET_1_IF_NEQ(u8AltFMQuality); // todo
        DAB_RET_1_IF_NEQ(u8Flags); // todo
        DAB_RET_1_IF_NEQ(u32FieldStrength);// exp 0x0261
        DAB_RET_1_IF_NEQ(u16TotalNumberLinks);// exp 0x0261
        DAB_RET_1_IF_NEQ(u16ActivatedLink);// exp 0x0261
        DAB_RET_1_IF_NEQ(bDataValid);
        DAB_RET_1_IF_NEQ(vecLinkList);
		DAB_RET_1_IF_NEQ(u8ECC);//Added by jab4kor
		DAB_RET_1_IF_NEQ(bSync);//Added by jab4kor

        return FALSE;
    }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("DAB_trTestModeDataProperty: bDataValid=%d  u32CurrentServId=%x  u32MSCBER=%x u32FICBER=%x u32NETBER=%x"
                            "  u8RSFEC=%d  u8DabMode=%d  u8ProtectionLevel=%d  u8ConcealmentLevel=%d ...",
                            bDataValid,
                            u32CurrentServId, 
                            u32MSCBER,
                            u32FICBER,
							u32NETBER,
                            u8RSFEC,
                            u8DabMode,
                            u8ProtectionLevel,
                            u8ConcealmentLevel));
            ETG_TRACE_USR4(("DAB_trTestModeDataProperty(cont): e8ServiceLinkingMode=%d u16AltFMPI=%d  u32AltFMFrequency=%d  u8AltFMQuality=%d  "
                            "u8Flags=%x  u32FieldStrength=%d u16TotalNumberLinks=%d  u16ActivatedLink=%d bSync=%d",
                            ETG_CENUM(DAB_tenServiceLinkingMode, e8ServiceLinkingMode),
                            u16AltFMPI,
                            u32AltFMFrequency,
                            u8AltFMQuality,
                            u8Flags,
                            u32FieldStrength,
                            u16TotalNumberLinks,
                            u16ActivatedLink,
							bSync));
        };

};

struct DAB_trBgTestModeDataProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL

    tU32 u32BgFrequency;
    tU32 u32BgCurrentEnsID;
    tU32 u32BgFICBER;
    tU16 u16BgFlags;
    tChar sBgDabMode[22];

    DAB_trBgTestModeDataProperty():
    u32BgFrequency(0),
    u32BgCurrentEnsID(0),
    u32BgFICBER(0),
    u16BgFlags(0)
    {
        sBgDabMode[0]=0;
    }

    bool operator!=(DAB_trBgTestModeDataProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(u32BgFrequency);
        DAB_RET_1_IF_NEQ(u32BgCurrentEnsID);
        DAB_RET_1_IF_NEQ(u32BgFICBER);
        DAB_RET_1_IF_NEQ(u16BgFlags);
        return FALSE;
    }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("DAB_trBgTestModeDataProperty: u32BGFrequency=%d  u32CurrentEnsID=%x  u32FICBER=%d u16BgFlags=%x sBgDabMode=%s",
                            u32BgFrequency,
                            u32BgCurrentEnsID,
                            u32BgFICBER,
                            u16BgFlags,
                            sBgDabMode));
        };

};


struct DAB_trAnnoStatusProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL

    tU16 u16CfgAnnoTypesMask;
    DAB::tenAnnoSource enCfgAnnoSource;

    tU16 u16AvailableAnnoTypesMask;
    tU16 u16ActiveAnnoTypesMask;
    DAB::tenAnnoSource enActiveAnnoSource;
    DAB::tenMeca_AnsAnnoActiveStatus enActiveAnnoStatus;
    
    DAB::tenAnnoSelectState enAnnoSelectState;
    tU16 u16SelectedAnnoTypesMask;
    
    DAB::trMecaLabel rAnnoServiceLabel;

	vector<DAB::trAnnoElement>vectorAnnoList;
    DAB_trAnnoStatusProperty():
        u16CfgAnnoTypesMask(0),
        enCfgAnnoSource(DAB::enAnnoSource_NONE),
        u16AvailableAnnoTypesMask(0),
        u16ActiveAnnoTypesMask(0),
        enActiveAnnoSource(DAB::enAnnoSource_NONE),
        enActiveAnnoStatus(DAB::enMeca_AnsAnnoActiveStatus_End),
        enAnnoSelectState(DAB::enAnnoSelectState_IDLE),
        u16SelectedAnnoTypesMask(),
        rAnnoServiceLabel(),
		vectorAnnoList()
    {}
    bool operator!=(DAB_trAnnoStatusProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(u16CfgAnnoTypesMask);
        DAB_RET_1_IF_NEQ(enCfgAnnoSource);
        DAB_RET_1_IF_NEQ(u16AvailableAnnoTypesMask);
        DAB_RET_1_IF_NEQ(u16ActiveAnnoTypesMask);
        DAB_RET_1_IF_NEQ(enActiveAnnoStatus);
        DAB_RET_1_IF_NEQ(enActiveAnnoSource);
        DAB_RET_1_IF_NEQ(enAnnoSelectState);
        DAB_RET_1_IF_NEQ(u16SelectedAnnoTypesMask);
        DAB_RET_1_IF_NEQ(rAnnoServiceLabel);
		DAB_RET_1_IF_NEQ(vectorAnnoList);

        return FALSE;
    }


    virtual tVoid vTrace() const {
        ETG_TRACE_USR4(("  DAB_trAnnoStatusProperty: u16CfgAnnoTypesMask=0x%04x u16AvailableAnnoTypesMask=0x%04x "
                        "u16ActiveAnnoTypesMask=0x%04x enActiveAnnoStatus=%d enAnnoSelectState=%d u16SelectedAnnoTypesMask=0x%04x vectorSize=%d label=%s ",
                        u16CfgAnnoTypesMask,
                        u16AvailableAnnoTypesMask,
                        u16ActiveAnnoTypesMask,
                        ETG_CENUM(tenMeca_AnsAnnoActiveStatus, enActiveAnnoStatus),
                        ETG_CENUM(tenAnnoSelectState, enAnnoSelectState),
                        u16SelectedAnnoTypesMask,
						(tU16)(vectorAnnoList.size()),
                        rAnnoServiceLabel.pcGetCString()));
    };

};

// todo this mesage should correspond to a wire-message
// most of the data correspond to Meca DAB_C_RDM_GET_RDM_INFO for which a notification 
// can be set in  in ADR via DAB_C_RDM_SET_AUTO_NOTIFICATION,
// witch b_notify_class == RDM,
// and  b_notify_message_mask_rdm==RDM_GET_RDM_INFO 


struct DAB_trTunerStatusProperty:
    public DAB_Message
{
    DAB_DISPATCH_IMPL

    DAB::DAB_tenTunerActivityType enTunerActivityType; // vHandleResponseTune+vHandleResponseSeek+vHandleResponseLearn+vSetTunerStatusIdsFlags  todo move
    //   midw_ext_fi_tcl_DAB_Frequency FreqTuned;
    tU32 u32Frequency;
    tChar sFreqLabel[DABTUN_FREQLABEL_BUFFER];

    DAB::DAB_tenFrequencyTableType enFrequencyTable;
    //   midw_ext_fi_tcl_b16_DAB_TunerStatusFlags b16Flags;


    tU16 u16TunerStatusFlags; // DAB_enTunerStatusFlagBitPos
    tU8  u8SignalQuality;
    tU8  u8AudioQuality;
    tU32 u32CurrentEnsembleId;
    tU32 u32CurrentServiceId;
    tU16 u16CurrentSCIDI;

    DAB_trTunerStatusProperty():
        enTunerActivityType(DAB_enTunerActivityType_INVALID),
        u32Frequency(0),
        enFrequencyTable(DAB_enFrequencyTableType_EUROPE),
        u16TunerStatusFlags(0),
        u8SignalQuality(0),
        u8AudioQuality(0),
        u32CurrentEnsembleId(0),
        u32CurrentServiceId(0),
        u16CurrentSCIDI(0)
    {
        sFreqLabel[0]=0;
    }

    bool operator!=(DAB_trTunerStatusProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(enTunerActivityType);
        DAB_RET_1_IF_NEQ(u32Frequency);
        DAB_RET_1_IF_NEQ(enFrequencyTable);
        DAB_RET_1_IF_NEQ(u16TunerStatusFlags);
        DAB_RET_1_IF_NEQ(u8SignalQuality);
        DAB_RET_1_IF_NEQ(u8AudioQuality);
        DAB_RET_1_IF_NEQ(u32CurrentEnsembleId);
        DAB_RET_1_IF_NEQ(u32CurrentServiceId);
        DAB_RET_1_IF_NEQ(u16CurrentSCIDI);
        return OSAL_s32MemoryCompare(sFreqLabel, b.sFreqLabel, DABTUN_FREQLABEL_BUFFER) ? TRUE : FALSE;
    } 
};


struct DAB_trMuteRequestProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL
    
    tBool bMuteRequest;
    tU8 u8Attenuation;
    tBool bMute_Interrupt;
    tBool bTA;
    
    DAB_trMuteRequestProperty():
        bMuteRequest(FALSE),
        u8Attenuation(0),
        bMute_Interrupt(FALSE),
        bTA(FALSE)
    {}
   
    bool operator!=(DAB_trMuteRequestProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(bMuteRequest);
        DAB_RET_1_IF_NEQ(u8Attenuation);
        DAB_RET_1_IF_NEQ(bMute_Interrupt);
        DAB_RET_1_IF_NEQ(bTA);
        return FALSE;
    }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  DAB_trMuteRequestProperty:bMuteRequest=%d u8Attenuation=%d bMute_Interrupt=%d bTA=%d",
                            bMuteRequest, u8Attenuation, bMute_Interrupt, bTA));
        };

};

#define INDEX_ZERO 0
#define INDEX_THREE 3
#define INDEX_FIVE 5
#define INDEX_EIGHT 8
struct DAB_trSetupStatusProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL
    
    tU32  u32SetupFlags;
    tU8 u8ConcealmentLevel;
    DAB_tenServiceLinkingMode e8ServiceLinkingMode;
    tU8 u8DABFMLinkingThreshold;
    tU8 u8DABFMLinkBackDelay;
    tU8 u8DABFMLinkingBackThreshold;
    tU8 u8DABFMLinkingBackValidationTime;
    tU8 u8EnsInsertThreshold;
    tU8 u8EnsDeleteThreshold;
    tU8 u8BgLearnUpdateTime;
    tU8 u8DABScanThreshold;
    
    DAB_trSetupStatusProperty():
        u32SetupFlags(0),
        e8ServiceLinkingMode(DAB_enServiceLinkingMode_DAB_FM),
        u8BgLearnUpdateTime(0),
        u8DABScanThreshold(0)
    {
        fc_dabtuner_config *poConfig=fc_dabtuner_config::instance();
        if (OSAL_NULL ==poConfig) {
            ETG_TRACE_FATAL(("DAB_trSetupStatusProperty config not available !!!"));
            u8ConcealmentLevel = DAB_CONFIG_DEFAULT_CONCEALMENT_LEVEL;
            u8DABFMLinkingThreshold = DAB_CONFIG_DEFAULT_TSU_TIME_CONSTANTS_DAB_FM[INDEX_FIVE];
            u8DABFMLinkBackDelay = DAB_CONFIG_DEFAULT_TSU_TIME_CONSTANTS_FM_DAB[INDEX_ZERO];
            u8DABFMLinkingBackThreshold = DAB_CONFIG_DEFAULT_TSU_TIME_CONSTANTS_FM_DAB[INDEX_FIVE];
            u8DABFMLinkingBackValidationTime = DAB_CONFIG_DEFAULT_TSU_TIME_CONSTANTS_DAB_FM[INDEX_THREE];
            u8EnsInsertThreshold = DAB_CONFIG_DEFAULT_RDM_QUALITY_CONFIG[INDEX_EIGHT];
            u8EnsDeleteThreshold = DAB_CONFIG_DEFAULT_RDM_QUALITY_CONFIG[INDEX_EIGHT];
        }
        else {
            u8ConcealmentLevel = poConfig->u8GetConcealmentLevel();
            u8DABFMLinkingThreshold = poConfig->pcu8GetTsuConstantsDabFm()[INDEX_FIVE];
            u8DABFMLinkBackDelay = poConfig->pcu8GetTsuConstantsFmDab()[INDEX_ZERO];
            u8DABFMLinkingBackThreshold = poConfig->pcu8GetTsuConstantsFmDab()[INDEX_FIVE];
            u8DABFMLinkingBackValidationTime = poConfig->pcu8GetTsuConstantsDabFm()[INDEX_THREE];
            u8EnsInsertThreshold = poConfig->pcu8GetRdmQualityConfig()[INDEX_EIGHT];
            u8EnsDeleteThreshold = poConfig->pcu8GetRdmQualityConfig()[INDEX_EIGHT];
        }
    }
   
    bool operator!=(DAB_trSetupStatusProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(u32SetupFlags);
        DAB_RET_1_IF_NEQ(u8ConcealmentLevel);
        DAB_RET_1_IF_NEQ(e8ServiceLinkingMode);
        DAB_RET_1_IF_NEQ(u8DABFMLinkingThreshold);
        DAB_RET_1_IF_NEQ(u8DABFMLinkBackDelay);
        DAB_RET_1_IF_NEQ(u8DABFMLinkingBackThreshold);
        DAB_RET_1_IF_NEQ(u8DABFMLinkingBackValidationTime);
        DAB_RET_1_IF_NEQ(u8EnsInsertThreshold);
        DAB_RET_1_IF_NEQ(u8EnsDeleteThreshold);
        DAB_RET_1_IF_NEQ(u8BgLearnUpdateTime);
        DAB_RET_1_IF_NEQ(u8DABScanThreshold);
        return FALSE;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR4(("  DAB_trSetupStatusProperty:u8ConcealmentLevel=0x%x u8DABFMLinkingThreshold=%d u8DABFMLinkBackDelay=%d "
        					"u8DABFMLinkingBackThreshold=%d u8DABFMLinkingBackValidationTime=%d u8EnsInsertThreshold=%d u8EnsDeleteThreshold=%d "
        					"u8BgLearnUpdateTime=%d u8DABScanThreshold=%d",
							u8ConcealmentLevel,
							u8DABFMLinkingThreshold,
							u8DABFMLinkBackDelay,
							u8DABFMLinkingBackThreshold,
							u8DABFMLinkingBackValidationTime,
							u8EnsInsertThreshold,
							u8EnsDeleteThreshold,
							u8BgLearnUpdateTime,
							u8DABScanThreshold));

        ETG_TRACE_USR4(("  DAB_trSetupStatusProperty:e8ServiceLinkingMode=0x%x ",
        		e8ServiceLinkingMode));
    };
};

struct DAB_trAvailabilityProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL
    
    tBool bIsDABAvailable;
    DAB::DAB_tenSourceState enSourceState;
    
    
    DAB_trAvailabilityProperty():
        bIsDABAvailable(FALSE),
        enSourceState(DAB::DAB_enSourceState_BG)
    {}

    DAB_trAvailabilityProperty(tBool bIsDABAvailable_, DAB::DAB_tenSourceState enSourceState_):
        bIsDABAvailable(bIsDABAvailable_),
        enSourceState(enSourceState_)
    {}

   
    bool operator!=(DAB_trAvailabilityProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(bIsDABAvailable);
        DAB_RET_1_IF_NEQ(enSourceState);

        return FALSE;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR4(("  DAB_trAvailabilityProperty:bIsDABAvailable=%d enSourceState=%d",
                        bIsDABAvailable,
                        ETG_CENUM(DAB_tenSourceState, enSourceState)));
    };
};
struct DAB_trAudioStatusProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL

    DAB::tenMeca_RdmMode enDabMode;
    DAB::tenMeca_RdmDrcMode enDrcMode;
    DAB::tenMeca_RdmMusicSpeech enMusicSpeech;
    DAB::tenMeca_RdmAudioMode enAudioMode;
    DAB::tenMeca_RdmDrcInfo enDrcInfo;
    DAB::tenMeca_RdmAlgorithm enAlgorithm;
    tU16 u16BitRateKbps;
    tBool bOriginal;
    tBool bCopyRight;
    tBool bDataValid;

    DAB_trAudioStatusProperty() {
        //u8DabMode = 0;
    	enDabMode = enMeca_RdmMode_DAB_INACTIVE;
        bDataValid = FALSE;
        enDrcMode = DAB::enMeca_RdmDrcMode_OFF;
        enMusicSpeech = DAB::enMeca_RdmMusicSpeech_NOT_SIGNALLED;
        enAudioMode = DAB::enMeca_RdmAudioMode_STEREO;
        enDrcInfo = DAB::enMeca_RdmDrcInfo_INVALID;
        enAlgorithm = DAB::enMeca_RdmAlgorithm_MPEG1;
        u16BitRateKbps = 0;
        bOriginal = FALSE;
        bCopyRight = FALSE;
    }
    bool operator!=(DAB_trAudioStatusProperty const& b) const
    {
    	DAB_RET_1_IF_NEQ(enDabMode);
        DAB_RET_1_IF_NEQ(enDrcMode);
        DAB_RET_1_IF_NEQ(enMusicSpeech);
        DAB_RET_1_IF_NEQ(enAudioMode);
        DAB_RET_1_IF_NEQ(enDrcInfo);
        DAB_RET_1_IF_NEQ(enAlgorithm);
        DAB_RET_1_IF_NEQ(u16BitRateKbps);
        DAB_RET_1_IF_NEQ(bOriginal);
        DAB_RET_1_IF_NEQ(bCopyRight);

        DAB_RET_1_IF_NEQ(bDataValid);

        return FALSE;
    }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  DAB_trAudioStatusProperty:"));
        };

};
struct DAB_trCompListProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL
    
    tBool bOpen;
    tU16 bActivatedElemId;
    vector<DAB::trChnListElem>vectorCompEntries;
    
    
    DAB_trCompListProperty():
        bOpen(FALSE),
        bActivatedElemId(0),
        vectorCompEntries()
    {}


   
    bool operator!=(DAB_trCompListProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(bOpen);
        DAB_RET_1_IF_NEQ(bActivatedElemId);
        DAB_RET_1_IF_NEQ(vectorCompEntries);

        return FALSE;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR4(("  DAB_trCompListProperty:bOpen=%d bActivatedElemId=%d numElem=%d",
                        bOpen,
                        bActivatedElemId,
                        (tU16)(vectorCompEntries.size())));
    };
};

struct DAB_trLearnUpdateStatusProperty: public DAB_Message
{
    DAB_DISPATCH_IMPL
    tU8 u8LearnPercentage;
    
    
    DAB_trLearnUpdateStatusProperty():
    	u8LearnPercentage(0)   //Updating the default value to zero. as per HMI requirement
    {}

    bool operator!=(DAB_trLearnUpdateStatusProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(u8LearnPercentage);

        return FALSE;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR4(("  DAB_trLearnUpdateStatusProperty:enAdrDiag=%d",
        		u8LearnPercentage));
    };
};

struct DAB_trErrLogProperty: public DAB_Message
{
    DAB_DISPATCH_IMPL
    DAB::DAB_tenTestResult enAdrDiag;
    
    
    DAB_trErrLogProperty():
        enAdrDiag(DAB::DAB_tenTestResult_NORESULT)
    {}


   
    bool operator!=(DAB_trErrLogProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(enAdrDiag);

        return FALSE;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR4(("  DAB_trErrLogProperty:enAdrDiag=%d",
                        ETG_CENUM(DAB_tenTestResult, enAdrDiag)));
    };
};

struct DAB_trTmcEncryption
{
    tBool bEncrypted;
    tU8 u8EncId;
    tU8 u8TestBits;

    DAB_trTmcEncryption():
        bEncrypted(FALSE), u8EncId(0),
        u8TestBits(0)
        {}
};

struct DAB_trTmcMsgSource
{
    tenTmcMediumType DabMediumTypeId;
    tU8 u8Number;

    DAB_trTmcMsgSource():
        DabMediumTypeId(enTmcMediumType_DAB), 
        u8Number(0)
        {}
};

struct DAB_trTmcDataListElement
{
    DAB_trTmcRdbId RdbId;
    tS16 s16Ecc;
    tU16 u16Aid;
    tU8 au8TmcData[5];
    tU16 u16ProgramId;
    DAB_trTmcEncryption TmcEncryption;
    DAB_trTmcMsgSource DabMsgSource;
    tCString sProgramServiceName;
    tCString sServiceProviderName;

    DAB_trTmcDataListElement():
        s16Ecc(-1), u16Aid(0),
        u16ProgramId(0)        
        {
            sProgramServiceName = "";
            sServiceProviderName = "";
            for(tU8 i=0; i<5; i++)
                au8TmcData[i] = 0;
        }
     
    bool operator!=(DAB_trTmcDataListElement const& b) const
        {
            DAB_RET_1_IF_NEQ(au8TmcData[0]);
            DAB_RET_1_IF_NEQ(au8TmcData[1]);
            DAB_RET_1_IF_NEQ(au8TmcData[2]);
            DAB_RET_1_IF_NEQ(au8TmcData[3]);
            DAB_RET_1_IF_NEQ(au8TmcData[4]);
            
            return FALSE;
        }
	 bool operator==(DAB_trTmcDataListElement const& b) const
        {
            DAB_RET_0_IF_NEQ(au8TmcData[0]);
            DAB_RET_0_IF_NEQ(au8TmcData[1]);
            DAB_RET_0_IF_NEQ(au8TmcData[2]);
            DAB_RET_0_IF_NEQ(au8TmcData[3]);
            DAB_RET_0_IF_NEQ(au8TmcData[4]);
            return FALSE;
        }
};

struct DAB_trTmcDataListProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL

    vector<DAB::DAB_trTmcDataListElement>vectorTmcDataListElements;

    DAB_trTmcDataListProperty():
        vectorTmcDataListElements()
    {}

    bool operator!=(DAB_trTmcDataListProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(vectorTmcDataListElements);

        return FALSE;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR4(("  DAB_trTmcDataListProperty: numElem=%d",
                        (tU16)(vectorTmcDataListElements.size())));
    };
};

struct DAB_trTmcAvailListProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL

    vector<DAB::trTmcAvailListElem>vectorTmcAvailEntries;
    
    DAB_trTmcAvailListProperty():
        vectorTmcAvailEntries()
    {}

    bool operator!=(DAB_trTmcAvailListProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(vectorTmcAvailEntries);
        return FALSE;
    }
	bool operator==(DAB_trTmcAvailListProperty const& b) const
    {
        DAB_RET_0_IF_NEQ(vectorTmcAvailEntries);
        return FALSE;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR4(("  DAB_trTmcAvailListProperty: numElem=%d",
                        (tU16)(vectorTmcAvailEntries.size())));
    };

};

struct DAB_trTmcTpegInfoProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL

    tU8 u8NumOfTmcServices;
    tU8 u8NumOfTpegServices;
    tU32 u32ReceivedTmcMsgs;
    tU32 u32ReceivedTpegMsgs;
    DAB_trTime rLastReceivedUriMsgTime;

    DAB_trTmcTpegInfoProperty(): 
        u8NumOfTmcServices(0),
        u8NumOfTpegServices(0),
        u32ReceivedTmcMsgs(0),
        u32ReceivedTpegMsgs(0),
        rLastReceivedUriMsgTime()

        {}
    bool operator!=(DAB_trTmcTpegInfoProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(u8NumOfTmcServices);
        DAB_RET_1_IF_NEQ(u8NumOfTpegServices);
        DAB_RET_1_IF_NEQ(u32ReceivedTmcMsgs);
        DAB_RET_1_IF_NEQ(u32ReceivedTpegMsgs);
        return FALSE;
    }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  DAB_trTmcTpegInfoProperty: u8NumOfTmcServices=%d u8NumOfTpegServices=%d u16ReceivedTmcMsgs=%d u16ReceivedTpegMsgs=%d",
                            u8NumOfTmcServices,
                            u8NumOfTpegServices,
                            u32ReceivedTmcMsgs,
                            u32ReceivedTpegMsgs));
        };
};


struct DAB_trComponentInfoProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL

    tU8 u8ComponentId;
    DAB::trMecaLabel rComponentLabel;
    tU16 u16Scidi;
    tU8 u8Subchannel;
    tU8 u8Scids;
    DAB::DAB_tenComponentType enComponentType;
    DAB::DAB_tenServiceType enServiceType;
    DAB::DAB_tenTMIdType enTMIIdType;
    tU8 u8AudioDataServiceComponentType;
	// Added Component Index -- Chait
    tU16 u16ComponentIndex;


    DAB_trComponentInfoProperty(): 
        u8ComponentId(0),
        rComponentLabel(),
        u16Scidi(0),
        u8Subchannel(0),
        u8Scids(0xff),
        enComponentType(DAB_enComponentType_PRIMARY),
        enServiceType(DAB_enServiceType_UNKNOWN),
        enTMIIdType(DAB_enTMIdType_MSC_AUDIO_STREAM),
        u8AudioDataServiceComponentType(0),
		u16ComponentIndex(0)
        {}
    bool operator!=(DAB_trComponentInfoProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(u8ComponentId);
        DAB_RET_1_IF_NEQ(rComponentLabel);
        DAB_RET_1_IF_NEQ(u16Scidi);
        DAB_RET_1_IF_NEQ(u8Subchannel);
        DAB_RET_1_IF_NEQ(u8Scids);
        DAB_RET_1_IF_NEQ(enComponentType);
        DAB_RET_1_IF_NEQ(enServiceType);
        DAB_RET_1_IF_NEQ(enTMIIdType);
        DAB_RET_1_IF_NEQ(u8AudioDataServiceComponentType);
        DAB_RET_1_IF_NEQ(u16ComponentIndex);
        return FALSE;
    }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  DAB_trComponentInfoProperty: u8ComponentId=%x"
                            "  u16Scidi=%x  u8Subchannel=%d u8Scids=%x enComponentType=%d  enServiceType=%d  enTMIIdType=%d"
                            "  u8AudioDataServiceComponentType=%x rLabel=%s",
                            u8ComponentId,
                            u16Scidi,
                            u8Subchannel,
                            u8Scids,
                            enComponentType,
                            enServiceType,
                            enTMIIdType,
                            u8AudioDataServiceComponentType,
                            rComponentLabel.pcGetCString()));
			ETG_TRACE_USR4(("  DAB_trComponentInfoProperty: u16ComponentIndex=%d", u16ComponentIndex ));
        };

};


    
struct DAB_trTsuSwitchProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL

    DAB::tenMeca_TsuTunerId enSelectedTunerId;
    DAB::tenMeca_TsuTunerId enRecomTunerId;
    tBool bUserRequest;
    trMecaProgrammeService rRecomService;

    DAB_trTsuSwitchProperty():
        enSelectedTunerId(enMeca_TsuTunerId_INVALID),
        enRecomTunerId(enMeca_TsuTunerId_INVALID),
        bUserRequest(FALSE)
    {}

    bool operator!=(DAB_trTsuSwitchProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(enSelectedTunerId);
        DAB_RET_1_IF_NEQ(enRecomTunerId);
        DAB_RET_1_IF_NEQ(bUserRequest);
        DAB_RET_1_IF_NEQ(rRecomService)

        return FALSE;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR4(("  DAB_trTsuSwitchProperty:%d --> %d (bUserRequest=%d srv=0x%08x)",
                        ETG_CENUM(tenMeca_TsuTunerId, enSelectedTunerId), 
                        ETG_CENUM(tenMeca_TsuTunerId, enRecomTunerId),
                        bUserRequest,
                        rRecomService.u32GetSID()));
    };
};

struct DAB_trMtcServiceFollowingProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL

    DAB::tenMeca_MtcTuner enMtcTuner;

    DAB_trMtcServiceFollowingProperty():
        enMtcTuner(enMeca_MtcTuner_MAIN_DAB_TUNER)
    {}

    bool operator!=(DAB_trMtcServiceFollowingProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(enMtcTuner)

        return FALSE;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR4(("  DAB_trMtcServiceFollowingProperty:%d )",
                        ETG_CENUM(DAB::tenMeca_MtcTuner, enMtcTuner)));
    };
};

/*
struct DAB_trActiveServiceInfoProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL

    tU32 u32ServiceId;
    DAB::trMecaLabel rServiceLabel;
    DAB::DAB_tenServiceType enServiceType;
    tU8 u8NumberOfAudioComponents;
    tU16 u16AnnoMask;
    DAB::DAB_tenSampleRate enSampleRate;
    tU16 u16BitRate;
    DAB::DAB_tenAudioMode enAudioMode;
    DAB::DAB_tenAudioCodec enAudioCodec;
    tBool b8Flag;
    tU8 u8PresetListElementId;

    DAB_trActiveServiceInfoProperty() :
        u32ServiceId(0)
        rServiceLabel(),
        enServiceType(DAB_enServiceType_UNKNOWN),
        u8NumberOfAudioComponents(0),
        u16AnnoMask(0),
        enSampleRate(DAB_enSampleRate_48KHZ),
        u16BitRate(0),
        enAudioMode(0),
        enAudioCodec(0),
        b8Flag(0),
        u8PresetListElementId(0)
        {}
    bool operator!=(DAB_trActiveServiceInfoProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(u32ServiceId);
        DAB_RET_1_IF_NEQ(rServiceLabel);
        DAB_RET_1_IF_NEQ(enServiceType);
        DAB_RET_1_IF_NEQ(u8NumberOfAudioComponents);
        DAB_RET_1_IF_NEQ(u16AnnoMask);
        DAB_RET_1_IF_NEQ(enSampleRate);
        DAB_RET_1_IF_NEQ(u16BitRate);
        DAB_RET_1_IF_NEQ(enAudioMode);
        DAB_RET_1_IF_NEQ(enAudioCodec);
        DAB_RET_1_IF_NEQ(b8Flag);
        DAB_RET_1_IF_NEQ(u8PresetListElementId);

        return FALSE;
    }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  DAB_trActiveServiceInfoProperty:"));
        };
};
*/
struct DAB_trEnsembleInfoProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL

    tU32 u32EnsembleId;
    DAB::trMecaLabel rEnsembleLabel;
    tU32 u32Frequency;
    tU8 u8NumberOfAudioServices;
    tU8 u8NumberOfDataServices;
    tU8 u8NumberOfAudioServiceComponents;
    tU8 u8NumberOfDataServiceComponents;
    tU8 u8ReceptionQuality;
    tU8 u8NumberOfEnsemble;
	tBool bReception;

    DAB_trEnsembleInfoProperty():
        u32EnsembleId(0),
        rEnsembleLabel(),
        u32Frequency(0),
        u8NumberOfAudioServices(0),
        u8NumberOfDataServices(0),
        u8NumberOfAudioServiceComponents(0),
        u8NumberOfDataServiceComponents(0),
        u8ReceptionQuality(0),
        u8NumberOfEnsemble(0),
		bReception(FALSE)
        {}
    bool operator!=(DAB_trEnsembleInfoProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(u32EnsembleId);
        DAB_RET_1_IF_NEQ(rEnsembleLabel);
        DAB_RET_1_IF_NEQ(u32Frequency);
        DAB_RET_1_IF_NEQ(u8NumberOfAudioServices);
        DAB_RET_1_IF_NEQ(u8NumberOfDataServices);
        DAB_RET_1_IF_NEQ(u8NumberOfAudioServiceComponents);
        DAB_RET_1_IF_NEQ(u8NumberOfDataServiceComponents);
        DAB_RET_1_IF_NEQ(u8ReceptionQuality);
        DAB_RET_1_IF_NEQ(u8NumberOfEnsemble);
		DAB_RET_1_IF_NEQ(bReception);
        return FALSE;
    }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  DAB_trEnsembleInfoProperty: u32EnsembleId=%d  "
                            "  u32Frequency=%d  u8NumberOfAudioServices=%d  u8NumberOfDataServices=%d"
                            "  u8NumberOfAudioServiceComponents=%d  u8NumberOfDataServiceComponents=%d"
                            "  u8ReceptionQuality=%d  u8NumberOfEnsemble=%d"
                            "  Label=%s",
                            u32EnsembleId,
                            u32Frequency,
                            u8NumberOfAudioServices,
                            u8NumberOfDataServices,
                            u8NumberOfAudioServiceComponents,
                            u8NumberOfDataServiceComponents,
                            u8ReceptionQuality,
                            u8NumberOfEnsemble,
                            rEnsembleLabel.pcGetCString()));
        };

};

struct DAB_trTpegDataListElement
{

    vector<tU8>vecTpegDataListElement;

    DAB_trTpegDataListElement():
        vecTpegDataListElement()
    {}
     
    bool operator!=(DAB_trTpegDataListElement const& b) const
        {
            DAB_RET_1_IF_NEQ(vecTpegDataListElement);
            
            return FALSE;
        }
	bool operator==(DAB_trTpegDataListElement const& b) const
	{
		DAB_RET_0_IF_NEQ(vecTpegDataListElement);
		return FALSE;
	}
};

struct DAB_trTpegDataListProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL

    vector<DAB::DAB_trTpegDataListElement>vecTpegDataListElements;

    DAB_trTpegDataListProperty():
        vecTpegDataListElements()
    {}

    bool operator!=(DAB_trTpegDataListProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(vecTpegDataListElements);

        return FALSE;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR4(("  DAB_trTpegDataListProperty: numElem=%d",
                        (tU16)(vecTpegDataListElements.size())));
    };
};

struct DAB_trPresetListProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL

	tU8 u8FirstElementID;
	tU8 u8NumberOfElements;	
	vector<trPresetInfo> vecPresetListInfo;

    DAB_trPresetListProperty():
		u8FirstElementID(0),
		u8NumberOfElements(0),
        vecPresetListInfo()
    {}

    bool operator!=(DAB_trPresetListProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(vecPresetListInfo);

        return FALSE;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR4(("  DAB_trPresetListProperty: numElem=%d",
                        (tU16)(vecPresetListInfo.size())));
    };
};

struct DAB_trMTCSetModeProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL

    DAB_tenMtcMode enMtcMode; 

    DAB_trMTCSetModeProperty():
        enMtcMode(DAB_enMtcMode_Off)
    {}

    bool operator!=(DAB_trMTCSetModeProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(enMtcMode)

        return FALSE;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR4(("  DAB_trMTCSetModeProperty:%d )",
                        ETG_CENUM(DAB_tenMtcMode, enMtcMode)));
    };
};

//struct DAB_trMTCSwitchStatusProperty: public DAB_Message 
//{
//    DAB_DISPATCH_IMPL
//
//	DAB_tenMtcRequestSwitch enMtcRequestSwitch;
//
//    DAB_trMTCSwitchStatusProperty():
//        enMtcRequestSwitch(DAB_enMtcRequestSwitch_DAB)
//    {}
//
//    bool operator!=(DAB_trMTCSwitchStatusProperty const& b) const
//    {
//        DAB_RET_1_IF_NEQ(enMtcRequestSwitch)
//
//        return FALSE;
//    }
//
//    virtual tVoid vTrace() const {
//        ETG_TRACE_USR4(("  DAB_trMTCSwitchStatusProperty:%d )",
//                        ETG_CENUM(DAB_tenMtcRequestSwitch, enMtcRequestSwitch)));
//    };
//};

struct DAB_trMTCTunerStatusProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL

	DAB_tenMtcTunerSwitchStatus e8MtcTunerSwitchStatus;

    DAB_trMTCTunerStatusProperty():
        e8MtcTunerSwitchStatus(DAB_enMtcTunerSwitchStatus_Main_Unknown)
    {}

    bool operator!=(DAB_trMTCTunerStatusProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(e8MtcTunerSwitchStatus)

        return FALSE;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR4(("  DAB_trMTCTunerStatusProperty:%d )",
                        ETG_CENUM(DAB_tenMtcTunerSwitchStatus, e8MtcTunerSwitchStatus)));
    };
};

//struct DAB_trMTCAnnouncementProperty: public DAB_Message 
//{
//    DAB_DISPATCH_IMPL
//
//	DAB_tenMtcActivityState enMtcActivityState;
//
//    DAB_trMTCAnnouncementProperty():
//        enMtcActivityState(DAB_enMtcActivityState_Stop)
//    {}
//
//    bool operator!=(DAB_trMTCAnnouncementProperty const& b) const
//    {
//        DAB_RET_1_IF_NEQ(enMtcActivityState)
//
//        return FALSE;
//    }
//
//    virtual tVoid vTrace() const {
//        ETG_TRACE_USR4(("  DAB_trMTCAnnouncementProperty:%d )",
//                        ETG_CENUM(DAB_tenMtcActivityState, enMtcActivityState)));
//    };
//};


struct DAB_trMTCSurveillanceMonitorProperty: public DAB_Message 
{
    DAB_DISPATCH_IMPL

	DAB_tenMtcStationType enMtcStationType;
	tU32 u32MtcStationFreq;
	tU32 u32MtcStationId;
	tU32 u32MtcEnsID;
	DAB_tenMtcTPState enMtcTPState;
	tU8 sMtcStationNameFM[8];
	tU8 sStationNameDAB[16];


    DAB_trMTCSurveillanceMonitorProperty():
    
		enMtcStationType(DAB_enMtcStationType_DAB),
		u32MtcStationFreq(0),
		u32MtcStationId(0),
		u32MtcEnsID(0),
		enMtcTPState(DAB_enMtcTPState_No_TP)
		{
			for(tU8 i=0; i<8; i++)
				sMtcStationNameFM[i] = 0;

			for(tU8 j=0; j<16; j++)
				sStationNameDAB[j] = 0;

		}

    bool operator!=(DAB_trMTCSurveillanceMonitorProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(enMtcStationType);
        DAB_RET_1_IF_NEQ(u32MtcStationFreq);
        DAB_RET_1_IF_NEQ(u32MtcStationId);
        DAB_RET_1_IF_NEQ(u32MtcEnsID);
        DAB_RET_1_IF_NEQ(enMtcTPState);
		for (tU8 i =0; i<8; i++)
			DAB_RET_1_IF_NEQ(sMtcStationNameFM[i]);
        
		for (tU8 j =0; j<8; j++)
			DAB_RET_1_IF_NEQ(sStationNameDAB[j]);
 
        return FALSE;
    }

    virtual tVoid vTrace() const {
        ETG_TRACE_USR4(("  DAB_trMTCSurveillanceMonitorProperty: e8MtcMtcStationType =%d "
			"  e8MtcStationFreq=%d  e8MtcStationId=%d  e8MtcEnsID=%d"
            "  e8MtcTPState=%d  sMtcStationNameFM=%s ",
             ETG_CENUM(DAB_tenMtcStationType, enMtcStationType), u32MtcStationFreq,
			 u32MtcStationId, u32MtcEnsID, 
			 ETG_CENUM(DAB_tenMtcTPState, enMtcTPState), sMtcStationNameFM));
		ETG_TRACE_USR4(("sStationNameDAB=%s", sStationNameDAB));
    };
};


struct DAB_trMsgSrvListUpdateProperty:public DAB_Message
{
    DAB_DISPATCH_IMPL

    DAB_trMsgSrvListUpdateProperty():vcAvailSrvList()
    {}
        
    DAB_trMsgSrvListUpdateProperty(vector<trChnListElem> const &vcAvailSrvList_):vcAvailSrvList(vcAvailSrvList_)
    {}

    bool operator!=(DAB_trMsgSrvListUpdateProperty const& b) const
    {
        DAB_RET_1_IF_NEQ(vcAvailSrvList);
		
        return FALSE;
    }

    virtual tVoid vTrace() const {
        DAB_FOREACH_CONST(vector<trChnListElem>, iter, vcAvailSrvList) {
            ETG_TRACE_USR1((   "  %08x %d %17s ", 
                                iter->rMecaId.u32GetSID(), 
                                iter->bReception,
                                iter->rLabel.pcGetCString()));
        }
    };
    vector<trChnListElem>vcAvailSrvList;
 };


struct DAB_trMsgSrvListUpdateNotification:public DAB_Message
{
    DAB_DISPATCH_IMPL
      
    DAB_trMsgSrvListUpdateNotification():
    bListUpdate(FALSE)
    {}

    bool operator!=(DAB_trMsgSrvListUpdateNotification const& b) const
    {
        DAB_RET_1_IF_NEQ(bListUpdate);
		
        return FALSE;
    }

    virtual tVoid vTrace() const {
         ETG_TRACE_USR1((" DAB_trMsgSrvListUpdateNotification: bListUpdate:%d )",
                    bListUpdate));
        }
    tBool bListUpdate;
 };


class dabdrv_properties:
    public DAB_Singleton<dabdrv_properties>,
    public DAB_MessageHandlerBase
{
    friend class DAB_Singleton<dabdrv_properties>;
public:
    DAB_TRACE_INFO_IMPL(FC_DABTUNER_TR_DRV_UTIL);
    virtual char const *pcGetName() const {return "dabdrv_properties";};

    DAB_tclProperty<DAB_trTunerStatusProperty> oTunerStatusProperty;
    DAB_tclProperty<DAB_trTestModeDataProperty> oTestModeDataProperty;
    DAB_tclProperty<DAB_trBgTestModeDataProperty> oBgTestModeDataProperty;
    DAB_tclProperty<DAB_trComponentInfoProperty> oComponentInfoProperty;
    DAB_tclProperty<DAB_trServiceInfoProperty> oServiceInfoProperty;
    DAB_tclProperty<DAB_trEnsembleInfoProperty> oEnsembleInfoProperty;
    DAB_tclProperty<DAB_trChnStateProperty> oChnStateProperty;
    DAB_tclProperty<DAB_trChnInfoProperty> oChnInfoProperty;
    DAB_tclProperty<DAB_trAnnoStatusProperty> oAnnoStatusProperty;
    DAB_tclProperty<DAB_trMuteRequestProperty> oMuteRequestProperty;
    DAB_tclProperty<DAB_trSetupStatusProperty> oSetupStatusProperty;
    DAB_tclProperty<DAB_trAvailabilityProperty> oAvailabilityProperty;
    DAB_tclProperty<DAB_trCompListProperty> oCompListProperty;
    DAB_tclProperty<DAB_trTmcAvailListProperty> oTmcAvailListProperty;
    DAB_tclProperty<DAB_trTmcTpegInfoProperty> oTmcTpegInfoProperty;

    DAB_tclProperty<DAB_trAudioStatusProperty> oAudioStatusProperty;
    DAB_tclProperty<DAB_trErrLogProperty> oErrLogProperty;
    DAB_tclProperty<DAB_trTmcDataListProperty> oTmcDataListProperty;
    DAB_tclProperty<DAB_trTsuSwitchProperty> oTsuSwitchProperty;
    DAB_tclProperty<DAB_trMtcServiceFollowingProperty> oMtcServiceFollowingProperty;
    DAB_tclProperty<DAB_trTpegDataListProperty> oTpegDataListProperty;
	DAB_tclProperty<DAB_trPresetListProperty> oPresetListDataProperty;
	DAB_tclProperty<DAB_trLearnUpdateStatusProperty> oLearnUpdateProperty;
	DAB_tclProperty<DAB_trADRMsgProperty> oADRMsgProperty;
	DAB_tclProperty<DAB_trLocalOffsetProperty> oLocalOffsetProperty;
	DAB_tclProperty<DAB_trMTCSetModeProperty> oMTCSetModeProperty;

	//DAB_tclProperty<DAB_trMTCSwitchStatusProperty> oMTCSwitchStatusProperty;

	DAB_tclProperty<DAB_trMTCTunerStatusProperty> oMTCTunerStatusProperty;

	//DAB_tclProperty<DAB_trMTCAnnouncementProperty> oMTCAnnouncementProperty;

	DAB_tclProperty<DAB_trMTCSurveillanceMonitorProperty> oMTCSurveillanceMonitorProperty;
    DAB_tclProperty<DAB_trMsgSrvListUpdateProperty> oSrvListUpdateProperty;
    DAB_tclProperty<DAB_trMsgSrvListUpdateNotification> oSrvListUpdateNotification;
};
}
#endif

