/************************************************************************
 * FILE:     dabdrv_mecaIf.cpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of dabdrv_meca
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      		 | Author                       | Modification
   
				
 *************************************************************************/
 
#include "dabdrv_mecaIf.h"

#define DAB_FBLOCK_LUNID 18
#define MTC_FBLOCK_LUNID 24

#include "dabdrv_main.hpp"
#ifndef _LINUXX86MAKE_
	#include "fc_dabtuner_service.h"
#endif


#include<string>
#include<cstring>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_DRV_MECAIF 
#include "trcGenProj/Header/dabdrv_mecaIf.cpp.trc.h"
#endif

int DAB_mecaIf_dummy=0;

using namespace DAB;



namespace DAB {

    //default constructor
    trMsgAdrInput::trMsgAdrInput():_rMostHdr() {
        _au8StaticData[0] = 0;
        _pu8DynData=OSAL_NULL;
        _pu8MecaData = _au8StaticData;
        _bValid = FALSE;
    };

    trMsgAdrInput::trMsgAdrInput(tU8 const *pu8Data, tU32 u32Len):
        _rMostHdr()
    {
        _au8StaticData[0]=0;
        _pu8DynData = OSAL_NULL;
        _pu8MecaData = _au8StaticData;            

        tS32 s32MecaLen = (tS32)(u32Len - (tU32)enAdrMsgOffset_MECA_DATA);
        _bValid =  FALSE;
        // check length
        if (s32MecaLen < 0) {
            return;
        }
        
        tU16 u16MecaLen = DABDRV_GET_U16(&pu8Data[enAdrMsgOffset_MECA_LEN]);
            
        if ((tS32)u16MecaLen == s32MecaLen) {
            _bValid =  TRUE;
        } else {
            return;
        }
            
        // check if memory has to be allocated
        if (u16MecaLen > DAB_ADR_INPUT_STATIC_LEN) {
            _pu8DynData = OSAL_NEW tU8[u16MecaLen];
            _pu8MecaData= _pu8DynData;

		if(NULL != _pu8MecaData)
		{
			OSAL_pvMemoryCopy(_pu8MecaData, &pu8Data[enAdrMsgOffset_MECA_DATA], u16MecaLen);
			}
		}
		else{
			OSAL_pvMemoryCopy(_pu8MecaData, &pu8Data[enAdrMsgOffset_MECA_DATA], u16MecaLen);
		}

        // read data
        _rMostHdr= trMostHdr(pu8Data);
        if (!_rMostHdr.bIsValid()) {
            _bValid = FALSE;
        }
    }

    trMsgAdrInput::trMsgAdrInput(const trMsgAdrInput &roFrom) {
        _bValid = roFrom._bValid;
        _rMostHdr = roFrom._rMostHdr;
        _au8StaticData[0]=0;
        _pu8MecaData = _au8StaticData;
        _pu8DynData = OSAL_NULL;
        
        if (!_bValid) {
            return;
        }

        if (OSAL_NULL != roFrom._pu8DynData) {
            _pu8DynData = OSAL_NEW tU8[_rMostHdr._u16MecaLen];
            _pu8MecaData = _pu8DynData;
        }

		if(NULL != _pu8MecaData)
		{
			OSAL_pvMemoryCopy(_pu8MecaData, roFrom._pu8MecaData, _rMostHdr._u16MecaLen);
		}

    }

    trMsgAdrInput::~trMsgAdrInput() {
        _pu8MecaData = OSAL_NULL;
        if (OSAL_NULL != _pu8DynData) {
            OSAL_DELETE[] _pu8DynData;
            _pu8DynData = OSAL_NULL;
        }
    }

    tVoid trMsgAdrInput::vTrace() const {
        ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_MSG, "  trMsgAdrInput:_bValid=%u, _u16ClientId=0x%x _u16FBlockId=0x%x"
                            "_u8InstanceId=%u _u16FktId=%04x _enOpType=%x _u16MecaLen=%u", 
                            _bValid,
                            _rMostHdr._u16ClientId,
                            _rMostHdr._u16FBlockId,
                            _rMostHdr._u8InstanceId,
                            ETG_CENUM(DAB_tenMecaMsgId, _rMostHdr._u16FktId),
                            ETG_CENUM(tenOpType,_rMostHdr._enOpType),
                            _rMostHdr._u16MecaLen));
    };

    tVoid trMsgDeBlockMeca::vTrace() const {
            ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_MSG, 
                                "  trMsgDeBlockMeca"));
    }


    // R:enMecaFktId_Startup
    struct trMeca_RConStartup:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            (tVoid)pu8Data;
            if (rMostHdr._u16MecaLen != 0) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  trMeca_RConStartup"));
        };
    };

    struct trMeca_RConTsuStartup:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            (tVoid)pu8Data;
            if (rMostHdr._u16MecaLen != 0) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  trMeca_RConTsuStartup"));
        };
    };
    // C:enMecaFktId_Registration
    struct trMeca_CConRegistration:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = enMecaFktId_ConRegistration;
            pRawOutput->enOpType = enOpType_SETGET;
            pRawOutput->u16DataLen = 0;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  trMeca_CConRegistration:"));
        };
    };


    // R:enMecaFktId_ConRegistration
    struct trMeca_RConRegistration:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            (tVoid)pu8Data;
            if (rMostHdr._u16MecaLen != 0) {
                //   return FALSE;
            }
            _rMostHdr=rMostHdr;
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  trMeca_RConRegistration:"));
        };
    };


    // C:enMecaFktId_ConDeRegistration
    struct trMeca_CConDeRegistration:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = enMecaFktId_ConRegistration;
            pRawOutput->enOpType = enOpType_SETGET;
            pRawOutput->u16DataLen = 0;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  trMeca_CConDeRegistration:"));
        };
    };


    // R:enMecaFktId_ConDeRegistration
    struct trMeca_RConDeRegistration:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            (tVoid)pu8Data;
            if (rMostHdr._u16MecaLen != 0) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  trMeca_RConDeRegistration:"));
        };
    };

    // C:enMecaFktId_ConPing
    struct trMeca_CConPing:
        public  trMsgMecaOut {
        DAB_DISPATCH_IMPL

        virtual tVoid vSerialize(trMsgMecaRawOutput *pRawOutput) const {
            pRawOutput->u16FBlockId = DAB_FBLOCK_ID;
            pRawOutput->u16MsgType = enMecaFktId_ConPing;
            pRawOutput->enOpType = enOpType_SETGET;
            pRawOutput->u16DataLen = 0;
        }

        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  trMeca_CConPing:"));
        };
    };

    // R:enMecaFktId_ConPing
    struct trMeca_RConPing:
        public trMsgMecaInput {
        DAB_DISPATCH_IMPL
    
        virtual tBool bParse(trMostHdr const &rMostHdr, tU8 const *pu8Data) {
            (tVoid)pu8Data;
            if (rMostHdr._u16MecaLen != 0) {
                return FALSE;
            }
            _rMostHdr=rMostHdr;
            return TRUE;
        }
        virtual tVoid vTrace() const {
            ETG_TRACE_USR4(("  trMeca_RConPing:"));
        };
    };


    // Timer


    // a timer-event
    struct dabdrv_msgPingTimer:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_MSG, 
                                "  dabdrv_msgPingTimer"));
        };

    };


} // namespace DAB

dabdrv_mecaIf::dabdrv_mecaIf(tVoid) {
    _enComState = enComState_Null;
    _u32PingTmrValMs = DAB_ADR_PING_TIMER_MS;
    _u32LastMecaRefId=0;
    _bMtcLunStartup = FALSE;
    _bDabLunStartup = FALSE;
    _bConRegistrationValid = FALSE;
    dabdrv_adrIf::instance();

}

tVoid dabdrv_mecaIf::vInit(tVoid) {

	ETG_TRACE_USR4(("dabdrv_mecaIf:vInit"));

    _oPingTmr.vInit(instance(),dabdrv_msgPingTimer());
    
    // subsribe
    vSubscribe<trMeca_RRdmTune>();
    vSubscribe<trMsgAdrInput>();
    vSubscribe<trMsgIpnError>();
    vSubscribe<trMsgDeBlockMeca>();
    vSubscribe<trMsgSrvCmdCriticalVoltage>();
    vSubscribe<trMsgAdrStatus>();
    vSubscribe<trMeca_RConPing>();

//#ifdef VARIANT_S_FTR_ENABLE_SSI_COMMUNICATION_ENABLE
    vEnterComState(enComState_WaitStartup);
//#else
/**rag6kor: As startup message is not available with INC, comState is set to Registration state*/
//	vEnterComState(enComState_WaitRegistration);
//#endi
}

tVoid dabdrv_mecaIf::vDeInit(tVoid) {
    _oPingTmr.vDeInit();
}


//lint -esym(1551, dabdrv_mecaIf::~dabdrv_mecaIf) prio3 Function may throw exception '...' in destructor
dabdrv_mecaIf::~dabdrv_mecaIf() {
}

tVoid dabdrv_mecaIf::vTraceState() const {
    ETG_TRACE_USR1(("  dabdrv_mecaIf STATE: _enComState=%d",
                    ETG_CENUM(tenComState, _enComState)));
}

tVoid dabdrv_mecaIf::vSendAdrMsg(tU16 u16FBlockId, tU16 u16FktId, tenOpType enOpType, tU16 u16MecaLen, tU8 const *pu8MecaData, trMecaSendSvInfo const *poSendInfo, tU8 u8InstanceId) {
	fc_dabtuner_config *poConfig=fc_dabtuner_config::instance();
	if(poConfig == nullptr)
	{
		return;
	}

	if ( (enComState_Normal != _enComState) && (u16FktId != (tU16)enMecaFktId_ConRegistration) ) {
		if ( u16FBlockId == DAB_MTC_FBLOCK_ID ) {
			ETG_TRACE_FATAL(("  dabdrv_mecaIf::vSendAdrMsg: sending of %d not possible in comState %d",
					ETG_CENUM(DAB_tenMecaTsuMsgId, u16FktId),
					ETG_CENUM(tenComState, _enComState)));
		}
		else {
			ETG_TRACE_FATAL(("  dabdrv_mecaIf::vSendAdrMsg: sending of %d not possible in comState %d",
					ETG_CENUM(DAB_tenMecaMsgId, u16FktId),
					ETG_CENUM(tenComState, _enComState)));
		}
		if (OSAL_NULL !=poSendInfo) {
			trMecaSendSvInfo oSendInfo=*poSendInfo;
			// this is dirty, the caller will get the sendInfo back with an dangling pointer!
			oSendInfo.vClear();
		}
		return;
	}
	trMsgAdrOutput oAdrOutput;
	oAdrOutput.u32MecaRefId=_u32LastMecaRefId;
	_u32LastMecaRefId++;
	oAdrOutput.u32Len = u16MecaLen + (tU16)enAdrMsgOffset_MECA_DATA;
	tU8 *pu8AdrMsg = oAdrOutput.au8AdrMsg;
	DABDRV_SET_U16(&pu8AdrMsg[enAdrMsgOffset_CLIENT_ID], DAB_CLIENT_ID);
	DABDRV_SET_U16(&pu8AdrMsg[enAdrMsgOffset_FBLOCK_ID], u16FBlockId);
	oAdrOutput.u16FBlockId = u16FBlockId;
	ten_MecaSendMode enSendMode=en_MecaSendMode_Normal;
	if (OSAL_NULL != poSendInfo) {
		enSendMode=poSendInfo->enSendMode;
		oAdrOutput.oSendInfo=*poSendInfo;
	}
	switch (enSendMode) {
	case en_MecaSendMode_Normal:
		oAdrOutput.u16WaitAnswer=0;
		break;
	case en_MecaSendMode_WaitAnswer:
		oAdrOutput.u16WaitAnswer=(tU16)(u16FktId+1);
		break;
	case en_MecaSendMode_Block:
		oAdrOutput.u16WaitAnswer=0xFFFF;
		break;
	default:
		oAdrOutput.u16WaitAnswer=0;
		break;
	}
	if ( u16FBlockId == DAB_MTC_FBLOCK_ID ) {
		u8InstanceId = DAB_MTC_INSTANCE_ID;
	}
	pu8AdrMsg[enAdrMsgOffset_INSTANCE_ID]=u8InstanceId;
	DABDRV_SET_U16(&pu8AdrMsg[enAdrMsgOffset_FKT_ID], u16FktId);
	pu8AdrMsg[enAdrMsgOffset_OP_TYPE]=(tU8)enOpType;
	DABDRV_SET_U16(&pu8AdrMsg[enAdrMsgOffset_MECA_LEN], u16MecaLen);
	OSAL_pvMemoryCopy(&pu8AdrMsg[enAdrMsgOffset_MECA_DATA], pu8MecaData, u16MecaLen);
	DAB_vPostMsg(dabdrv_adrIf::instance(), &oAdrOutput);
}

tVoid dabdrv_mecaIf::vSendMecaCommand( trMsgMecaOut const &roMecaOutMsg, trMecaSendSvInfo const *poSendInfo) {

    trMsgMecaRawOutput oMecaRawOutput;
    roMecaOutMsg.vSerialize(&oMecaRawOutput);
    if ( oMecaRawOutput.u16FBlockId  == DAB_FBLOCK_ID ) {
    ETG_TRACE_USR1(("dabdrv_mecaIf:vSendMecaCommand START: u16MsgType=%04x enOpType=%02x u16DataLen=%d msg follows:",
                    ETG_CENUM(DAB_tenMecaMsgId, oMecaRawOutput.u16MsgType),
                    ETG_CENUM(tenOpType,oMecaRawOutput.enOpType),
                    oMecaRawOutput.u16DataLen));
    }
    else {
        ETG_TRACE_USR1(("dabdrv_mecaIf:vSendMecaCommand START: u16MsgType=%04x enOpType=%02x u16DataLen=%d msg follows:",
                        ETG_CENUM(DAB_tenMecaTsuMsgId, oMecaRawOutput.enMecaTsuMsgId),
                        ETG_CENUM(tenOpType,oMecaRawOutput.enOpType),
                        oMecaRawOutput.u16DataLen));
    }
    if (OSAL_NULL != poSendInfo) {
        ETG_TRACE_USR1(("dabdrv_mecaIf:vSendMecaCommand:sendMode=%d timeOutMs=%d usesCallBack=%d",
                        ETG_CENUM(ten_MecaSendMode, poSendInfo->enSendMode), 
                        poSendInfo->u32TimeOutMs,
                        (OSAL_NULL == poSendInfo->poMsgCaller) ? 0 : 1));
    }
    roMecaOutMsg.vTrace();

    vSendMecaCommand(&oMecaRawOutput, poSendInfo);
    ETG_TRACE_USR4(("dabdrv_mecaIf:vSendMecaCommand END"))
}

tVoid dabdrv_mecaIf::vSendMecaCommand(trMsgMecaRawOutput const *poRawOutput, trMecaSendSvInfo const *poSendInfo) {
    if ( poRawOutput->u16FBlockId == DAB_FBLOCK_ID ) {
        vSendAdrMsg(poRawOutput->u16FBlockId,(tU16)poRawOutput->u16MsgType, poRawOutput->enOpType, poRawOutput->u16DataLen, poRawOutput->au8Data, poSendInfo);
    } else {
        vSendAdrMsg(poRawOutput->u16FBlockId,(tU16)poRawOutput->enMecaTsuMsgId, poRawOutput->enOpType, poRawOutput->u16DataLen, poRawOutput->au8Data, poSendInfo, DAB_MTC_INSTANCE_ID);
    }
}


// input is complete message according to "InternalMeCa_V0_1_Connection.pdf" 
// u32Len includes len-byte pu8Data[0]
tVoid dabdrv_mecaIf::vSimAdrInput(tU8 const *pu8Data, tU32 u32Len) const{
    ETG_TRACE_USR4(("dabdrv_mecaIf:vSimAdrInput(u32Len=%d) START",u32Len));
    trMsgAdrInput oAdrInput = trMsgAdrInput(pu8Data, u32Len);
    DAB_vPostMsg(this, &oAdrInput);
    ETG_TRACE_USR4(("dabdrv_mecaIf:vSimAdrInput END"));
}

// input is the meca-command-msg according to "InternalMeCa_V0_1_Connection.pdf" 
// u16MsgType = MsgPara
// pu8Data    = MsgBody
// u32Len     = MsgLength
tVoid dabdrv_mecaIf::vSimMecaCmdInput(tU16 u16FktId, tenOpType enOpType, tU16 u16MecaLen, tU8 const *pu8MecaData) {
    ETG_TRACE_USR4(("dabdrv_mecaIf:vSimMecaCmdInput START u16FktId=%04x enOpType=%02x u16MecaLen=%u",
                    ETG_CENUM(DAB_tenMecaMsgId,u16FktId), 
                    ETG_CENUM(tenOpType, enOpType),
                    u16MecaLen));
    trMostHdr rMostHdr;
    rMostHdr._u16FktId = u16FktId;
    rMostHdr._enOpType = enOpType;
    rMostHdr._u16MecaLen=u16MecaLen;
    vHandleMecaRsp(rMostHdr, pu8MecaData);
    ETG_TRACE_USR4(("dabdrv_mecaIf:vSimMecaCmdInput END"));
}


tVoid dabdrv_mecaIf::vEnterComState(tenComState enComState) {
    ETG_TRACE_COMP(("dabdrv_mecaIf:vEnterComState(%d->%d)",
                    ETG_CENUM(tenComState, _enComState),
                    ETG_CENUM(tenComState, enComState)));
#ifndef _LINUXX86MAKE_
    fc_dabtuner_tclCCAService* mCCAService = fc_dabtuner_tclCCAService::instance();
#endif
    if (_enComState == enComState) {
        return;
    }
    else {
        _enComState = enComState;
    }
    
    switch (_enComState) {
        case enComState_Null:
            // todo
            break;
        case enComState_WaitStartup:
            {
			#ifndef _LINUXX86MAKE_	
                mCCAService->vSetDABAvailability(FALSE);
				ETG_TRACE_USR1(("Com startup enComState_WaitStartup"));
			#endif
				
                DAB_vCallMsgCtor(dabdrv_main::instance(), trMsgDrvAdrDown());
                break;
            }
        case enComState_WaitRegistration:
            vSendMecaCommand(trMeca_CConRegistration());
            break;
        case enComState_WaitReRegistration:
            DAB_vCallMsgCtor(dabdrv_main::instance(), trMsgDrvAdrDown());
            vSendMecaCommand(trMeca_CConRegistration());
            break;
        case enComState_WaitDeRegistration:
            vSendMecaCommand(trMeca_CConDeRegistration());
            break;
        case enComState_AdrDead:
            break;
        case enComState_Normal:
        {
            ETG_TRACE_USR1((" init DAB PINGTIMER \n"));
		#ifndef _LINUXX86MAKE_	
			ETG_TRACE_USR1(("Com startup enComState_Normal"));
            mCCAService->vSetDABAvailability(TRUE);
		#endif
            vSendMecaCommand(trMeca_CConPing());
            _oPingTmr.vStart(_u32PingTmrValMs);
            DAB_vCallMsgCtor(dabdrv_main::instance(), trMsgDrvAdrUp());
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_INF4CV
            if(dabdrv_adrIf::instance()->blGetClearSystemFailureErrorLog())
            {
            	DAB_trErrLogProperty rProperty= dabdrv_properties::instance()->oErrLogProperty.oGet();
            	rProperty.enAdrDiag=DAB_tenTestResult_PASSED;
            	dabdrv_properties::instance()->oErrLogProperty.vSet(rProperty);
            	dabdrv_adrIf::instance()->vSetClearSystemFailureErrorLog(false);
            }
#endif
            // todo
            break;
        }
        default:
            break;
    }
}


tVoid dabdrv_mecaIf::vProcess(trMeca_RConStartup* poRConStartup) {
    (tVoid)poRConStartup;
    ETG_TRACE_USR1(("dabdrv_mecaIf:vProcess(trMeca_RConStartup)"));
    _bDabLunStartup = TRUE;
    vHandleStartup();
}
tVoid dabdrv_mecaIf::vProcess(trMeca_RConTsuStartup* poRConTsuStartup) {
    ETG_TRACE_USR1(("dabdrv_mecaIf:vProcess(trMeca_RConTsuStartup)"));
    (tVoid)poRConTsuStartup;
    _bMtcLunStartup = TRUE;
    vHandleStartup();
}
tVoid dabdrv_mecaIf::vHandleStartup() {
    ETG_TRACE_USR1(("dabdrv_mecaIf:vHandleStartup()"));
    if (_bMtcLunStartup && _bDabLunStartup) {
        vEnterComState(enComState_WaitRegistration);
    }
}
tVoid dabdrv_mecaIf::vProcess(trMeca_RConRegistration* poRConRegistration) {
    (tVoid)poRConRegistration;
    ETG_TRACE_USR1(("dabdrv_mecaIf:vProcess(trMeca_RConRegistration)"));
    _bConRegistrationValid = TRUE;
    if ( (enComState_WaitRegistration   == enGetComState())  ||
         (enComState_WaitReRegistration == enGetComState()) ) {
        vEnterComState(enComState_Normal);
    } else {
        vEnterComState(enComState_WaitStartup);
    }
}

tVoid dabdrv_mecaIf::vProcess(trMeca_RConDeRegistration* poRConDeRegistration) {
    (tVoid)poRConDeRegistration;
    ETG_TRACE_USR1(("dabdrv_mecaIf:vProcess(trMeca_RConDeRegistration)"));

    if (enComState_WaitDeRegistration == enGetComState()) {
        vEnterComState(enComState_Null);
    } else {
        vEnterComState(enComState_WaitStartup);
    }
}


tVoid dabdrv_mecaIf::vProcess(trMeca_RConPing* poRConPing) {
    (tVoid)poRConPing;
    ETG_TRACE_USR1((" dabdrv_mecaIf::vProcess(trMeca_RConPing)"));
	//vStartPingTimer();
   
}

tVoid dabdrv_mecaIf::vStartPingTimer()
{
	_oPingTmr.vStart(_u32PingTmrValMs);
    trMsgAdrMsgReceived rMsg;
    DAB_vPostMsg(dabdrv_adrIf::instance(), &rMsg);
}

tVoid dabdrv_mecaIf::vProcess(trMsgAdrInput* poMsgAdrInput){
    ETG_TRACE_USR4(("dabdrv_mecaIf:vProcess(trMsgAdrInput)"));


    if (!poMsgAdrInput->_bValid) {
        ETG_TRACE_FATAL(("dabdrv_mecaIf:vProcess(trMsgAdrInput) not valid Fkt-ID=%x",
                            poMsgAdrInput->_rMostHdr._u16FktId));
        return;
    }
    
    vHandleMecaRsp(poMsgAdrInput->_rMostHdr, poMsgAdrInput->_pu8MecaData);

    ETG_TRACE_USR4(("dabdrv_mecaIf:process trMsgAdrInput END")); 
}

tVoid dabdrv_mecaIf::vProcess(dabdrv_msgPingTimer* poPingTimer){
    (tVoid)poPingTimer;
    poPingTimer->vTrace();
    ETG_TRACE_USR1((" dabdrv_mecaIf::vProcess() vSendMecaCommand(trMeca_CConPing)"));
        vSendMecaCommand(trMeca_CConPing());
};

tVoid dabdrv_mecaIf::vProcess( trMeca_RRdmTune *poRTune) {
    (tVoid)poRTune;
    ETG_TRACE_USR4(("dabdrv_mecaIf:process trMeca_RRdmTune"));
    return;
}

tVoid dabdrv_mecaIf::vProcess( trMsgAdrStatus *poAdrStatus ) {
	ETG_TRACE_USR4(("dabdrv_mecaIf:process trMsgAdrStatus"));

	switch (poAdrStatus->enAdrStatus)
	{
	case enAdrStatus_Alive:
		dabdrv_adrIf::instance()->vStartAdrCommunication();
		_oPingTmr.vStart(_u32PingTmrValMs);
		break;
	case enAdrStatus_Dead:
		dabdrv_adrIf::instance()->vStopAdrCommunication();
		_oPingTmr.vStop();
		_bMtcLunStartup = FALSE;
		_bDabLunStartup = FALSE;
		vEnterComState(enComState_WaitStartup);
		break;
	default:
		ETG_TRACE_USR4(("dabdrv_mecaIf:process trMsgAdrStatus not handled"));
	}
}

tVoid dabdrv_mecaIf::vProcess( trMsgIpnError *poIpnError) {
    ETG_TRACE_USR4(("dabdrv_mecaIf:process trMsgIpnError: status=0x%x",
                    ETG_CENUM(tenIpnError, poIpnError->enIpnError)));

    if (poIpnError->enIpnError == enIpnError_AdrLost) {
        ETG_TRACE_USR4(("dabdrv_mecaIf:Adr Lost!"));
        DAB_trErrLogProperty rProperty= dabdrv_properties::instance()->oErrLogProperty.oGet();
        rProperty.enAdrDiag=DAB_tenTestResult_FAILED;
        dabdrv_properties::instance()->oErrLogProperty.vSet(rProperty);
    }
}

#define DABDRV_MECA_CONSTRUCT(id,type)          \
    case id:                                    \
    poMecaInput= OSAL_NEW type;                 \
    break

#define DABDRV_MECA_CONSTRUCT_LOCAL(id,type)          \
    case id:                                    \
    poMecaInput= OSAL_NEW type;                 \
    poCaller=DAB_poGetMsgCaller<dabdrv_mecaIf,type>();    \
    break

// Make meca-message according to MsgCode and Post message to subscribed message-handlers
tVoid dabdrv_mecaIf::vHandleMecaRsp(trMostHdr const &rMostHdr, tU8 const *pu8MecaData) {
    trMsgMecaInput *poMecaInput=OSAL_NULL;
    tBool bRes=FALSE;
    DAB_MsgCallerBase *poCaller=OSAL_NULL;
    static DAB_Profiler rProfiler("vHandleMecaRsp");

    if( rMostHdr._enOpType == enOpType_ERROR ) {
        ETG_TRACE_FATAL(("dabdrv_mecaIf:vHandleMecaRsp ERROR=%04x _u16FktId=%04x"
                        ,ETG_CENUM(tenIpnErrorType,pu8MecaData[0])
                        ,ETG_CENUM(DAB_tenMecaMsgId,rMostHdr._u16FktId)));
        if( pu8MecaData[0] == enIpnError_CLIENT_NOT_REGISTERED ) {
			/*Deblock as registration has to be sent*/
			trMsgDeblockAdrIf rMsgDeblockAdrIf;
			DAB_vPostMsg(dabdrv_adrIf::instance(), &rMsgDeblockAdrIf);
            vEnterComState(enComState_WaitReRegistration);
            return;
        }
		if(pu8MecaData[0] == enIpnError_CLIENT_ALREADY_REGISTERED)
		{	
			vEnterComState(enComState_Normal);
            return;
		}
    }
    if (rMostHdr._u16FBlockId == DABMECA_FBLOCK_ID_DAB) {
        ETG_TRACE_USR1(("dabdrv_mecaIf:vHandleMecaRsp FBlockID=%x _u16FktId=%04x data=%02x"
                        ,rMostHdr._u16FBlockId
                        ,ETG_CENUM(DAB_tenMecaMsgId,rMostHdr._u16FktId)
                        ,ETG_LIST_LEN(rMostHdr._u16MecaLen), ETG_LIST_PTR_T8(pu8MecaData)));
        //As part of bug 545323, print all the received data whos size is more than 223 as TTFIS prints only 223 bytes of received data
        tUShort totalMecaLen = rMostHdr._u16MecaLen, maxLenToPrint = 223;
        if (rMostHdr._u16MecaLen > maxLenToPrint)
        {
            string mecaDataRspStr((const char*)pu8MecaData,rMostHdr._u16MecaLen);
            auto itr = mecaDataRspStr.begin();
            tUShort pos = 0;
            while(itr < mecaDataRspStr.end())
            {
                string data = mecaDataRspStr.substr(pos,maxLenToPrint);
                tUShort len = (totalMecaLen  > maxLenToPrint) ? maxLenToPrint : totalMecaLen;
                ETG_TRACE_USR1(("data:%02x",ETG_LIST_LEN(len),ETG_LIST_PTR_T8((const char*)data.c_str())));
                if (totalMecaLen < maxLenToPrint)
                    break;
                totalMecaLen = (tUShort)(totalMecaLen - maxLenToPrint);
                itr += maxLenToPrint;
                pos = (tUShort)(pos + maxLenToPrint);
            }

        }
    } else {
        ETG_TRACE_USR1(("dabdrv_mecaIf:vHandleMecaRsp FBlockID=%x _u16FktId=%04x data=%02x"
                        ,rMostHdr._u16FBlockId
                        ,ETG_CENUM(DAB_tenMecaTsuMsgId,rMostHdr._u16FktId)
                    ,ETG_LIST_LEN(rMostHdr._u16MecaLen), ETG_LIST_PTR_T8(pu8MecaData)));
    } 

    if (rMostHdr._u16FBlockId == DABMECA_FBLOCK_ID_DAB) {
	#ifndef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC  //PSARCC30 -2112
	fc_dabtuner_config *poConfig=fc_dabtuner_config::instance();
	tBool bEPGEnable = poConfig->bGetDABEPGEnable();	
	tBool bSLSEnable = poConfig->bGetDABSLSEnable();

	if(bEPGEnable || bSLSEnable){
	#endif

	/* copy msg data to property*/
		
		ETG_TRACE_USR4(("dabdrv_mecaIf:vHandleMecaRsp Data size=%d",  rMostHdr._u16MecaLen ));
		DAB_trADRMsgProperty rProperty= dabdrv_properties::instance()->oADRMsgProperty.oGet();

		if ((rMostHdr._u16MecaLen != 0)&&
		((rMostHdr._u16FktId==MECA_DISPATCHER_R_DDM_SELECT_URI)||
		((rMostHdr._u16FktId==MECA_DISPATCHER_R_DB_QUERY)&&(pu8MecaData[1]==0xFF))
		||
		(rMostHdr._u16FktId==MECA_DISPATCHER_R_DDM_TIMESTAMP)||
		(rMostHdr._u16FktId==MECA_DISPATCHER_R_DDM_DATACHANNEL)|| 
		(rMostHdr._u16FktId==MECA_DISPATCHER_R_RDM_GET_RDM_STATUS)||
		(rMostHdr._u16FktId==MECA_DISPATCHER_R_RDM_AUDIO_GET_ASID)||
		(rMostHdr._u16FktId==MECA_DISPATCHER_R_RDM_AUDIO_GET_SCIDI))
		){
			rProperty.vectorADRMsg.clear();
			rProperty.u16FktId = rMostHdr._u16FktId;
			for (tU16 i=0;i< rMostHdr._u16MecaLen;i++) {
				//ETG_TRACE_USR4(("dabdrv_mecaIf:vHandleMecaRsp Data index %d : size=%d",
                 // i, rMostHdr._u16MecaLen ));
				rProperty.vectorADRMsg.push_back(pu8MecaData[i]);
			}			
			dabdrv_properties::instance()->oADRMsgProperty.vSet(rProperty);
		}
	#ifndef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
	}
	#endif
        switch (rMostHdr._u16FktId) {
            DABDRV_MECA_CONSTRUCT_LOCAL(enMecaFktId_ConStartup, trMeca_RConStartup);
            DABDRV_MECA_CONSTRUCT_LOCAL(enMecaFktId_ConPing, trMeca_RConPing);
            DABDRV_MECA_CONSTRUCT_LOCAL(enMecaFktId_ConRegistration, trMeca_RConRegistration);
            DABDRV_MECA_CONSTRUCT_LOCAL(enMecaFktId_ConDeRegistration, trMeca_RConDeRegistration);

            DABDRV_MECA_CONSTRUCT(enMecaFktId_ConSetWatchDog, trMeca_RConSetWatchDog);



            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_REJECT, trMeca_RDbReject);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_GET_DB_ENSEMBLE_LIST, trMeca_RDbGetDbEnsembleList);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_ENSEMBLE_GET_LABEL, trMeca_RDbEnsembleGetLabel);
            //        DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_ENSEMBLE_GET_SHORT_LABEL, trMeca_RDbEnsembleGetShortLabel);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_ENSEMBLE_GET_INT_TABLE_ID, trMeca_RDbEnsembleGetIntTableId);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_ENSEMBLE_GET_INFO, trMeca_RDbEnsembleGetInfo);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_ENSEMBLE_GET_SERVICE_LIST, trMeca_RDbEnsembleGetServiceList);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_ENSEMBLE_GET_REGION_LIST, trMeca_RDbEnsembleGetRegionList);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_ENSEMBLE_GET_REGION_LABEL, trMeca_RDbEnsembleGetRegionLabel);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_ENSEMBLE_GET_FREQUENCY_LIST, trMeca_RDbEnsembleGetFrequencyList);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_GET_DB_SERVICE_LIST, trMeca_RDbGetServiceList);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_SERVICE_GET_LABEL, trMeca_RDbServiceGetLabel);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_SERVICE_GET_COMPONENT_LIST, trMeca_RDbServiceGetComponentList);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_SERVICE_GET_ANNOUNCEMENT_SUPPORT, trMeca_RDbServiceGetAnnouncementSupport);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_SERVICE_GET_PTY, trMeca_RDbServiceGetPty);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_SERVICE_GET_AF, trMeca_RDbServiceGetAF);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_SERVICE_GET_INFO, trMeca_RDbServiceGetInfo);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_SERVICE_GET_LINK_INFO, trMeca_RDbServiceGetLinkInfo);
            //        DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_PROGRAMME_SERVICE_GET_SHORT_LABEL, trMeca_RDbProgrammeServiceGetShortLabel);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_SERVICE_COMPONENT_GET_SCIDI, trMeca_RDbServiceComponentGetSCIDI);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_SERVICE_COMPONENT_GET_LABEL, trMeca_RDbServiceComponentGetLabel);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_SERVICE_COMPONENT_GET_INFO, trMeca_RDbServiceComponentGetInfo);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_SERVICE_COMPONENT_GET_USER_APPLICATION_LIST, trMeca_RDbServiceComponentGetUserApplicationList);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_SERVICE_COMPONENT_GET_USER_APPLICATION_DATA, trMeca_RDbServiceComponentGetUserApplicationData);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_COMPONENT_GET_INFO, trMeca_RDbComponentGetInfo);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_SUBCHANNEL_GET_INFO, trMeca_RDbSubchannelGetInfo);
            //        DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_REGION_GET_LIST, trMeca_RDbRegionGetList);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_SET_AUTO_NOTIFICATION, trMeca_RDbSetAutoNotification);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_FILTERID, trMeca_RDbFilterId);
			 DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_CONFIG, trMeca_RDbConfig);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_QUERY, trMeca_RDbQuery);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DB_QUERY_TRIGGER, trMeca_RDbQueryTrigger);

            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_SET_MODE, trMeca_RRdmSetMode);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_SET_FILTER, trMeca_RRdmSetFilter);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_GET_SIGNAL_QUALITY, trMeca_RRdmGetSignalQuality);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_QUALITY_CONFIG, trMeca_RRdmQualityConfig);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_GET_FREQUENCY, trMeca_RRdmGetFrequency);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_TUNE,trMeca_RRdmTune);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_TUNE_FREQUENCY, trMeca_RRdmTuneFrequency);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_TUNE_EID, trMeca_RRdmTuneEid);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_GET_EID, trMeca_RRdmGetEid);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_GET_FREQUENCY_LABEL, trMeca_RRdmGetFrequencyLabel);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_SET_DAB_FREQUENCY_TABLE, trMeca_RRdmSetDabFrequencyTable);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_GET_DAB_FREQUENCY_TABLE, trMeca_RRdmGetDabFrequencyTable);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_GET_DAB_AUDIO_QUALITY, trMeca_RRdmGetDabAudioQuality);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_GET_AUDIO_STATUS, trMeca_RRdmGetAudioStatus);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_AUDIO_GET_ASID, trMeca_RRdmAudioGetAsid);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_AUDIO_PLAY_ASID, trMeca_RRdmAudioPlayAsid);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_AUDIO_PLAY_SERVICE, trMeca_RRdmAudioPlayService);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_AUDIO_PLAY_COMPONENT, trMeca_RRdmAudioPlayComponent);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_AUDIO_GET_SCIDI, trMeca_RRdmAudioGetScidi);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_AUDIO_COMPONENT_PLAY, trMeca_RRdmAudioComponentPlay);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_SET_DYNAMIC_RANGE_CONTROL, trMeca_RRdmSetDynamicRangeControl);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_GET_CURR_REGIONS, trMeca_RRdmGetCurrRegions);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_GET_COARSE_POSITION, trMeca_RRdmGetCoarsePosition);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_GET_CURR_TII_LIST, trMeca_RRdmGetCurrTiiList);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_GET_RDM_STATUS, trMeca_RRdmGetRdmStatus);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_GET_RDM_INFO, trMeca_RRdmGetRdmInfo);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_SET_AUTO_NOTIFICATION, trMeca_RRdmSetAutoNotification);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_LEARN, trMeca_RRdmLearn);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_GET_FREQUENCY_INFO, trMeca_RRdmFrequencyInfo);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_SERV_FOLLOW_SET_MODE, trMeca_RRdmServFollowSetMode);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_SERV_FOLLOW_ACTIVE, trMeca_RRdmServFollowActive);
		    DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_SCAN, trMeca_RRdmScan);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_STATION_PLAY, trMeca_RRdmStationPlay);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_STATION_STORE, trMeca_RRdmStationStore);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_STATION_GET_PRESET_STATION, trMeca_RRdmStationGetPresetStation);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_STATION_SET_PRESET_STATION, trMeca_RRdmStationSetPresetStation);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_STATION_SET_USER, trMeca_RRdmStationSetUser);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_TUNE_FREQUENCY_LABEL, trMeca_RRdmTuneFrequencyLabel);
			DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_SERV_FOLLOW_SET_ONESHOT_MODE, trMeca_RRdmServFollowSetOneShotMode);

            //        DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_STATION_GET_PRESET_STATION_NUMBER, trMeca_RRdmStationGetPresetStationNumber);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_RDM_STATION_TRAVELSTORE, trMeca_RRdmStationTravelStore);
                
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_ANS_ENABLE_ANNO, trMeca_RAnsEnableAnno);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_ANS_ANNO_ACTIVE_STATUS, trMeca_RAnsAnnoActiveStatus);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_ANS_SELECT_ANNO, trMeca_RAnsSelectAnno);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_ANS_SET_ANNO_MODE, trMeca_RAnsSetAnnoMode);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_ANS_GET_ANNO_INFO, trMeca_RAnsGetAnnoInfo);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_ANS_CONFIG_ANNO, trMeca_RAnsConfigAnno);

            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_AUD_SET_CONCEALMENT_LEVEL, trMeca_RAudSetConcealmentLevel);
			DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_PTYM_FILTER, trMeca_RPTYMSetFilter);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DDM_PACKET, trMeca_RDdmPacket);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DDM_DATAGROUP, trMeca_RDdmDataGroup);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DDM_PAD_DATAGROUP, trMeca_RDdmPadDataGroup);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DDM_STREAM, trMeca_RDdmStream);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DDM_PAD, trMeca_RDdmPad);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DDM_APP_TYPE_PAD, trMeca_RDdmAppTypePad);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DDM_RADIOTEXT_CURRENT, trMeca_RDdmRadioTextCurrent);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DDM_RADIOTEXT_CUR_CMD, trMeca_RDdmRadioTextCurrentCmd);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DDM_TMC_SC, trMeca_RDdmTmcSc);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DDM_TMC_CURRENT, trMeca_RDdmTmcCurrent);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DDM_TMC_EVENT, trMeca_RDdmTmcEvent);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DDM_RDI_SUBCH, trMeca_RDdmRdiSubCh);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DDM_SELECT_URI, trMeca_RDdmSelectUri);

            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_SYS_POWER, trMeca_RSysPower);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_SYS_GET_VERSION_INFO, trMeca_RSysVersionInfo);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_SYS_STORE, trMeca_RSysStore);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_SYS_RESTORE, trMeca_RSysReStore);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_SYS_STORAGE_PUT_FRAME, trMeca_RSysStoragePutFrame);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_SYS_STORAGE_FORMAT, trMeca_RSysStorageFormat);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_SYS_DIAG, trMeca_RSysDiag);


            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_TSU_SET_MODE, trMeca_RTsuSetMode);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_TSU_SWITCH_TUNER, trMeca_RTsuSwitchTuner);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_TSU_SWITCH_RECOMMENDATION, trMeca_RTsuSwitchRecommendation);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_TSU_SET_CONFIG, trMeca_RTsuSetConfig);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_TSU_SELECT_PSID, trMeca_RTsuSelectPsid);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_TSU_GET_QUALITY, trMeca_RTsuGetQuality);
                
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_EXP_SELECT, trMeca_RExpSelect);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_EXP_ID_VALUE_LABEL, trMeca_RExpIdValueLabel);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_EXP_ID_VALUE_UINT32, trMeca_RExpIdValueU32);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_EXP_ID_DESCRIPTION, trMeca_RExpIdDescription);

            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_R_DDM_DATACHANNEL, trMeca_RDdmDataChannel);
            default:
                break;
        }
    }
    else if (rMostHdr._u16FBlockId == DABMECA_FBLOCK_ID_DAB_MTC) {
        if(rMostHdr._enOpType == enOpType_ERROR)
        {
            return;
        }
        fc_dabtuner_config *poConfig=fc_dabtuner_config::instance();
        if( poConfig != nullptr)
        {
            if(poConfig->bIsSBRVariant())
            {
                switch(rMostHdr._u16FktId) {
                    DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_C_MTC_SURVEILLANCE, trMeca_RFMSurveillanceInfos);
                    default:
                        break;
                }
            }
            else
            {
                switch(rMostHdr._u16FktId) {
                    DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_C_MTC_SURVEILLANCE,trMeca_RMtcSurveillanceMonitor);
                    default:
                        break;
                }
            }
        }

        switch (rMostHdr._u16FktId) {
            DABDRV_MECA_CONSTRUCT_LOCAL(MECA_DISPATCHER_C_MTC_STARTUP, trMeca_RConTsuStartup);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_C_MTC_SET_MODE, trMeca_RMtcSetMode);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_C_MTC_SET_CONFIG, trMeca_RMtcSetConfig);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_C_MTC_SET_AUDIO, trMeca_RMtcSetAudio);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_C_MTC_TUNER_STATUS, trMeca_RMtcTunerStatus);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_C_MTC_TESTMODE, trMeca_RMtcTestmode);
            DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_C_MTC_SF_RESTART, trMeca_RMtcSfRestart);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_C_MTC_REQUEST_SWITCH, trMeca_RMtcRequestSwitch);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_C_MTC_ANNOUNCEMENT, trMeca_RMtcAnnouncementState);
            //DABDRV_MECA_CONSTRUCT(MECA_DISPATCHER_C_MTC_SURVEILLANCE, trMeca_RMtcSurveillanceMonitor);
            default:
                break;
        }

    }
	if(!_oPingTmr.bIsRunning())
	{
		ETG_TRACE_USR2(("dabdrv_mecaIf::vHandleMecaRsp:RConPing not recieved"));
		try
		{
			vStartPingTimer();
		}
		catch(std::bad_alloc)
		{
			ETG_TRACE_USR1(("Exception caught"));
		}
	}

    rProfiler.vStart(rMostHdr._u16FktId);
    if (poMecaInput) {
        // parse received byte-stream into poMecaInput by calling virtual bParse
        bRes = poMecaInput->bParse(rMostHdr, pu8MecaData);
        if (bRes) {
            if (etg_bIsTraceActive(FC_DABTUNER_TR_DRV_MECAIF, (tU16)TR_LEVEL_USER_2)) {
                ETG_TRACE_USR2(("dabdrv_mecaIf::vHandleMecaRsp: received:"));
                poMecaInput->vTrace();
            }
            if (enComState_Normal != enGetComState() ) {
                if (OSAL_NULL != poCaller) {
                    // all messages are handled in this class
                    poGetDispatcher()->vCallMsg(poCaller, poMecaInput);
                }
            }
            else {
                DAB_vCallMsg(poMecaInput);
            }
            vHandleBlocking(rMostHdr._u16FktId);
        }
        else {
            ETG_TRACE_ERR(("dabdrv_mecaIf:vHandleMecaRsp could not parse _u16FktId=%04x data=%*p",
                            ETG_CENUM(DAB_tenMecaMsgId,rMostHdr._u16FktId),
                            ETG_LIST_LEN(rMostHdr._u16MecaLen), ETG_LIST_PTR_T8(pu8MecaData)));
        }
        OSAL_DELETE poMecaInput;
    } else {
        ETG_TRACE_ERR(("dabdrv_mecaIf:vHandleMecaRsp _u16FktId=0x%x not SUPPORTED", ETG_CENUM(DAB_tenMecaMsgId,rMostHdr._u16FktId)));

    }
    rProfiler.u32GetMs();
}

tVoid dabdrv_mecaIf::vHandleBlocking(tU16 u16RcvFktId) const {
	fc_dabtuner_config *poConfig=fc_dabtuner_config::instance();
	if(poConfig == nullptr)
	{
		return;
	}

	tU16 u16WaitAnswer=dabdrv_adrIf::instance()->u16GetWaitAnswer();
	if (!u16WaitAnswer) {
		// do nothing
	}
	else if (u16WaitAnswer==0xFFFF) {
		ETG_TRACE_USR1(("dabdrv_mecaIf:vHandleBlocking:BLOCKED"));

	}
	else if (u16RcvFktId==u16WaitAnswer) {
		ETG_TRACE_USR1(("dabdrv_mecaIf:vHandleBlocking(received WaitAnswer=%04x)",
				ETG_CENUM(DAB_tenMecaMsgId,u16WaitAnswer)));
		trMsgDeblockAdrIf rMsgDeblockAdrIf;
		DAB_vPostMsg(dabdrv_adrIf::instance(), &rMsgDeblockAdrIf);
	}
	else {
		ETG_TRACE_USR1(("dabdrv_mecaIf:vHandleBlocking: WaitAnswer=(%04x) not received (%04x)",
				ETG_CENUM(DAB_tenMecaMsgId,u16WaitAnswer), ETG_CENUM(DAB_tenMecaMsgId,u16RcvFktId)));
	}
};

tVoid dabdrv_mecaIf::vProcess(trMsgDeBlockMeca* poDeBlockMeca) {

	fc_dabtuner_config *poConfig=fc_dabtuner_config::instance();
	if(poConfig == nullptr)
	{
		return;
	}
	(tVoid)poDeBlockMeca;
	ETG_TRACE_USR1(("dabdrv_mecaIf:trMsgDeBlockMeca"));
	trMsgDeblockAdrIf rMsgDeblockAdrIf;
	DAB_vPostMsg(dabdrv_adrIf::instance(), &rMsgDeblockAdrIf);
}

tVoid dabdrv_mecaIf::vProcess(trMsgSrvCmdCriticalVoltage* poCriticalVoltage) {

	ETG_TRACE_USR1(("dabdrv_mecaIf:poCriticalVoltage: bIsCritical=%d",
			poCriticalVoltage->bIsCritical));

	if (poCriticalVoltage->bIsCritical) {
		dabdrv_adrIf::instance()->vStopAdrCommunication();
		_bMtcLunStartup = FALSE;
		_bDabLunStartup = FALSE;
		_oPingTmr.vStop();
	} else {
		dabdrv_adrIf::instance()->vStartAdrCommunication();
		_oPingTmr.vStart(_u32PingTmrValMs);
	}
}


 


