/************************************************************************
 * FILE:       dabdrv_chnList.cpp
 * PROJECT:        g3g
 * SW-COMPONENT:   
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of dabdrv_chnList
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      		 | Author                       | Modification
   
				
 *************************************************************************/

#include "dabdrv_main.hpp"
#include "dabdrv_mecaIf.h"
#include "dabdrv_presets.hpp"
#include "dabdrv_chnList.hpp"
#include "dabdrv_chnInfo.hpp"
#include "dabdrv_presets.hpp"
#include "dabdrv_rdm.hpp"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS FC_DABTUNER_TR_DRV_SRVLST 
#include "trcGenProj/Header/dabdrv_chnList.cpp.trc.h"
#endif

// this class maintains a channellist filled with data from meca_db

/* interface for channel-selection:
   1.) get channel-information by intId, PSID, label
   2.) get newChannel(dir, steps, 

*/


#ifndef __RFC3072_HELPER_H__
#include "rfc3072_helper.h"
#endif


//#define OLD_STYLE_LIST_IMPLEMENTATION 1


using namespace DAB;


namespace DAB {

    // Timer
    struct trMsgChnListUpdateTimer:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_MSG, 
                                "trMsgChnListUpdateTimer"));
        };
    };

    struct trMsgChnListAfterLearnTimer:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_MSG, 
                                "trMsgChnListAfterLearnTimer"));
        };
    };


    struct trMsgChnList1sTick:
        public DAB_Message
    {
        DAB_DISPATCH_IMPL
        virtual tVoid vTrace() const {
            ETG_TRACE_USR1_CLS((FC_DABTUNER_TR_UTIL_MSG, 
                                "trMsgChnList1sTick"));
        };
    };








dabdrv_chnList::dabdrv_chnList():
    _mapEnsembleInfo(),
    _mapServiceInfo(),
    _mapDynSrvByLabel(),
	_mapQueryTagInfo(),
	_setDynSrvById(),
    _mapFrozenIndexBySrv(),
    _vectorFrozenSrvList(),
    _vectorSDSSrvList(),
	_mapFrozenIndexByEnsemble(),
	_vectorFrozenEnsmbleList(),
	_mapDynEnsByLabel(),
	m_u16EnsembleIndex(0),
	_vectorPTYCode(),
	_u32PtyFilter(PTY_MASK),
	_u8PtyFilterIndex(0),
	_u8NumEnsembles(0),
	_u8EnsembleCounter(0),
	_u8TagCounter(enMeca_TagID_Global_List_START),
	_bQueryEnsembleList(FALSE),
	_bBlockEnsembleListQuery(FALSE),
	_bEnableSDS(FALSE)
{
    _u32ChnListUpdateTimerValMs = DAB_CHN_LIST_UPDATE_TIMER_MS;
    _vectorFrozenSrvList.clear();

    _bFrozenChnListOpen = FALSE;
	_bFrozenEnsmbleListOpen = FALSE;

    _enListCheckState=enListState_Done;
    _enForcedUpdateState=enForcedUpdateState_Idle;
    _u32UpdateId=0;
    _u32NumChanges=1;
	
	_enChnListType = enChnList_Global;

	 m_enEnsListSortingType=enEnsembleListSortingType_LABEL;
	 m_enSrvListSortingType=enServiceListSortingType_LABEL;

    // commands from dabdrv_main
    vSubscribe<trMsgDrvStartComponent>();
    vSubscribe<trMsgDrvStopComponent>();

    vSubscribe<trMsgDrvCmdChnListPsidMonitor>();
    vSubscribe<trMsgDrvRspCurrentEnsemble>();
    vSubscribe<trMsgDrvEvtLearnDone>();
    vSubscribe<trMsgDrvEvtForcedLearnStart>();
	vSubscribe<trMsgSrvDataList>();
	vSubscribe<trMsgSrvCmdListUpdateStatus>();

    // meca-responses
#ifdef OLD_STYLE_LIST_IMPLEMENTATION
    vSubscribe<trMeca_RDbGetDbEnsembleList>();
    //vSubscribe<trMeca_RDbEnsembleGetFrequencyList>();
    vSubscribe<trMeca_RDbEnsembleGetServiceList>();
    vSubscribe<trMeca_RDbEnsembleGetInfo>();
    vSubscribe<trMeca_RDbServiceGetInfo>();
#endif
    // commands from server
    vSubscribe<trMsgSrvCmdSetChnList>();
	vSubscribe<trMsgSrvCmdSetEnsmbleList>();

	vSubscribe<trMeca_RPTYMSetFilter>();
	vSubscribe<trMeca_RDbQuery>();
	vSubscribe<trMeca_RRdmAudioPlayAsid>();
	vSubscribe<trMeca_RDbQueryTrigger>();
	vSubscribe<trMsgSrvValidateEPGData>();
}


tVoid dabdrv_chnList::vTraceSrvInfo(trChnListElem const &rSrvInfo) const{
    ETG_TRACE_USR4(("  %08x  %d %2d %17s",
                    rSrvInfo.rMecaId.u32GetSID(),
                    rSrvInfo.u16Scids,
                    rSrvInfo.bReception,
                    rSrvInfo.rLabel.pcGetCString()));
    
}

tVoid dabdrv_chnList::vTraceSrvInfo(trChnListServiceInfo const &rSrvInfo) const{
    vector<tU32> au32EnsIds;
    DAB_FOREACH_CONST(set<trMecaEnsemble>,ensIter,rSrvInfo.lEnsembles) {
        au32EnsIds.push_back(ensIter->u32GetID());
    }
    ETG_TRACE_USR4(("  %08x %d %d  0x%02x 0x%04x %d   %d  %17s ",
                    rSrvInfo.rProgService.u32GetSID(), 
                    rSrvInfo.u16Scids,
                    rSrvInfo.bPSFlag,
                    rSrvInfo.u8ReceptionQuality,
                    rSrvInfo.u16AvailableAnnoTypesMask,
                    rSrvInfo.bLabelPresent,
					rSrvInfo.u8PTY,
                    rSrvInfo.bLabelPresent ? rSrvInfo.rLabel.pcGetCString() : ""));
  ETG_TRACE_USR4(("list:%*p", ETG_LIST_LEN((tU16)(au32EnsIds.size())), ETG_LIST_PTR_T32(&au32EnsIds[0]))); 
}

tVoid dabdrv_chnList::vTraceEnsembleInfo(trEnsembleListElem const &rEnsInfo) const{

		 ETG_TRACE_USR4(("  %06u %08x %17s  %9s %d %d ",
			rEnsInfo.u32Frequency,
			rEnsInfo.rEnsemble.u16GetEID(),
			rEnsInfo.rEnsembleLabel.pcGetCString(),
			rEnsInfo.rEnsembleLabel.pcGetCStringShort(),
			rEnsInfo.u8NumberOfAudioServices,
			rEnsInfo.u8NumberOfDataServices));
}
tVoid dabdrv_chnList::vTraceAll() const {
    if (!bDoTraceUsr1()) {
        return;
    }
    ETG_TRACE_USR4(("dabdrv_chnList::vTraceAll: _u32NumChanges=%d"
                    "_bFrozenChnListOpen=%d _enForcedUpdateState=%d",
                    _u32NumChanges,
                    _bFrozenChnListOpen,
                    ETG_CENUM(tenForcedUpdateState, _enForcedUpdateState)));
    if (!bDoTraceUsr4()) {
        return;
    }
    ETG_TRACE_USR4(("dabdrv_chnList::vTraceAll:_mapEnsembleInfo"));
    ETG_TRACE_USR4(("  Ens      Qual Freq   Services"));
    DAB_FOREACH_MAP_CONST(trMecaEnsemble, trChnListEnsembleInfo,iEnsemblesIter, _mapEnsembleInfo) {
        const set<trMecaProgrammeService>&lServiceList=iEnsemblesIter->second.lServiceList;
        vector<tU32> au32SrvIds;
        DAB_FOREACH_CONST (set<trMecaProgrammeService>, srvIter, lServiceList) {
            au32SrvIds.push_back(srvIter->u32GetSID());
        }
        ETG_TRACE_USR4(("  %08x %d %02x   %6u %*p",
                        iEnsemblesIter->first.u32GetID(),
                        iEnsemblesIter->second.bReception,
                        iEnsemblesIter->second.u8Quality,
                        iEnsemblesIter->second.u32BestFreq,
                        ETG_LIST_LEN((tU16)(au32SrvIds.size())), ETG_LIST_PTR_T32(&au32SrvIds[0])));
    }

	ETG_TRACE_USR4(("dabdrv_chnList::vTraceAll:_vectorFrozenEnsmbleList"));

	ETG_TRACE_USR4(("  EnsFreq  InfoPresent FoundInList Quality labelPresent label EnsList"));
	 for (tU32 i=0;i<_vectorFrozenEnsmbleList.size();++i) {
		 vTraceEnsembleInfo(_vectorFrozenEnsmbleList[i]);     
    }

    

    ETG_TRACE_USR4(("dabdrv_chnList::vTraceAll:_mapServiceInfo"));
    ETG_TRACE_USR4(("  Srv      IPr Qual AnMsk  lbPr  label: EnsList "));
    DAB_FOREACH_MAP_CONST(trChnListKey, trChnListServiceInfo,iterSrvMap, _mapServiceInfo) {
        //        ETG_TRACE_USR4(("MeCaId=0x%08x", (*iterSrvMap).first.u32GetSID()));
        vTraceSrvInfo((*iterSrvMap).second);
    }

    ETG_TRACE_USR4(("dabdrv_chnList::vTraceAll:_mapDynSrvByLabel"));
    ETG_TRACE_USR4(("  Label                PSID"));

    DAB_FOREACH_MMAP_CONST(trMecaLabel, trChnListKey, iterDynSrv, _mapDynSrvByLabel) {
        ETG_TRACE_USR4(("  %17s %08x %d",
                        iterDynSrv->first.pcGetCString(),
                        iterDynSrv->second._rMecaId.u32GetSID(),
                        iterDynSrv->second._u16Scids));
        
    }

    ETG_TRACE_USR4(("dabdrv_chnList::vTraceAll:_vectorFrozenSrvList"));

    ETG_TRACE_USR4(("  Srv  InfoPresent FoundInList Quality labelPresent label EnsList ptycode"));
    for (tU32 i=0;i<_vectorFrozenSrvList.size();++i) {
        vTraceSrvInfo(_vectorFrozenSrvList[i]);

    }

    ETG_TRACE_USR4(("dabdrv_chnList::vTraceAll:_mapFrozenIndexBySrv (Srv : index)"));

    DAB_FOREACH_MAP_CONST(trChnListKey, tU16, iterIndexBySrv, _mapFrozenIndexBySrv) {
        ETG_TRACE_USR4(("0x%08x %d: %d",
                        iterIndexBySrv->first._rMecaId.u32GetSID(),
                        iterIndexBySrv->first._u16Scids,
                        iterIndexBySrv->second));
    }

	ETG_TRACE_USR4(("dabdrv_chnList::vTraceAll:_mapDynEnsByLabel"));
    ETG_TRACE_USR4(("  Label                EID"));

    DAB_FOREACH_MMAP_CONST(trMecaLabel, trMecaEnsemble, iterDynEns, _mapDynEnsByLabel) {
        ETG_TRACE_USR4(("  %17s %08x ", 
                        iterDynEns->first.pcGetCString(),
                        iterDynEns->second.u32GetID()));
        
    }

	ETG_TRACE_USR4(("dabdrv_chnList::vTraceAll:_mapFrozenIndexByEnsemble (Ensemble : index)"));
	DAB_FOREACH_MAP_CONST(trMecaEnsemble, tU16, iterIndexByEns, _mapFrozenIndexByEnsemble) {
        ETG_TRACE_USR4(("0x%08x : %d",
                        iterIndexByEns->first.u32GetID(),
                        iterIndexByEns->second));
    }


    ETG_TRACE_USR4(("dabdrv_chnList::vTraceAll:END"));
    
}

tVoid dabdrv_chnList::vInit() {
    // timer can not be created in constructor
    _oChnListUpdateTimer.vInit(instance(),trMsgChnListUpdateTimer());
    _oAfterLearnTimer.vInit(instance(),trMsgChnListAfterLearnTimer());
}

tVoid dabdrv_chnList::vDeInit() {
    _oChnListUpdateTimer.vDeInit();
    _oAfterLearnTimer.vDeInit();
}

tVoid dabdrv_chnList::vProcess(trMsgDrvStartComponent*) {
    _rMonitoredSrvId.vInvalidate();
    //_oChnListUpdateTimer.vStart(_u32ChnListUpdateTimerValMs);

    ETG_TRACE_USR4(("dabdrv_advisory::trMsgDrvStartComponent: Query for lists"));

    //Evaluate the ensemble list for the first time at startup
    trMsgDrvDbQuery rDbQueryEnsEval(Query_EnsembleList_Evaluate);
	DAB_vCallMsg(rDbQueryEnsEval);


    //Prepare the initial ensemble list
	trMsgDrvDbQuery rDbQueryEns(Query_EnsembleList_Prepare);
	DAB_vCallMsg(rDbQueryEns);

	 //Trigger for ensemble list
	trMsgDrvDbQueryTrigger rDbQueryTriggerEns(QueryTrigger_EnsembleList);
	DAB_vCallMsg(rDbQueryTriggerEns);

}

tBool dabdrv_chnList::bDynSrvListUpToDate() const {
    return (_u32NumChanges==0);
}

tBool dabdrv_chnList:: bIsEnsembleOk(trChnListEnsembleInfo &rEnsInfo) const {

	tBool bRes= bShowEnsemble(rEnsInfo.u8Quality) &&
		rEnsInfo.rLabel.bLabelValid;

    ETG_TRACE_USR4(("dabdrv_chnList::bIsEnsembleOk:ID=0x%08x u8ReceptionQuality=%d bShow=%d bLabelValid=%d bRes=%d",
					rEnsInfo.rEnsemble.u32GetID(),
					rEnsInfo.u8Quality,
                    (tU8)bShowEnsemble(rEnsInfo.u8Quality),
                   rEnsInfo.rLabel.bLabelValid,
                    bRes));
    return  bRes;
}

tBool dabdrv_chnList::bIsChnOk(trChnListServiceInfo &rSrvInfo) const {
    tBool bRes= rSrvInfo.bServiceInfoPresent && 
        bShowEnsemble(rSrvInfo.u8ReceptionQuality) &&
        rSrvInfo.rLabel.bLabelValid;
    ETG_TRACE_USR4(("dabdrv_chnList::bIsChnOk:SID=0x%08x u8ReceptionQuality=%x bShow=%d bServiceInfoPresent=%d bLabelValid=%d bRes=%d",
                    rSrvInfo.rProgService.u32GetSID(),
                    rSrvInfo.u8ReceptionQuality,
                    (tU8)bShowEnsemble(rSrvInfo.u8ReceptionQuality),
                    rSrvInfo.bServiceInfoPresent,
                    rSrvInfo.rLabel.bLabelValid,
                    bRes));
    ETG_TRACE_USR4(("dabdrv_chnList::bIsChnOk:PtyCode=%d,Ptyfilter=%x",rSrvInfo.u8PTY,_u32PtyFilter));

    if(_u32PtyFilter != PTY_MASK ){
        tU8 u8PTYCode = rSrvInfo.u8PTY;

        if((1<<u8PTYCode)&_u32PtyFilter){
            bRes &=TRUE;
        }
        else
        {
            bRes= FALSE;
        }
        
    }


    return  bRes;

}

 tVoid dabdrv_chnList::vPrepareDynEnsList()
 {
	 ETG_TRACE_USR4(("dabdrv_chnList::vPrepareDynEnsList:START"));
 
    _mapDynEnsByLabel.clear();

    tU32 u32NumAdded=0;
    tU32 u32NumIgnored=0;

	//vTraceAll();

    DAB_FOREACH_MAP(trMecaEnsemble, trChnListEnsembleInfo,iterEnsMap, _mapEnsembleInfo) {
        trChnListEnsembleInfo &rEnsInfo = (*iterEnsMap).second;

        if (bIsEnsembleOk(rEnsInfo)) {
            ETG_TRACE_USR4(("dabdrv_chnList::vPrepareDynEnsList:ADDING:EnsId=0x%08x label=%17s",
				rEnsInfo.rEnsemble.u32GetID(),
				rEnsInfo.rLabel.pcGetCString()));
			if(m_enEnsListSortingType == enEnsembleListSortingType_EID)
			{
				_setDynEnsById.insert(rEnsInfo.rEnsemble);
			}
			else if(m_enEnsListSortingType == enEnsembleListSortingType_Freq)
			{
				_mapDynEnsByFreq.insert(pair<tU32, trMecaEnsemble>(rEnsInfo.u32BestFreq,rEnsInfo.rEnsemble));
			}
			else
			{
				_mapDynEnsByLabel.insert(pair<trMecaLabel, trMecaEnsemble>(rEnsInfo.rLabel, rEnsInfo.rEnsemble));
			}
            u32NumAdded++;
        }
        else {
            ETG_TRACE_USR4(("dabdrv_chnList::vPrepareDynEnsList:IGNORE:EnsId=0x%08x label=%17s (qual=%d, labelValid=%d)",
                            rEnsInfo.rEnsemble.u32GetID(),
                            rEnsInfo.rLabel.pcGetCString(),
							rEnsInfo.u8Quality,
                            rEnsInfo.rLabel.bLabelValid));
            u32NumIgnored++;
        }
    }
    ETG_TRACE_USR4(("dabdrv_chnList::vPrepareDynEnsList:END(updated: u32NumAdded=%d u32NumIgnored=%d)",
                    u32NumAdded,
                    u32NumIgnored));

 }
// todo: alway keep current channel
tVoid dabdrv_chnList::vPrepareDynChnList() {
    ETG_TRACE_USR4(("dabdrv_chnList::vPrepareDynChnList:START"));
    if (bDynSrvListUpToDate()) {
        ETG_TRACE_USR4(("dabdrv_chnList::vPrepareDynChnList:END(uptodate)"));
        // list is still up to date
        return;
    }
    // clear all markers for changes
    _u32NumChanges=0;
    _mapDynSrvByLabel.clear();
   
	trMecaEnsemble rCurEnsemble;

		rCurEnsemble = (rGetEnsElemRelative(m_u16EnsembleIndex)).rEnsemble;


	ETG_TRACE_USR4(("dabdrv_chnList::vPrepareDynChnList:Services in Current Ensemble =0x%08x listType=%d",
		rCurEnsemble.u16GetEID(),
		_enChnListType));

	DAB_FOREACH_MAP(trChnListKey, trChnListServiceInfo,iterSrvMap, _mapServiceInfo) {
		trChnListServiceInfo &rSrvInfo = (*iterSrvMap).second;
		if(_enChnListType == enChnList_Global)
		{
			ETG_TRACE_USR4(("dabdrv_chnList::vPrepareDynChnList:Global List"));
			vUpdateDynChnList(rSrvInfo);		
		}
		else
		{
			DAB_IF_FIND_SET(trMecaEnsemble, ensIter, rSrvInfo.lEnsembles, rCurEnsemble) {

				ETG_TRACE_USR4(("dabdrv_chnList::vPrepareDynChnList:Current Ensemble List"));
				vUpdateDynChnList(rSrvInfo);
			}
		}
	}
	

}

tVoid dabdrv_chnList::vUpdateDynChnList(trChnListServiceInfo &rSrvInfo)
{
	tU32 u32NumAdded=0;
    tU32 u32NumIgnored=0;

	if (bIsChnOk(rSrvInfo)) {
				ETG_TRACE_USR4(("dabdrv_chnList::vUpdateDynChnList:ADDING:srvId=0x%08x label=%17s",
					rSrvInfo.rProgService.u32GetSID(),
					rSrvInfo.rLabel.pcGetCString()));
				if(m_enSrvListSortingType == enServiceListSortingType_SID)
				{
					_setDynSrvById.insert(rSrvInfo.rProgService);
				}
				else
				{
					_mapDynSrvByLabel.insert(pair<trMecaLabel, trChnListKey>(rSrvInfo.rLabel,rSrvInfo.rProgService));
				}
				u32NumAdded++;
			}
			else {
				ETG_TRACE_USR4(("dabdrv_chnList::vUpdateDynChnList:IGNORE:srvId=0x%08x label=%17s (infPres=%d, qual=%d, labelValid=%d)",
					rSrvInfo.rProgService.u32GetSID(),
					rSrvInfo.rLabel.pcGetCString(),
					rSrvInfo.bServiceInfoPresent, 
					rSrvInfo.u8ReceptionQuality,
					rSrvInfo.rLabel.bLabelValid));
				u32NumIgnored++;
			}

ETG_TRACE_USR4(("dabdrv_chnList::vUpdateDynChnList:END(updated: u32NumAdded=%d u32NumIgnored=%d)",
		u32NumAdded,
		u32NumIgnored));

}

tBool dabdrv_chnList::bIsInDynChnList(trMecaId const &rMecaId) const
{
	tBool bRes=FALSE;
    ETG_TRACE_USR4(("bIsInDynChnList: seace mecaId=0x%08x ",
                    rMecaId.u32GetSID()));

	 DAB_IF_FIND_SET(trMecaId, setIter, _setDynSrvById, rMecaId) {
               bRes=TRUE;
            }
	 return bRes;
                   
}

tBool dabdrv_chnList::bIsInDynChnList(trMecaId const &rMecaId, trMecaLabel const &rLabel) {
    tBool bRes=FALSE;
    ETG_TRACE_USR4(("bIsInDynChnList: seace mecaId=0x%08x label:",
                    rMecaId.u32GetSID()));
                   rLabel.vTrace();
                   
    DAB_FOREACH_MAP(trMecaLabel, trChnListKey,iterTest, _mapDynSrvByLabel) {
        ETG_TRACE_USR4(("bIsInDynChnList: check mecaId=0x%08x eq=%d label:",
                        iterTest->second._rMecaId.u32GetSID(),
                        iterTest->first == rLabel
                        ));
        iterTest->first.vTrace();
        if ((iterTest->first == rLabel)||(iterTest->second._rMecaId._u32Id == rMecaId._u32Id)) {
            bRes=TRUE;
            return bRes;
        }

    }
    DAB_IF_FIND_MAP(trMecaLabel, trChnListKey, iter, _mapDynSrvByLabel, rLabel) {
        while  (iter!=_mapDynSrvByLabel.end()) {
            if ( (iter->first.pcGetCString()==rLabel.pcGetCString()) ||

                 (iter->second._rMecaId._u32Id==rMecaId._u32Id) ){
                // found
                bRes=TRUE;
                break;
            }
            else {
                // try next, maybe it has the same label (multimap)
                ++iter;
            }
        }
    }
    return bRes;
}

tVoid dabdrv_chnList::vOpenFrozenEnsmbleList()
{
	ETG_TRACE_USR4(("dabdrv_chnList::vOpenFrozenEnsmbleList:START"));

	vPrepareDynEnsList();


	if(m_enEnsListSortingType == enEnsembleListSortingType_EID)
	{
		vUpdateFrozenEnsmbleListById();
	}
	else if(m_enEnsListSortingType == enEnsembleListSortingType_Freq)
	{
		vUpdateFrozenEnsmbleListByFreq();
	}
	else
	{
		vUpdateFrozenEnsmbleListByLabel();
	}

	

	ETG_TRACE_USR4(("dabdrv_chnList::vOpenFrozenEnsmbleList:END"));
}

tVoid dabdrv_chnList::vUpdateFrozenEnsmbleListById()
{
	tU16 u16Index=0;
	_mapFrozenIndexByEnsemble.clear();
	_vectorFrozenEnsmbleList.clear();

	ETG_TRACE_USR4(("dabdrv_chnList::vUpdateFrozenEnsmbleListById:create _vectorFrozenEnsmbleList"));
	_vectorFrozenEnsmbleList.resize(_setDynEnsById.size());

	 DAB_FOREACH(set<trMecaEnsemble>, iterDynEnsSet, _setDynEnsById) {

		trEnsembleListElem oEnsListElem;
		trChnListEnsembleInfo rEnsInfo= _mapEnsembleInfo[(*iterDynEnsSet)];
		
		oEnsListElem.u16Id = (tU16)(u16Index+1); 
		oEnsListElem.rEnsemble = rEnsInfo.rEnsemble;
		oEnsListElem.rEnsembleLabel = rEnsInfo.rLabel;
		oEnsListElem.u32Frequency = rEnsInfo.u32BestFreq;
		oEnsListElem.u8NumberOfAudioServices = rEnsInfo.u8NumberOfAudioServices;
		oEnsListElem.u8NumberOfDataServices = rEnsInfo.u8NumberOfDataServices;
		oEnsListElem.u8ReceptionQuality = rEnsInfo.u8Quality;
		oEnsListElem.bReception=bIsReceptionOk(rEnsInfo.u8Quality);
				
        
        _vectorFrozenEnsmbleList[u16Index]= oEnsListElem;

          ETG_TRACE_USR4(("dabdrv_chnList::vUpdateFrozenEnsmbleListById adding[%d]: EnsFrequency=%d EnsId=0x%08x label=%s",
			u16Index,
			oEnsListElem.u32Frequency,
			oEnsListElem.rEnsemble.u16GetEID(),
			oEnsListElem.rEnsembleLabel.pcGetCString()
			));
        _mapFrozenIndexByEnsemble[oEnsListElem.rEnsemble]= u16Index;
        u16Index++;
    }
}
tVoid dabdrv_chnList::vUpdateFrozenEnsmbleListByLabel()
{
	tU16 u16Index=0;
	_mapFrozenIndexByEnsemble.clear();
	_vectorFrozenEnsmbleList.clear();

	ETG_TRACE_USR4(("dabdrv_chnList::vOpenFrozenEnsmbleList:create _vectorFrozenEnsmbleList"));
	_vectorFrozenEnsmbleList.resize(_mapDynEnsByLabel.size());

	DAB_FOREACH_MMAP(trMecaLabel, trMecaEnsemble, iterDynEnsMap, _mapDynEnsByLabel) {

		trEnsembleListElem oEnsListElem;
		trChnListEnsembleInfo rEnsInfo= _mapEnsembleInfo[(*iterDynEnsMap).second];

		ETG_TRACE_USR4(("dabdrv_chnList::ensmeble:create _vectorFrozenEnsmbleList"));
		oEnsListElem.u16Id = (tU16)(u16Index+1); 
		oEnsListElem.rEnsemble = rEnsInfo.rEnsemble;
		oEnsListElem.rEnsembleLabel = rEnsInfo.rLabel;
		oEnsListElem.u32Frequency = rEnsInfo.u32BestFreq;
		oEnsListElem.u8NumberOfAudioServices = rEnsInfo.u8NumberOfAudioServices;
		oEnsListElem.u8NumberOfDataServices = rEnsInfo.u8NumberOfDataServices;
		oEnsListElem.u8ReceptionQuality = rEnsInfo.u8Quality;
		oEnsListElem.bReception=bIsReceptionOk(rEnsInfo.u8Quality);
		ETG_TRACE_USR4(("dabdrv_chnList::ensmeble:create _vectorFrozenEnsmbleList"));
		
        
        _vectorFrozenEnsmbleList[u16Index]= oEnsListElem;

          ETG_TRACE_USR4(("dabdrv_chnList::vOpenFrozenEnsmbleList adding[%d]: EnsFrequency=%d EnsId=0x%08x label=%s",
			u16Index,
			oEnsListElem.u32Frequency,
			oEnsListElem.rEnsemble.u16GetEID(),
			oEnsListElem.rEnsembleLabel.pcGetCString()
			));
        _mapFrozenIndexByEnsemble[oEnsListElem.rEnsemble]= u16Index;
        u16Index++;
    }
}
tVoid dabdrv_chnList::vUpdateFrozenEnsmbleListByFreq()
{
	tU16 u16Index=0;
	_mapFrozenIndexByEnsemble.clear();
	_vectorFrozenEnsmbleList.clear();

	ETG_TRACE_USR4(("dabdrv_chnList::vOpenFrozenEnsmbleList:create _vectorFrozenEnsmbleList"));
	_vectorFrozenEnsmbleList.resize(_mapDynEnsByFreq.size());

	DAB_FOREACH_MAP(tU32, trMecaEnsemble, iterDynEnsMap, _mapDynEnsByFreq) {

		trEnsembleListElem oEnsListElem;
		trChnListEnsembleInfo rEnsInfo= _mapEnsembleInfo[(*iterDynEnsMap).second];

		ETG_TRACE_USR4(("dabdrv_chnList::ensmeble:create _vectorFrozenEnsmbleList"));
		oEnsListElem.u16Id = (tU16)(u16Index+1); 
		oEnsListElem.rEnsemble = rEnsInfo.rEnsemble;
		oEnsListElem.rEnsembleLabel = rEnsInfo.rLabel;
		oEnsListElem.u32Frequency = rEnsInfo.u32BestFreq;
		oEnsListElem.u8NumberOfAudioServices = rEnsInfo.u8NumberOfAudioServices;
		oEnsListElem.u8NumberOfDataServices = rEnsInfo.u8NumberOfDataServices;
		oEnsListElem.u8ReceptionQuality = rEnsInfo.u8Quality;
		oEnsListElem.bReception=bIsReceptionOk(rEnsInfo.u8Quality);
		ETG_TRACE_USR4(("dabdrv_chnList::ensmeble:create _vectorFrozenEnsmbleList"));
		
        
        _vectorFrozenEnsmbleList[u16Index]= oEnsListElem;

          ETG_TRACE_USR4(("dabdrv_chnList::vOpenFrozenEnsmbleList adding[%d]: EnsFrequency=%d EnsId=0x%08x label=%s",
			u16Index,
			oEnsListElem.u32Frequency,
			oEnsListElem.rEnsemble.u16GetEID(),
			oEnsListElem.rEnsembleLabel.pcGetCString()
			));
        _mapFrozenIndexByEnsemble[oEnsListElem.rEnsemble]= u16Index;
        u16Index++;
    }
}

tVoid dabdrv_chnList::vOpenFrozenChnList(tBool bUpadteSDSSrvList) {
    ETG_TRACE_USR4(("dabdrv_chnList::vOpenFrozenChnList:START Sorting type=%d",m_enSrvListSortingType));

	//Clear the multi map as it is creating more memory. NCG3D-74972
	_mapDynSrvByLabel.clear();
    vPrepareDynChnList();

	if(m_enSrvListSortingType == enServiceListSortingType_SID)
	{
		vUpdateFrozenChnListbySID();
	}
	else
	{
        if (bUpadteSDSSrvList)
		    vUpdateFrozenChnListbyLabel(_vectorSDSSrvList);
        else
            vUpdateFrozenChnListbyLabel(_vectorFrozenSrvList);
	}   

    ETG_TRACE_USR4(("dabdrv_chnList::vOpenFrozenChnList:END"));
}

tVoid dabdrv_chnList::vUpdateFrozenChnListbyLabel(vector<trChnListElem>&_vectorSrvList)
{
	 tU16 u16Index=0;
    _mapFrozenIndexBySrv.clear();
    _vectorSrvList.clear();

    // add current service if not in _mapDynSrvByLabel
    tBool bTempAdded=FALSE;
    trChnListElem rCurChnInfo = dabdrv_chnInfo::instance()->rGetChnInfo();
    multimap<trMecaLabel, trChnListKey>::iterator addIter = _mapDynSrvByLabel.begin(); // assign for lint
    if (rCurChnInfo.rMecaId.bIsValid() && !bIsInDynChnList(rCurChnInfo.rMecaId, rCurChnInfo.rLabel)) {
        bTempAdded=TRUE;
        //if (!rCurChnInfo.rLabel.bLabelValid) {
        //    rCurChnInfo.rLabel=trMecaLabel(rCurChnInfo.rMecaId.u32GetSID());
        //}
        ETG_TRACE_USR4(("dabdrv_chnList::temp adding current Srv by label: 0x%08x label=%s",
                        rCurChnInfo.rMecaId.u32GetSID(),
                        rCurChnInfo.rLabel.pcGetCString()));
        addIter=  _mapDynSrvByLabel.insert(pair<trMecaLabel, trChnListKey>(rCurChnInfo.rLabel,rCurChnInfo.rMecaId));
    }
    _vectorSrvList.resize(_mapDynSrvByLabel.size());

    ETG_TRACE_USR4(("dabdrv_chnList::vUpdateFrozenChnListbyLabel:create _mapFrozenIndexBySrv"));

    DAB_FOREACH_MMAP(trMecaLabel, trChnListKey, iterDynSrvMap, _mapDynSrvByLabel) {
        trChnListElem oChnListElem;
		//PSARCC30-3357
        if (bTempAdded && rCurChnInfo.rMecaId._u32Id==(*iterDynSrvMap).second._rMecaId._u32Id) {
            oChnListElem=rCurChnInfo;
			oChnListElem.bReception=rCurChnInfo.bReception;
			ETG_TRACE_USR4(("dabdrv_chnList::vOpenFrozenChnList:oChnListElem.bReception=%d",oChnListElem.bReception));
        }
        else {
            oChnListElem=_mapServiceInfo[(*iterDynSrvMap).second];
        }
        oChnListElem.u16Id=(tU16)(u16Index+1);

		/*if(!oChnListElem.bPSFlag)
			oChnListElem.u8PresetNumber=0;
		else*/
        //In PSA secondary service can be stored in presets
			oChnListElem.u8PresetNumber=dabdrv_presets::instance()->u8GetPresetNumber(oChnListElem.rMecaId,(tU8)(oChnListElem.u16Scids));
			
			tU32 u32Id=oChnListElem.rMecaId.u32GetSID();
			ETG_TRACE_USR4(("dabdrv_chnList::vUpdateFrozenChnListbyLabel:SId=%x",u32Id));
	
			oChnListElem.sLogoLink = sGetLogoLink(oChnListElem.rMecaId._u32Id);
			ETG_TRACE_USR4(("dabdrv_chnList::vUpdateFrozenChnListbyLabel:SId=%x , logopath =%s",oChnListElem.u16Id ,oChnListElem.sLogoLink.c_str()));
	
			oChnListElem.u8DSSupport = u8GetDSSupport((u32Id&0xFFFF),(tU8)(oChnListElem.u16Scids));
        _vectorSrvList[u16Index]= oChnListElem;
        ETG_TRACE_USR4(("dabdrv_chnList::vUpdateFrozenChnListbyLabel:adding[%d]: id=%d srvId=0x%08x u8PresetNumber=%d label=%s",
                        u16Index,
                        oChnListElem.u16Id,
                        oChnListElem.rMecaId.u32GetSID(),
						oChnListElem.u8PresetNumber,
                        oChnListElem.rLabel.pcGetCString()));
		trChnListKey rChnListKey(oChnListElem.rMecaId,oChnListElem.u16Scids);
        _mapFrozenIndexBySrv[rChnListKey]=u16Index;
        u16Index++;
    }
    if (bTempAdded) {
        _mapDynSrvByLabel.erase(addIter);
    }
}
tVoid dabdrv_chnList::vUpdateFrozenChnListbySID()
{
	 tU16 u16Index=0;
    _mapFrozenIndexBySrv.clear();
    _vectorFrozenSrvList.clear();

    // add current service if not in _mapDynSrvByLabel
    tBool bTempAdded=FALSE;
    trChnListElem rCurChnInfo = dabdrv_chnInfo::instance()->rGetChnInfo();
    set<trMecaId>::iterator addIter = _setDynSrvById.begin(); // assign for lint
    if (rCurChnInfo.rMecaId.bIsValid() && !bIsInDynChnList(rCurChnInfo.rMecaId)) {
        bTempAdded=TRUE;
        if (!rCurChnInfo.rMecaId.bIsValid()) {
            rCurChnInfo.rMecaId=trMecaId(rCurChnInfo.rMecaId);
        }
        ETG_TRACE_USR4(("dabdrv_chnList::temp adding current Srv by ID: 0x%08x ",
                        rCurChnInfo.rMecaId.u32GetSID()));
		addIter=_setDynSrvById.insert(_setDynSrvById.begin(),rCurChnInfo.rMecaId);
    }
    _vectorFrozenSrvList.resize(_setDynSrvById.size());

    ETG_TRACE_USR4(("dabdrv_chnList::vUpdateFrozenChnListbySID:create _mapFrozenIndexBySrv"));

    DAB_FOREACH(set<trMecaId>, iterDynSrvSet, _setDynSrvById) {
        trChnListElem oChnListElem;
        if (bTempAdded && rCurChnInfo.rMecaId==(*iterDynSrvSet)) {
            oChnListElem=rCurChnInfo;
            oChnListElem.bReception=TRUE;
        }
        else {
            oChnListElem=_mapServiceInfo[(*iterDynSrvSet)];
        }
        oChnListElem.u16Id=(tU16)(u16Index+1);

			tU16 u16Id=oChnListElem.rMecaId.u16GetSID();
			ETG_TRACE_USR4(("dabdrv_chnList::vUpdateFrozenChnListbyLabel:SId=%x",u16Id));
		DAB_IF_FIND_MAP(tU32, string, iterLogoMap, _mapLogoList, u16Id) {
			ETG_TRACE_USR4(("dabdrv_chnList::vUpdateFrozenChnListbyLabel:SId=%x path=%s",oChnListElem.u16Id,iterLogoMap->second.c_str() ));
			oChnListElem.sLogoLink = iterLogoMap->second.c_str();
			ETG_TRACE_USR4(("dabdrv_chnList::vUpdateFrozenChnListbyLabel:SId=%x , logopath =%s",oChnListElem.u16Id ,oChnListElem.sLogoLink.c_str()));
		}
		

        //oChnListElem.u8PresetNumber=dabdrv_presets::instance()->u8GetPresetNumber(oChnListElem.rMecaId);
        _vectorFrozenSrvList[u16Index]= oChnListElem;
		ETG_TRACE_USR4(("dabdrv_chnList::vUpdateFrozenChnListbySID:adding[%d]: id=%d srvId=0x%08x label=%s",
                        u16Index,
                        oChnListElem.u16Id,
                        oChnListElem.rMecaId.u32GetSID(),
                        oChnListElem.rLabel.pcGetCString()));
		trChnListKey rChnListKey(oChnListElem.rMecaId,oChnListElem.u16Scids);
        _mapFrozenIndexBySrv[rChnListKey]=u16Index;
        u16Index++;
    }
    if (bTempAdded) {
        _setDynSrvById.erase(addIter);
    }
}


tVoid dabdrv_chnList::vProcess(trMsgSrvValidateEPGData *poMsgSrvValidateEPGData)
{
	(tVoid) poMsgSrvValidateEPGData;
	tBool bEPGPresent = FALSE;
	ETG_TRACE_USR4(("dabdrv_chnList::trMsgSrvValidateEPGData"));

	DAB_FOREACH_MMAP(tU32, tU8, _mapContentIDIter, _mapContentIDList)
	 {
		 DAB_FOREACH_MAP(trChnListKey, trChnListServiceInfo, _mapServiceInfoIter, _mapServiceInfo)
		{
			if((tU16)_mapContentIDIter->first == _mapServiceInfoIter->first._rMecaId.u16GetSID())
			{
				ETG_TRACE_USR4(("dabdrv_chnList::trMsgSrvValidateEPGData mapserviceinfo _mapContentIDIter->first = 0x%x, _mapServiceInfoIter->first = %d", _mapContentIDIter->first, _mapServiceInfoIter->first._rMecaId.u16GetSID()));
				bEPGPresent = TRUE;
				break;
			}
		 }
	}
	trMsgSrvRspValidateEPGData oSrvRspValidateEPGData;
	oSrvRspValidateEPGData.bEPGPresent = bEPGPresent;
	DAB_vCallMsg(oSrvRspValidateEPGData);
}


#ifdef OLD_STYLE_LIST_IMPLEMENTATION
tVoid dabdrv_chnList::vProcess(trMsgSrvCmdSetChnList *poCmdOpenList) {

	ETG_TRACE_USR4(("dabdrv_chnList::vProcess(trMsgSrvCmdSetChnList):PTY Mask=%u",
                       poCmdOpenList->u32PTYFilter));
	_rPreSetChnListCmd = *poCmdOpenList;
	_u32PtyFilter = poCmdOpenList->u32PTYFilter;

	if(poCmdOpenList->u32PTYFilter!=0){
		if(poCmdOpenList->enOperation == tenListOperation_OPEN){
		vExtractPtyCodes(poCmdOpenList->u32PTYFilter);
		_u8PtyFilterIndex =0;
		ETG_TRACE_USR4(("dabdrv_chnList::vSendPtyFilter for:by index PTY Code=%d ",_vectorPTYCode[0]));
		vSendPtyFilter(_vectorPTYCode[_u8PtyFilterIndex],enMeca_PtyCommand_set_filter);
	}
		else if(poCmdOpenList->enOperation == tenListOperation_GET){
			vSetChnList(poCmdOpenList);
		}
	else{

		vSendPtyFilter(0,enMeca_PtyCommand_delete_all_filter);
			vSetChnList(poCmdOpenList);
		}
	}
	else{
		vSetChnList(poCmdOpenList);
	}
}

#else
tVoid dabdrv_chnList::vProcess(trMsgSrvCmdSetChnList *poCmdOpenList) {

	ETG_TRACE_USR4(("dabdrv_chnList::vProcess(trMsgSrvCmdSetChnList):PTY Mask=%u",
                       poCmdOpenList->u32PTYFilter));
	_rPreSetChnListCmd = *poCmdOpenList;
	_u32PtyFilter = poCmdOpenList->u32PTYFilter;
	ETG_TRACE_USR4(("dabdrv_chnList::_u32PtyFilter=%x",poCmdOpenList->u32PTYFilter));

	if(poCmdOpenList->enOperation == tenListOperation_OPEN){

		vSetChnList(poCmdOpenList);
	/*	trMsgDrvDbQuery rDbQuery(Query_EnsembleList_Evaluate);
		DAB_vCallMsg(rDbQuery);*/

		//trMsgDrvDbQuery rDbQueryGlobal(Query_GlobalList_Prepare);
		//DAB_vCallMsg(rDbQueryGlobal);

		/*if(poCmdOpenList->u32PTYFilter==0){
			vSetChnList(poCmdOpenList);
		}*/
	}
	else
	{
		if(_u32PtyFilter==0) 
			_u32PtyFilter=PTY_MASK;

		vSetChnList(poCmdOpenList);
	}
}
#endif


tVoid dabdrv_chnList::vProcess(trMsgSrvCmdListUpdateStatus *poCmdListUpdateStatus){

	_bEnableSDS = poCmdListUpdateStatus->bEnableListUpadteStatus;
	//NCG3D-56562
	if(_bEnableSDS)
	{
		vSendUpdateSDS();
	}
}

tVoid dabdrv_chnList::vSendUpdateSDS(){
 
	ETG_TRACE_USR4(("dabdrv_chnList:vSendUpdateSDS"));

	if(!_bEnableSDS)
		return;

    m_enSrvListSortingType = enServiceListSortingType_LABEL;
    _enChnListType = enChnList_Global;

    vOpenFrozenChnList(TRUE);
    vTraceAll();
    DAB_trMsgSrvListUpdateProperty _oSrvListUpdateProperty(_vectorSDSSrvList);
    DAB_trMsgSrvListUpdateNotification _oSrvListUpdateNotification;
    _oSrvListUpdateNotification.bListUpdate = TRUE;
    _oSrvListUpdateProperty.vTrace();

     ETG_TRACE_USR4(("dabdrv_chnList::vTraceAll:_vectorSDSSrvList"));

    ETG_TRACE_USR4(("  Srv  InfoPresent FoundInList Quality labelPresent label EnsList ptycode"));
    for (tU32 i=0;i<_vectorSDSSrvList.size();++i) {
        vTraceSrvInfo(_vectorSDSSrvList[i]);

    }

    dabdrv_properties::instance()->oSrvListUpdateNotification.vSet(_oSrvListUpdateNotification);
    dabdrv_properties::instance()->oSrvListUpdateProperty.vSet(_oSrvListUpdateProperty);
    _oSrvListUpdateNotification.bListUpdate = FALSE;
    dabdrv_properties::instance()->oSrvListUpdateNotification.vSet(_oSrvListUpdateNotification);
    _vectorSDSSrvList.clear();

}




tVoid dabdrv_chnList::vSetChnList(trMsgSrvCmdSetChnList *poCmdOpenList){

	tBool bChanged = FALSE;
	DAB_tenResult enRes = DAB_enResult_FAILED;
	trMsgSrvRspSetChnList rResSrvList(_vectorFrozenSrvList);

	_enChnListType = poCmdOpenList->enListType;

	m_enSrvListSortingType= poCmdOpenList->enSrvListSortingType;
	m_u16EnsembleIndex = (tU16)(poCmdOpenList->u16EnsIndex-1);

	//Reset Preset select source
	dabdrv_presets::instance()->vResetSelectSource();

	switch (poCmdOpenList->enOperation) {
		case tenListOperation_OPEN:
			if (_bFrozenChnListOpen) {
				// close already opened list and reopen it again
				_vectorFrozenSrvList.clear();
				_mapFrozenIndexBySrv.clear();
				_bFrozenChnListOpen = FALSE;				
			}
			vMarkChanged();
			vOpenFrozenChnList();
			_bFrozenChnListOpen = TRUE;
			bChanged=TRUE;
			enRes=DAB_enResult_OK;		
			break;
		case tenListOperation_CLOSE:
			if (_bFrozenChnListOpen) {
				_vectorFrozenSrvList.clear();
				_mapFrozenIndexBySrv.clear();
				_bFrozenChnListOpen = FALSE;
			}
			bChanged=TRUE;
			_u8EnsembleCounter = 0;
			enRes=DAB_enResult_OK;
			break;
		case tenListOperation_GET:
			bChanged=TRUE;
			enRes=DAB_enResult_OK;
			break;
		case tenListOperation_INVALID:
		default:
			break;
	}

	ETG_TRACE_USR4(("dabdrv_chnList::trMsgSrvCmdSetChnList=%d",ETG_CENUM(DAB_tenResult, enRes)));
	rResSrvList.enRes=enRes;
	rResSrvList.u16TotalNumberOfElements=(tU16)_vectorFrozenSrvList.size();
	rResSrvList.u16NumElemToSend=0;
	rResSrvList.u16FirstIndexToSend=0;
	rResSrvList.bOpen=_bFrozenChnListOpen;
	if (poCmdOpenList->u16NumElem) {
		rResSrvList.u16FirstIndexToSend=(poCmdOpenList->u16FirstElem) ? (tU16)(poCmdOpenList->u16FirstElem - 1) : (tU16)0;
		if (rResSrvList.u16FirstIndexToSend<rResSrvList.u16TotalNumberOfElements) {
			// first index is valid
			if (rResSrvList.u16FirstIndexToSend+poCmdOpenList->u16NumElem>=rResSrvList.u16TotalNumberOfElements) {
				// correct num elem
				rResSrvList.u16NumElemToSend=(tU16)(rResSrvList.u16TotalNumberOfElements-rResSrvList.u16FirstIndexToSend);
			}
			else {
				rResSrvList.u16NumElemToSend=poCmdOpenList->u16NumElem;
			}
		}
	}
	// inform chn-info about status of frozen list first to let it fetch the frozenListId in case
	trMsgDrvCmdFrozenChnListStatus rFrozenChnListStatus(_bFrozenChnListOpen);
	DAB_vCallMsg(rFrozenChnListStatus);
	rResSrvList.u16ActivatedElemId= u16GetActivatedElemId(enFrozenList_Service);
	// todo: allocate _vectorFrozenSrvList once, dont use size() but numElem
	DAB_vCallMsg(rResSrvList);
	// send empty list of changes
	vUpdateChnListEmit(bChanged);

	vTraceAll();

}

tVoid dabdrv_chnList::vExtractPtyCodes(tU32 u32PtyMask)
{
	ETG_TRACE_USR4(("dabdrv_chnList::vExtractPtyCodes:PTY Mask=%u vectorSize=%d",u32PtyMask,(tU16)(_vectorPTYCode.size())));
	_vectorPTYCode.clear();
	for(tU8 i=0;i<=31;i++)
	{
		if((u32PtyMask >> i) & 0x01)
		{			
			_vectorPTYCode.push_back(i);

			ETG_TRACE_USR4(("dabdrv_chnList::vExtractPtyCodes:extrctd PTY Code=%d vectorSize=%d",i,(tU16)(_vectorPTYCode.size())));
		}
	}
}

tVoid dabdrv_chnList::vSendPtyFilter(tU8 u8PtyCode, tenMeca_PtyCommand enPtyCommand) const{

	ETG_TRACE_USR4(("dabdrv_chnList::vSendPtyFilter for:PTY Code=%d ",u8PtyCode));
	trMeca_CPTYMSetFilter rCPTYMSetFilter;
	rCPTYMSetFilter.enPTYCommand=enPtyCommand;
	rCPTYMSetFilter.u8IntTableId=0;
	rCPTYMSetFilter.u8PtyType=PTY_COARSE_PTY;
	rCPTYMSetFilter.u8PtyLanguage=0;
	rCPTYMSetFilter.u8PtyCode=u8PtyCode;
	dabdrv_mecaIf::instance()->vSendMecaCommand(rCPTYMSetFilter);
}

tVoid dabdrv_chnList::vProcess(trMeca_RPTYMSetFilter *poPTYMSetFilter) {

	if(_u32PtyFilter ){
		tU8 u8PtyCodeSent= _vectorPTYCode[_u8PtyFilterIndex];

		ETG_TRACE_USR4(("dabdrv_chnList::vProcess(trMeca_RPTYMSetFilter):SentPTYFilter=%d RecvdPTYFilter=%d result=%d",u8PtyCodeSent,poPTYMSetFilter->u8PtyCode,(poPTYMSetFilter->u8PtyCode==u8PtyCodeSent)));

		if(poPTYMSetFilter->u8PtyCode==u8PtyCodeSent){		
			trMeca_CDbGetDbEnsembleList oCGetEnsembleList;
			dabdrv_mecaIf::instance()->vSendMecaCommand(oCGetEnsembleList);
			trMeca_CDbEnsembleGetServiceList oCGetEnsembleServiceList;
			dabdrv_mecaIf::instance()->vSendMecaCommand(oCGetEnsembleServiceList);
				++_u8PtyFilterIndex;
				if(_u8PtyFilterIndex <_vectorPTYCode.size())
				{
					ETG_TRACE_USR4(("dabdrv_chnList::vProcess(trMeca_RPTYMSetFilter):Sending _vectorPTYCode index %d",_u8PtyFilterIndex));
					vSendPtyFilter(_vectorPTYCode[_u8PtyFilterIndex],enMeca_PtyCommand_add_filter);
				}
				else
				{
					ETG_TRACE_USR4(("dabdrv_chnList::vProcess(trMeca_RPTYMSetFilter):end of _vectorPTYCode "));
					_vectorPTYCode.clear();
					vSetChnList(&_rPreSetChnListCmd);
				}
		}
	}
	else
	{
		ETG_TRACE_USR4(("dabdrv_chnList::vProcess(trMeca_RPTYMSetFilter):_u32PtyFilter  is NULL"));
	}


}



tVoid dabdrv_chnList::vProcess(trMsgSrvCmdSetEnsmbleList *poCmdOpenList) {
   // tBool bChanged = FALSE;
    DAB_tenResult enRes = DAB_enResult_FAILED;
    trMsgSrvRspSetEnsmbleList rResEnsmbleList(_vectorFrozenEnsmbleList);
	m_enEnsListSortingType = poCmdOpenList->enEnsListSortingType;
    switch (poCmdOpenList->enOperation) {
        case tenListOperation_OPEN:
            if (_bFrozenEnsmbleListOpen) {
                // close already opened list and reopen it again
				_mapFrozenIndexByEnsemble.clear();
                _vectorFrozenEnsmbleList.clear();
                _bFrozenEnsmbleListOpen = FALSE;
            }
            vOpenFrozenEnsmbleList();
            _bFrozenEnsmbleListOpen = TRUE;
            //bChanged=TRUE;
            enRes=DAB_enResult_OK;
            break;
        case tenListOperation_CLOSE:
            if (_bFrozenEnsmbleListOpen) {
				_mapFrozenIndexByEnsemble.clear();
                _vectorFrozenEnsmbleList.clear();
                _bFrozenEnsmbleListOpen = FALSE;
            }
            //bChanged=TRUE;
            enRes=DAB_enResult_OK;
            break;
        case tenListOperation_GET:
            //bChanged=TRUE;
            enRes=DAB_enResult_OK;
            break;
        case tenListOperation_INVALID:
        default:
            break;
    }

    rResEnsmbleList.enRes=enRes;
    rResEnsmbleList.u16TotalNumberOfElements=(tU16)_vectorFrozenEnsmbleList.size();
    rResEnsmbleList.u16NumElemToSend=0;
    rResEnsmbleList.u16FirstIndexToSend=0;
    rResEnsmbleList.bOpen=_bFrozenEnsmbleListOpen;
    if (poCmdOpenList->u16NumElem) {
        rResEnsmbleList.u16FirstIndexToSend=(poCmdOpenList->u16FirstElem) ? (tU16)(poCmdOpenList->u16FirstElem - 1) : (tU16)0;
        if (rResEnsmbleList.u16FirstIndexToSend<rResEnsmbleList.u16TotalNumberOfElements) {
            // first index is valid
            if (rResEnsmbleList.u16FirstIndexToSend+poCmdOpenList->u16NumElem>=rResEnsmbleList.u16TotalNumberOfElements) {
                // correct num elem
                rResEnsmbleList.u16NumElemToSend=(tU16)(rResEnsmbleList.u16TotalNumberOfElements-rResEnsmbleList.u16FirstIndexToSend);
            }
            else {
                rResEnsmbleList.u16NumElemToSend=poCmdOpenList->u16NumElem;
            }
        }
    }
    //// inform chn-info about status of frozen list first to let it fetch the frozenListId in case
    trMsgDrvCmdFrozenEnsListStatus rFrozenEnsListStatus(_bFrozenEnsmbleListOpen);
    DAB_vCallMsg(rFrozenEnsListStatus);
    rResEnsmbleList.u16ActivatedElemId= u16GetActivatedElemId(enFrozenList_Ensemble);
    // todo: allocate _vectorFrozenSrvList once, dont use size() but numElem
    DAB_vCallMsg(rResEnsmbleList);
    //// send empty list of changes
    //vUpdateChnListEmit(bChanged);

    //vTraceAll();

}

tVoid dabdrv_chnList::vProcess(trMsgDrvStopComponent*) {
    _enListCheckState=enListState_Done;
    _enForcedUpdateState=enForcedUpdateState_Idle;
    DAB_vCallMsgCtor(this, trMsgSrvCmdSetChnList(tenListOperation_CLOSE));

    poGet1sTickTimer()->vStop();

}

tVoid dabdrv_chnList::vProcess(trMsgDrvCmdChnListPsidMonitor *poSetPsidMonitor) {
    ETG_TRACE_USR4(("dabdrv_chnList::trMsgDrvCmdChnListPsidMonitor:0x%08x",
                    poSetPsidMonitor->rMecaId.u32GetSID()));
    _rMonitoredSrvId=poSetPsidMonitor->rMecaId;
    // todo:direct response if monitored id is present
}

tVoid dabdrv_chnList::vProcess(trMsgDrvCmdChnListSetCurrentSrv *poSetCurrentSrv) {
    ETG_TRACE_USR4(("dabdrv_chnList::trMsgDrvCmdChnListSetCurrentSrv:0x%08x",
                    poSetCurrentSrv->rMecaId.u32GetSID()));
    _rCurrentSrvId=poSetCurrentSrv->rMecaId;
    // todo:direct response if monitored id is present
}


tVoid dabdrv_chnList::vRequestEnsFreqList(trMecaEnsemble const &rEns) const {

       
    trMeca_CDbEnsembleGetFrequencyList rCGetEnsFreqList;
    rCGetEnsFreqList.rRefEnsemble=rEns;
    rCGetEnsFreqList.u8NumFrequencies=10;
    //dabdrv_mecaIf::instance()->vSendMecaCommand(rCGetEnsFreqList);

}

tVoid dabdrv_chnList::vProcess(trMsgDrvRspCurrentEnsemble *poCurrentEnsemble) {
    ETG_TRACE_USR4(("dabdrv_chnList::trMsgDrvRspCurrentEnsemble:0x%08x",
                    poCurrentEnsemble->rEnsemble.u32GetID()));
    // only ensembles reported by rdm may have changed
    // check if ensembel is in our list

    _rCurrentEnsemble=poCurrentEnsemble->rEnsemble;

}

tVoid dabdrv_chnList::vProcess(trMsgDrvEvtLearnDone *poDrvEvtLearnDone) {
    (tVoid)poDrvEvtLearnDone;
    ETG_TRACE_USR4(("dabdrv_chnList::vProcess(trMsgDrvEvtLearnDone)"));

    if (_enForcedUpdateState==enForcedUpdateState_LearnRunning) {
        _oAfterLearnTimer.vStart(DAB_CHN_LIST_AFTER_LEARN_TIMER_MS);
        _enForcedUpdateState=enForcedUpdateState_AfterLearnWait;
    }
#if 0
    trMeca_CDbGetDbEnsembleList oCGetEnsembleList;
    dabdrv_mecaIf::instance()->vSendMecaCommand(oCGetEnsembleList);
#endif
}

tVoid dabdrv_chnList::vProcess(trMsgDrvEvtForcedLearnStart *poDrvEvtForcedLearnStart) {
    (tVoid)poDrvEvtForcedLearnStart;

    // mark that forced learn is running
    _enForcedUpdateState=enForcedUpdateState_LearnRunning;
    //  close chnList if open and add flag to chnListUpdate that forced learn is running
    DAB_vCallMsgCtor(this, trMsgSrvCmdSetChnList(tenListOperation_CLOSE));

	//  close ensList if open and add flag to enslistUpdate that forced learn is running
    DAB_vCallMsgCtor(this, trMsgSrvCmdSetEnsmbleList(tenListOperation_CLOSE));
    /*
      - add info to chnList that learn is running
      When learn is done:
      - start timer to wait for updates from dab-db. (2 s)
      - after timeout wait until all channel-label have been received
    */

}

tVoid dabdrv_chnList::vProcess(trMsgChnListUpdateTimer *) {
    ETG_TRACE_USR4(("dabdrv_chnList::trMsgChnListUpdateTimer:_rCurrentEnsemble=0x%08x",
                    _rCurrentEnsemble.u32GetID()));
    //vTraceAll();
    _oChnListUpdateTimer.vStart(_u32ChnListUpdateTimerValMs);
    // request service-list for current ensembles

    if (_enListCheckState==enListState_Done) {
        vStartSrvInfoCollect();
    }


#ifdef DAB_ENABLE_POLL_CHN_LIST 
    // todo, if autonotification for srvList works again with dab-module
    trMeca_CDbGetServiceList rCGetServiceList;
    rCGetServiceList.enServiceType=enMeca_ServiceType_AUDIO_SERVICE;
    rCGetServiceList.u8FilterId=(tU8)enDabDbFilterId_AudioService;
    rCGetServiceList.u8NumServBefore=0;
    rCGetServiceList.u8NumServBehind=0xFF;
    //dabdrv_mecaIf::instance()->vSendMecaCommand(rCGetServiceList);

#endif

}


tVoid dabdrv_chnList::vProcess(trMsgChnListAfterLearnTimer *) {
    ETG_TRACE_USR4(("dabdrv_chnList::trMsgChnListAfterLearnTimer"));
    _enForcedUpdateState=enForcedUpdateState_AfterLearnWaitLabel;
    //vStartSrvInfoCollect();//jab4kor: send evaluate for both lists
    //Evaluate the ensemble list
    trMsgDrvDbQuery rDbQueryEnsEval(Query_EnsembleList_Evaluate);
	DAB_vCallMsg(rDbQueryEnsEval);

	//Evaluate the global list
	//trMsgDrvDbQuery rDbQueryGlobalEval(Query_GlobalList_Evaluate);
	//DAB_vCallMsg(rDbQueryGlobalEval);
}
tVoid dabdrv_chnList::vProcess(trMsgChnList1sTick* ) {
    vRequestNextSrvInfo();
}




tVoid dabdrv_chnList::vRequestNextSrvInfo() {
    /*
      After forced learn we have to provide trigger when chnList is again in sync with dab-db:
      strategy: wait 2 sek after learn and than until _enListCheckState!=enListState_CheckVirgin
    */
    if (_enListCheckState!=enListState_CheckVirgin  && _enForcedUpdateState==enForcedUpdateState_AfterLearnWaitLabel) {
        ETG_TRACE_USR1(("vRequestNextSrvInfo:fored update end!"));
        // necessary updates after forced learn are done, update status.
        _enForcedUpdateState=enForcedUpdateState_Idle;
        vUpdateChnListEmit();
    }
    if (_enListCheckState==enListState_Done) {
        poGet1sTickTimer()->vStop();
        _oChnListUpdateTimer.vStart(_u32ChnListUpdateTimerValMs);

        return;
    }
    tBool bDone=FALSE;
    while(!bDone) {
		tenListElemUpdateState enListElemUpdateState=_iterUpdate->second.enListElemUpdateState;
        //ETG_TRACE_USR1(("vRequestNextSrvInfo:checking sid=0x%08x checkState=%d elemState=%d",
						//_iterUpdate->first.u32GetSID(),
                        //ETG_CENUM(tenListCheckState, _enListCheckState),
                        //ETG_CENUM(tenListElemUpdateState, enListElemUpdateState)));   //jab4kor: uncomment 
		if ((tU32)_enListCheckState==(tU32)enListElemUpdateState && _iterUpdate->second.u32UpdateId !=_u32UpdateId) {
            // mark that chn has been updated
            _iterUpdate->second.u32UpdateId =_u32UpdateId;
            // request chnInfo
            _rLastUpdatedService=_iterUpdate->first;
            trMeca_CDbServiceGetInfo rMecaCServiceGetInfo;
			rMecaCServiceGetInfo.rProgrammeService=_rLastUpdatedService._rMecaId.rGetProgService();
            dabdrv_mecaIf::instance()->vSendMecaCommand(rMecaCServiceGetInfo); 
            bDone=TRUE;
        }
        if (++_iterUpdate == _mapServiceInfo.end()) {
            ETG_TRACE_USR1(("vRequestNextSrvInfo:wrap MAP"));                        
            _iterUpdate=_mapServiceInfo.begin();
        }
        if (_iterUpdate==_iterUpdateStart) {
            ETG_TRACE_USR1(("vRequestNextSrvInfo:cycle done, new checkState=%d",
                            ETG_CENUM(tenListCheckState, _enListCheckState)));                        
            _enListCheckState = (tenListCheckState)((tU32)_enListCheckState+1);
            if (_enListCheckState==enListState_Done) {
                _rLastUpdatedService=_iterUpdate->first;
                break;
            } 
        }
    }
}

tVoid dabdrv_chnList::vStartSrvInfoCollect() {
    poGet1sTickTimer()->vStart();
    _u32UpdateId++;
    if (_mapServiceInfo.empty()) {
        _enListCheckState=enListState_Done;
        return;
    }
    //vTraceAll();
    _enListCheckState=enListState_CheckVirgin;
    _iterUpdate=_mapServiceInfo.upper_bound(_rLastUpdatedService);
    if (_iterUpdate==_mapServiceInfo.end()) {
        _iterUpdate=_mapServiceInfo.begin();
    }
	_iterUpdateStart=_iterUpdate;
    vRequestNextSrvInfo();     
}

#ifdef OLD_STYLE_LIST_IMPLEMENTATION
tVoid dabdrv_chnList::vProcess(trMeca_RDbServiceGetInfo* poSrvInfo) {

    tBool bChanged=FALSE;
    ETG_TRACE_USR4(("dabdrv_chnList(trMeca_RDbServiceGetInfo):got info for sid=0x%08x bChanged=%d",
                    poSrvInfo->rProgrammeService.u32GetSID(),bChanged));
    if ((poSrvInfo->enServiceType != enMeca_ServiceType_AUDIO_SERVICE) ||
        (FALSE == poSrvInfo->rLabel.bLabelValid)){
        return;
    }
    // if we received a valid label and the received label differs from the existing label, store it.

    DAB_IF_FIND_MAP(trMecaId, trChnListServiceInfo, iterSrvMap, _mapServiceInfo, poSrvInfo->rProgrammeService) {
        ETG_TRACE_USR4(("dabdrv_chnList(trMeca_RDbServiceGetInfo):updating sid=0x%08x",
                        poSrvInfo->rProgrammeService.u32GetSID()));
        trChnListServiceInfo &rServiceInfo = iterSrvMap->second;

        rServiceInfo.bServiceInfoPresent=TRUE;
        if (rServiceInfo.u16AvailableAnnoTypesMask != poSrvInfo->u16AnnoMask) {
            rServiceInfo.u16AvailableAnnoTypesMask=poSrvInfo->u16AnnoMask;
            bChanged=TRUE;
        }
		tU8 u8PTYCode=u8GetPtyCode(poSrvInfo);
		if (rServiceInfo.u8PTY != u8PTYCode) {
            rServiceInfo.u8PTY=u8PTYCode;
            bChanged=TRUE;
        }
        rServiceInfo.enListElemUpdateState=enListElemUpdateState_Ok;
        if (poSrvInfo->rLabel.bLabelValid) {
            if (!rServiceInfo.bLabelPresent || (rServiceInfo.rLabel !=poSrvInfo->rLabel )) {
                rServiceInfo.rLabel=poSrvInfo->rLabel;
                rServiceInfo.bLabelPresent = TRUE;
                bChanged=TRUE;
            }
        }
#ifdef DAB_REPLACE_INVALID_LABEL_BY_SID
        else {
            trMecaLabel rDefaultLabel=trMecaLabel(poSrvInfo->rProgrammeService.u32GetSID());
            rServiceInfo.enListElemUpdateState=enListElemUpdateState_Changed;

            if (rServiceInfo.rLabel != rDefaultLabel) {
                ETG_TRACE_USR1(("dabdrv_chnList:: replace label: =%17s",
                                rServiceInfo.rLabel.pcGetCString()));
                ETG_TRACE_USR1(("dabdrv_chnList:: with    label: =%17s",
                                rDefaultLabel.pcGetCString()));
                rServiceInfo.rLabel = rDefaultLabel;
                bChanged=TRUE;
            }
        }
#endif
        
        if (rServiceInfo.rLabel.bLabelValid && _rMonitoredSrvId.bIsValid() && poSrvInfo->rProgrammeService == _rMonitoredSrvId) {
            trMsgDrvRspChnListPsidMonitor rMonitor(_rMonitoredSrvId);
            _rMonitoredSrvId.vInvalidate();
            DAB_vCallMsg(rMonitor);
        }
        if (bChanged) {
            vMarkChanged();
            if (poSrvInfo->rProgrammeService == _rCurrentSrvId) {
                trMsgDrvRspChnListCurrentSrvChanged rSrvChanged(_rCurrentSrvId);
                DAB_vCallMsg(rSrvChanged);
            }
        }
        vTraceAll();
    } else {
        ETG_TRACE_USR4(("dabdrv_chnList(trMeca_RDbServiceGetInfo):unknown sid=0x%08x",
                        poSrvInfo->rProgrammeService.u32GetSID()));
    }

    if (poSrvInfo->rProgrammeService==_rLastUpdatedService && _enListCheckState!=enListState_Done && _enListCheckState!=enListState_CheckRefresh) {
        poGet1sTickTimer()->vStart();
        vRequestNextSrvInfo();  //jab4kor:check if required
    }
}
#endif

tU8 dabdrv_chnList::u8GetPtyCode(trMeca_RDbServiceGetInfo* poSrvInfo)const
{
	tU8 u8PTYMask=poSrvInfo->u8PtyMask;
	tU8 u8PTYCode=0;

	for(tU8 i=0;i<=enMeca_DbPtyCodeIndex_SECOND_FINE_CODE_DYNAMIC;i++)
	{
		if((u8PTYMask >> i) & 0x01)
		{
			u8PTYCode=poSrvInfo->u8PtyCodes[i];

			ETG_TRACE_USR4(("dabdrv_chnList::u8GetPTYCode: SID=0x%08x, PTYCode=%d",
				poSrvInfo->rProgrammeService.u32GetSID(),u8PTYCode ));
			break;
		}
	}
	return u8PTYCode;
}

tVoid dabdrv_chnList::vUpdateChnListClear() {
    _rMsgUpdateChnList.u16NumChanges=0;
    _rMsgUpdateChnList.vcChangedAvailablities.clear();
}
tVoid dabdrv_chnList::vUpdateChnListAdd(trMecaProgrammeService const &rSrv, tBool bReception) {
    ETG_TRACE_USR4(("dabdrv_chnList::vUpdateChnListAdd: SID=0x%08x, scids=%d bReception=%d _bFrozenChnListOpen=%d",
                    rSrv.u32GetSID(),rSrv.u16GetScids(), bReception, _bFrozenChnListOpen));

    if (_bFrozenChnListOpen) {
        trSrvIdAndQuality rSrvIdAndQuality;
        // check if the service is in the frozen list
        DAB_IF_FIND_MAP(trChnListKey, tU16, iter, _mapFrozenIndexBySrv, trChnListKey(rSrv)) {
            ETG_TRACE_USR4(("dabdrv_chnList::vUpdateChnListAdd:found in frozen list"));
            // we found the service in frozen list, set bReception=FALSE
            rSrvIdAndQuality.u16ListElemId= iter->second;
            rSrvIdAndQuality.bReception=bReception;
            _rMsgUpdateChnList.u16NumChanges++;
            _rMsgUpdateChnList.vcChangedAvailablities.push_back(rSrvIdAndQuality);
            // update reception also in frozen list:
			if((iter->second) < (_vectorFrozenSrvList.size())) //NCG3D-91090, NCG3D-84198
			{
			 _vectorFrozenSrvList[iter->second].bReception=rSrvIdAndQuality.bReception;
			}
        }
    }
}
tVoid dabdrv_chnList::vUpdateChnListEmit(tBool bForced) {
    ETG_TRACE_USR4(("dabdrv_chnList::vUpdateChnListEmit:_bFrozenChnListOpen=%d _enForcedUpdateState=%d _rMsg.bUpdateRunning=%d",
                    _bFrozenChnListOpen,
                    ETG_CENUM(tenForcedUpdateState, _enForcedUpdateState),
                    _rMsgUpdateChnList.bUpdateRunning));

    _rMsgUpdateChnList.bChnListActive=_bFrozenChnListOpen;
    if (!_bFrozenChnListOpen) {
        _rMsgUpdateChnList.u16ActivatedElemId=0;
    } else {
        _rMsgUpdateChnList.u16ActivatedElemId=u16GetActivatedElemId(enFrozenList_Service);
    }
    if (_rMsgUpdateChnList.bUpdateRunning != bFordedUpdateRunning()) {
        _rMsgUpdateChnList.bUpdateRunning=bFordedUpdateRunning();
        bForced=TRUE;
    }
    ETG_TRACE_USR4(("dabdrv_chnList::vUpdateChnListEmit:_bFrozenChnListOpen=%d bForced=%d u16NumChanges=%d forcedUpdateRunning=%d",
                    _bFrozenChnListOpen, bForced, _rMsgUpdateChnList.u16NumChanges, bFordedUpdateRunning()));
    if ((_bFrozenChnListOpen && _rMsgUpdateChnList.u16NumChanges) || bForced) {
        DAB_vCallMsg(_rMsgUpdateChnList);
    }
    vUpdateChnListClear();
}



tVoid dabdrv_chnList::vChangeSrvEnsList(trMecaProgrammeService const &rSrv, trMecaEnsemble const &rEns, tBool bDelete) {
    if (!_mapServiceInfo.count(rSrv)) {
        ETG_TRACE_USR4(("dabdrv_chnList::vChangeSrvEnsLis rSrv0x%08x not in map!!!", 
                        rSrv.u32GetSID()));
        return;
    }
	trChnListKey rchnListKey(rSrv);
    trChnListServiceInfo &rSrvInfo=_mapServiceInfo[rSrv];
    tU8 u8OldSrvQuality=rSrvInfo.u8ReceptionQuality;
	tBool bOldReception=rSrvInfo.bReception;
    ETG_TRACE_USR4(("dabdrv_chnList::vChangeSrvEnsList START: SID=0x%08x EID=0x%08x bDelete=%d oldQual=%u bOldReception=%d",
                    rSrv.u32GetSID(), rEns.u32GetID(), bDelete, u8OldSrvQuality,bOldReception));
    DAB_IF_FIND_SET(trMecaEnsemble, ensIter, rSrvInfo.lEnsembles, rEns) {
        ETG_TRACE_USR4(("Srv Exists"));
        if (bDelete) {
            rSrvInfo.lEnsembles.erase(ensIter);
        }        
    } else {
        ETG_TRACE_USR4(("Srv New"));
        if (!bDelete) {
            rSrvInfo.lEnsembles.insert(rEns);
        }
    }
    tU8 u8NewSrvQuality=0;
	tBool bNewReception=FALSE;
    if (rSrvInfo.lEnsembles.empty()) {
        ETG_TRACE_USR4(("Ens List empty, delete srv"));
        _mapServiceInfo.erase(rSrv);
    } else {
		trReceptionState rReceptionState=rUpdateReceptionQuality(rSrvInfo);
		u8NewSrvQuality=rReceptionState.u8ReceptionQuality;
		bNewReception=rReceptionState.bReception;

        ETG_TRACE_USR4(("new qual=%d new reception=%d", u8NewSrvQuality,bNewReception));
    }

	ETG_TRACE_USR4(("**********bOldReception: %d", bOldReception));
	//vUpdateChnListAdd(rSrv, bOldReception);

    //ETG_TRACE_USR4(("**********QUALITY:  u8OldSrvQuality = %x  u8NewSrvQuality = %x", u8OldSrvQuality, u8NewSrvQuality));

    //if (u8OldSrvQuality != u8NewSrvQuality) {
        if (bOldReception != bNewReception) {
            ETG_TRACE_USR4(("reception changed %x",bNewReception));
            vUpdateChnListAdd(rSrv, bNewReception);
        }
    //}
}


tVoid dabdrv_chnList::vUpdateFrozenEnsListQuality(trMecaEnsemble const &rEns,tU8 u8NewEnsQuality,tBool bNewReception)
{
	ETG_TRACE_USR4(("dabdrv_chnList::vUpdateFrozenEnsListQuality: EID=0x%08x,  _bFrozenChnListOpen=%d,  bNewReception=%d",
		rEns.u16GetEID(), _bFrozenEnsmbleListOpen,bNewReception));

	if (_bFrozenChnListOpen) {

		// check if the ensemble is in the frozen list
		DAB_IF_FIND_MAP(trMecaEnsemble, tU16, iter, _mapFrozenIndexByEnsemble, rEns) {
			ETG_TRACE_USR4(("dabdrv_chnList::vUpdateFrozenEnsListQuality:found in frozen list"));

            if(iter->second < _vectorFrozenEnsmbleList.size())
            {
			   _vectorFrozenEnsmbleList[iter->second].u8ReceptionQuality = u8NewEnsQuality;
			   _vectorFrozenEnsmbleList[iter->second].bReception = bNewReception;
			}
		}
	}
}

#ifdef OLD_STYLE_LIST_IMPLEMENTATION
tVoid dabdrv_chnList::vProcess(trMeca_RDbGetDbEnsembleList* poEnsembleList) {

    ETG_TRACE_USR4(("dabdrv_chnList(trMeca_RDbGetDbEnsembleList) START"));
    trMsgSrvRspUpdateChnList rMsgUpdateChnList;

  	DAB_trEnsembleInfoProperty rProperty=dabdrv_properties::instance()->oEnsembleInfoProperty.oGet();
    rProperty.u8NumberOfEnsemble=poEnsembleList->u8NumTotalEnsemble;
    dabdrv_properties::instance()->oEnsembleInfoProperty.vSet(rProperty);

    // get a sorted vector of all known trMecaEnsemble
    vector<trMecaEnsemble>vOldEnsembles;
    DAB_FOREACH_MAP(trMecaEnsemble, trChnListEnsembleInfo, oldEnsemblesIter, _mapEnsembleInfo) {
        vOldEnsembles.push_back(oldEnsemblesIter->first);
        ETG_TRACE_USR4(("oldEnsemble=0x%08x", oldEnsemblesIter->first.u32GetID()));
    }

    // sort the received vector (todo: should already be sorted)
    set<trMecaEnsembleListElement> oNewEnsembleInfoSet;
    DAB_FOREACH(vector<trMecaEnsembleListElement>, iter, poEnsembleList->lEnsembleList) {
        oNewEnsembleInfoSet.insert(*iter);
    }
    // get a sorted vector of all trMecaEnsemble in poEnsembleList
    vector<trMecaEnsemble>vNewEnsembles;
    DAB_FOREACH(set<trMecaEnsembleListElement>, newEnsemblesIter, oNewEnsembleInfoSet) {
        vNewEnsembles.push_back((*newEnsemblesIter).rEnsemble);
        ETG_TRACE_USR4(("newEnsemble=0x%08x", newEnsemblesIter->rEnsemble.u32GetID()));
            
    }

    if (vNewEnsembles != vOldEnsembles) {
        ETG_TRACE_USR4(("DIFF!"));
        
        set<trMecaEnsemble> vLostEnsembles;
        // find ensembles that are no longer there
        DAB_set_difference(vOldEnsembles.begin(), vOldEnsembles.end(), 
                           vNewEnsembles.begin(), vNewEnsembles.end(), 
                           vLostEnsembles);
        // remove from vOldEnsembles the lost ensembles
        DAB_removeFromSet(vOldEnsembles,vLostEnsembles);
        // find the added ensembles
        set<trMecaEnsemble> vAddEnsembles;
        DAB_set_difference(vNewEnsembles.begin(), vNewEnsembles.end(), 
                           vOldEnsembles.begin(), vOldEnsembles.end(), 
                           vAddEnsembles);

        // store information of added ensembles
        DAB_FOREACH(set<trMecaEnsemble>, iterAddEnsembles, vAddEnsembles) {
            vMarkChanged();
            ETG_TRACE_USR4(("addEnsemble=0x%08x", iterAddEnsembles->u32GetID()));
            const trMecaEnsemble &rEns =*iterAddEnsembles;
            trChnListEnsembleInfo rEnsInfo;
            rEnsInfo.rEnsemble=rEns;
            // request frequency-list
            vRequestEnsFreqList(rEns);
            DAB_IF_FIND_SET(trMecaEnsembleListElement, ensInfoIter, oNewEnsembleInfoSet, rEns) {
                rEnsInfo.u8Quality=ensInfoIter->u8Quality;
                ETG_TRACE_USR4(("qual=0x%08x", rEnsInfo.u8Quality));
            }
            _mapEnsembleInfo[rEns]=rEnsInfo;
        }

        // remove references to lost ensembles
        DAB_FOREACH(set<trMecaEnsemble>, iterLostEnsembles, vLostEnsembles) {
            vMarkChanged();
            ETG_TRACE_USR4(("lostEnsemble=0x%08x", iterLostEnsembles->u32GetID()));
        
            // for each lost ensemble ...
            trMecaEnsemble rLostEnsemble=*iterLostEnsembles;
            set<trMecaProgrammeService>lLostServiceList=_mapEnsembleInfo[rLostEnsemble].lServiceList;

			//update ensemble quality
			vUpdateFrozenEnsListQuality(rLostEnsemble);

            // delete the ensemble from each service it appears in
            DAB_FOREACH(set<trMecaProgrammeService>, iterLostServices, lLostServiceList) {
                const trMecaProgrammeService &rLostSrv=*iterLostServices;
                ETG_TRACE_USR4(("lostService=0x%08x", rLostSrv.u32GetSID()));

				
                // update service
                vChangeSrvEnsList(rLostSrv, rLostEnsemble, TRUE);
            }
            _mapEnsembleInfo.erase(rLostEnsemble);
            
        }
        vStartSrvInfoCollect();
    }

    // update quality of all services of ensembles that are still there
    DAB_FOREACH(vector<trMecaEnsemble>, iterKeptEnsembles, vOldEnsembles) {
        trMecaEnsemble &rEns=*iterKeptEnsembles;
        trChnListEnsembleInfo &rEnsInfo=_mapEnsembleInfo[rEns];
        tU8 u8OldEnsQuality=rEnsInfo.u8Quality;
        tU8 u8NewEnsQuality=0;
        DAB_IF_FIND_SET(trMecaEnsembleListElement, ensInfoIter, oNewEnsembleInfoSet, rEns) {
            u8NewEnsQuality=ensInfoIter->u8Quality;
        }
        
        ETG_TRACE_USR4(("Ensemble= 0x%08x  --> u8OldEnsQuality=0x%x  u8NewEnsQuality=0x%x", rEnsInfo.rEnsemble._u32Id, u8OldEnsQuality, u8NewEnsQuality));

		//update ensemble quality
		vUpdateFrozenEnsListQuality(rEns,u8NewEnsQuality);


        if (u8OldEnsQuality != u8NewEnsQuality) {
            // request frequency-list
            vRequestEnsFreqList(rEns);
            vMarkChanged();
            // the reception-quality of the ensemble has changed
            // update reception-quality of the ensemble
            rEnsInfo.u8Quality=u8NewEnsQuality;
            // update reception-quality of each service of the ensemble
            set<trMecaProgrammeService>&lServiceList=rEnsInfo.lServiceList;
            DAB_FOREACH(set<trMecaProgrammeService>, iterServices, lServiceList) {
                vChangeSrvEnsList(*iterServices, rEns);
            }
        }
    }

    vUpdateChnListEmit();


    ETG_TRACE_USR4(("dabdrv_chnList(trMeca_RDbGetDbEnsembleList) END"));

}
#else

tVoid dabdrv_chnList::vProcess(trMeca_RDbQueryTrigger* poRDbQueryTrigger){
	poRDbQueryTrigger->vTrace();
	switch(poRDbQueryTrigger->enQueryTriggerResponse){
		case enMeca_DbQueryTriggerCmdRsp_ONEVENT:
			if(poRDbQueryTrigger->u8QueryTag == enMeca_TagID_Ensemble_List)
			{
				ETG_TRACE_USR4(("dabdrv_chnList::OnEvent received on enMeca_TagID_Ensemble_List EQueryinProcess %d",_bBlockEnsembleListQuery ));

				/* NO pending servicelist query responses , hence ensemble list query could be sent*/
				/*(!_bBlockEnsembleListQuery)=> buffers onevent to send ensemble list query */
				if(!_bBlockEnsembleListQuery)
				{
					ETG_TRACE_USR4(("dabdrv_chnList::evaluating EnsembleList query"));
					//Evaluate the ensemble list
					trMsgDrvDbQuery rDbQueryEnsEval(Query_EnsembleList_Evaluate);
					DAB_vCallMsg(rDbQueryEnsEval);
					_bBlockEnsembleListQuery = TRUE;
				}
				else
				{
					ETG_TRACE_USR4(("dabdrv_chnList::Setting QueryEnsembleList trigger"));
					_bQueryEnsembleList =TRUE;
				}
			}
			break;		
		default:			
			break;
	}
}
tVoid dabdrv_chnList::vProcess(trMeca_RDbQuery* poRDbQuery){
	poRDbQuery->vTrace();
	switch(poRDbQuery->u8QueryTag){
		case enMeca_TagID_Ensemble_List:
			ETG_TRACE_USR4(("dabdrv_chnList::enMeca_TagID_Ensemble_List"));
			vOnEnsembleListQueryResponse(poRDbQuery);
			if(dabdrv_learn::instance()->vGetupdateCompleteStatus())
			{
				dabdrv_learn::instance()->vSetupdateCompleteStatus(FALSE);
				trMsgSrvRspCmdLearn oRspCmdLearn;
				oRspCmdLearn.enRes = DAB_enResult_OK;

				DAB_vCallMsg(oRspCmdLearn);
			}
			break;
		/*case enMeca_TagID_Global_List:
			ETG_TRACE_USR4(("dabdrv_chnList::enMeca_TagID_Global_List"));
			vOnEnsembleServiceListQueryResponse(poRDbQuery);
			break;*/
		default:
			if((poRDbQuery->u8QueryTag >= enMeca_TagID_Global_List_START) && (poRDbQuery->u8QueryTag <= enMeca_TagID_Global_List_END)){
				ETG_TRACE_USR4(("dabdrv_chnList::enMeca_TagID_Global_List"));
				vOnEnsembleServiceListQueryResponse(poRDbQuery);
			}
			break;
	}
}

//Added by jab4kor
tVoid dabdrv_chnList::vOnEnsembleListQueryResponse(trMeca_RDbQuery* poRDbQuery){
	ETG_TRACE_USR4(("dabdrv_chnList::vOnEnsembleListQueryResponse"));
	//check for _mapQueryTagInfo is added to avoid sending multiple ensemble servicelist queries 
	//before response for pevious query is received
	if( (poRDbQuery->enQueryResponse == enMeca_DbQueryResponse_OK)&&(_mapQueryTagInfo.size()==0)){
		tU32 u32DataLength = (tU32)(poRDbQuery->lu8UserData.size());
		tU8* pu8Data = NULL;
		pu8Data = OSAL_NEW tU8[u32DataLength];
		if (pu8Data == NULL){
			ETG_TRACE_USR4(("dabdrv_chnList::vOnEnsembleListQueryResponse result length %d",u32DataLength));
			return;
		}
		for(tU32 u32Index=0; u32Index<u32DataLength; u32Index++){
			pu8Data[u32Index] = poRDbQuery->lu8UserData[u32Index];
			}
		try
		{

	  	DAB_trEnsembleInfoProperty rProperty=dabdrv_properties::instance()->oEnsembleInfoProperty.oGet();
	    rProperty.u8NumberOfEnsemble=(tU8)(poRDbQuery->u16QueryNumRows);
		_u8NumEnsembles=(tU8)(poRDbQuery->u16QueryNumRows);
	    dabdrv_properties::instance()->oEnsembleInfoProperty.vSet(rProperty);

		ETG_TRACE_USR4(("dabdrv_chnList::vOnEnsembleListQueryResponse _u8NumEnsembles=%d",_u8NumEnsembles));

	    vector<trMecaEnsembleListElement>lEnsembleList;
		for (tU16 u16Row=1;u16Row<=poRDbQuery->u16QueryNumRows;u16Row++){

			tU8 u8ECC;
			tU16 u16EId;
			tU32 u32Freq;
			tU8 u8Quality;
			tBool bNotRXFlag;
			APILabel EnsLabel;
			tU8 u8NumberOfAudioServices;
			tU8 u8NumberOfDataServices;
			trMecaEnsembleListElement rEnsElem;
            tU8* pSiHash = sdxf_get_chunk_body_pointer(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,ONE));

            string sSiHashString = "";

            if(pSiHash!=NULL)
            {
                for(tU8 i=0; i<16;i++)
                {
                    sSiHashString += pSiHash[i];
                    ETG_TRACE_USR4(("dabdrv_chnList:vOnEnsembleListQueryResponse  pSiHash %x",pSiHash[i]));
                }
            }


            u8ECC=(tU8)(sdxf_chunk_get_numeric_uint32(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,TWO)));
            u16EId= (tU16)(sdxf_chunk_get_numeric_uint32(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,THREE)));
            sdxf_labelblob_2_apilabel(&EnsLabel, sdxf_chunk_get_blob(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,FOUR)));
            u8NumberOfAudioServices=(tU8)(sdxf_chunk_get_numeric_uint32(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,FIVE)));
            u8NumberOfDataServices=(tU8)(sdxf_chunk_get_numeric_uint32(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,SIX)));
            u32Freq=sdxf_chunk_get_numeric_uint32(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,SEVEN));
            u8Quality=(tU8)(sdxf_chunk_get_numeric_uint32(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,EIGHT)));
            bNotRXFlag=(tBool)(sdxf_chunk_get_numeric_uint32(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,NINE)));

#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
            SDXFBlob blobChunk = sdxf_chunk_get_blob(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,TEN));
            string FreqLabel((tChar*)blobChunk.content,blobChunk.length);
            ETG_TRACE_USR4(("dabdrv_chnList(trMeca_RDbQuery) %d",blobChunk.length));
            ETG_TRACE_USR4(("dabdrv_chnList:vOnEnsembleListQueryResponse   FreqLabel %s",FreqLabel.c_str()));
            rEnsElem.sFreqLabel = FreqLabel;
            ETG_TRACE_USR4(("dabdrv_chnList:rEnsElem.sFreqLabel %s",rEnsElem.sFreqLabel.c_str()));
#endif
			ETG_TRACE_USR4(("dabdrv_chnList(trMeca_RDbQuery) index=%d u8ECC=%04x u32EId=%04x u32Freq=%d bNotRXFlag=%d u8Quality=%d sEnsLabel=%s",
				u16Row,u8ECC,u16EId,u32Freq,bNotRXFlag,u8Quality,EnsLabel.b_label));
			ETG_TRACE_USR4(("dabdrv_chnList(trMeca_RDbQuery) index=%d u8NumberOfAudioServices=%d u8NumberOfDataServices=%d",
							u16Row,u8NumberOfAudioServices,u8NumberOfDataServices));


			rEnsElem.rEnsemble.vSetEID(u16EId);
			rEnsElem.rEnsemble.vSetECC(u8ECC);
			rEnsElem.u32Freq=u32Freq;
			rEnsElem.u8Quality=u8Quality;
			rEnsElem.bReception=!bNotRXFlag;
			rEnsElem.u8NumberOfAudioServices=u8NumberOfAudioServices;
			rEnsElem.u8NumberOfDataServices=u8NumberOfDataServices;
			rEnsElem.rLabel.vParse(EnsLabel); //NCG3D-153095
			rEnsElem.sSihash = sSiHashString;
			lEnsembleList.push_back(rEnsElem);
             ETG_TRACE_USR4(("dabdrv_chnList:vOnEnsembleListQueryResponse  Curr sSiHashString %s",sSiHashString.c_str()));

			 trMecaEnsemble rRefEnsemble;
			 rRefEnsemble.vSetEID(u16EId);
             rRefEnsemble.vSetECC(u8ECC);

			  DAB_IF_FIND_MAP(trMecaEnsemble, trChnListEnsembleInfo, iterEnsMap, _mapEnsembleInfo, rRefEnsemble) {
				      trChnListEnsembleInfo &rEnsInfo=_mapEnsembleInfo[rRefEnsemble];

					  ETG_TRACE_USR4(("dabdrv_chnList:vOnEnsembleListQueryResponse   Map sSiHashString %s",rEnsInfo.Sihash.c_str()));
					  if((rEnsInfo.Sihash.empty())||
						 (rEnsInfo.Sihash != sSiHashString)){
						  ETG_TRACE_USR4(("dabdrv_chnList(trMeca_RDbQuery)Adding sihash to map entry"));
						  rEnsInfo.Sihash = sSiHashString;
					  }
					  else if(rEnsInfo.Sihash == sSiHashString){
						  ETG_TRACE_USR4(("dabdrv_chnList(trMeca_RDbQuery) Map Sihash Matches Curr Sihash"));
						  continue;
					  }


			 }

			_u8TagCounter++;

			if(_u8TagCounter > enMeca_TagID_Global_List_END)
				_u8TagCounter = enMeca_TagID_Global_List_START;
			_mapQueryTagInfo[_u8TagCounter]=rEnsElem.rEnsemble;


            ETG_TRACE_USR1(("  vOnEnsembleListQueryResponse: Tag added EId=%x Ens-Quality=0x%x _u8TagCounter= %x",
                                                    rEnsElem.rEnsemble._u32Id,
                                                    rEnsElem.u8Quality,
                                                    _u8TagCounter));
            rEnsElem.rLabel.vTrace();

			vQueryEnsembleServiceList(u8ECC,u16EId,_u8TagCounter);

		}


		trEnsembleInfo rEnsembleInfo(lEnsembleList);
		vHandleEnsembleInfo(&rEnsembleInfo);

		if(_u8NumEnsembles == 0){
			_bBlockEnsembleListQuery = FALSE;
			/**Ensemble query is ZERO hence sending the Current Tuned station to the list.*/
			_mapEnsembleInfo.clear();
			_mapServiceInfo.clear();
			vSetChnList(&_rPreSetChnListCmd);
            vSendUpdateSDS();
        }

		/*if Sihash matches and service list query is skipped, then reset blocking of ensemble list query*/
		if(_mapQueryTagInfo.size()==0)
		{
			_bBlockEnsembleListQuery = FALSE;
		}
	}
	catch(std::bad_alloc)
		{
			ETG_TRACE_USR1(("Exception caught"));
		}
	if(pu8Data != NULL)
		{
			OSAL_DELETE[] pu8Data;
		}
	}
	else
	{
		ETG_TRACE_USR1(("dabdrv_chnList(vOnEnsembleListQueryResponse)Ensemble list info query failed: enQueryResponse=%d u16QueryNumRows=%d",
				ETG_CENUM(tenMeca_DbQueryResponse, poRDbQuery->enQueryResponse), poRDbQuery->u16QueryNumRows));
	}
}

tVoid dabdrv_chnList::vQueryEnsembleServiceList(tU8 u8ECC , tU16 u16EId, tU8 u8TagCounter)
{
	ETG_TRACE_USR1(("  vQueryEnsembleServiceList for u16EId=%x u8ECC=%x",u16EId, u8ECC));
	_u8EnsembleCounter = 0;
	trMeca_CDbQuery rCDbQuery;
	tChar sQuery[QUERY_LENGTH];
	sQuery[0]=0;
	rCDbQuery.enDbQueryCmd = enMeca_DbQueryCmd_EVALUATE;

	fc_dabtuner_config *poConfig=fc_dabtuner_config::instance();
	if (OSAL_NULL ==poConfig) {
		ETG_TRACE_FATAL(("dabdrv_main::vQueryEnsembleServiceList config not available !!!"));
		return;
	}
	tU8 _u8OEMType = poConfig->u8GetOEMType();
	//Query needs to be changed depending on the value of  _u8OEMType. Currently we have only one query which gives only primary services

	//rCDbQuery.u8QueryTag = (tU8)enMeca_TagID_Global_List;
	rCDbQuery.u8QueryTag = u8TagCounter;

	tBool bExcludeECC= FALSE;
	if(u8ECC==0xFF)
	{
		ETG_TRACE_USR4(("vQueryEnsembleServiceList: ECC %x is excluded",u8ECC));
		bExcludeECC = TRUE;
	}
	tChar ECC[FIELD_LENGTH];
	sprintf(ECC,"0x%x",u8ECC);
	tChar Eid[FIELD_LENGTH];
	sprintf(Eid,"0x%x",u16EId);

	if(_u8OEMType == (tU8)enKdsOEMType_SUZUKI)
	{
		tChar ptyFilter[FIELD_LENGTH+1];

		tU32 u32PTYFilter = dabdrv_chnList::instance()->u32GetPtyFilter();
		sprintf(ptyFilter,"0x%08x",u32PTYFilter);

		 ETG_TRACE_USR4(("dabdrv_db::vProcess(trMsgDrvDbQuery*) u32PTYFilter=%x ptyFilter=%s",
				 u32PTYFilter,ptyFilter));

		 if(!bExcludeECC)
		 {

		(tVoid)OSAL_szStringCopy(sQuery,DB_QUERY_INCLUDE_ECC );
		OSAL_szStringConcat(sQuery,ECC);
		OSAL_szStringConcat(sQuery," AND E.eid=");
		OSAL_szStringConcat(sQuery,Eid);
		OSAL_szStringConcat(sQuery,") AND (E.quality_word&0x80)=0");
		 }
		 else
		 {
			  (tVoid)OSAL_szStringCopy(sQuery, DB_QUERY_EXCLUDE_ECC);
			  OSAL_szStringConcat(sQuery,Eid);
			  OSAL_szStringConcat(sQuery," AND (E.quality_word&0x80)=0");
		 }		
            /*if( u32PTYFilter != PTY_MASK )
            {
        OSAL_szStringConcat(sQuery," AND ((1<<s_intpty)&");
        OSAL_szStringConcat(sQuery,ptyFilter);
        OSAL_szStringConcat(sQuery,")");
            }*/
		OSAL_szStringConcat(sQuery," GROUP BY S.r_service,C.scids");
	}
	else
	{
		if(!bExcludeECC)
		{
			(tVoid)OSAL_szStringCopy(sQuery, DB_QUERY_INCLUDE_ECC);
			OSAL_szStringConcat(sQuery,ECC);
			OSAL_szStringConcat(sQuery," AND E.eid=");
			OSAL_szStringConcat(sQuery,Eid);
			OSAL_szStringConcat(sQuery,") AND (E.quality_word&0x80)=0"
											 " GROUP BY S.r_service,C.scids");
		}
		else
		{
			(tVoid)OSAL_szStringCopy(sQuery, DB_QUERY_EXCLUDE_ECC);
			OSAL_szStringConcat(sQuery,Eid);
			OSAL_szStringConcat(sQuery," AND (E.quality_word&0x80)=0"
										 " GROUP BY S.r_service,C.scids");
		}

		ETG_TRACE_USR4(("dabdrv_chnList::vOnEnsembleServiceListQueryResponse ELSE"));
	}
	rCDbQuery.u8QueryStateId = 0;
	rCDbQuery.enDbQueryCmd = enMeca_DbQueryCmd_EVALUATE;

	tU32 u32Len=(tU32)(OSAL_u32StringLength(sQuery));
			rCDbQuery.lu8SqlStatement.reserve(u32Len+2);
			for (tU32 i=0; i < u32Len; i++) {
			rCDbQuery.lu8SqlStatement.push_back(sQuery[i]);
			}
		ETG_TRACE_USR4(("dabdrv_db::vProcess(trMsgDrvDbQuery* length=%d Query=%s",
			u32Len,
			sQuery));
		dabdrv_mecaIf::instance()->vSendMecaCommand(rCDbQuery);
}

tVoid dabdrv_chnList::vOnEnsembleServiceListQueryResponse(trMeca_RDbQuery* poRDbQuery) {

	ETG_TRACE_USR4(("dabdrv_chnList::vOnEnsembleServiceListQueryResponse"));

	trServiceInformation rServiceInformation;
	trEnsembleServiceList rEnsSrvInfo;
	tU8 u8ECC = 0x00;
	tU32 u32Eid = 0x0000;

	DAB_IF_FIND_MAP(tU8, trMecaEnsemble, iterTagMap, _mapQueryTagInfo, poRDbQuery->u8QueryTag) {
					u32Eid = iterTagMap->second.u32GetID();
					u8ECC = iterTagMap->second.u8GetECC();
					_mapQueryTagInfo.erase(poRDbQuery->u8QueryTag);			
		_bBlockEnsembleListQuery = FALSE;
		ETG_TRACE_USR4(("vOnEnsembleServiceListQueryResponse: Tag deleted u32Eid=%x tag =%x _bBlockEnsembleListQuery = %d",
			u32Eid, poRDbQuery->u8QueryTag ,_bBlockEnsembleListQuery));
			}
	if( (poRDbQuery->enQueryResponse == enMeca_DbQueryResponse_OK) /*&& (poRDbQuery->u16QueryNumRows != 0)*/){
		ETG_TRACE_USR4(("dabdrv_chnList::vOnEnsembleServiceListQueryResponse poRDbQuery->u16QueryNumRows =%d poRDbQuery->u8QueryTag = %d",poRDbQuery->u16QueryNumRows,poRDbQuery->u8QueryTag));
		//Increment EnsembleCounter
		_u8EnsembleCounter++;
			tU32 u32DataLength = (tU32)(poRDbQuery->lu8UserData.size());
			tU8* pu8Data = NULL;
			pu8Data = OSAL_NEW tU8[u32DataLength];
			if (pu8Data == NULL){
				ETG_TRACE_USR4(("dabdrv_chnList::vOnEnsembleServiceListQueryResponse result length %d",u32DataLength));
				return;
			}
			for(tU32 u32Index=0; u32Index<u32DataLength; u32Index++){
				pu8Data[u32Index] = poRDbQuery->lu8UserData[u32Index];
				}



			if(!u32Eid){
				OSAL_DELETE[] pu8Data;
				return;
			}


			trMecaEnsemble rEnsemble;
			rEnsemble.vSetEID((tU16)u32Eid);
			rEnsemble.vSetECC(u8ECC);
			ETG_TRACE_USR4(("vOnEnsembleServiceListQueryResponse u32Eid=%x rEnsemble.u16GetEID()=%x",
				u32Eid, rEnsemble.u16GetEID()));
			rEnsSrvInfo.rEnsemble = rEnsemble;
		try{

		for (tU16 u16Row=1;u16Row<=poRDbQuery->u16QueryNumRows;u16Row++){

			//tU16 u16Scidi;
			APIService rAPIservice;
			trMecaProgrammeService rMecaProgrammeService;
			
			trServiceInfo rServiceInfo;


			//u16Scidi=sdxf_chunk_get_numeric_uint32(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,ONE));
			//u32Eid=sdxf_chunk_get_numeric_uint32(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,ONE));
			//u8ECC=sdxf_chunk_get_numeric_uint32(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,TWO));

			if(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,SIX)!=NULL)
				rServiceInfo.u16Scids=(tU16)(sdxf_chunk_get_numeric_uint32(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,SIX)));
			else
				rServiceInfo.u16Scids=0;

					ETG_TRACE_USR4(("vOnEnsembleServiceListQueryResponse rServiceInfo.u16Scids=%d",
			rServiceInfo.u16Scids));

			sdxf_apiserviceblob_2_apiservice(&rAPIservice, sdxf_chunk_get_blob(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,TWO)));
			rMecaProgrammeService._enServiceType= (tenMeca_ServiceType)((sdxf_chunk_get_numeric_uint32(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,THREE)))+1);
			rMecaProgrammeService._u32Id =  (rAPIservice.r_programme_service.b_programme_service_type<<24)
																	| (rAPIservice.r_programme_service.b_ecc<<16)
																	| (rAPIservice.r_programme_service.w_service_id);
	#ifdef COMP_SERV_LIST

			rMecaProgrammeService._u16Scids = rServiceInfo.u16Scids;
	#endif
			rMecaProgrammeService.vTrace();

		
		


		//Update Service Info
		rServiceInfo.u32Eid=u32Eid;
		if(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,ONE)!=NULL)
			rServiceInfo.u8PresetNum=(tU8)(sdxf_chunk_get_numeric_uint32(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,ONE)));
		else
			rServiceInfo.u8PresetNum=0;
		rServiceInfo.rProgrammeService._enServiceType = rMecaProgrammeService._enServiceType; //Coverity
		rServiceInfo.rProgrammeService._u32Id =  rMecaProgrammeService._u32Id;
		rServiceInfo.rProgrammeService._u16Scids = rServiceInfo.u16Scids;
		rServiceInfo.rProgrammeService.vTrace();

			sdxf_labelblob_2_apilabel(&(rServiceInfo.ServiceLabel), sdxf_chunk_get_blob(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,FOUR)));
			sdxf_labelblob_2_apilabel(&(rServiceInfo.ComponentLabel), sdxf_chunk_get_blob(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,FIVE)));
			rServiceInfo.u8PSFlag=(tU8)(sdxf_chunk_get_numeric_uint32(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,SEVEN)));
			rServiceInfo.bNotRXFlag=(tBool)(sdxf_chunk_get_numeric_uint32(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,EIGHT)));
			if(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,NINE)!=NULL)
				rServiceInfo.u8ASUFlags=(tU8)(sdxf_chunk_get_numeric_uint32(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,NINE)));
			else
				rServiceInfo.u8ASUFlags=0;
			if( sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,TEN) != NULL ){
				rServiceInfo.u8PTY=(tU8)(sdxf_chunk_get_numeric_uint32(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,TEN)));
			}
			else{
				rServiceInfo.u8PTY=0;
			}
			if(!rServiceInfo.u8PSFlag){
				rServiceInfo.u8PresetNum=0;
			}

	#ifdef COMP_SERV_LIST
			//Fix for SUZUKI-22703 START
			/* As the primary service info is coming with scids value
			 * We assign 0 to the scids value while storing in the lists.
			 * */
			else{
				rServiceInfo.rProgrammeService._u16Scids = 0;
				rMecaProgrammeService._u16Scids = 0;
				rServiceInfo.u16Scids = 0;
			}
			//Fix for SUZUKI-22703 END
	#endif

			//sdxf_labelblob_2_apilabel(&(rServiceInfo.EnsembleLabel), sdxf_chunk_get_blob(sdxf_table_get_chunk(pu8Data,u32DataLength,u16Row,THIRTEEN)));
			DAB_IF_FIND_MAP(trMecaEnsemble, trChnListEnsembleInfo, iterEnsemble, _mapEnsembleInfo,rEnsemble){
				rServiceInfo.EnsembleLabel = iterEnsemble->second.rLabel;
				rServiceInfo.sFreqLabel = iterEnsemble->second.sFreqLabel;
                ETG_TRACE_USR4(("dabdrv_chnList:vOnEnsembleServiceListQueryResponse sFreqLabel %s",rServiceInfo.sFreqLabel.c_str()));
			}

			//rServiceInfo.u16Scidi=u16Scidi;
			ETG_TRACE_USR4(("vOnEnsembleServiceListQueryResponse u32Sid=%x bNotRXFlag=%d u8PSFlag=%d rServiceInfo.u16Scids=%d rServiceInfo.u8ASUFlags=%d rServiceInfo.u8PresetNum=%d",
					rMecaProgrammeService._u32Id,rServiceInfo.bNotRXFlag,rServiceInfo.u8PSFlag,rServiceInfo.u16Scids,rServiceInfo.u8ASUFlags,rServiceInfo.u8PresetNum));
			
			//CRQ 431566 label replace
			if (((DAB_CHNINFO_UPDATE_SUPERVISION_TIMER_MS)>0) && (rServiceInfo.ServiceLabel.b_label[0]=='\0'))
			{
				rServiceInfo.ServiceLabel.b_label_status = 0x01;
				rServiceInfo.ServiceLabel.b_charset = enDab_CharSetEBU;
				sprintf((char *)rServiceInfo.ServiceLabel.b_label, "%x", rServiceInfo.rProgrammeService._u32Id);
				ETG_TRACE_USR4(("replaced label if empty %s", rServiceInfo.ServiceLabel.b_label));
			}

		rEnsSrvInfo.lServiceList.insert(rMecaProgrammeService);
		rServiceInformation.vecServiceInfo.push_back(rServiceInfo);
	}

		vHandleEnsembleServiceInfo(&rEnsSrvInfo);
		vHandleServiceInfo(&rServiceInformation);

            vSendUpdateSDS();

		ETG_TRACE_USR4(("dabdrv_chnList::vOnEnsembleServiceListQueryResponse _u8NumEnsembles=%d _u8EnsembleCounter=%d",_u8NumEnsembles,_u8EnsembleCounter));
		if(_u8EnsembleCounter==_u8NumEnsembles){
			ETG_TRACE_USR4(("dabdrv_chnList::vOnEnsembleServiceListQueryResponse _u32PtyFilter=%d _u8EnsembleCounter=%d",_u32PtyFilter,_u8EnsembleCounter));						
			if(_u32PtyFilter==0){
				_mapEnsembleInfo.clear();
				_mapServiceInfo.clear();
			}
			if(_rPreSetChnListCmd.enOperation ==tenListOperation_OPEN){
			vSetChnList(&_rPreSetChnListCmd);
			}
		}
	}
		catch(std::bad_alloc)
			{
				ETG_TRACE_USR1(("Exception caught"));
			}

		if(pu8Data != NULL)
		{
			OSAL_DELETE[] pu8Data;
		}
	}

	else{
		ETG_TRACE_USR1(("dabdrv_chnList(vOnEnsembleServiceListQueryResponse)Global list info query failed: enQueryResponse=%d u16QueryNumRows=%d",
					ETG_CENUM(tenMeca_DbQueryResponse, poRDbQuery->enQueryResponse), poRDbQuery->u16QueryNumRows));
	}
        dabdrv_mecaIf::instance()->vSendMecaCommandCtor(trMeca_CRdmGetRdmInfo());
	/*if previous requests are cleared and there is a event, execute ensemble list query*/
	if((_bQueryEnsembleList)&&
		(_mapQueryTagInfo.size()==0))
	{
		ETG_TRACE_USR4(("dabdrv_chnList::vOnEnsembleServiceListQueryResponse:Ensmeble list  Query for previous trigger"));
		//Evaluate the ensemble list
		trMsgDrvDbQuery rDbQueryEnsEval(Query_EnsembleList_Evaluate);
		DAB_vCallMsg(rDbQueryEnsEval);	
		_bQueryEnsembleList = FALSE;
	}
}

//Added by jab4kor
tVoid dabdrv_chnList::vHandleEnsembleServiceInfo(trEnsembleServiceList* poEnsembleServiceInfo) {

	ETG_TRACE_USR4(("dabdrv_chnList::vProcess(trMsgDrvEnsembleServiceInfo)"));

	trMecaEnsemble rRefEnsemble=poEnsembleServiceInfo->rEnsemble;

	DAB_IF_FIND_MAP(trMecaEnsemble, trChnListEnsembleInfo, iterEnsMap, _mapEnsembleInfo, rRefEnsemble) {
	// do nothing
	} else {
			ETG_TRACE_USR4(("dabdrv_chnList::vHandleEnsembleServiceInfo(trEnsembleServiceInfo)ad unknown ens=0x%08x",
							rRefEnsemble.u32GetID()));
			trChnListEnsembleInfo &rEnsInfo=_mapEnsembleInfo[rRefEnsemble];
			rEnsInfo.rEnsemble=rRefEnsemble;
			iterEnsMap=_mapEnsembleInfo.find(rRefEnsemble);
		}

		set<trMecaProgrammeService> &lNewServiceList=poEnsembleServiceInfo->lServiceList;
		ETG_TRACE_USR4(("dabdrv_chnList::vHandleEnsembleServiceInfo(trEnsembleServiceInfo:lNewServiceList:)"));
		DAB_FOREACH_CONST(set<trMecaProgrammeService>, iter1, lNewServiceList) {
			(*iter1).vTrace();
		}

		set<trMecaProgrammeService> &lOldServiceList =  iterEnsMap->second.lServiceList;
		ETG_TRACE_USR4(("dabdrv_chnList::vHandleEnsembleServiceInfo(trEnsembleServiceInfo:lOldServiceList):"));
		DAB_FOREACH(set<trMecaProgrammeService>, iter2, lOldServiceList) {
			(*iter2).vTrace();
		}

		if (lNewServiceList ==lOldServiceList) {
			return;
		}

		    vMarkChanged();

		set<trMecaProgrammeService>lLostServices;
		DAB_set_difference(lOldServiceList.begin(), lOldServiceList.end(),
						   lNewServiceList.begin(), lNewServiceList.end(),
						   lLostServices);
		DAB_FOREACH(set<trMecaProgrammeService>, iterLostServices, lLostServices) {
			// remove reference to this ensemble from all lost services
			ETG_TRACE_USR4(("dabdrv_chnList::vHandleEnsembleServiceInfo(trEnsembleServiceInfo):lostSrv=0x%08x",
							iterLostServices->u32GetSID()));
			vChangeSrvEnsList(*iterLostServices, rRefEnsemble, TRUE);   //jab4kor: Check if this is required

		}
		// add the new services of this ensemble to service-list
		set<trMecaProgrammeService>lNewServices;
		DAB_set_difference(lNewServiceList.begin(), lNewServiceList.end(),
						   lOldServiceList.begin(), lOldServiceList.end(),
						   lNewServices);

		DAB_FOREACH(set<trMecaProgrammeService>, iterNewServices, lNewServices) {
			trChnListKey rListkey(*iterNewServices);
				if(!_mapServiceInfo.count(rListkey))  //NCG3D-86269
				{
					_mapServiceInfo.insert(std::make_pair(rListkey, trChnListServiceInfo()));				
				}
				//trChnListServiceInfo &rChnInfo = _mapServiceInfo[rListkey];
					_mapServiceInfo[rListkey].rProgService=*iterNewServices;
				ETG_TRACE_USR4(("dabdrv_chnList::vHandleEnsembleServiceInfo(trEnsembleServiceInfo):newSrv=0x%08x",
								_mapServiceInfo[rListkey].rProgService.u32GetSID()));
				vChangeSrvEnsList(*iterNewServices, rRefEnsemble);
			}

		// update the serviceList of this ensemble
		//if (!lNewServiceList.empty()) {
			iterEnsMap->second.lServiceList = lNewServiceList;
		//}
		vUpdateChnListEmit();
		//vStartSrvInfoCollect();

		    vTraceAll();
		    ETG_TRACE_USR4(("dabdrv_chnList::vHandleEnsembleServiceInfo(trEnsembleServiceInfo)END"));


}

//Added by jab4kor
tVoid dabdrv_chnList::vHandleServiceInfo(trServiceInformation* poServiceInformation) {

	DAB_FOREACH(vector<trServiceInfo>, iter, poServiceInformation->vecServiceInfo) {

		ETG_TRACE_USR4(("dabdrv_chnList(vHandleServiceInfo):got info for sid=0x%08x scids=%d",
			iter->rProgrammeService.u32GetSID(),iter->rProgrammeService.u16GetScids()));

	 tBool bChanged = FALSE;
	 trChnListKey rchnListKey(iter->rProgrammeService);
	 DAB_IF_FIND_MAP(trChnListKey, trChnListServiceInfo, iterSrvMap, _mapServiceInfo, rchnListKey) {
			ETG_TRACE_USR4(("dabdrv_chnList(vHandleServiceInfo):updating sid=0x%08x scids=%d",
					iter->rProgrammeService.u32GetSID(),iter->rProgrammeService.u16GetScids()));
			
			ETG_TRACE_USR4(("dabdrv_chnList(vHandleServiceInfo):updating _mapServiceInfo->sid=0x%08x _mapServiceInfo->scids=%d u8ASUFlags=0x%x",
				iterSrvMap->first._rMecaId._u32Id,iterSrvMap->first._u16Scids,iter->u8ASUFlags));

			trChnListServiceInfo &rServiceInfo = iterSrvMap->second;

			rServiceInfo.bServiceInfoPresent=TRUE;
			if (rServiceInfo.u16AvailableAnnoTypesMask != iter->u8ASUFlags) {
				rServiceInfo.u16AvailableAnnoTypesMask=iter->u8ASUFlags;
				bChanged=TRUE;
			}

			if (rServiceInfo.u8PTY != iter->u8PTY) {
				rServiceInfo.u8PTY=iter->u8PTY;
				bChanged=TRUE;
			}

			//if (rServiceInfo.u16SCIDI != iter->u16Scidi) {
				//rServiceInfo.u16SCIDI=iter->u16Scidi;
			//}
			rServiceInfo.enListElemUpdateState=enListElemUpdateState_Ok;
			rServiceInfo.bPSFlag=(tBool)iter->u8PSFlag;
			rServiceInfo.u16Scids=iter->u16Scids;
			rServiceInfo.u8PresetNumber=iter->u8PresetNum;
			rServiceInfo.bReception=!(iter->bNotRXFlag);
			trMecaLabel rSrvLabel;

			ETG_TRACE_USR4(("dabdrv_chnList(vHandleServiceInfo):bPSFlag=%d rServiceInfo.bReception=%d",
					rServiceInfo.bPSFlag,rServiceInfo.bReception));

			if(rServiceInfo.bPSFlag){
				ETG_TRACE_USR4(("dabdrv_chnList(vHandleServiceInfo):iter->ServiceLabel.w_8_char_abbreviation=%d",
					iter->ServiceLabel.w_8_char_abbreviation));
				rSrvLabel.vParse(iter->ServiceLabel);
				//rSrvLabel.vParse(const_cast<tU8*>(reinterpret_cast<tU8*>(&(iter->ServiceLabel))));
			}
			else{
				ETG_TRACE_USR4(("dabdrv_chnList(vHandleServiceInfo):iter->ComponentLabel.w_8_char_abbreviation=%d",
					iter->ComponentLabel.w_8_char_abbreviation));
				rSrvLabel.vParse(iter->ComponentLabel);
				//rSrvLabel.vParse(const_cast<tU8*>(reinterpret_cast<tU8*>(&(iter->ComponentLabel))));
			}

			rSrvLabel.vTrace();

			/*trMecaLabel rSrvLabel;
			rSrvLabel.vParse(const_cast<tU8*>(reinterpret_cast<tU8*>(&(iter->ServiceLabel))));

			rSrvLabel.vTrace();*/

			if (rSrvLabel.bLabelValid) {
				if (!rServiceInfo.bLabelPresent || (rServiceInfo.rLabel !=rSrvLabel )) {
					rServiceInfo.rLabel=rSrvLabel;
					rServiceInfo.bLabelPresent = TRUE;
					bChanged=TRUE;
				}
			}

			rServiceInfo.sFreqLabel = iter->sFreqLabel;
			ETG_TRACE_USR4(("dabdrv_chnList(vHandleServiceInfo):updating FreqLabel %s",rServiceInfo.sFreqLabel.c_str()));


			trMecaLabel rEnsembleLabel = iter->EnsembleLabel;
			ETG_TRACE_USR4(("dabdrv_chnList(vHandleServiceInfo):updating Ensemble Label"));
			rEnsembleLabel.vTrace();

			if (rEnsembleLabel.bLabelValid) {
				if (rServiceInfo.rEnsembleLabel !=rEnsembleLabel ) {
					rServiceInfo.rEnsembleLabel=rEnsembleLabel;
					bChanged=TRUE;
				}
			}


	#ifdef DAB_REPLACE_INVALID_LABEL_BY_SID
			else {
				trMecaLabel rDefaultLabel=trMecaLabel(poSrvInfo->rProgrammeService.u32GetSID());
				rServiceInfo.enListElemUpdateState=enListElemUpdateState_Changed;

				if (rServiceInfo.rLabel != rDefaultLabel) {
					rServiceInfo.rLabel = rDefaultLabel;
					bChanged=TRUE;
				}
			}
	#endif

			if (rServiceInfo.rLabel.bLabelValid && _rMonitoredSrvId.bIsValid() && iter->rProgrammeService == _rMonitoredSrvId) {
				trMsgDrvRspChnListPsidMonitor rMonitor(_rMonitoredSrvId);
				_rMonitoredSrvId.vInvalidate();
				DAB_vCallMsg(rMonitor);
			}
			if (bChanged) {
				vMarkChanged();
				if (iter->rProgrammeService == _rCurrentSrvId) {
					trMsgDrvRspChnListCurrentSrvChanged rSrvChanged(_rCurrentSrvId);
					DAB_vCallMsg(rSrvChanged);
				}
			}
			vTraceAll();
		} else {
			ETG_TRACE_USR4(("dabdrv_chnList(vHandleServiceInfo):unknown sid=0x%08x",
					iter->rProgrammeService.u32GetSID()));
		}
	}
}

tBool dabdrv_chnList::isEnsmbleReceptionStatusUpdated(vector<trChnListEnsembleInfo>&oldChnlListEnsembleInfo, set<trMecaEnsembleListElement>&newEnsembleInfoSet){

	DAB_FOREACH(vector<trChnListEnsembleInfo>, iterOld, oldChnlListEnsembleInfo){
		//const trMecaEnsemble &rEns = iterOld.rEnsemble;
		DAB_IF_FIND_SET(trMecaEnsembleListElement, iterNew, newEnsembleInfoSet, iterOld->rEnsemble){
			if(iterOld->bReception != iterNew->bReception)
				return TRUE;
		}
	}
	return FALSE;
}

//Added by jab4kor
tVoid dabdrv_chnList::vHandleEnsembleInfo(trEnsembleInfo* poMsgDrvEnsembleInfo) {
	 ETG_TRACE_USR4(("dabdrv_chnList(trMsgDrvEnsembleInfo) START"));

	 // get a sorted vector of all known trMecaEnsemble
	vector<trMecaEnsemble> vOldEnsembles;
	vector<trChnListEnsembleInfo> oldChnlListEnsembleInfoSet;
	DAB_FOREACH_MAP(trMecaEnsemble, trChnListEnsembleInfo, oldEnsemblesIter, _mapEnsembleInfo) {
		vOldEnsembles.push_back(oldEnsemblesIter->first);
		oldChnlListEnsembleInfoSet.push_back(oldEnsemblesIter->second);
		ETG_TRACE_USR4(("oldEnsemble=0x%08x", oldEnsemblesIter->first.u32GetID()));
	}

    // sort the received vector (todo: should already be sorted)
    set<trMecaEnsembleListElement> oNewEnsembleInfoSet;
    DAB_FOREACH(vector<trMecaEnsembleListElement>, iter, poMsgDrvEnsembleInfo->lEnsembleList) {
        oNewEnsembleInfoSet.insert(*iter);
    }

    // get a sorted vector of all trMecaEnsemble in poEnsembleList
   vector<trMecaEnsemble>vNewEnsembles;
   DAB_FOREACH(set<trMecaEnsembleListElement>, newEnsemblesIter, oNewEnsembleInfoSet) {
	   vNewEnsembles.push_back((*newEnsemblesIter).rEnsemble);
	   ETG_TRACE_USR4(("newEnsemble=0x%08x", newEnsemblesIter->rEnsemble.u32GetID()));

   }

   

   if ((vNewEnsembles != vOldEnsembles) ||
		   (isEnsmbleReceptionStatusUpdated(oldChnlListEnsembleInfoSet,oNewEnsembleInfoSet))) {
	   ETG_TRACE_USR4(("DIFF!"));

	   set<trMecaEnsemble> vLostEnsembles;
	   // find ensembles that are no longer there
	   DAB_set_difference(vOldEnsembles.begin(), vOldEnsembles.end(),
						  vNewEnsembles.begin(), vNewEnsembles.end(),
						  vLostEnsembles);
	   // remove from vOldEnsembles the lost ensembles
	   DAB_removeFromSet(vOldEnsembles,vLostEnsembles);
	   // find the added ensembles
	   set<trMecaEnsemble> vAddEnsembles;
	   DAB_set_difference(vNewEnsembles.begin(), vNewEnsembles.end(),
						  vOldEnsembles.begin(), vOldEnsembles.end(),
						  vAddEnsembles);

	   // store information of added ensembles
	   DAB_FOREACH(set<trMecaEnsemble>, iterAddEnsembles, vAddEnsembles) {
		   vMarkChanged();
		   ETG_TRACE_USR4(("addEnsemble=0x%08x", iterAddEnsembles->u32GetID()));
		   const trMecaEnsemble &rEns =*iterAddEnsembles;
		   trChnListEnsembleInfo rEnsInfo;
		   rEnsInfo.rEnsemble=rEns;
		   // request frequency-list
		   //vRequestEnsFreqList(rEns);

           DAB_IF_FIND_SET(trMecaEnsembleListElement, ensInfoIter, oNewEnsembleInfoSet, rEns) {
               rEnsInfo.u8Quality=ensInfoIter->u8Quality;
               rEnsInfo.u32BestFreq=ensInfoIter->u32Freq;
               rEnsInfo.bReception=ensInfoIter->bReception;
              
               rEnsInfo.u8NumberOfAudioServices=ensInfoIter->u8NumberOfAudioServices;
               rEnsInfo.u8NumberOfDataServices=ensInfoIter->u8NumberOfDataServices;
			   rEnsInfo.Sihash = ensInfoIter->sSihash;
			   rEnsInfo.sFreqLabel = ensInfoIter->sFreqLabel;
               if (ensInfoIter->rLabel.bLabelValid) {
                if (rEnsInfo.rLabel !=ensInfoIter->rLabel ) {
                    rEnsInfo.rLabel=ensInfoIter->rLabel;
                }
               ETG_TRACE_USR4(("qual=0x%08x BestFrequency=%d u8NumberOfAudioServices=%d u8NumberOfDataServices=%d newReception = %d Sihash %s"
                       , rEnsInfo.u8Quality
                       ,rEnsInfo.u32BestFreq
                       ,rEnsInfo.u8NumberOfAudioServices
                       ,rEnsInfo.u8NumberOfDataServices,
                       rEnsInfo.bReception,
					   rEnsInfo.Sihash.c_str()));
           }
           _mapEnsembleInfo[rEns]=rEnsInfo;
           }
       }
	   vSendUpdateSDS();

	   // remove references to lost ensembles
	   DAB_FOREACH(set<trMecaEnsemble>, iterLostEnsembles, vLostEnsembles) {
		   vMarkChanged();
		   ETG_TRACE_USR4(("lostEnsemble=0x%08x", iterLostEnsembles->u32GetID()));

		   // for each lost ensemble ...
		   trMecaEnsemble rLostEnsemble=*iterLostEnsembles;
		   set<trMecaProgrammeService>lLostServiceList=_mapEnsembleInfo[rLostEnsemble].lServiceList;

		//update ensemble quality
		vUpdateFrozenEnsListQuality(rLostEnsemble);

		   // delete the ensemble from each service it appears in
		   DAB_FOREACH(set<trMecaProgrammeService>, iterLostServices, lLostServiceList) {
			   const trMecaProgrammeService &rLostSrv=*iterLostServices;
			   ETG_TRACE_USR4(("lostService=0x%08x", rLostSrv.u32GetSID()));

			   // update service
			   vChangeSrvEnsList(rLostSrv, rLostEnsemble, TRUE);
		   }
		   _mapEnsembleInfo.erase(rLostEnsemble);
	   }
	   //vStartSrvInfoCollect();  //jab4kor: Check if required
  // }

	   // update quality of all services of ensembles that are still there
	   DAB_FOREACH(vector<trMecaEnsemble>, iterKeptEnsembles, vOldEnsembles) {
		   trMecaEnsemble &rEns=*iterKeptEnsembles;
		   trChnListEnsembleInfo &rEnsInfo=_mapEnsembleInfo[rEns];
		   tU8 u8OldEnsQuality=rEnsInfo.u8Quality;
		   tBool bOldReception=rEnsInfo.bReception;
		   tU8 u8NewEnsQuality=0;
		   tBool bNewReception=FALSE;
		   DAB_IF_FIND_SET(trMecaEnsembleListElement, ensInfoIter, oNewEnsembleInfoSet, rEns) {
			   u8NewEnsQuality=ensInfoIter->u8Quality;
			   bNewReception=ensInfoIter->bReception;
		   }

		   ETG_TRACE_USR4(("Ensemble= 0x%08x  --> u8OldEnsQuality=0x%x  u8NewEnsQuality=0x%x", rEnsInfo.rEnsemble._u32Id, u8OldEnsQuality, u8NewEnsQuality));
		   ETG_TRACE_USR4(("bOldReception= %d  --> bNewReception = %d", bOldReception,bNewReception));
		   //update ensemble quality
		   vUpdateFrozenEnsListQuality(rEns,u8NewEnsQuality,bNewReception);

		   if (u8OldEnsQuality != u8NewEnsQuality || bOldReception != bNewReception) {
			   // request frequency-list
			   vMarkChanged();
			   // the reception-quality of the ensemble has changed
			   // update reception-quality of the ensemble
			   rEnsInfo.u8Quality=u8NewEnsQuality;
			   rEnsInfo.bReception=bNewReception;
			   // update reception-quality of each service of the ensemble
			   set<trMecaProgrammeService>&lServiceList=rEnsInfo.lServiceList;
			   DAB_FOREACH(set<trMecaProgrammeService>, iterServices, lServiceList) {
				   vChangeSrvEnsList(*iterServices, rEns);
			   }
		   }

			/*if (bOldReception != bNewReception) {
			   vMarkChanged();
			   // the reception of the ensemble has changed
			   // update reception of the ensemble
			   rEnsInfo.bReception=bNewReception;
			   ETG_TRACE_USR4(("*********************************vChangeSrvEnsList START*********************"));
			   vTraceAll();
			   ETG_TRACE_USR4(("*********************************vChangeSrvEnsList END*********************"));
			   // update reception of each service of the ensemble
			   set<trMecaProgrammeService>&lServiceList=rEnsInfo.lServiceList;
			   DAB_FOREACH(set<trMecaProgrammeService>, iterServices, lServiceList) {
				   vChangeSrvEnsList(*iterServices, rEns);
			   }
		   }*/
	   }
	}

   vUpdateChnListEmit();

   ETG_TRACE_USR4(("dabdrv_chnList(trMsgDrvEnsembleInfo) END"));
}
#endif

#ifdef OLD_STYLE_LIST_IMPLEMENTATION
tVoid dabdrv_chnList::vProcess(trMeca_RDbEnsembleGetInfo* poEnsemble) {

    ETG_TRACE_USR4(("dabdrv_chnList(trMeca_RDbEnsembleGetInfo) START EID=%x  Receptionquality=0x%x",poEnsemble->rEnsemble._u32Id, poEnsemble->u8ReceptionQuality));

	DAB_IF_FIND_MAP(trMecaEnsemble, trChnListEnsembleInfo, iterEnsMap, _mapEnsembleInfo, poEnsemble->rEnsemble) 
	{
		ETG_TRACE_USR4(("dabdrv_chnList(trMeca_RDbEnsembleGetInfo):updating eid=0x%08x",
			poEnsemble->rEnsemble.u16GetEID()));
		trChnListEnsembleInfo &rEnsembleInfo = iterEnsMap->second;	
		/* update label info */
		if (poEnsemble->rLabel.bLabelValid) {
			if (rEnsembleInfo.rLabel !=poEnsemble->rLabel ) {
				rEnsembleInfo.rLabel=poEnsemble->rLabel;				           
			}
			/*update no of audio and data  services*/
			rEnsembleInfo.u8NumberOfAudioServices = poEnsemble->u8NumProgServices;
			rEnsembleInfo.u8NumberOfDataServices = poEnsemble->u8NumDataServices;
		}
		//Work-around by Surekha
   		trMsgDrvCmdSetEnsembleId rSetEnsembleId;
		rSetEnsembleId.rEnsemble = poEnsemble->rEnsemble;
		DAB_vCallMsg(rSetEnsembleId);

         

	}




    //trMeca_CDbGetDbEnsembleList oCGetEnsembleList;
    //dabdrv_mecaIf::instance()->vSendMecaCommand(oCGetEnsembleList);
}

/*tVoid dabdrv_chnList::vProcess(trMeca_RDbEnsembleGetFrequencyList *poEnsFreqList) {
    trMecaEnsemble &rRefEnsemble=poEnsFreqList->rRefEnsemble;
    
    ETG_TRACE_USR4(("dabdrv_chnList::vProcess(trMeca_RDbEnsembleGetFrequencyList)ens=0x%08x  u8NumFrequencies=%d",
                    rRefEnsemble.u32GetID(), poEnsFreqList->u8NumFrequencies));

    DAB_IF_FIND_MAP(trMecaEnsemble, trChnListEnsembleInfo, iterSrvMap,  _mapEnsembleInfo, rRefEnsemble) {
        // do nothing
    } else {
        ETG_TRACE_USR4(("dabdrv_chnList::vProcess(trMeca_RDbEnsembleGetFrequencyList)ignore ens=0x%08x",
                        rRefEnsemble.u32GetID()));
        return;
    }
    tU8 u8BestReception=0;
    tU32 u32BestFreq=0;
    for (tU8 u8Index=0; u8Index<poEnsFreqList->u8NumFrequencies;u8Index++) {
        tU8 u8Reception=poEnsFreqList->lReceptionQualityList[u8Index];
        if (u8Reception>=u8BestReception) {
            u8BestReception=u8Reception;
            u32BestFreq=poEnsFreqList->lFrequencyList[u8Index];
        }
    }
    ETG_TRACE_USR4(("dabdrv_chnList::vProcess(trMeca_RDbEnsembleGetFrequencyList)ens=0x%08x bestFreq=%u",
                    rRefEnsemble.u32GetID(),
                    u32BestFreq));
    iterSrvMap->second.u32BestFreq=u32BestFreq;



}*/

tVoid dabdrv_chnList::vProcess(trMeca_RDbEnsembleGetServiceList* poServiceList) {
#if 0
    if ( poServiceList->rRefEnsemble.u8GetECC() == 0) {
        return;
    }
#endif
    if (poServiceList->enServiceType != enMeca_ServiceType_AUDIO_SERVICE) {
        return;
    }
    if (poServiceList->u8FilterId !=(tU8)enDabDbFilterId_Default) {
        return;
    }
    trMecaEnsemble &rRefEnsemble=poServiceList->rRefEnsemble;
    ETG_TRACE_USR4(("dabdrv_chnList::vProcess(trMeca_RDbEnsembleGetServiceList)ens=0x%08x",
                    rRefEnsemble.u32GetID()));

    DAB_IF_FIND_MAP(trMecaEnsemble, trChnListEnsembleInfo, iterEnsMap, _mapEnsembleInfo, poServiceList->rRefEnsemble) {
        // do nothing
    } else {

        ETG_TRACE_USR4(("dabdrv_chnList::vProcess(trMeca_RDbEnsembleGetServiceList)ad unknown ens=0x%08x",
                        rRefEnsemble.u32GetID()));
        trChnListEnsembleInfo &rEnsInfo=_mapEnsembleInfo[rRefEnsemble];
        rEnsInfo.rEnsemble=rRefEnsemble;
        iterEnsMap=_mapEnsembleInfo.find(rRefEnsemble);
    }

    set<trMecaProgrammeService> &lNewServiceList=poServiceList->lServiceList;
    ETG_TRACE_USR4(("dabdrv_chnList::vProcess(trMeca_RDbEnsembleGetServiceList:lNewServiceList:)"));
    DAB_FOREACH_CONST(set<trMecaProgrammeService>, iter, lNewServiceList) {
        (*iter).vTrace();
    } 

    set<trMecaProgrammeService> &lOldServiceList =  iterEnsMap->second.lServiceList;
    ETG_TRACE_USR4(("dabdrv_chnList::vProcess(trMeca_RDbEnsembleGetServiceList:lOldServiceList):")); 
    DAB_FOREACH(set<trMecaProgrammeService>, iter2, lOldServiceList) {
        (*iter2).vTrace();
    }


    if (lNewServiceList ==lOldServiceList) {
        return;
    }

    vMarkChanged();

    set<trMecaProgrammeService>lLostServices;
    DAB_set_difference(lOldServiceList.begin(), lOldServiceList.end(), 
                       lNewServiceList.begin(), lNewServiceList.end(), 
                       lLostServices);
    DAB_FOREACH(set<trMecaProgrammeService>, iterLostServices, lLostServices) {
        // remove reference to this ensemble from all lost services
        ETG_TRACE_USR4(("dabdrv_chnList::vProcess(trMeca_RDbEnsembleGetServiceList):lostSrv=0x%08x",
                        iterLostServices->u32GetSID()));
        vChangeSrvEnsList(*iterLostServices, rRefEnsemble, TRUE);

    }
    // add the new services of this ensemble to service-list
    set<trMecaProgrammeService>lNewServices;
    DAB_set_difference(lNewServiceList.begin(), lNewServiceList.end(), 
                       lOldServiceList.begin(), lOldServiceList.end(),
                       lNewServices);


    DAB_FOREACH(set<trMecaProgrammeService>, iterNewServices, lNewServices) {
        trChnListServiceInfo &rChnInfo = _mapServiceInfo[trMecaId(*iterNewServices)];
        rChnInfo.rProgService=*iterNewServices;
        ETG_TRACE_USR4(("dabdrv_chnList::vProcess(trMeca_RDbEnsembleGetServiceList):newSrv=0x%08x",
                        iterNewServices->u32GetSID()));
        vChangeSrvEnsList(*iterNewServices, rRefEnsemble);
    }

    // update the serviceList of this ensemble
    if (!lNewServiceList.empty()) {
        iterEnsMap->second.lServiceList = lNewServiceList;
    }
    vUpdateChnListEmit();
    vStartSrvInfoCollect();
    
    // todo: polling of labels in current ensemble

    vTraceAll();
    ETG_TRACE_USR4(("dabdrv_chnList::vProcess(trMeca_RDbEnsembleGetServiceList)END"));

}
#endif

tU16 dabdrv_chnList::u16GetFrozenListId(trChnListKey const &rchnListKey) {
    tU16 u16ResId=0;
    ETG_TRACE_USR4(("dabdrv_chnList::u16GetFrozenListId  rchnListKey=0x%x scids=%d _bFrozenChnListOpen = %d", rchnListKey._rMecaId.u32GetSID(), rchnListKey._u16Scids,_bFrozenChnListOpen));
    if (_bFrozenChnListOpen) {
		DAB_IF_FIND_MAP(trChnListKey, tU16, iter, _mapFrozenIndexBySrv, rchnListKey) {
			u16ResId = (tU16)(iter->second + 1);
		}
		else return 0;
    }
	else{
		DAB_FOREACH_MAP_CONST(trChnListKey, trChnListServiceInfo,iterSrvMap, _mapServiceInfo) {
			u16ResId++;
			if(iterSrvMap->first==rchnListKey){
				break;
			}			
		}
	}
    ETG_TRACE_USR4(("dabdrv_chnList::u16GetFrozenListId END u16ResId = 0x%x",u16ResId));
    return u16ResId;
}

tU16 dabdrv_chnList::u16GetFrozenEnsembleListId(trMecaEnsemble const &rEnsemble) {
    tU16 u16ResId=0;
    if (_bFrozenEnsmbleListOpen) {
        DAB_IF_FIND_MAP(trMecaEnsemble, tU16, iter, _mapFrozenIndexByEnsemble, rEnsemble) {
            u16ResId = (tU16)(iter->second + 1);
        }
    }

    return u16ResId;
}

// checks reception-quality of all ensembles the service belongs to and stores the maximum in 
// the service-info.
trReceptionState dabdrv_chnList::rUpdateReceptionQuality(trChnListServiceInfo &rServiceInfo) {
	trReceptionState rReceptionState;
    //tU8 u8ResQuality=0;
    ETG_TRACE_USR4(("dabdrv_chnList::u8UpdateReceptionQuality:sid=0x%08x old=0x%02x START",
                    rServiceInfo.rProgService.u32GetSID(), rServiceInfo.u8ReceptionQuality));
    DAB_FOREACH(set<trMecaEnsemble>, ensIter, rServiceInfo.lEnsembles) {
        const trMecaEnsemble &rEns=*ensIter;
        DAB_IF_FIND_MAP(trMecaEnsemble, trChnListEnsembleInfo, ensInfoIter, _mapEnsembleInfo, rEns) {
            trChnListEnsembleInfo &rEnsInfo=ensInfoIter->second;
            tU8 u8Qual=rEnsInfo.u8Quality;
			rReceptionState.bReception=rEnsInfo.bReception;
            ETG_TRACE_USR4(("ens=0x%08x qual=0x%02x bReception=%d", rEns.u32GetID(), u8Qual, rReceptionState.bReception));
			//if ((tU8)enGetDbRecQuality(u8Qual)>(tU8)enGetDbRecQuality(rReceptionState.u8ReceptionQuality)) {
              //  ETG_TRACE_USR4(("u8ResQuality:0x%02x-->0x%02x", rReceptionState.u8ReceptionQuality, u8Qual));
                rReceptionState.u8ReceptionQuality=u8Qual;
            //}
        } else {
            ETG_TRACE_USR4(("ens=0x%08x not in map!!!", rEns.u32GetID()));
        }
    }
    ETG_TRACE_USR4(("dabdrv_chnList::u8UpdateReceptionQuality:sid=0x%08x service=%17s u8ResQuality=0x%02x-->0x%02x END",
                    rServiceInfo.rProgService.u32GetSID(), rServiceInfo.rLabel.pcGetCString(), rServiceInfo.u8ReceptionQuality, rReceptionState.u8ReceptionQuality));
    rServiceInfo.u8ReceptionQuality=rReceptionState.u8ReceptionQuality;
    return rReceptionState;
}

trEnsembleListElem dabdrv_chnList::rGetEnsembleInfo(trMecaEnsemble const &rMecaEnsemble)
{
	ETG_TRACE_USR4(("dabdrv_chnList::rGetEnsembleInfo: START..."));
	trEnsembleListElem rEnsInfo;

	 DAB_IF_FIND_MAP(trMecaEnsemble, trChnListEnsembleInfo, iter, _mapEnsembleInfo, rMecaEnsemble) {
        trChnListEnsembleInfo &rDbEnsInfo = iter->second;
		rEnsInfo.rEnsemble = rDbEnsInfo.rEnsemble;
		rEnsInfo.rEnsembleLabel = rDbEnsInfo.rLabel;
		rEnsInfo.u8NumberOfAudioServices = rDbEnsInfo.u8NumberOfAudioServices;
		rEnsInfo.u8NumberOfDataServices = rDbEnsInfo.u8NumberOfDataServices;
		rEnsInfo.u8ReceptionQuality = rDbEnsInfo.u8Quality;
    }
    ETG_TRACE_USR4(("dabdrv_chnList::rGetEnsembleInfo: END..."));
    rEnsInfo.vTrace();
    return rEnsInfo;
}

trChnListChnInfo dabdrv_chnList::rGetChnInfo(trMecaId const &rMecaId) {
    trChnListChnInfo rResInfo;
    rResInfo.bValid=FALSE;
	trChnListKey rchnListKey(rMecaId);
	DAB_IF_FIND_MAP(trChnListKey, trChnListServiceInfo, iter, _mapServiceInfo, rchnListKey) {
        rResInfo.bValid=TRUE;
        trChnListServiceInfo &rDbSrvInfo =iter->second;
        rResInfo.rMecaId=rMecaId;
        rResInfo.bLabelPresent=rDbSrvInfo.bLabelPresent;
        rResInfo.rLabel=rDbSrvInfo.rLabel;
        // we now take reception-quality from ensemble-list
        rResInfo.u8ReceptionQuality=rDbSrvInfo.u8ReceptionQuality;

        rResInfo.u16FrozenListId=u16GetFrozenListId(rchnListKey);
		rResInfo.u8PresetNumber=rDbSrvInfo.u8PresetNumber;
        //rResInfo.u8PresetNumber=dabdrv_presets::instance()->u8GetPresetNumber(rResInfo.rMecaId);
        rResInfo.u16AvailableAnnoTypesMask=rDbSrvInfo.u16AvailableAnnoTypesMask;
		rResInfo.u8PTY =rDbSrvInfo.u8PTY;
	}	
else
	{
		DAB_FOREACH(vector<trChnListElem>, srvListIter, _vectorFrozenSrvList)
		{
			if(srvListIter->rMecaId==rMecaId)
			{
				rResInfo.rMecaId=rMecaId;
				rResInfo.bLabelPresent=srvListIter->rLabel.bLabelValid;
				rResInfo.rLabel=srvListIter->rLabel;
				rResInfo.u16FrozenListId=u16GetFrozenListId(rchnListKey);
				rResInfo.u8PresetNumber=srvListIter->u8PresetNumber;
				rResInfo.u16AvailableAnnoTypesMask=0;
				rResInfo.u8PTY =srvListIter->u8PTY;
			}
		}
	}
    ETG_TRACE_USR4(("dabdrv_chnList::rGetChnInfo: ..."));
    rResInfo.vTrace();
    return rResInfo;
}

trChnListChnInfo dabdrv_chnList::rGetChnInfo(trMecaId const &rMecaId, tU16 u16Scids) {
    trChnListChnInfo rResInfo;
    rResInfo.bValid=FALSE;
	trChnListKey rchnListKey(rMecaId,u16Scids);
		ETG_TRACE_USR4(("dabdrv_chnList::_rMecaId._u32Id: 0x%x , _u16Scids=%d",
			rchnListKey._rMecaId._u32Id,
			rchnListKey._u16Scids));

	DAB_IF_FIND_MAP(trChnListKey, trChnListServiceInfo, iter, _mapServiceInfo, rchnListKey) {
		ETG_TRACE_USR4(("dabdrv_chnList::rGetChnInfo found in Map"));

        rResInfo.bValid=TRUE;
        trChnListServiceInfo &rDbSrvInfo =iter->second;
        rResInfo.rMecaId=rMecaId;
		rResInfo.u16Scids=u16Scids;
        rResInfo.bLabelPresent=rDbSrvInfo.bLabelPresent;
        rResInfo.rLabel=rDbSrvInfo.rLabel;
        // we now take reception-quality from ensemble-list
        rResInfo.u8ReceptionQuality=rDbSrvInfo.u8ReceptionQuality;

        rResInfo.u16FrozenListId=u16GetFrozenListId(rchnListKey);
		//rResInfo.u8PresetNumber=rDbSrvInfo.u8PresetNumber;
        rResInfo.u8PresetNumber=dabdrv_presets::instance()->u8GetPresetNumber(rResInfo.rMecaId,(tU8)(rResInfo.u16Scids));
        rResInfo.u16AvailableAnnoTypesMask=rDbSrvInfo.u16AvailableAnnoTypesMask;
		rResInfo.u8PTY =rDbSrvInfo.u8PTY;
		rResInfo.bPSFlag=rDbSrvInfo.bPSFlag;
    }
	else
	{
		DAB_FOREACH(vector<trChnListElem>, srvListIter, _vectorFrozenSrvList)
		{
			ETG_TRACE_USR4(("dabdrv_chnList::rGetChnInfo NOT found in Map"));
			if((srvListIter->rMecaId==rMecaId)&&
				(srvListIter->u16Scids == u16Scids))
			{
				ETG_TRACE_USR4(("dabdrv_chnList::rGetChnInfo found in _vectorFrozenSrvList"));
				rResInfo.rMecaId=rMecaId;
				rResInfo.bLabelPresent=srvListIter->rLabel.bLabelValid;
				rResInfo.rLabel=srvListIter->rLabel;
				rResInfo.u16FrozenListId=u16GetFrozenListId(rchnListKey);
				rResInfo.u8PresetNumber=srvListIter->u8PresetNumber;
				rResInfo.u16AvailableAnnoTypesMask=0;
				rResInfo.u8PTY =srvListIter->u8PTY;
				rResInfo.bPSFlag=srvListIter->bPSFlag;

			}
		}
	}
    ETG_TRACE_USR4(("dabdrv_chnList::rGetChnInfo: ..."));
    rResInfo.vTrace();
    return rResInfo;
}

trMecaLabel dabdrv_chnList::rGetChnLabel(trMecaId const &rMecaId) {
	trChnListKey rChnListKey(rMecaId); 
	DAB_IF_FIND_MAP(trChnListKey, trChnListServiceInfo, iter, _mapServiceInfo, rChnListKey) {
		return iter->second.rLabel;
	}
	else
	{
		//545325: If DAB reception gets lost, ADR will clear the mapService info after 10-15 mins.
		//When the Reception comes back again,MIDW takes the label from chnInfoProperty instead of taking the label from mapService which is empty.
		ETG_TRACE_USR4(("dabdrv_chnList::rGetChnLabel label not found in mapService,get the label from chninfoProperty"));
		DAB_trChnInfoProperty rChnInfoProperty=dabdrv_properties::instance()->oChnInfoProperty.oGet();
		trMecaLabel rLabel;
		rLabel = rChnInfoProperty.rLabel;
		rLabel.vTrace();
		return rLabel;
	}
}

tU16 dabdrv_chnList::u16GetActivatedElemId(tenFrozenListType rListType)  {
	tU16 u16FrozenListId= 0;
	if(rListType==enFrozenList_Ensemble)
	{
		u16FrozenListId= dabdrv_chnInfo::instance()->rGetCurrEnsembleInfo().u16FrozenListId;
	}
	else
	{
		//u16FrozenListId= dabdrv_chnInfo::instance()->rGetChnInfo().u16FrozenListId;
		trChnListChnInfo rChnListChnInfo =  dabdrv_chnInfo::instance()->rGetChnInfo();
		trChnListKey rChnListKey(rChnListChnInfo.rMecaId,rChnListChnInfo.u16Scids); 
		u16FrozenListId = u16GetFrozenListId(rChnListKey);
		dabdrv_chnInfo::instance()->vUpdateActivatedElemId(u16FrozenListId);

		//Update the frozen list element id in channelinfo
		DAB_trChnInfoProperty rProperty = dabdrv_properties::instance()->oChnInfoProperty.oGet();
		rProperty.u16FrozenListId = u16FrozenListId;
		dabdrv_properties::instance()->oChnInfoProperty.vSet(rProperty);

	}
	ETG_TRACE_USR4(("dabdrv_chnList::u16GetActivatedElemId: %d , listType=%d",
		u16FrozenListId,
		rListType));
	return u16FrozenListId;
}

trChnListElem dabdrv_chnList::rGetElemFromFrozenList(tU16 u16ElemId) {

    static trChnListElem rChnListElem;
    if (!_bFrozenChnListOpen || u16ElemId < 1) {
        return rChnListElem;
    }
	/**Vnd4kor: Solving prio 2 warning*/
    tS32 u16ElemIndex = (tU16)(u16ElemId-1);

    if (u16ElemIndex < (tS32)(_vectorFrozenSrvList.size())) {
        return _vectorFrozenSrvList[u16ElemIndex];
    }

    return rChnListElem;
}

trEnsembleListElem dabdrv_chnList::rGetEnsembleElemFromFrozenList(tU16 u16ElemId) {
	ETG_TRACE_USR4(("dabdrv_chnList::rGetEnsembleElemFromFrozenList: START..."));
    static trEnsembleListElem rEnsembleListElem;
    if (!_bFrozenEnsmbleListOpen || u16ElemId < 1) {
        return rEnsembleListElem;
    }
	/**Vnd4kor: Solving prio 2 warning*/
    tS32 u16ElemIndex = (tU16)(u16ElemId-1);

	ETG_TRACE_USR4(("dabdrv_chnList::rGetEnsembleElemFromFrozenList: ElementIndex=%d FrozenLisrSize=%d",
		u16ElemIndex,
		(tU16)(_vectorFrozenEnsmbleList.size())));
    if (u16ElemIndex < (tS32)(_vectorFrozenEnsmbleList.size())) {
        return _vectorFrozenEnsmbleList[u16ElemIndex];
    }
   ETG_TRACE_USR4(("dabdrv_chnList::rGetEnsembleElemFromFrozenList: END..."));
    return rEnsembleListElem;
}

trEnsembleListElem dabdrv_chnList::rGetEnsElemRelative(tS16 s16Steps){
	trEnsembleListElem rEnsListElem;

	if (_bFrozenEnsmbleListOpen) {
		tU16 u16FrozenListSize= (tU16)_vectorFrozenEnsmbleList.size();
		if ((!u16FrozenListSize)||(s16Steps>u16FrozenListSize)) {
			return rEnsListElem;
		}

		return _vectorFrozenEnsmbleList[s16Steps];

	}
	else
	{
		rEnsListElem = rGetMapEnsElemRelative(s16Steps);
	}
return rEnsListElem;
}
trEnsembleListElem dabdrv_chnList::rGetMapEnsElemRelative(tS16 s16Steps){
	trEnsembleListElem rEnsListElem;
		trMecaEnsemble rMecaEnsemble;
        vPrepareDynEnsList();
        trEnsListElemInfo rEnsInfo = dabdrv_chnInfo::instance()->rGetCurrEnsembleInfo();
        if (_mapDynEnsByLabel.empty()) {
            return rEnsListElem;
        }
        tBool bOk=FALSE;
		DAB_IF_FIND_MAP(trMecaLabel, trMecaEnsemble, iter, _mapDynEnsByLabel, rEnsInfo.rEnsembleElem.rEnsembleLabel) {
            bOk=TRUE;
            for (;;) {
                if (iter==_mapDynEnsByLabel.end()) {
                    iter=_mapDynEnsByLabel.begin();
                    bOk=FALSE;
                }
                if (iter->first!=rEnsInfo.rEnsembleElem.rEnsembleLabel) {
                    // wrong label, use iter anyway
                    break;
                    //return rChnListElem;
                }
				else if (iter->second==rEnsInfo.rEnsembleElem.rEnsemble) {
                    // found
                    break;
                }
                else {
                    // try next, maybe it has the same label (multimap)
                    ++iter;
                }
            }

            if (bOk) {
                while (s16Steps>0) {
                    ++iter;
                    if (iter==_mapDynEnsByLabel.end()) {
                        iter=_mapDynEnsByLabel.begin();
                    }
                    s16Steps--;
                }
                while (s16Steps<0) {
                    if (iter==_mapDynEnsByLabel.begin()) {
                        iter=_mapDynEnsByLabel.end();
                    }
                    --iter;
                    s16Steps++;
                    
                }
                rMecaEnsemble=iter->second;
            }
        } 

        if (!bOk) {
            rMecaEnsemble= (s16Steps>=0)? _mapDynEnsByLabel.begin()->second : _mapDynEnsByLabel.rbegin()->second;
        }
		if (rMecaEnsemble.bIsValid()) {
            DAB_IF_FIND_MAP(trMecaEnsemble, trChnListEnsembleInfo, iterEnsMap, _mapEnsembleInfo, rMecaEnsemble) {
                return iterEnsMap->second;
            }
            else {
                return _mapEnsembleInfo.begin()->second;
            }

	}
return rEnsListElem;
}



trChnListElem dabdrv_chnList::rGetElemRelative(tS16 s16Steps)
{
    trChnListElem rChnListElem;
    
    if (_bFrozenChnListOpen) {
        tU16 u16FrozenListSize= (tU16)_vectorFrozenSrvList.size();
        if (!u16FrozenListSize) {
            return rChnListElem;
        }
        tU16 u16ActiveElemId= u16GetActivatedElemId(enFrozenList_Service);
        if (!u16ActiveElemId) {
            return (s16Steps>=0) ?  *_vectorFrozenSrvList.begin() : *_vectorFrozenSrvList.rbegin();
        }
        tS16 s16Index= (tS16)(u16ActiveElemId-1);
        s16Index=(tS16)(s16Index + s16Steps);
        while (s16Index <0) {
            s16Index=(tS16)(s16Index + u16FrozenListSize);
        }
        tU16 u16Index=(tU16)s16Index;
        u16Index %= u16FrozenListSize;
        return _vectorFrozenSrvList[u16Index];

    }
    else {
        trMecaId rMecaId;
		tU16 u16Scids=0;
		m_u16EnsembleIndex =0;
        _enChnListType=enChnList_Global;
		vMarkChanged();
        vPrepareDynChnList();
        trChnListChnInfo rChnInfo = dabdrv_chnInfo::instance()->rGetChnInfo();
		rChnInfo.vTrace();
        if (_mapDynSrvByLabel.empty()) {
            return rChnListElem;
        }
        tBool bOk=FALSE;
        DAB_IF_FIND_MAP(trMecaLabel, trChnListKey, iter, _mapDynSrvByLabel, rChnInfo.rLabel) {
			ETG_TRACE_USR4(("dabdrv_chnList::rGetElemRelative: iter->second._rMecaId._u32Id=0x%0x  iter->second._u16Scids=%d", 
							iter->second._rMecaId._u32Id,iter->second._u16Scids));
			ETG_TRACE_USR4(("dabdrv_chnList::rGetElemRelative: rChnInfo.rMecaId._u32Id=0x%0x  rChnInfo.u16Scids=%d", 
							rChnInfo.rMecaId._u32Id,rChnInfo.u16Scids));
            bOk=TRUE;
            for (;;) {
                if (iter==_mapDynSrvByLabel.end()) {
                    iter=_mapDynSrvByLabel.begin();
                    bOk=FALSE;
                }
                if (strcmp(iter->first.pcGetCString(),rChnInfo.rLabel.pcGetCString())!=0) {
                    // wrong label, use iter anyway
                    break;
                    //return rChnListElem;
                }
				else if ((iter->second._rMecaId==rChnInfo.rMecaId) && (iter->second._u16Scids==rChnInfo.u16Scids)) {
					ETG_TRACE_USR4(("dabdrv_chnList::rGetElemRelative: Found iter->second._rMecaId._u32Id=0x%0x iter->second._u16Scids=%d",
						iter->second._rMecaId._u32Id,iter->second._u16Scids));
                    // found
                    break;
                }
                else {
                    // try next, maybe it has the same label (multimap)
                    ++iter;
                }
            }

            if (bOk) {
                while (s16Steps>0) {
                    ++iter;
                    if (iter==_mapDynSrvByLabel.end()) {
                        iter=_mapDynSrvByLabel.begin();
                    }
                    s16Steps--;
                }
                while (s16Steps<0) {
                    if (iter==_mapDynSrvByLabel.begin()) {
                        iter=_mapDynSrvByLabel.end();
                    }
                    --iter;
                    s16Steps++;
                    
                }
                rMecaId=iter->second._rMecaId;
				u16Scids=iter->second._u16Scids;
            }
        } 

        if (!bOk) {
            rMecaId= (s16Steps>=0)? _mapDynSrvByLabel.begin()->second._rMecaId : _mapDynSrvByLabel.rbegin()->second._rMecaId;
        }
        if (rMecaId.bIsValid()) {
			trChnListKey rChnListKey(rMecaId,u16Scids); 
			ETG_TRACE_USR4(("dabdrv_chnList::rGetElemRelative: rMecaId._u32Id=0x%0x  u16Scids=%d", rMecaId._u32Id,u16Scids));
			DAB_IF_FIND_MAP(trChnListKey, trChnListServiceInfo, iterServiceMap, _mapServiceInfo, rChnListKey) {
                return iterServiceMap->second;
            }
            else {
                return _mapServiceInfo.begin()->second;
            }
        }                
    }
    return rChnListElem;
}

tVoid dabdrv_chnList::vGetEnsemblesOfProgSrv(trMecaProgrammeService const &rRefProgSrv, set<trMecaEnsemble>&lEnsembles) const {
    lEnsembles.clear();
    ETG_TRACE_USR4(("dabdrv_testmode::vGetEnsemblesOfProgSrv:: rRefProgSrv:0x%08x",
                    rRefProgSrv.u32GetSID()));
	trChnListKey rChnListKey(rRefProgSrv); 
	DAB_IF_FIND_MAP_CONST(trChnListKey, trChnListServiceInfo, iter, _mapServiceInfo, rChnListKey) {
        lEnsembles= iter->second.lEnsembles; 
        DAB_FOREACH_CONST(set<trMecaEnsemble>, ensIter, lEnsembles) {
            ETG_TRACE_USR4(("dabdrv_testmode::vGetEnsemblesOfProgSrv:: ensemble:0x%08x",
                            ensIter->u32GetID()));

        }
    }
}

tU8 dabdrv_chnList::u8GetReceptionQualityOfEnsemble(trMecaEnsemble const &rRefEnsemble) const {
    tU8 u8ReceptionQuality=0;
    DAB_IF_FIND_MAP_CONST(trMecaEnsemble, trChnListEnsembleInfo, iter, _mapEnsembleInfo, rRefEnsemble) {
        u8ReceptionQuality=iter->second.u8Quality;
    }
    return u8ReceptionQuality;
}

tU32 dabdrv_chnList::u32GetFrequencyOfService(trMecaProgrammeService const &rProgSrv, tU32 u32RefFreq) const {
    // for all ensembles of service:
    // get reception-quality
    // return freq with best reception of u32RefFreq if found
    DAB_tenDbRecQuality enBestQual=DAB_tenDbRecQuality_Hide;
    tU32 u32BestFreq=0;
   // ETG_TRACE_USR4(("dabdrv_chnList::u32GetFrequencyOfService:: rProgSrv:0x%08x u32RefFreq=%u",
                  //  rProgSrv.u32GetSID(), u32RefFreq)); //jab4kor:uncomment
	trChnListKey rChnListKey(rProgSrv);
	DAB_IF_FIND_MAP_CONST(trChnListKey, trChnListServiceInfo, iterSrvInfo, _mapServiceInfo, rChnListKey) {
        DAB_FOREACH_CONST(set<trMecaEnsemble>, ensIter, iterSrvInfo->second.lEnsembles) {
            DAB_IF_FIND_MAP_CONST(trMecaEnsemble, trChnListEnsembleInfo, iterEnsInfo, _mapEnsembleInfo, *ensIter) {
                DAB_tenDbRecQuality enQual=enGetDbRecQuality(iterEnsInfo->second.u8Quality);
                if (u32RefFreq && iterEnsInfo->second.u32BestFreq==u32RefFreq) {
                    ETG_TRACE_USR4(("dabdrv_chnList::u32GetFrequencyOfService:: found ens with u32RefFreq"));
                    return u32RefFreq;
                }
                else if (enQual>=enBestQual) {
                    enBestQual=enQual;
                    u32BestFreq=iterEnsInfo->second.u32BestFreq;
                }
            }
            
        }
    }
    ETG_TRACE_USR4(("dabdrv_chnList::vGetEnsemblesOfProgSrv:: u32BestFreq:%u",
                    u32BestFreq));
    return  u32BestFreq;
    
}

    // can not be const because we provide data to _mapServiceInfo and _mapEnsembleInfo to avoid copy
tVoid dabdrv_chnList::vGetMachineData(trChnListMachineData *prChnListMachineData) {
    ETG_TRACE_USR4(("dabdrv_chnList::vGetMachineData:"));
    vTraceAll();
    prChnListMachineData->vLink(_mapServiceInfo, _mapEnsembleInfo);
};

tVoid dabdrv_chnList::vSetMachineData(trChnListMachineData const &rChnListMachineData) {
    ETG_TRACE_USR4(("dabdrv_chnList::vSetMachineData START"));
    _mapServiceInfo=rChnListMachineData.roGetSrvInfoConst();
    _mapEnsembleInfo=rChnListMachineData.roGetEnsInfoConst();
    vMarkChanged();
    DAB_vCallMsgCtor(this, trMsgDrvStopComponent());
    vTraceAll();
    ETG_TRACE_USR4(("dabdrv_chnList::vSetMachineData END"));

}

tVoid dabdrv_chnList::vGetEnsemblesWithTP(list<trMecaEnsemble>&lEnsembles, tU16 u16AnnoMask) const {
    if (!u16AnnoMask) {
        // default: check only for TP
        vSetAnsTypeBit(enMeca_AnsType_RoadTrafficFlash, &u16AnnoMask);
    }
    DAB_FOREACH_MAP_CONST(trMecaEnsemble, trChnListEnsembleInfo, ensIter, _mapEnsembleInfo) {
        // check all ensembles ...
        tU8 u8ReceptionQuality =ensIter->second.u8Quality;
        if (bIsReceptionOk(u8ReceptionQuality)) {
            // ... that are receivable ...
            DAB_FOREACH_CONST (set<trMecaProgrammeService>, srvIter, ensIter->second.lServiceList) {
                // ... check all services in the ensemble ...
				trChnListKey rChnListKey(*srvIter);
                DAB_IF_FIND_MAP_CONST(trChnListKey, trChnListServiceInfo, iter, _mapServiceInfo, rChnListKey) {
                    const trChnListServiceInfo &rDbSrvInfo =iter->second;
                    // ... if it has TP ...
                    if ((rDbSrvInfo.u16AvailableAnnoTypesMask & u16AnnoMask) == u16AnnoMask) {
                        // ... put it to the list of ensembles with the requested announcement
                        lEnsembles.push_back(ensIter->first);
                        break;
                    }
                }
            }
            
        }
    }
}



trChnListElem dabdrv_chnList::rGetFirstListElement() {

    return _mapServiceInfo.begin()->second;
    //return _mapServiceInfo.begin()->second;
}

tBool dabdrv_chnList::bGetEnsembleInfo(trMecaEnsemble const &rRefEnsemble, tU8 &u8Quality, set<trMecaProgrammeService>&lServiceList) {
    tBool bRes=FALSE;
    u8Quality=0;
    lServiceList.clear();
    
    DAB_IF_FIND_MAP(trMecaEnsemble, trChnListEnsembleInfo, iterSrvMap,  _mapEnsembleInfo, rRefEnsemble) {
        lServiceList=iterSrvMap->second.lServiceList;
        u8Quality=iterSrvMap->second.u8Quality;
        bRes=TRUE;
    }
    return bRes;
}

tBool dabdrv_chnList::bIsEnsembleContainsService(trMecaEnsemble const &rRefEnsemble, trMecaProgrammeService const &rService){
	 ETG_TRACE_USR4(("dabdrv_chnList::bIsEnsembleContainsService"));
	trChnListKey rchnListKey(rService);
	DAB_IF_FIND_MAP(trChnListKey, trChnListServiceInfo, iter_ServiceInfo, _mapServiceInfo, rchnListKey){
		trChnListServiceInfo& rServiceInfo = iter_ServiceInfo->second;
		DAB_IF_FIND_SET(trMecaEnsemble, ensIter, rServiceInfo.lEnsembles, rRefEnsemble){

			return TRUE;
			}
	    }
	return FALSE;
}

tVoid dabdrv_chnList::vProcess(trMeca_RRdmAudioPlayAsid* poMecaRRdmPlayPsid) {
	ETG_TRACE_USR4(("dabdrv_chnList::trMeca_RRdmAudioPlayAsid rMecaPgmService = %08x scids %d",poMecaRRdmPlayPsid->rMecaProgrammeService.u32GetSID(),poMecaRRdmPlayPsid->rMecaProgrammeService._u16Scids));

	trMecaProgrammeService	rMecaPgmService = poMecaRRdmPlayPsid->rMecaProgrammeService;
	tBool bFound = FALSE;
	if(PresetRecall == dabdrv_main::instance()->enGetPresetOrChannelRecallRequested()){
		DAB_FOREACH_MAP_CONST(trChnListKey, trChnListServiceInfo,iterSrvMap, _mapServiceInfo) {

		//In Suzuki even user presses the secondary component primary component should be palyed and updated to HMI
#ifndef VARIANT_S_FTR_ENABLE_FEATURE_SUZUKI
	    if((iterSrvMap->first._rMecaId._u32Id==rMecaPgmService._u32Id) && (iterSrvMap->first._u16Scids == rMecaPgmService._u16Scids )){
		
#else
            if((iterSrvMap->first._rMecaId._u32Id==rMecaPgmService._u32Id) && (iterSrvMap->second.bPSFlag)){
					rMecaPgmService._u16Scids = iterSrvMap->first._u16Scids;
			
#endif
					ETG_TRACE_USR4(("dabdrv_chnList::trMeca_RRdmAudioPlayAsid rMecaPgmService =%d",rMecaPgmService._u16Scids));
					bFound=TRUE;   //If the service is still available, then it will be found in the _mapServiceInfo
					// PSARCC30-1669 - As confirmed by Maik when manual update is going on if the user selects a preset, manual update will be aborted automatically form ADR.
					//In this case, PlayLSM should not be called and the PresetOrChannelRecallRequested
					//should not be changed to invalid when enMeca_RdmServiceState_WAITING_FOR_SERVICE so that we can block PlayLSM in main.cpp
					if (poMecaRRdmPlayPsid->enServiceState != enMeca_RdmServiceState_WAITING_FOR_SERVICE)
					{
						dabdrv_main::instance()->vSetPresetOrChannelRecallRequested(PresetOrChannelRecallInvalid);
					}
					dabdrv_chnInfo::instance()->bUpdateProperty(rMecaPgmService,FALSE);					
					break;
				}
			}
			//If it is not found in the _mapServiceInfo, then take the details from _vPresetInfos
			if(!bFound){
				dabdrv_presets::instance()->vUpdateUnavailablePresetInfo(poMecaRRdmPlayPsid);
			}
		}
	}
}


tVoid dabdrv_chnList::vProcess(trMsgSrvDataList* poMsgSrvDatalist){
	_mapLogoList = poMsgSrvDatalist->mapLogoList;
	_mapContentIDList = poMsgSrvDatalist->mmapContentIdList;
	ETG_TRACE_USR4(("vProcess(trMsgSrvLogoList* poMsgSrvDatalist) logo list Size %d",_mapLogoList.size()));
	DAB_trPresetListProperty rProperty=dabdrv_properties::instance()->oPresetListDataProperty.oGet();
	
	for(tU8 i=0; i < rProperty.vecPresetListInfo.size() ;i++){
		
		rProperty.vecPresetListInfo[i].sLogoLink =sGetLogoLink((rProperty.vecPresetListInfo[i].rPSID._u32Id) , TRUE);
	}
	dabdrv_properties::instance()->oPresetListDataProperty.vSet(rProperty);
	/*to update logo of current channel*/
	DAB_trChnInfoProperty rChnInfoProperty = dabdrv_properties::instance()->oChnInfoProperty.oGet();
	rChnInfoProperty.sLogoLink = sGetLogoLink((rChnInfoProperty.rMecaId._u32Id));
	dabdrv_properties::instance()->oChnInfoProperty.vSet(rChnInfoProperty);
	
}

string dabdrv_chnList::sGetLogoLink(tU32 u32ServiceId , tBool bRequireSmallLogo){
	ETG_TRACE_USR4(("dabdrv_chnList::sGetLogoLink:SId=%x ",u32ServiceId ));
	tU32 u32SID = (u32ServiceId & 0xFFFFFF);
	string sLogoLink="/0";
	DAB_IF_FIND_MAP(tU32, string, iterLogoMap, _mapLogoList, u32SID) {
		sLogoLink = iterLogoMap->second.c_str();				
	}
	ETG_TRACE_USR4(("dabdrv_chnList::sGetLogoLink:SId=%x , logopath =%s",u32SID ,sLogoLink.c_str()));

	tU8 u8ECC =(tU8)(u32SID >>16) ;
	tU16 u16ServiceId = (tU16)u32ServiceId;
	if(u8ECC == 0){
		ETG_TRACE_USR4(("dabdrv_chnList::sGetLogoLink:ECC is not present"));
		u8ECC = dabdrv_rdm::instance()->u8GetLastECC();
	}
	u8ECC &= 0xFF;	
	if(sLogoLink == "/0"){
		dabdrv_chnInfo::instance()->vGetStationLogoFilePath(u8ECC,u16ServiceId,sLogoLink,bRequireSmallLogo);
	}


	return sLogoLink;
}

tBool dabdrv_chnList::u8GetDSSupport(tU32 rService, tU8 u8Scid){
	
	DAB_FOREACH_MMAP(tU32, tU8, iter, _mapContentIDList) {
		ETG_TRACE_USR4(("dabdrv_chnList::u8GetDSSupport:content SId=%x  scid =%d ,Iter SId=%x  scid =%d ",rService ,u8Scid , iter->first ,iter->second));
		if((rService ==  iter->first)&&(u8Scid == iter->second ))
		{
			return TRUE;
		}
	}

	return FALSE;
}
