/**
* @copyright (c) 2015-2020 Robert Bosch Car Multimedia GmbH
* @addtogroup NavMiddleware
* @{
*/

#ifndef PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_MAPINFOS_H_
#define PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_MAPINFOS_H_

#include <stdint.h>
#include <cstring>
#include <string>
#include <sstream>
#include <vector>
#include <map>
#include <utility>
#include "InfoTypes.h"
#include "MapSettings.h"

namespace navmiddleware
{

/** The scrolling direction expressed such that 0° to 360°.
    Map will scroll to the opposite direction, so that required direction is visible on map*/
enum ScrollDirection
{
   /** Scrolls map in upward(0°) direction */
   SCROLL_DIRECTION_NORTH = 0,

   /** Scrolls map in up-right(315°) direction */
   SCROLL_DIRECTION_NORTH_EAST,

   /** Scrolls map in right(270°) direction */
   SCROLL_DIRECTION_EAST,

   /** Scrolls map in down-right(225°) direction */
   SCROLL_DIRECTION_SOUTH_EAST,

   /** Scrolls map in downward(180°) direction */
   SCROLL_DIRECTION_SOUTH,

   /** Scrolls map in down-left(135°) direction */
   SCROLL_DIRECTION_SOUTH_WEST,

   /** Scrolls map in left(90°) direction */
   SCROLL_DIRECTION_WEST,

   /** Scrolls map in up-left(45°) direction */
   SCROLL_DIRECTION_NORTH_WEST
};

inline ::std::string toString(ScrollDirection scrollDirection)
{
   switch (scrollDirection)
   {
   case SCROLL_DIRECTION_NORTH:
      return "SCROLL_DIRECTION_NORTH";
   case SCROLL_DIRECTION_NORTH_EAST:
      return "SCROLL_DIRECTION_NORTH_EAST";
   case SCROLL_DIRECTION_EAST:
      return "SCROLL_DIRECTION_EAST";
   case SCROLL_DIRECTION_SOUTH_EAST:
      return "SCROLL_DIRECTION_SOUTH_EAST";
   case SCROLL_DIRECTION_SOUTH:
      return "SCROLL_DIRECTION_SOUTH";
   case SCROLL_DIRECTION_SOUTH_WEST:
      return "SCROLL_DIRECTION_SOUTH_WEST";
   case SCROLL_DIRECTION_WEST:
      return "SCROLL_DIRECTION_WEST";
   case SCROLL_DIRECTION_NORTH_WEST:
      return "SCROLL_DIRECTION_NORTH_WEST";
   default:
      ::std::stringstream stream;
      stream << "SCROLL_DIRECTION_<" << static_cast<unsigned int>(scrollDirection) << ">";
      return stream.str();
   }
}

/** Specifies the scrolling speed for map scrolling*/
enum ScrollSpeed
{
   SCROLL_SPEED_0 = 0,
   SCROLL_SPEED_1,
   SCROLL_SPEED_2,
   SCROLL_SPEED_3,
   SCROLL_SPEED_4,
   SCROLL_SPEED_5,
   SCROLL_SPEED_6,
   SCROLL_SPEED_7,
   SCROLL_SPEED_8,
   SCROLL_SPEED_9
};

inline ::std::string toString(ScrollSpeed scrollSpeed)
{
   switch (scrollSpeed)
   {
   case SCROLL_SPEED_0:
      return "SCROLL_SPEED_0";
   case SCROLL_SPEED_1:
      return "SCROLL_SPEED_1";
   case SCROLL_SPEED_2:
      return "SCROLL_SPEED_2";
   case SCROLL_SPEED_3:
      return "SCROLL_SPEED_3";
   case SCROLL_SPEED_4:
      return "SCROLL_SPEED_4";
   case SCROLL_SPEED_5:
      return "SCROLL_SPEED_5";
   case SCROLL_SPEED_6:
      return "SCROLL_SPEED_6";
   case SCROLL_SPEED_7:
      return "SCROLL_SPEED_7";
   case SCROLL_SPEED_8:
      return "SCROLL_SPEED_8";
   case SCROLL_SPEED_9:
      return "SCROLL_SPEED_9";
   default:
      ::std::stringstream stream;
      stream << "SCROLL_SPEED_<" << static_cast<unsigned int>(scrollSpeed) << ">";
      return stream.str();
   }
}

/** Specifies if map rotation is clockwise, counter-clockwise or bounded*/
enum RotationType
{
   ROTATION_TYPE_CLOCKWISE,         /**< rotation is clockwise*/
   ROTATION_TYPE_COUNTERCLOCKWISE,  /**< rotation is counterclockwise*/
   ROTATION_TYPE_BOUNDED            /**< rotation is bounded*/
};

inline ::std::string toString(RotationType rotationType)
{
   switch (rotationType)
   {
   case ROTATION_TYPE_CLOCKWISE:
      return "ROTATION_TYPE_CLOCKWISE";
   case ROTATION_TYPE_COUNTERCLOCKWISE:
      return "ROTATION_TYPE_COUNTERCLOCKWISE";
   case ROTATION_TYPE_BOUNDED:
      return "ROTATION_TYPE_BOUNDED";
   default:
      ::std::stringstream stream;
      stream << "ROTATION_TYPE_<" << static_cast<unsigned int>(rotationType) << ">";
      return stream.str();
   }
}

/** Specifies the rotation speed for map rotation*/
enum RotationSpeed
{
   ROTATION_SPEED_1 = 1,
   ROTATION_SPEED_2,
   ROTATION_SPEED_3,
   ROTATION_SPEED_4,
   ROTATION_SPEED_5,
   ROTATION_SPEED_6,
   ROTATION_SPEED_7,
   ROTATION_SPEED_8,
   ROTATION_SPEED_9
};

inline ::std::string toString(RotationSpeed rotationSpeed)
{
   switch (rotationSpeed)
   {
   case ROTATION_SPEED_1:
      return "ROTATION_SPEED_1";
   case ROTATION_SPEED_2:
      return "ROTATION_SPEED_2";
   case ROTATION_SPEED_3:
      return "ROTATION_SPEED_3";
   case ROTATION_SPEED_4:
      return "ROTATION_SPEED_4";
   case ROTATION_SPEED_5:
      return "ROTATION_SPEED_5";
   case ROTATION_SPEED_6:
      return "ROTATION_SPEED_6";
   case ROTATION_SPEED_7:
      return "ROTATION_SPEED_7";
   case ROTATION_SPEED_8:
      return "ROTATION_SPEED_8";
   case ROTATION_SPEED_9:
      return "ROTATION_SPEED_9";
   default:
      ::std::stringstream stream;
      stream << "ROTATION_SPEED_<" << static_cast<unsigned int>(rotationSpeed) << ">";
      return stream.str();
   }
}

/**
 * The zoom type for zoom commands.
 */
enum ZoomType
{
   ZOOM_TYPE_ON_NOMINALSCALE_VALUES,            ///< map is zoomed in fixed steps, as specified in nominal scale property
   ZOOM_TYPE_ON_SUBSTEPS_NOMINALSCALE_VALUES    ///< map is zoomed in intermediate steps from fixed steps, for example,
   ///< 20% from current map scale.
};

inline ::std::string toString(ZoomType zoomType)
{
   switch (zoomType)
   {
   case ZOOM_TYPE_ON_NOMINALSCALE_VALUES:
      return "ZOOM_TYPE_ON_NOMINALSCALE_VALUES";
   case ZOOM_TYPE_ON_SUBSTEPS_NOMINALSCALE_VALUES:
      return "ZOOM_TYPE_ON_SUBSTEPS_NOMINALSCALE_VALUES";
   default:
      ::std::stringstream stream;
      stream << "ZOOM_TYPE_ON_<" << static_cast<unsigned int>(zoomType) << ">";
      return stream.str();
   }
}

/**
 * In case of showTrafficDetourInMap() call, specifies the type of tour to be displayed.
 */
enum TrafficRouteOverviewMode
{
   TRAFFIC_ROUTE_OVERVIEW_MODE__COMPLETE,    /**< show the whole route*/
   TRAFFIC_ROUTE_OVERVIEW_MODE__DETOUR       /**< show only a section of the route*/
};

inline ::std::string toString(TrafficRouteOverviewMode trafficRouteOverviewMode)
{
   switch (trafficRouteOverviewMode)
   {
   case TRAFFIC_ROUTE_OVERVIEW_MODE__COMPLETE:
      return "TRAFFIC_ROUTE_OVERVIEW_MODE__COMPLETE";
   case TRAFFIC_ROUTE_OVERVIEW_MODE__DETOUR:
      return "TRAFFIC_ROUTE_OVERVIEW_MODE__DETOUR";
   default:
      ::std::stringstream stream;
      stream << "TRAFFIC_ROUTE_OVERVIEW_MODE<" << static_cast<unsigned int>(trafficRouteOverviewMode) << ">";
      return stream.str();
   }
}

enum RoadNetworkIllustration
{
   ROAD_NETWORK_ILLUSTRATION__STANDARD,
   ROAD_NETWORK_ILLUSTRATION__MINIMAL
};

inline ::std::string toString(RoadNetworkIllustration roadNetworkIllustration)
{
   switch (roadNetworkIllustration)
   {
   case ROAD_NETWORK_ILLUSTRATION__STANDARD:
      return "ROAD_NETWORK_ILLUSTRATION__STANDARD";
   case ROAD_NETWORK_ILLUSTRATION__MINIMAL:
      return "ROAD_NETWORK_ILLUSTRATION__MINIMAL";
   default:
      ::std::stringstream stream;
      stream << "ROAD_NETWORK_ILLUSTRATION<" << static_cast<unsigned int>(roadNetworkIllustration) << ">";
      return stream.str();
   }
}

enum StreetViewMode
{
   /* only the front face of the street view will be shown */
   STREET_VIEW_MODE__BASIC,
   /* all available street view data will be shown */
   STREET_VIEW_MODE__FULL
};

inline ::std::string toString(StreetViewMode streetViewMode)
{
   switch (streetViewMode)
   {
   case STREET_VIEW_MODE__BASIC:
      return "STREET_VIEW_MODE__BASIC";
   case STREET_VIEW_MODE__FULL:
      return "STREET_VIEW_MODE__FULL";
   default:
      ::std::stringstream stream;
      stream << "STREET_VIEW_MODE__<" << static_cast<unsigned int>(streetViewMode) << ">";
      return stream.str();
   }
}

/** For displaying locations or POIs in map, specifies the startOffset of the list and the index of icon to be highlighted.*/
class VisibleListInfo
{
public:
   enum ListDisplayOption
   {
      /** Only the items given by m_startOffset, m_range will be visible and focused on in the map view,
          all others will be hidden.*/
      LIST_DISPLAY_OPTION__CURRENT_RANGE,
      /** All items of the current list will be displayed, the items given by m_startOffset, m_range will
          be focused on in the map view.*/
      LIST_DISPLAY_OPTION__ALL_ITEMS
   };

   ::std::string toString(ListDisplayOption listDisplayOption) const
   {
      switch (listDisplayOption)
      {
      case LIST_DISPLAY_OPTION__CURRENT_RANGE:
         return "LIST_DISPLAY_OPTION__CURRENT_RANGE";
      case LIST_DISPLAY_OPTION__ALL_ITEMS:
         return "LIST_DISPLAY_OPTION__ALL_ITEMS";
      default:
         ::std::stringstream stream;
         stream << "LIST_DISPLAY_OPTION__<" << static_cast<unsigned int>(listDisplayOption) << ">";
         return stream.str();
      }
   }

   struct ListItemDisplayOptions
   {
      ListItemDisplayOptions() :
         m_isListItemIndexDisplayEnabled(false),
         m_isAdditionalAttributesDisplayEnabled(false)
      {

      }

      bool getListItemIndexDisplayState() const
      {
         return m_isListItemIndexDisplayEnabled;
      }

      void setListItemIndexDisplayState(bool isEnabled)
      {
         m_isListItemIndexDisplayEnabled = isEnabled;
      }

      /** Currently not supported in Bosch Navigation.*/
      bool getAdditionalAttributesDisplayState() const
      {
         return m_isAdditionalAttributesDisplayEnabled;
      }

      /** Currently not supported in Bosch Navigation.*/
      void setAdditionalAttributesDisplayState(bool isEnabled)
      {
         m_isAdditionalAttributesDisplayEnabled = isEnabled;
      }

      bool operator==(const ListItemDisplayOptions& rhs) const
      {
         return ((m_isListItemIndexDisplayEnabled == rhs.m_isListItemIndexDisplayEnabled) &&
                 (m_isAdditionalAttributesDisplayEnabled == rhs.m_isAdditionalAttributesDisplayEnabled));
      }

      bool operator!=(const ListItemDisplayOptions& rhs) const
      {
         return !(*this == rhs);
      }


      ::std::string toString() const
      {
         ::std::stringstream stream;
         stream << "ListItemDisplayOptions: List item display=" << (m_isListItemIndexDisplayEnabled ? "true" : "false")
                << ", Additional attributes display=" << (m_isAdditionalAttributesDisplayEnabled ? "true" : "false")
                << ::std::endl;
         return stream.str();
      }

   private:
      /** controls the display of the item index*/
      bool m_isListItemIndexDisplayEnabled;
      /** controls the display of additional (POI) attributes (fuel prices, remaining parking spots ....)*/
      bool m_isAdditionalAttributesDisplayEnabled;

   };

   VisibleListInfo():
      m_startOffset(0),
      m_range(0),
      m_listDisplayOption(LIST_DISPLAY_OPTION__ALL_ITEMS)
   {}

   uint32_t getStartOffset() const
   {
      return m_startOffset;
   }

   void setStartOffset(uint32_t startOffset)
   {
      m_startOffset = startOffset;
   }

   const ValidValue<uint32_t>& getMapIconStartNumber() const
   {
      return m_mapIconStartNumber;
   }

   void setMapIconStartNumber(const ValidValue<uint32_t>& mapIconStartNumber)
   {
      m_mapIconStartNumber = mapIconStartNumber;
   }

   uint32_t getRange() const
   {
      return m_range;
   }

   void setRange(uint32_t startOffset)
   {
      m_range = startOffset;
   }

   const ValidValue<uint32_t>& getSelectedIndex() const
   {
      return m_selectedIndex;
   }

   void setSelectedIndex(const ValidValue<uint32_t>& selectedIndex)
   {
      m_selectedIndex = selectedIndex;
   }

   ListDisplayOption getListDisplayOption() const
   {
      return m_listDisplayOption;
   }

   void setListDisplayOption(ListDisplayOption listDisplayOption)
   {
      m_listDisplayOption = listDisplayOption;
   }

   const ListItemDisplayOptions& getListItemDisplayOptions() const
   {
      return m_listItemDisplayOptions;
   }

   void setListItemDisplayOptions(const ListItemDisplayOptions& listItemDisplayOptions)
   {
      m_listItemDisplayOptions = listItemDisplayOptions;
   }

   bool operator==(const VisibleListInfo& rhs) const
   {
      return ((m_startOffset == rhs.m_startOffset) &&
              (m_mapIconStartNumber == rhs.m_mapIconStartNumber) &&
              (m_range == rhs.m_range) &&
              (m_selectedIndex == rhs.m_selectedIndex) &&
              (m_listDisplayOption == rhs.m_listDisplayOption) &&
              (m_listItemDisplayOptions == rhs.m_listItemDisplayOptions));
   }

   bool operator!=(const VisibleListInfo& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "VisibleListInfo: Start offset=" << m_startOffset << ::std::endl
             << ", MapIconStartNumber = " << m_mapIconStartNumber.toString() << ::std::endl
             << ", Range=" << m_range << ::std::endl
             << ", Selected index=" << m_selectedIndex.toString() << ::std::endl
             << ", List display option=" << toString(m_listDisplayOption) << ::std::endl
             << ", List item display options=" << m_listItemDisplayOptions.toString()
             << ::std::endl;
      return stream.str();
   }

private:
   /** Start index of display list. It's value starts from 0.*/
   uint32_t   m_startOffset;
   /** Icon Start number for each page
    * If valid, this will be used to decide  map icon start number for each page
    * If invalid, m_startOffset will be used as map icon start number*/
   ValidValue<uint32_t>    m_mapIconStartNumber;
   /** Number of elements user wants to display.
    * eg. If user wants to display 3 elements then m_range value should be 3.
    * This value should be 0 to switch between different POI list.*/
   uint32_t   m_range;
   /** The index which user wants to highlight. */
   ValidValue<uint32_t>    m_selectedIndex;
   /** Indicates whether all list items or just the ones in the given range
    * should be displayed on map*/
   ListDisplayOption m_listDisplayOption;
   /** Controls the display of the list item index and additional attributes on map */
   ListItemDisplayOptions m_listItemDisplayOptions;
};

/** Alternatives to show in showRouteOverview()*/
enum ShowAlternatives
{
   SHOW_ALTERNATIVES_NONE, /**< show no alternative routes*/
   SHOW_ALTERNATIVES_ALL   /**< show all alternative routes*/
};

inline ::std::string toString(ShowAlternatives showAlternatives)
{
   switch (showAlternatives)
   {
   case SHOW_ALTERNATIVES_NONE:
      return "SHOW_ALTERNATIVES_NONE";
   case SHOW_ALTERNATIVES_ALL:
      return "SHOW_ALTERNATIVES_ALL";
   default:
      ::std::stringstream stream;
      stream << "SHOW_ALTERNATIVES_<" << static_cast<unsigned int>(showAlternatives) << ">";
      return stream.str();
   }
}

/** route segment to display in showRouteOverview()*/
enum DestinationPreference
{
   DESTINATION_PREFERENCE_CURRENT,  /**< show current destination*/
   DESTINATION_PREFERENCE_LAST      /**< show final destination*/
};

inline ::std::string toString(DestinationPreference destinationPreference)
{
   switch (destinationPreference)
   {
   case DESTINATION_PREFERENCE_CURRENT:
      return "DESTINATION_PREFERENCE_CURRENT";
   case DESTINATION_PREFERENCE_LAST:
      return "DESTINATION_PREFERENCE_LAST";
   default:
      ::std::stringstream stream;
      stream << "DESTINATION_PREFERENCE_<" << static_cast<unsigned int>(destinationPreference) << ">";
      return stream.str();
   }
}

enum IntersectionMapMode
{
   INTERSECTION_MAP_MODE_MANUAL,    ///< show intersection map split screen when user presses map hard key or
   ///< when it is enabled from map settings
   INTERSECTION_MAP_MODE_AUTOMATIC  ///< show intersection map split screen automatically
};

inline ::std::string toString(IntersectionMapMode intersectionMapMode)
{
   switch (intersectionMapMode)
   {
   case INTERSECTION_MAP_MODE_MANUAL:
      return "INTERSECTION_MAP_MODE_MANUAL";
   case INTERSECTION_MAP_MODE_AUTOMATIC:
      return "INTERSECTION_MAP_MODE_AUTOMATIC";
   default:
      ::std::stringstream stream;
      stream << "INTERSECTION_MAP_MODE_<" << static_cast<unsigned int>(intersectionMapMode) << ">";
      return stream.str();
   }
}

/** In addAvoidAreaPreview(), specify if avoid area should include freeways or not*/
enum AvoidAreaPreviewStyle
{
   AVOID_AREA_PREVIEW_STYLE_ALLOW_FREEWAYS = 0,
   AVOID_AREA_PREVIEW_STYLE_DENY_FREEWAYS
};

inline ::std::string toString(AvoidAreaPreviewStyle avoidAreaPreviewStyle)
{
   switch (avoidAreaPreviewStyle)
   {
   case AVOID_AREA_PREVIEW_STYLE_ALLOW_FREEWAYS:
      return "AVOID_AREA_PREVIEW_STYLE_ALLOW_FREEWAYS";
   case AVOID_AREA_PREVIEW_STYLE_DENY_FREEWAYS:
      return "AVOID_AREA_PREVIEW_STYLE_DENY_FREEWAYS";
   default:
      ::std::stringstream stream;
      stream << "AVOID_AREA_PREVIEW_STYLE_<" << static_cast<unsigned int>(avoidAreaPreviewStyle) << ">";
      return stream.str();
   }
}

/* Defines the line attributes */
class LineStyle
{
public:
   LineStyle()
      : m_width(0)
   {}

   LineStyle(const Color& color, uint16_t width)
      : m_color(color), m_width(width)
   {}

   const Color& getColor() const
   {
      return m_color;
   }

   void setColor(const Color& color)
   {
      m_color = color;
   }

   uint16_t getWidth() const
   {
      return m_width;
   }

   void setWidth(uint16_t width)
   {
      m_width = width;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "LineStyle: Color=" << m_color.toString()
             << ", Width=" << m_width
             << ::std::endl;
      return stream.str();
   }

   bool operator==(const LineStyle& rhs) const
   {
      return ((m_color == rhs.m_color) &&
              (m_width == rhs.m_width));
   }

   bool operator!=(const LineStyle& rhs) const
   {
      return !(*this == rhs);
   }

private:
   Color       m_color;    /* Specifies the color of the line in RGBA format */
   uint16_t    m_width;    /* Width of the line in pixel */
};

/* Defines the polygon attributes */
class PolyStyle
{
public:
   PolyStyle() {}

   explicit PolyStyle(const Color& color)
      : m_color(color)
   { }

   const Color& getColor() const
   {
      return m_color;
   }

   void setColor(const Color& color)
   {
      m_color = color;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "PolyStyle: Color=" << m_color.toString() << ::std::endl;
      return stream.str();
   }

   bool operator==(const PolyStyle& rhs) const
   {
      return (m_color == rhs.m_color);
   }

   bool operator!=(const PolyStyle& rhs) const
   {
      return !(*this == rhs);
   }

private:
   Color    m_color;    /* Specifies the color of the area in RGBA format */
};

/* Specifies the coordinates and style information for the dynamic area to be shown on the map.
 * Either LineStyle or PolyStyle (or both) must be valid.
 * If neither LineStyle nor PolyStyle is valid, no dynamic area will be seen on the map. */
class Polygon
{
public:
   Polygon() {}

   Polygon(const ::std::vector<GeoCoordinateDegree>& geocoordinateDegreeList,
           const ::navmiddleware::ValidValue<LineStyle>& lineStyle,
           const ::navmiddleware::ValidValue<PolyStyle>& polyStyle)
      : m_geocoordinateDegreeList(geocoordinateDegreeList)
      , m_lineStyle(lineStyle)
      , m_polyStyle(polyStyle)
   { }

   const ::std::vector<GeoCoordinateDegree>& getGeocoordinateDegreeList() const
   {
      return m_geocoordinateDegreeList;
   }

   void setGeocoordinateDegreeList(const ::std::vector<GeoCoordinateDegree>& dynamicElementsDataList)
   {
      m_geocoordinateDegreeList = dynamicElementsDataList;
   }

   const ::navmiddleware::ValidValue<LineStyle>& getLineStyle() const
   {
      return m_lineStyle;
   }

   void setLineStyle(const ::navmiddleware::ValidValue<LineStyle>& lineStyle)
   {
      m_lineStyle = lineStyle;
   }

   const ::navmiddleware::ValidValue<PolyStyle>& getPolyStyle() const
   {
      return m_polyStyle;
   }

   void setPolyStyle(const ::navmiddleware::ValidValue<PolyStyle>& polyStyle)
   {
      m_polyStyle = polyStyle;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "Polygon: GeoCoordinateDegree list=" << std::endl;
      for (::std::vector<GeoCoordinateDegree>::const_iterator iter = m_geocoordinateDegreeList.begin();
            iter != m_geocoordinateDegreeList.end(); ++iter)
      {
         stream << iter->toString() << ::std::endl;
      }
      stream << "LineStyle=" << (m_lineStyle.isValid() ? m_lineStyle.getValue().toString() : "Invalid") << std::endl
             << "PolyStyle=" << (m_polyStyle.isValid() ? m_polyStyle.getValue().toString() : "Invalid") << std::endl;
      return stream.str();
   }

   bool operator==(const Polygon& rhs) const
   {
      return ((m_geocoordinateDegreeList == rhs.m_geocoordinateDegreeList) &&
              (m_lineStyle == rhs.m_lineStyle) &&
              (m_polyStyle == rhs.m_polyStyle));
   }

   bool operator!=(const Polygon& rhs) const
   {
      return !(*this == rhs);
   }

private:
   /* The coordinates defining the dynamic area. */
   ::std::vector<GeoCoordinateDegree>        m_geocoordinateDegreeList;
   /* If a valid LineStyle is specified, the dynamic area will be outlined according to this style.
    * If no LineStyle is specified, the dynamic area will not be outlined. */
   ::navmiddleware::ValidValue<LineStyle>    m_lineStyle;
   /* If a valid PolyStyle is specified, the dynamic area will be filled according to this style.
    * If no PolyStyle is specified, the dynamic area will not be filled. */
   ::navmiddleware::ValidValue<PolyStyle>    m_polyStyle;
};

/* Defines the dynamic area attributes specified by the HMI */
class DynamicAreaUserStyle
{
public:
   DynamicAreaUserStyle() {}

   explicit DynamicAreaUserStyle(const Polygon& polygon)
      : m_polygon(polygon)
   {}

   const Polygon& getPolygon() const
   {
      return m_polygon;
   }

   void setPolygon(const Polygon& polygon)
   {
      m_polygon = polygon;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "DynamicAreaUserStyle: Polygon=" << m_polygon.toString() << ::std::endl;
      return stream.str();
   }

   bool operator==(const DynamicAreaUserStyle& rhs) const
   {
      return (m_polygon == rhs.m_polygon);
   }

   bool operator!=(const DynamicAreaUserStyle& rhs) const
   {
      return !(*this == rhs);
   }

private:
   Polygon  m_polygon;  /* Structure containing coordinates and style information for the polygon to be drawn. */
};

enum AddDynamicAreaUserStyleResult
{
   ADD_DYNAMIC_AREA_USER_STYLE_RESULT__UNKNOWN,    /** Status of requested add dynamic area unknown/not available. */
   ADD_DYNAMIC_AREA_USER_STYLE_RESULT__SUCCESS,    /** Request to add dynamic areas handled successfully. */
   ADD_DYNAMIC_AREA_USER_STYLE_RESULT__FAILURE     /** Request to add dynamic areas failed. */
};

inline ::std::string toString(AddDynamicAreaUserStyleResult addDynamicAreaUserStyleResult)
{
   switch (addDynamicAreaUserStyleResult)
   {
   case ADD_DYNAMIC_AREA_USER_STYLE_RESULT__UNKNOWN:
      return "ADD_DYNAMIC_AREA_USER_STYLE_RESULT__UNKNOWN";
   case ADD_DYNAMIC_AREA_USER_STYLE_RESULT__SUCCESS:
      return "ADD_DYNAMIC_AREA_USER_STYLE_RESULT__SUCCESS";
   case ADD_DYNAMIC_AREA_USER_STYLE_RESULT__FAILURE:
      return "ADD_DYNAMIC_AREA_USER_STYLE_RESULT__FAILURE";
   default:
      ::std::stringstream stream;
      stream << "ADD_DYNAMIC_AREA_USER_STYLE_RESULT__<" << static_cast<unsigned int>(addDynamicAreaUserStyleResult) << ">";
      return stream.str();
   }
}

/** Used to configure visibility of the crosshair*/
enum CrossHairVisibility
{
   CROSSHAIR_VISIBILITY_OFF = 0,
   CROSSHAIR_VISIBILITY_ON
};

inline ::std::string toString(CrossHairVisibility crossHairVisibility)
{
   switch (crossHairVisibility)
   {
   case CROSSHAIR_VISIBILITY_OFF:
      return "CROSSHAIR_VISIBILITY_OFF";
   case CROSSHAIR_VISIBILITY_ON:
      return "CROSSHAIR_VISIBILITY_ON";
   default:
      ::std::stringstream stream;
      stream << "CROSSHAIR_VISIBILITY_<" << static_cast<unsigned int>(crossHairVisibility) << ">";
      return stream.str();
   }
}

/** In freezeMap() call, specifies the visibility of the map.*/
enum MapViewVisibility
{
   MAP_VIEW_VISIBILITY__VISIBLE = 0,
   MAP_VIEW_VISIBILITY__HIDDEN
};

inline ::std::string toString(MapViewVisibility mapViewVisibility)
{
   switch (mapViewVisibility)
   {
   case MAP_VIEW_VISIBILITY__VISIBLE:
      return "MAP_VIEW_VISIBILITY__VISIBLE";
   case MAP_VIEW_VISIBILITY__HIDDEN:
      return "MAP_VIEW_VISIBILITY__HIDDEN";
   default:
      ::std::stringstream stream;
      stream << "MAP_VIEW_VISIBILITY__<" << static_cast<unsigned int>(mapViewVisibility) << ">";
      return stream.str();
   }
}

enum AvoidAreaVisibility
{
   /** no Avoid Area will be visible */
   AVOID_AREA_VISIBILITY__OFF,
   /** show Avoid Area via stored context coordinates on map */
   AVOID_AREA_VISIBILTY__MAP,
   /** HMI shows the Avoid Area Overlay */
   AVOID_AREA_VISIBILTY__HMI_OVERLAY
};

inline ::std::string toString(AvoidAreaVisibility avoidAreaVisibility)
{
   switch (avoidAreaVisibility)
   {
   case AVOID_AREA_VISIBILITY__OFF:
      return "AVOID_AREA_VISIBILITY__OFF";
   case AVOID_AREA_VISIBILTY__MAP:
      return "AVOID_AREA_VISIBILTY__MAP";
   case AVOID_AREA_VISIBILTY__HMI_OVERLAY:
      return "AVOID_AREA_VISIBILTY__HMI_OVERLAY";
   default:
      ::std::stringstream stream;
      stream << "AVOID_AREA_VISIBILTY__<" << static_cast<unsigned int>(avoidAreaVisibility) << ">";
      return stream.str();
   }
}

/**
 * Specifies whether to show the reachable area overview from carsor or destination.
 * If there are one or more waypoints, we show only the next waypoint.
 */
enum ReachableAreaVisibility
{
   REACHABLE_AREA_VISIBILITY__NONE, //[JP] Specific Extention,
   REACHABLE_AREA_VISIBILITY__CARSOR,
   REACHABLE_AREA_VISIBILITY__DESTINATION,
};

inline ::std::string toString(ReachableAreaVisibility reachableAreaVisibility)
{
   switch (reachableAreaVisibility)
   {
   case REACHABLE_AREA_VISIBILITY__NONE:
      return "REACHABLE_AREA_VISIBILITY__NONE";
   case REACHABLE_AREA_VISIBILITY__CARSOR:
      return "REACHABLE_AREA_VISIBILITY__CARSOR";
   case REACHABLE_AREA_VISIBILITY__DESTINATION:
      return "REACHABLE_AREA_VISIBILITY__DESTINATION";
   default:
      ::std::stringstream stream;
      stream << "REACHABLE_AREA_VISIBILTY__<" << static_cast<unsigned int>(reachableAreaVisibility) << ">";
      return stream.str();
   }
}

enum CoordinateAxis
{
   /** Refers to the x-axis*/
   COORDINATE_AXIS__RIGHT,
   /** Refers to the y-axis*/
   COORDINATE_AXIS__UP,
   /** Refers to the z-axis*/
   COORDINATE_AXIS__FORWARD
};

inline ::std::string toString(CoordinateAxis coordinateAxis)
{
   switch (coordinateAxis)
   {
   case COORDINATE_AXIS__RIGHT:
      return "COORDINATE_AXIS__RIGHT";
   case COORDINATE_AXIS__UP:
      return "COORDINATE_AXIS__UP";
   case COORDINATE_AXIS__FORWARD:
      return "COORDINATE_AXIS__FORWARD";
   default:
      ::std::stringstream stream;
      stream << "COORDINATE_AXIS__<" << static_cast<unsigned int>(coordinateAxis) << ">";
      return stream.str();
   }
}

/** specifies the image status in case of online images, for example, in case of street view preview and satellite image.*/
enum OnlineImageStatus
{
   ONLINE_IMAGE_STATUS__UNKNOWN,             /**< image data unknown*/
   ONLINE_IMAGE_STATUS__NO_DATA,             /**< no image data available*/
   ONLINE_IMAGE_STATUS__CONNECTION_ERROR,    /**< connection error while loading online image*/
   ONLINE_IMAGE_STATUS__COMPLETE,            /**< image data loading completed*/
   ONLINE_IMAGE_STATUS__NO_SUBSCRIPTION      /**< Online image is not subscribed*/
};

inline ::std::string toString(OnlineImageStatus onlineImageStatus)
{
   switch (onlineImageStatus)
   {
   case ONLINE_IMAGE_STATUS__UNKNOWN:
      return "ONLINE_IMAGE_STATUS__UNKNOWN";
   case ONLINE_IMAGE_STATUS__NO_DATA:
      return "ONLINE_IMAGE_STATUS__NO_DATA";
   case ONLINE_IMAGE_STATUS__CONNECTION_ERROR:
      return "ONLINE_IMAGE_STATUS__CONNECTION_ERROR";
   case ONLINE_IMAGE_STATUS__COMPLETE:
      return "ONLINE_IMAGE_STATUS__COMPLETE";
   case ONLINE_IMAGE_STATUS__NO_SUBSCRIPTION:
      return "ONLINE_IMAGE_STATUS__NO_SUBSCRIPTION";
   default:
      ::std::stringstream stream;
      stream << "ONLINE_IMAGE_STATUS__<" << static_cast<unsigned int>(onlineImageStatus) << ">";
      return stream.str();
   }
}

/** status of the pre-caching of the satellite view image.*/
class PrepareSatelliteViewInfo
{
public:
   PrepareSatelliteViewInfo():
      m_onlineImageStatus(ONLINE_IMAGE_STATUS__UNKNOWN)
   {}

   OnlineImageStatus getPrepareSatelliteViewStatus() const
   {
      return m_onlineImageStatus;
   }

   void setPrepareSatelliteViewStatus(OnlineImageStatus onlineImageStatus)
   {
      m_onlineImageStatus = onlineImageStatus;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("Satellite View Info :\n");
      stream << ::std::endl
             << "Online Image Status = " << ::navmiddleware::toString(m_onlineImageStatus) << ::std::endl;
      return stream.str();
   }

private:
   OnlineImageStatus m_onlineImageStatus;
};

/** status of the pre-caching of the street view image*/
class PrepareStreetViewStatusInfo
{
public:
   PrepareStreetViewStatusInfo():
      m_onlineImageStatus(ONLINE_IMAGE_STATUS__UNKNOWN)
   {}

   OnlineImageStatus getPrepareStreetViewStatus() const
   {
      return m_onlineImageStatus;
   }

   void setPrepareStreetViewStatus(OnlineImageStatus onlineImageStatus)
   {
      m_onlineImageStatus = onlineImageStatus;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("StreetView Status Info :\n");
      stream << ::std::endl
             << "Online Image Status = " << ::navmiddleware::toString(m_onlineImageStatus) << ::std::endl;
      return stream.str();
   }

private:
   OnlineImageStatus m_onlineImageStatus;
};

/** In configureMapLocationMarkers(), specifies details of the map location markers.*/
class MapLocationMarkers
{
public:
   MapLocationMarkers():
      m_locationMarkerType(MARKER_TYPE__POSITION)
   {}

   enum MarkerType
   {
      /** below enum is for position icon types */
      MARKER_TYPE__POSITION,
      /** below enum is for fully automated parking places */
      MARKER_TYPE__FAP
   };

   static ::std::string toString(MarkerType markerType)
   {
      switch (markerType)
      {
      case MARKER_TYPE__POSITION:
         return "MARKER_TYPE__POSITION";
      case MARKER_TYPE__FAP:
         return "MARKER_TYPE__FAP";
      default:
         ::std::stringstream stream;
         stream << "MARKER_TYPE__<" << static_cast<unsigned int>(markerType) << ">";
         return stream.str();
      }
   }

   enum MarkerDisplayMode
   {
      /** the new locations will be added with display mode as NORMAL */
      MARKER_DISPLAY_MODE__NORMAL,
      /** the new locations will be added with display mode as HIGHLIGHT */
      MARKER_DISPLAY_MODE__HIGHLIGHT
   };

   static ::std::string toString(MarkerDisplayMode markerDisplayMode)
   {
      switch (markerDisplayMode)
      {
      case MARKER_DISPLAY_MODE__NORMAL:
         return "MARKER_DISPLAY_MODE__NORMAL";
      case MARKER_DISPLAY_MODE__HIGHLIGHT:
         return "MARKER_DISPLAY_MODE__HIGHLIGHT";
      default:
         ::std::stringstream stream;
         stream << "MARKER_DISPLAY_MODE__<" << static_cast<unsigned int>(markerDisplayMode) << ">";
         return stream.str();
      }
   }

   struct Marker
   {
      Marker():
         m_markerDisplayMode(MARKER_DISPLAY_MODE__NORMAL)
      {}

      MarkerDisplayMode m_markerDisplayMode;    /**< display mode of the map location marker*/
      GeoCoordinateDegree m_geoCoordinate;      /**< geocoordinate where the marker is to be displayed*/
   };


   MapLocationMarkers(const ::std::vector< Marker>& markers, MarkerType locationMarkerType):
      m_markers(markers),
      m_locationMarkerType(locationMarkerType)
   {}

   void setMarkers(const ::std::vector< Marker >& markers)
   {
      m_markers = markers;
   }

   const ::std::vector< Marker >& getMarkers() const
   {
      return m_markers;
   }

   void setMarkerType(MarkerType locationMarkerType)
   {
      m_locationMarkerType = locationMarkerType;
   }

   MarkerType getMarkerType() const
   {
      return m_locationMarkerType;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("MapLocationMarkers payload:");
      stream << ::std::endl
             << "MarkerType " << toString(m_locationMarkerType) << ::std::endl;

      if(!m_markers.empty())
      {
         ::std::vector< Marker >::const_iterator iter = m_markers.begin();
         for(; iter != m_markers.end(); ++iter)
         {
            stream << ::std::endl
                   << "Marker List = [ " << iter->m_geoCoordinate.toString() << ", "
                   << toString(iter->m_markerDisplayMode) << " ]"
                   << ::std::endl;
         }
      }
      else
      {
         stream << "MapLocationMarker List is Empty!!! "<< ::std::endl;
      }
      return stream.str();
   }

private:
   /** if this vector is empty and MarkerType is MARKER_CONFIGURATION_OPERATION__REPLACE are given,
       all existing location markers of type m_locationMarkerType will be deleted*/
   ::std::vector<Marker> m_markers;
   MarkerType m_locationMarkerType;
};

enum MapLocationMarkerConfigurationOperation
{
   /** the new locations will be added to the already present location markers for the given MapLocationMarkerType */
   MAP_LOCATION_MARKER_CONFIGURATION_OPERATION__ADD,
   /** the new locations will replace the already present location markers for the given MapLocationMarkerType */
   MAP_LOCATION_MARKER_CONFIGURATION_OPERATION__REPLACE
};

inline ::std::string toString(MapLocationMarkerConfigurationOperation mapLocationMarkerConfigurationOperation)
{
   switch (mapLocationMarkerConfigurationOperation)
   {
   case MAP_LOCATION_MARKER_CONFIGURATION_OPERATION__ADD:
      return "MAP_LOCATION_MARKER_CONFIGURATION_OPERATION__ADD";
   case MAP_LOCATION_MARKER_CONFIGURATION_OPERATION__REPLACE:
      return "MAP_LOCATION_MARKER_CONFIGURATION_OPERATION__REPLACE";
   default:
      ::std::stringstream stream;
      stream << "MAP_LOCATION_MARKER_CONFIGURATION_OPERATION__<" << static_cast<unsigned int>(mapLocationMarkerConfigurationOperation) << ">";
      return stream.str();
   }
}

enum MapGeoFenceType
{
   /** Warning for car exiting the geo fence*/
   MAP_GEO_FENCE_TYPE__INCLUSION,
   /** Warning for car entering the geo fence */
   MAP_GEO_FENCE_TYPE__EXCLUSION
};

inline ::std::string toString(MapGeoFenceType mapGeoFenceType)
{
   switch (mapGeoFenceType)
   {
   case MAP_GEO_FENCE_TYPE__INCLUSION:
      return "MAP_GEO_FENCE_TYPE__INCLUSION";
   case MAP_GEO_FENCE_TYPE__EXCLUSION:
      return "MAP_GEO_FENCE_TYPE__EXCLUSION";
   default:
      ::std::stringstream stream;
      stream << "MAP_GEO_FENCE_TYPE__<" << static_cast<unsigned int>(mapGeoFenceType) << ">";
      return stream.str();
   }
}

struct GeoFence
{
   GeoFence(): m_geoFenceType(MAP_GEO_FENCE_TYPE__INCLUSION)
   {
   }

   GeoCoordinateDegree m_geoCoordinate1;             ///< First Coordinate
   ValidValue<GeoCoordinateDegree> m_geoCoordinate2; ///< Second coordinate: if this is not set, circle will be drawn
   ValidValue<uint32_t> m_radius;                    ///< radius of the circle in meters
   MapGeoFenceType m_geoFenceType;                   ///< specifies type of the geo-fence, that is, whether the car position
   ///< should be included or excluded from the geo-fence.
};

/** pixel position on the screen */
struct ScreenCoordinate
{
   uint16_t xPos;
   uint16_t yPos;
};

/** Contains the requested geo-coordinates after transformViewPositions() is called.*/
class TransformedViewPositionsInfo
{
public:
   TransformedViewPositionsInfo(): m_requestId(0)
   {
   }

   TransformedViewPositionsInfo(const ::std::vector< GeoCoordinateDegree >& geoCoordinates, RequestId requestId)
      : m_geoCoordinates(geoCoordinates)
      , m_requestId(requestId)
   {
   }

   void setGeoCoordinates(const ::std::vector< GeoCoordinateDegree >& geoCoordinates)
   {
      m_geoCoordinates = geoCoordinates;
   }

   const ::std::vector< GeoCoordinateDegree >& getCoordinates() const
   {
      return m_geoCoordinates;
   }

   void setRequestId(RequestId requestId)
   {
      m_requestId = requestId;
   }

   RequestId getRequestId() const
   {
      return m_requestId;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("TransformedViewPositions payload:");
      stream << ::std::endl
             << "Request Id " << m_requestId << ::std::endl;

      if(!m_geoCoordinates.empty())
      {
         ::std::vector< GeoCoordinateDegree >::const_iterator iter = m_geoCoordinates.begin();
         for(; iter != m_geoCoordinates.end(); ++iter)
         {
            stream << ::std::endl
                   << "Geo Coordinates List = [ " << iter->getLatitude() << ", "
                   << iter->getLongitude() << " ]"
                   << ::std::endl;
         }
      }
      else
      {
         stream << "Geo Coordinates List is Empty!!! "<< ::std::endl;
      }
      return stream.str();
   }

private:
   ::std::vector< GeoCoordinateDegree > m_geoCoordinates;
   RequestId m_requestId;
};

/** Provides info for drawing the avoid area overlay*/
class AvoidAreaOverlayColor
{
public:
   AvoidAreaOverlayColor():
      m_outlineWidth(0),
      m_innerFillColor(),
      m_outlineColor()
   {
   }

   AvoidAreaOverlayColor(uint16_t outlineWidth, const Color& innerFillColor, const Color& outlineColor)
      :m_outlineWidth(outlineWidth),
       m_innerFillColor(innerFillColor),
       m_outlineColor(outlineColor)
   {
   }

   bool operator==(const AvoidAreaOverlayColor& rhs ) const
   {
      return m_outlineWidth == rhs.m_outlineWidth  &&
             m_innerFillColor == rhs.m_innerFillColor &&
             m_outlineColor == rhs.m_outlineColor;
   }

   bool operator!=(const AvoidAreaOverlayColor& rhs ) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("AvoidAreaOverlayColor payload:\n");
      stream << "Outline Width = " << m_outlineWidth << ::std::endl
             << "InnerFill Color(R,G,B,Alpha) = " << m_innerFillColor.toString() << ::std::endl
             << "Outline Color(R,G,B,Alpha) = " << m_outlineColor.toString() << ::std::endl;

      return stream.str();
   }

private:
   /** outline or border thickness in pixels, 0 indicates no outline width */
   uint16_t m_outlineWidth;
   Color    m_innerFillColor;
   Color    m_outlineColor;
};

class AvoidAreaHMIOverlay
{
public:
   AvoidAreaHMIOverlay():m_avoidAreaId(0) {}

   AvoidAreaHMIOverlay(AvoidAreaId avoidAreaId,
                       const ::std::vector<ScreenCoordinate>& screenCoordinates,
                       const AvoidAreaOverlayColor& previewColorForFreewayInclusion,
                       const AvoidAreaOverlayColor& previewColorForFreewayExclusion)
      : m_avoidAreaId(avoidAreaId),
        m_overlayPosition (screenCoordinates),
        m_previewColorForFreewayInclusion(previewColorForFreewayInclusion),
        m_previewColorForFreewayExclusion(previewColorForFreewayExclusion)
   {
   }

   AvoidAreaId getAvoidAreaId() const
   {
      return m_avoidAreaId;
   }

   void setAvoidAreaId(AvoidAreaId avoidAreaId)
   {
      m_avoidAreaId = avoidAreaId;
   }

   void setOverlayPosition(const ::std::vector<ScreenCoordinate>& screenCoordinates)
   {
      m_overlayPosition = screenCoordinates;
   }

   const ::std::vector<ScreenCoordinate>& getOverlayPosition() const
   {
      return m_overlayPosition;
   }

   void setPreviewColorForFreewayInclusion(const AvoidAreaOverlayColor& previewColorForFreewayInclusion)
   {
      m_previewColorForFreewayInclusion = previewColorForFreewayInclusion;
   }

   const AvoidAreaOverlayColor& getPreviewColorForFreewayInclusion() const
   {
      return m_previewColorForFreewayInclusion;
   }

   void setPreviewColorForFreewayExclusion(const AvoidAreaOverlayColor& previewColorForFreewayExclusion)
   {
      m_previewColorForFreewayExclusion = previewColorForFreewayExclusion;
   }

   const AvoidAreaOverlayColor& getPreviewColorForFreewayExclusion() const
   {
      return m_previewColorForFreewayExclusion;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("AvoidAreaHMIOverlay payload:");
      stream << ::std::endl
             << "preview color for freeway inclusion = " << m_previewColorForFreewayInclusion.toString() << ::std::endl
             << "preview color for freeway exclusion = " << m_previewColorForFreewayExclusion.toString() << ::std::endl;
      if(!m_overlayPosition.empty())
      {
         for(unsigned int overlayPositionListIndex = 0; overlayPositionListIndex < m_overlayPosition.size(); ++overlayPositionListIndex)
         {
            stream << ::std::endl
                   << "Overlay Position List = [ " << m_overlayPosition[overlayPositionListIndex].xPos << ", "
                   << m_overlayPosition[overlayPositionListIndex].yPos << " ]"
                   << ::std::endl;
         }
      }
      else
      {
         stream << " Overlay Positions List is Empty!!! "<< ::std::endl;
      }
      return stream.str();
   }

private:
   AvoidAreaId                          m_avoidAreaId;
   ::std::vector<ScreenCoordinate>      m_overlayPosition;
   AvoidAreaOverlayColor                m_previewColorForFreewayInclusion;
   AvoidAreaOverlayColor                m_previewColorForFreewayExclusion;
};

class MapViewStatusInfos
{
public:
   enum DataLoadingStatus
   {
      /** Loading Map is unsuccessful */
      DATA_LOADING_STATUS_FAILURE = -1,
      /** Loading Map is unsuccessful */
      DATA_LOADING_STATUS_INVALID = 0,
      /** No data is available at the lat-long where the data is requested */
      DATA_LOADING_STATUS_NO_DATA,
      /** Only considers data that is sufficient to show map on screen. Includes terrain, sky, carsor, etc */
      DATA_LOADING_STATUS_BASE_DATA_LOADED,
      /** Map Data Loading is active and Minimum amount of data required for rendering Map has not been loaded yet. */
      DATA_LOADING_STATUS_STARTED,
      /** Minimum amount of data required for rendering MAP has been loaded.
       * This mainly refers to street & polygon). May also include Labels & POIs, but is not guaranteed if this is fully loaded */
      DATA_LOADING_STATUS_MINIMUM_LOADED,
      /** Map Data is completely loaded*/
      DATA_LOADING_STATUS_ALL_LOADED
   };

   static ::std::string toString(DataLoadingStatus dataLoadingStatus)
   {
      switch (dataLoadingStatus)
      {
      case DATA_LOADING_STATUS_FAILURE:
         return "DATA_LOADING_STATUS_FAILURE";
      case DATA_LOADING_STATUS_INVALID:
         return "DATA_LOADING_STATUS_INVALID";
      case DATA_LOADING_STATUS_NO_DATA:
         return "DATA_LOADING_STATUS_NO_DATA";
      case DATA_LOADING_STATUS_BASE_DATA_LOADED:
         return "DATA_LOADING_STATUS_BASE_DATA_LOADED";
      case DATA_LOADING_STATUS_STARTED:
         return "DATA_LOADING_STATUS_STARTED";
      case DATA_LOADING_STATUS_MINIMUM_LOADED:
         return "DATA_LOADING_STATUS_MINIMUM_LOADED";
      case DATA_LOADING_STATUS_ALL_LOADED:
         return "DATA_LOADING_STATUS_ALL_LOADED";
      default:
         ::std::stringstream stream;
         stream << "DATA_LOADING_STATUS_<" << static_cast<unsigned int>(dataLoadingStatus) << ">";
         return stream.str();
      }
   }

   enum DataRenderingStatus
   {
      RENDERING_STATUS_FAILURE = -1,
      RENDERING_STATUS_INVALID = 0,       ///< Invalid rendering status
      RENDERING_STATUS_ACTIVE,            ///< map view rendering is active
      RENDERING_STATUS_FREEZE,            ///< rendering is frozen
      RENDERING_STATUS_FREEZE_KEEP_DATA,  ///< rendering is frozen and map will not delete the map data.
      RENDERING_STATUS_UNCHANGED,         ///< client sets the rendering status to this value if it doesn't want the
      ///< rendering status to be changed. This is not a part of property update,
      ///< and can be set only by the client.
   };

   static ::std::string toString(DataRenderingStatus dataRenderingStatus)
   {
      switch (dataRenderingStatus)
      {
      case RENDERING_STATUS_FAILURE:
         return "RENDERING_STATUS_FAILURE";
      case RENDERING_STATUS_INVALID:
         return "RENDERING_STATUS_INVALID";
      case RENDERING_STATUS_ACTIVE:
         return "RENDERING_STATUS_ACTIVE";
      case RENDERING_STATUS_FREEZE:
         return "RENDERING_STATUS_FREEZE";
      case RENDERING_STATUS_UNCHANGED:
         return "RENDERING_STATUS_UNCHANGED";
      default:
         ::std::stringstream stream;
         stream << "DATA_STATUS_<" << static_cast<unsigned int>(dataRenderingStatus) << ">";
         return stream.str();
      }
   }

   MapViewStatusInfos() :
      m_dataLoadingStatus(DATA_LOADING_STATUS_ALL_LOADED),
      m_dataRenderingStatus(RENDERING_STATUS_UNCHANGED),
      m_mapViewId(MAP_VIEW_ID__PRIMARY)
   {
   }

   void setMapViewId(MapViewId mapViewId)
   {
      m_mapViewId = mapViewId;
   }

   MapViewId getMapViewId() const
   {
      return m_mapViewId;
   }

   void setLoadingStatus(DataLoadingStatus dataLoadingStatus)
   {
      m_dataLoadingStatus = dataLoadingStatus;
   }
   void setRenderingStatus(DataRenderingStatus dataRenderingStatus)
   {
      m_dataRenderingStatus = dataRenderingStatus;
   }

   DataLoadingStatus getLoadingStatus() const
   {
      return m_dataLoadingStatus;
   }
   DataRenderingStatus getRenderingStatus() const
   {
      return m_dataRenderingStatus;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("MapViewStatusInfos payload:\n");
      stream << "Data loading status " << toString(m_dataLoadingStatus) << ::std::endl
             << "Data rendering status " << toString(m_dataRenderingStatus) << ::std::endl
             << "Map View Id = " << ::navmiddleware::toString(m_mapViewId) << ::std::endl;
      return stream.str();
   }

private:
   DataLoadingStatus m_dataLoadingStatus;
   DataRenderingStatus m_dataRenderingStatus;
   MapViewId m_mapViewId;
};

struct MapScaleInfo
{
   MapScaleInfo()
      : m_mapScaleInMeters(0),
        m_mapScaleDistanceUnit(DISTANCEUNIT_METERS)
   {}

   MapScaleInfo(int32_t mapScaleInMeters,
                ::std::string mapScaleNoUnit,
                DistanceUnit mapScaleDistanceUnit,
                ::std::string formattedScaleBarMapScale)
      : m_mapScaleInMeters(mapScaleInMeters),
        m_mapScaleNoUnit(mapScaleNoUnit),
        m_mapScaleDistanceUnit(mapScaleDistanceUnit),
        m_formattedMapScale(formattedScaleBarMapScale)
   {}

   ::std::string toString() const
   {
      ::std::stringstream stream("MapScaleInfo payload:\n");
      stream << "Raw map scale value in meters = " << m_mapScaleInMeters << ::std::endl
             << "Map scale with no unit = " << m_mapScaleNoUnit << ::std::endl
             << "Map scale distance unit = " << ::navmiddleware::toString(m_mapScaleDistanceUnit) << ::std::endl
             << "Formatted map scale = " << m_formattedMapScale << ::std::endl;

      return stream.str();
   }

   bool operator==(const MapScaleInfo& rhs) const
   {
      return ((m_mapScaleInMeters == rhs.m_mapScaleInMeters) &&
              (m_mapScaleNoUnit == rhs.m_mapScaleNoUnit) &&
              (m_mapScaleDistanceUnit == rhs.m_mapScaleDistanceUnit) &&
              (m_formattedMapScale == rhs.m_formattedMapScale));
   }

   bool operator!=(const MapScaleInfo& rhs) const
   {
      return !(*this == rhs);
   }

   int32_t m_mapScaleInMeters;                  // Raw value of the map scale in meters.
   ::std::string m_mapScaleNoUnit;              // The formatted map scale without unit.
   DistanceUnit m_mapScaleDistanceUnit;         // enum value representing the unit.
   ::std::string m_formattedMapScale;           // The formatted map scale as a single string.
};

class MapCameraInfos
{
public:
   MapCameraInfos() :
      m_cameraHeading(0.0f),
      m_mapScale(0),
      /** @DEPRECATED, use m_scaleBarWidthInPercentage instead. */
      m_scaleBarWidthInPixel(0),
      m_scaleBarWidthInPercentage(0),
      m_mapViewId(MAP_VIEW_ID__PRIMARY),
      m_cameraPitch(0) {}

   void setMapViewId(MapViewId mapViewId)
   {
      m_mapViewId = mapViewId;
   }

   MapViewId getMapViewId() const
   {
      return m_mapViewId;
   }

   void setCameraHeading(float cameraHeading)
   {
      m_cameraHeading = cameraHeading;
   }

   /** camera heading, 0 = north, 90 = east, 180 = south, 270 = west.*/
   float getCameraHeading() const
   {
      return m_cameraHeading;
   }

   int32_t getMapScale() const
   {
      return m_mapScale;
   }

   void setMapScale(int32_t mapScale)
   {
      m_mapScale = mapScale;
   }

   /** @DEPRECATED, use setMapScaleInfo() instead. */
   void setFormattedMapScale(const ::std::string& mapScale)
   {
      m_formattedMapScale = mapScale;
   }

   /** @DEPRECATED, use getMapScaleInfo() instead. */
   const ::std::string& getFormattedMapScale() const
   {
      return m_formattedMapScale;
   }

   /** @DEPRECATED, use setScaleBarWidthInPercentage() instead. */
   void setScaleBarWidthInPixel(uint16_t scaleBarWidth)
   {
      m_scaleBarWidthInPixel = scaleBarWidth;
   }

   /** @DEPRECATED, use getScaleBarWidthInPercentage() instead. */
   uint16_t getScaleBarWidthInPixel() const
   {
      return m_scaleBarWidthInPixel;
   }

   void setScaleBarWidthInPercentage(uint32_t scaleBarWidth)
   {
      m_scaleBarWidthInPercentage = scaleBarWidth;
   }

   uint32_t getScaleBarWidthInPercentage() const
   {
      return m_scaleBarWidthInPercentage;
   }

   const MapScaleInfo& getMapScaleInfo() const
   {
      return m_mapScaleInfo;
   }

   void setMapScaleInfo(const MapScaleInfo& mapScaleInfo)
   {
      m_mapScaleInfo = mapScaleInfo;
   }

   uint16_t getCameraPitch() const
   {
      return m_cameraPitch;
   }

   void setCameraPitch(uint16_t cameraPitch)
   {
      m_cameraPitch = cameraPitch;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("MapCameraInfos payload:\n");
      stream << "Camera heading = " << m_cameraHeading << ::std::endl
             << "Map scale      = " << m_formattedMapScale << ::std::endl
             << "Scalebar width in pixel = " << m_scaleBarWidthInPixel << ::std::endl
             << "Scalebar width in percentage = " << m_scaleBarWidthInPercentage << ::std::endl
             << "Map View Id = " << ::navmiddleware::toString(m_mapViewId) << ::std::endl
             << "Map scale info = " << m_mapScaleInfo.toString() << std::endl
             << "Camera pitch = " << m_cameraPitch << ::std::endl;

      return stream.str();
   }

private:
   float m_cameraHeading;
   int32_t m_mapScale;
   ::std::string m_formattedMapScale;
   /** @DEPRECATED, use m_scaleBarWidthInPercentage instead. */
   uint16_t m_scaleBarWidthInPixel;
   uint32_t m_scaleBarWidthInPercentage;
   MapViewId m_mapViewId;
   MapScaleInfo m_mapScaleInfo;
   //Defines the pitch (trigonometric direction) of a camera: 0° = horizontal view, 90° (65535) = 2D view (top down)
   uint16_t m_cameraPitch;
};

/** Used to specify the map pitch to be set by setMapRepresentation()*/
enum MapPitch
{
   MAP_PITCH_INVALID,
   MAP_PITCH_2D,  /**< set the map in 2D view*/
   MAP_PITCH_3D   /**< set the map in 3D view*/
};

inline ::std::string toString(MapPitch mapPitch)
{
   switch (mapPitch)
   {
   case MAP_PITCH_INVALID:
      return "MAP_PITCH_INVALID";
   case MAP_PITCH_2D:
      return "MAP_PITCH_2D";
   case MAP_PITCH_3D:
      return "MAP_PITCH_3D";
   default:
      ::std::stringstream stream;
      stream << "MAP_PITCH_<" << static_cast<unsigned int>(mapPitch) << ">";
      return stream.str();
   }
}

struct MapRepresentation
{
   MapRepresentation() :
      m_mapOrientation(::navmiddleware::settings::MAP_ORIENTATION_INVALID),
      m_mapPitch(MAP_PITCH_INVALID)
   {

   }

   MapRepresentation(::navmiddleware::settings::MapOrientation mapOrientation, MapPitch mapPitch) :
      m_mapOrientation(mapOrientation),
      m_mapPitch(mapPitch)
   {

   }

   void reset()
   {
      m_mapOrientation = ::navmiddleware::settings::MAP_ORIENTATION_INVALID;
      m_mapPitch = MAP_PITCH_INVALID;
   }

   bool operator==(const MapRepresentation& rhs) const
   {
      return ((m_mapOrientation == rhs.m_mapOrientation) &&
              (m_mapPitch == rhs.m_mapPitch));
   }

   bool operator!=(const MapRepresentation& rhs) const
   {
      return !(*this == rhs);
   }

   ::navmiddleware::settings::MapOrientation m_mapOrientation;
   MapPitch m_mapPitch;
};

class MapRepresentationInfo
{
public:
   MapRepresentationInfo()
      : m_mapViewId(MAP_VIEW_ID__PRIMARY)
   {
   }

   void setMapViewId(MapViewId mapViewId)
   {
      m_mapViewId = mapViewId;
   }

   MapViewId getMapViewId() const
   {
      return m_mapViewId;
   }

   void setMapRepresentation(const MapRepresentation& mapRepresentation)
   {
      m_mapRepresentation = mapRepresentation;
   }

   const MapRepresentation& getMapRepresentation() const
   {
      return m_mapRepresentation;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("MapRepresentationInfo payload:\n");
      stream << ::std::endl
             << "Map view id = " << ::navmiddleware::toString(m_mapViewId) << ::std::endl
             << "Map orientation type = "
             << ::navmiddleware::settings::toString(m_mapRepresentation.m_mapOrientation) << ::std::endl
             << "Map pitch type = "
             << ::navmiddleware::toString(m_mapRepresentation.m_mapPitch) << ::std::endl;
      return stream.str();
   }

private:
   MapViewId         m_mapViewId;
   MapRepresentation m_mapRepresentation;
};

enum MapCameraMode
{
   MAP_MODE_INVALID = -1,
   /** Used to put the camera in free mode (i.e. there is no link target), map can be scrolled/zoomed, etc. */
   MAP_MODE_FREE = 0,
   /** Used to link the camera to the carsor's position. */
   MAP_MODE_CARSOR,
   /** Used to position the camera to fit an overview of a route. **/
   MAP_MODE_ROUTE_OVERVIEW,
   /** Used to position the camera to show an overview of a traffic message. */
   MAP_MODE_TRAFFIC_OVERVIEW,
   /** @DEPRECATED: This camera mode is deprecated and should not be used. This will be deleted in an upcoming release.
    * Used to make the camera do a flight along the route. */
   MAP_MODE_ROUTE_FLIGHT,
   /** Used when we want to enable touch interactions to control the camera.
    *  This camera mode is currently not supported, touch gestures are sent directly to display driver. */
   MAP_MODE_TOUCH,
   /** @DEPRECATED: This camera mode is deprecated and should not be used. This will be deleted in an upcoming release.
     * Used when want to allow camera movement to be controlled by a continuous movement operation, like continuous scrolling or rotation */
   MAP_MODE_CONTINUOUS_MOVEMENT,
   /** Used when we want to go into the settings split screen view */
   MAP_MODE_SETTINGS_VIEW,
   /** Used to position the camera to place the provided geocoordinate in the center of the map in 2D North-Up. */
   MAP_MODE_LOCATION_OVERVIEW,
   /** Used to display the weather map screen. */
   MAP_MODE_WEATHER_OVERVIEW,
   /** Used to position the camera at a guidance manoeuvre. */
   MAP_MODE_INTERSECTION_MAP,
   /** Used to display the map widget.*/
   MAP_MODE_WIDGET,
   /** Used to display the compass navi view.*/
   MAP_MODE_COMPASS_NAVI,
   /** Used to setup the camera to display a traffic detour.*/
   MAP_MODE_TRAFFIC_DETOUR,
   /** Used to display a map screen for changing the secondary view map scale from the settings.*/
   MAP_MODE_SETTINGS_SECONDARY_MAP_SCALE,
   /** Used to position the camera to display static satellite images. */
   MAP_MODE_SATELLITE_VIEW,
   /** Used to position the camera to display street view. */
   MAP_MODE_STREET_VIEW,
   /** Used to position the camera to show an overview of the avoided areas.*/
   MAP_MODE_AVOIDAREA_OVERVIEW,
   /** Used to position the camera to show an overview of the reachable areas.*/
   MAP_MODE_REACHABLEAREA_OVERVIEW,
   /** Used to show the WHERE_AM_I screen.*/
   MAP_MODE_WHERE_AM_I,
   /** Used to position the camera to show the geofence.*/
   MAP_MODE_GEO_FENCE,
   /** Used to position the camera to show a selected segment from the route (user selects a turnlist element) */
   MAP_MODE_ROUTE_SEGMENT_SELECTION,
   /** Used when we want to go into the screen where we edit the route. */
   MAP_MODE_EDIT_ROUTE_OVERVIEW,
   /** To Store Active Map views on shutdown. */
   MAP_MODE_CAPTURE,
   /** [JP] specific, NOT TO BE USED IN ROW,
    * Since the Junction image is rendered by Neusoft MW, No feature call available for this camera mode */
   MAP_MODE_JUNCTIONVIEW_MAP
};

inline ::std::string toString(MapCameraMode mapCameraMode)
{
   switch (mapCameraMode)
   {
   case MAP_MODE_INVALID:
      return "MAP_MODE_INVALID";
   case MAP_MODE_FREE:
      return "MAP_MODE_FREE";
   case MAP_MODE_CARSOR:
      return "MAP_MODE_CARSOR";
   case MAP_MODE_ROUTE_OVERVIEW:
      return "MAP_MODE_ROUTE_OVERVIEW";
   case MAP_MODE_TRAFFIC_OVERVIEW:
      return "MAP_MODE_TRAFFIC_OVERVIEW";
   case MAP_MODE_ROUTE_FLIGHT:
      return "MAP_MODE_ROUTE_FLIGHT";
   case MAP_MODE_TOUCH:
      return "MAP_MODE_TOUCH";
   case MAP_MODE_CONTINUOUS_MOVEMENT:
      return "MAP_MODE_CONTINUOUS_MOVEMENT";
   case MAP_MODE_SETTINGS_VIEW:
      return "MAP_MODE_SETTINGS_VIEW";
   case MAP_MODE_LOCATION_OVERVIEW:
      return "MAP_MODE_LOCATION_OVERVIEW";
   case MAP_MODE_WEATHER_OVERVIEW:
      return "MAP_MODE_WEATHER_OVERVIEW";
   case MAP_MODE_INTERSECTION_MAP:
      return "MAP_MODE_INTERSECTION_MAP";
   case MAP_MODE_WIDGET:
      return "MAP_MODE_WIDGET";
   case MAP_MODE_COMPASS_NAVI:
      return "MAP_MODE_COMPASS_NAVI";
   case MAP_MODE_TRAFFIC_DETOUR:
      return "MAP_MODE_TRAFFIC_DETOUR";
   case MAP_MODE_SETTINGS_SECONDARY_MAP_SCALE:
      return "MAP_MODE_SETTINGS_SECONDARY_MAP_SCALE";
   case MAP_MODE_SATELLITE_VIEW:
      return "MAP_MODE_SATELLITE_VIEW";
   case MAP_MODE_STREET_VIEW:
      return "MAP_MODE_STREET_VIEW";
   case MAP_MODE_AVOIDAREA_OVERVIEW:
      return "MAP_MODE_AVOIDAREA_OVERVIEW";
   case MAP_MODE_REACHABLEAREA_OVERVIEW:
      return "MAP_MODE_REACHABLEAREA_OVERVIEW";
   case MAP_MODE_WHERE_AM_I:
      return "MAP_MODE_WHERE_AM_I";
   case MAP_MODE_ROUTE_SEGMENT_SELECTION:
      return "MAP_MODE_ROUTE_SEGMENT_SELECTION";
   case MAP_MODE_EDIT_ROUTE_OVERVIEW:
      return "MAP_MODE_EDIT_ROUTE_OVERVIEW";
   case MAP_MODE_CAPTURE:
      return "MAP_MODE_CAPTURE";
   case MAP_MODE_JUNCTIONVIEW_MAP:
      return "MAP_MODE_JUNCTIONVIEW_MAP";   
   default:
      ::std::stringstream stream;
      stream << "MAP_MODE_<" << static_cast<unsigned int>(mapCameraMode) << ">";
      return stream.str();
   }
}

/** specifies the multiview type*/
enum MapMultiViewMode
{
   MULTIVIEW_OFF = 0,   /**< multiview is off*/
   MULTIVIEW_SPLIT      /**< multiview is set to split screen*/
};

inline ::std::string toString(MapMultiViewMode mapMultiViewMode)
{
   switch (mapMultiViewMode)
   {
   case MULTIVIEW_OFF:
      return "MULTIVIEW_OFF";
   case MULTIVIEW_SPLIT:
      return "MULTIVIEW_SPLIT";
   default:
      ::std::stringstream stream;
      stream << "MULTIVIEW_<" << static_cast<unsigned int>(mapMultiViewMode) << ">";
      return stream.str();
   }
}

/** Used to specify the dimensions for views*/
struct ViewDimensions
{
   ViewDimensions() :
      m_left(0),
      m_top(0),
      m_width(0),
      m_height(0)
   {
   }

   ViewDimensions(uint16_t left, uint16_t top, uint16_t width, uint16_t height) :
      m_left(left),
      m_top(top),
      m_width(width),
      m_height(height)
   {
   }


   bool operator==(const ViewDimensions& rhs) const
   {
      return ((m_left == rhs.m_left) &&
              (m_top == rhs.m_top) &&
              (m_width == rhs.m_width) &&
              (m_height == rhs.m_height));
   }

   bool operator!=(const ViewDimensions& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "ViewDimensions: (" << m_left << ", " << m_top << ")x(" << m_width << ", " << m_height << ")" << ::std::endl;
      return stream.str();
   }

   uint16_t m_left;     /**< x-coordinate on the screen of the top left corner of the required view*/
   uint16_t m_top;      /**< y-coordinate on the screen of the top left corner of the required view*/
   uint16_t m_width;    /**< width in pixels of the required view*/
   uint16_t m_height;   /**< height in pixels of the required view*/
};

class MapCameraModeInfo
{
public:
   MapCameraModeInfo()
      :m_mapCameraMode(MAP_MODE_INVALID),
       m_mapViewId(MAP_VIEW_ID__PRIMARY),
       m_isPreparationPending(false)
   {
   }

   void setMapViewId(MapViewId mapViewId)
   {
      m_mapViewId = mapViewId;
   }

   MapViewId getMapViewId() const
   {
      return m_mapViewId;
   }

   void setMapCameraModeInfo(MapCameraMode mapCameraMode)
   {
      m_mapCameraMode = mapCameraMode;
   }

   MapCameraMode getMapCameraModeInfo() const
   {
      return m_mapCameraMode;
   }

   void setIsPreparationPending(bool isPreparationPending)
   {
      m_isPreparationPending = isPreparationPending;
   }

   bool getIsPreparationPending() const
   {
      return m_isPreparationPending;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("MapCameraModeInfo payload:\n");
      stream << ::std::endl
             << "Camera mode type = " << ::navmiddleware::toString(m_mapCameraMode) << ::std::endl
             << "Map View Id = " << ::navmiddleware::toString(m_mapViewId) << ::std::endl
             << "Is Preparation Pending" << m_isPreparationPending << ::std::endl;
      return stream.str();
   }

private:
   MapCameraMode m_mapCameraMode;
   MapViewId m_mapViewId;
   bool m_isPreparationPending;
};

enum TouchGestureType
{
   /** Default value that does not stand for a particular gesture */
   TOUCH_GESTURE_INVALID = 0,

   /** The screen was tapped */
   TOUCH_GESTURE_TAPPED,

   /** The finger moves over the screen along a sufficiently long and straight line */
   TOUCH_GESTURE_SWIPE,

   /** Two fingers move over the screen along two opposite circular segments */
   TOUCH_GESTURE_ROTATE,

   /** Two Fingers move over the screen on a sufficiently a straight line in opposite directions
    * i.e either outwards or inwards */
   TOUCH_GESTURE_PINCH,

   /**  This gesture is sent when any one of the fingers part of touch interaction (single or multi-touch)
    *   moves a small amount. This gesture update will also mean that the TAP gesture is no longer possible,
    *   during the ongoing touch interaction. For CPU performance, Map will only send this gesture update
    *   if the MapCameraMode is not CAMERA_MODE_FREE or CAMERA_MODE_TOUCH, when the touch inter-action starts.
    *   This is a "live" gesture which means the gesture update will be sent as soon as the gesture is
    *   detected during the touch interaction, and not necessarily after the finger has been released.
    */
   TOUCH_GESTURE_MOVE
};

inline ::std::string toString(TouchGestureType touchGestureType)
{
   switch (touchGestureType)
   {
   case TOUCH_GESTURE_INVALID:
      return "TOUCH_GESTURE_INVALID";
   case TOUCH_GESTURE_TAPPED:
      return "TOUCH_GESTURE_TAPPED";
   case TOUCH_GESTURE_SWIPE:
      return "TOUCH_GESTURE_SWIPE";
   case TOUCH_GESTURE_ROTATE:
      return "TOUCH_GESTURE_ROTATE";
   case TOUCH_GESTURE_PINCH:
      return "TOUCH_GESTURE_PINCH";
   case TOUCH_GESTURE_MOVE:
      return "TOUCH_GESTURE_MOVE";
   default:
      ::std::stringstream stream;
      stream << "TOUCH_GESTURE_<" << static_cast<unsigned int>(touchGestureType) << ">";
      return stream.str();
   }
}

/** The type of tap gesture. */
enum TouchGestureTapType
{
   /** Default value that does not stand for a particular gesture */
   TOUCH_GESTURE_TAP_NONE = 0,

   /** The screen was tapped */
   TOUCH_GESTURE_SINGLE_TAP,

   /** Two short taps in rapid succession at nearly the same spot on the screen. */
   TOUCH_GESTURE_DOUBLE_TAP,

   /** The screen was touched and the finger was held at the same position */
   TOUCH_GESTURE_LONG_TAP
};

inline ::std::string toString(TouchGestureTapType touchGestureTapType)
{
   switch (touchGestureTapType)
   {
   case TOUCH_GESTURE_TAP_NONE:
      return "TOUCH_GESTURE_TAP_NONE";
   case TOUCH_GESTURE_SINGLE_TAP:
      return "TOUCH_GESTURE_SINGLE_TAP";
   case TOUCH_GESTURE_DOUBLE_TAP:
      return "TOUCH_GESTURE_DOUBLE_TAP";
   case TOUCH_GESTURE_LONG_TAP:
      return "TOUCH_GESTURE_LONG_TAP";
   default:
      ::std::stringstream stream;
      stream << "TOUCH_GESTURE_TAP_<" << static_cast<unsigned int>(touchGestureTapType) << ">";
      return stream.str();
   }
}

enum TouchGestureSwipeType
{
   TOUCH_GESTURE_SWIPE_TYPE__SINGLE_FINGER,
   TOUCH_GESTURE_SWIPE_TYPE__TWO_FINGER
};

inline ::std::string toString(TouchGestureSwipeType touchGestureSwipeType)
{
   switch (touchGestureSwipeType)
   {
   case TOUCH_GESTURE_SWIPE_TYPE__SINGLE_FINGER:
      return "TOUCH_GESTURE_SWIPE_TYPE__SINGLE_FINGER";
   case TOUCH_GESTURE_SWIPE_TYPE__TWO_FINGER:
      return "TOUCH_GESTURE_SWIPE_TYPE__TWO_FINGER";
   default:
      ::std::stringstream stream;
      stream << "TOUCH_GESTURE_SWIPE_TYPE_<" << static_cast<unsigned int>(touchGestureSwipeType) << ">";
      return stream.str();
   }
}

/** Position in pixel on the screen */
struct TouchEventScreenCoordinate
{
   uint32_t xPos;
   uint32_t yPos;
};

class TouchEventInfo
{
public:
   TouchEventInfo()
      : m_touchGestureType(TOUCH_GESTURE_INVALID),
        m_mapViewId(MAP_VIEW_ID__PRIMARY)
   {}

   void setMapViewId(MapViewId mapViewId)
   {
      m_mapViewId = mapViewId;
   }

   MapViewId getMapViewId() const
   {
      return m_mapViewId;
   }

   struct TouchGestureTap
   {
      TouchGestureTapType                 m_touchGestureTapType;
      TouchEventScreenCoordinate          m_touchEventScreenCoordinate;
      /**Tapped direction with respect to the y-axis such that 0° to 360°
       * Eg:  0° is up,
       * 90° is left,
       * 180° is down,
       * 270° is right.
       */
      uint16_t m_tappedDirection;
   };

   struct TouchGestureSwipe
   {
      /** The direction of the swipe with respect to the y-axis such that 0° to 360°
       * 0° is up,
       * 90° is left,
       * 180° is down,
       * 270° is right.
       */
      uint16_t m_swipeDirection;
      /*
       * The swipe speed in pixels per second
       */
      float m_speed;
      /*
       * The type of swipe gesture
       */
      TouchGestureSwipeType m_touchGestureSwipeType;
   };

   struct TouchGestureRotate
   {
      /** Positive value indicates clockwise rotation direction and negative value indicates
       * anti-clockwise rotation direction.
       * the value range should be [-360,360]
       */
      int16_t m_rotationAngle;
   };

   struct TouchGesturePinch
   {
      /** The factor by which the gap between the fingers changed due to Pinch Out or Pinch In
       * i.e ratio of pixel gap between fingers when touch interaction ended to pixel gap
       * between fingers when touch interaction started.
       * For Pinch Out, the factor is less than 1.f. For Pinch In, the factor is greater than 1.f
       */
      float m_pinchFactor;
   };

   union TouchGestureTypeDetail
   {
      TouchGestureTap           m_touchGestureTap;
      TouchGestureSwipe         m_touchGestureSwipe;
      TouchGestureRotate        m_touchGestureRotate;
      TouchGesturePinch         m_touchGesturePinch;
   };

   ::std::string toString() const
   {
      ::std::stringstream stream("TouchEventInfo payload:\n");

      if(m_touchGestureType == ::navmiddleware::TOUCH_GESTURE_TAPPED )
      {
         stream << "Touch Gesture Tap info : " << ::std::endl
                << "TouchGestureTapType = " << ::navmiddleware::toString(m_touchGestureTypeDetail.m_touchGestureTap.m_touchGestureTapType)<< ::std::endl
                << "TouchEventScreenCoordinate = [ " << m_touchGestureTypeDetail.m_touchGestureTap.m_touchEventScreenCoordinate.xPos << ", "
                << m_touchGestureTypeDetail.m_touchGestureTap.m_touchEventScreenCoordinate.yPos << " ]"
                << "Direction for LongTap = "<< m_touchGestureTypeDetail.m_touchGestureTap.m_tappedDirection << ::std::endl;
      }
      else if(m_touchGestureType == ::navmiddleware::TOUCH_GESTURE_SWIPE )
      {
         stream << "Touch Gesture Swipe info : " << ::std::endl
                << "SwipeDirection = "<< m_touchGestureTypeDetail.m_touchGestureSwipe.m_swipeDirection << ::std::endl
                << "Swipe Speed = "<< m_touchGestureTypeDetail.m_touchGestureSwipe.m_speed << ::std::endl
                << "Swipe Type = " << ::navmiddleware::toString(m_touchGestureTypeDetail.m_touchGestureSwipe.m_touchGestureSwipeType) << ::std::endl;
      }
      else if(m_touchGestureType == ::navmiddleware::TOUCH_GESTURE_ROTATE )
      {
         stream << " Touch Gesture Rotate Info: "  << ::std::endl
                << " Rotation Angle = " << m_touchGestureTypeDetail.m_touchGestureRotate.m_rotationAngle << ::std::endl;
      }
      else if(m_touchGestureType == ::navmiddleware::TOUCH_GESTURE_PINCH)
      {
         stream << " Touch Gesture Pinch Info: " << ::std::endl
                << " Pinch Factor = "<< m_touchGestureTypeDetail.m_touchGesturePinch.m_pinchFactor << ::std::endl;
      }
      else if(m_touchGestureType == ::navmiddleware::TOUCH_GESTURE_MOVE)
      {
         stream << " Touch Gesture Move Info: " << ::std::endl;
      }
      else
      {
         stream << "INVALID TOUCH GESTURE TYPE";
      }

      stream << "Map View Id = " << ::navmiddleware::toString(m_mapViewId) << ::std::endl;

      return stream.str();
   }

   TouchGestureType                  m_touchGestureType;
   TouchGestureTypeDetail            m_touchGestureTypeDetail;
   MapViewId                         m_mapViewId;
};

class NominalScaleInfos
{
public:
   NominalScaleInfos()
      : m_mapViewId(MAP_VIEW_ID__PRIMARY) {}

   class NominalScaleInfo
   {
   public:
      NominalScaleInfo()
      {
         m_mapScale = 0;
         m_autoPitch = 0;
      }

      void setMapScale(int32_t mapScale)
      {
         m_mapScale = mapScale;
      }

      int32_t getMapScale() const
      {
         return m_mapScale;
      }

      void setAutoPitch(uint32_t autoPitch)
      {
         m_autoPitch = autoPitch;
      }

      uint32_t getAutoPitch() const
      {
         return m_autoPitch;
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("NominalScaleInfo payload:\n");
         stream << ::std::endl
                << "Nominal map scale = " << m_mapScale << ::std::endl
                << "Nominal Auto Pitch = " << m_autoPitch << ::std::endl;
         return stream.str();
      }

      int32_t     m_mapScale;
      uint32_t    m_autoPitch;
   };

   void setMapViewId(MapViewId mapViewId)
   {
      m_mapViewId = mapViewId;
   }

   MapViewId getMapViewId() const
   {
      return m_mapViewId;
   }

   const std::vector<NominalScaleInfo>& getNominalScaleInfos() const
   {
      return m_nominalScaleInfo;
   }

   void setNominalScaleInfos(const std::vector<NominalScaleInfo>& nominalScaleInfos)
   {
      m_nominalScaleInfo = nominalScaleInfos;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("NominalScaleInfos payload:\n");
      stream << ::std::endl;

      for(unsigned int nominalScaleListIndex = 0; nominalScaleListIndex < getNominalScaleInfos().size(); nominalScaleListIndex++)
      {
         stream << "NominalScaleInfos :" << getNominalScaleInfos().at(nominalScaleListIndex).toString() << ::std::endl;
      }

      stream << "Map View Id = " << ::navmiddleware::toString(m_mapViewId) << ::std::endl;

      return stream.str();
   }

private:
   ::std::vector<NominalScaleInfo> m_nominalScaleInfo;
   MapViewId                       m_mapViewId;
};

class PitchRange
{
public:
   PitchRange()
   {
      m_minPitch = 0;
      m_maxPitch = 0;
   }
   void setMinPitch(uint16_t minPitch)
   {
      m_minPitch = minPitch;
   }
   void setMaxPitch(uint16_t maxPitch)
   {
      m_maxPitch = maxPitch;
   }
   uint16_t getMinPitch() const
   {
      return m_minPitch;
   }
   uint16_t getMaxPitch() const
   {
      return m_maxPitch;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("PitchRange payload:\n");
      stream << ::std::endl
             << "Minumum Pitch = " << m_minPitch << ::std::endl
             << "Maximum Pitch = " << m_maxPitch << ::std::endl;
      return stream.str();
   }

private:
   uint16_t m_minPitch;
   uint16_t m_maxPitch;
};

class MapScaleRange
{
public:
   MapScaleRange()
   {
      m_minMapScale = 0;
      m_maxMapScale = 0;
      m_mapViewId = MAP_VIEW_ID__PRIMARY;
   }
   MapScaleRange(int32_t minScale, int32_t maxScale, MapViewId mapViewId)
      : m_minMapScale(minScale)
      , m_maxMapScale(maxScale)
      , m_mapViewId(mapViewId)
   {}
   bool operator==(const MapScaleRange& rhs ) const
   {
      return (m_minMapScale == rhs.m_minMapScale)
             && (m_maxMapScale == rhs.m_maxMapScale);
   }

   bool operator!=(const MapScaleRange& rhs ) const
   {
      return !(*this == rhs);
   }

   void setMapViewId(MapViewId mapViewId)
   {
      m_mapViewId = mapViewId;
   }

   MapViewId getMapViewId() const
   {
      return m_mapViewId;
   }

   void setMinMapScale(int32_t minMapScale)
   {
      m_minMapScale = minMapScale;
   }
   void setMaxMapScale(int32_t maxMapScale)
   {
      m_maxMapScale = maxMapScale;
   }
   int32_t getMinMapScale() const
   {
      return m_minMapScale;
   }
   int32_t getMaxMapScale() const
   {
      return m_maxMapScale;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("MapScaleRange payload:\n");
      stream << ::std::endl
             << "Minumum map scale = " << m_minMapScale << ::std::endl
             << "Maximum map scale = " << m_maxMapScale << ::std::endl
             << "Map View Id = " << ::navmiddleware::toString(m_mapViewId) << ::std::endl;
      return stream.str();
   }

private:
   int32_t m_minMapScale;
   int32_t m_maxMapScale;
   MapViewId m_mapViewId;
};

class TouchActions
{
public:
   TouchActions()
      : m_touchActionMove(false)
      , m_touchActionScrollOnHold(false)
      , m_touchActionZoom(false)
      , m_touchActionRotate(false)
      , m_touchActionPitch(false)
      , m_touchActionGestureTap(false)
      , m_touchActionGestureSwipe(false)
      , m_touchActionGestureRotate(false)
      , m_touchActionGesturePinch(false)
      , m_touchActionGestureMove(false)
   { }

   void setTouchActionMove(bool move)
   {
      m_touchActionMove = move;
   }

   bool isTouchActionMoveEnabled() const
   {
      return m_touchActionMove;
   }

   void setTouchActionScrollOnHold(bool scrollOnHold)
   {
      m_touchActionScrollOnHold = scrollOnHold;
   }

   bool isTouchActionScrollOnHoldEnabled() const
   {
      return m_touchActionScrollOnHold;
   }

   void setTouchActionZoom(bool zoom)
   {
      m_touchActionZoom = zoom;
   }

   bool isTouchActionZoomEnabled() const
   {
      return m_touchActionZoom;
   }

   void setTouchActionRotate(bool rotate)
   {
      m_touchActionRotate = rotate;
   }

   bool isTouchActionRotateEnabled() const
   {
      return m_touchActionRotate;
   }

   void setTouchActionPitch(bool pitch)
   {
      m_touchActionPitch = pitch;
   }

   bool isTouchActionPitchEnabled() const
   {
      return m_touchActionPitch;
   }

   void setTouchActionGestureTap(bool touchActionGestureTap)
   {
      m_touchActionGestureTap = touchActionGestureTap;
   }

   bool isTouchActionGestureTapEnabled() const
   {
      return m_touchActionGestureTap;
   }

   void setTouchActionGestureSwipe(bool touchActionGestureSwipe)
   {
      m_touchActionGestureSwipe = touchActionGestureSwipe;
   }

   bool isTouchActionGestureSwipeEnabled() const
   {
      return m_touchActionGestureSwipe;
   }

   void setTouchActionGestureRotate(bool touchActionGestureRotate)
   {
      m_touchActionGestureRotate = touchActionGestureRotate;
   }

   bool isTouchActionGestureRotateEnabled() const
   {
      return m_touchActionGestureRotate;
   }

   void setTouchActionGesturePinch(bool touchActionGesturePinch)
   {
      m_touchActionGesturePinch = touchActionGesturePinch;
   }

   bool isTouchActionGesturePinchEnabled() const
   {
      return m_touchActionGesturePinch;
   }

   void setTouchActionGestureMove(bool touchActionGestureMove)
   {
      m_touchActionGestureMove = touchActionGestureMove;
   }

   bool isTouchActionGestureMoveEnabled() const
   {
      return m_touchActionGestureMove;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("TouchActions payload:\n");
      stream << ::std::endl
             << "Camera Move operation enabled = "           << m_touchActionMove          << ::std::endl
             << "Camera Scroll on Hold operation enabled = " << m_touchActionScrollOnHold  << ::std::endl
             << "Camera Zoom operation enabled = "           << m_touchActionZoom          << ::std::endl
             << "Camera Rotate operation enabled = "         << m_touchActionRotate        << ::std::endl
             << "Camera Pitch operation enabled = "          << m_touchActionPitch         << ::std::endl
             << "Touch Gesture Tap operation enabled = "     << m_touchActionGestureTap    << ::std::endl
             << "Touch Gesture Swipe operation enabled = "   << m_touchActionGestureSwipe  << ::std::endl
             << "Touch Gesture Rotate operation enabled = "  << m_touchActionGestureRotate << ::std::endl
             << "Touch Gesture Pinch operation enabled = "   << m_touchActionGesturePinch  << ::std::endl
             << "Touch Gesture Move operation enabled = "    << m_touchActionGestureMove   << ::std::endl;
      return stream.str();
   }

private:
   /** Camera operations*/
   /** The camera will follow the movement of the (index-) finger */
   bool m_touchActionMove;

   /** The camera will be allowed to perform the ScrollOnHold operation i.e the map moves continuously from
       finger to the center of the map, on detection of a hold gesture */
   bool m_touchActionScrollOnHold;

   /** The map scale will be changed (pinch) */
   bool m_touchActionZoom;

   /** The camera will be rotated */
   bool m_touchActionRotate;

   /** The camera pitch will be changed */
   bool m_touchActionPitch;

   /** Touch gestures*/
   /** Tap gesture (single tap or double tap) will be detected. */
   bool m_touchActionGestureTap;

   /** Touch gesture swipe will be detected. */
   bool m_touchActionGestureSwipe;

   /** Touch gesture rotate will be detected. */
   bool m_touchActionGestureRotate;

   /** Touch gesture pinch will be detected. */
   bool m_touchActionGesturePinch;

   /** Touch gesture move will be detected. */
   bool m_touchActionGestureMove;
};

struct MapIconInfo
{
   MapIconInfo()
   {
      m_showNumberedIcons = false;
      m_startOffset = 0;
   }

   MapIconInfo(const bool showNumberedIcons, const uint32_t startOffset, const ValidValue<uint32_t>& selectedIndex)
      :m_showNumberedIcons(showNumberedIcons)
      ,m_startOffset(startOffset)
      ,m_selectedIndex(selectedIndex)
   {}

   bool operator==(const MapIconInfo& rhs) const
   {
      return (m_showNumberedIcons == rhs.m_showNumberedIcons)
             && (m_startOffset == rhs.m_startOffset)
             && (m_selectedIndex == rhs.m_selectedIndex);
   }

   bool operator!=(const MapIconInfo& rhs) const
   {
      return (!(*this == rhs));
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("MapIconInfo payload:");
      stream << ::std::endl
             << "show numbered icons = " << m_showNumberedIcons << ::std::endl
             << "start offset = " << m_startOffset << ::std::endl
             << "selected index = " << m_selectedIndex.toString() << std::endl;
      return stream.str();
   }

   bool                  m_showNumberedIcons;
   /** m_startOffset value should start from 1 or higher. If the value is beyond the
      numbered icons available for display, it will be restricted internally e.g 200 */
   uint32_t              m_startOffset;
   /** The index user wants to highlight, should be higher than or equal to startOffset value and within the end range.
      if nothing to be highlighted, don't set any value */
   ValidValue<uint32_t>  m_selectedIndex;

};

struct MapMultiViewInfo
{
   MapMultiViewInfo()
   {
      m_multiViewMode = MULTIVIEW_OFF;
   }

   bool operator==(const MapMultiViewInfo& rhs) const
   {
      return (m_multiViewMode == rhs.m_multiViewMode);
   }

   bool operator!=(const MapMultiViewInfo& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("MapMultiViewInfo payload:\n");
      stream << ::std::endl
             << "Map multiview mode = " << ::navmiddleware::toString(m_multiViewMode) << ::std::endl;
      return stream.str();
   }

   MapMultiViewMode m_multiViewMode;
};

enum CameraAnimation
{
   CameraAnimation__ADAPTIVE,    /**< Best animation will be selected by map engine.*/
   CameraAnimation__IMMEDIATE,   /**< Camera settings will take effect immediately.*/
   CameraAnimation__FAST,        /**< Camera settings will take effect after a relatively short animation.*/
   CameraAnimation__MEDIUM,      /**< Camera settings will take effect after an animation.*/
   CameraAnimation__SLOW         /**< Camera settings will take effect after a relatively long animation.*/
};

inline ::std::string toString(CameraAnimation cameraAnimation)
{
   switch (cameraAnimation)
   {
   case CameraAnimation__ADAPTIVE:
      return "CameraAnimation__ADAPTIVE";
   case CameraAnimation__IMMEDIATE:
      return "CameraAnimation__IMMEDIATE";
   case CameraAnimation__FAST:
      return "CameraAnimation__FAST";
   case CameraAnimation__MEDIUM:
      return "CameraAnimation__MEDIUM";
   case CameraAnimation__SLOW:
      return "CameraAnimation__SLOW";
   default:
      ::std::stringstream stream;
      stream << "CameraAnimation__<" << static_cast<unsigned int>(cameraAnimation) << ">";
      return stream.str();
   }
}

/** Used to identify the screen to use in case of multiscreen*/
enum ScreenId
{
   SCREEN_ID__PRIMARY = 0,    /**< use the primary screen*/
   SCREEN_ID__SECONDARY,      /**< use the secondary screen*/
   SCREEN_ID__CLUSTER         /**< use the screen cluster*/
};

inline ::std::string toString(ScreenId screenId)
{
   switch (screenId)
   {
   case SCREEN_ID__PRIMARY:
      return "SCREEN_ID__PRIMARY";
   case SCREEN_ID__SECONDARY:
      return "SCREEN_ID__SECONDARY";
   case SCREEN_ID__CLUSTER:
      return "SCREEN_ID__CLUSTER";
   default:
      ::std::stringstream stream;
      stream << "SCREEN_ID__<" << static_cast<unsigned int>(screenId) << ">";
      return stream.str();
   }
}

class ViewModeOptions
{
public:
   enum ReachabilityDisplayMode
   {
      REACHABILITY_DISPLAY_MODE__AREA_AROUND_CARSOR,
      REACHABILITY_DISPLAY_MODE__AREA_AROUND_DESTINATION,
      REACHABILITY_DISPLAY_MODE__ON_ROUTE
   };

   ::std::string toString(ValidValue<ReachabilityDisplayMode> reachabilityDisplayMode) const
   {
      if(reachabilityDisplayMode.isValid())
      {
         switch (reachabilityDisplayMode.getValue())
         {
         case REACHABILITY_DISPLAY_MODE__AREA_AROUND_CARSOR:
            return "REACHABILITY_DISPLAY_MODE__AREA_AROUND_CARSOR";
         case REACHABILITY_DISPLAY_MODE__AREA_AROUND_DESTINATION:
            return "REACHABILITY_DISPLAY_MODE__AREA_AROUND_DESTINATION";
         case REACHABILITY_DISPLAY_MODE__ON_ROUTE:
            return "REACHABILITY_DISPLAY_MODE__ON_ROUTE";
         default:
            ::std::stringstream stream;
            stream << "REACHABILITY_DISPLAY_MODE__<" << static_cast<unsigned int>(reachabilityDisplayMode.getValue()) << ">";
            return stream.str();
         }
      }
      else
      {
         ::std::stringstream stream;
         stream << "reachabilityDisplayMode.m_isValid: " << reachabilityDisplayMode.isValid();
         return stream.str();
      }
   }

   enum UpperTrafficFlowDisplayLimit
   {
      /** No traffic flow will be displayed in map*/
      UPPER_TRAFFIC_FLOW_DISPLAY_LIMIT__DISABLED,
      /** Only no flow situations will be displayed in map*/
      UPPER_TRAFFIC_FLOW_DISPLAY_LIMIT__NO_FLOW,
      /** High impact and no flow situations will be displayed in map*/
      UPPER_TRAFFIC_FLOW_DISPLAY_LIMIT__HIGH_IMPACT,
      /** Low impact, high impact and no flow situations will be displayed in map*/
      UPPER_TRAFFIC_FLOW_DISPLAY_LIMIT__LOW_IMPACT,
      /** All traffic flow situations will be displayed in map*/
      UPPER_TRAFFIC_FLOW_DISPLAY_LIMIT__FREE_FLOW
   };

   ::std::string toString(UpperTrafficFlowDisplayLimit trafficFlowDisplayLimit) const
   {
      switch (trafficFlowDisplayLimit)
      {
      case UPPER_TRAFFIC_FLOW_DISPLAY_LIMIT__DISABLED:
         return "UPPER_TRAFFIC_FLOW_DISPLAY_LIMIT__DISABLED";
      case UPPER_TRAFFIC_FLOW_DISPLAY_LIMIT__NO_FLOW:
         return "UPPER_TRAFFIC_FLOW_DISPLAY_LIMIT__NO_FLOW";
      case UPPER_TRAFFIC_FLOW_DISPLAY_LIMIT__HIGH_IMPACT:
         return "UPPER_TRAFFIC_FLOW_DISPLAY_LIMIT__HIGH_IMPACT";
      case UPPER_TRAFFIC_FLOW_DISPLAY_LIMIT__LOW_IMPACT:
         return "UPPER_TRAFFIC_FLOW_DISPLAY_LIMIT__LOW_IMPACT";
      case UPPER_TRAFFIC_FLOW_DISPLAY_LIMIT__FREE_FLOW:
         return "UPPER_TRAFFIC_FLOW_DISPLAY_LIMIT__FREE_FLOW";
      default:
         ::std::stringstream stream;
         stream << "UPPER_TRAFFIC_FLOW_DISPLAY_LIMIT__<" << static_cast<unsigned int>(trafficFlowDisplayLimit) << ">";
         return stream.str();
      }
   }

   class MapVideoStreamInfo
   {
   public:
      MapVideoStreamInfo()
          : m_ipAddress("")
          , m_portNumber(0)
          , m_videoType(0)
      {
      }

      const ::std::string& getIpAddress() const
      {
         return m_ipAddress;
      }

      void setIpAddress(const ::std::string& value)
      {
         m_ipAddress = value;
      }

      uint16_t getPortNumber() const
      {
         return m_portNumber;
      }

      void setPortNumber(uint16_t value)
      {
         m_portNumber = value;
      }

      uint16_t getVideoType() const
      {
         return m_videoType;
      }

      void setVideoType(uint16_t value)
      {
         m_videoType = value;
      }

      bool operator==(const MapVideoStreamInfo& rhs) const
      {
         return ((m_ipAddress == rhs.m_ipAddress) &&
                 (m_portNumber == rhs.m_portNumber) &&
                 (m_videoType == rhs.m_videoType));
      }

      bool operator!=(const MapVideoStreamInfo& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream;
         stream << "MapVideoStreamInfo: IpAddress =" << m_ipAddress
                << ", PortNumber =" << m_portNumber
                << ", VideoType =" << m_videoType
                << ::std::endl;
         return stream.str();
      }

   private:
      ::std::string                         m_ipAddress;
      uint16_t                              m_portNumber;
      /** Video type defines the Map features to be enabled For eg: Roads, RoadNames, Landmarks etc */
      uint16_t                              m_videoType;
   };

   ViewModeOptions() {}
   ~ViewModeOptions() {}

   const ValidValue<bool>& areWeatherIconsVisible() const
   {
      return m_areWeatherIconsVisible;
   }

   void setWeatherIconsState(const ValidValue<bool>& value)
   {
      m_areWeatherIconsVisible = value;
   }

   const ValidValue<bool>& areSpeedCamsVisible() const
   {
      return m_areSpeedCamsVisible;
   }

   void setSpeedCamsState(const ValidValue<bool>& value)
   {
      m_areSpeedCamsVisible = value;
   }

   const ValidValue<bool>& areTrafficIconsVisible() const
   {
      return m_areTrafficIconsVisible;
   }

   void setEPoiDisplayVisibilityState(const ValidValue<bool>& value)
   {
      m_isEPoiDisplayForced = value;
   }

   const ValidValue<bool>& isEPoiDisplayForced() const
   {
      return m_isEPoiDisplayForced;
   }

   void setTrafficIconsState(const ValidValue<bool>& value)
   {
      m_areTrafficIconsVisible = value;
   }

   const ValidValue<bool>& arePoiIconsVisible() const
   {
      return m_arePoiIconsVisible;
   }

   void setPoiIconsState(const ValidValue<bool>& value)
   {
      m_arePoiIconsVisible = value;
   }

   const ValidValue<bool>& areAddressbookIconsVisible() const
   {
      return m_areAddressbookIconsVisible;
   }

   void setAddressbookIconsState(const ValidValue<bool>& value)
   {
      m_areAddressbookIconsVisible = value;
   }

   const ValidValue<UpperTrafficFlowDisplayLimit>& getUpperTrafficFlowDisplayLimit() const
   {
      return m_trafficFlowDisplayLimit;
   }

   void setUpperTrafficFlowDisplayLimit(const ValidValue<UpperTrafficFlowDisplayLimit>& upperTrafficFlowDisplayLimit)
   {
      m_trafficFlowDisplayLimit = upperTrafficFlowDisplayLimit;
   }

   const ValidValue<bool>& getInceptionMapState() const
   {
      return m_inceptionMapState;
   }

   void setInceptionMapState(const ValidValue<bool>& value)
   {
      m_inceptionMapState = value;
   }

   const ValidValue<MapVideoStreamInfo>& getMapVideoStreamInfo() const
   {
      return m_mapVideoStreamInfo;
   }

   void setMapVideoStreamInfo(const ValidValue<MapVideoStreamInfo>& value)
   {
      m_mapVideoStreamInfo = value;
   }

   const ValidValue<ReachabilityDisplayMode>& getReachabilityDisplayMode() const
   {
      return m_reachabilityDisplayMode;
   }

   void setReachabilityDisplayMode(const ValidValue<ReachabilityDisplayMode>& value)
   {
      m_reachabilityDisplayMode = value;
   }

   //<-- INF4CV
   void setGuidanceArrowVisibilityState(const ValidValue<bool>& value)
   {
      m_isGuidanceArrowVisible = value;
   }

   const ValidValue<bool>& isGuidanceArrowVisible() const
   {
      return m_isGuidanceArrowVisible;
   }
   //->

   bool operator==(const ViewModeOptions& rhs) const
   {
      return ((m_areWeatherIconsVisible == rhs.m_areWeatherIconsVisible) &&
              (m_areSpeedCamsVisible == rhs.m_areSpeedCamsVisible) &&
              (m_areTrafficIconsVisible == rhs.m_areTrafficIconsVisible) &&
              (m_arePoiIconsVisible == rhs.m_arePoiIconsVisible) &&
              (m_areAddressbookIconsVisible == rhs.m_areAddressbookIconsVisible) &&
              (m_trafficFlowDisplayLimit == rhs.m_trafficFlowDisplayLimit) &&
              (m_inceptionMapState == rhs.m_inceptionMapState) &&
              (m_mapVideoStreamInfo == rhs.m_mapVideoStreamInfo) &&
              (m_reachabilityDisplayMode == rhs.m_reachabilityDisplayMode) &&
              (m_isEPoiDisplayForced == rhs.m_isEPoiDisplayForced) &&
              (m_isGuidanceArrowVisible == rhs.m_isGuidanceArrowVisible));   //<-- INF4CV
   }

   bool operator!=(const ViewModeOptions& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "ViewModeOptions: WeatherIconsVisibility=" << m_areWeatherIconsVisible.toString() <<::std::endl
             << ", SpeedCamsVisibility=" << m_areSpeedCamsVisible.toString() <<::std::endl
             << ", TrafficIconsVisibility=" << m_areTrafficIconsVisible.toString() <<::std::endl
             << ", PoiIconsVisibility=" << m_arePoiIconsVisible.toString() <<::std::endl
             << ", AddressbookIconsVisibility=" << m_areAddressbookIconsVisible.toString() <<::std::endl
             << ", TrafficFlowDisplayOptions=" << toString(m_trafficFlowDisplayLimit.getValueOr(UPPER_TRAFFIC_FLOW_DISPLAY_LIMIT__DISABLED)) <<::std::endl
             << ", InceptionMapState=" << m_inceptionMapState.toString() <<::std::endl
             << ", MapVideoStreamInfo validity=" << m_mapVideoStreamInfo.isValid() <<::std::endl
             << ", MapVideoStreamInfo=" << m_mapVideoStreamInfo.getValueOr(MapVideoStreamInfo()).toString() <<::std::endl
             << ", ReachabilityDisplayMode=" << toString(m_reachabilityDisplayMode) <<::std::endl
             << ", Is EPoiDisplayForced =" << m_isEPoiDisplayForced.toString() <<::std::endl
             << ", Is GuidanceArrowVisible =" << m_isGuidanceArrowVisible.toString() <<::std::endl   //<-- INF4CV
             << ::std::endl;
      return stream.str();
   }

private:
   ValidValue<bool> m_areWeatherIconsVisible;
   ValidValue<bool> m_areSpeedCamsVisible;
   ValidValue<bool> m_areTrafficIconsVisible;
   ValidValue<bool> m_arePoiIconsVisible;
   ValidValue<bool> m_areAddressbookIconsVisible;
   ValidValue<bool> m_isEPoiDisplayForced;
   ValidValue<UpperTrafficFlowDisplayLimit> m_trafficFlowDisplayLimit;
   // this is a bent map which is visible only in 3D mode in Renault
   ValidValue<bool> m_inceptionMapState;
   ValidValue<MapVideoStreamInfo> m_mapVideoStreamInfo;
   ValidValue<ReachabilityDisplayMode> m_reachabilityDisplayMode;
   ValidValue<bool> m_isGuidanceArrowVisible;   //<-- INF4CV
};

/** Specifies grouping of related map views synchronized for view transition */
enum MapViewGroup
{
   // View group for map views to be displayed in the headunit
   MAP_VIEW_GROUP__HEADUNIT = 0,
   // View group for map views to be displayed in the cluster
   MAP_VIEW_GROUP__CLUSTER,
   // View group for map views to be displayed in rearseat entertainment
   MAP_VIEW_GROUP__REARSEAT
};

inline ::std::string toString(MapViewGroup mapViewGroup)
{
   switch (mapViewGroup)
   {
   case MAP_VIEW_GROUP__HEADUNIT:
      return "MAP_VIEW_GROUP__HEADUNIT";
   case MAP_VIEW_GROUP__CLUSTER:
      return "MAP_VIEW_GROUP__CLUSTER";
   case MAP_VIEW_GROUP__REARSEAT:
      return "MAP_VIEW_GROUP__REARSEAT";
   default:
      ::std::stringstream stream;
      stream << "MAP_VIEW_GROUP__<" << static_cast<unsigned int>(mapViewGroup) << ">";
      return stream.str();
   }
}

struct ViewModeConfiguration
{
   ViewModeConfiguration() :
      m_mapViewId(MAP_VIEW_ID__PRIMARY),
      m_screenId(SCREEN_ID__PRIMARY),
      m_mapCameraMode(MAP_MODE_INVALID)
   {
   }

   bool operator==(const ViewModeConfiguration& rhs) const
   {
      return ((m_mapViewId == rhs.m_mapViewId) &&
              (m_screenId == rhs.m_screenId) &&
              (m_mapCameraMode == rhs.m_mapCameraMode) &&
              (m_viewDimensions == rhs.m_viewDimensions) &&
              (m_activeAreaDimensions == rhs.m_activeAreaDimensions) &&
              (m_viewModeOptions.getValue() == rhs.m_viewModeOptions.getValue()));
   }

   bool operator!=(const ViewModeConfiguration& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "ViewModeConfiguration: MapViewId=" << ::navmiddleware::toString(m_mapViewId) << ::std::endl
             << ", ScreenId=" << ::navmiddleware::toString(m_screenId) << ::std::endl
             << ", CameraMode=" << ::navmiddleware::toString(m_mapCameraMode) << ::std::endl
             << ", " << m_viewDimensions.toString()
             << ", " << m_activeAreaDimensions.toString()
             << ", " << (m_viewModeOptions.isValid() ? m_viewModeOptions.getValue().toString() : "viewModeOptions not set")
             << ::std::endl;
      return stream.str();
   }

   MapViewId m_mapViewId;
   /** Identifies screen to show view on */
   ScreenId m_screenId;

   MapCameraMode m_mapCameraMode;
   /** view dimensions in screen coordinates */
   ViewDimensions m_viewDimensions;
   /** Active Area dimensions in screen coordinates */
   ViewDimensions m_activeAreaDimensions;
   ValidValue<ViewModeOptions> m_viewModeOptions;
};

/** Types for intersection map cluster view*/
enum IntersectionMapDataRoadClass
{
   IMD_UNKNOWN_ADMIN_ROAD_CLASS,             /**< Unknown road class.*/
   IMD_MOST_IMPORTANT_ADMIN_ROAD_CLASS,      /**< The most important road class.*/
   IMD_SECOND_IMPORTANT_ADMIN_ROAD_CLASS,    /**< Second important road class.*/
   IMD_THIRD_IMPORTANT_ADMIN_ROAD_CLASS,     /**< Third important road class*/
   IMD_FOURTH_IMPORTANT_ADMIN_ROAD_CLASS,    /**< Fourth important road class*/
   IMD_FIFTH_IMPORTANT_ADMIN_ROAD_CLASS,     /**< Fifth important road class*/
   IMD_SIXTH_IMPORTANT_ADMIN_ROAD_CLASS,     /**< Sixth important road class*/
   IMD_LEAST_IMPORTANT_ADMIN_ROAD_CLASS      /**< The least important road class.*/
};

inline ::std::string toString(IntersectionMapDataRoadClass intersectionMapDataRoadClass)
{
   switch (intersectionMapDataRoadClass)
   {
   case IMD_UNKNOWN_ADMIN_ROAD_CLASS:
      return "IMD_UNKNOWN_ADMIN_ROAD_CLASS";
   case IMD_MOST_IMPORTANT_ADMIN_ROAD_CLASS:
      return "IMD_MOST_IMPORTANT_ADMIN_ROAD_CLASS";
   case IMD_SECOND_IMPORTANT_ADMIN_ROAD_CLASS:
      return "IMD_SECOND_IMPORTANT_ADMIN_ROAD_CLASS";
   case IMD_THIRD_IMPORTANT_ADMIN_ROAD_CLASS:
      return "IMD_THIRD_IMPORTANT_ADMIN_ROAD_CLASS";
   case IMD_FOURTH_IMPORTANT_ADMIN_ROAD_CLASS:
      return "IMD_FOURTH_IMPORTANT_ADMIN_ROAD_CLASS";
   case IMD_FIFTH_IMPORTANT_ADMIN_ROAD_CLASS:
      return "IMD_FIFTH_IMPORTANT_ADMIN_ROAD_CLASS";
   case IMD_SIXTH_IMPORTANT_ADMIN_ROAD_CLASS:
      return "IMD_SIXTH_IMPORTANT_ADMIN_ROAD_CLASS";
   case IMD_LEAST_IMPORTANT_ADMIN_ROAD_CLASS:
      return "IMD_LEAST_IMPORTANT_ADMIN_ROAD_CLASS";
   default:
      ::std::stringstream stream;
      stream << "IMD_<" << static_cast<unsigned int>(intersectionMapDataRoadClass) << ">_IMPORTANT_ADMIN_ROAD_CLASS";
      return stream.str();
   }
}

struct IntersectionMapScreenPosition
{
   IntersectionMapScreenPosition() :
      m_xPos(0),
      m_yPos(0)
   {
   }

   IntersectionMapScreenPosition(int16_t xPos, int16_t yPos) :
      m_xPos(xPos),
      m_yPos(yPos)
   {
   }

   int16_t m_xPos;
   int16_t m_yPos;
};


struct IntersectionMapRoadLine
{
   IntersectionMapRoadLine()
   {
      m_roadclass = IMD_UNKNOWN_ADMIN_ROAD_CLASS;
   }

   IntersectionMapRoadLine(const IntersectionMapRoadLine& rhs)
   {
      m_coordinates = rhs.m_coordinates;
      m_roadclass = rhs.m_roadclass;
   }

   IntersectionMapRoadLine& operator=(const IntersectionMapRoadLine& rhs)
   {
      if (&rhs != this)
      {
         m_coordinates = rhs.m_coordinates;
         m_roadclass = rhs.m_roadclass;
      }
      return *this;
   }

   ::std::vector<IntersectionMapScreenPosition> m_coordinates;
   IntersectionMapDataRoadClass m_roadclass;
};


class IntersectionMapVectorDataInfos
{
public:

   class IntersectionMapVectorDataInfo
   {
   public:
      typedef ::std::vector<IntersectionMapRoadLine> IntersectionMapRoadLineList;

      IntersectionMapVectorDataInfo()
         : m_isValid(false)
         , m_maneuverId(0)
      {
      }

      bool isValid()
      {
         return m_isValid;
      }

      void setValidState(bool isValid)
      {
         m_isValid = isValid;
      }

      uint32_t getManeuverId() const
      {
         return m_maneuverId;
      }

      void setManeuverId(uint32_t maneuverId)
      {
         m_maneuverId = maneuverId;
      }

      const IntersectionMapRoadLineList& getRoads() const
      {
         return m_roads;
      }

      IntersectionMapRoadLineList& getRoads()
      {
         return m_roads;
      }

      void setRoads(const IntersectionMapRoadLineList& roads)
      {
         m_roads = roads;
      }

      const IntersectionMapRoadLineList& getRoute() const
      {
         return m_route;
      }

      IntersectionMapRoadLineList& getRoute()
      {
         return m_route;
      }

      void setRoute(const IntersectionMapRoadLineList& route)
      {
         m_route = route;
      }

      const ::std::vector<IntersectionMapScreenPosition>& getNoEntryIconsPositions() const
      {
         return m_noEntryIcons;
      }

      ::std::vector<IntersectionMapScreenPosition>& getNoEntryIconsPositions()
      {
         return m_noEntryIcons;
      }

      void setNoEntryIconsPositions(const ::std::vector<IntersectionMapScreenPosition>& positions)
      {
         m_noEntryIcons = positions;
      }

   private:
      bool m_isValid;
      uint32_t m_maneuverId;
      IntersectionMapRoadLineList m_roads;
      IntersectionMapRoadLineList m_route;
      ::std::vector<IntersectionMapScreenPosition> m_noEntryIcons;
   };

   const IntersectionMapVectorDataInfo& getInfo() const
   {
      return m_info;
   }

   IntersectionMapVectorDataInfo& getInfo()
   {
      return m_info;
   }

   void setInfo(const IntersectionMapVectorDataInfo& info)
   {
      m_info = info;
   }
private:
   IntersectionMapVectorDataInfo m_info;
};

class MapVideoStreamStatusInfo
{
public:
   MapVideoStreamStatusInfo()
      : m_mapVideoStreamStatus(MAP_VIDEO_STREAM_STATUS__UNAVAILABLE)
   {

   }

   enum MapVideoStreamStatus
   {
      MAP_VIDEO_STREAM_STATUS__UNAVAILABLE,
      MAP_VIDEO_STREAM_STATUS__AVAILABLE
   };

   static ::std::string toString(MapVideoStreamStatus mapVideoStreamStatus)
   {
      switch (mapVideoStreamStatus)
      {
      case MAP_VIDEO_STREAM_STATUS__UNAVAILABLE:
         return "MAP_VIDEO_STREAM_STATUS__UNAVAILABLE";
      case MAP_VIDEO_STREAM_STATUS__AVAILABLE:
         return "MAP_VIDEO_STREAM_STATUS__AVAILABLE";
      default:
         ::std::stringstream stream;
         stream << "MAP_VIDEO_STREAM_STATUS__<" << static_cast<unsigned int>(mapVideoStreamStatus) << ">";
         return stream.str();
      }
   }

   MapVideoStreamStatus getMapVideoStreamStatus() const
   {
      return m_mapVideoStreamStatus;
   }

   void setMapVideoStreamStatus(MapVideoStreamStatus value)
   {
      m_mapVideoStreamStatus = value;
   }

   bool operator==(const MapVideoStreamStatusInfo& rhs) const
   {
      return ((m_mapVideoStreamStatus == rhs.m_mapVideoStreamStatus));
   }

   bool operator!=(const MapVideoStreamStatusInfo& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "MapVideoStreamInfo: MapVideoStreamStatus =" << toString(m_mapVideoStreamStatus)
             << ::std::endl;
      return stream.str();
   }

private:
   MapVideoStreamStatus m_mapVideoStreamStatus;

};

/** information about the completed setMapCameraAndViewModes call signaled by the property "MAP_SET_MAP_CAMERA_AND_VIEW_MODES_DONE"*/
class MapCameraAndViewModesStatusInfo
{
public:
   MapCameraAndViewModesStatusInfo():
      m_mapViewGroup(MAP_VIEW_GROUP__HEADUNIT)
   {}

   MapViewGroup getMapViewGroup() const
   {
      return m_mapViewGroup;
   }

   void setMapViewGroup(MapViewGroup mapViewGroup)
   {
      m_mapViewGroup = mapViewGroup;
   }

   bool operator==(const MapCameraAndViewModesStatusInfo& rhs) const
   {
      return ((m_mapViewGroup == rhs.m_mapViewGroup));
   }

   bool operator!=(const MapCameraAndViewModesStatusInfo& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "MapCameraAndViewModesStatusInfo: MapViewGroup =" << ::navmiddleware::toString(m_mapViewGroup)
             << ::std::endl;
      return stream.str();
   }

private:
   MapViewGroup m_mapViewGroup;
};

/**
 * carsor position for the current intersection
 */
class IntersectionMapCarsorPosition
{
public:
   struct CarsorPosition
   {
      CarsorPosition()
         : xPos(0)
         , yPos(0) {}

      uint32_t xPos; /**< x-coordinate on screen of the carsor position*/
      uint32_t yPos; /**< y-coordinate on screen of the carsor position*/
   };

   enum CarsorState
   {
      ON,            /**< carsor position is within the view port*/
      OFF,           /**< carsor has passed the intersection*/
      STANDBY        /**< carsor is approaching the intersection*/
   };

   static ::std::string toString(CarsorState CarsorState)
   {
      switch (CarsorState)
      {
      case ON:
         return "ON";
      case OFF:
         return "OFF";
      case STANDBY:
         return "STANDBY";
      default:
         ::std::stringstream stream;
         stream << "CARSOR_STATE_<" << static_cast<unsigned int>(CarsorState) << ">";
         return stream.str();
      }
   }

   IntersectionMapCarsorPosition():
      m_maneuverId(0),
      m_carsorPosition(),
      m_carsorAngle(0),
      m_carsorState(OFF)
   {

   }

   void setManeuverId(uint32_t maneuverId)
   {
      m_maneuverId = maneuverId;
   }

   uint32_t getManeuverId() const
   {
      return m_maneuverId;
   }

   void setCarsorPosition(const CarsorPosition& carsorPosition)
   {
      m_carsorPosition.xPos = carsorPosition.xPos;
      m_carsorPosition.yPos = carsorPosition.yPos;
   }

   const CarsorPosition& getCarsorPosition() const
   {
      return m_carsorPosition;
   }

   void setCarsorAngle(int32_t carsorAngle)
   {
      m_carsorAngle = carsorAngle;
   }

   int32_t getCarsorAngle() const
   {
      return m_carsorAngle;
   }

   void setCarsorState(CarsorState carsorState)
   {
      m_carsorState = carsorState;
   }

   CarsorState getCarsorState() const
   {
      return m_carsorState;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "Intersection map ID = " << m_maneuverId << ::std::endl
             << "Carsor position: xPos = " << m_carsorPosition.xPos << " yPos = " << m_carsorPosition.yPos<<::std::endl
             << "Carsor angle  = " << m_carsorAngle << ::std::endl
             << "Carsor state  = " << toString(m_carsorState) << ::std::endl;

      return stream.str();
   }

private:
   uint32_t m_maneuverId;              ///< maneuver id of the current maneuver
   CarsorPosition m_carsorPosition;    ///< carsor position on the screen
   int32_t m_carsorAngle;              ///< Rotation angle of the carsor
   ///< (range: 0 to 360, 0: East, Counter-clockwise)
   CarsorState m_carsorState;
};

/**
 * Info about the street view preview image returned in response to getStreetViewPreviewImage() call.
 */
class StreetViewPreviewImageInfo
{
public:

   StreetViewPreviewImageInfo():
      m_width(0),
      m_height(0),
      m_onlineImageStatus(ONLINE_IMAGE_STATUS__UNKNOWN)
   {
   }

   void setWidth(uint32_t width)
   {
      m_width = width;
   }

   uint32_t getWidth() const
   {
      return m_width;
   }

   void setHeight(uint32_t height)
   {
      m_height = height;
   }

   uint32_t getHeight() const
   {
      return m_height;
   }

   void setGeoCoordinate(const GeoCoordinateDegree& geoCoordinate)
   {
      m_geoCoordinate = geoCoordinate;
   }

   const GeoCoordinateDegree& getGeoCoordinate() const
   {
      return m_geoCoordinate;
   }

   void setAddress(const ValidValue< ::std::string >& address)
   {
      m_address = address;
   }

   const ValidValue< ::std::string >& getAddress() const
   {
      return m_address;
   }

   void setImage(const Image& image)
   {
      m_image = image;
   }

   const Image& getImage() const
   {
      return m_image;
   }

   void setStreetViewPreviewImageStatus(OnlineImageStatus onlineImageStatus)
   {
      m_onlineImageStatus = onlineImageStatus;
   }

   OnlineImageStatus getStreetViewPreviewImageStatus() const
   {
      return m_onlineImageStatus;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "Width = " << m_width << ::std::endl
             << "Height = " << m_height << ::std::endl
             << "GeoCoordinateDegree: " << m_geoCoordinate.toString()<<::std::endl
             << "Address  = " << m_address.toString() << ::std::endl
             << "Image  = " << m_image.toString() << ::std::endl
             << "Street View Preview Online Image Status = " << ::navmiddleware::toString(m_onlineImageStatus) << ::std::endl;

      return stream.str();
   }

private:
   uint32_t m_width;                         /**< width of the preview image*/
   uint32_t m_height;                        /**< height of the preview image*/
   GeoCoordinateDegree m_geoCoordinate;      /**< geocoordinate of the location where preview image is requested*/
   ValidValue< ::std::string > m_address;    /**< address of the location where preview image is requested*/
   Image m_image;                            /**< preview image data*/
   OnlineImageStatus m_onlineImageStatus;    /**< online image status of the preview image*/
};

/**
 * Availability status of various weather features, to allow HMI to show the "Loading" image when a particular
 * weather feature requested by user is not available.
 */
class WeatherFeaturesStatusInfo
{
public:
   /** availability status of the weather feature*/
   enum WeatherFeatureAvailability
   {
      NOT_AVAILABLE = 0,   /**< weather feature is not available*/
      AVAILABLE            /**< weather feature is available*/
   };

   static ::std::string toString(WeatherFeatureAvailability weatherFeatureAvailability)
   {
      switch (weatherFeatureAvailability)
      {
      case NOT_AVAILABLE:
         return "NOT_AVAILABLE";
      case AVAILABLE:
         return "AVAILABLE";
      default:
         ::std::stringstream stream;
         stream << "WEATHER_FEATURE_AVAILABILITY_<" << static_cast<unsigned int>(weatherFeatureAvailability) << ">";
         return stream.str();
      }
   }

   WeatherFeaturesStatusInfo():
      m_stormAttributesAvailability(NOT_AVAILABLE),
      m_stormTrackAvailability(NOT_AVAILABLE),
      m_pressureMapAvailability(NOT_AVAILABLE),
      m_weatherIconsAvailability(NOT_AVAILABLE),
      m_weatherWindAvailability(NOT_AVAILABLE),
      m_weatherRadarAvailability(NOT_AVAILABLE),
      m_weatherAlertsAvailability(NOT_AVAILABLE),
      m_mapViewId(MAP_VIEW_ID__PRIMARY)
   {
   }

   void setMapViewId(MapViewId mapViewId)
   {
      m_mapViewId = mapViewId;
   }

   MapViewId getMapViewId() const
   {
      return m_mapViewId;
   }

   WeatherFeatureAvailability getStormAttributesAvailability() const
   {
      return m_stormAttributesAvailability;
   }

   void setStormAttributesAvailability(WeatherFeatureAvailability stormAttributesAvailability)
   {
      m_stormAttributesAvailability = stormAttributesAvailability;
   }

   WeatherFeatureAvailability getStormTrackAvailability() const
   {
      return m_stormTrackAvailability;
   }

   void setStormTrackAvailability(WeatherFeatureAvailability stormTrackAvailability)
   {
      m_stormTrackAvailability = stormTrackAvailability;
   }

   WeatherFeatureAvailability getPressureMapAvailability() const
   {
      return m_pressureMapAvailability;
   }

   void setPressureMapAvailability(WeatherFeatureAvailability pressureMapAvailability)
   {
      m_pressureMapAvailability = pressureMapAvailability;
   }

   WeatherFeatureAvailability getWeatherIconsAvailability() const
   {
      return m_weatherIconsAvailability;
   }

   void setWeatherIconsAvailability(WeatherFeatureAvailability weatherIconsAvailability)
   {
      m_weatherIconsAvailability = weatherIconsAvailability;
   }

   WeatherFeatureAvailability getWeatherWindAvailability() const
   {
      return m_weatherWindAvailability;
   }

   void setWeatherWindAvailability(WeatherFeatureAvailability weatherWindAvailability)
   {
      m_weatherWindAvailability = weatherWindAvailability;
   }

   WeatherFeatureAvailability getWeatherRadarAvailability() const
   {
      return m_weatherRadarAvailability;
   }

   void setWeatherRadarAvailability(WeatherFeatureAvailability weatherRadarAvailability)
   {
      m_weatherRadarAvailability = weatherRadarAvailability;
   }

   WeatherFeatureAvailability getWeatherAlertsAvailability() const
   {
      return m_weatherAlertsAvailability;
   }

   void setWeatherAlertsAvailability(WeatherFeatureAvailability weatherAlertsAvailability)
   {
      m_weatherAlertsAvailability = weatherAlertsAvailability;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("WeatherFeaturesStatusInfo payload:\n");
      stream << "StormAttributesAvailability = " << toString(m_stormAttributesAvailability) << ::std::endl
             << "StormTrackAvailability = " << toString(m_stormTrackAvailability) << ::std::endl
             << "PressureMapAvailability = " << toString(m_pressureMapAvailability) << ::std::endl
             << "WeatherIconsAvailability = " << toString(m_weatherIconsAvailability) << ::std::endl
             << "WeatherWindAvailability = " << toString(m_weatherWindAvailability) << ::std::endl
             << "WeatherRadarAvailability = " << toString(m_weatherRadarAvailability) << ::std::endl
             << "WeatherAlertsAvailability = " << toString(m_weatherAlertsAvailability) << ::std::endl
             << "Map View Id = " << ::navmiddleware::toString(m_mapViewId) << ::std::endl;

      return stream.str();
   }

private:
   /** Availability status for Weather Storm Attributes. */
   WeatherFeatureAvailability m_stormAttributesAvailability;
   /** Availability status for Weather Storm Track. */
   WeatherFeatureAvailability m_stormTrackAvailability;
   /** Availability status for Weather Pressure Map */
   WeatherFeatureAvailability m_pressureMapAvailability;
   /** Availability status for Weather Icons. */
   WeatherFeatureAvailability m_weatherIconsAvailability;
   /** Availability status for Weather Wind. */
   WeatherFeatureAvailability m_weatherWindAvailability;
   /** Availability status for Weather Radar. */
   WeatherFeatureAvailability m_weatherRadarAvailability;
   /** Availability status for Weather Alerts. */
   WeatherFeatureAvailability m_weatherAlertsAvailability;
   /* View id for the map */
   MapViewId m_mapViewId;
};

/**
 * When bringing the map into Camera mode "Free", this enum indicates whether to use the current camera configuration
 * or to restore the last free mode camera configuration.
 * If the user is entering Avoid area in free mode, then HMI has to set the CameraConfiguration to CAMERA_CONFIGURATION_AVOID_AREA.
 */
enum CameraConfiguration
{
   CAMERA_CONFIGURATION__CURRENT,    /**< use the current camera configuration*/
   CAMERA_CONFIGURATION__LAST,       /**< use the last free mode camera configuration*/
   CAMERA_CONFIGURATION__AVOID_AREA   /**< prepare the camera for the avoid area*/
};

inline ::std::string toString(CameraConfiguration cameraConfiguration)
{
   switch (cameraConfiguration)
   {
   case CAMERA_CONFIGURATION__CURRENT:
      return "CAMERA_CONFIGURATION__CURRENT";
   case CAMERA_CONFIGURATION__LAST:
      return "CAMERA_CONFIGURATION__LAST";
   case CAMERA_CONFIGURATION__AVOID_AREA:
      return "CAMERA_CONFIGURATION__AVOID_AREA";
   default:
      ::std::stringstream stream;
      stream << "CAMERA_CONFIGURATION__<" << static_cast<unsigned int>(cameraConfiguration) << ">";
      return stream.str();
   }
}

/**
 * Availability status of certain online features like street view and satellite view.
 */
class MapOnlineFeatureAvailabilityInfos
{
public:
   enum FeatureStatus
   {
      /**
       * Online feature is available.
       */
      FeatureStatus__AVAILABLE,
      /**
       * Online feature is unavailable.
       */
      FeatureStatus__UNAVAILABLE,
      /**
       * Online feature is not subscribed.
       */
      FeatureStatus__NOT_SUBSCRIBED,
      /**
       * Online feature is not provided.
       */
      FeatureStatus__NOT_PROVIDED
   };

   static ::std::string toString(FeatureStatus featureStatus)
   {
      switch (featureStatus)
      {
      case FeatureStatus__AVAILABLE:
         return "FeatureStatus__AVAILABLE";
      case FeatureStatus__UNAVAILABLE:
         return "FeatureStatus__UNAVAILABLE";
      case FeatureStatus__NOT_SUBSCRIBED:
         return "FeatureStatus__NOT_SUBSCRIBED";
      case FeatureStatus__NOT_PROVIDED:
         return "FeatureStatus__NOT_PROVIDED";
      default:
         ::std::stringstream stream;
         stream << "FeatureStatus__<" << static_cast<unsigned int>(featureStatus) << ">";
         return stream.str();
      }
   }

   MapOnlineFeatureAvailabilityInfos() :
      m_streetViewFeatureStatus(FeatureStatus__UNAVAILABLE),
      m_staticSatelliteViewFeatureStatus(FeatureStatus__UNAVAILABLE),
      m_dynamicSatelliteViewFeatureStatus(FeatureStatus__UNAVAILABLE)
   {

   }

   void setStreetViewFeatureStatus(FeatureStatus featureStatus)
   {
      m_streetViewFeatureStatus = featureStatus;
   }

   FeatureStatus getStreetViewFeatureStatus() const
   {
      return m_streetViewFeatureStatus;
   }

   void setStaticSatelliteViewFeatureStatus(FeatureStatus featureStatus)
   {
      m_staticSatelliteViewFeatureStatus = featureStatus;
   }

   FeatureStatus getStaticSatelliteViewFeatureStatus() const
   {
      return m_staticSatelliteViewFeatureStatus;
   }

   void setDynamicSatelliteViewFeatureStatus(FeatureStatus featureStatus)
   {
      m_dynamicSatelliteViewFeatureStatus = featureStatus;
   }

   FeatureStatus getDynamicSatelliteViewFeatureStatus() const
   {
      return m_dynamicSatelliteViewFeatureStatus;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("Map Online Feature Availability Infos payload:\n");
      stream << "Street View Feature Status = "
             << ::navmiddleware::MapOnlineFeatureAvailabilityInfos::toString(m_streetViewFeatureStatus) << ::std::endl
             << "Static Satellite View Feature Status = "
             << ::navmiddleware::MapOnlineFeatureAvailabilityInfos::toString(m_staticSatelliteViewFeatureStatus) << ::std::endl
             << "Dynamic Satellite View Feature Status = "
             << ::navmiddleware::MapOnlineFeatureAvailabilityInfos::toString(m_dynamicSatelliteViewFeatureStatus) << ::std::endl;

      return stream.str();
   }

private:
   FeatureStatus m_streetViewFeatureStatus;           /** availability status of street view feature.*/
   FeatureStatus m_staticSatelliteViewFeatureStatus;  /** availability status of satellite view feature.*/
   FeatureStatus m_dynamicSatelliteViewFeatureStatus; /** availability status of dynamic satellite view feature.*/
};

class MapCaptureResultInfo
{
public:
   enum MapCaptureResult
   {
      MAP_CAPTURE_RESULT__FAILURE,
      MAP_CAPTURE_RESULT__SUCCESS
   };

   static ::std::string toString(MapCaptureResult mapCaptureResult)
   {
      switch (mapCaptureResult)
      {
      case MAP_CAPTURE_RESULT__FAILURE:
         return "MapCaptureResult__Failure";
      case MAP_CAPTURE_RESULT__SUCCESS:
         return "MapCaptureResult__Success";
      default:
         ::std::stringstream stream;
         stream << "MapCaptureResul__<" << static_cast<unsigned int>(mapCaptureResult) << ">";
         return stream.str();
      }
   }

   MapCaptureResultInfo()
      : m_mapViewId(MAP_VIEW_ID__PRIMARY)
      , m_captureResult(MAP_CAPTURE_RESULT__FAILURE)
   {
   }

   void setMapViewId(MapViewId mapViewId)
   {
      m_mapViewId = mapViewId;
   }

   MapViewId getMapViewId() const
   {
      return m_mapViewId;
   }

   void setMapCapture(MapCaptureResult mapCaptureResult)
   {
      m_captureResult = mapCaptureResult;
   }

   MapCaptureResult getMapCaptureResult() const
   {
      return m_captureResult;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("Map Capture result Infos payload:\n");
      stream << "MapCaptureResult = "
             << ::navmiddleware::MapCaptureResultInfo::toString(m_captureResult) << ::std::endl
             << "Map View Id = " << ::navmiddleware::toString(m_mapViewId) << ::std::endl;

      return stream.str();
   }

private :
   MapViewId m_mapViewId;
   MapCaptureResult m_captureResult;
};

class MapDynamicIconInfo
{
public:
   MapDynamicIconInfo()
   {}

   MapDynamicIconInfo(const MapDynamicIconId& mapDynamicIconId, const GeoCoordinateDegree& geoCoordinate, const ::std::string& iconPath)
      : m_mapDynamicIconId(mapDynamicIconId)
      , m_geoCoordinate(geoCoordinate)
      , m_iconPath(iconPath)
   {}

   bool operator==(const MapDynamicIconInfo& rhs) const
   {
      return ((m_mapDynamicIconId == rhs.m_mapDynamicIconId) &&
              (m_geoCoordinate == rhs.m_geoCoordinate) &&
              (m_iconPath == rhs.m_iconPath));
   }

   bool operator!=(const MapDynamicIconInfo& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DynamicIconInfo payload:\n");
      stream << "m_mapDynamicIconId = " << m_mapDynamicIconId.toString() << ::std::endl
             << "m_geoCoordinate = " << m_geoCoordinate.toString() << ::std::endl
             << "m_iconPath = " << m_iconPath << ::std::endl;

      return stream.str();
   }

   void setMapDynamicIconId(const MapDynamicIconId& mapDynamicIconId)
   {
      m_mapDynamicIconId = mapDynamicIconId;
   }

   const MapDynamicIconId& getMapDynamicIconId() const
   {
      return m_mapDynamicIconId;
   }

   void setGeoCoordinate(const GeoCoordinateDegree& geoCoordinate)
   {
      m_geoCoordinate = geoCoordinate;
   }

   const GeoCoordinateDegree& getGeoCoodinate() const
   {
      return m_geoCoordinate;
   }

   void setIconPath(const ::std::string& iconPath)
   {
      m_iconPath = iconPath;
   }

   const ::std::string& getIconPath() const
   {
      return m_iconPath;
   }

private:
   MapDynamicIconId m_mapDynamicIconId;
   GeoCoordinateDegree m_geoCoordinate;
   ::std::string m_iconPath;
};

enum DynamicIconRequestStatus
{
   DYNAMIC_ICON_REQUEST_STATUS__UNKNOWN,           /** Dynamic icon status unknown /no response available*/
   DYNAMIC_ICON_REQUEST_STATUS__SUCCESS,           /** Requested Dynamic icon added successfully */
   DYNAMIC_ICON_REQUEST_STATUS__FAILURE            /** Dynamic icon addition failed - icon was not added */
};

inline ::std::string toString(DynamicIconRequestStatus dynamicIconRequestStatus)
{
   switch (dynamicIconRequestStatus)
   {
   case DYNAMIC_ICON_REQUEST_STATUS__UNKNOWN:
      return "DYNAMIC_ICON_REQUEST_STATUS__UNKNOWN";
   case DYNAMIC_ICON_REQUEST_STATUS__SUCCESS:
      return "DYNAMIC_ICON_REQUEST_STATUS__SUCCESS";
   case DYNAMIC_ICON_REQUEST_STATUS__FAILURE:
      return "DYNAMIC_ICON_REQUEST_STATUS__FAILURE";
   default:
      ::std::stringstream stream;
      stream << "DYNAMIC_ICON_REQUEST_STATUS__<" << static_cast<unsigned int>(dynamicIconRequestStatus) << ">";
      return stream.str();
   }
}

class DynamicIconRequestResult
{
public:
   DynamicIconRequestResult()
   {
      m_dynamicIconRequestResult.clear();
   }

   bool operator==(const DynamicIconRequestResult& rhs) const
   {
      return (m_dynamicIconRequestResult == rhs.m_dynamicIconRequestResult);
   }

   bool operator!=(const DynamicIconRequestResult& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DynamicIconRequestResult payload:\n");
      if(!m_dynamicIconRequestResult.empty())
      {
         for( ::std::map<MapDynamicIconId, DynamicIconRequestStatus>::const_iterator iter = m_dynamicIconRequestResult.begin();
               iter != m_dynamicIconRequestResult.end(); ++iter)
         {
            stream << iter->first.toString() << " -> Status = " << ::navmiddleware::toString(iter->second) << ::std::endl;
         }
      }
      return stream.str();
   }

   void setDynamicIconRequestResult(const ::std::map<MapDynamicIconId, DynamicIconRequestStatus>& dynamicIconRequestResult)
   {
      m_dynamicIconRequestResult = dynamicIconRequestResult;
   }

   const ::std::map<MapDynamicIconId, DynamicIconRequestStatus>& getDynamicIconRequestResult() const
   {
      return m_dynamicIconRequestResult;
   }

private:
   /**
    * This map will contain result for addDynamicIconsOnMap() and deleteDynamicIconsOnMap() operations
    * The number of entries in the map will be equal to the requested icon list size.
    * This result can be used to re-trigger the request on failure.
    * NOTE : The feedback is currently supported only for the addDynamicIconsOnMap() call.
    */
   ::std::map<MapDynamicIconId, DynamicIconRequestStatus> m_dynamicIconRequestResult;
};

class RouteVisibilityOptions
{
public:
   RouteVisibilityOptions() :
      m_showRoute(false),
      m_includeCarsor(false),
      m_showDestinationFlag(false)
   {}

   RouteVisibilityOptions(bool showRoute, bool includeCarsor, bool showDestinationFlag)
   {
      m_showRoute = showRoute;
      m_includeCarsor = includeCarsor;
      m_showDestinationFlag = showDestinationFlag;
   }

   bool operator==(const RouteVisibilityOptions& rhs) const
   {
      return ((m_showRoute == rhs.m_showRoute) &&
              (m_includeCarsor == rhs.m_includeCarsor) &&
              (m_showDestinationFlag == rhs.m_showDestinationFlag));
   }

   bool operator!=(const RouteVisibilityOptions& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("RouteViewOptions payload:\n");
      stream << "ShowRoute = " << (m_showRoute ? "true" : "false") << " "
             << "IncludeCarsor = " << (m_includeCarsor ? "true" : "false") << " "
             << "ShowDestinationFlag = " << (m_showDestinationFlag ? "true" : "false") << ::std::endl;

      return stream.str();
   }

   void setRouteVisibility(bool showRoute)
   {
      m_showRoute = showRoute;
   }

   bool getRouteVisibility() const
   {
      return m_showRoute;
   }

   void setCarsorInclusion(bool includeCarsor)
   {
      m_includeCarsor = includeCarsor;
   }

   bool getCarsorInclusion() const
   {
      return m_includeCarsor;
   }

   void setDestinationFlagVisibility(bool showDestinationFlag)
   {
      m_showDestinationFlag = showDestinationFlag;
   }

   bool getDestinationFlagVisibility() const
   {
      return m_showDestinationFlag;
   }

private:
   bool  m_showRoute;
   bool  m_includeCarsor;
   bool  m_showDestinationFlag;
};

} // namespace navmiddleware

#endif  // PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_MAPINFOS_H_
