/**
* @copyright (c) 2017-2020 Robert Bosch Car Multimedia GmbH
* @addtogroup NavMiddleware
* @{
*/

#ifndef PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_SETTINGS_LOCATIONSETTINGS_H_
#define PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_SETTINGS_LOCATIONSETTINGS_H_

#include <string>

namespace navmiddleware
{
namespace settings
{

class EvChargingStationFilterSettings
{
public:
   EvChargingStationFilterSettings():
      m_giveQuickChargingStationsOnly(false),
      m_giveAvailableSpotOnly(false),
      m_giveFreeOfCostOnly(false),
      m_giveOpen24HoursOnly(false),
      m_giveOpenForAtLeastNMinutesOnly(false),
      m_giveNcsOnly(false)
   {}

   EvChargingStationFilterSettings(
      bool giveQuickChargingStationsOnly,
      bool giveAvailableSpotOnly,
      bool giveFreeOfCostOnly,
      bool giveOpen24HoursOnly,
      bool giveOpenForAtLeastNMinutesOnly,
      bool giveNcsOnly)

      : m_giveQuickChargingStationsOnly(giveQuickChargingStationsOnly)
      , m_giveAvailableSpotOnly(giveAvailableSpotOnly)
      , m_giveFreeOfCostOnly(giveFreeOfCostOnly)
      , m_giveOpen24HoursOnly(giveOpen24HoursOnly)
      , m_giveOpenForAtLeastNMinutesOnly(giveOpenForAtLeastNMinutesOnly)
      , m_giveNcsOnly(giveNcsOnly)
   {}

   bool operator==(const EvChargingStationFilterSettings& rhs) const
   {
      return (m_giveQuickChargingStationsOnly == rhs.m_giveQuickChargingStationsOnly) &&
             (m_giveAvailableSpotOnly == rhs.m_giveAvailableSpotOnly) &&
             (m_giveFreeOfCostOnly == rhs.m_giveFreeOfCostOnly) &&
             (m_giveOpen24HoursOnly == rhs.m_giveOpen24HoursOnly) &&
             (m_giveOpenForAtLeastNMinutesOnly == rhs.m_giveOpenForAtLeastNMinutesOnly) &&
             (m_giveNcsOnly == rhs.m_giveNcsOnly);
   }
   bool operator!=(const EvChargingStationFilterSettings& rhs) const
   {
      return !(*this == rhs);
   }

   /**
    * Give charging stations supporting quick charging, only.
    */
   bool doGiveQuickChargingStationsOnly() const
   {
      return m_giveQuickChargingStationsOnly;
   }
   void setGiveQuickChargingStationsOnly(bool status)
   {
      m_giveQuickChargingStationsOnly = status;
   }

   /**
    * Give charging stations where currently some charging spot is available (not used for charging), only.
    */
   bool doGiveAvailableSpotOnly() const
   {
      return m_giveAvailableSpotOnly;
   }
   void setGiveAvailableSpotOnly(bool status)
   {
      m_giveAvailableSpotOnly = status;
   }

   /**
    * Give charging stations where vehicle can be charged for free (of cost), only.
    */
   bool doGiveFreeOfCostOnly() const
   {
      return m_giveFreeOfCostOnly;
   }
   void setGiveFreeOfCostOnly(bool status)
   {
      m_giveFreeOfCostOnly = status;
   }

   /**
    * Give charging stations which are open 24 hours, only.
    */
   bool doGiveOpen24HoursOnly() const
   {
      return m_giveOpen24HoursOnly;
   }
   void setGiveOpen24HoursOnly(bool status)
   {
      m_giveOpen24HoursOnly = status;
   }

   /**
    * Give charging stations which are currently open for a define time span of at least N minutes, only.
    * The number N of minutes is defined in another place.
    */
   bool doGiveOpenForAtLeastNMinutesOnly() const
   {
      return m_giveOpenForAtLeastNMinutesOnly;
   }
   void setGiveOpenForAtLeastNMinutesOnly(bool status)
   {
      m_giveOpenForAtLeastNMinutesOnly = status;
   }

   /**
    * Give NCS charging stations (only relevant for Japan), only.
    */
   bool doGiveNcsOnly() const
   {
      return m_giveNcsOnly;
   }
   void setGiveNcsOnly(bool status)
   {
      m_giveNcsOnly = status;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("EvChargingStationFilterSettings payload:\n");
      stream << "   m_giveQuickChargingStationsOnly = " << m_giveQuickChargingStationsOnly << ::std::endl
             << "   m_giveAvailableSpotOnly = " << m_giveAvailableSpotOnly << ::std::endl
             << "   m_giveFreeOfCostOnly = " << m_giveFreeOfCostOnly << ::std::endl
             << "   m_giveOpen24HoursOnly = " << m_giveOpen24HoursOnly << ::std::endl
             << "   m_giveOpenForAtLeastNMinutesOnly = " << m_giveOpenForAtLeastNMinutesOnly << ::std::endl
             << "   m_giveNcsOnly = " << m_giveNcsOnly << ::std::endl;
      return stream.str();
   }

private:
   bool m_giveQuickChargingStationsOnly; //@Deprecated: please use m_acceptQuickCharge from EvChargingSpeedFilterSettings
   bool m_giveAvailableSpotOnly;
   bool m_giveFreeOfCostOnly;
   bool m_giveOpen24HoursOnly;
   bool m_giveOpenForAtLeastNMinutesOnly;
   bool m_giveNcsOnly;
};

class PaymentMethodSettings
{
public:
   PaymentMethodSettings()
      : m_noPayment(false)
      , m_paymentMethodUnknown(false)
      , m_variablePaymentMethod(false)
      , m_acceptCash(false)
      , m_acceptCoins(false)
      , m_acceptEcCard(false)
      , m_acceptCreditCard(false)
      , m_acceptElectronicPurse(false)
      , m_acceptElectronicTollCollection(false)
      , m_acceptMobilePayment(false)
      , m_acceptMemberCard(false)
      , m_acceptServiceProviderPaymentMethod(false)
   {}

   PaymentMethodSettings(
      bool noPayment,
      bool paymentMethodUnknown,
      bool variablePaymentMethod,
      bool acceptCash,
      bool acceptCoins,
      bool acceptEcCard,
      bool acceptCreditCard,
      bool acceptElectronicPurse,
      bool acceptElectronicTollCollection,
      bool acceptMobilePayment,
      bool acceptMemberCard,
      bool acceptServiceProviderPaymentMethod)

      : m_noPayment(noPayment)
      , m_paymentMethodUnknown(paymentMethodUnknown)
      , m_variablePaymentMethod(variablePaymentMethod)
      , m_acceptCash(acceptCash)
      , m_acceptCoins(acceptCoins)
      , m_acceptEcCard(acceptEcCard)
      , m_acceptCreditCard(acceptCreditCard)
      , m_acceptElectronicPurse(acceptElectronicPurse)
      , m_acceptElectronicTollCollection(acceptElectronicTollCollection)
      , m_acceptMobilePayment(acceptMobilePayment)
      , m_acceptMemberCard(acceptMemberCard)
      , m_acceptServiceProviderPaymentMethod(acceptServiceProviderPaymentMethod)
   {}

   bool operator==(const PaymentMethodSettings& rhs) const
   {
      return (m_noPayment == rhs.m_noPayment) &&
             (m_paymentMethodUnknown == rhs.m_paymentMethodUnknown) &&
             (m_variablePaymentMethod == rhs.m_variablePaymentMethod) &&
             (m_acceptCash == rhs.m_acceptCash) &&
             (m_acceptCoins == rhs.m_acceptCoins) &&
             (m_acceptEcCard == rhs.m_acceptEcCard) &&
             (m_acceptCreditCard == rhs.m_acceptCreditCard) &&
             (m_acceptElectronicPurse == rhs.m_acceptElectronicPurse) &&
             (m_acceptElectronicTollCollection == rhs.m_acceptElectronicTollCollection) &&
             (m_acceptMobilePayment == rhs.m_acceptMobilePayment) &&
             (m_acceptMemberCard == rhs.m_acceptMemberCard) &&
             (m_acceptServiceProviderPaymentMethod == rhs.m_acceptServiceProviderPaymentMethod);
   }
   bool operator!=(const PaymentMethodSettings& rhs) const
   {
      return !(*this == rhs);
   }

   bool getNoPayment() const
   {
      return m_noPayment;
   }

   void setNoPayment(bool noPayment)
   {
      m_noPayment = noPayment;
   }

   bool getPaymentMethodUnknown() const
   {
      return m_paymentMethodUnknown;
   }

   void setPaymentMethodUnknown(bool paymentMethodUnknown)
   {
      m_paymentMethodUnknown = paymentMethodUnknown;
   }

   bool getVariablePaymentMethod() const
   {
      return m_variablePaymentMethod;
   }

   void setVariablePaymentMethod(bool variablePaymentMethod)
   {
      m_variablePaymentMethod = variablePaymentMethod;
   }

   bool getAcceptCash() const
   {
      return m_acceptCash;
   }

   void setAcceptCash(bool acceptCash)
   {
      m_acceptCash = acceptCash;
   }

   bool getAcceptCoins() const
   {
      return m_acceptCoins;
   }

   void setAcceptCoins(bool acceptCoins)
   {
      m_acceptCoins = acceptCoins;
   }

   bool getAcceptEcCard() const
   {
      return m_acceptEcCard;
   }

   void setAcceptEcCard(bool acceptEcCard)
   {
      m_acceptEcCard = acceptEcCard;
   }

   bool getAcceptCreditCard() const
   {
      return m_acceptCreditCard;
   }

   void setAcceptCreditCard(bool acceptCreditCard)
   {
      m_acceptCreditCard = acceptCreditCard;
   }

   bool getAcceptElectronicPurse() const
   {
      return m_acceptElectronicPurse;
   }

   void setAcceptElectronicPurse(bool acceptElectronicPurse)
   {
      m_acceptElectronicPurse = acceptElectronicPurse;
   }

   bool getAcceptElectronicTollCollection() const
   {
      return m_acceptElectronicTollCollection;
   }

   void setAcceptElectronicTollCollection(bool acceptElectronicTollCollection)
   {
      m_acceptElectronicTollCollection = acceptElectronicTollCollection;
   }

   bool getAcceptMobilePayment() const
   {
      return m_acceptMobilePayment;
   }

   void setAcceptMobilePayment(bool acceptMobilePayment)
   {
      m_acceptMobilePayment = acceptMobilePayment;
   }

   bool getAcceptMemberCard() const
   {
      return m_acceptMemberCard;
   }

   void setAcceptMemberCard(bool acceptMemberCard)
   {
      m_acceptMemberCard = acceptMemberCard;
   }

   bool getAcceptServiceProviderPaymentMethod() const
   {
      return m_acceptServiceProviderPaymentMethod;
   }

   void setAcceptServiceProviderPaymentMethod(bool acceptServiceProviderPaymentMethod)
   {
      m_acceptServiceProviderPaymentMethod = acceptServiceProviderPaymentMethod;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("PaymentMethodSettings payload:\n");
      stream << "   m_noPayment = " << m_noPayment << ::std::endl
             << "   m_paymentMethodUnknown = " << m_paymentMethodUnknown << ::std::endl
             << "   m_variablePaymentMethod = " << m_variablePaymentMethod << ::std::endl
             << "   m_acceptCash = " << m_acceptCash << ::std::endl
             << "   m_acceptCoins = " << m_acceptCoins << ::std::endl
             << "   m_acceptEcCard = " << m_acceptEcCard << ::std::endl
             << "   m_acceptCreditCard = " << m_acceptCreditCard << ::std::endl
             << "   m_acceptElectronicPurse = " << m_acceptElectronicPurse << ::std::endl
             << "   m_acceptElectronicTollCollection = " << m_acceptElectronicTollCollection << ::std::endl
             << "   m_acceptMobilePayment = " << m_acceptMobilePayment << ::std::endl
             << "   m_acceptMemberCard = " << m_acceptMemberCard << ::std::endl
             << "   m_acceptServiceProviderPaymentMethod = " << m_acceptServiceProviderPaymentMethod << ::std::endl;
      return stream.str();
   }

private:
   // POI's where no payment is needed shall be considered when this field is set.
   bool m_noPayment;
   // POI's where the payment methods are not known (not available in navdata) shall be considered when this field is set.
   bool m_paymentMethodUnknown;
   // POI's where the payment methods are not fixed (e.g, different payment methods accepted at different time of day)
   // shall be considered when this field is set.
   bool m_variablePaymentMethod;
   bool m_acceptCash;
   bool m_acceptCoins;
   bool m_acceptEcCard;
   bool m_acceptCreditCard;
   bool m_acceptElectronicPurse;
   bool m_acceptElectronicTollCollection;
   bool m_acceptMobilePayment;
   bool m_acceptMemberCard;
   bool m_acceptServiceProviderPaymentMethod;
};

class EvConnectorTypeFilterSettings
{
public:
   EvConnectorTypeFilterSettings()
      : m_acceptStandardHousehold(false)
      , m_acceptIec60309SinglePhaseIndustrial(false)
      , m_acceptIec62196_2_Type2(false)
      , m_acceptIec62196_2_Type3(false)
      , m_acceptSaeJ1772(false)
      , m_acceptCHAdeMO(false)
      , m_accepIec60309PhaseIndustrialRed(false)
      , m_acceptIec62196_2_Type2CCS(false)
      , m_acceptIec62196_2_Type1CCS(false)
      , m_acceptUnspecified(false)
   {}

   EvConnectorTypeFilterSettings(
      bool acceptStandardHousehold,
      bool acceptIec60309SinglePhaseIndustrial,
      bool acceptIec62196_2_Type2,
      bool acceptIec62196_2_Type3,
      bool acceptSaeJ1772,
      bool acceptCHAdeMO,
      bool accepIec60309PhaseIndustrialRed,
      bool acceptIec62196_2_Type2CCS,
      bool acceptIec62196_2_Type1CCS,
      bool acceptUnspecified)

      : m_acceptStandardHousehold(acceptStandardHousehold)
      , m_acceptIec60309SinglePhaseIndustrial(acceptIec60309SinglePhaseIndustrial)
      , m_acceptIec62196_2_Type2(acceptIec62196_2_Type2)
      , m_acceptIec62196_2_Type3(acceptIec62196_2_Type3)
      , m_acceptSaeJ1772(acceptSaeJ1772)
      , m_acceptCHAdeMO(acceptCHAdeMO)
      , m_accepIec60309PhaseIndustrialRed(accepIec60309PhaseIndustrialRed)
      , m_acceptIec62196_2_Type2CCS(acceptIec62196_2_Type2CCS)
      , m_acceptIec62196_2_Type1CCS(acceptIec62196_2_Type1CCS)
      , m_acceptUnspecified(acceptUnspecified)
   {}

   bool operator==(const EvConnectorTypeFilterSettings& rhs) const
   {
      return (m_acceptStandardHousehold == rhs.m_acceptStandardHousehold) &&
             (m_acceptIec60309SinglePhaseIndustrial == rhs.m_acceptIec60309SinglePhaseIndustrial) &&
             (m_acceptIec62196_2_Type2 == rhs.m_acceptIec62196_2_Type2) &&
             (m_acceptIec62196_2_Type3 == rhs.m_acceptIec62196_2_Type3) &&
             (m_acceptSaeJ1772 == rhs.m_acceptSaeJ1772) &&
             (m_acceptCHAdeMO == rhs.m_acceptCHAdeMO) &&
             (m_accepIec60309PhaseIndustrialRed == rhs.m_accepIec60309PhaseIndustrialRed) &&
             (m_acceptIec62196_2_Type2CCS == rhs.m_acceptIec62196_2_Type2CCS) &&
             (m_acceptIec62196_2_Type1CCS == rhs.m_acceptIec62196_2_Type1CCS) &&
             (m_acceptUnspecified == rhs.m_acceptUnspecified);
   }

   bool operator!=(const EvConnectorTypeFilterSettings& rhs) const
   {
      return !(*this == rhs);
   }

   bool getAcceptStandardHousehold() const
   {
      return m_acceptStandardHousehold;
   }

   void setAcceptStandardHousehold(bool acceptStandardHousehold)
   {
      m_acceptStandardHousehold = acceptStandardHousehold;
   }

   bool getAcceptIec60309SinglePhaseIndustrial() const
   {
      return m_acceptIec60309SinglePhaseIndustrial;
   }

   void setAcceptIec60309SinglePhaseIndustrial(bool acceptIec60309SinglePhaseIndustrial)
   {
      m_acceptIec60309SinglePhaseIndustrial = acceptIec60309SinglePhaseIndustrial;
   }

   void setAcceptIec62196_2_Type2(bool acceptIec62196_2_Type2)
   {
      m_acceptIec62196_2_Type2 = acceptIec62196_2_Type2;
   }

   bool getAcceptIec62196_2_Type2() const
   {
      return m_acceptIec62196_2_Type2;
   }

   // @DEPRECATED : Please use getAcceptIec62196_2_Type2()
   bool getAcceptIec62196_2_Type2Outlet() const
   {
      return m_acceptIec62196_2_Type2;
   }

   // @DEPRECATED : Please use setAcceptIec62196_2_Type2(bool)
   void setAcceptIec62196_2_Type2Outlet(bool acceptIec62196_2_Type2Outlet)
   {
      m_acceptIec62196_2_Type2 = acceptIec62196_2_Type2Outlet;
   }

   bool getAcceptIec62196_2_Type3() const
   {
      return m_acceptIec62196_2_Type3;
   }

   void setAcceptIec62196_2_Type3(bool acceptIec62196_2_Type3)
   {
      m_acceptIec62196_2_Type3 = acceptIec62196_2_Type3;
   }

   bool getAcceptSaeJ1772() const
   {
      return m_acceptSaeJ1772;
   }

   void setAcceptSaeJ1772(bool acceptSaeJ1772)
   {
      m_acceptSaeJ1772 = acceptSaeJ1772;
   }

   bool getAcceptCHAdeMO() const
   {
      return m_acceptCHAdeMO;
   }

   void setAcceptCHAdeMO(bool acceptCHAdeMO)
   {
      m_acceptCHAdeMO = acceptCHAdeMO;
   }

   bool getAcceptIec60309PhaseIndustrialRed() const
   {
      return m_accepIec60309PhaseIndustrialRed;
   }

   void setAcceptIec60309PhaseIndustrialRed(bool acceptIec60309PhaseIndustrialRed)
   {
      m_accepIec60309PhaseIndustrialRed = acceptIec60309PhaseIndustrialRed;
   }

   bool getAcceptIec62196_2_Type2CCS() const
   {
      return m_acceptIec62196_2_Type2CCS;
   }

   void setAcceptIec62196_2_Type2CCS(bool acceptIec62196_2_Type2CCS)
   {
      m_acceptIec62196_2_Type2CCS = acceptIec62196_2_Type2CCS;
   }

   bool getAcceptIec62196_2_Type1CCS() const
   {
      return m_acceptIec62196_2_Type1CCS;
   }

   void setAcceptIec62196_2_Type1CCS(bool acceptIec62196_2_Type1CCS)
   {
      m_acceptIec62196_2_Type1CCS = acceptIec62196_2_Type1CCS;
   }

   bool getAcceptUnspecified() const
   {
      return m_acceptUnspecified;
   }

   void setAcceptUnspecified(bool acceptUnspecified)
   {
      m_acceptUnspecified = acceptUnspecified;
   }
   ::std::string toString() const
   {
      ::std::stringstream stream("EvConnectorTypeFilterSettings payload:\n");
      stream << "   m_acceptStandardHousehold = " << m_acceptStandardHousehold << ::std::endl
             << "   m_acceptIec60309SinglePhaseIndustrial = " << m_acceptIec60309SinglePhaseIndustrial << ::std::endl
             << "   m_acceptIec62196_2_Type2 = " << m_acceptIec62196_2_Type2 << ::std::endl
             << "   m_acceptIec62196_2_Type3 = " << m_acceptIec62196_2_Type3 << ::std::endl
             << "   m_acceptSaeJ1772 = " << m_acceptSaeJ1772 << ::std::endl
             << "   m_acceptCHAdeMO = " << m_acceptCHAdeMO << ::std::endl
             << "   m_accepIec60309PhaseIndustrialRed = " << m_accepIec60309PhaseIndustrialRed << ::std::endl
             << "   m_acceptIec62196_2_Type2CCS = " << m_acceptIec62196_2_Type2CCS << ::std::endl
             << "   m_acceptIec62196_2_Type1CCS = " << m_acceptIec62196_2_Type1CCS << ::std::endl
             << "   m_acceptUnspecified = " << m_acceptUnspecified << ::std::endl;
      return stream.str();
   }

private:
   bool m_acceptStandardHousehold;              // "Domestic socket"
   bool m_acceptIec60309SinglePhaseIndustrial;  // "Blue industrial socket"
   bool m_acceptIec62196_2_Type2;               // "Mode 3, type 2 socket/connector" (diff unclear)
   bool m_acceptIec62196_2_Type3;               // "Mode 3, type 3 socket"
   bool m_acceptSaeJ1772;                       // "Mode 3, type 1 SAE J1772-2009"
   bool m_acceptCHAdeMO;                        // "CHAdeMO"
   bool m_accepIec60309PhaseIndustrialRed;      // "Red industrial socket"
   bool m_acceptIec62196_2_Type2CCS;            // "Mode 3, type 2 CCS"
   bool m_acceptIec62196_2_Type1CCS;            // "Mode 3, type 1 CCS"
   bool m_acceptUnspecified;                    // "Unknown"
};

class ParkingSettings
{
public:
   ParkingSettings()
      : m_giveParkAndRideParkingOnly(false)
      , m_giveParkingForReducedMobilityOnly(false)
      , m_useHeightRestriction(false)
   {}

   ParkingSettings(
      bool giveParkAndRideParkingOnly,
      bool giveParkingForReducedMobilityOnly,
      bool useHeightRestriction)

      : m_giveParkAndRideParkingOnly(giveParkAndRideParkingOnly)
      , m_giveParkingForReducedMobilityOnly(giveParkingForReducedMobilityOnly)
      , m_useHeightRestriction(useHeightRestriction)
   {}

   bool operator==(const ParkingSettings& rhs) const
   {
      return (m_giveParkAndRideParkingOnly == rhs.m_giveParkAndRideParkingOnly) &&
             (m_giveParkingForReducedMobilityOnly == rhs.m_giveParkingForReducedMobilityOnly) &&
             (m_useHeightRestriction == rhs.m_useHeightRestriction);
   }
   bool operator!=(const ParkingSettings& rhs) const
   {
      return !(*this == rhs);
   }

   bool doGiveParkAndRideParking() const
   {
      return m_giveParkAndRideParkingOnly;
   }

   void setGiveParkAndRideParking(bool useParkAndRideParking)
   {
      m_giveParkAndRideParkingOnly = useParkAndRideParking;
   }

   bool doGiveParkingForReducedMobility() const
   {
      return m_giveParkingForReducedMobilityOnly;
   }

   void setGiveParkingForReducedMobility(bool useParkingForReducedMobility)
   {
      m_giveParkingForReducedMobilityOnly = useParkingForReducedMobility;
   }

   bool getUseHeightRestriction() const
   {
      return m_useHeightRestriction;
   }

   void setUseHeightRestriction(bool useHeightRestriction)
   {
      m_useHeightRestriction = useHeightRestriction;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("ParkingSettings payload:\n");
      stream << "   m_giveParkAndRideParkingOnly = " << m_giveParkAndRideParkingOnly << ::std::endl
             << "   m_giveParkingForReducedMobilityOnly = " << m_giveParkingForReducedMobilityOnly << ::std::endl
             << "   m_useHeightRestriction = " << m_useHeightRestriction << ::std::endl;
      return stream.str();
   }

private:
   bool m_giveParkAndRideParkingOnly;
   bool m_giveParkingForReducedMobilityOnly;
   bool m_useHeightRestriction;
};

class EvChargingSpeedFilterSettings
{
public:
   EvChargingSpeedFilterSettings()
      : m_acceptNormalCharge(false)
      , m_acceptAcceleratedCharge(false)
      , m_acceptFastCharge(false)
      , m_acceptQuickCharge(false)
      , m_acceptQuickDrop(false)
      , m_acceptUnknown(false)
   {}

   bool operator==(const EvChargingSpeedFilterSettings& rhs) const
   {
      return (m_acceptNormalCharge == rhs.m_acceptNormalCharge) &&
             (m_acceptAcceleratedCharge == rhs.m_acceptAcceleratedCharge) &&
             (m_acceptFastCharge == rhs.m_acceptFastCharge) &&
             (m_acceptQuickCharge == rhs.m_acceptQuickCharge) &&
             (m_acceptQuickDrop == rhs.m_acceptQuickDrop) &&
             (m_acceptUnknown == rhs.m_acceptUnknown);
   }

   bool operator!=(const EvChargingSpeedFilterSettings& rhs) const
   {
      return !(*this == rhs);
   }

   bool getAcceptNormalCharge() const
   {
      return m_acceptNormalCharge;
   }

   void setAcceptNormalCharge(bool acceptNormalCharge)
   {
      m_acceptNormalCharge = acceptNormalCharge;
   }

   bool getAcceptAcceleratedCharge() const
   {
      return m_acceptAcceleratedCharge;
   }

   void setAcceptAcceleratedCharge(bool acceptAcceleratedCharge)
   {
      m_acceptAcceleratedCharge = acceptAcceleratedCharge;
   }

   bool getAcceptFastCharge() const
   {
      return m_acceptFastCharge;
   }

   void setAcceptFastCharge(bool acceptFastCharge)
   {
      m_acceptFastCharge = acceptFastCharge;
   }

   bool getAcceptQuickCharge() const
   {
      return m_acceptQuickCharge;
   }

   void setAcceptQuickCharge(bool acceptQuickCharge)
   {
      m_acceptQuickCharge = acceptQuickCharge;
   }

   void setAcceptQuickDrop(bool acceptQuickDrop)
   {
      m_acceptQuickDrop = acceptQuickDrop;
   }

   bool getAcceptQuickDrop() const
   {
      return m_acceptQuickDrop;
   }

   bool getAcceptUnknown() const
   {
      return m_acceptUnknown;
   }

   void setAcceptUnknown(bool acceptUnknown)
   {
      m_acceptUnknown = acceptUnknown;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("EvChargingSpeedFilterSettings payload:\n");
      stream << "   m_acceptNormalCharge = " << m_acceptNormalCharge << ::std::endl
             << "   m_acceptAcceleratedCharge = " << m_acceptAcceleratedCharge << ::std::endl
             << "   m_acceptFastCharge = " << m_acceptFastCharge << ::std::endl
             << "   m_acceptQuickCharge = " << m_acceptQuickCharge << ::std::endl
             << "   m_acceptQuickDrop = " << m_acceptQuickDrop << ::std::endl
             << "   m_acceptUnknown = " << m_acceptUnknown << ::std::endl;
      return stream.str();
   }

private:
   bool m_acceptNormalCharge;
   bool m_acceptAcceleratedCharge;
   bool m_acceptFastCharge;
   bool m_acceptQuickCharge;
   bool m_acceptQuickDrop;
   bool m_acceptUnknown;
};

} // settings
} // navmiddleware

#endif  // PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_SETTINGS_LOCATIONSETTINGS_H_
