/**
* @copyright (c) 2015-2020 Robert Bosch Car Multimedia GmbH
* @addtogroup NavMiddleware
* @{
*/

#ifndef PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_LOCATIONINFOS_H_
#define PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_LOCATIONINFOS_H_

#include <stdint.h>
#include <map>
#include <set>
#include <sstream>
#include <string>
#include <vector>
#include "InfoTypes.h"

namespace navmiddleware
{

enum ApplicationId
{
   APPLICATIONID__NAVIGATION,
   APPLICATIONID__SXM,
   APPLICATIONID__TOMTOM_WEATHER
};
inline ::std::string toString(ApplicationId appId)
{
   switch (appId)
   {
   case APPLICATIONID__NAVIGATION:
      return "APPLICATIONID__NAVIGATION";
   case APPLICATIONID__SXM:
      return "APPLICATIONID__SXM";
   case APPLICATIONID__TOMTOM_WEATHER:
      return "APPLICATIONID__TOMTOM_WEATHER";
   default:
      ::std::stringstream stream;
      stream << "APPLICATIONID__<" << static_cast<unsigned int>(appId) << ">";
      return stream.str();
   }
}

enum AddressInputInstance
{
   ADDRESS_INPUT_INSTANCE__REGULAR        = 0,
   ADDRESS_INPUT_INSTANCE__SEARCH_SCOPE   = 1
};
inline ::std::string toString(AddressInputInstance addressInputInstance)
{
   switch (addressInputInstance)
   {
   case ADDRESS_INPUT_INSTANCE__REGULAR:
      return "ADDRESS_INPUT_INSTANCE__REGULAR";
   case ADDRESS_INPUT_INSTANCE__SEARCH_SCOPE:
      return "ADDRESS_INPUT_INSTANCE__SEARCH_SCOPE";
   default:
      ::std::stringstream stream;
      stream << "ADDRESS_INPUT_INSTANCE__<" << static_cast<unsigned int>(addressInputInstance) << ">";
      return stream.str();
   }
}

struct LocationContext
{
   LocationContext():
      m_hmiContext(HMICONTEXT__GUI),
      m_appId(APPLICATIONID__NAVIGATION),
      m_inputInstanceNumber(0)
   {}

   enum HmiContext
   {
      HMICONTEXT__GUI,
      HMICONTEXT__SDS,
      HMICONTEXT__MAP
   };
   static ::std::string toString(HmiContext hmiContext)
   {
      switch (hmiContext)
      {
      case HMICONTEXT__GUI:
         return "HMICONTEXT__GUI";
      case HMICONTEXT__SDS:
         return "HMICONTEXT__SDS";
      case HMICONTEXT__MAP:
         return "HMICONTEXT__MAP";
      default:
         ::std::stringstream stream;
         stream << "HMICONTEXT__<" << static_cast<unsigned int>(hmiContext) << ">";
         return stream.str();
      }
   }

   bool operator == (const LocationContext& other) const
   {
      return (this->m_hmiContext == other.m_hmiContext)
             && (this->m_appId == other.m_appId)
             && (this->m_inputInstanceNumber == other.m_inputInstanceNumber);
   }

   bool operator != (const LocationContext& other) const
   {
      return (!(*this == other));
   }


   LocationContext(HmiContext hmiContext, ApplicationId appId, unsigned int inputInstanceNumber = 0)
      : m_hmiContext(hmiContext),
        m_appId(appId),
        m_inputInstanceNumber(inputInstanceNumber)
   {}

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "LocationContext payload:" << ::std::endl;
      stream << "   m_hmiContext          = " << toString(m_hmiContext) << ::std::endl;
      stream << "   m_appId               = " << ::navmiddleware::toString(m_appId) << ::std::endl;
      stream << "   m_inputInstanceNumber = " << m_inputInstanceNumber << ::std::endl;

      return stream.str();
   }

   HmiContext     m_hmiContext;
   ApplicationId  m_appId;
   unsigned int   m_inputInstanceNumber;
};

enum LocationSearchSource
{
   /** Search is done "offline" in onboard dataset (plus TPEG data for limited set of POI categories) only */
   LOCATIONSEARCHSOURCE__OFFLINE,
   /** Search is done at the "default" online search source only
       (e.g. TomTom Online Search in case of Bosch Navigation for Nissan) */
   LOCATIONSEARCHSOURCE__ONLINE,
   /** Search is done at the default online search source preferably, onboard search is used as fallback */
   LOCATIONSEARCHSOURCE__ON_AND_OFFLINE,
   /** Search is done at Google Places only */
   LOCATIONSEARCHSOURCE__GOOGLE_PLACES,
   //<-- INF4CV
   /** Search is done in POI data imported from USB stick */
   LOCATIONSEARCHSOURCE__USB_POI
   //->
};

inline ::std::string toString(LocationSearchSource searchSource)
{
   switch (searchSource)
   {
   case LOCATIONSEARCHSOURCE__OFFLINE:
      return "LOCATIONSEARCHSOURCE__OFFLINE";
   case LOCATIONSEARCHSOURCE__ONLINE:
      return "LOCATIONSEARCHSOURCE__ONLINE";
   case LOCATIONSEARCHSOURCE__ON_AND_OFFLINE:
      return "LOCATIONSEARCHSOURCE__ON_AND_OFFLINE";
   case LOCATIONSEARCHSOURCE__GOOGLE_PLACES:
      return "LOCATIONSEARCHSOURCE__GOOGLE_PLACES";
   //<-- INF4CV
   case LOCATIONSEARCHSOURCE__USB_POI:
      return "LOCATIONSEARCHSOURCE__USB_POI";
   //->
   default:
      ::std::stringstream stream;
      stream << "LOCATIONSEARCHSOURCE__<" << static_cast<unsigned int>(searchSource) << ">";
      return stream.str();
   }
}

struct LocationSearchScope
{
   enum Type
   {
      TYPE__AROUND_CURRENT_VEHICLE_POSITION,
      TYPE__AROUND_CURRENT_LOCATION_POSITION,
      TYPE__AROUND_WAYPOINT_POSITION,
      TYPE__ALONG_ROUTE,
      TYPE__ALONG_CURRENT_HIGHWAY,
      TYPE__ON_WHOLE_DATASET,
      TYPE__IN_CURRENT_COUNTRY,
      TYPE__IN_CURRENT_STATE,
      TYPE__IN_CURRENT_LOCATION_AREA,
      TYPE__CONNECTED_TO_CURRENT_LOCATION_POI
   };
   static ::std::string toString(Type type)
   {
      switch (type)
      {
      case TYPE__AROUND_CURRENT_VEHICLE_POSITION:
         return "TYPE__AROUND_CURRENT_VEHICLE_POSITION";
      case TYPE__AROUND_WAYPOINT_POSITION:
         return "TYPE__AROUND_WAYPOINT_POSITION";
      case TYPE__AROUND_CURRENT_LOCATION_POSITION:
         return "TYPE__AROUND_CURRENT_LOCATION_POSITION";
      case TYPE__ALONG_ROUTE:
         return "TYPE__ALONG_ROUTE";
      case TYPE__ALONG_CURRENT_HIGHWAY:
         return "TYPE__ALONG_CURRENT_HIGHWAY";
      case TYPE__ON_WHOLE_DATASET:
         return "TYPE__ON_WHOLE_DATASET";
      case TYPE__IN_CURRENT_COUNTRY:
         return "TYPE__IN_CURRENT_COUNTRY";
      case TYPE__IN_CURRENT_STATE:
         return "TYPE__IN_CURRENT_STATE";
      case TYPE__IN_CURRENT_LOCATION_AREA:
         return "TYPE__IN_CURRENT_LOCATION_AREA";
      case TYPE__CONNECTED_TO_CURRENT_LOCATION_POI:
         return "TYPE__CONNECTED_TO_CURRENT_LOCATION_POI";
      default:
         ::std::stringstream stream;
         stream << "TYPE__<" << static_cast<unsigned int>(type) << ">";
         return stream.str();
      }
   }

   enum SearchRadius
   {
      /** Search is done in a "default" area depending on the Type:
       *    - In case of TYPE__AROUND_... this is a fixed radius.
       *    - In case of TYPE__ALONG_ROUTE this is a small "corridor" along the route.
       *    - In case of TYPE__ALONG_CURRENT_HIGHWAY this is a small "corridor" along the possibly present highway
       *    - In case of TYPE__ON_WHOLE_DATASET there is (in fact) no limit.
       *    - In case of TYPE__IN_CURRENT_... the limit is defined by the area itself.
       *    - In case of TYPE__CONNECTED_TO_CURRENT_LOCATION_POI there is no explicit limit.
       */
      SEARCH_RADIUS__DEFAULT_AREA,

      /** Search is limited to the reachable area ("reachable" with state of charge of battery).
       *  This is allowed only for types TYPE__AROUND_CURRENT_VEHICLE_POSITION, TYPE__ALONG_ROUTE,
       *  and TYPE__ALONG_CURRENT_HIGHWAY.
       *  This is allowed for the type TYPE__AROUND_WAYPOINT_POSITION, only if it is next destination or way point.
       *  and for this type TYPE__AROUND_WAYPOINT_POSITION, Choice of waypoint position index is not possible,
       *  Next immediate waypoint/destination is considered
       *  If set for other types it would be ignored.
       */
      SEARCH_RADIUS__REACHABLE_AREA
   };
   static ::std::string toString(SearchRadius searchRadius)
   {
      switch (searchRadius)
      {
      case SEARCH_RADIUS__DEFAULT_AREA:
         return "SEARCH_RADIUS__DEFAULT_AREA";
      case SEARCH_RADIUS__REACHABLE_AREA:
         return "SEARCH_RADIUS__REACHABLE_AREA";
      default:
         ::std::stringstream stream;
         stream << "SEARCH_RADIUS__<" << static_cast<unsigned int>(searchRadius) << ">";
         return stream.str();
      }
   }

   explicit LocationSearchScope(
      Type type = TYPE__AROUND_CURRENT_VEHICLE_POSITION,
      unsigned int waypointIndex = 0,
      SearchRadius searchRadius = SEARCH_RADIUS__DEFAULT_AREA)
      : m_type(type)
      , m_waypointIndex(waypointIndex)
      , m_searchRadius(searchRadius)
   {}
   LocationSearchScope(Type type, SearchRadius searchRadius)
      : m_type(type)
      , m_waypointIndex(0)
      , m_searchRadius(searchRadius)
   {}
   bool operator==(const LocationSearchScope& other) const
   {
      return (this->m_type == other.m_type)
             && (this->m_waypointIndex == other.m_waypointIndex)
             && (this->m_searchRadius == other.m_searchRadius);
   }
   bool operator!=(const LocationSearchScope& other) const
   {
      return !(*this == other);
   }
   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "LocationSearchScope(" << "m_type = " << toString(m_type) << ", m_waypointIndex = " << m_waypointIndex
             << ", m_searchRadius = " << toString(m_searchRadius) << ")";
      return stream.str();
   }


   Type          m_type;
   unsigned int  m_waypointIndex;
   SearchRadius  m_searchRadius;
};


class OnlineFreeTextSearchAvailabilityInfo
{
public:
   OnlineFreeTextSearchAvailabilityInfo()
      : m_defaultOnlineSearchAvailability(ONLINE_SERVICE_AVAILABILITY__UNKNOWN)
      , m_tomTomOnlineSearchAvailability(ONLINE_SERVICE_AVAILABILITY__UNKNOWN)
      , m_googlePlacesSearchAvailability(ONLINE_SERVICE_AVAILABILITY__UNKNOWN)
   {
   }

   OnlineServiceAvailability getDefaultOnlineSearchAvailability() const
   {
      return m_defaultOnlineSearchAvailability;
   }
   void setDefaultOnlineSearchAvailability(OnlineServiceAvailability availability)
   {
      m_defaultOnlineSearchAvailability = availability;
   }

   OnlineServiceAvailability getTomTomOnlineSearchAvailability() const
   {
      return m_tomTomOnlineSearchAvailability;
   }
   void setTomTomOnlineSearchAvailability(OnlineServiceAvailability availability)
   {
      m_tomTomOnlineSearchAvailability = availability;
   }

   OnlineServiceAvailability getGooglePlacesSearchAvailability() const
   {
      return m_googlePlacesSearchAvailability;
   }
   void setGooglePlacesSearchAvailability(OnlineServiceAvailability availability)
   {
      m_googlePlacesSearchAvailability = availability;
   }

   bool operator==(const OnlineFreeTextSearchAvailabilityInfo& other) const
   {
      return (m_defaultOnlineSearchAvailability == other.m_defaultOnlineSearchAvailability)
             && (m_tomTomOnlineSearchAvailability == other.m_tomTomOnlineSearchAvailability)
             && (m_googlePlacesSearchAvailability == other.m_googlePlacesSearchAvailability);
   }
   bool operator!=(const OnlineFreeTextSearchAvailabilityInfo& other) const
   {
      return !(*this == other);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "OnlineFtsAvailability (default = " << ::navmiddleware::toString(m_defaultOnlineSearchAvailability)
             << ", TomTom = " << ::navmiddleware::toString(m_tomTomOnlineSearchAvailability)
             << ", Google = " << ::navmiddleware::toString(m_googlePlacesSearchAvailability)
             << ")";
      return stream.str();
   }

private:
   // availability if client (HMI) is starting FTS with "default" online source (using ONLINE or ON_AND_OFFLINE)
   OnlineServiceAvailability m_defaultOnlineSearchAvailability;
   // availability if client (HMI) is starting FTS explicitly with TomTom online source (currently not supported)
   OnlineServiceAvailability m_tomTomOnlineSearchAvailability;
   // availability if client (HMI) is starting FTS explicitly with Google Places online source
   OnlineServiceAvailability m_googlePlacesSearchAvailability;
};

class OnlinePoiAvailabilityInfo
{
public:
   OnlinePoiAvailabilityInfo()
      : m_tomTomOnlineSearchAvailability(ONLINE_SERVICE_AVAILABILITY__UNKNOWN)
   {
   }

   const ::std::map<PredefinedPoiCategory, OnlineServiceAvailability>& getOnlinePoiAvailability() const
   {
      return m_poiOnlineAvailabilityMap;
   }
   void setOnlinePoiAvailability(const ::std::map<PredefinedPoiCategory, OnlineServiceAvailability>& poiOnlineSearchAvailabilityMap)
   {
      m_poiOnlineAvailabilityMap = poiOnlineSearchAvailabilityMap;
   }

   OnlineServiceAvailability getOnlinePoiAvailability(PredefinedPoiCategory poiCategory) const
   {
      ::std::map<PredefinedPoiCategory, OnlineServiceAvailability>::const_iterator itr = m_poiOnlineAvailabilityMap.find(poiCategory);
      if(itr != m_poiOnlineAvailabilityMap.end())
      {
         return itr->second;
      }
      else
      {
         return m_tomTomOnlineSearchAvailability;
      }
   }

   OnlineServiceAvailability getTomTomOnlineSearchAvailability() const
   {
      return m_tomTomOnlineSearchAvailability;
   }
   void setTomTomOnlineSearchAvailability(OnlineServiceAvailability availability)
   {
      m_tomTomOnlineSearchAvailability = availability;
   }

   bool operator==(const OnlinePoiAvailabilityInfo& other) const
   {
      return ((m_poiOnlineAvailabilityMap == other.m_poiOnlineAvailabilityMap) &&
              (m_tomTomOnlineSearchAvailability == other.m_tomTomOnlineSearchAvailability));
   }
   bool operator!=(const OnlinePoiAvailabilityInfo& other) const
   {
      return !(*this == other);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "Online Poi AvailabilityInfo Payload:" << ::std::endl
             << "m_tomTomOnlineSearchAvailability = " << ::navmiddleware::toString(m_tomTomOnlineSearchAvailability) << ::std::endl;

      for (::std::map<PredefinedPoiCategory, OnlineServiceAvailability>::const_iterator it = m_poiOnlineAvailabilityMap.begin();
            it != m_poiOnlineAvailabilityMap.end(); ++it)
      {
         stream << ::navmiddleware::toString(it->first) << ":" <<::navmiddleware::toString( it->second) << std::endl;
      }
      return stream.str();
   }

private:
   ::std::map<PredefinedPoiCategory, OnlineServiceAvailability> m_poiOnlineAvailabilityMap;
   OnlineServiceAvailability                                    m_tomTomOnlineSearchAvailability;
};

enum FreeTextSearchType
{
   FREETEXTSEARCHTYPE__COMBINED,
   FREETEXTSEARCHTYPE__ADDRESS,
   FREETEXTSEARCHTYPE__POI,
   FREETEXTSEARCHTYPE__POICATEGORY // [JP]
};
inline ::std::string toString(FreeTextSearchType searchType)
{
   switch (searchType)
   {
   case FREETEXTSEARCHTYPE__COMBINED:
      return "FREETEXTSEARCHTYPE__COMBINED";
   case FREETEXTSEARCHTYPE__ADDRESS:
      return "FREETEXTSEARCHTYPE__ADDRESS";
   case FREETEXTSEARCHTYPE__POI:
      return "FREETEXTSEARCHTYPE__POI";
   case FREETEXTSEARCHTYPE__POICATEGORY:
      return "FREETEXTSEARCHTYPE__POICATEGORY";
   default:
      ::std::stringstream stream;
      stream << "FREETEXTSEARCHTYPE__<" << static_cast<unsigned int>(searchType) << ">";
      return stream.str();
   }
}


enum LocationSortOrder
{
   LOCATIONSORTORDER__UNSORTED,
   LOCATIONSORTORDER__ALPHABETICALLY,
   LOCATIONSORTORDER__BY_AIR_DISTANCE_NEAREST_FIRST,
   LOCATIONSORTORDER__BY_AIR_DISTANCE_FARTHEST_FIRST,
   LOCATIONSORTORDER__BY_AVAILABILITY,
   LOCATIONSORTORDER__BY_PRICE,
   LOCATIONSORTORDER__BY_RELEVANCE
};
inline ::std::string toString(LocationSortOrder sortOrder)
{
   switch (sortOrder)
   {
   case LOCATIONSORTORDER__UNSORTED:
      return "LOCATIONSORTORDER__UNSORTED";
   case LOCATIONSORTORDER__ALPHABETICALLY:
      return "LOCATIONSORTORDER__ALPHABETICALLY";
   case LOCATIONSORTORDER__BY_AIR_DISTANCE_NEAREST_FIRST:
      return "LOCATIONSORTORDER__BY_AIR_DISTANCE_NEAREST_FIRST";
   case LOCATIONSORTORDER__BY_AIR_DISTANCE_FARTHEST_FIRST:
      return "LOCATIONSORTORDER__BY_AIR_DISTANCE_FARTHEST_FIRST";
   case LOCATIONSORTORDER__BY_AVAILABILITY:
      return "LOCATIONSORTORDER__BY_AVAILABILITY";
   case LOCATIONSORTORDER__BY_PRICE:
      return "LOCATIONSORTORDER__BY_PRICE";
   case LOCATIONSORTORDER__BY_RELEVANCE:
      return "LOCATIONSORTORDER__BY_RELEVANCE";
   default:
      ::std::stringstream stream;
      stream << "LOCATIONSORTORDER__<" << static_cast<unsigned int>(sortOrder) << ">";
      return stream.str();
   }
}

enum AutoCompletionMode
{
   AUTO_COMPLETION_MODE__RESULTS_ONLY,
   AUTO_COMPLETION_MODE__AUTO_COMPLETION_ONLY,
   AUTO_COMPLETION_MODE__AUTO_COMPLETION_PLUS_RESULTS
};
inline ::std::string toString(AutoCompletionMode mode)
{
   switch (mode)
   {
   case AUTO_COMPLETION_MODE__RESULTS_ONLY:
      return "AUTO_COMPLETION_MODE__RESULTS_ONLY";
   case AUTO_COMPLETION_MODE__AUTO_COMPLETION_ONLY:
      return "AUTO_COMPLETION_MODE__AUTO_COMPLETION_ONLY";
   case AUTO_COMPLETION_MODE__AUTO_COMPLETION_PLUS_RESULTS:
      return "AUTO_COMPLETION_MODE__AUTO_COMPLETION_PLUS_RESULTS";
   default:
      ::std::stringstream stream;
      stream << "AUTO_COMPLETION_MODE__<" << static_cast<unsigned int>(mode) << ">";
      return stream.str();
   }
}

enum PoiCategoryFilter
{
   POI_CATEGORY_FILTER__NO_FILTER,
   POI_CATEGORY_FILTER__CHARGING_STATIONS_QUICK_CHARGING_SPEED,
   POI_CATEGORY_FILTER__CHARGING_STATIONS_ANY_CHARGING_SPEED
};

inline ::std::string toString(PoiCategoryFilter poiCategoryFilter)
{
   switch (poiCategoryFilter)
   {
   case POI_CATEGORY_FILTER__NO_FILTER:
      return "POI_CATEGORY_FILTER__NO_FILTER";
   case POI_CATEGORY_FILTER__CHARGING_STATIONS_QUICK_CHARGING_SPEED:
      return "POI_CATEGORY_FILTER__CHARGING_STATIONS_QUICK_CHARGING_SPEED";
   case POI_CATEGORY_FILTER__CHARGING_STATIONS_ANY_CHARGING_SPEED:
      return "POI_CATEGORY_FILTER__CHARGING_STATIONS_ANY_CHARGING_SPEED";
   default:
      ::std::stringstream stream;
      stream << "POI_CATEGORY_FILTER__<" << static_cast<unsigned int>(poiCategoryFilter) << ">";
      return stream.str();
   }
}

typedef ::std::map< ::std::string, unsigned int> T_StartingLetterIndexMap;

class LocationInfos
{
public:
   /**
    * Defines the Type of a LocationInfo entry
    */
   enum EntryType
   {
      /** Entry is a coordinate location (i.e. no or only limited address data available) */
      ENTRY_TYPE__COORDINATE_LOCATION,
      /** Entry is an address element (e.g. a city or street name) */
      ENTRY_TYPE__ADDRESS_ELEMENT,
      /** Entry is a POI location (e.g. some single fuel station) */
      ENTRY_TYPE__POI_LOCATION,
      /** Entry is an address category (e.g. city, street) */
      ENTRY_TYPE__ADDRESS_CATEGORY,
      /** Entry is a POI category (e.g. fuel stations) */
      ENTRY_TYPE__POI_CATEGORY,
      /** Entry is representing a whole route ("tour") */
      ENTRY_TYPE__ROUTE,
      /** Entry is representing a query (i.e. search term) auto-completion, e.g. provided by Google Places search */
      ENTRY_TYPE__QUERY_AUTO_COMPLETION
   };
   static ::std::string toString(EntryType entryType)
   {
      switch (entryType)
      {
      case ENTRY_TYPE__COORDINATE_LOCATION:
         return "ENTRY_TYPE__COORDINATE_LOCATION";
      case ENTRY_TYPE__ADDRESS_ELEMENT:
         return "ENTRY_TYPE__ADDRESS_ELEMENT";
      case ENTRY_TYPE__POI_LOCATION:
         return "ENTRY_TYPE__POI_LOCATION";
      case ENTRY_TYPE__ADDRESS_CATEGORY:
         return "ENTRY_TYPE__ADDRESS_CATEGORY";
      case ENTRY_TYPE__POI_CATEGORY:
         return "ENTRY_TYPE__POI_CATEGORY";
      case ENTRY_TYPE__ROUTE:
         return "ENTRY_TYPE__ROUTE";
      case ENTRY_TYPE__QUERY_AUTO_COMPLETION:
         return "ENTRY_TYPE__QUERY_AUTO_COMPLETION";
      default:
         ::std::stringstream stream;
         stream << "ENTRY_TYPE__<" << static_cast<unsigned int>(entryType) << ">";
         return stream.str();
      }
   }

   struct LocationInfo
   {
      LocationInfo()
         : m_index(0)
         , m_entryType(ENTRY_TYPE__COORDINATE_LOCATION)
         , m_locationSource(LOCATION_SOURCE__LOCATION_INPUT_ONBOARD)
         , m_isElementScope(false)
         , m_sourceId(0)
         , m_imageId(0)
         , m_imageBlob(0)
         , m_imageBlobLength(0)
         , m_entrySelectionStatus(ENTRY_SELECTION_STATUS__UNKNOWN)
      {}
      LocationInfo(
         unsigned int index,
         EntryType entryType,
         LocationSource locationSource,
         const ::std::string& elementName,
         const ValidValue<SdsGlobalNamedObjectIdInfo>& sdsGlobalNamedObjectIdInfo,
         const ::std::string& elementNameAsInNavigationData,
         const ::std::string& description,
         bool isElementScope,
         const ::std::string& distance,
         ValidValue<DirectionDescription> direction,
         ValidValue<GeoCoordinateDegree> coordinate,
         const LocationAttributeInfos& attributes,
         const Image& image,
         uint8_t sourceId,
         uint64_t imageId,
         const unsigned char* imageBlob,
         unsigned int imageBlobLength,
         EntrySelectionStatus entrySelectionStatus
      )
         : m_index(index)
         , m_entryType(entryType)
         , m_locationSource(locationSource)
         , m_elementName(elementName)
         , m_sdsGlobalNamedObjectIdInfo(sdsGlobalNamedObjectIdInfo)
         , m_elementNameAsInNavigationData(elementNameAsInNavigationData)
         , m_description(description)
         , m_isElementScope(isElementScope)
         , m_distance(distance)
         , m_direction(direction)
         , m_coordinate(coordinate)
         , m_attributes(attributes)
         , m_image(image)
         , m_sourceId(sourceId)
         , m_imageId(imageId)
         , m_imageBlob(imageBlob)
         , m_imageBlobLength(imageBlobLength)
         , m_entrySelectionStatus(entrySelectionStatus)
      {}
      ~LocationInfo()
      {}

      ::std::string toString() const
      {
         ::std::stringstream stream;
         stream << "LocationInfo payload:" << ::std::endl
                << "   m_index = " << m_index << ::std::endl
                << "   m_entryType = " << LocationInfos::toString(m_entryType) << ::std::endl
                << "   m_locationSource = " << ::navmiddleware::toString(m_locationSource) << ::std::endl
                << "   m_elementName = \"" << m_elementName << "\"" << ::std::endl
                << "   m_elementNameAsInNavigationData = \"" << m_elementNameAsInNavigationData << "\"" << ::std::endl
                << "   m_description = \"" << m_description << "\"" << ::std::endl
                << "   m_isElementScope = " << (m_isElementScope? "true" : "false") << ::std::endl
                << "   m_distance = \"" << m_distance << "\"" << ::std::endl
                << "   m_direction = " << (m_direction.isValid()? ::navmiddleware::toString(m_direction.getValue()) : "invalid") << ::std::endl
                << "   m_coordinate = " << (m_coordinate.isValid()? m_coordinate.getValue().toString() : "invalid") << ::std::endl
                << "   m_attributes = " << m_attributes.toString() << ::std::endl
                << "   m_image = " << m_image.toString() << ::std::endl
                << "   m_sourceId = " << static_cast<unsigned int>(m_sourceId) << ", m_imageId = " << m_imageId << ::std::endl
                << "   m_imageBlob = 0x" << static_cast<const void*>(m_imageBlob) << ::std::endl
                << "   m_imageBlobLength = " << m_imageBlobLength << ::std::endl
                << "   m_entrySelectionStatus = "<< ::navmiddleware::toString(m_entrySelectionStatus)<< ::std::endl;
         return stream.str();
      }

      unsigned int           m_index;
      EntryType              m_entryType;
      LocationSource         m_locationSource;
      ::std::string          m_elementName;
      // SDSGlobalId to be able to retrieve Phonemes in SDS context
      ValidValue<SdsGlobalNamedObjectIdInfo> m_sdsGlobalNamedObjectIdInfo;
      //If Display Name differs from Actual Name in Navigation Data this variable is filled
      //Used in HapticHandover usecase for SDS
      ::std::string          m_elementNameAsInNavigationData;
      ::std::string          m_description;
      bool                   m_isElementScope;
      ::std::string          m_distance;
      ValidValue<DirectionDescription> m_direction;
      ValidValue<GeoCoordinateDegree>  m_coordinate;
      LocationAttributeInfos m_attributes;
      Image                  m_image;
      // The following members will be @DEPRECATED and m_image should be used instead.
      uint8_t                m_sourceId; // means: image source
      uint64_t               m_imageId;
      const unsigned char*   m_imageBlob;
      unsigned int           m_imageBlobLength;
      EntrySelectionStatus   m_entrySelectionStatus;
   };

   enum ErrorCode
   {
      ERROR_CODE__NO_ERROR,
      ERROR_CODE__ONLINE_REQUEST_FAILED,
      ERROR_CODE__INTERNAL_ERROR,
      ERROR_CODE__NO_ONLINE_SUBSCRIPTION
   };
   static ::std::string toString(ErrorCode error)
   {
      switch (error)
      {
      case ERROR_CODE__NO_ERROR:
         return "ERROR_CODE__NO_ERROR";
      case ERROR_CODE__ONLINE_REQUEST_FAILED:
         return "ERROR_CODE__ONLINE_REQUEST_FAILED";
      case ERROR_CODE__INTERNAL_ERROR:
         return "ERROR_CODE__INTERNAL_ERROR";
      case ERROR_CODE__NO_ONLINE_SUBSCRIPTION:
         return "ERROR_CODE__NO_ONLINE_SUBSCRIPTION";
      default:
         ::std::stringstream stream;
         stream << "ERROR__<" << static_cast<unsigned int>(error) << ">";
         return stream.str();
      }
   }

   enum PreferredResultLocationType
   {
      PREFERRED_RESULT_LOCATION_TYPE__POI,
      PREFERRED_RESULT_LOCATION_TYPE__ADDRESS
   };
   static ::std::string toString(PreferredResultLocationType preferredResultLocationType)
   {
      switch(preferredResultLocationType)
      {
      case PREFERRED_RESULT_LOCATION_TYPE__POI:
         return "PREFERRED_RESULT_LOCATION_TYPE__POI";
      case PREFERRED_RESULT_LOCATION_TYPE__ADDRESS:
         return "PREFERRED_RESULT_LOCATION_TYPE__ADDRESS";
      default :
         ::std::stringstream stream;
         stream << "PREFERRED_RESULT_LOCATION_TYPE__<" << static_cast<unsigned int>(preferredResultLocationType) << ">";
         return stream.str();
      }
   }

   enum UpdateReason
   {
      UPDATE_REASON__START_INPUT,
      UPDATE_REASON__RESUME_INSTANCE,
      UPDATE_REASON__SET_SORT_ORDER,
      UPDATE_REASON__SET_SCOPE,
      UPDATE_REASON__SET_SOURCE,
      UPDATE_REASON__REQUEST_ENTRIES,
      UPDATE_REASON__SELECTION,
      UPDATE_REASON__UNDO_SELECTION
   };
   static ::std::string toString(UpdateReason updateReason)
   {
      switch (updateReason)
      {
      case UPDATE_REASON__START_INPUT:
         return "UPDATE_REASON__START_INPUT";
      case UPDATE_REASON__RESUME_INSTANCE:
         return "UPDATE_REASON__RESUME_INSTANCE";
      case UPDATE_REASON__SET_SORT_ORDER:
         return "UPDATE_REASON__SET_SORT_ORDER";
      case UPDATE_REASON__SET_SCOPE:
         return "UPDATE_REASON__SET_SCOPE";
      case UPDATE_REASON__SET_SOURCE:
         return "UPDATE_REASON__SET_SOURCE";
      case UPDATE_REASON__REQUEST_ENTRIES:
         return "UPDATE_REASON__REQUEST_ENTRIES";
      case UPDATE_REASON__SELECTION:
         return "UPDATE_REASON__SELECTION";
      case UPDATE_REASON__UNDO_SELECTION:
         return "UPDATE_REASON__UNDO_SELECTION";
      default:
         ::std::stringstream stream;
         stream << "UPDATE_REASON__<" << static_cast<unsigned int>(updateReason) << ">";
         return stream.str();
      }
   }

   LocationInfos()
      : m_countOfAllLocations(0)
      , m_currentSortOrder(LOCATIONSORTORDER__UNSORTED)
      , m_currentSearchSource(LOCATIONSEARCHSOURCE__OFFLINE)
      , m_currentAutoCompletionMode(AUTO_COMPLETION_MODE__RESULTS_ONLY)
      , m_areResultsOutstanding(false)
      , m_errorInfo(ERROR_CODE__NO_ERROR)
      , m_isResultTruncated(false)   //<-- INF4CV
   {
      m_availableSortOrders.insert(m_currentSortOrder);
   }

   const ::std::vector<LocationInfo>& getLocationInfos() const
   {
      return m_locationInfos;
   }
   unsigned int getCountOfAllLocations() const
   {
      return m_countOfAllLocations;
   }
   LocationSortOrder getCurrentSortOrder() const
   {
      return m_currentSortOrder;
   }
   const ::std::set<LocationSortOrder>& getAvailableSortOrders() const
   {
      return m_availableSortOrders;
   }
   const LocationSearchScope& getCurrentSearchScope() const
   {
      return m_currentSearchScope;
   }
   const ::std::string& getSearchScopeName() const
   {
      return m_searchScopeName;
   }
   LocationSearchSource getCurrentSearchSource() const
   {
      return m_currentSearchSource;
   }
   const ValidValue<LocationSearchSource>& getCurrentSourceOfResults() const
   {
      return m_currentSourceOfResults;
   }
   AutoCompletionMode getCurrentAutoCompletionMode() const
   {
      return m_currentAutoCompletionMode;
   }
   bool areResultsOutstanding() const
   {
      return m_areResultsOutstanding;
   }
   void setErrorCode(ErrorCode error)
   {
      m_errorInfo = error;
   }
   ErrorCode getErrorCode() const
   {
      return m_errorInfo;
   }
   void setPreferredResultLocationType(const ValidValue<PreferredResultLocationType>& preferredResultLocationType)
   {
      m_preferredResultLocationType = preferredResultLocationType;
   }
   const ValidValue<PreferredResultLocationType>& getPreferredResultLocationType() const
   {
      return m_preferredResultLocationType;
   }
   void setUpdateReason(const ValidValue<UpdateReason>& updateReason)
   {
      m_updateReason = updateReason;
   }
   const ValidValue<UpdateReason>& getUpdateReason() const
   {
      return m_updateReason;
   }
   ::std::vector<LocationInfo>& getLocationInfos()
   {
      return m_locationInfos;
   }
   void setCountOfAllLocations(unsigned int countOfAllLocations)
   {
      m_countOfAllLocations = countOfAllLocations;
   }
   void setCurrentSortOrder(LocationSortOrder sortOrder)
   {
      m_currentSortOrder = sortOrder;
   }
   void setAvailableSortOrders(const ::std::set<LocationSortOrder>& sortOrders)
   {
      m_availableSortOrders = sortOrders;
   }
   void setCurrentSearchScope(const LocationSearchScope& searchScope)
   {
      m_currentSearchScope = searchScope;
   }
   void setSearchScopeName(const ::std::string& searchScopeName)
   {
      m_searchScopeName = searchScopeName;
   }
   void setCurrentSearchSource(LocationSearchSource searchSource)
   {
      m_currentSearchSource = searchSource;
   }
   void setCurrentSourceOfResults(const ValidValue<LocationSearchSource>& currentSearchSource)
   {
      m_currentSourceOfResults = currentSearchSource;
   }
   void setCurrentAutoCompletionMode(AutoCompletionMode autoCompletionMode)
   {
      m_currentAutoCompletionMode = autoCompletionMode;
   }
   void setResultsOutstanding(bool status)
   {
      m_areResultsOutstanding = status;
   }

   //[JP]-->
   const ValidValue<T_StartingLetterIndexMap>& getStartingLetterIndexMap() const
   {
      return m_startingLetterIndexMap;
   }
   void setStartingLetterIndexMap(const ValidValue<T_StartingLetterIndexMap>& startingLetterIndexMap)
   {
      m_startingLetterIndexMap = startingLetterIndexMap;
   }
   //<--

   //<-- INF4CV
   bool isResultTruncated() const
   {
      return m_isResultTruncated;
   }
   void setResultTruncated(bool status)
   {
      m_isResultTruncated = status;
   }
   //->

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "LocationInfos payload:" << ::std::endl
             << "   m_countOfAllLocations = " << m_countOfAllLocations << ::std::endl
             << "   m_currentSortOrder = " << ::navmiddleware::toString(m_currentSortOrder) << ::std::endl
             << "   m_availableSortOrders = {";
      for (::std::set<LocationSortOrder>::const_iterator iter = m_availableSortOrders.begin();
            iter != m_availableSortOrders.end(); ++iter)
      {
         if (iter != m_availableSortOrders.begin())
         {
            stream << ", ";
         }
         stream << ::navmiddleware::toString(*iter);
      }
      stream << "}" << ::std::endl
             << "   m_currentSearchScope = " << m_currentSearchScope.toString() << ::std::endl
             << "   m_searchScopeName = " << m_searchScopeName << ::std::endl
             << "   m_currentSearchSource = " << ::navmiddleware::toString(m_currentSearchSource) << ::std::endl
             << "   m_currentAutoCompletionMode = " << ::navmiddleware::toString(m_currentAutoCompletionMode) << ::std::endl
             << "   m_areResultsOutstanding = " << (m_areResultsOutstanding? "true" : "false") << ::std::endl
             << "   m_isResultTruncated = " << (m_isResultTruncated? "true" : "false") << ::std::endl
             << "   m_errorInfo             = " << toString(m_errorInfo) << ::std::endl
             << "   m_locationInfos.size() = " << m_locationInfos.size() << ::std::endl;

      stream << "m_currentSourceOfResults is " << (m_currentSourceOfResults.isValid() ? ::navmiddleware::toString(m_currentSourceOfResults.getValue()) : "invalid") << ::std::endl;

      for(unsigned int locationListIndex = 0; locationListIndex < getLocationInfos().size(); locationListIndex++)
      {
         stream << "[" << locationListIndex << "] - " << m_locationInfos[locationListIndex].toString();
      }

      stream << "m_startingLetterIndexMap is " << (m_startingLetterIndexMap.isValid() ? "valid" : "invalid") << std::endl;
      if (m_startingLetterIndexMap.isValid())
      {
         const T_StartingLetterIndexMap& idxMap = m_startingLetterIndexMap.getValue();
         unsigned int k = 0;
         for (T_StartingLetterIndexMap::const_iterator it = idxMap.begin(); it != idxMap.end(); ++it, k++)
         {
            stream << "Letter/index[" << k << "] " << it->first << " [" << "] " << it->second << std::endl;
         }
      }

      stream<< "m_preferredResultLocationType is " << (m_preferredResultLocationType.isValid() ? toString(m_preferredResultLocationType.getValue()) : "invalid") << ::std::endl;

      stream<< "m_updateReason is " << (m_updateReason.isValid() ? toString(m_updateReason.getValue()) : "invalid") << ::std::endl;

      return stream.str();
   }

private:
   ::std::vector<LocationInfo>              m_locationInfos;
   unsigned int                             m_countOfAllLocations;
   LocationSortOrder                        m_currentSortOrder;
   ::std::set<LocationSortOrder>            m_availableSortOrders;
   LocationSearchScope                      m_currentSearchScope;
   ::std::string                            m_searchScopeName;
   LocationSearchSource                     m_currentSearchSource;
   /**
    * m_currentSourceOfResults flag indicates the source of the results.
    * If the results are online, flag is online.
    * If the results are offline, flag is offline.
    * If the results are mixed, flag is invalid.
    **/
   ValidValue<LocationSearchSource>         m_currentSourceOfResults;
   AutoCompletionMode                       m_currentAutoCompletionMode;
   bool                                     m_areResultsOutstanding;
   ErrorCode                                m_errorInfo;
   ValidValue<PreferredResultLocationType>  m_preferredResultLocationType;
   ValidValue<UpdateReason>                 m_updateReason;
   //[JP]
   ValidValue<T_StartingLetterIndexMap>     m_startingLetterIndexMap;
   bool                                     m_isResultTruncated;   //<-- INF4CV
};


/**
 * Holds all data required to fill a speller screen for form-based/hierarchical address or POI input.
 */
class ElementSpellerInfo
{
public:
   ElementSpellerInfo() : m_numberOfMatchingElements(65535),
      m_allowAutoJumpToResultList(false) {}

   /** Returns the currently used speller string */
   const ::std::string& getCurrentSpellerMatch() const
   {
      return m_currentSpellerMatch;
   }
   /** Returns a string containing the characters with can be input next matching at least one of the remaining elements */
   const ::std::string& getNextSpellerCharacters() const
   {
      return m_nextSpellerCharacters;
   }
   /** Returns the overall number of matching elements (a rough guess - not exact) */
   unsigned int getNumberOfMatchingElements() const
   {
      return m_numberOfMatchingElements;
   }
   /** Returns the status that is used to signal screen transition from speller view to the list view */
   bool getAllowAutoJumpToResultList() const
   {
      return m_allowAutoJumpToResultList;
   }
   void setCurrentSpellerMatch(const ::std::string& spellerMatch)
   {
      m_currentSpellerMatch = spellerMatch;
   }
   void setNextSpellerCharacters(const ::std::string& nextSpellerCharacters)
   {
      m_nextSpellerCharacters = nextSpellerCharacters;
   }
   void setNumberOfMatchingElements(unsigned int numberOfMatchingElements)
   {
      m_numberOfMatchingElements = numberOfMatchingElements;
   }
   void setAllowAutoJumpToResultList(bool status)
   {
      m_allowAutoJumpToResultList = status;
   }
   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "ElementSpellerInfo payload:" << ::std::endl
             << "   m_currentSpellerMatch = \"" << m_currentSpellerMatch << "\"" << ::std::endl
             << "   m_nextSpellerCharacters = \"" << m_nextSpellerCharacters << "\"" << ::std::endl
             << "   m_numberOfMatchingElements = " << m_numberOfMatchingElements << ::std::endl
             << "   m_allowAutoJumpToResultList = " << (m_allowAutoJumpToResultList ? "true" : "false") << ::std::endl;
      return stream.str();
   }

private:
   ::std::string m_currentSpellerMatch;
   ::std::string m_nextSpellerCharacters;
   unsigned int m_numberOfMatchingElements;
   bool m_allowAutoJumpToResultList;
};


/**
 * Defines the possible categories for form-based address input.
 */
enum AddressCategory { COUNTRY, STATE, CITY, STREET, HOUSE_NUMBER, CROSS_STREET, AMBIGUITY_RESOLUTION, PREFECTURE, CITY_TOWN_OR_WARD, WARD, TOWN, POSTAL_CODE };

inline ::std::string toString(AddressCategory category)
{
   switch (category)
   {
   case COUNTRY:
      return "COUNTRY";
   case STATE:
      return "STATE";
   case CITY:
      return "CITY";
   case STREET:
      return "STREET";
   case HOUSE_NUMBER:
      return "HOUSE_NUMBER";
   case CROSS_STREET:
      return "CROSS_STREET";
   case AMBIGUITY_RESOLUTION:
      return "AMBIGUITY_RESOLUTION";
      //<!-- JP
   case PREFECTURE:
      return "PREFECTURE";
   case CITY_TOWN_OR_WARD:
      return "CITY_TOWN_OR_WARD";
   case WARD:
      return "WARD";
   case TOWN:
      return "TOWN";
   case POSTAL_CODE:
      return "POSTAL_CODE";
      //->
   default:
      ::std::stringstream stream;
      stream << "<" << static_cast<unsigned int>(category) << ">";
      return stream.str();
   }
}


/**
 * Holds all data required to fill a form-based address input screen.
 */
class AddressInputInfo
{
public:
   AddressInputInfo()
      : m_currentPrefectureSubCC(0)
      , m_isCountrySelectable(false)
      , m_isStateSelectable(false)
      , m_isCitySelectable(false)
      , m_isStreetSelectable(false)
      , m_isHouseNumberSelectable(false)
      , m_isCrossStreetSelectable(false)
      , m_isPrefectureSelectable(false)
      , m_isCityTownOrWardSelectable(false)
      , m_isWardSelectable(false)
      , m_isTownSelectable(false)
      , m_isHouseNumberCorrectionRequired(false)
      , m_NoHouseNumbersAvailableInSelectedStreet(false)
      , m_isAmbiguityResultionRequired(false)
      , m_isValidDestination(false)
   {}

   struct SdsGlobalNamedObjectWithAddressType
   {
      AddressCategory m_addressCategory;
      SdsGlobalNamedObjectIdInfo m_sdsGlobalNamedObjectIdInfo;
      SdsGlobalNamedObjectWithAddressType()
         : m_addressCategory(COUNTRY)
      {
      }
   };

   /** Returns the currently selected country */
   const ::std::string& getCurrentCountry() const
   {
      return m_currentCountry;
   }
   /** Returns the currently selected state */
   const ::std::string& getCurrentState() const
   {
      return m_currentState;
   }
   /** Returns the currently selected city */
   const ::std::string& getCurrentCity() const
   {
      return m_currentCity;
   }
   /** Returns the currently selected City name as in Navigation Data, if returns empty use getCurrentCity()*/
   const ::std::string& getCurrentCityAsInNavigationData() const
   {
      return m_currentCityAsInNavigationData;
   }
   /** Returns the currently selected street */
   const ::std::string& getCurrentStreet() const
   {
      return m_currentStreet;
   }
   /** Returns the currently selected street name as in Navigation Data if returns empty use getCurrentStreet()*/
   const ::std::string& getCurrentStreetAsInNavigationData() const
   {
      return m_currentStreetAsInNavigationData;
   }
   /** Returns the currently selected house number */
   const ::std::string& getCurrentHouseNumber() const
   {
      return m_currentHouseNumber;
   }
   /** Returns the currently selected cross street (intersection) */
   const ::std::string& getCurrentCrossStreet() const
   {
      return m_currentCrossStreet;
   }

   const ::std::vector<SdsGlobalNamedObjectWithAddressType>& getSdsGlobalNamedObjectWithAddressTypeList() const
   {
      return m_sdsGlobalNamedObjectWithAddressTypeList;
   }

   ValidValue<AddressCategory> getActualAddressCategoryUnderCityAsInNavigationData() const
   {
      return m_actualAddressCategoryUnderCityAsInNavigationData;
   }
   //<!--JP
   /** Returns the currently selected Prefecture */
   const ::std::string& getCurrentPrefecture() const
   {
      return m_currentPrefecture;
   }
   /** Valid (m_currentPrefectureSubCC != 0) only when m_currentPrefecture is set **/
   uint64_t getCurrentPrefectureSubCC() const
   {
      return m_currentPrefectureSubCC;
   }
   /** Returns the currently selected City, Town or Ward */
   const ::std::string& getCurrentCityTownOrWard() const
   {
      return m_currentCityTownOrWard;
   }
   /** Returns the currently selected Ward */
   const ::std::string& getCurrentWard() const
   {
      return m_currentWard;
   }
   /** Returns the currently selected Town */
   const ::std::string& getCurrentTown() const
   {
      return m_currentTown;
   }
   //->
   /** Returns the HouseNo availability in selected street */
   bool getNoHouseNumbersAvailableInSelectedStreet() const
   {
      return m_NoHouseNumbersAvailableInSelectedStreet;
   }
   /** Returns the Current Coordinates for a valid destination */
   const ValidValue<GeoCoordinateDegree>& getCurrentCoordinates() const
   {
      return m_currentCoordinate;
   }
   const ValidValue<LocationAttributeInfos::BoundingBoxInfo>& getCurrentBoundingBox() const
   {
      return m_currentBoundingBox;
   }
   // Do we also need the language specific field names here (e.g. "Country"/"Land")?

   /** Returns true if country can be entered in current state */
   bool isCountrySelectable() const
   {
      return m_isCountrySelectable;
   }
   /** Returns true if state can be entered in current state */
   bool isStateSelectable() const
   {
      return m_isStateSelectable;
   }
   /** Returns true if city can be entered in current state */
   bool isCitySelectable() const
   {
      return m_isCitySelectable;
   }
   /** Returns true if street can be entered in current state */
   bool isStreetSelectable() const
   {
      return m_isStreetSelectable;
   }
   /** Returns true if house number can be entered in current state */
   bool isHouseNumberSelectable() const
   {
      return m_isHouseNumberSelectable;
   }
   /** Returns true if cross street can be entered in current state */
   bool isCrossStreetSelectable() const
   {
      return m_isCrossStreetSelectable;
   }
   //<!--JP
   /** Returns true if prefecture is selectable in the current state */
   bool isPrefectureSelectable() const
   {
      return m_isPrefectureSelectable;
   }
   /** Returns true if city, town or ward is selectable in the current state */
   bool isCityTownOrWardSelectable() const
   {
      return m_isCityTownOrWardSelectable;
   }
   /** Returns true if ward  is selectable in the current state */
   bool isWardSelectable() const
   {
      return m_isWardSelectable;
   }
   /** Returns true if town is selectable in the current state */
   bool isTownSelectable() const
   {
      return m_isTownSelectable;
   }
   //->
   /** Returns true if the entered house number needs correction */
   bool isHouseNumberCorrectionRequired() const
   {
      return m_isHouseNumberCorrectionRequired;
   }
   /** Returns true if the entered address data is not yet unique */
   bool isAmbiguityResultionRequired() const
   {
      return m_isAmbiguityResultionRequired;
   }
   /** Returns true if valid coordinates are available */
   bool isCurrentCoordinateValid() const
   {
      return m_currentCoordinate.isValid();
   }

   /** Returns true if the entered address (so far) is unique and could be used to start a route guidance */
   bool isValidDestination() const
   {
      return m_isValidDestination;
   }

   void setCurrentCountry(const ::std::string& country)
   {
      m_currentCountry = country;
   }
   void setCurrentState(const ::std::string& state)
   {
      m_currentState = state;
   }
   void setCurrentCity(const ::std::string& city)
   {
      m_currentCity = city;
   }
   void setCurrentCityAsInNavigationData(const ::std::string& currentCityAsInNavigationData)
   {
      m_currentCityAsInNavigationData = currentCityAsInNavigationData;
   }
   void setCurrentStreet(const ::std::string& street)
   {
      m_currentStreet = street;
   }
   void setCurrentStreetAsInNavigationData(const ::std::string& currentStreetAsInNavigationData)
   {
      m_currentStreetAsInNavigationData = currentStreetAsInNavigationData;
   }
   void setCurrentHouseNumber(const ::std::string& houseNumber)
   {
      m_currentHouseNumber = houseNumber;
   }
   void setCurrentCrossStreet(const ::std::string& crossStreet)
   {
      m_currentCrossStreet = crossStreet;
   }
   void setActualAddressCategoryUnderCityAsInNavigationData(const ValidValue<AddressCategory>& actualAddressCategoryUnderCityAsInNavigationData)
   {
      m_actualAddressCategoryUnderCityAsInNavigationData = actualAddressCategoryUnderCityAsInNavigationData;
   }
   //<!--JP
   void setCurrentPrefecture(const ::std::string& prefecture)
   {
      m_currentPrefecture = prefecture;
   }
   void setCurrentPrefectureSubCC(uint64_t prefectureSubCC)
   {
      m_currentPrefectureSubCC = prefectureSubCC;
   }
   void setCurrentCityTownOrWard(const ::std::string& cityTownOrWard)
   {
      m_currentCityTownOrWard = cityTownOrWard;
   }
   void setCurrentWard(const ::std::string& ward)
   {
      m_currentWard = ward;
   }
   void setCurrentTown(const ::std::string& town)
   {
      m_currentTown = town;
   }
   //->
   void setIsCountrySelectable(bool isSelectable)
   {
      m_isCountrySelectable = isSelectable;
   }
   void setIsStateSelectable(bool isSelectable)
   {
      m_isStateSelectable = isSelectable;
   }
   void setIsCitySelectable(bool isSelectable)
   {
      m_isCitySelectable = isSelectable;
   }
   void setIsStreetSelectable(bool isSelectable)
   {
      m_isStreetSelectable = isSelectable;
   }
   void setIsHouseNumberSelectable(bool isSelectable)
   {
      m_isHouseNumberSelectable = isSelectable;
   }
   void setIsCrossStreetSelectable(bool isSelectable)
   {
      m_isCrossStreetSelectable = isSelectable;
   }
   //<!--JP
   void setIsPrefectureSelectable(bool isSelectable)
   {
      m_isPrefectureSelectable = isSelectable;
   }
   void setIsCityTownOrWardSelectable(bool isSelectable)
   {
      m_isCityTownOrWardSelectable = isSelectable;
   }
   void setIsWardSelectable(bool isSelectable)
   {
      m_isWardSelectable = isSelectable;
   }
   void setIsTownSelectable(bool isSelectable)
   {
      m_isTownSelectable = isSelectable;
   }
   //->
   void setIsHouseNumberCorrectionRequired(bool isRequired)
   {
      m_isHouseNumberCorrectionRequired = isRequired;
   }
   void setNoHouseNumbersAvailableInSelectedStreet(bool isNotAvailable)
   {
      m_NoHouseNumbersAvailableInSelectedStreet = isNotAvailable;
   }
   void setIsAmbiguityResultionRequired(bool isRequired)
   {
      m_isAmbiguityResultionRequired = isRequired;
   }
   void setCurrentCoordinates(const GeoCoordinateDegree& coodinateDegree)
   {
      m_currentCoordinate.setValue(coodinateDegree);
   }
   void setCurrentBoundingBox(const LocationAttributeInfos::BoundingBoxInfo& boundingBox)
   {
      m_currentBoundingBox.setValue(boundingBox);
   }

   void setIsValidDestination(bool isValid)
   {
      m_isValidDestination = isValid;
   }

   void setSdsGlobalNamedObjectWithAddressTypeList(const ::std::vector<SdsGlobalNamedObjectWithAddressType>& sdsGlobalNamedObjectWithAddressTypeList)
   {
      m_sdsGlobalNamedObjectWithAddressTypeList = sdsGlobalNamedObjectWithAddressTypeList;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "AddressInputInfo payload:" << ::std::endl
             << "   Country = \""<< m_currentCountry << "\", "
             << "State = \"" << m_currentState << "\", "
             << "City = \"" << m_currentCity  << "\", "
             << "CityAsInData = \"" << m_currentCityAsInNavigationData  << "\", "
             << "Street = \"" << m_currentStreet << "\", "
             << "StreetAsInData = \"" << m_currentStreetAsInNavigationData << "\", "
             << "House Number = \"" << m_currentHouseNumber << "\", "
             << "Cross street = \"" << m_currentCrossStreet << "\", "
             << "ActualAddressCategoryUnderCityAsInData = \"" << navmiddleware::toString(m_actualAddressCategoryUnderCityAsInNavigationData.getValue()) << "\"" << ::std::endl
             << "Prefecture = "<< m_currentPrefecture << ", "
             << "Current prefecture SubCC" << m_currentPrefectureSubCC << ", "
             << "City town or ward = "<< m_currentCityTownOrWard << ", "
             << "Ward = "<< m_currentWard << ", "
             << "Town = "<< m_currentTown << ::std::endl
             << "   Is Country selectable = "<< m_isCountrySelectable << ::std::endl
             << "   Is State selectable = "<< m_isStateSelectable << ::std::endl
             << "   Is City selectable = "<< m_isCitySelectable << ::std::endl
             << "   Is Street selectable = "<< m_isStreetSelectable << ::std::endl
             << "   Is House number selectable = "<< m_isHouseNumberSelectable << ::std::endl
             << "   Is Cross street selectable = "<< m_isCrossStreetSelectable << ::std::endl
             << "   Is Prefecture selectable = " << m_isPrefectureSelectable << ::std::endl
             << "   Is City town or ward selectable = "<< m_isCityTownOrWardSelectable << ::std::endl
             << "   Is Ward selectable = "<< m_isWardSelectable << ::std::endl
             << "   Is Town selectable = "<< m_isTownSelectable << ::std::endl
             << "   Is house number correction required = "<< m_isHouseNumberCorrectionRequired << ::std::endl
             << "   Are no house numbers available in selected street = "<< m_NoHouseNumbersAvailableInSelectedStreet << ::std::endl
             << "   Is ambiguity resolution required = "<< m_isAmbiguityResultionRequired << ::std::endl
             << "   Is a valid destination = "<< m_isValidDestination << ::std::endl
             << "   Coordinate = " << (m_currentCoordinate.isValid()? m_currentCoordinate.getValue().toString() : "invalid") << ::std::endl
             << "   BoundingBox = " << (m_currentBoundingBox.isValid()? m_currentBoundingBox.getValue().toString() : "invalid") << ::std::endl;
      return stream.str();
   }

private:
   ::std::string m_currentCountry;
   ::std::string m_currentState;
   ::std::string m_currentCity;
   //If Display Name differs from Actual Name in Navigation Data this variable is filled
   //Used in HapticHandover usecase for SDS
   ::std::string m_currentCityAsInNavigationData;
   ::std::string m_currentStreet;
   //If Display Name differs from Actual Name in Navigation Data this variable is filled,
   //Used in HapticHandover usecase for SDS
   ::std::string m_currentStreetAsInNavigationData;
   ::std::string m_currentHouseNumber;
   ::std::string m_currentCrossStreet;
   ::std::vector<SdsGlobalNamedObjectWithAddressType> m_sdsGlobalNamedObjectWithAddressTypeList;
   ValidValue<AddressCategory> m_actualAddressCategoryUnderCityAsInNavigationData;
   //<!-- JP
   ::std::string m_currentPrefecture;
   // Represents ISO 3166-2:JP sub-country code, 64-bit in order to allow enhanced
   // internal matching. Valid (!= 0) only when m_currentPrefecture is set.
   uint64_t m_currentPrefectureSubCC;
   ::std::string m_currentCityTownOrWard;
   ::std::string m_currentWard;
   ::std::string m_currentTown;
   //->

   bool m_isCountrySelectable;
   bool m_isStateSelectable;
   bool m_isCitySelectable;
   bool m_isStreetSelectable;
   bool m_isHouseNumberSelectable;
   bool m_isCrossStreetSelectable;

   //<!-- JP
   bool m_isPrefectureSelectable;
   bool m_isCityTownOrWardSelectable;
   bool m_isWardSelectable;
   bool m_isTownSelectable;
   //->

   bool m_isHouseNumberCorrectionRequired;
   bool m_NoHouseNumbersAvailableInSelectedStreet;
   bool m_isAmbiguityResultionRequired;
   ValidValue<GeoCoordinateDegree> m_currentCoordinate;
   ValidValue<LocationAttributeInfos::BoundingBoxInfo> m_currentBoundingBox;

   bool m_isValidDestination;
};


class PoiInputInfo
{
public:
   PoiInputInfo():
      m_isValidDestination(false),
      m_isTopLevelEntryList(false),
      m_isPoiElementList(false),
      m_onlineSearchAvailability(ONLINE_SERVICE_AVAILABILITY__UNKNOWN)
   {}

   /** Returns true if a POI element was selected and could be used to start a route guidance */
   bool isValidDestination() const
   {
      return m_isValidDestination;
   }
   /** Returns true if current POI input scope is the up-most input level (categories or elements) */
   bool isTopLevelEntryList() const
   {
      return m_isTopLevelEntryList;
   }
   /** Returns true if current POI entry list represents a list of POI (leaf) elements (e.g. "McDonalds, Hildesheim)*/
   bool isPoiElementList() const
   {
      return m_isPoiElementList;
   }

   void setIsValidDestination(bool isValid)
   {
      m_isValidDestination = isValid;
   }
   void setIsTopLevelEntryList(bool isIsTopLevel)
   {
      m_isTopLevelEntryList = isIsTopLevel;
   }
   void setIsPoiElementList(bool isElementList)
   {
      m_isPoiElementList = isElementList;
   }

   const ValidValue<T_StartingLetterIndexMap>& getStartingLetterIndexMap() const
   {
      return m_startingLetterIndexMap;
   }
   void setStartingLetterIndexMap(const ValidValue<T_StartingLetterIndexMap>& startingLetterIndexMap)
   {
      m_startingLetterIndexMap = startingLetterIndexMap;
   }

   // @DEPRECATED:Please use OnlinePoiAvailabilityInfo
   OnlineServiceAvailability getPoiOnlineSearchAvailability() const
   {
      return m_onlineSearchAvailability;
   }
   // @DEPRECATED: Please use OnlinePoiAvailabilityInfo
   void setPoiOnlineSearchAvailability(OnlineServiceAvailability availability)
   {
      m_onlineSearchAvailability = availability;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "PoiInputInfo payload:" << ::std::endl
             << "   m_isValidDestination       = " << m_isValidDestination << ::std::endl
             << "   m_isTopLevelEntryList      = " << m_isTopLevelEntryList << ::std::endl
             << "   m_isPoiElementList         = " << m_isPoiElementList << ::std::endl
             << "   m_onlineSearchAvailability = " << ::navmiddleware::toString(m_onlineSearchAvailability) << ::std::endl
             << "   m_startingLetterIndexMap.isValid() = " << (m_startingLetterIndexMap.isValid()? "true" : "false") << ::std::endl;
      if (m_startingLetterIndexMap.isValid())
      {
         const T_StartingLetterIndexMap& idxMap = m_startingLetterIndexMap.getValue();
         for (T_StartingLetterIndexMap::const_iterator it = idxMap.begin(); it != idxMap.end(); ++it)
         {
            stream << it->first << ":" << it->second << std::endl;
         }
      }
      return stream.str();
   }

private:
   bool                                   m_isValidDestination;
   bool                                   m_isTopLevelEntryList;
   bool                                   m_isPoiElementList;
   ValidValue<T_StartingLetterIndexMap>   m_startingLetterIndexMap;
   OnlineServiceAvailability              m_onlineSearchAvailability;// @DEPRECATED
};

class LocationCategoryInfos
{
public:
   LocationCategoryInfos() {}

   struct LocationCategoryInfo
   {
      LocationCategoryInfo()
         : m_sourceId(0)
         , m_imageId(0)
         , m_imageBlob(0)
         , m_imageBlobLength(0)
         , m_poiCategoryId(0)
         , m_predefinedPoiCategory(PREDEFINED_POI_CATEGORY__UNDEFINED)
      {}
      LocationCategoryInfo(
         const ::std::string& categoryName,
         uint8_t sourceId,
         uint64_t imageId,
         const unsigned char* imageBlob,
         unsigned int imageBlobLength,
         const Image& image,
         PoiCategoryId poiCategoryId,
         PredefinedPoiCategory predefinedPoiCategory,
         const ValidValue<SdsGlobalNamedObjectIdInfo>& sdsGlobalNamedObjectIdInfo
      )
         : m_categoryName(categoryName)
         , m_sourceId(sourceId)
         , m_imageId(imageId)
         , m_imageBlob(imageBlob)
         , m_imageBlobLength(imageBlobLength)
         , m_image(image)
         , m_poiCategoryId(poiCategoryId)
         , m_predefinedPoiCategory(predefinedPoiCategory)
         , m_sdsGlobalNamedObjectIdInfo(sdsGlobalNamedObjectIdInfo)
      {}

      bool operator==(const LocationCategoryInfo& other) const
      {
         return (this->m_categoryName == other.m_categoryName)
                && (this->m_sourceId == other.m_sourceId)
                && (this->m_imageId == other.m_imageId)
                && (this->m_imageBlob == other.m_imageBlob)
                && (this->m_imageBlobLength == other.m_imageBlobLength)
                && (this->m_image == other.m_image)
                && (this->m_poiCategoryId == other.m_poiCategoryId)
                && (this->m_predefinedPoiCategory == other.m_predefinedPoiCategory)
                && (this->m_sdsGlobalNamedObjectIdInfo == other.m_sdsGlobalNamedObjectIdInfo);
      }
      bool operator!=(const LocationCategoryInfo& other) const
      {
         return !(*this == other);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream;
         stream << "LocationCategoryInfo payload:" << ::std::endl
                << "   m_categoryName = \"" << m_categoryName << "\"" << ::std::endl
                << "   m_sourceId = " << static_cast<unsigned int>(m_sourceId) << ::std::endl
                << "   m_imageID = " << m_imageId << ::std::endl
                << "   m_imageBlob-ptr = 0x" << static_cast<const void*>(m_imageBlob) << ::std::endl
                << "   m_imageBlobLength = " << m_imageBlobLength << ::std::endl
                << "   m_image = " << m_image.toString()<< ::std::endl
                << "   m_poiCategoryId = " << m_poiCategoryId << ::std::endl
                << "   m_predefinedPoiCategory = " << ::navmiddleware::toString(m_predefinedPoiCategory) << ::std::endl
                << "   m_sdsGlobalNamedObjectIdInfo = " << (m_sdsGlobalNamedObjectIdInfo.isValid() ?
                      m_sdsGlobalNamedObjectIdInfo.getValue().toString() : "No valid NDSId") << ::std::endl;
         return stream.str();
      }

      ::std::string          m_categoryName;
      uint8_t                m_sourceId;
      uint64_t               m_imageId;
      const unsigned char*   m_imageBlob;
      unsigned int           m_imageBlobLength;
      //m_image should be used. Other image related members will be @DEPRECATED.
      Image                  m_image;
      /** Internal NavCore POI category id - relevant for speech (SDS) component */
      PoiCategoryId          m_poiCategoryId;
      /** Contains the predefined POI category (relevant to HMI) if applicable */
      PredefinedPoiCategory  m_predefinedPoiCategory;
      /**Contains the NDS Id of the category to be able to request for Phoneme, Used in SDS context */
      ValidValue<SdsGlobalNamedObjectIdInfo> m_sdsGlobalNamedObjectIdInfo;
   };

   const ::std::vector<LocationCategoryInfo>& getLocationCategoryInfos() const
   {
      return m_locationCategoryInfos;
   }

   ::std::vector<LocationCategoryInfo>& getLocationCategoryInfos()
   {
      return m_locationCategoryInfos;
   }

   void setLanguageEntryInfo(const LanguageEntryInfo& languageEntryInfo)
   {
      m_languageEntryInfo = languageEntryInfo;
   }

   const LanguageEntryInfo& getLanguageEntryInfo() const
   {
      return m_languageEntryInfo;
   }

   bool operator==(const LocationCategoryInfos& other) const
   {
      return (this->m_locationCategoryInfos == other.m_locationCategoryInfos)
             && (this->m_languageEntryInfo == other.m_languageEntryInfo);
   }
   bool operator!=(const LocationCategoryInfos& other) const
   {
      return !(*this == other);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "LocationCategoryInfos payload:" << ::std::endl;
      stream << m_languageEntryInfo.toString();
      for(unsigned int i = 0; i < getLocationCategoryInfos().size(); ++i)
      {
         stream << "[" << i << "]: " << m_locationCategoryInfos[i].toString();
      }
      return stream.str();
   }

private:
   ::std::vector<LocationCategoryInfo>  m_locationCategoryInfos;
   /** Language code used for Translation of POI Categories (Used in SDS context only) */
   LanguageEntryInfo m_languageEntryInfo;
};

//Enumeration to indicate the failure to finalize location
enum LocationError
{
   LOCATION_ERROR__INTERNAL_FAILURE,
   LOCATION_ERROR__LOCATION_IS_OFF_MAP,
   LOCATION_ERROR__ABORT_BY_CLIENT
};
inline ::std::string toString(LocationError error)
{
   switch (error)
   {
   case LOCATION_ERROR__INTERNAL_FAILURE:
      return "LOCATION_ERROR__INTERNAL_FAILURE";
   case LOCATION_ERROR__LOCATION_IS_OFF_MAP:
      return "LOCATION_ERROR__LOCATION_IS_OFF_MAP";
   case LOCATION_ERROR__ABORT_BY_CLIENT:
      return "LOCATION_ERROR__ABORT_BY_CLIENT";
   default:
      ::std::stringstream stream;
      stream << "LOCATION_ERROR__<" << static_cast<unsigned int>(error) << ">";
      return stream.str();
   }
}

class LocationAttributeErrorInfo
{
public:
   LocationAttributeErrorInfo():
      m_locationAttributesError(LOCATION_ERROR__INTERNAL_FAILURE),
      m_requestId(0)
   {}

   void setRequestId(const RequestId& requestId)
   {
      m_requestId = requestId;
   }
   const RequestId& getRequestId() const
   {
      return m_requestId;
   }

   void setLocationAttributesError(const LocationError& locationError)
   {
      m_locationAttributesError = locationError;
   }
   const LocationError& getLocationAttributesError() const
   {
      return m_locationAttributesError;
   }

   bool operator==(const LocationAttributeErrorInfo& other) const
   {
      return (this->m_locationAttributesError == other.m_locationAttributesError)
             && (this->m_requestId == other.m_requestId);
   }
   bool operator!=(const LocationAttributeErrorInfo& other) const
   {
      return !(*this == other);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "LocationAttributeErrorInfo payload:" << ::std::endl;
      stream << "Location Error" << ::navmiddleware::toString(m_locationAttributesError);
      stream << "RequestId" << m_requestId;
      return stream.str();
   }

private:
   LocationError m_locationAttributesError;
   RequestId     m_requestId;
};

class AddressFormatInfos
{
public:
   struct AddressFormatInfo
   {
      AddressFormatInfo(
         const ::std::string& isoCountryCode,
         const ::std::string& addressFormat
      )
         :m_isoCountryCode(isoCountryCode),
          m_addressFormat(addressFormat)
      {}

      ::std::string toString() const
      {
         ::std::stringstream stream;
         stream << "AddressFormatInfo payload:" << ::std::endl
                << "   m_isoCountryCode = " << m_isoCountryCode << ::std::endl
                << "   m_addressFormat = " << m_addressFormat << ::std::endl;
         return stream.str();
      }

      // country code returned as ISO 3166-1-alpha-3.
      ::std::string m_isoCountryCode;
      /*
       * address format returned in xml tags.
       * Eg:<ROAD/><HNR/><ZIP/><PLACE/><COUNTRY/>
       * Supported Tags:<COUNTRY/><SUBCOUNTRY/><PLACE/><CITYDISTRICT/><ZIP/><HNR/><CROSSROAD/>
       */
      ::std::string m_addressFormat;
   };

   const ::std::vector<AddressFormatInfo>& getAddressFormatInfos() const
   {
      return m_addressFormatInfos;
   }
   ::std::vector<AddressFormatInfo>& getAddressFormatInfos()
   {
      return m_addressFormatInfos;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "AddressFormatInfos payload:" << ::std::endl
             << "   No of AddressFormats Available = " << getAddressFormatInfos().size()<< ::std::endl;
      for(unsigned int index = 0; index < getAddressFormatInfos().size(); index++)
      {
         stream << "[" << index << "] - " << m_addressFormatInfos[index].toString();
      }
      return stream.str();
   }
private:
   ::std::vector<AddressFormatInfo>    m_addressFormatInfos;
};

class RefreshInfo
{
public:
   enum RefreshStatus
   {
      REFRESH_STATUS__IN_PROGRESS,
      REFRESH_STATUS__INTERRUPTED,
      REFRESH_STATUS__COMPLETE
   };
   static ::std::string toString(RefreshStatus status)
   {
      switch(status)
      {
      case REFRESH_STATUS__IN_PROGRESS:
         return "REFRESH_STATUS__IN_PROGRESS";
      case REFRESH_STATUS__INTERRUPTED:
         return "REFRESH_STATUS__INTERRUPTED";
      case REFRESH_STATUS__COMPLETE:
         return "REFRESH_STATUS__COMPLETE";
      default:
         ::std::stringstream stream;
         stream << "REFRESH_STATUS__<" << static_cast<unsigned int>(status) << ">";
         return stream.str();
      }
   }

   RefreshInfo()
      :m_countofExpiredLocations(0)
      ,m_countofRefreshedLocations(0)
      ,m_countofDeletedLocations(0)
      ,m_countofOnboardMatchedLocations(0)
   {}

   RefreshInfo(
      const ValidValue<RefreshStatus>& refreshStatus,
      unsigned int expiredLocationsCount,
      unsigned int refreshedLocationsCount,
      unsigned int deletedLocationsCount,
      unsigned int onboardMatchedLocationsCount
   )
      :m_refreshStatus(refreshStatus),
       m_countofExpiredLocations(expiredLocationsCount),
       m_countofRefreshedLocations(refreshedLocationsCount),
       m_countofDeletedLocations(deletedLocationsCount),
       m_countofOnboardMatchedLocations(onboardMatchedLocationsCount)
   {}

   const ValidValue<RefreshStatus>& getRefreshStatus() const
   {
      return m_refreshStatus;
   }
   unsigned int getCountOfExpiredLocations() const
   {
      return m_countofExpiredLocations;
   }
   unsigned int getCountOfRefreshedLocations() const
   {
      return m_countofRefreshedLocations;
   }
   unsigned int getCountofDeltedLocations() const
   {
      return m_countofDeletedLocations;
   }
   unsigned int getCountOfOnboardMatchedLocations() const
   {
      return m_countofOnboardMatchedLocations;
   }

   void setRefreshStatus(const ValidValue<RefreshStatus>& refreshStatus)
   {
      m_refreshStatus = refreshStatus;
   }
   void setCountOfExpiredLocations(unsigned int expiredLocationsCount)
   {
      m_countofExpiredLocations = expiredLocationsCount;
   }
   void setCountOfRefreshedLocations(unsigned int refreshedLocationsCount)
   {
      m_countofRefreshedLocations = refreshedLocationsCount;
   }
   void setCountofDeltedLocations(unsigned int deletedLocationsCount)
   {
      m_countofDeletedLocations = deletedLocationsCount;
   }
   void setCountOfOnboardMatchedLocations(unsigned int onboardMatchedLocationsCount)
   {
      m_countofOnboardMatchedLocations = onboardMatchedLocationsCount;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "Refresh Info payload:" << ::std::endl
             << "m_refreshStatus = " << (m_refreshStatus.isValid() ? toString(m_refreshStatus.getValue()) : "invalid") << ::std::endl
             << "m_countofExpiredLocations = "<<m_countofExpiredLocations<<::std::endl
             << "m_countofRefreshedLocations = "<<m_countofRefreshedLocations<<::std::endl
             <<" m_countofDeletedLocations = "<<m_countofDeletedLocations<<::std::endl
             <<" m_countofOnboardMatchedLocations = "<<m_countofOnboardMatchedLocations<<::std::endl;
      return stream.str();
   }

private:
   ValidValue<RefreshStatus>  m_refreshStatus;
   // If Status is Valid , then read the below content.
   unsigned int               m_countofExpiredLocations;
   unsigned int               m_countofRefreshedLocations;
   unsigned int               m_countofDeletedLocations;
   unsigned int               m_countofOnboardMatchedLocations;
};

struct AutoCompletionConfiguration
{
public:
   enum AutoCompletionCategoryType
   {
      AutoCompletion_Category_Type__HOME,
      AutoCompletion_Category_Type__WORK,
      AutoCompletion_Category_Type__ADDRESS_BOOK,
      AutoCompletion_Category_Type__USER_TOURS,
      AutoCompletion_Category_Type__PREVIOUS_DESTINATIONS,
      AutoCompletion_Category_Type__SEARCH_HISTORY
   };

   ::std::string toString(AutoCompletionCategoryType categoryType)
   {
       switch(categoryType)
       {
       case  AutoCompletion_Category_Type__HOME:
         return "AutoCompletion_Category_Type__HOME";
       case  AutoCompletion_Category_Type__WORK:
         return "AutoCompletion_Category_Type__WORK";
       case  AutoCompletion_Category_Type__ADDRESS_BOOK:
          return "AutoCompletion_Category_Type__ADDRESS_BOOK";
       case  AutoCompletion_Category_Type__USER_TOURS:
          return "AutoCompletion_Category_Type__USER_TOURS";
       case  AutoCompletion_Category_Type__PREVIOUS_DESTINATIONS:
          return "AutoCompletion_Category_Type__PREVIOUS_DESTINATIONS";
       case  AutoCompletion_Category_Type__SEARCH_HISTORY:
          return "AutoCompletion_Category_Type__SEARCH_HISTORY";
       default :
          ::std::stringstream stream;
         stream << "AutoCompletion_Category_Type__" << static_cast<unsigned int>(categoryType) << ">";
         return stream.str();
       }
   }

   enum FtsSearchFilterType
   {
      Fts_Search_Filter_Type__ENTRY,
      Fts_Search_Filter_Type__TOUR,
      Fts_Search_Filter_Type__POI_CATEGORY
   };

   ::std::string toString(FtsSearchFilterType filterType)
   {
       switch(filterType)
       {
       case  Fts_Search_Filter_Type__ENTRY:
         return "Fts_Search_Filter_Type__ENTRY";
       case  Fts_Search_Filter_Type__TOUR:
         return "Fts_Search_Filter_Type__TOUR";
       case  Fts_Search_Filter_Type__POI_CATEGORY:
         return "Fts_Search_Filter_Type__POI_CATEGORY";
       default :
          ::std::stringstream stream;
         stream << "Fts_Search_Filter_Type__" << static_cast<unsigned int>(filterType) << ">";
         return stream.str();
       }
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "AutoCompletionConfiguration Payload:" << ::std::endl;

      if(m_ftsAutoCompletionCategoryForEmptySpeller.isValid())
      {
         ::std::vector<AutoCompletionCategoryType> ftsAutoCompletionCategoryForEmptySpeller =
                  m_ftsAutoCompletionCategoryForEmptySpeller.getValue();

         for(size_t emptySpellerList = 0; emptySpellerList < ftsAutoCompletionCategoryForEmptySpeller.size(); emptySpellerList++)
         {
            stream << "Payload for empty speller[" << emptySpellerList << "] - "
                   << (ftsAutoCompletionCategoryForEmptySpeller.at(emptySpellerList)) << ::std::endl;
         }
      }

      if(m_ftsAutoCompletionCategoryForNonEmptySpeller.isValid())
      {
         ::std::vector<AutoCompletionCategoryType> ftsAutoCompletionCategoryForNonEmptySpeller =
                  m_ftsAutoCompletionCategoryForNonEmptySpeller.getValue();

         for(size_t nonEmptySpellerList = 0; nonEmptySpellerList < ftsAutoCompletionCategoryForNonEmptySpeller.size(); nonEmptySpellerList++)
         {
            stream << "payload for non empty speller[" << nonEmptySpellerList << "] - "
                   << (ftsAutoCompletionCategoryForNonEmptySpeller.at(nonEmptySpellerList)) << ::std::endl;
         }
      }

      if(m_ftsSearchFilterType.isValid())
      {
         ::std::vector<FtsSearchFilterType> ftsSearchFilterType =
                  m_ftsSearchFilterType.getValue();

         for(size_t ftsSearchFilterList = 0; ftsSearchFilterList < ftsSearchFilterType.size(); ftsSearchFilterList++)
         {
            stream << "payload for fts search filter[" << ftsSearchFilterList << "] - "
                   << (ftsSearchFilterType.at(ftsSearchFilterList)) << ::std::endl;
         }
      }
      return stream.str();
   }

   /**
    * If Non Valid, default config is considered.
    * If Valid , client config is considered.
    * If Valid + Empty Vector, Wrong usage of Config from client.
      -> In case of Categories w.r.t Empty/NonEmpty Speller, Empty list is provided.
      -> In case of Filter Type, no filtering will be done, everything will be returned.
    */
    ValidValue< ::std::vector<AutoCompletionCategoryType> > m_ftsAutoCompletionCategoryForEmptySpeller;
    ValidValue< ::std::vector<AutoCompletionCategoryType> > m_ftsAutoCompletionCategoryForNonEmptySpeller;
    ValidValue< ::std::vector<FtsSearchFilterType> >        m_ftsSearchFilterType;
};

class PoiIconInfos
{
public:
   enum ImageVariant
   {
      IMAGE_VARIANT__NORMAL,
      IMAGE_VARIANT__DISABLED,
      IMAGE_VARIANT__SELECTED,
      IMAGE_VARIANT__PRESSED,
      IMAGE_VARIANT__EDIT_NORMAL,
      IMAGE_VARIANT__EDIT_DISABLED,
      IMAGE_VARIANT__EDIT_SELECTED,
      IMAGE_VARIANT__EDIT_PRESSED
   };

   static ::std::string toString(ImageVariant imageVariant)
   {
      switch (imageVariant)
      {
         case IMAGE_VARIANT__NORMAL:
            return "IMAGE_VARIANT__NORMAL";
         case IMAGE_VARIANT__DISABLED:
            return "IMAGE_VARIANT__DISABLED";
         case IMAGE_VARIANT__SELECTED:
            return "IMAGE_VARIANT__SELECTED";
         case IMAGE_VARIANT__PRESSED:
            return "IMAGE_VARIANT__PRESSED";
         case IMAGE_VARIANT__EDIT_NORMAL:
            return "IMAGE_VARIANT__EDIT_NORMAL";
         case IMAGE_VARIANT__EDIT_DISABLED:
            return "IMAGE_VARIANT__EDIT_DISABLED";
         case IMAGE_VARIANT__EDIT_SELECTED:
            return "IMAGE_VARIANT__EDIT_SELECTED";
         case IMAGE_VARIANT__EDIT_PRESSED:
            return "IMAGE_VARIANT__EDIT_PRESSED";
         default:
            ::std::stringstream stream;
            stream << "IMAGE_VARIANT__<" << static_cast<unsigned int>(imageVariant) << ">";
            return stream.str();
      }
   }

   static ::std::string toString(const ::std::vector<ImageVariant>& variants)
   {
      ::std::stringstream stream;
      for(size_t variantCount = 0 ; variantCount < variants.size(); ++variantCount )
      {
         stream << "Payload for Image Variants[" << variantCount << "] - "
                           << toString(variants[variantCount]) << ::std::endl;
      }
      return stream.str();
   }

   static ::std::string toString(const ::std::vector<PoiCategoryId>& poiCategories)
   {
      ::std::stringstream stream;
      for(size_t categoryCount = 0 ; categoryCount < poiCategories.size(); ++categoryCount )
      {
         stream << "Payload for PoiCategories [" << categoryCount << "] - "
                           << (poiCategories[categoryCount]) << ::std::endl;
      }
      return stream.str();
   }

  typedef std::map< ImageVariant, Image> ImageVariants;
  typedef std::map< PoiCategoryId, ImageVariants> PoiCategoryIcons;

   PoiIconInfos()
      : m_categoryIcons()
   {}

   explicit PoiIconInfos(std::map< PoiCategoryId, ImageVariants> categoryIcons)
      : m_categoryIcons(categoryIcons)
   {}

   void setPoiCategoryIcons(const PoiCategoryIcons& categoryIcons)
   {
      m_categoryIcons = categoryIcons;
   }

   const PoiCategoryIcons& getPoiCategoryIcons() const
   {
      return m_categoryIcons;
   }

   const Image& getPoiCategoryIcon(PoiCategoryId poiCategoryId, ImageVariant variant) const
   {
      PoiCategoryIcons::const_iterator itr = m_categoryIcons.find(poiCategoryId);
      if(itr != m_categoryIcons.end())
      {
         ImageVariants::const_iterator iter = itr->second.find(variant);
         if(iter != itr->second.end())
         {
            return iter->second;
         }
      }
      static const Image image;
      return image;
   }

   bool operator==(const PoiIconInfos& other) const
   {
      return (m_categoryIcons == other.m_categoryIcons);
   }
   bool operator!=(const PoiIconInfos& other) const
   {
      return !(*this == other);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "PoiShortcutIconInfos Payload:" << ::std::endl;
      for ( PoiCategoryIcons::const_iterator itrCategory = m_categoryIcons.begin(); itrCategory != m_categoryIcons.end(); ++itrCategory )
      {
         stream << "Poi Category Id : " << itrCategory->first << ::std::endl;
         const ImageVariants& imageVariants = itrCategory->second;
         for ( ImageVariants::const_iterator itrVariant = imageVariants.begin(); itrVariant != imageVariants.end(); ++itrVariant )
         {
            stream << "Variant : " << toString(itrVariant->first) << ::std::endl;
            stream << itrVariant->second.toString();
         }
      }
      return stream.str();
   }

private:
   PoiCategoryIcons m_categoryIcons;
};
//<-- INF4CV
/**
 * Informs about the status of USB POIs import
 */
class ImportUserPoisStatus
{
public:
   enum ImportStatus
   {
      IMPORT_STATUS__NOT_STARTED,
      IMPORT_STATUS__SUCCESS,
      IMPORT_STATUS__IN_PROGRESS,
      IMPORT_STATUS__ERROR
   };

   enum ErrorCode
   {
      ERROR_CODE__MAX_NUM_OF_PARALLEL_REQUESTS_EXCEEDED,
      ERROR_CODE__SERVICE_NOT_AVAILABLE,
      ERROR_CODE__INTERNAL_ERROR,
      ERROR_CODE__ILLEGAL_VALUE
   };

   static ::std::string toString(ErrorCode error)
   {
      switch (error)
      {
         case ERROR_CODE__MAX_NUM_OF_PARALLEL_REQUESTS_EXCEEDED:
            return "ERROR_CODE__MAX_NUM_OF_PARALLEL_REQUESTS_EXCEEDED";
         case ERROR_CODE__SERVICE_NOT_AVAILABLE:
            return "ERROR_CODE__SERVICE_NOT_AVAILABLE";
         case ERROR_CODE__INTERNAL_ERROR:
            return "ERROR_CODE__INTERNAL_ERROR";
         case ERROR_CODE__ILLEGAL_VALUE:
            return "ERROR_CODE__ILLEGAL_VALUE";
         default:
            ::std::stringstream stream;
            stream << "ERROR__<" << static_cast<unsigned int>(error) << ">";
            return stream.str();
      }
   }

   ImportUserPoisStatus()
      : m_ImportStatus(IMPORT_STATUS__NOT_STARTED)
      , m_numOfFilesProcessed(0)
      , m_numOfFilesImported(0)
      , m_numOfTotalFiles(0)
   {}

   ImportUserPoisStatus(
      ImportStatus ImportStatus,
      ValidValue<ErrorCode> errorCode = ValidValue<ErrorCode>(),
      uint32_t numOfFilesProcessed = 0,
      uint32_t numOfFilesImported = 0,
      uint32_t numOfTotalFiles = 0
   )
   : m_ImportStatus(ImportStatus)
   , m_errorCode(errorCode)
   , m_numOfFilesProcessed(numOfFilesProcessed)
   , m_numOfFilesImported(numOfFilesImported)
   , m_numOfTotalFiles(numOfTotalFiles)
   {}

   const ImportStatus& getStatus() const
   {
      return m_ImportStatus;
   }

   const ValidValue<ErrorCode>& getErrorCode() const
   {
      return m_errorCode;
   }

   void setNumOfFilesProcessed(unsigned int numOfFilesProcessed)
   {
      m_numOfFilesProcessed = numOfFilesProcessed;
   }

   uint32_t getNumOfFilesProcessed() const
   {
      return m_numOfFilesProcessed;
   }

   void setNumOfFilesImported(unsigned int numOfFilesImported)
   {
      m_numOfFilesImported = numOfFilesImported;
   }

   uint32_t getNumOfFilesImported() const
   {
      return m_numOfFilesImported;
   }

   void setNumOfTotalFiles(unsigned int numOfTotalFiles)
   {
      m_numOfTotalFiles = numOfTotalFiles;
   }

   uint32_t getNumOfTotalFiles() const
   {
      return m_numOfTotalFiles;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("ImportUserPoisStatus payload:\n");
      stream << ::std::endl << "Import Status = ";
      if (m_ImportStatus == IMPORT_STATUS__NOT_STARTED)
      {
         stream << "NOT STARTED" << ::std::endl;
      }
      else if (m_ImportStatus == IMPORT_STATUS__SUCCESS)
      {
         stream << "SUCCESS" << ::std::endl;
      }
      else if (m_ImportStatus == IMPORT_STATUS__IN_PROGRESS)
      {
         stream << "IN_PROGRESS" << ::std::endl;
      }
      else if (m_ImportStatus == IMPORT_STATUS__ERROR)
      {
         stream << "ERROR" << ::std::endl;
         stream << "Error Code = " << ImportUserPoisStatus::toString(m_errorCode.getValue()) << ::std::endl;
      }
      else
      {
         stream << "UNKNOWN";
      }
      stream << "Number of files processed = " << m_numOfFilesProcessed << ::std::endl;
      stream << "Number of files imported = " << m_numOfFilesImported << ::std::endl;
      stream << "Total number of files = " << m_numOfTotalFiles << ::std::endl;

      return stream.str();
   }

private:
   ImportStatus           m_ImportStatus;
   ValidValue<ErrorCode>  m_errorCode;
   uint32_t               m_numOfFilesProcessed;
   uint32_t               m_numOfFilesImported;
   uint32_t               m_numOfTotalFiles;
};
//->

struct ResumeFreeTextSearchStatusInfo
{
    ResumeFreeTextSearchStatusInfo():
      m_status(false)
   {
   }

    ResumeFreeTextSearchStatusInfo(bool status, LocationContext context)
      :m_status(status),
       m_context(context)
   {
   }

   bool operator == (const ResumeFreeTextSearchStatusInfo& other) const
   {
      return (this->m_status == other.m_status)
             && (this->m_context == other.m_context);
   }

   bool operator != (const ResumeFreeTextSearchStatusInfo& other) const
   {
      return (!(*this == other));
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "ResumeFreeTextSearchStatusInfo payload:" << ::std::endl;
      stream << "   m_status       =" << (m_status? "true" : "false") << ::std::endl;
      stream << "   m_context      =" << m_context.toString() <<::std::endl;

      return stream.str();
   }

   bool                m_status;
   LocationContext     m_context;
};

} // namespace navmiddleware

#endif  // PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_LOCATIONINFOS_H_
