/**
* @copyright (c) 2015-2022 Robert Bosch Car Multimedia GmbH
* @addtogroup NavMiddleware
* @{
*/

#ifndef PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_INFOTYPES_H_
#define PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_INFOTYPES_H_

#include <stdint.h>
#include <algorithm>
#include <map>
#include <set>
#include <sstream>
#include <string>
#include <vector>

namespace navmiddleware
{
// defining elementIds for different type of elements
typedef uint64_t LinkId;
typedef uint64_t NamedObjectId;
typedef uint64_t DestinationMemoryId;
typedef uint32_t SelectionCriterion;
typedef uint32_t PoiCategoryId;
typedef uint32_t NdsNamedObjectId;
typedef uint16_t NdsUpdateRegionSpecifier;
typedef ::std::vector<uint8_t> LocationId;
typedef uint64_t AvoidAreaId;

/**
 *  @brief Unique requestId which is returned to HMI
 *  as an asynchronous token for the interface calls
 */
typedef uint32_t RequestId;

enum SafetyCameraType
{
   SAFETY_CAMERA_TYPE__UNKNOWN,
   SAFETY_CAMERA_TYPE__FIXED_SPEED_CAMERA,
   SAFETY_CAMERA_TYPE__FIXED_SPEED_AND_RED_LIGHT_CAMERA,
   SAFETY_CAMERA_TYPE__RED_LIGHT_CAMERA,
   SAFETY_CAMERA_TYPE__MOBILE_SPEED_CAMERA,
   SAFETY_CAMERA_TYPE__AVERAGE_SPEED_ZONE,
   SAFETY_CAMERA_TYPE__MOBILE_SPEED_HOTSPOT_ZONE_CAMERA,
   SAFETY_CAMERA_TYPE__SPEED_ENFORCEMENT_ZONE,
   SAFETY_CAMERA_TYPE__RESTRICTED_AREA_CAMERA,
   SAFETY_CAMERA_TYPE__ACCIDENT_BLACK_SPOT_ZONE,
   SAFETY_CAMERA_TYPE__RAILWAY_CROSSING,
   SAFETY_CAMERA_TYPE__AUTOMATIC_NUMBER_PLATE_RECOGNITION_CAMERA,
   SAFETY_CAMERA_TYPE__BUS_LANE_CAMERA,
   SAFETY_CAMERA_TYPE__CONGESTION_CHARGE_CAMERA,
   SAFETY_CAMERA_TYPE__HIGH_OCCUPANCY_VEHICLE_LANE_CAMERA,
   SAFETY_CAMERA_TYPE__INTER_VEHICLE_DISTANCE_CAMERA,
   SAFETY_CAMERA_TYPE__LEFT_TURN_CAMERA,
   SAFETY_CAMERA_TYPE__RIGHT_TURN_CAMERA,
   SAFETY_CAMERA_TYPE__U_TURN_CAMERA,
   SAFETY_CAMERA_TYPE__RED_LIGHT_AND_U_TURN_CAMERA,
   SAFETY_CAMERA_TYPE__PARKING_CAMERA,
   SAFETY_CAMERA_TYPE__STOP_SIGN_CAMERA,
   SAFETY_CAMERA_TYPE__TOLL_BOOTH_CAMERA,
   SAFETY_CAMERA_TYPE__WEIGHT_LIMIT_CAMERA,
   SAFETY_CAMERA_TYPE__SPEED_BUMP,
   SAFETY_CAMERA_TYPE__RISK_ZONE,
   SAFETY_CAMERA_TYPE__DANGER_ZONE,
   SAFETY_CAMERA_TYPE__SCHOOL_ZONE,
   SAFETY_CAMERA_TYPE__NOISE_POLLUTION_CAMERA,
   SAFETY_CAMERA_TYPE__PECCANCY_CAMERA,
   SAFETY_CAMERA_TYPE__SPECIAL_LANE_CAMERA,
   SAFETY_CAMERA_TYPE__FAKE_CAMERA,
   SAFETY_CAMERA_TYPE__OTHER,
   SAFETY_CAMERA_TYPE__TOLL_ROAD_CAMERA,
   SAFETY_CAMERA_TYPE__TEST_CODE,
   SAFETY_CAMERA_TYPE__FREE_TEXT_ZONE,
   SAFETY_CAMERA_TYPE__GENERAL_VIGILANCE_ZONE,
   SAFETY_CAMERA_TYPE__AVERAGE_SPEED_START,
   SAFETY_CAMERA_TYPE__AVERAGE_SPEED_END
};

static ::std::string toString(SafetyCameraType safetyCameraType)
{
   switch (safetyCameraType)
   {
   case SAFETY_CAMERA_TYPE__UNKNOWN:
      return "SAFETY_CAMERA_TYPE__UNKNOWN";
   case SAFETY_CAMERA_TYPE__FIXED_SPEED_CAMERA:
      return "SAFETY_CAMERA_TYPE__FIXED_SPEED_CAMERA";
   case SAFETY_CAMERA_TYPE__FIXED_SPEED_AND_RED_LIGHT_CAMERA:
      return "SAFETY_CAMERA_TYPE__FIXED_SPEED_AND_RED_LIGHT_CAMERA";
   case SAFETY_CAMERA_TYPE__RED_LIGHT_CAMERA:
      return "SAFETY_CAMERA_TYPE__RED_LIGHT_CAMERA";
   case SAFETY_CAMERA_TYPE__MOBILE_SPEED_CAMERA:
      return "SAFETY_CAMERA_TYPE__MOBILE_SPEED_CAMERA";
   case SAFETY_CAMERA_TYPE__AVERAGE_SPEED_ZONE:
      return "SAFETY_CAMERA_TYPE__AVERAGE_SPEED_ZONE";
   case SAFETY_CAMERA_TYPE__MOBILE_SPEED_HOTSPOT_ZONE_CAMERA:
      return "SAFETY_CAMERA_TYPE__MOBILE_SPEED_HOTSPOT_ZONE_CAMERA";
   case SAFETY_CAMERA_TYPE__SPEED_ENFORCEMENT_ZONE:
      return "SAFETY_CAMERA_TYPE__SPEED_ENFORCEMENT_ZONE";
   case SAFETY_CAMERA_TYPE__RESTRICTED_AREA_CAMERA:
      return "SAFETY_CAMERA_TYPE__RESTRICTED_AREA_CAMERA";
   case SAFETY_CAMERA_TYPE__ACCIDENT_BLACK_SPOT_ZONE:
      return "SAFETY_CAMERA_TYPE__ACCIDENT_BLACK_SPOT_ZONE";
   case SAFETY_CAMERA_TYPE__RAILWAY_CROSSING:
      return "SAFETY_CAMERA_TYPE__RAILWAY_CROSSING";
   case SAFETY_CAMERA_TYPE__AUTOMATIC_NUMBER_PLATE_RECOGNITION_CAMERA:
      return "SAFETY_CAMERA_TYPE__AUTOMATIC_NUMBER_PLATE_RECOGNITION_CAMERA";
   case SAFETY_CAMERA_TYPE__BUS_LANE_CAMERA:
      return "SAFETY_CAMERA_TYPE__BUS_LANE_CAMERA";
   case SAFETY_CAMERA_TYPE__CONGESTION_CHARGE_CAMERA:
      return "SAFETY_CAMERA_TYPE__CONGESTION_CHARGE_CAMERA";
   case SAFETY_CAMERA_TYPE__HIGH_OCCUPANCY_VEHICLE_LANE_CAMERA:
      return "SAFETY_CAMERA_TYPE__HIGH_OCCUPANCY_VEHICLE_LANE_CAMERA";
   case SAFETY_CAMERA_TYPE__INTER_VEHICLE_DISTANCE_CAMERA:
      return "SAFETY_CAMERA_TYPE__INTER_VEHICLE_DISTANCE_CAMERA";
   case SAFETY_CAMERA_TYPE__LEFT_TURN_CAMERA:
      return "SAFETY_CAMERA_TYPE__LEFT_TURN_CAMERA";
   case SAFETY_CAMERA_TYPE__RIGHT_TURN_CAMERA:
      return "SAFETY_CAMERA_TYPE__RIGHT_TURN_CAMERA";
   case SAFETY_CAMERA_TYPE__U_TURN_CAMERA:
      return "SAFETY_CAMERA_TYPE__U_TURN_CAMERA";
   case SAFETY_CAMERA_TYPE__RED_LIGHT_AND_U_TURN_CAMERA:
      return "SAFETY_CAMERA_TYPE__RED_LIGHT_AND_U_TURN_CAMERA";
   case SAFETY_CAMERA_TYPE__PARKING_CAMERA:
      return "SAFETY_CAMERA_TYPE__PARKING_CAMERA";
   case SAFETY_CAMERA_TYPE__STOP_SIGN_CAMERA:
      return "SAFETY_CAMERA_TYPE__STOP_SIGN_CAMERA";
   case SAFETY_CAMERA_TYPE__TOLL_BOOTH_CAMERA:
      return "SAFETY_CAMERA_TYPE__TOLL_BOOTH_CAMERA";
   case SAFETY_CAMERA_TYPE__WEIGHT_LIMIT_CAMERA:
      return "SAFETY_CAMERA_TYPE__WEIGHT_LIMIT_CAMERA";
   case SAFETY_CAMERA_TYPE__SPEED_BUMP:
      return "SAFETY_CAMERA_TYPE__SPEED_BUMP";
   case SAFETY_CAMERA_TYPE__RISK_ZONE:
      return "SAFETY_CAMERA_TYPE__RISK_ZONE";
   case SAFETY_CAMERA_TYPE__DANGER_ZONE:
      return "SAFETY_CAMERA_TYPE__DANGER_ZONE";
   case SAFETY_CAMERA_TYPE__SCHOOL_ZONE:
      return "SAFETY_CAMERA_TYPE__SCHOOL_ZONE";
   case SAFETY_CAMERA_TYPE__NOISE_POLLUTION_CAMERA:
      return "SAFETY_CAMERA_TYPE__NOISE_POLLUTION_CAMERA";
   case SAFETY_CAMERA_TYPE__PECCANCY_CAMERA:
      return "SAFETY_CAMERA_TYPE__PECCANCY_CAMERA";
   case SAFETY_CAMERA_TYPE__SPECIAL_LANE_CAMERA:
      return "SAFETY_CAMERA_TYPE__SPECIAL_LANE_CAMERA";
   case SAFETY_CAMERA_TYPE__FAKE_CAMERA:
      return "SAFETY_CAMERA_TYPE__FAKE_CAMERA";
   case SAFETY_CAMERA_TYPE__OTHER:
      return "SAFETY_CAMERA_TYPE__OTHER";
   case SAFETY_CAMERA_TYPE__TOLL_ROAD_CAMERA:
      return "SAFETY_CAMERA_TYPE__TOLL_ROAD_CAMERA";
   case SAFETY_CAMERA_TYPE__TEST_CODE:
      return "SAFETY_CAMERA_TYPE__TEST_CODE";
   case SAFETY_CAMERA_TYPE__FREE_TEXT_ZONE:
      return "SAFETY_CAMERA_TYPE__FREE_TEXT_ZONE";
   case SAFETY_CAMERA_TYPE__GENERAL_VIGILANCE_ZONE:
      return "SAFETY_CAMERA_TYPE__GENERAL_VIGILANCE_ZONE";
   case SAFETY_CAMERA_TYPE__AVERAGE_SPEED_START:
      return "SAFETY_CAMERA_TYPE__AVERAGE_SPEED_START";
   case SAFETY_CAMERA_TYPE__AVERAGE_SPEED_END:
      return "SAFETY_CAMERA_TYPE__AVERAGE_SPEED_END";
   default:
      ::std::stringstream stream;
      stream << "SAFETY_CAMERA_TYPE__<" << static_cast<unsigned int>(safetyCameraType) << ">";
      return stream.str();
   }
}

enum ConnectedClusterType
{
   CONNECTED_CLUSTER_TYPE__NONE,
   CONNECTED_CLUSTER_TYPE__MCAN_CLUSTER_TYPE_1, // base functions
   CONNECTED_CLUSTER_TYPE__MCAN_CLUSTER_TYPE_2 ,// enlarge intersection view with calculated vector_data
   CONNECTED_CLUSTER_TYPE__SOME_IP_CLUSTER    // stream view, no MCAN cluster connected
};

inline ::std::string toString(ConnectedClusterType connectedClusterType)
{
   switch (connectedClusterType)
   {
   case CONNECTED_CLUSTER_TYPE__NONE:
      return "CONNECTED_CLUSTER_TYPE__NONE";
   case CONNECTED_CLUSTER_TYPE__MCAN_CLUSTER_TYPE_1:
      return "CONNECTED_CLUSTER_TYPE__MCAN_CLUSTER_TYPE_1";
   case CONNECTED_CLUSTER_TYPE__MCAN_CLUSTER_TYPE_2:
      return "CONNECTED_CLUSTER_TYPE__MCAN_CLUSTER_TYPE_2";
   case CONNECTED_CLUSTER_TYPE__SOME_IP_CLUSTER:
      return "CONNECTED_CLUSTER_TYPE__SOME_IP_CLUSTER";
   default:
      ::std::stringstream stream;
      stream << "CONNECTED_CLUSTER_TYPE__<" << static_cast<unsigned int>(connectedClusterType) << ">";
      return stream.str();
   }
}

class PoiId
{
public:
   PoiId()
      : m_sourceId(0), m_id(0LL)
   {
   }
   PoiId(uint8_t sourceId, uint64_t id)
      : m_sourceId(sourceId), m_id(id)
   {
   }
   bool operator==(const PoiId& other) const
   {
      return (this->m_sourceId == other.m_sourceId)
             && (this->m_id == other.m_id);
   }
   bool operator!=(const PoiId& other) const
   {
      return !(*this == other);
   }
   bool isValid() const
   {
      return (m_sourceId != 0);
   }
   uint8_t getSourceId() const
   {
      return m_sourceId;
   }
   uint64_t getId() const
   {
      return m_id;
   }
   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "PoiId(sourceId=" << static_cast<int>(m_sourceId) << ",id=" << m_id <<")";
      return stream.str();
   }

private:
   uint8_t m_sourceId;
   uint64_t m_id;
};

inline ::std::ostream& operator<<(::std::ostream& stream, const PoiId& poiId)
{
   return stream << poiId.toString();
}

/**
 * @brief Encapsulates values that can be invalid.
 *
 * The validation state of the value can be checked by isValid().
 * Only if the value isValid() it should be get by a call to getValue().
 * If value !isValid() an invalid handling has to be implemented at caller side.
 */
template <typename VALUETYPE>
class ValidValue
{
public:

   ValidValue()
      : m_isValid(false)
      , m_value(VALUETYPE())
   {}

   explicit ValidValue(const VALUETYPE& value)
      : m_isValid(true)
      , m_value(value)
   {}

   virtual ~ValidValue() {}

   /**
    * @brief Returns if the stored value is valid.
    */
   bool isValid() const
   {
      return m_isValid;
   }

   /**
    * @brief Returns the stored value.
    */
   VALUETYPE getValue() const
   {
      return m_value;
   }

   /**
    * @brief Returns the stored value, if stored value is valid. Otherwise the given default value is returned.
    */
   VALUETYPE getValueOr(const VALUETYPE& defaultValue) const
   {
      if(m_isValid)
      {
         return m_value;
      }
      else
      {
         return defaultValue;
      }
   }

   /**
    * @brief Invalidates the stored value.
    */
   void invalidate()
   {
      m_isValid = false;
   }

   /**
    * @brief Sets the stored value and switches valid state to true.
    */
   void setValue(const VALUETYPE& value)
   {
      m_value = value;
      setValid();
   }

   bool operator==(const ValidValue<VALUETYPE>& rhs) const
   {
      return (!m_isValid && !rhs.m_isValid) || (m_isValid && rhs.m_isValid && (m_value == rhs.m_value));
   }

   bool operator != (const ValidValue<VALUETYPE>& rhs) const
   {
      return !(*this == rhs);
   }

   ValidValue<VALUETYPE>& operator=(const ValidValue<VALUETYPE>& rhs)
   {
      if (&rhs != this)
      {
         m_isValid = rhs.m_isValid;
         m_value = rhs.m_value;
      }
      return *this;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "m_isValid: " << m_isValid;
      if(isValid())
      {
         stream << ", m_value: " << m_value;
      }
      return stream.str();
   }

private:

   void setValid()
   {
      m_isValid = true;
   }

   bool m_isValid;
   VALUETYPE m_value;
};

// specialization for uint32 type; fixes some failing traffic tests
template<> inline ValidValue<unsigned int>::ValidValue() : m_isValid(false), m_value(0) {}
template<> inline void ValidValue<unsigned int>::invalidate()
{
   m_isValid = false;
   m_value = 0;
}

// specialization for double type; fixes some failing traffic tests
template<> inline ValidValue<double>::ValidValue() : m_isValid(false), m_value(0.0) {}
template<> inline void ValidValue<double>::invalidate()
{
   m_isValid = false;
   m_value = 0.0;
}

template<typename VALUETYPE> inline ::std::string toString(const ValidValue<VALUETYPE>& validValue)
{
   if ( validValue.isValid() )
   {
      return validValue.getValue().toString();
   }
   else
   {
      return "invalid";
   }
}

enum DistanceUnitSystem
{
   DISTANCEUNITSYSTEM_METRIC,
   DISTANCEUNITSYSTEM_IMPERIAL,
   DISTANCEUNITSYSTEM_BRITISH_IMPERIAL
};

enum DistanceUnit
{
   DISTANCEUNIT_METERS = 0,
   DISTANCEUNIT_KILOMETERS,
   DISTANCEUNIT_MILES,
   DISTANCEUNIT_FEET,
   DISTANCEUNIT_YARD
};

inline ::std::string toString(DistanceUnit distanceUnit)
{
   switch (distanceUnit)
   {
   case DISTANCEUNIT_METERS:
      return "DISTANCEUNIT_METERS";
   case DISTANCEUNIT_KILOMETERS:
      return "DISTANCEUNIT_KILOMETERS";
   case DISTANCEUNIT_MILES:
      return "DISTANCEUNIT_MILES";
   case DISTANCEUNIT_FEET:
      return "DISTANCEUNIT_FEET";
   case DISTANCEUNIT_YARD:
      return "DISTANCEUNIT_YARD";
   default:
      ::std::stringstream stream;
      stream << "DistanceUnit__<" << static_cast<unsigned int>(distanceUnit) << ">";
      return stream.str();
   }
}

enum DateFormat
{
   DATEFORMAT_UNSET = 0,
   DATEFORMAT_DD_MM_YYYY_SLASH,
   DATEFORMAT_MM_DD_YYYY_SLASH,
   DATEFORMAT_YYYY_MM_DD_SLASH,
   DATEFORMAT_DD_MM_YYYY_DASH,
   DATEFORMAT_MM_DD_YYYY_DASH,
   DATEFORMAT_YYYY_MM_DD_DASH,
   DATEFORMAT_DD_MM_YYYY_DOT,
   DATEFORMAT_YYYY_MM_DD_JPN,
   DATEFORMAT_YYYY_MM_DD_KOR
};

enum TimeFormat
{
   TIMEFORMAT_UNSET = 0,
   TIMEFORMAT_12HOUR,
   TIMEFORMAT_24HOUR
};

enum VehicleType
{
   VEHICLE_TYPE__UNKNOWN,
   VEHICLE_TYPE__HBZ,
   VEHICLE_TYPE__HZC,
   VEHICLE_TYPE__LBZ,
   VEHICLE_TYPE__LJH,
   VEHICLE_TYPE__LJL,
   VEHICLE_TYPE__L02D,
   VEHICLE_TYPE__B02E,
   VEHICLE_TYPE__P02F,
   VEHICLE_TYPE__GD1A,
   VEHICLE_TYPE__D53H,
   VEHICLE_TYPE__L12F,
   VEHICLE_TYPE__L12M,
   VEHICLE_TYPE__B12P,
   VEHICLE_TYPE__P13A,
   VEHICLE_TYPE__L21B,
   VEHICLE_TYPE__P32R,
   VEHICLE_TYPE__P32S,
   VEHICLE_TYPE__J32U,
   VEHICLE_TYPE__P33A,
   VEHICLE_TYPE__P33B,
   VEHICLE_TYPE__P33C,
   VEHICLE_TYPE__P42M,
   VEHICLE_TYPE__L42N,
   VEHICLE_TYPE__L42P,
   VEHICLE_TYPE__P42R,
   VEHICLE_TYPE__L53H,
   VEHICLE_TYPE__H60A,
   VEHICLE_TYPE__P61G,
   VEHICLE_TYPE__H61L,
   VEHICLE_TYPE__P71A,
   VEHICLE_TYPE__XC2C,
   VEHICLE_TYPE__B13B,
   VEHICLE_TYPE__PNE,
   VEHICLE_TYPE__PZ1A,
   VEHICLE_TYPE__J32V,
   VEHICLE_TYPE__C53L,
   VEHICLE_TYPE__P60A,
   VEHICLE_TYPE__TRITON,
   VEHICLE_TYPE__B07,
   VEHICLE_TYPE__B10,
   VEHICLE_TYPE__BFB,
   VEHICLE_TYPE__BJA,
   VEHICLE_TYPE__FFK,
   VEHICLE_TYPE__FJK,
   VEHICLE_TYPE__HHA,
   VEHICLE_TYPE__HJB,
   VEHICLE_TYPE__HJD,
   VEHICLE_TYPE__HJE,
   VEHICLE_TYPE__JFC,
   VEHICLE_TYPE__KFB,
   VEHICLE_TYPE__KFK,
   VEHICLE_TYPE__KFD,
   VEHICLE_TYPE__KJK,
   VEHICLE_TYPE__LFD,
   VEHICLE_TYPE__LLF,
   VEHICLE_TYPE__LJC,
   VEHICLE_TYPE__J82,
   VEHICLE_TYPE__F82,
   VEHICLE_TYPE__AS1,
   VEHICLE_TYPE__W21A,
   VEHICLE_TYPE__W21B,
   VEHICLE_TYPE__P42Q,
   VEHICLE_TYPE__OTHER
};

enum RegionType
{
   REGIONTYPE_UNKNOWN = 255,
   REGIONTYPE_USA = 0,
   REGIONTYPE_CAN = 1,
   REGIONTYPE_MEX = 2,
   REGIONTYPE_UK = 3,
   REGIONTYPE_TKY = 4,
   REGIONTYPE_RUS = 5,
   REGIONTYPE_OTH_EUR = 6,
   REGIONTYPE_PRC = 7,
   REGIONTYPE_TWN = 8,
   REGIONTYPE_HKG_MACAU = 9,
   REGIONTYPE_GCC = 10,
   REGIONTYPE_EGP = 11,
   REGIONTYPE_ASR_NZR = 12,
   REGIONTYPE_BRA = 13,
   REGIONTYPE_AGT = 14,
   REGIONTYPE_OTH_LAC = 15,
   REGIONTYPE_SAF = 16,
   REGIONTYPE_THI = 17,
   REGIONTYPE_SGP = 18,
   REGIONTYPE_MLY = 19,
   REGIONTYPE_BRN = 20,
   REGIONTYPE_INN = 21,
   REGIONTYPE_VNM = 22,
   REGIONTYPE_PHL = 23,
   REGIONTYPE_IND = 24,
   REGIONTYPE_JPN = 25,
   REGIONTYPE_KOR = 26,
   REGIONTYPE_OTH_GOM = 27,
   REGIONTYPE_ISR = 28
};

enum LanguageCode
{
   LANGUAGE_CODE__UNKNOWN = 0,
   LANGUAGE_CODE__ALBANESE = 1,
   LANGUAGE_CODE__ARABIC,
   LANGUAGE_CODE__BULGARIEN,
   LANGUAGE_CODE__CHINESE_CANTONESE_SIMPLIFIED_CHINESE_CHARACTER,
   LANGUAGE_CODE__CHINESE_CANTONESE_TRADITIONAL_CHINESE_CHARACTER,
   LANGUAGE_CODE__CHINESE_MANDARIN_SIMPLIFIED_CHINESE_CHARACTER,
   LANGUAGE_CODE__CROATIAN,
   LANGUAGE_CODE__CZECH,
   LANGUAGE_CODE__DANISH,
   LANGUAGE_CODE__DUTCH,
   LANGUAGE_CODE__ENGLISH_AUSTRALIAN,
   LANGUAGE_CODE__ENGLISH_CANADIAN,
   LANGUAGE_CODE__ENGLISH_INDIA,
   LANGUAGE_CODE__ENGLISH_UK,
   LANGUAGE_CODE__ENGLISH_US,
   LANGUAGE_CODE__ENGLISH_US_FOR_JPN,
   LANGUAGE_CODE__ENGLISH_US_FOR_PRC,
   LANGUAGE_CODE__ESTONIAN,
   LANGUAGE_CODE__FINNISH,
   LANGUAGE_CODE__FLEMISH,
   LANGUAGE_CODE__FRENCH,
   LANGUAGE_CODE__FRENCH_CANADIAN,
   LANGUAGE_CODE__GERMAN,
   LANGUAGE_CODE__GREEK,
   LANGUAGE_CODE__HEBREW,
   LANGUAGE_CODE__HINDI,
   LANGUAGE_CODE__HUNGARIAN,
   LANGUAGE_CODE__INDONESIAN,
   LANGUAGE_CODE__ITALIAN,
   LANGUAGE_CODE__JAPANESE,
   LANGUAGE_CODE__KOREAN,
   LANGUAGE_CODE__LATVIAN,
   LANGUAGE_CODE__LITHUANIAN,
   LANGUAGE_CODE__MALAY,
   LANGUAGE_CODE__NORWEGIAN,
   LANGUAGE_CODE__PERSIAN_FARSI_IRANIAN,
   LANGUAGE_CODE__PHILIPPINES,
   LANGUAGE_CODE__POLISH,
   LANGUAGE_CODE__PORTUGUESE,
   LANGUAGE_CODE__PORTUGUESE_BRAZILIAN,
   LANGUAGE_CODE__ROMANIAN,
   LANGUAGE_CODE__RUSSIAN,
   LANGUAGE_CODE__SERBIAN,
   LANGUAGE_CODE__SLOVAKIAN,
   LANGUAGE_CODE__SLOVENIAN,
   LANGUAGE_CODE__SPANISH,
   LANGUAGE_CODE__SPANISH_LATIN_AMERICAN,
   LANGUAGE_CODE__SPANISH_MEXICAN,
   LANGUAGE_CODE__SWEDISH,
   LANGUAGE_CODE__TAIWANESE,
   LANGUAGE_CODE__THAI,
   LANGUAGE_CODE__TURKISH,
   LANGUAGE_CODE__UKRAINIAN,
   LANGUAGE_CODE__VIETNAMESE,
   LANGUAGE_CODE__UNSUPPORTED,
   LANGUAGE_CODE__UNUSED
};

enum RequiredFeatures
{
   RequiredFeatures__MAP_DISPLAY,
   RequiredFeatures__LOCATION_INPUT,
   RequiredFeatures__ROUTE_GUIDANCE
};

enum DirectionDescription
{
   DIRECTIONDESCRIPTION_NORTH = 0,
   DIRECTIONDESCRIPTION_NORTH_NORTH_EAST,
   DIRECTIONDESCRIPTION_NORTH_EAST,
   DIRECTIONDESCRIPTION_EAST_NORTH_EAST,
   DIRECTIONDESCRIPTION_EAST,
   DIRECTIONDESCRIPTION_EAST_SOUTH_EAST,
   DIRECTIONDESCRIPTION_SOUTH_EAST,
   DIRECTIONDESCRIPTION_SOUTH_SOUTH_EAST,
   DIRECTIONDESCRIPTION_SOUTH,
   DIRECTIONDESCRIPTION_SOUTH_SOUTH_WEST,
   DIRECTIONDESCRIPTION_SOUTH_WEST,
   DIRECTIONDESCRIPTION_WEST_SOUTH_WEST,
   DIRECTIONDESCRIPTION_WEST,
   DIRECTIONDESCRIPTION_WEST_NORTH_WEST,
   DIRECTIONDESCRIPTION_NORTH_WEST,
   DIRECTIONDESCRIPTION_NORTH_NORTH_WEST,
   DIRECTIONDESCRIPTION_SIZE
};

enum ExitPOICategory
{
   EXIT_POI_CATEGORY__RESTAURANT,
   EXIT_POI_CATEGORY__PETROL_STATION,
   EXIT_POI_CATEGORY__HOTEL,
   EXIT_POI_CATEGORY__ATM,
   EXIT_POI_CATEGORY__VEHICLE_REPAIR,
   EXIT_POI_CATEGORY__EV_CHARGING_STATION
};

inline ::std::string toString(ExitPOICategory exitPOICategory)
{
   switch (exitPOICategory)
   {
   case EXIT_POI_CATEGORY__RESTAURANT:
      return "EXIT_POI_CATEGORY__RESTAURANT";
   case EXIT_POI_CATEGORY__PETROL_STATION:
      return "EXIT_POI_CATEGORY__PETROL_STATION";
   case EXIT_POI_CATEGORY__HOTEL:
      return "EXIT_POI_CATEGORY__HOTEL";
   case EXIT_POI_CATEGORY__ATM:
      return "EXIT_POI_CATEGORY__ATM";
   case EXIT_POI_CATEGORY__VEHICLE_REPAIR:
      return "EXIT_POI_CATEGORY__VEHICLE_REPAIR";
   case EXIT_POI_CATEGORY__EV_CHARGING_STATION:
      return "EXIT_POI_CATEGORY__EV_CHARGING_STATION";
   default:
      ::std::stringstream stream;
      stream << "ExitPOICategory__<" << static_cast<unsigned int>(exitPOICategory) << ">";
      return stream.str();
   }
}

enum DestinationMemoryEntryCategory
{
   DESTINATION_MEMORY_ENTRY_CATEGORY__HOME,
   DESTINATION_MEMORY_ENTRY_CATEGORY__WORK,
   DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK,
   DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_GADGET,
   DESTINATION_MEMORY_ENTRY_CATEGORY__PERSONAL_EPOI,
   DESTINATION_MEMORY_ENTRY_CATEGORY__FAVORITE_WEATHER_ALERTS
};

enum WeatherReportType
{
   WEATHER_REPORT_TYPE__OVERVIEW,
   WEATHER_REPORT_TYPE__DAILY,
   WEATHER_REPORT_TYPE__SUBDAILY,
   WEATHER_REPORT_TYPE__TIMED,
   WEATHER_REPORT_TYPE__HISTORICAL
};

inline ::std::string toString(WeatherReportType weatherReportType)
{
   switch(weatherReportType)
   {
   case WEATHER_REPORT_TYPE__OVERVIEW:
      return "WEATHER_REPORT_TYPE__OVERVIEW";
   case WEATHER_REPORT_TYPE__DAILY:
      return "WEATHER_REPORT_TYPE__DAILY";
   case WEATHER_REPORT_TYPE__SUBDAILY:
      return "WEATHER_REPORT_TYPE__SUBDAILY";
   case WEATHER_REPORT_TYPE__TIMED:
      return "WEATHER_REPORT_TYPE__TIMED";
   case WEATHER_REPORT_TYPE__HISTORICAL:
      return "WEATHER_REPORT_TYPE__HISTORICAL";
   default:
      return "<unknown>";
   }
}

enum LocationType
{
   LOCATION_TYPE__CURRENT_VEHICLE_POSITION,
   LOCATION_TYPE__DESTINATION,
   LOCATION_TYPE__ARBITRARYLOCATION
};

inline ::std::string toString(LocationType locationType)
{
   switch(locationType)
   {
   case LOCATION_TYPE__CURRENT_VEHICLE_POSITION:
      return "LOCATION_TYPE__CURRENT_VEHICLE_POSITION";
   case LOCATION_TYPE__DESTINATION:
      return "LOCATION_TYPE__DESTINATION";
   case LOCATION_TYPE__ARBITRARYLOCATION:
      return "LOCATION_TYPE__ARBITRARYLOCATION";
   default:
      return "<unknown>";
   }
}

enum EntrySelectionStatus
{
   /** The entry is refreshed and valid to select */
   ENTRY_SELECTION_STATUS__VALID,
   /** The entry is expired and needs to be refreshed before selection*/
   ENTRY_SELECTION_STATUS__EXPIRED,
   /** The entry could not be refreshed, thus On-board Matching is necessary before selection */
   ENTRY_SELECTION_STATUS__ONBOARD_MATCHING_NEEDED,
   /** The entry could not be matched to an on-board entry, thus is not select-able */
   ENTRY_SELECTION_STATUS__ONBOARD_MATCHING_FAILED,
   /** The entry selection status is known */
   ENTRY_SELECTION_STATUS__UNKNOWN
};
inline ::std::string toString(EntrySelectionStatus entrySelectionStatus)
{
   switch (entrySelectionStatus)
   {
   case ENTRY_SELECTION_STATUS__VALID:
      return "ENTRY_SELECTION_STATUS__VALID";
   case ENTRY_SELECTION_STATUS__EXPIRED:
      return "ENTRY_SELECTION_STATUS__EXPIRED";
   case ENTRY_SELECTION_STATUS__ONBOARD_MATCHING_NEEDED:
      return "ENTRY_SELECTION_STATUS__ONBOARD_MATCHING_NEEDED";
   case ENTRY_SELECTION_STATUS__ONBOARD_MATCHING_FAILED:
      return "ENTRY_SELECTION_STATUS__ONBOARD_MATCHING_FAILED";
   case ENTRY_SELECTION_STATUS__UNKNOWN:
      return "ENTRY_SELECTION_STATUS__UNKNOWN";
   default:
      ::std::stringstream stream;
      stream << "ENTRY_SELECTION_STATUS__<" << static_cast<unsigned int>(entrySelectionStatus) << ">";
      return stream.str();
   }
}

enum UVIndex
{
   UVINDEX__UNKNOWN,
   UVINDEX__ONE_LOW,
   UVINDEX__TWO_LOW,
   UVINDEX__THREE_MODERATE,
   UVINDEX__FOUR_MODERATE,
   UVINDEX__FIVE_MODERATE,
   UVINDEX__SIX_HIGH,
   UVINDEX__SEVEN_HIGH,
   UVINDEX__EIGHT_VERY_HIGH,
   UVINDEX__NINE_VERY_HIGH,
   UVINDEX__TEN_VERY_HIGH,
   UVINDEX__ELEVEN_EXTREME
};

inline ::std::string toString(UVIndex uvIndex)
{
   switch(uvIndex)
   {
   case UVINDEX__UNKNOWN:
      return "UVINDEX__UNKNOWN";
   case UVINDEX__ONE_LOW:
      return "UVINDEX__ONE_LOW";
   case UVINDEX__TWO_LOW:
      return "UVINDEX__TWO_LOW";
   case UVINDEX__THREE_MODERATE:
      return "UVINDEX__THREE_MODERATE";
   case UVINDEX__FOUR_MODERATE:
      return "UVINDEX__FOUR_MODERATE";
   case UVINDEX__FIVE_MODERATE:
      return "UVINDEX__FIVE_MODERATE";
   case UVINDEX__SIX_HIGH:
      return "UVINDEX__SIX_HIGH";
   case UVINDEX__SEVEN_HIGH:
      return "UVINDEX__SEVEN_HIGH";
   case UVINDEX__EIGHT_VERY_HIGH:
      return "UVINDEX__EIGHT_VERY_HIGH";
   case UVINDEX__NINE_VERY_HIGH:
      return "UVINDEX__NINE_VERY_HIGH";
   case UVINDEX__TEN_VERY_HIGH:
      return "UVINDEX__TEN_VERY_HIGH";
   case UVINDEX__ELEVEN_EXTREME:
      return "UVINDEX__ELEVEN_EXTREME";
   default:
      return "<unknown>";
   }
}

enum CloudCoverage
{
   CLOUD_COVERAGE__UNKNOWN,
   CLOUD_COVERAGE__CLEAR,
   CLOUD_COVERAGE__FEWCLOUDS,
   CLOUD_COVERAGE__PARTLYCLOUDS,
   CLOUD_COVERAGE__CLOUDY,
   CLOUD_COVERAGE__VERYCLOUDY,
   CLOUD_COVERAGE__OVERCAST
};

inline ::std::string toString(CloudCoverage cloudCoverage)
{
   switch(cloudCoverage)
   {
   case CLOUD_COVERAGE__UNKNOWN:
      return "CLOUD_COVERAGE__UNKNOWN";
   case CLOUD_COVERAGE__CLEAR:
      return "CLOUD_COVERAGE__CLEAR";
   case CLOUD_COVERAGE__FEWCLOUDS:
      return "CLOUD_COVERAGE__FEWCLOUDS";
   case CLOUD_COVERAGE__PARTLYCLOUDS:
      return "CLOUD_COVERAGE__PARTLYCLOUDS";
   case CLOUD_COVERAGE__CLOUDY:
      return "CLOUD_COVERAGE__CLOUDY";
   case CLOUD_COVERAGE__VERYCLOUDY:
      return "CLOUD_COVERAGE__VERYCLOUDY";
   case CLOUD_COVERAGE__OVERCAST:
      return "CLOUD_COVERAGE__OVERCAST";
   default:
      return "<unknown>";
   }
}

enum WeatherSituation
{
   WEATHER_SITUATION__UNKNOWN,
   WEATHER_SITUATION__CLEAR_SKY,
   WEATHER_SITUATION__A_FEW_CLOUDS,
   WEATHER_SITUATION__PARTLY_CLOUDY,
   WEATHER_SITUATION__PARTLY_CLOUDY_SLIGHT_RAIN,
   WEATHER_SITUATION__PARTLY_CLOUDY_RAIN,
   WEATHER_SITUATION__PARTLY_CLOUDY_SLIGHT_SLEET,
   WEATHER_SITUATION__PARTLY_CLOUDY_SLEET,
   WEATHER_SITUATION__PARTLY_CLOUDY_SLIGHT_SNOW,
   WEATHER_SITUATION__PARTLY_CLOUDY_SNOW,
   WEATHER_SITUATION__PARTLY_CLOUDY_THUNDERSTORM,
   WEATHER_SITUATION__CLOUDY,
   WEATHER_SITUATION__CLOUDY_SLIGHT_RAIN,
   WEATHER_SITUATION__CLOUDY_RAIN,
   WEATHER_SITUATION__CLOUDY_SLIGHT_SLEET,
   WEATHER_SITUATION__CLOUDY_SLEET,
   WEATHER_SITUATION__CLOUDY_SLIGHT_SNOW,
   WEATHER_SITUATION__CLOUDY_SNOW,
   WEATHER_SITUATION__CLOUDY_THUNDERSTORM,
   WEATHER_SITUATION__OVERCAST,
   WEATHER_SITUATION__OVERCAST_SLIGHT_RAIN,
   WEATHER_SITUATION__OVERCAST_RAIN,
   WEATHER_SITUATION__OVERCAST_SEVERE_RAIN,
   WEATHER_SITUATION__OVERCAST_SLIGHT_SLEET,
   WEATHER_SITUATION__OVERCAST_SLEET,
   WEATHER_SITUATION__OVERCAST_SEVERE_SLEET,
   WEATHER_SITUATION__OVERCAST_SLIGHT_SNOW,
   WEATHER_SITUATION__OVERCAST_SNOW,
   WEATHER_SITUATION__OVERCAST_SEVERE_SNOW,
   WEATHER_SITUATION__OVERCAST_THUNDERSTORM,
   WEATHER_SITUATION__MIST,
   WEATHER_SITUATION__FOG
};

inline ::std::string toString(WeatherSituation weatherSituation)
{
   switch(weatherSituation)
   {
   case WEATHER_SITUATION__UNKNOWN:
      return "WEATHER_SITUATION__UNKNOWN";
   case WEATHER_SITUATION__CLEAR_SKY:
      return "WEATHER_SITUATION__CLEAR_SKY";
   case WEATHER_SITUATION__A_FEW_CLOUDS:
      return "WEATHER_SITUATION__A_FEW_CLOUDS";
   case WEATHER_SITUATION__PARTLY_CLOUDY:
      return "WEATHER_SITUATION__PARTLY_CLOUDY";
   case WEATHER_SITUATION__PARTLY_CLOUDY_SLIGHT_RAIN:
      return "WEATHER_SITUATION__PARTLY_CLOUDY_SLIGHT_RAIN";
   case WEATHER_SITUATION__PARTLY_CLOUDY_RAIN:
      return "WEATHER_SITUATION__PARTLY_CLOUDY_RAIN";
   case WEATHER_SITUATION__PARTLY_CLOUDY_SLIGHT_SLEET:
      return "WEATHER_SITUATION__PARTLY_CLOUDY_SLIGHT_SLEET";
   case WEATHER_SITUATION__PARTLY_CLOUDY_SLEET:
      return "WEATHER_SITUATION__PARTLY_CLOUDY_SLEET";
   case WEATHER_SITUATION__PARTLY_CLOUDY_SLIGHT_SNOW:
      return "WEATHER_SITUATION__PARTLY_CLOUDY_SLIGHT_SNOW";
   case WEATHER_SITUATION__PARTLY_CLOUDY_THUNDERSTORM:
      return "WEATHER_SITUATION__PARTLY_CLOUDY_THUNDERSTORM";
   case WEATHER_SITUATION__CLOUDY:
      return "WEATHER_SITUATION__CLOUDY";
   case WEATHER_SITUATION__CLOUDY_SLIGHT_RAIN:
      return "WEATHER_SITUATION__CLOUDY_SLIGHT_RAIN";
   case WEATHER_SITUATION__CLOUDY_RAIN:
      return "WEATHER_SITUATION__CLOUDY_RAIN";
   case WEATHER_SITUATION__CLOUDY_SLIGHT_SLEET:
      return "WEATHER_SITUATION__CLOUDY_SLIGHT_SLEET";
   case WEATHER_SITUATION__CLOUDY_SLEET:
      return "WEATHER_SITUATION__CLOUDY_SLEET";
   case WEATHER_SITUATION__CLOUDY_SLIGHT_SNOW:
      return "WEATHER_SITUATION__CLOUDY_SLIGHT_SNOW";
   case WEATHER_SITUATION__CLOUDY_THUNDERSTORM:
      return "WEATHER_SITUATION__CLOUDY_THUNDERSTORM";
   case WEATHER_SITUATION__OVERCAST:
      return "WEATHER_SITUATION__OVERCAST";
   case WEATHER_SITUATION__OVERCAST_SLIGHT_RAIN:
      return "WEATHER_SITUATION__OVERCAST_SLIGHT_RAIN";
   case WEATHER_SITUATION__OVERCAST_RAIN:
      return "WEATHER_SITUATION__OVERCAST_RAIN";
   case WEATHER_SITUATION__OVERCAST_SEVERE_RAIN:
      return "WEATHER_SITUATION__OVERCAST_SEVERE_RAIN";
   case WEATHER_SITUATION__OVERCAST_SLIGHT_SLEET:
      return "WEATHER_SITUATION__OVERCAST_SLIGHT_SLEET";
   case WEATHER_SITUATION__OVERCAST_SLEET:
      return "WEATHER_SITUATION__OVERCAST_SLEET";
   case WEATHER_SITUATION__OVERCAST_SEVERE_SLEET:
      return "WEATHER_SITUATION__OVERCAST_SEVERE_SLEET";
   case WEATHER_SITUATION__OVERCAST_SLIGHT_SNOW:
      return "WEATHER_SITUATION__OVERCAST_SLIGHT_SNOW";
   case WEATHER_SITUATION__OVERCAST_SEVERE_SNOW:
      return "WEATHER_SITUATION__OVERCAST_SEVERE_SNOW";
   case WEATHER_SITUATION__OVERCAST_THUNDERSTORM:
      return "WEATHER_SITUATION__OVERCAST_THUNDERSTORM";
   case WEATHER_SITUATION__MIST:
      return "WEATHER_SITUATION__MIST";
   case WEATHER_SITUATION__FOG:
      return "WEATHER_SITUATION__FOG";
   default:
      return "<unknown>";
   }
}

enum WeatherSeverity
{
   WEATHER_SEVERITY__UNKNOWN,
   WEATHER_SEVERITY__SEVERE_RAIN,
   WEATHER_SEVERITY__SEVERE_SLEET,
   WEATHER_SEVERITY__SEVERE_SNOW,
   WEATHER_SEVERITY__SEVERE_FOG,
   WEATHER_SEVERITY__SEVERE_STORM,
   WEATHER_SEVERITY__EXTREME_WEATHER_CONDITIONS,
   WEATHER_SEVERITY__VISIBILITY_REDUCED
};

inline ::std::string toString(WeatherSeverity weatherSeverity)
{
   switch(weatherSeverity)
   {
   case WEATHER_SEVERITY__UNKNOWN:
      return "WEATHER_SEVERITY__UNKNOWN";
   case WEATHER_SEVERITY__SEVERE_RAIN:
      return "WEATHER_SEVERITY__SEVERE_RAIN";
   case WEATHER_SEVERITY__SEVERE_SLEET:
      return "WEATHER_SEVERITY__SEVERE_SLEET";
   case WEATHER_SEVERITY__SEVERE_SNOW:
      return "WEATHER_SEVERITY__SEVERE_SNOW";
   case WEATHER_SEVERITY__SEVERE_FOG:
      return "WEATHER_SEVERITY__SEVERE_FOG";
   case WEATHER_SEVERITY__SEVERE_STORM:
      return "WEATHER_SEVERITY__SEVERE_STORM";
   case WEATHER_SEVERITY__EXTREME_WEATHER_CONDITIONS:
      return "WEATHER_SEVERITY__EXTREME_WEATHER_CONDITIONS";
   case WEATHER_SEVERITY__VISIBILITY_REDUCED:
      return "WEATHER_SEVERITY__VISIBILITY_REDUCED";
   default:
      return "<unknown>";
   }
}

enum ConnectedService
{
   CONNECTED_SERVICE__UNKNOWN,
   CONNECTED_SERVICE__TRAFFIC,
   CONNECTED_SERVICE__WEATHER,
   CONNECTED_SERVICE__FUEL_PRICE_INFO,
   CONNECTED_SERVICE__PARKING_INFO,
   CONNECTED_SERVICE__SPEED_ALERT_INFO,
   CONNECTED_SERVICE__ELECTROMOBILITY_INFO,
   CONNECTED_SERVICE__ONLINE_SEARCH
};

inline ::std::string toString(ConnectedService connectedService)
{
   switch (connectedService)
   {
   case CONNECTED_SERVICE__UNKNOWN:
      return "CONNECTED_SERVICE__UNKNOWN";
   case CONNECTED_SERVICE__TRAFFIC:
      return "CONNECTED_SERVICE__TRAFFIC";
   case CONNECTED_SERVICE__WEATHER:
      return "CONNECTED_SERVICE__WEATHER";
   case CONNECTED_SERVICE__FUEL_PRICE_INFO:
      return "CONNECTED_SERVICE__FUEL_PRICE_INFO";
   case CONNECTED_SERVICE__PARKING_INFO:
      return "CONNECTED_SERVICE__PARKING_INFO";
   case CONNECTED_SERVICE__SPEED_ALERT_INFO:
      return "CONNECTED_SERVICE__SPEED_ALERT_INFO";
   case CONNECTED_SERVICE__ELECTROMOBILITY_INFO:
      return "CONNECTED_SERVICE__ELECTROMOBILITY_INFO";
   case CONNECTED_SERVICE__ONLINE_SEARCH:
      return "CONNECTED_SERVICE__ONLINE_SEARCH";
   default:
      ::std::stringstream stream;
      stream << "CONNECTED_SERVICE__<" << static_cast<unsigned int>(connectedService) << ">";
      return stream.str();
   }
}

enum AdvancedLoggingState
{
   ADVANCED_LOGGING_STATE__ON,
   ADVANCED_LOGGING_STATE__OFF
};

inline ::std::string toString(AdvancedLoggingState state)
{
   switch(state)
   {
   case ADVANCED_LOGGING_STATE__ON:
      return "ADVANCED_LOGGING_STATE__ON";
   case ADVANCED_LOGGING_STATE__OFF:
      return "ADVANCED_LOGGING_STATE__OFF";
   default:
      ::std::stringstream stream;
      stream << "ADVANCED_LOGGING_STATE__<" << static_cast<unsigned int>(state) << ">";
      return stream.str();
   }
}

inline ::std::string toString(DestinationMemoryEntryCategory destinationMemoryEntryCategory)
{
   switch (destinationMemoryEntryCategory)
   {
   case DESTINATION_MEMORY_ENTRY_CATEGORY__HOME:
      return "DESTINATION_MEMORY_ENTRY_CATEGORY__HOME";
   case DESTINATION_MEMORY_ENTRY_CATEGORY__WORK:
      return "DESTINATION_MEMORY_ENTRY_CATEGORY__HOME";
   case DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK:
      return "DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_BOOK";
   case DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_GADGET:
      return "DESTINATION_MEMORY_ENTRY_CATEGORY__ADDRESS_GADGET";
   case DESTINATION_MEMORY_ENTRY_CATEGORY__PERSONAL_EPOI:
      return "DESTINATION_MEMORY_ENTRY_CATEGORY__PERSONAL_EPOI";
   case DESTINATION_MEMORY_ENTRY_CATEGORY__FAVORITE_WEATHER_ALERTS:
      return "DESTINATION_MEMORY_ENTRY_CATEGORY__FAVORITE_WEATHER_ALERTS";
   default:
      ::std::stringstream stream;
      stream << "DESTINATION_MEMORY_ENTRY_CATEGORY__<" << static_cast<unsigned int>(destinationMemoryEntryCategory) << ">";
      return stream.str();
   }
}

enum DestinationMemoryAttributesGroup
{
   DESTINATION_MEMORY_ATTRIBUTES_GROUP__NONE,
   DESTINATION_MEMORY_ATTRIBUTES_GROUP__CHARGING_STATION,
   DESTINATION_MEMORY_ATTRIBUTES_GROUP__FAMILY,
   DESTINATION_MEMORY_ATTRIBUTES_GROUP__FRIENDS,
   DESTINATION_MEMORY_ATTRIBUTES_GROUP__BUSINESS,
   DESTINATION_MEMORY_ATTRIBUTES_GROUP__RESTAURANT,
   DESTINATION_MEMORY_ATTRIBUTES_GROUP__LEISURE,
   DESTINATION_MEMORY_ATTRIBUTES_GROUP__SHOPPING,
   DESTINATION_MEMORY_ATTRIBUTES_GROUP__ACCOMODATION,
   DESTINATION_MEMORY_ATTRIBUTES_GROUP__OTHER
};

inline ::std::string toString(DestinationMemoryAttributesGroup destinationMemoryAttributesGroup)
{
   switch (destinationMemoryAttributesGroup)
   {
   case DESTINATION_MEMORY_ATTRIBUTES_GROUP__NONE:
      return "DESTINATION_MEMORY_ATTRIBUTES_GROUP__NONE";
   case DESTINATION_MEMORY_ATTRIBUTES_GROUP__CHARGING_STATION:
      return "DESTINATION_MEMORY_ATTRIBUTES_GROUP__CHARGING_STATION";
   case DESTINATION_MEMORY_ATTRIBUTES_GROUP__FAMILY:
      return "DESTINATION_MEMORY_ATTRIBUTES_GROUP__FAMILY";
   case DESTINATION_MEMORY_ATTRIBUTES_GROUP__FRIENDS:
      return "DESTINATION_MEMORY_ATTRIBUTES_GROUP__FRIENDS";
   case DESTINATION_MEMORY_ATTRIBUTES_GROUP__BUSINESS:
      return "DESTINATION_MEMORY_ATTRIBUTES_GROUP__BUSINESS";
   case DESTINATION_MEMORY_ATTRIBUTES_GROUP__RESTAURANT:
      return "DESTINATION_MEMORY_ATTRIBUTES_GROUP__RESTAURANT";
   case DESTINATION_MEMORY_ATTRIBUTES_GROUP__LEISURE:
      return "DESTINATION_MEMORY_ATTRIBUTES_GROUP__LEISURE";
   case DESTINATION_MEMORY_ATTRIBUTES_GROUP__SHOPPING:
      return "DESTINATION_MEMORY_ATTRIBUTES_GROUP__SHOPPING";
   case DESTINATION_MEMORY_ATTRIBUTES_GROUP__ACCOMODATION:
      return "DESTINATION_MEMORY_ATTRIBUTES_GROUP__ACCOMODATION";
   case DESTINATION_MEMORY_ATTRIBUTES_GROUP__OTHER:
      return "DESTINATION_MEMORY_ATTRIBUTES_GROUP__OTHER";
   default:
      ::std::stringstream stream;
      stream << "DESTINATION_MEMORY_ATTRIBUTES_GROUP__<" << static_cast<unsigned int>(destinationMemoryAttributesGroup) << ">";
      return stream.str();
   }
}

inline ::std::string toString(DirectionDescription directionDescription)
{
   switch (directionDescription)
   {
   case DIRECTIONDESCRIPTION_NORTH:
      return "DIRECTIONDESCRIPTION_NORTH";
   case DIRECTIONDESCRIPTION_NORTH_NORTH_EAST:
      return "DIRECTIONDESCRIPTION_NORTH_NORTH_EAST";
   case DIRECTIONDESCRIPTION_NORTH_EAST:
      return "DIRECTIONDESCRIPTION_NORTH_EAST";
   case DIRECTIONDESCRIPTION_EAST_NORTH_EAST:
      return "DIRECTIONDESCRIPTION_EAST_NORTH_EAST";
   case DIRECTIONDESCRIPTION_EAST:
      return "DIRECTIONDESCRIPTION_EAST";
   case DIRECTIONDESCRIPTION_EAST_SOUTH_EAST:
      return "DIRECTIONDESCRIPTION_EAST_SOUTH_EAST";
   case DIRECTIONDESCRIPTION_SOUTH_EAST:
      return "DIRECTIONDESCRIPTION_SOUTH_EAST";
   case DIRECTIONDESCRIPTION_SOUTH_SOUTH_EAST:
      return "DIRECTIONDESCRIPTION_SOUTH_SOUTH_EAST";
   case DIRECTIONDESCRIPTION_SOUTH:
      return "DIRECTIONDESCRIPTION_SOUTH";
   case DIRECTIONDESCRIPTION_SOUTH_SOUTH_WEST:
      return "DIRECTIONDESCRIPTION_SOUTH_SOUTH_WEST";
   case DIRECTIONDESCRIPTION_SOUTH_WEST:
      return "DIRECTIONDESCRIPTION_SOUTH_WEST";
   case DIRECTIONDESCRIPTION_WEST_SOUTH_WEST:
      return "DIRECTIONDESCRIPTION_WEST_SOUTH_WEST";
   case DIRECTIONDESCRIPTION_WEST:
      return "DIRECTIONDESCRIPTION_WEST";
   case DIRECTIONDESCRIPTION_WEST_NORTH_WEST:
      return "DIRECTIONDESCRIPTION_WEST_NORTH_WEST";
   case DIRECTIONDESCRIPTION_NORTH_WEST:
      return "DIRECTIONDESCRIPTION_NORTH_WEST";
   case DIRECTIONDESCRIPTION_NORTH_NORTH_WEST:
      return "DIRECTIONDESCRIPTION_NORTH_NORTH_WEST";
   default:
      return "<unknown>";
   }
}

enum LocationInputType
{
   LOCATIONINPUTTYPE_ADDRESS,
   LOCATIONINPUTTYPE_POI,
   LOCATIONINPUTTYPE_FTS,
   LOCATIONINPUTTYPE_SDS_ADDRESS,
   LOCATIONINPUTTYPE_MOTORWAY_IC //JP
};
inline ::std::string toString(LocationInputType inputType)
{
   switch (inputType)
   {
   case LOCATIONINPUTTYPE_ADDRESS:
      return "LOCATIONINPUTTYPE_ADDRESS";
   case LOCATIONINPUTTYPE_POI:
      return "LOCATIONINPUTTYPE_POI";
   case LOCATIONINPUTTYPE_FTS:
      return "LOCATIONINPUTTYPE_FTS";
   case LOCATIONINPUTTYPE_SDS_ADDRESS:
      return "LOCATIONINPUTTYPE_SDS_ADDRESS";
   case LOCATIONINPUTTYPE_MOTORWAY_IC:
      return "LOCATIONINPUTTYPE_MOTORWAY_IC";
   default:
      ::std::stringstream stream;
      stream << "LOCATIONINPUTTYPE_<" << static_cast<unsigned int>(inputType) << ">";
      return stream.str();
   }
}

/**
 * Describes the origin where the data (e.g. a location) is coming from
 */
enum LocationOrigin
{
   /** Origin is unknown */
   LOCATION_ORIGIN__UNKNOWN,
   /** Origin is the onboard dataset */
   LOCATION_ORIGIN__ONBOARD_DATASET,
   /** Origin is some external source (e.g. companion app)*/
   LOCATION_ORIGIN__UNKNOWN_EXTERNAL_SOURCE,
   /** Origin is TomTom online search */
   LOCATION_ORIGIN__TOMTOM_ONLINE_SEARCH,
   /** Origin is Google Places search */
   LOCATION_ORIGIN__GOOGLE_PLACES,
   /** Origin is a TPEG service */
   LOCATION_ORIGIN__TPEG
};
inline ::std::string toString(LocationOrigin origin)
{
   switch (origin)
   {
   case LOCATION_ORIGIN__UNKNOWN:
      return "LOCATION_ORIGIN__UNKNOWN";
   case LOCATION_ORIGIN__ONBOARD_DATASET:
      return "LOCATION_ORIGIN__ONBOARD_DATASET";
   case LOCATION_ORIGIN__UNKNOWN_EXTERNAL_SOURCE:
      return "LOCATION_ORIGIN__UNKNOWN_EXTERNAL_SOURCE";
   case LOCATION_ORIGIN__TOMTOM_ONLINE_SEARCH:
      return "LOCATION_ORIGIN__TOMTOM_ONLINE_SEARCH";
   case LOCATION_ORIGIN__GOOGLE_PLACES:
      return "LOCATION_ORIGIN__GOOGLE_PLACES";
   case LOCATION_ORIGIN__TPEG:
      return "LOCATION_ORIGIN__TPEG";
   default:
      ::std::stringstream stream;
      stream << "LOCATION_ORIGIN__<" << static_cast<unsigned int>(origin) << ">";
      return stream.str();
   }
}

/**
 * Describes the source where a location or an entry in a location (result) list comes from
 * With introduction of LocationOrigin, this represents mainly if a location is freshly input (i.e. coming from
 * onboard/online) or if it was added from one of the "history sources"
 */
enum LocationSource
{
   /** Location (entry) comes from Location Input onboard search */
   LOCATION_SOURCE__LOCATION_INPUT_ONBOARD,
   /** Location (entry) comes from Location Input online service */
   LOCATION_SOURCE__LOCATION_INPUT_ONLINE,
   /** Location (entry) comes from DestMem (unspecified category) */
   LOCATION_SOURCE__DEST_MEM_UNSPECIFIED,
   /** Location (entry) comes from DestMem last destinations/routes (Renault: history) */
   LOCATION_SOURCE__DEST_MEM_LAST_USED,
   /** Location (entry) comes from DestMem user address book/route book (Renault: favorites) */
   LOCATION_SOURCE__DEST_MEM_USER_STORED,
   /** Location (entry) comes from DestMem user's home location */
   LOCATION_SOURCE__DEST_MEM_HOME,
   /** Location (entry) comes from DestMem user's work location */
   LOCATION_SOURCE__DEST_MEM_WORK,
   /** Location (entry) was selected in the past (e.g. Renault "free text search history") */
   LOCATION_SOURCE__RECENTLY_SELECTED
};
inline ::std::string toString(LocationSource source)
{
   switch (source)
   {
   case LOCATION_SOURCE__LOCATION_INPUT_ONBOARD:
      return "LOCATION_SOURCE__LOCATION_INPUT_ONBOARD";
   case LOCATION_SOURCE__LOCATION_INPUT_ONLINE:
      return "LOCATION_SOURCE__LOCATION_INPUT_ONLINE";
   case LOCATION_SOURCE__DEST_MEM_UNSPECIFIED:
      return "LOCATION_SOURCE__DEST_MEM_UNSPECIFIED";
   case LOCATION_SOURCE__DEST_MEM_LAST_USED:
      return "LOCATION_SOURCE__DEST_MEM_LAST_DESTINATIONS";
   case LOCATION_SOURCE__DEST_MEM_USER_STORED:
      return "LOCATION_SOURCE__DEST_MEM_ADDRESS_BOOK";
   case LOCATION_SOURCE__DEST_MEM_HOME:
      return "LOCATION_SOURCE__DEST_MEM_HOME";
   case LOCATION_SOURCE__DEST_MEM_WORK:
      return "LOCATION_SOURCE__DEST_MEM_WORK";
   case LOCATION_SOURCE__RECENTLY_SELECTED:
      return "LOCATION_SOURCE__RECENTLY_SELECTED";
   default:
      ::std::stringstream stream;
      stream << "LOCATION_SOURCE__<" << static_cast<unsigned int>(source) << ">";
      return stream.str();
   }
}

struct LimitedFeatureMode
{
   LimitedFeatureMode()
      :m_isActive(false)
   {
      m_requiredFeatures.clear();  //<-- INF4CV
   }

   explicit LimitedFeatureMode(const ::std::set<RequiredFeatures>& reqFeature)
   {
      m_isActive = true;
      m_requiredFeatures = reqFeature;
   }

   bool operator==(const LimitedFeatureMode& rhs) const
   {
      return (!m_isActive == !rhs.m_isActive) && (m_requiredFeatures == rhs.m_requiredFeatures );
   }

   ::std::set<RequiredFeatures> m_requiredFeatures;
   bool                         m_isActive;
};

enum PositionSimulationMode
{
   POSITION_SIMULATION_MODE_OFF,
   POSITION_SIMULATION_MODE_ON,
   POSITION_SIMULATION_MODE_DRIVINGALONGROUTE_ON
};

enum HemisphereLatitude
{
   HEMISPHERE_LATITUDE__NONE,
   HEMISPHERE_LATITUDE__NORTHERN,
   HEMISPHERE_LATITUDE__SOUTHERN
};
inline ::std::string toString(HemisphereLatitude lat)
{
   switch (lat)
   {
   case HEMISPHERE_LATITUDE__NORTHERN:
      return "N";
   case HEMISPHERE_LATITUDE__SOUTHERN:
      return "S";
   default:
      return "";
   }
}

enum HemisphereLongitude
{
   HEMISPHERE_LONGITUDE__NONE,
   HEMISPHERE_LONGITUDE__EASTERN,
   HEMISPHERE_LONGITUDE__WESTERN
};
inline ::std::string toString(HemisphereLongitude lon)
{
   switch (lon)
   {
   case HEMISPHERE_LONGITUDE__EASTERN:
      return "E";
   case HEMISPHERE_LONGITUDE__WESTERN:
      return "W";
   default:
      return "";
   }
}

/**
 * Predefined POI categories are POI categories having a special meaning to HMI,
 * e.g. because HMI has to follow a special flow browsing those categories or have some fix shortcut for those categories
 */
enum PredefinedPoiCategory
{
   PREDEFINED_POI_CATEGORY__UNDEFINED, // Delivered POI category is not a predefined one
   PREDEFINED_POI_CATEGORY__ATM,
   PREDEFINED_POI_CATEGORY__EV_CHARGING_STATION,
   PREDEFINED_POI_CATEGORY__HOSPITAL,
   PREDEFINED_POI_CATEGORY__HOTEL_MOTEL,
   PREDEFINED_POI_CATEGORY__MOTORWAY_ENTRY,
   PREDEFINED_POI_CATEGORY__MOTORWAY_EXIT,
   PREDEFINED_POI_CATEGORY__PARKING,
   PREDEFINED_POI_CATEGORY__PETROL_STATION,
   PREDEFINED_POI_CATEGORY__RESTAURANT,
   PREDEFINED_POI_CATEGORY__VEHICLE_REPAIR,
   PREDEFINED_POI_CATEGORY__LEISURE
};
inline ::std::string toString(PredefinedPoiCategory category)
{
   switch (category)
   {
   case PREDEFINED_POI_CATEGORY__UNDEFINED:
      return "PREDEFINED_POI_CATEGORY__UNDEFINED";
   case PREDEFINED_POI_CATEGORY__ATM:
      return "PREDEFINED_POI_CATEGORY__ATM";
   case PREDEFINED_POI_CATEGORY__EV_CHARGING_STATION:
      return "PREDEFINED_POI_CATEGORY__EV_CHARGING_STATION";
   case PREDEFINED_POI_CATEGORY__HOSPITAL:
      return "PREDEFINED_POI_CATEGORY__HOSPITAL";
   case PREDEFINED_POI_CATEGORY__HOTEL_MOTEL:
      return "PREDEFINED_POI_CATEGORY__HOTEL_MOTEL";
   case PREDEFINED_POI_CATEGORY__MOTORWAY_ENTRY:
      return "PREDEFINED_POI_CATEGORY__MOTORWAY_ENTRY";
   case PREDEFINED_POI_CATEGORY__MOTORWAY_EXIT:
      return "PREDEFINED_POI_CATEGORY__MOTORWAY_EXIT";
   case PREDEFINED_POI_CATEGORY__PARKING:
      return "PREDEFINED_POI_CATEGORY__PARKING";
   case PREDEFINED_POI_CATEGORY__PETROL_STATION:
      return "PREDEFINED_POI_CATEGORY__PETROL_STATION";
   case PREDEFINED_POI_CATEGORY__RESTAURANT:
      return "PREDEFINED_POI_CATEGORY__RESTAURANT";
   case PREDEFINED_POI_CATEGORY__VEHICLE_REPAIR:
      return "PREDEFINED_POI_CATEGORY__VEHICLE_REPAIR";
   case PREDEFINED_POI_CATEGORY__LEISURE:
      return "PREDEFINED_POI_CATEGORY__LEISURE";
   default:
      ::std::stringstream stream;
      stream << "PREDEFINED_POI_CATEGORY__<" << static_cast<unsigned int>(category) << ">";
      return stream.str();
   }
}

enum ChargingType
{
   CHARGING_TYPE__STANDARD_CHARGE,
   CHARGING_TYPE__STANDARD_CHARGE_100V,
   CHARGING_TYPE__STANDARD_CHARGE_200V,
   CHARGING_TYPE__FAST_CHARGE,
   CHARGING_TYPE__ACCELERATED_CHARGE,
   CHARGING_TYPE__QUICK_DROP,
   CHARGING_TYPE__QUICK_CHARGE,
   CHARGING_TYPE__ULTRA_FAST_CHARGE,
   CHARGING_TYPE__UNKNOWN
};

inline ::std::string toString(ChargingType type)
{
   switch (type)
   {
   case CHARGING_TYPE__STANDARD_CHARGE:
      return "CHARGING_TYPE__STANDARD_CHARGE";
   case CHARGING_TYPE__STANDARD_CHARGE_100V:
      return "CHARGING_TYPE__STANDARD_CHARGE_100V";
   case CHARGING_TYPE__STANDARD_CHARGE_200V:
      return "CHARGING_TYPE__STANDARD_CHARGE_200V";
   case CHARGING_TYPE__FAST_CHARGE:
      return "CHARGING_TYPE__FAST_CHARGE";
   case CHARGING_TYPE__ACCELERATED_CHARGE:
      return "CHARGING_TYPE__ACCELERATED_CHARGE";
   case CHARGING_TYPE__QUICK_DROP:
      return "CHARGING_TYPE__QUICK_DROP";
   case CHARGING_TYPE__UNKNOWN:
      return "CHARGING_TYPE__UNKNOWN";
   default:
      ::std::stringstream stream;
      stream << "CHARGING_TYPE__<" << static_cast<unsigned int>(type) << ">";
      return stream.str();
   }
}

enum TrafficSense
{
   TRAFFIC_SENSE__RIGHTHAND_TRAFFIC = 0,
   TRAFFIC_SENSE__LEFTHAND_TRAFFIC = 1,
   TRAFFIC_SENSE__UNKNOWN = 2
};

inline ::std::string toString(TrafficSense trafficSense)
{
   switch (trafficSense)
   {
   case TRAFFIC_SENSE__RIGHTHAND_TRAFFIC:
      return "TRAFFIC_SENSE__RIGHTHAND_TRAFFIC";
   case TRAFFIC_SENSE__LEFTHAND_TRAFFIC:
      return "TRAFFIC_SENSE__LEFTHAND_TRAFFIC";
   case TRAFFIC_SENSE__UNKNOWN:
      return "TRAFFIC_SENSE__UNKNOWN";
   default:
      ::std::stringstream stream;
      stream << "TRAFFIC_SENSE__<" << static_cast<unsigned int>(trafficSense) << ">";
      return stream.str();
   }
}

/** This class defines a coordinate pair of latitude and longitude of unit degrees */
class GeoCoordinateDegree
{
public:

   GeoCoordinateDegree()
      :m_latitude(0.0)
      ,m_longitude(0.0)
   {}

   GeoCoordinateDegree(const double& latitude, const double& longitude)
      :m_latitude(latitude)
      ,m_longitude(longitude)
   {}

   double getLatitude() const
   {
      return m_latitude;
   }

   double getLongitude() const
   {
      return m_longitude;
   }

   bool operator==(const GeoCoordinateDegree& rhs ) const
   {
      return m_latitude == rhs.m_latitude && m_longitude == rhs.m_longitude;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "GeoCoordinateDegree Infos: ";

      stream << "Latitude = " << m_latitude << std::endl
             << "Longitude = " << m_longitude << std::endl;

      return stream.str();
   }

private:
   double  m_latitude;
   double  m_longitude;
};

struct PoiData
{
   PoiData()
      : m_selectionCriterion(0)
   {}
   PoiData(const PoiId& poiId, SelectionCriterion selectionCriterion, const GeoCoordinateDegree& coordinate)
      : m_poiId(poiId)
      , m_selectionCriterion(selectionCriterion)
      , m_coordinate(coordinate)
   {}

   bool operator == (const PoiData& rhs) const
   {
      return (m_poiId == rhs.m_poiId)&&
             (m_selectionCriterion == rhs.m_selectionCriterion)&&
             (m_coordinate == rhs.m_coordinate);
   }

   bool operator != (const PoiData& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "PoiData(" << "m_poiId = " << m_poiId << ", m_selectionCriterion = " << m_selectionCriterion
             << ", m_coordinate = " << m_coordinate.toString() << ")";
      return stream.str();
   }

   PoiId                m_poiId;
   SelectionCriterion   m_selectionCriterion;
   GeoCoordinateDegree  m_coordinate;
};

struct MapDynamicIconId
{
   MapDynamicIconId()
      : m_iconId(0)
      , m_iconIntent("")
   {}

   MapDynamicIconId(uint64_t iconId, const ::std::string& iconIntent)
      : m_iconId(iconId)
      , m_iconIntent(iconIntent)
   {}

   bool operator == (const MapDynamicIconId& rhs) const
   {
      return ((m_iconId == rhs.m_iconId)  &&
              (m_iconIntent == rhs.m_iconIntent));
   }

   bool operator != (const MapDynamicIconId& rhs) const
   {
      return !(*this == rhs);
   }

   bool operator < (const MapDynamicIconId& rhs) const
   {
      return ((m_iconId == rhs.m_iconId) ?
              (m_iconIntent < rhs.m_iconIntent) : (m_iconId < rhs.m_iconId));
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "MapDynamicIconId(" << "m_iconId = " << m_iconId
             << ", m_iconIntent = " << m_iconIntent << ")";
      return stream.str();
   }

   uint64_t          m_iconId;
   ::std::string     m_iconIntent;
};

class Image
{
public:
   enum Type
   {
      TYPE__UNDEFINED,
      TYPE__PNG,
      TYPE__JPEG
   };
   static ::std::string toString(Type type)
   {
      switch (type)
      {
      case TYPE__UNDEFINED:
         return "TYPE__UNDEFINED";
      case TYPE__PNG:
         return "TYPE__PNG";
      case TYPE__JPEG:
         return "TYPE__JPEG";
      default:
         ::std::stringstream stream;
         stream << "TYPE__<" << static_cast<unsigned int>(type) << ">";
         return stream.str();
      }
   }
   Image()
      : m_size(0)
      , m_blob(NULL)
      , m_type(TYPE__UNDEFINED) {}

   Image
   (
      unsigned int imageSize,
      const unsigned char* imageBlob,
      ValidValue<uint64_t> id = ValidValue<uint64_t>(),
      Type type = TYPE__PNG
   )
      : m_size(imageSize),
        m_blob(imageBlob),
        m_type(type),
        m_id(id) {}

   unsigned int getBlobSize() const
   {
      return m_size;
   }

   const unsigned char* getBlob() const
   {
      return m_blob;
   }

   void setBlobSize(unsigned int size)
   {
      m_size = size;
   }

   void setBlob(const unsigned char* blob)
   {
      m_blob = blob;
   }

   Type getType() const
   {
      return m_type;
   }

   void setType(Type type)
   {
      m_type = type;
   }

   const ValidValue<uint64_t>& getId() const
   {
      return m_id;
   }

   void setId(const ValidValue<uint64_t>& id)
   {
      m_id = id;
   }

   bool operator==(const Image& rhs) const
   {
      return (m_size == rhs.m_size) &&
             (m_blob == rhs.m_blob) &&
             (m_type == rhs.m_type) &&
             (m_id == rhs.m_id);
   }

   bool operator!=(const Image& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "Image: ";

      stream << "Type = " << toString(m_type) << std::endl
             << "Size = " << m_size << std::endl;
      if (m_id.isValid())
      {
         stream << "id = " << m_id.getValue() << std::endl;
      }

      return stream.str();
   }

private:
   unsigned int         m_size;
   const unsigned char* m_blob;
   Type                 m_type;
   ValidValue<uint64_t> m_id;
};

class LocationDetails
{
public:
   LocationDetails()
      : m_locationOrigin(LOCATION_ORIGIN__UNKNOWN)
   {}

   LocationDetails(const ::std::string& phoneNumber, const ::std::string& address, const Image& icon, LocationOrigin locationOrigin, const ::std::string& uniqueId)
      : m_phoneNumber(phoneNumber)
      , m_address(address)
      , m_icon(icon)
      , m_locationOrigin(locationOrigin)
      , m_uniqueId(uniqueId) {}

   const ::std::string& getPhoneNumber() const
   {
      return m_phoneNumber;
   }

   void setPhoneNumber(const ::std::string& phoneNumber)
   {
      m_phoneNumber = phoneNumber;
   }

   const ::std::string& getAddress() const
   {
      return m_address;
   }

   void setAddress(const ::std::string& address)
   {
      m_address = address;
   }

   const Image& getIcon() const
   {
      return m_icon;
   }

   void setIcon(const Image& icon)
   {
      m_icon = icon;
   }

   LocationOrigin getLocationOrigin() const
   {
      return m_locationOrigin;
   }

   void setLocationOrigin(LocationOrigin locationOrigin)
   {
      m_locationOrigin = locationOrigin;
   }

   const ::std::string& getUniqueId() const
   {
      return m_uniqueId;
   }

   void setUniqueId(const ::std::string& uniqueId)
   {
      m_uniqueId = uniqueId;
   }

   bool operator == (const LocationDetails& rhs) const
   {
      return (m_phoneNumber == rhs.m_phoneNumber)&&
             (m_address == rhs.m_address)&&
             (m_icon == rhs.m_icon)&&
             (m_locationOrigin == rhs.m_locationOrigin)&&
             (m_uniqueId == rhs.m_uniqueId);
   }

   bool operator != (const LocationDetails& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "LocationDetails: ";

      stream << "PhoneNumber = " << m_phoneNumber << ::std::endl
             << "Address = " << m_address << ::std::endl
             << "Icon size = " << m_icon.getBlobSize() << ::std::endl
             << "Location Origin = " << ::navmiddleware::toString(m_locationOrigin) << ::std::endl
             << "Unique External Location Id = " << m_uniqueId << ::std::endl;

      return stream.str();
   }

private:
   ::std::string m_phoneNumber;
   ::std::string m_address;
   Image m_icon;
   LocationOrigin m_locationOrigin; // Original source where the location is originated, For eg: Google destination coming from vNext should have locationOrigin set to Google
   ::std::string m_uniqueId; // UniqueId to identify the location. For eg: PlaceId incase of google destinations
};

class ExternalLocation
{
public:
   ExternalLocation() {}

   ExternalLocation(const ::std::string& name, const GeoCoordinateDegree& coordinate)
      : m_name(name)
      , m_coordinate(coordinate) {}

   ExternalLocation(const ::std::string& name, const GeoCoordinateDegree& coordinate, const ValidValue< LocationDetails >& locationDetails)
      : m_name(name)
      , m_coordinate(coordinate)
      , m_locationDetails(locationDetails) {}

   const ::std::string& getName() const
   {
      return m_name;
   }

   void setName(const ::std::string& name)
   {
      m_name = name;
   }

   const GeoCoordinateDegree& getCoordinate() const
   {
      return m_coordinate;
   }

   void setCoordinate(const GeoCoordinateDegree& coordinate)
   {
      m_coordinate = coordinate;
   }

   const ValidValue<LocationDetails>& getLocationDetails() const
   {
      return m_locationDetails;
   }

   void setLocationDetails(const ValidValue<LocationDetails>& locationDetails)
   {
      m_locationDetails = locationDetails;
   }

   bool operator == (const ExternalLocation& rhs) const
   {
      return (m_name == rhs.m_name)&&
             (m_coordinate == rhs.m_coordinate)&&
             (m_locationDetails == rhs.m_locationDetails);
   }

   bool operator != (const ExternalLocation& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "ExternalLocation: ";

      stream << "Name = " << m_name << ::std::endl
             << m_coordinate.toString()
             << (m_locationDetails.isValid() ? m_locationDetails.getValue().toString() : "LocationDetails: Undefined") << ::std::endl;

      return stream.str();
   }

private:
   ::std::string m_name;
   GeoCoordinateDegree m_coordinate;
   ValidValue< LocationDetails > m_locationDetails;
};

// this structure will be removed, since it is redundant to 'Image'.
// Image is extended to hold a generic replacement for 'internalMapId': 'iconId'
class IconData
{
public:
   IconData(
      const unsigned char* iconBlobAddress,
      uint32_t             iconBlobLength,
      uint64_t             internalMapId
   )
      : m_iconBlobAddress(iconBlobAddress),
        m_iconBlobLength(iconBlobLength),
        m_internalMapId(internalMapId)
   {
   }

   IconData()
      : m_iconBlobAddress(0),
        m_iconBlobLength(0),
        m_internalMapId(0)
   {
   }

   const unsigned char* getIconBlobAddress() const
   {
      return m_iconBlobAddress;
   }

   uint32_t getIconBlobLength() const
   {
      return m_iconBlobLength;
   }

   uint64_t getInternalMapId() const
   {
      return m_internalMapId;
   }

private:
   const unsigned char* m_iconBlobAddress;
   uint32_t             m_iconBlobLength;
   uint64_t             m_internalMapId;
};

enum EvChargingSpeed
{
   EV_CHARGING_SPEED__UNKNOWN,
   EV_CHARGING_SPEED__LEVEL_0,   // up to  8.0 kW
   EV_CHARGING_SPEED__LEVEL_1,   // up to 11.1 kW
   EV_CHARGING_SPEED__LEVEL_MAX, // up to 43.6 kW
   EV_CHARGING_SPEED__DC_CHARGE, // Nissan quick charge
   EV_CHARGING_SPEED__BATTERY_EXCHANGE
};
inline ::std::string toString(EvChargingSpeed chargingSpeed)
{
   switch (chargingSpeed)
   {
   case EV_CHARGING_SPEED__UNKNOWN:
      return "EV_CHARGING_SPEED__UNKNOWN";
   case EV_CHARGING_SPEED__LEVEL_0:
      return "EV_CHARGING_SPEED__LEVEL_0";
   case EV_CHARGING_SPEED__LEVEL_1:
      return "EV_CHARGING_SPEED__LEVEL_1";
   case EV_CHARGING_SPEED__LEVEL_MAX:
      return "EV_CHARGING_SPEED__LEVEL_MAX";
   case EV_CHARGING_SPEED__DC_CHARGE:
      return "EV_CHARGING_SPEED__DC_CHARGE";
   case EV_CHARGING_SPEED__BATTERY_EXCHANGE:
      return "EV_CHARGING_SPEED__BATTERY_EXCHANGE";
   default:
      ::std::stringstream stream;
      stream << "EV_CHARGING_SPEED__<" << static_cast<unsigned int>(chargingSpeed) << ">";
      return stream.str();
   }
}

//For more details on IEC 62196 refer: https://en.wikipedia.org/wiki/IEC_62196
enum EvConnectorType
{
   EV_CONNECTOR_TYPE__UNSPECIFIED,                              // Unknown
   EV_CONNECTOR_TYPE__STANDARD_HOUSEHOLD,                       // All regions,    "Domestic socket"
   EV_CONNECTOR_TYPE__IEC_60309_SINGLE_PHASE_INDUSTRIAL,        // Europe mainly,  "Blue industrial socket"
   EV_CONNECTOR_TYPE__IEC_60309_SINGLE_PHASE_INDUSTRIAL_32A,    // SinglePhaseBlue connector with 32A current
   EV_CONNECTOR_TYPE__IEC_60309_THREE_PHASE_INDUSTRIAL,         // Three phase red connector without current information
   EV_CONNECTOR_TYPE__IEC_60309_THREE_PHASE_INDUSTRIAL_16A,     // ThreePhaseRed connector with 16A current
   EV_CONNECTOR_TYPE__IEC_60309_THREE_PHASE_INDUSTRIAL_32A,     // ThreePhaseRed connector with 32A current
   EV_CONNECTOR_TYPE__IEC_60309_THREE_PHASE_INDUSTRIAL_63A,     // ThreePhaseRed connector with 63A current
   EV_CONNECTOR_TYPE__IEC_62196_2_TYPE_2_OUTLET,                // Europe mainly,   Mode 3, type 2 connector/Charging Spot Side Plug
   EV_CONNECTOR_TYPE__IEC_62196_2_TYPE_3,                       // Europe mainly,   Mode 3, type 3 Vehicle Inlet
   EV_CONNECTOR_TYPE__IEC_62196_2_TYPE_2_CCS,                   // Combo Type 2
   EV_CONNECTOR_TYPE__IEC_62196_2_TYPE_1_CCS,                   // Combo_Type 1
   EV_CONNECTOR_TYPE__CABLE_ATTACHED_WITH_IEC_62196_2_TYPE_1,
   EV_CONNECTOR_TYPE__CABLE_ATTACHED_WITH_IEC_62196_2_TYPE_2_MODE_1_3,
   EV_CONNECTOR_TYPE__CABLE_ATTACHED_WITH_IEC_62196_2_TYPE_3,
   EV_CONNECTOR_TYPE__SAE_J1772,                                 // SAE_J1771 same as Type-1
   EV_CONNECTOR_TYPE__CHADEMO,                                   // "CHAdeMO" Quick charging(DC)
   EV_CONNECTOR_TYPE__CHINA_SPECIFIC,                            // KDS entry
   EV_CONNECTOR_TYPE__NEMA_5_20,                                 // American, T-slot socket
   EV_CONNECTOR_TYPE__NEMA_5_15,                                 // American, Grounded (Type B), duplex socket
   EV_CONNECTOR_TYPE__NEMA_14_30,                                // Single-phase grounding socket
   EV_CONNECTOR_TYPE__NEMA_6_20,                                 // T-shaped hole
   EV_CONNECTOR_TYPE__NEMA_14_50,                                // Four-wire grounding devices (hot-hot-neutral-ground)
   EV_CONNECTOR_TYPE__SEV_TYPE13,                                // Swiss, Type J
   EV_CONNECTOR_TYPE__SEV_TYPE15,
   EV_CONNECTOR_TYPE__SEV_TYPE23,
   EV_CONNECTOR_TYPE__SEV_TYPE25,
   EV_CONNECTOR_TYPE__DOMESTIC_E_CEE_7_5,                          //French; Type E
   EV_CONNECTOR_TYPE__DOMESTIC_I,
   EV_CONNECTOR_TYPE__DOMESTIC_D_BS_546,
   EV_CONNECTOR_TYPE__DOMESTIC_F_CEE_7_4,
   EV_CONNECTOR_TYPE__DOMESTIC_EF_CEE_7_7,                         // Compatible with E and F
   EV_CONNECTOR_TYPE__DOMESTIC_H_SI_32,
   EV_CONNECTOR_TYPE__DOMESTIC_I_ASNZ_3112,
   EV_CONNECTOR_TYPE__DOMESTIC_I_CPCS_CCC,
   EV_CONNECTOR_TYPE__DOMESTIC_I_IRAM_2073,
   EV_CONNECTOR_TYPE__DOMESTIC_K,
   EV_CONNECTOR_TYPE__DOMESTIC_K_THAI,
   EV_CONNECTOR_TYPE__DOMESTIC_L,
   EV_CONNECTOR_TYPE__DOMESTIC_M,
   EV_CONNECTOR_TYPE__DOMESTIC_IEC_60906_1,
   EV_CONNECTOR_TYPE__DOMESTIC_G_BS_1363,
   EV_CONNECTOR_TYPE__EN50075_EUROPLUG,                             //European Standard, Flat non-wirable two-pole plugs
   EV_CONNECTOR_TYPE__KOREA_SPECIFIC_EVSE_PLUG
};

inline ::std::string toString(EvConnectorType connectorType)
{
   switch (connectorType)
   {
   case EV_CONNECTOR_TYPE__UNSPECIFIED:
      return "EV_CONNECTOR_TYPE__UNSPECIFIED";
   case EV_CONNECTOR_TYPE__STANDARD_HOUSEHOLD:
      return "EV_CONNECTOR_TYPE__STANDARD_HOUSEHOLD";
   case EV_CONNECTOR_TYPE__IEC_60309_SINGLE_PHASE_INDUSTRIAL:
      return "EV_CONNECTOR_TYPE__IEC_60309_SINGLE_PHASE_INDUSTRIAL";
   case EV_CONNECTOR_TYPE__IEC_60309_SINGLE_PHASE_INDUSTRIAL_32A:
      return "EV_CONNECTOR_TYPE__IEC_60309_SINGLE_PHASE_INDUSTRIAL_32A";
   case EV_CONNECTOR_TYPE__IEC_60309_THREE_PHASE_INDUSTRIAL:
      return "EV_CONNECTOR_TYPE__IEC_60309_THREE_PHASE_INDUSTRIAL";
   case EV_CONNECTOR_TYPE__IEC_60309_THREE_PHASE_INDUSTRIAL_16A:
      return "EV_CONNECTOR_TYPE__IEC_60309_THREE_PHASE_INDUSTRIAL_16A";
   case EV_CONNECTOR_TYPE__IEC_60309_THREE_PHASE_INDUSTRIAL_32A:
      return "EV_CONNECTOR_TYPE__IEC_60309_THREE_PHASE_INDUSTRIAL_32A";
   case EV_CONNECTOR_TYPE__IEC_60309_THREE_PHASE_INDUSTRIAL_63A:
      return "EV_CONNECTOR_TYPE__IEC_60309_THREE_PHASE_INDUSTRIAL_63A";
   case EV_CONNECTOR_TYPE__IEC_62196_2_TYPE_2_OUTLET:
      return "EV_CONNECTOR_TYPE__IEC_62196_2_TYPE_2_OUTLET";
   case EV_CONNECTOR_TYPE__IEC_62196_2_TYPE_3:
      return "EV_CONNECTOR_TYPE__IEC_62196_2_TYPE_3";
   case EV_CONNECTOR_TYPE__IEC_62196_2_TYPE_2_CCS:
      return "EV_CONNECTOR_TYPE__IEC_62196_2_TYPE_2_CCS";
   case EV_CONNECTOR_TYPE__IEC_62196_2_TYPE_1_CCS:
      return "EV_CONNECTOR_TYPE__IEC_62196_2_TYPE_1_CCS";
   case EV_CONNECTOR_TYPE__CABLE_ATTACHED_WITH_IEC_62196_2_TYPE_1:
      return "EV_CONNECTOR_TYPE__CABLE_ATTACHED_WITH_IEC_62196_2_TYPE_1";
   case EV_CONNECTOR_TYPE__CABLE_ATTACHED_WITH_IEC_62196_2_TYPE_2_MODE_1_3:
      return "EV_CONNECTOR_TYPE__CABLE_ATTACHED_WITH_IEC_62196_2_TYPE_2_MODE_1_3";
   case EV_CONNECTOR_TYPE__CABLE_ATTACHED_WITH_IEC_62196_2_TYPE_3:
      return "EV_CONNECTOR_TYPE__CABLE_ATTACHED_WITH_IEC_62196_2_TYPE_3";
   case EV_CONNECTOR_TYPE__SAE_J1772:
      return "EV_CONNECTOR_TYPE__SAE_J1772";
   case EV_CONNECTOR_TYPE__CHADEMO:
      return "EV_CONNECTOR_TYPE__CHADEMO";
   case EV_CONNECTOR_TYPE__CHINA_SPECIFIC:
      return "EV_CONNECTOR_TYPE__CHINA_SPECIFIC";
   case EV_CONNECTOR_TYPE__NEMA_5_20:
      return "EV_CONNECTOR_TYPE__NEMA_5_20";
   case EV_CONNECTOR_TYPE__NEMA_5_15:
      return "EV_CONNECTOR_TYPE__NEMA_5_15";
   case EV_CONNECTOR_TYPE__NEMA_14_30:
      return "EV_CONNECTOR_TYPE__NEMA_14_30";
   case EV_CONNECTOR_TYPE__NEMA_6_20:
      return "EV_CONNECTOR_TYPE__NEMA_6_20";
   case EV_CONNECTOR_TYPE__NEMA_14_50:
      return "EV_CONNECTOR_TYPE__NEMA_14_50";
   case EV_CONNECTOR_TYPE__SEV_TYPE13:
      return "EV_CONNECTOR_TYPE__SEV_TYPE13";
   case EV_CONNECTOR_TYPE__SEV_TYPE15:
      return "EV_CONNECTOR_TYPE__SEV_TYPE15";
   case EV_CONNECTOR_TYPE__SEV_TYPE23:
      return "EV_CONNECTOR_TYPE__SEV_TYPE23";
   case EV_CONNECTOR_TYPE__SEV_TYPE25:
      return "EV_CONNECTOR_TYPE__SEV_TYPE25";
   case EV_CONNECTOR_TYPE__DOMESTIC_E_CEE_7_5:
      return "EV_CONNECTOR_TYPE__DOMESTIC_E_CEE_7_5";
   case EV_CONNECTOR_TYPE__DOMESTIC_I:
      return "EV_CONNECTOR_TYPE__DOMESTIC_I";
   case EV_CONNECTOR_TYPE__DOMESTIC_D_BS_546:
      return "EV_CONNECTOR_TYPE__DOMESTIC_D_BS_546";
   case EV_CONNECTOR_TYPE__DOMESTIC_F_CEE_7_4:
      return "EV_CONNECTOR_TYPE__DOMESTIC_F_CEE_7_4";
   case EV_CONNECTOR_TYPE__DOMESTIC_EF_CEE_7_7:
      return "EV_CONNECTOR_TYPE__DOMESTIC_EF_CEE_7_7";
   case EV_CONNECTOR_TYPE__DOMESTIC_H_SI_32:
      return "EV_CONNECTOR_TYPE__DOMESTIC_H_SI_32";
   case EV_CONNECTOR_TYPE__DOMESTIC_I_ASNZ_3112:
      return "EV_CONNECTOR_TYPE__DOMESTIC_I_ASNZ_31120";
   case EV_CONNECTOR_TYPE__DOMESTIC_I_CPCS_CCC:
      return "EV_CONNECTOR_TYPE__DOMESTIC_I_CPCS_CCC";
   case EV_CONNECTOR_TYPE__DOMESTIC_I_IRAM_2073:
      return "EV_CONNECTOR_TYPE__DOMESTIC_I_IRAM_2073";
   case EV_CONNECTOR_TYPE__DOMESTIC_K:
      return "EV_CONNECTOR_TYPE__DOMESTIC_K";
   case EV_CONNECTOR_TYPE__DOMESTIC_K_THAI:
      return "EV_CONNECTOR_TYPE__DOMESTIC_K_THAI";
   case EV_CONNECTOR_TYPE__DOMESTIC_L:
      return "EV_CONNECTOR_TYPE__DOMESTIC_L";
   case EV_CONNECTOR_TYPE__DOMESTIC_M:
      return "EV_CONNECTOR_TYPE__DOMESTIC_M";
   case EV_CONNECTOR_TYPE__DOMESTIC_IEC_60906_1:
      return "EV_CONNECTOR_TYPE__DOMESTIC_IEC_60906_1";
   case EV_CONNECTOR_TYPE__DOMESTIC_G_BS_1363:
      return "EV_CONNECTOR_TYPE__DOMESTIC_G_BS_1363";
   case EV_CONNECTOR_TYPE__EN50075_EUROPLUG:
      return "EV_CONNECTOR_TYPE__EN50075_EUROPLUG";
   case EV_CONNECTOR_TYPE__KOREA_SPECIFIC_EVSE_PLUG:
      return "EV_CONNECTOR_TYPE__KOREA_SPECIFIC_EVSE_PLUG";
   default:
      ::std::stringstream stream;
      stream << "EV_CONNECTOR_TYPE__<" << static_cast<unsigned int>(connectorType) << ">";
      return stream.str();
   }
}

/**
 * Container class holding different kind of attributes associated with a location (either address or POI)
 */
class LocationAttributeInfos
{
public:
   /**
    * Base attribute info class
    */
   class LocationAttributeInfo
   {
   public:
      enum AttributeType
      {
         ATTRIBUTE_TYPE__UNDEFINED,
         ATTRIBUTE_TYPE__LOCATION_NAME,
         ATTRIBUTE_TYPE__LOCATION_NAME_PHONEME,
         ATTRIBUTE_TYPE__POI_CATEGORY,
         ATTRIBUTE_TYPE__PARKING,
         ATTRIBUTE_TYPE__OCCUPANCY,
         ATTRIBUTE_TYPE__FUEL_PRICE,
         ATTRIBUTE_TYPE__ADDRESS_DETAILS,
         ATTRIBUTE_TYPE__OPENING_HOURS,
         ATTRIBUTE_TYPE__OPENING_HOURS_DETAIL,
         ATTRIBUTE_TYPE__OPENING_STATUS,
         ATTRIBUTE_TYPE__PAYMENT_METHODS,
         ATTRIBUTE_TYPE__PHONE_NUMBER,
         ATTRIBUTE_TYPE__LAST_UPDATE_TIME,
         ATTRIBUTE_TYPE__LAST_UPDATE_STATUS,
         ATTRIBUTE_TYPE__DESTINATION_MEMORY_ALIAS,
         ATTRIBUTE_TYPE__BOUNDING_BOX,
         ATTRIBUTE_TYPE__BRAND_NAME,
         ATTRIBUTE_TYPE__EV_CHARGING_TYPE_OCCUPANCY,
         ATTRIBUTE_TYPE__EV_CHARGING_SPEEDS,
         ATTRIBUTE_TYPE__EV_CONNECTOR_TYPES,
         ATTRIBUTE_TYPE__ENERGY_PROVIDER,
         ATTRIBUTE_TYPE__AIR_DISTANCE,
         ATTRIBUTE_TYPE__WEBSITE,
         ATTRIBUTE_TYPE__CONTENT_PROVIDER,
         ATTRIBUTE_TYPE__PICTURE_REFERENCE,
         ATTRIBUTE_TYPE__USER_RANKING,
         ATTRIBUTE_TYPE__REVIEW,
         ATTRIBUTE_TYPE__SAFETYCAMERA_TYPE,
         ATTRIBUTE_TYPE__SAFETYCAMERA_SPEEDLIMIT,
         ATTRIBUTE_TYPE__HOUSE_NUMBER_RANGE,
         ATTRIBUTE_TYPE__BOOKING,
         ATTRIBUTE_TYPE__UNIQUE_ID,
         ATTRIBUTE_TYPE__ASSOCIATED_SERVICES,
         ATTRIBUTE_TYPE__SPECIAL_RESTRICTIONS,
         ATTRIBUTE_TYPE__SPECIAL_OFFER,
         ATTRIBUTE_TYPE__DIRECTION_DESCRIPTION,
         ATTRIBUTE_TYPE__PRICE_INFO,
         ATTRIBUTE_TYPE__DESCRIPTION   //<-- INF4CV. Description of USB POI entry
      };
      static ::std::string toString(AttributeType attributeType)
      {
         switch (attributeType)
         {
         case ATTRIBUTE_TYPE__UNDEFINED:
            return "ATTRIBUTE_TYPE__UNDEFINED";
         case ATTRIBUTE_TYPE__LOCATION_NAME:
            return "ATTRIBUTE_TYPE__LOCATION_NAME";
         case ATTRIBUTE_TYPE__LOCATION_NAME_PHONEME:
            return "ATTRIBUTE_TYPE__LOCATION_NAME_PHONEME";
         case ATTRIBUTE_TYPE__POI_CATEGORY:
            return "ATTRIBUTE_TYPE__POI_CATEGORY";
         case ATTRIBUTE_TYPE__PARKING:
            return "ATTRIBUTE_TYPE__PARKING";
         case ATTRIBUTE_TYPE__OCCUPANCY:
            return "ATTRIBUTE_TYPE__OCCUPANCY";
         case ATTRIBUTE_TYPE__FUEL_PRICE:
            return "ATTRIBUTE_TYPE__FUEL_PRICE";
         case ATTRIBUTE_TYPE__ADDRESS_DETAILS:
            return "ATTRIBUTE_TYPE__ADDRESS_DETAILS";
         case ATTRIBUTE_TYPE__OPENING_HOURS:
            return "ATTRIBUTE_TYPE__OPENING_HOURS";
         case ATTRIBUTE_TYPE__OPENING_HOURS_DETAIL:
            return "ATTRIBUTE_TYPE__OPENING_HOURS_DETAIL";
         case ATTRIBUTE_TYPE__OPENING_STATUS:
            return "ATTRIBUTE_TYPE__OPENING_STATUS";
         case ATTRIBUTE_TYPE__PAYMENT_METHODS:
            return "ATTRIBUTE_TYPE__PAYMENT_METHODS";
         case ATTRIBUTE_TYPE__PHONE_NUMBER:
            return "ATTRIBUTE_TYPE__PHONE_NUMBER";
         case ATTRIBUTE_TYPE__LAST_UPDATE_TIME:
            return "ATTRIBUTE_TYPE__LAST_UPDATE_TIME";
         case ATTRIBUTE_TYPE__LAST_UPDATE_STATUS:
            return "ATTRIBUTE_TYPE__LAST_UPDATE_STATUS";
         case ATTRIBUTE_TYPE__DESTINATION_MEMORY_ALIAS:
            return "ATTRIBUTE_TYPE__DESTINATION_MEMORY_ALIAS";
         case ATTRIBUTE_TYPE__BOUNDING_BOX:
            return "ATTRIBUTE_TYPE__BOUNDING_BOX";
         case ATTRIBUTE_TYPE__BRAND_NAME:
            return "ATTRIBUTE_TYPE__BRAND_NAME";
         case ATTRIBUTE_TYPE__EV_CHARGING_TYPE_OCCUPANCY:
            return "ATTRIBUTE_TYPE__EV_CHARGING_TYPE_OCCUPANCY";
         case ATTRIBUTE_TYPE__EV_CHARGING_SPEEDS:
            return "ATTRIBUTE_TYPE__EV_CHARGING_SPEEDS";
         case ATTRIBUTE_TYPE__EV_CONNECTOR_TYPES:
            return "ATTRIBUTE_TYPE__EV_CONNECTOR_TYPES";
         case ATTRIBUTE_TYPE__ENERGY_PROVIDER:
            return "ATTRIBUTE_TYPE__ENERGY_PROVIDER";
         case ATTRIBUTE_TYPE__AIR_DISTANCE:
            return "ATTRIBUTE_TYPE__AIR_DISTANCE";
         case ATTRIBUTE_TYPE__WEBSITE:
            return "ATTRIBUTE_TYPE__WEBSITE";
         case ATTRIBUTE_TYPE__CONTENT_PROVIDER:
            return "ATTRIBUTE_TYPE__CONTENT_PROVIDER";
         case ATTRIBUTE_TYPE__PICTURE_REFERENCE:
            return "ATTRIBUTE_TYPE__PICTURE_REFERENCE";
         case ATTRIBUTE_TYPE__USER_RANKING:
            return "ATTRIBUTE_TYPE__USER_RANKING";
         case ATTRIBUTE_TYPE__REVIEW:
            return "ATTRIBUTE_TYPE__REVIEW";
         case ATTRIBUTE_TYPE__SAFETYCAMERA_TYPE:
            return "ATTRIBUTE_TYPE__SAFETYCAMERA_TYPE";
         case ATTRIBUTE_TYPE__SAFETYCAMERA_SPEEDLIMIT:
            return "ATTRIBUTE_TYPE__SAFETYCAMERA_SPEEDLIMIT";
         case ATTRIBUTE_TYPE__HOUSE_NUMBER_RANGE:
            return "ATTRIBUTE_TYPE__HOUSE_NUMBER_RANGE";
         case ATTRIBUTE_TYPE__BOOKING:
            return "ATTRIBUTE_TYPE__BOOKING";
         case ATTRIBUTE_TYPE__UNIQUE_ID:
            return "ATTRIBUTE_TYPE__UNIQUE_ID";
         case ATTRIBUTE_TYPE__ASSOCIATED_SERVICES:
            return "ATTRIBUTE_TYPE__ASSOCIATED_SERVICES";
         case ATTRIBUTE_TYPE__SPECIAL_RESTRICTIONS:
            return "ATTRIBUTE_TYPE__SPECIAL_RESTRICTIONS";
         case ATTRIBUTE_TYPE__SPECIAL_OFFER:
            return "ATTRIBUTE_TYPE__SPECIAL_OFFER";
         case ATTRIBUTE_TYPE__DIRECTION_DESCRIPTION:
            return "ATTRIBUTE_TYPE__DIRECTION_DESCRIPTION";
         case ATTRIBUTE_TYPE__PRICE_INFO:
            return "ATTRIBUTE_TYPE__PRICE_INFO";
         //<-- INF4CV
         case ATTRIBUTE_TYPE__DESCRIPTION:
            return "ATTRIBUTE_TYPE__DESCRIPTION";
         //->
         default:
            ::std::stringstream stream;
            stream << "ATTRIBUTE_TYPE__<" << static_cast<unsigned int>(attributeType) << ">";
            return stream.str();
         }
      }
   protected:
      LocationAttributeInfo(AttributeType type, const ::std::string& typeName)
         : m_type(type)
         , m_typeName(typeName)
      {}

   public:
      virtual ~LocationAttributeInfo() {}
      virtual LocationAttributeInfo* clone() const = 0;

      bool operator==(const LocationAttributeInfo& rhs) const
      {
         return (m_type == rhs.m_type)
                && (m_typeName == rhs.m_typeName);
      }

      bool operator!=(const LocationAttributeInfo& rhs) const
      {
         return !(*this == rhs);
      }

      virtual ::std::string toString() const
      {
         ::std::stringstream stream("");
         stream << "Type = "<< toString(m_type) << ", "
                << "TypeName = "<< m_typeName;
         return stream.str();
      }

      AttributeType  m_type;
      ::std::string  m_typeName;
   };

   /**
    *  Generic attribute class usable for all kind of attributes expressible by a simple string
    */
   class StringAttributeInfo : public LocationAttributeInfo
   {
   public:
      explicit StringAttributeInfo(AttributeType type)
         : LocationAttributeInfo(type, "")
      {}
      virtual ~StringAttributeInfo() {}

      StringAttributeInfo(
         AttributeType type,
         const ::std::string& typeName,
         const ::std::string& content)

         : LocationAttributeInfo(type, typeName)
         , m_content(content)
      {}

      virtual StringAttributeInfo* clone() const
      {
         return new StringAttributeInfo(*this);
      }

      bool operator==(const StringAttributeInfo& rhs) const
      {
         return LocationAttributeInfo::operator== (rhs)
                && (m_content == rhs.m_content);
      }
      bool operator!=(const StringAttributeInfo& rhs) const
      {
         return !(*this == rhs);
      }
      virtual ::std::string toString() const
      {
         ::std::stringstream stream("StringAttributeInfo payload: ");
         stream << LocationAttributeInfo::toString() << ", "
                << "content = "<< m_content << ::std::endl;
         return stream.str();
      }

      ::std::string m_content;
   };

   /**
    * Info-Class with information to the category of a POI location
    */
   class PoiCategoryInfo : public LocationAttributeInfo
   {
   public:
      PoiCategoryInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__POI_CATEGORY, "")
         , m_predefinedPoiCategory(PREDEFINED_POI_CATEGORY__UNDEFINED)
         , m_poiCategoryId(0)
      {}
      virtual ~PoiCategoryInfo() {}

      PoiCategoryInfo(
         const ::std::string& name,
         const ::std::string& categoryName,
         const ::std::string& categoryNamePhoneme,
         PredefinedPoiCategory predefinedPoiCategory,
         PoiCategoryId poiCategoryId)

         : LocationAttributeInfo(ATTRIBUTE_TYPE__POI_CATEGORY, name)
         , m_categoryName(categoryName)
         , m_categoryNamePhoneme(categoryNamePhoneme)
         , m_predefinedPoiCategory(predefinedPoiCategory)
         , m_poiCategoryId(poiCategoryId)
      {}

      virtual LocationAttributeInfo* clone() const
      {
         return new PoiCategoryInfo(*this);
      }

      bool operator==(const PoiCategoryInfo& rhs) const
      {
         return LocationAttributeInfo::operator== (rhs)
                && (m_categoryName == rhs.m_categoryName)
                && (m_predefinedPoiCategory == rhs.m_predefinedPoiCategory)
                && (m_poiCategoryId == rhs.m_poiCategoryId)
                && (m_categoryNamePhoneme == rhs.m_categoryNamePhoneme);
      }
      bool operator!=(const PoiCategoryInfo& rhs) const
      {
         return !(*this == rhs);
      }
      virtual ::std::string toString() const
      {
         ::std::stringstream stream("PoiCategoryInfo payload: ");
         stream << LocationAttributeInfo::toString() << ", "
                << "m_categoryName = "<< m_categoryName << ", "
                << "m_predefinedPoiCategory = "<< ::navmiddleware::toString(m_predefinedPoiCategory) << ","
                << "m_poiCategoryId = " << m_poiCategoryId << ","
                << "m_categoryNamePhoneme = "<< m_categoryNamePhoneme << ::std::endl;
         return stream.str();
      }

      ::std::string m_categoryName;
      ::std::string m_categoryNamePhoneme;
      PredefinedPoiCategory m_predefinedPoiCategory;
      PoiCategoryId m_poiCategoryId;
   };

   /**
    * Info-Class with information to occupancy (of a POI)
    */
   class OccupancyInfo : public LocationAttributeInfo
   {
   public:
      enum Tendency
      {
         TENDENCY__RED,
         TENDENCY__YELLOW,
         TENDENCY__GREEN,
      };
      static ::std::string toString(Tendency tendency)
      {
         switch (tendency)
         {
         case TENDENCY__RED:
            return "TENDENCY__RED";
         case TENDENCY__YELLOW:
            return "TENDENCY__YELLOW";
         case TENDENCY__GREEN:
            return "TENDENCY__GREEN";
         default:
            ::std::stringstream stream;
            stream << "TENDENCY__<" << static_cast<unsigned int>(tendency) << ">";
            return stream.str();
         }
      }
      OccupancyInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__OCCUPANCY, "")
         , m_totalCapacity(0)
      {}
      virtual ~OccupancyInfo() {}

      OccupancyInfo(
         const ::std::string& typeName,
         unsigned int totalCapacity,
         ValidValue<unsigned int> availableSpaces,
         ValidValue<Tendency> tendency,
         const ::std::string& timeStamp)

         : LocationAttributeInfo(ATTRIBUTE_TYPE__OCCUPANCY, typeName)
         , m_totalCapacity(totalCapacity)
         , m_availableSpaces(availableSpaces)
         , m_tendency(tendency)
         , m_dataAcquisitionTime(timeStamp)
      {}

      virtual LocationAttributeInfo* clone() const
      {
         return new OccupancyInfo(*this);
      }

      bool operator==(const OccupancyInfo& rhs) const
      {
         return LocationAttributeInfo::operator== (rhs)
                && (m_totalCapacity == rhs.m_totalCapacity)
                && (m_availableSpaces == rhs.m_availableSpaces)
                && (m_tendency == rhs.m_tendency);
      }
      bool operator!=(const OccupancyInfo& rhs) const
      {
         return !(*this == rhs);
      }
      virtual ::std::string toString() const
      {
         ::std::stringstream stream("OccupancyInfo payload: ");
         stream << LocationAttributeInfo::toString() << ", "
                << "Capacity = "<< m_totalCapacity << ", "
                << "Available Spaces = "<< m_availableSpaces.toString() << ", "
                << "Tendency = "<< m_tendency.toString() << ", "
                << "DataAcquisionTime = "<< m_dataAcquisitionTime << ::std::endl;
         return stream.str();
      }

      unsigned int m_totalCapacity;
      ValidValue<unsigned int> m_availableSpaces;
      ValidValue<Tendency> m_tendency;
      ::std::string m_dataAcquisitionTime;
   };

   /**
    * Info-Class with information to Parking Space
    */
   class ParkingInfo : public OccupancyInfo
   {
   public:
        ParkingInfo()
           : m_isParkAndRide(false)
        {
           m_type = ATTRIBUTE_TYPE__PARKING;
        }
        virtual ~ParkingInfo() {}

        ParkingInfo(
           const ::std::string& typeName,
           unsigned int totalCapacity,
           ValidValue<unsigned int> availableSpaces,
           ValidValue<OccupancyInfo::Tendency> tendency,
           const ::std::string& timeStamp,
           const ::std::string& maxHeight,
           bool isParkAndRide)
           : OccupancyInfo(typeName, totalCapacity, availableSpaces, tendency, timeStamp)
           , m_maxHeight(maxHeight)
           , m_isParkAndRide(isParkAndRide)
        {
           m_type = ATTRIBUTE_TYPE__PARKING;
        }

        virtual LocationAttributeInfo* clone() const
        {
           return new ParkingInfo(*this);
        }

        bool operator==(const ParkingInfo& rhs) const
        {
           return OccupancyInfo::operator==(rhs)
                  && (m_maxHeight == rhs.m_maxHeight)
                  && (m_isParkAndRide == rhs.m_isParkAndRide);
        }
        bool operator!=(const ParkingInfo& rhs) const
        {
           return !(*this == rhs);
        }
        virtual ::std::string toString() const
        {
           ::std::stringstream stream("Parking payload: ");
           stream <<  OccupancyInfo::toString() << ", "
                  << "Max Height="<< m_maxHeight << ", "
                  << "Park&Ride="<< (m_isParkAndRide?"true":"false") << ::std::endl;
           return stream.str();
        }

      // m_maxHeight represent based on unit system. If MetricSystem -> meters, If ImperialSystem -> feet+inch.
      ::std::string m_maxHeight;
      bool m_isParkAndRide;
   };

   /**
    * Info-Class with information to Fuel Prices
    */
   class FuelPriceInfo : public LocationAttributeInfo
   {
   public:
      enum Rating
      {
         RATING__UNKNOWN,
         RATING__LOW,
         RATING__MEDIUM,
         RATING__HIGH
      };
      static ::std::string toString(Rating rating)
      {
         switch (rating)
         {
         case RATING__UNKNOWN:
            return "RATING__UNKNOWN";
         case RATING__LOW:
            return "RATING__LOW";
         case RATING__MEDIUM:
            return "RATING__MEDIUM";
         case RATING__HIGH:
            return "RATING__HIGH";
         default:
            ::std::stringstream stream;
            stream << "RATING__<" << static_cast<unsigned int>(rating) << ">";
            return stream.str();
         }
      }
      enum FuelUnit
      {
         FUEL_UNIT__UNDEFINED,
         FUEL_UNIT__LITER,
         FUEL_UNIT__IMPERIAL_GALLON,
         FUEL_UNIT__US_GALLON,
         FUEL_UNIT__CUBIC_METER,
         FUEL_UNIT__KILOWATT_HOUR,
         FUEL_UNIT__KILOGRAM,
         FUEL_UNIT__GAS_GALLON_EQUIVALENT,
      };
      static ::std::string toString(FuelUnit fuelUnit)
      {
         switch (fuelUnit)
         {
         case FUEL_UNIT__UNDEFINED:
            return "FUEL_UNIT__UNDEFINED";
         case FUEL_UNIT__LITER:
            return "FUEL_UNIT__LITER";
         case FUEL_UNIT__IMPERIAL_GALLON:
            return "FUEL_UNIT__IMPERIAL_GALLON";
         case FUEL_UNIT__US_GALLON:
            return "FUEL_UNIT__US_GALLON";
         case FUEL_UNIT__CUBIC_METER:
            return "FUEL_UNIT__CUBIC_METER";
         case FUEL_UNIT__KILOWATT_HOUR:
            return "FUEL_UNIT__KILOWATT_HOUR";
         case FUEL_UNIT__KILOGRAM:
            return "FUEL_UNIT__KILOGRAM";
         case FUEL_UNIT__GAS_GALLON_EQUIVALENT:
            return "FUEL_UNIT__GAS_GALLON_EQUIVALENT";
         default:
            ::std::stringstream stream;
            stream << "FUEL_UNIT__<" << static_cast<unsigned int>(fuelUnit) << ">";
            return stream.str();
         }
      }
      FuelPriceInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__FUEL_PRICE, "")
         , m_fuelUnit(FUEL_UNIT__UNDEFINED)
         , m_chargingSpeed(EV_CHARGING_SPEED__UNKNOWN)
         , m_rating(RATING__UNKNOWN)
      {}
      virtual ~FuelPriceInfo() {}

      FuelPriceInfo(const ::std::string& typeName,
                    const ::std::string& fuelType,
                    const ::std::string& price,
                    FuelUnit fuelUnit,
                    EvChargingSpeed chargingSpeed,
                    const ::std::string& timeStamp,
                    Rating rating)
         : LocationAttributeInfo(ATTRIBUTE_TYPE__FUEL_PRICE, typeName)
         , m_fuelType(fuelType)
         , m_price(price)
         , m_fuelUnit(fuelUnit)
         , m_chargingSpeed(chargingSpeed)
         , m_dataAcquisitionTime(timeStamp)
         , m_rating(rating)
      {}
      virtual LocationAttributeInfo* clone() const
      {
         return new FuelPriceInfo(*this);
      }
      bool operator==(const FuelPriceInfo& rhs) const
      {
         return LocationAttributeInfo::operator== (rhs)
                && (m_fuelType == rhs.m_fuelType)
                && (m_price == rhs.m_price)
                && (m_fuelUnit == rhs.m_fuelUnit)
                && (m_chargingSpeed == rhs.m_chargingSpeed)
                && (m_dataAcquisitionTime == rhs.m_dataAcquisitionTime)
                && (m_rating == rhs.m_rating);
      }
      bool operator!=(const FuelPriceInfo& rhs) const
      {
         return !(*this == rhs);
      }
      virtual ::std::string toString() const
      {
         ::std::stringstream stream("FuelPriceInfo payload: ");
         stream << LocationAttributeInfo::toString() << ", "
                << "Fuel Type = "<<m_fuelType << ", "
                << "Fuel Price = "<<m_price << ", "
                << "Fuel Unit = "<<toString(m_fuelUnit)<< ", ";
         if ( FUEL_UNIT__KILOWATT_HOUR == m_fuelUnit )
         {
            stream << "ChargingSpeed = " << ::navmiddleware::toString(m_chargingSpeed) << ", ";
         }
         stream << "Data Acquisition Time = "<< m_dataAcquisitionTime<<", "
                << "Rating = "<< toString(m_rating)<< ::std::endl;
         return stream.str();
      }

      ::std::string m_fuelType;
      ::std::string m_price;
      FuelUnit      m_fuelUnit;
      EvChargingSpeed m_chargingSpeed; // only used for FUEL_UNIT__KILOWATT_HOUR
      ::std::string m_dataAcquisitionTime;
      Rating        m_rating;

   };

   /**
    * Info-Class providing availability of EV charging spots of certain charging speed
    */
   class EvChargingTypeOccupancyInfo : public OccupancyInfo
   {
   public:
      EvChargingTypeOccupancyInfo()
         : m_chargingSpeed(EV_CHARGING_SPEED__UNKNOWN)
      {
         m_type = ATTRIBUTE_TYPE__EV_CHARGING_TYPE_OCCUPANCY;
      }
      virtual ~EvChargingTypeOccupancyInfo() {}

      EvChargingTypeOccupancyInfo(
         const ::std::string& typeName,
         EvChargingSpeed chargingSpeed,
         unsigned int totalCapacity,
         ValidValue<unsigned int> availableSpaces,
         ValidValue<OccupancyInfo::Tendency> tendency,
         const ::std::string& timeStamp)

         : OccupancyInfo(typeName, totalCapacity, availableSpaces, tendency, timeStamp)
         , m_chargingSpeed(chargingSpeed)
      {
         m_type = ATTRIBUTE_TYPE__EV_CHARGING_TYPE_OCCUPANCY;
      }

      virtual LocationAttributeInfo* clone() const
      {
         return new EvChargingTypeOccupancyInfo(*this);
      }

      bool operator==(const EvChargingTypeOccupancyInfo& rhs) const
      {
         return OccupancyInfo::operator==(rhs)
                && (m_chargingSpeed == rhs.m_chargingSpeed);
      }
      bool operator!=(const EvChargingTypeOccupancyInfo& rhs) const
      {
         return !(*this == rhs);
      }
      virtual ::std::string toString() const
      {
         ::std::stringstream stream("EvChargingTypeOccupancyInfo payload: ");
         stream << OccupancyInfo::toString() << ", "
                << "ChargingSpeed = "<< ::navmiddleware::toString(m_chargingSpeed) << ":" << ::std::endl;
         return stream.str();
      }

      EvChargingSpeed m_chargingSpeed;
   };

   /**
    * Info-Class with information about supported EV charging speeds
    */
   class EvChargingSpeedsInfo : public LocationAttributeInfo
   {
   public:
      EvChargingSpeedsInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__EV_CHARGING_SPEEDS, "")
      {}
      virtual ~EvChargingSpeedsInfo() {}

      EvChargingSpeedsInfo(
         const ::std::string& typeName,
         const ::std::vector<EvChargingSpeed>& chargingSpeeds)

         : LocationAttributeInfo(ATTRIBUTE_TYPE__EV_CHARGING_SPEEDS, typeName)
         , m_chargingSpeeds(chargingSpeeds)
      {}

      virtual LocationAttributeInfo* clone() const
      {
         return new EvChargingSpeedsInfo(*this);
      }

      bool operator==(const EvChargingSpeedsInfo& rhs) const
      {
         return LocationAttributeInfo::operator== (rhs)
                && (m_chargingSpeeds == rhs.m_chargingSpeeds);
      }
      bool operator!=(const EvChargingSpeedsInfo& rhs) const
      {
         return !(*this == rhs);
      }
      virtual ::std::string toString() const
      {
         ::std::stringstream stream("EvChargingSpeeds payload: ");
         stream << LocationAttributeInfo::toString() << ", "
                << "(num =" << m_chargingSpeeds.size() << "): ";
         for (unsigned int listIndex = 0; listIndex < m_chargingSpeeds.size(); ++listIndex)
         {
            stream << (listIndex > 0? ", ": "") << ::navmiddleware::toString(m_chargingSpeeds[listIndex]);
         }
         stream << ::std::endl;
         return stream.str();
      }

      ::std::vector<EvChargingSpeed> m_chargingSpeeds;
   };

   /**
    * Info-Class with information about supported EV connector types
    */
   class EvConnectorTypesInfo : public LocationAttributeInfo
   {
   public:
      EvConnectorTypesInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__EV_CONNECTOR_TYPES, "")
      {}
      virtual ~EvConnectorTypesInfo() {}

      EvConnectorTypesInfo(
         const ::std::string& typeName,
         const ::std::vector<EvConnectorType>& connectorTypes)

         : LocationAttributeInfo(ATTRIBUTE_TYPE__EV_CONNECTOR_TYPES, typeName)
         , m_connectorTypes(connectorTypes)
      {}

      virtual LocationAttributeInfo* clone() const
      {
         return new EvConnectorTypesInfo(*this);
      }

      bool operator==(const EvConnectorTypesInfo& rhs) const
      {
         return LocationAttributeInfo::operator== (rhs)
                && (m_connectorTypes == rhs.m_connectorTypes);
      }
      bool operator!=(const EvConnectorTypesInfo& rhs) const
      {
         return !(*this == rhs);
      }
      virtual ::std::string toString() const
      {
         ::std::stringstream stream("EvConnectorTypes payload: ");
         stream << LocationAttributeInfo::toString() << ", "
                << "(num =" << m_connectorTypes.size() << "): ";
         for (unsigned int listIndex = 0; listIndex < m_connectorTypes.size(); ++listIndex)
         {
            stream << (listIndex > 0? ", ": "") << ::navmiddleware::toString(m_connectorTypes[listIndex]);
         }
         stream << ::std::endl;
         return stream.str();
      }

      ::std::vector<EvConnectorType> m_connectorTypes;
   };

   /**
    * Info class to represent the address details
    */
   class AddressDetailInfo : public LocationAttributeInfo
   {
   public:
      enum AddressInfoType
      {
         ADDRESS_INFO_TYPE__FULL_ADDRESS,
         ADDRESS_INFO_TYPE__COUNTRY_NAME,
         ADDRESS_INFO_TYPE__COUNTRY_CODE,
         ADDRESS_INFO_TYPE__STATE_NAME,
         ADDRESS_INFO_TYPE__STATE_CODE,
         ADDRESS_INFO_TYPE__CITY_NAME,
         ADDRESS_INFO_TYPE__CITY_ADD_ON,
         ADDRESS_INFO_TYPE__CITY_DISTRICT,
         ADDRESS_INFO_TYPE__ZIP_CODE,
         ADDRESS_INFO_TYPE__STREET_NAME,
         ADDRESS_INFO_TYPE__HOUSE_NUMBER,
         ADDRESS_INFO_TYPE__CROSS_STREET_NAME,
         ADDRESS_INFO_TYPE__PREFECTURE_NAME,
         ADDRESS_INFO_TYPE__WARD_NAME,
         ADDRESS_INFO_TYPE__TOWN_NAME,
         ADDRESS_INFO_TYPE__BLOCK_NUMBER,
         ADDRESS_INFO_TYPE__BUILDING_NAME,
         ADDRESS_INFO_TYPE__FLOOR_NUMBER,
      };
      static ::std::string toString(AddressInfoType addressInfoType)
      {
          switch (addressInfoType)
          {
          case ADDRESS_INFO_TYPE__FULL_ADDRESS:
              return "ADDRESS_INFO_TYPE__FULL_ADDRESS";
          case ADDRESS_INFO_TYPE__COUNTRY_NAME:
              return "ADDRESS_INFO_TYPE__COUNTRY_NAME";
          case ADDRESS_INFO_TYPE__COUNTRY_CODE:
              return "ADDRESS_INFO_TYPE__COUNTRY_CODE";
          case ADDRESS_INFO_TYPE__STATE_NAME:
              return "ADDRESS_INFO_TYPE__STATE_NAME";
          case ADDRESS_INFO_TYPE__STATE_CODE:
              return "ADDRESS_INFO_TYPE__STATE_CODE";
          case ADDRESS_INFO_TYPE__CITY_NAME:
              return "ADDRESS_INFO_TYPE__CITY_NAME";
          case ADDRESS_INFO_TYPE__CITY_ADD_ON:
              return "ADDRESS_INFO_TYPE__CITY_ADD_ON";
          case ADDRESS_INFO_TYPE__CITY_DISTRICT:
              return "ADDRESS_INFO_TYPE__CITY_DISTRICT";
          case ADDRESS_INFO_TYPE__ZIP_CODE:
              return "ADDRESS_INFO_TYPE__ZIP_CODE";
          case ADDRESS_INFO_TYPE__STREET_NAME:
              return "ADDRESS_INFO_TYPE__STREET_NAME";
          case ADDRESS_INFO_TYPE__HOUSE_NUMBER:
              return "ADDRESS_INFO_TYPE__HOUSE_NUMBER";
          case ADDRESS_INFO_TYPE__CROSS_STREET_NAME:
              return "ADDRESS_INFO_TYPE__CROSS_STREET_NAME";
          case ADDRESS_INFO_TYPE__PREFECTURE_NAME:
              return "ADDRESS_INFO_TYPE__PREFECTURE_NAME";
          case ADDRESS_INFO_TYPE__WARD_NAME:
              return "ADDRESS_INFO_TYPE__WARD_NAME";
          case ADDRESS_INFO_TYPE__TOWN_NAME:
              return "ADDRESS_INFO_TYPE__TOWN_NAME";
          case ADDRESS_INFO_TYPE__BLOCK_NUMBER:
              return "ADDRESS_INFO_TYPE__BLOCK_NUMBER";
          case ADDRESS_INFO_TYPE__BUILDING_NAME:
              return "ADDRESS_INFO_TYPE__BUILDING_NAME";
          case ADDRESS_INFO_TYPE__FLOOR_NUMBER:
              return "ADDRESS_INFO_TYPE__FLOOR_NUMBER";
         default:
            ::std::stringstream stream;
            stream << "ADDRESS_INFO_TYPE__<" << static_cast<unsigned int>(addressInfoType) << ">";
            return stream.str();
         }
      }
      struct AddressInfoContent
      {
         ::std::string typeName;
         ::std::string value;
         ::std::string valuePhoneme;
         bool operator==(const AddressInfoContent& rhs) const
         {
            return ( (typeName == rhs.typeName) && (value == rhs.value) && (valuePhoneme == rhs.valuePhoneme));
         }
         bool operator!=(const AddressInfoContent& rhs) const
         {
            return !(*this == rhs);
         }
      };
      typedef ::std::map< AddressInfoType, AddressInfoContent> T_AddressInfoMap;

      AddressDetailInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__ADDRESS_DETAILS, "")
      {}
      virtual ~AddressDetailInfo() {}

      virtual LocationAttributeInfo* clone() const
      {
         return new AddressDetailInfo(*this);
      }
      bool operator==(const AddressDetailInfo& rhs) const
      {
         return LocationAttributeInfo::operator== (rhs)
                && (m_addressInfoMap == rhs.m_addressInfoMap);
      }
      bool operator!=(const AddressDetailInfo& rhs) const
      {
         return !(*this == rhs);
      }
      virtual ::std::string toString() const
      {
         ::std::stringstream stream("AddressInfo payload: ");
         stream << LocationAttributeInfo::toString() << ::std::endl;
         for (T_AddressInfoMap::const_iterator it = m_addressInfoMap.begin(); it != m_addressInfoMap.end(); ++it)
         {
            unsigned int count = 0;
            stream << "AddressInfoType[" << count << "] " << toString(it->first) << " [" << "] " << it->second.typeName << " [" << "] " << it->second.value << std::endl;
         }
         return stream.str();
      }
      void setAddressInfoMap(const T_AddressInfoMap& addressMap)
      {
         m_addressInfoMap = addressMap;
      }
      const T_AddressInfoMap& getAddressInfoMap() const
      {
         return m_addressInfoMap;
      }
      const ::std::string& getAddressValue(AddressInfoType type) const
      {
         T_AddressInfoMap::const_iterator it = m_addressInfoMap.find(type);
         if(it != m_addressInfoMap.end())
         {
            return it->second.value;
         }
         else
         {
            static const ::std::string str;
            return str;
         }
      }
   private :
      T_AddressInfoMap m_addressInfoMap;
   };

   /**
    * Info-Class with information related to the accepted payment methods
    */
   class PaymentMethodsInfo : public LocationAttributeInfo
   {
   public:
      enum PaymentMethod
      {
         PAYMENT_METHOD__NO_PAYMENT_REQUIRED,
         PAYMENT_METHOD__UNKNOWN_METHOD,
         PAYMENT_METHOD__CASH,
         PAYMENT_METHOD__COINS_ONLY,
         PAYMENT_METHOD__CREDIT_CARD,
         PAYMENT_METHOD__BANK_OR_DEBIT_CARD,
         PAYMENT_METHOD__ELECTRONIC_PURSE,
         PAYMENT_METHOD__ELECTRONIC_TOLL_COLLECTION,
         PAYMENT_METHOD__VARIABLE,
         PAYMENT_METHOD__SERVICE_PROVIDER_SPECIFIC,
         PAYMENT_METHOD__ATM,
         PAYMENT_METHOD__RFID,
         PAYMENT_METHOD__MOBILE_PHONE,
         PAYMENT_METHOD__SMART_CARD
      };
      static ::std::string toString(PaymentMethod paymentMethod)
      {
         switch (paymentMethod)
         {
         case PAYMENT_METHOD__NO_PAYMENT_REQUIRED:
            return "PAYMENT_METHOD__NO_PAYMENT_REQUIRED";
         case PAYMENT_METHOD__UNKNOWN_METHOD:
            return "PAYMENT_METHOD__UNKNOWN_METHOD";
         case PAYMENT_METHOD__CASH:
            return "PAYMENT_METHOD__CASH";
         case PAYMENT_METHOD__COINS_ONLY:
            return "PAYMENT_METHOD__COINS_ONLY";
         case PAYMENT_METHOD__CREDIT_CARD:
            return "PAYMENT_METHOD__CREDIT_CARD";
         case PAYMENT_METHOD__BANK_OR_DEBIT_CARD:
            return "PAYMENT_METHOD__BANK_OR_DEBIT_CARD";
         case PAYMENT_METHOD__ELECTRONIC_PURSE:
            return "PAYMENT_METHOD__ELECTRONIC_PURSE";
         case PAYMENT_METHOD__ELECTRONIC_TOLL_COLLECTION:
            return "PAYMENT_METHOD__ELECTRONIC_TOLL_COLLECTION";
         case PAYMENT_METHOD__VARIABLE:
            return "PAYMENT_METHOD__VARIABLE";
         case PAYMENT_METHOD__SERVICE_PROVIDER_SPECIFIC:
            return "PAYMENT_METHOD__SERVICE_PROVIDER_SPECIFIC";
         case PAYMENT_METHOD__ATM:
            return "PAYMENT_METHOD__ATM";
         case PAYMENT_METHOD__RFID:
            return "PAYMENT_METHOD__RFID";
         case PAYMENT_METHOD__MOBILE_PHONE:
            return "PAYMENT_METHOD__MOBILE_PHONE";
         case PAYMENT_METHOD__SMART_CARD:
            return "PAYMENT_METHOD__SMART_CARD";
         default:
            ::std::stringstream stream;
            stream << "PAYMENT_METHOD__<" << static_cast<unsigned int>(paymentMethod) << ">";
            return stream.str();
         }
      }
      PaymentMethodsInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__PAYMENT_METHODS, "")
      {}
      virtual ~PaymentMethodsInfo() {}

      PaymentMethodsInfo(
         const ::std::string& typeName,
         const ::std::vector< PaymentMethod >& acceptedMethods,
         const ::std::string& additionalInformation)

         : LocationAttributeInfo(ATTRIBUTE_TYPE__PAYMENT_METHODS, typeName)
         , m_acceptedMethods(acceptedMethods)
         , m_additionalInformation(additionalInformation)
      {}

      virtual LocationAttributeInfo* clone() const
      {
         return new PaymentMethodsInfo(*this);
      }

      bool operator==(const PaymentMethodsInfo& rhs) const
      {
         return LocationAttributeInfo::operator== (rhs)
                && (m_acceptedMethods == rhs.m_acceptedMethods)
                && (m_additionalInformation == rhs.m_additionalInformation);
      }
      bool operator!=(const PaymentMethodsInfo& rhs) const
      {
         return !(*this == rhs);
      }
      virtual ::std::string toString() const
      {
         ::std::stringstream stream("PaymentMethodsInfo payload: ");
         stream << LocationAttributeInfo::toString() << ", "
                << "Total Accepted Payment Methods = " << m_acceptedMethods.size() << ::std::endl;
         for(unsigned int listIndex = 0; listIndex < m_acceptedMethods.size(); ++listIndex)
         {
            stream << "  AcceptedPaymentMethod[" << listIndex << "] = " << toString(m_acceptedMethods[listIndex]) << ::std::endl;
         }
         stream << "  Additional information = " << m_additionalInformation << ::std::endl;
         return stream.str();
      }

      ::std::vector< PaymentMethod > m_acceptedMethods;
      ::std::string m_additionalInformation;
   };

   /**
    * Info-Class with information related to the destination memory alias info
    */
   class DestinationMemoryAliasInfo : public LocationAttributeInfo
   {
   public:
      DestinationMemoryAliasInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__DESTINATION_MEMORY_ALIAS, "")
         , m_destinationMemoryEntryCategory(DESTINATION_MEMORY_ENTRY_CATEGORY__HOME)
      {}
      virtual ~DestinationMemoryAliasInfo() {}

      DestinationMemoryAliasInfo(
         const ::std::string& typeName,
         const ::std::string& name,
         const ::std::vector< ::std::string >& phoneNumberList,
         DestinationMemoryEntryCategory destinationMemoryEntryCategory)

         : LocationAttributeInfo(ATTRIBUTE_TYPE__DESTINATION_MEMORY_ALIAS, typeName)
         , m_name(name)
         , m_phoneNumberList(phoneNumberList)
         , m_destinationMemoryEntryCategory(destinationMemoryEntryCategory)
      {}

      virtual LocationAttributeInfo* clone() const
      {
         return new DestinationMemoryAliasInfo(*this);
      }
      bool operator==(const DestinationMemoryAliasInfo& rhs) const
      {
         return LocationAttributeInfo::operator== (rhs)
                && (m_name == rhs.m_name)
                && (m_phoneNumberList == rhs.m_phoneNumberList)
                && (m_destinationMemoryEntryCategory == rhs.m_destinationMemoryEntryCategory);
      }
      bool operator!=(const DestinationMemoryAliasInfo& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("DestinationMemoryAliasInfo payload:\n");
         stream << LocationAttributeInfo::toString() << ", "
                << " Name = " << m_name << ::std::endl
                << " CountOfPhoneNumbers = " << m_phoneNumberList.size() << ::std::endl
                << " DestinationMemoryEntryCategory = " << ::navmiddleware::toString(m_destinationMemoryEntryCategory)<< ::std::endl;

         for(unsigned int listIndex = 0; listIndex < m_phoneNumberList.size(); ++listIndex)
         {
            stream << "  phoneNumberList[" << listIndex << "] = " << m_phoneNumberList[listIndex] << ::std::endl;
         }


         return stream.str();
      }

      ::std::string m_name;
      ::std::vector< ::std::string > m_phoneNumberList;
      DestinationMemoryEntryCategory m_destinationMemoryEntryCategory;
   };

   /**
    * Info-Class providing status info if a POI is open or closed and such
    */
   class OpeningStatusInfo : public LocationAttributeInfo
   {
   public:
      enum OpeningStatus
      {
         OPENING_STATUS__CLOSED_24_7,
         OPENING_STATUS__OPEN_24_7,
         OPENING_STATUS__CLOSED,
         OPENING_STATUS__OPEN,
         OPENING_STATUS__CLOSING_SOON,
         OPENING_STATUS__OPENING_SOON,
      };
      static ::std::string toString(OpeningStatus openingStatus)
      {
         switch (openingStatus)
         {
         case OPENING_STATUS__CLOSED_24_7:
            return "OPENING_STATUS__CLOSED_24_7";
         case OPENING_STATUS__OPEN_24_7:
            return "OPENING_STATUS__OPEN_24_7";
         case OPENING_STATUS__CLOSED:
            return "OPENING_STATUS__CLOSED";
         case OPENING_STATUS__OPEN:
            return "OPENING_STATUS__OPEN";
         case OPENING_STATUS__CLOSING_SOON:
            return "OPENING_STATUS__CLOSING_SOON";
         case OPENING_STATUS__OPENING_SOON:
            return "OPENING_STATUS__OPENING_SOON";
         default:
            ::std::stringstream stream;
            stream << "OPENING_STATUS__<" << static_cast<unsigned int>(openingStatus) << ">";
            return stream.str();
         }
      }
      OpeningStatusInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__OPENING_STATUS, "")
         , m_openingStatus(OPENING_STATUS__CLOSED_24_7)
      {}
      virtual ~OpeningStatusInfo() {}

      OpeningStatusInfo(
         const ::std::string& typeName,
         OpeningStatus openingStatus)

         : LocationAttributeInfo(ATTRIBUTE_TYPE__OPENING_STATUS, typeName)
         , m_openingStatus(openingStatus)
      {}

      virtual LocationAttributeInfo* clone() const
      {
         return new OpeningStatusInfo(*this);
      }
      bool operator==(const OpeningStatusInfo& rhs) const
      {
         return LocationAttributeInfo::operator==(rhs)
                && (m_openingStatus == rhs.m_openingStatus);
      }
      bool operator!=(const OpeningStatusInfo& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("OpeningStatusInfo payload:");
         stream << LocationAttributeInfo::toString() << ", "
                << " Opening Status = " << toString(m_openingStatus)<< ::std::endl;
         return stream.str();
      }

      OpeningStatus m_openingStatus;
   };

   /**
    * Info-Class providing the possible bounding box of a location (e.g. in case of country, city, etc.)
    */
   class BoundingBoxInfo : public LocationAttributeInfo
   {
   public:
      BoundingBoxInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__BOUNDING_BOX, "")
      {}

      BoundingBoxInfo(
         const ::std::string& typeName,
         const GeoCoordinateDegree& lowerLeftPosition,
         const GeoCoordinateDegree& upperRightPosition)
         : LocationAttributeInfo(ATTRIBUTE_TYPE__BOUNDING_BOX, typeName)
         , m_lowerLeftPosition(lowerLeftPosition)
         , m_upperRightPosition(upperRightPosition)
      {}
      virtual ~BoundingBoxInfo() {}

      virtual LocationAttributeInfo* clone() const
      {
         return new BoundingBoxInfo(*this);
      }

      bool operator==(const BoundingBoxInfo& rhs) const
      {
         return LocationAttributeInfo::operator== (rhs)
                && (m_lowerLeftPosition  == rhs.m_lowerLeftPosition)
                && (m_upperRightPosition == rhs.m_upperRightPosition);
      }
      bool operator!=(const BoundingBoxInfo& rhs) const
      {
         return !(*this == rhs);
      }
      virtual ::std::string toString() const
      {
         ::std::stringstream stream("Bounding box payload: ");
         stream << LocationAttributeInfo::toString() << ", "
                << "lower left = " << m_lowerLeftPosition.toString() << ::std::endl
                << "upper right = " << m_upperRightPosition.toString() << ::std::endl;
         return stream.str();
      }

      GeoCoordinateDegree m_lowerLeftPosition;
      GeoCoordinateDegree m_upperRightPosition;
   };

   /**
    * Info-Class providing the last update status for the dynamic attributes.
    */
   class LastUpdateStatusInfo : public LocationAttributeInfo
   {
   public:
      enum LastUpdateStatus
      {
         LASTUPDATE_STATUS__UNKNOWN,
         LASTUPDATE_STATUS__WITHIN_LAST_24_HOURS,
         LASTUPDATE_STATUS__BEFORE_LAST_24_HOURS
      };
      static ::std::string toString(LastUpdateStatus lastUpdateStatus)
      {
         switch (lastUpdateStatus)
         {
         case LASTUPDATE_STATUS__UNKNOWN:
            return "LASTUPDATE_STATUS__UNKNOWN";
         case LASTUPDATE_STATUS__WITHIN_LAST_24_HOURS:
            return "LASTUPDATE_STATUS__WITHIN_LAST_24_HOURS";
         case LASTUPDATE_STATUS__BEFORE_LAST_24_HOURS:
            return "LASTUPDATE_STATUS__BEFORE_LAST_24_HOURS";
         default:
            ::std::stringstream stream;
            stream << "LASTUPDATE_STATUS__<" << static_cast<unsigned int>(lastUpdateStatus) << ">";
            return stream.str();
         }
      }
      LastUpdateStatusInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__LAST_UPDATE_STATUS, "")
         , m_lastUpdateStatus(LASTUPDATE_STATUS__UNKNOWN)
      {}
      virtual ~LastUpdateStatusInfo() {}

      LastUpdateStatusInfo(
         const ::std::string& typeName,
         LastUpdateStatus lastUpdateStatus,
         ValidValue<unsigned int> daysSinceLastUpdate)

         : LocationAttributeInfo(ATTRIBUTE_TYPE__LAST_UPDATE_STATUS, typeName)
         , m_lastUpdateStatus(lastUpdateStatus)
         , m_daysSinceLastUpdate(daysSinceLastUpdate)
      {}

      virtual LocationAttributeInfo* clone() const
      {
         return new LastUpdateStatusInfo(*this);
      }
      bool operator==(const LastUpdateStatusInfo& rhs) const
      {
         return LocationAttributeInfo::operator==(rhs)
                && (m_lastUpdateStatus == rhs.m_lastUpdateStatus)
                && (m_daysSinceLastUpdate == rhs.m_daysSinceLastUpdate);
      }
      bool operator!=(const LastUpdateStatusInfo& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("LastUpdateStatusInfo payload:");
         stream << LocationAttributeInfo::toString() << ", "
                << " LastUpdate Status = " << toString(m_lastUpdateStatus)<< ::std::endl
                << " Days Since last Update = " << m_daysSinceLastUpdate.toString()<< ::std::endl;
         return stream.str();
      }


      LastUpdateStatus m_lastUpdateStatus; // @deprecated: Please use m_daysSincelastUpdate
      ValidValue<unsigned int> m_daysSinceLastUpdate; // Provides the days since the last update
   };

   /**
    * Info-Class providing the Picture Reference of a location.
    */
   class PictureReferenceInfo : public LocationAttributeInfo
   {
   public:
      PictureReferenceInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__PICTURE_REFERENCE, "")
         , m_sourceId(0)
      {}
      virtual ~PictureReferenceInfo() {}

      PictureReferenceInfo(
         const ::std::string&       typeName,
         unsigned int               sourceId,
         const ::std::string&       reference,
         ValidValue<unsigned int>   maxHeightInPixel,
         ValidValue<unsigned int>   maxWidthInPixel)
         : LocationAttributeInfo(ATTRIBUTE_TYPE__PICTURE_REFERENCE, typeName)
         , m_sourceId(sourceId)
         , m_reference(reference)
         , m_maxHeightInPixel(maxHeightInPixel)
         , m_maxWidthInPixel(maxWidthInPixel)
      {}

      virtual LocationAttributeInfo* clone() const
      {
         return new PictureReferenceInfo(*this);
      }

      bool operator==(const PictureReferenceInfo& rhs) const
      {
         return LocationAttributeInfo::operator== (rhs)
                && (m_sourceId  == rhs.m_sourceId)
                && (m_reference == rhs.m_reference)
                && (m_maxHeightInPixel == rhs.m_maxHeightInPixel)
                && (m_maxWidthInPixel == rhs.m_maxWidthInPixel);
      }
      bool operator!=(const PictureReferenceInfo& rhs) const
      {
         return !(*this == rhs);
      }
      virtual ::std::string toString() const
      {
         ::std::stringstream stream("PictureReferenceInfo payload: ");
         stream << LocationAttributeInfo::toString() << ", "
                << "SourceId = " << m_sourceId << ::std::endl
                << "Reference  = " << m_reference << ::std::endl
                << "Max Height in Pixel = " << m_maxHeightInPixel.toString() << ::std::endl
                << "Max Width in Pixel = " << m_maxWidthInPixel.toString() << ::std::endl;
         return stream.str();
      }

      unsigned int               m_sourceId;
      ::std::string              m_reference;
      ValidValue<unsigned int>   m_maxHeightInPixel;
      ValidValue<unsigned int>   m_maxWidthInPixel;
   };

   /**
    * Info-Class providing the user ranking info for the dynamic attributes.
    */
   class UserRankingInfo : public LocationAttributeInfo
   {
   public:
      UserRankingInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__USER_RANKING, "")
         , m_userRanking(0)
      {}
      virtual ~UserRankingInfo() {}

      UserRankingInfo(
         const ::std::string& typeName,
         float userRanking)

         : LocationAttributeInfo(ATTRIBUTE_TYPE__USER_RANKING, typeName)
         , m_userRanking(userRanking)
      {}

      virtual LocationAttributeInfo* clone() const
      {
         return new UserRankingInfo(*this);
      }
      bool operator==(const UserRankingInfo& rhs) const
      {
         return LocationAttributeInfo::operator==(rhs)
                && (m_userRanking == rhs.m_userRanking);
      }
      bool operator!=(const UserRankingInfo& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("UserRankingInfo payload:");
         stream << LocationAttributeInfo::toString() << ", "
                << "user Ranking = " << m_userRanking << ::std::endl;
         return stream.str();
      }

      float m_userRanking;
   };

   /**
    * Info-Class providing the user reviews for poi's.
    */
   class ReviewInfos : public LocationAttributeInfo
   {
   public:
      struct ReviewInfo
      {
         ReviewInfo()
            :m_rating(0)
         {}
         ReviewInfo(
            float rating,
            const ::std::string& author,
            const ::std::string& reviewContent,
            const ::std::string& lastUpdateStatus,
            const ::std::string& timeStamp
         )
            :m_rating(rating)
            ,m_author(author)
            ,m_reviewContent(reviewContent)
            ,m_lastUpdateStatus(lastUpdateStatus)
            ,m_timeStamp(timeStamp)
         {}
         bool operator==(const ReviewInfo& rhs) const
         {
            return (m_rating == rhs.m_rating)
                   && (m_author == rhs.m_author)
                   && (m_reviewContent == rhs.m_reviewContent)
                   && (m_lastUpdateStatus == rhs.m_lastUpdateStatus)
                   && (m_timeStamp == rhs.m_timeStamp);
         }
         bool operator!=(const ReviewInfo& rhs) const
         {
            return !(*this == rhs);
         }
         ::std::string toString() const
         {
            ::std::stringstream stream;
            stream << "rating = " << m_rating << ","
                   << "author = " << m_author << ","
                   << "reviewContent = " << m_reviewContent << ","
                   << "lastUpdateStatus = " << m_lastUpdateStatus << ","
                   << "timeStamp = " << m_timeStamp << ::std::endl;
            return stream.str();
         }
         float               m_rating;
         ::std::string       m_author;
         ::std::string       m_reviewContent;
         /** Relative time from Review creation time for eg: Two Months Ago */
         ::std::string       m_lastUpdateStatus;
         /** Review creation time */
         ::std::string       m_timeStamp;
      };
      ReviewInfos()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__REVIEW, "")
         , m_countOfAllReviews(0)
      {}

      ReviewInfos(
         const ::std::string& typeName,
         const ::std::vector<ReviewInfo>& reviews,
         unsigned int countOfAllReviews)
         : LocationAttributeInfo(ATTRIBUTE_TYPE__REVIEW, typeName)
         , m_reviews(reviews)
         , m_countOfAllReviews(countOfAllReviews)
      {}

      virtual LocationAttributeInfo* clone() const
      {
         return new ReviewInfos(*this);
      }

      bool operator==(const ReviewInfos& rhs) const
      {
         return LocationAttributeInfo::operator== (rhs)
                && (m_reviews == rhs.m_reviews)
                && (m_countOfAllReviews == rhs.m_countOfAllReviews);
      }
      bool operator!=(const ReviewInfos& rhs) const
      {
         return !(*this == rhs);
      }
      virtual ::std::string toString() const
      {
         ::std::stringstream stream("ReviewInfos payload: ");
         stream << LocationAttributeInfo::toString() << ", "
                << "Total No of Reviews=" << m_countOfAllReviews << ", "
                << "Count of Reviews Available=" << m_reviews.size() <<::std::endl;
         for (unsigned int listIndex = 0; listIndex < m_reviews.size(); ++listIndex)
         {
            stream << "[" << listIndex << "] = " << m_reviews[listIndex].toString();
         }
         stream << ::std::endl;
         return stream.str();
      }

      ::std::vector<ReviewInfo> m_reviews;
      unsigned int              m_countOfAllReviews;
   };

   /**
    * Info-Class providing the safety camera type info.
    */
   class SafetyCameraTypeInfo : public LocationAttributeInfo
   {
   public:
      SafetyCameraTypeInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__SAFETYCAMERA_TYPE, "")
         , m_safetyCameraType(SAFETY_CAMERA_TYPE__UNKNOWN)
      {}
      virtual ~SafetyCameraTypeInfo() {}

      SafetyCameraTypeInfo(
         const ::std::string& typeName,
         SafetyCameraType safetyCameraType)

         : LocationAttributeInfo(ATTRIBUTE_TYPE__SAFETYCAMERA_TYPE, typeName)
         , m_safetyCameraType(safetyCameraType)
      {}

      virtual LocationAttributeInfo* clone() const
      {
         return new SafetyCameraTypeInfo(*this);
      }
      bool operator==(const SafetyCameraTypeInfo& rhs) const
      {
         return LocationAttributeInfo::operator==(rhs)
                && (m_safetyCameraType == rhs.m_safetyCameraType);
      }
      bool operator!=(const SafetyCameraTypeInfo& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("SafetyCameraTypeInfo payload:");
         stream << LocationAttributeInfo::toString() << ", "
                << "Safety Camera Type = " << ::navmiddleware::toString(m_safetyCameraType) << ::std::endl;
         return stream.str();
      }

      SafetyCameraType m_safetyCameraType;
   };

   /**
    * Info-Class providing the safety camera speed limit info.
    */
   class SafetyCameraSpeedLimitInfo : public LocationAttributeInfo
   {
   public:

      enum SpeedUnit
      {
         SPEED_UNIT__KILOMETERS_PER_HOUR,
         SPEED_UNIT__MILES_PER_HOUR,
         SPEED_UNIT__CENTIMETERS_PER_SECOND
      };

      inline ::std::string toString(SpeedUnit speedUnit) const
      {
         switch (speedUnit)
         {
         case SPEED_UNIT__KILOMETERS_PER_HOUR:
            return "SPEED_UNIT__KILOMETERS_PER_HOUR";
         case SPEED_UNIT__MILES_PER_HOUR:
            return "SPEED_UNIT__MILES_PER_HOUR";
         case SPEED_UNIT__CENTIMETERS_PER_SECOND:
            return "SPEED_UNIT__CENTIMETERS_PER_SECOND";
         default:
            ::std::stringstream stream;
            stream << "SPEED_UNIT__<" << static_cast<unsigned int>(speedUnit) << ">";
            return stream.str();
         }
      }

      SafetyCameraSpeedLimitInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__SAFETYCAMERA_SPEEDLIMIT, "")
         , m_speedUnit(SPEED_UNIT__KILOMETERS_PER_HOUR)
         , m_speedLimit(0)
      {}
      virtual ~SafetyCameraSpeedLimitInfo() {}

      SafetyCameraSpeedLimitInfo(
         const ::std::string& typeName,
         SpeedUnit speedUnit,
         uint16_t speedLimit)

         : LocationAttributeInfo(ATTRIBUTE_TYPE__SAFETYCAMERA_SPEEDLIMIT, typeName)
         , m_speedUnit(speedUnit)
         , m_speedLimit(speedLimit)
      {}

      virtual LocationAttributeInfo* clone() const
      {
         return new SafetyCameraSpeedLimitInfo(*this);
      }
      bool operator==(const SafetyCameraSpeedLimitInfo& rhs) const
      {
         return LocationAttributeInfo::operator==(rhs)
                && (m_speedUnit == rhs.m_speedUnit)
                && (m_speedLimit == rhs.m_speedLimit);
      }
      bool operator!=(const SafetyCameraSpeedLimitInfo& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("SafetyCameraSpeedLimitInfo payload:");
         stream << LocationAttributeInfo::toString() << ", "
                << "Speed Unit = " << toString(m_speedUnit) << " , "
                << "Speed Limit = " << m_speedLimit << ::std::endl;
         return stream.str();
      }

      SpeedUnit m_speedUnit;
      uint16_t  m_speedLimit;
   };

   /**
    * Info-Class providing the house number range info.
    */
   class HouseNumberRangeInfo : public LocationAttributeInfo
   {
   public :
      HouseNumberRangeInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__HOUSE_NUMBER_RANGE, "")
      {}
      virtual ~HouseNumberRangeInfo() {}
      HouseNumberRangeInfo(const std::string& typeName,
                           const std::string& minHouseNo,
                           const std::string& maxHouseNo)
         : LocationAttributeInfo(ATTRIBUTE_TYPE__HOUSE_NUMBER_RANGE, typeName)
         , m_minHouseNo(minHouseNo)
         , m_maxHouseNo(maxHouseNo)
      {}
      virtual LocationAttributeInfo* clone() const
      {
         return new HouseNumberRangeInfo(*this);
      }
      bool operator==(const HouseNumberRangeInfo& rhs) const
      {
         return LocationAttributeInfo::operator== (rhs)
                && (m_minHouseNo == rhs.m_minHouseNo)
                && (m_maxHouseNo == rhs.m_maxHouseNo);
      }
      bool operator!=(const HouseNumberRangeInfo& rhs) const
      {
         return !(*this == rhs);
      }
      virtual ::std::string toString() const
      {
         ::std::stringstream stream("HouseNumberRangeInfo payload: ");
         stream << LocationAttributeInfo::toString() << ", "
                << "Min HouseNo = "<< m_minHouseNo << ", "
                << "Max HouseNo = "<< m_maxHouseNo <<::std::endl;
         return stream.str();
      }

      ::std::string m_minHouseNo;
      ::std::string m_maxHouseNo;
   };

   /**
    * Info-Class providing the Booking info.
    */
   class BookingInfo : public LocationAttributeInfo
   {
   public:
      enum Reservability
      {
         RESERVABILITY__UNKNOWN,
         RESERVABILITY__RESERVATION_REQUIRED,
         RESERVABILITY__RESERVATION_RECOMMENDED,
         RESERVABILITY__NOT_RESERVABLE
      };

      static ::std::string toString(Reservability reservability)
      {
         switch (reservability)
         {
         case RESERVABILITY__UNKNOWN:
            return "RESERVABILITY__UNKNOWN";
         case RESERVABILITY__RESERVATION_REQUIRED:
            return "RESERVABILITY__RESERVATION_REQUIRED";
         case RESERVABILITY__RESERVATION_RECOMMENDED:
            return "RESERVABILITY__RESERVATION_RECOMMENDED";
         case RESERVABILITY__NOT_RESERVABLE:
            return "RESERVABILITY__NOT_RESERVABLE";
         default:
            ::std::stringstream stream;
            stream << "RESERVABILITY__<" << static_cast<unsigned int>(reservability) << ">";
            return stream.str();
         }
      }

      BookingInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__BOOKING, "")
         , m_reservability(RESERVABILITY__UNKNOWN)
      {}

      virtual ~BookingInfo() {}

      BookingInfo(
         const ::std::string& typeName,
         const ::std::vector< ::std::string>& placeIds,
         const ::std::string& operatorId,
         const ::std::string& serviceProvider,
         Reservability reservability)

         : LocationAttributeInfo(ATTRIBUTE_TYPE__BOOKING, typeName)
         , m_placeIds(placeIds)
         , m_operatorId(operatorId)
         , m_serviceProvider(serviceProvider)
         , m_reservability(reservability)
      {}

      virtual LocationAttributeInfo* clone() const
      {
         return new BookingInfo(*this);
      }

      bool operator==(const BookingInfo& rhs) const
      {
         return LocationAttributeInfo::operator== (rhs)
                && (m_placeIds == rhs.m_placeIds)
                && (m_operatorId == rhs.m_operatorId)
                && (m_serviceProvider == rhs.m_serviceProvider)
                && (m_reservability == rhs.m_reservability);
      }
      bool operator!=(const BookingInfo& rhs) const
      {
         return !(*this == rhs);
      }
      virtual ::std::string toString() const
      {
         ::std::stringstream stream("BookingInfo payload: ");
         stream << LocationAttributeInfo::toString() << ", "
                << "Total Place Ids = " << m_placeIds.size() << ::std::endl;
         for(unsigned int listIndex = 0; listIndex < m_placeIds.size(); ++listIndex)
         {
            stream << "  Place Id[" << listIndex << "] = " << m_placeIds[listIndex] << ::std::endl;
         }

         stream << "OperatorId = "<< m_operatorId << ", "
                << "ServiceProvider = "<< m_serviceProvider << ", "
                << "Reservability = "<< toString(m_reservability) << ::std::endl;
         return stream.str();
      }

      // Unique identifier for each individual spot (eg charging spot in a charging stations), which will be used by external app for booking.
      ::std::vector< ::std::string > m_placeIds;
      ::std::string m_operatorId;
      ::std::string m_serviceProvider;
      // Provides reservability status from the service provider.(eg reservation status for a charging station, placeId should be used to initiate reservation request)
      Reservability m_reservability;
   };

   /**
    * Info-Class providing the Asscociated Service Type info.
    */
   class AssociatedServicesInfo : public LocationAttributeInfo
   {
   public:
      enum AssociatedService
      {
         ASSOCIATEDSERVICE__UNKNOWN,
         ASSOCIATEDSERVICE__HOTEL_MOTEL,
         ASSOCIATEDSERVICE__PARKING_FACILITY,
         ASSOCIATEDSERVICE__PETROL_STATION,
         ASSOCIATEDSERVICE__RESTAURANT,
         ASSOCIATEDSERVICE__WC,
         ASSOCIATEDSERVICE__KIOSK,
         ASSOCIATEDSERVICE__WIFI,
         ASSOCIATEDSERVICE__MEN_AND_WOMEN_SHOWERS,
         ASSOCIATEDSERVICE__LAUNDRY_FACILITIES,
         ASSOCIATEDSERVICE__REPAIR_FACILITY,
         ASSOCIATEDSERVICE__PICNIC_AREA,
         ASSOCIATEDSERVICE__REFRESHMENTS,
         ASSOCIATEDSERVICE__DRINKING_WATER,
         ASSOCIATEDSERVICE__FIRE_PLACE,
         ASSOCIATEDSERVICE__PUBLIC_PHONE,
         ASSOCIATEDSERVICE__EMERGENCY_PHONE,
         ASSOCIATEDSERVICE__CAMPING_AREA,
         ASSOCIATEDSERVICE__CARAVAN_PARK,
         ASSOCIATEDSERVICE__DUMP_STATION,
         ASSOCIATEDSERVICE__FIRST_AID,
         ASSOCIATEDSERVICE__WHEELCHAIR_WC,
         ASSOCIATEDSERVICE__WHEELCHAIR_ACCESS,
         ASSOCIATEDSERVICE__MINI_MARKET,
         ASSOCIATEDSERVICE__CARWASH,
         ASSOCIATEDSERVICE__PARK_AND_DRIVE,
         ASSOCIATEDSERVICE__PARK_AND_RIDE,
         ASSOCIATEDSERVICE__AIRPORT,
         ASSOCIATEDSERVICE__RAILWAY_STATION,
         ASSOCIATEDSERVICE__TOURIST_CENTER,
         ASSOCIATEDSERVICE__OVERNIGHT_ACCOMODATION,
         ASSOCIATEDSERVICE__VEHICLE_MAINTENANCE_FACILITY,
         ASSOCIATEDSERVICE__SHOP
      };

      static ::std::string toString(AssociatedService associatedService)
      {
         switch (associatedService)
         {
         case ASSOCIATEDSERVICE__UNKNOWN:
            return "ASSOCIATEDSERVICE__UNKNOWN";
         case ASSOCIATEDSERVICE__HOTEL_MOTEL:
            return "ASSOCIATEDSERVICE__HOTEL_MOTEL";
         case ASSOCIATEDSERVICE__PARKING_FACILITY:
            return "ASSOCIATEDSERVICE__PARKING_FACILITY";
         case ASSOCIATEDSERVICE__PETROL_STATION:
            return "ASSOCIATEDSERVICE__PETROL_STATION";
         case ASSOCIATEDSERVICE__RESTAURANT:
            return "ASSOCIATEDSERVICE__RESTAURANT";
         case ASSOCIATEDSERVICE__WC:
            return "ASSOCIATEDSERVICE__WC";
         case ASSOCIATEDSERVICE__KIOSK:
            return "ASSOCIATEDSERVICE__KIOSK";
         case ASSOCIATEDSERVICE__WIFI:
            return "ASSOCIATEDSERVICE__WIFI";
         case ASSOCIATEDSERVICE__MEN_AND_WOMEN_SHOWERS:
            return "ASSOCIATEDSERVICE__MEN_AND_WOMEN_SHOWERS";
         case ASSOCIATEDSERVICE__LAUNDRY_FACILITIES:
            return "ASSOCIATEDSERVICE__LAUNDRY_FACILITIES";
         case ASSOCIATEDSERVICE__REPAIR_FACILITY:
            return "ASSOCIATEDSERVICE__REPAIR_FACILITY";
         case ASSOCIATEDSERVICE__PICNIC_AREA:
            return "ASSOCIATEDSERVICE__PICNIC_AREA";
         case ASSOCIATEDSERVICE__REFRESHMENTS:
            return "ASSOCIATEDSERVICE__REFRESHMENTS";
         case ASSOCIATEDSERVICE__DRINKING_WATER:
            return "ASSOCIATEDSERVICE__DRINKING_WATER";
         case ASSOCIATEDSERVICE__FIRE_PLACE:
            return "ASSOCIATEDSERVICE__FIRE_PLACE";
         case ASSOCIATEDSERVICE__PUBLIC_PHONE:
            return "ASSOCIATEDSERVICE__PUBLIC_PHONE";
         case ASSOCIATEDSERVICE__EMERGENCY_PHONE:
            return "ASSOCIATEDSERVICE__EMERGENCY_PHONE";
         case ASSOCIATEDSERVICE__CAMPING_AREA:
            return "ASSOCIATEDSERVICE__CAMPING_AREA";
         case ASSOCIATEDSERVICE__CARAVAN_PARK:
            return "ASSOCIATEDSERVICE__CARAVAN_PARK";
         case ASSOCIATEDSERVICE__DUMP_STATION:
            return "ASSOCIATEDSERVICE__DUMP_STATION";
         case ASSOCIATEDSERVICE__FIRST_AID:
            return "ASSOCIATEDSERVICE__FIRST_AID";
         case ASSOCIATEDSERVICE__WHEELCHAIR_WC:
            return "ASSOCIATEDSERVICE__WHEELCHAIR_WC";
         case ASSOCIATEDSERVICE__WHEELCHAIR_ACCESS:
            return "ASSOCIATEDSERVICE__WHEELCHAIR_ACCESS";
         case ASSOCIATEDSERVICE__MINI_MARKET:
            return "ASSOCIATEDSERVICE__MINI_MARKET";
         case ASSOCIATEDSERVICE__CARWASH:
            return "ASSOCIATEDSERVICE__CARWASH";
         case ASSOCIATEDSERVICE__PARK_AND_DRIVE:
            return "ASSOCIATEDSERVICE__PARK_AND_DRIVE";
         case ASSOCIATEDSERVICE__PARK_AND_RIDE:
            return "ASSOCIATEDSERVICE__PARK_AND_RIDE";
         case ASSOCIATEDSERVICE__AIRPORT:
            return "ASSOCIATEDSERVICE__AIRPORT";
         case ASSOCIATEDSERVICE__RAILWAY_STATION:
            return "ASSOCIATEDSERVICE__RAILWAY_STATION";
         case ASSOCIATEDSERVICE__TOURIST_CENTER:
            return "ASSOCIATEDSERVICE__TOURIST_CENTER";
         case ASSOCIATEDSERVICE__OVERNIGHT_ACCOMODATION:
            return "ASSOCIATEDSERVICE__OVERNIGHT_ACCOMODATION";
         case ASSOCIATEDSERVICE__VEHICLE_MAINTENANCE_FACILITY:
            return "ASSOCIATEDSERVICE__VEHICLE_MAINTENANCE_FACILITY";
         case ASSOCIATEDSERVICE__SHOP:
            return "ASSOCIATEDSERVICE__SHOP";
         default:
            ::std::stringstream stream;
            stream << "ASSOCIATEDSERVICE__<" << static_cast<unsigned int>(associatedService) << ">";
            return stream.str();
         }
      }

      AssociatedServicesInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__ASSOCIATED_SERVICES, "")
      {}

      virtual ~AssociatedServicesInfo() {}

      AssociatedServicesInfo(
         const ::std::string& typeName,
         const ::std::vector< AssociatedService >& associatedServices)

         : LocationAttributeInfo(ATTRIBUTE_TYPE__ASSOCIATED_SERVICES, typeName)
         , m_associatedServices(associatedServices)
      {}

      virtual LocationAttributeInfo* clone() const
      {
         return new AssociatedServicesInfo(*this);
      }

      bool operator==(const AssociatedServicesInfo& rhs) const
      {
         return LocationAttributeInfo::operator== (rhs)
                && (m_associatedServices == rhs.m_associatedServices);
      }

      bool operator!=(const AssociatedServicesInfo& rhs) const
      {
         return !(*this == rhs);
      }

      virtual ::std::string toString () const
      {
         ::std::stringstream stream("AssociatedServicesInfo payload: ");
         stream << LocationAttributeInfo::toString() << ", "
                << "Total Associated Service Types = " << m_associatedServices.size() << ::std::endl;
         for(unsigned int listIndex =0; listIndex < m_associatedServices.size(); ++listIndex)
         {
            stream << "  AssociatedService[" << listIndex << "] = " << toString(m_associatedServices[listIndex]) << ::std::endl;
         }
         return stream.str();
      }

      ::std::vector< AssociatedService > m_associatedServices;
   };

   /**
    * Info-Class providing the Special Restrictions info.
    */
   class SpecialRestrictionsInfo : public LocationAttributeInfo
   {
   public:
      enum SpecialRestriction
      {
         SPECIAL_RESTRICTION__UNSPECIFIED,
         SPECIAL_RESTRICTION__NO_RESTRICTION,
         SPECIAL_RESTRICTION__GENERIC_RESTRICTION,
         SPECIAL_RESTRICTION__RESIDENTS_ONLY,
         SPECIAL_RESTRICTION__EMPLOYEES_ONLY,
         SPECIAL_RESTRICTION__AUTHORIZED_PERSONNEL_ONLY,
         SPECIAL_RESTRICTION__MEMBERS_ONLY,
         SPECIAL_RESTRICTION__PROVIDER_CUSTOMERS,
         SPECIAL_RESTRICTION__PROVIDER_AND_ROAMING_CUSTOMERS,
         SPECIAL_RESTRICTION__SHOPPERS,
         SPECIAL_RESTRICTION__HOTEL_GUESTS,
         SPECIAL_RESTRICTION__STAFF,
         SPECIAL_RESTRICTION__VISITORS,
         SPECIAL_RESTRICTION__FREETEXT
      };

      static ::std::string toString(SpecialRestriction specialRestriction)
      {
         switch (specialRestriction)
         {
         case SPECIAL_RESTRICTION__UNSPECIFIED:
            return "SPECIAL_RESTRICTION__UNSPECIFIED";
         case SPECIAL_RESTRICTION__NO_RESTRICTION:
            return "SPECIAL_RESTRICTION__NO_RESTRICTION";
         case SPECIAL_RESTRICTION__GENERIC_RESTRICTION:
            return "SPECIAL_RESTRICTION__GENERIC_RESTRICTION";
         case SPECIAL_RESTRICTION__RESIDENTS_ONLY:
            return "SPECIAL_RESTRICTION__RESIDENTS_ONLY";
         case SPECIAL_RESTRICTION__EMPLOYEES_ONLY:
            return "SPECIAL_RESTRICTION__EMPLOYEES_ONLY";
         case SPECIAL_RESTRICTION__AUTHORIZED_PERSONNEL_ONLY:
            return "SPECIAL_RESTRICTION__AUTHORIZED_PERSONNEL_ONLY";
         case SPECIAL_RESTRICTION__MEMBERS_ONLY:
            return "SPECIAL_RESTRICTION__MEMBERS_ONLY";
         case SPECIAL_RESTRICTION__PROVIDER_CUSTOMERS:
            return "SPECIAL_RESTRICTION__PROVIDER_CUSTOMERS";
         case SPECIAL_RESTRICTION__PROVIDER_AND_ROAMING_CUSTOMERS:
            return "SPECIAL_RESTRICTION__PROVIDER_AND_ROAMING_CUSTOMERS";
         case SPECIAL_RESTRICTION__SHOPPERS:
            return "SPECIAL_RESTRICTION__SHOPPERS";
         case SPECIAL_RESTRICTION__HOTEL_GUESTS:
            return "SPECIAL_RESTRICTION__HOTEL_GUESTS";
         case SPECIAL_RESTRICTION__STAFF:
            return "SPECIAL_RESTRICTION__STAFF";
         case SPECIAL_RESTRICTION__VISITORS:
            return "SPECIAL_RESTRICTION__VISITORS";
         case SPECIAL_RESTRICTION__FREETEXT:
            return "SPECIAL_RESTRICTION__FREETEXT";
         default:
            ::std::stringstream stream;
            stream << "SPECIAL_RESTRICTION__<" << static_cast<unsigned int>(specialRestriction) << ">";
            return stream.str();
         }
      }

      SpecialRestrictionsInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__SPECIAL_RESTRICTIONS, "")
      {}

      virtual ~SpecialRestrictionsInfo() {}

      SpecialRestrictionsInfo(
         const ::std::string& typeName,
         const ::std::vector< SpecialRestriction >& specialRestrictions,
         const ::std::string& freetextRestriction)

         : LocationAttributeInfo(ATTRIBUTE_TYPE__SPECIAL_RESTRICTIONS, typeName)
         , m_specialRestrictions(specialRestrictions)
         , m_freetextRestriction(freetextRestriction)
      {}

      virtual LocationAttributeInfo* clone() const
      {
         return new SpecialRestrictionsInfo(*this);
      }

      bool operator==(const SpecialRestrictionsInfo& rhs) const
      {
         return LocationAttributeInfo::operator== (rhs)
                && (m_specialRestrictions == rhs.m_specialRestrictions)
                && (m_freetextRestriction == rhs.m_freetextRestriction);
      }

      bool operator!=(const SpecialRestrictionsInfo& rhs) const
      {
         return !(*this == rhs);
      }

      virtual ::std::string toString () const
      {
         ::std::stringstream stream("SpecialRestrictionsInfo payload: ");
         stream << LocationAttributeInfo::toString() << ", "
                << "Total Special Restrictions Info = " << m_specialRestrictions.size() << ::std::endl;
         for(unsigned int listIndex =0; listIndex < m_specialRestrictions.size(); ++listIndex)
         {
            stream << "  SpecialRestriction[" << listIndex << "] = " << toString(m_specialRestrictions[listIndex]);
            if ( SPECIAL_RESTRICTION__FREETEXT == m_specialRestrictions[listIndex] )
            {
                stream << ", freetext = " << m_freetextRestriction;
            }
            stream << ::std::endl;
         }
         return stream.str();
      }

      ::std::vector< SpecialRestriction > m_specialRestrictions;
      ::std::string m_freetextRestriction;
   };

   /**
    * Info-Class providing the Special Restrictions info.
    */
   class OpeningHoursInfo : public LocationAttributeInfo
   {
   public:
      OpeningHoursInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__OPENING_HOURS_DETAIL, "")
         , m_OpeningHours()
         , m_ClosingDays()
         , m_Restrictions()
         , m_OpeningHoursException()
         , m_ClosingDaysException()
      {}

      virtual ~OpeningHoursInfo() {}

      OpeningHoursInfo(
         const ::std::string& typeName,
         const ::std::string& openingHours,
         const ::std::string& closingDays,
         const ::std::string& restrictions,
         const ::std::string& openingHoursException,
         const ::std::string& closingDaysException
      )
         : LocationAttributeInfo(ATTRIBUTE_TYPE__OPENING_HOURS_DETAIL, typeName)
         , m_OpeningHours(openingHours)
         , m_ClosingDays(closingDays)
         , m_Restrictions(restrictions)
         , m_OpeningHoursException(openingHoursException)
         , m_ClosingDaysException(closingDaysException)
      {}

      virtual LocationAttributeInfo* clone() const
      {
         return new OpeningHoursInfo(*this);
      }

      bool operator==(const OpeningHoursInfo& rhs) const
      {
         return LocationAttributeInfo::operator == (rhs)
                && (m_OpeningHours               == rhs.m_OpeningHours)
                && (m_ClosingDays                == rhs.m_ClosingDays)
                && (m_Restrictions               == rhs.m_Restrictions)
                && (m_OpeningHoursException      == rhs.m_OpeningHoursException)
                && (m_ClosingDaysException       == rhs.m_ClosingDaysException);
      }

      bool operator!=(const OpeningHoursInfo& rhs) const
      {
         return !(*this == rhs);
      }

      virtual ::std::string toString() const
      {
         ::std::stringstream stream("OpeningHoursInfo payload: ");
         stream << LocationAttributeInfo::toString()
                << ", opening hours = " << m_OpeningHours
                << ", closing days  = " << m_ClosingDays
                << ", restrictions  = " << m_Restrictions
                << ", opening hours exception = " << m_OpeningHoursException
                << ", closing days exception = " << m_ClosingDaysException << ::std::endl;
         return stream.str();
      }

      ::std::string m_OpeningHours;             // e.g for JPN:"1st July - 31th Aug 9:00 - 17:30" e.g for ROW:"16:00 CET - 22:00 CET"
      //[JP]-->
      ::std::string m_ClosingDays;              // e.g. "Every month the 7th"
      ::std::string m_Restrictions;             // e.g. "Cars cannot enter or exit parking lot from 23:00-9:00"
      ::std::string m_OpeningHoursException;    // e.g. "Except for the Obon period"
      ::std::string m_ClosingDaysException;     // e.g. "Except for public holidays"
      //<--
   };

   class DirectionDescriptionInfo : public LocationAttributeInfo
   {
   public:
      DirectionDescriptionInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__DIRECTION_DESCRIPTION, "")
         , m_directionDescription (DIRECTIONDESCRIPTION_NORTH)
      {}
      virtual ~DirectionDescriptionInfo() {}

      DirectionDescriptionInfo(
         const ::std::string& name,
         DirectionDescription directionDescription)

      : LocationAttributeInfo(ATTRIBUTE_TYPE__DIRECTION_DESCRIPTION, name)
      , m_directionDescription (directionDescription)
      {}

      virtual LocationAttributeInfo* clone() const
      {
         return new DirectionDescriptionInfo(*this);
      }

      bool operator==(const DirectionDescriptionInfo& rhs) const
      {
         return LocationAttributeInfo::operator== (rhs)
            && (m_directionDescription  == rhs.m_directionDescription);
      }
      bool operator!=(const DirectionDescriptionInfo& rhs) const
      {
         return !(*this == rhs);
      }
      virtual ::std::string toString() const
      {
         ::std::stringstream stream("DirectionInfo payload: ");
         stream << LocationAttributeInfo::toString() << ", "
            << "m_directionDescription  = "<< ::navmiddleware::toString(m_directionDescription) << ::std::endl;
         return stream.str();
      }

      DirectionDescription m_directionDescription;
   };

   /**
    * Info-Class with information to show pricing for poi categories, wherever applicable
    */
   class PriceInfo : public LocationAttributeInfo
   {
   public:
      enum FeeType
      {
         FeeType__UNKNOWN,
         FeeType__MINIMUM,
         FeeType__MAXIMUM,
         FeeType__ADDITIONAL,
         FeeType__SEASON_TICKET,
         FeeType__TEMPORARY_PRICE,
         FeeType__NIGHT_PRICE,
         FeeType__DAY_PRICE,
         FeeType__MONTH_PRICE,
         FeeType__YEAR_PRICE,
         FeeType__FIRST_HOUR_PRICE,
         FeeType__FREE_PARKING,
         FeeType__FLAT,
         FeeType__UNDEFINED,
      };
      static ::std::string toString(FeeType feeType)
      {
         switch (feeType)
         {
         case FeeType__UNKNOWN:
            return "FeeType__UNKNOWN";
         case FeeType__MINIMUM:
            return "FeeType__MINIMUM";
         case FeeType__MAXIMUM:
            return "FeeType__MAXIMUM";
         case FeeType__ADDITIONAL:
            return "FeeType__ADDITIONAL";
         case FeeType__SEASON_TICKET:
            return "FeeType__SEASON_TICKET";
         case FeeType__TEMPORARY_PRICE:
            return "FeeType__TEMPORARY_PRICE";
         case FeeType__NIGHT_PRICE:
            return "FeeType__NIGHT_PRICE";
         case FeeType__DAY_PRICE:
            return "FeeType__DAY_PRICE";
         case FeeType__MONTH_PRICE:
            return "FeeType__MONTH_PRICE";
         case FeeType__YEAR_PRICE:
            return "FeeType__YEAR_PRICE";
         case FeeType__FIRST_HOUR_PRICE:
            return "FeeType__FIRST_HOUR_PRICE";
         case FeeType__FREE_PARKING:
            return "FeeType__FREE_PARKING";
         case FeeType__FLAT:
            return "FeeType__FLAT";
         case FeeType__UNDEFINED:
            return "FeeType__UNDEFINED";
         default:
            ::std::stringstream stream;
            stream << "FEETYPE__<" << static_cast<unsigned int>(feeType) << ">";
            return stream.str();
         }
      }
      PriceInfo()
         : LocationAttributeInfo(ATTRIBUTE_TYPE__PRICE_INFO, "")
         , m_feeType(FeeType__UNKNOWN)
      {}
      virtual ~PriceInfo() {}
      PriceInfo(const ::std::string& typeName,
                FeeType feeType,
                const ::std::string& timeDuration,
                const ::std::string& price,
                const ::std::string& timeStamp)
         : LocationAttributeInfo(ATTRIBUTE_TYPE__PRICE_INFO, typeName)
         , m_feeType(feeType)
         , m_TimePeriodDuration(timeDuration)
         , m_price(price)
         , m_dataAcquisitionTime(timeStamp)
      {}
      virtual LocationAttributeInfo* clone() const
      {
         return new PriceInfo(*this);
      }
      bool operator==(const PriceInfo& rhs) const
      {
         return LocationAttributeInfo::operator== (rhs)
                && (m_feeType == rhs.m_feeType)
                && (m_TimePeriodDuration == rhs.m_TimePeriodDuration)
                && (m_price == rhs.m_price)
                && (m_dataAcquisitionTime == rhs.m_dataAcquisitionTime);
      }
      bool operator!=(const PriceInfo& rhs) const
      {
         return !(*this == rhs);
      }
      virtual ::std::string toString() const
      {
         ::std::stringstream stream("PriceInfo payload: ");
         stream << LocationAttributeInfo::toString() << ", "
                << "Fee Type = "<<toString(m_feeType) << ", "
                << "Time Period Duration = "<< m_TimePeriodDuration << ", "
                << "Price = "<< m_price<< ", "
                << "Data Acquisition Time = "<< m_dataAcquisitionTime<< ::std::endl;
         return stream.str();
      }
      FeeType       m_feeType;
      ::std::string m_TimePeriodDuration;
      ::std::string m_price;
      ::std::string m_dataAcquisitionTime;
   };

   // End of Info-Classes continue with ...

   // Other class member enumerations
   enum LocationType
   {
      LOCATIONTYPE_COORDINATE,
      LOCATIONTYPE_ADDRESS,
      LOCATIONTYPE_POI
   };

   inline ::std::string toString(LocationType locationType) const
   {
      switch (locationType)
      {
      case LOCATIONTYPE_COORDINATE:
         return "LOCATIONTYPE_COORDINATE";
      case LOCATIONTYPE_ADDRESS:
         return "LOCATIONTYPE_ADDRESS";
      case LOCATIONTYPE_POI:
         return "LOCATIONTYPE_POI";
      default:
         ::std::stringstream stream;
         stream << "LOCATIONTYPE_<" << static_cast<unsigned int>(locationType) << ">";
         return stream.str();
      }
   }

   enum MotorwaySelectionStatus
   {
      MOTORWAY_SELECTION_STATUS__UNKNOWN,
      MOTORWAY_SELECTION_STATUS__NO_MOTORWAY,
      MOTORWAY_SELECTION_STATUS__AVOIDABLE_MOTORWAY,
      MOTORWAY_SELECTION_STATUS__NONE_AVOIDABLE_MOTORWAY
   };

   inline ::std::string toString(MotorwaySelectionStatus motorwaySelectionStatus) const
   {
      switch (motorwaySelectionStatus)
      {
      case MOTORWAY_SELECTION_STATUS__UNKNOWN:
         return "MOTORWAY_SELECTION_STATUS__UNKNOWN";
      case MOTORWAY_SELECTION_STATUS__NO_MOTORWAY:
         return "MOTORWAY_SELECTION_STATUS__NO_MOTORWAY";
      case MOTORWAY_SELECTION_STATUS__AVOIDABLE_MOTORWAY:
         return "MOTORWAY_SELECTION_STATUS__AVOIDABLE_MOTORWAY";
      case MOTORWAY_SELECTION_STATUS__NONE_AVOIDABLE_MOTORWAY:
         return "MOTORWAY_SELECTION_STATUS__NONE_AVOIDABLE_MOTORWAY";
      default:
         ::std::stringstream stream;
         stream << "MOTORWAY_SELECTION_STATUS_<" << static_cast<unsigned int>(motorwaySelectionStatus) << ">";
         return stream.str();
      }
   }

   // implementation of LocationAttributeInfos class
   LocationAttributeInfos()
      : m_locationType(LOCATIONTYPE_COORDINATE)
      , m_origin(LOCATION_ORIGIN__UNKNOWN)
      , m_motorwaySelectionStatus(MOTORWAY_SELECTION_STATUS__UNKNOWN)
      , m_requestId(0)
   {
   }

   LocationAttributeInfos(const LocationAttributeInfos& rhs)
      : m_locationType(rhs.m_locationType)
      , m_origin(rhs.m_origin)
      , m_motorwaySelectionStatus(rhs.m_motorwaySelectionStatus)
      , m_coordinate(rhs.m_coordinate)
      , m_requestId(rhs.m_requestId)
   {
      m_attributes.reserve(rhs.m_attributes.size());
      for(::std::vector<const LocationAttributeInfo*>::const_iterator iter = rhs.m_attributes.begin();
            iter != rhs.m_attributes.end();
            ++iter)
      {
         if ( (*iter) )
         {
            m_attributes.push_back((*iter)->clone());
         }
      }
   }
   ~LocationAttributeInfos()
   {
      clearAttributes();
   }

   void clear()
   {
      clearAttributes();
      m_locationType = LOCATIONTYPE_COORDINATE;
      m_origin = LOCATION_ORIGIN__UNKNOWN;
      m_motorwaySelectionStatus = MOTORWAY_SELECTION_STATUS__UNKNOWN;
      m_coordinate.invalidate();
      m_requestId = 0;
   }

   LocationAttributeInfos& operator=(LocationAttributeInfos rhs)
   {
      m_attributes.swap(rhs.m_attributes);
      m_locationType = rhs.m_locationType;
      m_origin = rhs.m_origin;
      m_motorwaySelectionStatus = rhs.m_motorwaySelectionStatus;
      m_coordinate = rhs.m_coordinate;
      m_requestId = rhs.m_requestId;
      return *this;
   }
   bool operator==(const LocationAttributeInfos& rhs) const
   {
      bool status = (m_locationType == rhs.m_locationType) &&
                    (m_origin == rhs.m_origin) &&
                    (m_motorwaySelectionStatus == rhs.m_motorwaySelectionStatus) &&
                    (m_coordinate == rhs.m_coordinate) &&
                    (m_attributes.size() == rhs.m_attributes.size() &&
                    (m_requestId == rhs.m_requestId));
      if(status)
      {
         for(unsigned int i = 0; i < m_attributes.size(); ++i)
         {
            if( !(!m_attributes[i] && !rhs.m_attributes[i])
                  && ((m_attributes[i] && !rhs.m_attributes[i])
                      || (!m_attributes[i] && rhs.m_attributes[i])
                      || (*m_attributes[i] != *rhs.m_attributes[i])))
            {
               status = false;
               break;
            }
         }
      }
      return status;
   }
   bool operator!=(const LocationAttributeInfos& rhs) const
   {
      return !(*this == rhs);
   }
   ::std::string toString() const
   {
      ::std::stringstream stream("LocationAttributeInfos payload:\n");
      stream << "Location Type = " << toString(getLocationType()) << ::std::endl
             << "Origin = " << ::navmiddleware::toString(getOrigin()) << ::std::endl
             << "Motorway Selection Status = " << toString(getMotorwaySelectionStatus()) << ::std::endl
             << "m_coordinate = " << (m_coordinate.isValid()? m_coordinate.getValue().toString() : "invalid") << ::std::endl
             << "m_requestId = " << m_requestId << ::std::endl;

      stream << "Total Location Attributes Available =  " << m_attributes.size() << ::std::endl;
      for(unsigned int i = 0; i < m_attributes.size(); ++i)
      {
         const LocationAttributeInfo* attribute = m_attributes[i];
         stream << "["<< i << "]: ";
         if (attribute)
         {
            stream << attribute->toString();
         }
         else
         {
            stream << "NULL";
         }
      }
      stream << ::std::endl;
      return stream.str();
   }

   /**
    * Adds a single LocationAttributeInfo at the end of the attributes list.
    * The ownership of the passed pointer is taken over by this structure.
    */
   void addAttributeInfo(const LocationAttributeInfo* attribute)
   {
      if (attribute)
      {
         m_attributes.push_back(attribute);
      }
   }

   bool hasAttributes() const
   {
      return !m_attributes.empty();
   }

   /**
    * Get the overall list of attributes (regardless of their actual type).
    * The order of the attributes in the vector is random.
    * Each attribute of this vector is a concrete sub-class of LocationAttributeInfo. Before accessing the content
    * the caller has to check the attribute's type (m_type) and depending on this has to do a downcast to the
    * respective sub-class.
    */
   const ::std::vector<const LocationAttributeInfo*>& getAttributeInfos() const
   {
      return m_attributes;
   }

   /** Returns the name of the location, i.e. either POI name or address */
   const StringAttributeInfo* getLocationNameInfo() const
   {
      return static_cast<const StringAttributeInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__LOCATION_NAME));
   }
   /** Returns the phoneme of the location, i.e. phoneme for POI name no phoneme will be provided for FullAddress */
   const StringAttributeInfo* getLocationNamePhonemeInfo() const
   {
      return static_cast<const StringAttributeInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__LOCATION_NAME_PHONEME));
   }
   const PoiCategoryInfo* getPoiCategoryInfo() const
   {
      return static_cast<const PoiCategoryInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__POI_CATEGORY));
   }
   const OccupancyInfo* getOccupancyInfo() const
   {
      return static_cast<const OccupancyInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__OCCUPANCY));
   }
   const ParkingInfo* getParkingInfo() const
   {
      return static_cast<const ParkingInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__PARKING));
   }
   const FuelPriceInfo* getPreferredFuelPriceInfo() const
   {
      return static_cast<const FuelPriceInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__FUEL_PRICE));
   }
   ::std::vector<const FuelPriceInfo*> getFuelPriceInfos() const
   {
      // cppcheck-suppress compareBoolExpressionWithInt ; false positive - seems misinterpreting template brackets as comp op
      return getAttributeVector<FuelPriceInfo>(LocationAttributeInfo::ATTRIBUTE_TYPE__FUEL_PRICE);
   }
   ::std::vector<const EvChargingTypeOccupancyInfo*> getEvChargingTypeOccupancyInfos() const
   {
      // cppcheck-suppress compareBoolExpressionWithInt ; false positive - seems misinterpreting template brackets as comp op
      return getAttributeVector<EvChargingTypeOccupancyInfo>(LocationAttributeInfo::ATTRIBUTE_TYPE__EV_CHARGING_TYPE_OCCUPANCY);
   }
   const EvChargingSpeedsInfo* getEvChargingSpeedsInfo() const
   {
      return static_cast<const EvChargingSpeedsInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__EV_CHARGING_SPEEDS));
   }
   const EvConnectorTypesInfo* getEvConnectorTypesInfo() const
   {
      return static_cast<const EvConnectorTypesInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__EV_CONNECTOR_TYPES));
   }
   const StringAttributeInfo* getEnergyProviderInfo() const
   {
      return static_cast<const StringAttributeInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__ENERGY_PROVIDER));
   }
   const AddressDetailInfo* getAddressDetailInfos() const
   {
      return static_cast<const AddressDetailInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__ADDRESS_DETAILS));
   }
   const OpeningStatusInfo* getOpeningStatusInfo() const
   {
      return static_cast<const OpeningStatusInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__OPENING_STATUS));
   }
   const StringAttributeInfo* getOpeningHoursInfo() const  // depricated please refer below
   {
      return static_cast<const StringAttributeInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__OPENING_HOURS));
   }
   const OpeningHoursInfo* getOpeningHoursDetailInfo() const
   {
      return static_cast<const OpeningHoursInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__OPENING_HOURS_DETAIL));
   }
   const PaymentMethodsInfo* getPaymentMethodsInfo() const
   {
      return static_cast<const PaymentMethodsInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__PAYMENT_METHODS));
   }
   const StringAttributeInfo* getPhoneNumberInfo() const
   {
      return static_cast<const StringAttributeInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__PHONE_NUMBER));
   }
   const StringAttributeInfo* getLastUpdateTimeInfo() const
   {
      return static_cast<const StringAttributeInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__LAST_UPDATE_TIME));
   }
   const LastUpdateStatusInfo* getLastUpdateStatusInfo() const
   {
      return static_cast<const LastUpdateStatusInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__LAST_UPDATE_STATUS));
   }
   const DestinationMemoryAliasInfo* getDestinationMemoryAliasInfo() const
   {
      return static_cast<const DestinationMemoryAliasInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__DESTINATION_MEMORY_ALIAS));
   }
   const BoundingBoxInfo* getBoundingBoxInfo() const
   {
      return static_cast<const BoundingBoxInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__BOUNDING_BOX));
   }
   const StringAttributeInfo* getBrandNameInfo() const
   {
      return static_cast<const StringAttributeInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__BRAND_NAME));
   }
   const StringAttributeInfo* getAirDistanceInfo() const
   {
      return static_cast<const StringAttributeInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__AIR_DISTANCE));
   }
   const StringAttributeInfo* getContentProviderInfo() const
   {
      return static_cast<const StringAttributeInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__CONTENT_PROVIDER));
   }
   const StringAttributeInfo* getWebsiteInfo() const
   {
      return static_cast<const StringAttributeInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__WEBSITE));
   }
   const PictureReferenceInfo* getPictureReferenceInfo() const
   {
      return static_cast<const PictureReferenceInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__PICTURE_REFERENCE));
   }
   ::std::vector<const PictureReferenceInfo*> getPictureReferenceInfos() const
   {
      // cppcheck-suppress compareBoolExpressionWithInt ; false positive - seems misinterpreting template brackets as comp op
      return getAttributeVector<PictureReferenceInfo>(LocationAttributeInfo::ATTRIBUTE_TYPE__PICTURE_REFERENCE);
   }
   const ReviewInfos* getReviewInfos() const
   {
      return static_cast<const ReviewInfos*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__REVIEW));
   }
   const UserRankingInfo* getUserRankingInfo() const
   {
      return static_cast<const UserRankingInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__USER_RANKING));
   }
   const SafetyCameraTypeInfo* getSafetyCameraTypeInfo() const
   {
      return static_cast<const SafetyCameraTypeInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__SAFETYCAMERA_TYPE));
   }
   const SafetyCameraSpeedLimitInfo* getSafetyCameraSpeedLimitInfo() const
   {
      return static_cast<const SafetyCameraSpeedLimitInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__SAFETYCAMERA_SPEEDLIMIT));
   }
   const HouseNumberRangeInfo* getHouseNumberRangeInfo() const
   {
      return static_cast<const HouseNumberRangeInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__HOUSE_NUMBER_RANGE));
   }
   const BookingInfo* getBookingInfo() const
   {
      return static_cast<const BookingInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__BOOKING));
   }
   const StringAttributeInfo* getUniqueIdInfo() const
   {
      return static_cast<const StringAttributeInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__UNIQUE_ID));
   }
   const AssociatedServicesInfo* getAssociatedServicesInfo() const
   {
      return static_cast<const AssociatedServicesInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__ASSOCIATED_SERVICES));
   }
   const SpecialRestrictionsInfo* getSpecialRestrictionsInfo() const
   {
      return static_cast<const SpecialRestrictionsInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__SPECIAL_RESTRICTIONS));
   }
   const StringAttributeInfo* getSpecialOfferInfo() const
   {
      return static_cast<const StringAttributeInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__SPECIAL_OFFER));
   }
   const DirectionDescriptionInfo* getDirectionDescriptionInfo() const
   {
      return static_cast<const DirectionDescriptionInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__DIRECTION_DESCRIPTION));
   }
   ::std::vector<const PriceInfo*> getPricingInfo() const
   {
      // cppcheck-suppress compareBoolExpressionWithInt ; false positive - seems misinterpreting template brackets as comp op
      return getAttributeVector<PriceInfo>(LocationAttributeInfo::ATTRIBUTE_TYPE__PRICE_INFO);
   }
   //<-- INF4CV
   const StringAttributeInfo* getDescriptionInfo() const
   {
      return static_cast<const StringAttributeInfo*>(getAttributePtr(LocationAttributeInfo::ATTRIBUTE_TYPE__DESCRIPTION));
   }
   //->

   LocationType getLocationType() const
   {
      return m_locationType;
   }
   void setLocationType(LocationType locationType)
   {
      m_locationType = locationType;
   }

   LocationOrigin getOrigin() const
   {
      return m_origin;
   }
   void setOrigin(LocationOrigin origin)
   {
      m_origin = origin;
   }

   MotorwaySelectionStatus getMotorwaySelectionStatus() const
   {
      return m_motorwaySelectionStatus;
   }
   void setMotorwaySelectionStatus(MotorwaySelectionStatus motorwaySelectionStatus)
   {
      m_motorwaySelectionStatus = motorwaySelectionStatus;
   }

   //Returns the Current Coordinates for a valid destination
   const ValidValue<GeoCoordinateDegree>& getCoordinate() const
   {
      return m_coordinate;
   }
   void setCoordinate(const GeoCoordinateDegree& coordinate)
   {
      m_coordinate.setValue(coordinate);
   }

   RequestId getRequestId() const
   {
      return m_requestId;
   }
   void setRequestId(RequestId requestId)
   {
      m_requestId = requestId;
   }

private:
   template <class T>
   ::std::vector<const T*> getAttributeVector(LocationAttributeInfo::AttributeType type) const
   {
      ::std::vector<const T*> attributes;
      for(::std::vector<const LocationAttributeInfo*>::const_iterator iter = m_attributes.begin();
            iter != m_attributes.end();
            ++iter)
      {
         if ( (*iter) && ((*iter)->m_type == type) )
         {
            attributes.push_back(static_cast<const T*>(*iter));
         }
      }
      return attributes;
   }

   const LocationAttributeInfo* getAttributePtr(LocationAttributeInfo::AttributeType type) const
   {
      for(::std::vector<const LocationAttributeInfo*>::const_iterator iter = m_attributes.begin();
            iter != m_attributes.end();
            ++iter)
      {
         if ( (*iter) && ((*iter)->m_type == type) )
         {
            return (*iter);
         }
      }
      return NULL;
   }

   void clearAttributes()
   {
      for(::std::vector<const LocationAttributeInfo*>::const_iterator iter = m_attributes.begin();
            iter != m_attributes.end();
            ++iter)
      {
         delete *iter;
      }
      m_attributes.clear();
   }

   ::std::vector<const LocationAttributeInfo*> m_attributes;
   LocationType m_locationType;
   LocationOrigin m_origin;
   MotorwaySelectionStatus m_motorwaySelectionStatus;
   ValidValue<GeoCoordinateDegree> m_coordinate;
   RequestId                       m_requestId;
};

class WeatherReport
{
public:
   enum WeatherIconType
   {
      WEATHER_ICON_TYPE__DAY,
      WEATHER_ICON_TYPE__NIGHT
   };

   WeatherReport()
      : m_cloudCover(CLOUD_COVERAGE__UNKNOWN),
        m_uVIndex(UVINDEX__UNKNOWN),
        m_weatherSituation(WEATHER_SITUATION__UNKNOWN),
        m_weatherSeverity(WEATHER_SEVERITY__UNKNOWN),
        m_weatherIconType(WEATHER_ICON_TYPE__DAY),
        m_weatherReportTimeHour(-1)
   {}

   ~WeatherReport() {}


   void setCurrentTemperatureInMilliKelvin(const ValidValue<int>& temperature)
   {
      m_currentTemperatureInMilliKelvin = temperature;
   }

   const ValidValue<int>& getCurrentTemperatureInMilliKelvin() const
   {
      return m_currentTemperatureInMilliKelvin;
   }

   void setCurrentTemperature(const ::std::string& temperature)
   {
      m_currentTemperature = temperature;
   }

   const ::std::string& getCurrentTemperature() const
   {
      return m_currentTemperature;
   }

   void setMinTemperatureInMilliKelvin(const ValidValue<int>& minTemperature)
   {
      m_minimumTemperatureInMilliKelvin = minTemperature;
   }

   const ValidValue<int>& getMinTemperatureInMilliKelvin() const
   {
      return m_minimumTemperatureInMilliKelvin;
   }

   void setMinTemperature(const ::std::string& minTemperature)
   {
      m_minTemperature = minTemperature;
   }

   const ::std::string& getMinTemperature() const
   {
      return m_minTemperature;
   }

   void setMaxTemperatureInMilliKelvin(const ValidValue<int>& maxTemperature)
   {
      m_maximumTemperatureInMilliKelvin = maxTemperature;
   }

   const ValidValue<int>& getMaxTemperatureInMilliKelvin() const
   {
      return m_maximumTemperatureInMilliKelvin;
   }

   void setMaxTemperature(const ::std::string& maxTemperature)
   {
      m_maxTemperature = maxTemperature;
   }

   const ::std::string& getMaxTemperature() const
   {
      return m_maxTemperature;
   }

   void setHumidityInPercentage(const ValidValue<int>& humidity)
   {
      m_humidityInPercentage = humidity;
   }

   const ValidValue<int>& getHumidityInPercentage() const
   {
      return m_humidityInPercentage;
   }

   void setRelativeHumidity(const ::std::string& relativeHumidity)
   {
      m_relativeHumidity = relativeHumidity;
   }

   const ::std::string& getRelativeHumidity() const
   {
      return m_relativeHumidity;
   }

   void setWindSpeedInCentimeterSeconds(const ValidValue<int>& windSpeed)
   {
      m_windSpeedInCentimeterSecond = windSpeed;
   }

   const ValidValue<int>& getWindSpeedInCentimeterSeconds() const
   {
      return m_windSpeedInCentimeterSecond;
   }

   void setWindSpeed(const ::std::string& windSpeed)
   {
      m_windSpeed = windSpeed;
   }

   const ::std::string& getWindSpeed() const
   {
      return m_windSpeed;
   }

   void setWindDirectionInDegree(const ValidValue<int>& windDirection)
   {
      m_windDirectionInDegree = windDirection;
   }

   const ValidValue<int>& getWindDirectionInDegree() const
   {
      return m_windDirectionInDegree;
   }

   void setPrecipitation(const ::std::string& precipitationProbability)
   {
      m_precipitationProbability = precipitationProbability;
   }

   const ::std::string& getPrecipitation() const
   {
      return m_precipitationProbability;
   }

   void setPrecipitationInPercentage(const ValidValue<int>& precipitation)
   {
      m_precipitationInPercentage = precipitation;
   }

   const ValidValue<int>& getPrecipitationInPercentage() const
   {
      return m_precipitationInPercentage;
   }

   void setCloudCoverage(CloudCoverage cloudCover)
   {
      m_cloudCover = cloudCover;
   }

   CloudCoverage getCloudCoverage() const
   {
      return m_cloudCover;
   }

   void setUvIndex(UVIndex uVIndex)
   {
      m_uVIndex = uVIndex;
   }

   UVIndex getUvIndex() const
   {
      return m_uVIndex;
   }

   void setWeatherSituation(WeatherSituation weatherSituation)
   {
      m_weatherSituation = weatherSituation;
   }

   WeatherSituation getWeatherSituation() const
   {
      return m_weatherSituation;
   }

   void setWeatherSituationIcon(const Image& weatherSituationIcon)
   {
      m_weatherSituationIcon = weatherSituationIcon;
   }

   const Image& getWeatherSituationIcon() const
   {
      return m_weatherSituationIcon;
   }

   void setWeatherSeverity(WeatherSeverity weatherSeverity)
   {
      m_weatherSeverity = weatherSeverity;
   }

   WeatherSeverity getWeatherSeverity() const
   {
      return m_weatherSeverity;
   }

   const Image& getWeatherAlertIcon() const
   {
      return m_weatherAlertIcon;
   }

   void setWeatherAlertIcon(const Image& weatherAlertIcon)
   {
      m_weatherAlertIcon = weatherAlertIcon;
   }

   void setCityName(const ::std::string& cityName)
   {
      m_cityName = cityName;
   }

   const ::std::string& getCityName() const
   {
      return m_cityName;
   }

   void setWeatherIconType(WeatherIconType iconType)
   {
      m_weatherIconType = iconType;
   }

   WeatherIconType getWeatherIconType() const
   {
      return m_weatherIconType;
   }

   void setWeatherReportTime(const ::std::string& weatherReportTime)
   {
      m_weatherReportTime = weatherReportTime;
   }

   void setWeatherReportTimeHour(int8_t weatherReportTimeHour)
   {
      m_weatherReportTimeHour = weatherReportTimeHour;
   }

   const ::std::string& getWeatherReportTime() const
   {
      return m_weatherReportTime;
   }

   int8_t getWeatherReportTimeHour() const
   {
      return m_weatherReportTimeHour;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("WeatherReport payload:\n");
      stream << ::std::endl
             << "Current Temperature in MilliKelvin: " << m_currentTemperatureInMilliKelvin.toString() << ::std::endl
             << "Current Temperature: " << m_currentTemperature << ::std::endl
             << "Minimum Temperature in MilliKelvin: " << m_minimumTemperatureInMilliKelvin.toString() << ::std::endl
             << "Minimum Temperature: " << m_minTemperature << ::std::endl
             << "Maximum Temperature in MilliKelvin: " << m_maximumTemperatureInMilliKelvin.toString() << ::std::endl
             << "Maximum Temperature: " << m_maxTemperature << ::std::endl
             << "Relative humidity in Percentage: " << m_humidityInPercentage.toString() << ::std::endl
             << "Relative humidity: " << m_relativeHumidity << ::std::endl
             << "Wind speed InCentimeterSecond: " << m_windSpeedInCentimeterSecond.toString() << ::std::endl
             << "Wind speed: " << m_windSpeed << ::std::endl
             << "Wind Direction in percentage: "<< m_windDirectionInDegree.toString() << ::std::endl
             << "Precipitation in percentage: " << m_precipitationInPercentage.toString() << ::std::endl
             << "PrecipitationProbability: " << m_precipitationProbability << ::std::endl
             << "CloudCoverage: " << ::navmiddleware::toString(m_cloudCover) << ::std::endl
             << "UV Index: " << ::navmiddleware::toString(m_uVIndex) << ::std::endl
             << "Weather Situation : " << ::navmiddleware::toString(m_weatherSituation) << ::std::endl
             << "WeatherSituation Icon size: " << m_weatherSituationIcon.getBlobSize() << ::std::endl
             << "Weather Severity : " << ::navmiddleware::toString(m_weatherSeverity) << ::std::endl
             << "WeatherAlert Icon size: " << m_weatherAlertIcon.getBlobSize() << ::std::endl
             << "City Name: " << m_cityName << ::std::endl
             << "WeatherIconType: " << m_weatherIconType << ::std::endl
             << "weatherReportTimeHour: " << static_cast<int>(m_weatherReportTimeHour) << ::std::endl
             << "WeatherReportTime(string): " << m_weatherReportTime << ::std::endl;
      return stream.str();
   }

   bool operator==(const WeatherReport& rhs) const
   {
      return (m_currentTemperatureInMilliKelvin == rhs.m_currentTemperatureInMilliKelvin) &&
             (m_currentTemperature == rhs.m_currentTemperature) &&
             (m_minimumTemperatureInMilliKelvin == rhs.m_minimumTemperatureInMilliKelvin) &&
             (m_minTemperature == rhs.m_minTemperature) &&
             (m_maximumTemperatureInMilliKelvin == rhs.m_maximumTemperatureInMilliKelvin) &&
             (m_maxTemperature == rhs.m_maxTemperature) &&
             (m_humidityInPercentage == rhs.m_humidityInPercentage) &&
             (m_relativeHumidity == rhs.m_relativeHumidity) &&
             (m_windSpeedInCentimeterSecond == rhs.m_windSpeedInCentimeterSecond) &&
             (m_windSpeed == rhs.m_windSpeed) &&
             (m_windDirectionInDegree == rhs.m_windDirectionInDegree) &&
             (m_precipitationInPercentage == rhs.m_precipitationInPercentage) &&
             (m_precipitationProbability == rhs.m_precipitationProbability) &&
             (m_cloudCover == rhs.m_cloudCover) &&
             (m_uVIndex == rhs.m_uVIndex) &&
             (m_weatherSituation == rhs.m_weatherSituation) &&
             (m_weatherSituationIcon == rhs.m_weatherSituationIcon) &&
             (m_weatherSeverity == rhs.m_weatherSeverity) &&
             (m_weatherAlertIcon == rhs.m_weatherAlertIcon) &&
             (m_cityName == rhs.m_cityName) &&
             (m_weatherIconType == rhs.m_weatherIconType) &&
             (m_weatherReportTimeHour == rhs.m_weatherReportTimeHour) &&
             (m_weatherReportTime == rhs.m_weatherReportTime);
   }

   bool operator!=(const WeatherReport& rhs) const
   {
      return !(*this == rhs);
   }

private:
   ValidValue<int> m_currentTemperatureInMilliKelvin;
   ::std::string m_currentTemperature;
   ValidValue<int> m_minimumTemperatureInMilliKelvin;
   ::std::string m_minTemperature;
   ValidValue<int> m_maximumTemperatureInMilliKelvin;
   ::std::string m_maxTemperature;
   ValidValue<int> m_humidityInPercentage;
   ::std::string m_relativeHumidity;
   ValidValue<int> m_windSpeedInCentimeterSecond;
   ::std::string m_windSpeed;
   ValidValue<int> m_windDirectionInDegree;
   ValidValue<int> m_precipitationInPercentage;
   ::std::string m_precipitationProbability;
   CloudCoverage m_cloudCover;
   UVIndex m_uVIndex;
   WeatherSituation m_weatherSituation;
   Image m_weatherSituationIcon;
   WeatherSeverity m_weatherSeverity;
   Image m_weatherAlertIcon;
   ::std::string m_cityName;
   WeatherIconType m_weatherIconType;
   /** weatherReportTimeHour in 24 hour format*/
   int8_t m_weatherReportTimeHour;
   /** m_weatherReportTime will provide only hour value based on time format settings*/
   ::std::string m_weatherReportTime;
};

class WeatherDataInfo
{
public:
   WeatherDataInfo()
      : m_locationType(LOCATION_TYPE__CURRENT_VEHICLE_POSITION)
      , m_weatherReportType(WEATHER_REPORT_TYPE__DAILY)
   {}

   void setWeatherReportType(WeatherReportType weatherReportType)
   {
      m_weatherReportType = weatherReportType;
   }

   WeatherReportType getWeatherReportType() const
   {
      return m_weatherReportType;
   }

   const ::std::vector<WeatherReport>& getWeatherReportList() const
   {
      return m_weatherReportList;
   }

   ::std::vector<WeatherReport>& getWeatherReportList()
   {
      return m_weatherReportList;
   }

   void setLocationType(LocationType locationType)
   {
      m_locationType = locationType;
   }

   LocationType getLocationType() const
   {
      return m_locationType;
   }

   void setDestinationIndex(const ValidValue<unsigned int>& destinationIndex)
   {
      m_destinationIndex = destinationIndex;
   }

   const ValidValue<unsigned int>& getDestinationIndex() const
   {
      return m_destinationIndex ;
   }

   void setUnknownWeatherSituationIcon(const ValidValue<Image>& unknownWeatherSituationIcon)
   {
      m_unknownWeatherSituationIcon = unknownWeatherSituationIcon;
   }

   const ValidValue<Image>& getUnknownWeatherSituationIcon() const
   {
      return m_unknownWeatherSituationIcon;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "WeatherDataInfo payload:" << ::std::endl
             << "\tm_locationType: " << ::navmiddleware::toString(m_locationType) << ::std::endl
             << "\tm_destinationIndex Valid: " << m_destinationIndex.toString() << ::std::endl;

      for (unsigned int i = 0; i < m_weatherReportList.size(); i++)
      {
         stream << "\t-> " << m_weatherReportList[i].toString();
      }

      if(m_unknownWeatherSituationIcon.isValid())
      {
         stream << "m_unknownWeatherSituationIcon blob size -> " << m_unknownWeatherSituationIcon.getValue().getBlobSize() << ::std::endl;
      }
      else
      {
         stream << "m_unknownWeatherSituationIcon is invalid" << ::std::endl;
      }

      return stream.str();
   }

   bool operator==(const WeatherDataInfo& rhs) const
   {
      return (m_locationType == rhs.m_locationType) &&
             (m_destinationIndex == rhs.m_destinationIndex) &&
             (m_weatherReportType == rhs.m_weatherReportType) &&
             (m_weatherReportList == rhs.m_weatherReportList);
   }

   bool operator!=(const WeatherDataInfo& rhs)const
   {
      return !(*this == rhs);
   }

private:
   /* Providing the location type in infos for scenarios where weather report is fetched at a different location type instead of requested location type.
    * This can happen if the requested location type is within the threshold distance(if any) from CVP.
    * In such cases, weather report for CVP will be fetched instead of requested location type.
    */
   LocationType m_locationType;
   ValidValue< unsigned int> m_destinationIndex;
   WeatherReportType m_weatherReportType;
//   unknownWeatherSituationIcon would be available to HMI only when the WeatherReport is empty
   ValidValue< Image> m_unknownWeatherSituationIcon;
   ::std::vector<WeatherReport> m_weatherReportList;
};

enum OnlineServiceAvailability
{
   /** Connection state is unknown (initial state) */
   ONLINE_SERVICE_AVAILABILITY__UNKNOWN,
   /** Connection to the online service is established */
   ONLINE_SERVICE_AVAILABILITY__AVAILABLE,
   /** Connection to the online service is not possible due to network issues
       This state is set if the first attempt to get an internet connection failed.
       Reason maybe a connection problem, e.g. phone not paired, but also a poor reception quality.
       So this state can be used to give a hint to check if setup is correct for internet connection. */
   ONLINE_SERVICE_AVAILABILITY__NO_INTERNET_CONNECTION,
   /** The authentication with the online service center failed */
   ONLINE_SERVICE_AVAILABILITY__AUTHENTICATION_INVALID,
   /** Internet connection got lost (but was available before), data cannot be received.
       This is normally triggered when entering an area with poor reception quality.
       This state is normally no indication for the user to check the setup. */
   ONLINE_SERVICE_AVAILABILITY__CONNECTION_LOST,
   /** No connection, due to a reason not covered by the other values */
   ONLINE_SERVICE_AVAILABILITY__NO_CONNECTION,
   /** The online service is provided but not subscribed by the device, i.e. respective buttons should be disabled by HMI */
   ONLINE_SERVICE_AVAILABILITY__NO_SUBSCRIPTION,
   /** The online service provided and subscribed by the device, but user activated the privacy mode */
   ONLINE_SERVICE_AVAILABILITY__PRIVACYMODE_ON,
   /** The online service is not provided for the device at all, i.e. respective buttons should be removed from HMI */
   ONLINE_SERVICE_AVAILABILITY__NOT_PROVIDED
};

inline ::std::string toString(OnlineServiceAvailability onlineServiceAvailability)
{
   switch(onlineServiceAvailability)
   {
   case ONLINE_SERVICE_AVAILABILITY__UNKNOWN:
      return "ONLINE_SERVICE_AVAILABILITY__UNKNOWN";
   case ONLINE_SERVICE_AVAILABILITY__AVAILABLE:
      return "ONLINE_SERVICE_AVAILABILITY__AVAILABLE";
   case ONLINE_SERVICE_AVAILABILITY__NO_INTERNET_CONNECTION:
      return "ONLINE_SERVICE_AVAILABILITY__NO_INTERNET_CONNECTION";
   case ONLINE_SERVICE_AVAILABILITY__AUTHENTICATION_INVALID:
      return "ONLINE_SERVICE_AVAILABILITY__AUTHENTICATION_INVALID";
   case ONLINE_SERVICE_AVAILABILITY__CONNECTION_LOST:
      return "ONLINE_SERVICE_AVAILABILITY__CONNECTION_LOST";
   case ONLINE_SERVICE_AVAILABILITY__NO_CONNECTION:
      return "ONLINE_SERVICE_AVAILABILITY__NO_CONNECTION";
   case ONLINE_SERVICE_AVAILABILITY__NO_SUBSCRIPTION:
      return "ONLINE_SERVICE_AVAILABILITY__NO_SUBSCRIPTION";
   case ONLINE_SERVICE_AVAILABILITY__PRIVACYMODE_ON:
      return "ONLINE_SERVICE_AVAILABILITY__PRIVACYMODE_ON";
   case ONLINE_SERVICE_AVAILABILITY__NOT_PROVIDED:
      return "ONLINE_SERVICE_AVAILABILITY__NOT_PROVIDED";
   default:
      ::std::stringstream stream;
      stream << "ONLINE_SERVICE_AVAILABILITY__<" << static_cast<unsigned int>(onlineServiceAvailability) << ">";
      return stream.str();
   }
}

struct Color
{
   Color():
      m_red(0),
      m_green(0),
      m_blue(0),
      m_alpha(0) {}

   Color(unsigned char red, unsigned char green, unsigned char blue, unsigned char alpha):
      m_red(red),m_green(green),m_blue(blue),m_alpha(alpha) {}

   Color(const Color& routeColor):
      m_red(routeColor.m_red),
      m_green(routeColor.m_green),
      m_blue(routeColor.m_blue),
      m_alpha(routeColor.m_alpha) {}

   Color& operator=(const Color& rhs)
   {
      if (&rhs != this)
      {
         m_red = rhs.m_red;
         m_green = rhs.m_green;
         m_blue = rhs.m_blue;
         m_alpha = rhs.m_alpha;
      }
      return *this;
   }

   bool operator==(const Color& rhs)const
   {
      return((m_red == rhs.m_red)&&
             (m_green == rhs.m_green)&&
             (m_blue == rhs.m_blue)&&
             (m_alpha == rhs.m_alpha));
   }

   bool operator!=(const Color& rhs)const
   {
      return ! (*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;

      stream << static_cast<int>(m_red)<< ","<< static_cast<int>(m_green)
             << ","<< static_cast<int>(m_blue)<<","<< static_cast<int>(m_alpha)<< ::std::endl;

      return stream.str();
   }

   unsigned char m_red;
   unsigned char m_green;
   unsigned char m_blue;
   unsigned char m_alpha;
};

class LanguageEntryInfo
{
public:
   LanguageEntryInfo()
   {
   }

   void setIsoLanguageCode(const ::std::string& isoLanugageCode)
   {
      m_isoLanguageCode = isoLanugageCode;
   }

   const ::std::string& getIsoLanguageCode() const
   {
      return m_isoLanguageCode;
   }

   void setIsoCountryCode(const ::std::string& isoCountryCode)
   {
      m_isoCountryCode = isoCountryCode;
   }

   const ::std::string& getIsoCountryCode() const
   {
      return m_isoCountryCode;
   }

   void setIsoScriptCode(const ::std::string& isoScriptcode)
   {
      m_isoScriptCode = isoScriptcode;
   }

   const ::std::string& getIsoScriptCode() const
   {
      return m_isoScriptCode;
   }

   bool operator==(const LanguageEntryInfo& rhs) const
   {
      return (m_isoLanguageCode == rhs.m_isoLanguageCode)
             && (m_isoCountryCode == rhs.m_isoCountryCode)
             && (m_isoScriptCode == rhs.m_isoScriptCode);
   }

   bool operator!=(const LanguageEntryInfo& rhs) const
   {
      return (!(*this == rhs));
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;

      stream << " isoLanguageCode" << m_isoLanguageCode<< ::std::endl
             << " isoCountryCode"  << m_isoCountryCode << ::std::endl
             << " isoScriptCode"  << m_isoScriptCode << ::std::endl;

      return stream.str();
   }

private:
   ::std::string m_isoLanguageCode;
   ::std::string m_isoCountryCode;
   ::std::string m_isoScriptCode;
};

class RoadNumberInfo
{
public:
   RoadNumberInfo() {}

   bool operator == (const RoadNumberInfo& rhs) const
   {
      return (m_iconData == rhs.m_iconData)&&
             (m_cardinalDirection == rhs.m_cardinalDirection);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("RoadNumberData:\n");
      stream << "road number icon blob size: " << m_iconData.getBlobSize() << ::std::endl
             << "road number icon blob address: " << static_cast<const void*>(m_iconData.getBlob()) << ::std::endl;
      stream << "road number cardinal direction: " << m_cardinalDirection << ::std::endl;
      return stream.str();
   }
   Image m_iconData;
   // road number direction as string e.g. "NORTH" in English and "NORD" in French based on the current regional language.
   // currently supported only for USA and Canada
   ::std::string m_cardinalDirection;
};

// Template function to print payload of list of info objects
template<typename InfoObjectType> ::std::string toString(::std::vector<InfoObjectType> infoObjects)
{
   ::std::stringstream stream;
   for(size_t index = 0; index < infoObjects.size(); ++index)
   {
      stream << "[" << index << "] - " << infoObjects[index].toString();
   }
   return stream.str();
}

// Template function to print payload of map of enum and info object
template<typename EnumType, typename InfoObjectType> ::std::string toString(::std::map<EnumType, InfoObjectType> infoObjectDict)
{
   ::std::stringstream stream;
   typename ::std::map<EnumType, InfoObjectType>::iterator iter;
   for(iter = infoObjectDict.begin(); iter != infoObjectDict.end(); ++iter)
   {
      stream << ::std::endl
             << "[" << toString(iter->first) << "]" << ::std::endl
             << iter->second.toString();
   }
   return stream.str();
}

enum MapViewId
{
   MAP_VIEW_ID__PRIMARY = 0,
   MAP_VIEW_ID__SECONDARY,
   MAP_VIEW_ID__THREE,
   MAP_VIEW_ID__FOUR
};

inline ::std::string toString(MapViewId mapViewId)
{
   switch (mapViewId)
   {
   case MAP_VIEW_ID__PRIMARY:
      return "MAP_VIEW_ID__PRIMARY";
   case MAP_VIEW_ID__SECONDARY:
      return "MAP_VIEW_ID__SECONDARY";
   case MAP_VIEW_ID__THREE:
      return "MAP_VIEW_ID__THREE";
   case MAP_VIEW_ID__FOUR:
      return "MAP_VIEW_ID__FOUR";
   default:
      ::std::stringstream stream;
      stream << "MAP_VIEW_ID__<" << static_cast<unsigned int>(mapViewId) << ">";
      return stream.str();
   }
}

// ID type for line type object
class LineId
{
public:

   LineId() :
      m_tile(0),
      m_updateRegion(0),
      m_line(0),
      m_lineType(0)
   {}

   bool operator == (const LineId& rhs) const
   {
      return ((m_tile == rhs.m_tile) &&
              (m_updateRegion == rhs.m_updateRegion) &&
              (m_line == rhs.m_line) &&
              (m_lineType == rhs.m_lineType));
   }

   bool operator != (const LineId& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("LineId payload:\n");

      stream << ::std::endl
             << "LineId= " << m_line << ::std::endl
             << "Update region = " << m_updateRegion << ::std::endl
             << "Tile ID = " << m_tile << ::std::endl
             << "Line Type = " << static_cast<uint32_t>(m_lineType) << ::std::endl;

      return stream.str();
   }

   uint32_t m_tile;
   uint16_t m_updateRegion;
   uint16_t m_line;
   uint8_t  m_lineType; //The type of the line (ex. Road, Railway line etc.)
};

class VehicleRestrictionsInfo
{
public:
   //<-- INF4CV
   enum RestrictionType
   {
     RESTRICTION_TYPE__HARD_PHYSICAL,
     RESTRICTION_TYPE__HARD_GENERAL,
     RESTRICTION_TYPE__SOFT_PHYSICAL,
     RESTRICTION_TYPE__SOFT_GENERAL,
     RESTRICTION_TYPE__UNKNOWN
   };

   inline ::std::string toString(const RestrictionType restrictionType) const
   {
      switch (restrictionType)
      {
      case RESTRICTION_TYPE__HARD_PHYSICAL:
         return "RESTRICTION_TYPE__HARD_PHYSICAL";
      case RESTRICTION_TYPE__HARD_GENERAL:
         return "RESTRICTION_TYPE__HARD_GENERAL";
      case RESTRICTION_TYPE__SOFT_PHYSICAL:
         return "RESTRICTION_TYPE__SOFT_PHYSICAL";
      case RESTRICTION_TYPE__SOFT_GENERAL:
         return "RESTRICTION_TYPE__SOFT_GENERAL";
      default:
         ::std::stringstream stream;
         stream << "RESTRICTION_TYPE__<" << static_cast<unsigned int>(restrictionType) << ">";
         return stream.str();
      }
   }
   //->

   enum VehicleType
   {
      VEHICLE_TYPE__UNKNOWN,
      VEHICLE_TYPE__MOTORIZED_VEHICLE,
      VEHICLE_TYPE__PERSONAL_CAR,
      VEHICLE_TYPE__TRUCK,
      VEHICLE_TYPE__MOTORCYCLE,
      VEHICLE_TYPE__BUS,
      VEHICLE_TYPE__NOT_LIGHT_MOTORIZED_VEHICLE
   };

   inline ::std::string toString(const VehicleType vehicleType) const
   {
      switch (vehicleType)
      {
      case VEHICLE_TYPE__UNKNOWN:
         return "VEHICLE_TYPE__UNKNOWN";
      case VEHICLE_TYPE__MOTORIZED_VEHICLE:
         return "VEHICLE_TYPE__MOTORIZED_VEHICLE";
      case VEHICLE_TYPE__PERSONAL_CAR:
         return "VEHICLE_TYPE__PERSONAL_CAR";
      case VEHICLE_TYPE__TRUCK:
         return "VEHICLE_TYPE__TRUCK";
      case VEHICLE_TYPE__MOTORCYCLE:
         return "VEHICLE_TYPE__MOTORCYCLE";
      case VEHICLE_TYPE__BUS:
         return "VEHICLE_TYPE__BUS";
      case VEHICLE_TYPE__NOT_LIGHT_MOTORIZED_VEHICLE:
         return "VEHICLE_TYPE__NOT_LIGHT_MOTORIZED_VEHICLE";
      default:
         ::std::stringstream stream;
         stream << "VEHICLE_TYPE__<" << static_cast<unsigned int>(vehicleType) << ">";
         return stream.str();
      }
   }

   class CargoType
   {
   public:
      CargoType()
         :m_hasSpecialLoad(false)
         ,m_hasWaterPolluting(false)
         ,m_hasExplosive(false)
         ,m_hasGasses(false)
         ,m_hasFlammableLiquids(false)
         ,m_hasFlammableSolids(false)
         ,m_hasOxidizersAndOrganicPeroxides(false)
         ,m_hasToxicAndInfectiousSubstances(false)
         ,m_hasRadioactiveSubstances(false)
         ,m_hasCorrosiveSubstances(false)
         ,m_hasOtherDangerousGoods(false)
      {}

      bool operator == (const CargoType& rhs) const
      {
         return ((m_hasSpecialLoad == rhs.m_hasSpecialLoad) &&
                 (m_hasWaterPolluting == rhs.m_hasWaterPolluting) &&
                 (m_hasExplosive == rhs.m_hasExplosive) &&
                 (m_hasGasses == rhs.m_hasGasses) &&
                 (m_hasFlammableLiquids == rhs.m_hasFlammableLiquids) &&
                 (m_hasFlammableSolids == rhs.m_hasFlammableSolids) &&
                 (m_hasOxidizersAndOrganicPeroxides == rhs.m_hasOxidizersAndOrganicPeroxides) &&
                 (m_hasToxicAndInfectiousSubstances == rhs.m_hasToxicAndInfectiousSubstances) &&
                 (m_hasRadioactiveSubstances == rhs.m_hasRadioactiveSubstances) &&
                 (m_hasCorrosiveSubstances == rhs.m_hasCorrosiveSubstances) &&
                 (m_hasOtherDangerousGoods == rhs.m_hasOtherDangerousGoods));
      }

      bool operator != (const CargoType& rhs) const
      {
         return !(*this == rhs);
      }

      bool m_hasSpecialLoad;
      bool m_hasWaterPolluting;
      bool m_hasExplosive;
      bool m_hasGasses;
      bool m_hasFlammableLiquids;
      bool m_hasFlammableSolids;
      bool m_hasOxidizersAndOrganicPeroxides;
      bool m_hasToxicAndInfectiousSubstances;
      bool m_hasRadioactiveSubstances;
      bool m_hasCorrosiveSubstances;
      bool m_hasOtherDangerousGoods;

      ::std::string toString() const
      {
         ::std::stringstream stream("CargoType payload:\n");
         stream << (m_hasSpecialLoad ? "SpecialLoad, " : "")
                << (m_hasWaterPolluting ? "WaterPolluting, " : "")
                << (m_hasExplosive ? "Explosive, " : "")
                << (m_hasGasses ? "Gasses, " : "")
                << (m_hasFlammableLiquids ? "FlammableLiquids, " : "")
                << (m_hasFlammableSolids ? "FlammableSolids, " : "")
                << (m_hasOxidizersAndOrganicPeroxides ? "OxidizersAndOrganicPeroxides, " : "")
                << (m_hasToxicAndInfectiousSubstances ? "ToxicAndInfectiousSubstances, " : "")
                << (m_hasRadioactiveSubstances ? "RadioactiveSubstances, " : "")
                << (m_hasCorrosiveSubstances ? "CorrosiveSubstances, " : "")
                << (m_hasOtherDangerousGoods ? "OtherDangerousGoods " : "") << ::std::endl;
         return stream.str();
      }
   };

   //<-- INF4CV
   VehicleRestrictionsInfo()
      : m_unitSystemForRestriction(DISTANCEUNITSYSTEM_METRIC)
      , m_unitSystemForWeightRestriction(DISTANCEUNITSYSTEM_METRIC)
      , m_heightOfVehicle(0)
      , m_widthOfVehicle(0)
      , m_lengthOfVehicle(0)
      , m_maxWeightOfVehicle(0)
      , m_maxWeightPerAxleOfVehicle(0)
      , m_restrictionTypeForDimension(RESTRICTION_TYPE__UNKNOWN)
      , m_restrictionTypeForWeight(RESTRICTION_TYPE__UNKNOWN)
      , m_restrictionTypeForCargo(RESTRICTION_TYPE__UNKNOWN)
      , m_restrictionTypeForVehicleType(RESTRICTION_TYPE__UNKNOWN)
   {
   }
   //->

   bool operator == (const VehicleRestrictionsInfo& rhs) const
   {
      return ((m_lengthRestrictionValueWithUnit == rhs.m_lengthRestrictionValueWithUnit) &&
              (m_widthRestrictionValueWithUnit == rhs.m_widthRestrictionValueWithUnit) &&
              (m_heightRestrictionValueWithUnit == rhs.m_heightRestrictionValueWithUnit) &&
              (m_weightRestrictionValueWithUnit == rhs.m_weightRestrictionValueWithUnit) &&
              (m_weightPerAxleRestrictionValueWithUnit == rhs.m_weightPerAxleRestrictionValueWithUnit) &&   //<-- INF4CV
              (m_vehicleTypeRestriction == rhs.m_vehicleTypeRestriction) &&
              (m_cargoTypeRestriction == rhs.m_cargoTypeRestriction) &&
              (m_lineId == rhs.m_lineId) &&
              //<-- INF4CV
              (m_unitSystemForRestriction == rhs.m_unitSystemForRestriction) &&
              (m_unitSystemForWeightRestriction == rhs.m_unitSystemForWeightRestriction) &&
              (m_heightOfVehicle == rhs.m_heightOfVehicle) &&
              (m_widthOfVehicle == rhs.m_widthOfVehicle) &&
              (m_lengthOfVehicle == rhs.m_lengthOfVehicle) &&
              (m_maxWeightOfVehicle == rhs.m_maxWeightOfVehicle) &&
              (m_maxWeightPerAxleOfVehicle == rhs.m_maxWeightPerAxleOfVehicle) &&
              (m_restrictionTypeForDimension == rhs.m_restrictionTypeForDimension) &&
              (m_restrictionTypeForWeight == rhs.m_restrictionTypeForWeight) &&
              (m_restrictionTypeForCargo == rhs.m_restrictionTypeForCargo) &&
              (m_restrictionTypeForVehicleType == rhs.m_restrictionTypeForVehicleType));
              //->
   }

   bool operator != (const VehicleRestrictionsInfo& rhs) const
   {
      return !(*this == rhs);
   }

   void setLengthRestrictionValueWithUnit(const ::std::string& lengthRestrictionValueWithUnit)
   {
      m_lengthRestrictionValueWithUnit = lengthRestrictionValueWithUnit;
   }

   const ::std::string& getLengthRestrictionValueWithUnit() const
   {
      return m_lengthRestrictionValueWithUnit;
   }

   void setWidthRestrictionValueWithUnit(const ::std::string& widthRestrictionValueWithUnit)
   {
      m_widthRestrictionValueWithUnit = widthRestrictionValueWithUnit;
   }

   const ::std::string& getWidthRestrictionValueWithUnit() const
   {
      return m_widthRestrictionValueWithUnit;
   }

   void setHeightRestrictionValueWithUnit(const ::std::string& heightRestrictionValueWithUnit)
   {
      m_heightRestrictionValueWithUnit = heightRestrictionValueWithUnit;
   }

   const ::std::string& getHeightRestrictionValueWithUnit() const
   {
      return m_heightRestrictionValueWithUnit;
   }

   void setWeightRestrictionValueWithUnit(const ::std::string& weightRestrictionValueWithUnit)
   {
      m_weightRestrictionValueWithUnit = weightRestrictionValueWithUnit;
   }

   const ::std::string& getWeightRestrictionValueWithUnit() const
   {
      return m_weightRestrictionValueWithUnit;
   }

   //<-- INF4CV
   void setWeightPerAxleRestrictionValueWithUnit(const ::std::string& weightPerAxleRestrictionValueWithUnit)
   {
      m_weightPerAxleRestrictionValueWithUnit = weightPerAxleRestrictionValueWithUnit;
   }

   const ::std::string& getWeightPerAxleRestrictionValueWithUnit() const
   {
      return m_weightPerAxleRestrictionValueWithUnit;
   }
   //->

   void setVehicleTypeRestriction(const ::std::vector<VehicleType>& vehicleTypeRestriction)
   {
      m_vehicleTypeRestriction = vehicleTypeRestriction;
   }

   const ::std::vector<VehicleType>& getVehicleTypeRestriction() const
   {
      return m_vehicleTypeRestriction;
   }

   void setCargoTypeRestriction(const ValidValue<CargoType>& cargoTypeRestriction)
   {
      m_cargoTypeRestriction = cargoTypeRestriction;
   }

   const ValidValue<CargoType>& getCargoTypeRestriction() const
   {
      return m_cargoTypeRestriction;
   }

   void setLineId(const ValidValue<LineId>& lineId)
   {
      m_lineId = lineId;
   }

   const ValidValue<LineId>& getLineId() const
   {
      return m_lineId;
   }

   //<-- INF4CV
   DistanceUnitSystem getUnitSystem() const
   {
      return m_unitSystemForRestriction;
   }

   void setUnitSystem(DistanceUnitSystem unitSystem)
   {
      m_unitSystemForRestriction = unitSystem;
   }

   DistanceUnitSystem getUnitSystemForWeightRestriction() const
   {
      return m_unitSystemForWeightRestriction;
   }

   void setUnitSystemForWeightRestriction(DistanceUnitSystem unitSystem)
   {
      m_unitSystemForWeightRestriction = unitSystem;
   }

   unsigned int getHeightOfVehicle() const
   {
      return m_heightOfVehicle;
   }

   void setHeightOfVehicle(unsigned int height)
   {
      m_heightOfVehicle = height;
   }

   unsigned int getWidthOfVehicle() const
   {
      return m_widthOfVehicle;
   }

   void setWidthOfVehicle(unsigned int width)
   {
      m_widthOfVehicle = width;
   }

   unsigned int getLengthOfVehicle() const
   {
      return m_lengthOfVehicle;
   }

   void setLengthOfVehicle(unsigned int length)
   {
      m_lengthOfVehicle = length;
   }

   unsigned int getMaxWeightOfVehicle() const
   {
      return m_maxWeightOfVehicle;
   }

   void setMaxWeightOfVehicle(unsigned int weight)
   {
      m_maxWeightOfVehicle = weight;
   }

   unsigned int getMaxWeightPerAxleOfVehicle() const
   {
      return m_maxWeightPerAxleOfVehicle;
   }

   void setMaxWeightPerAxleOfVehicle(unsigned int weightPerAxle)
   {
      m_maxWeightPerAxleOfVehicle = weightPerAxle;
   }

   RestrictionType getRestrictionTypeForDimension() const
   {
      return m_restrictionTypeForDimension;
   }

   void setRestrictionTypeForDimension(RestrictionType restrictionType)
   {
      m_restrictionTypeForDimension = restrictionType;
   }

   RestrictionType getRestrictionTypeForWeight() const
   {
   return m_restrictionTypeForWeight;
   }

   void setRestrictionTypeForWeight(RestrictionType restrictionType)
   {
   m_restrictionTypeForWeight = restrictionType;
   }

   RestrictionType getRestrictionTypeForCargo() const
   {
   return m_restrictionTypeForCargo;
   }

   void setRestrictionTypeForCargo(RestrictionType restrictionType)
   {
   m_restrictionTypeForCargo = restrictionType;
   }

   RestrictionType getRestrictionTypeForVehicleType() const
   {
      return m_restrictionTypeForVehicleType;
   }

   void setRestrictionTypeForVehicleType(RestrictionType restrictionType)
   {
      m_restrictionTypeForVehicleType = restrictionType;
   }
   //->

   ::std::string toString() const
   {
      ::std::stringstream stream("VehicleRestrictionsInfo payload:\n");
      stream << "m_lengthRestrictionValueWithUnit: " << m_lengthRestrictionValueWithUnit << ::std::endl;
      stream << "m_widthRestrictionValueWithUnit: " << m_widthRestrictionValueWithUnit << ::std::endl;
      stream << "m_heightRestrictionValueWithUnit: " << m_heightRestrictionValueWithUnit << ::std::endl;
      stream << "m_weightRestrictionValueWithUnit: " << m_weightRestrictionValueWithUnit << ::std::endl;
      stream << "m_weightPerAxleRestrictionValueWithUnit: " << m_weightPerAxleRestrictionValueWithUnit << ::std::endl;   //<-- INF4CV
      //<-- INF4CV
      stream << "m_unitSystemForRestriction: " << m_unitSystemForRestriction << ::std::endl;
      stream << "m_unitSystemForWeightRestriction: " << m_unitSystemForWeightRestriction << ::std::endl;
      stream << "m_lengthOfVehicle: " << m_lengthOfVehicle << ::std::endl;
      stream << "m_widthOfVehicle: " << m_widthOfVehicle << ::std::endl;
      stream << "m_heightOfVehicle: " << m_heightOfVehicle << ::std::endl;
      stream << "m_maxWeightOfVehicle: " << m_maxWeightOfVehicle << ::std::endl;
      stream << "m_maxWeightPerAxleOfVehicle: " << m_maxWeightPerAxleOfVehicle << ::std::endl;
      stream << "m_restrictionTypeForDimension: " << toString(m_restrictionTypeForDimension) << ::std::endl;
      stream << "m_restrictionTypeForWeight: " << toString(m_restrictionTypeForWeight) << ::std::endl;
      stream << "m_restrictionTypeForCargo: " << toString(m_restrictionTypeForCargo) << ::std::endl;
      stream << "m_restrictionTypeForVehicleType: " << toString(m_restrictionTypeForVehicleType) << ::std::endl;
      //->
      if(!m_vehicleTypeRestriction.empty())
      {
         ::std::vector< VehicleType>::const_iterator vehicleTypeItr = m_vehicleTypeRestriction.begin();
         for (; vehicleTypeItr !=  m_vehicleTypeRestriction.end(); ++vehicleTypeItr)
         {
            stream  << toString(*vehicleTypeItr) << " ";
         }
         stream << ::std::endl;
      }
      else
      {
         stream << "m_vehicleTypeRestriction: none" << ::std::endl;
      }

      if(m_cargoTypeRestriction.isValid())
      {
         stream << "m_cargoTypeRestriction: " << m_cargoTypeRestriction.getValue().toString() << ::std::endl;
      }
      else
      {
         stream << "m_cargoTypeRestriction: none" << ::std::endl;
      }

      if(m_lineId.isValid())
      {
         stream << "m_lineId: " << m_lineId.getValue().toString()<< ::std::endl;
      }
      else
      {
         stream << "m_lineId: none" << ::std::endl;
      }
      return stream.str();
   }

private:
   //<-- INF4CV
   /**
    * PresCtrl sets the restriction values as string in the below units.
    * Metric: centimeter, Imperial: inch (For length, width and height)
    * Metric: kilogram, Imperial: pound (For weight)
    */
   //->
   // Contains length restriction value on current road segment. Empty if no restriction.
   ::std::string m_lengthRestrictionValueWithUnit;
   // Contains width restriction value on current road segment. Empty if no restriction.
   ::std::string m_widthRestrictionValueWithUnit;
   // Contains height restriction value on current road segment. Empty if no restriction.
   ::std::string m_heightRestrictionValueWithUnit;
   // Contains weight restriction value on current road segment. Empty if no restriction.
   ::std::string m_weightRestrictionValueWithUnit;
   //<-- INF4CV
   // Contains weight per axle restriction value on current road segment. Empty if no restriction.
   ::std::string m_weightPerAxleRestrictionValueWithUnit;
   //->
   // Contains vehicle type restriction on current road segment. Invalid if no restriction.
   ::std::vector<VehicleType> m_vehicleTypeRestriction;
   // Contains cargo type restriction on current road segment. Invalid if no restriction.
   ValidValue<CargoType> m_cargoTypeRestriction;
   // Contains lineId information to highlight the line on map
   ValidValue<LineId> m_lineId;

   //<-- INF4CV
   DistanceUnitSystem m_unitSystemForRestriction;
   DistanceUnitSystem m_unitSystemForWeightRestriction;
   /**
    * PresCtrl sets the restriction values as integer in the below units.
    * Metric: millimeter, Imperial: inch (For length, width and height)
    * Metric: kilogram, Imperial: pound (For weight)
    */
   // Restriction value is set to 0 if there is no restriction
   unsigned int m_heightOfVehicle;
   unsigned int m_widthOfVehicle;
   unsigned int m_lengthOfVehicle;
   unsigned int m_maxWeightOfVehicle;
   unsigned int m_maxWeightPerAxleOfVehicle;

   RestrictionType m_restrictionTypeForDimension;
   RestrictionType m_restrictionTypeForWeight;
   RestrictionType m_restrictionTypeForCargo;
   RestrictionType m_restrictionTypeForVehicleType;
   //->
};

class CountryAndStateDetailsInfo
{
public:
   CountryAndStateDetailsInfo()
      :m_trafficSense(TRAFFIC_SENSE__RIGHTHAND_TRAFFIC)
      ,m_isMetricSystemUsed(false)
   {
   }

   class CountrySpeedLimitInfo
   {
   public:
      CountrySpeedLimitInfo()
         :m_speedLimit(0)
      {
      }

      class TimeRangeOfDayInfo
      {
      public:
         TimeRangeOfDayInfo()
            :m_isInclusive(false) {}

         bool operator == (const TimeRangeOfDayInfo& rhs) const
         {
            return (m_startTime == rhs.m_startTime)&&
                   (m_endTime == rhs.m_endTime) &&
                   (m_isInclusive == rhs.m_isInclusive);
         }

         bool operator != (const TimeRangeOfDayInfo& rhs) const
         {
            return !(*this == rhs);
         }

         ::std::string toString() const
         {
            ::std::stringstream stream;
            stream << "TimeRangeOfDayInfo payload" << ::std::endl;
            stream << "m_startTime = " << m_startTime << ::std::endl;
            stream << "m_endTime = " << m_endTime << ::std::endl;
            stream << "m_isInclusive = " << m_isInclusive << ::std::endl;
            return stream.str();
         }
         ::std::string m_startTime;
         ::std::string m_endTime;
         bool m_isInclusive;
      };

      class DateRangeOfYearInfo
      {
      public:
         DateRangeOfYearInfo()
            :m_isInclusive(false) {}

         bool operator == (const DateRangeOfYearInfo& rhs) const
         {
            return (m_startDate == rhs.m_startDate) &&
                   (m_endDate == rhs.m_endDate) &&
                   (m_isInclusive == rhs.m_isInclusive);
         }

         bool operator != (const DateRangeOfYearInfo& rhs) const
         {
            return !(*this == rhs);
         }

         ::std::string toString() const
         {
            ::std::stringstream stream;
            stream << "DateRangeOfYearInfo payload" << ::std::endl;
            stream << "m_startDate = " << m_startDate << ::std::endl;
            stream << "m_endDate = " << m_endDate << ::std::endl;
            stream <<"m_isInclusive = " << m_isInclusive<< ::std::endl;
            return stream.str();
         }

         // Default format used to represent start and end date is DD/MM/YYYY
         ::std::string m_startDate;
         ::std::string m_endDate;
         bool m_isInclusive;
      };

      class TimeRangeInfo
      {
      public:

         bool operator == (const TimeRangeInfo& rhs) const
         {
            return (m_timeRangeOfDayInfo == rhs.m_timeRangeOfDayInfo)
                   && (m_dateRangeOfYearInfo == rhs.m_dateRangeOfYearInfo);
         }
         bool operator != (const TimeRangeInfo& rhs) const
         {
            return !(*this == rhs);
         }

         ::std::string toString() const
         {
            ::std::stringstream stream;
            stream << "m_timeRangeOfDayInfo  = " << m_timeRangeOfDayInfo.toString() << ","
                   << "m_dateRangeOfYearInfo = " << m_dateRangeOfYearInfo.toString() << ::std::endl;
            return stream.str();
         }

         TimeRangeOfDayInfo  m_timeRangeOfDayInfo;
         DateRangeOfYearInfo m_dateRangeOfYearInfo;
      };

      enum SpeedLimitRoadType
      {
         SPEED_LIMIT_ROAD_TYPE__URBAN,
         SPEED_LIMIT_ROAD_TYPE__OUTSIDE_URBAN,
         SPEED_LIMIT_ROAD_TYPE__CONTROLLED_ACCESS,
         SPEED_LIMIT_ROAD_TYPE__MOTORWAY
      };

      inline ::std::string toString(const SpeedLimitRoadType speedLimitRoadType) const
      {
         switch(speedLimitRoadType)
         {
         case SPEED_LIMIT_ROAD_TYPE__URBAN:
            return "SPEED_LIMIT_ROAD_TYPE__URBAN";
         case SPEED_LIMIT_ROAD_TYPE__OUTSIDE_URBAN:
            return "SPEED_LIMIT_ROAD_TYPE__OUTSIDE_URBAN";
         case SPEED_LIMIT_ROAD_TYPE__CONTROLLED_ACCESS:
            return "SPEED_LIMIT_ROAD_TYPE__CONTROLLED_ACCESS";
         case SPEED_LIMIT_ROAD_TYPE__MOTORWAY:
            return "SPEED_LIMIT_ROAD_TYPE__MOTORWAY";
         default:
            ::std::stringstream stream;
            stream << "SPEED_LIMIT_ROAD_TYPE__<" << static_cast<unsigned int>(speedLimitRoadType) << ">";
            return stream.str();
         }
      }

      class FuzzyTimeInfo
      {
      public:
         FuzzyTimeInfo()
            : m_fuzzyTime(FUZZY_TIME__EXTERNAL)
            , m_isInclusive(false) {}

         enum FuzzyTime
         {
            FUZZY_TIME__EXTERNAL,
            FUZZY_TIME__DAWN,
            FUZZY_TIME__DUSK,
            FUZZY_TIME__SCHOOL,
            FUZZY_TIME__HOLIDAY,
            FUZZY_TIME__WINTER,
            FUZZY_TIME__SPRING,
            FUZZY_TIME__SUMMER,
            FUZZY_TIME__AUTUMN,
            FUZZY_TIME__HIGH_TIDE,
            FUZZY_TIME__LOW_TIDE,
            FUZZY_TIME__HIGH_WATER,
            FUZZY_TIME__LOW_WATER,
            FUZZY_TIME__WET,
            FUZZY_TIME__DRY,
            FUZZY_TIME__PEAK_HOURS,
            FUZZY_TIME__OFF_PEAK_HOURS,
            FUZZY_TIME__MORNING,
            FUZZY_TIME__EVENING_RUSH_HOUR,
            FUZZY_TIME__MORNING_RUSH_HOUR,
            FUZZY_TIME__DAY,
            FUZZY_TIME__NIGHT,
            FUZZY_TIME__NON_SCHOOL_HOURS,
            FUZZY_TIME__SCHOOL_HOURS,
            FUZZY_TIME__WHEN_CHILDREN_ARE_PRESENT,
            FUZZY_TIME__SUNRISE_TILL_SUNSET,
            FUZZY_TIME__SUNSET_TILL_SUNRISE,
            FUZZY_TIME__AFTERNOON,
            FUZZY_TIME__EVENT,
            FUZZY_TIME__MARKET_HOURS,
            FUZZY_TIME__UNDEFINED_OCCASION,
            FUZZY_TIME__RACE_DAYS,
            FUZZY_TIME__POLLUTION,
            FUZZY_TIME__EVENING,
            FUZZY_TIME__BUSINESS_HOURS,
            FUZZY_TIME__SKI_SEASON,
            FUZZY_TIME__TOURIST_SEASON,
            FUZZY_TIME__CHURCH_HOURS,
            FUZZY_TIME__SUMMER_SCHOOL,
            FUZZY_TIME__FUNERAL,
            FUZZY_TIME__HUNTING_SEASON,
            FUZZY_TIME__MILITARY_EXERCISE
         };

         inline ::std::string toString(const FuzzyTime fuzzyTime) const
         {
            switch(fuzzyTime)
            {
            case FUZZY_TIME__EXTERNAL:
               return "FUZZY_TIME__EXTERNAL";
            case FUZZY_TIME__DAWN:
               return "FUZZY_TIME__DAWN";
            case FUZZY_TIME__DUSK:
               return "FUZZY_TIME__DUSK";
            case FUZZY_TIME__SCHOOL:
               return "FUZZY_TIME__SCHOOL";
            case FUZZY_TIME__HOLIDAY:
               return "FUZZY_TIME__HOLIDAY";
            case FUZZY_TIME__WINTER:
               return "FUZZY_TIME__WINTER";
            case FUZZY_TIME__SPRING:
               return "FUZZY_TIME__SPRING";
            case FUZZY_TIME__SUMMER:
               return "FUZZY_TIME__SUMMER";
            case FUZZY_TIME__AUTUMN:
               return "FUZZY_TIME__AUTUMN";
            case FUZZY_TIME__HIGH_TIDE:
               return "FUZZY_TIME__HIGH_TIDE";
            case FUZZY_TIME__LOW_TIDE:
               return "FUZZY_TIME__LOW_TIDE";
            case FUZZY_TIME__HIGH_WATER:
               return "FUZZY_TIME__HIGH_WATER";
            case FUZZY_TIME__LOW_WATER:
               return "FUZZY_TIME__LOW_WATER";
            case FUZZY_TIME__WET:
               return "FUZZY_TIME__WET";
            case FUZZY_TIME__DRY:
               return "FUZZY_TIME__DRY";
            case FUZZY_TIME__PEAK_HOURS:
               return "FUZZY_TIME__PEAK_HOURS";
            case FUZZY_TIME__OFF_PEAK_HOURS:
               return "FUZZY_TIME__OFF_PEAK_HOURS";
            case FUZZY_TIME__MORNING:
               return "FUZZY_TIME__MORNING";
            case FUZZY_TIME__EVENING_RUSH_HOUR:
               return "FUZZY_TIME__EVENING_RUSH_HOUR";
            case FUZZY_TIME__MORNING_RUSH_HOUR:
               return "FUZZY_TIME__MORNING_RUSH_HOUR";
            case FUZZY_TIME__DAY:
               return "FUZZY_TIME__DAY";
            case FUZZY_TIME__NIGHT:
               return "FUZZY_TIME__NIGHT";
            case FUZZY_TIME__NON_SCHOOL_HOURS:
               return "FUZZY_TIME__NON_SCHOOL_HOURS";
            case FUZZY_TIME__SCHOOL_HOURS:
               return "FUZZY_TIME__SCHOOL_HOURS";
            case FUZZY_TIME__WHEN_CHILDREN_ARE_PRESENT:
               return "FUZZY_TIME__WHEN_CHILDREN_ARE_PRESENT";
            case FUZZY_TIME__SUNRISE_TILL_SUNSET:
               return "FUZZY_TIME__SUNRISE_TILL_SUNSET";
            case FUZZY_TIME__SUNSET_TILL_SUNRISE:
               return "FUZZY_TIME__SUNSET_TILL_SUNRISE";
            case FUZZY_TIME__AFTERNOON:
               return "FUZZY_TIME__AFTERNOON";
            case FUZZY_TIME__EVENT:
               return "FUZZY_TIME__EVENT";
            case FUZZY_TIME__MARKET_HOURS:
               return "FUZZY_TIME__MARKET_HOURS";
            case FUZZY_TIME__UNDEFINED_OCCASION:
               return "FUZZY_TIME__UNDEFINED_OCCASION";
            case FUZZY_TIME__RACE_DAYS:
               return "FUZZY_TIME__RACE_DAYS";
            case FUZZY_TIME__POLLUTION:
               return "FUZZY_TIME__POLLUTION";
            case FUZZY_TIME__EVENING:
               return "FUZZY_TIME__EVENING";
            case FUZZY_TIME__BUSINESS_HOURS:
               return "FUZZY_TIME__BUSINESS_HOURS";
            case FUZZY_TIME__SKI_SEASON:
               return "FUZZY_TIME__SKI_SEASON";
            case FUZZY_TIME__TOURIST_SEASON:
               return "FUZZY_TIME__TOURIST_SEASON";
            case FUZZY_TIME__CHURCH_HOURS:
               return "FUZZY_TIME__CHURCH_HOURS";
            case FUZZY_TIME__SUMMER_SCHOOL:
               return "FUZZY_TIME__SUMMER_SCHOOL";
            case FUZZY_TIME__FUNERAL:
               return "FUZZY_TIME__FUNERAL";
            case FUZZY_TIME__HUNTING_SEASON:
               return "FUZZY_TIME__HUNTING_SEASON";
            case FUZZY_TIME__MILITARY_EXERCISE:
               return "FUZZY_TIME__MILITARY_EXERCISE";
            default:
               ::std::stringstream stream;
               stream << "FUZZY_TIME__<" << static_cast<unsigned int>(fuzzyTime) << ">";
               return stream.str();
            }
         }

         bool operator == (const FuzzyTimeInfo& rhs) const
         {
            return (m_fuzzyTime == rhs.m_fuzzyTime)
                   && (m_isInclusive == rhs.m_isInclusive);
         }

         bool operator != (const FuzzyTimeInfo& rhs) const
         {
            return !(*this == rhs);
         }

         ::std::string toString() const
         {
            ::std::stringstream stream;
            stream << "FuzzyTimeInfo payload" << ::std::endl;
            stream <<  "m_fuzzyTime = " << toString(m_fuzzyTime) << ","
                   <<"m_isInclusive = " << m_isInclusive<< ::std::endl;
            return stream.str();
         }

         FuzzyTime m_fuzzyTime;
         bool m_isInclusive;
      };

      class WeatherInfo
      {
      public:

         WeatherInfo()
            : m_weather(WEATHER__INDEPENDENT)
            , m_isInclusive(false) {}

         enum Weather
         {
            WEATHER__INDEPENDENT,
            WEATHER__SNOW,
            WEATHER__RAIN,
            WEATHER__SUNSHINE,
            WEATHER__FOG,
            WEATHER__THAW,
            WEATHER__STRONG_WIND,
            WEATHER__AVALANCHE,
            WEATHER__ICE
         };

         inline ::std::string toString(const Weather weather) const
         {
            switch(weather)
            {
            case WEATHER__INDEPENDENT:
               return "WEATHER__INDEPENDENT";
            case WEATHER__SNOW:
               return "WEATHER__SNOW";
            case WEATHER__RAIN:
               return "WEATHER__RAIN";
            case WEATHER__SUNSHINE:
               return "WEATHER__SUNSHINE";
            case WEATHER__FOG:
               return "WEATHER__FOG";
            case WEATHER__THAW:
               return "WEATHER__THAW";
            case WEATHER__STRONG_WIND:
               return "WEATHER__STRONG_WIND";
            case WEATHER__AVALANCHE:
               return "WEATHER__AVALANCHE";
            case WEATHER__ICE:
               return "WEATHER__ICE";
            default:
               ::std::stringstream stream;
               stream << "WEATHER__<" << static_cast<unsigned int>(weather) << ">";
               return stream.str();
            }
         }

         bool operator == (const WeatherInfo& rhs) const
         {
            return (m_weather == rhs.m_weather)
                   && (m_isInclusive == rhs.m_isInclusive);
         }

         bool operator != (const WeatherInfo& rhs) const
         {
            return !(*this == rhs);
         }

         ::std::string toString() const
         {
            ::std::stringstream stream;
            stream << "WeatherInfo payload" << ::std::endl;
            stream <<  "m_weather = " << toString(m_weather) << ","
                   <<"m_isInclusive = " << m_isInclusive<< ::std::endl;
            return stream.str();
         }

         Weather m_weather;
         bool m_isInclusive;
      };

      bool operator == (const CountrySpeedLimitInfo& rhs) const
      {
         return (m_speedLimitWithUnit == rhs.m_speedLimitWithUnit)
                && (m_speedLimitRoadType == rhs.m_speedLimitRoadType)
                && (m_timeRangeInfos == rhs.m_timeRangeInfos)
                && (m_fuzzyTimeInfos == rhs.m_fuzzyTimeInfos)
                && (m_weatherInfos == rhs.m_weatherInfos);
      }

      bool operator != (const CountrySpeedLimitInfo& rhs) const
      {
         return !(*this == rhs);
      }

      ::std::string toString() const
      {
         ::std::stringstream stream;
         stream << "CountrySpeedLimitInfo payload" << ::std::endl;
         stream << "m_speedLimitWithUnit = " << m_speedLimitWithUnit<< ::std::endl;
         stream << "m_speedLimit = " << m_speedLimit<< ::std::endl;

         if(m_speedLimitRoadType.isValid())
         {
            stream << "m_speedLimitRoadType = " << toString(m_speedLimitRoadType.getValue())<< ::std::endl;
         }
         else
         {
            stream << "m_speedLimitRoadType is invalid " << ::std::endl;
         }

         stream << "m_timeRangeInfos" << ::std::endl;
         ::navmiddleware::toString<TimeRangeInfo>(m_timeRangeInfos);

         stream << "m_fuzzyTimeInfos" << ::std::endl;
         ::navmiddleware::toString<FuzzyTimeInfo>(m_fuzzyTimeInfos);

         stream << "m_weatherInfos" << ::std::endl;
         ::navmiddleware::toString<WeatherInfo>(m_weatherInfos);

         return stream.str();
      }

      ::std::string m_speedLimitWithUnit;
      unsigned int m_speedLimit;
      ValidValue < SpeedLimitRoadType > m_speedLimitRoadType;
      ::std::vector< TimeRangeInfo > m_timeRangeInfos;
      ::std::vector< FuzzyTimeInfo> m_fuzzyTimeInfos;
      ::std::vector< WeatherInfo> m_weatherInfos;
   };

   bool operator == (const CountryAndStateDetailsInfo& rhs) const
   {
      return (m_countryName == rhs.m_countryName)
             && (m_isoCountryCode == rhs.m_isoCountryCode)
             && (m_subCountryName == rhs.m_subCountryName)
             && (m_isoSubCountryCode == rhs.m_isoSubCountryCode)
             && (m_trafficSense == rhs.m_trafficSense)
             && (m_isMetricSystemUsed == rhs.m_isMetricSystemUsed)
             && (m_supportedIsoLangageCodeList == rhs.m_supportedIsoLangageCodeList)
             && (m_countrySpeedLimitInfos == rhs.m_countrySpeedLimitInfos);
   }

   bool operator != (const CountryAndStateDetailsInfo& rhs) const
   {
      return !(*this == rhs);
   }

   void setCountryName(const ::std::string& countryName)
   {
      m_countryName = countryName;
   }

   const ::std::string& getCountryName() const
   {
      return m_countryName;
   }

   void setIsoCountryCode(const ::std::string& isoCountryCode)
   {
      m_isoCountryCode = isoCountryCode;
   }

   const ::std::string& getIsoCountryCode() const
   {
      return m_isoCountryCode;
   }

   void setSubCountryName(const ::std::string& subCountryName)
   {
      m_subCountryName = subCountryName;
   }

   const ::std::string& getSubCountryName() const
   {
      return m_subCountryName;
   }

   void setIsoSubCountryCode(const ::std::string& isoSubCountryCode)
   {
      m_isoSubCountryCode = isoSubCountryCode;
   }

   const ::std::string& getIsoSubCountryCode() const
   {
      return m_isoSubCountryCode;
   }

   void setCountryFlagImageData( const ValidValue<Image>& countryFlagImageData)
   {
      m_countryFlagImageData = countryFlagImageData;
   }

   const ValidValue<Image>& getCountryFlagImageData() const
   {
      return m_countryFlagImageData;
   }

   void setSubCountryFlagImageData( const ValidValue<Image>& subCountryFlagImageData)
   {
      m_subCountryFlagImageData = subCountryFlagImageData;
   }

   const ValidValue<Image>& getSubCountryFlagImageData() const
   {
      return m_subCountryFlagImageData;
   }

   void setTrafficSense(TrafficSense trafficSense)
   {
      m_trafficSense = trafficSense;
   }

   TrafficSense getTrafficSense() const
   {
      return m_trafficSense;
   }

   void setMetricSystemUsed(bool isMetricSystemUsed)
   {
      m_isMetricSystemUsed = isMetricSystemUsed;
   }

   bool isMetricSystemUsed() const
   {
      return m_isMetricSystemUsed;
   }

   const ::std::vector< ::std::string>& getSupportedIsoLangageCodeList() const
   {
      return m_supportedIsoLangageCodeList;
   }

   void setSupportedIsoLangageCodeList(const ::std::vector< ::std::string>& supportedIsoLangageCodeList)
   {
      m_supportedIsoLangageCodeList = supportedIsoLangageCodeList;
   }

   const ::std::vector< CountrySpeedLimitInfo >& getCountrySpeedLimitInfos() const
   {
      return m_countrySpeedLimitInfos;
   }

   void setCountrySpeedLimitInfos(const ::std::vector< CountrySpeedLimitInfo >& countrySpeedLimitInfos)
   {
      m_countrySpeedLimitInfos = countrySpeedLimitInfos;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("CountryAndStateDetailInfos payload:\n");

      stream << "m_countryName: " << m_countryName << ::std::endl;
      stream << "m_isoCountryCode: " << m_isoCountryCode << ::std::endl;
      stream << "m_subCountryName: " << m_subCountryName << ::std::endl;
      stream << "m_isoSubCountryCode: " << m_isoSubCountryCode << ::std::endl;

      if(m_countryFlagImageData.isValid())
      {
         stream << "m_countryFlagImageData blob size: " << m_countryFlagImageData.getValue().getBlobSize()<< ::std::endl;
      }
      else
      {
         stream << "m_countryFlagImageDatais invalid " << ::std::endl;
      }
      if(m_subCountryFlagImageData.isValid())
      {
         stream << "m_subCountryFlagImageData blob size: " << m_subCountryFlagImageData.getValue().getBlobSize()<< ::std::endl;
      }
      else
      {
         stream << "m_subCountryFlagImageData is invalid " << ::std::endl;
      }

      stream << "m_trafficSense: " << ::navmiddleware::toString(m_trafficSense) << ::std::endl;
      stream << "m_isMetricSystemUsed: " << (m_isMetricSystemUsed? "true": "false") << ::std::endl;

      stream << "m_supportedIsoLangageCodeList" << ::std::endl;
      ::std::vector< ::std::string>::const_iterator languageCodeListItr = m_supportedIsoLangageCodeList.begin();
      for (; languageCodeListItr !=  m_supportedIsoLangageCodeList.end(); ++languageCodeListItr)
      {
         stream  << languageCodeListItr->c_str();
      }

      stream << "m_countrySpeedLimitInfos size " << m_countrySpeedLimitInfos.size() << ::std::endl;
      ::navmiddleware::toString<CountrySpeedLimitInfo>(m_countrySpeedLimitInfos);

      return stream.str();
   }

private:

   ::std::string m_countryName;
   ::std::string m_isoCountryCode;
   ::std::string m_subCountryName;
   ::std::string m_isoSubCountryCode;
   ValidValue<Image> m_countryFlagImageData;
   ValidValue<Image> m_subCountryFlagImageData;
   TrafficSense m_trafficSense;
   // set to TRUE, if country uses metric system. Otherwise,the country uses imperial system.
   bool m_isMetricSystemUsed;

   // List of languages supported in the country. First entry in the list is the default official language
   ::std::vector< ::std::string> m_supportedIsoLangageCodeList;

   // Provides details about speed limits in the country
   ::std::vector< CountrySpeedLimitInfo > m_countrySpeedLimitInfos;
};

enum DriveMode
{
   DRIVEMODE__NORMAL = 0,
   DRIVEMODE__ECO,
   DRIVEMODE__SPORT,
   DRIVEMODE__RACE
};

inline ::std::string toString(DriveMode driveMode)
{
   switch (driveMode)
   {
   case DRIVEMODE__NORMAL:
      return "DRIVEMODE__NORMAL";
   case DRIVEMODE__ECO:
      return "DRIVEMODE__ECO";
   case DRIVEMODE__SPORT:
      return "DRIVEMODE__SPORT";
   case DRIVEMODE__RACE:
      return "DRIVEMODE__RACE";
   default:
      ::std::stringstream stream;
      stream << "DRIVEMODE__<" << static_cast<unsigned int>(driveMode) << ">";
      return stream.str();
   }
}

class SdsGlobalUpdateRegionInfo
{
public:
   SdsGlobalUpdateRegionInfo()
      : m_updateRegionId(0)
      , m_productId(0)
      , m_supplierId(0)
   {
   }

   void setUpdateRegionId(uint16_t updateRegionId)
   {
      m_updateRegionId = updateRegionId;
   }

   uint16_t getUpdateRegionId() const
   {
      return m_updateRegionId;
   }

   void setProductId(uint32_t productId)
   {
      m_productId = productId;
   }

   uint32_t getProductId() const
   {
      return m_productId;
   }

   void setSupplierId(uint8_t supplierId)
   {
      m_supplierId = supplierId;
   }

   uint8_t getSupplierId() const
   {
      return m_supplierId;
   }

   bool operator==(const SdsGlobalUpdateRegionInfo& rhs) const
   {
      return ((this->m_updateRegionId == rhs.m_updateRegionId)
              && (this->m_productId == rhs.m_productId)
              && (this->m_supplierId == rhs.m_supplierId));
   }

   bool operator!=(const SdsGlobalUpdateRegionInfo& other) const
   {
      return !(*this == other);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SdsGlobalUpdateRegionInfo payload:" << ::std::endl
             << "   UpdateRegionId = " << m_updateRegionId << ::std::endl
             << "   ProductId = " << m_productId << ::std::endl
             << "   SupplierId = " << static_cast<unsigned int>(m_supplierId) << ::std::endl;

      return stream.str();
   }

private:
   uint16_t m_updateRegionId;
   uint32_t m_productId;
   uint8_t  m_supplierId;
};

class SdsGlobalNamedObjectIdInfo
{
public:
   SdsGlobalNamedObjectIdInfo()
      : m_namedObjectId(0)
   {
   }

   void setSdsGlobalUpdateRegionInfo(const SdsGlobalUpdateRegionInfo& sdsGlobalUpdateRegionInfo)
   {
      m_sdsGlobalUpdateRegionInfo = sdsGlobalUpdateRegionInfo;
   }

   const SdsGlobalUpdateRegionInfo& getSdsGlobalUpdateRegionInfo() const
   {
      return m_sdsGlobalUpdateRegionInfo;
   }

   void setNamedObjectId(uint32_t namedObjectId)
   {
      m_namedObjectId = namedObjectId;
   }

   uint32_t getNamedObjectId() const
   {
      return m_namedObjectId;
   }

   bool operator==(const SdsGlobalNamedObjectIdInfo& rhs) const
   {
      return (this->m_sdsGlobalUpdateRegionInfo == rhs.m_sdsGlobalUpdateRegionInfo)
             && (this->m_namedObjectId == rhs.m_namedObjectId);
   }

   bool operator!=(const SdsGlobalNamedObjectIdInfo& other) const
   {
      return !(*this == other);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "SdsGlobalNamedObjectIdInfo payload:" << ::std::endl
             << "   SdsGlobalUpdateRegionInfo = " << m_sdsGlobalUpdateRegionInfo.toString() << ::std::endl
             << "   NamedObjectId = " << m_namedObjectId << ::std::endl;

      return stream.str();
   }

private:
   SdsGlobalUpdateRegionInfo  m_sdsGlobalUpdateRegionInfo;
   uint32_t                   m_namedObjectId;
};

enum RoadClass
{
   ROAD_CLASS__NONE,
   ROAD_CLASS__MOTORWAY,
   ROAD_CLASS__FREEWAY,
   ROAD_CLASS__HIGHWAY,
   ROAD_CLASS__ARTERIAL,
   ROAD_CLASS__COLLECTOR,
   ROAD_CLASS__LOCAL,
   ROAD_CLASS__ADRESSABLESTREET,
   ROAD_CLASS__TRAIL
};
inline ::std::string toString(RoadClass roadClass)
{
   switch (roadClass)
   {
   case ROAD_CLASS__NONE:
      return "ROAD_CLASS__NONE";
   case ROAD_CLASS__MOTORWAY:
      return "ROAD_CLASS__MOTORWAY";
   case ROAD_CLASS__FREEWAY:
      return "ROAD_CLASS__FREEWAY";
   case ROAD_CLASS__HIGHWAY:
      return "ROAD_CLASS__HIGHWAY";
   case ROAD_CLASS__ARTERIAL:
      return "ROAD_CLASS__ARTERIAL";
   case ROAD_CLASS__COLLECTOR:
      return "ROAD_CLASS__COLLECTOR";
   case ROAD_CLASS__LOCAL:
      return "ROAD_CLASS__LOCAL";
   case ROAD_CLASS__ADRESSABLESTREET:
      return "ROAD_CLASS__ADRESSABLESTREET";
   case ROAD_CLASS__TRAIL:
      return "ROAD_CLASS__TRAIL";
   default:
      ::std::stringstream stream;
      stream << "ROAD_CLASS__" << static_cast<unsigned int>(roadClass) << ">";
      return stream.str();
   }
}

//<-- INF4CV
struct ResidentialOrLocalMode
{
   ResidentialOrLocalMode()
      :m_isActive(false)
   {
   }

   explicit ResidentialOrLocalMode(const ::std::vector< ::std::string >& blackListCountries)
   {
      m_isActive = true;
      m_blackListCountries = blackListCountries;
   }

   bool operator==(const ResidentialOrLocalMode& rhs) const
   {
      return ((m_isActive == rhs.m_isActive) && (m_blackListCountries == rhs.m_blackListCountries));
   }

   bool operator!=(const ResidentialOrLocalMode& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("ResidentialOrLocalMode payload: ");
      stream << "m_isActive = " << m_isActive << ::std::endl;
      stream << "Total black list countries = " << m_blackListCountries.size() << ::std::endl;
      for(unsigned int listIndex = 0; listIndex < m_blackListCountries.size(); ++listIndex)
      {
         stream << "  BlackListCountry[" << listIndex << "] = " << m_blackListCountries[listIndex] << ::std::endl;
      }
      return stream.str();
   }

   ::std::vector< ::std::string > m_blackListCountries;
   bool                           m_isActive;
};
//->

} // namespace navmiddleware

#endif  // PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_INFOTYPES_H_
