/**
* @copyright (c) 2017-2020 Robert Bosch Car Multimedia GmbH
* @addtogroup NavMiddleware
* @{
*/

#ifndef PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_ELECTRICVEHICLEINFOS_H_
#define PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_ELECTRICVEHICLEINFOS_H_

#include <sstream>
#include <string>
#include <vector>
#include "InfoTypes.h"

namespace navmiddleware
{

class BatteryInfos
{
public:

   BatteryInfos():
      m_currentStateOfChargeDisplay(0),
      m_currentStateOfChargeMeter(0),
      m_catalogBatteryCharge(0),
      m_fullCharge(0) {}

   BatteryInfos(const BatteryInfos& batteryInfos):
      m_currentStateOfChargeDisplay(batteryInfos.m_currentStateOfChargeDisplay),
      m_currentStateOfChargeMeter(batteryInfos.m_currentStateOfChargeMeter),
      m_catalogBatteryCharge(batteryInfos.m_catalogBatteryCharge),
      m_fullCharge(batteryInfos.m_fullCharge) {}

   bool operator == (const BatteryInfos& rhs) const
   {
      return((m_currentStateOfChargeDisplay == rhs.m_currentStateOfChargeDisplay)&&
             (m_currentStateOfChargeMeter == rhs.m_currentStateOfChargeMeter)&&
             (m_catalogBatteryCharge == rhs.m_catalogBatteryCharge)&&
             (m_fullCharge == rhs.m_fullCharge));
   }

   BatteryInfos& operator = (const BatteryInfos& rhs)
   {
      if (&rhs != this)
      {
         m_currentStateOfChargeDisplay = rhs.m_currentStateOfChargeDisplay;
         m_currentStateOfChargeMeter = rhs.m_currentStateOfChargeMeter;
         m_catalogBatteryCharge = rhs.m_catalogBatteryCharge;
         m_fullCharge = rhs.m_fullCharge;
      }
      return *this;
   }

   bool operator != (const BatteryInfos& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("BatteryInfos payload:\n");

      stream << ::std::endl
             << "Current state of charge display =" << m_currentStateOfChargeDisplay  << ::std::endl
             << "Current state of charge meter =" << m_currentStateOfChargeMeter << ::std::endl
             << "Catalog battery charge =" << m_catalogBatteryCharge << ::std::endl
             << "Full charge = " << m_fullCharge << ::std::endl;

      return stream.str();
   }

   // Current state of charge display percentage
   unsigned int m_currentStateOfChargeDisplay;
   // Relative current state of charge in vehicle meter display
   unsigned int m_currentStateOfChargeMeter;
   // Catalog Battery charge in Ws
   unsigned int m_catalogBatteryCharge;
   // maximum battery charge in Ws
   unsigned int m_fullCharge;

};

class ChargingInfos
{
public:

   ChargingInfos():
      m_chargingType(CHARGING_TYPE__UNKNOWN) {}

   bool operator == (const ChargingInfos& rhs) const
   {
      return(m_chargingType == rhs.m_chargingType);
   }

   ChargingInfos& operator = (const ChargingInfos& rhs)
   {
      if (&rhs != this)
      {
         m_chargingType = rhs.m_chargingType;
      }
      return *this;
   }

   bool operator != (const ChargingInfos& rhs) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("ChargingInfos payload:\n");

      stream << ::std::endl
             << "Type of charging =" << ::navmiddleware::toString(m_chargingType) << ::std::endl;

      return stream.str();
   }

   // Type of Charging mode
   ChargingType m_chargingType;
};

class ElectricVehicleInfos
{
public:

   ElectricVehicleInfos()
      : m_driveMode(DRIVEMODE__NORMAL)
   {
   }

   bool operator==(const ElectricVehicleInfos& rhs) const
   {
      return (m_batteryInfos == rhs.m_batteryInfos)&&
             (m_driveMode == rhs.m_driveMode);
   }

   bool operator!=(const ElectricVehicleInfos& rhs) const
   {
      return !(*this == rhs);
   }

   const BatteryInfos& getBatteryInfos() const
   {
      return m_batteryInfos;
   }

   void setBatteryInfos(const BatteryInfos& batteryInfos)
   {
      m_batteryInfos = batteryInfos;
   }

   const ChargingInfos& getChargingInfos() const
   {
      return m_chargingInfos;
   }

   void setChargingInfos(const ChargingInfos& chargingInfos)
   {
      m_chargingInfos = chargingInfos;
   }

   DriveMode getDriveMode() const
   {
      return m_driveMode;
   }

   void setDriveMode(DriveMode driveMode)
   {
      m_driveMode = driveMode;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("ElectricVehicleInfos payload:\n");
      stream << ::std::endl
             << "Battery infos = " << m_batteryInfos.toString() << ::std::endl
             << "Charging infos = "<< m_chargingInfos.toString() << ::std::endl
             << "Drive mode = " << ::navmiddleware::toString(m_driveMode) << ::std::endl;
      return stream.str();
   }

private:

   BatteryInfos      m_batteryInfos;
   ChargingInfos     m_chargingInfos;
   DriveMode         m_driveMode;

};

} //namespace navmiddleware

#endif  // PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_ELECTRICVEHICLEINFOS_H_
