/**
* @copyright (c) 2015-2020 Robert Bosch Car Multimedia GmbH
* @addtogroup NavMiddleware
* @{
*/

#ifndef PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_DESTINATIONMEMORYINFOS_H_
#define PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_DESTINATIONMEMORYINFOS_H_

#include <stdint.h>
#include <string>
#include <sstream>
#include <vector>
#include "InfoTypes.h"
#include "RouteSettings.h"

namespace navmiddleware
{

class DepartureInfo
{
public:
   DepartureInfo(const ::std::string& name, double longitude, double latitude)
      : m_name(name)
      , m_longitude(longitude)
      , m_latitude(latitude)
   {}

   DepartureInfo()
      : m_name()
      , m_longitude(0)
      , m_latitude(0)
   {}

   void setName(const ::std::string name)
   {
      m_name = name;
   }

   void setLatitude(double latitude)
   {
      m_latitude = latitude;
   }

   void setLongitude(double longitude)
   {
      m_longitude = longitude;
   }

   ::std::string getName() const
   {
      return m_name;
   }

   double getLongitude() const
   {
      return m_longitude;
   }

   double getLatitude() const
   {
      return m_latitude;
   }

   bool operator==(const DepartureInfo& rhs) const
   {
      return (m_name == rhs.m_name) &&
        (m_longitude == rhs.m_longitude) &&
        (m_latitude == rhs.m_latitude);
   }

   bool operator!=(const DepartureInfo& rhs) const
   {
      return !(rhs == *this);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DepartureInfo:");
      stream << ::std::endl
        << "Name = " << m_name << ::std::endl
        << "Longitude = " << m_longitude << ::std::endl
        << "Latitude = " << m_latitude << ::std::endl;
      return stream.str();
   }

private:
   ::std::string  m_name;
   double         m_longitude;
   double         m_latitude;
};

class DestinationMemoryDestination
{
public:
   DestinationMemoryDestination()
      : m_longitude(0)
      , m_latitude(0)
      , m_locationOrigin(LOCATION_ORIGIN__UNKNOWN)
      , m_entrySelectionStatus(ENTRY_SELECTION_STATUS__UNKNOWN)
   {}

   DestinationMemoryDestination(const ::std::string& name, double longitude, double latitude,
                                const LocationAttributeInfos& attributes, LocationOrigin locationOrigin,
                                const std::string& uniquePlaceId, EntrySelectionStatus entrySelectionStatus,
                                const LanguageEntryInfo& languageEntryInfo)
      : m_name(name)
      , m_longitude(longitude)
      , m_latitude(latitude)
      , m_attributes(attributes)
      , m_languageInfo(languageEntryInfo)
      , m_locationOrigin(locationOrigin)
      , m_uniquePlaceId(uniquePlaceId)
      , m_entrySelectionStatus(entrySelectionStatus)
   {}

   bool operator==(const DestinationMemoryDestination& rhs) const
   {
      return m_name == rhs.m_name &&
             m_longitude == rhs.m_longitude &&
             m_latitude == rhs.m_latitude &&
             m_attributes == rhs.m_attributes &&
             m_languageInfo == rhs.m_languageInfo &&
             m_locationOrigin == rhs.m_locationOrigin &&
             m_uniquePlaceId == rhs.m_uniquePlaceId &&
             m_entrySelectionStatus == rhs.m_entrySelectionStatus;
   }

   bool operator!=(const DestinationMemoryDestination& rhs) const
   {
      return !(*this == rhs);
   }

   void setName(const ::std::string& name)
   {
      m_name = name;
   }

   ::std::string getName() const
   {
      return m_name;
   }

   void setLongitude(double longitude)
   {
      m_longitude = longitude;
   }

   double getLongitude() const
   {
      return m_longitude;
   }

   void setLatitude(double latitude)
   {
      m_latitude = latitude;
   }

   double getLatitude() const
   {
      return m_latitude;
   }

   void setAttributes(const LocationAttributeInfos& attributes)
   {
      m_attributes = attributes;
   }

   const LocationAttributeInfos& getAttributes() const
   {
      return m_attributes;
   }

   void setLanguageInfo(const LanguageEntryInfo& language)
   {
      m_languageInfo = language;
   }

   const LanguageEntryInfo& getLanguageInfo() const
   {
      return m_languageInfo;
   }

   void setLocationOrigin(LocationOrigin locationOrigin)
   {
      m_locationOrigin = locationOrigin;
   }

   LocationOrigin getLocationOrigin() const
   {
      return m_locationOrigin;
   }

   void setPlaceId(const ::std::string& placeId)
   {
      m_uniquePlaceId = placeId;
   }

   const std::string& getPlaceId() const
   {
      return m_uniquePlaceId;
   }

   void setDestinationRefreshStatus(EntrySelectionStatus entrySelectionStatus)
   {
      m_entrySelectionStatus = entrySelectionStatus;
   }

   EntrySelectionStatus getDestinationRefreshStatus() const
   {
      return m_entrySelectionStatus;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DestinationMemoryDestination payload:\n");
      stream << ::std::endl
             << "Destination Name = " << m_name << ::std::endl
             << "Destination Longitude = " << m_longitude << ::std::endl
             << "Destination Latitude = " << m_latitude << ::std::endl
             << m_attributes.toString() << ::std::endl
             << "Language info = " << m_languageInfo.toString() << ::std::endl
             << "LocationOrigin = " << ::navmiddleware::toString(m_locationOrigin) << ::std::endl
             << "UniquePlaceId = " << m_uniquePlaceId<< ::std::endl
             << "EntrySelectionStatus = " << ::navmiddleware::toString(m_entrySelectionStatus) << ::std::endl;
      return stream.str();
   }

private:
   ::std::string                       m_name;
   double                              m_longitude;
   double                              m_latitude;
   LocationAttributeInfos              m_attributes;
   LanguageEntryInfo                   m_languageInfo;
   LocationOrigin                      m_locationOrigin;
   std::string                         m_uniquePlaceId;
   EntrySelectionStatus                m_entrySelectionStatus;
};

class HistoryItem
{
public:
   HistoryItem()
      : m_creationTimeStamp(0)
      , m_lastGuidedTimeStamp(0)
   {}
   HistoryItem(int64_t creationTimeStamp, int64_t lastGuidedTimeStamp)
      : m_creationTimeStamp(creationTimeStamp)
      , m_lastGuidedTimeStamp(lastGuidedTimeStamp)
   {}

   virtual ~HistoryItem() {}

   virtual bool operator==(const HistoryItem& rhs) const
   {
      return m_creationTimeStamp == rhs.m_creationTimeStamp &&
             m_lastGuidedTimeStamp == rhs.m_lastGuidedTimeStamp;
   }

   virtual bool operator!=(const HistoryItem& rhs) const
   {
      return !(*this == rhs);
   }

   void setCreationTimeStamp(int64_t creationTimeStamp)
   {
      m_creationTimeStamp = creationTimeStamp;
   }

   int64_t getCreationTimeStamp() const
   {
      return m_creationTimeStamp;
   }

   void setLastGuidedTimeStamp(int64_t lastGuidedTimeStamp)
   {
      m_lastGuidedTimeStamp = lastGuidedTimeStamp;
   }

   int64_t getLastGuidedTimeStamp() const
   {
      return m_lastGuidedTimeStamp;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "Entry Creation Time = " << m_creationTimeStamp << ::std::endl;
      stream << "Entry Last Guided Time = " << m_lastGuidedTimeStamp << ::std::endl;
      return stream.str();
   }

private:
   int64_t m_creationTimeStamp; // milliseconds epoch time
   int64_t m_lastGuidedTimeStamp; // milliseconds epoch time
};

class DestinationMemoryLastDestination : public HistoryItem
{
public:
   DestinationMemoryLastDestination()
      : HistoryItem(0, 0)
   {}

   DestinationMemoryLastDestination(const DestinationMemoryDestination& destinationMemoryDestination, int64_t creationTimeStamp, int64_t lastGuidedTimeStamp)
      : HistoryItem(creationTimeStamp, lastGuidedTimeStamp)
      , m_destinationMemoryDestination(destinationMemoryDestination)
   {}

   bool operator==(const DestinationMemoryLastDestination& rhs) const
   {
      return HistoryItem::operator==(rhs) &&
             m_destinationMemoryDestination == rhs.m_destinationMemoryDestination;
   }

   bool operator!=(const DestinationMemoryLastDestination& rhs) const
   {
      return !(*this == rhs);
   }

   void setDestination(const DestinationMemoryDestination& destinationMemoryDestination)
   {
      m_destinationMemoryDestination = destinationMemoryDestination;
   }

   const DestinationMemoryDestination& getDestination() const
   {
      return m_destinationMemoryDestination;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DestinationMemoryDestination payload:\n");
      stream << ::std::endl
             << "Last Destination = " << m_destinationMemoryDestination.toString() << ::std::endl
             << HistoryItem::toString() << ::std::endl
             << ::std::endl;
      return stream.str();
   }

private:
   DestinationMemoryDestination m_destinationMemoryDestination;
};

typedef ::std::vector<DestinationMemoryAttributesGroup> DestinationMemoryAttributesGroupList;

class DestinationMemoryEntryAttributes
{
public:

   enum Sound
   {
      SOUND__NONE,
      SOUND__CHIMENON,
      SOUND__MELODY,
      SOUND__SOUND,
      SOUND__DOG,
      SOUND__CAT,
      SOUND__WAVE,
      SOUND__CHURCH_BELL,
      SOUND__MUSIC_BOX1,
      SOUND__MUSIC_BOX2,
      SOUND__MARIMBA,
      SOUND__TRUMPET,
      SOUND__HORN,
      SOUND__SCRATCH,
      SOUND__SIZEOF
   };

   enum Distance
   {
      DISTANCE__FIFTY_METERS = 50,
      DISTANCE__HUNDRED_METERS = 100,
      DISTANCE__THREE_HUNDRED_METERS = 300,
      DISTANCE__FIVE_HUNDRED_METERS = 500,
      DISTANCE__SIZEOF = 4
   };

   DestinationMemoryEntryAttributes()
      : m_sound(SOUND__NONE),
        m_distance(DISTANCE__THREE_HUNDRED_METERS),
        m_showOnMap(false)
   {}

   void setName(const ::std::string& name)
   {
      m_name = name;
   }

   const ::std::string& getName() const
   {
      return m_name;
   }

   void setVoiceTag(uint64_t voiceTag)
   {
      m_voiceTag.setValue(voiceTag);
   }

   const ::navmiddleware::ValidValue<uint64_t>& getVoiceTag() const
   {
      return m_voiceTag;
   }

   void invalidateVoiceTag()
   {
      m_voiceTag.invalidate();
   }

   void setDistance(Distance distance)
   {
      m_distance = distance;
   }

   Distance getDistance() const
   {
      return m_distance;
   }

   void setSound(Sound sound)
   {
      m_sound = sound;
   }

   Sound getSound() const
   {
      return m_sound;
   }

   // Set rounded integer degree in the range of [0, 360) clockwise
   void setDirection(uint16_t direction)
   {
      m_direction.setValue(direction);
   }

   const ::navmiddleware::ValidValue<uint16_t>& getDirection() const
   {
      return m_direction;
   }

   void invalidateDirection()
   {
      m_direction.invalidate();
   }

   void setGroupList(const DestinationMemoryAttributesGroupList & groupList)
   {
      m_groupList = groupList;
   }

   const DestinationMemoryAttributesGroupList& getGroupList() const
   {
      return m_groupList;
   }

   void setIconId(uint64_t iconId)
   {
      m_iconId.setValue(iconId);
   }

   const ::navmiddleware::ValidValue<uint64_t>& getIconId() const
   {
      return m_iconId;
   }

   void invalidateIconId()
   {
      m_iconId.invalidate();
   }

   void setUserPoiIconData(const ValidValue<Image>& userPoiIcon)
   {
      m_userPoiIconData = userPoiIcon;
   }

   const ValidValue<Image>& getUserPoiIconData() const
   {
      return m_userPoiIconData;
   }

   void setShowOnMap(bool showOnMap)
   {
      m_showOnMap = showOnMap;
   }

   bool isShownOnMap() const
   {
      return m_showOnMap;
   }

   void setPhoneNumber(const ::std::vector< ::std::string >& phoneNumberList)
   {
      m_phoneNumberList = phoneNumberList;
   }

   const ::std::vector< ::std::string >& getPhoneNumber() const
   {
      return m_phoneNumberList;
   }

   void setChargingType(ChargingType chargingType)
   {
      m_chargingType.setValue(chargingType);
   }

   const ::navmiddleware::ValidValue<ChargingType>& getChargingType() const
   {
      return m_chargingType;
   }

   void invalidateChargingType()
   {
      m_chargingType.invalidate();
   }

   void setPhoneme(const ::std::string& phoneme)
   {
      m_phoneme.setValue(phoneme);
   }

   const ::navmiddleware::ValidValue< ::std::string >& getPhoneme() const
   {
      return m_phoneme;
   }

   void invalidatePhoneme()
   {
      m_phoneme.invalidate();
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("AddressBookAttributes payload:\n");

      stream << ::std::endl
             << "AddressBookAttribute Name = " << m_name << ::std::endl
             << "AddressBookAttribute No. of Phone Numbers = " << m_phoneNumberList.size() << ::std::endl
             << "AddressBookAttribute SoundSelection = " << m_sound << ::std::endl
             << "AddressBookAttribute Distance = " << m_distance << ::std::endl
             << "AddressBookAttribute VoiceTag = " << m_voiceTag.toString() << ::std::endl
             << "AddressBookAttribute Direction = " << m_direction.toString() << ::std::endl
             << "AddressBookAttribute Icon Id = " << m_iconId.toString() << ::std::endl
             << "AddressBookAttribute User POI Icon Image size : "
             << m_userPoiIconData.getValueOr(Image()).toString() << ::std::endl
             << "AddressBookAttribute Show On Map = " << m_showOnMap << ::std::endl
             << "AddressBookAttribute No. of Group Selection = " << m_groupList.size() << ::std::endl
             << "AddressBookAttribute ChargingType = " << m_chargingType.toString() << ::std::endl
             << "AddressBookAttribute Phoneme = " << m_phoneme.getValueOr("(not set)") << ::std::endl
             << ::std::endl;

      for (size_t ndx = 0; ndx < m_groupList.size(); ndx++)
      {
         stream << "AddressBookAttribute Group [" << ndx << "] = " << ::navmiddleware::toString(m_groupList[ndx]) << ::std::endl;
      }

      return stream.str();
   }

private:
   ::std::string                                   m_name;
   ValidValue<uint64_t>                            m_voiceTag;
   Sound                                           m_sound;
   ValidValue<uint64_t>                            m_iconId;
   ValidValue<Image>                               m_userPoiIconData;
   ValidValue<uint16_t>                            m_direction; //Set rounded integer degree in [0, 360)
   Distance                                        m_distance;
   DestinationMemoryAttributesGroupList            m_groupList;
   std::vector< ::std::string >                    m_phoneNumberList;
   bool                                            m_showOnMap;
   ValidValue<ChargingType>                        m_chargingType;
   ValidValue < ::std::string >                    m_phoneme; // [JP] phonetic representation of the entries name, JP: full width Katakana
};

class DestinationMemoryEntry
{
public:
   typedef uint64_t Id;

   enum SortingAlgorithm
   {
      SORTINGALGORITHM_UNSORTED,
      SORTINGALGORITHM_ALPHABETICAL_BY_NAME_ASC,
      SORTINGALGORITHM_ALPHABETICAL_BY_NAME_DESC,
      SORTINGALGORITHM_ALPHABETICAL_BY_ALIAS_ASC,
      SORTINGALGORITHM_ALPHABETICAL_BY_ALIAS_DESC,
      SORTINGALGORITHM_BY_START_GUIDANCE_TIME_NEWEST_FIRST,
      SORTINGALGORITHM_BY_START_GUIDANCE_TIME_OLDEST_FIRST,
      SORTINGALGORITHM_BY_ENTRY_POSITION_ASC,
      SORTINGALGORITHM_BY_ENTRY_POSITION_DESC,
      SORTINGALGORITHM_BY_ICON_AND_ALPHABETICAL_BY_NAME_ASC,
      SORTINGALGORITHM_BY_ICON_AND_ALPHABETICAL_BY_ALIAS_ASC,
      SORTINGALGORITHM_BY_GROUP_AND_ALPHABETICAL_BY_NAME_ASC,
      SORTINGALGORITHM_BY_GROUP_AND_ALPHABETICAL_BY_ALIAS_ASC,
      SORTINGALGORITHM_BY_CREATION_TIME_NEWEST_FIRST,
      SORTINGALGORITHM_BY_CREATION_TIME_OLDEST_FIRST
   };

public:
   DestinationMemoryEntry()
      : m_id(0)
      , m_position(0)
   {}

   DestinationMemoryEntry(Id id, const DestinationMemoryDestination& destination, uint64_t position, ValidValue< DestinationMemoryEntryAttributes > attributes)
      : m_id(id)
      , m_destination(destination)
      , m_position(position)
      , m_attributes(attributes)
   {}

   void setId(Id id)
   {
      m_id = id;
   }

   Id getId() const
   {
      return m_id;
   }

   void setDestination(const DestinationMemoryDestination& destination)
   {
      m_destination = destination;
   }

   const DestinationMemoryDestination& getDestination() const
   {
      return m_destination;
   }

   void setPosition(uint64_t position)
   {
      m_position = position;
   }

   uint64_t getPosition() const
   {
      return m_position;
   }

   void setAttributes(const DestinationMemoryEntryAttributes& attributes)
   {
      m_attributes.setValue(attributes);
   }

   const ValidValue< DestinationMemoryEntryAttributes >& getAttributes() const
   {
      return m_attributes;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DestinationMemoryEntry payload:\n");
      stream << ::std::endl
             << "Entry ID = " << m_id << ::std::endl
             << "Entry Destination = " << m_destination.toString() << ::std::endl
             << "Entry Position = " << m_position << ::std::endl
             << "Entry Attributes = " << m_attributes.getValue().toString() << ::std::endl
             << ::std::endl;
      return stream.str();
   }

private:
   Id                                             m_id;
   DestinationMemoryDestination                   m_destination;
   uint64_t                                       m_position;
   ValidValue< DestinationMemoryEntryAttributes > m_attributes;
};

class DestinationMemoryEntryList
{
public:
   DestinationMemoryEntryList()
      : m_entryListCount(0)
   {}
   void addEntry(const DestinationMemoryEntry& entry)
   {
      m_entryList.push_back(entry);
   }

   const ::std::vector<DestinationMemoryEntry>& getEntryList() const
   {
      return m_entryList;
   }

   void clearEntryList()
   {
      m_entryList.clear();
   }

   void setEntryListCount(uint32_t entryListCount)
   {
      m_entryListCount = entryListCount;
   }

   uint32_t getEntryListCount() const
   {
      return m_entryListCount;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DestinationMemoryEntryList payload:\n\n");
      for (unsigned int entryListIndex = 0; entryListIndex < m_entryList.size(); entryListIndex++)
      {
         stream << "Entry list Index = " << entryListIndex << ::std::endl
                << "Entry = " << m_entryList[entryListIndex].toString() << ::std::endl;
      }
      stream << "Entry list count = " << m_entryListCount << std::endl;
      return stream.str();
   }

private:
   ::std::vector<DestinationMemoryEntry> m_entryList;
   /* m_entryListCount provides total number of entries stored */
   uint32_t m_entryListCount;
};

class DestinationMemoryLastDestinationHistory : public HistoryItem
{
public:
   DestinationMemoryLastDestinationHistory()
      : HistoryItem(0,0)
      , m_entryId(0)
   {}

   DestinationMemoryLastDestinationHistory(const DestinationMemoryEntry::Id& entryId,
                                           int64_t creationTimeStamp, int64_t lastGuidedTimeStamp)
      : HistoryItem(creationTimeStamp, lastGuidedTimeStamp)
      , m_entryId(entryId)
   {}

   bool operator==(const DestinationMemoryLastDestinationHistory& rhs) const
   {
      return HistoryItem::operator==(rhs) &&
             m_entryId == rhs.m_entryId;
   }

   bool operator!=(const DestinationMemoryLastDestinationHistory& rhs) const
   {
      return !(*this == rhs);
   }

   void setEntryId(const DestinationMemoryEntry::Id& entryId)
   {
      m_entryId = entryId;
   }

   const DestinationMemoryEntry::Id& getEntryId() const
   {
      return m_entryId;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DestinationMemoryLastDestinationHistory payload:\n\n");
      stream << "EntryId = " << m_entryId<< ::std::endl;
      stream << HistoryItem::toString() << ::std::endl;
      return stream.str();
   }

private:
   DestinationMemoryEntry::Id m_entryId;
};

class DestinationMemoryLastDestinationHistoryList
{
public:
   void setLastDestinationHistoryList(const ::std::vector<DestinationMemoryLastDestinationHistory>&  lastDestinationHistoryList)
   {
      m_lastDestinationHistoryList = lastDestinationHistoryList;
   }

   const ::std::vector<DestinationMemoryLastDestinationHistory>& getLastDestinationHistoryList() const
   {
      return m_lastDestinationHistoryList;
   }

   void clearLastDestinationHistoryList()
   {
      m_lastDestinationHistoryList.clear();
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DestinationMemoryLastDestinationHistoryList payload:\n\n");
      for (size_t lastDestinationHistoryListIndex = 0; lastDestinationHistoryListIndex < m_lastDestinationHistoryList.size(); lastDestinationHistoryListIndex++)
      {
         stream << "LastDestinationHistory list Index = " << lastDestinationHistoryListIndex << ::std::endl
                << "LastDestinationHistory = " << m_lastDestinationHistoryList[lastDestinationHistoryListIndex].toString() << ::std::endl;
      }
      return stream.str();
   }

private:
   ::std::vector<DestinationMemoryLastDestinationHistory> m_lastDestinationHistoryList;
};

class DestinationMemoryUpdatedLastDestinationList
{
public:
   void setUpdatedLastDestinationList(const ::std::vector< DestinationMemoryEntry::Id >& updatedLastDestinationList)
   {
      m_updatedLastDestinationList = updatedLastDestinationList;
   }

   const ::std::vector< DestinationMemoryEntry::Id >& getUpdatedLastDestinationList() const
   {
      return m_updatedLastDestinationList;
   }

   void clearUpdatedLastDestinationList()
   {
      m_updatedLastDestinationList.clear();
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DestinationMemoryUpdatedLastDestinationList payload:\n");
      stream << ::std::endl;
      for (size_t ndx = 0; ndx < m_updatedLastDestinationList.size(); ndx++)
      {
         stream << "Id list Index = " << ndx << ::std::endl
                << "Id  = " << m_updatedLastDestinationList[ndx] << ::std::endl;
      }

      return stream.str();
   }
private:
   ::std::vector< DestinationMemoryEntry::Id > m_updatedLastDestinationList;
};

class DestinationMemoryWaypoint
{
public:
   DestinationMemoryWaypoint()
      : m_routeCriterion(::navmiddleware::settings::ROUTE_CRITERION_FAST)
   {}

   DestinationMemoryWaypoint(const DestinationMemoryDestination& destination, ::navmiddleware::settings::RouteCriterion routeCriterion)
      : m_destination(destination)
      , m_routeCriterion(routeCriterion)
   {}

   void setDestination(const DestinationMemoryDestination& destination)
   {
      m_destination = destination;
   }

   const DestinationMemoryDestination& getDestination() const
   {
      return m_destination;
   }

   void setRouteCriterion(::navmiddleware::settings::RouteCriterion routeCriterion)
   {
      m_routeCriterion = routeCriterion;
   }

   ::navmiddleware::settings::RouteCriterion getRouteCriterion() const
   {
      return m_routeCriterion;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DestinationMemoryWaypoint payload:\n\n");
      stream << "Waypoint = " << m_destination.toString() << ::std::endl;
      stream << "RouteCriterion = " << m_routeCriterion << ::std::endl;
      return stream.str();
   }

private:
   DestinationMemoryDestination              m_destination;
   ::navmiddleware::settings::RouteCriterion m_routeCriterion;
};

class DestinationMemoryRoute
{
public:
   typedef uint64_t Id;
   enum Category
   {
      CATEGORY__FAVORITE_ROUTE
   };

public:
   DestinationMemoryRoute()
      : m_id(0)
   {}

   DestinationMemoryRoute(Id id, const ::std::string& name, const ::std::vector< DestinationMemoryWaypoint >& waypointList)
      : m_id(id), m_name(name), m_waypointList(waypointList)
   {}

   void setId(uint64_t id)
   {
      m_id = id;
   }

   uint64_t getId() const
   {
      return( m_id );
   }

   void setName(const ::std::string& name)
   {
      m_name = name;
   }

   const ::std::string& getName() const
   {
      return m_name;
   }

   void setFinalDestinationName(const ::std::string& finalDestinationName)
   {
      m_finalDestinationName = finalDestinationName;
   }

   const ::std::string& getFinalDestinationName() const
   {
      return m_finalDestinationName;
   }

   void setWaypointsName(const ::std::string& waypointsName)
   {
      m_waypointsName = waypointsName;
   }

   const ::std::string& getWaypointsName() const
   {
      return m_waypointsName;
   }

   void setWaypointList(const ::std::vector< DestinationMemoryWaypoint >& waypointList)
   {
      m_waypointList = waypointList;
   }

   const ::std::vector< DestinationMemoryWaypoint >& getWaypointList() const
   {
      return m_waypointList;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DestinationMemoryRoute payload:\n");

      stream << ::std::endl
             << "Route ID = " << m_id << ::std::endl
             << "Route Name = " << m_name << ::std::endl
             << "Final Destination Name = " << m_finalDestinationName << ::std::endl
             << "Waypoints Name = " << m_waypointsName << ::std::endl;

      for (size_t waypointListIndex = 0; waypointListIndex < m_waypointList.size(); waypointListIndex++)
      {
         stream << "Route Waypoint Index = " << waypointListIndex << ::std::endl
                << "Waypoint = " << m_waypointList[waypointListIndex].toString() << ::std::endl;
      }

      return stream.str();
   }

private:
   Id                                       m_id;
   ::std::string                            m_name;
   ::std::string                            m_finalDestinationName; // name(address) of the final destination
   /* name constructed by concatenating waypoint names(addresses) without final destination
    * if the software variant is SOFTWAREVARIANT_RIVI there is language dependent prefix "via " */
   ::std::string                            m_waypointsName;
   ::std::vector<DestinationMemoryWaypoint> m_waypointList;
};

class DestinationMemoryRouteList
{
public:
   void addRoute(const DestinationMemoryRoute& storedRoute)
   {
      m_routeList.push_back(storedRoute);
   }

   const ::std::vector<DestinationMemoryRoute>& getRouteList() const
   {
      return m_routeList;
   }

   void clearRouteList()
   {
      m_routeList.clear();
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DestinationMemoryRouteList payload:\n");
      stream << ::std::endl;

      for (unsigned int routeListIndex = 0; routeListIndex < m_routeList.size(); routeListIndex++)
      {
         stream << "Route list Index = " << routeListIndex << ::std::endl
                << "Route  =" << m_routeList[routeListIndex].toString() << ::std::endl;
      }

      return stream.str();
   }

private:
   ::std::vector<DestinationMemoryRoute> m_routeList;
};

enum DestinationMemoryItemType
{
   DESTINATION_MEMORY_ITEM_ENTRY,
   DESTINATION_MEMORY_ITEM_ROUTE,
};

/*
 * Composite type for handling of both Destination Memory Route (which is a list of waypoints) and
 * Destination Memory Entry objects (which references Destination with distinct category and user).
 */

class DestinationMemoryItem
{
public:
   struct IdElement
   {
      IdElement()
         : m_id(0), m_type(::navmiddleware::DESTINATION_MEMORY_ITEM_ENTRY)
      {}

      IdElement(uint64_t id, DestinationMemoryItemType type)
         : m_id(id), m_type(type)
      {}

      uint64_t                  m_id;
      DestinationMemoryItemType m_type;
   };

public:
   DestinationMemoryItem()
   {}

   IdElement getIdElement() const
   {
      return( m_idElement );
   }

   void setEntry(const DestinationMemoryEntry& entry)
   {
      m_idElement.m_id = entry.getId();
      m_idElement.m_type = ::navmiddleware::DESTINATION_MEMORY_ITEM_ENTRY;
      m_entry = entry;
   }

   void setRoute(const DestinationMemoryRoute& route)
   {
      m_idElement.m_id = route.getId();
      m_idElement.m_type = ::navmiddleware::DESTINATION_MEMORY_ITEM_ROUTE;
      m_route = route;
   }

   bool hasEntry() const
   {
      return m_idElement.m_type == ::navmiddleware::DESTINATION_MEMORY_ITEM_ENTRY;
   }

   bool hasRoute() const
   {
      return m_idElement.m_type == ::navmiddleware::DESTINATION_MEMORY_ITEM_ROUTE;
   }

   const DestinationMemoryEntry& getEntry() const
   {
      return m_entry;
   }

   const DestinationMemoryRoute& getRoute() const
   {
      return m_route;
   }

   // Format for the timestamp string is: YYYY-MM-DD
   void setTimestampDate(const ::std::string& timestampDate)
   {
      m_timestampDate = timestampDate;
   }

   ::std::string getTimestampDate() const
   {
      return m_timestampDate;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DestinationMemoryItem payload:\n");

      stream << ::std::endl
             << "Item ID = " << m_idElement.m_id << ::std::endl
             << "Timestamp date = " << m_timestampDate << ::std::endl;

      if(m_idElement.m_type == ::navmiddleware::DESTINATION_MEMORY_ITEM_ENTRY )
      {
         stream << "Entry  =" << m_entry.toString() << ::std::endl;
      }
      else if(m_idElement.m_type == ::navmiddleware::DESTINATION_MEMORY_ITEM_ROUTE )
      {
         stream << "Route  =" << m_route.toString() << ::std::endl;
      }
      else
      {
         stream << "INVALID DESTINATION MEMORY ITEM TYPE";
      }

      return stream.str();
   }

private:
   IdElement                  m_idElement;
   DestinationMemoryEntry     m_entry;
   DestinationMemoryRoute     m_route;
   ::std::string              m_timestampDate;
};

class DestinationMemoryItemList
{
public:
   DestinationMemoryItemList()
      : m_itemListCount(0)
   {}
   enum Filter
   {
      /* Only last guided single destination entries should be fetched/removed.
       * DestinationMemoryItem results are to be of the type DESTINATION_MEMORY_ITEM_ENTRY. */
      FILTER_ENTRY,
      /* Only last guided tours should be fetched/removed.
       * DestinationMemoryItem results are to be of the type DESTINATION_MEMORY_ITEM_ROUTE. */
      FILTER_ROUTE,
      /* Both last guided single destination entries and last guided tour waypoint entries should be fetched/removed.
       * DestinationMemoryItem results are to be of the type DESTINATION_MEMORY_ITEM_ENTRY. */
      FILTER_ENTRY_AND_ROUTE_WAYPOINT,
      /* Both last guided single destination entries and last guided tours should be fetched/removed.
       * DestinationMemoryItem results are to be of the types DESTINATION_MEMORY_ITEM_ENTRY and DESTINATION_MEMORY_ITEM_ROUTE. */
      FILTER_ENTRY_AND_ROUTE,
      FILTER_SIZEOF
   };

   void addItem(const DestinationMemoryItem& item)
   {
      m_itemList.push_back(item);
   }

   const ::std::vector<DestinationMemoryItem>& getItemList() const
   {
      return m_itemList;
   }

   ::std::vector<DestinationMemoryItem>& getItemListMutable()
   {
      return m_itemList;
   }

   void clearItemList()
   {
      m_itemList.clear();
   }

   void setItemListCount(uint32_t itemListCount)
   {
      m_itemListCount = itemListCount;
   }

   uint32_t getItemListCount() const
   {
      return m_itemListCount;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DestinationMemoryItemList payload:\n");
      stream << ::std::endl;

      for (size_t ndx = 0; ndx < m_itemList.size(); ndx++)
      {
         stream << "Item list Index = " << ndx << ::std::endl
                << "Item  = " << m_itemList[ndx].toString() << ::std::endl;
      }
      stream << "Item list count = " << m_itemListCount << std::endl;
      return stream.str();
   }

private:
   ::std::vector<DestinationMemoryItem> m_itemList;
   /* m_itemListCount provides total number of items stored in history/previous destination */
   uint32_t m_itemListCount;
};

enum DestinationMemoryStoreErrorStatus
{
   DESTINATION_MEMORY_STORE_ERROR_STATUS__MEMORY_FULL,
   DESTINATION_MEMORY_STORE_ERROR_STATUS__INTERNAL_ERROR
};

/**
 * Informs about the status of the storeEntry functions.
 *
 */
class DestinationMemoryStoreEntryStatus
{
public:
   enum StoreEntryStatus
   {
      STOREENTRYSTATUS__NOT_STARTED,
      STOREENTRYSTATUS__OK,
      STOREENTRYSTATUS__ERROR,
      STOREENTRYSTATUS__ENTRY_IS_OFFMAP
   };

   DestinationMemoryStoreEntryStatus()
      : m_storeEntryStatus(STOREENTRYSTATUS__NOT_STARTED)
      , m_storedEntryId(0ULL)
   {}

   DestinationMemoryStoreEntryStatus(
      StoreEntryStatus storeEntryStatus,
      DestinationMemoryEntry::Id storedEntryId = 0ULL,
      ValidValue<DestinationMemoryStoreErrorStatus> storeEntryExtendedErrorStatus = ValidValue<DestinationMemoryStoreErrorStatus>(),
      const ValidValue<DestinationMemoryDestination>& destination = ValidValue<DestinationMemoryDestination>(),
      const ValidValue<DestinationMemoryEntryAttributes>& attributes = ValidValue<DestinationMemoryEntryAttributes>())
      : m_storeEntryStatus(storeEntryStatus)
      , m_storedEntryId(storedEntryId)
      , m_storeEntryExtendedErrorStatus(storeEntryExtendedErrorStatus)
      , m_destination(destination)
      , m_attributes(attributes)
   {}

   DestinationMemoryStoreEntryStatus::StoreEntryStatus getStatus() const
   {
      return m_storeEntryStatus;
   }

   const ValidValue<DestinationMemoryStoreErrorStatus>& getStoreEntryExtendedErrorStatus() const
   {
      return m_storeEntryExtendedErrorStatus;
   }

   DestinationMemoryEntry::Id getStoredEntryId() const
   {
      return m_storedEntryId;
   }

   void setDestination(const ValidValue<DestinationMemoryDestination>& destination)
   {
      m_destination = destination;
   }

   const ValidValue<DestinationMemoryDestination>& getDestination() const
   {
      return m_destination;
   }

   void setAttributes(const ValidValue<DestinationMemoryEntryAttributes>& attributes)
   {
      m_attributes = attributes;
   }

   const ValidValue<DestinationMemoryEntryAttributes>& getAttributes() const
   {
      return m_attributes;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DestinationMemoryStoreEntryStatus payload:\n");
      stream << ::std::endl << "Status = ";
      if(m_storeEntryStatus == STOREENTRYSTATUS__NOT_STARTED )
      {
         stream << "NOT_STARTED" << ::std::endl;
      }
      else if(m_storeEntryStatus == STOREENTRYSTATUS__OK )
      {
         stream << "OK" << ::std::endl;
      }
      else if(m_storeEntryStatus == STOREENTRYSTATUS__ERROR )
      {
         stream << "ERROR" << ::std::endl;
         stream << "Extended error status: " << m_storeEntryExtendedErrorStatus.toString() << ::std::endl;
      }
      else if(m_storeEntryStatus == STOREENTRYSTATUS__ENTRY_IS_OFFMAP )
      {
         stream << "ENTRY_IS_OFFMAP" << ::std::endl;
      }
      else
      {
         stream << "UNKNOWN";
      }
      stream << "entry Id = " << m_storedEntryId << ::std::endl;
      if(m_destination.isValid())
      {
         stream << "Entry Destination = " << m_destination.getValue().toString() << ::std::endl;
      }
      else
      {
         stream << "Destination Object Invalid"<< ::std::endl;
      }
      if(m_attributes.isValid())
      {
         stream << "Destination Attributes = " << m_attributes.getValue().toString() << ::std::endl;
      }
      else
      {
         stream << "Destination Attributes Object Invalid"<< ::std::endl;
      }
      return stream.str();
   }

private:
   StoreEntryStatus                               m_storeEntryStatus;
   DestinationMemoryEntry::Id                     m_storedEntryId;
   ValidValue<DestinationMemoryStoreErrorStatus>  m_storeEntryExtendedErrorStatus;
   ValidValue<DestinationMemoryDestination>       m_destination;
   ValidValue<DestinationMemoryEntryAttributes>   m_attributes;
};
class DestinationMemoryStoreEntriesStatus
{
public:
   void addEntryStatus(const DestinationMemoryStoreEntryStatus& entryStatus)
   {
      m_entryStatusList.push_back(entryStatus);
   }

   const ::std::vector<DestinationMemoryStoreEntryStatus>& getEntryStatusList() const
   {
      return m_entryStatusList;
   }

   void clearEntryStatusList()
   {
      m_entryStatusList.clear();
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DestinationMemoryStoreEntryStatus payload:\n\n");
      for ( unsigned int entryStatusIndex = 0; entryStatusIndex < m_entryStatusList.size(); entryStatusIndex++)
      {
         stream << "Store entry index= " << entryStatusIndex << ::std::endl
                << "Status = " << m_entryStatusList[entryStatusIndex].toString() << ::std::endl;
      }
      return stream.str();
   }

private:
   ::std::vector<DestinationMemoryStoreEntryStatus> m_entryStatusList;
};
class DestinationMemoryStatus
{
public:
   DestinationMemoryStatus()
      : m_numberOfAddressbookEntries(0)
      , m_numberOfRoutes(0)
      , m_numberOfHomeEntries(0)
      , m_numberOfWorkEntries(0)
      , m_numberOfAddressGadgetEntries(0)
      , m_numberOfPersonalEPOIEntries(0)
      , m_numberOfFavoriteWeatherAlertEntries(0)
   {
      for(int32_t ndx = 0; ndx < DestinationMemoryItemList::FILTER_SIZEOF; ++ndx)
      {
         m_numberOfPreviousDestinationItems[ndx] = 0;
      }
   }

   uint32_t getNumberOfAddressBookEntries() const
   {
      return( m_numberOfAddressbookEntries );
   }

   void setNumberOfAddressbookEntries(uint32_t numberOfAddressbookEntries)
   {
      m_numberOfAddressbookEntries = numberOfAddressbookEntries;
   }

   uint32_t getNumberOfRoutes() const
   {
      return( m_numberOfRoutes );
   }

   void setNumberOfRoutes(uint32_t numberOfRoutes)
   {
      m_numberOfRoutes = numberOfRoutes;
   }

   uint32_t getNumberOfHomeEntries() const
   {
      return( m_numberOfHomeEntries );
   }

   void setNumberOfHomeEntries(uint32_t numberOfHomeEntries)
   {
      m_numberOfHomeEntries = numberOfHomeEntries;
   }

   uint32_t getNumberOfWorkEntries() const
   {
      return( m_numberOfWorkEntries );
   }

   void setNumberOfWorkEntries(uint32_t numberOfWorkEntries)
   {
      m_numberOfWorkEntries = numberOfWorkEntries;
   }

   uint32_t getNumberOfAddressGadgetEntries() const
   {
      return( m_numberOfAddressGadgetEntries );
   }

   void setNumberOfAddressGadgetEntries(uint32_t numberOfAddressGadgetEntries)
   {
      m_numberOfAddressGadgetEntries = numberOfAddressGadgetEntries;
   }

   uint32_t getNumberOfPersonalEPOIEntries() const
   {
      return( m_numberOfPersonalEPOIEntries );
   }

   void setNumberOfPersonalEPOIEntries(uint32_t numberOfPersonalEPOIEntries)
   {
      m_numberOfPersonalEPOIEntries = numberOfPersonalEPOIEntries;
   }

   uint32_t getNumberOfFavoriteWeatherAlertEntries() const
   {
      return( m_numberOfFavoriteWeatherAlertEntries );
   }

   void setNumberOfFavoriteWeatherAlertEntries(uint32_t numberOfFavoriteWeatherAlertEntries)
   {
      m_numberOfFavoriteWeatherAlertEntries = numberOfFavoriteWeatherAlertEntries;
   }

   uint32_t getNumberOfPreviousDestionationItems(DestinationMemoryItemList::Filter filter) const
   {
      return (filter < DestinationMemoryItemList::FILTER_SIZEOF ? m_numberOfPreviousDestinationItems[filter] : 0);
   }

   void setNumberOfPreviousDestinationItems(DestinationMemoryItemList::Filter filter, uint32_t numberOfPreviousDestinationItems)
   {
      if(filter < DestinationMemoryItemList::FILTER_SIZEOF)
      {
         m_numberOfPreviousDestinationItems[filter] = numberOfPreviousDestinationItems;
      }
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DestinationMemoryStatus payload:\n\n");
      stream << "Number. of stored address book entries = " << m_numberOfAddressbookEntries
             << "\n Number. of stored home entries = " << m_numberOfHomeEntries
             << "\n Number. of stored work entries = " << m_numberOfWorkEntries
             << "\n Number. of address gadget entries = " << m_numberOfAddressGadgetEntries
             << "\n Number. of personal EPOI entries = " << m_numberOfPersonalEPOIEntries
             << "\n Number. of favorite weather alert entries = " << m_numberOfFavoriteWeatherAlertEntries
             << "\n Number. of stored previous destinations = " << m_numberOfPreviousDestinationItems[DestinationMemoryItemList::FILTER_ENTRY]
             << "\n Number. of stored previous routes = " << m_numberOfPreviousDestinationItems[DestinationMemoryItemList::FILTER_ROUTE]
             << "\n Number. of stored previous destinations and route waypoints = " << m_numberOfPreviousDestinationItems[DestinationMemoryItemList::FILTER_ENTRY_AND_ROUTE_WAYPOINT]
             << "\n Number. of stored previous destinations and routes = " << m_numberOfPreviousDestinationItems[DestinationMemoryItemList::FILTER_ENTRY_AND_ROUTE]
             << "\n Number. of stored routes = " << m_numberOfRoutes << ::std::endl;
      return stream.str();
   }

private:
   uint32_t m_numberOfAddressbookEntries;
   uint32_t m_numberOfRoutes;
   uint32_t m_numberOfHomeEntries;
   uint32_t m_numberOfWorkEntries;
   uint32_t m_numberOfAddressGadgetEntries;
   uint32_t m_numberOfPersonalEPOIEntries;
   uint32_t m_numberOfFavoriteWeatherAlertEntries;
   uint32_t m_numberOfPreviousDestinationItems[DestinationMemoryItemList::FILTER_SIZEOF];
};

class DestinationMemoryStoreRouteStatus
{
public:
   enum StoreRouteStatus
   {
      STOREROUTESTATUS__NOT_STARTED,
      STOREROUTESTATUS__OK,
      STOREROUTESTATUS__ERROR,
   };

   DestinationMemoryStoreRouteStatus()
      : m_storeRouteStatus(STOREROUTESTATUS__NOT_STARTED),
        m_storedRouteId(0ULL)
   {}

   DestinationMemoryStoreRouteStatus(
      StoreRouteStatus storeRouteStatus,
      DestinationMemoryRoute::Id storedRouteId = 0ULL,
      ValidValue<DestinationMemoryStoreErrorStatus> storeRouteExtendedErrorStatus = ValidValue<DestinationMemoryStoreErrorStatus>())
      : m_storeRouteStatus(storeRouteStatus)
      , m_storedRouteId(storedRouteId)
      , m_storeRouteExtendedErrorStatus(storeRouteExtendedErrorStatus)
   {}

   DestinationMemoryStoreRouteStatus::StoreRouteStatus getStatus() const
   {
      return m_storeRouteStatus;
   }

   const ValidValue<DestinationMemoryStoreErrorStatus>& getStoreRouteExtendedErrorStatus() const
   {
      return m_storeRouteExtendedErrorStatus;
   }

   DestinationMemoryRoute::Id getStoredRouteId() const
   {
      return m_storedRouteId;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("DestinationMemoryStoreRouteStatus payload:\n");
      stream << ::std::endl << "Status = ";
      switch(m_storeRouteStatus)
      {
      case STOREROUTESTATUS__NOT_STARTED :
         stream << "NOT_STARTED" << ::std::endl;
         break;
      case STOREROUTESTATUS__OK :
         stream << "OK" << ::std::endl;
         break;
      case STOREROUTESTATUS__ERROR :
         stream << "ERROR" << ::std::endl;
         stream << "Extended error status: " << m_storeRouteExtendedErrorStatus.toString() << ::std::endl;
         break;
      default :
         stream << "UNKNOWN";
         break;
      }
      return stream.str();
   }

private:
   StoreRouteStatus                               m_storeRouteStatus;
   DestinationMemoryRoute::Id                     m_storedRouteId;
   ValidValue<DestinationMemoryStoreErrorStatus>  m_storeRouteExtendedErrorStatus;
};

} /* namespace navmiddleware */

#endif // PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_DESTINATIONMEMORYINFOS_H_

