/**
* @copyright (c) 2015-2021 Robert Bosch Car Multimedia GmbH
* @addtogroup NavMiddleware
* @{
*/

#ifndef PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_CONFIGURATIONINFOS_H_
#define PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_CONFIGURATIONINFOS_H_

#include "InfoTypes.h"
#include "RouteSettings.h"
#include <sstream>
#include <string>
#include <vector>
#include <utility>

namespace navmiddleware
{

class ConfigurationInfos
{
public:

   /**
   * Represents determines variant handling between Renault-Nissan etc.
   */
   enum SoftwareVariant
   {
      SOFTWAREVARIANT_NIVI = 0,
      SOFTWAREVARIANT_NIVI_2,
      SOFTWAREVARIANT_RIVI,
      SOFTWAREVARIANT_RIVI_2,
      SOFTWAREVARIANT_PIVI
   };

   enum NavVariant
   {
      NAV_VARIANT__FULL_NAV = 0,
      NAV_VARIANT__FAKE_NAV
   };

   enum LightMode
   {
      LIGHT_MODE_DAY_AUTOMATIC,
      LIGHT_MODE_DAY_PERMANENT,
      LIGHT_MODE_NIGHT_AUTOMATIC,
      LIGHT_MODE_NIGHT_PERMANENT
   };

   enum TimeZoneAndDSTConsideration
   {
      TIMEZONE_AND_DST_OFFSET_CONSIDERATION__NOT_REQUIRED=0,
      TIMEZONE_AND_DST_OFFSET_CONSIDERATION__REQUIRED
   };

   enum DisplayAspectRatio
   {
      DISPLAY_ASPECT_RATIO__15_9,
      DISPLAY_ASPECT_RATIO__17_9
   };

   enum DisplayResolution
   {
      DISPLAY_RESOLUTION__UNKNOWN = 0,
      DISPLAY_RESOLUTION__SVGA,
      DISPLAY_RESOLUTION__WXGA
   };

   /**
   * Represents different user profiles for the system
   */
   enum UserProfile
   {
      USER_PROFILE__ANONYMOUS=0,
      USER_PROFILE__1,
      USER_PROFILE__2,
      USER_PROFILE__3,
      USER_PROFILE__4,
      USER_PROFILE__5,
      USER_PROFILE__6
   };

   enum VehicleModeType
   {
      VEHICLE_MODE_TYPE__UNKNOWN = 0,
      VEHICLE_MODE_TYPE__ICE,
      VEHICLE_MODE_TYPE__EV,
      VEHICLE_MODE_TYPE__HEV,
      VEHICLE_MODE_TYPE__PHEV
   };

   enum VehicleFuelType
   {
      VEHICLE_FUEL_TYPE__UNKNOWN = 0,
      VEHICLE_FUEL_TYPE__PETROL,
      VEHICLE_FUEL_TYPE__DIESEL,
      VEHICLE_FUEL_TYPE__HYBRID_DIESEL,
      VEHICLE_FUEL_TYPE__HYBRID_PETROL,
      VEHICLE_FUEL_TYPE__ELECTRICITY,
      VEHICLE_FUEL_TYPE__PLUG_IN_HYBRID
   };

   enum VehicleDriveTrain
   {
      VEHICLE_DRIVE_TRAIN__UNKNOWN,
      VEHICLE_DRIVE_TRAIN__FRONT_WHEEL_DRIVE,
      VEHICLE_DRIVE_TRAIN__REAR_WHEEL_DRIVE,
      VEHICLE_DRIVE_TRAIN__ALL_WHEEL_DRIVE
   };

   enum VehicleBrandType
   {
      VEHICLE_BRAND_TYPE__UNKNOWN = 0,
      VEHICLE_BRAND_TYPE__RENAULT,
      VEHICLE_BRAND_TYPE__NISSAN,
      VEHICLE_BRAND_TYPE__INFINITI,
      VEHICLE_BRAND_TYPE__DACIA,
      VEHICLE_BRAND_TYPE__DAIMLER,
      VEHICLE_BRAND_TYPE__LADA,
      VEHICLE_BRAND_TYPE__SMART,
      VEHICLE_BRAND_TYPE__MITSUBISHI,
      VEHICLE_BRAND_TYPE__ALPINE,
      VEHICLE_BRAND_TYPE__RENAULT_TRUCKS
   };

   enum SystemPowerMode
   {
      SYSTEM_POWER_MODE__IGNITION_ON,
      SYSTEM_POWER_MODE__IGNITION_OFF
   };

   enum SystemType
   {
      SYSTEM_TYPE__UNKNOWN = 0,
      SYSTEM_TYPE__ENTRY,
      SYSTEM_TYPE__ENTRY_ACCESS,
      SYSTEM_TYPE__MID
   };

   enum TCUStatus
   {
      TCU_STATUS__UNKNOWN = 0,
      TCU_STATUS__WITHOUT_TCU,
      TCU_STATUS__WITH_TCU_NOT_READY,
      TCU_STATUS__WITH_TCU_READY,
   };

   enum SystemState
   {
      SYSTEM_STATE__BACKGROUND = 0,
      SYSTEM_STATE__PREPARE_SHUTDOWN,
      SYSTEM_STATE__SHUTDOWN,
      SYSTEM_STATE__SUSPEND,
      SYSTEM_STATE__STANDBY,
      SYSTEM_STATE__OFF,
      SYSTEM_STATE__DOWNLOAD,
      SYSTEM_STATE__ON,
      SYSTEM_STATE__DOOR_OPEN,
      SYSTEM_STATE__IGNITION,
      SYSTEM_STATE__DIAGNOSIS,
      SYSTEM_STATE__PROFILE,
      SYSTEM_STATE__MMI_STANDBY_PWR_SAVE_1,
      SYSTEM_STATE__MMI_STANDBY_PWR_SAVE_2,
      SYSTEM_STATE__MMI_STANDBY,
      SYSTEM_STATE__MMI_ON,
      SYSTEM_STATE__MMI_STANDBY_RESCTRICTED,
      SYSTEM_STATE__MMI_ON_DIAG,
      SYSTEM_STATE__MMI_ON_TEL,
      SYSTEM_STATE__MMI_ON_SWDL,
      SYSTEM_STATE__MMI_STANDBY_CUSTOMER_SWDL,
      SYSTEM_STATE__MMI_STANDBY_PWR_SAVE,
      SYSTEM_STATE__OVER_TMP,
      SYSTEM_STATE__SAFE,
      SYSTEM_STATE__STATE_MAX,
      SYSTEM_STATE__UNKNOWN
   };

   enum DeviceVariant
   {
      DEVICEVARIANT__LANDSCAPE = 0,
      DEVICEVARIANT__PORTRAIT
   };

   enum EvHvacType
   {
      EV_HVAC_TYPE__NONE = 0,
      EV_HVAC_TYPE__X12G
   };

   enum EvAllianceReachAreaType
   {
      EV_ALLIANCE_REACH_AREA_TYPE__PATATOID = 0,
      EV_ALLIANCE_REACH_AREA_TYPE__CIRCLE
   };

   struct ScreenDimensions
   {
      ScreenDimensions(unsigned int screenWidth, unsigned int screenHeight, DisplayAspectRatio displayAspectRatioVal, DisplayResolution displayResolutionVal)
         : width(screenWidth), height(screenHeight), displayAspectRatio(displayAspectRatioVal), displayResolution(displayResolutionVal) {}
      ScreenDimensions() : width(0), height(0), displayAspectRatio(DISPLAY_ASPECT_RATIO__15_9), displayResolution(DISPLAY_RESOLUTION__UNKNOWN) {}

      bool operator==(const ScreenDimensions& rhs) const
      {
         return width == rhs.width &&
                height == rhs.height &&
                displayAspectRatio == rhs.displayAspectRatio &&
                displayResolution == rhs.displayResolution;
      }

      bool operator!=(const ScreenDimensions& rhs) const
      {
         return !(*this == rhs);
      }

      unsigned int width;
      unsigned int height;
      DisplayAspectRatio displayAspectRatio;
      //Illustrates the display resolutions, the view dimensions is also expected in addition.
      DisplayResolution displayResolution;
   };

   enum PrivacyMode
   {
      PRIVACY_MODE__UNKNOWN = 0,
      PRIVACY_MODE__ON,
      PRIVACY_MODE__OFF
   };

   enum LocationInputEntryType
   {
      LOCATION_INPUT_ENTRY_TYPE__UNKNOWN,
      LOCATION_INPUT_ENTRY_TYPE__EUR, // Form based Input.
      LOCATION_INPUT_ENTRY_TYPE__NAR, // User guided Entry flow.
      LOCATION_INPUT_ENTRY_TYPE__JPN  // User guided fixed Entry flow.
   };

   enum FotaBearerType
   {
      FOTA_BEARER_TYPE__NOT_USE = 0,
      FOTA_BEARER_TYPE__USE,
      FOTA_BEARER_TYPE__ONLY_USE_FOTA,
      FOTA_BEARER_TYPE__ONLY_USE_MAP
   };

   enum ChargeFactorType
   {
      CHARGE_FACTOR_TYPE__UNSPECIFIED,                   // Not Present(Default Value)
      CHARGE_FACTOR_TYPE__STANDARD,                      // Normal Charge
      CHARGE_FACTOR_TYPE__ACCELERATED,
      CHARGE_FACTOR_TYPE__FAST,
      CHARGE_FACTOR_TYPE__DC,                            // QuickCharge
      CHARGE_FACTOR_TYPE__NORMAL_ACCELERATED_AND_FAST,   // Normal + Accelerated + Fast
      CHARGE_FACTOR_TYPE__NORMAL_ACCELERATED_FAST_AND_DC // Normal + Accelerated + Fast + DC
   };

   static ::std::string toString(ChargeFactorType chargeFactorType)
   {
      switch(chargeFactorType)
      {
      case CHARGE_FACTOR_TYPE__UNSPECIFIED:
         return "CHARGE_FACTOR_TYPE__UNSPECIFIED";
      case CHARGE_FACTOR_TYPE__STANDARD:
         return "CHARGE_FACTOR_TYPE__STANDARD";
      case CHARGE_FACTOR_TYPE__ACCELERATED:
         return "CHARGE_FACTOR_TYPE__ACCELERATED";
      case CHARGE_FACTOR_TYPE__FAST:
         return "CHARGE_FACTOR_TYPE__FAST";
      case CHARGE_FACTOR_TYPE__DC:
         return "CHARGE_FACTOR_TYPE__DC";
      case CHARGE_FACTOR_TYPE__NORMAL_ACCELERATED_AND_FAST:
         return "CHARGE_FACTOR_TYPE__NORMAL_ACCELERATED_AND_FAST";
      case CHARGE_FACTOR_TYPE__NORMAL_ACCELERATED_FAST_AND_DC:
         return "CHARGE_FACTOR_TYPE__NORMAL_ACCELERATED_FAST_AND_DC";
      default:
         ::std::stringstream stream;
         stream << "CHARGE_FACTOR_TYPE__<" << static_cast<unsigned int>(chargeFactorType) << ">";
         return stream.str();
      }
   }

   enum VehicleInletType
   {
      VEHICLE_INLET_TYPE__UNSPECIFIED,
      VEHICLE_INLET_TYPE__IEC_62196_2_TYPE_1,                      // Type 1
      VEHICLE_INLET_TYPE__IEC_62196_2_TYPE_2,                      // Type 2
      VEHICLE_INLET_TYPE__IEC_62196_3_TYPE_2_CCS,                  // Combo 2
      VEHICLE_INLET_TYPE__SAE_J1772_CCS,                           // Combo 1
      VEHICLE_INLET_TYPE__CHINA_SPECIFIC_AC,                       // AC charging - GBT 20234.2
      VEHICLE_INLET_TYPE__CHINA_SPECIFIC_DC,                       // DC charging - GBT 20234.3
      VEHICLE_INLET_TYPE__CHINA_SPECIFIC_AC_AND_CHINA_SPECIFIC_DC,
      VEHICLE_INLET_TYPE__CHINA_SPECIFIC_AC_AND_CHADEMO,
      VEHICLE_INLET_TYPE__CHINA_SPECIFIC_AC_AND_COMBO_2,
      VEHICLE_INLET_TYPE__CHADEMO,
      VEHICLE_INLET_TYPE__CHADEMO_AND_COMBO_1,
      VEHICLE_INLET_TYPE__CHADEMO_AND_COMBO_2,
      VEHICLE_INLET_TYPE__COMBO_1_AND_TYPE_2,
      VEHICLE_INLET_TYPE__COMBO_1_AND_CHINA_SPECIFIC_AC,
      VEHICLE_INLET_TYPE__COMBO_2_AND_TYPE_1
   };

   static ::std::string toString(VehicleInletType vehicleInletType)
   {
      switch(vehicleInletType)
      {
      case VEHICLE_INLET_TYPE__UNSPECIFIED:
         return "VEHICLE_INLET_TYPE__UNSPECIFIED";
      case VEHICLE_INLET_TYPE__IEC_62196_2_TYPE_1:
         return "VEHICLE_INLET_TYPE__IEC_62196_2_TYPE_1";
      case VEHICLE_INLET_TYPE__IEC_62196_2_TYPE_2:
         return "VEHICLE_INLET_TYPE__IEC_62196_2_TYPE_2";
      case VEHICLE_INLET_TYPE__IEC_62196_3_TYPE_2_CCS:
         return "VEHICLE_INLET_TYPE__IEC_62196_3_TYPE_2_CCS";
      case VEHICLE_INLET_TYPE__SAE_J1772_CCS:
         return "VEHICLE_INLET_TYPE__SAE_J1772_CCS";
      case  VEHICLE_INLET_TYPE__CHINA_SPECIFIC_AC:
         return "VEHICLE_INLET_TYPE__CHINA_SPECIFIC_AC";
      case  VEHICLE_INLET_TYPE__CHINA_SPECIFIC_DC:
         return "VEHICLE_INLET_TYPE__CHINA_SPECIFIC_DC";
      case VEHICLE_INLET_TYPE__CHINA_SPECIFIC_AC_AND_CHINA_SPECIFIC_DC:
         return "VEHICLE_INLET_TYPE__CHINA_SPECIFIC_AC_AND_CHINA_SPECIFIC_DC";
      case VEHICLE_INLET_TYPE__CHINA_SPECIFIC_AC_AND_CHADEMO:
         return "VEHICLE_INLET_TYPE__CHINA_SPECIFIC_AC_AND_CHADEMO";
      case VEHICLE_INLET_TYPE__CHINA_SPECIFIC_AC_AND_COMBO_2:
         return "VEHICLE_INLET_TYPE__CHINA_SPECIFIC_AC_AND_COMBO_2";
      case VEHICLE_INLET_TYPE__CHADEMO:
         return "VEHICLE_INLET_TYPE__CHADEMO";
      case VEHICLE_INLET_TYPE__CHADEMO_AND_COMBO_1:
         return "VEHICLE_INLET_TYPE__CHADEMO_AND_COMBO_1";
      case VEHICLE_INLET_TYPE__CHADEMO_AND_COMBO_2:
         return "VEHICLE_INLET_TYPE__CHADEMO_AND_COMBO_2";
      case VEHICLE_INLET_TYPE__COMBO_1_AND_TYPE_2:
         return "VEHICLE_INLET_TYPE__COMBO_1_AND_TYPE_2";
      case VEHICLE_INLET_TYPE__COMBO_1_AND_CHINA_SPECIFIC_AC:
         return "VEHICLE_INLET_TYPE__COMBO_1_AND_CHINA_SPECIFIC_AC";
      case VEHICLE_INLET_TYPE__COMBO_2_AND_TYPE_1:
         return "VEHICLE_INLET_TYPE__COMBO_2_AND_TYPE_1";
      default:
         ::std::stringstream stream;
         stream << "VEHICLE INLET TYPE__<" << static_cast<unsigned int>(vehicleInletType) << ">";
         return stream.str();
      }
   }

   class CompatibleChargingConfigurationInfo
   {
   public:
      enum ChargingCableGroup                       // Meaning of Groups Chargespot side plug -> Vehicle side plug
      {
         CHARGING_CABLE_GROUP__INVALID = 0,         // Invalid
         CHARGING_CABLE_GROUP__1,                   // Standard Household -> IEC 62196-2 Type 1/Type 2/China Specific
         CHARGING_CABLE_GROUP__2,                   // IEC 62196-2 Type 2 -> IEC 62196-2 Type 1/Type 2
         CHARGING_CABLE_GROUP__3,                   // IEC 62196-2 Type 3 -> IEC 62196-2 Type 1/Type 2
         CHARGING_CABLE_GROUP__4,                   // 309-Single Phase-Industrial -> IEC 62196-2 Type 1/Type 2
         CHARGING_CABLE_GROUP__5,                   // China Infrastructure -> China specific (AC Charging)
         CHARGING_CABLE_GROUP__6,                   // Korea specific EVSE plug -> IEC 62196-2 Type 1/Type 2
         CHARGING_CABLE_GROUP__7,                   // Rest of the world: Group 1 + Group 2+ Group 4
         CHARGING_CABLE_GROUP__8,                   // France specific: Group 1 + Group 2 + Group 3 + Group 4
         CHARGING_CABLE_GROUP__9                    // China specific: Group 1 + Group 5
      };

      static ::std::string toString(ChargingCableGroup chargingCableGroup)
      {
         switch(chargingCableGroup)
         {
         case CHARGING_CABLE_GROUP__INVALID:
            return "CHARGING_CABLE_GROUP__INVALID";
         case CHARGING_CABLE_GROUP__1:
            return "CHARGING_CABLE_GROUP__1";
         case CHARGING_CABLE_GROUP__2:
            return "CHARGING_CABLE_GROUP__2";
         case CHARGING_CABLE_GROUP__3:
            return "CHARGING_CABLE_GROUP__3";
         case CHARGING_CABLE_GROUP__4:
            return "CHARGING_CABLE_GROUP__4";
         case CHARGING_CABLE_GROUP__5:
            return "CHARGING_CABLE_GROUP__5";
         case CHARGING_CABLE_GROUP__6:
            return "CHARGING_CABLE_GROUP__6";
         case CHARGING_CABLE_GROUP__7:
            return "CHARGING_CABLE_GROUP__7";
         case CHARGING_CABLE_GROUP__8:
            return "CHARGING_CABLE_GROUP__8";
         case CHARGING_CABLE_GROUP__9:
            return "CHARGING_CABLE_GROUP__9";
         default:
            ::std::stringstream stream;
            stream << "CHARGING_CABLE_GROUP__<" << static_cast<unsigned int>(chargingCableGroup) << ">";
            return stream.str();
         }
      }

      CompatibleChargingConfigurationInfo()
         : m_chargingCableGroup(CHARGING_CABLE_GROUP__INVALID)
         , m_vehicleConnectorType(EV_CONNECTOR_TYPE__UNSPECIFIED)
         , m_vehicleInletType(VEHICLE_INLET_TYPE__UNSPECIFIED)
         , m_spotSidePlugType(EV_CONNECTOR_TYPE__UNSPECIFIED)
         , m_chargeFactorType(CHARGE_FACTOR_TYPE__UNSPECIFIED) {}

      bool operator==(const CompatibleChargingConfigurationInfo& rhs) const
      {
         return m_chargingCableGroup == rhs.m_chargingCableGroup &&
                m_vehicleConnectorType == rhs.m_vehicleConnectorType &&
                m_vehicleInletType == rhs.m_vehicleInletType &&
                m_spotSidePlugType == rhs.m_spotSidePlugType &&
                m_chargeFactorType == rhs.m_chargeFactorType;
      }

      bool operator!=(const CompatibleChargingConfigurationInfo& rhs) const
      {
         return !(*this == rhs);
      }

      void setChargingCableGroup(ChargingCableGroup chargingCableGroup)
      {
         m_chargingCableGroup = chargingCableGroup;
      }

      ChargingCableGroup getChargingCableGroup() const
      {
         return m_chargingCableGroup;
      }

      void setVehicleConnectorType(EvConnectorType vehicleConnectorType)
      {
         m_vehicleConnectorType = vehicleConnectorType;
      }

      EvConnectorType getVehicleConnectorType() const
      {
         return m_vehicleConnectorType;
      }

      void setVehicleInletType(VehicleInletType vehicleInletType)
      {
         m_vehicleInletType = vehicleInletType;
      }

      VehicleInletType getVehicleInletType() const
      {
         return m_vehicleInletType;
      }

      void setSpotSidePlugType(EvConnectorType spotSidePlugType)
      {
         m_spotSidePlugType = spotSidePlugType;
      }

      EvConnectorType getSpotSidePlugType() const
      {
         return m_spotSidePlugType;
      }

      void setChargeFactorType(ChargeFactorType chargeFactorType)
      {
         m_chargeFactorType = chargeFactorType;
      }

      ChargeFactorType getChargeFactorType() const
      {
         return m_chargeFactorType;
      }

      ::std::string toString() const
      {
         ::std::stringstream stream("CompatibleChargingConfigurationInfo: ");
         stream << "ChargingCableGroup = "<< toString(m_chargingCableGroup) << ", "
                << "VehicleConnectorType = "<< ::navmiddleware::toString(m_vehicleConnectorType) << ", "
                << "VehicleInletType = "<< ::navmiddleware::ConfigurationInfos::toString(m_vehicleInletType) << ", "
                << "SpotSidePlugType = "<< ::navmiddleware::toString(m_spotSidePlugType) << ", "
                << "ChargeFactorType    = "<< ::navmiddleware::ConfigurationInfos::toString(m_chargeFactorType);
         return stream.str();
      }

   private:
      ChargingCableGroup m_chargingCableGroup;
      EvConnectorType    m_vehicleConnectorType;
      VehicleInletType   m_vehicleInletType;
      EvConnectorType    m_spotSidePlugType;
      ChargeFactorType   m_chargeFactorType;
   };

   struct DateTime
   {
      DateTime(unsigned int year, unsigned int month, unsigned int day, unsigned int hour, unsigned int minute, unsigned int second)
         :m_year(year), m_month(month), m_day(day), m_hour(hour), m_minute(minute), m_second(second) {}

      DateTime() : m_year(0), m_month(0), m_day(0), m_hour(0), m_minute(0), m_second(0) {}

      bool operator==(const DateTime& rhs)const
      {
         return m_year == rhs.m_year &&
                m_month == rhs.m_month &&
                m_day == rhs.m_day &&
                m_hour == rhs.m_hour &&
                m_minute == rhs.m_minute &&
                m_second == rhs.m_second;
      }

      bool operator!=(const DateTime& rhs) const
      {
         return !(*this == rhs);
      }

      unsigned int m_year;
      unsigned int m_month;
      unsigned int m_day;
      unsigned int m_hour;
      unsigned int m_minute;
      unsigned int m_second;
   };

   ConfigurationInfos()
      : m_softwareVariant(SOFTWAREVARIANT_NIVI)
      , m_navVariant(NAV_VARIANT__FULL_NAV)
      , m_userProfile(USER_PROFILE__ANONYMOUS)
      , m_hmiLanguageCodeAsString("")
      , m_sdsLanguageCodeAsString("")
      , m_trafficTextLanguageCodeAsString("")
      , m_clusterLanguageCodeAsString("")
      , m_hmiLanguageCode(LANGUAGE_CODE__UNKNOWN)
      , m_sdsLanguageCode(LANGUAGE_CODE__UNKNOWN)
      , m_trafficTextLanguageCode(LANGUAGE_CODE__UNKNOWN)
      , m_clusterLanguageCode(LANGUAGE_CODE__UNKNOWN)
      , m_distanceUnitSystem(DISTANCEUNITSYSTEM_METRIC)
      , m_dateFormat(DATEFORMAT_UNSET)
      , m_timeFormat(TIMEFORMAT_UNSET)
      , m_regionType(REGIONTYPE_UNKNOWN)
      , m_vehicleType(VEHICLE_TYPE__UNKNOWN)
      , m_vehicleModeType(VEHICLE_MODE_TYPE__UNKNOWN)
      , m_vehicleBrandType(VEHICLE_BRAND_TYPE__UNKNOWN)
      , m_vehicleFuelType(VEHICLE_FUEL_TYPE__UNKNOWN)
      , m_vehicleDriveTrain(VEHICLE_DRIVE_TRAIN__UNKNOWN)
      , m_limitedFeatureMode()
      , m_lightMode(LIGHT_MODE_DAY_AUTOMATIC)
      , m_navigationUnitIdentifier("")
      , m_systemPowerMode(SYSTEM_POWER_MODE__IGNITION_OFF)
      , m_systemType(SYSTEM_TYPE__UNKNOWN)
      , m_tcuStatus(TCU_STATUS__UNKNOWN)
      , m_systemState(SYSTEM_STATE__UNKNOWN)
      , m_deviceVariant(DEVICEVARIANT__LANDSCAPE)
      , m_defaultPosition(52.117013,9.891959)
      , m_defaultPositionHeading(0.f)
      , m_displayTimeOffset(0)
      , m_timeZoneAndDSTConsideration(TIMEZONE_AND_DST_OFFSET_CONSIDERATION__NOT_REQUIRED)
      , m_implicitMapViewCreationFlag(true)
      , m_isTrafficRegion(true)
      , m_genericConfiguration()
      , m_privacyMode(PRIVACY_MODE__UNKNOWN)
      , m_locationInputEntryType(LOCATION_INPUT_ENTRY_TYPE__UNKNOWN)
      , m_vehicleIdentificationNumber("")
      , m_fotaBearerType(FOTA_BEARER_TYPE__NOT_USE)
      , m_evHvacType(EV_HVAC_TYPE__NONE)
      , m_evAllianceReachAreaType(EV_ALLIANCE_REACH_AREA_TYPE__PATATOID)
      , m_isTrafficSignRecognitionSupported(false)
      , m_residentialOrLocalMode()   //<-- INF4CV
   {

   }

   bool operator==(const ConfigurationInfos& rhs) const
   {
      return m_softwareVariant == rhs.m_softwareVariant &&
             m_navVariant == rhs.m_navVariant &&
             m_screenSize == rhs.m_screenSize &&
             m_userProfile == rhs.m_userProfile &&
             m_hmiLanguageCode == rhs.m_hmiLanguageCode &&
             m_sdsLanguageCode == rhs.m_sdsLanguageCode &&
             m_trafficTextLanguageCode == rhs.m_trafficTextLanguageCode &&
             m_clusterLanguageCode == rhs.m_clusterLanguageCode &&
             m_distanceUnitSystem == rhs.m_distanceUnitSystem &&
             m_dateFormat == rhs.m_dateFormat &&
             m_timeFormat == rhs.m_timeFormat &&
             m_regionType == rhs.m_regionType &&
             m_vehicleType == rhs.m_vehicleType &&
             m_vehicleModeType == rhs.m_vehicleModeType &&
             m_vehicleBrandType == rhs.m_vehicleBrandType &&
             m_vehicleFuelType == rhs.m_vehicleFuelType &&
             m_vehicleDriveTrain == rhs.m_vehicleDriveTrain &&
             m_limitedFeatureMode == rhs.m_limitedFeatureMode &&
             m_lightMode == rhs.m_lightMode &&
             m_navigationUnitIdentifier == rhs.m_navigationUnitIdentifier &&
             m_defaultVehicleProfile == rhs.m_defaultVehicleProfile &&
             m_systemPowerMode == rhs.m_systemPowerMode &&
             m_systemType == rhs.m_systemType &&
             m_tcuStatus == rhs.m_tcuStatus &&
             m_systemState == rhs.m_systemState &&
             m_defaultPosition == rhs.m_defaultPosition &&
             m_defaultPositionHeading == rhs.m_defaultPositionHeading &&
             m_displayTimeOffset == rhs.m_displayTimeOffset &&
             m_timeZoneAndDSTConsideration == rhs.m_timeZoneAndDSTConsideration &&
             m_implicitMapViewCreationFlag == rhs.m_implicitMapViewCreationFlag &&
             m_isTrafficRegion == rhs.m_isTrafficRegion &&
             m_genericConfiguration == rhs.m_genericConfiguration &&
             m_deviceVariant == rhs.m_deviceVariant &&
             m_privacyMode == rhs.m_privacyMode &&
             m_locationInputEntryType == rhs.m_locationInputEntryType &&
             m_vehicleIdentificationNumber == rhs.m_vehicleIdentificationNumber &&
             m_fotaBearerType == rhs.m_fotaBearerType &&
             m_compatibleChargingConfigurationInfo == rhs.m_compatibleChargingConfigurationInfo &&
             m_evHvacType == rhs.m_evHvacType &&
             m_evAllianceReachAreaType == rhs.m_evAllianceReachAreaType &&
             m_isTrafficSignRecognitionSupported == rhs.m_isTrafficSignRecognitionSupported &&
             m_productionDateTime == rhs.m_productionDateTime &&
             m_residentialOrLocalMode == rhs.m_residentialOrLocalMode;   //<-- INF4CV
   }

   bool operator!=(const ConfigurationInfos& rhs) const
   {
      return !(*this == rhs);
   }

   SoftwareVariant getSoftwareVariant() const
   {
      return m_softwareVariant;
   }

   void setSoftwareVariant(SoftwareVariant softwareVariant)
   {
      m_softwareVariant = softwareVariant;
   }

   NavVariant getNavVariant() const
   {
      return m_navVariant;
   }

   void setNavVariant(NavVariant navVariant)
   {
      m_navVariant = navVariant;
   }

   const ScreenDimensions& getScreenSize() const
   {
      return m_screenSize;
   }

   void setScreenSize(const ScreenDimensions& screenSize)
   {
      m_screenSize = screenSize;
   }

   UserProfile getUserProfile() const
   {
      return m_userProfile;
   }

   void setUserProfile(UserProfile userProfile)
   {
      m_userProfile = userProfile;
   }

   void setHMILanguage(const ::std::string& hmiLanguageCode, LanguageCode languageCode)
   {
      m_hmiLanguageCodeAsString = hmiLanguageCode;
      m_hmiLanguageCode = languageCode;
   }

   const ::std::string& getHMILanguageAsString() const
   {
      return m_hmiLanguageCodeAsString;
   }

   LanguageCode getHMILanguage() const
   {
      return m_hmiLanguageCode;
   }

   void setSDSLanguage(const ::std::string& sdsLanguageCode, LanguageCode languageCode)
   {
      m_sdsLanguageCodeAsString = sdsLanguageCode;
      m_sdsLanguageCode = languageCode;
   }

   const ::std::string& getSDSLanguageAsString() const
   {
      return m_sdsLanguageCodeAsString;
   }

   LanguageCode getSDSLanguage() const
   {
      return m_sdsLanguageCode;
   }

   void setTrafficTextLanguage(const ::std::string& trafficTextLanguageCode, LanguageCode languageCode)
   {
      m_trafficTextLanguageCodeAsString = trafficTextLanguageCode;
      m_trafficTextLanguageCode         = languageCode;
   }

   const ::std::string& getTrafficTextLanguageAsString() const
   {
      return m_trafficTextLanguageCodeAsString;
   }

   LanguageCode getTrafficTextLanguage() const
   {
      return m_trafficTextLanguageCode;
   }

   void setClusterLanguage(const ::std::string& clusterLanguageCode, LanguageCode languageCode)
   {
      m_clusterLanguageCodeAsString = clusterLanguageCode;
      m_clusterLanguageCode = languageCode;
   }

   const ::std::string& getClusterLanguageAsString() const
   {
      return m_clusterLanguageCodeAsString;
   }

   LanguageCode getClusterLanguage() const
   {
      return m_clusterLanguageCode;
   }

   DistanceUnitSystem getDistanceUnitSystem() const
   {
      return m_distanceUnitSystem;
   }

   void setDistanceUnitSystem(DistanceUnitSystem distanceUnitSystem)
   {
      m_distanceUnitSystem = distanceUnitSystem;
   }

   DateFormat getDateFormat() const
   {
      return m_dateFormat;
   }

   void setDateFormat(DateFormat dateFormat)
   {
      m_dateFormat = dateFormat;
   }

   TimeFormat getTimeFormat() const
   {
      return m_timeFormat;
   }

   void setTimeFormat(TimeFormat timeFormat)
   {
      m_timeFormat = timeFormat;
   }

   RegionType getRegionType() const
   {
      return m_regionType;
   }

   void setRegionType(RegionType regionType)
   {
      m_regionType = regionType;
   }

   VehicleType getVehicleType() const
   {
      return m_vehicleType;
   }

   void setVehicleType(VehicleType vehicleType)
   {
      m_vehicleType = vehicleType;
   }

   VehicleModeType getVehicleModeType() const
   {
      return m_vehicleModeType;
   }

   void setVehicleModeType(VehicleModeType vehicleModeType)
   {
      m_vehicleModeType = vehicleModeType;
   }

   VehicleFuelType getVehicleFuelType() const
   {
      return m_vehicleFuelType;
   }

   void setVehicleFuelType(VehicleFuelType vehicleFuelType)
   {
      m_vehicleFuelType = vehicleFuelType;
   }

   VehicleDriveTrain getVehicleDriveTrain() const
   {
      return m_vehicleDriveTrain;
   }

   void setVehicleDriveTrain(VehicleDriveTrain vehicleDriveTrain)
   {
      m_vehicleDriveTrain = vehicleDriveTrain;
   }

   VehicleBrandType getVehicleBrandType() const
   {
      return m_vehicleBrandType;
   }

   void setVehicleBrandType(VehicleBrandType vehicleBrandType)
   {
      m_vehicleBrandType = vehicleBrandType;
   }

   bool isNarRegion() const
   {
      if(::navmiddleware::REGIONTYPE_USA == m_regionType
            ||
            ::navmiddleware::REGIONTYPE_MEX == m_regionType
            ||
            ::navmiddleware::REGIONTYPE_CAN == m_regionType)
      {
         return true;
      }
      return false;
   }

   const LimitedFeatureMode& getLimitedFeatureMode() const
   {
      return m_limitedFeatureMode;
   }

   void setLimitedFeatureMode(const LimitedFeatureMode& limitedFeatureMode)
   {
      m_limitedFeatureMode = limitedFeatureMode;
   }

   LightMode getLightMode() const
   {
      return m_lightMode;
   }

   void setLightMode(LightMode lightMode)
   {
      m_lightMode = lightMode;
   }

   bool getImplicitMapViewCreationFlag() const
   {
      return m_implicitMapViewCreationFlag;
   }

   void setImplicitMapViewCreationFlag(bool implicitMapViewCreationFlag)
   {
      m_implicitMapViewCreationFlag = implicitMapViewCreationFlag;
   }

   bool isTrafficRegion() const
   {
      return m_isTrafficRegion;
   }

   void setIsTrafficRegion(bool isTrafficRegion)
   {
      m_isTrafficRegion = isTrafficRegion;
   }

   const ::std::string& getNavigationUnitIdentifier() const
   {
      return m_navigationUnitIdentifier;
   }

   void setNavigationUnitIdentifier(const ::std::string& navigationUnitIdentifier)
   {
      m_navigationUnitIdentifier = navigationUnitIdentifier;
   }

   SystemType getSystemType() const
   {
      return m_systemType;
   }

   void setSystemType(SystemType systemType)
   {
      m_systemType = systemType;
   }

   TCUStatus getTCUStatus() const
   {
      return m_tcuStatus;
   }

   void setTCUStatus(TCUStatus tcuStatus)
   {
      m_tcuStatus = tcuStatus;
   }

   SystemState getSystemState() const
   {
      return m_systemState;
   }

   void setSystemState(SystemState SystemState)
   {
      m_systemState = SystemState;
   }

   const GeoCoordinateDegree& getDefaultPosition() const
   {
      return m_defaultPosition;
   }

   void setDefaultPosition(const GeoCoordinateDegree& defaultPosition)
   {
      m_defaultPosition = defaultPosition;
   }

   float getDefaultPositionHeading() const
   {
      return m_defaultPositionHeading;
   }

   void setDefaultPositionHeading(float defaultPositionHeading)
   {
      m_defaultPositionHeading = defaultPositionHeading;
   }

   int32_t getDisplayTimeOffset() const
   {
      return m_displayTimeOffset;
   }

   void setDisplayTimeOffset(int32_t displayTimeOffset)
   {
      m_displayTimeOffset = displayTimeOffset;
   }

   TimeZoneAndDSTConsideration getTimeZoneAndDSTConsideration() const
   {
      return m_timeZoneAndDSTConsideration;
   }

   void setTimeZoneAndDSTConsideration(TimeZoneAndDSTConsideration timeZoneAndDSTConsideration)
   {
      m_timeZoneAndDSTConsideration = timeZoneAndDSTConsideration;
   }

   const ::navmiddleware::settings::VehicleProfile& getDefaultVehicleProfile() const
   {
      return m_defaultVehicleProfile;
   }

   void setDefaultVehicleProfile(const ::navmiddleware::settings::VehicleProfile& vehicleProfile)
   {
      m_defaultVehicleProfile = vehicleProfile;
   }

   const ::std::vector< ::std::pair< ::std::string, ::std::string> >& getGenericConfiguration() const
   {
      return m_genericConfiguration;
   }

   void setGenericConfiguration(const ::std::vector< ::std::pair< ::std::string, ::std::string> >& genericConfiguration)
   {
      m_genericConfiguration = genericConfiguration;
   }

   SystemPowerMode getSystemPowerMode() const
   {
      return m_systemPowerMode;
   }

   void setSystemPowerMode(SystemPowerMode systemPowerMode)
   {
      m_systemPowerMode = systemPowerMode;
   }

   void setDeviceVariant(DeviceVariant deviceVariant)
   {
      m_deviceVariant = deviceVariant;
   }

   DeviceVariant getDeviceVariant() const
   {
      return m_deviceVariant;
   }

   void setPrivacyMode(PrivacyMode privacyMode)
   {
      m_privacyMode = privacyMode;
   }

   PrivacyMode getPrivacyMode() const
   {
      return m_privacyMode;
   }

   void setLocationInputEntryType(LocationInputEntryType locationInputEntryType)
   {
      m_locationInputEntryType = locationInputEntryType;
   }

   LocationInputEntryType getLocationInputEntryType() const
   {
      return m_locationInputEntryType;
   }

   void setVehicleIdentificationNumber(const ::std::string& vehicleIdentificationNumber)
   {
      m_vehicleIdentificationNumber = vehicleIdentificationNumber;
   }

   const ::std::string& getVehicleIdentificationNumber() const
   {
      return(m_vehicleIdentificationNumber);
   }

   void setFotaBearerType(FotaBearerType fotaBearerType)
   {
      m_fotaBearerType = fotaBearerType;
   }

   FotaBearerType getFotaBearerType() const
   {
      return m_fotaBearerType;
   }

   void setCompatibleChargingConfigurationInfo(const CompatibleChargingConfigurationInfo& compatibleChargingConfigurationInfo)
   {
      m_compatibleChargingConfigurationInfo = compatibleChargingConfigurationInfo;
   }

   const CompatibleChargingConfigurationInfo& getCompatibleChargingConfigurationInfo() const
   {
      return m_compatibleChargingConfigurationInfo;
   }

   void setEvHvacType(EvHvacType evHvacType)
   {
      m_evHvacType = evHvacType;
   }

   EvHvacType getEvHvacType() const
   {
      return m_evHvacType;
   }

   void setEvAllianceReachAreaType(EvAllianceReachAreaType evAllianceReachAreaType)
   {
      m_evAllianceReachAreaType = evAllianceReachAreaType;
   }

   EvAllianceReachAreaType getEvAllianceReachAreaType() const
   {
      return m_evAllianceReachAreaType;
   }

   void setTrafficSignRecognitionSupported(bool isSupported)
   {
      m_isTrafficSignRecognitionSupported = isSupported;
   }

   bool isTrafficSignRecognitionSupported() const
   {
      return m_isTrafficSignRecognitionSupported;
   }

   void setProductionDate( const DateTime& productiondate)
   {
      m_productionDateTime = productiondate;
   }

   const DateTime& getProductionDate() const
   {
      return(m_productionDateTime);
   }

   //<-- INF4CV
   const ResidentialOrLocalMode& getResidentialOrLocalMode() const
   {
      return m_residentialOrLocalMode;
   }

   void setResidentialOrLocalMode(const ResidentialOrLocalMode& residentialOrLocalMode)
   {
      m_residentialOrLocalMode = residentialOrLocalMode;
   }
   //->

   ::std::string toString() const
   {
      ::std::stringstream genericConfigurationString;
      for (::std::vector< ::std::pair< ::std::string, ::std::string> > ::const_iterator it = m_genericConfiguration.begin(); it != m_genericConfiguration.end(); ++it)
      {
         genericConfigurationString << "Key: " << it->first << " Value: " << it->second << ::std::endl;
      }
      ::std::stringstream stream("ConfigurationInfos payload:\n");
      stream << ::std::endl
             << "Software Variant = " << m_softwareVariant << ::std::endl
             << "Navigation Variant = "<< toString(m_navVariant) << ::std::endl
             << "Screen Dimensions - Height = " << m_screenSize.height << ", "<< "Width = " << m_screenSize.width << ", " << "Display aspect ratio = "
             << m_screenSize.displayAspectRatio << ::std::endl
             << "User Profile = " << m_userProfile << ::std::endl
             << "Language Codes as String: HMI language code = " << m_hmiLanguageCodeAsString << ", "<< "SDS language code = " << m_sdsLanguageCodeAsString
             << ", " << "Cluster language code = " << m_clusterLanguageCodeAsString << ::std::endl
             << "Language Codes as Enum : HMI language code = " << m_hmiLanguageCode << ", " << "SDS language code = " << m_sdsLanguageCode
             << ", " << "Cluster language code = " << m_clusterLanguageCode << ::std::endl
             << "Distance Unit type = " << m_distanceUnitSystem << ::std::endl
             << "Date format = " << dateFormatToString(m_dateFormat) << ::std::endl
             << "Time format = " << timeFormatToString(m_timeFormat) << ::std::endl
             << "Region type = " << m_regionType << ::std::endl
             << "Vehicle type = " << m_vehicleType << ::std::endl
             << "Vehicle mode type = " << toString(m_vehicleModeType) << ::std::endl
             << "Vehicle fuel type = " << toString(m_vehicleFuelType) << ::std::endl
             << "Vehicle drive train = " << toString(m_vehicleDriveTrain) << ::std::endl
             << "Vehicle brand type = " << toString(m_vehicleBrandType) << ::std::endl
             << "Limited Features active = " << m_limitedFeatureMode.m_isActive <<::std::endl
             << "Light Mode  = " << m_lightMode <<::std::endl
             << "Navigation Unit Identifier  = " << m_navigationUnitIdentifier <<::std::endl
             << "System power mode = " << toString(m_systemPowerMode) << ::std::endl
             << "Time offset between display time and system time = " << m_displayTimeOffset << ::std::endl
             << "Time zone and DST offset considered for ETA calculation = " << m_timeZoneAndDSTConsideration << ::std::endl
             << "MW responsible for view creation on start-up = " << m_implicitMapViewCreationFlag << ::std::endl
             << "Is Traffic region = " << m_isTrafficRegion << ::std::endl
             << "SystemType = " << toString(m_systemType) << ::std::endl
             << "TCU Status = " << toString(m_tcuStatus) << ::std::endl
             << "SystemState = " << toString(m_systemState) << ::std::endl
             << "Device Variant = " << toString(m_deviceVariant) << ::std::endl
             << "Generic configuration {: " << genericConfigurationString.str() << "}" << ::std::endl
             << "Privacy mode = " << toString(m_privacyMode) << ::std::endl
             << "Location Input Entry Type = " << toString(m_locationInputEntryType) << ::std::endl
             << "Vehicle Identification Number = " << m_vehicleIdentificationNumber << ::std::endl
             << "Fota Bearer Type = " << toString(m_fotaBearerType) << ::std::endl
             << "CompatibleChargingConfigurationInfo = " << m_compatibleChargingConfigurationInfo.toString() << ::std::endl
             << "EV_HVAC Type = " << toString(m_evHvacType) << ::std::endl
             << "EV Alliance Reach Area Type = " << toString(m_evAllianceReachAreaType) << ::std::endl
             << "Is traffic sign recognition supported: "<< (m_isTrafficSignRecognitionSupported ? "true": "false") << ::std::endl
             << "Production Date = " <<toString(m_productionDateTime)<< ::std::endl
             << "Residential or local mode: "<< m_residentialOrLocalMode.toString() << ::std::endl;   //<-- INF4CV
      return stream.str();
   }

private:

   ::std::string toString(NavVariant navVariant) const
   {
      switch ( navVariant )
      {
         case NAV_VARIANT__FULL_NAV:
            return "NAV_VARIANT__FULL_NAV";
         case NAV_VARIANT__FAKE_NAV:
            return "NAV_VARIANT__FAKE_NAV";
      }
      ::std::stringstream stream;
      stream << "NAV_VARIANT__" << static_cast<unsigned int>(navVariant) << ">";
      return stream.str();
   };

   ::std::string dateFormatToString(DateFormat dateFormat) const
   {
      ::std::string dateFormatString("DATEFORMAT_");
      switch( dateFormat )
      {
      case DATEFORMAT_UNSET:
         return( dateFormatString.append("UNSET"));
      case DATEFORMAT_DD_MM_YYYY_SLASH:
         return( dateFormatString.append("DD_MM_YYYY_SLASH"));
      case DATEFORMAT_MM_DD_YYYY_SLASH:
         return( dateFormatString.append("MM_DD_YYYY_SLASH"));
      case DATEFORMAT_YYYY_MM_DD_SLASH:
         return( dateFormatString.append("YYYY_MM_DD_SLASH"));
      case DATEFORMAT_DD_MM_YYYY_DASH:
         return( dateFormatString.append("DD_MM_YYYY_DASH"));
      case DATEFORMAT_MM_DD_YYYY_DASH:
         return( dateFormatString.append("MM_DD_YYYY_DASH"));
      case DATEFORMAT_YYYY_MM_DD_DASH:
         return( dateFormatString.append("YYYY_MM_DD_DASH"));
      case DATEFORMAT_DD_MM_YYYY_DOT:
         return( dateFormatString.append("DD_MM_YYYY_DOT"));
      case DATEFORMAT_YYYY_MM_DD_JPN:
         return( dateFormatString.append("YYYY_MM_DD_JPN"));
      case DATEFORMAT_YYYY_MM_DD_KOR:
         return( dateFormatString.append("YYYY_MM_DD_KOR"));
      default:
         return( dateFormatString.append(" not mapped"));
      }
   }

   ::std::string timeFormatToString(TimeFormat timeFormat) const
   {
      ::std::string timeFormatString = "TIMEFORMAT_";
      switch( timeFormat )
      {
      case TIMEFORMAT_UNSET:
         return( timeFormatString.append("UNSET"));
      case TIMEFORMAT_12HOUR:
         return( timeFormatString.append("12HOUR"));
      case TIMEFORMAT_24HOUR:
         return( timeFormatString.append("24HOUR"));
      default:
         return( timeFormatString.append(" not mapped"));
      }
   }

   ::std::string toString(VehicleModeType vehicleModeType) const
   {
      switch ( vehicleModeType )
      {
         case VEHICLE_MODE_TYPE__UNKNOWN:
            return "VEHICLE_MODE_TYPE__UNKNOWN";
         case VEHICLE_MODE_TYPE__ICE:
            return "VEHICLE_MODE_TYPE__ICE";
         case VEHICLE_MODE_TYPE__EV:
            return "VEHICLE_MODE_TYPE__EV";
         case VEHICLE_MODE_TYPE__HEV:
            return "VEHICLE_MODE_TYPE__HEV";
         case VEHICLE_MODE_TYPE__PHEV:
            return "VEHICLE_MODE_TYPE__PHEV";
      }
       ::std::stringstream stream;
       stream << "VEHICLE_MODE_TYPE__" << static_cast<unsigned int>(vehicleModeType) << ">";
       return stream.str();
   };

   ::std::string toString(VehicleFuelType vehicleFuelType) const
   {
      switch (vehicleFuelType)
      {
      case VEHICLE_FUEL_TYPE__UNKNOWN:
         return "VEHICLE_FUEL_TYPE__UNKNOWN";
      case VEHICLE_FUEL_TYPE__PETROL:
         return "VEHICLE_FUEL_TYPE__PETROL";
      case VEHICLE_FUEL_TYPE__DIESEL:
         return "VEHICLE_FUEL_TYPE__DIESEL";
      case VEHICLE_FUEL_TYPE__HYBRID_DIESEL:
         return "VEHICLE_FUEL_TYPE__HYBRID_DIESEL";
      case VEHICLE_FUEL_TYPE__HYBRID_PETROL:
         return "VEHICLE_FUEL_TYPE__HYBRID_PETROL";
      case VEHICLE_FUEL_TYPE__ELECTRICITY:
         return "VEHICLE_FUEL_TYPE__ELECTRICITY";
      case VEHICLE_FUEL_TYPE__PLUG_IN_HYBRID:
         return "VEHICLE_FUEL_TYPE__PLUG_IN_HYBRID";
      default:
         ::std::stringstream stream;
         stream << "VEHICLE_FUEL_TYPE__<" << static_cast<unsigned int>(vehicleFuelType) << ">";
         return stream.str();
      }
   }

   ::std::string toString(VehicleDriveTrain vehicleDriveTrain) const
   {
      switch ( vehicleDriveTrain )
      {
         case VEHICLE_DRIVE_TRAIN__UNKNOWN:
            return "VEHICLE_DRIVE_TRAIN__UNKNOWN";
         case VEHICLE_DRIVE_TRAIN__FRONT_WHEEL_DRIVE:
            return "VEHICLE_DRIVE_TRAIN__FRONT_WHEEL_DRIVE";
         case VEHICLE_DRIVE_TRAIN__REAR_WHEEL_DRIVE:
            return "VEHICLE_DRIVE_TRAIN__REAR_WHEEL_DRIVE";
         case VEHICLE_DRIVE_TRAIN__ALL_WHEEL_DRIVE:
            return "VEHICLE_DRIVE_TRAIN__ALL_WHEEL_DRIVE";
      }
      ::std::stringstream stream;
      stream << "VEHICLE_DRIVE_TRAIN__" << static_cast<unsigned int>(vehicleDriveTrain) << ">";
      return stream.str();
   };

   ::std::string toString(VehicleBrandType vehicleBrandType) const
   {
      switch (vehicleBrandType)
      {
      case VEHICLE_BRAND_TYPE__UNKNOWN:
         return "VEHICLE_BRAND_TYPE__UNKNOWN";
      case VEHICLE_BRAND_TYPE__RENAULT:
         return "VEHICLE_BRAND_TYPE__RENAULT";
      case VEHICLE_BRAND_TYPE__NISSAN:
         return "VEHICLE_BRAND_TYPE__NISSAN";
      case VEHICLE_BRAND_TYPE__INFINITI:
         return "VEHICLE_BRAND_TYPE__INFINITI";
      case VEHICLE_BRAND_TYPE__DACIA:
         return "VEHICLE_BRAND_TYPE__DACIA";
      case VEHICLE_BRAND_TYPE__DAIMLER:
         return "VEHICLE_BRAND_TYPE__DAIMLER";
      case VEHICLE_BRAND_TYPE__LADA:
         return "VEHICLE_BRAND_TYPE__LADA";
      case VEHICLE_BRAND_TYPE__SMART:
         return "VEHICLE_BRAND_TYPE__SMART";
      case VEHICLE_BRAND_TYPE__MITSUBISHI:
         return "VEHICLE_BRAND_TYPE__MITSUBISHI";
      case VEHICLE_BRAND_TYPE__ALPINE:
         return "VEHICLE_BRAND_TYPE__ALPINE";
      case VEHICLE_BRAND_TYPE__RENAULT_TRUCKS:
         return "VEHICLE_BRAND_TYPE__RENAULT_TRUCKS";
      default:
         ::std::stringstream stream;
         stream << "VEHICLE_BRAND_TYPE__<" << static_cast<unsigned int>(vehicleBrandType) << ">";
         return stream.str();
      }
   }

   ::std::string toString(SystemPowerMode systemPowerMode) const
   {
      switch (systemPowerMode)
      {
      case SYSTEM_POWER_MODE__IGNITION_ON:
         return "SYSTEM_POWER_MODE__IGNITION_ON";
      case SYSTEM_POWER_MODE__IGNITION_OFF:
         return "SYSTEM_POWER_MODE__IGNITION_OFF";
      default:
         ::std::stringstream stream;
         stream << "SYSTEM_POWER_MODE__<" << static_cast<unsigned int>(systemPowerMode) << ">";
         return stream.str();
      }
   }

   ::std::string toString(SystemType systemType) const
   {
      switch (systemType)
      {
      case SYSTEM_TYPE__ENTRY:
         return "SYSTEM_TYPE__ENTRY";
      case SYSTEM_TYPE__ENTRY_ACCESS:
         return "SYSTEM_TYPE__ENTRY_ACCESS";
      case SYSTEM_TYPE__MID:
         return "SYSTEM_TYPE__MID";
      case SYSTEM_TYPE__UNKNOWN:
         return "SYSTEM_TYPE__MID";
      default:
         ::std::stringstream stream;
         stream << "SYSTEM_TYPE__<" << static_cast<unsigned int>(systemType) << ">";
         return stream.str();
      }
   }

   ::std::string toString(TCUStatus tcuStatus) const
   {
      switch (tcuStatus)
      {
      case TCU_STATUS__WITH_TCU_NOT_READY:
         return "TCU_STATUS__WITH_TCU_NOT_READY";
      case TCU_STATUS__WITH_TCU_READY:
         return "TCU_STATUS__WITH_TCU_READY";
      case TCU_STATUS__WITHOUT_TCU:
         return "TCU_STATUS__WITHOUT_TCU";
      case TCU_STATUS__UNKNOWN:
         return "TCU_STATUS__UNKNOWN";
      default:
         ::std::stringstream stream;
         stream << "TCU_Status__<" << static_cast<unsigned int>(tcuStatus) << ">";
         return stream.str();
      }
   }

   ::std::string toString(DeviceVariant deviceVariant) const
   {
      switch (deviceVariant)
      {
      case DEVICEVARIANT__LANDSCAPE:
         return "DEVICEVARIANT__LANDSCAPE";
      case DEVICEVARIANT__PORTRAIT:
         return "DEVICEVARIANT__PORTRAIT";
      default:
         ::std::stringstream stream;
         stream << "DEVICEVARIANT__<" << static_cast<unsigned int>(deviceVariant) << ">";
         return stream.str();
      }
   }

   ::std::string toString(SystemState systemState) const
   {
      switch(systemState)
      {
      case SYSTEM_STATE__BACKGROUND:
         return "SYSTEM_STATE__BACKGROUND";
      case SYSTEM_STATE__PREPARE_SHUTDOWN:
         return "SYSTEM_STATE__PREPARE_SHUTDOWN";
      case SYSTEM_STATE__SHUTDOWN:
         return "SYSTEM_STATE__SHUTDOWN";
      case SYSTEM_STATE__SUSPEND:
         return "SYSTEM_STATE__SUSPEND";
      case SYSTEM_STATE__STANDBY:
         return "SYSTEM_STATE__STANDBY";
      case SYSTEM_STATE__OFF:
         return "SYSTEM_STATE__OFF";
      case SYSTEM_STATE__DOWNLOAD:
         return "SYSTEM_STATE__DOWNLOAD";
      case SYSTEM_STATE__ON:
         return "SYSTEM_STATE__ON";
      case SYSTEM_STATE__DOOR_OPEN:
         return "SYSTEM_STATE__DOOR_OPEN";
      case SYSTEM_STATE__IGNITION:
         return "SYSTEM_STATE__IGNITION";
      case SYSTEM_STATE__DIAGNOSIS:
         return "SYSTEM_STATE__DIAGNOSIS";
      case SYSTEM_STATE__PROFILE:
         return "SYSTEM_STATE__PROFILE";
      case SYSTEM_STATE__MMI_STANDBY_PWR_SAVE_1:
         return "SYSTEM_STATE__MMI_STANDBY_PWR_SAVE_1";
      case SYSTEM_STATE__MMI_STANDBY_PWR_SAVE_2:
         return "SYSTEM_STATE__MMI_STANDBY_PWR_SAVE_2";
      case SYSTEM_STATE__MMI_STANDBY:
         return "SYSTEM_STATE__MMI_STANDBY";
      case SYSTEM_STATE__MMI_ON:
         return "SYSTEM_STATE__MMI_ON";
      case SYSTEM_STATE__MMI_STANDBY_RESCTRICTED:
         return "SYSTEM_STATE__MMI_STANDBY_RESCTRICTED";
      case SYSTEM_STATE__MMI_ON_DIAG:
         return "SYSTEM_STATE__MMI_ON_DIAG";
      case SYSTEM_STATE__MMI_ON_TEL:
         return "SYSTEM_STATE__MMI_ON_TEL";
      case SYSTEM_STATE__MMI_ON_SWDL:
         return "SYSTEM_STATE__MMI_ON_SWDL";
      case SYSTEM_STATE__MMI_STANDBY_CUSTOMER_SWDL:
         return "SYSTEM_STATE__MMI_STANDBY_CUSTOMER_SWDL";
      case SYSTEM_STATE__MMI_STANDBY_PWR_SAVE:
         return "SYSTEM_STATE__MMI_STANDBY_PWR_SAVE";
      case SYSTEM_STATE__OVER_TMP:
         return "SYSTEM_STATE__OVER_TMP";
      case SYSTEM_STATE__SAFE:
         return "SYSTEM_STATE__SAFE";
      case SYSTEM_STATE__STATE_MAX:
         return "SYSTEM_STATE__STATE_MAX";
      case SYSTEM_STATE__UNKNOWN:
         return "SYSTEM_STATE__UNKNOWN";
      default:
         ::std::stringstream stream;
         stream << "SYSTEM_STATE__<" << static_cast<unsigned int>(systemState) << ">";
         return stream.str();
      }
   }

   ::std::string toString(PrivacyMode privacyMode) const
   {
      switch(privacyMode)
      {
      case PRIVACY_MODE__UNKNOWN:
         return("PRIVACY_MODE__UNKNOWN");
      case PRIVACY_MODE__ON:
         return("PRIVACY_MODE__ON");
      case PRIVACY_MODE__OFF:
         return("PRIVACY_MODE__OFF");
      default:
         ::std::stringstream stream;
         stream << "PRIVACY_MODE__<" << static_cast<unsigned int>(privacyMode) << ">";
         return stream.str();
      }
   }

   ::std::string toString(LocationInputEntryType locationInputEntryType) const
   {
      switch(locationInputEntryType)
      {
      case LOCATION_INPUT_ENTRY_TYPE__UNKNOWN:
         return "LOCATION_INPUT_ENTRY_TYPE__UNKNOWN";
      case LOCATION_INPUT_ENTRY_TYPE__EUR:
         return "LOCATION_INPUT_ENTRY_TYPE__EUR";
      case LOCATION_INPUT_ENTRY_TYPE__NAR:
         return "LOCATION_INPUT_ENTRY_TYPE__NAR";
      case LOCATION_INPUT_ENTRY_TYPE__JPN:
         return "LOCATION_INPUT_ENTRY_TYPE__JPN";
      default:
         ::std::stringstream stream;
         stream << "LOCATION_INPUT_ENTRY_TYPE__<" << static_cast<unsigned int>(locationInputEntryType) << ">";
         return stream.str();
      }
   }

   ::std::string toString(FotaBearerType fotaBearerType) const
   {
      switch(fotaBearerType)
      {
      case FOTA_BEARER_TYPE__NOT_USE:
         return "FOTA_BEARER_TYPE__NOT_USE";
      case FOTA_BEARER_TYPE__USE:
         return "FOTA_BEARER_TYPE__USE";
      case FOTA_BEARER_TYPE__ONLY_USE_FOTA:
         return "FOTA_BEARER_TYPE__ONLY_USE_FOTA";
      case FOTA_BEARER_TYPE__ONLY_USE_MAP:
         return "FOTA_BEARER_TYPE__ONLY_USE_MAP";
      default:
         ::std::stringstream stream;
         stream << "FOTA_BEARER_TYPE__<" << static_cast<unsigned int>(fotaBearerType) << ">";
         return stream.str();
      }
   }

   ::std::string toString(EvHvacType evHvacType) const
   {
      switch (evHvacType)
      {
      case EV_HVAC_TYPE__NONE:
         return "EV_HVAC_TYPE__NONE";
      case EV_HVAC_TYPE__X12G:
         return "EV_HVAC_TYPE__X12G";
      default:
         ::std::stringstream stream;
         stream << "EV_HVAC_TYPE__<" << static_cast<unsigned int>(evHvacType) << ">";
         return stream.str();
      }
   }

   ::std::string toString(EvAllianceReachAreaType evAllianceReachAreaType) const
   {
      switch (evAllianceReachAreaType)
      {
      case EV_ALLIANCE_REACH_AREA_TYPE__PATATOID:
         return "EV_ALLIANCE_REACH_AREA_TYPE__PATATOID";
      case EV_ALLIANCE_REACH_AREA_TYPE__CIRCLE:
         return "EV_ALLIANCE_REACH_AREA_TYPE__CIRCLE";
      default:
         ::std::stringstream stream;
         stream << "EV_ALLIANCE_REACH_AREA_TYPE__<" << static_cast<unsigned int>(evAllianceReachAreaType) << ">";
         return stream.str();
      }
   }

   ::std::string toString(const DateTime& productionDateTime) const
   {
      ::std::stringstream stream;
      stream <<"Year "  <<productionDateTime.m_year
             <<"Month " <<productionDateTime.m_month
             <<"Day "   <<productionDateTime.m_day
             <<"Hour "  <<productionDateTime.m_hour
             <<"Minute "<<productionDateTime.m_minute
             <<"Second "<<productionDateTime.m_second<< ::std::endl;
      return stream.str();
   }

   SoftwareVariant      m_softwareVariant;
   NavVariant           m_navVariant;
   ScreenDimensions     m_screenSize;
   UserProfile          m_userProfile;
   ::std::string        m_hmiLanguageCodeAsString;
   ::std::string        m_sdsLanguageCodeAsString;
   ::std::string        m_trafficTextLanguageCodeAsString;
   ::std::string        m_clusterLanguageCodeAsString;
   LanguageCode         m_hmiLanguageCode;
   LanguageCode         m_sdsLanguageCode;
   LanguageCode         m_trafficTextLanguageCode;
   LanguageCode         m_clusterLanguageCode;
   DistanceUnitSystem   m_distanceUnitSystem;
   DateFormat           m_dateFormat;
   TimeFormat           m_timeFormat;
   RegionType           m_regionType;
   VehicleType          m_vehicleType;
   VehicleModeType      m_vehicleModeType;
   VehicleBrandType     m_vehicleBrandType;
   VehicleFuelType      m_vehicleFuelType;
   VehicleDriveTrain    m_vehicleDriveTrain;
   LimitedFeatureMode   m_limitedFeatureMode;
   LightMode            m_lightMode;
   ::std::string        m_navigationUnitIdentifier;
   ::navmiddleware::settings::VehicleProfile m_defaultVehicleProfile;
   SystemPowerMode      m_systemPowerMode;
   SystemType           m_systemType;
   TCUStatus            m_tcuStatus;
   SystemState          m_systemState;
   DeviceVariant        m_deviceVariant;

   /**
    * startup position after factory reset
    */
   GeoCoordinateDegree  m_defaultPosition;
   float                m_defaultPositionHeading;

   // m_displayTimeOffset - Offset value between display time and system time in seconds
   int32_t              m_displayTimeOffset;

   // m_timeZoneAndDSTConsideration - This value gives information on whether UtcOffset difference between
   // start position and destination should be considered while calculating ETA.
   TimeZoneAndDSTConsideration m_timeZoneAndDSTConsideration;
   /*
    * The HMI is responsible for setting this flag.
    * If it is set to true, then MW is responsible for reading the settings and creating the views on start-up.
    * If it is set to false, then HMI is responsible for reading the settings and creating the views on start-up.
    */
   bool                 m_implicitMapViewCreationFlag;

   /*
    * This value is set by HMI. It tells whether the current region is a traffic or a non-traffic region.
    * For a non-traffic region, traffic information will be deactivated from MW side.
    */
   bool                 m_isTrafficRegion;

   /**
    * key-value structure to provide arbitrary values to be used/agreed between HMI and MW
    */
   ::std::vector< ::std::pair< ::std::string, ::std::string> > m_genericConfiguration;

   /**
    * This value is set by HMI, to indicate that privacy mode is activated or not.
    */
   PrivacyMode m_privacyMode;

   /**
    * This value is set by HMI, to indicate the input screen layout used for Location Input.
    */
   LocationInputEntryType m_locationInputEntryType ;
   ::std::string        m_vehicleIdentificationNumber;

   /**
    * This value is set by HMI, to indicate fota bearer type configured for the region.
    */
   FotaBearerType         m_fotaBearerType;

   /**
    * KDS configuration for Vehicle compatible Charging information
    */
   CompatibleChargingConfigurationInfo m_compatibleChargingConfigurationInfo;

   /**
    * This value is set by HMI to indicate if HVAC status bar will be shown.
    */
   EvHvacType             m_evHvacType;
   /**
    * KDS configuration for Electric Vehicle to configure Reachable Area Map as circle or patatoid/polygon
    */
   EvAllianceReachAreaType   m_evAllianceReachAreaType;
   /*
    * The HMI is responsible for setting this flag.
    * To be set to true if traffic sign recognition is supported.
    * Default values for speed limit settings depends on availability of traffic sign recognition.
    */
   bool                 m_isTrafficSignRecognitionSupported;
   /**
    * Production date time refers to the time when the target was flashed with IVI software in the production unit.
    * The date time is local to the location where production was done.
    */
   DateTime             m_productionDateTime;
   //<-- INF4CV
   /**
    * This value is set by HMI to indicate the black list countries for which LCV/Truck dimension
    * and load prohibitions (which are combined with AUTHORIZATION attribute having relevance for Residential/Delivery/Local Vehicles)
    * will NOT be ignored.
    */
   ResidentialOrLocalMode m_residentialOrLocalMode;
   //->
};

class UserProfileActionInfos
{
public:
   enum UserProfileActionOperation
   {
      USER_PROFILE_ACTION_OPERATION__UNKNOWN,
      USER_PROFILE_ACTION_OPERATION__COPY,
      USER_PROFILE_ACTION_OPERATION__DELETE,
      USER_PROFILE_ACTION_OPERATION__SET
   };

   inline ::std::string toString(UserProfileActionOperation operation) const
   {
      switch (operation)
      {
      case USER_PROFILE_ACTION_OPERATION__UNKNOWN:
         return("USER_PROFILE_ACTION_OPERATION__UNKNOWN");
      case USER_PROFILE_ACTION_OPERATION__COPY:
         return("USER_PROFILE_ACTION_OPERATION__COPY");
      case USER_PROFILE_ACTION_OPERATION__DELETE:
         return("USER_PROFILE_ACTION_OPERATION__DELETE");
      case USER_PROFILE_ACTION_OPERATION__SET:
         return("USER_PROFILE_ACTION_OPERATION__SET");
      default:
         ::std::stringstream stream;
         stream << "UserProfileActionOperation not mapped: " << static_cast<unsigned int>(operation);
         return stream.str();
      }
   }

   enum UserProfileActionStatus
   {
      USER_PROFILE_ACTION_STATUS__UNKNOWN,
      USER_PROFILE_ACTION_STATUS__OK,
      USER_PROFILE_ACTION_STATUS__FAILURE
   };

   inline ::std::string toString(UserProfileActionStatus status) const
   {
      switch (status)
      {
      case USER_PROFILE_ACTION_STATUS__UNKNOWN:
         return("USER_PROFILE_ACTION_STATUS__UNKNOWN");
      case USER_PROFILE_ACTION_STATUS__OK:
         return("USER_PROFILE_ACTION_STATUS__OK");
      case USER_PROFILE_ACTION_STATUS__FAILURE:
         return("USER_PROFILE_ACTION_STATUS__FAILURE");
      default:
         ::std::stringstream stream;
         stream << "UserProfileActionStatus not mapped: " << static_cast<unsigned int>(status);
         return stream.str();
      }
   }

   UserProfileActionInfos()
      : m_userProfile(::navmiddleware::ConfigurationInfos::USER_PROFILE__ANONYMOUS),
        m_action(USER_PROFILE_ACTION_OPERATION__UNKNOWN),
        m_status(USER_PROFILE_ACTION_STATUS__UNKNOWN) {}

   UserProfileActionInfos(::navmiddleware::ConfigurationInfos::UserProfile userProfile,
                          UserProfileActionOperation action,
                          UserProfileActionStatus status)
      : m_userProfile(userProfile),
        m_action(action),
        m_status(status) {}

   ::std::string toString() const
   {
      ::std::stringstream stream("UserProfileActionInfos payload:\n");

      stream << ::std::endl
             << "UserProfile = " << m_userProfile << ::std::endl
             << "UserProfileActionOperation = " << toString(m_action) << ::std::endl
             << "UserProfileActionStatus = " << toString(m_status) << ::std::endl
             << ::std::endl;

      return stream.str();
   }

   UserProfileActionOperation getAction() const
   {
      return m_action;
   }

   void setAction(UserProfileActionOperation action)
   {
      m_action = action;
   }

   UserProfileActionStatus getStatus() const
   {
      return m_status;
   }

   void setStatus(UserProfileActionStatus status)
   {
      m_status = status;
   }

   ::navmiddleware::ConfigurationInfos::UserProfile getUserProfile() const
   {
      return m_userProfile;
   }

   void setUserProfile(::navmiddleware::ConfigurationInfos::UserProfile userProfile)
   {
      m_userProfile = userProfile;
   }

private:
   ::navmiddleware::ConfigurationInfos::UserProfile m_userProfile;
   UserProfileActionOperation                       m_action;
   UserProfileActionStatus                          m_status;
};

class AvoidAreaInfo
{
public:

   enum AvoidAreaSize
   {
      AVOID_AREA_SIZE__LARGE,
      AVOID_AREA_SIZE__MEDIUM_LARGE,
      AVOID_AREA_SIZE__MEDIUM,
      AVOID_AREA_SIZE__SMALL_MEDIUM,
      AVOID_AREA_SIZE__SMALL
   };

   enum AvoidAreaOptions
   {
      AVOID_AREA_OPTIONS__NONE/* motorways will also be avoided */,
      AVOID_AREA_OPTIONS__USE_MOTORWAYS
   };

public:
   AvoidAreaInfo(AvoidAreaId id, const ::std::string& name, const ::navmiddleware::GeoCoordinateDegree& geoCoordinate,
                 AvoidAreaSize size, uint32_t scale, uint32_t edgeLength, AvoidAreaOptions avoidAreaOptions )
      :m_id(id),
       m_avoidAreaName(name),
       m_avoidAreaSize(size),
       m_avoidAreaMapScale(scale),
       m_avoidAreaEdgeLength(edgeLength),
       m_avoidAreaOptions(avoidAreaOptions),
       m_centerGeoCoordinate(geoCoordinate)
   {}
   AvoidAreaInfo()
      :m_id(0),
       m_avoidAreaName(""),
       m_avoidAreaSize(AVOID_AREA_SIZE__LARGE),
       m_avoidAreaMapScale(0),
       m_avoidAreaEdgeLength(0),
       m_avoidAreaOptions(AVOID_AREA_OPTIONS__USE_MOTORWAYS),
       m_centerGeoCoordinate()
   {}

   AvoidAreaId getId() const
   {
      return m_id;
   }

   void setName(const ::std::string& name)
   {
      m_avoidAreaName = name;
   }

   const ::std::string& getName() const
   {
      return m_avoidAreaName;
   }

   void setSize(AvoidAreaSize size)
   {
      m_avoidAreaSize = size;
   }

   AvoidAreaSize getSize() const
   {
      return m_avoidAreaSize;
   }

   void setScale(uint32_t scale)
   {
      m_avoidAreaMapScale = scale;
   }

   uint32_t getScale() const
   {
      return m_avoidAreaMapScale;
   }

   void setEdgeLength(uint32_t edgeLength)
   {
      m_avoidAreaEdgeLength = edgeLength;
   }

   uint32_t getEdgeLength() const
   {
      return m_avoidAreaEdgeLength;
   }

   void setAvoidAreaOptions(AvoidAreaOptions avoidAreaOptions)
   {
      m_avoidAreaOptions = avoidAreaOptions;
   }

   AvoidAreaOptions getAvoidAreaOptions() const
   {
      return m_avoidAreaOptions;
   }

   void setCenterGeoCoordinate (const ::navmiddleware::GeoCoordinateDegree& coord)
   {
      m_centerGeoCoordinate = coord;
   }

   const ::navmiddleware::GeoCoordinateDegree& getCenterGeoCoordinate() const
   {
      return m_centerGeoCoordinate;
   }

   bool operator==(const AvoidAreaInfo& rhs ) const
   {
      return m_id == rhs.m_id  &&
             m_avoidAreaSize == rhs.m_avoidAreaSize &&
             m_avoidAreaName == rhs.m_avoidAreaName &&
             m_avoidAreaMapScale == rhs.m_avoidAreaMapScale &&
             m_avoidAreaOptions  == rhs.m_avoidAreaOptions &&
             m_avoidAreaEdgeLength == rhs.m_avoidAreaEdgeLength &&
             m_centerGeoCoordinate == rhs.m_centerGeoCoordinate;
   }

   bool operator!=(const AvoidAreaInfo& rhs ) const
   {
      return !(*this == rhs);
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("AvoidAreaInfo payload:\n");

      stream << ::std::endl
             << "Avoid Area Id = " << m_id << ::std::endl
             << "Avoid Area Name = " << m_avoidAreaName << ::std::endl
             << "Avoid Area size = " << m_avoidAreaSize << ::std::endl
             << "Avoid Area scale = " << m_avoidAreaMapScale << ::std::endl
             << "Avoid Area edgeLength = " << m_avoidAreaEdgeLength << ::std::endl
             << "Avoid Area avoidAreaOptions = " << m_avoidAreaOptions << ::std::endl
             << "Avoid Area geoCoordinateDegree = " << m_centerGeoCoordinate.toString()
             << ::std::endl;

      return stream.str();
   }

private:
   AvoidAreaId                           m_id;
   ::std::string                         m_avoidAreaName;
   AvoidAreaSize                         m_avoidAreaSize;
   uint32_t                              m_avoidAreaMapScale;
   uint32_t                              m_avoidAreaEdgeLength;
   AvoidAreaOptions                      m_avoidAreaOptions;
   ::navmiddleware::GeoCoordinateDegree  m_centerGeoCoordinate;
};

class AvoidAreaInfos
{
public:

   const ::std::vector<AvoidAreaInfo>& getAvoidAreaInfos() const
   {
      return m_avoidAreaInfoList;
   }
   ::std::vector<AvoidAreaInfo>& getAvoidAreaInfos()
   {
      return m_avoidAreaInfoList;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("AvoidAreaInfos payload:\n");
      stream << ::std::endl;

      for(unsigned int index = 0; index < m_avoidAreaInfoList.size(); index++)
      {
         stream << "AvoidAreaInfo list Index = " << index << ::std::endl
                << "AvoidAreaInfo Info =" << m_avoidAreaInfoList[index].toString() << ::std::endl;
      }
      return stream.str();
   }

private:
   ::std::vector<AvoidAreaInfo> m_avoidAreaInfoList;
};

//Configuration manager
enum ConfigItemType
{
   CONFIG_ITEM_TYPE__UNKNOWN = 0,
   CONFIG_ITEM_TYPE__PRIVATE = 1,
   CONFIG_ITEM_TYPE__PUBLIC = 2
};

inline ::std::string toString(ConfigItemType type)
{
   switch (type)
   {
   case CONFIG_ITEM_TYPE__UNKNOWN:
      return("CONFIG_ITEM_TYPE__UNKNOWN");
   case CONFIG_ITEM_TYPE__PRIVATE:
      return("CONFIG_ITEM_TYPE__PRIVATE");
   case CONFIG_ITEM_TYPE__PUBLIC:
      return("CONFIG_ITEM_TYPE__PUBLIC");
   default:
      return("ConfigItemType not mapped");
   }
}

enum ConfigResultCode
{
   CONFIG_RESULT_CODE__UNKNOWN = 0,
   CONFIG_RESULT_CODE__SUCCESS = 1,
   CONFIG_RESULT_CODE__FAILED = 2,
   CONFIG_RESULT_CODE__UNKNOWN_GROUP_NAME = 3
};

inline ::std::string toString(ConfigResultCode resultCode)
{
   switch (resultCode)
   {
   case CONFIG_RESULT_CODE__UNKNOWN:
      return("CONFIG_RESULT_CODE__UNKNOWN");
   case CONFIG_RESULT_CODE__SUCCESS:
      return("CONFIG_RESULT_CODE__SUCCESS");
   case CONFIG_RESULT_CODE__FAILED:
      return("CONFIG_RESULT_CODE__FAILED");
   case CONFIG_RESULT_CODE__UNKNOWN_GROUP_NAME:
      return("CONFIG_RESULT_CODE__UNKNOWN_GROUP_NAME");
   default:
      return("ConfigResultCode not mapped");
   }
}

class ConfigItem
{
public:
   ConfigItem()
      : m_key()
      , m_value()
   {}

   ConfigItem(const ::std::string& key, const ::std::string& value)
      : m_key(key)
      , m_value(value)
   {}

   virtual ~ConfigItem()
   {}

   const ::std::string& getKey() const
   {
      return(m_key);
   }

   void setKey(const ::std::string& key)
   {
      m_key = key;
   }

   const ::std::string& getValue() const
   {
      return(m_value);
   }

   void setValue(const ::std::string& value)
   {
      m_value = value;
   }

   ::std::string toString() const
   {
      ::std::stringstream resultString;
      resultString << "Key:" << m_key << ", Value:" << m_value << ::std::endl;
      return(resultString.str());
   }

private:
   ::std::string m_key;
   ::std::string m_value;
};

typedef ::std::vector< ConfigItem> ConfigItemList;

class ConfigItemsGetResult
{
public:
   ConfigItemsGetResult()
      : m_resultCode(CONFIG_RESULT_CODE__UNKNOWN)
   {}

   ConfigItemsGetResult(ConfigResultCode resultCode, const ConfigItemList& configItems)
      : m_resultCode(resultCode)
      , m_configItems(configItems)
   {}

   virtual ~ConfigItemsGetResult()
   {}


   const ::std::vector<ConfigItem>& getConfigItems() const
   {
      return m_configItems;
   }

   void setConfigItems(const ::std::vector<ConfigItem>& configItems)
   {
      m_configItems = configItems;
   }

   ConfigResultCode getResultCode() const
   {
      return m_resultCode;
   }

   void setResultCode(ConfigResultCode resultCode)
   {
      m_resultCode = resultCode;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream("ConfigItemsGetResult payload:\n");
      stream << ::navmiddleware::toString(m_resultCode) << ::std::endl;
      for (::std::vector< ConfigItem>::const_iterator it = m_configItems.begin(); it != m_configItems.end(); ++it)
      {
         stream << it->toString();
      }
      return(stream.str());
   }

private:
   ConfigResultCode m_resultCode;
   ConfigItemList m_configItems;
};

class RegisteredConfigItems
{
public:
   RegisteredConfigItems()
      : m_configItemType(CONFIG_ITEM_TYPE__UNKNOWN)
   {}

   RegisteredConfigItems(ConfigItemType configItemType, const ::std::vector< ::std::string>& configItemGroupNames)
      : m_configItemType(configItemType)
      , m_configItemGroupNames(configItemGroupNames)
   {}

   virtual ~RegisteredConfigItems()
   {}

   const ::std::vector< ::std::string>& getConfigItemGroupNames() const
   {
      return m_configItemGroupNames;
   }

   void setConfigItemGroupNames(const ::std::vector< ::std::string>& configItemGroupNames)
   {
      m_configItemGroupNames = configItemGroupNames;
   }

   ConfigItemType getConfigItemType() const
   {
      return m_configItemType;
   }

   void setConfigItemType(ConfigItemType configItemType)
   {
      m_configItemType = configItemType;
   }

   ::std::string toString() const
   {
      ::std::stringstream stream;
      stream << "ConfigItemType: " << ::navmiddleware::toString(m_configItemType) << ::std::endl;
      for (::std::vector< ::std::string>::const_iterator it = m_configItemGroupNames.begin(); it != m_configItemGroupNames.end(); ++it)
      {
         stream << "ConfigItemGroupNames: " << *it << ::std::endl;
      }
      return(stream.str());
   }

private:
   ConfigItemType m_configItemType;
   ::std::vector< ::std::string > m_configItemGroupNames;
};

class RegisteredConfigItemsList
{
public:
   RegisteredConfigItemsList()
   {}

   explicit RegisteredConfigItemsList(const ::std::vector< RegisteredConfigItems>& registeredConfigItemsList)
      : m_registeredConfigItemsList(registeredConfigItemsList)
   {}

   virtual ~RegisteredConfigItemsList()
   {}

   const ::std::vector<RegisteredConfigItems>& getRegisteredConfigItemsList() const
   {
      return m_registeredConfigItemsList;
   }

   void setRegisteredConfigItemsList(const ::std::vector<RegisteredConfigItems>& registeredConfigItemsList)
   {
      m_registeredConfigItemsList = registeredConfigItemsList;
   }
   ::std::string toString() const
   {
      ::std::stringstream stream("RegisteredConfigItemsList payload:\n");
      for ( ::std::vector< RegisteredConfigItems>::const_iterator it = m_registeredConfigItemsList.begin();
            it != m_registeredConfigItemsList.end(); ++it)
      {
         stream << it->toString();
      }
      return(stream.str());
   }

private:
   ::std::vector< RegisteredConfigItems> m_registeredConfigItemsList;
};

} //namespace navmiddleware

#endif  // PRES_CTRL_AIVI_PRES_CTRL_SRC_NAVMIDDLEWARE_INFO_CONFIGURATIONINFOS_H_
