/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     MemoryStream.h
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2012 Robert Bosch Car Multimedia GmbH
 *
 *\par History:
 * See history of revision control system
 ***************************************************************************/

#ifndef ASF_STREAM_MEMORYSTREAM_H
#define ASF_STREAM_MEMORYSTREAM_H

#include <string>
#include "asf/core/Logger.h"
#include "asf/core/Types.h"

#define STREAM_CHUNKSIZE 2048  // 16

namespace asf {
namespace cca {

class CcaMostStream;
}
}  // namespace asf

namespace asf {
namespace stream {

class MemoryStream {
    /*The friend relationship is needed here, because of the take() and release()
     * methods of CcaStreamMost which modify the inner members of MemoryStream taking
     * over the ownership of the stream.
     */
    friend class ::asf::cca::CcaMostStream;

public:
    MemoryStream(size_t chunkSize = STREAM_CHUNKSIZE);

    MemoryStream(uint8* stream_, size_t bufferSize);

    virtual ~MemoryStream();

    virtual uint8* getStream() const;

    virtual size_t getBufferSize() const { return _bufferSize; }

    size_t getReadIndex() const { return _readIndex; }

    void setReadIndex(size_t readIndex) {
        LOG_ASSERT(readIndex <= _usedSize);
        _readIndex = readIndex;
    }

    void rewindReadIndex() { setReadIndex(0); }

    size_t getSize() const;

    MemoryStream& operator<<(const uint8& u8Value);

    MemoryStream& operator<<(const uint16& u16Value);

    MemoryStream& operator<<(const uint32& u32Value);

    MemoryStream& operator<<(const uint64& u64Value);

    MemoryStream& operator<<(const double& dValue);

    MemoryStream& operator<<(const float& dValue);

    MemoryStream& operator<<(const int8& s8Value) { return operator<<((const uint8&)s8Value); }

    MemoryStream& operator<<(const int16& s16Value) { return operator<<((const uint16&)s16Value); }

    MemoryStream& operator<<(const int32& s32Value) { return operator<<((const uint32&)s32Value); }

    MemoryStream& operator<<(const int64& s64Value) { return operator<<((const uint64&)s64Value); }

    MemoryStream& operator<<(const bool& bValue) { return operator<<((const uint8&)bValue); }

    MemoryStream& operator>>(uint8& u8Value);

    MemoryStream& operator>>(uint16& u16Value);

    MemoryStream& operator>>(uint32& u32Value);

    MemoryStream& operator>>(uint64& u64Value);

    MemoryStream& operator>>(double& dValue);

    MemoryStream& operator>>(float& dValue);

    MemoryStream& operator>>(int8& s8Value) { return operator>>((uint8&)s8Value); }

    MemoryStream& operator>>(int16& s16Value) { return operator>>((uint16&)s16Value); }

    MemoryStream& operator>>(int32& s32Value) { return operator>>((uint32&)s32Value); }

    MemoryStream& operator>>(int64& s64Value) { return operator>>((uint64&)s64Value); }

    MemoryStream& operator>>(bool& bValue) { return operator>>((uint8&)bValue); }

    void writeString(const char* szString);

    void readString(char* szString, uint16 u16Size);

    void writeBuffer(const void* buffer, size_t length);

    void readBuffer(void* buffer, size_t length);

    MemoryStream& operator<<(const std::string& str);

    MemoryStream& operator>>(std::string& str);

    void deleteStream();

private:
    MemoryStream(const MemoryStream&);

    MemoryStream& operator=(const MemoryStream&);

    DECLARE_CLASS_LOGGER();

    uint8* pCheckStreamSizeAndAlloc(size_t numberOfBytes);

    size_t _readIndex;

    size_t _bufferSize;

    size_t _usedSize;

    size_t _chunkSize;

    uint8* _stream;

    bool _bIsInternalStream;
};

class SerializableIF {
public:
    virtual ~SerializableIF() {}

    virtual void serialize(MemoryStream& rfOutStream) = 0;

    virtual void deserialize(MemoryStream& rfInStream) = 0;

    virtual size_t getSize() = 0;
};

MemoryStream& operator<<(MemoryStream& rfOutStream, SerializableIF& rfSerial);

MemoryStream& operator>>(MemoryStream& rfInStream, SerializableIF& rfSerial);

}  // namespace stream
}  // namespace asf

#endif  // ASF_STREAM_MEMORYSTREAM_H
