/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     DBusStubRegistry.h
 *\brief
 *
 *\author   CM-AI/PJ-G31
 *          stefan.baron3@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2012 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/

#ifndef ASF_DBUS_DBUSSTUBREGISTRY_H
#define ASF_DBUS_DBUSSTUBREGISTRY_H

#include "asf/core/ComponentDescription.h"
#include "asf/core/Logger.h"
#include "asf/core/Types.h"
#include "asf/threading/Mutex.h"

#include <map>
#include <string>
#include <vector>

class DBusStubRegistryTest;

namespace asf {
namespace dbus {

class DBusStubDelegateCallbackIF;
class DBusStub;
class DBusStubDelegate;

/**
 * The DBusStubRegistry is used to determine the stub and with it the
 * component which belongs to a received dbus message. Whenever a dbus
 * stub message is received a lookup
 * in the registry happens.
 *
 * The DBusStubRegistry stores all available stubs and provides finder
 * method to lookup a certain stub by a the serviceId of a
 * received method.
 *
 * There is a single stub registry for each DBusConnector.
 * Whenever a D-Bus stub is created it adds itself to the registry.
 * On destruction it removes itself from the registry.
 *
 * The DBusStubRegistry is thread safe and can be used concurrently from
 * multiple threads.
 *
 * The DBusStubRegistry creates and deletes Introspectable and Properties
 * stubs for D-Bus objects.
 */
class DBusStubRegistry {
public:
    /**
     * Adds a listener for interfaces.
     *
     * The listener will be called when an interfaces is added underneath the given objectPath
     */
    void addInterfaceListener(DBusStubDelegateCallbackIF* cb, const ::std::string& objectPath);

    bool deregisterStubDelegate(DBusStubDelegate* stubDelegate);

    /**
     * Return all the child object paths of the given object path.
     * Returns only the first level of child objects.
     * The returned paths consists only of the relative object path.
     */
    void findChildObjects(const std::string& objectPath, std::vector< std::string >& childObjects);

    /**
     * Return the list of object paths underneath the given objectPath.
     */
    void findSubObjectPaths(const std::string& objectPath, std::vector< std::string >& objects);

    /**
     * Return the stubs of a given objectPath.
     */
    void findStubDelegates(const std::string& objectPath,
                           std::vector< DBusStubDelegate* >& stubDelegates);

    /**
     * Return the stub of a given busname, objectPath and interface.
     */
    DBusStubDelegate* findStubDelegate(const std::string& objectPath,
                                       const std::string& interfaceName);

    /**
     * Check if a given object path is present in the registry
     */
    bool hasObjectPath(const std::string& objectPath);

    void registerStubDelegate(DBusStubDelegate* stub);

    /**
     * Remove a listener previously added.
     * Returns true when the listener was removed, otherwise false.
     */
    bool removeInterfaceListener(DBusStubDelegateCallbackIF* cb);

private:
    void deleteStub(const std::string& objectPath, const std::string& interfaceName);

    void notifyListeners(const ::std::vector< DBusStubDelegate* >& stubDelegates,
                         bool removal,
                         bool objectPathFlag);

    struct ServiceId {
        std::string _objectPath;
        std::string _interfaceName;

        ServiceId(const std::string& objectPath, const std::string& interfaceName)
            : _objectPath(objectPath), _interfaceName(interfaceName) {}

        bool operator<(const ServiceId& a) const;
    };

    typedef std::map< ServiceId, DBusStubDelegate* > ServiceIdToStubDelegateMap;

    /**
     * Stores the registered StubDelegates
     */
    ServiceIdToStubDelegateMap _stubDelegateMap;

    class Listener {
    public:
        Listener(DBusStubDelegateCallbackIF* cb, const ::std::string objectPath)
            : _cb(cb), _objectPath(objectPath) {}

        DBusStubDelegateCallbackIF* _cb;

        ::std::string _objectPath;
    };

    typedef std::vector< Listener > Listeners;

    Listeners _listeners;

    ::asf::threading::Mutex _lock;

    friend class ::DBusStubRegistryTest;

    DECLARE_CLASS_LOGGER();
};

}  // namespace dbus
}  // namespace asf

#endif
