/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     DBusProxyRegistry.h
 *\brief
 *
 *\author   CM-AI/PJ-G31
 *          stefan.baron3@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2012 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/

#ifndef ASF_DBUS_DBUSPROXYREGISTRY_H
#define ASF_DBUS_DBUSPROXYREGISTRY_H

#include "asf/core/IncomingOutgoingIF.h"
#include "asf/core/Logger.h"
#include "asf/core/Proxy.h"
#include "asf/core/Types.h"
#include "asf/threading/Mutex.h"

#include <dbus/dbus.h>
#include <map>
#include <string>
#include <vector>

namespace asf {
namespace core {

class ConnectionIF;
}
}  // namespace asf

namespace asf {
namespace dbus {

class DBusProxyDelegate;

/**
 * The DBusProxyRegistry is used to determine the proxy and with it the
 * component which belongs to a received D-Bus signal. D-Bus messages
 * don't make use of the DBusProxyRegistry. Whenever a D-Bus
 * proxy signal (a signal from server to client) is received a lookup
 * in the registry happens.
 *
 * The DBusProxyRegistry stores all proxies and provides finder
 * method to lookup a certain proxy by the busname, object path and interface name
 * of a received signal.
 *
 * There is a single proxy registry for each D-Bus connector in the system. Whenever a D-Bus proxy
 * instance is created it adds itself to the registry. On destruction it
 * removes itself from the registry.
 *
 * Each services proxy instantiated a DBusDaemon and a DBusProperties proxy. These internal
 * proxies will also be registered to the DBusProxyRegistry. Please note, that for a given
 * busname, object path and interface name multiple proxies could be registered.
 *
 * The DBusProxyRegistry is thread safe and can be used concurrently from
 * multiple threads.
 */
class DBusProxyRegistry {
public:
    typedef std::vector< ::boost::shared_ptr< DBusProxyDelegate > > ProxyDelegateList;

    bool registerProxy(const ::boost::shared_ptr< DBusProxyDelegate >& proxy);

    bool deregisterProxy(const ::boost::shared_ptr< DBusProxyDelegate >& proxy);

    ::boost::shared_ptr< ProxyDelegateList > findProxy(const std::string& busName,
                                                       const std::string& objectPath,
                                                       const std::string& interfaceName);

    /**
     * This method returns all D-Bus daemon proxies for a given busname.
     */
    ::boost::shared_ptr< ProxyDelegateList > findAllDaemonProxiesOnBus(const std::string& busName);

    void onConnect(::asf::core::ConnectionIFSharedPtr connection);

    void onDisconnect(::asf::core::ConnectionIFSharedPtr connection);

private:
    /**
     * Returns all registered proxy delegates for the current component thread and the given
     * connection.
     */
    ::boost::shared_ptr< ProxyDelegateList >
    findAllProxyDelegatesForConnectionAndCurrentComponentThread(
        ::asf::core::ConnectionIFSharedPtr connection);

    struct ServiceId {
        std::string _busName;
        std::string _objectPath;
        std::string _interfaceName;

        bool operator<(const ServiceId& a) const;
    };

    typedef std::multimap< ServiceId, ::boost::weak_ptr< DBusProxyDelegate > >
        ServiceIdToProxyMultiMap;

    ServiceIdToProxyMultiMap _proxyMultiMap;

    ::asf::threading::Mutex _lock;

    DECLARE_CLASS_LOGGER();
};

}  // namespace dbus
}  // namespace asf

#endif
