/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/

/*!
 *\file     DBusDispatcher.h
 *\brief
 *
 *\author   CM-AI/ECA2
 *          stefan.baron3@de.bosch.com
 *
 *\par Copyright:
 *(c) 2015-2015 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef DBUSDISPATCHER_H_
#define DBUSDISPATCHER_H_

#include "asf/core/Logger.h"

#include "asf/threading/Mutex.h"

#include "com/bosch/cm/asf/lang/dbus/Connectors.h"

#include "IDBusDispatcher.h"

#include <dbus/dbus.h>

#include <sys/poll.h>

#include <map>
#include <queue>

class DBusDispatcherTest;

namespace asf {
namespace dbus {

/**
 * The DBusSimpleDispatcher makes use of the dbus_connection_read_write_dispatch() function which
 * reads and writes D-Bus messages on the given connection. The
 * drawback of using the dispatcher is, that the dbus_connection_read_write_dispatch() blocks for a
 * given timeout. This decreases the reactivity of D-Bus components.
 * Simple in that case means, that this dispatcher should only be used for non-time critical D-Bus
 * applications.
 */
class DBusSimpleDispatcher : public IDBusDispatcher {
public:
    DBusSimpleDispatcher(
        ::DBusConnection* dbusConnection,
        ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector& connectorOptions)
        : _dbusConnection(dbusConnection), _timeout(connectorOptions.getPollingTimeout()) {
        LOG_INFO("Instantiating simple dispatcher for connection %p with polling timeout %u ms",
                 _dbusConnection,
                 _timeout);
    };

    ~DBusSimpleDispatcher(){};

    void dispatch() {
        dbus_connection_read_write_dispatch(_dbusConnection, static_cast< int >(_timeout));
    };

    void send(::DBusMessage* message) {
        if (message && TRUE == dbus_connection_send(_dbusConnection, message, 0)) {
            LOG_DEBUG("Message %p sent on connection %p", message, _dbusConnection);
        }
    }

    void shutdown(){};

private:
    ::DBusConnection* _dbusConnection;

    ::asf::ms_t _timeout;

    DECLARE_CLASS_LOGGER();
};

/**
 * This class is the default dispatcher which will be instantiated. Dispatching D-Bus messages are
 * realized by file description in connection with the D-Bus watch
 * mechanism. The dispatcher gets active immediately on pending messages.
 */
class DBusDispatcher : public IDBusDispatcher {
public:
    DBusDispatcher(::DBusConnection* dbusConnection);

    ~DBusDispatcher();

    void dispatch();

    void send(::DBusMessage* message);

    void shutdown();

private:
    static dbus_bool_t add_watch(DBusWatch* watch, void* data);

    static void remove_watch(DBusWatch* watch, void* data);

    static void toggle_watch(DBusWatch* watch, void* data);

    void init();

    void deinit() const;

    void addWatch(DBusWatch* watch);

    bool removeWatch(DBusWatch* watch);

    void toggleWatch(DBusWatch* watch);

    bool isWatchActive(DBusWatch* watch);

    void fdDispatch(short revents, DBusWatch* watch);

    void readEventFd(int fd) const;

    void writeEventFd(int fd) const;

    bool isConnected() const {
        return (_dbusConnection && TRUE == dbus_connection_get_is_connected(_dbusConnection));
    }

    ::DBusConnection* _dbusConnection;

    typedef std::map< ::DBusWatch*, pollfd > WatchMap;

    WatchMap _watchMap;

    ::asf::threading::Mutex _lockWatchMap;

    /**
     * The _shutdownFd is used with eventfd() to wake up the blocking poll() function in shutdown
     * case.
     */
    struct pollfd _shutdownFd;

    /**
     * The send queue stores all messages to send out.
     */
    ::asf::threading::Mutex _lockDBusConnection;

    bool _shutdownCalled;

    friend class ::DBusDispatcherTest;

    DECLARE_CLASS_LOGGER();
};

}  // namespace dbus
}  // namespace asf

#endif /* DBUSDISPATCHER_H_ */
