/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2017
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     Uri.h
 *\brief    Uniform Resource Identifier
 *
 *          Non-Hierarchical URIs:
 *          ----------------------
 *
 *          mailto:christoph.perick@de.bosch.com
 *          \____/ \___________________________/
 *            |                  |
 *          scheme              path
 *
 *
 *          Examples:
 *          news:comp.os.linux.answers
 *          mailto:john.doe@example.com
 *          tel:+1-816-555-1212
 *
 *
 *          Hierarchical URIs:
 *          ------------------
 *
 *           ftp://christoph.perick:password@www.bosch.com:8021
 *           \_/   \______________/ \______/ \___________/ \__/
 *            |            |            |          |        |
 *          scheme      username     password     host     port
 *                 \__________________________________________/
 *                                       |
 *                                   authority
 *
 *           http://www.bosch.com/resource/?key=value#page1
 *           \__/   \___________/\_______/  \_______/ \___/
 *            |            |         |          |       |
 *          scheme     authority    path      query   fragment
 *
 *
 *          Examples:
 *          http://www.bosch.com
 *          file:///bin/bash
 *          cms-bin://localhost:123456/echoservice
 *          cms-json://localhost:123456/echoservice
 *          cms-someip://localhost:123456/echoservice
 *          cms-bin-uds://%2Ftmp%2Fechoserver.socket/echoservice
 *          cms-json-uds://%2Ftmp%2Fechoserver.socket/echoservice
 *          cms-ws://localhost:123456/echoservice
 *          cca://osalqueue:42
 *
 *
 *\author   CM/ESA2
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2017 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_CORE_URI_H
#define ASF_CORE_URI_H

#include <string>

namespace asf {
namespace core {

class Uri {
public:
    Uri(const std::string& uri);

    const std::string& getUri() const { return _uri; }

    const std::string& getScheme() const { return _scheme; }

    const std::string& getUsername() const { return _username; }

    const std::string& getPassword() const { return _password; }

    const std::string& getHost() const { return _host; }

    int getPort() const { return _port; }

    const std::string& getPath() const { return _path; }

    const std::string& getQuery() const { return _query; }

    const std::string& getFragment() const { return _fragment; }

    bool isHierachical() const { return _isHierachical; }

    bool isValid() const { return _isValid; }

private:
    Uri() : _isValid(true), _isHierachical(false), _port(0) {}

    std::string decode(const std::string& input);

    std::string encode(const std::string& input) const;

    std::string extractFromHead(std::string& input, std::string delimiter) const;

    std::string extractFromTail(std::string& input, std::string delimiter) const;

    std::string charToHex(const char c) const;

    unsigned char hexToChar(const std::string& str);

    void parse();

    std::string _uri;

    bool _isValid;

    bool _isHierachical;

    std::string _scheme;

    std::string _username;

    std::string _password;

    std::string _host;

    int _port;

    std::string _path;

    std::string _query;

    std::string _fragment;
};
}  // namespace core
}  // namespace asf

#endif  // ASF_CORE_URI_H
