/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012 - 2013
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     TokenPool.cpp
 *\brief    This class is used as a pool for tokens, e.g. CCT (Client Completion Token) in CMS.
 *
 *\author   CM-AI/PJ-CF12.2
 *          denys.barankevych@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2013 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/

#ifndef TOKENPOOL_H_
#define TOKENPOOL_H_

#include "asf/core/Logger.h"
#include "asf/core/Types.h"
#include "asf/threading/Guard.h"
#include "asf/threading/Mutex.h"

#include <cassert>
#include <queue>

namespace asf {
namespace core {

template < typename T >
class TokenPool {
public:
    /**
     * maxValue - maximum value of the returned token (e.g. UINT32_MAX)
     * gapSize - defines when returned tokens may be reused. If then number of returned tokens is >=
     * gapSize,
     *              tokens may be reused. This is done to avoid constant reuse of the same
     * token.
     * queueMaxSize - maximum number of returned tokens that can be hold
     */
    TokenPool(const T maxValue,
              const typename std::queue< T >::size_type gapSize,
              const typename std::queue< T >::size_type queueMaxSize)
        : _maxValue(maxValue), _gapSize(gapSize), _queueMaxSize(queueMaxSize), _queue(), _peak(0) {}

    /**
     * Method has 2 cases:
     * 1. create new token by incrementing the peak value
     * 2. reusing a token from a queue if queue contains more than gapSize elements
     *       This is done to have some gap in tokens to avoid reuse of the same token often,
     *       thus making it easier to read logger output
     *
     * Assert will be raised if pool is empty: queue is empty and peak ==  maxValue
     */
    T getToken() {
        ::asf::threading::Guard< ::asf::threading::Mutex > guard(_lock);

        if ((_queue.size() >= _gapSize) && (!_queue.empty())) {
            T id = _queue.front();
            _queue.pop();
            return id;
        } else {
            LOG_SYSTEM_LOGGER();
            // This error is critical. New message can't get an ID,
            // because new token can not be assigned.
            // This may happen when a huge number of requests are pending.
            LOG_ASSERT_FATAL(_peak < _maxValue);
            return ++_peak;
        }
    }

    /**
     * After the token was used it may be returned back to the tokenPool
     * and reused later
     *
     * Assert will be raised if number of returned tokens exceeds maxValue.
     * This may happen when returnToken was called more than maxValue times
     * which in normal case should not happen. Only by mistake.
     *
     * Another assert will be raised if returned id is greater than biggest token.
     */
    void returnToken(T id) {
        ::asf::threading::Guard< ::asf::threading::Mutex > guard(_lock);

        LOG_SYSTEM_LOGGER();
        // This error is critical. A token was released which has been never assigned.
        // This may lead to 2 identical tokens in the queue.
        LOG_ASSERT_FATAL_MSG(id <= _peak, "Can't return token, it has not been created yet");
        // This error is critical. This means that a huge number of tokens was released.
        // This may happen if a lot of responses were processed at a time with no new request
        // coming.
        LOG_ASSERT_FATAL_MSG(_queue.size() < _maxValue,
                             "Invalid state of token pool, queue can't accept more values");

        if (_queue.size() < _queueMaxSize) {
            _queue.push(id);
        }
    }

    /**
     * Value represents how many tokens were newly created. If peak > gapSize this may mean
     * that (peak - gapSize) simultaneous requests are pending in the system
     */
    T getPeakToken() const { return _peak; }

    /**
     * This is a getter for tests. It is not allowed to use it for other purposes.
     */
    const std::queue< T >& test_getQueue() const { return _queue; }

private:
    const T _maxValue;
    const typename std::queue< T >::size_type _gapSize;
    const typename std::queue< T >::size_type _queueMaxSize;
    std::queue< T > _queue;
    T _peak;
    ::asf::threading::Mutex _lock;
};

}  // namespace core
}  // namespace asf

#endif /* TOKENPOOL_H_ */
