/****************************************************************************
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2012
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 ***************************************************************************/
/*!
 *\file     Timer.cpp
 *\brief
 *
 *\author   CM-AI/PJ-CF15
 *          christoph.perick@de.bosch.com
 *
 *\par Copyright:
 *(c) 2012-2012 Robert Bosch Car Multimedia GmbH
 ***************************************************************************/
#ifndef ASF_CORE_TIMER_H
#define ASF_CORE_TIMER_H

#include "asf/core.h"
#include "asf/core/Logger.h"
#include "asf/core/Payload.h"

namespace asf {
namespace core {

class Timer;

class TimerMessage;

class TimerCallbackIF {
public:
    virtual ~TimerCallbackIF() {}

    /**
     * The user has to implement the pure virtual function onExpired(..).
     * The callback will be called when the timer expires.
     * @param timer Reference to the timer, which is responsible for the invocation.
     * @param data Informations about the timer callback (repeat count, reason)
     */
    virtual void onExpired(Timer& timer, ::boost::shared_ptr< TimerPayload > data) = 0;
};

class TimerImplIF;

/**
 * A Timer is used to stimulate a component based on the expiration of a time
 * interval. When the timer expires a message is send to the component owning
 * the timer. A user provided callback is executed when the message is processed.
 */
class Timer {
public:
    static const uint32 Periodic = 0;

    static const uint32 Once = 1;

    /**
     * Creates a new timer. The timer is not configured and inactive after creating.
     * You have to call the start(...) methods to activate the timer.
     */
    Timer();

    /**
     * Creates a new timer and configured it with the given values
     * @param callback The callback which will be called when the timer expires
     * @param interval The time interval in milliseconds which has to expire
     * before calling the callback.
     * @param repeatCount How often the timer will expire. A value of null
     * means the timer will run endless until stop() is called (see constant Periodic).
     * A value of 1 means the timer will expire once (see constant Once). A value
     * > 1 means the timer will repeat n times.
     */
    Timer(TimerCallbackIF& callback, ms_t interval, size_t repeatCount = Once);

    /**
     * Destructs a timer.
     * It is not allowed to destruct a timer before isActive() returns false.
     */
    virtual ~Timer();

    /**
     * Starts the timer and reuses the configuration parameters which have
     * been set before. It is not valid to call this method when no configuration
     * was set before.
     *
     * @returns act, asynchronous completion token. The act is specified later
     * in the callback and can be utilized to pair a start() invocation with
     * a callback stimulation (see TimerPayload::getAct()).

     * Note: Pairing of start() and callback stimulation must be based on the same timer object.
     * A reference to the timer object on which the start() was invoked is always part of
     'onExpired()'
     */
    act_t start();

    /**
     * Starts the timer. Once the time interval expires the callback will
     * be called (reason E_STOPPED).
     *
     * If the timer is already started it will be stopped implicitly.
     *
     * @param callback The callback which will be called when the timer expires
     * @param interval The time interval in milliseconds which has to expire
     * before calling the callback. An interval of 0ms stops the timer,
     * act_t is then returned as 0.
     * @param repeatCount How often the timer will expire. A value of null
     * means the timer will run endless until stop() is called (see Periodic).
     * A value of 1 means the timer will expire once (see SingleShot). A value
     * > 1 means the timer will repeat n times.
     *
     * @returns act, asynchronous completion token. The act is specified later
     * in the callback and can be utilized to pair a start() invocation with
     * a callback stimulation (see TimerPayload::getAct()).
     *
     * Note: Pairing of start() and callback stimulation must be based on the same timer object.
     * A reference to the timer object on which the start() was invoked is always part of
     * 'onExpired()'
     */
    act_t start(TimerCallbackIF& callback, ms_t interval, size_t repeatCount = Once);

    /**
     * Stops the timer. No new timer messages will be created. Be aware
     * that timer messages which are already queued get still processed
     * after calling stop.
     */
    void stop();

    /**
     * Returns true whether the timer is active and the timer callback will be called
     * in future. isActive turns to false when no timer messages are pending
     * It is not allowed to destruct a timer before this function returns false.
     */
    bool isActive() const;

    /**
     * Returns true if a previously started timer was stopped with help of the stop()
     * method or the timer expired by reaching the repeat count.
     */
    bool isStopped() const;

    /**
     * Returns the act returned by the most recent call to any of the start() methods.
     * Returns 0 if no start method was called before.
     */
    act_t getAct() const;

private:
    Timer(const Timer&);

    Timer& operator=(const Timer&);

    void messageProcessed();

    DECLARE_CLASS_LOGGER();

    TimerImplIF* _poImpl;

    friend class TimerMessage;
};
}  // namespace core
}  // namespace asf

#endif  // ASF_CORE_TIMER_H
